import React from 'react';
import PropTypes from 'prop-types';
import { DESCENDING } from './constants';
import { sortConfigFromProp } from './utils';
import { sortConfigSingle } from './proptypes';

const withSortHandler = ComponentToWrap => {
    const SortHandler = class extends React.Component {
        constructor(props){
            super(props);
            let sort;
            if (props.sort !== undefined){
                sort = sortConfigFromProp(props.sort);
            } else {
                sort = sortConfigFromProp(props.defaultSort);
            }
            this.state = {
                sort : sort
            };
            this.updateSort = this.updateSort.bind(this);
        }

        getChildContext(){
            return {
                updateSort: this.updateSort,
                sort: this.state.sort
            };
        }

        componentWillReceiveProps(nextProps){
            const newSort = sortConfigFromProp(nextProps.defaultSort);
            const newState = {};
            if (!(JSON.stringify(this.props.defaultSort) === JSON.stringify(newSort))){
                newState.sort = newSort;
            }
            this.setState(newState);
        }

        updateSort(column, order){
            const { onChangeSort, sort } = this.props;
            const hasOnChangeSort = onChangeSort !== undefined;
            if (sort === undefined || hasOnChangeSort){
                this.setState({sort: {columnName: column, order: order}}, ()=>{
                    hasOnChangeSort && onChangeSort({columnName: column, order: order}); 
                });
            }
        }

        sortObjects(){
            let { sort } = this.state;
            const { objects, columnConfigs, sortValueGetter : defaultSortValueGetter } = this.props;
            if (sort === undefined){
                return objects;
            }
            const { columnName, order } = sort;
            const columnConfig = columnConfigs.find((c)=>c.name === columnName);
            let sortValueGetter;
            if (columnConfig !== undefined && columnConfig.sortValueGetter !== undefined) {
                sortValueGetter = columnConfig.sortValueGetter;
            } else {
                sortValueGetter = defaultSortValueGetter;
            }
  
            // temporary array holds objects with position and sort-value
            let mapped = objects.map(function(el, i) {
                const value = sortValueGetter({value: el[columnName], object: el});
                return { index: i, value: value };
            });
  
  
            //reverse order on "descending" sort option
            const multplier = order === DESCENDING ? -1 : 1;
  
            // sorting the mapped array containing the reduced values
            mapped.sort((a, b) => {
                if (a.value == null){
                    return multplier * -1;
                }
                if (b.value == null){
                    return multplier * 1;
                }
                return multplier * (+(a.value > b.value) || +(a.value === b.value) - 1);
            });
  
            // container for the resulting order
            return mapped.map(function(el){
                return objects[el.index];
            });
  
        }

        render(){
            const props = this.props;
            const objects = this.sortObjects();

            return(
                <ComponentToWrap {...props} objects={objects} />
            );
        }
    };
    SortHandler.childContextTypes = {
        sort: PropTypes.object,
        updateSort: PropTypes.func
    };
    SortHandler.displayName = `SortHandler(${ComponentToWrap.displayName || ComponentToWrap.name || 'Component'})`;
    
    if (process.env.NODE_ENV !== 'production'){
        SortHandler.propTypes = {
            objects: PropTypes.array.isRequired,
            defaultSort: PropTypes.oneOfType([
                PropTypes.string,
                sortConfigSingle
            ]),
            sort: function(props, propName, componentName, ...rest) {
                if (!props[propName]){
                    return null;
                }
                if (!props.onChangeSort){
                    return new Error('You provided a `sort` prop without an `onChangeSort` handler. This will lead to constant sorting. If you want to set the default sorting, use the `defaultSort` prop.'); 
                }
                return PropTypes.oneOfType([
                    PropTypes.string,
                    sortConfigSingle
                ])(props, propName, componentName, ...rest);
            },
            onChangeSort : PropTypes.func,
            columnConfigs: PropTypes.arrayOf(PropTypes.shape({
                sortValueGetter: PropTypes.func
            })).isRequired,
            sortValueGetter: PropTypes.func.isRequired
        };
    }
    
    SortHandler.defaultProps = {
        sortValueGetter:  ({value}) => {
            if (value){
                const valueType = typeof value;
                switch (valueType){
                    case 'number':
                    case 'boolean':
                        return value;
                    case 'string':
                        return value.toLowerCase();
                    default:
                        return JSON.stringify(value).toLowerCase();
                }
            }
            return value;
        }
    };

    return SortHandler;
};

export default withSortHandler;
