import { appGetters, pageGetters, pageMutations, tagsViewGetters, tagsViewMutations } from '../../store';
import ContextMenu from '../../component/ContextMenu';
import HorizontalScroller from '../../component/HorizontalScroller';
import { refreshPage } from '../../helper';
import { getRouterKey, getRouterTitle, isRedirectRouter } from '../../config/logic';
export default {
  name: 'TagsView',

  data() {
    return {
      activeKey: '',
      selectedTag: undefined,
      contextMenu: {
        show: false,
        top: 0,
        left: 0
      }
    };
  },

  computed: {
    visitedViews: () => tagsViewGetters.visitedViews,
    menus: () => appGetters.menus,

    contextMenuItems() {
      return [{
        content: '刷新',
        click: this.refreshSelectedTag
      }, this.visitedViews.length <= 1 || this.selectedTag && this.isAffix(this.selectedTag) ? undefined : {
        content: '关闭',
        click: () => this.closeSelectedTag(this.selectedTag)
      }, {
        content: '关闭其他',
        click: this.closeOthersTags
      }, {
        content: '关闭全部',
        click: this.closeAllTags
      }];
    },

    tagsSlot() {
      return this._u([{
        key: 'default',
        fn: this.renderTags,
        proxy: true
      }]);
    }

  },
  watch: {
    $route(to, from) {
      tagsViewGetters.enableChangeTransition && this.decideRouteTransition(to, from);
      if (isRedirectRouter(to)) return;
      this.setActiveKey(to);
      this.addTag(to);
      this.$nextTick(this.moveToCurrentTag);
    }

  },
  methods: {
    setActiveKey(to) {
      this.activeKey = getRouterKey(to);
    },

    isAffix(view) {
      return view.meta.affix;
    },

    decideRouteTransition(to, from) {
      const {
        next,
        prev
      } = pageGetters.transition;
      const fromKey = getRouterKey(from);
      const toKey = getRouterKey(to);
      let transitionName = prev;
      const fromIndex = this.visitedViews.findIndex(i => i.key === fromKey);
      const toIndex = this.visitedViews.findIndex(i => i.key === toKey);

      if (toIndex === -1 || fromIndex < toIndex) {
        transitionName = next;
      }

      pageMutations.transition({
        curr: transitionName
      });
    },

    getAffixTags(menus) {
      return menus.reduce((affixTags, {
        fullPath,
        children,
        meta
      }) => {
        if (meta.affix === true) {
          const {
            route
          } = this.$router.resolve(fullPath);
          affixTags.push({ ...route,
            meta: {
              affix: true,
              ...route.meta,
              title: getRouterTitle(route)
            }
          });
        }

        if (children) {
          const tempTags = this.getAffixTags(children);
          tempTags.length && affixTags.push(...tempTags);
        }

        return affixTags;
      }, []);
    },

    initTags() {
      this.getAffixTags(this.menus).forEach(tagsViewMutations.addTagOnly);
      this.addTag(this.$route);
    },

    addTag(route) {
      tagsViewMutations.addTagAndCache({ ...route,
        meta: { ...route.meta,
          title: getRouterTitle(route)
        }
      });
    },

    moveToCurrentTag() {
      const scroller = this.$refs['scroller'];
      const cur = Array.from(scroller.$el.children).find(el => el.classList.contains('active'));
      cur && scroller.moveToTarget(cur);
    },

    refreshSelectedTag() {
      refreshPage(this.$router, this.selectedTag);
    },

    closeSelectedTag(view, e) {
      if (this.visitedViews.length <= 1 || this.isAffix(view)) return;
      e && e.preventDefault();
      tagsViewMutations.delTagAndCache(view);
      this.activeKey === view.key && this.gotoLastTag();
    },

    closeOthersTags() {
      tagsViewMutations.delOtherTagAndCache(this.selectedTag);
      this.gotoLastTag();
    },

    closeAllTags() {
      tagsViewMutations.delAllTagAndCache();
      this.gotoLastTag(true);
    },

    gotoLastTag(refresh = false) {
      const views = this.visitedViews;

      if (views.length === 0) {
        return this.$router.push('/');
      }

      const latest = views[views.length - 1];
      this.activeKey === latest.key ? refresh && refreshPage(this.$router) : this.$nextTick(() => this.$router.push(latest));
    },

    openContextMenu(tag, e) {
      e.preventDefault();
      const contextMenuWidth = 105;
      const {
        left: elLeft,
        width: elWidth
      } = this.$el.getBoundingClientRect();
      const maxLeft = elWidth + elLeft - contextMenuWidth;
      const left = e.clientX;
      this.contextMenu.left = (left > maxLeft ? maxLeft : left) + 15;
      this.contextMenu.top = e.clientY;
      this.contextMenu.show = true;
      this.selectedTag = tag;
    },

    renderTags() {
      const h = this.$createElement;
      return this.visitedViews.map((view, _, arr) => {
        const active = this.activeKey === view.key;
        const className = {
          'tags-view-item': true,
          active
        };
        const affix = this.isAffix(view);
        const showClose = !affix && arr.length > 1;
        const on = {
          contextmenu: e => this.openContextMenu(view, e)
        };

        const onIconClick = e => {
          e.stopPropagation();
          this.closeSelectedTag(view, e);
        };

        if (!active) {
          on.click = () => this.$router.push(view, () => undefined);
        }

        return h("div", {
          "key": view.key,
          "class": className,
          "on": { ...on
          }
        }, [h("span", [view.meta.title]), showClose && h("i", {
          "class": "el-icon-close",
          "on": {
            "click": onIconClick
          }
        })]);
      });
    }

  },

  mounted() {
    this.setActiveKey(this.$route);
    this.initTags();
  },

  render() {
    const h = arguments[0];
    return h("nav", {
      "class": "tags-view"
    }, [h(HorizontalScroller, {
      "ref": "scroller",
      "scopedSlots": this.tagsSlot
    }), h(ContextMenu, {
      "attrs": {
        "items": this.contextMenuItems,
        "left": this.contextMenu.left,
        "top": this.contextMenu.top
      },
      "model": {
        value: this.contextMenu.show,
        callback: $$v => {
          this.$set(this.contextMenu, "show", $$v);
        }
      }
    })]);
  }

};