import MenuItem from '../../component/ElMenu/item';
import SubMenu from '../../component/ElMenu/sub';
import { isEmpty } from "../../util";

function getSingleMenu(menu) {
  const {
    children,
    meta: {
      alwaysShow = true
    } = {}
  } = menu;
  if (!children) return menu;
  if (children.length === 0) return { ...menu,
    children: undefined
  };
  if (children.length === 1) return alwaysShow ? null : getSingleMenu(children[0]);
  return null;
}

export default {
  name: 'NavMenu',
  props: {
    menus: {
      type: Array,
      default: () => []
    },
    theme: {
      type: String,
      default: 'light'
    },
    inlineIndent: Number,
    showParentOnCollapse: Boolean,
    switchTransitionName: String,
    menuIconSlot: Function,
    menuContentSlot: Function,
    mode: {
      type: String,
      default: 'vertical'
    },
    collapse: Boolean,
    defaultActive: String,
    defaultOpeneds: Array,
    uniqueOpened: Boolean
  },
  computed: {
    children() {
      const h = this.$createElement;
      let children = this.renderMenus(h, this.menus);
      const {
        switchTransitionName: name
      } = this;

      if (!isEmpty(name)) {
        children = h('transition-group', {
          props: {
            name
          }
        }, children);
      }

      return children;
    },

    themeClass() {
      return `el-menu--${this.theme}`;
    },

    menuClass() {
      return `el-menu--${this.mode} ${this.themeClass}`;
    }

  },
  methods: {
    onSelect(...args) {
      this.$emit('select', ...args);
    },

    renderMenuIcon(h, menu, depth) {
      const slot = this.menuIconSlot;
      if (slot) return slot(h, {
        menu,
        depth
      });
      const icon = menu.meta.icon;
      return icon && h("i", {
        "class": `menu-icon ${icon}`
      });
    },

    renderMenuContent(h, menu, depth) {
      const slot = this.menuContentSlot;
      return slot ? slot(h, {
        menu,
        depth
      }) : h("span", [menu.meta.title]);
    },

    renderSingleMenu(h, menu, depth) {
      const {
        fullPath
      } = menu;
      return h(MenuItem, {
        "key": fullPath,
        "attrs": {
          "index": fullPath,
          "inline-indent": this.inlineIndent
        }
      }, [this.renderMenuIcon(h, menu, depth), h("template", {
        "slot": "title"
      }, [this.renderMenuContent(h, menu, depth)])]);
    },

    renderSubMenu(h, menu, children, depth) {
      const {
        fullPath
      } = menu;
      const noContent = depth === 1 && this.collapse && this.mode === 'vertical';
      return h(SubMenu, {
        "key": fullPath,
        "attrs": {
          "index": fullPath,
          "inline-indent": this.inlineIndent,
          "popper-class": this.themeClass,
          "popper-append-to-body": true
        }
      }, [h("template", {
        "slot": "title"
      }, [this.renderMenuIcon(h, menu, depth), !noContent && this.renderMenuContent(h, menu, depth)]), children]);
    },

    renderChildrenWithParentMenu(h, menu, children, depth) {
      return [h("div", {
        "class": "popover-menu__title"
      }, [this.renderMenuIcon(h, menu, depth), this.renderMenuContent(h, menu, depth)]), h("div", {
        "class": "el-menu el-menu--inline"
      }, [children])];
    },

    renderMenus(h, menus, depth = 1) {
      return menus.map(menu => {
        const singleMenu = getSingleMenu(menu);

        if (singleMenu) {
          return this.renderSingleMenu(h, singleMenu, depth);
        }

        let children = this.renderMenus(h, menu.children, depth + 1);

        if (this.collapse && this.showParentOnCollapse) {
          children = this.renderChildrenWithParentMenu(h, menu, children, depth + 1);
        }

        return this.renderSubMenu(h, menu, children, depth);
      });
    }

  },

  render() {
    const h = arguments[0];
    return h("el-menu", {
      "ref": "el-menu",
      "class": this.menuClass,
      "attrs": {
        "mode": this.mode,
        "collapse": this.collapse,
        "collapse-transition": false,
        "unique-opened": this.uniqueOpened,
        "default-active": this.defaultActive,
        "default-openeds": this.defaultOpeneds
      },
      "on": {
        "select": this.onSelect
      }
    }, [this.children]);
  }

};