import _mergeJSXProps from "@vue/babel-helper-vue-jsx-merge-props";
import { pageGetters, tagsViewGetters } from '../../store';
import { getRouterKey } from '../../config/logic';
import { isEmpty } from '../../util';

function pruneCacheEntry(key, instance) {
  if (!instance) return;
  const {
    cache,
    _vnode: current
  } = instance;
  const entry = cache[key];

  if (entry && (!current || entry.tag !== current.tag)) {
    entry.componentInstance.$destroy();
  }

  delete cache[key];
  instance.keys = [];
}

function getCtorId(instance) {
  const opt = instance.$vnode.componentOptions;
  return opt && opt.Ctor.cid;
}

export default {
  name: 'PageView',

  data() {
    return {
      renderView: true
    };
  },

  computed: {
    routerViewKey() {
      return getRouterKey(this.$route);
    },

    useKeepAlive() {
      return tagsViewGetters.enabled && tagsViewGetters.enableCache;
    },

    keepAliveData() {
      const timestamp = Date.now();
      const exclude = tagsViewGetters.cachedViews.map(key => key + timestamp);
      return {
        props: {
          exclude
        },
        on: {
          'hook:updated': this.onKeepAliveUpdated
        }
      };
    },

    cachedKeyMap() {
      return tagsViewGetters.cachedViews.reduce((map, key) => {
        map[key] = 1;
        return map;
      }, {});
    },

    useTransition() {
      return pageGetters.enableTransition;
    },

    transitionName() {
      return pageGetters.transition.curr;
    }

  },
  methods: {
    getKeepAliveInstance() {
      if (!this._vnode) return;
      const vnode = this._vnode.children[0];

      if (!pageGetters.enableTransition) {
        return vnode.componentInstance;
      }

      return vnode.child._vnode.componentInstance;
    },

    onKeepAliveUpdated() {
      const instance = this.getKeepAliveInstance();
      const {
        cache
      } = instance;
      const hasTransition = pageGetters.enableTransition;
      Object.keys(cache).forEach(key => {
        const routerKey = hasTransition ? key.replace(/__transition-(\d+)-/, '') : key;
        !this.cachedKeyMap[routerKey] && pruneCacheEntry(key, instance);

        if (process.env.NODE_ENV === 'development') {
          if (cache[key]) {
            const comp = cache[key].componentInstance;
            const lastCtorId = cache[key]._ctorId;
            const ctorId = cache[key]._ctorId = getCtorId(comp);

            if (!isEmpty(lastCtorId) && lastCtorId !== ctorId) {
              pruneCacheEntry(key, instance);
              this.rerenderRouterView();
            }
          }
        }
      });
    },

    rerenderRouterView() {
      if (!this.renderView) return;
      this.renderView = false;
      this.$nextTick(() => this.renderView = true);
    }

  },

  render() {
    const h = arguments[0];
    let view = this.renderView && h("router-view", {
      "key": this.routerViewKey
    });

    if (this.useKeepAlive) {
      view = h("keep-alive", _mergeJSXProps([{}, this.keepAliveData]), [view]);
    }

    if (this.useTransition) {
      view = h("transition", {
        "attrs": {
          "name": this.transitionName,
          "mode": "out-in"
        }
      }, [view]);
    }

    return h("div", {
      "class": "page-view"
    }, [view]);
  }

};