import menuMixin from '../../mixin/menu';
import { appGetters, appMutations, headerGetters } from '../../store';
import NavMenu from '../../component/NavMenu';
import LoadingSpinner from '../../component/LoadingSpinner';
import { getRouterActiveMenu, isRedirectRouter } from '../../config/logic';
import { getMenuByFullPath } from '../../store/app';
import { findFirstLeaf } from '../../util';
export default {
  name: 'HorizontalResizableMenu',
  mixins: [menuMixin],

  data() {
    return {
      lastVisibleIndex: undefined,
      seed: 0
    };
  },

  computed: {
    menus() {
      const menus = appGetters.menus;

      switch (appGetters.navMode) {
        case 'head':
          return menus;

        case 'mix':
          return menus.map(menu => ({ ...menu,
            children: undefined
          }));

        default:
          return [];
      }
    },

    realMenus() {
      const {
        lastVisibleIndex,
        menus
      } = this;

      if (lastVisibleIndex === undefined) {
        return menus;
      }

      const fullPath = `_${this.seed}`;

      if (lastVisibleIndex === -1) {
        return [{
          fullPath,
          meta: {
            icon: 'el-icon-menu'
          },
          children: menus
        }];
      }

      const visible = menus.slice(0, lastVisibleIndex + 1);
      const hidden = menus.slice(lastVisibleIndex + 1);
      visible.push({
        fullPath,
        meta: {
          title: '...'
        },
        children: hidden
      });
      return visible;
    }

  },
  watch: {
    $route: {
      immediate: true,

      handler(to) {
        if (this.setActiveRootMenu(to)) {
          this.setActiveMenu(appGetters.navMode, to);
        }
      }

    },

    lastVisibleIndex() {
      this.seed++;
    }

  },
  methods: {
    setActiveRootMenu(route) {
      const {
        matched
      } = route;

      if (matched.length === 0 || isRedirectRouter(route)) {
        return false;
      }

      const [root] = matched;
      root && appMutations.activeRootMenu(root.path || '/');
      return true;
    },

    setActiveMenu(navMode = appGetters.navMode, route = this.$route) {
      const oldVal = this.activeMenu;

      if (navMode === 'mix') {
        this.activeMenu = appGetters.activeRootMenu;
      } else this.activeMenu = getRouterActiveMenu(route);

      this.activeMenu !== oldVal && this.resetActiveMenu();
    },

    onSelect(index) {
      if (appGetters.navMode === 'mix') {
        return this.onSelectRootMenu(index);
      }

      this.actionOnSelectMenu(index);
    },

    onSelectRootMenu(index) {
      const root = getMenuByFullPath(index);
      const {
        leaf,
        hasOtherLeaf
      } = findFirstLeaf(root);

      if (!leaf || appGetters.activeRootMenu === index && hasOtherLeaf) {
        return;
      }

      this.actionOnSelectMenu(leaf.fullPath);
    },

    getMenuEl() {
      return this.$el;
    },

    setChildrenWidth() {
      const ul = this.getMenuEl();
      if (!ul) return;
      const menuItemNodes = ul.children;
      if (!menuItemNodes || menuItemNodes.length === 0) return;
      this.menuItemSizes = Array.from(menuItemNodes).map(i => i.getBoundingClientRect().width);
      this.originalTotalWidth = this.menuItemSizes.reduce((acc, cur) => acc + cur, 0);
    },

    setOverflowedIndicatorWidth() {
      const ul = document.createElement('ul');
      ul.className = 'el-menu--horizontal el-menu el-menu--horizontal';
      ul.style.position = 'fixed';
      ul.style.top = '-100px';
      ul.style.right = '-1000px';
      ul.innerHTML = `<li class="el-submenu"><div class="el-submenu__title"><span>...</span></div></li>`;
      document.body.appendChild(ul);
      this.overflowedIndicatorWidth = ul.children[0].offsetWidth + 1;
      document.body.removeChild(ul);
    },

    resize() {
      const width = this.getMenuEl().getBoundingClientRect().width;
      let lastVisibleIndex = undefined;

      if (this.originalTotalWidth > width) {
        lastVisibleIndex = -1;
        const {
          menuItemSizes,
          overflowedIndicatorWidth
        } = this;

        for (let i = menuItemSizes.length - 1, sum = 0; i >= 0; i--) {
          sum += menuItemSizes[i];

          if (sum + overflowedIndicatorWidth > width) {
            break;
          }

          lastVisibleIndex += 1;
        }
      }

      this.lastVisibleIndex = lastVisibleIndex;
    }

  },

  created() {
    this.$watch('realMenus', v => {
      this.setActiveMenu();
      this.setDefaultActiveMenu(v);
    }, {
      immediate: true
    });
  },

  async mounted() {
    await this.$nextTick();
    this.setOverflowedIndicatorWidth();
    this.setChildrenWidth();
    this.resizeObserver = new window.ResizeObserver(this.resize);
    this.resizeObserver.observe(this.getMenuEl());
  },

  beforeDestroy() {
    if (this.resizeObserver) {
      this.resizeObserver.disconnect();
      this.resizeObserver = null;
    }
  },

  render() {
    const h = arguments[0];

    if (appGetters.loadingMenu) {
      return h("div", {
        "style": "position: relative;width: 100%;height: 100%"
      }, [h(LoadingSpinner)]);
    }

    if (this.menus.length === 0) return;
    return h(NavMenu, {
      "ref": "nav-menu",
      "attrs": {
        "menus": this.realMenus,
        "theme": headerGetters.theme,
        "mode": "horizontal",
        "default-active": this.defaultActive,
        "menu-icon-slot": headerGetters.menuIconSlot,
        "menu-content-slot": headerGetters.menuContentSlot
      },
      "on": {
        "select": this.onSelect
      }
    });
  }

};