function pruneCacheEntry(key, instance) {
  if (!instance) return;
  const {
    cache,
    _vnode: current
  } = instance;
  const entry = cache[key];

  if (entry && (!current || entry.tag !== current.tag)) {
    entry.componentInstance.$destroy();
  }

  delete cache[key];
  instance.keys = [];
}

function getCtorId(instance) {
  const opt = instance.$vnode.componentOptions;
  return opt && opt.Ctor.cid;
}

export default {
  name: 'CachedRouterView',
  inheritAttrs: true,
  props: {
    cacheable: {
      type: Boolean,
      default: true
    },
    tag: {
      type: String,
      default: 'div'
    },
    transitionProps: Object,
    keyFn: Function,
    includes: {
      type: Array,
      default: () => []
    }
  },

  data() {
    return {
      renderView: true
    };
  },

  computed: {
    routerViewKey() {
      return this.$props.keyFn(this.$route);
    },

    cachedKeyMap() {
      return this.$props.includes.reduce((map, key) => {
        map[key] = 1;
        return map;
      }, {});
    }

  },
  methods: {
    getKeepAliveInstance() {
      if (!this._vnode) return;
      let vnode = this._vnode.children[0];
      const wrappedByTransition = vnode.componentOptions.tag === 'transition';

      if (wrappedByTransition) {
        vnode = vnode.child._vnode;
      }

      return {
        instance: vnode.componentInstance,
        wrappedByTransition
      };
    },

    onKeepAliveUpdated() {
      const {
        instance,
        wrappedByTransition
      } = this.getKeepAliveInstance();
      const {
        cache
      } = instance;
      Object.keys(cache).forEach(key => {
        const cachedKey = wrappedByTransition ? key.replace(/__transition-(\d+)-/, '') : key;
        !this.cachedKeyMap[cachedKey] && pruneCacheEntry(key, instance);

        if (process.env.NODE_ENV === 'development') {
          if (cache[key]) {
            const comp = cache[key].componentInstance;
            const lastCtorId = cache[key]._ctorId;
            const ctorId = cache[key]._ctorId = getCtorId(comp);

            if (lastCtorId != null && lastCtorId !== ctorId) {
              pruneCacheEntry(key, instance);
              this.rerenderRouterView();
            }
          }
        }
      });
    },

    rerenderRouterView() {
      if (!this.renderView) return;
      this.renderView = false;
      this.$nextTick(() => this.renderView = true);
    }

  },

  render(h) {
    const {
      cacheable,
      transitionProps,
      tag,
      includes
    } = this.$props;
    this.$data.$_tmp_ = includes.length;
    let view = this.renderView && h('router-view', {
      key: this.routerViewKey
    });

    if (cacheable) {
      const data = {
        on: {
          'hook:updated': this.onKeepAliveUpdated
        }
      };
      view = h('keep-alive', data, [view]);
    }

    if (transitionProps) {
      view = h('transition', {
        props: transitionProps
      }, [view]);
    }

    return tag ? h(tag, [view]) : view;
  }

};