import { appGetters, pageGetters, pageMutations, tagsViewGetters, tagsViewMutations } from '../../store';
import useContextMenu from '../../component/ContextMenu/functionalUse';
import HorizontalScroller from '../../component/HorizontalScroller';
import { refreshPage } from '../../helper';
import { getRouterKey, getRouterTitle, isRedirectRouter } from '../../config/logic';
import { isAffix, getAffixTagsFromMenuTree, renderDefaultStyleTag } from './util';
export default {
  name: 'TagsView',

  data() {
    return {
      activeKey: '',
      selectedTag: undefined,
      contextMenu: {
        show: false,
        top: 0,
        left: 0
      }
    };
  },

  computed: {
    contextMenuItems() {
      const {
        visitedViews
      } = tagsViewGetters;
      const closeable = visitedViews.length > 1 && !isAffix(this.selectedTag);
      return [{
        content: '刷新',
        click: this.refreshSelectedTag
      }, {
        content: closeable && '关闭',
        click: this.closeSelectedTag
      }, {
        content: '关闭其他',
        click: this.closeOthersTags
      }, {
        content: '关闭全部',
        click: this.closeAllTags
      }].filter(i => i.content);
    },

    tagsSlot() {
      return this._u([{
        key: 'default',
        fn: this.renderTags,
        proxy: true
      }]);
    }

  },
  watch: {
    $route(to, from) {
      tagsViewGetters.enableChangeTransition && this.decideRouteTransition(to, from);
      if (isRedirectRouter(to)) return;
      this.setActiveKey(to);
      this.addTag(to);
      this.$nextTick(this.moveToCurrentTag);
    }

  },
  methods: {
    setActiveKey(route) {
      this.activeKey = getRouterKey(route);
    },

    decideRouteTransition(to, from) {
      const {
        next,
        prev
      } = pageGetters.transition;
      const {
        visitedViews
      } = tagsViewGetters;
      const fromKey = getRouterKey(from);
      const toKey = getRouterKey(to);
      const fromIndex = visitedViews.findIndex(i => i.key === fromKey);
      const toIndex = visitedViews.findIndex(i => i.key === toKey);
      let transitionName = prev;

      if (toIndex === -1 || fromIndex < toIndex) {
        transitionName = next;
      }

      pageMutations.transition({
        curr: transitionName
      });
    },

    initTags() {
      getAffixTagsFromMenuTree(this.$router, appGetters.menus).forEach(tagsViewMutations.addTagOnly);
      this.addTag(this.$route);
    },

    addTag(route) {
      tagsViewMutations.addTagAndCache({ ...route,
        meta: { ...route.meta,
          title: getRouterTitle(route)
        }
      });
    },

    moveToCurrentTag() {
      const {
        scroller
      } = this.$refs;
      const cur = Array.from(scroller.$el.children).find(el => el.classList.contains('active'));
      cur && scroller.moveToTarget(cur);
    },

    gotoLastTag(refresh = false) {
      const views = tagsViewGetters.visitedViews;
      const router = this.$router;

      if (views.length === 0) {
        return router.push('/');
      }

      const latest = views[views.length - 1];
      this.activeKey === latest.key ? refresh && refreshPage(router) : this.$nextTick(() => router.push(latest));
    },

    refreshSelectedTag(view = this.selectedTag) {
      refreshPage(this.$router, view);
    },

    closeSelectedTag(view = this.selectedTag) {
      if (tagsViewGetters.visitedViews.length <= 1 || isAffix(view)) return;
      tagsViewMutations.delTagAndCache(view);
      this.activeKey === view.key && this.gotoLastTag();
    },

    closeOthersTags() {
      tagsViewMutations.delOtherTagAndCache(this.selectedTag);
      this.gotoLastTag();
    },

    closeAllTags() {
      tagsViewMutations.delAllTagAndCache();
      this.gotoLastTag(true);
    },

    openContextMenu(tag, e) {
      e.preventDefault();
      this.$contextmenu && this.$contextmenu();
      this.selectedTag = tag;
      this.$contextmenu = useContextMenu(this.contextMenuItems, {
        left: e.clientX + 15,
        top: e.clientY + 5
      });
    },

    renderTags() {
      const {
        $createElement: h,
        $router,
        activeKey
      } = this;
      const {
        itemSlot,
        visitedViews
      } = tagsViewGetters;
      const renderFn = itemSlot || renderDefaultStyleTag;
      return visitedViews.map(view => {
        const active = activeKey === view.key;
        const showClose = !isAffix(view) && visitedViews.length > 1;
        const on = {
          contextmenu: e => this.openContextMenu(view, e)
        };

        const onClose = e => {
          e.stopPropagation();
          this.closeSelectedTag(view);
        };

        if (!active) {
          on.click = () => $router.push(view, () => undefined);
        }

        return renderFn(h, {
          key: view.key,
          active,
          on,
          title: view.meta.title,
          close: showClose && onClose
        });
      });
    }

  },

  mounted() {
    this.setActiveKey(this.$route);
    this.initTags();
  },

  render() {
    const h = arguments[0];
    return h("nav", {
      "class": "tags-view"
    }, [h(HorizontalScroller, {
      "ref": "scroller",
      "scopedSlots": this.tagsSlot
    })]);
  }

};