/**
 * 判断是否为空值，undefined、null、'' 都视为空值
 *
 * @param str         不定参数
 * @return {boolean}  若为空值，返回true，否则返回false
 */
export function isEmpty(...str) {
  return str.some(i => i === undefined || i === null || i === '')
}

/**
 * 防抖
 *
 * @param func {function}      原函数
 * @param wait {number}        防抖间隔，单位毫秒
 * @param immediate {boolean}  是否立即执行一次
 * @return {function}          经过防抖包装后的函数
 */
export function debounce(func, wait = 100, immediate = false) {
  let timeout, args, context, timestamp, result

  const later = function() {
    // 据上一次触发时间间隔
    const last = new Date().getTime() - timestamp

    // 上次被包装函数被调用时间间隔 last 小于设定时间间隔 wait
    if (last < wait && last > 0) {
      timeout = window.setTimeout(later, wait - last)
    }
    else {
      timeout = null
      // 如果设定为immediate===true，因为开始边界已经调用过了此处无需调用
      if (!immediate) {
        result = func.apply(context, args)
        if (!timeout) context = args = null
      }
    }
  }

  return function() {
    context = this
    args = arguments
    timestamp = new Date().getTime()
    const callNow = immediate && !timeout
    // 如果延时不存在，重新设定延时
    if (!timeout) timeout = window.setTimeout(later, wait)
    if (callNow) {
      result = func.apply(context, args)
      context = args = null
    }

    return result
  }
}

/**
 * 以深度优先找到根节点的第一个叶子节点，并判断是否有其他的叶子节点
 *
 * @param node
 * @param hasOtherLeaf
 * @returns {{hasOtherLeaf: boolean, leaf: ({children}|*)}|*}
 */
export function findFirstLeaf(node, hasOtherLeaf = false) {
  if (!node || !node.children || node.children.length === 0) {
    return { leaf: node, hasOtherLeaf }
  }

  return findFirstLeaf(node.children[0], hasOtherLeaf || node.children.length > 1)
}
