import { Fragment, defineComponent, getCurrentInstance, ref, onUpdated, onMounted, watch, nextTick, provide, h, renderSlot } from 'vue';
import { isPromise, NOOP } from '@vue/shared';
import '../../../utils/index.mjs';
import '../../../constants/index.mjs';
import { ElIcon } from '../../icon/index.mjs';
import { Plus } from '@element-plus/icons-vue';
import '../../../tokens/index.mjs';
import TabNav from './tab-nav.mjs';
import { buildProps, definePropType } from '../../../utils/vue/props.mjs';
import { UPDATE_MODEL_EVENT, INPUT_EVENT } from '../../../constants/event.mjs';
import { tabsRootContextKey } from '../../../tokens/tabs.mjs';
import { EVENT_CODE } from '../../../constants/aria.mjs';

const tabsProps = buildProps({
  type: {
    type: String,
    values: ["card", "border-card", ""],
    default: ""
  },
  activeName: {
    type: String,
    default: ""
  },
  closable: Boolean,
  addable: Boolean,
  modelValue: {
    type: [String, Number],
    default: ""
  },
  editable: Boolean,
  tabPosition: {
    type: String,
    values: ["top", "right", "bottom", "left"],
    default: "top"
  },
  beforeLeave: {
    type: definePropType(Function),
    default: () => true
  },
  stretch: Boolean
});
const tabsEmits = {
  [UPDATE_MODEL_EVENT]: (tabName) => typeof tabName === "string" || typeof tabName === "number",
  [INPUT_EVENT]: (tabName) => typeof tabName === "string" || typeof tabName === "number",
  "tab-click": (pane, ev) => ev instanceof Event,
  edit: (paneName, action) => action === "remove" || action === "add",
  "tab-remove": (paneName) => typeof paneName === "string" || typeof paneName === "number",
  "tab-add": () => true
};
const getPaneInstanceFromSlot = (vnode, paneInstanceList = []) => {
  const children = vnode.children || [];
  Array.from(children).forEach((node) => {
    let type = node.type;
    type = type.name || type;
    if (type === "ElTabPane" && node.component) {
      paneInstanceList.push(node.component);
    } else if (type === Fragment || type === "template") {
      getPaneInstanceFromSlot(node, paneInstanceList);
    }
  });
  return paneInstanceList;
};
var Tabs = defineComponent({
  name: "ElTabs",
  props: tabsProps,
  emits: tabsEmits,
  setup(props, { emit, slots, expose }) {
    const instance = getCurrentInstance();
    const nav$ = ref();
    const panes = ref([]);
    const currentName = ref(props.modelValue || props.activeName || "0");
    const paneStatesMap = {};
    const updatePaneInstances = (isForceUpdate = false) => {
      if (slots.default) {
        const children = instance.subTree.children;
        const content = Array.from(children).find(({ props: props2 }) => (props2 == null ? void 0 : props2.class) === "el-tabs__content");
        if (!content)
          return;
        const paneInstanceList = getPaneInstanceFromSlot(content).map((paneComponent) => paneStatesMap[paneComponent.uid]);
        const panesChanged = !(paneInstanceList.length === panes.value.length && paneInstanceList.every((pane, index) => pane.uid === panes.value[index].uid));
        if (isForceUpdate || panesChanged) {
          panes.value = paneInstanceList;
        }
      } else if (panes.value.length !== 0) {
        panes.value = [];
      }
    };
    const changeCurrentName = (value) => {
      currentName.value = value;
      emit(INPUT_EVENT, value);
      emit(UPDATE_MODEL_EVENT, value);
    };
    const setCurrentName = (value) => {
      var _a;
      if (currentName.value === value)
        return;
      const canLeave = (_a = props.beforeLeave) == null ? void 0 : _a.call(props, value, currentName.value);
      if (isPromise(canLeave)) {
        canLeave.then(() => {
          var _a2, _b;
          changeCurrentName(value);
          (_b = (_a2 = nav$.value) == null ? void 0 : _a2.removeFocus) == null ? void 0 : _b.call(_a2);
        }, NOOP);
      } else if (canLeave !== false) {
        changeCurrentName(value);
      }
    };
    const handleTabClick = (tab, tabName, event) => {
      if (tab.props.disabled)
        return;
      setCurrentName(tabName);
      emit("tab-click", tab, event);
    };
    const handleTabRemove = (pane, ev) => {
      if (pane.props.disabled)
        return;
      ev.stopPropagation();
      emit("edit", pane.props.name, "remove");
      emit("tab-remove", pane.props.name);
    };
    const handleTabAdd = () => {
      emit("edit", null, "add");
      emit("tab-add");
    };
    onUpdated(() => updatePaneInstances());
    onMounted(() => updatePaneInstances());
    watch(() => props.activeName, (modelValue) => setCurrentName(modelValue));
    watch(() => props.modelValue, (modelValue) => setCurrentName(modelValue));
    watch(currentName, async () => {
      var _a, _b;
      updatePaneInstances(true);
      await nextTick();
      await ((_a = nav$.value) == null ? void 0 : _a.$nextTick());
      (_b = nav$.value) == null ? void 0 : _b.scrollToActiveTab();
    });
    provide(tabsRootContextKey, {
      props,
      currentName,
      updatePaneState: (pane) => paneStatesMap[pane.uid] = pane
    });
    expose({
      currentName
    });
    return () => {
      const newButton = props.editable || props.addable ? h("span", {
        class: "el-tabs__new-tab",
        tabindex: "0",
        onClick: handleTabAdd,
        onKeydown: (ev) => {
          if (ev.code === EVENT_CODE.enter)
            handleTabAdd();
        }
      }, [h(ElIcon, { class: "is-icon-plus" }, { default: () => h(Plus) })]) : null;
      const header = h("div", { class: ["el-tabs__header", `is-${props.tabPosition}`] }, [
        newButton,
        h(TabNav, {
          currentName: currentName.value,
          editable: props.editable,
          type: props.type,
          panes: panes.value,
          stretch: props.stretch,
          ref: nav$,
          onTabClick: handleTabClick,
          onTabRemove: handleTabRemove
        })
      ]);
      const panels = h("div", { class: "el-tabs__content" }, [
        renderSlot(slots, "default")
      ]);
      return h("div", {
        class: {
          "el-tabs": true,
          "el-tabs--card": props.type === "card",
          [`el-tabs--${props.tabPosition}`]: true,
          "el-tabs--border-card": props.type === "border-card"
        }
      }, props.tabPosition !== "bottom" ? [header, panels] : [panels, header]);
    };
  }
});

export { Tabs as default, tabsEmits, tabsProps };
//# sourceMappingURL=tabs.mjs.map
