import { getCurrentInstance, toRef, ref, shallowRef, computed, unref, watch } from 'vue';
import '../../../utils/index.mjs';
import { enforceUnit, sum } from './utils.mjs';
import { useColumns } from './use-columns.mjs';
import { SortOrder, oppositeOrderMap } from './constants.mjs';
import { isNumber } from '@vueuse/core';
import { isUndefined } from '../../../utils/types.mjs';
import { isObject } from '@vue/shared';

function useTable(props) {
  const emit = getCurrentInstance().emit;
  const {
    columns,
    columnsStyles,
    columnsTotalWidth,
    fixedColumnsOnLeft,
    fixedColumnOnRight,
    hasFixedColumns,
    mainColumns,
    getColumn
  } = useColumns(toRef(props, "columns"), toRef(props, "fixed"));
  const expandedRowKeys = ref(props.defaultExpandedRowKeys || []);
  const depthMap = ref({});
  const hoveringRowKey = shallowRef(null);
  const resetIndex = shallowRef(null);
  const isResetting = shallowRef(false);
  const isScrolling = shallowRef(false);
  const containerRef = ref();
  const mainTableRef = ref();
  const leftTableRef = ref();
  const rightTableRef = ref();
  const scrollPos = ref({ scrollLeft: 0, scrollTop: 0 });
  const lastRenderedRowIndex = ref(-1);
  const rowHeights = shallowRef({});
  const leftTableHeights = shallowRef({});
  const mainTableHeights = shallowRef({});
  const rightTableHeights = shallowRef({});
  const rowsHeight = computed(() => {
    const { rowHeight, estimatedRowHeight } = props;
    const _data = unref(data);
    if (isNumber(estimatedRowHeight)) {
      return _data.length * estimatedRowHeight;
    }
    return _data.length * rowHeight;
  });
  const flattenedData = computed(() => {
    const depths = {};
    const { data: data2, rowKey } = props;
    const _expandedRowKeys = unref(expandedRowKeys);
    if (!_expandedRowKeys || !_expandedRowKeys.length)
      return data2;
    const array = [];
    const keysSet = /* @__PURE__ */ new Set();
    _expandedRowKeys.forEach((x) => keysSet.add(x));
    let copy = data2.slice();
    copy.forEach((x) => depths[x[rowKey]] = 0);
    while (copy.length > 0) {
      const item = copy.shift();
      array.push(item);
      if (keysSet.has(item[rowKey]) && Array.isArray(item.children) && item.children.length > 0) {
        copy = [...item.children, ...copy];
        item.children.forEach((child) => depths[child[rowKey]] = depths[item[rowKey]] + 1);
      }
    }
    depthMap.value = depths;
    return array;
  });
  const data = computed(() => {
    const { data: data2, expandColumnKey } = props;
    return expandColumnKey ? unref(flattenedData) : data2;
  });
  const bodyWidth = computed(() => {
    const { fixed, width, vScrollbarSize } = props;
    const ret = width - vScrollbarSize;
    return fixed ? Math.max(Math.round(unref(columnsTotalWidth)), ret) : ret;
  });
  const rootStyle = computed(() => {
    const { style = {}, height, width } = props;
    return enforceUnit({
      ...style,
      height,
      width
    });
  });
  const headerWidth = computed(() => unref(bodyWidth) + (props.fixed ? props.vScrollbarSize : 0));
  const mainTableHeight = computed(() => {
    const { height = 0, maxHeight = 0, footerHeight, hScrollbarSize } = props;
    if (maxHeight > 0) {
      const _fixedRowsHeight = unref(fixedRowsHeight);
      const _rowsHeight = unref(rowsHeight);
      const _headerHeight = unref(headerHeight);
      const total = _headerHeight + _fixedRowsHeight + _rowsHeight + hScrollbarSize;
      return Math.min(total, maxHeight - footerHeight);
    }
    return height - footerHeight;
  });
  const fixedTableHeight = computed(() => {
    const { maxHeight } = props;
    const tableHeight = unref(mainTableHeight);
    if (isNumber(maxHeight) && maxHeight > 0)
      return tableHeight;
    const totalHeight = unref(rowsHeight) + unref(headerHeight) + unref(fixedRowsHeight);
    return Math.min(tableHeight, totalHeight);
  });
  const mapColumn = (column) => column.width;
  const leftTableWidth = computed(() => sum(unref(fixedColumnsOnLeft).map(mapColumn)));
  const rightTableWidth = computed(() => sum(unref(fixedColumnOnRight).map(mapColumn)));
  const headerHeight = computed(() => sum(props.headerHeight));
  const fixedRowsHeight = computed(() => {
    var _a;
    return (((_a = props.fixedData) == null ? void 0 : _a.length) || 0) * props.rowHeight;
  });
  const windowHeight = computed(() => {
    return unref(mainTableHeight) - unref(headerHeight) - unref(fixedRowsHeight);
  });
  function doScroll(params) {
    var _a, _b, _c;
    const { scrollTop } = params;
    (_a = mainTableRef.value) == null ? void 0 : _a.scrollTo(params);
    (_b = leftTableRef.value) == null ? void 0 : _b.scrollToTop(scrollTop);
    (_c = rightTableRef.value) == null ? void 0 : _c.scrollToTop(scrollTop);
  }
  function scrollTo(params) {
    scrollPos.value = params;
    doScroll(params);
  }
  function scrollToTop(scrollTop) {
    scrollPos.value.scrollTop = scrollTop;
    doScroll(unref(scrollPos));
  }
  function scrollToLeft(scrollLeft) {
    var _a, _b;
    scrollPos.value.scrollLeft = scrollLeft;
    (_b = (_a = mainTableRef.value) == null ? void 0 : _a.scrollTo) == null ? void 0 : _b.call(_a, unref(scrollPos));
  }
  function onMaybeEndReached() {
    const { onEndReached } = props;
    if (!onEndReached)
      return;
    const { scrollTop } = unref(scrollPos);
    const _totalHeight = unref(rowsHeight);
    const clientHeight = unref(windowHeight);
    const heightUntilEnd = _totalHeight - (scrollTop + clientHeight) + props.hScrollbarSize;
    if (unref(lastRenderedRowIndex) >= 0 && _totalHeight !== unref(rowsHeight)) {
      onEndReached(heightUntilEnd);
    }
  }
  function updateRows() {
    isResetting.value = true;
    resetAfterIndex(unref(resetIndex), false);
    resetIndex.value = null;
    isResetting.value = false;
  }
  function resetAfterIndex(index, forceUpdate = false) {
    if (isUndefined(props.estimatedRowHeight))
      return;
    [mainTableRef, leftTableRef, rightTableRef].forEach((tableRef) => {
      const table = unref(tableRef);
      if (table)
        table.resetAfterRowIndex(index, forceUpdate);
    });
  }
  function onScroll(params) {
    var _a;
    scrollTo(params);
    (_a = props.onScroll) == null ? void 0 : _a.call(props, params);
  }
  function onVerticalScroll({ scrollTop }) {
    const { scrollTop: currentScrollTop } = unref(scrollPos);
    if (scrollTop !== currentScrollTop)
      scrollToTop(scrollTop);
  }
  function onRowsRendered(params) {
    var _a;
    (_a = props.onRowRendered) == null ? void 0 : _a.call(props, params);
    if (params.rowCacheEnd > unref(lastRenderedRowIndex)) {
      lastRenderedRowIndex.value = params.rowCacheEnd;
    }
  }
  function onRowHovered({ hovered, rowKey }) {
    hoveringRowKey.value = hovered ? rowKey : null;
  }
  function onRowExpanded({
    expanded,
    rowData,
    rowIndex,
    rowKey
  }) {
    var _a, _b;
    const _expandedRowKeys = [...unref(expandedRowKeys)];
    const currentKeyIndex = _expandedRowKeys.indexOf(rowKey);
    if (expanded) {
      if (currentKeyIndex === -1)
        _expandedRowKeys.push(rowKey);
    } else {
      if (currentKeyIndex > -1)
        _expandedRowKeys.splice(currentKeyIndex, 1);
    }
    expandedRowKeys.value = _expandedRowKeys;
    emit("update:expandedRowKeys", _expandedRowKeys);
    (_a = props.onRowExpand) == null ? void 0 : _a.call(props, {
      expanded,
      rowData,
      rowIndex,
      rowKey
    });
    (_b = props.onExpandedRowsChange) == null ? void 0 : _b.call(props, _expandedRowKeys);
  }
  function onColumnSorted(e) {
    var _a;
    const { key } = e.currentTarget.dataset;
    if (!key)
      return;
    const { sortState, sortBy } = props;
    let order = SortOrder.ASC;
    if (isObject(sortState)) {
      order = oppositeOrderMap[sortState[key]];
    } else {
      order = oppositeOrderMap[sortBy.order];
    }
    (_a = props.onColumnSort) == null ? void 0 : _a.call(props, { column: getColumn(key), key, order });
  }
  function onRowHeightChanged(rowKey, height, rowIdx) {
    const resetIdx = unref(resetIndex);
    if (resetIdx === null) {
      resetIndex.value = rowIdx;
    } else {
      if (resetIdx > rowIdx) {
        resetIndex.value = rowIdx;
      }
    }
    rowHeights.value[rowKey] = height;
  }
  function onFixedRowHeightChanged({
    rowKey,
    height,
    rowIndex
  }, fixedDir) {
    if (!fixedDir) {
      mainTableHeights.value[rowKey] = height;
    } else {
      if (fixedDir === "right") {
        rightTableHeights.value[rowKey] = height;
      } else {
        leftTableHeights.value[rowKey] = height;
      }
    }
    const _rowHeights = unref(rowHeights);
    if (_rowHeights[rowKey] !== height) {
      onRowHeightChanged(rowKey, height, rowIndex);
    }
  }
  watch(() => unref(scrollPos).scrollTop, (cur, prev) => {
    if (cur > prev)
      onMaybeEndReached();
  });
  watch(lastRenderedRowIndex, () => onMaybeEndReached());
  watch(() => props.expandedRowKeys, (val) => expandedRowKeys.value = val, {
    deep: true
  });
  watch(rowHeights, () => {
    updateRows();
  });
  return {
    columns,
    containerRef,
    mainTableRef,
    leftTableRef,
    rightTableRef,
    isResetting,
    isScrolling,
    hoveringRowKey,
    hasFixedColumns,
    columnsStyles,
    columnsTotalWidth,
    data,
    expandedRowKeys,
    depthMap,
    fixedColumnsOnLeft,
    fixedColumnOnRight,
    mainColumns,
    bodyWidth,
    rootStyle,
    headerWidth,
    mainTableHeight,
    fixedTableHeight,
    leftTableWidth,
    rightTableWidth,
    scrollTo,
    scrollToLeft,
    scrollToTop,
    onColumnSorted,
    onRowHovered,
    onRowExpanded,
    onRowsRendered,
    onRowHeightChanged,
    onFixedRowHeightChanged,
    onScroll,
    onVerticalScroll
  };
}

export { useTable };
//# sourceMappingURL=use-table.mjs.map
