import { CreateElement, VNode } from 'vue'
import { ElementUIComponent } from './component'

/** The node of the tree */
export interface TreeNode {
  id?: any,
  label?: string,
  isLeaf?: boolean,
  children?: TreeNode[]
}

export interface RenderContent {
  /**
   * Render function for a specific node
   *
   * @param h The render function
   * @param data The data object containing the specific node
   */
  (h: CreateElement, data: { node: TreeNode }): VNode
}

export interface FilterNodeMethod {
  /**
   * Filter method for each node
   *
   * @param value The query string
   * @param data The original data object
   * @param node Tree node
   */
  (value: string, data: TreeNode, node: any): boolean
}

export interface AllowDragMethod {
  /**
   * Function executed before dragging a node
   *
   * @param node The node to be dragged
   */
  (node: any): boolean
}

export interface AllowDropMethod {
  /**
   * Function executed before the dragging node is dropped
   *
   * @param draggingNode The dragging node
   * @param dropNode The target node
   * @param type Drop type
   */
  (draggingNode: any, dropNode: any, type: string): boolean
}

/** Tree Component */
export declare class ElTree extends ElementUIComponent {
  /** Tree data */
  data: TreeNode[]

  /** Text displayed when data is void */
  emptyText: string

  /** Unique identity key name for nodes, its value should be unique across the whole tree */
  nodeKey: string

  /** Configuration options, see the following table */
  props: object

  /** Method for loading subtree data */
  load: (node: TreeNode, resolve: Function) => void

  /** Render function for tree node */
  renderContent: RenderContent

  /** Whether current node is highlighted */
  highlightCurrent: boolean

  /** Whether to expand all nodes by default */
  defaultExpandAll: boolean

  /** Whether to expand or collapse node when clicking on the node. If false, then expand or collapse node only when clicking on the arrow icon. */
  expandOnClickNode: boolean

  /** Whether to check or uncheck node when clicking on the node, if false, the node can only be checked or unchecked by clicking on the checkbox. */
  checkOnClickNode: boolean

  /** Whether to expand father node when a child node is expanded */
  autoExpandParent: boolean

  /** Array of keys of initially expanded nodes */
  defaultExpandedKeys: any[]

  /** Whether node is selectable */
  showCheckbox: boolean

  /** Whether checked state of a node not affects its father and child nodes when show-checkbox is true */
  checkStrictly: boolean

  /** Array of keys of initially checked nodes */
  defaultCheckedKeys: any[]

  /** This function will be executed on each node when use filter method. If return false, tree node will be hidden. */
  filterNodeMethod: FilterNodeMethod

  /** Whether only one node among the same level can be expanded at one time */
  accordion: boolean

  /** Horizontal indentation of nodes in adjacent levels in pixels */
  indent: number

  /** Whether enable tree nodes drag and drop */
  draggable: boolean

  /** Function to be executed before dragging a node */
  allowDrag: AllowDragMethod

  /** Function to be executed before the dragging node is dropped */
  allowDrop: AllowDropMethod

  /**
   * Filter all tree nodes. Filtered nodes will be hidden
   *
   * @param value The value to be used as first parameter for `filter-node-method`
   */
  filter (value: any): void

  /**
   * Update the children of the node which specified by the key
   * 
   * @param key the key of the node which children will be updated
   * @param data the children data
   */
  updateKeyChildren (key: any, data: TreeNode[]): void

  /**
   * If the node can be selected (`show-checkbox` is `true`), it returns the currently selected array of nodes
   *
   * @param leafOnly If the `leafOnly` is `true`, it only returns the currently selected array of sub-nodes
   * @param includeHalfChecked If the `includeHalfChecked` is `true`, the return value contains halfchecked nodes
   */
  getCheckedNodes (leafOnly?: boolean, includeHalfChecked?: boolean): TreeNode[]

  /**
   * Set certain nodes to be checked. Only works when `node-key` is assigned
   *
   * @param nodes An array of nodes to be checked
   * @param leafOnly If the parameter is true, it only returns the currently selected array of sub-nodes
   */
  setCheckedNodes (nodes: TreeNode[], leafOnly?: boolean): void

  /**
   * If the node can be selected (`show-checkbox` is `true`), it returns the currently selected array of nodes' keys
   *
   * @param leafOnly If the `leafOnly` is `true`, it only returns the currently selected array of sub-nodes
   */
  getCheckedKeys (leafOnly?: boolean): any[]

  /**
   * Set certain nodes to be checked. Only works when `node-key` is assigned
   *
   * @param keys An array of node's keys to be checked
   * @param leafOnly If the parameter is true, it only returns the currently selected array of sub-nodes
   */
  setCheckedKeys (keys: any[], leafOnly?: boolean): void

  /**
   * Set node to be checked or not. Only works when `node-key` is assigned
   *
   * @param data Node's key or data to be checked
   * @param checked Indicating the node checked or not
   * @param deep Indicating whether to checked state deeply or not
   */
  setChecked (data: TreeNode | any, checked: boolean, deep: boolean): void

  /**
   * If the node can be selected (`show-checkbox` is `true`), it returns the currently half selected array of nodes
   */
  getHalfCheckedNodes (): void

  /**
   * If the node can be selected (`show-checkbox` is `true`), it returns the currently half selected array of nodes' keys
   */
  getHalfCheckedKeys (): void;

  /**
   * Return the highlight node's key (null if no node is highlighted)
   */
  getCurrentKey (): any

  /**
   * Set highlighted node by key, only works when node-key is assigned
   *
   * @param key The node's key to be highlighted
   */
  setCurrentKey (key: any): void

  /**
   * Return the highlight node (null if no node is highlighted)
   */
  getCurrentNode (): TreeNode

  /**
   * Set highlighted node, only works when node-key is assigned
   *
   * @param node The node to be highlighted
   */
  setCurrentNode (node: TreeNode): void

  /**
   * Get node by node key or node data
   * 
   * @param by node key or node data
   */
  getNode (by: TreeNode | any): TreeNode

  /**
   * Remove node by key or node data or node instance
   * 
   * @param by key or node data or node instance
   */
  remove (by: TreeNode | any): void

  /**
   * Append a child node to specified node
   * 
   * @param childData the data of appended node
   * @param parent key or node data or node instance of the parent node
   */
  append (childData: TreeNode, parent: TreeNode | any): void

  /**
   * insert a node before specified node
   * 
   * @param data the data of inserted node
   * @param ref key or node data or node instance of the reference node
   */
  insertBefore (data: TreeNode, ref: TreeNode | any): void

  /**
   * insert a node after specified node
   * 
   * @param data the data of inserted node
   * @param ref key or node data or node instance of the reference node
   */
  insertAfter (data: TreeNode, ref: TreeNode | any): void
}
