/**
 * Here is the list of every allowed parameter when using Emitter#on:
 * @type {Object}
 */
var __allowedOptions = {
  once: 'boolean',
  scope: 'object'
};

/**
 * Incremental id used to order event handlers.
 */
var __order = 0;

/**
 * A simple helper to shallowly merge two objects. The second one will "win"
 * over the first one.
 *
 * @param  {object}  o1 First target object.
 * @param  {object}  o2 Second target object.
 * @return {object}     Returns the merged object.
 */
function shallowMerge(o1, o2) {
  var o = {},
      k;

  for (k in o1) o[k] = o1[k];
  for (k in o2) o[k] = o2[k];

  return o;
}

/**
 * Is the given variable a plain JavaScript object?
 *
 * @param  {mixed}  v   Target.
 * @return {boolean}    The boolean result.
 */
function isPlainObject(v) {
  return v &&
          typeof v === 'object' &&
          !Array.isArray(v) &&
          !(v instanceof Function) &&
          !(v instanceof RegExp);
}

/**
 * Iterate over an object that may have ES6 Symbols.
 *
 * @param  {object}   object  Object on which to iterate.
 * @param  {function} fn      Iterator function.
 * @param  {object}   [scope] Optional scope.
 */
function forIn(object, fn, scope) {
  var symbols,
      k,
      i,
      l;

  for (k in object)
    fn.call(scope || null, k, object[k]);

  if (Object.getOwnPropertySymbols) {
    symbols = Object.getOwnPropertySymbols(object);

    for (i = 0, l = symbols.length; i < l; i++)
      fn.call(scope || null, symbols[i], object[symbols[i]]);
  }
}

/**
 * The emitter's constructor. It initializes the handlers-per-events store and
 * the global handlers store.
 *
 * Emitters are useful for non-DOM events communication. Read its methods
 * documentation for more information about how it works.
 *
 * @return {Emitter}         The fresh new instance.
 */
var Emitter = function() {
  this._enabled = true;

  // Dirty trick that will set the necessary properties to the emitter
  this.unbindAll();
};

/**
 * This method unbinds every handlers attached to every or any events. So, these
 * functions will no more be executed when the related events are emitted. If
 * the functions were not bound to the events, nothing will happen, and no error
 * will be thrown.
 *
 * Usage:
 * ******
 * > myEmitter.unbindAll();
 *
 * @return {Emitter}      Returns this.
 */
Emitter.prototype.unbindAll = function() {

  this._handlers = {};
  this._handlersAll = [];
  this._handlersComplex = [];

  return this;
};


/**
 * This method binds one or more functions to the emitter, handled to one or a
 * suite of events. So, these functions will be executed anytime one related
 * event is emitted.
 *
 * It is also possible to bind a function to any emitted event by not specifying
 * any event to bind the function to.
 *
 * Recognized options:
 * *******************
 *  - {?boolean} once   If true, the handlers will be unbound after the first
 *                      execution. Default value: false.
 *  - {?object}  scope  If a scope is given, then the listeners will be called
 *                      with this scope as "this".
 *
 * Variant 1:
 * **********
 * > myEmitter.on('myEvent', function(e) { console.log(e); });
 * > // Or:
 * > myEmitter.on('myEvent', function(e) { console.log(e); }, { once: true });
 *
 * @param  {EventName} event   The event to listen to.
 * @param  {Handler}   handler The function to bind.
 * @param  {?object}   options Some options.
 * @return {Emitter}           Returns this.
 *
 * Variant 2:
 * **********
 * > myEmitter.on(
 * >   ['myEvent1', 'myEvent2'],
 * >   function(e) { console.log(e); }
 * > );
 * > // Or:
 * > myEmitter.on(
 * >   ['myEvent1', 'myEvent2'],
 * >   function(e) { console.log(e); }
 * >   { once: true }}
 * > );
 *
 * @param  {EventName[]} events  The events to listen to.
 * @param  {Handler}     handler The function to bind.
 * @param  {?object}     options Some options.
 * @return {Emitter}             Returns this.
 *
 * Variant 3:
 * **********
 * > myEmitter.on({
 * >   myEvent1: function(e) { console.log(e); },
 * >   myEvent2: function(e) { console.log(e); }
 * > });
 * > // Or:
 * > myEmitter.on({
 * >   myEvent1: function(e) { console.log(e); },
 * >   myEvent2: function(e) { console.log(e); }
 * > }, { once: true });
 *
 * @param  {Object<EventName, Handler>} bindings An object containing
 *                                               pairs event / function.
 * @param  {?object}                    options  Some options.
 * @return {Emitter}                             Returns this.
 *
 * Variant 4:
 * **********
 * > myEmitter.on(function(e) { console.log(e); });
 * > // Or:
 * > myEmitter.on(function(e) { console.log(e); }, { once: true });
 *
 * @param  {Handler} handler The function to bind to every events.
 * @param  {?object} options Some options.
 * @return {Emitter}         Returns this.
 */
Emitter.prototype.on = function(a, b, c) {
  var i,
      l,
      k,
      event,
      eArray,
      handlersList,
      bindingObject;

  // Variant 3
  if (isPlainObject(a)) {
    forIn(a, function(name, fn) {
      this.on(name, fn, b);
    }, this);

    return this;
  }

  // Variant 4
  if (typeof a === 'function') {
    c = b;
    b = a;
    a = null;
  }

  eArray = [].concat(a);

  for (i = 0, l = eArray.length; i < l; i++) {
    event = eArray[i];

    bindingObject = {
      order: __order++,
      fn: b
    };

    // Defining the list in which the handler should be inserted
    if (typeof event === 'string' || typeof event === 'symbol') {
      if (!this._handlers[event])
        this._handlers[event] = [];
      handlersList = this._handlers[event];
      bindingObject.type = event;
    }
    else if (event instanceof RegExp) {
      handlersList = this._handlersComplex;
      bindingObject.pattern = event;
    }
    else if (event === null) {
      handlersList = this._handlersAll;
    }
    else {
      throw Error('Emitter.on: invalid event.');
    }

    // Appending needed properties
    for (k in c || {})
      if (__allowedOptions[k])
        bindingObject[k] = c[k];

    handlersList.push(bindingObject);
  }

  return this;
};


/**
 * This method works exactly as the previous #on, but will add an options object
 * if none is given, and set the option "once" to true.
 *
 * The polymorphism works exactly as with the #on method.
 */
Emitter.prototype.once = function() {
  var args = Array.prototype.slice.call(arguments),
      li = args.length - 1;

  if (isPlainObject(args[li]) && args.length > 1)
    args[li] = shallowMerge(args[li], {once: true});
  else
    args.push({once: true});

  return this.on.apply(this, args);
};


/**
 * This method unbinds one or more functions from events of the emitter. So,
 * these functions will no more be executed when the related events are emitter.
 * If the functions were not bound to the events, nothing will happen, and no
 * error will be thrown.
 *
 * Variant 1:
 * **********
 * > myEmitter.off('myEvent', myHandler);
 *
 * @param  {EventName} event   The event to unbind the handler from.
 * @param  {Handler}   handler The function to unbind.
 * @return {Emitter}           Returns this.
 *
 * Variant 2:
 * **********
 * > myEmitter.off(['myEvent1', 'myEvent2'], myHandler);
 *
 * @param  {EventName[]} events  The events to unbind the handler from.
 * @param  {Handler}     handler The function to unbind.
 * @return {Emitter}             Returns this.
 *
 * Variant 3:
 * **********
 * > myEmitter.off({
 * >   myEvent1: myHandler1,
 * >   myEvent2: myHandler2
 * > });
 *
 * @param  {Object<EventName, Handler>} bindings An object containing pairs
 *                                               event / function.
 * @return {Emitter}                             Returns this.
 *
 * Variant 4:
 * **********
 * > myEmitter.off(myHandler);
 *
 * @param  {Handler} handler The function to unbind from every events.
 * @return {Emitter}         Returns this.
 *
 * Variant 5:
 * **********
 * > myEmitter.off(event);
 *
 * @param  {EventName} event The event we should unbind.
 * @return {Emitter}         Returns this.
 */
function filter(target, fn) {
  target = target || [];

  var a = [],
      l,
      i;

  for (i = 0, l = target.length; i < l; i++)
    if (target[i].fn !== fn)
      a.push(target[i]);

  return a;
}

Emitter.prototype.off = function(events, fn) {
  var i,
      n,
      k,
      event;

  // Variant 4:
  if (arguments.length === 1 && typeof events === 'function') {
    fn = arguments[0];

    var keys = Object.keys(this._handlers)
      .concat(Object.getOwnPropertySymbols(this._handlers));

    // Handlers bound to events:
    for (i = 0; i < keys.length; i++) {
      k = keys[i];

      this._handlers[k] = filter(this._handlers[k], fn);

      if (this._handlers[k].length === 0)
        delete this._handlers[k];
    }

    // Generic Handlers
    this._handlersAll = filter(this._handlersAll, fn);

    // Complex handlers
    this._handlersComplex = filter(this._handlersComplex, fn);
  }

  // Variant 5
  else if (arguments.length === 1 &&
            (typeof events === 'string' || typeof events === 'symbol')) {
    delete this._handlers[events];
  }

  // Variant 1 and 2:
  else if (arguments.length === 2) {
    var eArray = [].concat(events);

    for (i = 0, n = eArray.length; i < n; i++) {
      event = eArray[i];

      this._handlers[event] = filter(this._handlers[event], fn);

      if ((this._handlers[event] || []).length === 0)
        delete this._handlers[event];
    }
  }

  // Variant 3
  else if (isPlainObject(events)) {
    forIn(events, this.off, this);
  }

  return this;
};

/**
 * This method retrieve the listeners attached to a particular event.
 *
 * @param  {?EventName} event Name of the event.
 * @return {array}            Array of handler functions.
 */
Emitter.prototype.listeners = function(event) {
  var handlers = this._handlersAll || [],
      complex = false,
      h,
      i,
      l;

  if (!event)
    throw Error('Emitter.listeners: no event provided.');

  handlers = handlers.concat(this._handlers[event] || []);

  for (i = 0, l = this._handlersComplex.length; i < l; i++) {
    h = this._handlersComplex[i];

    if (typeof event === 'string' && ~event.search(h.pattern)) {
      complex = true;
      handlers.push(h);
    }
  }

  // If we have any complex handlers, we need to sort
  if (this._handlersAll.length || complex)
    return handlers.sort(function(a, b) {
      return a.order - b.order;
    });
  else
    return handlers.slice(0);
};

/**
 * This method emits the specified event(s), and executes every handlers bound
 * to the event(s).
 *
 * Variant 1:
 * **********
 * > myEmitter.emit('myEvent');
 * > myEmitter.emit('myEvent', myData);
 *
 * @param  {EventName} event The event to emit.
 * @param  {object?}   data  Some data.
 * @return {Emitter}         Returns this.
 *
 * Variant 2:
 * **********
 * > myEmitter.emit(['myEvent1', 'myEvent2']);
 * > myEmitter.emit(['myEvent1', 'myEvent2'], myData);
 *
 * @param  {EventName[]} events The events to emit.
 * @param  {object?}     data   Some data.
 * @return {Emitter}            Returns this.
 *
 * Variant 3:
 * **********
 * > myEmitter.emit({myEvent1: myData1, myEvent2: myData2});
 *
 * @param  {Object<EventName, any>} events The events to emit.
 * @return {Emitter}                       Returns this.
 */
Emitter.prototype.emit = function(events, data) {

  // Short exit if the emitter is disabled
  if (!this._enabled)
    return this;

  // Object variant
  if (isPlainObject(events)) {
    forIn(events, this.emit, this);
    return this;
  }

  var eArray = [].concat(events),
      onces = [],
      event,
      parent,
      handlers,
      handler,
      i,
      j,
      l,
      m;

  for (i = 0, l = eArray.length; i < l; i++) {
    handlers = this.listeners(eArray[i]);

    for (j = 0, m = handlers.length; j < m; j++) {
      handler = handlers[j];
      event = {
        type: eArray[i],
        target: this
      };

      if (arguments.length > 1)
        event.data = data;

      handler.fn.call('scope' in handler ? handler.scope : this, event);

      if (handler.once)
        onces.push(handler);
    }

    // Cleaning onces
    for (j = onces.length - 1; j >= 0; j--) {
      if (onces[j].type)
        parent = this._handlers[onces[j].type];
      else if (onces[j].pattern)
        parent = this._handlersComplex;
      else
        parent = this._handlersAll;

      var onceIndex = parent.indexOf(onces[j]);
      if (onceIndex !== -1) {
        parent.splice(onceIndex, 1);
      }
    }
  }

  return this;
};


/**
 * This method will unbind all listeners and make it impossible to ever rebind
 * any listener to any event.
 */
Emitter.prototype.kill = function() {

  this.unbindAll();
  this._handlers = null;
  this._handlersAll = null;
  this._handlersComplex = null;
  this._enabled = false;

  // Nooping methods
  this.unbindAll =
  this.on =
  this.once =
  this.off =
  this.emit =
  this.listeners = Function.prototype;
};


/**
 * This method disabled the emitter, which means its emit method will do
 * nothing.
 *
 * @return {Emitter} Returns this.
 */
Emitter.prototype.disable = function() {
  this._enabled = false;

  return this;
};


/**
 * This method enables the emitter.
 *
 * @return {Emitter} Returns this.
 */
Emitter.prototype.enable = function() {
  this._enabled = true;

  return this;
};


/**
 * Version:
 */
Emitter.version = '3.2.0';


/**
 * Export:
 */
module.exports = Emitter;
