/**
 * @author       Yannick Deubel (https://github.com/yandeu)
 * @copyright    Copyright (c) 2020 Yannick Deubel; Project Url: https://github.com/enable3d/enable3d
 * @license      {@link https://github.com/enable3d/enable3d/blob/master/LICENSE|GNU GPLv3}
 */
import { Clock } from 'three';
import { CSG } from '@enable3d/three-graphics/jsm/csg';
import * as Plugins from '@enable3d/three-graphics/jsm/plugins';
// https://www.typescriptlang.org/docs/handbook/utility-types.html Pick<T,K>
// export class Scene3D implements Partial<ThreeGraphics> {
export class Scene3D {
    constructor(sceneConfig = {}) {
        this.sceneConfig = sceneConfig;
        // core modules from three-graphics
        this.scenes = new Map();
        this.__config = {};
        this._isRunning = false;
        this._deconstructor = [];
        const { key = Math.random().toString(), enableXR = false } = sceneConfig;
        this.__config.sceneKey = key;
        this.__config.enableXR = enableXR;
    }
    /** Pass all objects you want to destroy on scene restart or stop. */
    get deconstructor() {
        return {
            /**
             * Pass an your objects.
             * @example
             * // this is what the deconstructor does on
             * // scene restart or stop to all objects added:
             * await object.dispose?.()
             * await object.destroy?.()
             * if (typeof object === 'function') await object?.()
             * object = null
             */
            add: (...object) => {
                object.forEach(o => {
                    this._deconstructor.push(o);
                });
            }
        };
    }
    initializeScene(plugins) {
        const { renderer, parent, canvas, scene, scenes, camera, cache, physics, sceneConfig } = plugins;
        this.scene = scene;
        this.scenes = scenes;
        this.camera = camera;
        this.cache = cache;
        this.physics = physics;
        this.renderer = renderer;
        this.parent = parent;
        this.canvas = canvas;
        const { autoStart, textureAnisotropy } = sceneConfig;
        // plugins
        this.load = new Plugins.Loaders(this.cache, textureAnisotropy);
        this.lights = new Plugins.Lights(this.scene);
        this.transform = new Plugins.Transform(this.camera, this.renderer);
        this.csg = CSG;
        this.heightMap = new Plugins.HeightMap(this.scene);
        this.factories = new Plugins.Factories(this.scene);
        this.misc = new Plugins.Misc(this.scene, this.renderer, this.factories);
        this.ws = new Plugins.WarpSpeed(scene, renderer, camera, this.lights, this.physics, this.load, this.factories);
        this.mixers = new Plugins.Mixers();
        this.cameras = new Plugins.Cameras();
        this.clock = new Clock();
        // add vr camera
        if (this.__config.enableXR) {
            this.webXR = new Plugins.WebXR(this.renderer, this.scene);
        }
        if (autoStart)
            this.start(this.__config.sceneKey);
    }
    get sceneKey() {
        return this.__config.sceneKey;
    }
    /** Destroys a object and its body. */
    destroy(obj) {
        this.physics?.destroy(obj.body);
        this.scene.remove(obj);
        // @ts-expect-error: ExtendedObject3D and ExtendedMesh can't be null.
        obj = null;
    }
    async warpSpeed(...features) {
        return await this.ws.warpSpeed(...features);
    }
    get animationMixers() {
        return this.mixers.mixers;
    }
    get make() {
        return this.factories.make;
    }
    get add() {
        return this.factories.add;
    }
    haveSomeFun(numberOfElements = 20) {
        Plugins.HaveSomeFun(numberOfElements, this.physics);
    }
    isRunning() {
        return this._isRunning;
    }
    async start(key, data) {
        // start another scene
        // and stop this one
        if (key && key !== this.__config.sceneKey) {
            this.stop();
            this.scenes.get(key)?._start(data);
        }
        else {
            this._start(data);
        }
    }
    async _start(data) {
        await this.init?.(data);
        await this._preload();
        await this._create();
        // console.log('start')
        this.renderer.setAnimationLoop(() => {
            this._update();
        });
        this._isRunning = true;
    }
    async restart(data) {
        await this.stop();
        await this.start(this.__config.sceneKey, data);
    }
    async stop() {
        this._isRunning = false;
        this.renderer.setAnimationLoop(null);
        // reset clock
        this.clock.start();
        for (let object of this._deconstructor) {
            await object.dispose?.();
            await object.destroy?.();
            if (typeof object === 'function')
                await object?.();
            object = null;
        }
        this._deconstructor = [];
        // destroy all rigid bodies
        if (this.physics?.rigidBodies)
            for (let i = this.physics.rigidBodies.length - 1; i >= 0; i--) {
                this.physics.destroy(this.physics.rigidBodies[i]);
            }
        // destroy all three objects
        for (let i = this.scene.children.length - 1; i >= 0; i--) {
            this.scene.remove(this.scene.children[i]);
        }
    }
    setSize(width, height) {
        this.renderer.setSize(width, height);
        if (typeof this.camera.aspect !== 'undefined')
            this.camera.aspect = width / height;
        this.camera.updateProjectionMatrix();
    }
    setPixelRatio(ratio) {
        this.renderer.setPixelRatio(ratio);
    }
    // public get controls() {
    //   return {
    //     pointerDrag: (autoStart = true) => new PointerDrag(this.canvas, autoStart),
    //     pointerLock: (autoLock = true) => new PointerLock(this.canvas, autoLock),
    //     joystick: () => new JoyStick(),
    //     firstPerson: (target: ExtendedObject3D, config: FirstPersonControlsConfig = {}) =>
    //       new FirstPersonControls(this.camera, target, config),
    //     thirdPerson: (target: ExtendedObject3D, config: ThirdPersonControlsConfig = {}) =>
    //       new ThirdPersonControls(this.camera, target, config)
    //   }
    // }
    init(data = {}) { }
    preload() { }
    create() { }
    update(_time, _delta) { }
    /** Will be called before THREE.WebGLRenderer.render() */
    preRender() { }
    /** Will be called after THREE.WebGLRenderer.render() */
    postRender() { }
    async _preload() {
        await this.preload?.();
    }
    async _create() {
        await this.create?.();
    }
    _update() {
        const delta = this.clock.getDelta() * 1000;
        const time = this.clock.getElapsedTime();
        this.update?.(parseFloat(time.toFixed(3)), parseInt(delta.toString()));
        this.physics?.update(delta);
        this.physics?.updateDebugger();
        this.animationMixers.update(delta);
        this.preRender();
        if (this.composer)
            this.composer.render();
        else
            this.renderer.render(this.scene, this.camera);
        this.postRender();
    }
}
//# sourceMappingURL=scene3d.js.map