import _Object$keys from "babel-runtime/core-js/object/keys";
import _Promise from "babel-runtime/core-js/promise";
import _classCallCheck from "babel-runtime/helpers/classCallCheck";
import _createClass from "babel-runtime/helpers/createClass";
import EventEmitter from "event-emitter";
import { extend, borders, uuid, isNumber, bounds as _bounds, defer, createBlobUrl, revokeBlobUrl, serialize } from "../../utils/core";
import EpubCFI from "../../utils/epubcfi";
import Contents from "../../rendition/contents";
import { EVENTS } from "../../utils/constants";
import { Pane, Highlight, Underline } from "marks-pane";
import globalRequest from "../../utils/request";
import Hook from "../../utils/hook";

var IframeView = function () {
	function IframeView(section, options) {
		_classCallCheck(this, IframeView);

		this.settings = extend({
			ignoreClass: "",
			axis: options.layout && options.layout.props.flow === "scrolled" ? "vertical" : "horizontal",
			direction: undefined,
			width: 0,
			height: 0,
			layout: undefined,
			globalLayoutProperties: {},
			method: "url"
		}, options || {});

		this.id = "epubjs-view-" + uuid();
		this.section = section;
		this.index = section.index;

		this.element = this.container(this.settings.axis);

		this.added = false;
		this.displayed = false;
		this.rendered = false;

		// this.width  = this.settings.width;
		// this.height = this.settings.height;

		this.fixedWidth = 0;
		this.fixedHeight = 0;

		// Blank Cfi for Parsing
		this.epubcfi = new EpubCFI();

		this.layout = this.settings.layout;
		// Dom events to listen for
		// this.listenedEvents = ["keydown", "keyup", "keypressed", "mouseup", "mousedown", "click", "touchend", "touchstart"];

		this.pane = undefined;
		this.highlights = {};
		this.underlines = {};
		this.marks = {};
	}

	_createClass(IframeView, [{
		key: "container",
		value: function container(axis) {
			var element = document.createElement("div");

			element.classList.add("epub-view");

			// this.element.style.minHeight = "100px";
			element.style.height = "0px";
			element.style.width = "0px";
			element.style.overflow = "hidden";
			element.style.position = "relative";
			element.style.display = "block";

			if (axis && axis == "horizontal") {
				element.style.flex = "none";
			} else {
				element.style.flex = "initial";
			}

			return element;
		}
	}, {
		key: "create",
		value: function create() {

			if (this.iframe) {
				return this.iframe;
			}

			if (!this.element) {
				this.element = this.createContainer();
			}

			this.iframe = document.createElement("iframe");
			this.iframe.id = this.id;
			this.iframe.scrolling = "no"; // Might need to be removed: breaks ios width calculations
			this.iframe.style.overflow = "hidden";
			this.iframe.seamless = "seamless";
			// Back up if seamless isn't supported
			this.iframe.style.border = "none";

			this.iframe.setAttribute("enable-annotation", "true");

			this.resizing = true;

			// this.iframe.style.display = "none";
			this.element.style.visibility = "hidden";
			this.iframe.style.visibility = "hidden";

			this.iframe.style.width = "0";
			this.iframe.style.height = "0";
			this._width = 0;
			this._height = 0;

			this.element.setAttribute("ref", this.index);

			this.element.appendChild(this.iframe);
			this.added = true;

			this.elementBounds = _bounds(this.element);

			// if(width || height){
			//   this.resize(width, height);
			// } else if(this.width && this.height){
			//   this.resize(this.width, this.height);
			// } else {
			//   this.iframeBounds = bounds(this.iframe);
			// }


			if ("srcdoc" in this.iframe) {
				this.supportsSrcdoc = true;
			} else {
				this.supportsSrcdoc = false;
			}

			// if (!this.settings.method) {
			// 	this.settings.method = this.supportsSrcdoc ? "srcdoc" : "write";
			// }

			return this.iframe;
		}
	}, {
		key: "render",
		value: function render(request, show) {
			var contents = void 0;
			// view.onLayout = this.layout.format.bind(this.layout);
			this.create();

			// Fit to size of the container, apply padding
			this.size();

			if (this.settings.method === "url") {
				contents = this.section.href;
			} else if (contents) {
				contents = this.section.contents;
			} else {
				contents = globalRequest(this.section.href);
			}

			// Render Chain
			return this.load(contents).then(function () {
				var _this = this;

				// apply the layout function to the contents
				this.layout.format(this.contents);

				// find and report the writingMode axis
				var writingMode = this.contents.writingMode();
				var axis = writingMode.indexOf("vertical") === 0 ? "vertical" : "horizontal";

				this.setAxis(axis);
				this.emit(EVENTS.VIEWS.AXIS, axis);

				// Listen for events that require an expansion of the iframe
				this.addListeners();

				return new _Promise(function (resolve, reject) {
					// Expand the iframe to the full size of the content
					_this.expand();
					resolve();
				});
			}.bind(this), function (e) {
				this.emit(EVENTS.VIEWS.LOAD_ERROR, e);
				return new _Promise(function (resolve, reject) {
					reject(e);
				});
			}.bind(this)).then(function () {
				this.emit(EVENTS.VIEWS.RENDERED, this.section);
			}.bind(this));
		}
	}, {
		key: "reset",
		value: function reset() {
			if (this.iframe) {
				this.iframe.style.width = "0";
				this.iframe.style.height = "0";
				this._width = 0;
				this._height = 0;
				this._textWidth = undefined;
				this._contentWidth = undefined;
				this._textHeight = undefined;
				this._contentHeight = undefined;
			}
			this._needsReframe = true;
		}

		// Determine locks base on settings

	}, {
		key: "size",
		value: function size(_width, _height) {
			var width = _width || this.settings.width;
			var height = _height || this.settings.height;

			if (this.layout.name === "pre-paginated") {
				this.lock("both", width, height);
			} else if (this.settings.axis === "horizontal") {
				this.lock("height", width, height);
			} else {
				this.lock("width", width, height);
			}

			this.settings.width = width;
			this.settings.height = height;
		}

		// Lock an axis to element dimensions, taking borders into account

	}, {
		key: "lock",
		value: function lock(what, width, height) {
			var elBorders = borders(this.element);
			var iframeBorders;

			if (this.iframe) {
				iframeBorders = borders(this.iframe);
			} else {
				iframeBorders = { width: 0, height: 0 };
			}

			if (what == "width" && isNumber(width)) {
				this.lockedWidth = width - elBorders.width - iframeBorders.width;
				// this.resize(this.lockedWidth, width); //  width keeps ratio correct
			}

			if (what == "height" && isNumber(height)) {
				this.lockedHeight = height - elBorders.height - iframeBorders.height;
				// this.resize(width, this.lockedHeight);
			}

			if (what === "both" && isNumber(width) && isNumber(height)) {

				this.lockedWidth = width - elBorders.width - iframeBorders.width;
				this.lockedHeight = height - elBorders.height - iframeBorders.height;
				// this.resize(this.lockedWidth, this.lockedHeight);
			}

			if (this.displayed && this.iframe) {

				// this.contents.layout();
				this.expand();
			}
		}

		// Resize a single axis based on content dimensions

	}, {
		key: "expand",
		value: function expand(force) {
			var width = this.lockedWidth;
			var height = this.lockedHeight;
			var columns;

			if (!this.iframe || this._expanding) return;

			this._expanding = true;

			if (this.layout.name === "pre-paginated") {
				width = this.layout.columnWidth;
				height = this.layout.height;
			}
			// Expand Horizontally
			else if (this.settings.axis === "horizontal") {
					// Get the width of the text
					width = this.contents.textWidth();

					if (width % this.layout.pageWidth > 0) {
						width = Math.ceil(width / this.layout.pageWidth) * this.layout.pageWidth;
					}

					if (this.settings.forceEvenPages) {
						columns = width / this.layout.delta;
						if (this.layout.divisor > 1 && this.layout.name === "reflowable" && columns % 2 > 0) {
							// add a blank page
							width += this.layout.gap + this.layout.columnWidth;
						}
					}
				} // Expand Vertically
				else if (this.settings.axis === "vertical") {
						height = this.contents.scrollHeight();
					}

			// Only Resize if dimensions have changed or
			// if Frame is still hidden, so needs reframing
			if (this._needsReframe || width != this._width || height != this._height) {
				this.reframe(width, height);
			}

			this._expanding = false;
		}
	}, {
		key: "reframe",
		value: function reframe(width, height) {
			var size;

			if (isNumber(width)) {
				this.element.style.width = width + "px";
				this.iframe.style.width = width + "px";
				this._width = width;
			}

			if (isNumber(height)) {
				this.element.style.height = height + "px";
				this.iframe.style.height = height + "px";
				this._height = height;
			}

			var widthDelta = this.prevBounds ? width - this.prevBounds.width : width;
			var heightDelta = this.prevBounds ? height - this.prevBounds.height : height;

			size = {
				width: width,
				height: height,
				widthDelta: widthDelta,
				heightDelta: heightDelta
			};

			this.pane && this.pane.render();

			this.onResize(this, size);

			this.emit(EVENTS.VIEWS.RESIZED, size);

			this.prevBounds = size;

			this.elementBounds = _bounds(this.element);
		}
	}, {
		key: "load",
		value: function load(contents) {
			var _this2 = this;

			var loading = new defer();
			var loaded = loading.promise;

			if (!this.iframe) {
				loading.reject(new Error("No Iframe Available"));
				return loaded;
			}

			this.iframe.onload = function (event) {

				this.onLoad(event, loading);
			}.bind(this);

			if (this.settings.method != "url") {
				contents.then(function (r) {
					var text = serialize(r);
					if (_this2.settings.method === "blobUrl") {
						_this2.blobUrl = createBlobUrl(text, "application/xhtml+xml");
						_this2.iframe.src = _this2.blobUrl;
					} else if (_this2.settings.method === "srcdoc") {
						_this2.iframe.srcdoc = text;
					}
				});
			} else {
				this.iframe.src = contents;
			}

			return loaded;
		}
	}, {
		key: "onLoad",
		value: function onLoad(event, promise) {
			var _this3 = this;

			this.window = this.iframe.contentWindow;
			this.document = this.iframe.contentDocument;

			this.contents = new Contents(this.document, this.document.body, this.section.cfiBase, this.section.index);

			this.rendering = false;

			/*
   TODO: this seems not needed with replace cannonical
   var link = this.document.querySelector("link[rel='canonical']");
   if (link) {
   	link.setAttribute("href", this.section.canonical);
   } else {
   	link = this.document.createElement("link");
   	link.setAttribute("rel", "canonical");
   	link.setAttribute("href", this.section.canonical);
   	this.document.querySelector("head").appendChild(link);
   }
   */

			this.contents.on(EVENTS.CONTENTS.EXPAND, function () {
				if (_this3.displayed && _this3.iframe) {
					_this3.expand();
					if (_this3.contents) {
						_this3.layout.format(_this3.contents);
					}
				}
			});

			this.contents.on(EVENTS.CONTENTS.RESIZE, function (e) {
				if (_this3.displayed && _this3.iframe) {
					_this3.expand();
					if (_this3.contents) {
						_this3.layout.format(_this3.contents);
					}
				}
			});

			promise.resolve(this.contents);
		}
	}, {
		key: "setLayout",
		value: function setLayout(layout) {
			this.layout = layout;

			if (this.contents) {
				this.layout.format(this.contents);
				this.expand();
			}
		}
	}, {
		key: "setAxis",
		value: function setAxis(axis) {

			// Force vertical for scrolled
			if (this.layout.props.flow === "scrolled") {
				axis = "vertical";
			}

			this.settings.axis = axis;

			if (axis == "horizontal") {
				this.element.style.flex = "none";
			} else {
				this.element.style.flex = "initial";
			}

			this.size();
		}
	}, {
		key: "addListeners",
		value: function addListeners() {
			//TODO: Add content listeners for expanding
		}
	}, {
		key: "removeListeners",
		value: function removeListeners(layoutFunc) {
			//TODO: remove content listeners for expanding
		}
	}, {
		key: "display",
		value: function display(request) {
			var displayed = new defer();

			if (!this.displayed) {

				this.render(request).then(function () {

					this.emit(EVENTS.VIEWS.DISPLAYED, this);
					this.onDisplayed(this);

					this.displayed = true;
					displayed.resolve(this);
				}.bind(this), function (err) {
					displayed.reject(err, this);
				});
			} else {
				displayed.resolve(this);
			}

			return displayed.promise;
		}
	}, {
		key: "show",
		value: function show() {

			this.element.style.visibility = "visible";

			if (this.iframe) {
				this.iframe.style.visibility = "visible";
			}

			this.emit(EVENTS.VIEWS.SHOWN, this);
		}
	}, {
		key: "hide",
		value: function hide() {
			// this.iframe.style.display = "none";
			this.element.style.visibility = "hidden";
			this.iframe.style.visibility = "hidden";

			this.stopExpanding = true;
			this.emit(EVENTS.VIEWS.HIDDEN, this);
		}
	}, {
		key: "offset",
		value: function offset() {
			return {
				top: this.element.offsetTop,
				left: this.element.offsetLeft
			};
		}
	}, {
		key: "width",
		value: function width() {
			return this._width;
		}
	}, {
		key: "height",
		value: function height() {
			return this._height;
		}
	}, {
		key: "position",
		value: function position() {
			return this.element.getBoundingClientRect();
		}
	}, {
		key: "locationOf",
		value: function locationOf(target) {
			// var parentPos = this.iframe.getBoundingClientRect();
			var targetPos = this.contents.locationOf(target, this.settings.ignoreClass);

			return {
				"left": targetPos.left,
				"top": targetPos.top
			};
		}
	}, {
		key: "onDisplayed",
		value: function onDisplayed(view) {
			// Stub, override with a custom functions
		}
	}, {
		key: "onResize",
		value: function onResize(view, e) {
			// Stub, override with a custom functions
		}
	}, {
		key: "bounds",
		value: function bounds(force) {
			if (force || !this.elementBounds) {
				this.elementBounds = _bounds(this.element);
			}

			return this.elementBounds;
		}
	}, {
		key: "highlight",
		value: function highlight(cfiRange) {
			var _this4 = this;

			var data = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
			var cb = arguments[2];

			if (!this.contents) {
				return;
			}
			var range = this.contents.range(cfiRange);

			var emitter = function emitter() {
				_this4.emit(EVENTS.VIEWS.MARK_CLICKED, cfiRange, data);
			};

			data["epubcfi"] = cfiRange;

			if (!this.pane) {
				this.pane = new Pane(this.iframe, this.element);
			}

			var m = new Highlight(range, "epubjs-hl", data, { "fill": "yellow", "fill-opacity": "0.3", "mix-blend-mode": "multiply" });
			var h = this.pane.addMark(m);

			this.highlights[cfiRange] = { "mark": h, "element": h.element, "listeners": [emitter, cb] };

			h.element.setAttribute("ref", "epubjs-hl");
			h.element.addEventListener("click", emitter);
			h.element.addEventListener("touchstart", emitter);

			if (cb) {
				h.element.addEventListener("click", cb);
				h.element.addEventListener("touchstart", cb);
			}
			return h;
		}
	}, {
		key: "underline",
		value: function underline(cfiRange) {
			var _this5 = this;

			var data = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
			var cb = arguments[2];

			if (!this.contents) {
				return;
			}
			var range = this.contents.range(cfiRange);
			var emitter = function emitter() {
				_this5.emit(EVENTS.VIEWS.MARK_CLICKED, cfiRange, data);
			};

			data["epubcfi"] = cfiRange;

			if (!this.pane) {
				this.pane = new Pane(this.iframe, this.element);
			}

			var m = new Underline(range, "epubjs-ul", data, { "stroke": "black", "stroke-opacity": "0.3", "mix-blend-mode": "multiply" });
			var h = this.pane.addMark(m);

			this.underlines[cfiRange] = { "mark": h, "element": h.element, "listeners": [emitter, cb] };

			h.element.setAttribute("ref", "epubjs-ul");
			h.element.addEventListener("click", emitter);
			h.element.addEventListener("touchstart", emitter);

			if (cb) {
				h.element.addEventListener("click", cb);
				h.element.addEventListener("touchstart", cb);
			}
			return h;
		}
	}, {
		key: "mark",
		value: function mark(cfiRange) {
			var _this6 = this;

			var data = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
			var cb = arguments[2];


			if (!this.contents) {
				return;
			}

			if (cfiRange in this.marks) {
				var item = this.marks[cfiRange];
				return item;
			}

			var range = this.contents.range(cfiRange);
			if (!range) {
				return;
			}
			var container = range.commonAncestorContainer;
			var parent = container.nodeType === 1 ? container : container.parentNode;

			var emitter = function emitter(e) {
				_this6.emit(EVENTS.VIEWS.MARK_CLICKED, cfiRange, data);
			};

			if (range.collapsed && container.nodeType === 1) {
				range = new Range();
				range.selectNodeContents(container);
			} else if (range.collapsed) {
				// Webkit doesn't like collapsed ranges
				range = new Range();
				range.selectNodeContents(parent);
			}

			var top = void 0,
			    right = void 0,
			    left = void 0;

			if (this.layout.name === "pre-paginated" || this.settings.axis !== "horizontal") {
				var pos = range.getBoundingClientRect();
				top = pos.top;
				right = pos.right;
			} else {
				// Element might break columns, so find the left most element
				var rects = range.getClientRects();
				var rect = void 0;
				for (var i = 0; i != rects.length; i++) {
					rect = rects[i];
					if (!left || rect.left < left) {
						left = rect.left;
						right = left + this.layout.columnWidth - this.layout.gap;
						top = rect.top;
					}
				}
			}

			var mark = this.document.createElement("a");
			mark.setAttribute("ref", "epubjs-mk");
			mark.style.position = "absolute";
			mark.style.top = top + "px";
			mark.style.left = right + "px";

			mark.dataset["epubcfi"] = cfiRange;

			if (data) {
				_Object$keys(data).forEach(function (key) {
					mark.dataset[key] = data[key];
				});
			}

			if (cb) {
				mark.addEventListener("click", cb);
				mark.addEventListener("touchstart", cb);
			}

			mark.addEventListener("click", emitter);
			mark.addEventListener("touchstart", emitter);

			this.element.appendChild(mark);

			this.marks[cfiRange] = { "element": mark, "listeners": [emitter, cb] };

			return parent;
		}
	}, {
		key: "unhighlight",
		value: function unhighlight(cfiRange) {
			var item = void 0;
			if (cfiRange in this.highlights) {
				item = this.highlights[cfiRange];

				this.pane.removeMark(item.mark);
				item.listeners.forEach(function (l) {
					if (l) {
						item.element.removeEventListener("click", l);
					}
				});
				delete this.highlights[cfiRange];
			}
		}
	}, {
		key: "ununderline",
		value: function ununderline(cfiRange) {
			var item = void 0;
			if (cfiRange in this.underlines) {
				item = this.underlines[cfiRange];
				this.pane.removeMark(item.mark);
				item.listeners.forEach(function (l) {
					if (l) {
						item.element.removeEventListener("click", l);
					}
				});
				delete this.underlines[cfiRange];
			}
		}
	}, {
		key: "unmark",
		value: function unmark(cfiRange) {
			var item = void 0;
			if (cfiRange in this.marks) {
				item = this.marks[cfiRange];
				this.element.removeChild(item.element);
				item.listeners.forEach(function (l) {
					if (l) {
						item.element.removeEventListener("click", l);
					}
				});
				delete this.marks[cfiRange];
			}
		}
	}, {
		key: "destroy",
		value: function destroy() {

			for (var cfiRange in this.highlights) {
				this.unhighlight(cfiRange);
			}

			for (var _cfiRange in this.underlines) {
				this.ununderline(_cfiRange);
			}

			for (var _cfiRange2 in this.marks) {
				this.unmark(_cfiRange2);
			}

			if (this.blobUrl) {
				revokeBlobUrl(this.blobUrl);
			}

			if (this.displayed) {
				this.displayed = false;

				this.removeListeners();

				this.stopExpanding = true;
				this.element.removeChild(this.iframe);

				this.iframe = null;

				this._textWidth = null;
				this._textHeight = null;
				this._width = null;
				this._height = null;
			}
			// this.element.style.height = "0px";
			// this.element.style.width = "0px";
		}
	}]);

	return IframeView;
}();

EventEmitter(IframeView.prototype);

export default IframeView;