"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prettierDifferences = void 0;
const path = require("path");
const chalk = require("chalk");
const prettier = require("prettier");
const prettier_linter_helpers_1 = require("prettier-linter-helpers");
const report_1 = require("./report");
const { INSERT, DELETE, REPLACE } = prettier_linter_helpers_1.generateDifferences;
const prettierDifferences = ({ context, source, options, offset = 0, }) => {
    // prettier.format() may throw a SyntaxError if it cannot parse the
    // source code it is given. Usually for JS files this isn't a
    // problem as ESLint will report invalid syntax before trying to
    // pass it to the prettier plugin. However this might be a problem
    // for non-JS languages that are handled by a plugin. Notably Vue
    // files throw an error if they contain unclosed elements, such as
    // `<template><div></template>. In this case report an error at the
    // point at which parsing failed.
    let prettierSource;
    try {
        prettierSource = prettier.format(source, options);
    }
    catch (err) {
        if (!(err instanceof Error)) {
            throw err;
        }
        // UndefinedParserError
        if (err.message.startsWith('No parser could be inferred for file')) {
            console.warn(chalk.yellow('warning'), '[prettier-vue]', `No parser could be inferred for "${path.extname(options.filepath || '')}" format`);
            return;
        }
        if (!(err instanceof SyntaxError)) {
            throw err;
        }
        let message = `Parsing error: ${err.message}`;
        const error = err;
        // Prettier's message contains a codeframe style preview of the
        // invalid code and the line/column at which the error occurred.
        // ESLint shows those pieces of information elsewhere already so
        // remove them from the message
        if (error.codeFrame) {
            message = message.replace(`\n${error.codeFrame}`, '');
        }
        if (error.loc) {
            message = message.replace(/ \(\d+:\d+\)$/, '');
        }
        context.report({ message, loc: error.loc });
        return;
    }
    if (source !== prettierSource) {
        const differences = (0, prettier_linter_helpers_1.generateDifferences)(source, prettierSource);
        differences.forEach((difference) => {
            switch (difference.operation) {
                case INSERT:
                    (0, report_1.reportInsert)(context, difference.offset + offset, difference.insertText);
                    break;
                case DELETE:
                    (0, report_1.reportDelete)(context, difference.offset + offset, difference.deleteText);
                    break;
                case REPLACE:
                    (0, report_1.reportReplace)(context, difference.offset + offset, difference.deleteText, difference.insertText);
                    break;
                default:
            }
        });
    }
};
exports.prettierDifferences = prettierDifferences;
