import type { Engine } from '../Engine';
import type { Loadable } from '../Interfaces/Loadable';
import { Canvas } from '../Graphics/Canvas';
import type { EventKey, Handler, Subscription } from '../EventEmitter';
import { EventEmitter } from '../EventEmitter';
export interface DefaultLoaderOptions {
    /**
     * List of loadables
     */
    loadables?: Loadable<any>[];
}
export type LoaderEvents = {
    beforeload: void;
    afterload: void;
    useraction: void;
    loadresourcestart: Loadable<any>;
    loadresourceend: Loadable<any>;
};
export declare const LoaderEvents: {
    BeforeLoad: string;
    AfterLoad: string;
    UserAction: string;
    LoadResourceStart: string;
    LoadResourceEnd: string;
};
export type LoaderConstructor = new (...args: any[]) => DefaultLoader;
/**
 * Returns true if the constructor is for an Excalibur Loader
 */
export declare function isLoaderConstructor(x: any): x is LoaderConstructor;
export declare class DefaultLoader implements Loadable<Loadable<any>[]> {
    data: Loadable<any>[];
    events: EventEmitter<LoaderEvents>;
    canvas: Canvas;
    private _resources;
    get resources(): readonly Loadable<any>[];
    private _numLoaded;
    engine: Engine;
    /**
     * @param options Optionally provide the list of resources you want to load at constructor time
     */
    constructor(options?: DefaultLoaderOptions);
    /**
     * Called by the engine before loading
     * @param engine
     */
    onInitialize(engine: Engine): void;
    /**
     * Return a promise that resolves when the user interacts with the loading screen in some way, usually a click.
     *
     * It's important to implement this in order to unlock the audio context in the browser. Browsers automatically prevent
     * audio from playing until the user performs an action.
     *
     */
    onUserAction(): Promise<void>;
    /**
     * Overridable lifecycle method, called directly before loading starts
     */
    onBeforeLoad(): Promise<void>;
    /**
     * Overridable lifecycle method, called after loading has completed
     */
    onAfterLoad(): Promise<void>;
    /**
     * Add a resource to the loader to load
     * @param loadable  Resource to add
     */
    addResource(loadable: Loadable<any>): void;
    /**
     * Add a list of resources to the loader to load
     * @param loadables  The list of resources to load
     */
    addResources(loadables: Loadable<any>[]): void;
    markResourceComplete(): void;
    /**
     * Returns the progress of the loader as a number between [0, 1] inclusive.
     */
    get progress(): number;
    /**
     * Returns true if the loader has completely loaded all resources
     */
    isLoaded(): boolean;
    private _totalTimeMs;
    /**
     * Optionally override the onUpdate
     * @param engine
     * @param elapsed
     */
    onUpdate(engine: Engine, elapsed: number): void;
    /**
     * Optionally override the onDraw
     */
    onDraw(ctx: CanvasRenderingContext2D): void;
    private _loadingFuture;
    areResourcesLoaded(): Promise<void>;
    /**
     * Not meant to be overridden
     *
     * Begin loading all of the supplied resources, returning a promise
     * that resolves when loading of all is complete AND the user has interacted with the loading screen
     */
    load(): Promise<Loadable<any>[]>;
    emit<TEventName extends EventKey<LoaderEvents>>(eventName: TEventName, event: LoaderEvents[TEventName]): void;
    emit(eventName: string, event?: any): void;
    on<TEventName extends EventKey<LoaderEvents>>(eventName: TEventName, handler: Handler<LoaderEvents[TEventName]>): Subscription;
    on(eventName: string, handler: Handler<unknown>): Subscription;
    once<TEventName extends EventKey<LoaderEvents>>(eventName: TEventName, handler: Handler<LoaderEvents[TEventName]>): Subscription;
    once(eventName: string, handler: Handler<unknown>): Subscription;
    off<TEventName extends EventKey<LoaderEvents>>(eventName: TEventName, handler: Handler<LoaderEvents[TEventName]>): void;
    off(eventName: string, handler: Handler<unknown>): void;
    off(eventName: string): void;
}
