import type { EventKey, Handler, Subscription } from './EventEmitter';
import { EventEmitter } from './EventEmitter';
import { PointerScope } from './Input/PointerScope';
import type { CanUpdate, CanDraw, CanInitialize } from './Interfaces/LifecycleEvents';
import type { Vector } from './Math/vector';
import type { ViewportDimension } from './Screen';
import { Screen, DisplayMode, Resolution } from './Screen';
import type { ScreenElement } from './ScreenElement';
import type { Actor } from './Actor';
import type { Timer } from './Timer';
import type { TileMap } from './TileMap';
import { DefaultLoader } from './Director/DefaultLoader';
import { VisibleEvent, HiddenEvent, GameStartEvent, GameStopEvent, PreUpdateEvent, PostUpdateEvent, PreFrameEvent, PostFrameEvent, PreDrawEvent, PostDrawEvent, InitializeEvent } from './Events';
import { Color } from './Color';
import type { SceneConstructor } from './Scene';
import { Scene } from './Scene';
import { Entity } from './EntityComponentSystem/Entity';
import type { DebugStats } from './Debug/DebugConfig';
import { DebugConfig } from './Debug/DebugConfig';
import { BrowserEvents } from './Util/Browser';
import type { AntialiasOptions, ExcaliburGraphicsContext } from './Graphics';
import { ExcaliburGraphicsContext2DCanvas } from './Graphics';
import type { Clock } from './Util/Clock';
import type { InputMapper } from './Input/InputMapper';
import type { GoToOptions, SceneMap, StartOptions, SceneWithOptions, WithRoot } from './Director/Director';
import { Director, DirectorEvents } from './Director/Director';
import { InputHost } from './Input/InputHost';
import type { PhysicsConfig } from './Collision/PhysicsConfig';
import type { DeepRequired } from './Util/Required';
import type { Context } from './Context';
import type { GarbageCollectionOptions } from './GarbageCollector';
export type EngineEvents = DirectorEvents & {
    fallbackgraphicscontext: ExcaliburGraphicsContext2DCanvas;
    initialize: InitializeEvent<Engine>;
    visible: VisibleEvent;
    hidden: HiddenEvent;
    start: GameStartEvent;
    stop: GameStopEvent;
    preupdate: PreUpdateEvent<Engine>;
    postupdate: PostUpdateEvent<Engine>;
    preframe: PreFrameEvent;
    postframe: PostFrameEvent;
    predraw: PreDrawEvent;
    postdraw: PostDrawEvent;
};
export declare const EngineEvents: {
    readonly NavigationStart: "navigationstart";
    readonly Navigation: "navigation";
    readonly NavigationEnd: "navigationend";
    readonly FallbackGraphicsContext: "fallbackgraphicscontext";
    readonly Initialize: "initialize";
    readonly Visible: "visible";
    readonly Hidden: "hidden";
    readonly Start: "start";
    readonly Stop: "stop";
    readonly PreUpdate: "preupdate";
    readonly PostUpdate: "postupdate";
    readonly PreFrame: "preframe";
    readonly PostFrame: "postframe";
    readonly PreDraw: "predraw";
    readonly PostDraw: "postdraw";
};
/**
 * Enum representing the different mousewheel event bubble prevention
 */
export declare enum ScrollPreventionMode {
    /**
     * Do not prevent any page scrolling
     */
    None = 0,
    /**
     * Prevent page scroll if mouse is over the game canvas
     */
    Canvas = 1,
    /**
     * Prevent all page scrolling via mouse wheel
     */
    All = 2
}
/**
 * Defines the available options to configure the Excalibur engine at constructor time.
 */
export interface EngineOptions<TKnownScenes extends string = any> {
    /**
     * Optionally configure the width of the viewport in css pixels
     */
    width?: number;
    /**
     * Optionally configure the height of the viewport in css pixels
     */
    height?: number;
    /**
     * Optionally configure the width & height of the viewport in css pixels.
     * Use `viewport` instead of {@apilink EngineOptions.width} and {@apilink EngineOptions.height}, or vice versa.
     */
    viewport?: ViewportDimension;
    /**
     * Optionally specify the size the logical pixel resolution, if not specified it will be width x height.
     * See {@apilink Resolution} for common presets.
     */
    resolution?: Resolution;
    /**
     * Optionally specify antialiasing (smoothing), by default true (smooth pixels)
     *
     *  * `true` - useful for high resolution art work you would like smoothed, this also hints excalibur to load images
     * with default blending {@apilink ImageFiltering.Blended}
     *
     *  * `false` - useful for pixel art style art work you would like sharp, this also hints excalibur to load images
     * with default blending {@apilink ImageFiltering.Pixel}
     *
     * * {@apilink AntialiasOptions} Optionally deeply configure the different antialiasing settings, **WARNING** thar be dragons here.
     * It is recommended you stick to `true` or `false` unless you understand what you're doing and need to control rendering to
     * a high degree.
     */
    antialiasing?: boolean | AntialiasOptions;
    /**
     * Optionally specify excalibur garbage collection, by default true.
     *
     * * `true` - garbage collection defaults are enabled (default)
     *
     * * `false` - garbage collection is completely disabled (not recommended)
     *
     * * {@apilink GarbageCollectionOptions} Optionally deeply configure garbage collection settings, **WARNING** thar be dragons here.
     * It is recommended you stick to `true` or `false` unless you understand what you're doing, it is possible to get into a downward
     * spiral if collection timings are set too low where you are stuck in repeated collection.
     */
    garbageCollection?: boolean | GarbageCollectionOptions;
    /**
     * Quick convenience property to configure Excalibur to use special settings for "pretty" anti-aliased pixel art
     *
     * 1. Turns on special shader condition to blend for pixel art and enables various antialiasing settings,
     *  notice blending is ON for this special mode.
     *
     * Equivalent to:
     * ```javascript
     * antialiasing: {
     *  pixelArtSampler: true,
     *  canvasImageRendering: 'auto',
     *  filtering: ImageFiltering.Blended,
     *  webglAntialiasing: true
     * }
     * ```
     */
    pixelArt?: boolean;
    /**
     * Specify any UV padding you want use in pixels, this brings sampling into the texture if you're using
     * a sprite sheet in one image to prevent sampling bleed.
     *
     * Defaults:
     * * `antialiasing: false` or `filtering: ImageFiltering.Pixel` - 0.0;
     * * `pixelArt: true` - 0.25
     * * All else 0.01
     */
    uvPadding?: number;
    /**
     * Optionally hint the graphics context into a specific power profile
     *
     * Default "high-performance"
     */
    powerPreference?: 'default' | 'high-performance' | 'low-power';
    /**
     * Optionally upscale the number of pixels in the canvas. Normally only useful if you need a smoother look to your assets, especially
     * {@apilink Text} or Pixel Art assets.
     *
     * **WARNING** It is recommended you try using `antialiasing: true` before adjusting pixel ratio. Pixel ratio will consume more memory
     * and on mobile may break if the internal size of the canvas exceeds 4k pixels in width or height.
     *
     * Default is based the display's pixel ratio, for example a HiDPI screen might have the value 2;
     */
    pixelRatio?: number;
    /**
     * Optionally configure the native canvas transparent backdrop
     */
    enableCanvasTransparency?: boolean;
    /**
     * Optionally specify the target canvas DOM element to render the game in
     */
    canvasElementId?: string;
    /**
     * Optionally specify the target canvas DOM element directly
     */
    canvasElement?: HTMLCanvasElement;
    /**
     * Optionally enable the right click context menu on the canvas
     *
     * Default if unset is false
     */
    enableCanvasContextMenu?: boolean;
    /**
     * Optionally snap graphics to nearest pixel, default is false
     */
    snapToPixel?: boolean;
    /**
     * The {@apilink DisplayMode} of the game, by default {@apilink DisplayMode.FitScreen} with aspect ratio 4:3 (800x600).
     * Depending on this value, {@apilink width} and {@apilink height} may be ignored.
     */
    displayMode?: DisplayMode;
    /**
     * Configures the pointer scope. Pointers scoped to the 'Canvas' can only fire events within the canvas viewport; whereas, 'Document'
     * (default) scoped will fire anywhere on the page.
     */
    pointerScope?: PointerScope;
    /**
     * Suppress boot up console message, which contains the "powered by Excalibur message"
     */
    suppressConsoleBootMessage?: boolean;
    /**
     * Suppress minimum browser feature detection, it is not recommended users of excalibur switch this off. This feature ensures that
     * the currently running browser meets the minimum requirements for running excalibur. This can be useful if running on non-standard
     * browsers or if there is a bug in excalibur preventing execution.
     */
    suppressMinimumBrowserFeatureDetection?: boolean;
    /**
     * Suppress HiDPI auto detection and scaling, it is not recommended users of excalibur switch off this feature. This feature detects
     * and scales the drawing canvas appropriately to accommodate HiDPI screens.
     */
    suppressHiDPIScaling?: boolean;
    /**
     * Suppress play button, it is not recommended users of excalibur switch this feature. Some browsers require a user gesture (like a click)
     * for certain browser features to work like web audio.
     */
    suppressPlayButton?: boolean;
    /**
     * Sets the focus of the window, this is needed when hosting excalibur in a cross-origin iframe in order for certain events
     * (like keyboard) to work.
     * For example: itch.io or codesandbox.io
     *
     * By default set to true,
     */
    grabWindowFocus?: boolean;
    /**
     * Scroll prevention method.
     */
    scrollPreventionMode?: ScrollPreventionMode;
    /**
     * Optionally set the background color
     */
    backgroundColor?: Color;
    /**
     * Optionally set the maximum fps if not set Excalibur will go as fast as the device allows.
     *
     * You may want to constrain max fps if your game cannot maintain fps consistently, it can look and feel better to have a 30fps game than
     * one that bounces between 30fps and 60fps
     */
    maxFps?: number;
    /**
     * Optionally configure a fixed update timestep in milliseconds, this can be desirable if you need the physics simulation to be very stable. When
     * set the update step and physics will use the same elapsed time for each tick even if the graphical framerate drops. In order for the
     * simulation to be correct, excalibur will run multiple updates in a row (at the configured update elapsed) to catch up, for example
     * there could be X updates and 1 draw each clock step.
     *
     * **NOTE:** This does come at a potential perf cost because each catch-up update will need to be run if the fixed rate is greater than
     * the current instantaneous framerate, or perf gain if the fixed rate is less than the current framerate.
     *
     * By default is unset and updates will use the current instantaneous framerate with 1 update and 1 draw each clock step.
     *
     * **WARN:** `fixedUpdateTimestep` takes precedence over `fixedUpdateFps` use whichever is most convenient.
     */
    fixedUpdateTimestep?: number;
    /**
     * Optionally configure a fixed update fps, this can be desirable if you need the physics simulation to be very stable. When set
     * the update step and physics will use the same elapsed time for each tick even if the graphical framerate drops. In order for the
     * simulation to be correct, excalibur will run multiple updates in a row (at the configured update elapsed) to catch up, for example
     * there could be X updates and 1 draw each clock step.
     *
     * **NOTE:** This does come at a potential perf cost because each catch-up update will need to be run if the fixed rate is greater than
     * the current instantaneous framerate, or perf gain if the fixed rate is less than the current framerate.
     *
     * By default is unset and updates will use the current instantaneous framerate with 1 update and 1 draw each clock step.
     *
     * **WARN:** `fixedUpdateTimestep` takes precedence over `fixedUpdateFps` use whichever is most convenient.
     */
    fixedUpdateFps?: number;
    /**
     * Default `true`, optionally configure excalibur to use optimal draw call sorting, to opt out set this to `false`.
     *
     * Excalibur will automatically sort draw calls by z and priority into renderer batches for maximal draw performance,
     * this can disrupt a specific desired painter order.
     *
     */
    useDrawSorting?: boolean;
    /**
     * Optionally provide a custom handler for the webgl context lost event
     */
    handleContextLost?: (e: Event) => void;
    /**
     * Optionally provide a custom handler for the webgl context restored event
     */
    handleContextRestored?: (e: Event) => void;
    /**
     * Optionally configure how excalibur handles poor performance on a player's browser
     */
    configurePerformanceCanvas2DFallback?: {
        /**
         * By default `false`, this will switch the internal graphics context to Canvas2D which can improve performance on non hardware
         * accelerated browsers.
         */
        allow: boolean;
        /**
         * By default `false`, if set to `true` a dialogue will be presented to the player about their browser and how to potentially
         * address any issues.
         */
        showPlayerMessage?: boolean;
        /**
         * Default `{ numberOfFrames: 100, fps: 20 }`, optionally configure excalibur to fallback to the 2D Canvas renderer
         * if bad performance is detected.
         *
         * In this example of the default if excalibur is running at 20fps or less for 100 frames it will trigger the fallback to the 2D
         * Canvas renderer.
         */
        threshold?: {
            numberOfFrames: number;
            fps: number;
        };
    };
    /**
     * Optionally configure the physics simulation in excalibur
     *
     * If false, Excalibur will not produce a physics simulation.
     *
     * Default is configured to use {@apilink SolverStrategy.Arcade} physics simulation
     */
    physics?: boolean | PhysicsConfig;
    /**
     * Optionally specify scenes with their transitions and loaders to excalibur's scene {@apilink Director}
     *
     * Scene transitions can can overridden dynamically by the `Scene` or by the call to `.goToScene`
     */
    scenes?: SceneMap<TKnownScenes>;
}
/**
 * The Excalibur Engine
 *
 * The {@apilink Engine} is the main driver for a game. It is responsible for
 * starting/stopping the game, maintaining state, transmitting events,
 * loading resources, and managing the scene.
 */
export declare class Engine<TKnownScenes extends string = any> implements CanInitialize, CanUpdate, CanDraw {
    static Context: Context<Engine | null>;
    static useEngine(): Engine;
    static InstanceCount: number;
    /**
     * Anything run under scope can use `useEngine()` to inject the current engine
     * @param cb
     */
    scope: <TReturn>(cb: () => TReturn) => TReturn;
    private _garbageCollector;
    readonly garbageCollectorConfig: GarbageCollectionOptions | null;
    /**
     * Current Excalibur version string
     *
     * Useful for plugins or other tools that need to know what features are available
     */
    readonly version: string;
    /**
     * Listen to and emit events on the Engine
     */
    events: EventEmitter<EngineEvents>;
    /**
     * Excalibur browser events abstraction used for wiring to native browser events safely
     */
    browser: BrowserEvents;
    /**
     * Screen abstraction
     */
    screen: Screen;
    /**
     * Scene director, manages all scenes, scene transitions, and loaders in excalibur
     */
    director: Director<TKnownScenes>;
    /**
     * Direct access to the engine's canvas element
     */
    canvas: HTMLCanvasElement;
    /**
     * Direct access to the ExcaliburGraphicsContext used for drawing things to the screen
     */
    graphicsContext: ExcaliburGraphicsContext;
    /**
     * Direct access to the canvas element ID, if an ID exists
     */
    canvasElementId: string;
    /**
     * Direct access to the physics configuration for excalibur
     */
    physics: DeepRequired<PhysicsConfig>;
    /**
     * Optionally set the maximum fps if not set Excalibur will go as fast as the device allows.
     *
     * You may want to constrain max fps if your game cannot maintain fps consistently, it can look and feel better to have a 30fps game than
     * one that bounces between 30fps and 60fps
     */
    maxFps: number;
    /**
     * Optionally configure a fixed update fps, this can be desirable if you need the physics simulation to be very stable. When set
     * the update step and physics will use the same elapsed time for each tick even if the graphical framerate drops. In order for the
     * simulation to be correct, excalibur will run multiple updates in a row (at the configured update elapsed) to catch up, for example
     * there could be X updates and 1 draw each clock step.
     *
     * **NOTE:** This does come at a potential perf cost because each catch-up update will need to be run if the fixed rate is greater than
     * the current instantaneous framerate, or perf gain if the fixed rate is less than the current framerate.
     *
     * By default is unset and updates will use the current instantaneous framerate with 1 update and 1 draw each clock step.
     *
     * **WARN:** `fixedUpdateTimestep` takes precedence over `fixedUpdateFps` use whichever is most convenient.
     */
    readonly fixedUpdateFps?: number;
    /**
     * Optionally configure a fixed update timestep in milliseconds, this can be desirable if you need the physics simulation to be very stable. When
     * set the update step and physics will use the same elapsed time for each tick even if the graphical framerate drops. In order for the
     * simulation to be correct, excalibur will run multiple updates in a row (at the configured update elapsed) to catch up, for example
     * there could be X updates and 1 draw each clock step.
     *
     * **NOTE:** This does come at a potential perf cost because each catch-up update will need to be run if the fixed rate is greater than
     * the current instantaneous framerate, or perf gain if the fixed rate is less than the current framerate.
     *
     * By default is unset and updates will use the current instantaneous framerate with 1 update and 1 draw each clock step.
     *
     * **WARN:** `fixedUpdateTimestep` takes precedence over `fixedUpdateFps` use whichever is most convenient.
     */
    readonly fixedUpdateTimestep?: number;
    /**
     * Direct access to the excalibur clock
     */
    clock: Clock;
    readonly pointerScope: PointerScope;
    readonly grabWindowFocus: boolean;
    /**
     * The width of the game canvas in pixels (physical width component of the
     * resolution of the canvas element)
     */
    get canvasWidth(): number;
    /**
     * Returns half width of the game canvas in pixels (half physical width component)
     */
    get halfCanvasWidth(): number;
    /**
     * The height of the game canvas in pixels, (physical height component of
     * the resolution of the canvas element)
     */
    get canvasHeight(): number;
    /**
     * Returns half height of the game canvas in pixels (half physical height component)
     */
    get halfCanvasHeight(): number;
    /**
     * Returns the width of the engine's visible drawing surface in pixels including zoom and device pixel ratio.
     */
    get drawWidth(): number;
    /**
     * Returns half the width of the engine's visible drawing surface in pixels including zoom and device pixel ratio.
     */
    get halfDrawWidth(): number;
    /**
     * Returns the height of the engine's visible drawing surface in pixels including zoom and device pixel ratio.
     */
    get drawHeight(): number;
    /**
     * Returns half the height of the engine's visible drawing surface in pixels including zoom and device pixel ratio.
     */
    get halfDrawHeight(): number;
    /**
     * Returns whether excalibur detects the current screen to be HiDPI
     */
    get isHiDpi(): boolean;
    /**
     * Access engine input like pointer, keyboard, or gamepad
     */
    input: InputHost;
    /**
     * Map multiple input sources to specific game actions actions
     */
    inputMapper: InputMapper;
    private _inputEnabled;
    /**
     * Access Excalibur debugging functionality.
     *
     * Useful when you want to debug different aspects of built in engine features like
     *   * Transform
     *   * Graphics
     *   * Colliders
     */
    debug: DebugConfig;
    /**
     * Access {@apilink stats} that holds frame statistics.
     */
    get stats(): DebugStats;
    /**
     * The current {@apilink Scene} being drawn and updated on screen
     */
    get currentScene(): Scene;
    /**
     * The current {@apilink Scene} being drawn and updated on screen
     */
    get currentSceneName(): string;
    /**
     * The default {@apilink Scene} of the game, use {@apilink Engine.goToScene} to transition to different scenes.
     */
    get rootScene(): Scene;
    /**
     * Contains all the scenes currently registered with Excalibur
     */
    get scenes(): {
        [key: string]: Scene | SceneConstructor | SceneWithOptions;
    };
    /**
     * Indicates whether the engine is set to fullscreen or not
     */
    get isFullscreen(): boolean;
    /**
     * Indicates the current {@apilink DisplayMode} of the engine.
     */
    get displayMode(): DisplayMode;
    private _suppressPlayButton;
    /**
     * Returns the calculated pixel ration for use in rendering
     */
    get pixelRatio(): number;
    /**
     * Indicates whether audio should be paused when the game is no longer visible.
     */
    pauseAudioWhenHidden: boolean;
    /**
     * Indicates whether the engine should draw with debug information
     */
    private _isDebug;
    get isDebug(): boolean;
    /**
     * Sets the background color for the engine.
     */
    backgroundColor: Color;
    /**
     * Sets the Transparency for the engine.
     */
    enableCanvasTransparency: boolean;
    /**
     * Hints the graphics context to truncate fractional world space coordinates
     */
    get snapToPixel(): boolean;
    set snapToPixel(shouldSnapToPixel: boolean);
    /**
     * The action to take when a fatal exception is thrown
     */
    onFatalException: (e: any) => void;
    /**
     * The mouse wheel scroll prevention mode
     */
    pageScrollPreventionMode: ScrollPreventionMode;
    private _logger;
    private _toaster;
    private _compatible;
    private _timescale;
    private _loader;
    private _isInitialized;
    emit<TEventName extends EventKey<EngineEvents>>(eventName: TEventName, event: EngineEvents[TEventName]): void;
    emit(eventName: string, event?: any): void;
    on<TEventName extends EventKey<EngineEvents>>(eventName: TEventName, handler: Handler<EngineEvents[TEventName]>): Subscription;
    on(eventName: string, handler: Handler<unknown>): Subscription;
    once<TEventName extends EventKey<EngineEvents>>(eventName: TEventName, handler: Handler<EngineEvents[TEventName]>): Subscription;
    once(eventName: string, handler: Handler<unknown>): Subscription;
    off<TEventName extends EventKey<EngineEvents>>(eventName: TEventName, handler: Handler<EngineEvents[TEventName]>): void;
    off(eventName: string, handler: Handler<unknown>): void;
    off(eventName: string): void;
    /**
     * Default {@apilink EngineOptions}
     */
    private static _DEFAULT_ENGINE_OPTIONS;
    private _originalOptions;
    readonly _originalDisplayMode: DisplayMode;
    /**
     * Creates a new game using the given {@apilink EngineOptions}. By default, if no options are provided,
     * the game will be rendered full screen (taking up all available browser window space).
     * You can customize the game rendering through {@apilink EngineOptions}.
     *
     * Example:
     *
     * ```js
     * var game = new ex.Engine({
     *   width: 0, // the width of the canvas
     *   height: 0, // the height of the canvas
     *   enableCanvasTransparency: true, // the transparencySection of the canvas
     *   canvasElementId: '', // the DOM canvas element ID, if you are providing your own
     *   displayMode: ex.DisplayMode.FullScreen, // the display mode
     *   pointerScope: ex.PointerScope.Document, // the scope of capturing pointer (mouse/touch) events
     *   backgroundColor: ex.Color.fromHex('#2185d0') // background color of the engine
     * });
     *
     * // call game.start, which is a Promise
     * game.start().then(function () {
     *   // ready, set, go!
     * });
     * ```
     */
    constructor(options?: EngineOptions<TKnownScenes>);
    private _handleWebGLContextLost;
    private _performanceThresholdTriggered;
    private _fpsSamples;
    private _monitorPerformanceThresholdAndTriggerFallback;
    /**
     * Switches the engine's graphics context to the 2D Canvas.
     * @warning Some features of Excalibur will not work in this mode.
     */
    useCanvas2DFallback(): void;
    private _disposed;
    /**
     * Attempts to completely clean up excalibur resources, including removing the canvas from the dom.
     *
     * To start again you will need to new up an Engine.
     */
    dispose(): void;
    isDisposed(): boolean;
    /**
     * Returns a BoundingBox of the top left corner of the screen
     * and the bottom right corner of the screen.
     */
    getWorldBounds(): import("./").BoundingBox;
    /**
     * Gets the current engine timescale factor (default is 1.0 which is 1:1 time)
     */
    get timescale(): number;
    /**
     * Sets the current engine timescale factor. Useful for creating slow-motion effects or fast-forward effects
     * when using time-based movement.
     */
    set timescale(value: number);
    /**
     * Adds a {@apilink Timer} to the {@apilink currentScene}.
     * @param timer  The timer to add to the {@apilink currentScene}.
     */
    addTimer(timer: Timer): Timer;
    /**
     * Removes a {@apilink Timer} from the {@apilink currentScene}.
     * @param timer  The timer to remove to the {@apilink currentScene}.
     */
    removeTimer(timer: Timer): Timer;
    /**
     * Adds a {@apilink Scene} to the engine, think of scenes in Excalibur as you
     * would levels or menus.
     * @param key  The name of the scene, must be unique
     * @param scene The scene to add to the engine
     */
    addScene<TScene extends string>(key: TScene, scene: Scene | SceneConstructor | SceneWithOptions): Engine<TKnownScenes | TScene>;
    /**
     * Removes a {@apilink Scene} instance from the engine
     * @param scene  The scene to remove
     */
    removeScene(scene: Scene | SceneConstructor): void;
    /**
     * Removes a scene from the engine by key
     * @param key  The scene key to remove
     */
    removeScene(key: string): void;
    /**
     * Adds a {@apilink Scene} to the engine, think of scenes in Excalibur as you
     * would levels or menus.
     * @param sceneKey  The key of the scene, must be unique
     * @param scene     The scene to add to the engine
     */
    add(sceneKey: string, scene: Scene | SceneConstructor | SceneWithOptions): void;
    /**
     * Adds a {@apilink Timer} to the {@apilink currentScene}.
     * @param timer  The timer to add to the {@apilink currentScene}.
     */
    add(timer: Timer): void;
    /**
     * Adds a {@apilink TileMap} to the {@apilink currentScene}, once this is done the TileMap
     * will be drawn and updated.
     */
    add(tileMap: TileMap): void;
    /**
     * Adds an actor to the {@apilink currentScene} of the game. This is synonymous
     * to calling `engine.currentScene.add(actor)`.
     *
     * Actors can only be drawn if they are a member of a scene, and only
     * the {@apilink currentScene} may be drawn or updated.
     * @param actor  The actor to add to the {@apilink currentScene}
     */
    add(actor: Actor): void;
    add(entity: Entity): void;
    /**
     * Adds a {@apilink ScreenElement} to the {@apilink currentScene} of the game,
     * ScreenElements do not participate in collisions, instead the
     * remain in the same place on the screen.
     * @param screenElement  The ScreenElement to add to the {@apilink currentScene}
     */
    add(screenElement: ScreenElement): void;
    /**
     * Removes a scene instance from the engine
     * @param scene  The scene to remove
     */
    remove(scene: Scene | SceneConstructor): void;
    /**
     * Removes a scene from the engine by key
     * @param sceneKey  The scene to remove
     */
    remove(sceneKey: string): void;
    /**
     * Removes a {@apilink Timer} from the {@apilink currentScene}.
     * @param timer  The timer to remove to the {@apilink currentScene}.
     */
    remove(timer: Timer): void;
    /**
     * Removes a {@apilink TileMap} from the {@apilink currentScene}, it will no longer be drawn or updated.
     */
    remove(tileMap: TileMap): void;
    /**
     * Removes an actor from the {@apilink currentScene} of the game. This is synonymous
     * to calling `engine.currentScene.removeChild(actor)`.
     * Actors that are removed from a scene will no longer be drawn or updated.
     * @param actor  The actor to remove from the {@apilink currentScene}.
     */
    remove(actor: Actor): void;
    /**
     * Removes a {@apilink ScreenElement} to the scene, it will no longer be drawn or updated
     * @param screenElement  The ScreenElement to remove from the {@apilink currentScene}
     */
    remove(screenElement: ScreenElement): void;
    /**
     * Changes the current scene with optionally supplied:
     * * Activation data
     * * Transitions
     * * Loaders
     *
     * Example:
     * ```typescript
     * game.goToScene('myScene', {
     *   sceneActivationData: {any: 'thing at all'},
     *   destinationIn: new FadeInOut({duration: 1000, direction: 'in'}),
     *   sourceOut: new FadeInOut({duration: 1000, direction: 'out'}),
     *   loader: MyLoader
     * });
     * ```
     *
     * Scenes are defined in the Engine constructor
     * ```typescript
     * const engine = new ex.Engine({
        scenes: {...}
      });
     * ```
     * Or by adding dynamically
     *
     * ```typescript
     * engine.addScene('myScene', new ex.Scene());
     * ```
     * @param destinationScene
     * @param options
     */
    goToScene<TData = undefined>(destinationScene: WithRoot<TKnownScenes>, options?: GoToOptions<TData>): Promise<void>;
    /**
     * Transforms the current x, y from screen coordinates to world coordinates
     * @param point  Screen coordinate to convert
     */
    screenToWorldCoordinates(point: Vector): Vector;
    /**
     * Transforms a world coordinate, to a screen coordinate
     * @param point  World coordinate to convert
     */
    worldToScreenCoordinates(point: Vector): Vector;
    /**
     * Initializes the internal canvas, rendering context, display mode, and native event listeners
     */
    private _initialize;
    toggleInputEnabled(enabled: boolean): void;
    onInitialize(engine: Engine): void;
    /**
     * Gets whether the actor is Initialized
     */
    get isInitialized(): boolean;
    private _overrideInitialize;
    /**
     * Updates the entire state of the game
     * @param elapsed  Number of milliseconds elapsed since the last update.
     */
    private _update;
    /**
     * @internal
     */
    _preupdate(elapsed: number): void;
    /**
     * Safe to override method
     * @param engine The reference to the current game engine
     * @param elapsed  The time elapsed since the last update in milliseconds
     */
    onPreUpdate(engine: Engine, elapsed: number): void;
    /**
     * @internal
     */
    _postupdate(elapsed: number): void;
    /**
     * Safe to override method
     * @param engine The reference to the current game engine
     * @param elapsed  The time elapsed since the last update in milliseconds
     */
    onPostUpdate(engine: Engine, elapsed: number): void;
    /**
     * Draws the entire game
     * @param elapsed  Number of milliseconds elapsed since the last draw.
     */
    private _draw;
    /**
     * @internal
     */
    _predraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * Safe to override method to hook into pre draw
     * @param ctx {@link ExcaliburGraphicsContext} for drawing
     * @param elapsed  Number of milliseconds elapsed since the last draw.
     */
    onPreDraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * @internal
     */
    _postdraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * Safe to override method to hook into pre draw
     * @param ctx {@link ExcaliburGraphicsContext} for drawing
     * @param elapsed  Number of milliseconds elapsed since the last draw.
     */
    onPostDraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * Enable or disable Excalibur debugging functionality.
     * @param toggle a value that debug drawing will be changed to
     */
    showDebug(toggle: boolean): void;
    /**
     * Toggle Excalibur debugging functionality.
     */
    toggleDebug(): boolean;
    /**
     * Returns true when loading is totally complete and the player has clicked start
     */
    get loadingComplete(): boolean;
    private _isLoading;
    private _hideLoader;
    private _isReadyFuture;
    get ready(): boolean;
    isReady(): Promise<void>;
    /**
     * Starts the internal game loop for Excalibur after loading
     * any provided assets.
     * @param loader  Optional {@apilink Loader} to use to load resources. The default loader is {@apilink Loader},
     * override to provide your own custom loader.
     *
     * Note: start() only resolves AFTER the user has clicked the play button
     */
    start(loader?: DefaultLoader): Promise<void>;
    /**
     * Starts the internal game loop for Excalibur after configuring any routes, loaders, or transitions
     * @param startOptions Optional {@apilink StartOptions} to configure the routes for scenes in Excalibur
     *
     * Note: start() only resolves AFTER the user has clicked the play button
     */
    start(sceneName: WithRoot<TKnownScenes>, options?: StartOptions): Promise<void>;
    /**
     * Starts the internal game loop after any loader is finished
     * @param loader
     */
    start(loader?: DefaultLoader): Promise<void>;
    /**
     * Returns the current frames elapsed milliseconds
     */
    currentFrameElapsedMs: number;
    /**
     * Returns the current frame lag when in fixed update mode
     */
    currentFrameLagMs: number;
    private _lagMs;
    private _mainloop;
    /**
     * Stops Excalibur's main loop, useful for pausing the game.
     */
    stop(): void;
    /**
     * Returns the Engine's running status, Useful for checking whether engine is running or paused.
     */
    isRunning(): boolean;
    private _screenShotRequests;
    /**
     * Takes a screen shot of the current viewport and returns it as an
     * HTML Image Element.
     * @param preserveHiDPIResolution in the case of HiDPI return the full scaled backing image, by default false
     */
    screenshot(preserveHiDPIResolution?: boolean): Promise<HTMLImageElement>;
    private _checkForScreenShots;
    /**
     * Another option available to you to load resources into the game.
     * Immediately after calling this the game will pause and the loading screen
     * will appear.
     * @param loader  Some {@apilink Loadable} such as a {@apilink Loader} collection, {@apilink Sound}, or {@apilink Texture}.
     */
    load(loader: DefaultLoader, hideLoader?: boolean): Promise<void>;
}
