import type { Scene } from '../Scene';
import type { Component, ComponentCtor } from './Component';
import { Entity } from './Entity';
import { EntityManager } from './EntityManager';
import type { Query } from './Query';
import { QueryManager } from './QueryManager';
import { System, SystemType } from './System';
import type { SystemCtor } from './SystemManager';
import { SystemManager } from './SystemManager';
import type { TagQuery } from './TagQuery';
/**
 * The World is a self-contained entity component system for a particular context.
 */
export declare class World {
    scene: Scene;
    private _logger;
    queryManager: QueryManager;
    entityManager: EntityManager;
    systemManager: SystemManager;
    /**
     * The context type is passed to the system updates
     * @param scene
     */
    constructor(scene: Scene);
    /**
     * Query the ECS world for entities that match your components
     * @param requiredTypes
     */
    query<TKnownComponentCtors extends ComponentCtor<Component>>(requiredTypes: TKnownComponentCtors[]): Query<TKnownComponentCtors>;
    queryTags<TKnownTags extends string>(requiredTags: TKnownTags[]): TagQuery<TKnownTags>;
    /**
     * Update systems by type and time elapsed in milliseconds
     */
    update(type: SystemType, elapsed: number): void;
    /**
     * Add an entity to the ECS world
     * @param entity
     */
    add(entity: Entity): void;
    /**
     * Add a system to the ECS world
     * @param system
     */
    add(system: System): void;
    add(system: SystemCtor<System>): void;
    /**
     * Get a system out of the ECS world
     */
    get(system: SystemCtor<System>): System;
    /**
     * Remove an entity from the ECS world
     * @param entity
     */
    remove(entity: Entity, deferred?: boolean): void;
    /**
     * Remove a system from the ECS world
     * @param system
     */
    remove(system: System): void;
    get entities(): Entity<any>[];
    clearEntities(): void;
    clearSystems(): void;
}
