import type { Vector } from '../Math/vector';
import { BoundingBox } from '../Collision/Index';
import { Color } from '../Color';
import type { ExcaliburGraphicsContext } from './Context/ExcaliburGraphicsContext';
import type { FontOptions, FontRenderer } from './FontCommon';
import { BaseAlign, Direction, FontStyle, FontUnit, TextAlign } from './FontCommon';
import type { GraphicOptions } from './Graphic';
import { Graphic } from './Graphic';
import type { RasterOptions } from './Raster';
import { ImageFiltering } from './Filtering';
/**
 * Represents a system or web font in Excalibur
 *
 * If no options specified, the system sans-serif 10 pixel is used
 *
 * If loading a custom web font be sure to have the font loaded before you use it https://erikonarheim.com/posts/dont-test-fonts/
 */
export declare class Font extends Graphic implements FontRenderer {
    /**
     * Set the font filtering mode, by default set to {@apilink ImageFiltering.Blended} regardless of the engine default smoothing
     *
     * If you have a pixel style font that may be a reason to switch this to {@apilink ImageFiltering.Pixel}
     */
    filtering: ImageFiltering;
    constructor(options?: FontOptions & GraphicOptions & RasterOptions);
    clone(): Font;
    /**
     * Font quality determines the size of the underlying raster text, higher quality means less jagged edges.
     * If quality is set to 1, then just enough raster bitmap is generated to render the text.
     *
     * You can think of quality as how zoomed in to the text you can get before seeing jagged edges.
     *
     * (Default 2)
     */
    quality: number;
    padding: number;
    smoothing: boolean;
    lineWidth: number;
    lineDash: number[];
    color: Color;
    strokeColor?: Color;
    family: string;
    style: FontStyle;
    bold: boolean;
    unit: FontUnit;
    textAlign: TextAlign;
    baseAlign: BaseAlign;
    direction: Direction;
    /**
     * Font line height in pixels, default line height if unset
     */
    lineHeight: number | undefined;
    size: number;
    shadow?: {
        blur?: number;
        offset?: Vector;
        color?: Color;
    };
    get fontString(): string;
    private _textBounds;
    get localBounds(): BoundingBox;
    protected _drawImage(_ex: ExcaliburGraphicsContext, _x: number, _y: number): void;
    protected _rotate(ex: ExcaliburGraphicsContext): void;
    protected _flip(ex: ExcaliburGraphicsContext): void;
    private _textMeasurement;
    measureTextWithoutCache(text: string, maxWidth?: number): BoundingBox;
    /**
     * Returns a BoundingBox that is the total size of the text including multiple lines
     *
     * Does not include any padding or adjustment
     * @param text
     * @returns BoundingBox
     */
    measureText(text: string, maxWidth?: number): BoundingBox;
    protected _postDraw(ex: ExcaliburGraphicsContext): void;
    render(ex: ExcaliburGraphicsContext, text: string, colorOverride: Color, x: number, y: number, maxWidth?: number): void;
}
