import { Vector } from '../Math/vector';
import { Graphic } from './Graphic';
import type { HasTick } from './Animation';
import type { ExcaliburGraphicsContext } from './Context/ExcaliburGraphicsContext';
import { BoundingBox } from '../Collision/Index';
import { Component } from '../EntityComponentSystem/Component';
import type { Material } from './Context/material';
import type { Color } from '../Color';
/**
 * Type guard for checking if a Graphic HasTick (used for graphics that change over time like animations)
 * @param graphic
 */
export declare function hasGraphicsTick(graphic: Graphic): graphic is Graphic & HasTick;
export interface GraphicsShowOptions {
    offset?: Vector;
    anchor?: Vector;
}
export interface GraphicsComponentOptions {
    onPostDraw?: (ex: ExcaliburGraphicsContext, elapsed: number) => void;
    onPreDraw?: (ex: ExcaliburGraphicsContext, elapsed: number) => void;
    onPreTransformDraw?: (ex: ExcaliburGraphicsContext, elapsed: number) => void;
    onPostTransformDraw?: (ex: ExcaliburGraphicsContext, elapsed: number) => void;
    /**
     * Name of current graphic to use
     */
    current?: string;
    /**
     * Optionally set the color of the graphics component
     */
    color?: Color;
    /**
     * Optionally set a material to use on the graphic
     */
    material?: Material;
    /**
     * Optionally copy instances of graphics by calling .clone(), you may set this to false to avoid sharing graphics when added to the
     * component for performance reasons. By default graphics are not copied and are shared when added to the component.
     */
    copyGraphics?: boolean;
    /**
     * Optional visible flag, if the graphics component is not visible it will not be displayed
     */
    visible?: boolean;
    /**
     * Optional opacity
     */
    opacity?: number;
    /**
     * List of graphics and optionally the options per graphic
     */
    graphics?: {
        [graphicName: string]: Graphic | {
            graphic: Graphic;
            options?: GraphicsShowOptions | undefined;
        };
    };
    /**
     * Optional offset in absolute pixels to shift all graphics in this component from each graphic's anchor (default is top left corner)
     */
    offset?: Vector;
    /**
     * Optional anchor
     */
    anchor?: Vector;
}
/**
 * Component to manage drawings, using with the position component
 */
export declare class GraphicsComponent extends Component {
    private _logger;
    private _current;
    private _graphics;
    private _options;
    material: Material | null;
    /**
     * Draws after the entity transform has been applied, but before graphics component graphics have been drawn
     */
    onPreDraw?: (ctx: ExcaliburGraphicsContext, elapsed: number) => void;
    /**
     * Draws after the entity transform has been applied, and after graphics component graphics has been drawn
     */
    onPostDraw?: (ctx: ExcaliburGraphicsContext, elapsed: number) => void;
    /**
     * Draws before the entity transform has been applied before any any graphics component drawing
     */
    onPreTransformDraw?: (ctx: ExcaliburGraphicsContext, elapsed: number) => void;
    /**
     * Draws after the entity transform has been applied, and after all graphics component drawing
     */
    onPostTransformDraw?: (ctx: ExcaliburGraphicsContext, elapsed: number) => void;
    private _color?;
    /**
     * Sets or gets wether any drawing should be visible in this component
     * @deprecated use isVisible
     */
    get visible(): boolean;
    /**
     * Sets or gets wether any drawing should be visible in this component
     * @deprecated use isVisible
     */
    set visible(val: boolean);
    /**
     * Sets or gets wether any drawing should be visible in this component
     */
    isVisible: boolean;
    /**
     * Optionally force the graphic onscreen, default false. Not recommend to use for perf reasons, only if you known what you're doing.
     */
    forceOnScreen: boolean;
    /**
     * Sets or gets wither all drawings should have an opacity applied
     */
    opacity: number;
    private _offset;
    /**
     * Offset to apply to graphics by default
     */
    get offset(): Vector;
    set offset(value: Vector);
    private _anchor;
    /**
     * Anchor to apply to graphics by default
     */
    get anchor(): Vector;
    set anchor(value: Vector);
    /**
     * Sets the color of the actor's current graphic
     */
    get color(): Color | undefined;
    set color(v: Color | undefined);
    /**
     * Flip all graphics horizontally along the y-axis
     */
    flipHorizontal: boolean;
    /**
     * Flip all graphics vertically along the x-axis
     */
    flipVertical: boolean;
    /**
     * If set to true graphics added to the component will be copied. This can effect performance, but is useful if you don't want
     * changes to a graphic to effect all the places it is used.
     */
    copyGraphics: boolean;
    constructor(options?: GraphicsComponentOptions);
    getGraphic(name: string): Graphic | undefined;
    getOptions(name: string): GraphicsShowOptions | undefined;
    /**
     * Get registered graphics names
     */
    getNames(): string[];
    /**
     * Returns the currently displayed graphic
     */
    get current(): Graphic | undefined;
    /**
     * Returns the currently displayed graphic offsets
     */
    get currentOptions(): GraphicsShowOptions | undefined;
    /**
     * Returns all graphics associated with this component
     */
    get graphics(): {
        [graphicName: string]: Graphic;
    };
    /**
     * Returns all graphics options associated with this component
     */
    get options(): {
        [graphicName: string]: GraphicsShowOptions | undefined;
    };
    /**
     * Adds a named graphic to this component, if the name is "default" or not specified, it will be shown by default without needing to call
     * @param graphic
     */
    add(graphic: Graphic, options?: GraphicsShowOptions): Graphic;
    add(name: string, graphic: Graphic, options?: GraphicsShowOptions): Graphic;
    /**
     * Removes a registered graphic, if the removed graphic is the current it will switch to the default
     * @param name
     */
    remove(name: string): void;
    /**
     * Use a graphic only, will set the default graphic. Returns the new {@apilink Graphic}
     *
     * Optionally override the stored options
     * @param nameOrGraphic
     * @param options
     */
    use<T extends Graphic = Graphic>(nameOrGraphic: string | T, options?: GraphicsShowOptions): T;
    /**
     * Hide currently shown graphic
     */
    hide(): void;
    private _localBounds?;
    set localBounds(bounds: BoundingBox);
    recalculateBounds(): void;
    /**
     * Get local bounds of graphics component
     */
    get localBounds(): BoundingBox;
    /**
     * Get world bounds of graphics component
     */
    get bounds(): BoundingBox;
    /**
     * Update underlying graphics if necessary, called internally
     * @param elapsed
     * @internal
     */
    update(elapsed: number, idempotencyToken?: number): void;
    clone(): GraphicsComponent;
}
