import type { SpriteOptions } from './Sprite';
import { Sprite } from './Sprite';
import type { Loadable } from '../Interfaces/Index';
import { ImageFiltering } from './Filtering';
import { ImageWrapping } from './Wrapping';
import type { GraphicOptions } from './Graphic';
export interface ImageSourceOptions {
    filtering?: ImageFiltering;
    wrapping?: ImageWrapConfiguration | ImageWrapping;
    bustCache?: boolean;
}
export interface ImageWrapConfiguration {
    x: ImageWrapping;
    y: ImageWrapping;
}
export declare const ImageSourceAttributeConstants: {
    readonly Filtering: "filtering";
    readonly WrappingX: "wrapping-x";
    readonly WrappingY: "wrapping-y";
};
export declare class ImageSource implements Loadable<HTMLImageElement> {
    private _logger;
    private _resource;
    filtering?: ImageFiltering;
    wrapping?: ImageWrapConfiguration;
    /**
     * The original size of the source image in pixels
     */
    get width(): number;
    /**
     * The original height of the source image in pixels
     */
    get height(): number;
    private _src?;
    /**
     * Returns true if the Texture is completely loaded and is ready
     * to be drawn.
     */
    isLoaded(): boolean;
    /**
     * Access to the underlying html image element
     */
    data: HTMLImageElement;
    get image(): HTMLImageElement;
    private _readyFuture;
    /**
     * Promise the resolves when the image is loaded and ready for use, does not initiate loading
     */
    ready: Promise<HTMLImageElement>;
    readonly path: string;
    /**
     * The path to the image, can also be a data url like 'data:image/'
     * @param pathOrBase64 {string} Path to the image resource relative from the HTML document hosting the game, or absolute
     * @param options
     */
    constructor(pathOrBase64: string, options?: ImageSourceOptions);
    /**
     * The path to the image, can also be a data url like 'data:image/'
     * @param pathOrBase64 {string} Path to the image resource relative from the HTML document hosting the game, or absolute
     * @param bustCache {boolean} Should excalibur add a cache busting querystring?
     * @param filtering {ImageFiltering} Optionally override the image filtering set by {@apilink EngineOptions.antialiasing}
     */
    constructor(pathOrBase64: string, bustCache: boolean, filtering?: ImageFiltering);
    /**
     * Create an ImageSource from and HTML <image> tag element
     * @param image
     */
    static fromHtmlImageElement(image: HTMLImageElement, options?: ImageSourceOptions): ImageSource;
    static fromHtmlCanvasElement(image: HTMLCanvasElement, options?: ImageSourceOptions): ImageSource;
    static fromSvgString(svgSource: string, options?: ImageSourceOptions): ImageSource;
    /**
     * Should excalibur add a cache busting querystring? By default false.
     * Must be set before loading
     */
    get bustCache(): boolean;
    set bustCache(val: boolean);
    /**
     * Begins loading the image and returns a promise that resolves when the image is loaded
     */
    load(): Promise<HTMLImageElement>;
    /**
     * Build a sprite from this ImageSource
     */
    toSprite(options?: Omit<GraphicOptions & SpriteOptions, 'image'>): Sprite;
    /**
     * Unload images from memory
     */
    unload(): void;
}
