import type { GraphicOptions } from './Graphic';
import { Graphic } from './Graphic';
import type { ExcaliburGraphicsContext } from './Context/ExcaliburGraphicsContext';
import type { ImageSource } from './ImageSource';
import { Vector } from '../Math/vector';
/**
 * Nine slice stretch mode
 */
export declare enum NineSliceStretch {
    /**
     * Stretch the image across a dimension
     */
    Stretch = "stretch",
    /**
     * Tile the image across a dimension
     */
    Tile = "tile",
    /**
     * Tile the image across a dimension but only by whole image amounts
     */
    TileFit = "tile-fit"
}
export type NineSliceConfig = GraphicOptions & {
    /**
     * Final width of the nine slice graphic
     */
    width: number;
    /**
     * Final height of the nine slice graphic
     */
    height: number;
    /**
     *  Image source that's loaded from a Loader or individually
     *
     */
    source: ImageSource;
    /**
     *  Configuration for the source
     *
     *  Details for the source image, including:
     *
     *  width and height as numbers of the source image
     *
     *  and the 9 slice margins
     */
    sourceConfig: {
        width: number;
        height: number;
        topMargin: number;
        leftMargin: number;
        bottomMargin: number;
        rightMargin: number;
    };
    /**
     *  Configuration for the destination
     *
     *  Details for the destination image, including:
     *
     *  stretching strategies for horizontal and vertical stretching
     *
     *  and flag for drawing the center tile if desired
     */
    destinationConfig: {
        /**
         * Draw the center part of the nine slice, if false it's a completely transparent gap
         */
        drawCenter: boolean;
        /**
         * Horizontal stretch configuration
         */
        horizontalStretch: NineSliceStretch;
        /**
         * Vertical stretch configuration
         */
        verticalStretch: NineSliceStretch;
    };
};
export declare class NineSlice extends Graphic {
    private _imgSource;
    private _sourceSprite?;
    private _canvasA;
    private _canvasB;
    private _canvasC;
    private _canvasD;
    private _canvasE;
    private _canvasF;
    private _canvasG;
    private _canvasH;
    private _canvasI;
    private _logger;
    private _config;
    constructor(config: NineSliceConfig);
    /**
     * Sets the target width of the 9 slice (pixels), and recalculates the 9 slice if desired (auto)
     * @param newWidth
     * @param auto
     */
    setTargetWidth(newWidth: number, auto?: boolean): void;
    /**
     * Sets the target height of the 9 slice (pixels), and recalculates the 9 slice if desired (auto)
     * @param newHeight
     * @param auto
     */
    setTargetHeight(newHeight: number, auto?: boolean): void;
    /**
     *  Sets the 9 slice margins (pixels), and recalculates the 9 slice if desired (auto)
     */
    setMargins(left: number, top: number, right: number, bottom: number, auto?: boolean): void;
    /**
     *  Sets the stretching strategy for the 9 slice, and recalculates the 9 slice if desired (auto)
     *
     */
    setStretch(type: 'horizontal' | 'vertical' | 'both', stretch: NineSliceStretch, auto?: boolean): void;
    /**
     *  Returns the config of the 9 slice
     */
    getConfig(): NineSliceConfig;
    /**
     * Draws 1 of the 9 tiles based on parameters passed in
     * context is the ExcaliburGraphicsContext from the _drawImage function
     * destinationSize is the size of the destination image as a vector (width,height)
     * targetCanvas is the canvas to draw to
     * horizontalStretch and verticalStretch are the horizontal and vertical stretching strategies
     * marginW and marginH are optional margins for the 9 slice for positioning
     * @param context
     * @param targetCanvas
     * @param destinationSize
     * @param horizontalStretch
     * @param verticalStretch
     * @param marginWidth
     * @param marginHeight
     */
    protected _drawTile(context: ExcaliburGraphicsContext, targetCanvas: HTMLCanvasElement, destinationSize: Vector, horizontalStretch: NineSliceStretch, verticalStretch: NineSliceStretch, marginWidth?: number, marginHeight?: number): void;
    /**
     *  Draws the 9 slices to the canvas
     */
    protected _drawImage(ex: ExcaliburGraphicsContext, x: number, y: number): void;
    /**
     * Slices the source sprite into the 9 slice canvases internally
     */
    protected _initialize(): void;
    /**
     * Clones the 9 slice
     */
    clone(): NineSlice;
    /**
     * Returns the number of tiles
     */
    protected _getNumberOfTiles(tileSize: number, destinationSize: number, strategy: NineSliceStretch): number;
    /**
     * Returns the position and size of the tile
     */
    protected _calculateParams(tileNum: number, numTiles: number, tileSize: number, destinationSize: number, strategy: NineSliceStretch): {
        tempPosition: number;
        tempSize: number;
    };
}
