import type { ImageSource } from './ImageSource';
import type { SourceView } from './Sprite';
import { Sprite } from './Sprite';
import type { GraphicOptions } from './Graphic';
import type { TiledSpriteOptions } from './TiledSprite';
import { TiledSprite } from './TiledSprite';
import { Vector } from '../Math/vector';
/**
 * Specify sprite sheet spacing options, useful if your sprites are not tightly packed
 * and have space between them.
 */
export interface SpriteSheetSpacingDimensions {
    /**
     * The starting point to offset and start slicing the sprite sheet from the top left of the image.
     * Default is (0, 0)
     */
    originOffset?: {
        x?: number;
        y?: number;
    } | Vector;
    /**
     * The margin between sprites.
     * Default is (0, 0)
     */
    margin?: {
        x?: number;
        y?: number;
    } | Vector;
}
/**
 * Sprite sheet options for slicing up images
 */
export interface SpriteSheetGridOptions {
    /**
     * Source image to use for each sprite
     */
    image: ImageSource;
    /**
     * Grid definition for the sprite sheet
     */
    grid: {
        /**
         * Number of rows in the sprite sheet
         */
        rows: number;
        /**
         * Number of columns in the sprite sheet
         */
        columns: number;
        /**
         * Width of each individual sprite
         */
        spriteWidth: number;
        /**
         * Height of each individual sprite
         */
        spriteHeight: number;
    };
    /**
     * Optionally specify any spacing information between sprites
     */
    spacing?: SpriteSheetSpacingDimensions;
}
export interface SpriteSheetSparseOptions {
    /**
     * Source image to use for each sprite
     */
    image: ImageSource;
    /**
     * List of source view rectangles to create a sprite sheet from
     */
    sourceViews: SourceView[];
}
export interface SpriteSheetOptions {
    /**
     * Source sprites for the sprite sheet
     */
    sprites: Sprite[];
    /**
     * Optionally specify the number of rows in a sprite sheet (default 1 row)
     */
    rows?: number;
    /**
     * Optionally specify the number of columns in a sprite sheet (default sprites.length)
     */
    columns?: number;
}
export interface GetSpriteOptions extends GraphicOptions {
}
/**
 * Represents a collection of sprites from a source image with some organization in a grid
 */
export declare class SpriteSheet {
    readonly sprites: Sprite[];
    readonly rows: number;
    readonly columns: number;
    /**
     * Build a new sprite sheet from a list of sprites
     *
     * Use {@apilink SpriteSheet.fromImageSource} to create a SpriteSheet from an {@apilink ImageSource} organized in a grid
     * @param options
     */
    constructor(options: SpriteSheetOptions);
    /**
     * Find a sprite by their x/y integer coordinates in the SpriteSheet, for example `getSprite(0, 0)` is the {@apilink Sprite} in the top-left
     * and `getSprite(1, 0)` is the sprite one to the right.
     * @param x
     * @param y
     */
    getSprite(x: number, y: number, options?: GetSpriteOptions): Sprite;
    /**
     * Find a sprite by their x/y integer coordinates in the SpriteSheet and configures tiling to repeat by default,
     * for example `getTiledSprite(0, 0)` is the {@apilink TiledSprite} in the top-left
     * and `getTiledSprite(1, 0)` is the sprite one to the right.
     *
     * Example:
     *
     * ```typescript
     * spriteSheet.getTiledSprite(1, 0, {
     * width: game.screen.width,
     * height: 200,
     * wrapping: {
     * x: ex.ImageWrapping.Repeat,
     * y: ex.ImageWrapping.Clamp
     * }
     * });
     * ```
     * @param x
     * @param y
     * @param options
     */
    getTiledSprite(x: number, y: number, options?: Partial<Omit<TiledSpriteOptions & GraphicOptions, 'image'>>): TiledSprite;
    /**
     * Create a sprite sheet from a sparse set of {@apilink SourceView} rectangles
     * @param options
     */
    static fromImageSourceWithSourceViews(options: SpriteSheetSparseOptions): SpriteSheet;
    /**
     * Create a SpriteSheet from an {@apilink ImageSource} organized in a grid
     *
     * Example:
     * ```
     * const spriteSheet = SpriteSheet.fromImageSource({
     *   image: imageSource,
     *   grid: {
     *     rows: 5,
     *     columns: 2,
     *     spriteWidth: 32, // pixels
     *     spriteHeight: 32 // pixels
     *   },
     *   // Optionally specify spacing
     *   spacing: {
     *     // pixels from the top left to start the sprite parsing
     *     originOffset: {
     *       x: 5,
     *       y: 5
     *     },
     *     // pixels between each sprite while parsing
     *     margin: {
     *       x: 1,
     *       y: 1
     *     }
     *   }
     * })
     * ```
     * @param options
     */
    static fromImageSource(options: SpriteSheetGridOptions): SpriteSheet;
    clone(): SpriteSheet;
}
