import type { Engine } from '../Engine';
import { GlobalCoordinates } from '../Math/global-coordinates';
import type { Vector } from '../Math/vector';
import { PointerEvent } from './PointerEvent';
import { WheelEvent } from './WheelEvent';
import { PointerAbstraction } from './PointerAbstraction';
import type { EventKey, Handler, Subscription } from '../EventEmitter';
import { EventEmitter } from '../EventEmitter';
export type NativePointerEvent = globalThis.PointerEvent;
export type NativeMouseEvent = globalThis.MouseEvent;
export type NativeTouchEvent = globalThis.TouchEvent;
export type NativeWheelEvent = globalThis.WheelEvent;
export type PointerEvents = {
    move: PointerEvent;
    down: PointerEvent;
    up: PointerEvent;
    wheel: WheelEvent;
};
export declare const PointerEvents: {
    Move: string;
    Down: string;
    Up: string;
    Wheel: string;
};
export interface PointerInitOptions {
    grabWindowFocus?: boolean;
}
/**
 * The PointerEventProcessor is responsible for collecting all the events from the canvas and transforming them into GlobalCoordinates
 */
export declare class PointerEventReceiver {
    readonly target: GlobalEventHandlers & EventTarget;
    engine: Engine;
    events: EventEmitter<PointerEvents>;
    primary: PointerAbstraction;
    private _activeNativePointerIdsToNormalized;
    lastFramePointerCoords: Map<number, GlobalCoordinates>;
    currentFramePointerCoords: Map<number, GlobalCoordinates>;
    currentFramePointerDown: Map<number, boolean>;
    lastFramePointerDown: Map<number, boolean>;
    currentFrameDown: PointerEvent[];
    currentFrameUp: PointerEvent[];
    currentFrameMove: PointerEvent[];
    currentFrameCancel: PointerEvent[];
    currentFrameWheel: WheelEvent[];
    private _enabled;
    constructor(target: GlobalEventHandlers & EventTarget, engine: Engine);
    toggleEnabled(enabled: boolean): void;
    /**
     * Creates a new PointerEventReceiver with a new target and engine while preserving existing pointer event
     * handlers.
     * @param target
     * @param engine
     */
    recreate(target: GlobalEventHandlers & EventTarget, engine: Engine): PointerEventReceiver;
    private _pointers;
    /**
     * Locates a specific pointer by id, creates it if it doesn't exist
     * @param index
     */
    at(index: number): PointerAbstraction;
    /**
     * The number of pointers currently being tracked by excalibur
     */
    count(): number;
    /**
     * Is the specified pointer id down this frame
     * @param pointerId
     */
    isDown(pointerId: number): boolean;
    /**
     * Was the specified pointer id down last frame
     * @param pointerId
     */
    wasDown(pointerId: number): boolean;
    /**
     * Whether the Pointer is currently dragging.
     */
    isDragging(pointerId: number): boolean;
    /**
     * Whether the Pointer just started dragging.
     */
    isDragStart(pointerId: number): boolean;
    /**
     * Whether the Pointer just ended dragging.
     */
    isDragEnd(pointerId: number): boolean;
    emit<TEventName extends EventKey<PointerEvents>>(eventName: TEventName, event: PointerEvents[TEventName]): void;
    emit(eventName: string, event?: any): void;
    on<TEventName extends EventKey<PointerEvents>>(eventName: TEventName, handler: Handler<PointerEvents[TEventName]>): Subscription;
    on(eventName: string, handler: Handler<unknown>): Subscription;
    once<TEventName extends EventKey<PointerEvents>>(eventName: TEventName, handler: Handler<PointerEvents[TEventName]>): Subscription;
    once(eventName: string, handler: Handler<unknown>): Subscription;
    off<TEventName extends EventKey<PointerEvents>>(eventName: TEventName, handler: Handler<PointerEvents[TEventName]>): void;
    off(eventName: string, handler: Handler<unknown>): void;
    off(eventName: string): void;
    /**
     * Called internally by excalibur
     *
     * Updates the current frame pointer info and emits raw pointer events
     *
     * This does not emit events to entities, see PointerSystem
     * @internal
     */
    update(): void;
    /**
     * Clears the current frame event and pointer data
     */
    clear(): void;
    private _boundHandle;
    private _boundWheel;
    /**
     * Initializes the pointer event receiver so that it can start listening to native
     * browser events.
     */
    init(options?: PointerInitOptions): void;
    detach(): void;
    /**
     * Take native pointer id and map it to index in active pointers
     * @param nativePointerId
     */
    private _normalizePointerId;
    /**
     * Responsible for handling and parsing pointer events
     */
    private _handle;
    private _handleWheel;
    /**
     * Triggers an excalibur pointer event in a world space pos
     *
     * Useful for testing pointers in excalibur
     * @param type
     * @param pos
     */
    triggerEvent(type: 'down' | 'up' | 'move' | 'cancel', pos: Vector): void;
    private _nativeButtonToPointerButton;
    private _stringToPointerType;
}
