import type { Vector } from './vector';
export interface BezierCurveOptions {
    /**
     * [start, control1, control2, end]
     */
    controlPoints: [start: Vector, control1: Vector, control2: Vector, end: Vector];
    /**
     * Quality when sampling uniform points on the curve. Samples = 4 * quality;
     *
     * For bigger 'uniform' curves you may want to increase quality
     *
     * Default 4
     */
    quality?: number;
}
/**
 * BezierCurve that supports cubic Bezier curves.
 */
export declare class BezierCurve {
    private _distLookup;
    private _controlPoints;
    private _arcLength;
    readonly quality: number;
    constructor(options: BezierCurveOptions);
    get arcLength(): number;
    get controlPoints(): readonly [start: Vector, control1: Vector, control2: Vector, end: Vector];
    set controlPoints(points: [start: Vector, control1: Vector, control2: Vector, end: Vector]);
    setControlPoint(index: 0 | 1 | 2 | 3, point: Vector): void;
    private _calculateLookup;
    private _getTimeGivenDistance;
    /**
     * Get the point on the Bezier curve at a certain time
     * @param time Between 0-1
     */
    getPoint(time: number): Vector;
    /**
     * Get the tangent of the Bezier curve at a certain time
     * @param time Between 0-1
     */
    getTangent(time: number): Vector;
    /**
     * Get the tangent of the Bezier curve where the distance is uniformly distributed over time
     * @param time
     */
    getUniformTangent(time: number): Vector;
    /**
     * Get the normal of the Bezier curve at a certain time
     * @param time Between 0-1
     */
    getNormal(time: number): Vector;
    /**
     * Get the normal of the Bezier curve where the distance is uniformly distributed over time
     * @param time
     */
    getUniformNormal(time: number): Vector;
    /**
     * Points are spaced uniformly across the length of the curve over time
     * @param time
     */
    getUniformPoint(time: number): Vector;
    clone(): BezierCurve;
}
