import type { Sprite } from '../Graphics/Sprite';
import { SpriteSheet } from '../Graphics/SpriteSheet';
import { Animation } from '../Graphics/Animation';
import type { Loadable } from '../Interfaces/Index';
import { ImageSource } from '../Graphics/ImageSource';
/**
 * The {@apilink Texture} object allows games built in Excalibur to load image resources.
 * {@apilink Texture} is an {@apilink Loadable} which means it can be passed to a {@apilink Loader}
 * to pre-load before starting a level or game.
 */
export declare class Gif implements Loadable<ImageSource[]> {
    path: string;
    private _resource;
    /**
     * The width of the texture in pixels
     */
    width: number;
    /**
     * The height of the texture in pixels
     */
    height: number;
    private _stream?;
    private _gif?;
    private _images;
    private _animation?;
    data: ImageSource[];
    private _sprites;
    /**
     * @param path       Path to the image resource
     * @param bustCache  Optionally load texture with cache busting
     */
    constructor(path: string, bustCache?: boolean);
    /**
     * Should excalibur add a cache busting querystring? By default false.
     * Must be set before loading
     */
    get bustCache(): boolean;
    set bustCache(val: boolean);
    /**
     * Begins loading the texture and returns a promise to be resolved on completion
     */
    load(): Promise<ImageSource[]>;
    isLoaded(): boolean;
    /**
     * Return a frame of the gif as a sprite by id
     * @param id
     */
    toSprite(id?: number): Sprite | null;
    /**
     * Return the gif as a spritesheet
     */
    toSpriteSheet(): SpriteSheet | null;
    /**
     * Transform the GIF into an animation with duration per frame
     * @param durationPerFrame Optionally override duration per frame
     */
    toAnimation(durationPerFrame?: number): Animation | null;
    get readCheckBytes(): number[];
}
export interface GifFrame {
    sentinel: number;
    type: string;
    leftPos: number;
    topPos: number;
    width: number;
    height: number;
    lctFlag: boolean;
    lctBytes: [number, number, number][];
    interlaced: boolean;
    sorted: boolean;
    reserved: boolean[];
    lctSize: number;
    lzwMinCodeSize: number;
    pixels: number[];
    delayTime: number;
    delayMs: number;
}
export declare class Stream {
    data: Uint8Array;
    len: number;
    position: number;
    constructor(dataArray: ArrayBuffer);
    readByte: () => number;
    readBytes: (n: number) => number[];
    read: (n: number) => string;
    readUnsigned: () => number;
}
interface GifBlock {
    sentinel: number;
    type: string;
}
interface GCExtBlock extends GifBlock {
    type: 'ext';
    label: number;
    extType: string;
    reserved: boolean[];
    disposalMethod: number;
    userInputFlag: boolean;
    transparentColorFlag: boolean;
    delayTime: number;
    transparentColorIndex: number;
    terminator: number;
}
/**
 * GifParser for binary format
 *
 * Roughly based on the documentation https://giflib.sourceforge.net/whatsinagif/index.html
 */
export declare class GifParser {
    private _st;
    private _handler;
    frames: GifFrame[];
    images: HTMLImageElement[];
    private _currentFrameCanvas;
    private _currentFrameContext;
    globalColorTableBytes: [number, number, number][];
    checkBytes: number[];
    private _gce?;
    private _hdr?;
    constructor(stream: Stream);
    parseColorTableBytes: (entries: number) => [number, number, number][];
    readSubBlocks: () => string;
    parseHeader: () => void;
    parseExt: (block: GCExtBlock) => void;
    parseImg: (img: GifFrame) => void;
    parseBlocks: () => void;
    arrayToImage: (frame: GifFrame, colorTable: [number, number, number][]) => void;
}
export {};
