import { ScreenElement } from './ScreenElement';
import type { ActivateEvent, DeactivateEvent } from './Events';
import { InitializeEvent, PreUpdateEvent, PostUpdateEvent, PreDrawEvent, PostDrawEvent, PreDebugDrawEvent, PostDebugDrawEvent } from './Events';
import { Timer } from './Timer';
import type { Engine } from './Engine';
import { TileMap } from './TileMap';
import { Camera } from './Camera';
import { Actor } from './Actor';
import type { CanInitialize, CanActivate, CanDeactivate, CanUpdate, CanDraw, SceneActivationContext } from './Interfaces/LifecycleEvents';
import { Trigger } from './Trigger';
import { World } from './EntityComponentSystem/World';
import { Entity } from './EntityComponentSystem/Entity';
import type { ExcaliburGraphicsContext } from './Graphics';
import { PhysicsWorld } from './Collision/PhysicsWorld';
import type { EventKey, Handler, Subscription } from './EventEmitter';
import { EventEmitter } from './EventEmitter';
import type { Color } from './Color';
import type { DefaultLoader } from './Director/DefaultLoader';
import type { Transition } from './Director';
import { InputHost } from './Input/InputHost';
export declare class PreLoadEvent {
    loader: DefaultLoader;
}
export type SceneEvents = {
    initialize: InitializeEvent<Scene>;
    activate: ActivateEvent;
    deactivate: DeactivateEvent;
    preupdate: PreUpdateEvent;
    postupdate: PostUpdateEvent;
    predraw: PreDrawEvent;
    postdraw: PostDrawEvent;
    predebugdraw: PreDebugDrawEvent;
    postdebugdraw: PostDebugDrawEvent;
    preload: PreLoadEvent;
    transitionstart: Transition;
    transitionend: Transition;
};
export declare const SceneEvents: {
    readonly Initialize: "initialize";
    readonly Activate: "activate";
    readonly Deactivate: "deactivate";
    readonly PreUpdate: "preupdate";
    readonly PostUpdate: "postupdate";
    readonly PreDraw: "predraw";
    readonly PostDraw: "postdraw";
    readonly PreDebugDraw: "predebugdraw";
    readonly PostDebugDraw: "postdebugdraw";
    readonly PreLoad: "preload";
    readonly TransitionStart: "transitionstart";
    readonly TransitionEnd: "transitionend";
};
export type SceneConstructor = new (...args: any[]) => Scene;
/**
 *
 */
export declare function isSceneConstructor(x: any): x is SceneConstructor;
/**
 * {@apilink Actor | `Actors`} are composed together into groupings called Scenes in
 * Excalibur. The metaphor models the same idea behind real world
 * actors in a scene. Only actors in scenes will be updated and drawn.
 *
 * Typical usages of a scene include: levels, menus, loading screens, etc.
 */
export declare class Scene<TActivationData = unknown> implements CanInitialize, CanActivate<TActivationData>, CanDeactivate, CanUpdate, CanDraw {
    private _logger;
    events: EventEmitter<SceneEvents>;
    /**
     * Gets or sets the current camera for the scene
     */
    camera: Camera;
    /**
     * Scene specific background color
     */
    backgroundColor?: Color;
    /**
     * The ECS world for the scene
     */
    world: World;
    /**
     * The Excalibur physics world for the scene. Used to interact
     * with colliders included in the scene.
     *
     * Can be used to perform scene ray casts, track colliders, broadphase, and narrowphase.
     */
    physics: PhysicsWorld;
    /**
     * The actors in the current scene
     */
    get actors(): readonly Actor[];
    /**
     * The entities in the current scene
     */
    get entities(): readonly Entity[];
    /**
     * The triggers in the current scene
     */
    get triggers(): readonly Trigger[];
    /**
     * The {@apilink TileMap}s in the scene, if any
     */
    get tileMaps(): readonly TileMap[];
    /**
     * Access to the Excalibur engine
     */
    engine: Engine;
    /**
     * Access scene specific input, handlers on this only fire when this scene is active.
     */
    input: InputHost;
    private _isInitialized;
    private _timers;
    get timers(): readonly Timer[];
    private _cancelQueue;
    constructor();
    emit<TEventName extends EventKey<SceneEvents>>(eventName: TEventName, event: SceneEvents[TEventName]): void;
    emit(eventName: string, event?: any): void;
    on<TEventName extends EventKey<SceneEvents>>(eventName: TEventName, handler: Handler<SceneEvents[TEventName]>): Subscription;
    on(eventName: string, handler: Handler<unknown>): Subscription;
    once<TEventName extends EventKey<SceneEvents>>(eventName: TEventName, handler: Handler<SceneEvents[TEventName]>): Subscription;
    once(eventName: string, handler: Handler<unknown>): Subscription;
    off<TEventName extends EventKey<SceneEvents>>(eventName: TEventName, handler: Handler<SceneEvents[TEventName]>): void;
    off(eventName: string, handler: Handler<unknown>): void;
    off(eventName: string): void;
    /**
     * Event hook to provide Scenes a way of loading scene specific resources.
     *
     * This is called before the Scene.onInitialize during scene transition. It will only ever fire once for a scene.
     * @param loader
     */
    onPreLoad(loader: DefaultLoader): void;
    /**
     * Event hook fired directly before transition, either "in" or "out" of the scene
     *
     * This overrides the Engine scene definition. However transitions specified in goToScene take highest precedence
     *
     * ```typescript
     * // Overrides all
     * Engine.goToScene('scene', { destinationIn: ..., sourceOut: ... });
     * ```
     *
     * This can be used to configure custom transitions for a scene dynamically
     */
    onTransition(direction: 'in' | 'out'): Transition | undefined;
    /**
     * This is called before the first update of the {@apilink Scene}. Initializes scene members like the camera. This method is meant to be
     * overridden. This is where initialization of child actors should take place.
     */
    onInitialize(engine: Engine): void;
    /**
     * This is called when the scene is made active and started. It is meant to be overridden,
     * this is where you should setup any DOM UI or event handlers needed for the scene.
     */
    onActivate(context: SceneActivationContext<TActivationData>): void;
    /**
     * This is called when the scene is made transitioned away from and stopped. It is meant to be overridden,
     * this is where you should cleanup any DOM UI or event handlers needed for the scene.
     */
    onDeactivate(context: SceneActivationContext): void;
    /**
     * Safe to override onPreUpdate lifecycle event handler. Synonymous with `.on('preupdate', (evt) =>{...})`
     *
     * `onPreUpdate` is called directly before a scene is updated.
     * @param engine reference to the engine
     * @param elapsed  Number of milliseconds elapsed since the last draw.
     */
    onPreUpdate(engine: Engine, elapsed: number): void;
    /**
     * Safe to override onPostUpdate lifecycle event handler. Synonymous with `.on('preupdate', (evt) =>{...})`
     *
     * `onPostUpdate` is called directly after a scene is updated.
     * @param engine reference to the engine
     * @param elapsed  Number of milliseconds elapsed since the last draw.
     */
    onPostUpdate(engine: Engine, elapsed: number): void;
    /**
     * Safe to override onPreDraw lifecycle event handler. Synonymous with `.on('preupdate', (evt) =>{...})`
     *
     * `onPreDraw` is called directly before a scene is drawn.
     *
     */
    onPreDraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * Safe to override onPostDraw lifecycle event handler. Synonymous with `.on('preupdate', (evt) =>{...})`
     *
     * `onPostDraw` is called directly after a scene is drawn.
     *
     */
    onPostDraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * Initializes actors in the scene
     */
    private _initializeChildren;
    /**
     * Gets whether or not the {@apilink Scene} has been initialized
     */
    get isInitialized(): boolean;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Initializes the scene before the first update, meant to be called by engine not by users of
     * Excalibur
     * @internal
     */
    _initialize(engine: Engine): Promise<void>;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Activates the scene with the base behavior, then calls the overridable `onActivate` implementation.
     * @internal
     */
    _activate(context: SceneActivationContext<TActivationData>): Promise<void>;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Deactivates the scene with the base behavior, then calls the overridable `onDeactivate` implementation.
     * @internal
     */
    _deactivate(context: SceneActivationContext<never>): Promise<void>;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _preupdate handler for {@apilink onPreUpdate} lifecycle event
     * @internal
     */
    _preupdate(engine: Engine, elapsed: number): void;
    /**
     *  It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _preupdate handler for {@apilink onPostUpdate} lifecycle event
     * @internal
     */
    _postupdate(engine: Engine, elapsed: number): void;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _predraw handler for {@apilink onPreDraw} lifecycle event
     * @internal
     */
    _predraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _postdraw handler for {@apilink onPostDraw} lifecycle event
     * @internal
     */
    _postdraw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * Updates all the actors and timers in the scene. Called by the {@apilink Engine}.
     * @param engine  Reference to the current Engine
     * @param elapsed   The number of milliseconds since the last update
     */
    update(engine: Engine, elapsed: number): void;
    /**
     * Draws all the actors in the Scene. Called by the {@apilink Engine}.
     * @param ctx    The current rendering context
     * @param elapsed  The number of milliseconds since the last draw
     */
    draw(ctx: ExcaliburGraphicsContext, elapsed: number): void;
    /**
     * Draws all the actors' debug information in the Scene. Called by the {@apilink Engine}.
     * @param ctx  The current rendering context
     */
    debugDraw(ctx: ExcaliburGraphicsContext): void;
    /**
     * Checks whether an actor is contained in this scene or not
     */
    contains(actor: Actor): boolean;
    /**
     * Adds a {@apilink Timer} to the current {@apilink Scene}.
     * @param timer  The timer to add to the current {@apilink Scene}.
     */
    add(timer: Timer): void;
    /**
     * Adds a {@apilink TileMap} to the {@apilink Scene}, once this is done the {@apilink TileMap} will be drawn and updated.
     */
    add(tileMap: TileMap): void;
    /**
     * Adds a {@apilink Trigger} to the {@apilink Scene}, once this is done the {@apilink Trigger} will listen for interactions with other actors.
     * @param trigger
     */
    add(trigger: Trigger): void;
    /**
     * Adds an actor to the scene, once this is done the {@apilink Actor} will be drawn and updated.
     * @param actor  The actor to add to the current scene
     */
    add(actor: Actor): void;
    /**
     * Adds an {@apilink Entity} to the scene, once this is done the {@apilink Actor} will be drawn and updated.
     * @param entity The entity to add to the current scene
     */
    add(entity: Entity): void;
    /**
     * Adds a {@apilink ScreenElement} to the scene.
     * @param screenElement  The ScreenElement to add to the current scene
     */
    add(screenElement: ScreenElement): void;
    /**
     * Removes a {@apilink Timer} from it's current scene
     * and adds it to this scene.
     *
     * Useful if you want to have an object be present in only 1 scene at a time.
     * @param timer The Timer to transfer to the current scene
     */
    transfer(timer: Timer): void;
    /**
     * Removes a {@apilink TileMap} from it's current scene
     * and adds it to this scene.
     *
     * Useful if you want to have an object be present in only 1 scene at a time.
     * @param tileMap The TileMap to transfer to the current scene
     */
    transfer(tileMap: TileMap): void;
    /**
     * Removes a {@apilink Trigger} from it's current scene
     * and adds it to this scene.
     *
     * Useful if you want to have an object be present in only 1 scene at a time.
     * @param trigger The Trigger to transfer to the current scene
     */
    transfer(trigger: Trigger): void;
    /**
     * Removes an {@apilink Actor} from it's current scene
     * and adds it to this scene.
     *
     * Useful if you want to have an object be present in only 1 scene at a time.
     * @param actor The Actor to transfer to the current scene
     */
    transfer(actor: Actor): void;
    /**
     * Removes an {@apilink Entity} from it's current scene
     * and adds it to this scene.
     *
     * Useful if you want to have an object be present in only 1 scene at a time.
     * @param entity The Entity to transfer to the current scene
     */
    transfer(entity: Entity): void;
    /**
     * Removes a {@apilink ScreenElement} from it's current scene
     * and adds it to this scene.
     *
     * Useful if you want to have an object be present in only 1 scene at a time.
     * @param screenElement The ScreenElement to transfer to the current scene
     */
    transfer(screenElement: ScreenElement): void;
    /**
     * Removes a {@apilink Timer} from the current scene, it will no longer be updated.
     * @param timer  The timer to remove to the current scene.
     */
    remove(timer: Timer): void;
    /**
     * Removes a {@apilink TileMap} from the scene, it will no longer be drawn or updated.
     * @param tileMap {TileMap}
     */
    remove(tileMap: TileMap): void;
    /**
     * Removes an actor from the scene, it will no longer be drawn or updated.
     * @param actor  The actor to remove from the current scene.
     */
    remove(actor: Actor): void;
    remove(entity: Entity): void;
    /**
     * Removes a {@apilink ScreenElement} to the scene, it will no longer be drawn or updated
     * @param screenElement  The ScreenElement to remove from the current scene
     */
    remove(screenElement: ScreenElement): void;
    /**
     * Removes all entities and timers from the scene, optionally indicate whether deferred should or shouldn't be used.
     *
     * By default entities use deferred removal
     * @param deferred
     */
    clear(deferred?: boolean): void;
    /**
     * Adds a {@apilink Timer} to the scene
     * @param timer  The timer to add
     */
    addTimer(timer: Timer): Timer;
    /**
     * Removes a {@apilink Timer} from the scene.
     * @warning Can be dangerous, use {@apilink cancelTimer} instead
     * @param timer  The timer to remove
     */
    removeTimer(timer: Timer): Timer;
    /**
     * Cancels a {@apilink Timer}, removing it from the scene nicely
     * @param timer  The timer to cancel
     */
    cancelTimer(timer: Timer): Timer;
    /**
     * Tests whether a {@apilink Timer} is active in the scene
     */
    isTimerActive(timer: Timer): boolean;
    isCurrentScene(): boolean;
    private _collectActorStats;
}
