import type { Scene } from './Scene';
import type * as ex from './index';
export interface TimerOptions {
    /**
     * If true the timer repeats every interval infinitely
     */
    repeats?: boolean;
    /**
     * If a number is specified then it will only repeat a number of times
     */
    numberOfRepeats?: number;
    /**
     * @deprecated use action: () => void, will be removed in v1.0
     */
    fcn?: () => void;
    /**
     * Action to perform every time the timer fires
     */
    action?: () => void;
    /**
     * Interval in milliseconds for the timer to fire
     */
    interval: number;
    /**
     * Optionally specify a random range of milliseconds for the timer to fire
     */
    randomRange?: [number, number];
    /**
     * Optionally provide a random instance to use for random behavior, otherwise a new random will be created seeded from the current time.
     */
    random?: ex.Random;
}
/**
 * The Excalibur timer hooks into the internal timer and fires callbacks,
 * after a certain interval, optionally repeating.
 */
export declare class Timer {
    private _logger;
    private static _MAX_ID;
    id: number;
    private _elapsedTime;
    private _totalTimeAlive;
    private _running;
    private _numberOfTicks;
    private _callbacks;
    interval: number;
    repeats: boolean;
    maxNumberOfRepeats: number;
    randomRange: [number, number];
    random: ex.Random;
    private _baseInterval;
    private _generateRandomInterval;
    private _complete;
    get complete(): boolean;
    scene: Scene;
    constructor(options: TimerOptions);
    /**
     * Adds a new callback to be fired after the interval is complete
     * @param action The callback to be added to the callback list, to be fired after the interval is complete.
     */
    on(action: () => void): void;
    /**
     * Removes a callback from the callback list to be fired after the interval is complete.
     * @param action The callback to be removed from the callback list, to be fired after the interval is complete.
     */
    off(action: () => void): void;
    /**
     * Updates the timer after a certain number of milliseconds have elapsed. This is used internally by the engine.
     * @param elapsed  Number of elapsed milliseconds since the last update.
     */
    update(elapsed: number): void;
    /**
     * Resets the timer so that it can be reused, and optionally reconfigure the timers interval.
     *
     * Warning** you may need to call `timer.start()` again if the timer had completed
     * @param newInterval If specified, sets a new non-negative interval in milliseconds to refire the callback
     * @param newNumberOfRepeats If specified, sets a new non-negative upper limit to the number of time this timer executes
     */
    reset(newInterval?: number, newNumberOfRepeats?: number): void;
    get timesRepeated(): number;
    getTimeRunning(): number;
    /**
     * @returns milliseconds until the next action callback, if complete will return 0
     */
    get timeToNextAction(): number;
    /**
     * @returns milliseconds elapsed toward the next action
     */
    get timeElapsedTowardNextAction(): number;
    get isRunning(): boolean;
    /**
     * Pauses the timer, time will no longer increment towards the next call
     */
    pause(): Timer;
    /**
     * Resumes the timer, time will now increment towards the next call.
     */
    resume(): Timer;
    /**
     * Starts the timer, if the timer was complete it will restart the timer and reset the elapsed time counter
     */
    start(): Timer;
    /**
     * Stops the timer and resets the elapsed time counter towards the next action invocation
     */
    stop(): Timer;
    /**
     * Cancels the timer, preventing any further executions.
     */
    cancel(): void;
}
