import { ActionContext } from './ActionContext';
import { Component } from '../EntityComponentSystem/Component';
import type { Entity } from '../EntityComponentSystem/Entity';
import type { Actor } from '../Actor';
import { MotionComponent } from '../EntityComponentSystem/Components/MotionComponent';
import { TransformComponent } from '../EntityComponentSystem/Components/TransformComponent';
import type { Vector, RotationType } from '../Math';
import type { EasingFunction } from '../Util/EasingFunctions';
import type { ActionQueue } from './ActionQueue';
import type { Action } from './Action';
import type { Color } from '../Color';
import type { CurveToOptions } from './Action/CurveTo';
import type { CurveByOptions } from './Action/CurveBy';
import type { MoveToOptions } from './Action/MoveTo';
import type { MoveByOptions, RotateByOptions, RotateToOptions, ScaleByOptions, ScaleToOptions } from './index';
export interface ActionContextMethods extends Pick<ActionContext, keyof ActionContext> {
}
export declare class ActionsComponent extends Component implements ActionContextMethods {
    dependencies: (typeof TransformComponent | typeof MotionComponent)[];
    private _ctx;
    onAdd(entity: Entity): void;
    onRemove(): void;
    private _getCtx;
    /**
     * Returns the internal action queue
     * @returns action queue
     */
    getQueue(): ActionQueue;
    /**
     * Runs a specific action in the action queue
     * @param action
     */
    runAction(action: Action): ActionContext;
    /**
     * Updates the internal action context, performing action and moving through the internal queue
     * @param elapsed
     */
    update(elapsed: number): void;
    /**
     * Clears all queued actions from the Actor
     */
    clearActions(): void;
    /**
     * Animates an actor with a specified bezier curve by an offset to the current position, the start point is assumed
     * to be the actors current position
     * @param options
     */
    curveBy(options: CurveByOptions): ActionContext;
    /**
     * Animates an actor with a specified bezier curve to an absolute world space coordinate, the start point is assumed
     * to be the actors current position
     * @param options
     */
    curveTo(options: CurveToOptions): ActionContext;
    /**
     * This method will move an actor to the specified `x` and `y` position over the
     * specified duration using a given {@apilink EasingFunctions} and return back the actor. This
     * method is part of the actor 'Action' fluent API allowing action chaining.
     * @param pos       The x,y vector location to move the actor to
     * @param duration  The time it should take the actor to move to the new location in milliseconds
     * @param easingFcn Use {@apilink EasingFunctions} or a custom function to use to calculate position, Default is {@apilink EasingFunctions.Linear}
     * @deprecated use new moveTo({pos: Vector, duration: number, easing: EasingFunction})
     */
    easeTo(pos: Vector, duration: number, easingFcn?: EasingFunction): ActionContext;
    /**
     * This method will move an actor to the specified `x` and `y` position over the
     * specified duration using a given {@apilink EasingFunctions} and return back the actor. This
     * method is part of the actor 'Action' fluent API allowing action chaining.
     * @param x         The x location to move the actor to
     * @param y         The y location to move the actor to
     * @param duration  The time it should take the actor to move to the new location in milliseconds
     * @param easingFcn Use {@apilink EasingFunctions} or a custom function to use to calculate position, Default is {@apilink EasingFunctions.Linear}
     * @deprecated use new moveTo({pos: Vector, duration: number, easing: EasingFunction})
     */
    easeTo(x: number, y: number, duration: number, easingFcn?: EasingFunction): ActionContext;
    /**
     *
     * @param offset
     * @param duration
     * @param easingFcn
     * @deprecated use new moveBy({pos: Vector, duration: number, easing: EasingFunction})
     */
    easeBy(offset: Vector, duration: number, easingFcn?: EasingFunction): ActionContext;
    /**
     *
     * @param offsetX
     * @param offsetY
     * @param duration
     * @param easingFcn
     * @deprecated use new moveBy({pos: Vector, duration: number, easing: EasingFunction})
     */
    easeBy(offsetX: number, offsetY: number, duration: number, easingFcn?: EasingFunction): ActionContext;
    /**
     * Moves an actor to a specified {@link Vector} in a given duration in milliseconds.
     * You may optionally specify an {@link EasingFunction}
     * @param options
     */
    moveTo(options: MoveToOptions): ActionContext;
    /**
     * This method will move an actor to the specified x and y position at the
     * speed specified (in pixels per second) and return back the actor. This
     * method is part of the actor 'Action' fluent API allowing action chaining.
     * @param pos    The x,y vector location to move the actor to
     * @param speed  The speed in pixels per second to move
     */
    moveTo(pos: Vector, speed: number): ActionContext;
    /**
     * This method will move an actor to the specified x and y position at the
     * speed specified (in pixels per second) and return back the actor. This
     * method is part of the actor 'Action' fluent API allowing action chaining.
     * @param x      The x location to move the actor to
     * @param y      The y location to move the actor to
     * @param speed  The speed in pixels per second to move
     */
    moveTo(x: number, y: number, speed: number): ActionContext;
    /**
     * Moves an actor by a specified offset {@link Vector} in a given duration in milliseconds.
     * You may optionally specify an {@link EasingFunction}
     * @param options
     */
    moveBy(options: MoveByOptions): ActionContext;
    /**
     * This method will move an actor by the specified x offset and y offset from its current position, at a certain speed.
     * This method is part of the actor 'Action' fluent API allowing action chaining.
     * @param offset The (x, y) offset to apply to this actor
     * @param speed  The speed in pixels per second the actor should move
     */
    moveBy(offset: Vector, speed: number): ActionContext;
    /**
     * This method will move an actor by the specified x offset and y offset from its current position, at a certain speed.
     * This method is part of the actor 'Action' fluent API allowing action chaining.
     * @param xOffset     The x offset to apply to this actor
     * @param yOffset     The y location to move the actor to
     * @param speed  The speed in pixels per second the actor should move
     */
    moveBy(xOffset: number, yOffset: number, speed: number): ActionContext;
    /**
     * Rotates an actor to a specified angle over a duration in milliseconds,
     * you make pick a rotation strategy {@link RotationType} to pick the direction
     * @param options
     */
    rotateTo(options: RotateToOptions): ActionContext;
    /**
     * This method will rotate an actor to the specified angle at the speed
     * specified (in radians per second) and return back the actor. This
     * method is part of the actor 'Action' fluent API allowing action chaining.
     * @param angle  The angle to rotate to in radians
     * @param speed         The angular velocity of the rotation specified in radians per second
     * @param rotationType  The {@apilink RotationType} to use for this rotation
     */
    rotateTo(angle: number, speed: number, rotationType?: RotationType): ActionContext;
    /**
     * Rotates an actor by a specified offset angle over a duration in milliseconds,
     * you make pick a rotation strategy {@link RotationType} to pick the direction
     * @param options
     */
    rotateBy(options: RotateByOptions): ActionContext;
    /**
     * This method will rotate an actor by the specified angle offset, from it's current rotation given a certain speed
     * in radians/sec and return back the actor. This method is part
     * of the actor 'Action' fluent API allowing action chaining.
     * @param angleRadiansOffset  The angle to rotate to in radians relative to the current rotation
     * @param speed          The speed in radians/sec the actor should rotate at
     * @param rotationType  The {@apilink RotationType} to use for this rotation, default is shortest path
     */
    rotateBy(angleRadiansOffset: number, speed: number, rotationType?: RotationType): ActionContext;
    /**
     * Scales an actor to a specified scale {@link Vector} over a duration
     * @param options
     */
    scaleTo(options: ScaleToOptions): ActionContext;
    /**
     * This method will scale an actor to the specified size at the speed
     * specified (in magnitude increase per second) and return back the
     * actor. This method is part of the actor 'Action' fluent API allowing
     * action chaining.
     * @param size    The scale to adjust the actor to over time
     * @param speed   The speed of scaling specified in magnitude increase per second
     */
    scaleTo(size: Vector, speed: Vector): ActionContext;
    /**
     * This method will scale an actor to the specified size at the speed
     * specified (in magnitude increase per second) and return back the
     * actor. This method is part of the actor 'Action' fluent API allowing
     * action chaining.
     * @param sizeX   The scaling factor to apply on X axis
     * @param sizeY   The scaling factor to apply on Y axis
     * @param speedX  The speed of scaling specified in magnitude increase per second on X axis
     * @param speedY  The speed of scaling specified in magnitude increase per second on Y axis
     */
    scaleTo(sizeX: number, sizeY: number, speedX: number, speedY: number): ActionContext;
    /**
     * Scales an actor by a specified scale offset {@link Vector} over a duration in milliseconds
     * @param options
     */
    scaleBy(options: ScaleByOptions): ActionContext;
    /**
     * This method will scale an actor by an amount relative to the current scale at a certain speed in scale units/sec
     * and return back the actor. This method is part of the
     * actor 'Action' fluent API allowing action chaining.
     * @param offset   The scaling factor to apply to the actor
     * @param speed    The speed to scale at in scale units/sec
     */
    scaleBy(offset: Vector, speed: number): ActionContext;
    /**
     * This method will scale an actor by an amount relative to the current scale at a certain speed in scale units/sec
     * and return back the actor. This method is part of the
     * actor 'Action' fluent API allowing action chaining.
     * @param sizeOffsetX   The scaling factor to apply on X axis
     * @param sizeOffsetY   The scaling factor to apply on Y axis
     * @param speed    The speed to scale at in scale units/sec
     */
    scaleBy(sizeOffsetX: number, sizeOffsetY: number, speed: number): ActionContext;
    /**
     * This method will cause an actor to blink (become visible and not
     * visible). Optionally, you may specify the number of blinks. Specify the amount of time
     * the actor should be visible per blink, and the amount of time not visible.
     * This method is part of the actor 'Action' fluent API allowing action chaining.
     * @param timeVisible     The amount of time to stay visible per blink in milliseconds
     * @param timeNotVisible  The amount of time to stay not visible per blink in milliseconds
     * @param numBlinks       The number of times to blink
     */
    blink(timeVisible: number, timeNotVisible: number, numBlinks?: number): ActionContext;
    /**
     * This method will cause an actor's opacity to change from its current value
     * to the provided value by a specified time (in milliseconds). This method is
     * part of the actor 'Action' fluent API allowing action chaining.
     * @param opacity  The ending opacity
     * @param duration     The time it should take to fade the actor (in milliseconds)
     */
    fade(opacity: number, duration: number): ActionContext;
    /**
     * This will cause an actor to flash a specific color for a period of time
     * @param color
     * @param duration The duration in milliseconds
     */
    flash(color: Color, duration?: number): ActionContext;
    /**
     * This method will delay the next action from executing for a certain
     * amount of time (in milliseconds). This method is part of the actor
     * 'Action' fluent API allowing action chaining.
     * @param duration  The amount of time to delay the next action in the queue from executing in milliseconds
     */
    delay(duration: number): ActionContext;
    /**
     * This method will add an action to the queue that will remove the actor from the
     * scene once it has completed its previous  Any actions on the
     * action queue after this action will not be executed.
     */
    die(): ActionContext;
    /**
     * This method allows you to call an arbitrary method as the next action in the
     * action queue. This is useful if you want to execute code in after a specific
     * action, i.e An actor arrives at a destination after traversing a path
     */
    callMethod(method: () => any): ActionContext;
    /**
     * This method will cause the actor to repeat all of the actions built in
     * the `repeatBuilder` callback. If the number of repeats
     * is not specified it will repeat forever. This method is part of
     * the actor 'Action' fluent API allowing action chaining
     *
     * ```typescript
     * // Move up in a zig-zag by repeated moveBy's
     * actor.actions.repeat(repeatCtx => {
     * repeatCtx.moveBy(10, 0, 10);
     * repeatCtx.moveBy(0, 10, 10);
     * }, 5);
     * ```
     * @param repeatBuilder The builder to specify the repeatable list of actions
     * @param times  The number of times to repeat all the previous actions in the action queue. If nothing is specified the actions
     * will repeat forever
     */
    repeat(repeatBuilder: (repeatContext: ActionContext) => any, times?: number): ActionContext;
    /**
     * This method will cause the actor to repeat all of the actions built in
     * the `repeatBuilder` callback. If the number of repeats
     * is not specified it will repeat forever. This method is part of
     * the actor 'Action' fluent API allowing action chaining
     *
     * ```typescript
     * // Move up in a zig-zag by repeated moveBy's
     * actor.actions.repeat(repeatCtx => {
     * repeatCtx.moveBy(10, 0, 10);
     * repeatCtx.moveBy(0, 10, 10);
     * }, 5);
     * ```
     * @param repeatBuilder The builder to specify the repeatable list of actions
     */
    repeatForever(repeatBuilder: (repeatContext: ActionContext) => any): ActionContext;
    /**
     * This method will cause the entity to follow another at a specified distance
     * @param entity           The entity to follow
     * @param followDistance  The distance to maintain when following, if not specified the actor will follow at the current distance.
     */
    follow(entity: Actor, followDistance?: number): ActionContext;
    /**
     * This method will cause the entity to move towards another until they
     * collide "meet" at a specified speed.
     * @param entity  The entity to meet
     * @param speed  The speed in pixels per second to move, if not specified it will match the speed of the other actor
     */
    meet(entity: Actor, speed?: number): ActionContext;
    /**
     * Returns a promise that resolves when the current action queue up to now
     * is finished.
     */
    toPromise(): Promise<void>;
}
