import type { PostCollisionEvent, PreCollisionEvent, CollisionStartEvent, CollisionEndEvent, EnterViewPortEvent, ExitViewPortEvent, PreDrawEvent, PostDrawEvent, PreDebugDrawEvent, PostDebugDrawEvent, ActionStartEvent, ActionCompleteEvent } from './Events';
import { KillEvent, PostKillEvent, PreKillEvent } from './Events';
import type { Engine } from './Engine';
import type { Color } from './Color';
import type { CanInitialize, CanUpdate, CanBeKilled } from './Interfaces/LifecycleEvents';
import type { Scene } from './Scene';
import { Logger } from './Util/Log';
import { Vector } from './Math/vector';
import { BodyComponent } from './Collision/BodyComponent';
import type { Eventable } from './Interfaces/Evented';
import type { PointerEvents } from './Interfaces/PointerEventHandlers';
import { CollisionType } from './Collision/CollisionType';
import type { EntityEvents } from './EntityComponentSystem/Entity';
import { Entity } from './EntityComponentSystem/Entity';
import { TransformComponent } from './EntityComponentSystem/Components/TransformComponent';
import { MotionComponent } from './EntityComponentSystem/Components/MotionComponent';
import { GraphicsComponent } from './Graphics/GraphicsComponent';
import { ColliderComponent } from './Collision/ColliderComponent';
import type { Collider, CollisionContact, CollisionGroup, Side } from './Collision/Index';
import type { PointerEvent } from './Input/PointerEvent';
import type { WheelEvent } from './Input/WheelEvent';
import { PointerComponent } from './Input/PointerComponent';
import { ActionsComponent } from './Actions/ActionsComponent';
import { CoordPlane } from './Math/coord-plane';
import type { EventKey, Handler, Subscription } from './EventEmitter';
import { EventEmitter } from './EventEmitter';
/**
 * Type guard for checking if something is an Actor
 * @param x
 */
export declare function isActor(x: any): x is Actor;
/**
 * Actor constructor options
 */
export type ActorArgs = ColliderArgs & {
    /**
     * Optionally set the name of the actor, default is 'anonymous'
     */
    name?: string;
    /**
     * Optionally set the x position of the actor, default is 0
     */
    x?: number;
    /**
     * Optionally set the y position of the actor, default is 0
     */
    y?: number;
    /**
     * Optionally set the (x, y) position of the actor as a vector, default is (0, 0)
     */
    pos?: Vector;
    /**
     * Optionally set the coordinate plane of the actor, default is {@apilink CoordPlane.World} meaning actor is subject to camera positioning
     */
    coordPlane?: CoordPlane;
    /**
     * Optionally set the velocity of the actor in pixels/sec
     */
    vel?: Vector;
    /**
     * Optionally set the acceleration of the actor in pixels/sec^2
     */
    acc?: Vector;
    /**
     * Optionally se the rotation in radians (180 degrees = Math.PI radians)
     */
    rotation?: number;
    /**
     * Optionally set the angular velocity of the actor in radians/sec (180 degrees = Math.PI radians)
     */
    angularVelocity?: number;
    /**
     * Optionally set the scale of the actor's transform
     */
    scale?: Vector;
    /**
     * Optionally set the z index of the actor, default is 0
     */
    z?: number;
    /**
     * Optionally set the color of an actor, only used if no graphics are present
     * If a width/height or a radius was set a default graphic will be added
     */
    color?: Color;
    /**
     * Optionally set the color of an actor, only used if no graphics are present
     * If a width/height or a radius was set a default graphic will be added
     */
    opacity?: number;
    /**
     * Optionally set the visibility of the actor
     */
    visible?: boolean;
    /**
     * Optionally set the anchor for graphics in the actor
     */
    anchor?: Vector;
    /**
     * Optionally set the anchor for graphics in the actor
     */
    offset?: Vector;
    /**
     * Optionally set the collision type
     */
    collisionType?: CollisionType;
    /**
     * Optionally supply a {@apilink CollisionGroup}
     */
    collisionGroup?: CollisionGroup;
    /**
     * Optionally silence excalibur warning warnings
     */
    silenceWarnings?: boolean;
};
type ColliderArgs = // custom collider
{
    /**
     * Optionally supply a collider for an actor, if supplied ignores any supplied width/height
     *
     * No default graphigc is created in this case
     */
    collider?: Collider;
    width?: undefined;
    height?: undefined;
    radius?: undefined;
    color?: undefined;
} | {
    /**
     * Optionally set the width of a box collider for the actor
     */
    width?: number;
    /**
     * Optionally set the height of a box collider for the actor
     */
    height?: number;
    /**
     * Optionally set the color of a rectangle graphic for the actor
     */
    color?: Color;
    collider?: undefined;
    radius?: undefined;
} | {
    /**
     * Optionally set the radius of the circle collider for the actor
     */
    radius?: number;
    /**
     * Optionally set the color on a circle graphic for the actor
     */
    color?: Color;
    collider?: undefined;
    width?: undefined;
    height?: undefined;
};
export type ActorEvents = EntityEvents & {
    collisionstart: CollisionStartEvent;
    collisionend: CollisionEndEvent;
    precollision: PreCollisionEvent;
    postcollision: PostCollisionEvent;
    kill: KillEvent;
    prekill: PreKillEvent;
    postkill: PostKillEvent;
    predraw: PreDrawEvent;
    postdraw: PostDrawEvent;
    pretransformdraw: PreDrawEvent;
    posttransformdraw: PostDrawEvent;
    predebugdraw: PreDebugDrawEvent;
    postdebugdraw: PostDebugDrawEvent;
    pointerup: PointerEvent;
    pointerdown: PointerEvent;
    pointerenter: PointerEvent;
    pointerleave: PointerEvent;
    pointermove: PointerEvent;
    pointercancel: PointerEvent;
    pointerwheel: WheelEvent;
    pointerdragstart: PointerEvent;
    pointerdragend: PointerEvent;
    pointerdragenter: PointerEvent;
    pointerdragleave: PointerEvent;
    pointerdragmove: PointerEvent;
    enterviewport: EnterViewPortEvent;
    exitviewport: ExitViewPortEvent;
    actionstart: ActionStartEvent;
    actioncomplete: ActionCompleteEvent;
};
export declare const ActorEvents: {
    readonly CollisionStart: "collisionstart";
    readonly CollisionEnd: "collisionend";
    readonly PreCollision: "precollision";
    readonly PostCollision: "postcollision";
    readonly Kill: "kill";
    readonly PreKill: "prekill";
    readonly PostKill: "postkill";
    readonly PreDraw: "predraw";
    readonly PostDraw: "postdraw";
    readonly PreTransformDraw: "pretransformdraw";
    readonly PostTransformDraw: "posttransformdraw";
    readonly PreDebugDraw: "predebugdraw";
    readonly PostDebugDraw: "postdebugdraw";
    readonly PointerUp: "pointerup";
    readonly PointerDown: "pointerdown";
    readonly PointerEnter: "pointerenter";
    readonly PointerLeave: "pointerleave";
    readonly PointerMove: "pointermove";
    readonly PointerCancel: "pointercancel";
    readonly Wheel: "pointerwheel";
    readonly PointerDrag: "pointerdragstart";
    readonly PointerDragEnd: "pointerdragend";
    readonly PointerDragEnter: "pointerdragenter";
    readonly PointerDragLeave: "pointerdragleave";
    readonly PointerDragMove: "pointerdragmove";
    readonly EnterViewPort: "enterviewport";
    readonly ExitViewPort: "exitviewport";
    readonly ActionStart: "actionstart";
    readonly ActionComplete: "actioncomplete";
};
/**
 * The most important primitive in Excalibur is an `Actor`. Anything that
 * can move on the screen, collide with another `Actor`, respond to events,
 * or interact with the current scene, must be an actor. An `Actor` **must**
 * be part of a {@apilink Scene} for it to be drawn to the screen.
 */
export declare class Actor extends Entity implements Eventable, PointerEvents, CanInitialize, CanUpdate, CanBeKilled {
    events: EventEmitter<ActorEvents>;
    /**
     * Set defaults for all Actors
     */
    static defaults: {
        anchor: Vector;
    };
    /**
     * The physics body the is associated with this actor. The body is the container for all physical properties, like position, velocity,
     * acceleration, mass, inertia, etc.
     */
    body: BodyComponent;
    /**
     * Access the Actor's built in {@apilink TransformComponent}
     */
    transform: TransformComponent;
    /**
     * Access the Actor's built in {@apilink MotionComponent}
     */
    motion: MotionComponent;
    /**
     * Access to the Actor's built in {@apilink GraphicsComponent}
     */
    graphics: GraphicsComponent;
    /**
     * Access to the Actor's built in {@apilink ColliderComponent}
     */
    collider: ColliderComponent;
    /**
     * Access to the Actor's built in {@apilink PointerComponent} config
     */
    pointer: PointerComponent;
    /**
     * Useful for quickly scripting actor behavior, like moving to a place, patrolling back and forth, blinking, etc.
     *
     *  Access to the Actor's built in {@apilink ActionsComponent} which forwards to the
     * {@apilink ActionContext | `Action context`} of the actor.
     */
    actions: ActionsComponent;
    private _silenceWarnings;
    /**
     * Gets the position vector of the actor in pixels
     */
    get pos(): Vector;
    /**
     * Sets the position vector of the actor in pixels
     */
    set pos(thePos: Vector);
    /**
     * Gets the position vector of the actor from the last frame
     */
    get oldPos(): Vector;
    /**
     * Gets the global position vector of the actor from the last frame
     */
    get oldGlobalPos(): Vector;
    /**
     * Sets the position vector of the actor in the last frame
     */
    set oldPos(thePos: Vector);
    /**
     * Gets the velocity vector of the actor in pixels/sec
     */
    get vel(): Vector;
    /**
     * Sets the velocity vector of the actor in pixels/sec
     */
    set vel(theVel: Vector);
    /**
     * Gets the velocity vector of the actor from the last frame
     */
    get oldVel(): Vector;
    /**
     * Sets the velocity vector of the actor from the last frame
     */
    set oldVel(theVel: Vector);
    /**
     * Gets the acceleration vector of the actor in pixels/second/second. An acceleration pointing down such as (0, 100) may be
     * useful to simulate a gravitational effect.
     */
    get acc(): Vector;
    /**
     * Sets the acceleration vector of teh actor in pixels/second/second
     */
    set acc(theAcc: Vector);
    /**
     * Sets the acceleration of the actor from the last frame. This does not include the global acc {@apilink Physics.acc}.
     */
    set oldAcc(theAcc: Vector);
    /**
     * Gets the acceleration of the actor from the last frame. This does not include the global acc {@apilink Physics.acc}.
     */
    get oldAcc(): Vector;
    /**
     * Gets the rotation of the actor in radians. 1 radian = 180/PI Degrees.
     */
    get rotation(): number;
    /**
     * Sets the rotation of the actor in radians. 1 radian = 180/PI Degrees.
     */
    set rotation(theAngle: number);
    /**
     * Gets the rotational velocity of the actor in radians/second
     */
    get angularVelocity(): number;
    /**
     * Sets the rotational velocity of the actor in radians/sec
     */
    set angularVelocity(angularVelocity: number);
    get scale(): Vector;
    set scale(scale: Vector);
    private _anchor;
    /**
     * The anchor to apply all actor related transformations like rotation,
     * translation, and scaling. By default the anchor is in the center of
     * the actor. By default it is set to the center of the actor (.5, .5)
     *
     * An anchor of (.5, .5) will ensure that drawings are centered.
     *
     * Use `anchor.setTo` to set the anchor to a different point using
     * values between 0 and 1. For example, anchoring to the top-left would be
     * `Actor.anchor.setTo(0, 0)` and top-right would be `Actor.anchor.setTo(0, 1)`.
     */
    get anchor(): Vector;
    set anchor(vec: Vector);
    private _handleAnchorChange;
    private _offset;
    /**
     * The offset in pixels to apply to all actor graphics
     *
     * Default offset of (0, 0)
     */
    get offset(): Vector;
    set offset(vec: Vector);
    private _handleOffsetChange;
    /**
     * Indicates whether the actor is physically in the viewport
     */
    get isOffScreen(): boolean;
    /**
     * Convenience reference to the global logger
     */
    logger: Logger;
    /**
     * Draggable helper
     */
    private _draggable;
    private _dragging;
    private _pointerDragStartHandler;
    private _pointerDragEndHandler;
    private _pointerDragMoveHandler;
    private _pointerDragLeaveHandler;
    get draggable(): boolean;
    set draggable(isDraggable: boolean);
    /**
     * Sets the color of the actor's current graphic
     */
    get color(): Color;
    set color(v: Color);
    /**
     *
     * @param config
     */
    constructor(config?: ActorArgs);
    clone(): Actor;
    /**
     * `onInitialize` is called before the first update of the actor. This method is meant to be
     * overridden. This is where initialization of child actors should take place.
     *
     * Synonymous with the event handler `.on('initialize', (evt) => {...})`
     */
    onInitialize(engine: Engine): void;
    /**
     * Initializes this actor and all it's child actors, meant to be called by the Scene before first update not by users of Excalibur.
     *
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     * @internal
     */
    _initialize(engine: Engine): void;
    emit<TEventName extends EventKey<ActorEvents>>(eventName: TEventName, event: ActorEvents[TEventName]): void;
    emit(eventName: string, event?: any): void;
    on<TEventName extends EventKey<ActorEvents>>(eventName: TEventName, handler: Handler<ActorEvents[TEventName]>): Subscription;
    on(eventName: string, handler: Handler<unknown>): Subscription;
    once<TEventName extends EventKey<ActorEvents>>(eventName: TEventName, handler: Handler<ActorEvents[TEventName]>): Subscription;
    once(eventName: string, handler: Handler<unknown>): Subscription;
    off<TEventName extends EventKey<ActorEvents>>(eventName: TEventName, handler: Handler<ActorEvents[TEventName]>): void;
    off(eventName: string, handler: Handler<unknown>): void;
    off(eventName: string): void;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _prekill handler for {@apilink onPreKill} lifecycle event
     * @internal
     */
    _prekill(scene: Scene): void;
    /**
     * Safe to override onPreKill lifecycle event handler. Synonymous with `.on('prekill', (evt) =>{...})`
     *
     * `onPreKill` is called directly before an actor is killed and removed from its current {@apilink Scene}.
     */
    onPreKill(scene: Scene): void;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _prekill handler for {@apilink onPostKill} lifecycle event
     * @internal
     */
    _postkill(scene: Scene): void;
    /**
     * Safe to override onPostKill lifecycle event handler. Synonymous with `.on('postkill', (evt) => {...})`
     *
     * `onPostKill` is called directly after an actor is killed and remove from its current {@apilink Scene}.
     */
    onPostKill(scene: Scene): void;
    /**
     * If the current actor is a member of the scene, this will remove
     * it from the scene graph. It will no longer be drawn or updated.
     */
    kill(): void;
    /**
     * If the current actor is killed, it will now not be killed.
     */
    unkill(): void;
    /**
     * Indicates wether the actor has been killed.
     */
    isKilled(): boolean;
    /**
     * Gets the z-index of an actor. The z-index determines the relative order an actor is drawn in.
     * Actors with a higher z-index are drawn on top of actors with a lower z-index
     */
    get z(): number;
    /**
     * Sets the z-index of an actor and updates it in the drawing list for the scene.
     * The z-index determines the relative order an actor is drawn in.
     * Actors with a higher z-index are drawn on top of actors with a lower z-index
     * @param newZ new z-index to assign
     */
    set z(newZ: number);
    /**
     * Get the center point of an actor (global position)
     */
    get center(): Vector;
    /**
     * Get the local center point of an actor
     */
    get localCenter(): Vector;
    get width(): number;
    get height(): number;
    /**
     * Gets this actor's rotation taking into account any parent relationships
     * @returns Rotation angle in radians
     * @deprecated Use {@apilink globalRotation} instead
     */
    getGlobalRotation(): number;
    /**
     * The actor's rotation (in radians) taking into account any parent relationships
     */
    get globalRotation(): number;
    /**
     * Gets an actor's world position taking into account parent relationships, scaling, rotation, and translation
     * @returns Position in world coordinates
     * @deprecated Use {@apilink globalPos} instead
     */
    getGlobalPos(): Vector;
    /**
     * The actor's world position taking into account parent relationships, scaling, rotation, and translation
     */
    get globalPos(): Vector;
    /**
     * Gets the global scale of the Actor
     * @deprecated Use {@apilink globalScale} instead
     */
    getGlobalScale(): Vector;
    /**
     * The global scale of the Actor
     */
    get globalScale(): Vector;
    /**
     * The global z-index of the actor
     */
    get globalZ(): number;
    /**
     * Tests whether the x/y specified are contained in the actor
     * @param x  X coordinate to test (in world coordinates)
     * @param y  Y coordinate to test (in world coordinates)
     * @param recurse checks whether the x/y are contained in any child actors (if they exist).
     */
    contains(x: number, y: number, recurse?: boolean): boolean;
    /**
     * Returns true if the two actor.collider's surfaces are less than or equal to the distance specified from each other
     * @param actor     Actor to test
     * @param distance  Distance in pixels to test
     */
    within(actor: Actor, distance: number): boolean;
    /**
     * Called by the Engine, updates the state of the actor
     * @internal
     * @param engine The reference to the current game engine
     * @param elapsed  The time elapsed since the last update in milliseconds
     */
    update(engine: Engine, elapsed: number): void;
    /**
     * Safe to override onPreUpdate lifecycle event handler. Synonymous with `.on('preupdate', (evt) =>{...})`
     *
     * `onPreUpdate` is called directly before an actor is updated.
     * @param engine The reference to the current game engine
     * @param elapsed  The time elapsed since the last update in milliseconds
     */
    onPreUpdate(engine: Engine, elapsed: number): void;
    /**
     * Safe to override onPostUpdate lifecycle event handler. Synonymous with `.on('postupdate', (evt) =>{...})`
     *
     * `onPostUpdate` is called directly after an actor is updated.
     * @param engine The reference to the current game engine
     * @param elapsed  The time elapsed since the last update in milliseconds
     */
    onPostUpdate(engine: Engine, elapsed: number): void;
    /**
     * Fires before every collision resolution for a confirmed contact
     * @param self
     * @param other
     * @param side
     * @param contact
     */
    onPreCollisionResolve(self: Collider, other: Collider, side: Side, contact: CollisionContact): void;
    /**
     * Fires after every resolution for a confirmed contact.
     * @param self
     * @param other
     * @param side
     * @param contact
     */
    onPostCollisionResolve(self: Collider, other: Collider, side: Side, contact: CollisionContact): void;
    /**
     * Fires once when 2 entities with a ColliderComponent first start colliding or touching, if the Colliders stay in contact this
     * does not continue firing until they separate and re-collide.
     * @param self
     * @param other
     * @param side
     * @param contact
     */
    onCollisionStart(self: Collider, other: Collider, side: Side, contact: CollisionContact): void;
    /**
     * Fires once when 2 entities with a ColliderComponent separate after having been in contact.
     * @param self
     * @param other
     * @param side
     * @param lastContact
     */
    onCollisionEnd(self: Collider, other: Collider, side: Side, lastContact: CollisionContact): void;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _preupdate handler for {@apilink onPreUpdate} lifecycle event
     * @param engine The reference to the current game engine
     * @param elapsed  The time elapsed since the last update in milliseconds
     * @internal
     */
    _preupdate(engine: Engine, elapsed: number): void;
    /**
     * It is not recommended that internal excalibur methods be overridden, do so at your own risk.
     *
     * Internal _preupdate handler for {@apilink onPostUpdate} lifecycle event
     * @param engine The reference to the current game engine
     * @param elapsed  The time elapsed since the last update in milliseconds
     * @internal
     */
    _postupdate(engine: Engine, elapsed: number): void;
}
export {};
