import type { Engine } from '../Engine';
import type { Scene } from '../Scene';
import { Entity, TransformComponent } from '../EntityComponentSystem';
import { GraphicsComponent } from '../Graphics';
import type { EasingFunction } from '../Util/EasingFunctions';
import type { CoroutineInstance } from '../Util/Coroutine';
export interface TransitionOptions {
    /**
     * Transition duration in milliseconds
     */
    duration: number;
    /**
     * Optionally hides the loader during the transition
     *
     * If either the out or in transition have this set to true, then the loader will be hidden.
     *
     * Default false
     */
    hideLoader?: boolean;
    /**
     * Optionally blocks user input during a transition
     *
     * Default false
     */
    blockInput?: boolean;
    /**
     * Optionally specify a easing function, by default linear
     */
    easing?: EasingFunction;
    /**
     * Optionally specify a transition direction, by default 'out'
     *
     * * For 'in' direction transitions start at 1 and complete is at 0
     * * For 'out' direction transitions start at 0 and complete is at 1
     */
    direction?: 'out' | 'in';
}
/**
 * Base Transition that can be extended to provide custom scene transitions in Excalibur.
 */
export declare class Transition extends Entity {
    private _logger;
    transform: TransformComponent;
    graphics: GraphicsComponent;
    readonly hideLoader: boolean;
    readonly blockInput: boolean;
    readonly duration: number;
    readonly easing: EasingFunction;
    readonly direction: 'out' | 'in';
    private _completeFuture;
    protected _engine?: Engine;
    private _co?;
    started: boolean;
    private _currentDistance;
    private _currentProgress;
    done: Promise<void>;
    /**
     * Returns a number between [0, 1] indicating what state the transition is in.
     *
     * * For 'out' direction transitions start at 0 and end at 1
     * * For 'in' direction transitions start at 1 and end at 0
     */
    get progress(): number;
    get complete(): boolean;
    constructor(options: TransitionOptions);
    /**
     * Overridable lifecycle method, called before each update.
     *
     * **WARNING BE SURE** to call `super.updateTransition()` if overriding in your own custom implementation
     * @param engine
     * @param elapsed
     */
    updateTransition(engine: Engine, elapsed: number): void;
    /**
     * Overridable lifecycle method, called right before the previous scene has deactivated.
     *
     * This gives incoming transition a chance to grab info from previous scene if desired
     * @param scene
     */
    onPreviousSceneDeactivate(scene: Scene): Promise<void>;
    /**
     * Overridable lifecycle method, called once at the beginning of the transition
     *
     * `progress` is given between 0 and 1
     * @param progress
     */
    onStart(progress: number): void;
    /**
     * Overridable lifecycle method, called every frame of the transition
     *
     * `progress` is given between 0 and 1
     * @param progress
     */
    onUpdate(progress: number): void;
    /**
     * Overridable lifecycle method, called at the end of the transition,
     *
     * `progress` is given between 0 and 1
     * @param progress
     */
    onEnd(progress: number): void;
    /**
     * Overridable lifecycle method, called when the transition is reset
     *
     * Use this to override and provide your own reset logic for internal state in custom transition implementations
     */
    onReset(): void;
    /**
     * reset() is called by the engine to reset transitions
     */
    reset(): void;
    /**
     * @internal
     */
    _addToTargetScene(engine: Engine, targetScene: Scene): CoroutineInstance;
    /**
     * Called internally by excalibur to swap scenes with transition
     * @internal
     */
    _play(): Promise<void>;
    /**
     * execute() is called by the engine every frame to update the Transition lifecycle onStart/onUpdate/onEnd
     * @internal
     */
    _execute(): void;
}
