import type { Scene } from './Scene';
import type { Vector } from './Math/vector';
import type { Actor } from './Actor';
import type { Trigger } from './Trigger';
import type { FrameStats } from './Debug';
import type { Engine } from './Engine';
import type { TileMap } from './TileMap';
import type { Side } from './Collision/Side';
import type { CollisionContact } from './Collision/Detection/CollisionContact';
import type { Collider } from './Collision/Colliders/Collider';
import type { Entity } from './EntityComponentSystem/Entity';
import type { OnInitialize, OnPreUpdate, OnPostUpdate, SceneActivationContext, OnAdd, OnRemove } from './Interfaces/LifecycleEvents';
import type { ExcaliburGraphicsContext } from './Graphics';
import type { Axes, Buttons, Gamepad } from './Input/Gamepad';
import type { Action } from './Actions/Action';
export declare enum EventTypes {
    Kill = "kill",
    PreKill = "prekill",
    PostKill = "postkill",
    PreDraw = "predraw",
    PostDraw = "postdraw",
    PreDebugDraw = "predebugdraw",
    PostDebugDraw = "postdebugdraw",
    PreUpdate = "preupdate",
    PostUpdate = "postupdate",
    PreFrame = "preframe",
    PostFrame = "postframe",
    PreCollision = "precollision",
    CollisionStart = "collisionstart",
    CollisionEnd = "collisionend",
    PostCollision = "postcollision",
    Initialize = "initialize",
    Activate = "activate",
    Deactivate = "deactivate",
    ExitViewport = "exitviewport",
    EnterViewport = "enterviewport",
    ExitTrigger = "exit",
    EnterTrigger = "enter",
    Connect = "connect",
    Disconnect = "disconnect",
    Button = "button",
    Axis = "axis",
    Visible = "visible",
    Hidden = "hidden",
    Start = "start",
    Stop = "stop",
    PointerUp = "pointerup",
    PointerDown = "pointerdown",
    PointerMove = "pointermove",
    PointerEnter = "pointerenter",
    PointerLeave = "pointerleave",
    PointerCancel = "pointercancel",
    PointerWheel = "pointerwheel",
    Up = "up",
    Down = "down",
    Move = "move",
    Enter = "enter",
    Leave = "leave",
    Cancel = "cancel",
    Wheel = "wheel",
    Press = "press",
    Release = "release",
    Hold = "hold",
    PointerDragStart = "pointerdragstart",
    PointerDragEnd = "pointerdragend",
    PointerDragEnter = "pointerdragenter",
    PointerDragLeave = "pointerdragleave",
    PointerDragMove = "pointerdragmove",
    ActionStart = "actionstart",
    ActionComplete = "actioncomplete",
    Add = "add",
    Remove = "remove"
}
export type kill = 'kill';
export type prekill = 'prekill';
export type postkill = 'postkill';
export type predraw = 'predraw';
export type postdraw = 'postdraw';
export type predebugdraw = 'predebugdraw';
export type postdebugdraw = 'postdebugdraw';
export type preupdate = 'preupdate';
export type postupdate = 'postupdate';
export type preframe = 'preframe';
export type postframe = 'postframe';
export type precollision = 'precollision';
export type collisionstart = 'collisionstart';
export type collisionend = 'collisionend';
export type postcollision = 'postcollision';
export type initialize = 'initialize';
export type activate = 'activate';
export type deactivate = 'deactivate';
export type exitviewport = 'exitviewport';
export type enterviewport = 'enterviewport';
export type exittrigger = 'exit';
export type entertrigger = 'enter';
export type connect = 'connect';
export type disconnect = 'disconnect';
export type button = 'button';
export type axis = 'axis';
export type subscribe = 'subscribe';
export type unsubscribe = 'unsubscribe';
export type visible = 'visible';
export type hidden = 'hidden';
export type start = 'start';
export type stop = 'stop';
export type pointerup = 'pointerup';
export type pointerdown = 'pointerdown';
export type pointermove = 'pointermove';
export type pointerenter = 'pointerenter';
export type pointerleave = 'pointerleave';
export type pointercancel = 'pointercancel';
export type pointerwheel = 'pointerwheel';
export type up = 'up';
export type down = 'down';
export type move = 'move';
export type enter = 'enter';
export type leave = 'leave';
export type cancel = 'cancel';
export type wheel = 'wheel';
export type press = 'press';
export type release = 'release';
export type hold = 'hold';
export type pointerdragstart = 'pointerdragstart';
export type pointerdragend = 'pointerdragend';
export type pointerdragenter = 'pointerdragenter';
export type pointerdragleave = 'pointerdragleave';
export type pointerdragmove = 'pointerdragmove';
export type add = 'add';
export type remove = 'remove';
/**
 * Base event type in Excalibur that all other event types derive from. Not all event types are thrown on all Excalibur game objects,
 * some events are unique to a type, others are not.
 *
 */
export declare class GameEvent<T, U = T> {
    /**
     * Target object for this event.
     */
    target: T;
    /**
     * Other target object for this event
     */
    other: U | null;
    /**
     * If set to false, prevents event from propagating to other actors. If true it will be propagated
     * to all actors that apply.
     */
    get bubbles(): boolean;
    set bubbles(value: boolean);
    private _bubbles;
    /**
     * Prevents event from bubbling
     */
    stopPropagation(): void;
}
/**
 * The 'kill' event is emitted on actors when it is killed. The target is the actor that was killed.
 */
export declare class KillEvent extends GameEvent<Entity> {
    self: Entity;
    constructor(self: Entity);
}
/**
 * The 'prekill' event is emitted directly before an actor is killed.
 */
export declare class PreKillEvent extends GameEvent<Actor> {
    self: Actor;
    constructor(self: Actor);
}
/**
 * The 'postkill' event is emitted directly after the actor is killed.
 */
export declare class PostKillEvent extends GameEvent<Actor> {
    self: Actor;
    constructor(self: Actor);
}
/**
 * The 'start' event is emitted on engine when has started and is ready for interaction.
 */
export declare class GameStartEvent extends GameEvent<Engine> {
    self: Engine;
    constructor(self: Engine);
}
/**
 * The 'stop' event is emitted on engine when has been stopped and will no longer take input, update or draw.
 */
export declare class GameStopEvent extends GameEvent<Engine> {
    self: Engine;
    constructor(self: Engine);
}
/**
 * The 'predraw' event is emitted on actors, scenes, and engine before drawing starts. Actors' predraw happens inside their graphics
 * transform so that all drawing takes place with the actor as the origin.
 *
 */
export declare class PreDrawEvent extends GameEvent<Entity | Scene | Engine | TileMap> {
    ctx: ExcaliburGraphicsContext;
    elapsed: number;
    self: Entity | Scene | Engine | TileMap;
    constructor(ctx: ExcaliburGraphicsContext, elapsed: number, self: Entity | Scene | Engine | TileMap);
}
/**
 * The 'postdraw' event is emitted on actors, scenes, and engine after drawing finishes. Actors' postdraw happens inside their graphics
 * transform so that all drawing takes place with the actor as the origin.
 *
 */
export declare class PostDrawEvent extends GameEvent<Entity | Scene | Engine | TileMap> {
    ctx: ExcaliburGraphicsContext;
    elapsed: number;
    self: Entity | Scene | Engine | TileMap;
    constructor(ctx: ExcaliburGraphicsContext, elapsed: number, self: Entity | Scene | Engine | TileMap);
}
/**
 * The 'pretransformdraw' event is emitted on actors/entities before any graphics transforms have taken place.
 * Useful if you need to completely customize the draw or modify the transform before drawing in the draw step (for example needing
 * latest camera positions)
 *
 */
export declare class PreTransformDrawEvent extends GameEvent<Entity> {
    ctx: ExcaliburGraphicsContext;
    elapsed: number;
    self: Entity;
    constructor(ctx: ExcaliburGraphicsContext, elapsed: number, self: Entity);
}
/**
 * The 'posttransformdraw' event is emitted on actors/entities after all graphics have been draw and transforms reset.
 * Useful if you need to completely custom the draw after everything is done.
 *
 */
export declare class PostTransformDrawEvent extends GameEvent<Entity> {
    ctx: ExcaliburGraphicsContext;
    elapsed: number;
    self: Entity;
    constructor(ctx: ExcaliburGraphicsContext, elapsed: number, self: Entity);
}
/**
 * The 'predebugdraw' event is emitted on actors, scenes, and engine before debug drawing starts.
 */
export declare class PreDebugDrawEvent extends GameEvent<Entity | Actor | Scene | Engine> {
    ctx: ExcaliburGraphicsContext;
    self: Entity | Actor | Scene | Engine;
    constructor(ctx: ExcaliburGraphicsContext, self: Entity | Actor | Scene | Engine);
}
/**
 * The 'postdebugdraw' event is emitted on actors, scenes, and engine after debug drawing starts.
 */
export declare class PostDebugDrawEvent extends GameEvent<Entity | Actor | Scene | Engine> {
    ctx: ExcaliburGraphicsContext;
    self: Entity | Actor | Scene | Engine;
    constructor(ctx: ExcaliburGraphicsContext, self: Entity | Actor | Scene | Engine);
}
/**
 * The 'preupdate' event is emitted on actors, scenes, camera, and engine before the update starts.
 */
export declare class PreUpdateEvent<T extends OnPreUpdate = Entity> extends GameEvent<T> {
    engine: Engine;
    elapsed: number;
    self: T;
    constructor(engine: Engine, elapsed: number, self: T);
}
/**
 * The 'postupdate' event is emitted on actors, scenes, camera, and engine after the update ends.
 */
export declare class PostUpdateEvent<T extends OnPostUpdate = Entity> extends GameEvent<T> {
    engine: Engine;
    elapsed: number;
    self: T;
    constructor(engine: Engine, elapsed: number, self: T);
}
/**
 * The 'preframe' event is emitted on the engine, before the frame begins.
 */
export declare class PreFrameEvent extends GameEvent<Engine> {
    engine: Engine;
    prevStats: FrameStats;
    constructor(engine: Engine, prevStats: FrameStats);
}
/**
 * The 'postframe' event is emitted on the engine, after a frame ends.
 */
export declare class PostFrameEvent extends GameEvent<Engine> {
    engine: Engine;
    stats: FrameStats;
    constructor(engine: Engine, stats: FrameStats);
}
/**
 * Event received when a gamepad is connected to Excalibur. {@apilink Gamepads} receives this event.
 */
export declare class GamepadConnectEvent extends GameEvent<Gamepad> {
    index: number;
    gamepad: Gamepad;
    constructor(index: number, gamepad: Gamepad);
}
/**
 * Event received when a gamepad is disconnected from Excalibur. {@apilink Gamepads} receives this event.
 */
export declare class GamepadDisconnectEvent extends GameEvent<Gamepad> {
    index: number;
    gamepad: Gamepad;
    constructor(index: number, gamepad: Gamepad);
}
/**
 * Gamepad button event. See {@apilink Gamepads} for information on responding to controller input. {@apilink Gamepad} instances receive this event;
 */
export declare class GamepadButtonEvent extends GameEvent<Gamepad> {
    /**
     * The Gamepad {@apilink Buttons} if not known by excalibur {@apilink Buttons.Unknown} is returned, use index to disambiguate.
     */
    button: Buttons;
    /**
     * The canonical index of the gamepad button from the system
     */
    index: number;
    /**
     * A numeric value between 0 and 1
     */
    value: number;
    /**
     * Reference to the gamepad
     */
    self: Gamepad;
    /**
     * @param button  The Gamepad {@apilink Buttons} if not known by excalibur {@apilink Buttons.Unknown} is returned, use index to disambiguate.
     * @param index   The canonical index of the gamepad button from the system
     * @param value   A numeric value between 0 and 1
     * @param self    Reference to the gamepad
     */
    constructor(
    /**
     * The Gamepad {@apilink Buttons} if not known by excalibur {@apilink Buttons.Unknown} is returned, use index to disambiguate.
     */
    button: Buttons, 
    /**
     * The canonical index of the gamepad button from the system
     */
    index: number, 
    /**
     * A numeric value between 0 and 1
     */
    value: number, 
    /**
     * Reference to the gamepad
     */
    self: Gamepad);
}
/**
 * Gamepad axis event. See {@apilink Gamepads} for information on responding to controller input. {@apilink Gamepad} instances receive this event;
 */
export declare class GamepadAxisEvent extends GameEvent<Gamepad> {
    /**
     * The Gamepad {@apilink Axis}
     */
    axis: Axes;
    /**
     * A numeric value between -1 and 1, 0 is the neutral axis position.
     */
    value: number;
    /**
     * Reference to the gamepad
     */
    self: Gamepad;
    /**
     * @param axis  The Gamepad axis
     * @param value A numeric value between -1 and 1
     * @param self Reference to the gamepad
     */
    constructor(
    /**
     * The Gamepad {@apilink Axis}
     */
    axis: Axes, 
    /**
     * A numeric value between -1 and 1, 0 is the neutral axis position.
     */
    value: number, 
    /**
     * Reference to the gamepad
     */
    self: Gamepad);
}
/**
 * Event received by the {@apilink Engine} when the browser window is visible on a screen.
 */
export declare class VisibleEvent extends GameEvent<Engine> {
    self: Engine;
    constructor(self: Engine);
}
/**
 * Event received by the {@apilink Engine} when the browser window is hidden from all screens.
 */
export declare class HiddenEvent extends GameEvent<Engine> {
    self: Engine;
    constructor(self: Engine);
}
/**
 * Event thrown on an {@apilink Actor | `actor`} when a collision will occur this frame if it resolves
 */
export declare class PreCollisionEvent<T extends Collider = Collider> extends GameEvent<T> {
    self: T;
    other: T;
    side: Side;
    intersection: Vector;
    contact: CollisionContact;
    /**
     * @param self          The actor the event was thrown on
     * @param other         The actor that will collided with the current actor
     * @param side          The side that will be collided with the current actor
     * @param intersection  Intersection vector
     */
    constructor(self: T, other: T, side: Side, intersection: Vector, contact: CollisionContact);
}
/**
 * Event thrown on an {@apilink Actor | `actor`} when a collision has been resolved (body reacted) this frame
 */
export declare class PostCollisionEvent<T extends Collider = Collider> extends GameEvent<T> {
    self: T;
    other: T;
    side: Side;
    intersection: Vector;
    contact: CollisionContact;
    /**
     * @param self          The actor the event was thrown on
     * @param other         The actor that did collide with the current actor
     * @param side          The side that did collide with the current actor
     * @param intersection  Intersection vector
     */
    constructor(self: T, other: T, side: Side, intersection: Vector, contact: CollisionContact);
}
export declare class ContactStartEvent<T extends Collider = Collider> {
    self: T;
    other: T;
    side: Side;
    contact: CollisionContact;
    constructor(self: T, other: T, side: Side, contact: CollisionContact);
}
export declare class ContactEndEvent<T extends Collider = Collider> {
    self: T;
    other: T;
    side: Side;
    lastContact: CollisionContact;
    constructor(self: T, other: T, side: Side, lastContact: CollisionContact);
}
export declare class CollisionPreSolveEvent<T extends Collider = Collider> {
    self: T;
    other: T;
    side: Side;
    intersection: Vector;
    contact: CollisionContact;
    constructor(self: T, other: T, side: Side, intersection: Vector, contact: CollisionContact);
}
export declare class CollisionPostSolveEvent<T extends Collider = Collider> {
    self: T;
    other: T;
    side: Side;
    intersection: Vector;
    contact: CollisionContact;
    constructor(self: T, other: T, side: Side, intersection: Vector, contact: CollisionContact);
}
/**
 * Event thrown the first time an {@apilink Actor | `actor`} collides with another, after an actor is in contact normal collision events are fired.
 */
export declare class CollisionStartEvent<T extends Collider = Collider> extends GameEvent<T> {
    self: T;
    other: T;
    side: Side;
    contact: CollisionContact;
    /**
     *
     * @param self
     * @param other
     * @param side
     * @param contact
     */
    constructor(self: T, other: T, side: Side, contact: CollisionContact);
}
/**
 * Event thrown when the {@apilink Actor | `actor`} is no longer colliding with another
 */
export declare class CollisionEndEvent<T extends Collider = Collider> extends GameEvent<T> {
    self: T;
    other: T;
    side: Side;
    lastContact: CollisionContact;
    /**
     *
     */
    constructor(self: T, other: T, side: Side, lastContact: CollisionContact);
}
/**
 * Event thrown on an {@apilink Actor}, {@apilink Scene}, and {@apilink Engine} only once before the first update call
 */
export declare class InitializeEvent<T extends OnInitialize = Entity> extends GameEvent<T> {
    engine: Engine;
    self: T;
    /**
     * @param engine  The reference to the current engine
     */
    constructor(engine: Engine, self: T);
}
/**
 * Event thrown on a {@apilink Scene} on activation
 */
export declare class ActivateEvent<TData = undefined> extends GameEvent<Scene> {
    context: SceneActivationContext<TData>;
    self: Scene;
    /**
     * @param context  The context for the scene activation
     */
    constructor(context: SceneActivationContext<TData>, self: Scene);
}
/**
 * Event thrown on a {@apilink Scene} on deactivation
 */
export declare class DeactivateEvent extends GameEvent<Scene> {
    context: SceneActivationContext<never>;
    self: Scene;
    /**
     * @param context  The context for the scene deactivation
     */
    constructor(context: SceneActivationContext<never>, self: Scene);
}
/**
 * Event thrown on an {@apilink Actor} when the graphics bounds completely leaves the screen.
 */
export declare class ExitViewPortEvent extends GameEvent<Entity> {
    self: Entity;
    constructor(self: Entity);
}
/**
 * Event thrown on an {@apilink Actor} when any part of the graphics bounds are on screen.
 */
export declare class EnterViewPortEvent extends GameEvent<Entity> {
    self: Entity;
    constructor(self: Entity);
}
export declare class EnterTriggerEvent extends GameEvent<Trigger> {
    self: Trigger;
    entity: Entity;
    constructor(self: Trigger, entity: Entity);
}
export declare class ExitTriggerEvent extends GameEvent<Trigger> {
    self: Trigger;
    entity: Entity;
    constructor(self: Trigger, entity: Entity);
}
/**
 * Event thrown on an {@apilink Actor} when an action starts.
 */
export declare class ActionStartEvent extends GameEvent<Entity> {
    action: Action;
    self: Entity;
    constructor(action: Action, self: Entity);
}
/**
 * Event thrown on an {@apilink Actor} when an action completes.
 */
export declare class ActionCompleteEvent extends GameEvent<Entity> {
    action: Action;
    self: Entity;
    constructor(action: Action, self: Entity);
}
/**
 * Event thrown on an [[Actor]] when an Actor added to scene.
 */
export declare class AddEvent<T extends OnAdd> extends GameEvent<T> {
    engine: Engine;
    self: T;
    constructor(engine: Engine, self: T);
}
/**
 * Event thrown on an [[Actor]] when an Actor removed from scene.
 */
export declare class RemoveEvent<T extends OnRemove> extends GameEvent<T> {
    engine: Engine;
    self: T;
    constructor(engine: Engine, self: T);
}
