import { Color } from '../../Color';
import type { ExcaliburGraphicsContext } from './ExcaliburGraphicsContext';
import type { Shader, UniformDictionary } from './shader';
import type { ImageSource } from '../ImageSource';
import type { ImageFiltering } from '../Filtering';
export interface MaterialOptions {
    /**
     * Name the material for debugging
     */
    name?: string;
    /**
     * Excalibur graphics context to create the material (only WebGL is supported at the moment)
     */
    graphicsContext?: ExcaliburGraphicsContext;
    /**
     * Optionally specify a vertex shader
     *
     * If none supplied the default will be used
     *
     * ```
     *  #version 300 es
     *  // vertex position in local space
     *  in vec2 a_position;
     *  in vec2 a_uv;
     *  out vec2 v_uv;
     *  // orthographic projection matrix
     *  uniform mat4 u_matrix;
     *  // world space transform matrix
     *  uniform mat4 u_transform;
     *  void main() {
     *    // Set the vertex position using the ortho & transform matrix
     *    gl_Position = u_matrix * u_transform * vec4(a_position, 0.0, 1.0);
     *    // Pass through the UV coord to the fragment shader
     *    v_uv = a_uv;
     *  }
     * ```
     */
    vertexSource?: string;
    /**
     * Add custom fragment shader
     *
     * *Note: Excalibur image alpha's are pre-multiplied
     *
     * Pre-built varyings:
     *
     * * `in vec2 v_uv` - UV coordinate
     * * `in vec2 v_screenuv` - UV coordinate
     *
     * Pre-built uniforms:
     *
     * * `uniform sampler2D u_graphic` - The current graphic displayed by the GraphicsComponent
     * * `uniform vec2 u_resolution` - The current resolution of the screen
     * * `uniform vec2 u_size;` - The current size of the graphic
     * * `uniform vec4 u_color` - The current color of the material
     * * `uniform float u_opacity` - The current opacity of the graphics context
     *
     */
    fragmentSource: string;
    /**
     * Add custom color, by default ex.Color.Transparent
     */
    color?: Color;
    /**
     * Add additional images to the material, you are limited by the GPU's maximum texture slots
     *
     * Specify a dictionary of uniform sampler names to ImageSource
     */
    images?: Record<string, ImageSource>;
    /**
     * Optionally set starting uniforms on a shader
     */
    uniforms?: UniformDictionary;
}
export interface MaterialImageOptions {
    filtering?: ImageFiltering;
}
export declare class Material {
    private _logger;
    private _name;
    private _shader;
    private _color;
    private _initialized;
    private _fragmentSource;
    private _vertexSource;
    private _images;
    private _uniforms;
    constructor(options: MaterialOptions);
    private _initialize;
    get uniforms(): UniformDictionary;
    get images(): Record<string, ImageSource>;
    get color(): Color;
    set color(c: Color);
    get name(): string;
    get isUsingScreenTexture(): boolean;
    update(callback: (shader: Shader) => any): void;
    getShader(): Shader | null;
    addImageSource(samplerName: string, image: ImageSource): void;
    removeImageSource(samplerName: string): void;
    use(): void;
}
