import type { Engine } from '../Engine';
import { Color } from '../Color';
/**
 * Logging level that Excalibur will tag
 */
export declare enum LogLevel {
    Debug = 0,
    Info = 1,
    Warn = 2,
    Error = 3,
    Fatal = 4
}
/**
 * Static singleton that represents the logging facility for Excalibur.
 * Excalibur comes built-in with a {@apilink ConsoleAppender} and {@apilink ScreenAppender}.
 * Derive from {@apilink Appender} to create your own logging appenders.
 */
export declare class Logger {
    private static _INSTANCE;
    private _appenders;
    constructor();
    /**
     * Gets or sets the default logging level. Excalibur will only log
     * messages if equal to or above this level. Default: {@apilink LogLevel.Info}
     */
    defaultLevel: LogLevel;
    /**
     * Gets the current static instance of Logger
     */
    static getInstance(): Logger;
    /**
     * Adds a new {@apilink Appender} to the list of appenders to write to
     */
    addAppender(appender: Appender): void;
    /**
     * Clears all appenders from the logger
     */
    clearAppenders(): void;
    /**
     * Logs a message at a given LogLevel
     * @param level  The LogLevel`to log the message at
     * @param args   An array of arguments to write to an appender
     */
    private _log;
    private _logOnceSet;
    private _logOnce;
    /**
     * Writes a log message at the {@apilink LogLevel.Debug} level
     * @param args  Accepts any number of arguments
     */
    debug(...args: any[]): void;
    /**
     * Writes a log message once at the {@apilink LogLevel.Fatal} level, if it sees the same args again it wont log
     * @param args  Accepts any number of arguments
     */
    debugOnce(...args: any[]): void;
    /**
     * Writes a log message at the {@apilink LogLevel.Info} level
     * @param args  Accepts any number of arguments
     */
    info(...args: any[]): void;
    /**
     * Writes a log message once at the {@apilink LogLevel.Info} level, if it sees the same args again it wont log
     * @param args  Accepts any number of arguments
     */
    infoOnce(...args: any[]): void;
    /**
     * Writes a log message at the {@apilink LogLevel.Warn} level
     * @param args  Accepts any number of arguments
     */
    warn(...args: any[]): void;
    /**
     * Writes a log message once at the {@apilink LogLevel.Warn} level, if it sees the same args again it won't log
     * @param args  Accepts any number of arguments
     */
    warnOnce(...args: any[]): void;
    /**
     * Writes a log message at the {@apilink LogLevel.Error} level
     * @param args  Accepts any number of arguments
     */
    error(...args: any[]): void;
    /**
     * Writes a log message once at the {@apilink LogLevel.Error} level, if it sees the same args again it won't log
     * @param args  Accepts any number of arguments
     */
    errorOnce(...args: any[]): void;
    /**
     * Writes a log message at the {@apilink LogLevel.Fatal} level
     * @param args  Accepts any number of arguments
     */
    fatal(...args: any[]): void;
    /**
     * Writes a log message once at the {@apilink LogLevel.Fatal} level, if it sees the same args again it won't log
     * @param args  Accepts any number of arguments
     */
    fatalOnce(...args: any[]): void;
}
/**
 * Contract for any log appender (such as console/screen)
 */
export interface Appender {
    /**
     * Logs a message at the given {@apilink LogLevel}
     * @param level  Level to log at
     * @param args   Arguments to log
     */
    log(level: LogLevel, args: any[]): void;
}
/**
 * Console appender for browsers (i.e. `console.log`)
 */
export declare class ConsoleAppender implements Appender {
    /**
     * Logs a message at the given {@apilink LogLevel}
     * @param level  Level to log at
     * @param args   Arguments to log
     */
    log(level: LogLevel, args: any[]): void;
}
export interface ScreenAppenderOptions {
    engine: Engine;
    /**
     * Optionally set the width of the overlay canvas
     */
    width?: number;
    /**
     * Optionally set the height of the overlay canvas
     */
    height?: number;
    /**
     * Adjust the text offset from the left side of the screen
     */
    xPos?: number;
    /**
     * Provide a text color
     */
    color?: Color;
    /**
     * Optionally set the CSS zindex of the overlay canvas
     */
    zIndex?: number;
}
/**
 * On-screen (canvas) appender
 */
export declare class ScreenAppender implements Appender {
    private _messages;
    canvas: HTMLCanvasElement;
    private _ctx;
    private _pos;
    private _color;
    private _options;
    constructor(options: ScreenAppenderOptions);
    private _positionScreenAppenderCanvas;
    /**
     * Logs a message at the given {@apilink LogLevel}
     * @param level  Level to log at
     * @param args   Arguments to log
     */
    log(level: LogLevel, args: any[]): void;
}
