#!/usr/bin/env node
"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const fs_1 = require("fs");
const minimist = require("minimist");
const rdf_string_1 = require("rdf-string");
const index_1 = require("../index");
// tslint:disable-next-line:no-var-requires
const n3 = require('n3');
process.argv.splice(0, 2);
const args = minimist(process.argv);
if (args._.length === 0 || args._.length > 2 || args.h || args.help) {
    // Print command usage
    process.stderr.write(`fetch-sparql-endpoint Sends a query to a SPARQL endpoint

Usage:
  fetch-sparql-endpoint https://dbpedia.org/sparql [-q] 'SELECT * WHERE { ?s ?p ?o } 100'
  fetch-sparql-endpoint https://dbpedia.org/sparql -f query.sparql
  cat query.sparql | fetch-sparql-endpoint https://dbpedia.org/sparql

Options:
  -q            evaluate the given SPARQL query string
  -f            evaluate the SPARQL query in the given file
  --help        print this help message
`);
    process.exit(1);
}
function getQuery() {
    return __awaiter(this, void 0, void 0, function* () {
        if (args._.length > 1) {
            return args._[1];
        }
        else if (args.q) {
            return args.q;
        }
        else if (args.f) {
            return fs_1.readFileSync(args.f, { encoding: 'utf8' });
        }
        else {
            // tslint:disable-next-line:no-var-requires
            return yield require('stream-to-string')(process.stdin);
        }
    });
}
const endpoint = args._[0];
getQuery().then((query) => {
    const fetcher = new index_1.SparqlEndpointFetcher();
    const queryType = fetcher.getQueryType(query);
    switch (queryType) {
        case 'SELECT':
            querySelect(fetcher, query);
            break;
        case 'ASK':
            queryAsk(fetcher, query);
            break;
        case 'CONSTRUCT':
            queryConstruct(fetcher, query);
            break;
    }
});
function querySelect(fetcher, query) {
    fetcher.fetchBindings(endpoint, query)
        .then((bindingsStream) => {
        bindingsStream.on('data', (bindings) => {
            for (const variable of Object.keys(bindings)) {
                bindings[variable] = rdf_string_1.termToString(bindings[variable]);
            }
            process.stdout.write(JSON.stringify(bindings) + '\n');
        });
    })
        .catch((error) => {
        process.stderr.write(error.message + '\n');
        process.exit(1);
    });
}
function queryAsk(fetcher, query) {
    fetcher.fetchAsk(endpoint, query)
        .then((answer) => {
        process.stdout.write(answer + '\n');
    })
        .catch((error) => {
        process.stderr.write(error.message + '\n');
        process.exit(1);
    });
}
function queryConstruct(fetcher, query) {
    fetcher.fetchTriples(endpoint, query)
        .then((tripleStream) => {
        tripleStream
            .pipe(new n3.StreamWriter(index_1.SparqlEndpointFetcher.CONTENTTYPE_TURTLE))
            .pipe(process.stdout);
    })
        .catch((error) => {
        process.stderr.write(error.message + '\n');
        process.exit(1);
    });
}
//# sourceMappingURL=fetch-sparql-endpoint.js.map