/// <reference types="node" />
import "isomorphic-fetch";
import * as RDF from "rdf-js";
import { ISettings, SparqlJsonParser } from "sparqljson-parse";
import { SparqlXmlParser } from "sparqlxml-parse";
/**
 * A SparqlEndpointFetcher can send queries to SPARQL endpoints,
 * and retrieve and parse the results.
 */
export declare class SparqlEndpointFetcher {
    static CONTENTTYPE_SPARQL_JSON: string;
    static CONTENTTYPE_SPARQL_XML: string;
    static CONTENTTYPE_SPARQL: string;
    static CONTENTTYPE_TURTLE: string;
    readonly fetchCb: (input?: Request | string, init?: RequestInit) => Promise<Response>;
    readonly sparqlParsers: {
        [contentType: string]: ISparqlResultsParser;
    };
    readonly sparqlJsonParser: SparqlJsonParser;
    readonly sparqlXmlParser: SparqlXmlParser;
    constructor(args?: ISparqlEndpointFetcherArgs);
    /**
     * Get the query type of the given query.
     *
     * This will parse the query and thrown an exception on syntax errors.
     *
     * @param {string} query A query.
     * @return {"SELECT" | "ASK" | "CONSTRUCT" | "UNKNOWN"} The query type.
     */
    getQueryType(query: string): "SELECT" | "ASK" | "CONSTRUCT" | "UNKNOWN";
    /**
     * Send a SELECT query to the given endpoint URL and return the resulting bindings stream.
     * @see IBindings
     * @param {string} endpoint A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query    A SPARQL query string.
     * @return {Promise<NodeJS.ReadableStream>} A stream of {@link IBindings}.
     */
    fetchBindings(endpoint: string, query: string): Promise<NodeJS.ReadableStream>;
    /**
     * Send an ASK query to the given endpoint URL and return a promise resolving to the boolean answer.
     * @param {string} endpoint A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query    A SPARQL query string.
     * @return {Promise<boolean>} A boolean resolving to the answer.
     */
    fetchAsk(endpoint: string, query: string): Promise<boolean>;
    /**
     * Send a CONSTRUCT/DESCRIBE query to the given endpoint URL and return the resulting triple stream.
     * @param {string} endpoint A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query    A SPARQL query string.
     * @return {Promise<Stream>} A stream of triples.
     */
    fetchTriples(endpoint: string, query: string): Promise<RDF.Stream>;
    /**
     * Send a query to the given endpoint URL and return the resulting stream.
     *
     * This will only accept responses with the application/sparql-results+json content type.
     *
     * @param {string} endpoint     A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query        A SPARQL query string.
     * @param {string} acceptHeader The HTTP accept to use.
     * @return {Promise<[string, NodeJS.ReadableStream]>} The content type and SPARQL endpoint response stream.
     */
    fetchRawStream(endpoint: string, query: string, acceptHeader: string): Promise<[string, NodeJS.ReadableStream]>;
}
export interface ISparqlEndpointFetcherArgs extends ISettings {
    fetch?: (input?: Request | string, init?: RequestInit) => Promise<Response>;
}
export interface IBindings {
    [key: string]: RDF.Term;
}
export interface ISparqlResultsParser {
    parseResultsStream(sparqlResponseStream: NodeJS.ReadableStream): NodeJS.ReadableStream;
    parseBooleanStream(sparqlResponseStream: NodeJS.ReadableStream): Promise<boolean>;
}
