"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
require("isomorphic-fetch");
const sparqljs_1 = require("sparqljs");
const sparqljson_parse_1 = require("sparqljson-parse");
const sparqlxml_parse_1 = require("sparqlxml-parse");
// tslint:disable-next-line:no-var-requires
const n3 = require('n3');
/**
 * A SparqlEndpointFetcher can send queries to SPARQL endpoints,
 * and retrieve and parse the results.
 */
class SparqlEndpointFetcher {
    constructor(args) {
        args = args || {};
        this.fetchCb = args.fetch || fetch;
        this.sparqlJsonParser = new sparqljson_parse_1.SparqlJsonParser(args);
        this.sparqlXmlParser = new sparqlxml_parse_1.SparqlXmlParser(args);
        this.sparqlParsers = {
            [SparqlEndpointFetcher.CONTENTTYPE_SPARQL_JSON]: {
                parseBooleanStream: (sparqlResponseStream) => this.sparqlJsonParser.parseJsonBooleanStream(sparqlResponseStream),
                parseResultsStream: (sparqlResponseStream) => this.sparqlJsonParser.parseJsonResultsStream(sparqlResponseStream),
            },
            [SparqlEndpointFetcher.CONTENTTYPE_SPARQL_XML]: {
                parseBooleanStream: (sparqlResponseStream) => this.sparqlXmlParser.parseXmlBooleanStream(sparqlResponseStream),
                parseResultsStream: (sparqlResponseStream) => this.sparqlXmlParser.parseXmlResultsStream(sparqlResponseStream),
            },
        };
    }
    /**
     * Get the query type of the given query.
     *
     * This will parse the query and thrown an exception on syntax errors.
     *
     * @param {string} query A query.
     * @return {"SELECT" | "ASK" | "CONSTRUCT" | "UNKNOWN"} The query type.
     */
    getQueryType(query) {
        const parsedQuery = new sparqljs_1.Parser().parse(query);
        return parsedQuery.type === 'query'
            ? (parsedQuery.queryType === 'DESCRIBE' ? 'CONSTRUCT' : parsedQuery.queryType) : "UNKNOWN";
    }
    /**
     * Send a SELECT query to the given endpoint URL and return the resulting bindings stream.
     * @see IBindings
     * @param {string} endpoint A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query    A SPARQL query string.
     * @return {Promise<NodeJS.ReadableStream>} A stream of {@link IBindings}.
     */
    fetchBindings(endpoint, query) {
        return __awaiter(this, void 0, void 0, function* () {
            const [contentType, responseStream] = yield this
                .fetchRawStream(endpoint, query, SparqlEndpointFetcher.CONTENTTYPE_SPARQL);
            const parser = this.sparqlParsers[contentType];
            if (!parser) {
                throw new Error('Unknown SPARQL results content type: ' + contentType);
            }
            return parser.parseResultsStream(responseStream);
        });
    }
    /**
     * Send an ASK query to the given endpoint URL and return a promise resolving to the boolean answer.
     * @param {string} endpoint A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query    A SPARQL query string.
     * @return {Promise<boolean>} A boolean resolving to the answer.
     */
    fetchAsk(endpoint, query) {
        return __awaiter(this, void 0, void 0, function* () {
            const [contentType, responseStream] = yield this
                .fetchRawStream(endpoint, query, SparqlEndpointFetcher.CONTENTTYPE_SPARQL);
            const parser = this.sparqlParsers[contentType];
            if (!parser) {
                throw new Error('Unknown SPARQL results content type: ' + contentType);
            }
            return parser.parseBooleanStream(responseStream);
        });
    }
    /**
     * Send a CONSTRUCT/DESCRIBE query to the given endpoint URL and return the resulting triple stream.
     * @param {string} endpoint A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query    A SPARQL query string.
     * @return {Promise<Stream>} A stream of triples.
     */
    fetchTriples(endpoint, query) {
        return __awaiter(this, void 0, void 0, function* () {
            const rawStream = (yield this.fetchRawStream(endpoint, query, SparqlEndpointFetcher.CONTENTTYPE_TURTLE))[1];
            return rawStream.pipe(new n3.StreamParser({ format: SparqlEndpointFetcher.CONTENTTYPE_TURTLE }));
        });
    }
    /**
     * Send a query to the given endpoint URL and return the resulting stream.
     *
     * This will only accept responses with the application/sparql-results+json content type.
     *
     * @param {string} endpoint     A SPARQL endpoint URL. (without the `?query=` suffix).
     * @param {string} query        A SPARQL query string.
     * @param {string} acceptHeader The HTTP accept to use.
     * @return {Promise<[string, NodeJS.ReadableStream]>} The content type and SPARQL endpoint response stream.
     */
    fetchRawStream(endpoint, query, acceptHeader) {
        return __awaiter(this, void 0, void 0, function* () {
            const url = endpoint + '?query=' + encodeURIComponent(query);
            // Initiate request
            const headers = new Headers();
            headers.append('Accept', acceptHeader);
            const httpResponse = yield this.fetchCb(url, { headers });
            // Wrap WhatWG readable stream into a Node.js readable stream
            // If the body already is a Node.js stream (in the case of node-fetch), don't do explicit conversion.
            const responseStream = require('is-stream')(httpResponse.body)
                ? httpResponse.body : require('node-web-streams').toNodeReadable(httpResponse.body);
            // Determine the content type and emit it to the stream
            let contentType = httpResponse.headers.get('Content-Type') || '';
            if (contentType.indexOf(';') > 0) {
                contentType = contentType.substr(0, contentType.indexOf(';'));
            }
            // Emit an error if the server returned an invalid response
            if (!httpResponse.ok) {
                setImmediate(() => responseStream.emit('error', new Error('Invalid SPARQL endpoint (' + endpoint + ') response: ' + httpResponse.statusText)));
            }
            return [contentType, responseStream];
        });
    }
}
SparqlEndpointFetcher.CONTENTTYPE_SPARQL_JSON = 'application/sparql-results+json';
SparqlEndpointFetcher.CONTENTTYPE_SPARQL_XML = 'application/sparql-results+xml';
SparqlEndpointFetcher.CONTENTTYPE_SPARQL = `${SparqlEndpointFetcher.CONTENTTYPE_SPARQL_JSON};q=1.0,${SparqlEndpointFetcher.CONTENTTYPE_SPARQL_XML};q=0.7`;
SparqlEndpointFetcher.CONTENTTYPE_TURTLE = 'text/turtle';
exports.SparqlEndpointFetcher = SparqlEndpointFetcher;
//# sourceMappingURL=SparqlEndpointFetcher.js.map