/* eslint-env mocha */
"use strict";

const fs = require("fs");
const assert = require("proclaim");
const yaml = require("yamlparser");
const uaParser = require("../lib/ua_parser");

function msg(name, actual, expected) {
	return "Expected " + name + " to be " + JSON.stringify(expected) + " got " + JSON.stringify(actual) + " instead.";
}

describe("useragent-parser should pass tests from the ua-parser/uap-core project", function() {
	this.timeout(30000);
	it("parses Mozilla/5.0 (iPad; U; CPU OS 3_2 like Mac OS X; en-us) AppleWebKit/531.21.10 (KHTML, like Gecko) Version/9.0.4 Mobile/7B367 Safari/531.21.10 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU OS 3_2 like Mac OS X; en-us) AppleWebKit/531.21.10 (KHTML, like Gecko) Version/9.0.4 Mobile/7B367 Safari/531.21.10");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 10_2_1 like Mac OS X) AppleWebKit/602.4.6 (KHTML, like Gecko) Snapchat/10.44.1.1 (iPhone7,2; iOS 10.2.1; gzip) correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 10_2_1 like Mac OS X) AppleWebKit/602.4.6 (KHTML, like Gecko) Snapchat/10.44.1.1 (iPhone7,2; iOS 10.2.1; gzip)");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU OS 3_2 like Mac OS X; en-us) AppleWebKit/531.21.10 (KHTML, like Gecko) Version/4.0.4 Mobile/7B367 Safari/531.21.10 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU OS 3_2 like Mac OS X; en-us) AppleWebKit/531.21.10 (KHTML, like Gecko) Version/4.0.4 Mobile/7B367 Safari/531.21.10");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPod; U; CPU iPhone OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2 Mobile/8H7 Safari/6533.18.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPod; U; CPU iPhone OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2 Mobile/8H7 Safari/6533.18.5");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 7_0_3 like Mac OS X) AppleWebKit/537.51.1 (KHTML, like Gecko) Coast/3.1.0.79792 Mobile/11B511 Safari/7534.48.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 7_0_3 like Mac OS X) AppleWebKit/537.51.1 (KHTML, like Gecko) Coast/3.1.0.79792 Mobile/11B511 Safari/7534.48.3");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 7_1_1 like Mac OS X) AppleWebKit/537.51.2 (KHTML, like Gecko) OPiOS/8.0.1.80062 Mobile/11D201 Safari/9537.53 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 7_1_1 like Mac OS X) AppleWebKit/537.51.2 (KHTML, like Gecko) OPiOS/8.0.1.80062 Mobile/11D201 Safari/9537.53");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_6 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Whale/0.9.1.679 Mobile/15D100 Safari/604.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_6 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Whale/0.9.1.679 Mobile/15D100 Safari/604.5.6");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPod; U; CPU iPhone OS 4_3_5 like Mac OS X; en-US) AppleWebKit/534.12 (KHTML, like Gecko) Puffin/1.3.3102MS Mobile Safari/534.12 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPod; U; CPU iPhone OS 4_3_5 like Mac OS X; en-US) AppleWebKit/534.12 (KHTML, like Gecko) Puffin/1.3.3102MS Mobile Safari/534.12");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 8_3 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) FxiOS/1.0 Mobile/12F69 Safari/600.1.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 8_3 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) FxiOS/1.0 Mobile/12F69 Safari/600.1.4");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; CPU iPhone OS 8_3 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) FxiOS/1.0 Mobile/12F69 Safari/600.1.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; CPU iPhone OS 8_3 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) FxiOS/1.0 Mobile/12F69 Safari/600.1.4");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 8_2 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) Mobile/12D508 [FBAN/GroupsForiOS;FBAV/9.0;FBBV/7752968;FBDV/iPhone7,2;FBMD/iPhone;FBSN/iPhone OS;FBSV/8.2;FBSS/2; FBCR/Telekom.de;FBID/phone;FBLC/de_ correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 8_2 like Mac OS X) AppleWebKit/600.1.4 (KHTML, like Gecko) Mobile/12D508 [FBAN/GroupsForiOS;FBAV/9.0;FBBV/7752968;FBDV/iPhone7,2;FBMD/iPhone;FBSN/iPhone OS;FBSV/8.2;FBSS/2; FBCR/Telekom.de;FBID/phone;FBLC/de_");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_4_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15G77 [FBAN/FBIOS;FBDV/iPhone10,4;FBMD/iPhone;FBSN/iOS;FBSV/11.4.1;FBSS/2;FBCR/A1;FBID/phone;FBLC/de_DE;FBOP/5;FBRV/122166081] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_4_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15G77 [FBAN/FBIOS;FBDV/iPhone10,4;FBMD/iPhone;FBSN/iOS;FBSV/11.4.1;FBSS/2;FBCR/A1;FBID/phone;FBLC/de_DE;FBOP/5;FBRV/122166081]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 10_2_1 like Mac OS X) AppleWebKit/602.4.6 (KHTML, like Gecko) Mobile/14D27 [FBAN/MessengerForiOS;FBAV/124.0.0.50.70;FBBV/63293619;FBDV/iPhone7,1;FBMD/iPhone;FBSN/iOS;FBSV/10.2.1;FBSS/3;FBCR/Viettel;FBID/phone;FBLC/vi_VN;FBOP/5;FBRV/0] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 10_2_1 like Mac OS X) AppleWebKit/602.4.6 (KHTML, like Gecko) Mobile/14D27 [FBAN/MessengerForiOS;FBAV/124.0.0.50.70;FBBV/63293619;FBDV/iPhone7,1;FBMD/iPhone;FBSN/iOS;FBSV/10.2.1;FBSS/3;FBCR/Viettel;FBID/phone;FBLC/vi_VN;FBOP/5;FBRV/0]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_4_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15G77 [FBAN/FBIOS;FBAV/194.0.0.38.99;FBBV/127868476;FBDV/iPhone7,2;FBMD/iPhone;FBSN/iOS;FBSV/11.4.1;FBSS/2;FBCR/OrangeBotswana;FBID/phone;FBLC/en_GB;FBOP/5;FBRV/128807018] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_4_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15G77 [FBAN/FBIOS;FBAV/194.0.0.38.99;FBBV/127868476;FBDV/iPhone7,2;FBMD/iPhone;FBSN/iOS;FBSV/11.4.1;FBSS/2;FBCR/OrangeBotswana;FBID/phone;FBLC/en_GB;FBOP/5;FBRV/128807018]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2 Mobile/8H7 Safari/WKWebView correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2 Mobile/8H7 Safari/WKWebView");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_0 like Mac OS X) AppleWebKit/604.1.38 (KHTML, like Gecko) Mobile/15A372 Safari Line/7.12.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_0 like Mac OS X) AppleWebKit/604.1.38 (KHTML, like Gecko) Mobile/15A372 Safari Line/7.12.0");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU OS 5_1_1 like Mac OS X; en-us) AppleWebKit/534.46.0 (KHTML, like Gecko) Chrome/19.0.1084.60 Mobile/9B206 Safari/7534.48.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU OS 5_1_1 like Mac OS X; en-us) AppleWebKit/534.46.0 (KHTML, like Gecko) Chrome/19.0.1084.60 Mobile/9B206 Safari/7534.48.3");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 10_2_1 like Mac OS X) AppleWebKit/602.4.6 (KHTML, like Gecko) Mobile/14D27 SznProhlizec/4.4i correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 10_2_1 like Mac OS X) AppleWebKit/602.4.6 (KHTML, like Gecko) Mobile/14D27 SznProhlizec/4.4i");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_5 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Mobile/15D60 Instagram 33.0.0.11.96 (iPhone9,3; iOS 11_2_5; en_AU; en-AU; scale=2.00; gamut=wide; 750x1334) correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_5 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Mobile/15D60 Instagram 33.0.0.11.96 (iPhone9,3; iOS 11_2_5; en_AU; en-AU; scale=2.00; gamut=wide; 750x1334)");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_6 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Mobile/15D100 Flipboard/4.2.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_6 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Mobile/15D100 Flipboard/4.2.2");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_6 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Snapchat/10.38.0.25 (iPhone8,1; iOS 11.2.6; gzip) correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_6 like Mac OS X) AppleWebKit/604.5.6 (KHTML, like Gecko) Snapchat/10.38.0.25 (iPhone8,1; iOS 11.2.6; gzip)");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_0_2 like Mac OS X) AppleWebKit/604.1.34 (KHTML, like Gecko) GSA/36.0.169645775 Mobile/15A421 Safari/604.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_0_2 like Mac OS X) AppleWebKit/604.1.34 (KHTML, like Gecko) GSA/36.0.169645775 Mobile/15A421 Safari/604.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 6_0_1 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10A525 [Pinterest/iOS] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 6_0_1 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10A525 [Pinterest/iOS]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 6_1_3 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10B329 [FBAN/FBIOS;FBAV/6.5.1;FBBV/377040;FBDV/iPhone4,1;FBMD/iPhone;FBSN/iPhone OS;FBSV/6.1.3;FBSS/2; FBCR/Telekom.de;FBID/phone;FBLC/de_DE; correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 6_1_3 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10B329 [FBAN/FBIOS;FBAV/6.5.1;FBBV/377040;FBDV/iPhone4,1;FBMD/iPhone;FBSN/iPhone OS;FBSV/6.1.3;FBSS/2; FBCR/Telekom.de;FBID/phone;FBLC/de_DE;");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 6_1_3 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10B329 [FBAN/FBIOS;FBAV/6.2;FBBV/228172;FBDV/iPhone5,1;FBMD/iPhone;FBSN/iPhone OS;FBSV/6.1.3;FBSS/2; FBCR/o2-de;FBID/phone;FBLC/pt_BR;FBOP/1] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 6_1_3 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10B329 [FBAN/FBIOS;FBAV/6.2;FBBV/228172;FBDV/iPhone5,1;FBMD/iPhone;FBSN/iPhone OS;FBSV/6.1.3;FBSS/2; FBCR/o2-de;FBID/phone;FBLC/pt_BR;FBOP/1]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; CPU OS 5_0 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Mobile/9A334 [FBAN/FBIOS;FBAV/6.5.1;FBBV/377040;FBDV/iPad2,1;FBMD/iPad;FBSN/iPhone OS;FBSV/5.0;FBSS/1; FBCR/;FBID/tablet;FBLC/de_DE;FBOP/1] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; CPU OS 5_0 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Mobile/9A334 [FBAN/FBIOS;FBAV/6.5.1;FBBV/377040;FBDV/iPad2,1;FBMD/iPad;FBSN/iPhone OS;FBSV/5.0;FBSS/1; FBCR/;FBID/tablet;FBLC/de_DE;FBOP/1]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 5_1_1 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Mobile/9B206 [FBAN/FBIOS;FBAV/6.1;FBBV/201075;FBDV/iPhone3,1;FBMD/iPhone;FBSN/iPhone OS;FBSV/5.1.1;FBSS/2; FBCR/Vodafone.de;FBID/phone;FBLC/en_US;FBOP/1] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 5_1_1 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Mobile/9B206 [FBAN/FBIOS;FBAV/6.1;FBBV/201075;FBDV/iPhone3,1;FBMD/iPhone;FBSN/iPhone OS;FBSV/5.1.1;FBSS/2; FBCR/Vodafone.de;FBID/phone;FBLC/en_US;FBOP/1]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 6_1_4 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10B350 [Pinterest/iOS] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 6_1_4 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10B350 [Pinterest/iOS]");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone 5; CPU iPhone OS 7_0_6 like Mac OS X) AppleWebKit/537.51.1 (KHTML, like Gecko) Version/6.0 MQQBrowser/5.0.5 Mobile/11B651 Safari/8536.25 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone 5; CPU iPhone OS 7_0_6 like Mac OS X) AppleWebKit/537.51.1 (KHTML, like Gecko) Version/6.0 MQQBrowser/5.0.5 Mobile/11B651 Safari/8536.25");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MQQBrowser/371 Mozilla/5.0 (iPhone 4S; CPU iPhone OS 6_0_1 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10A523 Safari/7534.48.3 correctly", function() {
		const results = uaParser("MQQBrowser/371 Mozilla/5.0 (iPhone 4S; CPU iPhone OS 6_0_1 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10A523 Safari/7534.48.3");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/9 (Compatible; MSIE:9.0; iPhone; BlackBerry9700; AppleWebKit/24.746; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/9 (Compatible; MSIE:9.0; iPhone; BlackBerry9700; AppleWebKit/24.746; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MQQBrowser/39 Mozilla/5.0 (iPhone 4S; CPU iPhone OS 6_0_1 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10A523 Safari/7534.48.3 correctly", function() {
		const results = uaParser("MQQBrowser/39 Mozilla/5.0 (iPhone 4S; CPU iPhone OS 6_0_1 like Mac OS X) AppleWebKit/536.26 (KHTML, like Gecko) Mobile/10A523 Safari/7534.48.3");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MQQBrowser/391 Mozilla/5.0 (iPhone 4S; CPU iPhone OS 5_0 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Mobile/9A334 Safari/7534.48.3 correctly", function() {
		const results = uaParser("MQQBrowser/391 Mozilla/5.0 (iPhone 4S; CPU iPhone OS 5_0 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Mobile/9A334 Safari/7534.48.3");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Mobile");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU like Mac OS X; FIT_LANG_REPLACE) AppleWebKit/532+ (KHTML, like Gecko) Version/3.0 Mobile/1A538b Safari/419.3 Midori/0.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU like Mac OS X; FIT_LANG_REPLACE) AppleWebKit/532+ (KHTML, like Gecko) Version/3.0 Mobile/1A538b Safari/419.3 Midori/0.4");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 9_3_4 like Mac OS X) AppleWebKit/601.1 (KHTML, like Gecko) Outlook-iOS-Android/1.0 Mobile/13G35 Safari/601.1.46\") correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 9_3_4 like Mac OS X) AppleWebKit/601.1 (KHTML, like Gecko) Outlook-iOS-Android/1.0 Mobile/13G35 Safari/601.1.46\")");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_5 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Version/11.2.5 Mobile/9B179 Safari/7534.48.3 OktaMobile/5.10.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_2_5 like Mac OS X) AppleWebKit/534.46 (KHTML, like Gecko) Version/11.2.5 Mobile/9B179 Safari/7534.48.3 OktaMobile/5.10.2");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (iPhone; Opera Mini/7.1.32694/27.1407; U; en) Presto/2.8.119 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (iPhone; Opera Mini/7.1.32694/27.1407; U; en) Presto/2.8.119 Version/11.10");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "32694" || "", msg("results.patch", results.patch || "", "32694" || ""));
	});
	it("parses Opera/9.80 (iPad; Opera Mini/7.1.32694/27.1407; U; en) Presto/2.8.119 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (iPad; Opera Mini/7.1.32694/27.1407; U; en) Presto/2.8.119 Version/11.10");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "32694" || "", msg("results.patch", results.patch || "", "32694" || ""));
	});
	it("parses Opera/9.80 (iPhone; Opera Mini/7.0.4/28.2555; U; fr) Presto/2.8.119 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (iPhone; Opera Mini/7.0.4/28.2555; U; fr) Presto/2.8.119 Version/11.10");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Opera/9.80 (J2ME/iPhone;Opera Mini/5.0.019802/886; U; ja)Presto/2.4.15 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/iPhone;Opera Mini/5.0.019802/886; U; ja)Presto/2.4.15");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "019802" || "", msg("results.patch", results.patch || "", "019802" || ""));
	});
	it("parses Opera/9.80 (J2ME/iPhone;Opera Mini/5.0.019802/886; U; ja)Presto/ 2.4.15 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/iPhone;Opera Mini/5.0.019802/886; U; ja)Presto/ 2.4.15");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "019802" || "", msg("results.patch", results.patch || "", "019802" || ""));
	});
	it("parses Opera/9.80 (J2ME/iPhone;Opera Mini/5.0.019802/886; U; ja) Presto/2.4.15 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/iPhone;Opera Mini/5.0.019802/886; U; ja) Presto/2.4.15");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "019802" || "", msg("results.patch", results.patch || "", "019802" || ""));
	});
	it("parses Opera/9.80 (iPhone; Opera Mini/5.0.019802/886; U; ja) Presto/2.4.15 correctly", function() {
		const results = uaParser("Opera/9.80 (iPhone; Opera Mini/5.0.019802/886; U; ja) Presto/2.4.15");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "019802" || "", msg("results.patch", results.patch || "", "019802" || ""));
	});
	it("parses Opera/9.80 (iPhone; Opera Mini/5.0.019802/886; U; en) Presto/2.4.15 correctly", function() {
		const results = uaParser("Opera/9.80 (iPhone; Opera Mini/5.0.019802/886; U; en) Presto/2.4.15");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "019802" || "", msg("results.patch", results.patch || "", "019802" || ""));
	});
	it("parses Opera/9.80 (iPhone; Opera Mini/5.0.019802/22.414; U; de) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (iPhone; Opera Mini/5.0.019802/22.414; U; de) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "019802" || "", msg("results.patch", results.patch || "", "019802" || ""));
	});
	it("parses Opera/9.80 (iPhone; Opera Mini/5.0.019802/18.738; U; en) Presto/2.4.15 correctly", function() {
		const results = uaParser("Opera/9.80 (iPhone; Opera Mini/5.0.019802/18.738; U; en) Presto/2.4.15");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "019802" || "", msg("results.patch", results.patch || "", "019802" || ""));
	});
	it("parses Opera/9.80 (iPhone; Opera Mini/5.0.0176/764; U; en) Presto/2.4.154.15 correctly", function() {
		const results = uaParser("Opera/9.80 (iPhone; Opera Mini/5.0.0176/764; U; en) Presto/2.4.154.15");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0176" || "", msg("results.patch", results.patch || "", "0176" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; xxxx like Mac OS X; en) AppleWebKit/24.838; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; xxxx like Mac OS X; en) AppleWebKit/24.838; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; fr; CPU iPhone OS 4_2_1 like Mac OS X; fr) AppleWebKit/23.405; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; fr; CPU iPhone OS 4_2_1 like Mac OS X; fr) AppleWebKit/23.405; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; CPU iPhone OS 3_0 like Mac OS X; en-us) AppleWebKit/23.411; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; CPU iPhone OS 3_0 like Mac OS X; en-us) AppleWebKit/23.411; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; CPU iPhone OS 3_0 like Mac OS X; en-us) AppleWebKit/23.377; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/5.0 (iPhone; U; CPU iPhone OS 3_0 like Mac OS X; en-us) AppleWebKit/23.377; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/4.3.24214; iPhone; CPU iPhone OS 4_2_1 like Mac OS X; AppleWebKit/24.783; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/4.3.24214; iPhone; CPU iPhone OS 4_2_1 like Mac OS X; AppleWebKit/24.783; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "24214" || "", msg("results.patch", results.patch || "", "24214" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/4.0 (iPad; U; CPU OS 3_2 like Mac OS X; en-us) AppleWebKit/23.411; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/4.0 (iPad; U; CPU OS 3_2 like Mac OS X; en-us) AppleWebKit/23.411; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU OS 7_0 like Mac OS X) AppleWebKit/537.51.1 (KHTML, like Gecko) Version/9.3 Mobile/11A465 Safari/9537.53 VLC for iOS/2.7.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU OS 7_0 like Mac OS X) AppleWebKit/537.51.1 (KHTML, like Gecko) Version/9.3 Mobile/11A465 Safari/9537.53 VLC for iOS/2.7.2");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0+(iPad;+CPU+OS+9_3_1+like+Mac+OS+X)+AppleWebKit/601.1.46+(KHTML,+like+Gecko)+Version/9.0+Mobile/13E238+Safari/601.1 correctly", function() {
		const results = uaParser("Mozilla/5.0+(iPad;+CPU+OS+9_3_1+like+Mac+OS+X)+AppleWebKit/601.1.46+(KHTML,+like+Gecko)+Version/9.0+Mobile/13E238+Safari/601.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0+(iPhone;+CPU+iPhone+OS+9_3_1+like+Mac+OS+X)+AppleWebKit/601.1.46+(KHTML,+like+Gecko)+Version/9.0+Mobile/13E238+Safari/601.1 correctly", function() {
		const results = uaParser("Mozilla/5.0+(iPhone;+CPU+iPhone+OS+9_3_1+like+Mac+OS+X)+AppleWebKit/601.1.46+(KHTML,+like+Gecko)+Version/9.0+Mobile/13E238+Safari/601.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2 Mobile/8H7 Safari correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU OS 4_3_2 like Mac OS X; en-us) AppleWebKit/533.17.9 (KHTML, like Gecko) Version/5.0.2 Mobile/8H7 Safari");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 10_0_2 like Mac OS X) AppleWebKit/602.1.50 (KHTML, like Gecko) AppleNews/608.0.1 Version/2.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 10_0_2 like Mac OS X) AppleWebKit/602.1.50 (KHTML, like Gecko) AppleNews/608.0.1 Version/2.0.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 7_1_2 like Mac OS X) AppleWebKit/537.51.2 (KHTML, like Gecko) Mobile/11D257 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 7_1_2 like Mac OS X) AppleWebKit/537.51.2 (KHTML, like Gecko) Mobile/11D257");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPod touch; CPU iPhone OS 9_3_2 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Mobile/13F69 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPod touch; CPU iPhone OS 9_3_2 like Mac OS X) AppleWebKit/601.1.46 (KHTML, like Gecko) Mobile/13F69");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 10_3_3 like Mac OS X) AppleWebKit/603.3.8 (KHTML, like Gecko) Version/10.0 Mobile/14G60 Safari/602.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 10_3_3 like Mac OS X) AppleWebKit/603.3.8 (KHTML, like Gecko) Version/10.0 Mobile/14G60 Safari/602.1");
		assert.strictEqual(results.family, "Mobile Safari", msg("results.family", results.family, "Mobile Safari"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 11_3 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/11.0 Mobile/15E148 Safari/604.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 11_3 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/11.0 Mobile/15E148 Safari/604.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 12_3_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/12.1.1 Mobile/15E148 Safari/604.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 12_3_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/12.1.1 Mobile/15E148 Safari/604.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "12", msg("results.major", results.major || "", "12"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 13_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.3 Mobile/15E148 Safari/604.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 13_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.3 Mobile/15E148 Safari/604.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "13", msg("results.major", results.major || "", "13"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 13_3 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.4 Mobile/15E148 Safari/604.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 13_3 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/13.0.4 Mobile/15E148 Safari/604.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "13", msg("results.major", results.major || "", "13"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 14_0_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/14.0 Mobile/15E148 Safari/604.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 14_0_1 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Version/14.0 Mobile/15E148 Safari/604.1");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "14", msg("results.major", results.major || "", "14"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; CPU iPhone OS 15_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; CPU iPhone OS 15_2 like Mac OS X) AppleWebKit/605.1.15 (KHTML, like Gecko) Mobile/15E148");
		assert.strictEqual(results.family, "Mobile Safari/WKWebView", msg("results.family", results.family, "Mobile Safari/WKWebView"));
		assert.strictEqual(results.major || "", "15", msg("results.major", results.major || "", "15"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; en; rv:1.8.1.6) Gecko/20070809 Camino/1.5.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; en; rv:1.8.1.6) Gecko/20070809 Camino/1.5.1");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X; en; rv:1.8.1.6) Gecko/20070809 Camino/1.5.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X; en; rv:1.8.1.6) Gecko/20070809 Camino/1.5.1");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X; en; rv:1.8.1.11) Gecko/20071128 Camino/1.5.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X; en; rv:1.8.1.11) Gecko/20071128 Camino/1.5.4");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; en; rv:1.8.1.12) Gecko/20080206 Camino/1.5.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; en; rv:1.8.1.12) Gecko/20080206 Camino/1.5.5");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X Mach-O; en; rv:1.8.1.12) Gecko/20080206 Camino/1.5.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X Mach-O; en; rv:1.8.1.12) Gecko/20080206 Camino/1.5.5");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; it; rv:1.8.1.21) Gecko/20090327 Camino/1.6.7 (MultiLang) (like Firefox/2.0.0.21pre) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; it; rv:1.8.1.21) Gecko/20090327 Camino/1.6.7 (MultiLang) (like Firefox/2.0.0.21pre)");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; en; rv:1.8.1.21) Gecko/20090327 Camino/1.6.7 (like Firefox/2.0.0.21pre) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X Mach-O; en; rv:1.8.1.21) Gecko/20090327 Camino/1.6.7 (like Firefox/2.0.0.21pre)");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en; rv:1.9.0.18) Gecko/2010021619 Camino/2.0.2 (like Firefox/3.0.18) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en; rv:1.9.0.18) Gecko/2010021619 Camino/2.0.2 (like Firefox/3.0.18)");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X 10.4; en; rv:1.9.0.19) Gecko/2010051911 Camino/2.0.3 (like Firefox/3.0.19) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X 10.4; en; rv:1.9.0.19) Gecko/2010051911 Camino/2.0.3 (like Firefox/3.0.19)");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; nl; rv:1.9.0.19) Gecko/2010051911 Camino/2.0.3 (MultiLang) (like Firefox/3.0.19) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; nl; rv:1.9.0.19) Gecko/2010051911 Camino/2.0.3 (MultiLang) (like Firefox/3.0.19)");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.4; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.4; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; PPC Mac OS X 10.4; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; PPC Mac OS X 10.4; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; PPC Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; PPC Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 Camino/2.2.1");
		assert.strictEqual(results.family, "Camino", msg("results.family", results.family, "Camino"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.0; rv:2.0b6pre) Gecko/20100907 Firefox/4.0b6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.0; rv:2.0b6pre) Gecko/20100907 Firefox/4.0b6pre");
		assert.strictEqual(results.family, "Firefox Beta", msg("results.family", results.family, "Firefox Beta"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b6pre" || "", msg("results.patch", results.patch || "", "b6pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0b6pre) Gecko/20100903 Firefox/4.0b6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0b6pre) Gecko/20100903 Firefox/4.0b6pre");
		assert.strictEqual(results.family, "Firefox Beta", msg("results.family", results.family, "Firefox Beta"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b6pre" || "", msg("results.patch", results.patch || "", "b6pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; rv:2.0b6pre) Gecko/20100903 Firefox/4.0b6pre Firefox/4.0b6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; rv:2.0b6pre) Gecko/20100903 Firefox/4.0b6pre Firefox/4.0b6pre");
		assert.strictEqual(results.family, "Firefox Beta", msg("results.family", results.family, "Firefox Beta"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b6pre" || "", msg("results.patch", results.patch || "", "b6pre" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0b6pre) Gecko/20100907 Firefox/4.0b6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0b6pre) Gecko/20100907 Firefox/4.0b6pre");
		assert.strictEqual(results.family, "Firefox Beta", msg("results.family", results.family, "Firefox Beta"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b6pre" || "", msg("results.patch", results.patch || "", "b6pre" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686; rv:2.0b6pre) Gecko/20100907 Firefox/4.0b6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686; rv:2.0b6pre) Gecko/20100907 Firefox/4.0b6pre");
		assert.strictEqual(results.family, "Firefox Beta", msg("results.family", results.family, "Firefox Beta"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b6pre" || "", msg("results.patch", results.patch || "", "b6pre" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; sv-SE; rv:1.9.1.3) Gecko/20090824 Firefox/3.5.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; sv-SE; rv:1.9.1.3) Gecko/20090824 Firefox/3.5.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.6) Gecko/20100117 Gentoo Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.6) Gecko/20100117 Gentoo Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; zh-CN; rv:1.9.1.6) Gecko/20091216 Fedora/3.5.6-1.fc11 Firefox/3.5.6 GTB6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; zh-CN; rv:1.9.1.6) Gecko/20091216 Fedora/3.5.6-1.fc11 Firefox/3.5.6 GTB6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; es-ES; rv:1.9.1.6) Gecko/20091201 SUSE/3.5.6-1.1.1 Firefox/3.5.6 GTB6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; es-ES; rv:1.9.1.6) Gecko/20091201 SUSE/3.5.6-1.1.1 Firefox/3.5.6 GTB6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.6) Gecko/20100118 Gentoo Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.6) Gecko/20100118 Gentoo Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-GB; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6 GTB6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-GB; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6 GTB6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6 GTB7.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6 GTB7.0");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.6) Gecko/20091201 SUSE/3.5.6-1.1.1 Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.6) Gecko/20091201 SUSE/3.5.6-1.1.1 Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; cs-CZ; rv:1.9.1.6) Gecko/20100107 Fedora/3.5.6-1.fc12 Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; cs-CZ; rv:1.9.1.6) Gecko/20100107 Fedora/3.5.6-1.fc12 Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; ca; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; ca; rv:1.9.1.6) Gecko/20091215 Ubuntu/9.10 (karmic) Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; it; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; it; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; id; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; id; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.1.6) Gecko/20091201 MRA 5.4 (build 02647) Firefox/3.5.6 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.1.6) Gecko/20091201 MRA 5.4 (build 02647) Firefox/3.5.6 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; nl; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; nl; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 MRA 5.5 (build 02842) Firefox/3.5.6 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 MRA 5.5 (build 02842) Firefox/3.5.6 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 MRA 5.5 (build 02842) Firefox/3.5.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 MRA 5.5 (build 02842) Firefox/3.5.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 GTB6 (.NET CLR 3.5.30729) FBSMTWB correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 GTB6 (.NET CLR 3.5.30729) FBSMTWB");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 3.5.30729) FBSMTWB correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 3.5.30729) FBSMTWB");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-US; rv:1.9.1.6) Gecko/20091201 Firefox/3.5.6 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2) Gecko/20100305 Gentoo Firefox/3.5.7 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2) Gecko/20100305 Gentoo Firefox/3.5.7");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; cs-CZ; rv:1.9.1.7) Gecko/20100106 Ubuntu/9.10 (karmic) Firefox/3.5.7 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; cs-CZ; rv:1.9.1.7) Gecko/20100106 Ubuntu/9.10 (karmic) Firefox/3.5.7");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; es-ES; rv:1.9.1.7) Gecko/20091222 SUSE/3.5.7-1.1.1 Firefox/3.5.7 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; es-ES; rv:1.9.1.7) Gecko/20091222 SUSE/3.5.7-1.1.1 Firefox/3.5.7");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; ja; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7 GTB6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; ja; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7 GTB6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; de; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7 (.NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.30729; .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; de; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7 (.NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.30729; .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; fr; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7 (.NET CLR 3.0.04506.648) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; fr; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7 (.NET CLR 3.0.04506.648)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; fa; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; fa; rv:1.9.1.7) Gecko/20091221 Firefox/3.5.7");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.7) Gecko/20091221 MRA 5.5 (build 02842) Firefox/3.5.7 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.7) Gecko/20091221 MRA 5.5 (build 02842) Firefox/3.5.7 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; ru; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc12 Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; ru; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc12 Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; es-ES; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc11 Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; es-ES; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc11 Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.8) Gecko/20100318 Gentoo Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.8) Gecko/20100318 Gentoo Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; zh-CN; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc12 Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; zh-CN; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc12 Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; ja-JP; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc12 Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; ja-JP; rv:1.9.1.8) Gecko/20100216 Fedora/3.5.8-1.fc12 Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; es-AR; rv:1.9.1.8) Gecko/20100214 Ubuntu/9.10 (karmic) Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; es-AR; rv:1.9.1.8) Gecko/20100214 Ubuntu/9.10 (karmic) Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.8) Gecko/20100214 Ubuntu/9.10 (karmic) Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.8) Gecko/20100214 Ubuntu/9.10 (karmic) Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; FreeBSD i386; ja-JP; rv:1.9.1.8) Gecko/20100305 Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; FreeBSD i386; ja-JP; rv:1.9.1.8) Gecko/20100305 Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; sl; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; sl; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 (.NET CLR 3.5.30729) FirePHP/0.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 (.NET CLR 3.5.30729) FirePHP/0.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-TW; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 GTB6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-TW; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 GTB6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; ja; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 GTB7.0 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; ja; rv:1.9.1.8) Gecko/20100202 Firefox/3.5.8 GTB7.0 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; it; rv:1.9.1.9) Gecko/20100402 Ubuntu/9.10 (karmic) Firefox/3.5.9 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; it; rv:1.9.1.9) Gecko/20100402 Ubuntu/9.10 (karmic) Firefox/3.5.9 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; it; rv:1.9.1.9) Gecko/20100330 Fedora/3.5.9-2.fc12 Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; it; rv:1.9.1.9) Gecko/20100330 Fedora/3.5.9-2.fc12 Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.1.9) Gecko/20100317 SUSE/3.5.9-0.1.1 Firefox/3.5.9 GTB7.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.1.9) Gecko/20100317 SUSE/3.5.9-0.1.1 Firefox/3.5.9 GTB7.0");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; es-CL; rv:1.9.1.9) Gecko/20100402 Ubuntu/9.10 (karmic) Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; es-CL; rv:1.9.1.9) Gecko/20100402 Ubuntu/9.10 (karmic) Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; cs-CZ; rv:1.9.1.9) Gecko/20100317 SUSE/3.5.9-0.1.1 Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; cs-CZ; rv:1.9.1.9) Gecko/20100317 SUSE/3.5.9-0.1.1 Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; nl; rv:1.9.1.9) Gecko/20100401 Ubuntu/9.10 (karmic) Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; nl; rv:1.9.1.9) Gecko/20100401 Ubuntu/9.10 (karmic) Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; hu-HU; rv:1.9.1.9) Gecko/20100330 Fedora/3.5.9-1.fc12 Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; hu-HU; rv:1.9.1.9) Gecko/20100330 Fedora/3.5.9-1.fc12 Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; es-ES; rv:1.9.1.9) Gecko/20100317 SUSE/3.5.9-0.1 Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; es-ES; rv:1.9.1.9) Gecko/20100317 SUSE/3.5.9-0.1 Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.9) Gecko/20100401 Ubuntu/9.10 (karmic) Firefox/3.5.9 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.9) Gecko/20100401 Ubuntu/9.10 (karmic) Firefox/3.5.9 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.9) Gecko/20100315 Ubuntu/9.10 (karmic) Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.9) Gecko/20100315 Ubuntu/9.10 (karmic) Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.4) Gecko/20091028 Ubuntu/9.10 (karmic) Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.4) Gecko/20091028 Ubuntu/9.10 (karmic) Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; tr; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; tr; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; hu; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; hu; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; fr; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; fr; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; et; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; et; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; nl; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; nl; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; es-ES; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB5 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; es-ES; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB5 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; de; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB7.0 (.NET CLR 3.0.30618) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; de; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB7.0 (.NET CLR 3.0.30618)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; ca; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB7.0 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; ca; rv:1.9.1.9) Gecko/20100315 Firefox/3.5.9 GTB7.0 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.7; en-US; rv:1.9.2.2) Gecko/20100316 Firefox/3.6.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.7; en-US; rv:1.9.2.2) Gecko/20100316 Firefox/3.6.2");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.2.3) Gecko/20100403 Fedora/3.6.3-4.fc13 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.2.3) Gecko/20100403 Fedora/3.6.3-4.fc13 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.0 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.0 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; cs; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; cs; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; ca; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; ca; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; sv-SE; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; sv-SE; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; pl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; pl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; nl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; nl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; SunOS sun4u; en-US; rv:1.9.2.3) Gecko/20100403 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; SunOS sun4u; en-US; rv:1.9.2.3) Gecko/20100403 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.3) Gecko/20100403 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.3) Gecko/20100403 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; de; rv:1.9.2.3) Gecko/20100401 SUSE/3.6.3-1.1 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; de; rv:1.9.2.3) Gecko/20100401 SUSE/3.6.3-1.1 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; ko-KR; rv:1.9.2.3) Gecko/20100423 Ubuntu/10.04 (lucid) Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; ko-KR; rv:1.9.2.3) Gecko/20100423 Ubuntu/10.04 (lucid) Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.3) Gecko/20100404 Ubuntu/10.04 (lucid) Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.3) Gecko/20100404 Ubuntu/10.04 (lucid) Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.2.3) Gecko/20100423 Ubuntu/10.04 (lucid) Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.2.3) Gecko/20100423 Ubuntu/10.04 (lucid) Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X 10.5; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X 10.5; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.0");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-US; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-GB; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en-GB; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.5; nl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.5; nl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.5; fr; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.5; fr; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; zh-CN; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; zh-CN; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; ru; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; ru; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; pl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; pl; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; it; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; it; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; hu; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; hu; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; es-ES; rv:1.9.2.3) Gecko/20100401 Firefox/3.6.3 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.4) Gecko/20100614 Ubuntu/10.04 (lucid) Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.4) Gecko/20100614 Ubuntu/10.04 (lucid) Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.4) Gecko/20100625 Gentoo Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.4) Gecko/20100625 Gentoo Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; ru; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; ru; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; ja; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; ja; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; cs; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; cs; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; zh-CN; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; zh-CN; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; ja; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; ja; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; fr; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; fr; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100527 Firefox/3.6.4 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100527 Firefox/3.6.4 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100527 Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100527 Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-TW; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 GTB7.0 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-TW; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 GTB7.0 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-CN; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-CN; rv:1.9.2.4) Gecko/20100513 Firefox/3.6.4 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-CN; rv:1.9.2.4) Gecko/20100503 Firefox/3.6.4 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-CN; rv:1.9.2.4) Gecko/20100503 Firefox/3.6.4 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; nb-NO; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; nb-NO; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; ko; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; ko; rv:1.9.2.4) Gecko/20100523 Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; cs; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; cs; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X 10.5; en-US; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 GTB7.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X 10.5; en-US; rv:1.9.2.4) Gecko/20100611 Firefox/3.6.4 GTB7.0");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6 GTB7.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6 GTB7.0");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.6) Gecko/20100628 Ubuntu/10.04 (lucid) Firefox/3.6.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; pt-PT; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; pt-PT; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; it; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 ( .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; it; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 ( .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; zh-CN; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; zh-CN; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; nl; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; nl; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; it; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 ( .NET CLR 3.5.30729; .NET4.0E) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; it; rv:1.9.2.6) Gecko/20100625 Firefox/3.6.6 ( .NET CLR 3.5.30729; .NET4.0E)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.7) Gecko/20100723 Fedora/3.6.7-1.fc13 Firefox/3.6.7 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.7) Gecko/20100723 Fedora/3.6.7-1.fc13 Firefox/3.6.7");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.7) Gecko/20100726 CentOS/3.6-3.el5.centos Firefox/3.6.7 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.7) Gecko/20100726 CentOS/3.6-3.el5.centos Firefox/3.6.7");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; hu; rv:1.9.2.7) Gecko/20100713 Firefox/3.6.7 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; hu; rv:1.9.2.7) Gecko/20100713 Firefox/3.6.7 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; pt-PT; rv:1.9.2.7) Gecko/20100713 Firefox/3.6.7 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; pt-PT; rv:1.9.2.7) Gecko/20100713 Firefox/3.6.7 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.8) Gecko/20100804 Gentoo Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.8) Gecko/20100804 Gentoo Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.8) Gecko/20100723 SUSE/3.6.8-0.1.1 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.8) Gecko/20100723 SUSE/3.6.8-0.1.1 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; zh-CN; rv:1.9.2.8) Gecko/20100722 Ubuntu/10.04 (lucid) Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; zh-CN; rv:1.9.2.8) Gecko/20100722 Ubuntu/10.04 (lucid) Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; ru; rv:1.9.2.8) Gecko/20100723 Ubuntu/10.04 (lucid) Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; ru; rv:1.9.2.8) Gecko/20100723 Ubuntu/10.04 (lucid) Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; fi-FI; rv:1.9.2.8) Gecko/20100723 Ubuntu/10.04 (lucid) Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; fi-FI; rv:1.9.2.8) Gecko/20100723 Ubuntu/10.04 (lucid) Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.8) Gecko/20100727 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.2.8) Gecko/20100727 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de-DE; rv:1.9.2.8) Gecko/20100725 Gentoo Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de-DE; rv:1.9.2.8) Gecko/20100725 Gentoo Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; FreeBSD i386; de-CH; rv:1.9.2.8) Gecko/20100729 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; FreeBSD i386; de-CH; rv:1.9.2.8) Gecko/20100729 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; zh-CN; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; zh-CN; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; pt-BR; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; pt-BR; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; he; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; he; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; fr; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 GTB7.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; fr; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 GTB7.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-GB; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 ( .NET CLR 3.5.30729; .NET4.0C) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-GB; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 ( .NET CLR 3.5.30729; .NET4.0C)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; de; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; de; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; de; rv:1.9.2.3) Gecko/20121221 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; de; rv:1.9.2.3) Gecko/20121221 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.8) Gecko/20100722 BTRS86393 Firefox/3.6.8 ( .NET CLR 3.5.30729; .NET4.0C) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.2.8) Gecko/20100722 BTRS86393 Firefox/3.6.8 ( .NET CLR 3.5.30729; .NET4.0C)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; zh-TW; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; zh-TW; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-CN; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; zh-CN; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; tr; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 ( .NET CLR 3.5.30729; .NET4.0E) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; tr; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 ( .NET CLR 3.5.30729; .NET4.0E)");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; ro; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; ro; rv:1.9.2.8) Gecko/20100722 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en; rv:1.9.2.8) Gecko/20100805 Firefox/3.6.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10.6; en; rv:1.9.2.8) Gecko/20100805 Firefox/3.6.8");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.2; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.2; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux armv7l; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux armv7l; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (WindowsCE 6.0; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (WindowsCE 6.0; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko/20100101 Firefox/4.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.2; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.2; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.2; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.2; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (WindowsCE 6.0; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (WindowsCE 6.0; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux armv7l; rv:2.0.1) Gecko Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux armv7l; rv:2.0.1) Gecko Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (WindowsCE 6.0; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (WindowsCE 6.0; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux armv7l; rv:2.1.1) Gecko/ Firefox/5.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux armv7l; rv:2.1.1) Gecko/ Firefox/5.0.1");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; en-US; rv:1.9.1.7) Gecko/20100104 SeaMonkey/2.0.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; en-US; rv:1.9.1.7) Gecko/20100104 SeaMonkey/2.0.2");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; de; rv:1.9.1.7) Gecko/20100104 SeaMonkey/2.0.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; de; rv:1.9.1.7) Gecko/20100104 SeaMonkey/2.0.2");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; de; rv:1.9.1.8) Gecko/20100205 SeaMonkey/2.0.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; de; rv:1.9.1.8) Gecko/20100205 SeaMonkey/2.0.3");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.1.9) Gecko/20100428 Lightning/1.0b1 SeaMonkey/2.0.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; fr; rv:1.9.1.9) Gecko/20100428 Lightning/1.0b1 SeaMonkey/2.0.4");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.9) Gecko/20100317 SUSE/2.0.4-3.2 Lightning/1.0b1 SeaMonkey/2.0.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.9) Gecko/20100317 SUSE/2.0.4-3.2 Lightning/1.0b1 SeaMonkey/2.0.4");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.1.9) Gecko/20100317 Lightning/1.0b1 SeaMonkey/2.0.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.1.9) Gecko/20100317 Lightning/1.0b1 SeaMonkey/2.0.4");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.10) Gecko/20100504 Lightning/1.0b1 Mnenhy/0.8.2 SeaMonkey/2.0.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; de; rv:1.9.1.10) Gecko/20100504 Lightning/1.0b1 Mnenhy/0.8.2 SeaMonkey/2.0.5");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; ru; rv:1.9.1.10) Gecko/20100504 Lightning/1.0b1 SeaMonkey/2.0.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; ru; rv:1.9.1.10) Gecko/20100504 Lightning/1.0b1 SeaMonkey/2.0.5");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; en-US; rv:1.9.1.10) Gecko/20100504 Lightning/1.0b1 SeaMonkey/2.0.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; en-US; rv:1.9.1.10) Gecko/20100504 Lightning/1.0b1 SeaMonkey/2.0.5");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.11) Gecko/20100721 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.11) Gecko/20100721 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.11) Gecko/20100714 SUSE/2.0.6-2.1 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.1.11) Gecko/20100714 SUSE/2.0.6-2.1 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux ia64; de; rv:1.9.1.11) Gecko/20100820 Lightning/1.0b2pre SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux ia64; de; rv:1.9.1.11) Gecko/20100820 Lightning/1.0b2pre SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.11) Gecko/20100722 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.11) Gecko/20100722 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 9.0; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 9.0; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.0; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11) Gecko/20100722 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11) Gecko/20100722 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; de; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; de; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.0; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.0; en-US; rv:1.9.1.11) Gecko/20100701 SeaMonkey/2.0.6");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.1.11pre) Gecko/20100630 SeaMonkey/2.0.6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.1.11pre) Gecko/20100630 SeaMonkey/2.0.6pre");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6pre" || "", msg("results.patch", results.patch || "", "6pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.1.11pre) Gecko/20100629 SeaMonkey/2.0.6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.1.11pre) Gecko/20100629 SeaMonkey/2.0.6pre");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6pre" || "", msg("results.patch", results.patch || "", "6pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11pre) Gecko/20100515 SeaMonkey/2.0.6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11pre) Gecko/20100515 SeaMonkey/2.0.6pre");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6pre" || "", msg("results.patch", results.patch || "", "6pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11pre) Gecko/20100508 SeaMonkey/2.0.6pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.1.11pre) Gecko/20100508 SeaMonkey/2.0.6pre");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6pre" || "", msg("results.patch", results.patch || "", "6pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.2; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.2; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (WindowsCE 6.0; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (WindowsCE 6.0; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.5; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10.6; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686 on x86_64; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux armv7l; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux armv7l; rv:2.0.1) Gecko/20100101 Firefox/4.0.1 SeaMonkey/2.1.1");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.3a4pre) Gecko/20100404 SeaMonkey/2.1a1pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.3a4pre) Gecko/20100404 SeaMonkey/2.1a1pre");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "a1pre" || "", msg("results.patch", results.patch || "", "a1pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.3a3pre) Gecko/20100312 SeaMonkey/2.1a1pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.2; en-CA; rv:1.9.3a3pre) Gecko/20100312 SeaMonkey/2.1a1pre");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "a1pre" || "", msg("results.patch", results.patch || "", "a1pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows; Windows NT 5.2; rv:2.0b3pre) Gecko/20100803 SeaMonkey/2.1a3pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; Windows NT 5.2; rv:2.0b3pre) Gecko/20100803 SeaMonkey/2.1a3pre");
		assert.strictEqual(results.family, "SeaMonkey", msg("results.family", results.family, "SeaMonkey"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "a3pre" || "", msg("results.patch", results.patch || "", "a3pre" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; en-US) AppleWebKit/531.9 (KHTML, like Gecko) AdobeAIR/2.5.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; en-US) AppleWebKit/531.9 (KHTML, like Gecko) AdobeAIR/2.5.1");
		assert.strictEqual(results.family, "AdobeAIR", msg("results.family", results.family, "AdobeAIR"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_6_3; en-us; Silk/1.1.0-80) AppleWebKit/533.16 (KHTML, like Gecko) Version/5.0 Safari/533.16 Silk-Accelerated=true correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_6_3; en-us; Silk/1.1.0-80) AppleWebKit/533.16 (KHTML, like Gecko) Version/5.0 Safari/533.16 Silk-Accelerated=true");
		assert.strictEqual(results.family, "Amazon Silk", msg("results.family", results.family, "Amazon Silk"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0-80" || "", msg("results.patch", results.patch || "", "0-80" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; en-us; KFTT Build/IML74K) AppleWebKit/535.19 (KHTML, like Gecko) Silk/2.0 Safari/535.19 Silk-Accelerated=false correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; en-us; KFTT Build/IML74K) AppleWebKit/535.19 (KHTML, like Gecko) Silk/2.0 Safari/535.19 Silk-Accelerated=false");
		assert.strictEqual(results.family, "Amazon Silk", msg("results.family", results.family, "Amazon Silk"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; en-us; KFOT Build/IML74K) AppleWebKit/535.19 (KHTML, like Gecko) Silk/2.1 Safari/535.19 Silk-Accelerated=true correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; en-us; KFOT Build/IML74K) AppleWebKit/535.19 (KHTML, like Gecko) Silk/2.1 Safari/535.19 Silk-Accelerated=true");
		assert.strictEqual(results.family, "Amazon Silk", msg("results.family", results.family, "Amazon Silk"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; en-us; KFTT Build/IML74K) AppleWebKit/535.19 (KHTML, like Gecko) Silk/2.2 Safari/535.19 Silk-Accelerated=true correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; en-us; KFTT Build/IML74K) AppleWebKit/535.19 (KHTML, like Gecko) Silk/2.2 Safari/535.19 Silk-Accelerated=true");
		assert.strictEqual(results.family, "Amazon Silk", msg("results.family", results.family, "Amazon Silk"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.0.3; en-gb; KFTT Build/IML74K) AppleWebKit/537.36 (KHTML, like Gecko) Silk/3.25 like Chrome/34.0.1847.137 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.0.3; en-gb; KFTT Build/IML74K) AppleWebKit/537.36 (KHTML, like Gecko) Silk/3.25 like Chrome/34.0.1847.137 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Amazon Silk", msg("results.family", results.family, "Amazon Silk"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "25", msg("results.minor", results.minor || "", "25"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.2.2; en-gb; HTC Desire Build/FRG83G) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.2.2; en-gb; HTC Desire Build/FRG83G) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.3.3; en-fr; HTC/WildfireS/1.33.163.2 Build/GRI40) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.3.3; en-fr; HTC/WildfireS/1.33.163.2 Build/GRI40) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.3.4; en-us; Kindle Fire Build/GINGERBREAD) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.3.4; en-us; Kindle Fire Build/GINGERBREAD) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Linux;U;Android 2.3.5;en-us;TECNO T3 Build/master) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux;U;Android 2.3.5;en-us;TECNO T3 Build/master) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 3.0.1; en-us; GT-P7510 Build/HRI83) AppleWebKit/534.13 (KHTML, like Gecko) Version/4.0 Safari/534.13 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 3.0.1; en-us; GT-P7510 Build/HRI83) AppleWebKit/534.13 (KHTML, like Gecko) Version/4.0 Safari/534.13");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.0.3; en-us; KFTT Build/IML74K) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Safari/534.30 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.0.3; en-us; KFTT Build/IML74K) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Safari/534.30");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.0.3; en-us; KFOT Build/IML74K) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Safari/534.30 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.0.3; en-us; KFOT Build/IML74K) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Safari/534.30");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.0.3; en-us; Amaze_4G Build/IML74K) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.0.3; en-us; Amaze_4G Build/IML74K) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.0.4; en-us; PJ83100/2.20.502.7 Build/IMM76D) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.0.4; en-us; PJ83100/2.20.502.7 Build/IMM76D) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.0");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; zh_CN) AppleWebKit/534.7 (KHTML, like Gecko) Chrome/7.0 baidubrowser/1.x Safari/534.7 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; zh_CN) AppleWebKit/534.7 (KHTML, like Gecko) Chrome/7.0 baidubrowser/1.x Safari/534.7");
		assert.strictEqual(results.family, "Baidu Browser", msg("results.family", results.family, "Baidu Browser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; Trident/4.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; .NET4.0C; .NET4.0E; baidubrowser 1.x) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; Trident/4.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; .NET4.0C; .NET4.0E; baidubrowser 1.x)");
		assert.strictEqual(results.family, "Baidu Browser", msg("results.family", results.family, "Baidu Browser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.15 Safari/535.11 QQBrowser/6.13.13719.201 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.15 Safari/535.11 QQBrowser/6.13.13719.201");
		assert.strictEqual(results.family, "QQ Browser", msg("results.family", results.family, "QQ Browser"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "13", msg("results.minor", results.minor || "", "13"));
		assert.equal(results.patch || "", "13719" || "", msg("results.patch", results.patch || "", "13719" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.2; WOW64; Trident/6.0; QQBrowser/7.6.21433.400) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.2; WOW64; Trident/6.0; QQBrowser/7.6.21433.400)");
		assert.strictEqual(results.family, "QQ Browser", msg("results.family", results.family, "QQ Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "21433" || "", msg("results.patch", results.patch || "", "21433" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.1.1; zh-cn; MI 2S Build/JRO03L) AppleWebKit/537.36 (KHTML, like Gecko)Version/4.0 MQQBrowser/5.0 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.1.1; zh-cn; MI 2S Build/JRO03L) AppleWebKit/537.36 (KHTML, like Gecko)Version/4.0 MQQBrowser/5.0 Mobile Safari/537.36");
		assert.strictEqual(results.family, "QQ Browser Mobile", msg("results.family", results.family, "QQ Browser Mobile"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MQQBrowser/1.0/Mozilla/5.0 (compatible; MSIE 10.0; Windows Phone 8.0; Trident/6.0; IEMobile/10.0; NOKIA; RM-910apacprc200) correctly", function() {
		const results = uaParser("MQQBrowser/1.0/Mozilla/5.0 (compatible; MSIE 10.0; Windows Phone 8.0; Trident/6.0; IEMobile/10.0; NOKIA; RM-910apacprc200)");
		assert.strictEqual(results.family, "QQ Browser Mobile", msg("results.family", results.family, "QQ Browser Mobile"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses QQBrowser (Linux; U; zh-cn; HTC Hero Build/FRF91) correctly", function() {
		const results = uaParser("QQBrowser (Linux; U; zh-cn; HTC Hero Build/FRF91)");
		assert.strictEqual(results.family, "QQ Browser", msg("results.family", results.family, "QQ Browser"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (BB10; Touch) AppleWebKit/537.3+ (KHTML, like Gecko) Version/10.0.9.388 Mobile Safari/537.3+ correctly", function() {
		const results = uaParser("Mozilla/5.0 (BB10; Touch) AppleWebKit/537.3+ (KHTML, like Gecko) Version/10.0.9.388 Mobile Safari/537.3+");
		assert.strictEqual(results.family, "BlackBerry WebKit", msg("results.family", results.family, "BlackBerry WebKit"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (PlayBook; U; RIM Tablet OS 1.0.0; en-US) AppleWebKit/534.8+ (KHTML, like Gecko) Version/0.0.1 Safari/534.8+ correctly", function() {
		const results = uaParser("Mozilla/5.0 (PlayBook; U; RIM Tablet OS 1.0.0; en-US) AppleWebKit/534.8+ (KHTML, like Gecko) Version/0.0.1 Safari/534.8+");
		assert.strictEqual(results.family, "BlackBerry WebKit", msg("results.family", results.family, "BlackBerry WebKit"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (BlackBerry; U; BlackBerry 9800; en-GB) AppleWebKit/534.1+ (KHTML, like Gecko) Version/6.0.0.141 Mobile Safari/534.1+ correctly", function() {
		const results = uaParser("Mozilla/5.0 (BlackBerry; U; BlackBerry 9800; en-GB) AppleWebKit/534.1+ (KHTML, like Gecko) Version/6.0.0.141 Mobile Safari/534.1+");
		assert.strictEqual(results.family, "BlackBerry WebKit", msg("results.family", results.family, "BlackBerry WebKit"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (BlackBerry; U; BlackBerry 9800; en-US) AppleWebKit/534.1  (KHTML, like Gecko) Version/6.0.0.91 Mobile Safari/534.1  correctly", function() {
		const results = uaParser("Mozilla/5.0 (BlackBerry; U; BlackBerry 9800; en-US) AppleWebKit/534.1  (KHTML, like Gecko) Version/6.0.0.91 Mobile Safari/534.1 ");
		assert.strictEqual(results.family, "BlackBerry WebKit", msg("results.family", results.family, "BlackBerry WebKit"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; BOLT/2.101) AppleWebKit/530  (KHTML, like Gecko) Version/4.0 Safari/530.17 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; BOLT/2.101) AppleWebKit/530  (KHTML, like Gecko) Version/4.0 Safari/530.17");
		assert.strictEqual(results.family, "BOLT", msg("results.family", results.family, "BOLT"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "101", msg("results.minor", results.minor || "", "101"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Bunjalloo/0.7.6(Nintendo DS;U;en) correctly", function() {
		const results = uaParser("Bunjalloo/0.7.6(Nintendo DS;U;en)");
		assert.strictEqual(results.family, "Bunjalloo", msg("results.family", results.family, "Bunjalloo"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.0; Trident/4.0; chromeframe; SLCC1; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.0; Trident/4.0; chromeframe; SLCC1; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; chromeframe; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; Sleipnir 2.8.5)3.0.30729) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; chromeframe; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; Sleipnir 2.8.5)3.0.30729)");
		assert.strictEqual(results.family, "Sleipnir", msg("results.family", results.family, "Sleipnir"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Windows NT 6.1; Catchpoint) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/43.0.2357.81 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Windows NT 6.1; Catchpoint) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/43.0.2357.81 Safari/537.36");
		assert.strictEqual(results.family, "Catchpoint", msg("results.family", results.family, "Catchpoint"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Windows NT 6.1; Catchpoint bot) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/43.0.2357.81 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Windows NT 6.1; Catchpoint bot) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/43.0.2357.81 Safari/537.36");
		assert.strictEqual(results.family, "Catchpoint bot", msg("results.family", results.family, "Catchpoint bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; chromeframe/11.0.660.0) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; chromeframe/11.0.660.0)");
		assert.strictEqual(results.family, "Chrome Frame", msg("results.family", results.family, "Chrome Frame"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "660" || "", msg("results.patch", results.patch || "", "660" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.1.2; Nexus 5X Build/N2G47W; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.1.2; Nexus 5X Build/N2G47W; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Chrome Mobile WebView", msg("results.family", results.family, "Chrome Mobile WebView"));
		assert.strictEqual(results.major || "", "58", msg("results.major", results.major || "", "58"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3029" || "", msg("results.patch", results.patch || "", "3029" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.4.4; SHV31 Build/S2280) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/33.0.0.0 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.4.4; SHV31 Build/S2280) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/33.0.0.0 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Chrome Mobile WebView", msg("results.family", results.family, "Chrome Mobile WebView"));
		assert.strictEqual(results.major || "", "33", msg("results.major", results.major || "", "33"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.2; Galaxy Nexus Build/JOP40C) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.166 Mobile Safari/535.19 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.2; Galaxy Nexus Build/JOP40C) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.166 Mobile Safari/535.19");
		assert.strictEqual(results.family, "Chrome Mobile", msg("results.family", results.family, "Chrome Mobile"));
		assert.strictEqual(results.major || "", "18", msg("results.major", results.major || "", "18"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1025" || "", msg("results.patch", results.patch || "", "1025" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.1.1; SPH-L710 Build/JRO03L) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.166 Mobile Safari/535.19 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.1.1; SPH-L710 Build/JRO03L) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.166 Mobile Safari/535.19");
		assert.strictEqual(results.family, "Chrome Mobile", msg("results.family", results.family, "Chrome Mobile"));
		assert.strictEqual(results.major || "", "18", msg("results.major", results.major || "", "18"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1025" || "", msg("results.patch", results.patch || "", "1025" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.4.2; Nexus 5 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/35.0.1916.122 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.4.2; Nexus 5 Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/35.0.1916.122 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Chrome Mobile", msg("results.family", results.family, "Chrome Mobile"));
		assert.strictEqual(results.major || "", "35", msg("results.major", results.major || "", "35"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1916" || "", msg("results.patch", results.patch || "", "1916" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US) AppleWebKit/534.16 (KHTML, like Gecko) Ubuntu/10.10 Chromium/10.0.648.133 Chrome/10.0.648.133 Safari/534.16 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US) AppleWebKit/534.16 (KHTML, like Gecko) Ubuntu/10.10 Chromium/10.0.648.133 Chrome/10.0.648.133 Safari/534.16");
		assert.strictEqual(results.family, "Chromium", msg("results.family", results.family, "Chromium"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "648" || "", msg("results.patch", results.patch || "", "648" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US) AppleWebKit/532.5 (KHTML, like Gecko) Comodo_Dragon/4.1.1.11 Chrome/4.1.249.1042 Safari/532.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US) AppleWebKit/532.5 (KHTML, like Gecko) Comodo_Dragon/4.1.1.11 Chrome/4.1.249.1042 Safari/532.5");
		assert.strictEqual(results.family, "Comodo Dragon", msg("results.family", results.family, "Comodo Dragon"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.16) Gecko/20110302 Conkeror/0.9.2 (Debian-0.9.2+git100804-1) correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.16) Gecko/20110302 Conkeror/0.9.2 (Debian-0.9.2+git100804-1)");
		assert.strictEqual(results.family, "Conkeror", msg("results.family", results.family, "Conkeror"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:2.0) Gecko/20110408 conkeror/0.9.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:2.0) Gecko/20110408 conkeror/0.9.3");
		assert.strictEqual(results.family, "Conkeror", msg("results.family", results.family, "Conkeror"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (SAMSUNG; SAMSUNG-GT-S8500/S8500XXJEE; U; Bada/1.0; nl-nl) AppleWebKit/533.1 (KHTML, like Gecko) Dolfin/2.0 Mobile WVGA SMM-MMS/1.2.0 OPN-B correctly", function() {
		const results = uaParser("Mozilla/5.0 (SAMSUNG; SAMSUNG-GT-S8500/S8500XXJEE; U; Bada/1.0; nl-nl) AppleWebKit/533.1 (KHTML, like Gecko) Dolfin/2.0 Mobile WVGA SMM-MMS/1.2.0 OPN-B");
		assert.strictEqual(results.family, "Dolfin", msg("results.family", results.family, "Dolfin"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses facebookexternalhit/1.1 (+http://www.facebook.com/externalhit_uatext.php) correctly", function() {
		const results = uaParser("facebookexternalhit/1.1 (+http://www.facebook.com/externalhit_uatext.php)");
		assert.strictEqual(results.family, "FacebookBot", msg("results.family", results.family, "FacebookBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LinkedInBot/1.0 (compatible; Mozilla/5.0; Jakarta Commons-HttpClient/3.1 +http://www.linkedin.com) correctly", function() {
		const results = uaParser("LinkedInBot/1.0 (compatible; Mozilla/5.0; Jakarta Commons-HttpClient/3.1 +http://www.linkedin.com)");
		assert.strictEqual(results.family, "LinkedInBot", msg("results.family", results.family, "LinkedInBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Google (+https://developers.google.com/+/web/snippet/) correctly", function() {
		const results = uaParser("Google (+https://developers.google.com/+/web/snippet/)");
		assert.strictEqual(results.family, "GooglePlusBot", msg("results.family", results.family, "GooglePlusBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Firefox/11.0 (via ggpht.com GoogleImageProxy) correctly", function() {
		const results = uaParser("Firefox/11.0 (via ggpht.com GoogleImageProxy)");
		assert.strictEqual(results.family, "GmailImageProxy", msg("results.family", results.family, "GmailImageProxy"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Twitterbot/1.0 correctly", function() {
		const results = uaParser("Twitterbot/1.0");
		assert.strictEqual(results.family, "Twitterbot", msg("results.family", results.family, "Twitterbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WhatsApp/2.17.70 W correctly", function() {
		const results = uaParser("WhatsApp/2.17.70 W");
		assert.strictEqual(results.family, "WhatsApp", msg("results.family", results.family, "WhatsApp"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "17", msg("results.minor", results.minor || "", "17"));
		assert.equal(results.patch || "", "70" || "", msg("results.patch", results.patch || "", "70" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.1pre) Gecko/20090717 Ubuntu/9.04 (jaunty) Shiretoko/3.5.1pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; en-US; rv:1.9.1.1pre) Gecko/20090717 Ubuntu/9.04 (jaunty) Shiretoko/3.5.1pre");
		assert.strictEqual(results.family, "Firefox (Shiretoko)", msg("results.family", results.family, "Firefox (Shiretoko)"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1pre" || "", msg("results.patch", results.patch || "", "1pre" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686 (x86_64); rv:2.0b4) Gecko/20100818 Firefox/4.0b4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686 (x86_64); rv:2.0b4) Gecko/20100818 Firefox/4.0b4");
		assert.strictEqual(results.family, "Firefox Beta", msg("results.family", results.family, "Firefox Beta"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b4" || "", msg("results.patch", results.patch || "", "b4" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:2.0b8pre) Gecko/20101031 Firefox-4.0/4.0b8pre correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:2.0b8pre) Gecko/20101031 Firefox-4.0/4.0b8pre");
		assert.strictEqual(results.family, "Firefox Beta", msg("results.family", results.family, "Firefox Beta"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b8pre" || "", msg("results.patch", results.patch || "", "b8pre" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; SunOS i86pc; en-US; rv:1.8.0.5) Gecko/20060728 Firefox/1.5.0.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; SunOS i86pc; en-US; rv:1.8.0.5) Gecko/20060728 Firefox/1.5.0.5");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.12) Gecko/20101027 Ubuntu/10.04 (lucid) Firefox/3.6.12 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US; rv:1.9.2.12) Gecko/20101027 Ubuntu/10.04 (lucid) Firefox/3.6.12");
		assert.strictEqual(results.family, "Firefox", msg("results.family", results.family, "Firefox"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "12" || "", msg("results.patch", results.patch || "", "12" || ""));
	});
	it("parses Mozilla/5.0 (Mobile; rv:15.0) Gecko/15.0 Firefox/15.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Mobile; rv:15.0) Gecko/15.0 Firefox/15.0");
		assert.strictEqual(results.family, "Firefox Mobile", msg("results.family", results.family, "Firefox Mobile"));
		assert.strictEqual(results.major || "", "15", msg("results.major", results.major || "", "15"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/534+ (KHTML, like Gecko) FireWeb/1.0.0.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/534+ (KHTML, like Gecko) FireWeb/1.0.0.0");
		assert.strictEqual(results.family, "FireWeb", msg("results.family", results.family, "FireWeb"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)");
		assert.strictEqual(results.family, "Googlebot", msg("results.family", results.family, "Googlebot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (DTV) AppleWebKit/531.2+ (KHTML, like Gecko) Espial/6.1.6 AQUOSBrowser/2.0 (US01DTV;V;0001;0001) correctly", function() {
		const results = uaParser("Mozilla/5.0 (DTV) AppleWebKit/531.2+ (KHTML, like Gecko) Espial/6.1.6 AQUOSBrowser/2.0 (US01DTV;V;0001;0001)");
		assert.strictEqual(results.family, "Espial", msg("results.family", results.family, "Espial"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses iBrowser/Mini2.8 (Nokia5130c-2/07.97) correctly", function() {
		const results = uaParser("iBrowser/Mini2.8 (Nokia5130c-2/07.97)");
		assert.strictEqual(results.family, "iBrowser Mini", msg("results.family", results.family, "iBrowser Mini"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ICE Browser/5.05 (Java 1.4.0; Windows 2000 5.0 x86) correctly", function() {
		const results = uaParser("ICE Browser/5.05 (Java 1.4.0; Windows 2000 5.0 x86)");
		assert.strictEqual(results.family, "ICE Browser", msg("results.family", results.family, "ICE Browser"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "05", msg("results.minor", results.minor || "", "05"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:2.0) Gecko/20110417 IceCat/4.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:2.0) Gecko/20110417 IceCat/4.0");
		assert.strictEqual(results.family, "IceCat", msg("results.family", results.family, "IceCat"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0; XBLWP7; ZuneWP7) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/5.0; XBLWP7; ZuneWP7)");
		assert.strictEqual(results.family, "IE Large Screen", msg("results.family", results.family, "IE Large Screen"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows Phone OS 7.0; Trident/3.1; IEMobile/7.0; SAMSUNG; SGH-i917) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows Phone OS 7.0; Trident/3.1; IEMobile/7.0; SAMSUNG; SGH-i917)");
		assert.strictEqual(results.family, "IE Mobile", msg("results.family", results.family, "IE Mobile"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 9.0; Windows Phone OS 7.5; Trident/5.0; IEMobile/9.0; SAMSUNG; SGH-i917) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 9.0; Windows Phone OS 7.5; Trident/5.0; IEMobile/9.0; SAMSUNG; SGH-i917)");
		assert.strictEqual(results.family, "IE Mobile", msg("results.family", results.family, "IE Mobile"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 9.0; Windows Phone OS 7.5; Trident/5.0; IEMobile/9.0; NOKIA; Lumia 800) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 9.0; Windows Phone OS 7.5; Trident/5.0; IEMobile/9.0; NOKIA; Lumia 800)");
		assert.strictEqual(results.family, "IE Mobile", msg("results.family", results.family, "IE Mobile"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 10.0; Windows Phone 8.0; Trident/6.0; IEMobile/10.0; ARM; Touch; NOKIA; Lumia 920) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 10.0; Windows Phone 8.0; Trident/6.0; IEMobile/10.0; ARM; Touch; NOKIA; Lumia 920)");
		assert.strictEqual(results.family, "IE Mobile", msg("results.family", results.family, "IE Mobile"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; GTB6; .NET CLR 2.0.50727; .NET CLR 1.1.4322) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; GTB6; .NET CLR 2.0.50727; .NET CLR 1.1.4322)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 WebTV/2.6 (compatible; MSIE 4.0) correctly", function() {
		const results = uaParser("Mozilla/4.0 WebTV/2.6 (compatible; MSIE 4.0)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.17; Mac_PowerPC) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.17; Mac_PowerPC)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "17", msg("results.minor", results.minor || "", "17"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.2; ARM; Trident/6.0) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.2; ARM; Trident/6.0)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; GTB6; chromeframe; .NET CLR 2.0.50727; .NET CLR 1.1.4322; .NET CLR 3.0.04506.648; .NET CLR 3.5.21022; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; GTB6; chromeframe; .NET CLR 2.0.50727; .NET CLR 1.1.4322; .NET CLR 3.0.04506.648; .NET CLR 3.5.21022; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Java/1.6.0_43 correctly", function() {
		const results = uaParser("Java/1.6.0_43");
		assert.strictEqual(results.family, "Java", msg("results.family", results.family, "Java"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "43" || "", msg("results.patch", results.patch || "", "43" || ""));
	});
	it("parses Java/1.7.0_71 correctly", function() {
		const results = uaParser("Java/1.7.0_71");
		assert.strictEqual(results.family, "Java", msg("results.family", results.family, "Java"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "71" || "", msg("results.patch", results.patch || "", "71" || ""));
	});
	it("parses Java/1.8.0_25 correctly", function() {
		const results = uaParser("Java/1.8.0_25");
		assert.strictEqual(results.family, "Java", msg("results.family", results.family, "Java"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "25" || "", msg("results.patch", results.patch || "", "25" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; en-US) AppleWebKit/528.5+ (KHTML, like Gecko, Safari/528.5+) Version/4.0 Kindle/3.0 (screen 600x800; rotate) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; en-US) AppleWebKit/528.5+ (KHTML, like Gecko, Safari/528.5+) Version/4.0 Kindle/3.0 (screen 600x800; rotate)");
		assert.strictEqual(results.family, "Kindle", msg("results.family", results.family, "Kindle"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux; de-DE) AppleWebKit/527  (KHTML, like Gecko, Safari/419.3) konqueror/4.3.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux; de-DE) AppleWebKit/527  (KHTML, like Gecko, Safari/419.3) konqueror/4.3.1");
		assert.strictEqual(results.family, "Konqueror", msg("results.family", results.family, "Konqueror"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.2.17) Gecko/20110414 Lightning/1.0b3pre Thunderbird/3.1.10 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.2.17) Gecko/20110414 Lightning/1.0b3pre Thunderbird/3.1.10");
		assert.strictEqual(results.family, "Lightning", msg("results.family", results.family, "Lightning"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "b3pre" || "", msg("results.patch", results.patch || "", "b3pre" || ""));
	});
	it("parses Mozilla/5.0 (Linux x86_64) AppleWebKit/534.26+ WebKitGTK+/1.4.1 luakit/f3a2dbe correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux x86_64) AppleWebKit/534.26+ WebKitGTK+/1.4.1 luakit/f3a2dbe");
		assert.strictEqual(results.family, "LuaKit", msg("results.family", results.family, "LuaKit"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.47 Safari/535.11 MRCHROME correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.1) AppleWebKit/535.11 (KHTML, like Gecko) Chrome/17.0.963.47 Safari/535.11 MRCHROME");
		assert.strictEqual(results.family, "Mail.ru Chromium Browser", msg("results.family", results.family, "Mail.ru Chromium Browser"));
		assert.strictEqual(results.major || "", "17", msg("results.major", results.major || "", "17"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "963" || "", msg("results.patch", results.patch || "", "963" || ""));
	});
	it("parses Midori/0.2 (X11; Linux; U; en-us) WebKit/531.2  correctly", function() {
		const results = uaParser("Midori/0.2 (X11; Linux; U; en-us) WebKit/531.2 ");
		assert.strictEqual(results.family, "Midori", msg("results.family", results.family, "Midori"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.3a1) Gecko/20100208 MozillaDeveloperPreview/3.7a1 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.3a1) Gecko/20100208 MozillaDeveloperPreview/3.7a1 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "MozillaDeveloperPreview", msg("results.family", results.family, "MozillaDeveloperPreview"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "a1" || "", msg("results.patch", results.patch || "", "a1" || ""));
	});
	it("parses NCSA_Mosaic/2.0 (Windows 3.1) correctly", function() {
		const results = uaParser("NCSA_Mosaic/2.0 (Windows 3.1)");
		assert.strictEqual(results.family, "NCSA Mosaic", msg("results.family", results.family, "NCSA Mosaic"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (PLAYSTATION 3; 3.55) correctly", function() {
		const results = uaParser("Mozilla/5.0 (PLAYSTATION 3; 3.55)");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (PLAYSTATION 3 4.31) AppleWebKit/531.22.8 (KHTML, like Gecko) correctly", function() {
		const results = uaParser("Mozilla/5.0 (PLAYSTATION 3 4.31) AppleWebKit/531.22.8 (KHTML, like Gecko)");
		assert.strictEqual(results.family, "NetFront NX", msg("results.family", results.family, "NetFront NX"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (PSP (PlayStation Portable); 2.00) correctly", function() {
		const results = uaParser("Mozilla/4.0 (PSP (PlayStation Portable); 2.00)");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (PlayStation Vita 1.81) AppleWebKit/531.22.8 (KHTML, like Gecko) Silk/3.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (PlayStation Vita 1.81) AppleWebKit/531.22.8 (KHTML, like Gecko) Silk/3.2");
		assert.strictEqual(results.family, "NetFront NX", msg("results.family", results.family, "NetFront NX"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; Linux 2.6.10) NetFront/3.3 Kindle/1.0 (screen 600x800) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; Linux 2.6.10) NetFront/3.3 Kindle/1.0 (screen 600x800)");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Nintendo 3DS; U; ; en) Version/1.7498.US correctly", function() {
		const results = uaParser("Mozilla/5.0 (Nintendo 3DS; U; ; en) Version/1.7498.US");
		assert.strictEqual(results.family, "NetFront NX", msg("results.family", results.family, "NetFront NX"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Nintendo WiiU) AppleWebKit/534.52 (KHTML, like Gecko) NX/2.1.0.8.21 NintendoBrowser/1.0.0.7494.US correctly", function() {
		const results = uaParser("Mozilla/5.0 (Nintendo WiiU) AppleWebKit/534.52 (KHTML, like Gecko) NX/2.1.0.8.21 NintendoBrowser/1.0.0.7494.US");
		assert.strictEqual(results.family, "NetFront NX", msg("results.family", results.family, "NetFront NX"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses HUAWEI-M750/001.00 ACS-NetFront/3.2 correctly", function() {
		const results = uaParser("HUAWEI-M750/001.00 ACS-NetFront/3.2");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (BREW 3.1.5; U; en-us; Sanyo; NetFront/3.5.1/AMB) Boost SCP3810 correctly", function() {
		const results = uaParser("Mozilla/4.0 (BREW 3.1.5; U; en-us; Sanyo; NetFront/3.5.1/AMB) Boost SCP3810");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses NetFront/3.5.1 (BREW 3.1.5; U; en-us; LG; NetFront/3.5.1/WAP) Sprint LN240 MMP/2.0 Profile/MIDP-2.1 Configuration/CLDC-1.1 correctly", function() {
		const results = uaParser("NetFront/3.5.1 (BREW 3.1.5; U; en-us; LG; NetFront/3.5.1/WAP) Sprint LN240 MMP/2.0 Profile/MIDP-2.1 Configuration/CLDC-1.1");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/4.0 (Brew MP 1.0.2; U; en-us; Sanyo; NetFront/3.5.1/AMB) Sprint E4100 correctly", function() {
		const results = uaParser("Mozilla/4.0 (Brew MP 1.0.2; U; en-us; Sanyo; NetFront/3.5.1/AMB) Sprint E4100");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses PantechP6010/JNUS11072011 BMP/1.0.2 DeviceId/141020 NetFront/4.1 OMC/1.5.3 Profile/MIDP-2.1 Configuration/CLDC-1.1 correctly", function() {
		const results = uaParser("PantechP6010/JNUS11072011 BMP/1.0.2 DeviceId/141020 NetFront/4.1 OMC/1.5.3 Profile/MIDP-2.1 Configuration/CLDC-1.1");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NetFront/4.2 (BMP 1.0.4; U; en-us; LG; NetFront/4.2/AMB) Boost LG272 MMP/2.0 Profile/MIDP-2.1 Configuration/CLDC-1.1 correctly", function() {
		const results = uaParser("NetFront/4.2 (BMP 1.0.4; U; en-us; LG; NetFront/4.2/AMB) Boost LG272 MMP/2.0 Profile/MIDP-2.1 Configuration/CLDC-1.1");
		assert.strictEqual(results.family, "NetFront", msg("results.family", results.family, "NetFront"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (SymbianOS/9.4; U; Series60/5.0 Nokia5800d-1/21.0.025; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/413 (KHTML, like Gecko) Safari/413 correctly", function() {
		const results = uaParser("Mozilla/5.0 (SymbianOS/9.4; U; Series60/5.0 Nokia5800d-1/21.0.025; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/413 (KHTML, like Gecko) Safari/413");
		assert.strictEqual(results.family, "Nokia Browser", msg("results.family", results.family, "Nokia Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Symbian/3; Series60/5.2 NokiaN8-00/013.016; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/525 (KHTML, like Gecko) Version/3.0 BrowserNG/7.2.8.10 3gpp-gba correctly", function() {
		const results = uaParser("Mozilla/5.0 (Symbian/3; Series60/5.2 NokiaN8-00/013.016; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/525 (KHTML, like Gecko) Version/3.0 BrowserNG/7.2.8.10 3gpp-gba");
		assert.strictEqual(results.family, "Nokia Browser", msg("results.family", results.family, "Nokia Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Symbian/3; Series60/5.2 NokiaN8-00/012.002; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/533.4 (KHTML, like Gecko) NokiaBrowser/7.3.0 Mobile Safari/533.4 3gpp-gba correctly", function() {
		const results = uaParser("Mozilla/5.0 (Symbian/3; Series60/5.2 NokiaN8-00/012.002; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/533.4 (KHTML, like Gecko) NokiaBrowser/7.3.0 Mobile Safari/533.4 3gpp-gba");
		assert.strictEqual(results.family, "Nokia Browser", msg("results.family", results.family, "Nokia Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Symbian/3; Series60/5.3 Nokia701/111.020.0307; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/533.4 (KHTML, like Gecko) NokiaBrowser/7.4.1.14 Mobile Safari/533.4 3gpp-gba correctly", function() {
		const results = uaParser("Mozilla/5.0 (Symbian/3; Series60/5.3 Nokia701/111.020.0307; Profile/MIDP-2.1 Configuration/CLDC-1.1 ) AppleWebKit/533.4 (KHTML, like Gecko) NokiaBrowser/7.4.1.14 Mobile Safari/533.4 3gpp-gba");
		assert.strictEqual(results.family, "Nokia Browser", msg("results.family", results.family, "Nokia Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (MeeGo; NokiaN9) AppleWebKit/534.13 (KHTML, like Gecko) NokiaBrowser/8.5.0 Mobile Safari/534.13 correctly", function() {
		const results = uaParser("Mozilla/5.0 (MeeGo; NokiaN9) AppleWebKit/534.13 (KHTML, like Gecko) NokiaBrowser/8.5.0 Mobile Safari/534.13");
		assert.strictEqual(results.family, "Nokia Browser", msg("results.family", results.family, "Nokia Browser"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses ALCATEL-OT510A/382 ObigoInternetBrowser/Q05A correctly", function() {
		const results = uaParser("ALCATEL-OT510A/382 ObigoInternetBrowser/Q05A");
		assert.strictEqual(results.family, "Obigo", msg("results.family", results.family, "Obigo"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Huawei/1.0/0HuaweiG2800/WAP2.0/Obigo-Browser/Q03C MMS/Obigo-MMS/1.2 correctly", function() {
		const results = uaParser("Huawei/1.0/0HuaweiG2800/WAP2.0/Obigo-Browser/Q03C MMS/Obigo-MMS/1.2");
		assert.strictEqual(results.family, "Obigo", msg("results.family", results.family, "Obigo"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PantechP7040/JLUS04042011 Browser/Obigo/Q05A OMC/1.5.3 Profile/MIDP-2.1 Configuration/CLDC-1.1 correctly", function() {
		const results = uaParser("PantechP7040/JLUS04042011 Browser/Obigo/Q05A OMC/1.5.3 Profile/MIDP-2.1 Configuration/CLDC-1.1");
		assert.strictEqual(results.family, "Obigo", msg("results.family", results.family, "Obigo"));
		assert.strictEqual(results.major || "", "05", msg("results.major", results.major || "", "05"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses OneBrowser/3.0 (SAMSUNG-GT-S5253/S5253DDKJ2) correctly", function() {
		const results = uaParser("OneBrowser/3.0 (SAMSUNG-GT-S5253/S5253DDKJ2)");
		assert.strictEqual(results.family, "ONE Browser", msg("results.family", results.family, "ONE Browser"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses OneBrowser/3.0 (NokiaC2-00/03.42) correctly", function() {
		const results = uaParser("OneBrowser/3.0 (NokiaC2-00/03.42)");
		assert.strictEqual(results.family, "ONE Browser", msg("results.family", results.family, "ONE Browser"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SAMSUNG-C3053/1.0 Openwave/6.2.3 Profile/MIDP-2.0 Configuration/CLDC-1.1 UP.Browser/6.2.3.3.c.1.101 (GUI) MMP/2.0 correctly", function() {
		const results = uaParser("SAMSUNG-C3053/1.0 Openwave/6.2.3 Profile/MIDP-2.0 Configuration/CLDC-1.1 UP.Browser/6.2.3.3.c.1.101 (GUI) MMP/2.0");
		assert.strictEqual(results.family, "Openwave", msg("results.family", results.family, "Openwave"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Opera/9.80 (VRE; Opera Mini/4.2/28.2794; U; en) Presto/2.8.119 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (VRE; Opera Mini/4.2/28.2794; U; en) Presto/2.8.119 Version/11.10");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (BREW; Opera Mini/5.1.191/27.2202; U; en) Presto/2.8.119 240X400 LG VN271 correctly", function() {
		const results = uaParser("Opera/9.80 (BREW; Opera Mini/5.1.191/27.2202; U; en) Presto/2.8.119 240X400 LG VN271");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "191" || "", msg("results.patch", results.patch || "", "191" || ""));
	});
	it("parses Opera/9.80 (Series 60; Opera Mini/6.24455/25.677; U; fr) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (Series 60; Opera Mini/6.24455/25.677; U; fr) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "24455", msg("results.minor", results.minor || "", "24455"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (BlackBerry; Opera Mini/7.0.31437/28.3030; U; en) Presto/2.8.119 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (BlackBerry; Opera Mini/7.0.31437/28.3030; U; en) Presto/2.8.119 Version/11.10");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "31437" || "", msg("results.patch", results.patch || "", "31437" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/9.80 (S60; SymbOS; Opera Mobi/23.348; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/9.80 (S60; SymbOS; Opera Mobi/23.348; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "80", msg("results.minor", results.minor || "", "80"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (J2ME/MIDP; Opera Mini/9.80 (J2ME/22.478; U; en) Presto/2.5.25 Version/10.54 correctly", function() {
		const results = uaParser("Opera/9.80 (J2ME/MIDP; Opera Mini/9.80 (J2ME/22.478; U; en) Presto/2.5.25 Version/10.54");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "80", msg("results.minor", results.minor || "", "80"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (Android; Opera Mini/7.6.35766/35.5706; U; en) Presto/2.8.119 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (Android; Opera Mini/7.6.35766/35.5706; U; en) Presto/2.8.119 Version/11.10");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "35766" || "", msg("results.patch", results.patch || "", "35766" || ""));
	});
	it("parses Opera/9.80 (S60; SymbOS; Opera Mobi/275; U; es-ES) Presto/2.4.13 Version/10.00 correctly", function() {
		const results = uaParser("Opera/9.80 (S60; SymbOS; Opera Mobi/275; U; es-ES) Presto/2.4.13 Version/10.00");
		assert.strictEqual(results.family, "Opera Mobile", msg("results.family", results.family, "Opera Mobile"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "00", msg("results.minor", results.minor || "", "00"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (Android 3.2; Linux; Opera Tablet/ADR-1106291546; U; en) Presto/2.8.149 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (Android 3.2; Linux; Opera Tablet/ADR-1106291546; U; en) Presto/2.8.149 Version/11.10");
		assert.strictEqual(results.family, "Opera Tablet", msg("results.family", results.family, "Opera Tablet"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.30 (Nintendo Wii; U; ; 3642; en) correctly", function() {
		const results = uaParser("Opera/9.30 (Nintendo Wii; U; ; 3642; en)");
		assert.strictEqual(results.family, "Opera", msg("results.family", results.family, "Opera"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "30", msg("results.minor", results.minor || "", "30"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.50 (Nintendo DSi; Opera/507; U; en-US) correctly", function() {
		const results = uaParser("Opera/9.50 (Nintendo DSi; Opera/507; U; en-US)");
		assert.strictEqual(results.family, "Opera", msg("results.family", results.family, "Opera"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "50", msg("results.minor", results.minor || "", "50"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (Windows NT 5.1; U; ru) Presto/2.5.24 Version/10.53 correctly", function() {
		const results = uaParser("Opera/9.80 (Windows NT 5.1; U; ru) Presto/2.5.24 Version/10.53");
		assert.strictEqual(results.family, "Opera", msg("results.family", results.family, "Opera"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "53", msg("results.minor", results.minor || "", "53"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_7_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/28.0.1500.20 Safari/537.36 OPR/15.0.1147.18 (Edition Next) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_7_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/28.0.1500.20 Safari/537.36 OPR/15.0.1147.18 (Edition Next)");
		assert.strictEqual(results.family, "Opera", msg("results.family", results.family, "Opera"));
		assert.strictEqual(results.major || "", "15", msg("results.major", results.major || "", "15"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1147" || "", msg("results.patch", results.patch || "", "1147" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android; 4.1.2; GT-I9100 Build/000000) AppleWebKit/537.22 (KHTML, like Gecko) Chrome/25.0.1234.12 Mobile Safari/537.22 OPR/14.0.123.123 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android; 4.1.2; GT-I9100 Build/000000) AppleWebKit/537.22 (KHTML, like Gecko) Chrome/25.0.1234.12 Mobile Safari/537.22 OPR/14.0.123.123");
		assert.strictEqual(results.family, "Opera Mobile", msg("results.family", results.family, "Opera Mobile"));
		assert.strictEqual(results.major || "", "14", msg("results.major", results.major || "", "14"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "123" || "", msg("results.patch", results.patch || "", "123" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/40.0.2214.10 Safari/537.36 OPR/27.0.1689.22 (Edition developer) correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/40.0.2214.10 Safari/537.36 OPR/27.0.1689.22 (Edition developer)");
		assert.strictEqual(results.family, "Opera", msg("results.family", results.family, "Opera"));
		assert.strictEqual(results.major || "", "27", msg("results.major", results.major || "", "27"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1689" || "", msg("results.patch", results.patch || "", "1689" || ""));
	});
	it("parses SomethingWeNeverKnewExisted correctly", function() {
		const results = uaParser("SomethingWeNeverKnewExisted");
		assert.strictEqual(results.family, "Other", msg("results.family", results.family, "Other"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Series40; NokiaC2-03/07.48; Profile/MIDP-2.1 Configuration/CLDC-1.1) Gecko/20100401 S40OviBrowser/2.2.0.0.33 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Series40; NokiaC2-03/07.48; Profile/MIDP-2.1 Configuration/CLDC-1.1) Gecko/20100401 S40OviBrowser/2.2.0.0.33");
		assert.strictEqual(results.family, "Ovi Browser", msg("results.family", results.family, "Ovi Browser"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Series40; NokiaX2-05/08.35; Profile/MIDP-2.1 Configuration/CLDC-1.1) Gecko/20100401 S40OviBrowser/2.0.2.68.14 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Series40; NokiaX2-05/08.35; Profile/MIDP-2.1 Configuration/CLDC-1.1) Gecko/20100401 S40OviBrowser/2.0.2.68.14");
		assert.strictEqual(results.family, "Ovi Browser", msg("results.family", results.family, "Ovi Browser"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.1; rv:2.0) Gecko/20110407 Firefox/4.0.3 PaleMoon/4.0.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.1; rv:2.0) Gecko/20110407 Firefox/4.0.3 PaleMoon/4.0.3");
		assert.strictEqual(results.family, "Pale Moon", msg("results.family", results.family, "Pale Moon"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:3.0) Goanna/20170207 PaleMoon/27.1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:3.0) Goanna/20170207 PaleMoon/27.1.0");
		assert.strictEqual(results.family, "Pale Moon", msg("results.family", results.family, "Pale Moon"));
		assert.strictEqual(results.major || "", "27", msg("results.major", results.major || "", "27"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:3.0) Gecko/20100101 Goanna/20170207 PaleMoon/27.1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:3.0) Gecko/20100101 Goanna/20170207 PaleMoon/27.1.0");
		assert.strictEqual(results.family, "Pale Moon", msg("results.family", results.family, "Pale Moon"));
		assert.strictEqual(results.major || "", "27", msg("results.major", results.major || "", "27"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:45.9) Gecko/20100101 Goanna/3.0 Firefox/45.9 PaleMoon/27.1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:45.9) Gecko/20100101 Goanna/3.0 Firefox/45.9 PaleMoon/27.1.0");
		assert.strictEqual(results.family, "Pale Moon", msg("results.family", results.family, "Pale Moon"));
		assert.strictEqual(results.major || "", "27", msg("results.major", results.major || "", "27"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (LG-T500 AppleWebkit/531 Browser/Phantom/V2.0 Widget/LGMW/3.0 MMS/LG-MMS-V1.0/1.2 Java/ASVM/1.1 Profile/MIDP-2.1 Configuration/CLDC-1.1) correctly", function() {
		const results = uaParser("Mozilla/5.0 (LG-T500 AppleWebkit/531 Browser/Phantom/V2.0 Widget/LGMW/3.0 MMS/LG-MMS-V1.0/1.2 Java/ASVM/1.1 Profile/MIDP-2.1 Configuration/CLDC-1.1)");
		assert.strictEqual(results.family, "Phantom Browser", msg("results.family", results.family, "Phantom Browser"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pingdom.com_bot_version_1.4_(http://www.pingdom.com/) correctly", function() {
		const results = uaParser("Pingdom.com_bot_version_1.4_(http://www.pingdom.com/)");
		assert.strictEqual(results.family, "PingdomBot", msg("results.family", results.family, "PingdomBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Unknown; Linux x86_64) AppleWebKit/534.34 (KHTML, like Gecko) PingdomTMS/0.8.5 Safari/534.34 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Unknown; Linux x86_64) AppleWebKit/534.34 (KHTML, like Gecko) PingdomTMS/0.8.5 Safari/534.34");
		assert.strictEqual(results.family, "PingdomBot", msg("results.family", results.family, "PingdomBot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses NewRelicPinger/1.0 (1025794) correctly", function() {
		const results = uaParser("NewRelicPinger/1.0 (1025794)");
		assert.strictEqual(results.family, "NewRelicPingerBot", msg("results.family", results.family, "NewRelicPingerBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.4 (KHTML, like Gecko) Chrome/98 Safari/537.4 (StatusCake) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.4 (KHTML, like Gecko) Chrome/98 Safari/537.4 (StatusCake)");
		assert.strictEqual(results.family, "StatusCakeBot", msg("results.family", results.family, "StatusCakeBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/3.0 (Planetweb/2.100 JS SSL US; Dreamcast US) correctly", function() {
		const results = uaParser("Mozilla/3.0 (Planetweb/2.100 JS SSL US; Dreamcast US)");
		assert.strictEqual(results.family, "Planetweb", msg("results.family", results.family, "Planetweb"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "100", msg("results.minor", results.minor || "", "100"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PyAMF/0.6.1 correctly", function() {
		const results = uaParser("PyAMF/0.6.1");
		assert.strictEqual(results.family, "PyAMF", msg("results.family", results.family, "PyAMF"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses python-requests/0.14 CPython/2.6 Linux/2.6-43-server correctly", function() {
		const results = uaParser("python-requests/0.14 CPython/2.6 Linux/2.6-43-server");
		assert.strictEqual(results.family, "Python Requests", msg("results.family", results.family, "Python Requests"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "14", msg("results.minor", results.minor || "", "14"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11i; Linux; C) AppleWebKikt/533.3 (KHTML, like Gecko) QtCarBrowser Safari/533.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11i; Linux; C) AppleWebKikt/533.3 (KHTML, like Gecko) QtCarBrowser Safari/533.3");
		assert.strictEqual(results.family, "QtCarBrowser", msg("results.family", results.family, "QtCarBrowser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Rackspace Monitoring/1.1 (https://monitoring.api.rackspacecloud.com) correctly", function() {
		const results = uaParser("Rackspace Monitoring/1.1 (https://monitoring.api.rackspacecloud.com)");
		assert.strictEqual(results.family, "RackspaceBot", msg("results.family", results.family, "RackspaceBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; BSD Four; en-US) AppleWebKit/533.3 (KHTML, like Gecko) rekonq Safari/533.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; BSD Four; en-US) AppleWebKit/533.3 (KHTML, like Gecko) rekonq Safari/533.3");
		assert.strictEqual(results.family, "Rekonq", msg("results.family", results.family, "Rekonq"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686) AppleWebKit/534.34 (KHTML, like Gecko) rekonq/1.0 Safari/534.34 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686) AppleWebKit/534.34 (KHTML, like Gecko) rekonq/1.0 Safari/534.34");
		assert.strictEqual(results.family, "Rekonq", msg("results.family", results.family, "Rekonq"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US) AppleWebKit/534.3 (KHTML, like Gecko) RockMelt/0.8.34.841 Chrome/6.0.472.63 Safari/534.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US) AppleWebKit/534.3 (KHTML, like Gecko) RockMelt/0.8.34.841 Chrome/6.0.472.63 Safari/534.3");
		assert.strictEqual(results.family, "RockMelt", msg("results.family", results.family, "RockMelt"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "34" || "", msg("results.patch", results.patch || "", "34" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; PPC Mac OS X; en-us) AppleWebKit/418.8 (KHTML, like Gecko) Safari/419.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; PPC Mac OS X; en-us) AppleWebKit/418.8 (KHTML, like Gecko) Safari/419.3");
		assert.strictEqual(results.family, "Safari", msg("results.family", results.family, "Safari"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_6_5; en-us) AppleWebKit/533.18.1 (KHTML, like Gecko) Version/5.0.2 Safari/533.18.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_6_5; en-us) AppleWebKit/533.18.1 (KHTML, like Gecko) Version/5.0.2 Safari/533.18.5");
		assert.strictEqual(results.family, "Safari", msg("results.family", results.family, "Safari"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_5_7; en-us) AppleWebKit/530.17 (KHTML, like Gecko) Version/4.0 Safari/530.17 Skyfire/2.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_5_7; en-us) AppleWebKit/530.17 (KHTML, like Gecko) Version/4.0 Safari/530.17 Skyfire/2.0");
		assert.strictEqual(results.family, "Skyfire", msg("results.family", results.family, "Skyfire"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.21 (KHTML, like Gecko) Snowshoe/1.0.0 Safari/537.21 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.21 (KHTML, like Gecko) Snowshoe/1.0.0 Safari/537.21");
		assert.strictEqual(results.family, "Snowshoe", msg("results.family", results.family, "Snowshoe"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.17) Gecko/20110414 Thunderbird/3.1.10 ThunderBrowse/3.3.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en-US; rv:1.9.2.17) Gecko/20110414 Thunderbird/3.1.10 ThunderBrowse/3.3.5");
		assert.strictEqual(results.family, "ThunderBrowse", msg("results.family", results.family, "ThunderBrowse"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:24.0) Gecko/20100101 Thunderbird/24.2.0 Lightning/2.6.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:24.0) Gecko/20100101 Thunderbird/24.2.0 Lightning/2.6.4");
		assert.strictEqual(results.family, "Thunderbird", msg("results.family", results.family, "Thunderbird"));
		assert.strictEqual(results.major || "", "24", msg("results.major", results.major || "", "24"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:24.0) Gecko/20100101 Thunderbird/24.2.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:24.0) Gecko/20100101 Thunderbird/24.2.0");
		assert.strictEqual(results.family, "Thunderbird", msg("results.family", results.family, "Thunderbird"));
		assert.strictEqual(results.major || "", "24", msg("results.major", results.major || "", "24"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; rv:45.0) Gecko/20100101 Thunderbird/45.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; rv:45.0) Gecko/20100101 Thunderbird/45.0");
		assert.strictEqual(results.family, "Thunderbird", msg("results.family", results.family, "Thunderbird"));
		assert.strictEqual(results.major || "", "45", msg("results.major", results.major || "", "45"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.2; WOW64; rv:7.0.1) Gecko/20151105 Postbox/4.0.8 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.2; WOW64; rv:7.0.1) Gecko/20151105 Postbox/4.0.8");
		assert.strictEqual(results.family, "Postbox", msg("results.family", results.family, "Postbox"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/4.0 (compatible; Lotus-Notes/6.0; Windows-NT) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; Lotus-Notes/6.0; Windows-NT)");
		assert.strictEqual(results.family, "Lotus Notes", msg("results.family", results.family, "Lotus Notes"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YahooMobileMail/1.0 (Android Mail; 1.3.10) (supersonic;HTC;PC36100;2.3.5/GRJ90) correctly", function() {
		const results = uaParser("YahooMobileMail/1.0 (Android Mail; 1.3.10) (supersonic;HTC;PC36100;2.3.5/GRJ90)");
		assert.strictEqual(results.family, "YahooMobileMail", msg("results.family", results.family, "YahooMobileMail"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Barca/2.8.2 correctly", function() {
		const results = uaParser("Barca/2.8.2");
		assert.strictEqual(results.family, "Barca", msg("results.family", results.family, "Barca"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses BarcaPro/1.4.12 correctly", function() {
		const results = uaParser("BarcaPro/1.4.12");
		assert.strictEqual(results.family, "Barca", msg("results.family", results.family, "Barca"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "12" || "", msg("results.patch", results.patch || "", "12" || ""));
	});
	it("parses The Bat! 4.0.0.22 correctly", function() {
		const results = uaParser("The Bat! 4.0.0.22");
		assert.strictEqual(results.family, "The Bat!", msg("results.family", results.family, "The Bat!"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses MailBar/1.3.2 (Mac OS X Version 10.11.1 (Build 15B42)) correctly", function() {
		const results = uaParser("MailBar/1.3.2 (Mac OS X Version 10.11.1 (Build 15B42))");
		assert.strictEqual(results.family, "MailBar", msg("results.family", results.family, "MailBar"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/534.34 (KHTML, like Gecko) kmail2/4.14.2 Safari/534.34 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/534.34 (KHTML, like Gecko) kmail2/4.14.2 Safari/534.34");
		assert.strictEqual(results.family, "kmail2", msg("results.family", results.family, "kmail2"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "14", msg("results.minor", results.minor || "", "14"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1; .NET CLR 1.1.4322; .NET CLR 2.0.50727; MSOffice 12) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1; .NET CLR 1.1.4322; .NET CLR 2.0.50727; MSOffice 12)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2007", msg("results.major", results.major || "", "2007"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0; Trident/4.0; InfoPath.2; MSOffice 14) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0; Trident/4.0; InfoPath.2; MSOffice 14)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2010", msg("results.major", results.major || "", "2010"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; Trident/6.0; Microsoft Outlook 15.0.4420) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; Trident/6.0; Microsoft Outlook 15.0.4420)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2013", msg("results.major", results.major || "", "2013"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_8_2) AppleWebKit/536.26.14 (KHTML, like Gecko) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_8_2) AppleWebKit/536.26.14 (KHTML, like Gecko)");
		assert.strictEqual(results.family, "Apple Mail", msg("results.family", results.family, "Apple Mail"));
		assert.strictEqual(results.major || "", "536", msg("results.major", results.major || "", "536"));
		assert.strictEqual(results.minor || "", "26", msg("results.minor", results.minor || "", "26"));
		assert.equal(results.patch || "", "14" || "", msg("results.patch", results.patch || "", "14" || ""));
	});
	it("parses Airmail 1.0 rv:148 (Macintosh; Mac OS X 10.8.3; en_BE) correctly", function() {
		const results = uaParser("Airmail 1.0 rv:148 (Macintosh; Mac OS X 10.8.3; en_BE)");
		assert.strictEqual(results.family, "Airmail", msg("results.family", results.family, "Airmail"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Airmail 1.0.6 rv:196 (Macintosh; Mac OS X 10.8.4; en_GB) correctly", function() {
		const results = uaParser("Airmail 1.0.6 rv:196 (Macintosh; Mac OS X 10.8.4; en_GB)");
		assert.strictEqual(results.family, "Airmail", msg("results.family", results.family, "Airmail"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.45.0 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.45.0 Safari/537.36");
		assert.strictEqual(results.family, "Whale", msg("results.family", results.family, "Whale"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "45" || "", msg("results.patch", results.patch || "", "45" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.50.3 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.50.3 Safari/537.36");
		assert.strictEqual(results.family, "Whale", msg("results.family", results.family, "Whale"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "50" || "", msg("results.patch", results.patch || "", "50" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 6.0.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.113 Whale/0.9.5.0 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 6.0.1) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.113 Whale/0.9.5.0 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Whale", msg("results.family", results.family, "Whale"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.1.2; Nexus 6P Build/WHALE) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.50.3 Mobile Safari/537.36 sidebar webpanel correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.1.2; Nexus 6P Build/WHALE) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.50.3 Mobile Safari/537.36 sidebar webpanel");
		assert.strictEqual(results.family, "Whale", msg("results.family", results.family, "Whale"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "50" || "", msg("results.patch", results.patch || "", "50" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.1.2; Nexus 6P Build/WHALE) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.50.3 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.1.2; Nexus 6P Build/WHALE) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.220 Whale/1.3.50.3 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Whale", msg("results.family", results.family, "Whale"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "50" || "", msg("results.patch", results.patch || "", "50" || ""));
	});
	it("parses J2ME/UCWEB7.0.3.45/139/7682 correctly", function() {
		const results = uaParser("J2ME/UCWEB7.0.3.45/139/7682");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses NOKIA6120c/UC Browser7.4.0.65/28/352 correctly", function() {
		const results = uaParser("NOKIA6120c/UC Browser7.4.0.65/28/352");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses UCWEB/3.0 (iPhone; CPU OS_6; en-US)AppleWebKit/534.1 U3/3.0.0 Mobile correctly", function() {
		const results = uaParser("UCWEB/3.0 (iPhone; CPU OS_6; en-US)AppleWebKit/534.1 U3/3.0.0 Mobile");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses UCWEB/2.0 (Linux; U; Opera Mini/7.1.32052/30.2697; en-US; GT-S5302) U2/1.0.0 UCBrowser/9.3.0.440 Mobile correctly", function() {
		const results = uaParser("UCWEB/2.0 (Linux; U; Opera Mini/7.1.32052/30.2697; en-US; GT-S5302) U2/1.0.0 UCBrowser/9.3.0.440 Mobile");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses IUC(U;iOS 5.1.1;Zh-cn;320*480;)/UCWEB7.9.0.94/41/997 correctly", function() {
		const results = uaParser("IUC(U;iOS 5.1.1;Zh-cn;320*480;)/UCWEB7.9.0.94/41/997");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Nokia5320di/UCWEB8.0.3.99/28/999 correctly", function() {
		const results = uaParser("Nokia5320di/UCWEB8.0.3.99/28/999");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Nokia201/2.0 (11.21) Profile/MIDP-2.1 Configuration/CLDC-1.1 Mozilla/5.0 (Java; U; en-us; nokia201) UCBrowser8.3.0.154/70/355/UCWEB Mobile correctly", function() {
		const results = uaParser("Nokia201/2.0 (11.21) Profile/MIDP-2.1 Configuration/CLDC-1.1 Mozilla/5.0 (Java; U; en-us; nokia201) UCBrowser8.3.0.154/70/355/UCWEB Mobile");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (S60V5; U; en-us; NokiaC5-03) AppleWebKit/530.13 (KHTML, like Gecko) UCBrowser/8.7.0.218/50/352/UCWEB Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (S60V5; U; en-us; NokiaC5-03) AppleWebKit/530.13 (KHTML, like Gecko) UCBrowser/8.7.0.218/50/352/UCWEB Mobile");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/50.0.2661.102 UBrowser/5.7.14488.1025 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/50.0.2661.102 UBrowser/5.7.14488.1025 Safari/537.36");
		assert.strictEqual(results.family, "UC Browser", msg("results.family", results.family, "UC Browser"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "14488" || "", msg("results.patch", results.patch || "", "14488" || ""));
	});
	it("parses Alcatel-OH5/1.0 UP.Browser/6.1.0.7.7 (GUI) MMP/1.0 correctly", function() {
		const results = uaParser("Alcatel-OH5/1.0 UP.Browser/6.1.0.7.7 (GUI) MMP/1.0");
		assert.strictEqual(results.family, "UP.Browser", msg("results.family", results.family, "UP.Browser"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/534+ (KHTML, like Gecko) Version/5.1.1 Safari/534.51.22 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/534+ (KHTML, like Gecko) Version/5.1.1 Safari/534.51.22");
		assert.strictEqual(results.family, "WebKit Nightly", msg("results.family", results.family, "WebKit Nightly"));
		assert.strictEqual(results.major || "", "534", msg("results.major", results.major || "", "534"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.1+ (KHTML, like Gecko) Version/5.1.1 Safari/534.51.22 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.1+ (KHTML, like Gecko) Version/5.1.1 Safari/534.51.22");
		assert.strictEqual(results.family, "WebKit Nightly", msg("results.family", results.family, "WebKit Nightly"));
		assert.strictEqual(results.major || "", "537", msg("results.major", results.major || "", "537"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_7_4) AppleWebKit/537.8+ (KHTML, like Gecko) Version/6.0 Safari/536.25 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_7_4) AppleWebKit/537.8+ (KHTML, like Gecko) Version/6.0 Safari/536.25");
		assert.strictEqual(results.family, "WebKit Nightly", msg("results.family", results.family, "WebKit Nightly"));
		assert.strictEqual(results.major || "", "537", msg("results.major", results.major || "", "537"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (hp-tablet; Linux; hpwOS/3.0.0; U; en-US) AppleWebKit/534.6 (KHTML, like Gecko) wOSBrowser/233.58 Safari/534.6 TouchPad/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (hp-tablet; Linux; hpwOS/3.0.0; U; en-US) AppleWebKit/534.6 (KHTML, like Gecko) wOSBrowser/233.58 Safari/534.6 TouchPad/1.0");
		assert.strictEqual(results.family, "webOS Browser", msg("results.family", results.family, "webOS Browser"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (webOS/1.2; U; en-US) AppleWebKit/525.27.1 (KHTML, like Gecko) Version/1.0 Safari/525.27.1 Desktop/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (webOS/1.2; U; en-US) AppleWebKit/525.27.1 (KHTML, like Gecko) Version/1.0 Safari/525.27.1 Desktop/1.0");
		assert.strictEqual(results.family, "webOS Browser", msg("results.family", results.family, "webOS Browser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (hp-tablet; Linux; hpwOS/3.0.5; U; en-US) AppleWebKit/534.6 (KHTML, like Gecko) wOSBrowser/234.83 Safari/534.6 TouchPad/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (hp-tablet; Linux; hpwOS/3.0.5; U; en-US) AppleWebKit/534.6 (KHTML, like Gecko) wOSBrowser/234.83 Safari/534.6 TouchPad/1.0");
		assert.strictEqual(results.family, "webOS Browser", msg("results.family", results.family, "webOS Browser"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; nl-NL) AppleWebKit/534.3 (KHTML, like Gecko) WeTab-Browser Safari/534.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; nl-NL) AppleWebKit/534.3 (KHTML, like Gecko) WeTab-Browser Safari/534.3");
		assert.strictEqual(results.family, "WeTab", msg("results.family", results.family, "WeTab"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.4 (KHTML, like Gecko) Chrome/22.0.1104.222 YaBrowser/1.5.1104.222 Safari/537.4 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.4 (KHTML, like Gecko) Chrome/22.0.1104.222 YaBrowser/1.5.1104.222 Safari/537.4");
		assert.strictEqual(results.family, "Yandex Browser", msg("results.family", results.family, "Yandex Browser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1104" || "", msg("results.patch", results.patch || "", "1104" || ""));
	});
	it("parses Mozilla/5.0 YottaaMonitor; correctly", function() {
		const results = uaParser("Mozilla/5.0 YottaaMonitor;");
		assert.strictEqual(results.family, "YottaaMonitor", msg("results.family", results.family, "YottaaMonitor"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X) AppleWebKit/534.34 (KHTML, like Gecko) PhantomJS/1.6.0 Safari/534.34 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X) AppleWebKit/534.34 (KHTML, like Gecko) PhantomJS/1.6.0 Safari/534.34");
		assert.strictEqual(results.family, "PhantomJS", msg("results.family", results.family, "PhantomJS"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (IE 11.0; Windows NT 6.3; Trident/7.0; .NET4.0E; .NET4.0C; rv:11.0) like Gecko correctly", function() {
		const results = uaParser("Mozilla/5.0 (IE 11.0; Windows NT 6.3; Trident/7.0; .NET4.0E; .NET4.0C; rv:11.0) like Gecko");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 9.0; AOL 9.7; AOLBuild 4343.19; Windows NT 6.1; WOW64; Trident/5.0; FunWebProducts) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 9.0; AOL 9.7; AOLBuild 4343.19; Windows NT 6.1; WOW64; Trident/5.0; FunWebProducts)");
		assert.strictEqual(results.family, "AOL", msg("results.family", results.family, "AOL"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "4343" || "", msg("results.patch", results.patch || "", "4343" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.3; Win64; x64; Trident/7.0; rv:11.0) like Gecko correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.3; Win64; x64; Trident/7.0; rv:11.0) like Gecko");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HbbTV/1.1.1 (;Samsung;SmartTV2013;T-FXPDEUC-1102.2;;) WebKit correctly", function() {
		const results = uaParser("HbbTV/1.1.1 (;Samsung;SmartTV2013;T-FXPDEUC-1102.2;;) WebKit");
		assert.strictEqual(results.family, "HbbTV", msg("results.family", results.family, "HbbTV"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses HbbTV/1.2.1 (;Panasonic;VIERA 2013;3.672;4101-0003 0002-0000;) correctly", function() {
		const results = uaParser("HbbTV/1.2.1 (;Panasonic;VIERA 2013;3.672;4101-0003 0002-0000;)");
		assert.strictEqual(results.family, "HbbTV", msg("results.family", results.family, "HbbTV"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses HbbTV/1.1.1 (;;;;;) firetv-firefox-plugin 1.1.20 correctly", function() {
		const results = uaParser("HbbTV/1.1.1 (;;;;;) firetv-firefox-plugin 1.1.20");
		assert.strictEqual(results.family, "HbbTV", msg("results.family", results.family, "HbbTV"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64; Trident/7.0; rv:11.0) like Gecko/20100101 Firefox/12.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64; Trident/7.0; rv:11.0) like Gecko/20100101 Firefox/12.0");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/40.0.2214.89 Vivaldi/1.0.83.38 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/40.0.2214.89 Vivaldi/1.0.83.38 Safari/537.36");
		assert.strictEqual(results.family, "Vivaldi", msg("results.family", results.family, "Vivaldi"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "83" || "", msg("results.patch", results.patch || "", "83" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/42.0.2311.135 Safari/537.36 Edge/12.9600 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/42.0.2311.135 Safari/537.36 Edge/12.9600");
		assert.strictEqual(results.family, "Edge", msg("results.family", results.family, "Edge"));
		assert.strictEqual(results.major || "", "12", msg("results.major", results.major || "", "12"));
		assert.strictEqual(results.minor || "", "9600", msg("results.minor", results.minor || "", "9600"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (Vodafone/1.0/LG-GU280/v10a Browser/Obigo-Q7.3 MMS/LG-MMS-V1.0/1.2 Java/ASVM/1.1 Profile/MIDP-2.1 Configuration/CLDC-1.1) correctly", function() {
		const results = uaParser("Mozilla/4.0 (Vodafone/1.0/LG-GU280/v10a Browser/Obigo-Q7.3 MMS/LG-MMS-V1.0/1.2 Java/ASVM/1.1 Profile/MIDP-2.1 Configuration/CLDC-1.1)");
		assert.strictEqual(results.family, "Obigo", msg("results.family", results.family, "Obigo"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows CE; IEMobile 6.12) Vodafone/1.0/HTC_Elf/1.11.164.2 correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows CE; IEMobile 6.12) Vodafone/1.0/HTC_Elf/1.11.164.2");
		assert.strictEqual(results.family, "IE Mobile", msg("results.family", results.family, "IE Mobile"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "12", msg("results.minor", results.minor || "", "12"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows CE; IEMobile 8.12; MSIEMobile 6.0) Vodafone/1.0/HTC_HD2/1.44.162.6 (70494) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows CE; IEMobile 8.12; MSIEMobile 6.0) Vodafone/1.0/HTC_HD2/1.44.162.6 (70494)");
		assert.strictEqual(results.family, "IE Mobile", msg("results.family", results.family, "IE Mobile"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "12", msg("results.minor", results.minor || "", "12"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Mobile; Windows Phone 8.1; Android 4.0; ARM; Trident/7.0; Touch; rv:11.0; IEMobile/11.0; NOKIA; Lumia 920) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Mobile; Windows Phone 8.1; Android 4.0; ARM; Trident/7.0; Touch; rv:11.0; IEMobile/11.0; NOKIA; Lumia 920)");
		assert.strictEqual(results.family, "IE Mobile", msg("results.family", results.family, "IE Mobile"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (SymbianOS/9.2; U; Series60/3.1 Vodafone/1.0/SamsungSGHi560/I560AEHB1 Profile/MIDP-2.0 Configuration/CLDC-1.1 ) AppleWebKit/413 (KHTML, like Gecko) Safari/413 correctly", function() {
		const results = uaParser("Mozilla/5.0 (SymbianOS/9.2; U; Series60/3.1 Vodafone/1.0/SamsungSGHi560/I560AEHB1 Profile/MIDP-2.0 Configuration/CLDC-1.1 ) AppleWebKit/413 (KHTML, like Gecko) Safari/413");
		assert.strictEqual(results.family, "Nokia OSS Browser", msg("results.family", results.family, "Nokia OSS Browser"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Vodafone/1.0/LG-KC910/V08h Browser/Teleca-Q7.1 MMS/LG-MMS-V1.0/1.2 MediaPlayer/LGPlayer/1.0 Java/ASVM/1.1 Profile/MIDP-2.1 Configuration/CLDC-1.1) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Vodafone/1.0/LG-KC910/V08h Browser/Teleca-Q7.1 MMS/LG-MMS-V1.0/1.2 MediaPlayer/LGPlayer/1.0 Java/ASVM/1.1 Profile/MIDP-2.1 Configuration/CLDC-1.1)");
		assert.strictEqual(results.family, "Teleca Browser", msg("results.family", results.family, "Teleca Browser"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Vodafone/1.0/0Vodafone715/B116 Browser/Obigo-Browser/Q04A MMS/Obigo-MMS/Q04A SyncML/HW-SyncML/1.0 Java/QVM/4.1 Profile/MIDP-2.0 Configuration/CLDC-1.1 correctly", function() {
		const results = uaParser("Vodafone/1.0/0Vodafone715/B116 Browser/Obigo-Browser/Q04A MMS/Obigo-MMS/Q04A SyncML/HW-SyncML/1.0 Java/QVM/4.1 Profile/MIDP-2.0 Configuration/CLDC-1.1");
		assert.strictEqual(results.family, "Obigo", msg("results.family", results.family, "Obigo"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (Android 1.6; Linux; Opera Mobi/ADR-1107051709; U; en) Presto/2.8.149 Version/11.10 correctly", function() {
		const results = uaParser("Opera/9.80 (Android 1.6; Linux; Opera Mobi/ADR-1107051709; U; en) Presto/2.8.149 Version/11.10");
		assert.strictEqual(results.family, "Opera Mobile", msg("results.family", results.family, "Opera Mobile"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (Windows Mobile; WCE; Opera Mobi/WMD-50430; U; en) Presto/2.4.13 Version/10.00 correctly", function() {
		const results = uaParser("Opera/9.80 (Windows Mobile; WCE; Opera Mobi/WMD-50430; U; en) Presto/2.4.13 Version/10.00");
		assert.strictEqual(results.family, "Opera Mobile", msg("results.family", results.family, "Opera Mobile"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "00", msg("results.minor", results.minor || "", "00"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses (Opera) Vodafone/1.0/HPiPAQDataMessenger/1.00.00 Browser/Opera/9.5 Profile/MIDP-2.0 Configuration/CLDC-1.1 Opera/9.5 (Microsoft Windows; PPC; Opera Mobi/15142; U; en) correctly", function() {
		const results = uaParser("(Opera) Vodafone/1.0/HPiPAQDataMessenger/1.00.00 Browser/Opera/9.5 Profile/MIDP-2.0 Configuration/CLDC-1.1 Opera/9.5 (Microsoft Windows; PPC; Opera Mobi/15142; U; en)");
		assert.strictEqual(results.family, "Opera Mobile", msg("results.family", results.family, "Opera Mobile"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HTC MAX 4G Opera/9.5 (Microsoft Windows; PPC; Opera Mobi/1409; U; ru) correctly", function() {
		const results = uaParser("HTC MAX 4G Opera/9.5 (Microsoft Windows; PPC; Opera Mobi/1409; U; ru)");
		assert.strictEqual(results.family, "Opera Mobile", msg("results.family", results.family, "Opera Mobile"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses iBrowser/3.0/Mozilla/5.0 (Linux; U; Android 2.3.6; yy-yy; Karbonn A2 Build/GRK39F) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 correctly", function() {
		const results = uaParser("iBrowser/3.0/Mozilla/5.0 (Linux; U; Android 2.3.6; yy-yy; Karbonn A2 Build/GRK39F) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Series40; NokiaC3-01/05.60; Profile/MIDP-2.1 Configuration/CLDC-1.1) Gecko/20100401 S40OviBrowser/2.2.0.0.31 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Series40; NokiaC3-01/05.60; Profile/MIDP-2.1 Configuration/CLDC-1.1) Gecko/20100401 S40OviBrowser/2.2.0.0.31");
		assert.strictEqual(results.family, "Ovi Browser", msg("results.family", results.family, "Ovi Browser"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses iRAPP/1.16.0 NokiaN95_8GB/31.0.015; Series60/3.1 Profile/MIDP-2.0 Configuration/CLDC-1. correctly", function() {
		const results = uaParser("iRAPP/1.16.0 NokiaN95_8GB/31.0.015; Series60/3.1 Profile/MIDP-2.0 Configuration/CLDC-1.");
		assert.strictEqual(results.family, "iRAPP", msg("results.family", results.family, "iRAPP"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "16", msg("results.minor", results.minor || "", "16"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses iRAPP/1.3.0 Nokia5230/50.0.101 Series60/5.0 Profile/MIDP-2.1 Configuration/CLDC-1.1 3gpp-gba correctly", function() {
		const results = uaParser("iRAPP/1.3.0 Nokia5230/50.0.101 Series60/5.0 Profile/MIDP-2.1 Configuration/CLDC-1.1 3gpp-gba");
		assert.strictEqual(results.family, "iRAPP", msg("results.family", results.family, "iRAPP"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses iRAPP/3.5.0 NokiaN8-00/111.040.1511 Series60/5.3 Profile/MIDP-2.1 Configuration/CLDC-1.1 3gpp-gba correctly", function() {
		const results = uaParser("iRAPP/3.5.0 NokiaN8-00/111.040.1511 Series60/5.3 Profile/MIDP-2.1 Configuration/CLDC-1.1 3gpp-gba");
		assert.strictEqual(results.family, "iRAPP", msg("results.family", results.family, "iRAPP"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LG-GD710/V10f; Mozilla/5.0 (Profile/MIDP-2.0 Configuration/CLDC-1.1; Opera Mini/att/4.2.14812; U; en) Opera 9.50 correctly", function() {
		const results = uaParser("LG-GD710/V10f; Mozilla/5.0 (Profile/MIDP-2.0 Configuration/CLDC-1.1; Opera Mini/att/4.2.14812; U; en) Opera 9.50");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "14812" || "", msg("results.patch", results.patch || "", "14812" || ""));
	});
	it("parses SAMSUNG-SGH-A897/A897UCJC1; Mozilla/5.0 (Profile/MIDP-2.0 Configuration/CLDC-1.1; Opera Mini/att/4.2.15304; U; fr-US) Opera 9.50 correctly", function() {
		const results = uaParser("SAMSUNG-SGH-A897/A897UCJC1; Mozilla/5.0 (Profile/MIDP-2.0 Configuration/CLDC-1.1; Opera Mini/att/4.2.15304; U; fr-US) Opera 9.50");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "15304" || "", msg("results.patch", results.patch || "", "15304" || ""));
	});
	it("parses MQQBrowser/20 (Linux; U; 2.3.3; en-us; HTC Desire S Build/GRI40;480*800) correctly", function() {
		const results = uaParser("MQQBrowser/20 (Linux; U; 2.3.3; en-us; HTC Desire S Build/GRI40;480*800)");
		assert.strictEqual(results.family, "QQ Browser Mobile", msg("results.family", results.family, "QQ Browser Mobile"));
		assert.strictEqual(results.major || "", "20", msg("results.major", results.major || "", "20"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MQQBrowser/Mini2.8 (ZTE-X990/X990_V2_Z12_ESFR_D18F100) correctly", function() {
		const results = uaParser("MQQBrowser/Mini2.8 (ZTE-X990/X990_V2_Z12_ESFR_D18F100)");
		assert.strictEqual(results.family, "QQ Browser Mini", msg("results.family", results.family, "QQ Browser Mini"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MQQBrowser/Mini3.1 (SonyEricssonJ105i/R1HA035) correctly", function() {
		const results = uaParser("MQQBrowser/Mini3.1 (SonyEricssonJ105i/R1HA035)");
		assert.strictEqual(results.family, "QQ Browser Mini", msg("results.family", results.family, "QQ Browser Mini"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses QQBrowser/14 (Linux; U; 2.2.2; en-us; Motorola XT316 BUILD/FRG83G) Mobile/0050 correctly", function() {
		const results = uaParser("QQBrowser/14 (Linux; U; 2.2.2; en-us; Motorola XT316 BUILD/FRG83G) Mobile/0050");
		assert.strictEqual(results.family, "QQ Browser", msg("results.family", results.family, "QQ Browser"));
		assert.strictEqual(results.major || "", "14", msg("results.major", results.major || "", "14"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Dolphin 6.5.1 (iPad; iPhone OS 6.1.3; de_DE) correctly", function() {
		const results = uaParser("Dolphin 6.5.1 (iPad; iPhone OS 6.1.3; de_DE)");
		assert.strictEqual(results.family, "Dolphin", msg("results.family", results.family, "Dolphin"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Dolphin 7.4 (iPhone; iPhone OS 7.0.2; de_DE) correctly", function() {
		const results = uaParser("Dolphin 7.4 (iPhone; iPhone OS 7.0.2; de_DE)");
		assert.strictEqual(results.family, "Dolphin", msg("results.family", results.family, "Dolphin"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Dolphin 7.5.1 (iPhone; iPhone OS 7.0.3; de_DE) correctly", function() {
		const results = uaParser("Dolphin 7.5.1 (iPhone; iPhone OS 7.0.3; de_DE)");
		assert.strictEqual(results.family, "Dolphin", msg("results.family", results.family, "Dolphin"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.1-update1; zh-cn; XT701 Build/STCU_31.05.4) AppleWebKit/530.17 (KHTML, like Gecko) Version/4.0 Mobile Safari/530.17 DolphinHDCN/7.0.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.1-update1; zh-cn; XT701 Build/STCU_31.05.4) AppleWebKit/530.17 (KHTML, like Gecko) Version/4.0 Mobile Safari/530.17 DolphinHDCN/7.0.1");
		assert.strictEqual(results.family, "Dolphin", msg("results.family", results.family, "Dolphin"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.2; en-gb; GT-P1000 Build/FROYO) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 DolphinHDCN/6.3.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.2; en-gb; GT-P1000 Build/FROYO) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 DolphinHDCN/6.3.1");
		assert.strictEqual(results.family, "Dolphin", msg("results.family", results.family, "Dolphin"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.1.1; en-us; Nexus 7 Build/JRO03D) AppleWebKit/534.30 (KHTML, like Gecko) Dolphin/INT-1.0.4 Mobile Safari/534.30 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.1.1; en-us; Nexus 7 Build/JRO03D) AppleWebKit/534.30 (KHTML, like Gecko) Dolphin/INT-1.0.4 Mobile Safari/534.30");
		assert.strictEqual(results.family, "Dolphin", msg("results.family", results.family, "Dolphin"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_6_3; en-us) AppleWebKit/533.16 (KHTML, like Gecko) Version/5.0 Safari/533.16 DolphinHDCN/6.1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X 10_6_3; en-us) AppleWebKit/533.16 (KHTML, like Gecko) Version/5.0 Safari/533.16 DolphinHDCN/6.1.0");
		assert.strictEqual(results.family, "Dolphin", msg("results.family", results.family, "Dolphin"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0(miniGUI/3.x; U; Linux i686; en-US) AppleWebKit/534.26 (KHTML, like Gecko) mDolphin/3.0 chrome/10.0 Safria/534.26 correctly", function() {
		const results = uaParser("Mozilla/5.0(miniGUI/3.x; U; Linux i686; en-US) AppleWebKit/534.26 (KHTML, like Gecko) mDolphin/3.0 chrome/10.0 Safria/534.26");
		assert.strictEqual(results.family, "mDolphin", msg("results.family", results.family, "mDolphin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0(miniGUI/3.x; U; Linux i686; en; ) AppleWebKit/533.9.0 (KHTML, like Gecko) mDolphin/3.0.0 Safria/533.9.0 correctly", function() {
		const results = uaParser("Mozilla/5.0(miniGUI/3.x; U; Linux i686; en; ) AppleWebKit/533.9.0 (KHTML, like Gecko) mDolphin/3.0.0 Safria/533.9.0");
		assert.strictEqual(results.family, "mDolphin", msg("results.family", results.family, "mDolphin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses CFNetwork, iPhone OS 5.1.1, iPhone4,1 correctly", function() {
		const results = uaParser("CFNetwork, iPhone OS 5.1.1, iPhone4,1");
		assert.strictEqual(results.family, "CFNetwork", msg("results.family", results.family, "CFNetwork"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CFNetwork, iPhone OS 7.0.4, iPhone5,2 correctly", function() {
		const results = uaParser("CFNetwork, iPhone OS 7.0.4, iPhone5,2");
		assert.strictEqual(results.family, "CFNetwork", msg("results.family", results.family, "CFNetwork"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CFNetwork, iPhone OS 7.0, iPhone4,1 correctly", function() {
		const results = uaParser("CFNetwork, iPhone OS 7.0, iPhone4,1");
		assert.strictEqual(results.family, "CFNetwork", msg("results.family", results.family, "CFNetwork"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Safari5530.17 CFNetwork/438.12 Darwin/9.7.0 (i386) (Macmini2,1) correctly", function() {
		const results = uaParser("Safari5530.17 CFNetwork/438.12 Darwin/9.7.0 (i386) (Macmini2,1)");
		assert.strictEqual(results.family, "CFNetwork", msg("results.family", results.family, "CFNetwork"));
		assert.strictEqual(results.major || "", "438", msg("results.major", results.major || "", "438"));
		assert.strictEqual(results.minor || "", "12", msg("results.minor", results.minor || "", "12"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Safari/6533.18.5 CFNetwork/454.9.8 Darwin/10.4.0 (i386) (MacBookPro7,1) correctly", function() {
		const results = uaParser("Safari/6533.18.5 CFNetwork/454.9.8 Darwin/10.4.0 (i386) (MacBookPro7,1)");
		assert.strictEqual(results.family, "Safari", msg("results.family", results.family, "Safari"));
		assert.strictEqual(results.major || "", "6533", msg("results.major", results.major || "", "6533"));
		assert.strictEqual(results.minor || "", "18", msg("results.minor", results.minor || "", "18"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Safari/7536.30.1 CFNetwork/520.5.1 Darwin/11.4.2 (i386) (MacBook3,1) correctly", function() {
		const results = uaParser("Safari/7536.30.1 CFNetwork/520.5.1 Darwin/11.4.2 (i386) (MacBook3,1)");
		assert.strictEqual(results.family, "Safari", msg("results.family", results.family, "Safari"));
		assert.strictEqual(results.major || "", "7536", msg("results.major", results.major || "", "7536"));
		assert.strictEqual(results.minor || "", "30", msg("results.minor", results.minor || "", "30"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Reader Notifier/5 CFNetwork/596.3.3 Darwin/12.3.0 (x86_64) (MacBookPro7,1) correctly", function() {
		const results = uaParser("Reader Notifier/5 CFNetwork/596.3.3 Darwin/12.3.0 (x86_64) (MacBookPro7,1)");
		assert.strictEqual(results.family, "Reader Notifier", msg("results.family", results.family, "Reader Notifier"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Safari/9537.71 CFNetwork/673.0.2 Darwin/13.0.1 (x86_64) (MacBookPro11,1) correctly", function() {
		const results = uaParser("Safari/9537.71 CFNetwork/673.0.2 Darwin/13.0.1 (x86_64) (MacBookPro11,1)");
		assert.strictEqual(results.family, "Safari", msg("results.family", results.family, "Safari"));
		assert.strictEqual(results.major || "", "9537", msg("results.major", results.major || "", "9537"));
		assert.strictEqual(results.minor || "", "71", msg("results.minor", results.minor || "", "71"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DEPoker-iPad/1.0.2 CFNetwork/548.1.4 Darwin/11.0.0 correctly", function() {
		const results = uaParser("DEPoker-iPad/1.0.2 CFNetwork/548.1.4 Darwin/11.0.0");
		assert.strictEqual(results.family, "DEPoker", msg("results.family", results.family, "DEPoker"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses JDSports-iPad/1.1 CFNetwork/672.0.8 Darwin/14.0.0 correctly", function() {
		const results = uaParser("JDSports-iPad/1.1 CFNetwork/672.0.8 Darwin/14.0.0");
		assert.strictEqual(results.family, "JDSports", msg("results.family", results.family, "JDSports"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AngryBirdsBlack-iPhone/1.1.0 CFNetwork/548.1.4 Darwin/11.0.0 correctly", function() {
		const results = uaParser("AngryBirdsBlack-iPhone/1.1.0 CFNetwork/548.1.4 Darwin/11.0.0");
		assert.strictEqual(results.family, "AngryBirdsBlack", msg("results.family", results.family, "AngryBirdsBlack"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Bing for iPad/1.1.2 CFNetwork/485.13.9 Darwin/11.0.0 correctly", function() {
		const results = uaParser("Bing for iPad/1.1.2 CFNetwork/485.13.9 Darwin/11.0.0");
		assert.strictEqual(results.family, "Bing for iPad", msg("results.family", results.family, "Bing for iPad"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses NightstandPaid-iPad/1.3.1 CFNetwork/548.1.4 Darwin/11.0.0 correctly", function() {
		const results = uaParser("NightstandPaid-iPad/1.3.1 CFNetwork/548.1.4 Darwin/11.0.0");
		assert.strictEqual(results.family, "NightstandPaid", msg("results.family", results.family, "NightstandPaid"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Glo-De-iPad/1.4.7 CFNetwork/672.0.2 Darwin/14.0.0 correctly", function() {
		const results = uaParser("Glo-De-iPad/1.4.7 CFNetwork/672.0.2 Darwin/14.0.0");
		assert.strictEqual(results.family, "Glo-De", msg("results.family", results.family, "Glo-De"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Island for iPhone/1.95 CFNetwork/672.0.2 Darwin/14.0.0 correctly", function() {
		const results = uaParser("Island for iPhone/1.95 CFNetwork/672.0.2 Darwin/14.0.0");
		assert.strictEqual(results.family, "Island for iPhone", msg("results.family", results.family, "Island for iPhone"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "95", msg("results.minor", results.minor || "", "95"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WormsiPhone-iPad/2.3 CFNetwork/548.1.4 Darwin/11.0.0 correctly", function() {
		const results = uaParser("WormsiPhone-iPad/2.3 CFNetwork/548.1.4 Darwin/11.0.0");
		assert.strictEqual(results.family, "WormsiPhone", msg("results.family", results.family, "WormsiPhone"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Rummy LITE iPad/2.3.0 CFNetwork/609.1.4 Darwin/13.0.0 correctly", function() {
		const results = uaParser("Rummy LITE iPad/2.3.0 CFNetwork/609.1.4 Darwin/13.0.0");
		assert.strictEqual(results.family, "Rummy LITE iPad", msg("results.family", results.family, "Rummy LITE iPad"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses MobileRSSFree-iPad/3.1 CFNetwork/467.12 Darwin/10.3.1 correctly", function() {
		const results = uaParser("MobileRSSFree-iPad/3.1 CFNetwork/467.12 Darwin/10.3.1");
		assert.strictEqual(results.family, "MobileRSSFree", msg("results.family", results.family, "MobileRSSFree"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MobileRSSFree-iPad/3.1.4 CFNetwork/485.13.9 Darwin/11.0.0 correctly", function() {
		const results = uaParser("MobileRSSFree-iPad/3.1.4 CFNetwork/485.13.9 Darwin/11.0.0");
		assert.strictEqual(results.family, "MobileRSSFree", msg("results.family", results.family, "MobileRSSFree"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses babbelIndonesian-iPad/4.0.1 CFNetwork/672.0.8 Darwin/14.0.0 correctly", function() {
		const results = uaParser("babbelIndonesian-iPad/4.0.1 CFNetwork/672.0.8 Darwin/14.0.0");
		assert.strictEqual(results.family, "babbelIndonesian", msg("results.family", results.family, "babbelIndonesian"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses WeltMobile-iPad/4.2 CFNetwork/609.1.4 Darwin/13.0.0 correctly", function() {
		const results = uaParser("WeltMobile-iPad/4.2 CFNetwork/609.1.4 Darwin/13.0.0");
		assert.strictEqual(results.family, "WeltMobile", msg("results.family", results.family, "WeltMobile"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IMPlusFull-iPad/7.9.1 CFNetwork/548.0.4 Darwin/11.0.0 correctly", function() {
		const results = uaParser("IMPlusFull-iPad/7.9.1 CFNetwork/548.0.4 Darwin/11.0.0");
		assert.strictEqual(results.family, "IMPlusFull", msg("results.family", results.family, "IMPlusFull"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Cooliris/1.3 CFNetwork/342.1 Darwin/9.4.1 correctly", function() {
		const results = uaParser("Cooliris/1.3 CFNetwork/342.1 Darwin/9.4.1");
		assert.strictEqual(results.family, "Cooliris", msg("results.family", results.family, "Cooliris"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Poof/1.0 CFNetwork/485.12.7 Darwin/10.4.0 correctly", function() {
		const results = uaParser("Poof/1.0 CFNetwork/485.12.7 Darwin/10.4.0");
		assert.strictEqual(results.family, "Poof", msg("results.family", results.family, "Poof"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Parking Mania Free/1.9.5.0 CFNetwork/548.0.4 Darwin/11.0.0 correctly", function() {
		const results = uaParser("Parking Mania Free/1.9.5.0 CFNetwork/548.0.4 Darwin/11.0.0");
		assert.strictEqual(results.family, "Parking Mania Free", msg("results.family", results.family, "Parking Mania Free"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Planet Boing!/1.4.8 CFNetwork/609.1.4 Darwin/13.0.0 correctly", function() {
		const results = uaParser("Planet Boing!/1.4.8 CFNetwork/609.1.4 Darwin/13.0.0");
		assert.strictEqual(results.family, "Planet Boing!", msg("results.family", results.family, "Planet Boing!"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses PlayTube/1.7 CFNetwork/672.0.2 Darwin/14.0.0 correctly", function() {
		const results = uaParser("PlayTube/1.7 CFNetwork/672.0.2 Darwin/14.0.0");
		assert.strictEqual(results.family, "PlayTube", msg("results.family", results.family, "PlayTube"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.2.1; es-us) AppleWebKit/534.12 (KHTML, like Gecko) Puffin/1.3.2913S Mobile Safari/534.12 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.2.1; es-us) AppleWebKit/534.12 (KHTML, like Gecko) Puffin/1.3.2913S Mobile Safari/534.12");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "2913" || "", msg("results.patch", results.patch || "", "2913" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 3.2.1; es-es) AppleWebKit/534.35 (KHTML, like Gecko) Chrome/11.0.696.65 Safari/534.35 Puffin/2.0.6440M Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 3.2.1; es-es) AppleWebKit/534.35 (KHTML, like Gecko) Chrome/11.0.696.65 Safari/534.35 Puffin/2.0.6440M Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "6440" || "", msg("results.patch", results.patch || "", "6440" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 3.2; en-gb) AppleWebKit/534.35 (KHTML, like Gecko) Chrome/11.0.696.65 Safari/534.35 Puffin/2.0.5932M Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 3.2; en-gb) AppleWebKit/534.35 (KHTML, like Gecko) Chrome/11.0.696.65 Safari/534.35 Puffin/2.0.5932M Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "5932" || "", msg("results.patch", results.patch || "", "5932" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.1.1; en-us) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/2.9909AT Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.1.1; en-us) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/2.9909AT Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "9909", msg("results.minor", results.minor || "", "9909"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.2.2; de-de) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558AT Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.2.2; de-de) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558AT Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "11558", msg("results.minor", results.minor || "", "11558"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.2.2; en-us) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558AP Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.2.2; en-us) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558AP Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "11558", msg("results.minor", results.minor || "", "11558"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686; th-TH@calendar=gregorian) AppleWebKit/534.12 (KHTML, like Gecko) Puffin/1.3.2665MS Safari/534.12 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686; th-TH@calendar=gregorian) AppleWebKit/534.12 (KHTML, like Gecko) Puffin/1.3.2665MS Safari/534.12");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "2665" || "", msg("results.patch", results.patch || "", "2665" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; ar-AE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.10990IT correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; ar-AE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.10990IT");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "10990", msg("results.minor", results.minor || "", "10990"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; ar-SA) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11546IP correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; ar-SA) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11546IP");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "11546", msg("results.minor", results.minor || "", "11546"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; ar-ae) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/2.10977AP correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; ar-ae) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/2.10977AP");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "10977", msg("results.minor", results.minor || "", "10977"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; de-at) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/2.10977AT correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; de-at) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/2.10977AT");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "10977", msg("results.minor", results.minor || "", "10977"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU OS 7_0_6 like Mac OS X; de-DE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558IT Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU OS 7_0_6 like Mac OS X; de-DE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558IT Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "11558", msg("results.minor", results.minor || "", "11558"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; U; CPU iPhone OS 7_0_6 like Mac OS X; de-DE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558IP Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; U; CPU iPhone OS 7_0_6 like Mac OS X; de-DE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.11558IP Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "11558", msg("results.minor", results.minor || "", "11558"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPod; U; CPU iPhone OS 5_1_1 like Mac OS X; de-DE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.9174IP Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPod; U; CPU iPhone OS 5_1_1 like Mac OS X; de-DE) AppleWebKit/534.35 (KHTML, like Gecko)  Chrome/11.0.696.65 Safari/534.35 Puffin/3.9174IP Mobile");
		assert.strictEqual(results.family, "Puffin", msg("results.family", results.family, "Puffin"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "9174", msg("results.minor", results.minor || "", "9174"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MacAppStore/2.0 (Macintosh; OS X 10.10.2; 14C81f) AppleWebKit/0600.3.10.2 correctly", function() {
		const results = uaParser("MacAppStore/2.0 (Macintosh; OS X 10.10.2; 14C81f) AppleWebKit/0600.3.10.2");
		assert.strictEqual(results.family, "MacAppStore", msg("results.family", results.family, "MacAppStore"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses iTunes/12.0.1 (Macintosh; OS X 10.9.2) AppleWebKit/537.74.9 correctly", function() {
		const results = uaParser("iTunes/12.0.1 (Macintosh; OS X 10.9.2) AppleWebKit/537.74.9");
		assert.strictEqual(results.family, "iTunes", msg("results.family", results.family, "iTunes"));
		assert.strictEqual(results.major || "", "12", msg("results.major", results.major || "", "12"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Ant/Ant-Nutch-1.1 (Ant Nutch Crawler; http://www.ant.com; crawler@ant.com) correctly", function() {
		const results = uaParser("Ant/Ant-Nutch-1.1 (Ant Nutch Crawler; http://www.ant.com; crawler@ant.com)");
		assert.strictEqual(results.family, "Ant-Nutch", msg("results.family", results.family, "Ant-Nutch"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Axtaris Web Crawler/Axtaris-1.0.1 (Hello from Axtaris.org | Web-Crawler; http://axtaris.org; crawler at axtaris dot org) correctly", function() {
		const results = uaParser("Axtaris Web Crawler/Axtaris-1.0.1 (Hello from Axtaris.org | Web-Crawler; http://axtaris.org; crawler at axtaris dot org)");
		assert.strictEqual(results.family, "Axtaris", msg("results.family", results.family, "Axtaris"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.3; WOW64; Trident/7.0; rv:11.0) like Gecko PTST/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.3; WOW64; Trident/7.0; rv:11.0) like Gecko PTST/1.0");
		assert.strictEqual(results.family, "WebPageTest.org bot", msg("results.family", results.family, "WebPageTest.org bot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/64.0.3282.0 Safari/537.36 PTST/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/64.0.3282.0 Safari/537.36 PTST/1.0");
		assert.strictEqual(results.family, "WebPageTest.org bot", msg("results.family", results.family, "WebPageTest.org bot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 6.0.1; Moto G (4) Build/MPJ24.139-64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.146 Mobile Safari/537.36 PTST/180521.140508 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 6.0.1; Moto G (4) Build/MPJ24.139-64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.146 Mobile Safari/537.36 PTST/180521.140508");
		assert.strictEqual(results.family, "WebPageTest.org bot", msg("results.family", results.family, "WebPageTest.org bot"));
		assert.strictEqual(results.major || "", "180521", msg("results.major", results.major || "", "180521"));
		assert.strictEqual(results.minor || "", "140508", msg("results.minor", results.minor || "", "140508"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 6.0.1; Moto G (4) Build/MPJ24.139-64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.81 Mobile Safari/537.36 PTST/391 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 6.0.1; Moto G (4) Build/MPJ24.139-64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.81 Mobile Safari/537.36 PTST/391");
		assert.strictEqual(results.family, "WebPageTest.org bot", msg("results.family", results.family, "WebPageTest.org bot"));
		assert.strictEqual(results.major || "", "391", msg("results.major", results.major || "", "391"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; Datanyze; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.181 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Datanyze; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/65.0.3325.181 Safari/537.36");
		assert.strictEqual(results.family, "Datanyze", msg("results.family", results.family, "Datanyze"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CazoodleBot/CazoodleBot-0.1 (CazoodleBot Crawler; http://www.cazoodle.com/cazoodlebot; cazoodlebot@cazoodle.com) correctly", function() {
		const results = uaParser("CazoodleBot/CazoodleBot-0.1 (CazoodleBot Crawler; http://www.cazoodle.com/cazoodlebot; cazoodlebot@cazoodle.com)");
		assert.strictEqual(results.family, "CazoodleBot", msg("results.family", results.family, "CazoodleBot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Isara-Search/Isara-1.0 (A non-profit web crawler operated by a charity organization.; www.isara.org; webmaster@isara.org) correctly", function() {
		const results = uaParser("Isara-Search/Isara-1.0 (A non-profit web crawler operated by a charity organization.; www.isara.org; webmaster@isara.org)");
		assert.strictEqual(results.family, "Isara", msg("results.family", results.family, "Isara"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses noobot Spider/Noobot-1.2 (noobot-bot; http://www.noobot.fr) correctly", function() {
		const results = uaParser("noobot Spider/Noobot-1.2 (noobot-bot; http://www.noobot.fr)");
		assert.strictEqual(results.family, "Noobot", msg("results.family", results.family, "Noobot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses asked/Nutch-0.8 (web crawler; http://asked.jp; epicurus at gmail dot com) correctly", function() {
		const results = uaParser("asked/Nutch-0.8 (web crawler; http://asked.jp; epicurus at gmail dot com)");
		assert.strictEqual(results.family, "Nutch", msg("results.family", results.family, "Nutch"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses nutch crawler/Nutch-2.2 correctly", function() {
		const results = uaParser("nutch crawler/Nutch-2.2");
		assert.strictEqual(results.family, "Nutch", msg("results.family", results.family, "Nutch"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Spider/Nutch-2.3-SNAPSHOT (Webcrawler) correctly", function() {
		const results = uaParser("Spider/Nutch-2.3-SNAPSHOT (Webcrawler)");
		assert.strictEqual(results.family, "Nutch", msg("results.family", results.family, "Nutch"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SheenBot/SheenBot-1.0.4 (Sheen web crawler) correctly", function() {
		const results = uaParser("SheenBot/SheenBot-1.0.4 (Sheen web crawler)");
		assert.strictEqual(results.family, "SheenBot", msg("results.family", results.family, "SheenBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses SaladSpoon/ShopSalad 1.0 (Search Engine crawler for ShopSalad.com; http://shopsalad.com/en/partners.html; crawler AT shopsalad.com) correctly", function() {
		const results = uaParser("SaladSpoon/ShopSalad 1.0 (Search Engine crawler for ShopSalad.com; http://shopsalad.com/en/partners.html; crawler AT shopsalad.com)");
		assert.strictEqual(results.family, "ShopSalad", msg("results.family", results.family, "ShopSalad"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TailsweepBlogCrawler/Tailsweep-2.9-SNAPSHOT (http://www.tailsweep.com/; bot at [tailsweep] dot com) correctly", function() {
		const results = uaParser("TailsweepBlogCrawler/Tailsweep-2.9-SNAPSHOT (http://www.tailsweep.com/; bot at [tailsweep] dot com)");
		assert.strictEqual(results.family, "Tailsweep", msg("results.family", results.family, "Tailsweep"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SAE/fetchurl-22wy0njxnm WordPress/3.2.1; http://1.webfront.sinaapp.com correctly", function() {
		const results = uaParser("SAE/fetchurl-22wy0njxnm WordPress/3.2.1; http://1.webfront.sinaapp.com");
		assert.strictEqual(results.family, "fetchurl", msg("results.family", results.family, "fetchurl"));
		assert.strictEqual(results.major || "", "22", msg("results.major", results.major || "", "22"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RedBot/redbot-1.0 (Rediff.com Crawler; redbot at rediff dot com) correctly", function() {
		const results = uaParser("RedBot/redbot-1.0 (Rediff.com Crawler; redbot at rediff dot com)");
		assert.strictEqual(results.family, "redbot", msg("results.family", results.family, "redbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IS Alpha/zcspider-0.1 correctly", function() {
		const results = uaParser("IS Alpha/zcspider-0.1");
		assert.strictEqual(results.family, "zcspider", msg("results.family", results.family, "zcspider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; 008/0.85; http://www.80legs.com/webcrawler.html) Gecko/2008032620 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; 008/0.85; http://www.80legs.com/webcrawler.html) Gecko/2008032620");
		assert.strictEqual(results.family, "008", msg("results.family", results.family, "008"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "85", msg("results.minor", results.minor || "", "85"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Altresium/4.6 (+http://www.altresium.com/bot.html) correctly", function() {
		const results = uaParser("Altresium/4.6 (+http://www.altresium.com/bot.html)");
		assert.strictEqual(results.family, "Altresium", msg("results.family", results.family, "Altresium"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Argus/1.1 (Nutch; http://www.simpy.com/bot.html; feedback at simpy dot com) correctly", function() {
		const results = uaParser("Argus/1.1 (Nutch; http://www.simpy.com/bot.html; feedback at simpy dot com)");
		assert.strictEqual(results.family, "Argus", msg("results.family", results.family, "Argus"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Argus/2.8.65 CFNetwork/609 Darwin/13.0.0 correctly", function() {
		const results = uaParser("Argus/2.8.65 CFNetwork/609 Darwin/13.0.0");
		assert.strictEqual(results.family, "Argus", msg("results.family", results.family, "Argus"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "65" || "", msg("results.patch", results.patch || "", "65" || ""));
	});
	it("parses DoCoMo/2.0 P05A(c100;TB;W24H15) (compatible; BaiduMobaider/1.0;  http://www.baidu.jp/spider/) correctly", function() {
		const results = uaParser("DoCoMo/2.0 P05A(c100;TB;W24H15) (compatible; BaiduMobaider/1.0;  http://www.baidu.jp/spider/)");
		assert.strictEqual(results.family, "BaiduMobaider", msg("results.family", results.family, "BaiduMobaider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BoardReader/1.0 (http://boardreader.com/info/robots.htm)-Machine1 correctly", function() {
		const results = uaParser("BoardReader/1.0 (http://boardreader.com/info/robots.htm)-Machine1");
		assert.strictEqual(results.family, "BoardReader", msg("results.family", results.family, "BoardReader"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DNSGroup/0.1 (DNS Group Crawler; http://www.dnsgroup.com/; crawler@dnsgroup.com) correctly", function() {
		const results = uaParser("DNSGroup/0.1 (DNS Group Crawler; http://www.dnsgroup.com/; crawler@dnsgroup.com)");
		assert.strictEqual(results.family, "DNSGroup", msg("results.family", results.family, "DNSGroup"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DataparkSearch/4.35-02122005 ( http://www.dataparksearch.org/) correctly", function() {
		const results = uaParser("DataparkSearch/4.35-02122005 ( http://www.dataparksearch.org/)");
		assert.strictEqual(results.family, "DataparkSearch", msg("results.family", results.family, "DataparkSearch"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "35", msg("results.minor", results.minor || "", "35"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses EDI/1.6.0 (Edacious & Intelligent Web Crawler) correctly", function() {
		const results = uaParser("EDI/1.6.0 (Edacious & Intelligent Web Crawler)");
		assert.strictEqual(results.family, "EDI", msg("results.family", results.family, "EDI"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Goodzer/2.0; crawler@goodzer.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Goodzer/2.0; crawler@goodzer.com)");
		assert.strictEqual(results.family, "Goodzer", msg("results.family", results.family, "Goodzer"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Grub/2.0 (Grub.org crawler; http://www.grub.org/; bot@grub.org) correctly", function() {
		const results = uaParser("Grub/2.0 (Grub.org crawler; http://www.grub.org/; bot@grub.org)");
		assert.strictEqual(results.family, "Grub", msg("results.family", results.family, "Grub"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses INGRID/2.0 (http://spsearch.ilse.nl/; Startpagina dochter links spider) correctly", function() {
		const results = uaParser("INGRID/2.0 (http://spsearch.ilse.nl/; Startpagina dochter links spider)");
		assert.strictEqual(results.family, "INGRID", msg("results.family", results.family, "INGRID"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses INGRID/3.0 MT (webcrawler@NOSPAMexperimental.net; http://aanmelden.ilse.nl/?aanmeld_mode=webhints) correctly", function() {
		const results = uaParser("INGRID/3.0 MT (webcrawler@NOSPAMexperimental.net; http://aanmelden.ilse.nl/?aanmeld_mode=webhints)");
		assert.strictEqual(results.family, "INGRID", msg("results.family", results.family, "INGRID"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Infohelfer/1.2.0; +http://www.infohelfer.de/crawler.php) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Infohelfer/1.2.0; +http://www.infohelfer.de/crawler.php)");
		assert.strictEqual(results.family, "Infohelfer", msg("results.family", results.family, "Infohelfer"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses LOOQ/0.1 alfa (LOOQ Crawler for european sites; http://looq.eu; root (at) looq dot eu) correctly", function() {
		const results = uaParser("LOOQ/0.1 alfa (LOOQ Crawler for european sites; http://looq.eu; root (at) looq dot eu)");
		assert.strictEqual(results.family, "LOOQ", msg("results.family", results.family, "LOOQ"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LinkedInBot/1.0 (compatible; Mozilla/5.0; Jakarta Commons-HttpClient/3.1  http://www.linkedin.com) correctly", function() {
		const results = uaParser("LinkedInBot/1.0 (compatible; Mozilla/5.0; Jakarta Commons-HttpClient/3.1  http://www.linkedin.com)");
		assert.strictEqual(results.family, "LinkedInBot", msg("results.family", results.family, "LinkedInBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; PathDefender/1.0; +http://www.pathdefender.com/help/crawler) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; PathDefender/1.0; +http://www.pathdefender.com/help/crawler)");
		assert.strictEqual(results.family, "PathDefender", msg("results.family", results.family, "PathDefender"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Peew/1.0; http://www.peew.de/crawler/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Peew/1.0; http://www.peew.de/crawler/)");
		assert.strictEqual(results.family, "Peew", msg("results.family", results.family, "Peew"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PostPost/1.0 ( http://postpo.st/crawlers) correctly", function() {
		const results = uaParser("PostPost/1.0 ( http://postpo.st/crawlers)");
		assert.strictEqual(results.family, "PostPost", msg("results.family", results.family, "PostPost"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Steeler/1.3 (http://www.tkl.iis.u-tokyo.ac.jp/~crawler/) correctly", function() {
		const results = uaParser("Steeler/1.3 (http://www.tkl.iis.u-tokyo.ac.jp/~crawler/)");
		assert.strictEqual(results.family, "Steeler", msg("results.family", results.family, "Steeler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Steeler/3.3 (http://www.tkl.iis.u-tokyo.ac.jp/~crawler/) correctly", function() {
		const results = uaParser("Steeler/3.3 (http://www.tkl.iis.u-tokyo.ac.jp/~crawler/)");
		assert.strictEqual(results.family, "Steeler", msg("results.family", results.family, "Steeler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Steeler/3.5; http://www.tkl.iis.u-tokyo.ac.jp/~crawler/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Steeler/3.5; http://www.tkl.iis.u-tokyo.ac.jp/~crawler/)");
		assert.strictEqual(results.family, "Steeler", msg("results.family", results.family, "Steeler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses VSE/1.0 (testcrawler@hotmail.com) correctly", function() {
		const results = uaParser("VSE/1.0 (testcrawler@hotmail.com)");
		assert.strictEqual(results.family, "VSE", msg("results.family", results.family, "VSE"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses VSE/1.0 (vsecrawler@hotmail.com) correctly", function() {
		const results = uaParser("VSE/1.0 (vsecrawler@hotmail.com)");
		assert.strictEqual(results.family, "VSE", msg("results.family", results.family, "VSE"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; WebCrunch/1.2; +http://webcrunch.net/crawler) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; WebCrunch/1.2; +http://webcrunch.net/crawler)");
		assert.strictEqual(results.family, "WebCrunch", msg("results.family", results.family, "WebCrunch"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WebZIP/7.0 (http://www.spidersoft.com) correctly", function() {
		const results = uaParser("WebZIP/7.0 (http://www.spidersoft.com)");
		assert.strictEqual(results.family, "WebZIP", msg("results.family", results.family, "WebZIP"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Y!J-BRI/0.0.1 crawler ( http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html ) correctly", function() {
		const results = uaParser("Y!J-BRI/0.0.1 crawler ( http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html )");
		assert.strictEqual(results.family, "Y!J-BRI", msg("results.family", results.family, "Y!J-BRI"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Y!J-BRW/1.0 crawler (http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html) correctly", function() {
		const results = uaParser("Y!J-BRW/1.0 crawler (http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html)");
		assert.strictEqual(results.family, "Y!J-BRW", msg("results.family", results.family, "Y!J-BRW"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YahooSeeker/1.0 (compatible; Mozilla 4.0; MSIE 5.5; http://help.yahoo.com/help/us/shop/merchant/) correctly", function() {
		const results = uaParser("YahooSeeker/1.0 (compatible; Mozilla 4.0; MSIE 5.5; http://help.yahoo.com/help/us/shop/merchant/)");
		assert.strictEqual(results.family, "YahooSeeker", msg("results.family", results.family, "YahooSeeker"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses envolk/1.7 ( http://www.envolk.com/envolkspider.html) correctly", function() {
		const results = uaParser("envolk/1.7 ( http://www.envolk.com/envolkspider.html)");
		assert.strictEqual(results.family, "envolk", msg("results.family", results.family, "envolk"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses sproose/1.0beta (sproose bot; http://www.sproose.com/bot.html; crawler@sproose.com) correctly", function() {
		const results = uaParser("sproose/1.0beta (sproose bot; http://www.sproose.com/bot.html; crawler@sproose.com)");
		assert.strictEqual(results.family, "sproose", msg("results.family", results.family, "sproose"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses wminer/1.2 (North America Web Crawler; http://www.wminer.com/bot; info at wminer dot com) correctly", function() {
		const results = uaParser("wminer/1.2 (North America Web Crawler; http://www.wminer.com/bot; info at wminer dot com)");
		assert.strictEqual(results.family, "wminer", msg("results.family", results.family, "wminer"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 4.0; MSIECrawler; Windows 95) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 4.0; MSIECrawler; Windows 95)");
		assert.strictEqual(results.family, "MSIECrawler", msg("results.family", results.family, "MSIECrawler"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/4.0; FDM; MSIECrawler; Media Center PC 5.0) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 9.0; Windows NT 6.1; Trident/4.0; FDM; MSIECrawler; Media Center PC 5.0)");
		assert.strictEqual(results.family, "MSIECrawler", msg("results.family", results.family, "MSIECrawler"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Apache-HttpClient/4.0-beta2 (java 1.5) correctly", function() {
		const results = uaParser("Apache-HttpClient/4.0-beta2 (java 1.5)");
		assert.strictEqual(results.family, "Apache-HttpClient", msg("results.family", results.family, "Apache-HttpClient"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.1-update1; de-de; E10i Build/2.1.1.C.0.0) AppleWebKit/530.17 (KHTML, like Gecko) Version/4.0 Mobile Safari/530.17 Google-HTTP-Java-Client/1.10.3-beta (gzip) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.1-update1; de-de; E10i Build/2.1.1.C.0.0) AppleWebKit/530.17 (KHTML, like Gecko) Version/4.0 Mobile Safari/530.17 Google-HTTP-Java-Client/1.10.3-beta (gzip)");
		assert.strictEqual(results.family, "Google-HTTP-Java-Client", msg("results.family", results.family, "Google-HTTP-Java-Client"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Google-HTTP-Java-Client/1.13.1-beta (gzip) correctly", function() {
		const results = uaParser("Google-HTTP-Java-Client/1.13.1-beta (gzip)");
		assert.strictEqual(results.family, "Google-HTTP-Java-Client", msg("results.family", results.family, "Google-HTTP-Java-Client"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "13", msg("results.minor", results.minor || "", "13"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Google-YouTubeSample/1.0 Google-HTTP-Java-Client/1.7.0-beta (gzip) Google-HTTP-Java-Client/1.7.0-beta (gzip) correctly", function() {
		const results = uaParser("Google-YouTubeSample/1.0 Google-HTTP-Java-Client/1.7.0-beta (gzip) Google-HTTP-Java-Client/1.7.0-beta (gzip)");
		assert.strictEqual(results.family, "Google-HTTP-Java-Client", msg("results.family", results.family, "Google-HTTP-Java-Client"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses JNLP/1.7.0 javaws/10.17.2.02 (<internal>) Java/1.7.0_17 correctly", function() {
		const results = uaParser("JNLP/1.7.0 javaws/10.17.2.02 (<internal>) Java/1.7.0_17");
		assert.strictEqual(results.family, "JNLP", msg("results.family", results.family, "JNLP"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses JNLP/6.0 javaws/1.6.0_14 (b08) Java/1.6.0_14 correctly", function() {
		const results = uaParser("JNLP/6.0 javaws/1.6.0_14 (b08) Java/1.6.0_14");
		assert.strictEqual(results.family, "JNLP", msg("results.family", results.family, "JNLP"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Python-urllib2) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Python-urllib2)");
		assert.strictEqual(results.family, "Python-urllib", msg("results.family", results.family, "Python-urllib"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Python-urllib/1.15 correctly", function() {
		const results = uaParser("Python-urllib/1.15");
		assert.strictEqual(results.family, "Python-urllib", msg("results.family", results.family, "Python-urllib"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "15", msg("results.minor", results.minor || "", "15"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IMVU Client/498.0 Python-urllib/2.7 correctly", function() {
		const results = uaParser("IMVU Client/498.0 Python-urllib/2.7");
		assert.strictEqual(results.family, "Python-urllib", msg("results.family", results.family, "Python-urllib"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Python-urllib/3.4 correctly", function() {
		const results = uaParser("Python-urllib/3.4");
		assert.strictEqual(results.family, "Python-urllib", msg("results.family", results.family, "Python-urllib"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TLSProber/0.8 correctly", function() {
		const results = uaParser("TLSProber/0.8");
		assert.strictEqual(results.family, "TLSProber", msg("results.family", results.family, "TLSProber"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Asynchronous WinHTTP/1.0 correctly", function() {
		const results = uaParser("Asynchronous WinHTTP/1.0");
		assert.strictEqual(results.family, "WinHTTP", msg("results.family", results.family, "WinHTTP"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 1470.net crawler correctly", function() {
		const results = uaParser("1470.net crawler");
		assert.strictEqual(results.family, "1470.net crawler", msg("results.family", results.family, "1470.net crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 50.nu/0.01 (  http://50.nu/bot.html ) correctly", function() {
		const results = uaParser("50.nu/0.01 (  http://50.nu/bot.html )");
		assert.strictEqual(results.family, "50.nu", msg("results.family", results.family, "50.nu"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 8bo Crawler Bot correctly", function() {
		const results = uaParser("8bo Crawler Bot");
		assert.strictEqual(results.family, "8bo Crawler Bot", msg("results.family", results.family, "8bo Crawler Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Aboundex/0.2 (http://www.aboundex.com/crawler/) correctly", function() {
		const results = uaParser("Aboundex/0.2 (http://www.aboundex.com/crawler/)");
		assert.strictEqual(results.family, "Aboundex", msg("results.family", results.family, "Aboundex"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Accoona-AI-Agent/1.1.1 (crawler at accoona dot com) correctly", function() {
		const results = uaParser("Accoona-AI-Agent/1.1.1 (crawler at accoona dot com)");
		assert.strictEqual(results.family, "Accoona-AI-Agent", msg("results.family", results.family, "Accoona-AI-Agent"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Accoona-Biz-Agent/1.1.1 crawler@accoona.com correctly", function() {
		const results = uaParser("Accoona-Biz-Agent/1.1.1 crawler@accoona.com");
		assert.strictEqual(results.family, "Accoona-Biz-Agent", msg("results.family", results.family, "Accoona-Biz-Agent"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses AdsBot-Google correctly", function() {
		const results = uaParser("AdsBot-Google");
		assert.strictEqual(results.family, "AdsBot-Google", msg("results.family", results.family, "AdsBot-Google"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AppEngine-Google; ( http://code.google.com/appengine) correctly", function() {
		const results = uaParser("AppEngine-Google; ( http://code.google.com/appengine)");
		assert.strictEqual(results.family, "AppEngine-Google", msg("results.family", results.family, "AppEngine-Google"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Inne: Mozilla/2.0 (compatible; Ask Jeeves/Teoma;  http://sp.ask.com/docs/about/tech_crawling.html) correctly", function() {
		const results = uaParser("Inne: Mozilla/2.0 (compatible; Ask Jeeves/Teoma;  http://sp.ask.com/docs/about/tech_crawling.html)");
		assert.strictEqual(results.family, "Ask Jeeves", msg("results.family", results.family, "Ask Jeeves"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BaiDuSpider correctly", function() {
		const results = uaParser("BaiDuSpider");
		assert.strictEqual(results.family, "BaiDuSpider", msg("results.family", results.family, "BaiDuSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Baiduspider correctly", function() {
		const results = uaParser("Baiduspider");
		assert.strictEqual(results.family, "Baiduspider", msg("results.family", results.family, "Baiduspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 'Mozilla/5.0 (compatible; Baiduspider/2.0; +ht' correctly", function() {
		const results = uaParser("'Mozilla/5.0 (compatible; Baiduspider/2.0; +ht'");
		assert.strictEqual(results.family, "Baiduspider", msg("results.family", results.family, "Baiduspider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Baiduspider-cpro; +http://www.baidu.com/search/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Baiduspider-cpro; +http://www.baidu.com/search/spider.html)");
		assert.strictEqual(results.family, "Baiduspider-cpro", msg("results.family", results.family, "Baiduspider-cpro"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Baiduspider-image ( http://www.baidu.com/search/spider.htm) correctly", function() {
		const results = uaParser("Baiduspider-image ( http://www.baidu.com/search/spider.htm)");
		assert.strictEqual(results.family, "Baiduspider-image", msg("results.family", results.family, "Baiduspider-image"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Baiduspider-testbranch correctly", function() {
		const results = uaParser("Baiduspider-testbranch");
		assert.strictEqual(results.family, "Baiduspider-testbranch", msg("results.family", results.family, "Baiduspider-testbranch"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/534  (KHTML, like Gecko) BingPreview/1.0b correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/534  (KHTML, like Gecko) BingPreview/1.0b");
		assert.strictEqual(results.family, "BingPreview", msg("results.family", results.family, "BingPreview"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BlogBridge 6.7 (http://www.blogbridge.com/) 1.6.0_18 correctly", function() {
		const results = uaParser("BlogBridge 6.7 (http://www.blogbridge.com/) 1.6.0_18");
		assert.strictEqual(results.family, "BlogBridge", msg("results.family", results.family, "BlogBridge"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BoardReader Blog Indexer(http://boardreader.com) correctly", function() {
		const results = uaParser("BoardReader Blog Indexer(http://boardreader.com)");
		assert.strictEqual(results.family, "BoardReader Blog Indexer", msg("results.family", results.family, "BoardReader Blog Indexer"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BoardReader Favicon Fetcher /1.0 info@boardreader.com correctly", function() {
		const results = uaParser("BoardReader Favicon Fetcher /1.0 info@boardreader.com");
		assert.strictEqual(results.family, "BoardReader Favicon Fetcher", msg("results.family", results.family, "BoardReader Favicon Fetcher"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla 4.0(compatible; BotSeer/1.0;  http://botseer.ist.psu.edu) correctly", function() {
		const results = uaParser("Mozilla 4.0(compatible; BotSeer/1.0;  http://botseer.ist.psu.edu)");
		assert.strictEqual(results.family, "BotSeer", msg("results.family", results.family, "BotSeer"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CRAWL-E/0.6.4 correctly", function() {
		const results = uaParser("CRAWL-E/0.6.4");
		assert.strictEqual(results.family, "CRAWL-E", msg("results.family", results.family, "CRAWL-E"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Charlotte/1.0b; charlotte@beta.spider.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Charlotte/1.0b; charlotte@beta.spider.com)");
		assert.strictEqual(results.family, "Charlotte", msg("results.family", results.family, "Charlotte"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Checklinks/1.3 (pywikipedia robot; http://toolserver.org/~dispenser/view/Checklinks) correctly", function() {
		const results = uaParser("Checklinks/1.3 (pywikipedia robot; http://toolserver.org/~dispenser/view/Checklinks)");
		assert.strictEqual(results.family, "Checklinks", msg("results.family", results.family, "Checklinks"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Comodo HTTP(S) Crawler - http://www.instantssl.com/crawler correctly", function() {
		const results = uaParser("Comodo HTTP(S) Crawler - http://www.instantssl.com/crawler");
		assert.strictEqual(results.family, "Comodo HTTP(S) Crawler", msg("results.family", results.family, "Comodo HTTP(S) Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Comodo-Webinspector-Crawler 2.1 correctly", function() {
		const results = uaParser("Comodo-Webinspector-Crawler 2.1");
		assert.strictEqual(results.family, "Comodo-Webinspector-Crawler", msg("results.family", results.family, "Comodo-Webinspector-Crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Comodo-Webinspector-Crawler 2.2.2, http://www.comodorobot.com correctly", function() {
		const results = uaParser("Comodo-Webinspector-Crawler 2.2.2, http://www.comodorobot.com");
		assert.strictEqual(results.family, "Comodo-Webinspector-Crawler", msg("results.family", results.family, "Comodo-Webinspector-Crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses ConveraCrawler/0.9c ( http://www.authoritativeweb.com/crawl) correctly", function() {
		const results = uaParser("ConveraCrawler/0.9c ( http://www.authoritativeweb.com/crawl)");
		assert.strictEqual(results.family, "ConveraCrawler", msg("results.family", results.family, "ConveraCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CrawlConvera0.1 (CrawlConvera@yahoo.com) correctly", function() {
		const results = uaParser("CrawlConvera0.1 (CrawlConvera@yahoo.com)");
		assert.strictEqual(results.family, "CrawlConvera", msg("results.family", results.family, "CrawlConvera"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE or Firefox mutant; not on Windows server; + http://tab.search.daum.net/aboutWebSearch.html) Daumoa/3.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE or Firefox mutant; not on Windows server; + http://tab.search.daum.net/aboutWebSearch.html) Daumoa/3.0");
		assert.strictEqual(results.family, "Daumoa", msg("results.family", results.family, "Daumoa"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Firefox compatible; MS IE compatible;  http://search.daum.net/) Daumoa-feedfetcher/2.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Firefox compatible; MS IE compatible;  http://search.daum.net/) Daumoa-feedfetcher/2.0");
		assert.strictEqual(results.family, "Daumoa-feedfetcher", msg("results.family", results.family, "Daumoa-feedfetcher"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Feed Seeker Bot (RSS Feed Seeker http://www.MyNewFavoriteThing.com/fsb.php) correctly", function() {
		const results = uaParser("Feed Seeker Bot (RSS Feed Seeker http://www.MyNewFavoriteThing.com/fsb.php)");
		assert.strictEqual(results.family, "Feed Seeker Bot", msg("results.family", results.family, "Feed Seeker Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Flamingo_SearchEngine (+http://www.flamingosearch.com/bot) correctly", function() {
		const results = uaParser("Flamingo_SearchEngine (+http://www.flamingosearch.com/bot)");
		assert.strictEqual(results.family, "Flamingo_SearchEngine", msg("results.family", results.family, "Flamingo_SearchEngine"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FollowSite Bot ( http://www.followsite.com/bot.html ) correctly", function() {
		const results = uaParser("FollowSite Bot ( http://www.followsite.com/bot.html )");
		assert.strictEqual(results.family, "FollowSite Bot", msg("results.family", results.family, "FollowSite Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Genieo/1.0 http://www.genieo.com/webfilter.html correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Genieo/1.0 http://www.genieo.com/webfilter.html");
		assert.strictEqual(results.family, "Genieo", msg("results.family", results.family, "Genieo"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible;  MSIE 5.01; GomezAgent 2.0; Windows NT) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible;  MSIE 5.01; GomezAgent 2.0; Windows NT)");
		assert.strictEqual(results.family, "GomezAgent", msg("results.family", results.family, "GomezAgent"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Googlebot v2.1 (+http://www.google.com/bot.html) (http://www.openwebspider.org/) correctly", function() {
		const results = uaParser("Googlebot v2.1 (+http://www.google.com/bot.html) (http://www.openwebspider.org/)");
		assert.strictEqual(results.family, "Googlebot", msg("results.family", results.family, "Googlebot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Googlebot/2.1;  http://www.google.com/bot.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Googlebot/2.1;  http://www.google.com/bot.html)");
		assert.strictEqual(results.family, "Googlebot", msg("results.family", results.family, "Googlebot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Googlebot/2.2 correctly", function() {
		const results = uaParser("Googlebot/2.2");
		assert.strictEqual(results.family, "Googlebot", msg("results.family", results.family, "Googlebot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; Mozilla 4.0(compatible; Googlebot/5.0;  http://www.google.com/googlebot); en-US;) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; Mozilla 4.0(compatible; Googlebot/5.0;  http://www.google.com/googlebot); en-US;)");
		assert.strictEqual(results.family, "Googlebot", msg("results.family", results.family, "Googlebot"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Googlebot-Image/1.0 correctly", function() {
		const results = uaParser("Googlebot-Image/1.0");
		assert.strictEqual(results.family, "Googlebot-Image", msg("results.family", results.family, "Googlebot-Image"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Googlebot-Mobile (compatible; Googlebot-Mobile/2.1; +http://www.google.com/bot.html) correctly", function() {
		const results = uaParser("Googlebot-Mobile (compatible; Googlebot-Mobile/2.1; +http://www.google.com/bot.html)");
		assert.strictEqual(results.family, "Googlebot-Mobile", msg("results.family", results.family, "Googlebot-Mobile"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DoCoMo//2.0 N905i(c100;TB;W24H16) (compatible; Googlebot-Mobile/2.1; +http://www.google.com/bot.html) correctly", function() {
		const results = uaParser("DoCoMo//2.0 N905i(c100;TB;W24H16) (compatible; Googlebot-Mobile/2.1; +http://www.google.com/bot.html)");
		assert.strictEqual(results.family, "Googlebot-Mobile", msg("results.family", results.family, "Googlebot-Mobile"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Googlebot-News correctly", function() {
		const results = uaParser("Googlebot-News");
		assert.strictEqual(results.family, "Googlebot-News", msg("results.family", results.family, "Googlebot-News"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Googlebot-Video/1.0 correctly", function() {
		const results = uaParser("Googlebot-Video/1.0");
		assert.strictEqual(results.family, "Googlebot-Video", msg("results.family", results.family, "Googlebot-Video"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Googlebot-richsnippets correctly", function() {
		const results = uaParser("Googlebot-richsnippets");
		assert.strictEqual(results.family, "Googlebot-richsnippets", msg("results.family", results.family, "Googlebot-richsnippets"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Google Web Preview correctly", function() {
		const results = uaParser("Google Web Preview");
		assert.strictEqual(results.family, "Other", msg("results.family", results.family, "Other"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.78 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112.78 Safari/537.36");
		assert.strictEqual(results.family, "Chrome", msg("results.family", results.family, "Chrome"));
		assert.strictEqual(results.major || "", "60", msg("results.major", results.major || "", "60"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3112" || "", msg("results.patch", results.patch || "", "3112" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_6) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/60.0.3112 Safari/537.36");
		assert.strictEqual(results.family, "Chrome", msg("results.family", results.family, "Chrome"));
		assert.strictEqual(results.major || "", "60", msg("results.major", results.major || "", "60"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3112" || "", msg("results.patch", results.patch || "", "3112" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko; Google Web Preview) Chrome/27.0 .1453 Safari/537.36. correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko; Google Web Preview) Chrome/27.0 .1453 Safari/537.36.");
		assert.strictEqual(results.family, "Chrome", msg("results.family", results.family, "Chrome"));
		assert.strictEqual(results.major || "", "27", msg("results.major", results.major || "", "27"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HiddenMarket-1.0-beta (www.hiddenmarket.net/crawler.php) correctly", function() {
		const results = uaParser("HiddenMarket-1.0-beta (www.hiddenmarket.net/crawler.php)");
		assert.strictEqual(results.family, "HiddenMarket", msg("results.family", results.family, "HiddenMarket"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HooWWWer/2.1.0 ( http://cosco.hiit.fi/search/hoowwwer/ | mailto:crawler-info<at>hiit.fi) correctly", function() {
		const results = uaParser("HooWWWer/2.1.0 ( http://cosco.hiit.fi/search/hoowwwer/ | mailto:crawler-info<at>hiit.fi)");
		assert.strictEqual(results.family, "HooWWWer", msg("results.family", results.family, "HooWWWer"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses ICC-Crawler(Mozilla-compatible; http://kc.nict.go.jp/icc/crawl.html; icc-crawl(at)ml(dot)nict(dot)go(dot)jp) correctly", function() {
		const results = uaParser("ICC-Crawler(Mozilla-compatible; http://kc.nict.go.jp/icc/crawl.html; icc-crawl(at)ml(dot)nict(dot)go(dot)jp)");
		assert.strictEqual(results.family, "ICC-Crawler", msg("results.family", results.family, "ICC-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ICC-Crawler/2.0 (Mozilla-compatible; ; http://kc.nict.go.jp/project1/crawl.html) correctly", function() {
		const results = uaParser("ICC-Crawler/2.0 (Mozilla-compatible; ; http://kc.nict.go.jp/project1/crawl.html)");
		assert.strictEqual(results.family, "ICC-Crawler", msg("results.family", results.family, "ICC-Crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IconSurf/2.0 favicon finder (see http://iconsurf.com/robot.html) correctly", function() {
		const results = uaParser("IconSurf/2.0 favicon finder (see http://iconsurf.com/robot.html)");
		assert.strictEqual(results.family, "IconSurf", msg("results.family", results.family, "IconSurf"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IlTrovatore/1.2 (IlTrovatore; http://www.iltrovatore.it/bot.html; bot@iltrovatore.it) correctly", function() {
		const results = uaParser("IlTrovatore/1.2 (IlTrovatore; http://www.iltrovatore.it/bot.html; bot@iltrovatore.it)");
		assert.strictEqual(results.family, "IlTrovatore", msg("results.family", results.family, "IlTrovatore"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IlTrovatore-Setaccio ( http://www.iltrovatore.it) correctly", function() {
		const results = uaParser("IlTrovatore-Setaccio ( http://www.iltrovatore.it)");
		assert.strictEqual(results.family, "IlTrovatore-Setaccio", msg("results.family", results.family, "IlTrovatore-Setaccio"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; InfuzApp/1.0; +http://www.infuz.com/bot.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; InfuzApp/1.0; +http://www.infuz.com/bot.html)");
		assert.strictEqual(results.family, "InfuzApp", msg("results.family", results.family, "InfuzApp"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses InternetArchive-1.0 correctly", function() {
		const results = uaParser("InternetArchive-1.0");
		assert.strictEqual(results.family, "InternetArchive", msg("results.family", results.family, "InternetArchive"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses InternetArchive/0.8-dev (Nutch; http://lucene.apache.org/nutch/bot.html; nutch-agent@lucene.apache.org) correctly", function() {
		const results = uaParser("InternetArchive/0.8-dev (Nutch; http://lucene.apache.org/nutch/bot.html; nutch-agent@lucene.apache.org)");
		assert.strictEqual(results.family, "InternetArchive", msg("results.family", results.family, "InternetArchive"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses KDDI-CA34 UP.Browser/6.2.0.10.2.2(GUI)MMP/2.0 (compatible; KDDI-Googlebot-Mobile/2.1; http://www.google.com/bot.html) correctly", function() {
		const results = uaParser("KDDI-CA34 UP.Browser/6.2.0.10.2.2(GUI)MMP/2.0 (compatible; KDDI-Googlebot-Mobile/2.1; http://www.google.com/bot.html)");
		assert.strictEqual(results.family, "KDDI-Googlebot-Mobile", msg("results.family", results.family, "KDDI-Googlebot-Mobile"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses kalooga/KaloogaBot (Kalooga; http://www.kalooga.com/info.html?page=crawler) correctly", function() {
		const results = uaParser("kalooga/KaloogaBot (Kalooga; http://www.kalooga.com/info.html?page=crawler)");
		assert.strictEqual(results.family, "KaloogaBot", msg("results.family", results.family, "KaloogaBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Kraken/0.1; http://linkfluence.net/; bot@linkfluence.net) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Kraken/0.1; http://linkfluence.net/; bot@linkfluence.net)");
		assert.strictEqual(results.family, "Kraken", msg("results.family", results.family, "Kraken"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Kurzor/1.0 (Kurzor; http://adcenter.hu/docs/en/bot.html; cursor@easymail.hu) correctly", function() {
		const results = uaParser("Kurzor/1.0 (Kurzor; http://adcenter.hu/docs/en/bot.html; cursor@easymail.hu)");
		assert.strictEqual(results.family, "Kurzor", msg("results.family", results.family, "Kurzor"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LEIA/3.01pr (LEIAcrawler; leia@gseek.com; http://www.gseek.com) correctly", function() {
		const results = uaParser("LEIA/3.01pr (LEIAcrawler; leia@gseek.com; http://www.gseek.com)");
		assert.strictEqual(results.family, "LEIA", msg("results.family", results.family, "LEIA"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows compatible LesnikBot) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows compatible LesnikBot)");
		assert.strictEqual(results.family, "LesnikBot", msg("results.family", results.family, "LesnikBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Linguee Bot (bot@linguee.com) correctly", function() {
		const results = uaParser("Linguee Bot (bot@linguee.com)");
		assert.strictEqual(results.family, "Linguee Bot", msg("results.family", results.family, "Linguee Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LinkAider (http://linkaider.com/crawler/) correctly", function() {
		const results = uaParser("LinkAider (http://linkaider.com/crawler/)");
		assert.strictEqual(results.family, "LinkAider", msg("results.family", results.family, "LinkAider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Lite Bot0316B correctly", function() {
		const results = uaParser("Lite Bot0316B");
		assert.strictEqual(results.family, "Lite Bot", msg("results.family", results.family, "Lite Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Llaut/1.0 (http://mnm.uib.es/~gallir/llaut/bot.html) correctly", function() {
		const results = uaParser("Llaut/1.0 (http://mnm.uib.es/~gallir/llaut/bot.html)");
		assert.strictEqual(results.family, "Llaut", msg("results.family", results.family, "Llaut"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Linux x86_64; Mail.RU_Bot/Fast/2.0; +http://go.mail.ru/help/robots) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Linux x86_64; Mail.RU_Bot/Fast/2.0; +http://go.mail.ru/help/robots)");
		assert.strictEqual(results.family, "Mail.RU_Bot", msg("results.family", results.family, "Mail.RU_Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Linux x86_64; Mail.RU_Bot/2.0; +http://go.mail.ru/help/robots) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Linux x86_64; Mail.RU_Bot/2.0; +http://go.mail.ru/help/robots)");
		assert.strictEqual(results.family, "Mail.RU_Bot", msg("results.family", results.family, "Mail.RU_Bot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mediapartners-Google correctly", function() {
		const results = uaParser("Mediapartners-Google");
		assert.strictEqual(results.family, "Mediapartners-Google", msg("results.family", results.family, "Mediapartners-Google"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DoCoMo/2.0 SH905i(c100;TB;W24H16) (compatible; Mediapartners-Google/2.1; +http://www.google.com/bot.html) correctly", function() {
		const results = uaParser("DoCoMo/2.0 SH905i(c100;TB;W24H16) (compatible; Mediapartners-Google/2.1; +http://www.google.com/bot.html)");
		assert.strictEqual(results.family, "Mediapartners-Google", msg("results.family", results.family, "Mediapartners-Google"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft Bing Mobile SocialStreams Bot correctly", function() {
		const results = uaParser("Microsoft Bing Mobile SocialStreams Bot");
		assert.strictEqual(results.family, "Microsoft Bing Mobile SocialStreams Bot", msg("results.family", results.family, "Microsoft Bing Mobile SocialStreams Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft MSN SocialStreams Bot,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Microsoft MSN SocialStreams Bot,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Microsoft MSN SocialStreams Bot", msg("results.family", results.family, "Microsoft MSN SocialStreams Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NING/1.0 correctly", function() {
		const results = uaParser("NING/1.0");
		assert.strictEqual(results.family, "NING", msg("results.family", results.family, "NING"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Netvibes (http://www.netvibes.com) correctly", function() {
		const results = uaParser("Netvibes (http://www.netvibes.com)");
		assert.strictEqual(results.family, "Netvibes", msg("results.family", results.family, "Netvibes"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NewsGator/2.0 Bot (http://www.newsgator.com) correctly", function() {
		const results = uaParser("NewsGator/2.0 Bot (http://www.newsgator.com)");
		assert.strictEqual(results.family, "NewsGator", msg("results.family", results.family, "NewsGator"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NewsGator/3.0,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("NewsGator/3.0,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "NewsGator", msg("results.family", results.family, "NewsGator"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NewsGator FetchLinks extension/0.2.0 (http://graemef.com) correctly", function() {
		const results = uaParser("NewsGator FetchLinks extension/0.2.0 (http://graemef.com)");
		assert.strictEqual(results.family, "NewsGator FetchLinks extension", msg("results.family", results.family, "NewsGator FetchLinks extension"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses NewsGatorOnline/2.0 (http://www.newsgator.com) correctly", function() {
		const results = uaParser("NewsGatorOnline/2.0 (http://www.newsgator.com)");
		assert.strictEqual(results.family, "NewsGatorOnline", msg("results.family", results.family, "NewsGatorOnline"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SapphireWebCrawler/1.0 (Sapphire Web Crawler using Nutch; http://boston.lti.cs.cmu.edu/crawler/; mhoy@cs.cmu.edu) correctly", function() {
		const results = uaParser("SapphireWebCrawler/1.0 (Sapphire Web Crawler using Nutch; http://boston.lti.cs.cmu.edu/crawler/; mhoy@cs.cmu.edu)");
		assert.strictEqual(results.family, "Nutch; http:", msg("results.family", results.family, "Nutch; http:"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NutchCVS/0.06-dev (Nutch running at UW; http://www.nutch.org/docs/en/bot.html; sycrawl@cs.washington.edu) correctly", function() {
		const results = uaParser("NutchCVS/0.06-dev (Nutch running at UW; http://www.nutch.org/docs/en/bot.html; sycrawl@cs.washington.edu)");
		assert.strictEqual(results.family, "NutchCVS", msg("results.family", results.family, "NutchCVS"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "06", msg("results.minor", results.minor || "", "06"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NutchOSUOSL/0.05-dev (Nutch; http://www.nutch.org/docs/en/bot.html; nutch-agent@lists.sourceforge.net) correctly", function() {
		const results = uaParser("NutchOSUOSL/0.05-dev (Nutch; http://www.nutch.org/docs/en/bot.html; nutch-agent@lists.sourceforge.net)");
		assert.strictEqual(results.family, "NutchOSUOSL", msg("results.family", results.family, "NutchOSUOSL"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "05", msg("results.minor", results.minor || "", "05"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NutchOrg/0.03-dev (Nutch; http://www.nutch.org/docs/bot.html; nutch-agent@lists.sourceforge.net) correctly", function() {
		const results = uaParser("NutchOrg/0.03-dev (Nutch; http://www.nutch.org/docs/bot.html; nutch-agent@lists.sourceforge.net)");
		assert.strictEqual(results.family, "NutchOrg", msg("results.family", results.family, "NutchOrg"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "03", msg("results.minor", results.minor || "", "03"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ObjectsSearch/0.2 (ObjectsSearch; http://www.ObjectsSearch.com/bot.html; support@thesoftwareobjects.com) correctly", function() {
		const results = uaParser("ObjectsSearch/0.2 (ObjectsSearch; http://www.ObjectsSearch.com/bot.html; support@thesoftwareobjects.com)");
		assert.strictEqual(results.family, "ObjectsSearch", msg("results.family", results.family, "ObjectsSearch"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Orbiter ( http://www.dailyorbit.com/bot.htm) correctly", function() {
		const results = uaParser("Orbiter ( http://www.dailyorbit.com/bot.htm)");
		assert.strictEqual(results.family, "Orbiter", msg("results.family", results.family, "Orbiter"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PagePeeker.com correctly", function() {
		const results = uaParser("PagePeeker.com");
		assert.strictEqual(results.family, "PagePeeker", msg("results.family", results.family, "PagePeeker"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PagesInventory (robot +http://www.pagesinventory.com) correctly", function() {
		const results = uaParser("PagesInventory (robot +http://www.pagesinventory.com)");
		assert.strictEqual(results.family, "PagesInventory", msg("results.family", results.family, "PagesInventory"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; PaxleFramework/0.1.1;  http://www.paxle.net/en/bot) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; PaxleFramework/0.1.1;  http://www.paxle.net/en/bot)");
		assert.strictEqual(results.family, "PaxleFramework", msg("results.family", results.family, "PaxleFramework"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Peeplo Screenshot Bot/0.20 ( abuse at peeplo dot_com ) correctly", function() {
		const results = uaParser("Peeplo Screenshot Bot/0.20 ( abuse at peeplo dot_com )");
		assert.strictEqual(results.family, "Peeplo Screenshot Bot", msg("results.family", results.family, "Peeplo Screenshot Bot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "20", msg("results.minor", results.minor || "", "20"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PlantyNet_WebRobot_V1.9 babo@plantynet.com correctly", function() {
		const results = uaParser("PlantyNet_WebRobot_V1.9 babo@plantynet.com");
		assert.strictEqual(results.family, "PlantyNet_WebRobot", msg("results.family", results.family, "PlantyNet_WebRobot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pompos/1.2 http://pompos.iliad.fr correctly", function() {
		const results = uaParser("Pompos/1.2 http://pompos.iliad.fr");
		assert.strictEqual(results.family, "Pompos", msg("results.family", results.family, "Pompos"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Reaper/1.3.6 (Linux; U; Android 4.3; de-DE; GT-I9300 Build/JSS15J) correctly", function() {
		const results = uaParser("Reaper/1.3.6 (Linux; U; Android 4.3; de-DE; GT-I9300 Build/JSS15J)");
		assert.strictEqual(results.family, "Reaper", msg("results.family", results.family, "Reaper"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Reaper/2.07 ( http://www.sitesearch.ca/reaper) correctly", function() {
		const results = uaParser("Reaper/2.07 ( http://www.sitesearch.ca/reaper)");
		assert.strictEqual(results.family, "Reaper", msg("results.family", results.family, "Reaper"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "07", msg("results.minor", results.minor || "", "07"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RedCarpet/1.0 (http://www.redcarpet-inc.com/robots.html) correctly", function() {
		const results = uaParser("RedCarpet/1.0 (http://www.redcarpet-inc.com/robots.html)");
		assert.strictEqual(results.family, "RedCarpet", msg("results.family", results.family, "RedCarpet"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RedCarpet/2.1 CFNetwork/672.0.2 Darwin/14.0.0 correctly", function() {
		const results = uaParser("RedCarpet/2.1 CFNetwork/672.0.2 Darwin/14.0.0");
		assert.strictEqual(results.family, "RedCarpet", msg("results.family", results.family, "RedCarpet"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Riddler (http://riddler.io/about.html) correctly", function() {
		const results = uaParser("Riddler (http://riddler.io/about.html)");
		assert.strictEqual(results.family, "Riddler", msg("results.family", results.family, "Riddler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Scrapy/0.16.4 (+http://scrapy.org) correctly", function() {
		const results = uaParser("Scrapy/0.16.4 (+http://scrapy.org)");
		assert.strictEqual(results.family, "Scrapy", msg("results.family", results.family, "Scrapy"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "16", msg("results.minor", results.minor || "", "16"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Scrapy/0.22.2 (+http://scrapy.org) correctly", function() {
		const results = uaParser("Scrapy/0.22.2 (+http://scrapy.org)");
		assert.strictEqual(results.family, "Scrapy", msg("results.family", results.family, "Scrapy"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "22", msg("results.minor", results.minor || "", "22"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Simpy/1.1 (Simpy; http://www.simpy.com/?ref=bot; feedback at simpy dot com) correctly", function() {
		const results = uaParser("Simpy/1.1 (Simpy; http://www.simpy.com/?ref=bot; feedback at simpy dot com)");
		assert.strictEqual(results.family, "Simpy", msg("results.family", results.family, "Simpy"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/3.0 (Slurp.so/1.0; slurp@inktomi.com; http://www.inktomi.com/slurp.html) correctly", function() {
		const results = uaParser("Mozilla/3.0 (Slurp.so/1.0; slurp@inktomi.com; http://www.inktomi.com/slurp.html)");
		assert.strictEqual(results.family, "Slurp", msg("results.family", results.family, "Slurp"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Slurp/2.0 correctly", function() {
		const results = uaParser("Slurp/2.0");
		assert.strictEqual(results.family, "Slurp", msg("results.family", results.family, "Slurp"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Yahoo!; U; Slurp/3.0; http://help.yahoo.com/help/us/ysearch/slurp) Mozilla/5.0 () correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Yahoo!; U; Slurp/3.0; http://help.yahoo.com/help/us/ysearch/slurp) Mozilla/5.0 ()");
		assert.strictEqual(results.family, "Slurp", msg("results.family", results.family, "Slurp"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Speedy Spider; http://www.entireweb.com/about/search_tech/speedy_spider/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Speedy Spider; http://www.entireweb.com/about/search_tech/speedy_spider/)");
		assert.strictEqual(results.family, "Speedy Spider", msg("results.family", results.family, "Speedy Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Squrl Java/1.6.0_22 correctly", function() {
		const results = uaParser("Squrl Java/1.6.0_22");
		assert.strictEqual(results.family, "Squrl Java", msg("results.family", results.family, "Squrl Java"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses TheUsefulbot_2.3.62 (bot@theusefulnet.com) correctly", function() {
		const results = uaParser("TheUsefulbot_2.3.62 (bot@theusefulnet.com)");
		assert.strictEqual(results.family, "TheUsefulbot", msg("results.family", results.family, "TheUsefulbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; de-DE; rv:1.9.0.19) Gecko/2010091808 ThumbShotsBot (KFSW 3.0.6-3) correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; de-DE; rv:1.9.0.19) Gecko/2010091808 ThumbShotsBot (KFSW 3.0.6-3)");
		assert.strictEqual(results.family, "ThumbShotsBot", msg("results.family", results.family, "ThumbShotsBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Thumbshots.ru; +http://thumbshots.ru/bot) Firefox/3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Thumbshots.ru; +http://thumbshots.ru/bot) Firefox/3");
		assert.strictEqual(results.family, "Thumbshots.ru", msg("results.family", results.family, "Thumbshots.ru"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Vagabondo/Wapspider; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Vagabondo/Wapspider; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/)");
		assert.strictEqual(results.family, "Vagabondo", msg("results.family", results.family, "Vagabondo"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Vagabondo/2.1; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Vagabondo/2.1; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/)");
		assert.strictEqual(results.family, "Vagabondo", msg("results.family", results.family, "Vagabondo"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible;  Vagabondo/2.2; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible;  Vagabondo/2.2; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/)");
		assert.strictEqual(results.family, "Vagabondo", msg("results.family", results.family, "Vagabondo"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible;  Vagabondo/2.3; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible;  Vagabondo/2.3; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/)");
		assert.strictEqual(results.family, "Vagabondo", msg("results.family", results.family, "Vagabondo"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible;  Vagabondo/4.0Beta; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible;  Vagabondo/4.0Beta; webcrawler at wise-guys dot nl; http://webagent.wise-guys.nl/)");
		assert.strictEqual(results.family, "Vagabondo", msg("results.family", results.family, "Vagabondo"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.0; Windows 95) VoilaBot BETA 1.2 (http://www.voila.com/) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.0; Windows 95) VoilaBot BETA 1.2 (http://www.voila.com/)");
		assert.strictEqual(results.family, "VoilaBot", msg("results.family", results.family, "VoilaBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Votay bot/4.0;  http://www.votay.com/arts/comics/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Votay bot/4.0;  http://www.votay.com/arts/comics/)");
		assert.strictEqual(results.family, "Votay bot", msg("results.family", results.family, "Votay bot"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; WASALive Bot ; http://blog.wasalive.com/wasalive-bots/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; WASALive Bot ; http://blog.wasalive.com/wasalive-bots/)");
		assert.strictEqual(results.family, "WASALive Bot", msg("results.family", results.family, "WASALive Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU iPhone OS 7_0 like Mac OS X; de-DE) WIRED/3.2.3.11.87970 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU iPhone OS 7_0 like Mac OS X; de-DE) WIRED/3.2.3.11.87970");
		assert.strictEqual(results.family, "WIRE", msg("results.family", results.family, "WIRE"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WIRE/0.11 (Linux; i686; Robot,Spider,Crawler,aromano@cli.di.unipi.it) correctly", function() {
		const results = uaParser("WIRE/0.11 (Linux; i686; Robot,Spider,Crawler,aromano@cli.di.unipi.it)");
		assert.strictEqual(results.family, "WIRE", msg("results.family", results.family, "WIRE"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "11", msg("results.minor", results.minor || "", "11"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WIRE/1.0 (Linux;i686;Bot,Robot,Spider,Crawler) correctly", function() {
		const results = uaParser("WIRE/1.0 (Linux;i686;Bot,Robot,Spider,Crawler)");
		assert.strictEqual(results.family, "WIRE", msg("results.family", results.family, "WIRE"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; Windows NT 6.1; WOW64; rv:2.0b8pre; .NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; MS-RTC LM 8; OfficeLiveConnector.1.4; OfficeLivePatch.1.3; SLCC1; SLCC2; Media Center PC 6.0; GTB6.4; InfoPath.2; en-US; FunWebProducts; Zango 10.1.181.0; SV1; PRTG Network Monitor (www.paessler.com)) Gecko/20101114 Firefox/4.0b8pre QuickTime/7.6.2 Songbird/1.1.2 Web-sniffer/1.0.36 lftp/3.7.4 libwww-perl/5.820 GSiteCrawler/v1.12 rev. 260 Snoopy v1.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; Windows NT 6.1; WOW64; rv:2.0b8pre; .NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; MS-RTC LM 8; OfficeLiveConnector.1.4; OfficeLivePatch.1.3; SLCC1; SLCC2; Media Center PC 6.0; GTB6.4; InfoPath.2; en-US; FunWebProducts; Zango 10.1.181.0; SV1; PRTG Network Monitor (www.paessler.com)) Gecko/20101114 Firefox/4.0b8pre QuickTime/7.6.2 Songbird/1.1.2 Web-sniffer/1.0.36 lftp/3.7.4 libwww-perl/5.820 GSiteCrawler/v1.12 rev. 260 Snoopy v1.2");
		assert.strictEqual(results.family, "Web-sniffer", msg("results.family", results.family, "Web-sniffer"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "36" || "", msg("results.patch", results.patch || "", "36" || ""));
	});
	it("parses Mozilla/5.0 (compatible; WebThumbnail/2.2; Website Thumbnail Generator; +http://webthumbnail.org) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; WebThumbnail/2.2; Website Thumbnail Generator; +http://webthumbnail.org)");
		assert.strictEqual(results.family, "WebThumb", msg("results.family", results.family, "WebThumb"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux i686 (x86_64); en-US; rv:1.9.0.17) Gecko WebThumb/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux i686 (x86_64); en-US; rv:1.9.0.17) Gecko WebThumb/1.0");
		assert.strictEqual(results.family, "WebThumb", msg("results.family", results.family, "WebThumb"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WhatWeb/0.4.8 correctly", function() {
		const results = uaParser("WhatWeb/0.4.8");
		assert.strictEqual(results.family, "WhatWeb", msg("results.family", results.family, "WhatWeb"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Jetpack by WordPress.com correctly", function() {
		const results = uaParser("Jetpack by WordPress.com");
		assert.strictEqual(results.family, "WordPress", msg("results.family", results.family, "WordPress"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WordPress/1.2.1 PHP/4.3.9-1 correctly", function() {
		const results = uaParser("WordPress/1.2.1 PHP/4.3.9-1");
		assert.strictEqual(results.family, "WordPress", msg("results.family", results.family, "WordPress"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Pachacutec/0.5 (qpImageBuilders.com); WordPress/3.1; http://contacto-latino.com/news correctly", function() {
		const results = uaParser("Pachacutec/0.5 (qpImageBuilders.com); WordPress/3.1; http://contacto-latino.com/news");
		assert.strictEqual(results.family, "WordPress", msg("results.family", results.family, "WordPress"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Wotbox/0.7-alpha (bot@wotbox.com; http://www.wotbox.com) correctly", function() {
		const results = uaParser("Wotbox/0.7-alpha (bot@wotbox.com; http://www.wotbox.com)");
		assert.strictEqual(results.family, "Wotbox", msg("results.family", results.family, "Wotbox"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Wotbox/2.01 ( http://www.wotbox.com/bot/) correctly", function() {
		const results = uaParser("Wotbox/2.01 ( http://www.wotbox.com/bot/)");
		assert.strictEqual(results.family, "Wotbox", msg("results.family", results.family, "Wotbox"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Xenu Link Sleuth 1.1f correctly", function() {
		const results = uaParser("Xenu Link Sleuth 1.1f");
		assert.strictEqual(results.family, "Xenu Link Sleuth", msg("results.family", results.family, "Xenu Link Sleuth"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Xenu's Link Sleuth 1.0p correctly", function() {
		const results = uaParser("Xenu's Link Sleuth 1.0p");
		assert.strictEqual(results.family, "Xenu's Link Sleuth", msg("results.family", results.family, "Xenu's Link Sleuth"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Xerka WebBot v1.0.0 [AIDO_CREA_ab] correctly", function() {
		const results = uaParser("Xerka WebBot v1.0.0 [AIDO_CREA_ab]");
		assert.strictEqual(results.family, "Xerka WebBot", msg("results.family", results.family, "Xerka WebBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Yahoo! Slurp China; http://misc.yahoo.com.cn/help.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Yahoo! Slurp China; http://misc.yahoo.com.cn/help.html)");
		assert.strictEqual(results.family, "Yahoo! Slurp", msg("results.family", results.family, "Yahoo! Slurp"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Yahoo! Slurp/3.0 ; http://help.yahoo.com/help/us/ysearch/slurp) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Yahoo! Slurp/3.0 ; http://help.yahoo.com/help/us/ysearch/slurp)");
		assert.strictEqual(results.family, "Yahoo! Slurp", msg("results.family", results.family, "Yahoo! Slurp"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LG-C1500 UP.Browser/6.2.3 (GUI) MMP/1.0 (compatible;YahooSeeker/M1A1-R2D2; http://help.yahoo.com/help/us/ysearch/crawling/crawling-01.html) correctly", function() {
		const results = uaParser("LG-C1500 UP.Browser/6.2.3 (GUI) MMP/1.0 (compatible;YahooSeeker/M1A1-R2D2; http://help.yahoo.com/help/us/ysearch/crawling/crawling-01.html)");
		assert.strictEqual(results.family, "YahooSeeker", msg("results.family", results.family, "YahooSeeker"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots),gzip(gfe) AppEngine-Google; (+http://code.google.com/appengine; appid: twitter-mirror),gzip(gfe),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; YandexBot/3.0; +http://yandex.com/bots),gzip(gfe) AppEngine-Google; (+http://code.google.com/appengine; appid: twitter-mirror),gzip(gfe),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "YandexBot", msg("results.family", results.family, "YandexBot"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Yeti-FeedItemCrawler/1.0 (NHN Corp.;+http://help.naver.com/robots/) correctly", function() {
		const results = uaParser("Yeti-FeedItemCrawler/1.0 (NHN Corp.;+http://help.naver.com/robots/)");
		assert.strictEqual(results.family, "Yeti", msg("results.family", results.family, "Yeti"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; ) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; YodaoBot/1.0; http://www.yodao.com/help/webmaster/spider/; )");
		assert.strictEqual(results.family, "YodaoBot", msg("results.family", results.family, "YodaoBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible;YodaoBot-Image/1.0;http://www.youdao.com/help/webmaster/spider/;) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible;YodaoBot-Image/1.0;http://www.youdao.com/help/webmaster/spider/;)");
		assert.strictEqual(results.family, "YodaoBot-Image", msg("results.family", results.family, "YodaoBot-Image"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YowedoBot/Yowedo 1.0 (Search Engine crawler for yowedo.com; http://yowedo.com/en/partners.html; crawler@yowedo.com) correctly", function() {
		const results = uaParser("YowedoBot/Yowedo 1.0 (Search Engine crawler for yowedo.com; http://yowedo.com/en/partners.html; crawler@yowedo.com)");
		assert.strictEqual(results.family, "Yowedo", msg("results.family", results.family, "Yowedo"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Zao-Crawler correctly", function() {
		const results = uaParser("Zao-Crawler");
		assert.strictEqual(results.family, "Zao", msg("results.family", results.family, "Zao"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ZeBot_www.ze.bz (ze.bz@hotmail.com) correctly", function() {
		const results = uaParser("ZeBot_www.ze.bz (ze.bz@hotmail.com)");
		assert.strictEqual(results.family, "ZeBot_www.ze.bz", msg("results.family", results.family, "ZeBot_www.ze.bz"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ZooShot 0.42 correctly", function() {
		const results = uaParser("ZooShot 0.42");
		assert.strictEqual(results.family, "ZooShot", msg("results.family", results.family, "ZooShot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "42", msg("results.minor", results.minor || "", "42"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE; ZyBorg; Win32) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE; ZyBorg; Win32)");
		assert.strictEqual(results.family, "ZyBorg", msg("results.family", results.family, "ZyBorg"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 compatible ZyBorg/1.0 (ZyBorg@WISEnutbot.com; http://www.WISEnutbot.com) correctly", function() {
		const results = uaParser("Mozilla/4.0 compatible ZyBorg/1.0 (ZyBorg@WISEnutbot.com; http://www.WISEnutbot.com)");
		assert.strictEqual(results.family, "ZyBorg", msg("results.family", results.family, "ZyBorg"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AltaVista Intranet V2.0 Compaq Altavista Eval sveand@altavista.net correctly", function() {
		const results = uaParser("AltaVista Intranet V2.0 Compaq Altavista Eval sveand@altavista.net");
		assert.strictEqual(results.family, "altavista", msg("results.family", results.family, "altavista"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; archive.bibalex.org_bot; +http://www.test.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; archive.bibalex.org_bot; +http://www.test.com)");
		assert.strictEqual(results.family, "archive.bibalex.org_bot", msg("results.family", results.family, "archive.bibalex.org_bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; archive.org_bot/heritrix-1.15.1-x  http://pandora.nla.gov.au/crawl.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; archive.org_bot/heritrix-1.15.1-x  http://pandora.nla.gov.au/crawl.html)");
		assert.strictEqual(results.family, "archive.org_bot", msg("results.family", results.family, "archive.org_bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; special_archiver/3.2.0 +http://www.loc.gov/webarchiving/notice_to_webmasters.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; special_archiver/3.2.0 +http://www.loc.gov/webarchiving/notice_to_webmasters.html)");
		assert.strictEqual(results.family, "archiver", msg("results.family", results.family, "archiver"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.2; .NET CLR 1.1.4322; .NET CLR 2.0.50215; baiduspider) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.2; .NET CLR 1.1.4322; .NET CLR 2.0.50215; baiduspider)");
		assert.strictEqual(results.family, "baiduspider", msg("results.family", results.family, "baiduspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses baiduspider-mobile-gate correctly", function() {
		const results = uaParser("baiduspider-mobile-gate");
		assert.strictEqual(results.family, "baiduspider-mobile-gate", msg("results.family", results.family, "baiduspider-mobile-gate"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses bingbot correctly", function() {
		const results = uaParser("bingbot");
		assert.strictEqual(results.family, "bingbot", msg("results.family", results.family, "bingbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; bingbot/2.0  http://www.bing.com/bingbot.htm) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; bingbot/2.0  http://www.bing.com/bingbot.htm)");
		assert.strictEqual(results.family, "bingbot", msg("results.family", results.family, "bingbot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses boitho.com-dc/0.4 ( http://www.boitho.com/dcbot.html ) correctly", function() {
		const results = uaParser("boitho.com-dc/0.4 ( http://www.boitho.com/dcbot.html )");
		assert.strictEqual(results.family, "boitho.com-dc", msg("results.family", results.family, "boitho.com-dc"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses boitho.com-dc/0.86 ( http://www.boitho.com/dcbot.html ) correctly", function() {
		const results = uaParser("boitho.com-dc/0.86 ( http://www.boitho.com/dcbot.html )");
		assert.strictEqual(results.family, "boitho.com-dc", msg("results.family", results.family, "boitho.com-dc"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "86", msg("results.minor", results.minor || "", "86"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; clumboot; +http://localhost/clumpit/bot.php) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; clumboot; +http://localhost/clumpit/bot.php)");
		assert.strictEqual(results.family, "clumboot", msg("results.family", results.family, "clumboot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses findlinks/2.6 ( http://wortschatz.uni-leipzig.de/findlinks/) correctly", function() {
		const results = uaParser("findlinks/2.6 ( http://wortschatz.uni-leipzig.de/findlinks/)");
		assert.strictEqual(results.family, "findlinks", msg("results.family", results.family, "findlinks"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 compatible FurlBot/Furl Search 2.0 (FurlBot; http://www.furl.net; wn.furlbot@looksmart.net) correctly", function() {
		const results = uaParser("Mozilla/4.0 compatible FurlBot/Furl Search 2.0 (FurlBot; http://www.furl.net; wn.furlbot@looksmart.net)");
		assert.strictEqual(results.family, "furlbot", msg("results.family", results.family, "furlbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses gonzo1[D] mailto:crawleradmin.t-info@telekom.de correctly", function() {
		const results = uaParser("gonzo1[D] mailto:crawleradmin.t-info@telekom.de");
		assert.strictEqual(results.family, "gonzo1", msg("results.family", results.family, "gonzo1"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses grub-client-1.5.3; (grub-client-1.5.3; Crawl your own stuff with http://grub.org) correctly", function() {
		const results = uaParser("grub-client-1.5.3; (grub-client-1.5.3; Crawl your own stuff with http://grub.org)");
		assert.strictEqual(results.family, "grub-client", msg("results.family", results.family, "grub-client"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses gsa-crawler correctly", function() {
		const results = uaParser("gsa-crawler");
		assert.strictEqual(results.family, "gsa-crawler", msg("results.family", results.family, "gsa-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; cdlwas_bot; heritrix/1.14.1 +http://webarchives.cdlib.org/p/webmasters) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; cdlwas_bot; heritrix/1.14.1 +http://webarchives.cdlib.org/p/webmasters)");
		assert.strictEqual(results.family, "heritrix", msg("results.family", results.family, "heritrix"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "14", msg("results.minor", results.minor || "", "14"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (compatible; heritrix/3.2.0 +http://suki.ling.helsinki.fi/eng/project.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; heritrix/3.2.0 +http://suki.ling.helsinki.fi/eng/project.html)");
		assert.strictEqual(results.family, "heritrix", msg("results.family", results.family, "heritrix"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses holmes/2.3 correctly", function() {
		const results = uaParser("holmes/2.3");
		assert.strictEqual(results.family, "holmes", msg("results.family", results.family, "holmes"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses holmes/3.12.4 (http://morfeo.centrum.cz/bot) correctly", function() {
		const results = uaParser("holmes/3.12.4 (http://morfeo.centrum.cz/bot)");
		assert.strictEqual(results.family, "holmes", msg("results.family", results.family, "holmes"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "12", msg("results.minor", results.minor || "", "12"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses htdig/3.1.6 (romieu@bastide-medical.fr) correctly", function() {
		const results = uaParser("htdig/3.1.6 (romieu@bastide-medical.fr)");
		assert.strictEqual(results.family, "htdig", msg("results.family", results.family, "htdig"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses ia_archiver/8.7 (Windows NT 5.0; ) correctly", function() {
		const results = uaParser("ia_archiver/8.7 (Windows NT 5.0; )");
		assert.strictEqual(results.family, "ia_archiver", msg("results.family", results.family, "ia_archiver"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ichiro/1.0 (ichiro@nttr.co.jp) correctly", function() {
		const results = uaParser("ichiro/1.0 (ichiro@nttr.co.jp)");
		assert.strictEqual(results.family, "ichiro", msg("results.family", results.family, "ichiro"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ichiro/5.0 (http://help.goo.ne.jp/door/crawler.html) correctly", function() {
		const results = uaParser("ichiro/5.0 (http://help.goo.ne.jp/door/crawler.html)");
		assert.strictEqual(results.family, "ichiro", msg("results.family", results.family, "ichiro"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DoCoMo/2.0 P900i(c100;TB;W24H11)(compatible; ichiro/mobile goo;  http://help.goo.ne.jp/door/crawler.html) correctly", function() {
		const results = uaParser("DoCoMo/2.0 P900i(c100;TB;W24H11)(compatible; ichiro/mobile goo;  http://help.goo.ne.jp/door/crawler.html)");
		assert.strictEqual(results.family, "ichiro/mobile", msg("results.family", results.family, "ichiro/mobile"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 08_800w_web1 (larbin2.6.3@unspecified.mail) correctly", function() {
		const results = uaParser("08_800w_web1 (larbin2.6.3@unspecified.mail)");
		assert.strictEqual(results.family, "larbin", msg("results.family", results.family, "larbin"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; .NET CLR 1.1.4322; LYCOSA;+http://lycosa.se) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1; .NET CLR 1.1.4322; LYCOSA;+http://lycosa.se)");
		assert.strictEqual(results.family, "lycos", msg("results.family", results.family, "lycos"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses masidani_bot_v0.3 (info@masidani.com) correctly", function() {
		const results = uaParser("masidani_bot_v0.3 (info@masidani.com)");
		assert.strictEqual(results.family, "masidani_bot", msg("results.family", results.family, "masidani_bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses mozDex/0.05-dev (mozDex; http://www.mozdex.com/bot.html; spider@mozdex.com) correctly", function() {
		const results = uaParser("mozDex/0.05-dev (mozDex; http://www.mozdex.com/bot.html; spider@mozdex.com)");
		assert.strictEqual(results.family, "mozDex", msg("results.family", results.family, "mozDex"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "05", msg("results.minor", results.minor || "", "05"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses msnbot/2.0 (+http://search.msn.com/msnbot.htm) correctly", function() {
		const results = uaParser("msnbot/2.0 (+http://search.msn.com/msnbot.htm)");
		assert.strictEqual(results.family, "msnbot", msg("results.family", results.family, "msnbot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.2.13) Gecko/20101203 Firefox/3.6.13 (.NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; msnbot/2.1) Jakarta Commons-HttpClient/3.0-rc3 PHPCrawl GStreamer souphttpsrc libsoup/2.27.4 PycURL/7.19.0 XML-RPC for PHP 2.2.1 GoogleFriendConnect/1.0 HTMLParser/1.6 gPodder/0.15.2 ( http://gpodder.org/) anw webtool LoadControl/1.3 WinHttp urlgrabber/3.1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.9.2.13) Gecko/20101203 Firefox/3.6.13 (.NET CLR 1.1.4322; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; msnbot/2.1) Jakarta Commons-HttpClient/3.0-rc3 PHPCrawl GStreamer souphttpsrc libsoup/2.27.4 PycURL/7.19.0 XML-RPC for PHP 2.2.1 GoogleFriendConnect/1.0 HTMLParser/1.6 gPodder/0.15.2 ( http://gpodder.org/) anw webtool LoadControl/1.3 WinHttp urlgrabber/3.1.0");
		assert.strictEqual(results.family, "msnbot", msg("results.family", results.family, "msnbot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses msnbot-media/1.0 ( http://search.msn.com/msnbot.htm) correctly", function() {
		const results = uaParser("msnbot-media/1.0 ( http://search.msn.com/msnbot.htm)");
		assert.strictEqual(results.family, "msnbot-media", msg("results.family", results.family, "msnbot-media"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses msnbot-media/2.0b (+http://search.msn.com/msnbot.htm) correctly", function() {
		const results = uaParser("msnbot-media/2.0b (+http://search.msn.com/msnbot.htm)");
		assert.strictEqual(results.family, "msnbot-media", msg("results.family", results.family, "msnbot-media"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MSRBOT (http://research.microsoft.com/research/sv/msrbot) correctly", function() {
		const results = uaParser("MSRBOT (http://research.microsoft.com/research/sv/msrbot)");
		assert.strictEqual(results.family, "msrbot", msg("results.family", results.family, "msrbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; scooter; .NET CLR 1.0.3705) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; scooter; .NET CLR 1.0.3705)");
		assert.strictEqual(results.family, "scooter", msg("results.family", results.family, "scooter"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; focuseekbot) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; focuseekbot)");
		assert.strictEqual(results.family, "seekbot", msg("results.family", results.family, "seekbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses semanticdiscovery/0.2(http://www.semanticdiscovery.com/sd/robot.html) correctly", function() {
		const results = uaParser("semanticdiscovery/0.2(http://www.semanticdiscovery.com/sd/robot.html)");
		assert.strictEqual(results.family, "semanticdiscovery", msg("results.family", results.family, "semanticdiscovery"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses semanticdiscovery/2.0(http://www.semanticdiscovery.com/robot.html) correctly", function() {
		const results = uaParser("semanticdiscovery/2.0(http://www.semanticdiscovery.com/robot.html)");
		assert.strictEqual(results.family, "semanticdiscovery", msg("results.family", results.family, "semanticdiscovery"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses voyager/1.0 correctly", function() {
		const results = uaParser("voyager/1.0");
		assert.strictEqual(results.family, "voyager", msg("results.family", results.family, "voyager"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses voyager/2.0 (http://www.kosmix.com/crawler.html) correctly", function() {
		const results = uaParser("voyager/2.0 (http://www.kosmix.com/crawler.html)");
		assert.strictEqual(results.family, "voyager", msg("results.family", results.family, "voyager"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses http://www.almaden.ibm.com/cs/crawler correctly", function() {
		const results = uaParser("http://www.almaden.ibm.com/cs/crawler");
		assert.strictEqual(results.family, "www.almaden.ibm.com", msg("results.family", results.family, "www.almaden.ibm.com"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 449 Overture-WebCrawler/3.8/Fresh (atw-crawler at fast dot no; http://fast.no/support/crawler.asp correctly", function() {
		const results = uaParser("449 Overture-WebCrawler/3.8/Fresh (atw-crawler at fast dot no; http://fast.no/support/crawler.asp");
		assert.strictEqual(results.family, "449 Overture-WebCrawler", msg("results.family", results.family, "449 Overture-WebCrawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; 80bot/0.71; http://www.80legs.com/spider.html;) Gecko/2008032620 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; 80bot/0.71; http://www.80legs.com/spider.html;) Gecko/2008032620");
		assert.strictEqual(results.family, "80bot", msg("results.family", results.family, "80bot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "71", msg("results.minor", results.minor || "", "71"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses A6-Indexer/1.0 (http://www.a6corp.com/a6-web-scraping-policy/) correctly", function() {
		const results = uaParser("A6-Indexer/1.0 (http://www.a6corp.com/a6-web-scraping-policy/)");
		assert.strictEqual(results.family, "A6-Indexer", msg("results.family", results.family, "A6-Indexer"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AcquiloSpider/1.0,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("AcquiloSpider/1.0,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "AcquiloSpider", msg("results.family", results.family, "AcquiloSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Aleksika Spider/1.0 ( http://www.aleksika.com/) correctly", function() {
		const results = uaParser("Aleksika Spider/1.0 ( http://www.aleksika.com/)");
		assert.strictEqual(results.family, "Aleksika Spider", msg("results.family", results.family, "Aleksika Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AmorankSpider/0.1; +http://amorank.com/webcrawler.html correctly", function() {
		const results = uaParser("AmorankSpider/0.1; +http://amorank.com/webcrawler.html");
		assert.strictEqual(results.family, "AmorankSpider", msg("results.family", results.family, "AmorankSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AnomaliesBot/0.06-dev (The Anomalies Network Search Spider; http://www.anomalies.net; info@anomalies.net) correctly", function() {
		const results = uaParser("AnomaliesBot/0.06-dev (The Anomalies Network Search Spider; http://www.anomalies.net; info@anomalies.net)");
		assert.strictEqual(results.family, "AnomaliesBot", msg("results.family", results.family, "AnomaliesBot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "06", msg("results.minor", results.minor || "", "06"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Automattic Analytics Crawler/0.1; http://wordpress.com/crawler/ correctly", function() {
		const results = uaParser("Automattic Analytics Crawler/0.1; http://wordpress.com/crawler/");
		assert.strictEqual(results.family, "Automattic Analytics Crawler", msg("results.family", results.family, "Automattic Analytics Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Automattic Analytics Crawler/0.2;+http://wordpress.com/crawler/ correctly", function() {
		const results = uaParser("Automattic Analytics Crawler/0.2;+http://wordpress.com/crawler/");
		assert.strictEqual(results.family, "Automattic Analytics Crawler", msg("results.family", results.family, "Automattic Analytics Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; BLEXBot/1.0; +http://webmeup.com/crawler.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; BLEXBot/1.0; +http://webmeup.com/crawler.html)");
		assert.strictEqual(results.family, "BLEXBot", msg("results.family", results.family, "BLEXBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BabalooSpider/1.2 (BabalooSpider; http://www.babaloo.si; spider@babaloo.si) correctly", function() {
		const results = uaParser("BabalooSpider/1.2 (BabalooSpider; http://www.babaloo.si; spider@babaloo.si)");
		assert.strictEqual(results.family, "BabalooSpider", msg("results.family", results.family, "BabalooSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BabalooSpider/1.3 (BabalooSpider; http://www.babaloo.si; spider@babaloo.si) correctly", function() {
		const results = uaParser("BabalooSpider/1.3 (BabalooSpider; http://www.babaloo.si; spider@babaloo.si)");
		assert.strictEqual(results.family, "BabalooSpider", msg("results.family", results.family, "BabalooSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BebopBot/2.5.1 (compatible; media crawler V1;  http://www.apassion4jazz.net/bebopbot.html;) correctly", function() {
		const results = uaParser("BebopBot/2.5.1 (compatible; media crawler V1;  http://www.apassion4jazz.net/bebopbot.html;)");
		assert.strictEqual(results.family, "BebopBot", msg("results.family", results.family, "BebopBot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (compatible; BlinkaCrawler/1.0;  http://www.blinka.jp/crawler/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; BlinkaCrawler/1.0;  http://www.blinka.jp/crawler/)");
		assert.strictEqual(results.family, "BlinkaCrawler", msg("results.family", results.family, "BlinkaCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BlogRangerCrawler/1.0 correctly", function() {
		const results = uaParser("BlogRangerCrawler/1.0");
		assert.strictEqual(results.family, "BlogRangerCrawler", msg("results.family", results.family, "BlogRangerCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Brekiri crawler/1.0 correctly", function() {
		const results = uaParser("Brekiri crawler/1.0");
		assert.strictEqual(results.family, "Brekiri crawler", msg("results.family", results.family, "Brekiri crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BurstFindCrawler/1.1 (crawler.burstfind.com; http://crawler.burstfind.com; crawler@burstfind.com) correctly", function() {
		const results = uaParser("BurstFindCrawler/1.1 (crawler.burstfind.com; http://crawler.burstfind.com; crawler@burstfind.com)");
		assert.strictEqual(results.family, "BurstFindCrawler", msg("results.family", results.family, "BurstFindCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CCBot/1.0 ( http://www.commoncrawl.org/bot.html) correctly", function() {
		const results = uaParser("CCBot/1.0 ( http://www.commoncrawl.org/bot.html)");
		assert.strictEqual(results.family, "CCBot", msg("results.family", results.family, "CCBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CCBot/2.0 (http://commoncrawl.org/faq/) correctly", function() {
		const results = uaParser("CCBot/2.0 (http://commoncrawl.org/faq/)");
		assert.strictEqual(results.family, "CCBot", msg("results.family", results.family, "CCBot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CamontSpider/1.0  http://epweb2.ph.bham.ac.uk/user/slater/camont/info.html correctly", function() {
		const results = uaParser("CamontSpider/1.0  http://epweb2.ph.bham.ac.uk/user/slater/camont/info.html");
		assert.strictEqual(results.family, "CamontSpider", msg("results.family", results.family, "CamontSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CazoodleBot/0.1 (CazoodleBot Crawler; http://www.cazoodle.com; mqbot@cazoodle.com) correctly", function() {
		const results = uaParser("CazoodleBot/0.1 (CazoodleBot Crawler; http://www.cazoodle.com; mqbot@cazoodle.com)");
		assert.strictEqual(results.family, "CazoodleBot", msg("results.family", results.family, "CazoodleBot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; CloudServerMarketSpider/1.0; +http://www.cloudservermarket.com/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; CloudServerMarketSpider/1.0; +http://www.cloudservermarket.com/spider.html)");
		assert.strictEqual(results.family, "CloudServerMarketSpider", msg("results.family", results.family, "CloudServerMarketSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; CompSpyBot/1.0; +http://www.compspy.com/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; CompSpyBot/1.0; +http://www.compspy.com/spider.html)");
		assert.strictEqual(results.family, "CompSpyBot", msg("results.family", results.family, "CompSpyBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ConveraMultiMediaCrawler/0.1 ( http://www.authoritativeweb.com/crawl) correctly", function() {
		const results = uaParser("ConveraMultiMediaCrawler/0.1 ( http://www.authoritativeweb.com/crawl)");
		assert.strictEqual(results.family, "ConveraMultiMediaCrawler", msg("results.family", results.family, "ConveraMultiMediaCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CosmixCrawler/0.1 correctly", function() {
		const results = uaParser("CosmixCrawler/0.1");
		assert.strictEqual(results.family, "CosmixCrawler", msg("results.family", results.family, "CosmixCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Crawl/0.1 langcrawl/0.1 correctly", function() {
		const results = uaParser("Crawl/0.1 langcrawl/0.1");
		assert.strictEqual(results.family, "Crawl", msg("results.family", results.family, "Crawl"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; CrawlBot/1.0.0) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; CrawlBot/1.0.0)");
		assert.strictEqual(results.family, "CrawlBot", msg("results.family", results.family, "CrawlBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Crawllybot/0.1/0.1 (Crawllybot/0.1; http://www.crawlly.com; crawler@crawlly.com) correctly", function() {
		const results = uaParser("Crawllybot/0.1/0.1 (Crawllybot/0.1; http://www.crawlly.com; crawler@crawlly.com)");
		assert.strictEqual(results.family, "Crawllybot", msg("results.family", results.family, "Crawllybot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Crawly/1.9;  http://92.51.162.40/crawler.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Crawly/1.9;  http://92.51.162.40/crawler.html)");
		assert.strictEqual(results.family, "Crawly", msg("results.family", results.family, "Crawly"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Crawlzilla/1.0 (Crawlzilla; http://www.crawlzilla.com; crawler@crawlzilla.com) correctly", function() {
		const results = uaParser("Crawlzilla/1.0 (Crawlzilla; http://www.crawlzilla.com; crawler@crawlzilla.com)");
		assert.strictEqual(results.family, "Crawlzilla", msg("results.family", results.family, "Crawlzilla"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CydralSpider/3.2 (Cydral Image Search; http://www.cydral.com) correctly", function() {
		const results = uaParser("CydralSpider/3.2 (Cydral Image Search; http://www.cydral.com)");
		assert.strictEqual(results.family, "CydralSpider", msg("results.family", results.family, "CydralSpider"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DealGates Bot/1.1 by Luc Michalski (http://spider.dealgates.com/bot.html) correctly", function() {
		const results = uaParser("DealGates Bot/1.1 by Luc Michalski (http://spider.dealgates.com/bot.html)");
		assert.strictEqual(results.family, "DealGates Bot", msg("results.family", results.family, "DealGates Bot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Denodo IECrawler/4.5,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Denodo IECrawler/4.5,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Denodo IECrawler", msg("results.family", results.family, "Denodo IECrawler"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Diffbot/0.1 correctly", function() {
		const results = uaParser("Diffbot/0.1");
		assert.strictEqual(results.family, "Diffbot", msg("results.family", results.family, "Diffbot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DomainCrawler/2.0 (info@domaincrawler.com; http://www.domaincrawler.com/bot correctly", function() {
		const results = uaParser("DomainCrawler/2.0 (info@domaincrawler.com; http://www.domaincrawler.com/bot");
		assert.strictEqual(results.family, "DomainCrawler", msg("results.family", results.family, "DomainCrawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DotBot/1.0.1 (http://www.dotnetdotcom.org/, crawler@dotnetdotcom.org) correctly", function() {
		const results = uaParser("DotBot/1.0.1 (http://www.dotnetdotcom.org/, crawler@dotnetdotcom.org)");
		assert.strictEqual(results.family, "DotBot", msg("results.family", results.family, "DotBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (compatible; DotBot/1.1; http://www.dotnetdotcom.org/, crawler@dotnetdotcom.org) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; DotBot/1.1; http://www.dotnetdotcom.org/, crawler@dotnetdotcom.org)");
		assert.strictEqual(results.family, "DotBot", msg("results.family", results.family, "DotBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DotSpotsBot/0.2 (crawler; support at dotspots.com) correctly", function() {
		const results = uaParser("DotSpotsBot/0.2 (crawler; support at dotspots.com)");
		assert.strictEqual(results.family, "DotSpotsBot", msg("results.family", results.family, "DotSpotsBot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ERACrawler/1.0 correctly", function() {
		const results = uaParser("ERACrawler/1.0");
		assert.strictEqual(results.family, "ERACrawler", msg("results.family", results.family, "ERACrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; EventGuruBot/1.0; +http://www.eventguru.com/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; EventGuruBot/1.0; +http://www.eventguru.com/spider.html)");
		assert.strictEqual(results.family, "EventGuruBot", msg("results.family", results.family, "EventGuruBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Ex-Crawler/0.1.5a; powered by ex-crawler; +http://www.ex-crawler.de/) Java/1.6.0_20,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Ex-Crawler/0.1.5a; powered by ex-crawler; +http://www.ex-crawler.de/) Java/1.6.0_20,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Ex-Crawler", msg("results.family", results.family, "Ex-Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses ExactSeek Crawler/0.1 correctly", function() {
		const results = uaParser("ExactSeek Crawler/0.1");
		assert.strictEqual(results.family, "ExactSeek Crawler", msg("results.family", results.family, "ExactSeek Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ExactSeekCrawler/1.0 correctly", function() {
		const results = uaParser("ExactSeekCrawler/1.0");
		assert.strictEqual(results.family, "ExactSeekCrawler", msg("results.family", results.family, "ExactSeekCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FANGCrawl/0.01 correctly", function() {
		const results = uaParser("FANGCrawl/0.01");
		assert.strictEqual(results.family, "FANGCrawl", msg("results.family", results.family, "FANGCrawl"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FAST Enterprise Crawler/6 (www.fastsearch.com) correctly", function() {
		const results = uaParser("FAST Enterprise Crawler/6 (www.fastsearch.com)");
		assert.strictEqual(results.family, "FAST Enterprise Crawler", msg("results.family", results.family, "FAST Enterprise Crawler"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FAST-WebCrawler/2.2.11 (crawler@fast.no; http://www.fast.no/faq/faqfastwebsearch/faqfastwebcrawler.html) correctly", function() {
		const results = uaParser("FAST-WebCrawler/2.2.11 (crawler@fast.no; http://www.fast.no/faq/faqfastwebsearch/faqfastwebcrawler.html)");
		assert.strictEqual(results.family, "FAST-WebCrawler", msg("results.family", results.family, "FAST-WebCrawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "11" || "", msg("results.patch", results.patch || "", "11" || ""));
	});
	it("parses FAST-WebCrawler/3.7 (atw-crawler at fast dot no; http://fast.no/support/crawler.asp) correctly", function() {
		const results = uaParser("FAST-WebCrawler/3.7 (atw-crawler at fast dot no; http://fast.no/support/crawler.asp)");
		assert.strictEqual(results.family, "FAST-WebCrawler", msg("results.family", results.family, "FAST-WebCrawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NokiaN70/. FASTMobileCrawl/6.6 Profile/MIDP-2.0 Configuration/CLDC-1.1 correctly", function() {
		const results = uaParser("NokiaN70/. FASTMobileCrawl/6.6 Profile/MIDP-2.0 Configuration/CLDC-1.1");
		assert.strictEqual(results.family, "FASTMobileCrawl", msg("results.family", results.family, "FASTMobileCrawl"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FlaxCrawler/1.0,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("FlaxCrawler/1.0,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "FlaxCrawler", msg("results.family", results.family, "FlaxCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FyberSpider/1.3 (http://www.fybersearch.com/fyberspider.php) correctly", function() {
		const results = uaParser("FyberSpider/1.3 (http://www.fybersearch.com/fyberspider.php)");
		assert.strictEqual(results.family, "FyberSpider", msg("results.family", results.family, "FyberSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GarlikCrawler/1.2 (http://garlik.com/ correctly", function() {
		const results = uaParser("GarlikCrawler/1.2 (http://garlik.com/");
		assert.strictEqual(results.family, "GarlikCrawler", msg("results.family", results.family, "GarlikCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GematchCrawler/2.1 (http://www.gematch.com/crawler.html) correctly", function() {
		const results = uaParser("GematchCrawler/2.1 (http://www.gematch.com/crawler.html)");
		assert.strictEqual(results.family, "GematchCrawler", msg("results.family", results.family, "GematchCrawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Gigabot/2.0/gigablast.com/spider.html correctly", function() {
		const results = uaParser("Gigabot/2.0/gigablast.com/spider.html");
		assert.strictEqual(results.family, "Gigabot", msg("results.family", results.family, "Gigabot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Gigabot/3.0 (http://www.gigablast.com/spider.html) correctly", function() {
		const results = uaParser("Gigabot/3.0 (http://www.gigablast.com/spider.html)");
		assert.strictEqual(results.family, "Gigabot", msg("results.family", results.family, "Gigabot"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GingerCrawler/1.0 (Language Assistant for Dyslexics; www.gingersoftware.com/crawler_agent.htm; support at ginger software dot com) correctly", function() {
		const results = uaParser("GingerCrawler/1.0 (Language Assistant for Dyslexics; www.gingersoftware.com/crawler_agent.htm; support at ginger software dot com)");
		assert.strictEqual(results.family, "GingerCrawler", msg("results.family", results.family, "GingerCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GoGuidesBot/0.0.1 (GoGuides Indexing Spider; http://www.goguides.org/spider.html) correctly", function() {
		const results = uaParser("GoGuidesBot/0.0.1 (GoGuides Indexing Spider; http://www.goguides.org/spider.html)");
		assert.strictEqual(results.family, "GoGuidesBot", msg("results.family", results.family, "GoGuidesBot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (compatible; GrapeshotCrawler/2.0; +http://www.grapeshot.co.uk/crawler.php) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; GrapeshotCrawler/2.0; +http://www.grapeshot.co.uk/crawler.php)");
		assert.strictEqual(results.family, "GrapeshotCrawler", msg("results.family", results.family, "GrapeshotCrawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HPI FeedCrawler/0.1 (+http://www.hpi.uni-potsdam.de/meinel/bross/feedcrawler) correctly", function() {
		const results = uaParser("HPI FeedCrawler/0.1 (+http://www.hpi.uni-potsdam.de/meinel/bross/feedcrawler)");
		assert.strictEqual(results.family, "HPI FeedCrawler", msg("results.family", results.family, "HPI FeedCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HRCrawler/2.0 (XF86; MacOS x86_64) AppleWebKit/537.31 (KHTML, like Gecko) correctly", function() {
		const results = uaParser("HRCrawler/2.0 (XF86; MacOS x86_64) AppleWebKit/537.31 (KHTML, like Gecko)");
		assert.strictEqual(results.family, "HRCrawler", msg("results.family", results.family, "HRCrawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Hailoobot/1.2;  http://www.hailoo.com/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Hailoobot/1.2;  http://www.hailoo.com/spider.html)");
		assert.strictEqual(results.family, "Hailoobot", msg("results.family", results.family, "Hailoobot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Hatena::Crawler/0.01 correctly", function() {
		const results = uaParser("Hatena::Crawler/0.01");
		assert.strictEqual(results.family, "Hatena::Crawler", msg("results.family", results.family, "Hatena::Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; HiveCrawler/1.2 http://www.businessinsider.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; HiveCrawler/1.2 http://www.businessinsider.com)");
		assert.strictEqual(results.family, "HiveCrawler", msg("results.family", results.family, "HiveCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Host-Spy Crawler/1.2; +http://www.host-spy.com/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Host-Spy Crawler/1.2; +http://www.host-spy.com/)");
		assert.strictEqual(results.family, "Host-Spy Crawler", msg("results.family", results.family, "Host-Spy Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HttpSpider/0.91 correctly", function() {
		const results = uaParser("HttpSpider/0.91");
		assert.strictEqual(results.family, "HttpSpider", msg("results.family", results.family, "HttpSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "91", msg("results.minor", results.minor || "", "91"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HuaweiSymantecSpider/1.0 DSE-support@huaweisymantec.com (compatible; MSIE 7.0; Windows NT 5.1; Trident/4.0; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR ; http://www.huaweisymantec.com/cn/IRL/spider) correctly", function() {
		const results = uaParser("HuaweiSymantecSpider/1.0 DSE-support@huaweisymantec.com (compatible; MSIE 7.0; Windows NT 5.1; Trident/4.0; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR ; http://www.huaweisymantec.com/cn/IRL/spider)");
		assert.strictEqual(results.family, "HuaweiSymantecSpider", msg("results.family", results.family, "HuaweiSymantecSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IRLbot/1.0 ( http://irl.cs.tamu.edu/crawler) correctly", function() {
		const results = uaParser("IRLbot/1.0 ( http://irl.cs.tamu.edu/crawler)");
		assert.strictEqual(results.family, "IRLbot", msg("results.family", results.family, "IRLbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IRLbot/3.0 (compatible; MSIE 6.0; http://irl.cs.tamu.edu/crawler) correctly", function() {
		const results = uaParser("IRLbot/3.0 (compatible; MSIE 6.0; http://irl.cs.tamu.edu/crawler)");
		assert.strictEqual(results.family, "IRLbot", msg("results.family", results.family, "IRLbot"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; Iplexx Spider/1.1) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; Iplexx Spider/1.1)");
		assert.strictEqual(results.family, "Iplexx Spider", msg("results.family", results.family, "Iplexx Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Jambot/0.1.1 (Jambot; http://www.jambot.com/blog; crawler@jambot.com) correctly", function() {
		const results = uaParser("Jambot/0.1.1 (Jambot; http://www.jambot.com/blog; crawler@jambot.com)");
		assert.strictEqual(results.family, "Jambot", msg("results.family", results.family, "Jambot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Jomjaibot/1.0 Crawl (+http://www.jomjaibot.com/) correctly", function() {
		const results = uaParser("Jomjaibot/1.0 Crawl (+http://www.jomjaibot.com/)");
		assert.strictEqual(results.family, "Jomjaibot", msg("results.family", results.family, "Jomjaibot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses KIT webcrawler/0.2.4 correctly", function() {
		const results = uaParser("KIT webcrawler/0.2.4");
		assert.strictEqual(results.family, "KIT webcrawler", msg("results.family", results.family, "KIT webcrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Kyoto-Crawler/2.0 (Mozilla-compatible; kyoto-crawler-contact(at)nlp(dot)kuee(dot)kyoto-u(dot)ac(dot)jp; http://nlp.ist.i.kyoto-u.ac.jp/) correctly", function() {
		const results = uaParser("Kyoto-Crawler/2.0 (Mozilla-compatible; kyoto-crawler-contact(at)nlp(dot)kuee(dot)kyoto-u(dot)ac(dot)jp; http://nlp.ist.i.kyoto-u.ac.jp/)");
		assert.strictEqual(results.family, "Kyoto-Crawler", msg("results.family", results.family, "Kyoto-Crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LB-Crawler/1.0 correctly", function() {
		const results = uaParser("LB-Crawler/1.0");
		assert.strictEqual(results.family, "LB-Crawler", msg("results.family", results.family, "LB-Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LSSRocketCrawler/1.0 LightspeedSystems correctly", function() {
		const results = uaParser("LSSRocketCrawler/1.0 LightspeedSystems");
		assert.strictEqual(results.family, "LSSRocketCrawler", msg("results.family", results.family, "LSSRocketCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.7) Gecko/20040707 Lightningspider/0.9.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 5.1; en-US; rv:1.7) Gecko/20040707 Lightningspider/0.9.2");
		assert.strictEqual(results.family, "Lightningspider", msg("results.family", results.family, "Lightningspider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (compatible; LikaholixCrawler/1.0; +http://mylikes.com/about/crawler) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; LikaholixCrawler/1.0; +http://mylikes.com/about/crawler)");
		assert.strictEqual(results.family, "LikaholixCrawler", msg("results.family", results.family, "LikaholixCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Logict IPv6 Crawler/1.0 (http://ipv6search.logict.net),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Logict IPv6 Crawler/1.0 (http://ipv6search.logict.net),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Logict IPv6 Crawler", msg("results.family", results.family, "Logict IPv6 Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MPICrawler/0.1 +http://kermit.news.cs.nyu.edu/crawler.html,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("MPICrawler/0.1 +http://kermit.news.cs.nyu.edu/crawler.html,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "MPICrawler", msg("results.family", results.family, "MPICrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/8.01 (J2ME/MIDP; MXit WebBot/1.8.3.119) Opera Mini/3.1 correctly", function() {
		const results = uaParser("Opera/8.01 (J2ME/MIDP; MXit WebBot/1.8.3.119) Opera Mini/3.1");
		assert.strictEqual(results.family, "MXit WebBot", msg("results.family", results.family, "MXit WebBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses MetaGeneratorCrawler/1.3.8 (www.metagenerator.info) correctly", function() {
		const results = uaParser("MetaGeneratorCrawler/1.3.8 (www.metagenerator.info)");
		assert.strictEqual(results.family, "MetaGeneratorCrawler", msg("results.family", results.family, "MetaGeneratorCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MetamojiCrawler/1.0; +http://www.metamoji.com/jp/crawler.html correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MetamojiCrawler/1.0; +http://www.metamoji.com/jp/crawler.html");
		assert.strictEqual(results.family, "MetamojiCrawler", msg("results.family", results.family, "MetamojiCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MonkeyCrawl/0.05 (MonkeyCrawl; http://www.monkeymethods.org;  ) correctly", function() {
		const results = uaParser("MonkeyCrawl/0.05 (MonkeyCrawl; http://www.monkeymethods.org;  )");
		assert.strictEqual(results.family, "MonkeyCrawl", msg("results.family", results.family, "MonkeyCrawl"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "05", msg("results.minor", results.minor || "", "05"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla crawl/5.0 (compatible; fairshare.cc +http://fairshare.cc) correctly", function() {
		const results = uaParser("Mozilla crawl/5.0 (compatible; fairshare.cc +http://fairshare.cc)");
		assert.strictEqual(results.family, "Mozilla crawl", msg("results.family", results.family, "Mozilla crawl"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; NLCrawler/2.0.15; Linux 2.6.3-7; i686; en_US)KHTML/3.4.89 (like Gecko) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; NLCrawler/2.0.15; Linux 2.6.3-7; i686; en_US)KHTML/3.4.89 (like Gecko)");
		assert.strictEqual(results.family, "NLCrawler", msg("results.family", results.family, "NLCrawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "15" || "", msg("results.patch", results.patch || "", "15" || ""));
	});
	it("parses NMG Spider/0.3 (szukanko.com) correctly", function() {
		const results = uaParser("NMG Spider/0.3 (szukanko.com)");
		assert.strictEqual(results.family, "NMG Spider", msg("results.family", results.family, "NMG Spider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NalezenCzBot/1.0 (http://www.nalezen.cz/about-crawler) correctly", function() {
		const results = uaParser("NalezenCzBot/1.0 (http://www.nalezen.cz/about-crawler)");
		assert.strictEqual(results.family, "NalezenCzBot", msg("results.family", results.family, "NalezenCzBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NationalDirectory-WebSpider/1.3 correctly", function() {
		const results = uaParser("NationalDirectory-WebSpider/1.3");
		assert.strictEqual(results.family, "NationalDirectory-WebSpider", msg("results.family", results.family, "NationalDirectory-WebSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; NetSeer crawler/2.0; +http://www.netseer.com/crawler.html; crawler@netseer.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; NetSeer crawler/2.0; +http://www.netseer.com/crawler.html; crawler@netseer.com)");
		assert.strictEqual(results.family, "NetSeer crawler", msg("results.family", results.family, "NetSeer crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NetWhatCrawler/0.06-dev (NetWhatCrawler from NetWhat.com; http://www.netwhat.com; support@netwhat.com) correctly", function() {
		const results = uaParser("NetWhatCrawler/0.06-dev (NetWhatCrawler from NetWhat.com; http://www.netwhat.com; support@netwhat.com)");
		assert.strictEqual(results.family, "NetWhatCrawler", msg("results.family", results.family, "NetWhatCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "06", msg("results.minor", results.minor || "", "06"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses compatible; Netseer crawler/2.0; +http://www.netseer.com/crawler.html; crawler@netseer.com correctly", function() {
		const results = uaParser("compatible; Netseer crawler/2.0; +http://www.netseer.com/crawler.html; crawler@netseer.com");
		assert.strictEqual(results.family, "Netseer crawler", msg("results.family", results.family, "Netseer crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses New-Sogou-Spider/1.0 (compatible; MSIE 5.5; Windows 98) correctly", function() {
		const results = uaParser("New-Sogou-Spider/1.0 (compatible; MSIE 5.5; Windows 98)");
		assert.strictEqual(results.family, "New-Sogou-Spider", msg("results.family", results.family, "New-Sogou-Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NewzCrawler/1.9 (compatible; MSIE 6.00; Newz Crawler 1.9; http://www.newzcrawler.com/ ) correctly", function() {
		const results = uaParser("NewzCrawler/1.9 (compatible; MSIE 6.00; Newz Crawler 1.9; http://www.newzcrawler.com/ )");
		assert.strictEqual(results.family, "NewzCrawler", msg("results.family", results.family, "NewzCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NodejsSpider/1.0 correctly", function() {
		const results = uaParser("NodejsSpider/1.0");
		assert.strictEqual(results.family, "NodejsSpider", msg("results.family", results.family, "NodejsSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; .None-Crawler/0.1 +http://domains.ericbinek.None/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; .None-Crawler/0.1 +http://domains.ericbinek.None/)");
		assert.strictEqual(results.family, "None-Crawler", msg("results.family", results.family, "None-Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses OmniExplorer_Bot/6.70 ( http://www.omni-explorer.com) WorldIndexer correctly", function() {
		const results = uaParser("OmniExplorer_Bot/6.70 ( http://www.omni-explorer.com) WorldIndexer");
		assert.strictEqual(results.family, "OmniExplorer_Bot", msg("results.family", results.family, "OmniExplorer_Bot"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "70", msg("results.minor", results.minor || "", "70"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; OpenCrawler/0.1.6.6; http://code.google.com/p/opencrawler/),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; OpenCrawler/0.1.6.6; http://code.google.com/p/opencrawler/),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "OpenCrawler", msg("results.family", results.family, "OpenCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses OpenWebSpider/0.6 (http://www.openwebspider.org) correctly", function() {
		const results = uaParser("OpenWebSpider/0.6 (http://www.openwebspider.org)");
		assert.strictEqual(results.family, "OpenWebSpider", msg("results.family", results.family, "OpenWebSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Overture-WebCrawler/3.8/Fresh (atw-crawler at fast dot no; http://fast.no/support/crawler.asp) correctly", function() {
		const results = uaParser("Overture-WebCrawler/3.8/Fresh (atw-crawler at fast dot no; http://fast.no/support/crawler.asp)");
		assert.strictEqual(results.family, "Overture-WebCrawler", msg("results.family", results.family, "Overture-WebCrawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PArchiveCrawler/1.0 correctly", function() {
		const results = uaParser("PArchiveCrawler/1.0");
		assert.strictEqual(results.family, "PArchiveCrawler", msg("results.family", results.family, "PArchiveCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PercolateCrawler/4 (ops@percolate.com) correctly", function() {
		const results = uaParser("PercolateCrawler/4 (ops@percolate.com)");
		assert.strictEqual(results.family, "PercolateCrawler", msg("results.family", results.family, "PercolateCrawler"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pete-Spider/1.1 correctly", function() {
		const results = uaParser("Pete-Spider/1.1");
		assert.strictEqual(results.family, "Pete-Spider", msg("results.family", results.family, "Pete-Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PicSpider/1.1 (spider@217-20-118-26.internetserviceteam.com; http://www.bildkiste.de) correctly", function() {
		const results = uaParser("PicSpider/1.1 (spider@217-20-118-26.internetserviceteam.com; http://www.bildkiste.de)");
		assert.strictEqual(results.family, "PicSpider", msg("results.family", results.family, "PicSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PluckItCrawler/1.0 (compatible; Mozilla 4.0; MSIE 5.5; http://www.pluck.com;) correctly", function() {
		const results = uaParser("PluckItCrawler/1.0 (compatible; Mozilla 4.0; MSIE 5.5; http://www.pluck.com;)");
		assert.strictEqual(results.family, "PluckItCrawler", msg("results.family", results.family, "PluckItCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; PornSpider/1.0; +http://www.pornspider.net) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; PornSpider/1.0; +http://www.pornspider.net)");
		assert.strictEqual(results.family, "PornSpider", msg("results.family", results.family, "PornSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PortalBSpider/2.0 (spider@portalb.com) correctly", function() {
		const results = uaParser("PortalBSpider/2.0 (spider@portalb.com)");
		assert.strictEqual(results.family, "PortalBSpider", msg("results.family", results.family, "PortalBSpider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; ProCogBot/1.0; +http://www.procog.com/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; ProCogBot/1.0; +http://www.procog.com/spider.html)");
		assert.strictEqual(results.family, "ProCogBot", msg("results.family", results.family, "ProCogBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ProloCrawler/1.0 (http://www.prolo.com) correctly", function() {
		const results = uaParser("ProloCrawler/1.0 (http://www.prolo.com)");
		assert.strictEqual(results.family, "ProloCrawler", msg("results.family", results.family, "ProloCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PsSpider/0.7 (spider@bildkiste.de) correctly", function() {
		const results = uaParser("PsSpider/0.7 (spider@bildkiste.de)");
		assert.strictEqual(results.family, "PsSpider", msg("results.family", results.family, "PsSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RSSIncludeBot/1.0 (http://www.rssinclude.com/spider) correctly", function() {
		const results = uaParser("RSSIncludeBot/1.0 (http://www.rssinclude.com/spider)");
		assert.strictEqual(results.family, "RSSIncludeBot", msg("results.family", results.family, "RSSIncludeBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1; Trident/4.0; Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1) ;  Embedded Web Browser from: http://bsalsa.com/; RSScrawler/4.0 (compatible; MSIE 6.0; Windows NT 5.0); .NET CLR 2.0.50727 correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.1; Trident/4.0; Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; SV1) ;  Embedded Web Browser from: http://bsalsa.com/; RSScrawler/4.0 (compatible; MSIE 6.0; Windows NT 5.0); .NET CLR 2.0.50727");
		assert.strictEqual(results.family, "RSScrawler", msg("results.family", results.family, "RSScrawler"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RSpider/1.0 correctly", function() {
		const results = uaParser("RSpider/1.0");
		assert.strictEqual(results.family, "RSpider", msg("results.family", results.family, "RSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RyzeCrawler/1.1.0 (+http://www.ryze.nl/crawler/) correctly", function() {
		const results = uaParser("RyzeCrawler/1.1.0 (+http://www.ryze.nl/crawler/)");
		assert.strictEqual(results.family, "RyzeCrawler", msg("results.family", results.family, "RyzeCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses SMXCrawler/1.0 (www.socialmetrix.com) correctly", function() {
		const results = uaParser("SMXCrawler/1.0 (www.socialmetrix.com)");
		assert.strictEqual(results.family, "SMXCrawler", msg("results.family", results.family, "SMXCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; SRCCN!Spider/1.1; +http://site.srccn.com/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; SRCCN!Spider/1.1; +http://site.srccn.com/spider.html)");
		assert.strictEqual(results.family, "SRCCN!Spider", msg("results.family", results.family, "SRCCN!Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; SWEBot/1.0; +http://swebot-crawler.net) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; SWEBot/1.0; +http://swebot-crawler.net)");
		assert.strictEqual(results.family, "SWEBot", msg("results.family", results.family, "SWEBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sangfor Spider/0.7 correctly", function() {
		const results = uaParser("Sangfor Spider/0.7");
		assert.strictEqual(results.family, "Sangfor Spider", msg("results.family", results.family, "Sangfor Spider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Sailfish 3.0; Mobile; rv:45.0) Gecko/45.0 Firefox/45.0 SailfishBrowser/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Sailfish 3.0; Mobile; rv:45.0) Gecko/45.0 Firefox/45.0 SailfishBrowser/1.0");
		assert.strictEqual(results.family, "Sailfish Browser", msg("results.family", results.family, "Sailfish Browser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Sailfish 3.0; Mobile; rv:45.0) Gecko/45.0 Firefox/45.0 SailfishBrowser/1.2.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Sailfish 3.0; Mobile; rv:45.0) Gecko/45.0 Firefox/45.0 SailfishBrowser/1.2.3");
		assert.strictEqual(results.family, "Sailfish Browser", msg("results.family", results.family, "Sailfish Browser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 5.0.1; SAMSUNG GT-I9506-ORANGE Build/LRX22C) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/2.1 Chrome/34.0.1847.76 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 5.0.1; SAMSUNG GT-I9506-ORANGE Build/LRX22C) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/2.1 Chrome/34.0.1847.76 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Samsung Internet", msg("results.family", results.family, "Samsung Internet"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 5.0.2; SAMSUNG SM-T800 Build/LRX22G) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/3.0 Chrome/38.0.2125.102 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 5.0.2; SAMSUNG SM-T800 Build/LRX22G) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/3.0 Chrome/38.0.2125.102 Safari/537.36");
		assert.strictEqual(results.family, "Samsung Internet", msg("results.family", results.family, "Samsung Internet"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 5.1.1; SAMSUNG SM-G920F Build/LMY47X) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/3.2 Chrome/38.0.2125.102 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 5.1.1; SAMSUNG SM-G920F Build/LMY47X) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/3.2 Chrome/38.0.2125.102 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Samsung Internet", msg("results.family", results.family, "Samsung Internet"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 5.0.2; SAMSUNG SM-T710 Build/LRX22G) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/3.5 Chrome/38.0.2125.102 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 5.0.2; SAMSUNG SM-T710 Build/LRX22G) AppleWebKit/537.36 (KHTML, like Gecko) SamsungBrowser/3.5 Chrome/38.0.2125.102 Safari/537.36");
		assert.strictEqual(results.family, "Samsung Internet", msg("results.family", results.family, "Samsung Internet"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ScSpider/0.2 correctly", function() {
		const results = uaParser("ScSpider/0.2");
		assert.strictEqual(results.family, "ScSpider", msg("results.family", results.family, "ScSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ScollSpider/2.0 ( http://www.webwobot.com/ScollSpider.php) correctly", function() {
		const results = uaParser("ScollSpider/2.0 ( http://www.webwobot.com/ScollSpider.php)");
		assert.strictEqual(results.family, "ScollSpider", msg("results.family", results.family, "ScollSpider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Screaming Frog SEO Spider/1.10 correctly", function() {
		const results = uaParser("Screaming Frog SEO Spider/1.10");
		assert.strictEqual(results.family, "Screaming Frog SEO Spider", msg("results.family", results.family, "Screaming Frog SEO Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Screaming Frog SEO Spider/2,01 correctly", function() {
		const results = uaParser("Screaming Frog SEO Spider/2,01");
		assert.strictEqual(results.family, "Screaming Frog SEO Spider", msg("results.family", results.family, "Screaming Frog SEO Spider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SearchSpider/1.2.10 correctly", function() {
		const results = uaParser("SearchSpider/1.2.10");
		assert.strictEqual(results.family, "SearchSpider", msg("results.family", results.family, "SearchSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "10" || "", msg("results.patch", results.patch || "", "10" || ""));
	});
	it("parses Searchspider/1.2 (SearchSpider; http://www.searchspider.com; webmaster@searchspider.com) correctly", function() {
		const results = uaParser("Searchspider/1.2 (SearchSpider; http://www.searchspider.com; webmaster@searchspider.com)");
		assert.strictEqual(results.family, "Searchspider", msg("results.family", results.family, "Searchspider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SimpleCrawler/0.1 correctly", function() {
		const results = uaParser("SimpleCrawler/0.1");
		assert.strictEqual(results.family, "SimpleCrawler", msg("results.family", results.family, "SimpleCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SlugBug Spider/0.1 beta (SlugBug.com search engine; http://www.slugbug.com) correctly", function() {
		const results = uaParser("SlugBug Spider/0.1 beta (SlugBug.com search engine; http://www.slugbug.com)");
		assert.strictEqual(results.family, "SlugBug Spider", msg("results.family", results.family, "SlugBug Spider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SmartAndSimpleWebCrawler/1.3 (https://crawler.dev.java.net),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("SmartAndSimpleWebCrawler/1.3 (https://crawler.dev.java.net),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "SmartAndSimpleWebCrawler", msg("results.family", results.family, "SmartAndSimpleWebCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Linux; Socialradarbot/2.0; en-US; crawler@infegy.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Linux; Socialradarbot/2.0; en-US; crawler@infegy.com)");
		assert.strictEqual(results.family, "Socialradarbot", msg("results.family", results.family, "Socialradarbot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou Orion spider/4.0( http://www.sogou.com/docs/help/webmasters.htm#07) correctly", function() {
		const results = uaParser("Sogou Orion spider/4.0( http://www.sogou.com/docs/help/webmasters.htm#07)");
		assert.strictEqual(results.family, "Sogou Orion spider", msg("results.family", results.family, "Sogou Orion spider"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou Pic Spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07) correctly", function() {
		const results = uaParser("Sogou Pic Spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07)");
		assert.strictEqual(results.family, "Sogou Pic Spider", msg("results.family", results.family, "Sogou Pic Spider"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou Push Spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07) correctly", function() {
		const results = uaParser("Sogou Push Spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07)");
		assert.strictEqual(results.family, "Sogou Push Spider", msg("results.family", results.family, "Sogou Push Spider"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou develop spider/4.0( http://www.sogou.com/docs/help/webmasters.htm#07) correctly", function() {
		const results = uaParser("Sogou develop spider/4.0( http://www.sogou.com/docs/help/webmasters.htm#07)");
		assert.strictEqual(results.family, "Sogou develop spider", msg("results.family", results.family, "Sogou develop spider"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou head spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07) correctly", function() {
		const results = uaParser("Sogou head spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07)");
		assert.strictEqual(results.family, "Sogou head spider", msg("results.family", results.family, "Sogou head spider"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou web spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07) correctly", function() {
		const results = uaParser("Sogou web spider/3.0( http://www.sogou.com/docs/help/webmasters.htm#07)");
		assert.strictEqual(results.family, "Sogou web spider", msg("results.family", results.family, "Sogou web spider"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou-Test-Spider/4.0 (compatible; MSIE 5.5; Windows 98) correctly", function() {
		const results = uaParser("Sogou-Test-Spider/4.0 (compatible; MSIE 5.5; Windows 98)");
		assert.strictEqual(results.family, "Sogou-Test-Spider", msg("results.family", results.family, "Sogou-Test-Spider"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Sosoimagespider/2.0; +http://help.soso.com/soso-image-spider.htm) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Sosoimagespider/2.0; +http://help.soso.com/soso-image-spider.htm)");
		assert.strictEqual(results.family, "Sosoimagespider", msg("results.family", results.family, "Sosoimagespider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Sosospider/2.0; +http://help.soso.com/webspider.htm) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Sosospider/2.0; +http://help.soso.com/webspider.htm)");
		assert.strictEqual(results.family, "Sosospider", msg("results.family", results.family, "Sosospider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Spider/5.0 correctly", function() {
		const results = uaParser("Spider/5.0");
		assert.strictEqual(results.family, "Spider", msg("results.family", results.family, "Spider"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SpokeSpider/1.0 (http://support.spoke.com/webspider/) Mozilla/5.0 (not really) correctly", function() {
		const results = uaParser("SpokeSpider/1.0 (http://support.spoke.com/webspider/) Mozilla/5.0 (not really)");
		assert.strictEqual(results.family, "SpokeSpider", msg("results.family", results.family, "SpokeSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; StoneSunSpider/1.1) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; StoneSunSpider/1.1)");
		assert.strictEqual(results.family, "StoneSunSpider", msg("results.family", results.family, "StoneSunSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; StreamScraper/1.0; +http://code.google.com/p/streamscraper/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; StreamScraper/1.0; +http://code.google.com/p/streamscraper/)");
		assert.strictEqual(results.family, "StreamScraper", msg("results.family", results.family, "StreamScraper"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.5; SuperSpider/139; Windows 98; Win 9x 4.90) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.5; SuperSpider/139; Windows 98; Win 9x 4.90)");
		assert.strictEqual(results.family, "SuperSpider", msg("results.family", results.family, "SuperSpider"));
		assert.strictEqual(results.major || "", "139", msg("results.major", results.major || "", "139"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses T3census-Crawler/1.0 correctly", function() {
		const results = uaParser("T3census-Crawler/1.0");
		assert.strictEqual(results.family, "T3census-Crawler", msg("results.family", results.family, "T3census-Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TECOMAC-Crawler/0.3 correctly", function() {
		const results = uaParser("TECOMAC-Crawler/0.3");
		assert.strictEqual(results.family, "TECOMAC-Crawler", msg("results.family", results.family, "TECOMAC-Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Tasapspider/0.9 correctly", function() {
		const results = uaParser("Tasapspider/0.9");
		assert.strictEqual(results.family, "Tasapspider", msg("results.family", results.family, "Tasapspider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TinEye-bot/0.02 (see http://www.tineye.com/crawler.html) correctly", function() {
		const results = uaParser("TinEye-bot/0.02 (see http://www.tineye.com/crawler.html)");
		assert.strictEqual(results.family, "TinEye-bot", msg("results.family", results.family, "TinEye-bot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "02", msg("results.minor", results.minor || "", "02"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Top10Ranking Spider/3.1 ( http://www.top10Ranking.nl/, Top10ranking.nl heeft op een aantal woorden uw posities in Google gecheckt) correctly", function() {
		const results = uaParser("Top10Ranking Spider/3.1 ( http://www.top10Ranking.nl/, Top10ranking.nl heeft op een aantal woorden uw posities in Google gecheckt)");
		assert.strictEqual(results.family, "Top10Ranking Spider", msg("results.family", results.family, "Top10Ranking Spider"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TouTrix crawler/1.0 correctly", function() {
		const results = uaParser("TouTrix crawler/1.0");
		assert.strictEqual(results.family, "TouTrix crawler", msg("results.family", results.family, "TouTrix crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; TridentSpider/3.1) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; TridentSpider/3.1)");
		assert.strictEqual(results.family, "TridentSpider", msg("results.family", results.family, "TridentSpider"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TurnitinBot/1.5 (http://www.turnitin.com/robot/crawlerinfo.html) correctly", function() {
		const results = uaParser("TurnitinBot/1.5 (http://www.turnitin.com/robot/crawlerinfo.html)");
		assert.strictEqual(results.family, "TurnitinBot", msg("results.family", results.family, "TurnitinBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TurnitinBot/3.0 (http://www.turnitin.com/robot/crawlerinfo.html) correctly", function() {
		const results = uaParser("TurnitinBot/3.0 (http://www.turnitin.com/robot/crawlerinfo.html)");
		assert.strictEqual(results.family, "TurnitinBot", msg("results.family", results.family, "TurnitinBot"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Linux i686; en-US; URLfilterDB-crawler/1.0) ufdb/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Linux i686; en-US; URLfilterDB-crawler/1.0) ufdb/1.0");
		assert.strictEqual(results.family, "URLfilterDB-crawler", msg("results.family", results.family, "URLfilterDB-crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (URLfilterDB-crawler/1.1) ufdb/1.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (URLfilterDB-crawler/1.1) ufdb/1.0");
		assert.strictEqual(results.family, "URLfilterDB-crawler", msg("results.family", results.family, "URLfilterDB-crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses VinjaVideoSpider/1.1 correctly", function() {
		const results = uaParser("VinjaVideoSpider/1.1");
		assert.strictEqual(results.family, "VinjaVideoSpider", msg("results.family", results.family, "VinjaVideoSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses VisBot/2.0 (Visvo.com Crawler; http://www.visvo.com/bot.html; bot@visvo.com) correctly", function() {
		const results = uaParser("VisBot/2.0 (Visvo.com Crawler; http://www.visvo.com/bot.html; bot@visvo.com)");
		assert.strictEqual(results.family, "VisBot", msg("results.family", results.family, "VisBot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; VodpodCrawler/1.0; +http://vodpod.com/site/help) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; VodpodCrawler/1.0; +http://vodpod.com/site/help)");
		assert.strictEqual(results.family, "VodpodCrawler", msg("results.family", results.family, "VodpodCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WSDLSpider/1.0 (http://www.wsdlworld.com) correctly", function() {
		const results = uaParser("WSDLSpider/1.0 (http://www.wsdlworld.com)");
		assert.strictEqual(results.family, "WSDLSpider", msg("results.family", results.family, "WSDLSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; WangIDSpider/1.0; +http://www.wangid.com/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; WangIDSpider/1.0; +http://www.wangid.com/spider.html)");
		assert.strictEqual(results.family, "WangIDSpider", msg("results.family", results.family, "WangIDSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Web-Robot/5.0 (en-US; web-robot.com/policy.html) Web-Robot Crawler/2.0.3 correctly", function() {
		const results = uaParser("Web-Robot/5.0 (en-US; web-robot.com/policy.html) Web-Robot Crawler/2.0.3");
		assert.strictEqual(results.family, "Web-Robot", msg("results.family", results.family, "Web-Robot"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WebAlta Crawler/1.3.18 (http://www.webalta.net/ru/about_webmaster.html) (Windows; U; Windows NT 5.1; ru-RU) correctly", function() {
		const results = uaParser("WebAlta Crawler/1.3.18 (http://www.webalta.net/ru/about_webmaster.html) (Windows; U; Windows NT 5.1; ru-RU)");
		assert.strictEqual(results.family, "WebAlta Crawler", msg("results.family", results.family, "WebAlta Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "18" || "", msg("results.patch", results.patch || "", "18" || ""));
	});
	it("parses WebAlta Crawler/2.0 (http://www.webalta.net/ru/about_webmaster.html) (Windows; U; Windows NT 5.1; ru-RU) correctly", function() {
		const results = uaParser("WebAlta Crawler/2.0 (http://www.webalta.net/ru/about_webmaster.html) (Windows; U; Windows NT 5.1; ru-RU)");
		assert.strictEqual(results.family, "WebAlta Crawler", msg("results.family", results.family, "WebAlta Crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WebIndexer/1-dev (Web Indexer; mailto://webindexerv1@yahoo.com; webindexerv1@yahoo.com) correctly", function() {
		const results = uaParser("WebIndexer/1-dev (Web Indexer; mailto://webindexerv1@yahoo.com; webindexerv1@yahoo.com)");
		assert.strictEqual(results.family, "WebIndexer", msg("results.family", results.family, "WebIndexer"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Webbot/0.1; http://www.webbot.ru/bot.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Webbot/0.1; http://www.webbot.ru/bot.html)");
		assert.strictEqual(results.family, "Webbot", msg("results.family", results.family, "Webbot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Webspider/1.0 (web spider;  ) correctly", function() {
		const results = uaParser("Webspider/1.0 (web spider;  )");
		assert.strictEqual(results.family, "Webspider", msg("results.family", results.family, "Webspider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WinWebBot/1.0; (Balaena Ltd, UK); http://www.balaena.com/winwebbot.html; winwebbot@balaena.com;) correctly", function() {
		const results = uaParser("WinWebBot/1.0; (Balaena Ltd, UK); http://www.balaena.com/winwebbot.html; winwebbot@balaena.com;)");
		assert.strictEqual(results.family, "WinWebBot", msg("results.family", results.family, "WinWebBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WinkBot/0.06 (Wink.com search engine web crawler; http://www.wink.com/Wink:WinkBot; winkbot@wink.com) correctly", function() {
		const results = uaParser("WinkBot/0.06 (Wink.com search engine web crawler; http://www.wink.com/Wink:WinkBot; winkbot@wink.com)");
		assert.strictEqual(results.family, "WinkBot", msg("results.family", results.family, "WinkBot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "06", msg("results.minor", results.minor || "", "06"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Yahoo-MMCrawler/3.x (mm dash crawler at trd dot overture dot com) correctly", function() {
		const results = uaParser("Yahoo-MMCrawler/3.x (mm dash crawler at trd dot overture dot com)");
		assert.strictEqual(results.family, "Yahoo-MMCrawler", msg("results.family", results.family, "Yahoo-MMCrawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Yahoo-MMCrawler/4.0; mailto:vertical-crawl-support@yahoo-inc.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Yahoo-MMCrawler/4.0; mailto:vertical-crawl-support@yahoo-inc.com)");
		assert.strictEqual(results.family, "Yahoo-MMCrawler", msg("results.family", results.family, "Yahoo-MMCrawler"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Yahoo-Newscrawler/3.9 (news-search-crawler at yahoo-inc dot com) correctly", function() {
		const results = uaParser("Yahoo-Newscrawler/3.9 (news-search-crawler at yahoo-inc dot com)");
		assert.strictEqual(results.family, "Yahoo-Newscrawler", msg("results.family", results.family, "Yahoo-Newscrawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Yahoo-VerticalCrawler-FormerWebCrawler/3.9 crawler at trd dot overture dot com; http://www.alltheweb.com/help/webmaster/crawler correctly", function() {
		const results = uaParser("Yahoo-VerticalCrawler-FormerWebCrawler/3.9 crawler at trd dot overture dot com; http://www.alltheweb.com/help/webmaster/crawler");
		assert.strictEqual(results.family, "Yahoo-VerticalCrawler-FormerWebCrawler", msg("results.family", results.family, "Yahoo-VerticalCrawler-FormerWebCrawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YoonoCrawler/1.0 (crawler@yoono.com) correctly", function() {
		const results = uaParser("YoonoCrawler/1.0 (crawler@yoono.com)");
		assert.strictEqual(results.family, "YoonoCrawler", msg("results.family", results.family, "YoonoCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; YoudaoBot/1.0; http://www.youdao.com/help/webmaster/spider/; ) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; YoudaoBot/1.0; http://www.youdao.com/help/webmaster/spider/; )");
		assert.strictEqual(results.family, "YoudaoBot", msg("results.family", results.family, "YoudaoBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; ZemlyaCrawl/1.0; +http://zemlyaozer.com/bot) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; ZemlyaCrawl/1.0; +http://zemlyaozer.com/bot)");
		assert.strictEqual(results.family, "ZemlyaCrawl", msg("results.family", results.family, "ZemlyaCrawl"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Zeusbot/0.8.1 (Ulysseek's web-crawling robot; http://www.zeusbot.com; agent@zeusbot.com) correctly", function() {
		const results = uaParser("Zeusbot/0.8.1 (Ulysseek's web-crawling robot; http://www.zeusbot.com; agent@zeusbot.com)");
		assert.strictEqual(results.family, "Zeusbot", msg("results.family", results.family, "Zeusbot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses agbot/1.0 (AgHaven.com search engine crawler; http://search.aghaven.com; bot@aghaven.com) correctly", function() {
		const results = uaParser("agbot/1.0 (AgHaven.com search engine crawler; http://search.aghaven.com; bot@aghaven.com)");
		assert.strictEqual(results.family, "agbot", msg("results.family", results.family, "agbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; archive_crawler/3.0.0-SNAPSHOT-20091205.013431  http://www.archive.org/details/archive_crawler) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; archive_crawler/3.0.0-SNAPSHOT-20091205.013431  http://www.archive.org/details/archive_crawler)");
		assert.strictEqual(results.family, "archive_crawler", msg("results.family", results.family, "archive_crawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses audioCrawlerBot/1.0 (http://www.audiocrawler.com/) correctly", function() {
		const results = uaParser("audioCrawlerBot/1.0 (http://www.audiocrawler.com/)");
		assert.strictEqual(results.family, "audioCrawlerBot", msg("results.family", results.family, "audioCrawlerBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses awesomebar_scraper/1.0 correctly", function() {
		const results = uaParser("awesomebar_scraper/1.0");
		assert.strictEqual(results.family, "awesomebar_scraper", msg("results.family", results.family, "awesomebar_scraper"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses blog_crawler/1.0 correctly", function() {
		const results = uaParser("blog_crawler/1.0");
		assert.strictEqual(results.family, "blog_crawler", msg("results.family", results.family, "blog_crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; ca-crawler/1.0) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; ca-crawler/1.0)");
		assert.strictEqual(results.family, "ca-crawler", msg("results.family", results.family, "ca-crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses crawl/0.4 langcrawl/0.1 correctly", function() {
		const results = uaParser("crawl/0.4 langcrawl/0.1");
		assert.strictEqual(results.family, "crawl", msg("results.family", results.family, "crawl"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; crawler/3.0.0 +http://www.notconfigured.com/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; crawler/3.0.0 +http://www.notconfigured.com/)");
		assert.strictEqual(results.family, "crawler", msg("results.family", results.family, "crawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses dCrawlBot/1.0.1120 correctly", function() {
		const results = uaParser("dCrawlBot/1.0.1120");
		assert.strictEqual(results.family, "dCrawlBot", msg("results.family", results.family, "dCrawlBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1120" || "", msg("results.patch", results.patch || "", "1120" || ""));
	});
	it("parses deepcrawler/3.1 (http://www.queusearch.com/whatis_deepcrawler.php),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("deepcrawler/3.1 (http://www.queusearch.com/whatis_deepcrawler.php),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "deepcrawler", msg("results.family", results.family, "deepcrawler"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses envolk[ITS]spider/1.6 ( http://www.envolk.com/envolkspider.html) correctly", function() {
		const results = uaParser("envolk[ITS]spider/1.6 ( http://www.envolk.com/envolkspider.html)");
		assert.strictEqual(results.family, "envolk[ITS]spider", msg("results.family", results.family, "envolk[ITS]spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses fastlwspider/1.0 correctly", function() {
		const results = uaParser("fastlwspider/1.0");
		assert.strictEqual(results.family, "fastlwspider", msg("results.family", results.family, "fastlwspider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses i1searchbot/2.0 (i1search web crawler; http://www.i1search.com; crawler@i1search.com) correctly", function() {
		const results = uaParser("i1searchbot/2.0 (i1search web crawler; http://www.i1search.com; crawler@i1search.com)");
		assert.strictEqual(results.family, "i1searchbot", msg("results.family", results.family, "i1searchbot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; iaskspider/1.0; MSIE 6.0) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; iaskspider/1.0; MSIE 6.0)");
		assert.strictEqual(results.family, "iaskspider", msg("results.family", results.family, "iaskspider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses iaskspider/2.0( http://iask.com/help/help_index.html) correctly", function() {
		const results = uaParser("iaskspider/2.0( http://iask.com/help/help_index.html)");
		assert.strictEqual(results.family, "iaskspider", msg("results.family", results.family, "iaskspider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Liquida.it-Crawler/1.0 ( crawler@liquida.it +http://www.liquida.it ) correctly", function() {
		const results = uaParser("Liquida.it-Crawler/1.0 ( crawler@liquida.it +http://www.liquida.it )");
		assert.strictEqual(results.family, "it-Crawler", msg("results.family", results.family, "it-Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses it2media-domain-crawler/1.0 on crawler-prod.it2media.de correctly", function() {
		const results = uaParser("it2media-domain-crawler/1.0 on crawler-prod.it2media.de");
		assert.strictEqual(results.family, "it2media-domain-crawler", msg("results.family", results.family, "it2media-domain-crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses jrCrawler/1.0b correctly", function() {
		const results = uaParser("jrCrawler/1.0b");
		assert.strictEqual(results.family, "jrCrawler", msg("results.family", results.family, "jrCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; loc-crawl/1.10.1 +http://www.google.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; loc-crawl/1.10.1 +http://www.google.com)");
		assert.strictEqual(results.family, "loc-crawl", msg("results.family", results.family, "loc-crawl"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (compatible; loc-crawler/0.11.0 +http://loc.gov),gzip(gfe),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; loc-crawler/0.11.0 +http://loc.gov),gzip(gfe),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "loc-crawler", msg("results.family", results.family, "loc-crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "11", msg("results.minor", results.minor || "", "11"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses DoCoMo/2.0 P904i( m65bot/0.1; c; http://m65.jp/bot.html ) correctly", function() {
		const results = uaParser("DoCoMo/2.0 P904i( m65bot/0.1; c; http://m65.jp/bot.html )");
		assert.strictEqual(results.family, "m65bot", msg("results.family", results.family, "m65bot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses magpie-crawler/1.1 (U; Linux amd64; en-GB;  http://www.brandwatch.net) correctly", function() {
		const results = uaParser("magpie-crawler/1.1 (U; Linux amd64; en-GB;  http://www.brandwatch.net)");
		assert.strictEqual(results.family, "magpie-crawler", msg("results.family", results.family, "magpie-crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses noxtrumbot/1.0 (crawler@noxtrum.com) correctly", function() {
		const results = uaParser("noxtrumbot/1.0 (crawler@noxtrum.com)");
		assert.strictEqual(results.family, "noxtrumbot", msg("results.family", results.family, "noxtrumbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses nyobot/1.1 (Noyb.com search engine crawler; http://www.noyb.com; bot@noyb.com),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("nyobot/1.1 (Noyb.com search engine crawler; http://www.noyb.com; bot@noyb.com),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "nyobot", msg("results.family", results.family, "nyobot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; oBot/2.3.1; +http://filterdb.iss.net/crawler/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; oBot/2.3.1; +http://filterdb.iss.net/crawler/)");
		assert.strictEqual(results.family, "oBot", msg("results.family", results.family, "oBot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses omgilibot/0.3 +http://www.omgili.com/Crawler.html correctly", function() {
		const results = uaParser("omgilibot/0.3 +http://www.omgili.com/Crawler.html");
		assert.strictEqual(results.family, "omgilibot", msg("results.family", results.family, "omgilibot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; rogerBot/1.0; UrlCrawler; http://www.seomoz.org/dp/rogerbot) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; rogerBot/1.0; UrlCrawler; http://www.seomoz.org/dp/rogerbot)");
		assert.strictEqual(results.family, "rogerBot", msg("results.family", results.family, "rogerBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses rogerbot/1.1 (http://moz.com/help/pro/what-is-rogerbot-, rogerbot-crawler+pr1-crawler-14@moz.com) correctly", function() {
		const results = uaParser("rogerbot/1.1 (http://moz.com/help/pro/what-is-rogerbot-, rogerbot-crawler+pr1-crawler-14@moz.com)");
		assert.strictEqual(results.family, "rogerbot", msg("results.family", results.family, "rogerbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses semantics webbot/1.0 correctly", function() {
		const results = uaParser("semantics webbot/1.0");
		assert.strictEqual(results.family, "semantics webbot", msg("results.family", results.family, "semantics webbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; socialbm_bot/1.0; +http://spider.socialbm.net) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; socialbm_bot/1.0; +http://spider.socialbm.net)");
		assert.strictEqual(results.family, "socialbm_bot", msg("results.family", results.family, "socialbm_bot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses tCrawler/0.1,gzip(gfe),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("tCrawler/0.1,gzip(gfe),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "tCrawler", msg("results.family", results.family, "tCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses tivraSpider/1.0 (crawler@tivra.com) correctly", function() {
		const results = uaParser("tivraSpider/1.0 (crawler@tivra.com)");
		assert.strictEqual(results.family, "tivraSpider", msg("results.family", results.family, "tivraSpider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses webscraper/1.0 correctly", function() {
		const results = uaParser("webscraper/1.0");
		assert.strictEqual(results.family, "webscraper", msg("results.family", results.family, "webscraper"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses wume_crawler/1.1 (http://wume.cse.lehigh.edu/~xiq204/crawler/) correctly", function() {
		const results = uaParser("wume_crawler/1.1 (http://wume.cse.lehigh.edu/~xiq204/crawler/)");
		assert.strictEqual(results.family, "wume_crawler", msg("results.family", results.family, "wume_crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Abrave Spider v5.3 Robot 2 (http://robot.abrave.com) correctly", function() {
		const results = uaParser("Abrave Spider v5.3 Robot 2 (http://robot.abrave.com)");
		assert.strictEqual(results.family, "3 Robot", msg("results.family", results.family, "3 Robot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Apexoo Spider 1.1 correctly", function() {
		const results = uaParser("Apexoo Spider 1.1");
		assert.strictEqual(results.family, "Apexoo Spider", msg("results.family", results.family, "Apexoo Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BT Crawler 1.0,gzip(gfe),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("BT Crawler 1.0,gzip(gfe),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "BT Crawler", msg("results.family", results.family, "BT Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Bot Blocker Crawler 1.0 (btw, IncrediBILL says \"HI!\") correctly", function() {
		const results = uaParser("Bot Blocker Crawler 1.0 (btw, IncrediBILL says \"HI!\")");
		assert.strictEqual(results.family, "Bot Blocker Crawler", msg("results.family", results.family, "Bot Blocker Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BurstFind Crawler 1.1 - www.burstfind.com correctly", function() {
		const results = uaParser("BurstFind Crawler 1.1 - www.burstfind.com");
		assert.strictEqual(results.family, "BurstFind Crawler", msg("results.family", results.family, "BurstFind Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Comodo Spider 1.1 correctly", function() {
		const results = uaParser("Comodo Spider 1.1");
		assert.strictEqual(results.family, "Comodo Spider", msg("results.family", results.family, "Comodo Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Comodo Spider 1.2 correctly", function() {
		const results = uaParser("Comodo Spider 1.2");
		assert.strictEqual(results.family, "Comodo Spider", msg("results.family", results.family, "Comodo Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Crawler 0.1,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Crawler 0.1,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Crawler", msg("results.family", results.family, "Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ExB Language Crawler 2.1.5 (+http://www.exb.de/crawler) correctly", function() {
		const results = uaParser("ExB Language Crawler 2.1.5 (+http://www.exb.de/crawler)");
		assert.strictEqual(results.family, "ExB Language Crawler", msg("results.family", results.family, "ExB Language Crawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (compatible; FAST Crawler 6.3) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; FAST Crawler 6.3)");
		assert.strictEqual(results.family, "FAST Crawler", msg("results.family", results.family, "FAST Crawler"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FAST EnterpriseCrawler 6 correctly", function() {
		const results = uaParser("FAST EnterpriseCrawler 6");
		assert.strictEqual(results.family, "FAST EnterpriseCrawler", msg("results.family", results.family, "FAST EnterpriseCrawler"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses schibstedsokbot (compatible; Mozilla/5.0; MSIE 5.0; FAST FreshCrawler 6; Contact: webcrawl@schibstedsok.no;) correctly", function() {
		const results = uaParser("schibstedsokbot (compatible; Mozilla/5.0; MSIE 5.0; FAST FreshCrawler 6; Contact: webcrawl@schibstedsok.no;)");
		assert.strictEqual(results.family, "FAST FreshCrawler", msg("results.family", results.family, "FAST FreshCrawler"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; FatBot 2.0; http://www.thefind.com/crawler) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; FatBot 2.0; http://www.thefind.com/crawler)");
		assert.strictEqual(results.family, "FatBot", msg("results.family", results.family, "FatBot"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Feedjit Favicon Crawler 1.0 correctly", function() {
		const results = uaParser("Feedjit Favicon Crawler 1.0");
		assert.strictEqual(results.family, "Feedjit Favicon Crawler", msg("results.family", results.family, "Feedjit Favicon Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HubSpot Crawler 1.0 http://www.hubspot.com/ correctly", function() {
		const results = uaParser("HubSpot Crawler 1.0 http://www.hubspot.com/");
		assert.strictEqual(results.family, "HubSpot Crawler", msg("results.family", results.family, "HubSpot Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Jaxified Crawler 1.0a (+http://www.jaxified.com/),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Jaxified Crawler 1.0a (+http://www.jaxified.com/),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Jaxified Crawler", msg("results.family", results.family, "Jaxified Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LinksCrawler 0.1beta correctly", function() {
		const results = uaParser("LinksCrawler 0.1beta");
		assert.strictEqual(results.family, "LinksCrawler", msg("results.family", results.family, "LinksCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Liquida Spider 1.0 +http://liquida.com/ correctly", function() {
		const results = uaParser("Liquida Spider 1.0 +http://liquida.com/");
		assert.strictEqual(results.family, "Liquida Spider", msg("results.family", results.family, "Liquida Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses My Spider 1.0,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("My Spider 1.0,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "My Spider", msg("results.family", results.family, "My Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NWSpider 0.9 correctly", function() {
		const results = uaParser("NWSpider 0.9");
		assert.strictEqual(results.family, "NWSpider", msg("results.family", results.family, "NWSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Netchart Adv Crawler 1.0 correctly", function() {
		const results = uaParser("Netchart Adv Crawler 1.0");
		assert.strictEqual(results.family, "Netchart Adv Crawler", msg("results.family", results.family, "Netchart Adv Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (NimbleCrawler 0.1) obeys NimbleCrawler directive contact jpump@looksmart.net correctly", function() {
		const results = uaParser("Mozilla/4.0 (NimbleCrawler 0.1) obeys NimbleCrawler directive contact jpump@looksmart.net");
		assert.strictEqual(results.family, "NimbleCrawler", msg("results.family", results.family, "NimbleCrawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows;) NimbleCrawler 2.0.1 obeys UserAgent NimbleCrawler For problems contact: crawler@healthline.com correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows;) NimbleCrawler 2.0.1 obeys UserAgent NimbleCrawler For problems contact: crawler@healthline.com");
		assert.strictEqual(results.family, "NimbleCrawler", msg("results.family", results.family, "NimbleCrawler"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses OMGCrawler 1.0 correctly", function() {
		const results = uaParser("OMGCrawler 1.0");
		assert.strictEqual(results.family, "OMGCrawler", msg("results.family", results.family, "OMGCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SeekOn Spider 1.9(+http://www.seekon.com/spider.html) correctly", function() {
		const results = uaParser("SeekOn Spider 1.9(+http://www.seekon.com/spider.html)");
		assert.strictEqual(results.family, "SeekOn Spider", msg("results.family", results.family, "SeekOn Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Toms Spider 0.3 correctly", function() {
		const results = uaParser("Toms Spider 0.3");
		assert.strictEqual(results.family, "Toms Spider", msg("results.family", results.family, "Toms Spider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Tutorial Crawler 1.4 correctly", function() {
		const results = uaParser("Tutorial Crawler 1.4");
		assert.strictEqual(results.family, "Tutorial Crawler", msg("results.family", results.family, "Tutorial Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RB2B-bot v0.1 (Using Fast Enterprise Crawler 6 search@reedbusiness.com) correctly", function() {
		const results = uaParser("RB2B-bot v0.1 (Using Fast Enterprise Crawler 6 search@reedbusiness.com)");
		assert.strictEqual(results.family, "Using Fast Enterprise Crawler", msg("results.family", results.family, "Using Fast Enterprise Crawler"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WocBot/Mozilla/5.0 (Wocodi Web Crawler 1.0; http://www.wocodi.com/crawler; crawler@wocodi.com) correctly", function() {
		const results = uaParser("WocBot/Mozilla/5.0 (Wocodi Web Crawler 1.0; http://www.wocodi.com/crawler; crawler@wocodi.com)");
		assert.strictEqual(results.family, "Wocodi Web Crawler", msg("results.family", results.family, "Wocodi Web Crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; WordSurfer Spider 2.2;)) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; WordSurfer Spider 2.2;))");
		assert.strictEqual(results.family, "WordSurfer Spider", msg("results.family", results.family, "WordSurfer Spider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses XSpider 0.01;http://uncool.oicp.net/spider.html correctly", function() {
		const results = uaParser("XSpider 0.01;http://uncool.oicp.net/spider.html");
		assert.strictEqual(results.family, "XSpider", msg("results.family", results.family, "XSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 7.0) XSpider 7 correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 7.0) XSpider 7");
		assert.strictEqual(results.family, "XSpider", msg("results.family", results.family, "XSpider"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Xaldon WebSpider 2.7.b6 correctly", function() {
		const results = uaParser("Xaldon WebSpider 2.7.b6");
		assert.strictEqual(results.family, "Xaldon WebSpider", msg("results.family", results.family, "Xaldon WebSpider"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses echocrawl 2.0 correctly", function() {
		const results = uaParser("echocrawl 2.0");
		assert.strictEqual(results.family, "echocrawl", msg("results.family", results.family, "echocrawl"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses enicura crawler 1.0,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("enicura crawler 1.0,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "enicura crawler", msg("results.family", results.family, "enicura crawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; lworld spider 1.0; Windows NT 5.1) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; lworld spider 1.0; Windows NT 5.1)");
		assert.strictEqual(results.family, "lworld spider", msg("results.family", results.family, "lworld spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FAST-mSEARCH Crawler 0.1 (bergum@fast.no) correctly", function() {
		const results = uaParser("FAST-mSEARCH Crawler 0.1 (bergum@fast.no)");
		assert.strictEqual(results.family, "FAST-mSEARCH Crawler", msg("results.family", results.family, "FAST-mSEARCH Crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 123metaspider-Bot (Version: 1.04, powered by www.123metaspider.com) correctly", function() {
		const results = uaParser("123metaspider-Bot (Version: 1.04, powered by www.123metaspider.com)");
		assert.strictEqual(results.family, "123metaspider-Bot", msg("results.family", results.family, "123metaspider-Bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses 360Spider correctly", function() {
		const results = uaParser("360Spider");
		assert.strictEqual(results.family, "360Spider", msg("results.family", results.family, "360Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Win98; en-US; rv:1.8webcrawler) http://skateboarddirectory.com correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Win98; en-US; rv:1.8webcrawler) http://skateboarddirectory.com");
		assert.strictEqual(results.family, "8webcrawler", msg("results.family", results.family, "8webcrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AESOP_com_SpiderMan correctly", function() {
		const results = uaParser("AESOP_com_SpiderMan");
		assert.strictEqual(results.family, "AESOP_com_SpiderMan", msg("results.family", results.family, "AESOP_com_SpiderMan"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AISearchBot (Email: aisearchbot@gmail.com; If your web site doesn't want to be crawled, please send us a email.) correctly", function() {
		const results = uaParser("AISearchBot (Email: aisearchbot@gmail.com; If your web site doesn't want to be crawled, please send us a email.)");
		assert.strictEqual(results.family, "AISearchBot", msg("results.family", results.family, "AISearchBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; DAWINCI ANTIPLAG SPIDER) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; DAWINCI ANTIPLAG SPIDER)");
		assert.strictEqual(results.family, "ANTIPLAG SPIDER", msg("results.family", results.family, "ANTIPLAG SPIDER"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AcquiaCrawler,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("AcquiaCrawler,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "AcquiaCrawler", msg("results.family", results.family, "AcquiaCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Adaxas Spider (http://www.adaxas.net/) correctly", function() {
		const results = uaParser("Adaxas Spider (http://www.adaxas.net/)");
		assert.strictEqual(results.family, "Adaxas Spider", msg("results.family", results.family, "Adaxas Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0) AddSugarSpiderBot www.idealobserver.com correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.0) AddSugarSpiderBot www.idealobserver.com");
		assert.strictEqual(results.family, "AddSugarSpiderBot", msg("results.family", results.family, "AddSugarSpiderBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AdnormCrawler www.adnorm.com/crawler correctly", function() {
		const results = uaParser("AdnormCrawler www.adnorm.com/crawler");
		assert.strictEqual(results.family, "AdnormCrawler", msg("results.family", results.family, "AdnormCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses www.website-analyzer.net Website Analyzer Crawler correctly", function() {
		const results = uaParser("www.website-analyzer.net Website Analyzer Crawler");
		assert.strictEqual(results.family, "Analyzer Crawler", msg("results.family", results.family, "Analyzer Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Apexoo Spider (http://www.apexoo.com/spider/) correctly", function() {
		const results = uaParser("Apexoo Spider (http://www.apexoo.com/spider/)");
		assert.strictEqual(results.family, "Apexoo Spider", msg("results.family", results.family, "Apexoo Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses AppCodes crawler - looking for iOS app mentions. More info: support@appcodes.com. Robots.txt id: AppCodesCrawler correctly", function() {
		const results = uaParser("AppCodes crawler - looking for iOS app mentions. More info: support@appcodes.com. Robots.txt id: AppCodesCrawler");
		assert.strictEqual(results.family, "AppCodes crawler", msg("results.family", results.family, "AppCodes crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ArchitextSpider correctly", function() {
		const results = uaParser("ArchitextSpider");
		assert.strictEqual(results.family, "ArchitextSpider", msg("results.family", results.family, "ArchitextSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Arikus_Spider correctly", function() {
		const results = uaParser("Arikus_Spider");
		assert.strictEqual(results.family, "Arikus_Spider", msg("results.family", results.family, "Arikus_Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0 compatible; Asterias Crawler v4;  http://www.singingfish.com/help/spider.html; webmaster@singingfish.com); SpiderThread  Revision: 3.0 correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0 compatible; Asterias Crawler v4;  http://www.singingfish.com/help/spider.html; webmaster@singingfish.com); SpiderThread  Revision: 3.0");
		assert.strictEqual(results.family, "Asterias Crawler", msg("results.family", results.family, "Asterias Crawler"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible: AstraSpider V.2.1 : astrafind.com) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible: AstraSpider V.2.1 : astrafind.com)");
		assert.strictEqual(results.family, "AstraSpider", msg("results.family", results.family, "AstraSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Autonomy Spider correctly", function() {
		const results = uaParser("Autonomy Spider");
		assert.strictEqual(results.family, "Autonomy Spider", msg("results.family", results.family, "Autonomy Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses axadine/  (Axadine Crawler; http://www.axada.de/;  ) correctly", function() {
		const results = uaParser("axadine/  (Axadine Crawler; http://www.axada.de/;  )");
		assert.strictEqual(results.family, "Axadine Crawler", msg("results.family", results.family, "Axadine Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; BIXOCRAWLER; +http://wiki.github.com/bixo/bixo/bixocrawler; bixo-dev@yahoogroups.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; BIXOCRAWLER; +http://wiki.github.com/bixo/bixo/bixocrawler; bixo-dev@yahoogroups.com)");
		assert.strictEqual(results.family, "BIXOCRAWLER", msg("results.family", results.family, "BIXOCRAWLER"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; BOTW Spider;  http://botw.org) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; BOTW Spider;  http://botw.org)");
		assert.strictEqual(results.family, "BOTW Spider", msg("results.family", results.family, "BOTW Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BacklinkCrawler (http://www.backlinktest.com/crawler.html) correctly", function() {
		const results = uaParser("BacklinkCrawler (http://www.backlinktest.com/crawler.html)");
		assert.strictEqual(results.family, "BacklinkCrawler", msg("results.family", results.family, "BacklinkCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BaiduImagespider ( http://help.baidu.jp/system/05.html) correctly", function() {
		const results = uaParser("BaiduImagespider ( http://help.baidu.jp/system/05.html)");
		assert.strictEqual(results.family, "BaiduImagespider", msg("results.family", results.family, "BaiduImagespider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BarraHomeCrawler (albertof@barrahome.org) correctly", function() {
		const results = uaParser("BarraHomeCrawler (albertof@barrahome.org)");
		assert.strictEqual(results.family, "BarraHomeCrawler", msg("results.family", results.family, "BarraHomeCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ZoomInfo::Beehive Crawler correctly", function() {
		const results = uaParser("ZoomInfo::Beehive Crawler");
		assert.strictEqual(results.family, "Beehive Crawler", msg("results.family", results.family, "Beehive Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BeijingCrawler correctly", function() {
		const results = uaParser("BeijingCrawler");
		assert.strictEqual(results.family, "BeijingCrawler", msg("results.family", results.family, "BeijingCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BejiBot Crawler (BNL Services; http://www.bejjan.net/crawler/) correctly", function() {
		const results = uaParser("BejiBot Crawler (BNL Services; http://www.bejjan.net/crawler/)");
		assert.strictEqual(results.family, "BejiBot", msg("results.family", results.family, "BejiBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BravoBrian SpiderEngine MarcoPolo correctly", function() {
		const results = uaParser("BravoBrian SpiderEngine MarcoPolo");
		assert.strictEqual(results.family, "BravoBrian SpiderEngine", msg("results.family", results.family, "BravoBrian SpiderEngine"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BrightCrawler (http://www.brightcloud.com/brightcrawler.asp) correctly", function() {
		const results = uaParser("BrightCrawler (http://www.brightcloud.com/brightcrawler.asp)");
		assert.strictEqual(results.family, "BrightCrawler", msg("results.family", results.family, "BrightCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BuildCMS crawler (http://www.buildcms.com/crawler) correctly", function() {
		const results = uaParser("BuildCMS crawler (http://www.buildcms.com/crawler)");
		assert.strictEqual(results.family, "BuildCMS crawler", msg("results.family", results.family, "BuildCMS crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ByWebSite-Search-Spider,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("ByWebSite-Search-Spider,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "ByWebSite-Search-Spider", msg("results.family", results.family, "ByWebSite-Search-Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CFG_SPIDER_USER_AGENT correctly", function() {
		const results = uaParser("CFG_SPIDER_USER_AGENT");
		assert.strictEqual(results.family, "CFG_SPIDER_USER_AGENT", msg("results.family", results.family, "CFG_SPIDER_USER_AGENT"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (CMS Crawler: http://www.cmscrawler.com) correctly", function() {
		const results = uaParser("Mozilla/4.0 (CMS Crawler: http://www.cmscrawler.com)");
		assert.strictEqual(results.family, "CMS Crawler", msg("results.family", results.family, "CMS Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CMS crawler ( http://buytaert.net/crawler/) correctly", function() {
		const results = uaParser("CMS crawler ( http://buytaert.net/crawler/)");
		assert.strictEqual(results.family, "CMS crawler", msg("results.family", results.family, "CMS crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CRAZYWEBCRAWLER 0.9.0, http://www.crazywebcrawler.com correctly", function() {
		const results = uaParser("CRAZYWEBCRAWLER 0.9.0, http://www.crazywebcrawler.com");
		assert.strictEqual(results.family, "CRAZYWEBCRAWLER", msg("results.family", results.family, "CRAZYWEBCRAWLER"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses CSimpleSpider Robot correctly", function() {
		const results = uaParser("CSimpleSpider Robot");
		assert.strictEqual(results.family, "CSimpleSpider", msg("results.family", results.family, "CSimpleSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Cityreview Robot (+http://www.cityreview.org/crawler/) correctly", function() {
		const results = uaParser("Cityreview Robot (+http://www.cityreview.org/crawler/)");
		assert.strictEqual(results.family, "Cityreview Robot", msg("results.family", results.family, "Cityreview Robot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Exalead Cloudview Crawler,gzip(gfe),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Exalead Cloudview Crawler,gzip(gfe),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Cloudview Crawler", msg("results.family", results.family, "Cloudview Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Comodo-Certificates-Spider correctly", function() {
		const results = uaParser("Comodo-Certificates-Spider");
		assert.strictEqual(results.family, "Comodo-Certificates-Spider", msg("results.family", results.family, "Comodo-Certificates-Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Computer_and_Automation_Research_Institute_Crawler (crawler@info.ilab.sztaki.hu) correctly", function() {
		const results = uaParser("Computer_and_Automation_Research_Institute_Crawler (crawler@info.ilab.sztaki.hu)");
		assert.strictEqual(results.family, "Computer_and_Automation_Research_Institute_Crawler", msg("results.family", results.family, "Computer_and_Automation_Research_Institute_Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Http Connector Spider, contact Alcatel-Lucent IDOL Search correctly", function() {
		const results = uaParser("Http Connector Spider, contact Alcatel-Lucent IDOL Search");
		assert.strictEqual(results.family, "Connector Spider", msg("results.family", results.family, "Connector Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Content Crawler correctly", function() {
		const results = uaParser("Content Crawler");
		assert.strictEqual(results.family, "Content Crawler", msg("results.family", results.family, "Content Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CowCrawler/CowCrawler-dev (+http://beta.cow.com),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("CowCrawler/CowCrawler-dev (+http://beta.cow.com),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "CowCrawler", msg("results.family", results.family, "CowCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; CrawlDaddy v0.3.0 abot v1.1.1.0 http://code.google.com/p/abot) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MSIE 9.0; Windows NT 6.1; CrawlDaddy v0.3.0 abot v1.1.1.0 http://code.google.com/p/abot)");
		assert.strictEqual(results.family, "CrawlDaddy", msg("results.family", results.family, "CrawlDaddy"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses CrawlFire - you can disable this Robot: http://pastebin.de/25277 correctly", function() {
		const results = uaParser("CrawlFire - you can disable this Robot: http://pastebin.de/25277");
		assert.strictEqual(results.family, "CrawlFire", msg("results.family", results.family, "CrawlFire"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CrawlWave/1.2 (crawlwave[at]circular.gr http://www.spiderwave.aueb.gr/ correctly", function() {
		const results = uaParser("CrawlWave/1.2 (crawlwave[at]circular.gr http://www.spiderwave.aueb.gr/");
		assert.strictEqual(results.family, "CrawlWave", msg("results.family", results.family, "CrawlWave"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CrawlerBoy Pinpoint.com correctly", function() {
		const results = uaParser("CrawlerBoy Pinpoint.com");
		assert.strictEqual(results.family, "CrawlerBoy", msg("results.family", results.family, "CrawlerBoy"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.0; Windows 98; DigExt; Crayon Crawler) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.0; Windows 98; DigExt; Crayon Crawler)");
		assert.strictEqual(results.family, "Crayon Crawler", msg("results.family", results.family, "Crayon Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DRKSpider - Website link validator - http://www.drk.com.ar/spider/,gzip(gfe),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("DRKSpider - Website link validator - http://www.drk.com.ar/spider/,gzip(gfe),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "DRKSpider", msg("results.family", results.family, "DRKSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DefaultCrawlTest/0.6 (Ram Crawl Test; devarajaswami at yahoo dot com) correctly", function() {
		const results = uaParser("DefaultCrawlTest/0.6 (Ram Crawl Test; devarajaswami at yahoo dot com)");
		assert.strictEqual(results.family, "DefaultCrawlTest", msg("results.family", results.family, "DefaultCrawlTest"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FFC Trap Door Spider correctly", function() {
		const results = uaParser("FFC Trap Door Spider");
		assert.strictEqual(results.family, "Door Spider", msg("results.family", results.family, "Door Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DoubleVerify Crawler correctly", function() {
		const results = uaParser("DoubleVerify Crawler");
		assert.strictEqual(results.family, "DoubleVerify Crawler", msg("results.family", results.family, "DoubleVerify Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses EasouSpider correctly", function() {
		const results = uaParser("EasouSpider");
		assert.strictEqual(results.family, "EasouSpider", msg("results.family", results.family, "EasouSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses EcoGrader Crawler: Beta correctly", function() {
		const results = uaParser("EcoGrader Crawler: Beta");
		assert.strictEqual(results.family, "EcoGrader Crawler", msg("results.family", results.family, "EcoGrader Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Adknowledge Engage Crawler correctly", function() {
		const results = uaParser("Adknowledge Engage Crawler");
		assert.strictEqual(results.family, "Engage Crawler", msg("results.family", results.family, "Engage Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ByWebSite Search Engine Spider correctly", function() {
		const results = uaParser("ByWebSite Search Engine Spider");
		assert.strictEqual(results.family, "Engine Spider", msg("results.family", results.family, "Engine Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses EverbeeCrawler correctly", function() {
		const results = uaParser("EverbeeCrawler");
		assert.strictEqual(results.family, "EverbeeCrawler", msg("results.family", results.family, "EverbeeCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Ex-Crawler/v0.1.5ALPHA SVN rev58; powered by ex-crawler; +http://www.ex-crawler.de/) Java/1.6.0_20,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Ex-Crawler/v0.1.5ALPHA SVN rev58; powered by ex-crawler; +http://www.ex-crawler.de/) Java/1.6.0_20,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Ex-Crawler", msg("results.family", results.family, "Ex-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ExactSeek_Spider correctly", function() {
		const results = uaParser("ExactSeek_Spider");
		assert.strictEqual(results.family, "ExactSeek_Spider", msg("results.family", results.family, "ExactSeek_Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; FastCrawler3, support-fastcrawler3@fast.no) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; FastCrawler3, support-fastcrawler3@fast.no)");
		assert.strictEqual(results.family, "FastCrawler3", msg("results.family", results.family, "FastCrawler3"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Finderbots finder bot; +http://wiki.github.com/bixo/bixo/bixocrawler; bixo-dev@yahoogroups.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Finderbots finder bot; +http://wiki.github.com/bixo/bixo/bixocrawler; bixo-dev@yahoogroups.com)");
		assert.strictEqual(results.family, "Finderbots", msg("results.family", results.family, "Finderbots"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Findexa Crawler (http://www.findexa.no/gulesider/article26548.ece) correctly", function() {
		const results = uaParser("Findexa Crawler (http://www.findexa.no/gulesider/article26548.ece)");
		assert.strictEqual(results.family, "Findexa Crawler", msg("results.family", results.family, "Findexa Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/3.0 (compatible; Fluffy the spider; http://www.searchhippo.com/; info@searchhippo.com) correctly", function() {
		const results = uaParser("Mozilla/3.0 (compatible; Fluffy the spider; http://www.searchhippo.com/; info@searchhippo.com)");
		assert.strictEqual(results.family, "Fluffy the spider", msg("results.family", results.family, "Fluffy the spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.01; Windows NT 5.0) (Atsameip FreeCrawl!) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.01; Windows NT 5.0) (Atsameip FreeCrawl!)");
		assert.strictEqual(results.family, "FreeCrawl", msg("results.family", results.family, "FreeCrawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Fujiko Spider (fujiko@inelegant.org) correctly", function() {
		const results = uaParser("Fujiko Spider (fujiko@inelegant.org)");
		assert.strictEqual(results.family, "Fujiko Spider", msg("results.family", results.family, "Fujiko Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; FunkyCrawler; ) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; FunkyCrawler; )");
		assert.strictEqual(results.family, "FunkyCrawler", msg("results.family", results.family, "FunkyCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GSiteCrawler/v1.12 rev. 260 (http://gsitecrawler.com/) correctly", function() {
		const results = uaParser("GSiteCrawler/v1.12 rev. 260 (http://gsitecrawler.com/)");
		assert.strictEqual(results.family, "GSiteCrawler", msg("results.family", results.family, "GSiteCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GeekTools Crawler - http://domains.geek-tools.org correctly", function() {
		const results = uaParser("GeekTools Crawler - http://domains.geek-tools.org");
		assert.strictEqual(results.family, "GeekTools Crawler", msg("results.family", results.family, "GeekTools Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla 4.0 - GetMeLinked Spider www.GetMeLinked.com Web Directory correctly", function() {
		const results = uaParser("Mozilla 4.0 - GetMeLinked Spider www.GetMeLinked.com Web Directory");
		assert.strictEqual(results.family, "GetMeLinked Spider", msg("results.family", results.family, "GetMeLinked Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Gnam Gnam Spider correctly", function() {
		const results = uaParser("Gnam Gnam Spider");
		assert.strictEqual(results.family, "Gnam Spider", msg("results.family", results.family, "Gnam Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; GoGuidesBot; http://www.goguides.org/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; GoGuidesBot; http://www.goguides.org/spider.html)");
		assert.strictEqual(results.family, "GoGuidesBot", msg("results.family", results.family, "GoGuidesBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GoScraper correctly", function() {
		const results = uaParser("GoScraper");
		assert.strictEqual(results.family, "GoScraper", msg("results.family", results.family, "GoScraper"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses GrowerIdeas Crawler/GrowerIdeas-nutch-1.6 (Crawls URLs for indexing content for our new search startup which aims to provide simple and smart search across curated content. For more info please contact helloworld@growerideas.com. If you think correctly", function() {
		const results = uaParser("GrowerIdeas Crawler/GrowerIdeas-nutch-1.6 (Crawls URLs for indexing content for our new search startup which aims to provide simple and smart search across curated content. For more info please contact helloworld@growerideas.com. If you think");
		assert.strictEqual(results.family, "GrowerIdeas Crawler", msg("results.family", results.family, "GrowerIdeas Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Huaweisymantecspider (compatible; MSIE 7.0; Windows NT 5.1; Trident/4.0; .NET CLR 2.0.50727),gzip(gfe) correctly", function() {
		const results = uaParser("Huaweisymantecspider (compatible; MSIE 7.0; Windows NT 5.1; Trident/4.0; .NET CLR 2.0.50727),gzip(gfe)");
		assert.strictEqual(results.family, "Huaweisymantecspider", msg("results.family", results.family, "Huaweisymantecspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BlogPulse (ISSpider-3.0) correctly", function() {
		const results = uaParser("BlogPulse (ISSpider-3.0)");
		assert.strictEqual(results.family, "ISSpider", msg("results.family", results.family, "ISSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IWE Spider v.01 - www.pavka.com.au correctly", function() {
		const results = uaParser("IWE Spider v.01 - www.pavka.com.au");
		assert.strictEqual(results.family, "IWE Spider", msg("results.family", results.family, "IWE Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IXE Crawler correctly", function() {
		const results = uaParser("IXE Crawler");
		assert.strictEqual(results.family, "IXE Crawler", msg("results.family", results.family, "IXE Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Imperia-LinkSpider,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Imperia-LinkSpider,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Imperia-LinkSpider", msg("results.family", results.family, "Imperia-LinkSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses netEstate Impressumscrawler (+http://www.netestate.de/De/Loesungen/Impressumscrawler) correctly", function() {
		const results = uaParser("netEstate Impressumscrawler (+http://www.netestate.de/De/Loesungen/Impressumscrawler)");
		assert.strictEqual(results.family, "Impressumscrawler", msg("results.family", results.family, "Impressumscrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Inar_spider2 (tspyyp@tom.com) correctly", function() {
		const results = uaParser("Inar_spider2 (tspyyp@tom.com)");
		assert.strictEqual(results.family, "Inar_spider2", msg("results.family", results.family, "Inar_spider2"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IOI/2.0 (ISC Open Index crawler; http://index.isc.org/; bot@index.isc.org) correctly", function() {
		const results = uaParser("IOI/2.0 (ISC Open Index crawler; http://index.isc.org/; bot@index.isc.org)");
		assert.strictEqual(results.family, "Index crawler", msg("results.family", results.family, "Index crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.5; Windows 98; Infocrawler; Alexa Toolbar) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.5; Windows 98; Infocrawler; Alexa Toolbar)");
		assert.strictEqual(results.family, "Infocrawler", msg("results.family", results.family, "Infocrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; Inspyder-Crawler; http://www.inspyder.com) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; Inspyder-Crawler; http://www.inspyder.com)");
		assert.strictEqual(results.family, "Inspyder-Crawler", msg("results.family", results.family, "Inspyder-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Willow Internet Crawler by Twotrees V2.1 correctly", function() {
		const results = uaParser("Willow Internet Crawler by Twotrees V2.1");
		assert.strictEqual(results.family, "Internet Crawler", msg("results.family", results.family, "Internet Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (SKIZZLE! Distributed Internet Spider v1.0 - www.SKIZZLE.com) correctly", function() {
		const results = uaParser("Mozilla/4.0 (SKIZZLE! Distributed Internet Spider v1.0 - www.SKIZZLE.com)");
		assert.strictEqual(results.family, "Internet Spider", msg("results.family", results.family, "Internet Spider"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses IssueCrawler correctly", function() {
		const results = uaParser("IssueCrawler");
		assert.strictEqual(results.family, "IssueCrawler", msg("results.family", results.family, "IssueCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses JUST-CRAWLER( http://www.justsystems.com/jp/tech/crawler/) correctly", function() {
		const results = uaParser("JUST-CRAWLER( http://www.justsystems.com/jp/tech/crawler/)");
		assert.strictEqual(results.family, "JUST-CRAWLER", msg("results.family", results.family, "JUST-CRAWLER"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Jayde Crawler. http://www.jayde.com correctly", function() {
		const results = uaParser("Jayde Crawler. http://www.jayde.com");
		assert.strictEqual(results.family, "Jayde Crawler", msg("results.family", results.family, "Jayde Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses JikeSpider; +http://shoulu.jike.com/spider.html correctly", function() {
		const results = uaParser("JikeSpider; +http://shoulu.jike.com/spider.html");
		assert.strictEqual(results.family, "JikeSpider", msg("results.family", results.family, "JikeSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible;WI Job Roboter Spider Version 3;+http://www.webintegration.at) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible;WI Job Roboter Spider Version 3;+http://www.webintegration.at)");
		assert.strictEqual(results.family, "Job Roboter", msg("results.family", results.family, "Job Roboter"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses JobSpider_BA/1.1 correctly", function() {
		const results = uaParser("JobSpider_BA/1.1");
		assert.strictEqual(results.family, "JobSpider_BA", msg("results.family", results.family, "JobSpider_BA"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses KiwiStatus (NZS.com)/0.2 (NZS.com KiwiStatus Spider,  Local Search New Zealand; http://www.nzs.com; bot-at-nzs dot com) correctly", function() {
		const results = uaParser("KiwiStatus (NZS.com)/0.2 (NZS.com KiwiStatus Spider,  Local Search New Zealand; http://www.nzs.com; bot-at-nzs dot com)");
		assert.strictEqual(results.family, "KiwiStatus Spider", msg("results.family", results.family, "KiwiStatus Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Kotoss Crawler http://hitokoto.kotoss.com,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("Kotoss Crawler http://hitokoto.kotoss.com,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Kotoss Crawler", msg("results.family", results.family, "Kotoss Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Kyluka crawl; crawl@kyluka.com; http://www.kyluka.com/static/crawl.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Kyluka crawl; crawl@kyluka.com; http://www.kyluka.com/static/crawl.html)");
		assert.strictEqual(results.family, "Kyluka crawl", msg("results.family", results.family, "Kyluka crawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LNSpiderguy correctly", function() {
		const results = uaParser("LNSpiderguy");
		assert.strictEqual(results.family, "LNSpiderguy", msg("results.family", results.family, "LNSpiderguy"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LarbinWebCrawler (internet@bredband.net) correctly", function() {
		const results = uaParser("LarbinWebCrawler (internet@bredband.net)");
		assert.strictEqual(results.family, "LarbinWebCrawler", msg("results.family", results.family, "LarbinWebCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LargeSmall Crawler correctly", function() {
		const results = uaParser("LargeSmall Crawler");
		assert.strictEqual(results.family, "LargeSmall Crawler", msg("results.family", results.family, "LargeSmall Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Lijit Crawler (+http://www.lijit.com/robot/crawler) correctly", function() {
		const results = uaParser("Lijit Crawler (+http://www.lijit.com/robot/crawler)");
		assert.strictEqual(results.family, "Lijit Crawler", msg("results.family", results.family, "Lijit Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DocWeb Link Crawler (http://doc.php.net) correctly", function() {
		const results = uaParser("DocWeb Link Crawler (http://doc.php.net)");
		assert.strictEqual(results.family, "Link Crawler", msg("results.family", results.family, "Link Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Lipperhey Spider; http://www.lipperhey.com/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Lipperhey Spider; http://www.lipperhey.com/)");
		assert.strictEqual(results.family, "Lipperhey Spider", msg("results.family", results.family, "Lipperhey Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LookUpCrawler - lookupcanada.ca [ZSEBOT] correctly", function() {
		const results = uaParser("LookUpCrawler - lookupcanada.ca [ZSEBOT]");
		assert.strictEqual(results.family, "LookUpCrawler", msg("results.family", results.family, "LookUpCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Lycos_Spider_(modspider) correctly", function() {
		const results = uaParser("Lycos_Spider_(modspider)");
		assert.strictEqual(results.family, "Lycos_Spider_", msg("results.family", results.family, "Lycos_Spider_"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.5; Windows NT 5.0) METASpider correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.5; Windows NT 5.0) METASpider");
		assert.strictEqual(results.family, "METASpider", msg("results.family", results.family, "METASpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MQbot http://metaquerier.cs.uiuc.edu/crawler correctly", function() {
		const results = uaParser("MQbot http://metaquerier.cs.uiuc.edu/crawler");
		assert.strictEqual(results.family, "MQbot", msg("results.family", results.family, "MQbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MSIndianWebcrawl correctly", function() {
		const results = uaParser("MSIndianWebcrawl");
		assert.strictEqual(results.family, "MSIndianWebcrawl", msg("results.family", results.family, "MSIndianWebcrawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MSR-ISRCCrawler correctly", function() {
		const results = uaParser("MSR-ISRCCrawler");
		assert.strictEqual(results.family, "MSR-ISRCCrawler", msg("results.family", results.family, "MSR-ISRCCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MedSpider v0.0.1 correctly", function() {
		const results = uaParser("MedSpider v0.0.1");
		assert.strictEqual(results.family, "MedSpider", msg("results.family", results.family, "MedSpider"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Social Media Crawler using your Home URL on Twitter,Facebook,Myspace,Linkedin by ProfileCapture - contact profilecapture@gmail.com to report any problems with my crawling. http://profilecapture.com correctly", function() {
		const results = uaParser("Social Media Crawler using your Home URL on Twitter,Facebook,Myspace,Linkedin by ProfileCapture - contact profilecapture@gmail.com to report any problems with my crawling. http://profilecapture.com");
		assert.strictEqual(results.family, "Media Crawler", msg("results.family", results.family, "Media Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible); MSIE 5.0; Medialab Spider correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible); MSIE 5.0; Medialab Spider");
		assert.strictEqual(results.family, "Medialab Spider", msg("results.family", results.family, "Medialab Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DomainsDB.net MetaCrawler v.0.9.7b (http://domainsdb.net/) correctly", function() {
		const results = uaParser("DomainsDB.net MetaCrawler v.0.9.7b (http://domainsdb.net/)");
		assert.strictEqual(results.family, "MetaCrawler", msg("results.family", results.family, "MetaCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses FAST MetaWeb Crawler (helpdesk at fastsearch dot com) correctly", function() {
		const results = uaParser("FAST MetaWeb Crawler (helpdesk at fastsearch dot com)");
		assert.strictEqual(results.family, "MetaWeb Crawler", msg("results.family", results.family, "MetaWeb Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CC Metadata Scaper http://wiki.creativecommons.org/Metadata_Scraper correctly", function() {
		const results = uaParser("CC Metadata Scaper http://wiki.creativecommons.org/Metadata_Scraper");
		assert.strictEqual(results.family, "Metadata_Scraper", msg("results.family", results.family, "Metadata_Scraper"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MicrosoftPrototypeCrawler (How's my crawling? mailto:newbiecrawler@hotmail.com) correctly", function() {
		const results = uaParser("MicrosoftPrototypeCrawler (How's my crawling? mailto:newbiecrawler@hotmail.com)");
		assert.strictEqual(results.family, "MicrosoftPrototypeCrawler", msg("results.family", results.family, "MicrosoftPrototypeCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; MixrankBot; crawler@mixrank.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; MixrankBot; crawler@mixrank.com)");
		assert.strictEqual(results.family, "MixrankBot", msg("results.family", results.family, "MixrankBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sogou Mobile Spider1.0 (http://wap.sogou.com) correctly", function() {
		const results = uaParser("Sogou Mobile Spider1.0 (http://wap.sogou.com)");
		assert.strictEqual(results.family, "Mobile Spider1", msg("results.family", results.family, "Mobile Spider1"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla-Firefox-Spider(Wenanry) correctly", function() {
		const results = uaParser("Mozilla-Firefox-Spider(Wenanry)");
		assert.strictEqual(results.family, "Mozilla-Firefox-Spider", msg("results.family", results.family, "Mozilla-Firefox-Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MultiCrawler, http://sw.deri.org/2006/04/multicrawler/robots.html correctly", function() {
		const results = uaParser("MultiCrawler, http://sw.deri.org/2006/04/multicrawler/robots.html");
		assert.strictEqual(results.family, "MultiCrawler", msg("results.family", results.family, "MultiCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows NT 6.1; en; rv:1.9.1.3; MySpaceScraper) Gecko/20090824 Firefox/3.5.3 (.NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows NT 6.1; en; rv:1.9.1.3; MySpaceScraper) Gecko/20090824 Firefox/3.5.3 (.NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "MySpaceScraper", msg("results.family", results.family, "MySpaceScraper"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Norbert the Spider(Burf.com) correctly", function() {
		const results = uaParser("Norbert the Spider(Burf.com)");
		assert.strictEqual(results.family, "Norbert the Spider", msg("results.family", results.family, "Norbert the Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NuSearch Spider www.nusearch.com correctly", function() {
		const results = uaParser("NuSearch Spider www.nusearch.com");
		assert.strictEqual(results.family, "NuSearch Spider", msg("results.family", results.family, "NuSearch Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Nusearch Spider (compatible; MSIE 6.0) correctly", function() {
		const results = uaParser("Nusearch Spider (compatible; MSIE 6.0)");
		assert.strictEqual(results.family, "Nusearch Spider", msg("results.family", results.family, "Nusearch Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses OpenWebSpider (link collector; http://links.port30.se/cia.html) v (http://www.openwebspider.org/) correctly", function() {
		const results = uaParser("OpenWebSpider (link collector; http://links.port30.se/cia.html) v (http://www.openwebspider.org/)");
		assert.strictEqual(results.family, "OpenWebSpider", msg("results.family", results.family, "OpenWebSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; OpenX Spider; http://www.openx.org) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; OpenX Spider; http://www.openx.org)");
		assert.strictEqual(results.family, "OpenX Spider", msg("results.family", results.family, "OpenX Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; OpenindexSpider; +http://www.openindex.io/en/webmasters/spider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; OpenindexSpider; +http://www.openindex.io/en/webmasters/spider.html)");
		assert.strictEqual(results.family, "OpenindexSpider", msg("results.family", results.family, "OpenindexSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Orgbybot/OrgbyBot v1.2 (Spidering the net for Orgby; http://www.orgby.com/  ; Orgby.com Search Engine) correctly", function() {
		const results = uaParser("Orgbybot/OrgbyBot v1.2 (Spidering the net for Orgby; http://www.orgby.com/  ; Orgby.com Search Engine)");
		assert.strictEqual(results.family, "Orgbybot", msg("results.family", results.family, "Orgbybot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PDFBot (crawler@pdfind.com) correctly", function() {
		const results = uaParser("PDFBot (crawler@pdfind.com)");
		assert.strictEqual(results.family, "PDFBot", msg("results.family", results.family, "PDFBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.0; Trident/4.0; SLCC1; .NET CLR 2.0.50727; .NET CLR 1.1.4322; InfoPath.2; .NET CLR 3.5.21022; .NET CLR 3.5.30729; MS-RTC LM 8; OfficeLiveConnector.1.4; OfficeLivePatch.1.3; .NET CLR 3.0.30729) Jakarta Commons-HttpClient/3.0-rc3 PHPCrawl GStreamer souphttpsrc libsoup/2.27.4 PycURL/7.19.0 XML-RPC for PHP 2.2.1 GoogleFriendConnect/1.0 HTMLParser/1.6 gPodder/0.15.2 ( http://gpodder.org/) anw webtool LoadControl/1.3 WinHttp urlgrabber/3.1.0 correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 8.0; Windows NT 6.0; Trident/4.0; SLCC1; .NET CLR 2.0.50727; .NET CLR 1.1.4322; InfoPath.2; .NET CLR 3.5.21022; .NET CLR 3.5.30729; MS-RTC LM 8; OfficeLiveConnector.1.4; OfficeLivePatch.1.3; .NET CLR 3.0.30729) Jakarta Commons-HttpClient/3.0-rc3 PHPCrawl GStreamer souphttpsrc libsoup/2.27.4 PycURL/7.19.0 XML-RPC for PHP 2.2.1 GoogleFriendConnect/1.0 HTMLParser/1.6 gPodder/0.15.2 ( http://gpodder.org/) anw webtool LoadControl/1.3 WinHttp urlgrabber/3.1.0");
		assert.strictEqual(results.family, "PHPCrawl", msg("results.family", results.family, "PHPCrawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Fast PartnerSite Crawler correctly", function() {
		const results = uaParser("Fast PartnerSite Crawler");
		assert.strictEqual(results.family, "PartnerSite Crawler", msg("results.family", results.family, "PartnerSite Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Patwebbot (http://www.herz-power.de/technik.html) correctly", function() {
		const results = uaParser("Patwebbot (http://www.herz-power.de/technik.html)");
		assert.strictEqual(results.family, "Patwebbot", msg("results.family", results.family, "Patwebbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PeerFactor Crawler correctly", function() {
		const results = uaParser("PeerFactor Crawler");
		assert.strictEqual(results.family, "PeerFactor Crawler", msg("results.family", results.family, "PeerFactor Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PhoenixWebBot Beta correctly", function() {
		const results = uaParser("PhoenixWebBot Beta");
		assert.strictEqual(results.family, "PhoenixWebBot", msg("results.family", results.family, "PhoenixWebBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses pipeLiner/0.10 (PipeLine Spider; http://www.pipeline-search.com/webmaster.html) correctly", function() {
		const results = uaParser("pipeLiner/0.10 (PipeLine Spider; http://www.pipeline-search.com/webmaster.html)");
		assert.strictEqual(results.family, "PipeLine Spider", msg("results.family", results.family, "PipeLine Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ProjectWF-java-test-crawler correctly", function() {
		const results = uaParser("ProjectWF-java-test-crawler");
		assert.strictEqual(results.family, "ProjectWF-java-test-crawler", msg("results.family", results.family, "ProjectWF-java-test-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Punk Spider/PunkSPIDER-v0.1 correctly", function() {
		const results = uaParser("Punk Spider/PunkSPIDER-v0.1");
		assert.strictEqual(results.family, "Punk Spider", msg("results.family", results.family, "Punk Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses QuerySeekerSpider ( http://queryseeker.com/bot.html ) correctly", function() {
		const results = uaParser("QuerySeekerSpider ( http://queryseeker.com/bot.html )");
		assert.strictEqual(results.family, "QuerySeekerSpider", msg("results.family", results.family, "QuerySeekerSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses QuickFinder Crawler correctly", function() {
		const results = uaParser("QuickFinder Crawler");
		assert.strictEqual(results.family, "QuickFinder Crawler", msg("results.family", results.family, "QuickFinder Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses OpenLink Virtuoso RDF crawler correctly", function() {
		const results = uaParser("OpenLink Virtuoso RDF crawler");
		assert.strictEqual(results.family, "RDF crawler", msg("results.family", results.family, "RDF crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Jaxified Public RSS Crawler ( http://www.jaxified.com/) correctly", function() {
		const results = uaParser("Jaxified Public RSS Crawler ( http://www.jaxified.com/)");
		assert.strictEqual(results.family, "RSS Crawler", msg("results.family", results.family, "RSS Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RSS-SPIDER (RSS Feed Seeker http://www.MyNewFavoriteThing.com/fsb.php) correctly", function() {
		const results = uaParser("RSS-SPIDER (RSS Feed Seeker http://www.MyNewFavoriteThing.com/fsb.php)");
		assert.strictEqual(results.family, "RSS-SPIDER", msg("results.family", results.family, "RSS-SPIDER"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RavenCrawler correctly", function() {
		const results = uaParser("RavenCrawler");
		assert.strictEqual(results.family, "RavenCrawler", msg("results.family", results.family, "RavenCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ReadPath_Spider correctly", function() {
		const results = uaParser("ReadPath_Spider");
		assert.strictEqual(results.family, "ReadPath_Spider", msg("results.family", results.family, "ReadPath_Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses LinkStar Research Crawler (http://linkstar.com/),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("LinkStar Research Crawler (http://linkstar.com/),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "Research Crawler", msg("results.family", results.family, "Research Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Research spider - ak1835@albany.edu correctly", function() {
		const results = uaParser("Research spider - ak1835@albany.edu");
		assert.strictEqual(results.family, "Research spider", msg("results.family", results.family, "Research spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses RoboCrawl (www.canadiancontent.net) correctly", function() {
		const results = uaParser("RoboCrawl (www.canadiancontent.net)");
		assert.strictEqual(results.family, "RoboCrawl", msg("results.family", results.family, "RoboCrawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.2; WOW64) Runet-Research-Crawler (itrack.ru/research/cmsrate; rating@itrack.ru) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.2; WOW64) Runet-Research-Crawler (itrack.ru/research/cmsrate; rating@itrack.ru)");
		assert.strictEqual(results.family, "Runet-Research-Crawler", msg("results.family", results.family, "Runet-Research-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Teragram/SAS Crawler correctly", function() {
		const results = uaParser("Teragram/SAS Crawler");
		assert.strictEqual(results.family, "SAS Crawler", msg("results.family", results.family, "SAS Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SCrawlTest/CR1 (CWD; http://sproose.com; crawl@sproose.com) correctly", function() {
		const results = uaParser("SCrawlTest/CR1 (CWD; http://sproose.com; crawl@sproose.com)");
		assert.strictEqual(results.family, "SCrawlTest", msg("results.family", results.family, "SCrawlTest"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; SISTRIX Crawler; http://crawler.sistrix.net/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; SISTRIX Crawler; http://crawler.sistrix.net/)");
		assert.strictEqual(results.family, "SISTRIX Crawler", msg("results.family", results.family, "SISTRIX Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux i686) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/30.0.1599.101 Safari/537.36; SSL-Crawler: http://crawler.dcsec.uni-hannover.de correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux i686) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/30.0.1599.101 Safari/537.36; SSL-Crawler: http://crawler.dcsec.uni-hannover.de");
		assert.strictEqual(results.family, "SSL-Crawler", msg("results.family", results.family, "SSL-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SWAT Crawler. AGH University project. In case of problem contact: opal@tempus.metal.agh.edu.pl. Thanks. correctly", function() {
		const results = uaParser("SWAT Crawler. AGH University project. In case of problem contact: opal@tempus.metal.agh.edu.pl. Thanks.");
		assert.strictEqual(results.family, "SWAT Crawler", msg("results.family", results.family, "SWAT Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SandCrawler - Compatibility Testing correctly", function() {
		const results = uaParser("SandCrawler - Compatibility Testing");
		assert.strictEqual(results.family, "SandCrawler", msg("results.family", results.family, "SandCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sangfor Spider correctly", function() {
		const results = uaParser("Sangfor Spider");
		assert.strictEqual(results.family, "Sangfor Spider", msg("results.family", results.family, "Sangfor Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/3.0 (compatible; ScollSpider; http://www.webwobot.com) correctly", function() {
		const results = uaParser("Mozilla/3.0 (compatible; ScollSpider; http://www.webwobot.com)");
		assert.strictEqual(results.family, "ScollSpider", msg("results.family", results.family, "ScollSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ScreenerBot Crawler Beta 2.0 (+http://www.ScreenerBot.com) correctly", function() {
		const results = uaParser("ScreenerBot Crawler Beta 2.0 (+http://www.ScreenerBot.com)");
		assert.strictEqual(results.family, "ScreenerBot", msg("results.family", results.family, "ScreenerBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SearchSpider.com/1.1 correctly", function() {
		const results = uaParser("SearchSpider.com/1.1");
		assert.strictEqual(results.family, "SearchSpider", msg("results.family", results.family, "SearchSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ownCloud Server Crawler correctly", function() {
		const results = uaParser("ownCloud Server Crawler");
		assert.strictEqual(results.family, "Server Crawler", msg("results.family", results.family, "Server Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Shim-Crawler correctly", function() {
		const results = uaParser("Shim-Crawler");
		assert.strictEqual(results.family, "Shim-Crawler", msg("results.family", results.family, "Shim-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ShowyouBot (http://showyou.com/crawler) correctly", function() {
		const results = uaParser("ShowyouBot (http://showyou.com/crawler)");
		assert.strictEqual(results.family, "crawler", msg("results.family", results.family, "crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.0) SiteCheck-sitecrawl by Siteimprove.com correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 5.0) SiteCheck-sitecrawl by Siteimprove.com");
		assert.strictEqual(results.family, "SiteCheck-sitecrawl", msg("results.family", results.family, "SiteCheck-sitecrawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SiteCrawler,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("SiteCrawler,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "SiteCrawler", msg("results.family", results.family, "SiteCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.5 (compatible; MSIE 5.5; Windows NT 5.1) Sitespider+ b432.1 correctly", function() {
		const results = uaParser("Mozilla/5.5 (compatible; MSIE 5.5; Windows NT 5.1) Sitespider+ b432.1");
		assert.strictEqual(results.family, "Sitespider", msg("results.family", results.family, "Sitespider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SocialSpider-Finder/0.2 correctly", function() {
		const results = uaParser("SocialSpider-Finder/0.2");
		assert.strictEqual(results.family, "SocialSpider", msg("results.family", results.family, "SocialSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sosoblogspider+(+http://help.soso.com/soso-blog-spider.htm) correctly", function() {
		const results = uaParser("Sosoblogspider+(+http://help.soso.com/soso-blog-spider.htm)");
		assert.strictEqual(results.family, "Sosoblogspider", msg("results.family", results.family, "Sosoblogspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sosoimagespider ( http://help.soso.com/soso-image-spider.htm) correctly", function() {
		const results = uaParser("Sosoimagespider ( http://help.soso.com/soso-image-spider.htm)");
		assert.strictEqual(results.family, "Sosoimagespider", msg("results.family", results.family, "Sosoimagespider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Sosospider+(+help.soso.com/webspider.htm) correctly", function() {
		const results = uaParser("Sosospider+(+help.soso.com/webspider.htm)");
		assert.strictEqual(results.family, "Sosospider", msg("results.family", results.family, "Sosospider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; SpeedySpider; www.entireweb.com) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; SpeedySpider; www.entireweb.com)");
		assert.strictEqual(results.family, "SpeedySpider", msg("results.family", results.family, "SpeedySpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SpiderKU/0.9 correctly", function() {
		const results = uaParser("SpiderKU/0.9");
		assert.strictEqual(results.family, "SpiderKU", msg("results.family", results.family, "SpiderKU"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; SpiderLing (a SPIDER for LINGustic research); +http://nlp.fi.muni.cz/projects/biwec/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; SpiderLing (a SPIDER for LINGustic research); +http://nlp.fi.muni.cz/projects/biwec/)");
		assert.strictEqual(results.family, "SpiderLing", msg("results.family", results.family, "SpiderLing"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SpiderMan 3.0.1-2-11-111 (CP/M;8-bit) correctly", function() {
		const results = uaParser("SpiderMan 3.0.1-2-11-111 (CP/M;8-bit)");
		assert.strictEqual(results.family, "SpiderMan", msg("results.family", results.family, "SpiderMan"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/4.0 (compatible; SpiderView 1.0;unix) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; SpiderView 1.0;unix)");
		assert.strictEqual(results.family, "SpiderView", msg("results.family", results.family, "SpiderView"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Spider_Monkey/7.06 (SpiderMonkey.ca info at http://SpiderMonkey.ca /sm.shtml) correctly", function() {
		const results = uaParser("Spider_Monkey/7.06 (SpiderMonkey.ca info at http://SpiderMonkey.ca /sm.shtml)");
		assert.strictEqual(results.family, "Spider_Monkey", msg("results.family", results.family, "Spider_Monkey"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "06", msg("results.minor", results.minor || "", "06"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Spiderlytics/1.0; +spider@spiderlytics.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Spiderlytics/1.0; +spider@spiderlytics.com)");
		assert.strictEqual(results.family, "Spiderlytics", msg("results.family", results.family, "Spiderlytics"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PKU Student Spider correctly", function() {
		const results = uaParser("PKU Student Spider");
		assert.strictEqual(results.family, "Student Spider", msg("results.family", results.family, "Student Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Symfony Spider (http://symfony.com/spider) correctly", function() {
		const results = uaParser("Symfony Spider (http://symfony.com/spider)");
		assert.strictEqual(results.family, "Symfony Spider", msg("results.family", results.family, "Symfony Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Synthesio Crawler release MonaLisa (contact at synthesio dot fr) correctly", function() {
		const results = uaParser("Synthesio Crawler release MonaLisa (contact at synthesio dot fr)");
		assert.strictEqual(results.family, "Synthesio Crawler", msg("results.family", results.family, "Synthesio Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TAMU_CS_IRL_CRAWLER/1.0 correctly", function() {
		const results = uaParser("TAMU_CS_IRL_CRAWLER/1.0");
		assert.strictEqual(results.family, "TAMU_CS_IRL_CRAWLER", msg("results.family", results.family, "TAMU_CS_IRL_CRAWLER"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TTop-Crawler,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("TTop-Crawler,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "TTop-Crawler", msg("results.family", results.family, "TTop-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TayaCrawler (Beta; v0.1; tayadev@nexdegree.com) correctly", function() {
		const results = uaParser("TayaCrawler (Beta; v0.1; tayadev@nexdegree.com)");
		assert.strictEqual(results.family, "TayaCrawler", msg("results.family", results.family, "TayaCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TelemetrySpider2/0.1 linux correctly", function() {
		const results = uaParser("TelemetrySpider2/0.1 linux");
		assert.strictEqual(results.family, "TelemetrySpider2", msg("results.family", results.family, "TelemetrySpider2"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Theme Spider ( http://www.themespider.com/spider.html) correctly", function() {
		const results = uaParser("Theme Spider ( http://www.themespider.com/spider.html)");
		assert.strictEqual(results.family, "Theme Spider", msg("results.family", results.family, "Theme Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Thesis_php_crawler correctly", function() {
		const results = uaParser("Thesis_php_crawler");
		assert.strictEqual(results.family, "Thesis_php_crawler", msg("results.family", results.family, "Thesis_php_crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Trends Crawler, Real time trends bot (info@trendscrawler.com) correctly", function() {
		const results = uaParser("Trends Crawler, Real time trends bot (info@trendscrawler.com)");
		assert.strictEqual(results.family, "Trends Crawler", msg("results.family", results.family, "Trends Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses TwitSpider correctly", function() {
		const results = uaParser("TwitSpider");
		assert.strictEqual(results.family, "TwitSpider", msg("results.family", results.family, "TwitSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Twitmunin Crawler http://www.twitmunin.com correctly", function() {
		const results = uaParser("Twitmunin Crawler http://www.twitmunin.com");
		assert.strictEqual(results.family, "Twitmunin Crawler", msg("results.family", results.family, "Twitmunin Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; TwitterCrawler) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; TwitterCrawler)");
		assert.strictEqual(results.family, "TwitterCrawler", msg("results.family", results.family, "TwitterCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses UCMore Crawler App correctly", function() {
		const results = uaParser("UCMore Crawler App");
		assert.strictEqual(results.family, "UCMore Crawler", msg("results.family", results.family, "UCMore Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses UOLCrawler (soscrawler@uol.com.br) correctly", function() {
		const results = uaParser("UOLCrawler (soscrawler@uol.com.br)");
		assert.strictEqual(results.family, "UOLCrawler", msg("results.family", results.family, "UOLCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Links4US-Crawler, ( http://links4us.com/) correctly", function() {
		const results = uaParser("Links4US-Crawler, ( http://links4us.com/)");
		assert.strictEqual(results.family, "US-Crawler", msg("results.family", results.family, "US-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses USyd-NLP-Spider (http://www.it.usyd.edu.au/~vinci/bot.html) correctly", function() {
		const results = uaParser("USyd-NLP-Spider (http://www.it.usyd.edu.au/~vinci/bot.html)");
		assert.strictEqual(results.family, "USyd-NLP-Spider", msg("results.family", results.family, "USyd-NLP-Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; UnisterBot; crawler@unister.de) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; UnisterBot; crawler@unister.de)");
		assert.strictEqual(results.family, "UnisterBot", msg("results.family", results.family, "UnisterBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0, VM-Crawler/cs version info  ofni noisrev sc correctly", function() {
		const results = uaParser("Mozilla/4.0, VM-Crawler/cs version info  ofni noisrev sc");
		assert.strictEqual(results.family, "VM-Crawler", msg("results.family", results.family, "VM-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WEBB Crawler - see: http://badcheese.com/robots.html correctly", function() {
		const results = uaParser("WEBB Crawler - see: http://badcheese.com/robots.html");
		assert.strictEqual(results.family, "WEBB Crawler", msg("results.family", results.family, "WEBB Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WSDL Crawler correctly", function() {
		const results = uaParser("WSDL Crawler");
		assert.strictEqual(results.family, "WSDL Crawler", msg("results.family", results.family, "WSDL Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses EmeraldShield.com WebBot correctly", function() {
		const results = uaParser("EmeraldShield.com WebBot");
		assert.strictEqual(results.family, "com WebBot", msg("results.family", results.family, "com WebBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WebCompanyCrawler correctly", function() {
		const results = uaParser("WebCompanyCrawler");
		assert.strictEqual(results.family, "WebCompanyCrawler", msg("results.family", results.family, "WebCompanyCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WebCrawler v1.3 correctly", function() {
		const results = uaParser("WebCrawler v1.3");
		assert.strictEqual(results.family, "WebCrawler", msg("results.family", results.family, "WebCrawler"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses WebVulnCrawl.blogspot.com/1.0 libwww-perl/5.803 correctly", function() {
		const results = uaParser("WebVulnCrawl.blogspot.com/1.0 libwww-perl/5.803");
		assert.strictEqual(results.family, "WebVulnCrawl", msg("results.family", results.family, "WebVulnCrawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses CyberPatrol SiteCat Webbot (http://www.cyberpatrol.com/cyberpatrolcrawler.asp) correctly", function() {
		const results = uaParser("CyberPatrol SiteCat Webbot (http://www.cyberpatrol.com/cyberpatrolcrawler.asp)");
		assert.strictEqual(results.family, "SiteCat Webbot", msg("results.family", results.family, "SiteCat Webbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses EricssonR320/R1A (Fast Wireless Crawler) correctly", function() {
		const results = uaParser("EricssonR320/R1A (Fast Wireless Crawler)");
		assert.strictEqual(results.family, "Wireless Crawler", msg("results.family", results.family, "Wireless Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses X-Crawler correctly", function() {
		const results = uaParser("X-Crawler");
		assert.strictEqual(results.family, "X-Crawler", msg("results.family", results.family, "X-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses XYZ Spider correctly", function() {
		const results = uaParser("XYZ Spider");
		assert.strictEqual(results.family, "XYZ Spider", msg("results.family", results.family, "XYZ Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Y!J-BRJ/YATS crawler (http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html) correctly", function() {
		const results = uaParser("Y!J-BRJ/YATS crawler (http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html)");
		assert.strictEqual(results.family, "YATS crawler", msg("results.family", results.family, "YATS crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Y!J-BRO/YFSJ crawler (compatible; Mozilla 4.0; MSIE 5.5; http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html; YahooFeedSeekerJp/2.0) correctly", function() {
		const results = uaParser("Y!J-BRO/YFSJ crawler (compatible; Mozilla 4.0; MSIE 5.5; http://help.yahoo.co.jp/help/jp/search/indexing/indexing-15.html; YahooFeedSeekerJp/2.0)");
		assert.strictEqual(results.family, "YFSJ crawler", msg("results.family", results.family, "YFSJ crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YRL_ODP_CRAWLER correctly", function() {
		const results = uaParser("YRL_ODP_CRAWLER");
		assert.strictEqual(results.family, "YRL_ODP_CRAWLER", msg("results.family", results.family, "YRL_ODP_CRAWLER"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Android 3.0; YRSpider; +http://www.yunrang.com/yrspider.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Android 3.0; YRSpider; +http://www.yunrang.com/yrspider.html)");
		assert.strictEqual(results.family, "YRSpider", msg("results.family", results.family, "YRSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YebolBot (Email: yebolbot@gmail.com; If the web crawling affects your web service, or you don't like to be crawled by us, please email us. We'll stop crawling immediately.) correctly", function() {
		const results = uaParser("YebolBot (Email: yebolbot@gmail.com; If the web crawling affects your web service, or you don't like to be crawled by us, please email us. We'll stop crawling immediately.)");
		assert.strictEqual(results.family, "YebolBot", msg("results.family", results.family, "YebolBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; Yet-Another-Spider; ) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; Yet-Another-Spider; )");
		assert.strictEqual(results.family, "Yet-Another-Spider", msg("results.family", results.family, "Yet-Another-Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YisouSpider correctly", function() {
		const results = uaParser("YisouSpider");
		assert.strictEqual(results.family, "YisouSpider", msg("results.family", results.family, "YisouSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses NL-Crawler correctly", function() {
		const results = uaParser("NL-Crawler");
		assert.strictEqual(results.family, "NL-Crawler", msg("results.family", results.family, "NL-Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; YoudaoBot-rts/1.0; http://www.youdao.com/help/webmaster/spider/; ) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; YoudaoBot-rts/1.0; http://www.youdao.com/help/webmaster/spider/; )");
		assert.strictEqual(results.family, "YoudaoBot", msg("results.family", results.family, "YoudaoBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ZIBB Crawler (email address / WWW address) correctly", function() {
		const results = uaParser("ZIBB Crawler (email address / WWW address)");
		assert.strictEqual(results.family, "ZIBB Crawler", msg("results.family", results.family, "ZIBB Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ZillaCrawler correctly", function() {
		const results = uaParser("ZillaCrawler");
		assert.strictEqual(results.family, "ZillaCrawler", msg("results.family", results.family, "ZillaCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ZoomSpider - wrensoft.com correctly", function() {
		const results = uaParser("ZoomSpider - wrensoft.com");
		assert.strictEqual(results.family, "ZoomSpider", msg("results.family", results.family, "ZoomSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Zspider (+http://www.zhanzhangsou.com/index.htm) correctly", function() {
		const results = uaParser("Zspider (+http://www.zhanzhangsou.com/index.htm)");
		assert.strictEqual(results.family, "Zspider", msg("results.family", results.family, "Zspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; acedo crawler extension) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; acedo crawler extension)");
		assert.strictEqual(results.family, "acedo crawler", msg("results.family", results.family, "acedo crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses acquia-crawler (detected bad behaviour? please tell us at it@acquia.com),gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("acquia-crawler (detected bad behaviour? please tell us at it@acquia.com),gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "acquia-crawler", msg("results.family", results.family, "acquia-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.21 (KHTML, like Gecko) adspider Safari/537.21 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/537.21 (KHTML, like Gecko) adspider Safari/537.21");
		assert.strictEqual(results.family, "adspider", msg("results.family", results.family, "adspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses amphetameme crawler (crawler@amphetameme.com) correctly", function() {
		const results = uaParser("amphetameme crawler (crawler@amphetameme.com)");
		assert.strictEqual(results.family, "amphetameme crawler", msg("results.family", results.family, "amphetameme crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; ayna-crawler  http://www.ayna.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; ayna-crawler  http://www.ayna.com)");
		assert.strictEqual(results.family, "ayna-crawler", msg("results.family", results.family, "ayna-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Bender; http://sites.google.com/site/bendercrawler) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Bender; http://sites.google.com/site/bendercrawler)");
		assert.strictEqual(results.family, "bendercrawler", msg("results.family", results.family, "bendercrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses snap.com beta crawler v0 correctly", function() {
		const results = uaParser("snap.com beta crawler v0");
		assert.strictEqual(results.family, "beta crawler", msg("results.family", results.family, "beta crawler"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses blackspider correctly", function() {
		const results = uaParser("blackspider");
		assert.strictEqual(results.family, "blackspider", msg("results.family", results.family, "blackspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; bmdspider; windows 5.1) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; bmdspider; windows 5.1)");
		assert.strictEqual(results.family, "bmdspider", msg("results.family", results.family, "bmdspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses boitho crawler correctly", function() {
		const results = uaParser("boitho crawler");
		assert.strictEqual(results.family, "boitho crawler", msg("results.family", results.family, "boitho crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses cis455crawler correctly", function() {
		const results = uaParser("cis455crawler");
		assert.strictEqual(results.family, "cis455crawler", msg("results.family", results.family, "cis455crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses http://www.Syntryx.com/ ANT Chassis 9.27; Mozilla/4.0 compatible crawler correctly", function() {
		const results = uaParser("http://www.Syntryx.com/ ANT Chassis 9.27; Mozilla/4.0 compatible crawler");
		assert.strictEqual(results.family, "compatible crawler", msg("results.family", results.family, "compatible crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses crawler4j (http://code.google.com/p/crawler4j/) correctly", function() {
		const results = uaParser("crawler4j (http://code.google.com/p/crawler4j/)");
		assert.strictEqual(results.family, "crawler4j", msg("results.family", results.family, "crawler4j"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses dtSearchSpider correctly", function() {
		const results = uaParser("dtSearchSpider");
		assert.strictEqual(results.family, "dtSearchSpider", msg("results.family", results.family, "dtSearchSpider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Enterprise_Search/1.0 110 (http://www.innerprise.net/es-spider.asp) correctly", function() {
		const results = uaParser("Enterprise_Search/1.0 110 (http://www.innerprise.net/es-spider.asp)");
		assert.strictEqual(results.family, "es-spider", msg("results.family", results.family, "es-spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses eseek-crawler.0.5 (crawler@exactseek.com) correctly", function() {
		const results = uaParser("eseek-crawler.0.5 (crawler@exactseek.com)");
		assert.strictEqual(results.family, "eseek-crawler", msg("results.family", results.family, "eseek-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses exooba/exooba crawler (exooba; exooba) correctly", function() {
		const results = uaParser("exooba/exooba crawler (exooba; exooba)");
		assert.strictEqual(results.family, "exooba crawler", msg("results.family", results.family, "exooba crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses flatlandbot/allspark (Flatland Industries Web Spider; http://www.flatlandindustries.com/flatlandbot; jason@flatlandindustries.com) correctly", function() {
		const results = uaParser("flatlandbot/allspark (Flatland Industries Web Spider; http://www.flatlandindustries.com/flatlandbot; jason@flatlandindustries.com)");
		assert.strictEqual(results.family, "flatlandbot", msg("results.family", results.family, "flatlandbot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses haupia-crawler correctly", function() {
		const results = uaParser("haupia-crawler");
		assert.strictEqual(results.family, "haupia-crawler", msg("results.family", results.family, "haupia-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses hitcrawler_0.1 (ringken@gmail.com) correctly", function() {
		const results = uaParser("hitcrawler_0.1 (ringken@gmail.com)");
		assert.strictEqual(results.family, "hitcrawler_0", msg("results.family", results.family, "hitcrawler_0"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses iaskspider2 (iask@staff.sina.com.cn) correctly", function() {
		const results = uaParser("iaskspider2 (iask@staff.sina.com.cn)");
		assert.strictEqual(results.family, "iaskspider2", msg("results.family", results.family, "iaskspider2"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses visaduhoc.info Crawler correctly", function() {
		const results = uaParser("visaduhoc.info Crawler");
		assert.strictEqual(results.family, "info Crawler", msg("results.family", results.family, "info Crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ip-web-crawler.com correctly", function() {
		const results = uaParser("ip-web-crawler.com");
		assert.strictEqual(results.family, "ip-web-crawler", msg("results.family", results.family, "ip-web-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses jikespider \"Mozilla/5.0 correctly", function() {
		const results = uaParser("jikespider \"Mozilla/5.0");
		assert.strictEqual(results.family, "jikespider", msg("results.family", results.family, "jikespider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 5.5; Windows NT 5.0) compatible; kototoi-crawl@yl.is.s.u-tokyo.ac.jp correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 5.5; Windows NT 5.0) compatible; kototoi-crawl@yl.is.s.u-tokyo.ac.jp");
		assert.strictEqual(results.family, "kototoi-crawl", msg("results.family", results.family, "kototoi-crawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses lb-spider correctly", function() {
		const results = uaParser("lb-spider");
		assert.strictEqual(results.family, "lb-spider", msg("results.family", results.family, "lb-spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ldspider (BTC 2011 crawl, harth@kit.edu, http://code.google.com/p/ldspider/wiki/Robots) correctly", function() {
		const results = uaParser("ldspider (BTC 2011 crawl, harth@kit.edu, http://code.google.com/p/ldspider/wiki/Robots)");
		assert.strictEqual(results.family, "ldspider", msg("results.family", results.family, "ldspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; lemurwebcrawler admin@lemurproject.org; +http://boston.lti.cs.cmu.edu/crawler_12/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; lemurwebcrawler admin@lemurproject.org; +http://boston.lti.cs.cmu.edu/crawler_12/)");
		assert.strictEqual(results.family, "lemurwebcrawler", msg("results.family", results.family, "lemurwebcrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses lmspider (lmspider@scansoft.com) correctly", function() {
		const results = uaParser("lmspider (lmspider@scansoft.com)");
		assert.strictEqual(results.family, "lmspider", msg("results.family", results.family, "lmspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses lworldspider correctly", function() {
		const results = uaParser("lworldspider");
		assert.strictEqual(results.family, "lworldspider", msg("results.family", results.family, "lworldspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses trunk.ly spider contact@trunk.ly correctly", function() {
		const results = uaParser("trunk.ly spider contact@trunk.ly");
		assert.strictEqual(results.family, "ly spider", msg("results.family", results.family, "ly spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Vodafone mCrawler (bergum@fast.no) correctly", function() {
		const results = uaParser("Vodafone mCrawler (bergum@fast.no)");
		assert.strictEqual(results.family, "mCrawler", msg("results.family", results.family, "mCrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses media-percbotspider <ops@percolate.com> correctly", function() {
		const results = uaParser("media-percbotspider <ops@percolate.com>");
		assert.strictEqual(results.family, "media-percbotspider", msg("results.family", results.family, "media-percbotspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses DoCoMo/2.0 N902iS(c100;TB;W24H12)(compatible; moba-crawler; http://crawler.dena.jp/) correctly", function() {
		const results = uaParser("DoCoMo/2.0 N902iS(c100;TB;W24H12)(compatible; moba-crawler; http://crawler.dena.jp/)");
		assert.strictEqual(results.family, "moba-crawler", msg("results.family", results.family, "moba-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Nokia6680/1.0 ((4.04.07) SymbianOS/8.0 Series60/2.6 Profile/MIDP-2.0 Configuration/CLDC-1.1 (for mobile crawler) ) correctly", function() {
		const results = uaParser("Nokia6680/1.0 ((4.04.07) SymbianOS/8.0 Series60/2.6 Profile/MIDP-2.0 Configuration/CLDC-1.1 (for mobile crawler) )");
		assert.strictEqual(results.family, "mobile crawler", msg("results.family", results.family, "mobile crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses n4p_bot (crawler@n4p.com) correctly", function() {
		const results = uaParser("n4p_bot (crawler@n4p.com)");
		assert.strictEqual(results.family, "n4p_bot", msg("results.family", results.family, "n4p_bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses na-Webcrawler (helpdesk@newsaktuell.de) correctly", function() {
		const results = uaParser("na-Webcrawler (helpdesk@newsaktuell.de)");
		assert.strictEqual(results.family, "na-Webcrawler", msg("results.family", results.family, "na-Webcrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses nuSearch Spider <a href='http://www.nusearch.com'>www.nusearch.com</a> (compatible; MSIE 4.01; Windows NT) correctly", function() {
		const results = uaParser("nuSearch Spider <a href='http://www.nusearch.com'>www.nusearch.com</a> (compatible; MSIE 4.01; Windows NT)");
		assert.strictEqual(results.family, "nuSearch Spider", msg("results.family", results.family, "nuSearch Spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses parallelContextFocusCrawler1.1parallelContextFocusCrawler1.1 correctly", function() {
		const results = uaParser("parallelContextFocusCrawler1.1parallelContextFocusCrawler1.1");
		assert.strictEqual(results.family, "parallelContextFocusCrawler1", msg("results.family", results.family, "parallelContextFocusCrawler1"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses persomm-spider/v1.0 correctly", function() {
		const results = uaParser("persomm-spider/v1.0");
		assert.strictEqual(results.family, "persomm-spider", msg("results.family", results.family, "persomm-spider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; http://www.pregnancycrawler.com) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; http://www.pregnancycrawler.com)");
		assert.strictEqual(results.family, "pregnancycrawler", msg("results.family", results.family, "pregnancycrawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; ptd-crawler;  http://bixolabs.com/crawler/ptd/; crawler@bixolabs.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; ptd-crawler;  http://bixolabs.com/crawler/ptd/; crawler@bixolabs.com)");
		assert.strictEqual(results.family, "ptd-crawler", msg("results.family", results.family, "ptd-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; pub-crawler; +http://wiki.github.com/bixo/bixo/bixocrawler; bixo-dev@yahoogroups.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; pub-crawler; +http://wiki.github.com/bixo/bixo/bixocrawler; bixo-dev@yahoogroups.com)");
		assert.strictEqual(results.family, "pub-crawler", msg("results.family", results.family, "pub-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses pythonic-crawler (suzuki@tkl.iis.u-tokyo.ac.jp) correctly", function() {
		const results = uaParser("pythonic-crawler (suzuki@tkl.iis.u-tokyo.ac.jp)");
		assert.strictEqual(results.family, "pythonic-crawler", msg("results.family", results.family, "pythonic-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.2; WOW64) Russian CMS rating crawler (itrack.ru/cmsrate, avlasov@itrack.ru) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.2; WOW64) Russian CMS rating crawler (itrack.ru/cmsrate, avlasov@itrack.ru)");
		assert.strictEqual(results.family, "rating crawler", msg("results.family", results.family, "rating crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses camoca/camoca-n.1.2 (super-agent; search crawler; info at does not exist dot com) correctly", function() {
		const results = uaParser("camoca/camoca-n.1.2 (super-agent; search crawler; info at does not exist dot com)");
		assert.strictEqual(results.family, "search crawler", msg("results.family", results.family, "search crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; sgbot v0.01a, sgcrawlerbot@gmail.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; sgbot v0.01a, sgcrawlerbot@gmail.com)");
		assert.strictEqual(results.family, "sgbot", msg("results.family", results.family, "sgbot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses spiderpig/0.1,gzip(gfe),gzip(gfe) correctly", function() {
		const results = uaParser("spiderpig/0.1,gzip(gfe),gzip(gfe)");
		assert.strictEqual(results.family, "spiderpig", msg("results.family", results.family, "spiderpig"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ssearch_bot (sSearch Crawler; http://www.semantissimo.de) correctly", function() {
		const results = uaParser("ssearch_bot (sSearch Crawler; http://www.semantissimo.de)");
		assert.strictEqual(results.family, "ssearch_bot", msg("results.family", results.family, "ssearch_bot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; suggybot v0.01a, http://blog.suggy.com/was-ist-suggy/suggy-webcrawler/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; suggybot v0.01a, http://blog.suggy.com/was-ist-suggy/suggy-webcrawler/)");
		assert.strictEqual(results.family, "suggybot", msg("results.family", results.family, "suggybot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "01", msg("results.minor", results.minor || "", "01"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; uMBot-FC/1.0; mailto: crawling@ubermetrics-technologies.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; uMBot-FC/1.0; mailto: crawling@ubermetrics-technologies.com)");
		assert.strictEqual(results.family, "uMBot", msg("results.family", results.family, "uMBot"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses unchaos_crawler_2.0.2 (search.engine@unchaos.com) correctly", function() {
		const results = uaParser("unchaos_crawler_2.0.2 (search.engine@unchaos.com)");
		assert.strictEqual(results.family, "unchaos_crawler_2", msg("results.family", results.family, "unchaos_crawler_2"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses updated/0.1-alpha (updated crawler; http://www.updated.com; crawler@updated.com) correctly", function() {
		const results = uaParser("updated/0.1-alpha (updated crawler; http://www.updated.com; crawler@updated.com)");
		assert.strictEqual(results.family, "updated crawler", msg("results.family", results.family, "updated crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Yahoo-Test/4.0 mailto:vertical-crawl-support@yahoo-inc.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Yahoo-Test/4.0 mailto:vertical-crawl-support@yahoo-inc.com)");
		assert.strictEqual(results.family, "vertical-crawl", msg("results.family", results.family, "vertical-crawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses yp-crawl@attinteractive.com correctly", function() {
		const results = uaParser("yp-crawl@attinteractive.com");
		assert.strictEqual(results.family, "yp-crawl", msg("results.family", results.family, "yp-crawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses yrspider (Mozilla/5.0 (compatible; YRSpider;  http://www.yunrang.com/yrspider.html)) correctly", function() {
		const results = uaParser("yrspider (Mozilla/5.0 (compatible; YRSpider;  http://www.yunrang.com/yrspider.html))");
		assert.strictEqual(results.family, "yrspider", msg("results.family", results.family, "yrspider"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Yahoo-Test/4.0 ysm-keystone-crawl-support@yahoo-inc.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Yahoo-Test/4.0 ysm-keystone-crawl-support@yahoo-inc.com)");
		assert.strictEqual(results.family, "ysm-keystone-crawl", msg("results.family", results.family, "ysm-keystone-crawl"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; zyklop; +http://www.seoratio.de/zyklop-crawler/) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; zyklop; +http://www.seoratio.de/zyklop-crawler/)");
		assert.strictEqual(results.family, "zyklop-crawler", msg("results.family", results.family, "zyklop-crawler"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.2.1; en-gb; CUBOT ONE Build/JOP40D) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.2.1; en-gb; CUBOT ONE Build/JOP40D) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30");
		assert.strictEqual(results.family, "Android", msg("results.family", results.family, "Android"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9) AppleWebKit/537.71 (KHTML, like Gecko) Version/7.0 Safari/537.71 (Rival IQ, rivaliq.com) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9) AppleWebKit/537.71 (KHTML, like Gecko) Version/7.0 Safari/537.71 (Rival IQ, rivaliq.com)");
		assert.strictEqual(results.family, "Rival IQ", msg("results.family", results.family, "Rival IQ"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; en-gb) AppleWebKit/418.9.1 (KHTML, like Gecko) SiteCon/8.10.9 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; en-gb) AppleWebKit/418.9.1 (KHTML, like Gecko) SiteCon/8.10.9");
		assert.strictEqual(results.family, "SiteCon", msg("results.family", results.family, "SiteCon"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_4) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.9.133 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_4) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.9.133 Safari/537.36");
		assert.strictEqual(results.family, "Spotify", msg("results.family", results.family, "Spotify"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.9.133 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.9.133 Safari/537.36");
		assert.strictEqual(results.family, "Spotify", msg("results.family", results.family, "Spotify"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_7_5) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.4.90 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_7_5) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.4.90 Safari/537.36");
		assert.strictEqual(results.family, "Spotify", msg("results.family", results.family, "Spotify"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.3.101 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.3.101 Safari/537.36");
		assert.strictEqual(results.family, "Spotify", msg("results.family", results.family, "Spotify"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_5) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.8.59 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_5) AppleWebKit/537.36 (KHTML, like Gecko) Spotify/1.0.8.59 Safari/537.36");
		assert.strictEqual(results.family, "Spotify", msg("results.family", results.family, "Spotify"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Android 5.0; Tablet; rv:41.0) Gecko/41.0 Firefox/41.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Android 5.0; Tablet; rv:41.0) Gecko/41.0 Firefox/41.0");
		assert.strictEqual(results.family, "Firefox Mobile", msg("results.family", results.family, "Firefox Mobile"));
		assert.strictEqual(results.major || "", "41", msg("results.major", results.major || "", "41"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses [FBAN/FB4A;FBAV/3.4;FBBV/258875;FBDM/{density=0.75,width=240,height=320};FBLC/tr_TR;FBCR/o2 - de;FBPN/com.facebook.katana;FBDV/LG-E400;FBSV/2.3.6;] correctly", function() {
		const results = uaParser("[FBAN/FB4A;FBAV/3.4;FBBV/258875;FBDM/{density=0.75,width=240,height=320};FBLC/tr_TR;FBCR/o2 - de;FBPN/com.facebook.katana;FBDV/LG-E400;FBSV/2.3.6;]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses [FBAN/FB4A;FBAV/2.3;FBBV/149649;FBDM/{density=1.5,width=480,height=800};FBLC/es_ES;FBCR/;FBPN/com.facebook.katana;FBDV/LG-P920;FBSV/2.2.2;] correctly", function() {
		const results = uaParser("[FBAN/FB4A;FBAV/2.3;FBBV/149649;FBDM/{density=1.5,width=480,height=800};FBLC/es_ES;FBCR/;FBPN/com.facebook.katana;FBDV/LG-P920;FBSV/2.2.2;]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 2.3.4; en-us; SCH-R720 Build/GINGERBREAD) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 [FBAN/FB4A;FBAV/1.8.1;FBPN/com.facebook.katana;FBDV/SCH-R720;FBSV/2.3.4;FBDM/{density=1.0,width=320,height= correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 2.3.4; en-us; SCH-R720 Build/GINGERBREAD) AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1 [FBAN/FB4A;FBAV/1.8.1;FBPN/com.facebook.katana;FBDV/SCH-R720;FBSV/2.3.4;FBDM/{density=1.0,width=320,height=");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.2.2; de-de; 706_v92_jbla_fhd Build/JDQ39) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30 [FB_IAB/FB4A;FBAV/24.0.0.30.15;] correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.2.2; de-de; 706_v92_jbla_fhd Build/JDQ39) AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30 [FB_IAB/FB4A;FBAV/24.0.0.30.15;]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "24", msg("results.major", results.major || "", "24"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU iPhone OS 4_3_2 like Mac OS X; de_DE) AppleWebKit (KHTML, like Gecko) Mobile [FBAN/FBForIPhone;FBAV/4.0.2;FBBV/4020.0;FBDV/iPad1,1;FBMD/iPad;FBSN/iPhone OS;FBSV/4.3.2;FBSS/1; FBCR/;FBID/tablet;FBLC/de_DE;FBSF/1.0] correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU iPhone OS 4_3_2 like Mac OS X; de_DE) AppleWebKit (KHTML, like Gecko) Mobile [FBAN/FBForIPhone;FBAV/4.0.2;FBBV/4020.0;FBDV/iPad1,1;FBMD/iPad;FBSN/iPhone OS;FBSV/4.3.2;FBSS/1; FBCR/;FBID/tablet;FBLC/de_DE;FBSF/1.0]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (iPad; U; CPU iPhone OS 4_3_5 like Mac OS X; de_DE) AppleWebKit (KHTML, like Gecko) Mobile [FBAN/FBForIPhone;FBAV/4.0.3;FBBV/4030.0;FBDV/iPad2,2;FBMD/iPad;FBSN/iPhone OS;FBSV/4.3.5;FBSS/1; FBCR/Telekom.de;FBID/tablet;FBLC/de_DE;FBS correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPad; U; CPU iPhone OS 4_3_5 like Mac OS X; de_DE) AppleWebKit (KHTML, like Gecko) Mobile [FBAN/FBForIPhone;FBAV/4.0.3;FBBV/4030.0;FBDV/iPad2,2;FBMD/iPad;FBSN/iPhone OS;FBSV/4.3.5;FBSS/1; FBCR/Telekom.de;FBID/tablet;FBLC/de_DE;FBS");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (iPhone; U; CPU iPhone OS 4_1 like Mac OS X; fr_FR) AppleWebKit (KHTML, like Gecko) Mobile [FBAN/FBForIPhone;FBAV/4.0;FBBV/4000.0;FBDV/iPhone1,2;FBMD/iPhone;FBSN/iPhone OS;FBSV/4.1;FBSS/1; FBCR/Carrier;FBID/phone;FBLC/fr_FR;FBSF/1. correctly", function() {
		const results = uaParser("Mozilla/5.0 (iPhone; U; CPU iPhone OS 4_1 like Mac OS X; fr_FR) AppleWebKit (KHTML, like Gecko) Mobile [FBAN/FBForIPhone;FBAV/4.0;FBBV/4000.0;FBDV/iPhone1,2;FBMD/iPhone;FBSN/iPhone OS;FBSV/4.1;FBSS/1; FBCR/Carrier;FBID/phone;FBLC/fr_FR;FBSF/1.");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Dalvik/1.2.0 (Linux; U; Android 2.2.2; HTC Desire Build/FRG83G) [FBAN/Orca-Android;FBAV/2.6.1-release;FBLC/de_DE;FBBV/288543;FBCR/o2 - de;FBMF/HTC;FBBD/htc_wwe;FBDV/HTC Desire;FBSV/2.2.2] correctly", function() {
		const results = uaParser("Dalvik/1.2.0 (Linux; U; Android 2.2.2; HTC Desire Build/FRG83G) [FBAN/Orca-Android;FBAV/2.6.1-release;FBLC/de_DE;FBBV/288543;FBCR/o2 - de;FBMF/HTC;FBBD/htc_wwe;FBDV/HTC Desire;FBSV/2.2.2]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses [FBAN/FB4A;FBAV/3.6;FBBV/330148;FBDM/{density=0.75,width=240,height=320};FBLC/de_DE;FBCR/o2 - de;FBPN/com.facebook.katana;FBDV/GT-S5570;FBSV/2.2.1;FBCA/armeabi:unknown;] correctly", function() {
		const results = uaParser("[FBAN/FB4A;FBAV/3.6;FBBV/330148;FBDM/{density=0.75,width=240,height=320};FBLC/de_DE;FBCR/o2 - de;FBPN/com.facebook.katana;FBDV/GT-S5570;FBSV/2.2.1;FBCA/armeabi:unknown;]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses [FBAN/PAAA;FBAV/1.7;FBDM/{density=2.0,width=720,height=1280};FBLC/es_ES;FB_FW/2;FBSN/Android;FBCR/FONIC;FBDV/GT-I9300;FBSV/4.1.2;] correctly", function() {
		const results = uaParser("[FBAN/PAAA;FBAV/1.7;FBDM/{density=2.0,width=720,height=1280};FBLC/es_ES;FB_FW/2;FBSN/Android;FBCR/FONIC;FBDV/GT-I9300;FBSV/4.1.2;]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses [FBAN/PAAA;FBAV/1.9;FBDM/{density=2.0,width=720,height=1280};FBLC/de_DE;FB_FW/2;FBSN/Android;FBCR/o2 - de;FBDV/GT-I9300;FBSV/4.3;] correctly", function() {
		const results = uaParser("[FBAN/PAAA;FBAV/1.9;FBDM/{density=2.0,width=720,height=1280};FBLC/de_DE;FB_FW/2;FBSN/Android;FBCR/o2 - de;FBDV/GT-I9300;FBSV/4.3;]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses [FBAN/FB4A;FBAV/130.0.321;FBBV/149649;FBDM/{density=1.5,width=480,height=800};FBLC/es_ES;FBCR/;FBPN/com.facebook.katana;FBDV/LG-P920;FBSV/2.2.2;] correctly", function() {
		const results = uaParser("[FBAN/FB4A;FBAV/130.0.321;FBBV/149649;FBDM/{density=1.5,width=480,height=800};FBLC/es_ES;FBCR/;FBPN/com.facebook.katana;FBDV/LG-P920;FBSV/2.2.2;]");
		assert.strictEqual(results.family, "Facebook", msg("results.family", results.family, "Facebook"));
		assert.strictEqual(results.major || "", "130", msg("results.major", results.major || "", "130"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "321" || "", msg("results.patch", results.patch || "", "321" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 6.0.1; SM-A910F Build/MMB29M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36 [FB_IAB/MESSENGER;FBAV/120.0.0.14.84;] correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 6.0.1; SM-A910F Build/MMB29M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36 [FB_IAB/MESSENGER;FBAV/120.0.0.14.84;]");
		assert.strictEqual(results.family, "Facebook Messenger", msg("results.family", results.family, "Facebook Messenger"));
		assert.strictEqual(results.major || "", "120", msg("results.major", results.major || "", "120"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.4.2; A3-A11 Build/KOT49H) AppleWebKit/537.36 (KHTML like Gecko) Version/4.0 Chrome/30.0.0.0 Safari/537.36 [Pinterest/Android] correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.4.2; A3-A11 Build/KOT49H) AppleWebKit/537.36 (KHTML like Gecko) Version/4.0 Chrome/30.0.0.0 Safari/537.36 [Pinterest/Android]");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pinterest for Android Tablet/1.8.4 (SGP321; 4.3) correctly", function() {
		const results = uaParser("Pinterest for Android Tablet/1.8.4 (SGP321; 4.3)");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Pinterest for Android Tablet/4.3.1 (A7600-H; 4.4.2) correctly", function() {
		const results = uaParser("Pinterest for Android Tablet/4.3.1 (A7600-H; 4.4.2)");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Pinterest for Android/1.1.1 (endeavoru; 4.1.1) correctly", function() {
		const results = uaParser("Pinterest for Android/1.1.1 (endeavoru; 4.1.1)");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Pinterest for Android/3.6.2 (klte; 4.4.2) correctly", function() {
		const results = uaParser("Pinterest for Android/3.6.2 (klte; 4.4.2)");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Pinterest/0.1 correctly", function() {
		const results = uaParser("Pinterest/0.1");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pinterest/0.2 (+http://www.pinterest.com/) correctly", function() {
		const results = uaParser("Pinterest/0.2 (+http://www.pinterest.com/)");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pinterest for Android/1.1.12 (endeavoru; 4.1.1) correctly", function() {
		const results = uaParser("Pinterest for Android/1.1.12 (endeavoru; 4.1.1)");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "12" || "", msg("results.patch", results.patch || "", "12" || ""));
	});
	it("parses Pinterest/3.2 CFNetwork/672.0.8 Darwin/14.0.0 correctly", function() {
		const results = uaParser("Pinterest/3.2 CFNetwork/672.0.8 Darwin/14.0.0");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pinterest/3.3.3 CFNetwork/609.1.4 Darwin/13.0.0 correctly", function() {
		const results = uaParser("Pinterest/3.3.3 CFNetwork/609.1.4 Darwin/13.0.0");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Pinterest/3356 CFNetwork/711.0.6 Darwin/14.0.0 correctly", function() {
		const results = uaParser("Pinterest/3356 CFNetwork/711.0.6 Darwin/14.0.0");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "3356", msg("results.major", results.major || "", "3356"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Pinterest/4.1.3 CFNetwork/672.1.14 Darwin/14.0.0 correctly", function() {
		const results = uaParser("Pinterest/4.1.3 CFNetwork/672.1.14 Darwin/14.0.0");
		assert.strictEqual(results.family, "Pinterest", msg("results.family", results.family, "Pinterest"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Pinterest/0.2 (+https://www.pinterest.com/bot.html) correctly", function() {
		const results = uaParser("Pinterest/0.2 (+https://www.pinterest.com/bot.html)");
		assert.strictEqual(results.family, "Pinterestbot", msg("results.family", results.family, "Pinterestbot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Pinterestbot/1.0; +https://www.pinterest.com/bot.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Pinterestbot/1.0; +https://www.pinterest.com/bot.html)");
		assert.strictEqual(results.family, "Pinterestbot", msg("results.family", results.family, "Pinterestbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 6.0.1; Nexus 5X Build/MMB29P) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2272.96 Mobile Safari/537.36 (compatible; Pinterestbot/1.0; +https://www.pinterest.com/bot.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 6.0.1; Nexus 5X Build/MMB29P) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/41.0.2272.96 Mobile Safari/537.36 (compatible; Pinterestbot/1.0; +https://www.pinterest.com/bot.html)");
		assert.strictEqual(results.family, "Pinterestbot", msg("results.family", results.family, "Pinterestbot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) qutebrowser/0.2.1 Safari/538.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) qutebrowser/0.2.1 Safari/538.1");
		assert.strictEqual(results.family, "qutebrowser", msg("results.family", results.family, "qutebrowser"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_4) AppleWebKit/537.36 (KHTML, like Gecko) pagedraw/0.1.0 Chrome/49.0.2623.75 Electron/0.37.8 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_4) AppleWebKit/537.36 (KHTML, like Gecko) pagedraw/0.1.0 Chrome/49.0.2623.75 Electron/0.37.8 Safari/537.36");
		assert.strictEqual(results.family, "Electron", msg("results.family", results.family, "Electron"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "37", msg("results.minor", results.minor || "", "37"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X) AppleWebKit/538.1 (KHTML, like Gecko) QupZilla/1.6.3 Safari/538.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X) AppleWebKit/538.1 (KHTML, like Gecko) QupZilla/1.6.3 Safari/538.1");
		assert.strictEqual(results.family, "QupZilla", msg("results.family", results.family, "QupZilla"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (OS/2 Warp 4.5) AppleWebKit/537.21 (KHTML, like Gecko) QupZilla/1.6.4 Safari/537.21 correctly", function() {
		const results = uaParser("Mozilla/5.0 (OS/2 Warp 4.5) AppleWebKit/537.21 (KHTML, like Gecko) QupZilla/1.6.4 Safari/537.21");
		assert.strictEqual(results.family, "QupZilla", msg("results.family", results.family, "QupZilla"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Unknown; UNIX BSD/SYSV system) AppleWebKit/534.34 (KHTML, like Gecko) QupZilla/1.7.0 Safari/534.34 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Unknown; UNIX BSD/SYSV system) AppleWebKit/534.34 (KHTML, like Gecko) QupZilla/1.7.0 Safari/534.34");
		assert.strictEqual(results.family, "QupZilla", msg("results.family", results.family, "QupZilla"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.21 (KHTML, like Gecko) QupZilla/1.6.1 Safari/537.21 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.21 (KHTML, like Gecko) QupZilla/1.6.1 Safari/537.21");
		assert.strictEqual(results.family, "QupZilla", msg("results.family", results.family, "QupZilla"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.03 beta 3 Safari/538.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.03 beta 3 Safari/538.1");
		assert.strictEqual(results.family, "Otter", msg("results.family", results.family, "Otter"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "03" || "", msg("results.patch", results.patch || "", "03" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.04 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; Win64; x64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.04");
		assert.strictEqual(results.family, "Otter", msg("results.family", results.family, "Otter"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "04" || "", msg("results.patch", results.patch || "", "04" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.21 (KHTML, like Gecko) Otter/0.9.04-dev correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.21 (KHTML, like Gecko) Otter/0.9.04-dev");
		assert.strictEqual(results.family, "Otter", msg("results.family", results.family, "Otter"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "04" || "", msg("results.patch", results.patch || "", "04" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.3.01-dev Safari/538.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.3.01-dev Safari/538.1");
		assert.strictEqual(results.family, "Otter", msg("results.family", results.family, "Otter"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "01" || "", msg("results.patch", results.patch || "", "01" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.03 beta 3 Safari/538.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.03 beta 3 Safari/538.1");
		assert.strictEqual(results.family, "Otter", msg("results.family", results.family, "Otter"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "03" || "", msg("results.patch", results.patch || "", "03" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.05 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/538.1 (KHTML, like Gecko) Otter/0.9.05");
		assert.strictEqual(results.family, "Otter", msg("results.family", results.family, "Otter"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "05" || "", msg("results.patch", results.patch || "", "05" || ""));
	});
	it("parses Mozilla/5.0 (Windows Phone 10.0; Android 4.2.1; NOKIA; Lumia 930) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/39.0.2171.71 Mobile Safari/537.36 Edge/12.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows Phone 10.0; Android 4.2.1; NOKIA; Lumia 930) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/39.0.2171.71 Mobile Safari/537.36 Edge/12.0");
		assert.strictEqual(results.family, "Edge Mobile", msg("results.family", results.family, "Edge Mobile"));
		assert.strictEqual(results.major || "", "12", msg("results.major", results.major || "", "12"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_2) AppleWebKit/537.36 (KHTML, like Gecko) brave/0.7.11 Chrome/47.0.2526.110 Brave/0.36.5 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_2) AppleWebKit/537.36 (KHTML, like Gecko) brave/0.7.11 Chrome/47.0.2526.110 Brave/0.36.5 Safari/537.36");
		assert.strictEqual(results.family, "Brave", msg("results.family", results.family, "Brave"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "11" || "", msg("results.patch", results.patch || "", "11" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) brave/0.7.12 Chrome/47.0.2526.110 Brave/0.36.7 Safari/537.36  correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) brave/0.7.12 Chrome/47.0.2526.110 Brave/0.36.7 Safari/537.36 ");
		assert.strictEqual(results.family, "Brave", msg("results.family", results.family, "Brave"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "12" || "", msg("results.patch", results.patch || "", "12" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.3; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) brave/0.7.10 Chrome/47.0.2526.110 Brave/0.36.5 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.3; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) brave/0.7.10 Chrome/47.0.2526.110 Brave/0.36.5 Safari/537.36");
		assert.strictEqual(results.family, "Brave", msg("results.family", results.family, "Brave"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "10" || "", msg("results.patch", results.patch || "", "10" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) HeadlessChrome Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) HeadlessChrome Safari/537.36");
		assert.strictEqual(results.family, "HeadlessChrome", msg("results.family", results.family, "HeadlessChrome"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_13_1) AppleWebKit/537.36 (KHTML, like Gecko) HeadlessChrome/62.0.3202.89 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_13_1) AppleWebKit/537.36 (KHTML, like Gecko) HeadlessChrome/62.0.3202.89 Safari/537.36");
		assert.strictEqual(results.family, "HeadlessChrome", msg("results.family", results.family, "HeadlessChrome"));
		assert.strictEqual(results.major || "", "62", msg("results.major", results.major || "", "62"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3202" || "", msg("results.patch", results.patch || "", "3202" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Ubuntu Chromium/59.0.3071.109 HeadlessChrome/59.0.3071.109 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Ubuntu Chromium/59.0.3071.109 HeadlessChrome/59.0.3071.109 Safari/537.36");
		assert.strictEqual(results.family, "HeadlessChrome", msg("results.family", results.family, "HeadlessChrome"));
		assert.strictEqual(results.major || "", "59", msg("results.major", results.major || "", "59"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3071" || "", msg("results.patch", results.patch || "", "3071" || ""));
	});
	it("parses Roku/DVP-6.2 (096.02E06005A) correctly", function() {
		const results = uaParser("Roku/DVP-6.2 (096.02E06005A)");
		assert.strictEqual(results.family, "Roku", msg("results.family", results.family, "Roku"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Roku/DVP-5.0 (025.00E08043A) correctly", function() {
		const results = uaParser("Roku/DVP-5.0 (025.00E08043A)");
		assert.strictEqual(results.family, "Roku", msg("results.family", results.family, "Roku"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Roku/DVP-5.1 (025.01E01195A) correctly", function() {
		const results = uaParser("Roku/DVP-5.1 (025.01E01195A)");
		assert.strictEqual(results.family, "Roku", msg("results.family", results.family, "Roku"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft Office/12.0 (Windows NT 6.1; Microsoft Office Outlook 12.0.6739; Pro) correctly", function() {
		const results = uaParser("Microsoft Office/12.0 (Windows NT 6.1; Microsoft Office Outlook 12.0.6739; Pro)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2007", msg("results.major", results.major || "", "2007"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft Office/14.0 (Windows NT 6.1; Microsoft Outlook 14.0.5128; Pro) correctly", function() {
		const results = uaParser("Microsoft Office/14.0 (Windows NT 6.1; Microsoft Outlook 14.0.5128; Pro)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2010", msg("results.major", results.major || "", "2010"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft Office/16.0 (Microsoft Outlook Mail 16.0.6525; Pro) correctly", function() {
		const results = uaParser("Microsoft Office/16.0 (Microsoft Outlook Mail 16.0.6525; Pro)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2016", msg("results.major", results.major || "", "2016"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft Office/16.0 (Windows NT 10.0; Microsoft Outlook 16.0.6326; Pro) correctly", function() {
		const results = uaParser("Microsoft Office/16.0 (Windows NT 10.0; Microsoft Outlook 16.0.6326; Pro)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2016", msg("results.major", results.major || "", "2016"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 10.0; WOW64; Trident/8.0; .NET4.0C; .NET4.0E; .NET CLR 2.0.50727; .NET CLR 3.0.30729; .NET CLR 3.5.30729; Microsoft Outlook 16.0.6366; ms-office; MSOffice 16) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 10.0; WOW64; Trident/8.0; .NET4.0C; .NET4.0E; .NET CLR 2.0.50727; .NET CLR 3.0.30729; .NET CLR 3.5.30729; Microsoft Outlook 16.0.6366; ms-office; MSOffice 16)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2016", msg("results.major", results.major || "", "2016"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; ms-office; MSOffice 16) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; ms-office; MSOffice 16)");
		assert.strictEqual(results.family, "Outlook", msg("results.family", results.family, "Outlook"));
		assert.strictEqual(results.major || "", "2016", msg("results.major", results.major || "", "2016"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-Express/7.0 (MSIE 7.0; Windows NT 6.1; Trident/4.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; TmstmpExt) correctly", function() {
		const results = uaParser("Outlook-Express/7.0 (MSIE 7.0; Windows NT 6.1; Trident/4.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; TmstmpExt)");
		assert.strictEqual(results.family, "Windows Live Mail", msg("results.family", results.family, "Windows Live Mail"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-Express/7.0 (MSIE 7.0; Windows NT 5.1; Trident/4.0; AskTB5.6; TmstmpExt) correctly", function() {
		const results = uaParser("Outlook-Express/7.0 (MSIE 7.0; Windows NT 5.1; Trident/4.0; AskTB5.6; TmstmpExt)");
		assert.strictEqual(results.family, "Windows Live Mail", msg("results.family", results.family, "Windows Live Mail"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-Express/7.0 (MSIE 7.0; Windows NT 6.1; WOW64; Trident/4.0; SLCC2; Media Center PC 6.0; OfficeLiveConnector.1.4; OfficeLivePatch.1.3; InfoPath.3; FDM; TmstmpExt) correctly", function() {
		const results = uaParser("Outlook-Express/7.0 (MSIE 7.0; Windows NT 6.1; WOW64; Trident/4.0; SLCC2; Media Center PC 6.0; OfficeLiveConnector.1.4; OfficeLivePatch.1.3; InfoPath.3; FDM; TmstmpExt)");
		assert.strictEqual(results.family, "Windows Live Mail", msg("results.family", results.family, "Windows Live Mail"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-Express/7.0 (MSIE 6.0; Windows NT 5.1; SV1; GTB6.3; .NET CLR 2.0.50727; .NET CLR 3.0.04506.30; InfoPath.2; .NET CLR 3.0.04506.648; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; OfficeLiveConnector.1.3; OfficeLivePatch.0.0; TmstmpExt) correctly", function() {
		const results = uaParser("Outlook-Express/7.0 (MSIE 6.0; Windows NT 5.1; SV1; GTB6.3; .NET CLR 2.0.50727; .NET CLR 3.0.04506.30; InfoPath.2; .NET CLR 3.0.04506.648; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; OfficeLiveConnector.1.3; OfficeLivePatch.0.0; TmstmpExt)");
		assert.strictEqual(results.family, "Windows Live Mail", msg("results.family", results.family, "Windows Live Mail"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-Express/7.0 (MSIE 8; Windows NT 5.1; Trident/4.0; GTB7.0; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; TmstmpExt) correctly", function() {
		const results = uaParser("Outlook-Express/7.0 (MSIE 8; Windows NT 5.1; Trident/4.0; GTB7.0; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; TmstmpExt)");
		assert.strictEqual(results.family, "Windows Live Mail", msg("results.family", results.family, "Windows Live Mail"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-Express/7.0 (MSIE 8.0; Windows NT 5.1; Trident/4.0; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; InfoPath.1; TmstmpExt) correctly", function() {
		const results = uaParser("Outlook-Express/7.0 (MSIE 8.0; Windows NT 5.1; Trident/4.0; .NET CLR 2.0.50727; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; InfoPath.1; TmstmpExt)");
		assert.strictEqual(results.family, "Windows Live Mail", msg("results.family", results.family, "Windows Live Mail"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-Express/7.0 (MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; HPDTDF; .NET4.0C; BRI/2; AskTbLOL/5.12.5.17640; TmstmpExt) correctly", function() {
		const results = uaParser("Outlook-Express/7.0 (MSIE 9.0; Windows NT 6.1; WOW64; Trident/5.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; HPDTDF; .NET4.0C; BRI/2; AskTbLOL/5.12.5.17640; TmstmpExt)");
		assert.strictEqual(results.family, "Windows Live Mail", msg("results.family", results.family, "Windows Live Mail"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 5.0.1; GT-I9505 Build/LRX22C) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.111 YaBrowser/16.2.1.1239.00 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 5.0.1; GT-I9505 Build/LRX22C) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/47.0.2526.111 YaBrowser/16.2.1.1239.00 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Yandex Browser", msg("results.family", results.family, "Yandex Browser"));
		assert.strictEqual(results.major || "", "16", msg("results.major", results.major || "", "16"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.4.2; SM-G800F Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/36.0.1985.135 MobileIron/1.6.0 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.4.2; SM-G800F Build/KOT49H) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/36.0.1985.135 MobileIron/1.6.0 Mobile Safari/537.36");
		assert.strictEqual(results.family, "MobileIron", msg("results.family", results.family, "MobileIron"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.4.4; GT-I9195I Build/KTU84P) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/43.0.2357.130 Crosswalk/14.43.343.17 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.4.4; GT-I9195I Build/KTU84P) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/43.0.2357.130 Crosswalk/14.43.343.17 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Crosswalk", msg("results.family", results.family, "Crosswalk"));
		assert.strictEqual(results.major || "", "14", msg("results.major", results.major || "", "14"));
		assert.strictEqual(results.minor || "", "43", msg("results.minor", results.minor || "", "43"));
		assert.equal(results.patch || "", "343" || "", msg("results.patch", results.patch || "", "343" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 6.0.1; Z831 Build/MMB29M) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/45.0.2454.94 Mobile Crosswalk/11.45.2454.20160425 Mobile Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 6.0.1; Z831 Build/MMB29M) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/45.0.2454.94 Mobile Crosswalk/11.45.2454.20160425 Mobile Safari/537.36");
		assert.strictEqual(results.family, "Crosswalk", msg("results.family", results.family, "Crosswalk"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "45", msg("results.minor", results.minor || "", "45"));
		assert.equal(results.patch || "", "2454" || "", msg("results.patch", results.patch || "", "2454" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 5.1; FTJ152B Build/LMY47D; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/51.0.2704.81 Mobile Safari/537.36 Line/6.4.1 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 5.1; FTJ152B Build/LMY47D; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/51.0.2704.81 Mobile Safari/537.36 Line/6.4.1");
		assert.strictEqual(results.family, "LINE", msg("results.family", results.family, "LINE"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 4.1.2; GT-S7710 Build/JZO54K) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.166 Mobile correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 4.1.2; GT-S7710 Build/JZO54K) AppleWebKit/535.19 (KHTML, like Gecko) Chrome/18.0.1025.166 Mobile");
		assert.strictEqual(results.family, "Chrome Mobile", msg("results.family", results.family, "Chrome Mobile"));
		assert.strictEqual(results.major || "", "18", msg("results.major", results.major || "", "18"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1025" || "", msg("results.patch", results.patch || "", "1025" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 4.4.4; de-de; SM-G850F Build/KTU84P) AppleWebKit/537.16 (KHTML, like Gecko) Version/4.0 Mobile Safari/537.16 Chrome/33.0.0.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 4.4.4; de-de; SM-G850F Build/KTU84P) AppleWebKit/537.16 (KHTML, like Gecko) Version/4.0 Mobile Safari/537.16 Chrome/33.0.0.0");
		assert.strictEqual(results.family, "Chrome Mobile WebView", msg("results.family", results.family, "Chrome Mobile WebView"));
		assert.strictEqual(results.major || "", "33", msg("results.major", results.major || "", "33"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 6.0.1; ru-ru; Redmi 4 Build/MMB29M) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/61.0.3163.128 Mobile Safari/537.36 XiaoMi/MiuiBrowser/10.3.6-g correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 6.0.1; ru-ru; Redmi 4 Build/MMB29M) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/61.0.3163.128 Mobile Safari/537.36 XiaoMi/MiuiBrowser/10.3.6-g");
		assert.strictEqual(results.family, "MiuiBrowser", msg("results.family", results.family, "MiuiBrowser"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Linux; U; Android 7.1.2; ru-ru; Redmi 4A Build/N2G47H) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/61.0.3163.128 Mobile Safari/537.36 XiaoMi/Mint Browser/1.3.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; U; Android 7.1.2; ru-ru; Redmi 4A Build/N2G47H) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/61.0.3163.128 Mobile Safari/537.36 XiaoMi/Mint Browser/1.3.3");
		assert.strictEqual(results.family, "Mint Browser", msg("results.family", results.family, "Mint Browser"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Web0S; Linux/SmartTV) AppleWebKit/537.41 (KHTML, like Gecko) Large Screen WebAppManager Safari/537.41 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Web0S; Linux/SmartTV) AppleWebKit/537.41 (KHTML, like Gecko) Large Screen WebAppManager Safari/537.41");
		assert.strictEqual(results.family, "Safari", msg("results.family", results.family, "Safari"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses MacOutlook/15.27.0.161010 (Intelx64 Mac OS X Version 10.11.6 (Build 15G1108)) correctly", function() {
		const results = uaParser("MacOutlook/15.27.0.161010 (Intelx64 Mac OS X Version 10.11.6 (Build 15G1108))");
		assert.strictEqual(results.family, "MacOutlook", msg("results.family", results.family, "MacOutlook"));
		assert.strictEqual(results.major || "", "15", msg("results.major", results.major || "", "15"));
		assert.strictEqual(results.minor || "", "27", msg("results.minor", results.minor || "", "27"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_6) AppleWebKit/601.7.8 (KHTML, like Gecko) Slack_SSB/2.0.3 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_6) AppleWebKit/601.7.8 (KHTML, like Gecko) Slack_SSB/2.0.3");
		assert.strictEqual(results.family, "Slack Desktop Client", msg("results.family", results.family, "Slack Desktop Client"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_4) AppleWebKit/537.36 (KHTML, like Gecko) AtomShell/2.6.0 Chrome/56.0.2924.87 Electron/1.6.3 Safari/537.36 MacAppStore/16.5.0 Slack_SSB/2.6.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_4) AppleWebKit/537.36 (KHTML, like Gecko) AtomShell/2.6.0 Chrome/56.0.2924.87 Electron/1.6.3 Safari/537.36 MacAppStore/16.5.0 Slack_SSB/2.6.0");
		assert.strictEqual(results.family, "Slack Desktop Client", msg("results.family", results.family, "Slack Desktop Client"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_6) AppleWebKit/537.36 (KHTML, like Gecko) Slack/2.6.0-beta18998559 Chrome/56.0.2924.87 AtomShell/1.6.3 Safari/537.36 Slack_SSB/2.6.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_11_6) AppleWebKit/537.36 (KHTML, like Gecko) Slack/2.6.0-beta18998559 Chrome/56.0.2924.87 AtomShell/1.6.3 Safari/537.36 Slack_SSB/2.6.0");
		assert.strictEqual(results.family, "Slack Desktop Client", msg("results.family", results.family, "Slack Desktop Client"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_4) AppleWebKit/603.1.30 (KHTML, like Gecko) HipChat/732 (modern) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_4) AppleWebKit/603.1.30 (KHTML, like Gecko) HipChat/732 (modern)");
		assert.strictEqual(results.family, "HipChat Desktop Client", msg("results.family", results.family, "HipChat Desktop Client"));
		assert.strictEqual(results.major || "", "732", msg("results.major", results.major || "", "732"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses HipChat Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) QtWebEngine/5.7.0 Chrome/49.0.2623.111 Safari/537.36 correctly", function() {
		const results = uaParser("HipChat Mozilla/5.0 (Windows NT 6.2; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) QtWebEngine/5.7.0 Chrome/49.0.2623.111 Safari/537.36");
		assert.strictEqual(results.family, "HipChat Desktop Client", msg("results.family", results.family, "HipChat Desktop Client"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft-CryptoAPI/6.1 correctly", function() {
		const results = uaParser("Microsoft-CryptoAPI/6.1");
		assert.strictEqual(results.family, "Microsoft-CryptoAPI", msg("results.family", results.family, "Microsoft-CryptoAPI"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft SkyDriveSync 17.3.6517.0809 ship; Windows NT 6.1 Service Pack 1 (7601) correctly", function() {
		const results = uaParser("Microsoft SkyDriveSync 17.3.6517.0809 ship; Windows NT 6.1 Service Pack 1 (7601)");
		assert.strictEqual(results.family, "Microsoft SkyDriveSync", msg("results.family", results.family, "Microsoft SkyDriveSync"));
		assert.strictEqual(results.major || "", "17", msg("results.major", results.major || "", "17"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "6517" || "", msg("results.patch", results.patch || "", "6517" || ""));
	});
	it("parses ExchangeServicesClient/14.02.0051.000 correctly", function() {
		const results = uaParser("ExchangeServicesClient/14.02.0051.000");
		assert.strictEqual(results.family, "ExchangeServicesClient", msg("results.family", results.family, "ExchangeServicesClient"));
		assert.strictEqual(results.major || "", "14", msg("results.major", results.major || "", "14"));
		assert.strictEqual(results.minor || "", "02", msg("results.minor", results.minor || "", "02"));
		assert.equal(results.patch || "", "0051" || "", msg("results.patch", results.patch || "", "0051" || ""));
	});
	it("parses Mac OS X/10.11.6 (15G1004); ExchangeWebServices/6.0 (243); Mail/9.3 (3124) correctly", function() {
		const results = uaParser("Mac OS X/10.11.6 (15G1004); ExchangeWebServices/6.0 (243); Mail/9.3 (3124)");
		assert.strictEqual(results.family, "ExchangeWebServices", msg("results.family", results.family, "ExchangeWebServices"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Slackbot-LinkExpanding 1.0 (+https://api.slack.com/robots) correctly", function() {
		const results = uaParser("Slackbot-LinkExpanding 1.0 (+https://api.slack.com/robots)");
		assert.strictEqual(results.family, "Slackbot-LinkExpanding", msg("results.family", results.family, "Slackbot-LinkExpanding"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Slack-ImgProxy 1.136 (+https://api.slack.com/robots) correctly", function() {
		const results = uaParser("Slack-ImgProxy 1.136 (+https://api.slack.com/robots)");
		assert.strictEqual(results.family, "Slack-ImgProxy", msg("results.family", results.family, "Slack-ImgProxy"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "136", msg("results.minor", results.minor || "", "136"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses okhttp/3.4.2 correctly", function() {
		const results = uaParser("okhttp/3.4.2");
		assert.strictEqual(results.family, "okhttp", msg("results.family", results.family, "okhttp"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_0) AppleWebKit/537.11 (KHTML, like Gecko) Chrome/23.0.1300.0 Iron/23.0.1300.0 Safari/537.11 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_10_0) AppleWebKit/537.11 (KHTML, like Gecko) Chrome/23.0.1300.0 Iron/23.0.1300.0 Safari/537.11");
		assert.strictEqual(results.family, "Iron", msg("results.family", results.family, "Iron"));
		assert.strictEqual(results.major || "", "23", msg("results.major", results.major || "", "23"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "1300" || "", msg("results.patch", results.patch || "", "1300" || ""));
	});
	it("parses Mozilla/5.0 (X11; U; Linux x86_64; en-US) AppleWebKit/534.13 (KHTML, like Gecko) Iron/9.0.600.2 Chrome/9.0.600.2 Safari/534.13 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; U; Linux x86_64; en-US) AppleWebKit/534.13 (KHTML, like Gecko) Iron/9.0.600.2 Chrome/9.0.600.2 Safari/534.13");
		assert.strictEqual(results.family, "Iron", msg("results.family", results.family, "Iron"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "600" || "", msg("results.patch", results.patch || "", "600" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/52.0.2750.0 Iron Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/52.0.2750.0 Iron Safari/537.36");
		assert.strictEqual(results.family, "Iron", msg("results.family", results.family, "Iron"));
		assert.strictEqual(results.major || "", "52", msg("results.major", results.major || "", "52"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2750" || "", msg("results.patch", results.patch || "", "2750" || ""));
	});
	it("parses Mozilla/5.0 (Android; U; en-US) AppleWebKit/533.19.4 (KHTML, like Gecko) AdobeAIR/23.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Android; U; en-US) AppleWebKit/533.19.4 (KHTML, like Gecko) AdobeAIR/23.0");
		assert.strictEqual(results.family, "AdobeAIR", msg("results.family", results.family, "AdobeAIR"));
		assert.strictEqual(results.major || "", "23", msg("results.major", results.major || "", "23"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; U; Intel Mac OS X; en-US) AppleWebKit/533.19.4 (KHTML, like Gecko) AdobeAIR/19.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; U; Intel Mac OS X; en-US) AppleWebKit/533.19.4 (KHTML, like Gecko) AdobeAIR/19.0");
		assert.strictEqual(results.family, "AdobeAIR", msg("results.family", results.family, "AdobeAIR"));
		assert.strictEqual(results.major || "", "19", msg("results.major", results.major || "", "19"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (iOS; U; en-US) AppleWebKit/533.19.4 (KHTML, like Gecko) AdobeAIR/19.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (iOS; U; en-US) AppleWebKit/533.19.4 (KHTML, like Gecko) AdobeAIR/19.0");
		assert.strictEqual(results.family, "AdobeAIR", msg("results.family", results.family, "AdobeAIR"));
		assert.strictEqual(results.major || "", "19", msg("results.major", results.major || "", "19"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Kurio/3.0.8 Build 65303(Android Kitkat 4.4.4; Phone) correctly", function() {
		const results = uaParser("Kurio/3.0.8 Build 65303(Android Kitkat 4.4.4; Phone)");
		assert.strictEqual(results.family, "Kurio App", msg("results.family", results.family, "Kurio App"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses BacaBerita App/5.5.0 (Linux; U; Android 4.4.4; en-us) Mobile Safari correctly", function() {
		const results = uaParser("BacaBerita App/5.5.0 (Linux; U; Android 4.4.4; en-us) Mobile Safari");
		assert.strictEqual(results.family, "BacaBerita App", msg("results.family", results.family, "BacaBerita App"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.36 (KHTML, like Gecko) MxBrowser/4.5.2.2000 Chrome/30.0.1551.0 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.36 (KHTML, like Gecko) MxBrowser/4.5.2.2000 Chrome/30.0.1551.0 Safari/537.36");
		assert.strictEqual(results.family, "Maxthon", msg("results.family", results.family, "Maxthon"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.36 (KHTML, like Gecko) Maxthon/4.4.5.1000 Chrome/30.0.1599.101 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.36 (KHTML, like Gecko) Maxthon/4.4.5.1000 Chrome/30.0.1599.101 Safari/537.36");
		assert.strictEqual(results.family, "Maxthon", msg("results.family", results.family, "Maxthon"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.0; SM-G930P Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36 MxBrowser/4.5.10.7000 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.0; SM-G930P Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36 MxBrowser/4.5.10.7000");
		assert.strictEqual(results.family, "Maxthon", msg("results.family", results.family, "Maxthon"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "10" || "", msg("results.patch", results.patch || "", "10" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.6 (KHTML, like Gecko) Chrome/18.0.1025.133 Safari/537.6 Midori/0.5 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.6 (KHTML, like Gecko) Chrome/18.0.1025.133 Safari/537.6 Midori/0.5");
		assert.strictEqual(results.family, "Midori", msg("results.family", results.family, "Midori"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses curl/7.29.0 correctly", function() {
		const results = uaParser("curl/7.29.0");
		assert.strictEqual(results.family, "curl", msg("results.family", results.family, "curl"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "29", msg("results.minor", results.minor || "", "29"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Debian APT-HTTP/1.3 (1.0.1ubuntu2) correctly", function() {
		const results = uaParser("Debian APT-HTTP/1.3 (1.0.1ubuntu2)");
		assert.strictEqual(results.family, "Debian APT-HTTP", msg("results.family", results.family, "Debian APT-HTTP"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses jupdate correctly", function() {
		const results = uaParser("jupdate");
		assert.strictEqual(results.family, "jupdate", msg("results.family", results.family, "jupdate"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses libcurl-agent/1.0 correctly", function() {
		const results = uaParser("libcurl-agent/1.0");
		assert.strictEqual(results.family, "libcurl-agent", msg("results.family", results.family, "libcurl-agent"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses libwww-perl/6.05 correctly", function() {
		const results = uaParser("libwww-perl/6.05");
		assert.strictEqual(results.family, "libwww-perl", msg("results.family", results.family, "libwww-perl"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "05", msg("results.minor", results.minor || "", "05"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Microsoft-CryptoAPI/6.3 correctly", function() {
		const results = uaParser("Microsoft-CryptoAPI/6.3");
		assert.strictEqual(results.family, "Microsoft-CryptoAPI", msg("results.family", results.family, "Microsoft-CryptoAPI"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses OpenBSD ftp correctly", function() {
		const results = uaParser("OpenBSD ftp");
		assert.strictEqual(results.family, "OpenBSD ftp", msg("results.family", results.family, "OpenBSD ftp"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SophosAgent/1.0 (type= spa ) correctly", function() {
		const results = uaParser("SophosAgent/1.0 (type= spa )");
		assert.strictEqual(results.family, "SophosAgent", msg("results.family", results.family, "SophosAgent"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses SophosUpdateManager/1.5.7.50 SDDS/2.0 (u= EO2ANA123G c= 6342da15-f351-4ab7-9656-3f5f2d50885d ) correctly", function() {
		const results = uaParser("SophosUpdateManager/1.5.7.50 SDDS/2.0 (u= EO2ANA123G c= 6342da15-f351-4ab7-9656-3f5f2d50885d )");
		assert.strictEqual(results.family, "SophosUpdateManager", msg("results.family", results.family, "SophosUpdateManager"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses Ubuntu APT-HTTP/1.3 (0.7.20.2ubuntu6) correctly", function() {
		const results = uaParser("Ubuntu APT-HTTP/1.3 (0.7.20.2ubuntu6)");
		assert.strictEqual(results.family, "Ubuntu APT-HTTP", msg("results.family", results.family, "Ubuntu APT-HTTP"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses urlgrabber/3.10 yum/3.4.3 correctly", function() {
		const results = uaParser("urlgrabber/3.10 yum/3.4.3");
		assert.strictEqual(results.family, "urlgrabber", msg("results.family", results.family, "urlgrabber"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses urlgrabber/3.9.1 yum/3.2.29 correctly", function() {
		const results = uaParser("urlgrabber/3.9.1 yum/3.2.29");
		assert.strictEqual(results.family, "urlgrabber", msg("results.family", results.family, "urlgrabber"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Wget/1.14 (linux-gnu) correctly", function() {
		const results = uaParser("Wget/1.14 (linux-gnu)");
		assert.strictEqual(results.family, "Wget", msg("results.family", results.family, "Wget"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "14", msg("results.minor", results.minor || "", "14"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Windows-Update-Agent/7.9.9600.17729 Client-Protocol/1.21 correctly", function() {
		const results = uaParser("Windows-Update-Agent/7.9.9600.17729 Client-Protocol/1.21");
		assert.strictEqual(results.family, "Windows-Update-Agent", msg("results.family", results.family, "Windows-Update-Agent"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "9600" || "", msg("results.patch", results.patch || "", "9600" || ""));
	});
	it("parses Windows-Update-Agent/7.9.9600.18094 Client-Protocol/1.21 correctly", function() {
		const results = uaParser("Windows-Update-Agent/7.9.9600.18094 Client-Protocol/1.21");
		assert.strictEqual(results.family, "Windows-Update-Agent", msg("results.family", results.family, "Windows-Update-Agent"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "9600" || "", msg("results.patch", results.patch || "", "9600" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 5.1.1; MI NOTE Pro Build/LMY47V; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/48.0.2564.116 Mobile Safari/537.36 baidubrowser/7.7.13.0 (Baidu; P1 5.1.1) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 5.1.1; MI NOTE Pro Build/LMY47V; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/48.0.2564.116 Mobile Safari/537.36 baidubrowser/7.7.13.0 (Baidu; P1 5.1.1)");
		assert.strictEqual(results.family, "Baidu Browser", msg("results.family", results.family, "Baidu Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "13" || "", msg("results.patch", results.patch || "", "13" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Dragon/36.1.1.21 Chrome/36.0.1985.97 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) Dragon/36.1.1.21 Chrome/36.0.1985.97 Safari/537.36");
		assert.strictEqual(results.family, "Dragon", msg("results.family", results.family, "Dragon"));
		assert.strictEqual(results.major || "", "36", msg("results.major", results.major || "", "36"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Windows; U; Windows CE 5.1; rv:1.8.1a3) Gecko/20060610 Minimo/0.016 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows; U; Windows CE 5.1; rv:1.8.1a3) Gecko/20060610 Minimo/0.016");
		assert.strictEqual(results.family, "Minimo", msg("results.family", results.family, "Minimo"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "016", msg("results.minor", results.minor || "", "016"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Opera/9.80 (MAUI Runtime; Opera Mini/4.4.39008/37.9178; U; en) Presto/2.12.423 Version/12.16 correctly", function() {
		const results = uaParser("Opera/9.80 (MAUI Runtime; Opera Mini/4.4.39008/37.9178; U; en) Presto/2.12.423 Version/12.16");
		assert.strictEqual(results.family, "Opera Mini", msg("results.family", results.family, "Opera Mini"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "39008" || "", msg("results.patch", results.patch || "", "39008" || ""));
	});
	it("parses Mozilla/5.0 (compatible; 008/0.83; http://www.80legs.com/spider.html) Gecko/2008032620 correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; 008/0.83; http://www.80legs.com/spider.html) Gecko/2008032620");
		assert.strictEqual(results.family, "008", msg("results.family", results.family, "008"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "83", msg("results.minor", results.minor || "", "83"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_2) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/53.0.2785.21 Safari/537.36 MMS/1.0.2459.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_2) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/53.0.2785.21 Safari/537.36 MMS/1.0.2459.0");
		assert.strictEqual(results.family, "Opera Neon", msg("results.family", results.family, "Opera Neon"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "2459" || "", msg("results.patch", results.patch || "", "2459" || ""));
	});
	it("parses PANTECH-EUROPA-U4000-orange/1.0 Obigo/Q04C MMS/1.2.0 profile/MIDP-2.0 configuration/CLDC-1.1 correctly", function() {
		const results = uaParser("PANTECH-EUROPA-U4000-orange/1.0 Obigo/Q04C MMS/1.2.0 profile/MIDP-2.0 configuration/CLDC-1.1");
		assert.strictEqual(results.family, "Obigo", msg("results.family", results.family, "Obigo"));
		assert.strictEqual(results.major || "", "04", msg("results.major", results.major || "", "04"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses masscan/1.0 (https://github.com/robertdavidgraham/masscan) correctly", function() {
		const results = uaParser("masscan/1.0 (https://github.com/robertdavidgraham/masscan)");
		assert.strictEqual(results.family, "masscan", msg("results.family", results.family, "masscan"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/63.0.3239.132 Safari/537.36 SznProhlizec/4.3.0-251281 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/63.0.3239.132 Safari/537.36 SznProhlizec/4.3.0-251281");
		assert.strictEqual(results.family, "Seznam prohlížeč", msg("results.family", results.family, "Seznam prohlížeč"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.1.2; Redmi 4X Build/N2G47H) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/64.0.3282.137 Mobile Safari/537.36 SznProhlizec/5.2.1a correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.1.2; Redmi 4X Build/N2G47H) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/64.0.3282.137 Mobile Safari/537.36 SznProhlizec/5.2.1a");
		assert.strictEqual(results.family, "Seznam prohlížeč", msg("results.family", results.family, "Seznam prohlížeč"));
		assert.strictEqual(results.major || "", "5", msg("results.major", results.major || "", "5"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_0) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/55.0.2883.87 Safari/537.36 SznProhlizec/3.8.4 NWjs/0.19.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_0) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/55.0.2883.87 Safari/537.36 SznProhlizec/3.8.4 NWjs/0.19.6");
		assert.strictEqual(results.family, "Seznam prohlížeč", msg("results.family", results.family, "Seznam prohlížeč"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.3; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) coc_coc_browser/42.0 CoRom/36.0.1985.144 Chrome/36.0.1985.144 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.3; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) coc_coc_browser/42.0 CoRom/36.0.1985.144 Chrome/36.0.1985.144 Safari/537.36");
		assert.strictEqual(results.family, "Coc Coc", msg("results.family", results.family, "Coc Coc"));
		assert.strictEqual(results.major || "", "42", msg("results.major", results.major || "", "42"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) coc_coc_browser/49.0 Chrome/43.0.2357.138 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.1) AppleWebKit/537.36 (KHTML, like Gecko) coc_coc_browser/49.0 Chrome/43.0.2357.138 Safari/537.36");
		assert.strictEqual(results.family, "Coc Coc", msg("results.family", results.family, "Coc Coc"));
		assert.strictEqual(results.major || "", "49", msg("results.major", results.major || "", "49"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.36 (KHTML, like Gecko) coc_coc_browser/50.0.125 Chrome/44.0.2403.125 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 5.1) AppleWebKit/537.36 (KHTML, like Gecko) coc_coc_browser/50.0.125 Chrome/44.0.2403.125 Safari/537.36");
		assert.strictEqual(results.family, "Coc Coc", msg("results.family", results.family, "Coc Coc"));
		assert.strictEqual(results.major || "", "50", msg("results.major", results.major || "", "50"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "125" || "", msg("results.patch", results.patch || "", "125" || ""));
	});
	it("parses Mozilla/5.0 (compatible; Qwantify/2.4w; +https://www.qwant.com/)/2.4w correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; Qwantify/2.4w; +https://www.qwant.com/)/2.4w");
		assert.strictEqual(results.family, "Qwantify", msg("results.family", results.family, "Qwantify"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/42.0.2311.135 Safari/537.36 Edge/12 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/42.0.2311.135 Safari/537.36 Edge/12");
		assert.strictEqual(results.family, "Edge", msg("results.family", results.family, "Edge"));
		assert.strictEqual(results.major || "", "12", msg("results.major", results.major || "", "12"));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Bloglovin/1.0 (http://www.bloglovin.com; 1000 subscribers) correctly", function() {
		const results = uaParser("Bloglovin/1.0 (http://www.bloglovin.com; 1000 subscribers)");
		assert.strictEqual(results.family, "Bloglovin", msg("results.family", results.family, "Bloglovin"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Feedbin feed-id:1033517 - 1000 subscribers correctly", function() {
		const results = uaParser("Feedbin feed-id:1033517 - 1000 subscribers");
		assert.strictEqual(results.family, "Feedbin", msg("results.family", results.family, "Feedbin"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Tiny Tiny RSS/16.3 (http://tt-rss.org/) correctly", function() {
		const results = uaParser("Tiny Tiny RSS/16.3 (http://tt-rss.org/)");
		assert.strictEqual(results.family, "Tiny Tiny RSS", msg("results.family", results.family, "Tiny Tiny RSS"));
		assert.strictEqual(results.major || "", "16", msg("results.major", results.major || "", "16"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mtps Feed Aggregation System correctly", function() {
		const results = uaParser("Mtps Feed Aggregation System");
		assert.strictEqual(results.family, "Mtps Feed Aggregation System", msg("results.family", results.family, "Mtps Feed Aggregation System"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Stringer correctly", function() {
		const results = uaParser("Stringer");
		assert.strictEqual(results.family, "Stringer", msg("results.family", results.family, "Stringer"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Box Sync/4.0.7848;Darwin/10.13;i386/64bit correctly", function() {
		const results = uaParser("Box Sync/4.0.7848;Darwin/10.13;i386/64bit");
		assert.strictEqual(results.family, "Box Sync", msg("results.family", results.family, "Box Sync"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "7848" || "", msg("results.patch", results.patch || "", "7848" || ""));
	});
	it("parses Box/1.2.93;Darwin/10.13;i386/64bit correctly", function() {
		const results = uaParser("Box/1.2.93;Darwin/10.13;i386/64bit");
		assert.strictEqual(results.family, "Box", msg("results.family", results.family, "Box"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "93" || "", msg("results.patch", results.patch || "", "93" || ""));
	});
	it("parses Mozilla/5.0 (Macintosh; Intel Mac OS X 10_13_0) AppleWebKit/537.36 (KHTML, like Gecko) BoxNotes/1.3.0 Chrome/56.0.2924.87 Electron/1.6.8 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_13_0) AppleWebKit/537.36 (KHTML, like Gecko) BoxNotes/1.3.0 Chrome/56.0.2924.87 Electron/1.6.8 Safari/537.36");
		assert.strictEqual(results.family, "BoxNotes", msg("results.family", results.family, "BoxNotes"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Box Sync/4.0.7848;Windows/8.1;x86 Family 6 Model 158 Stepping 9, GenuineIntel/32bit correctly", function() {
		const results = uaParser("Box Sync/4.0.7848;Windows/8.1;x86 Family 6 Model 158 Stepping 9, GenuineIntel/32bit");
		assert.strictEqual(results.family, "Box Sync", msg("results.family", results.family, "Box Sync"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "7848" || "", msg("results.patch", results.patch || "", "7848" || ""));
	});
	it("parses Box Sync/4.0.7848;Windows/10;Intel64 Family 6 Model 158 Stepping 9, GenuineIntel/64bit correctly", function() {
		const results = uaParser("Box Sync/4.0.7848;Windows/10;Intel64 Family 6 Model 158 Stepping 9, GenuineIntel/64bit");
		assert.strictEqual(results.family, "Box Sync", msg("results.family", results.family, "Box Sync"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "7848" || "", msg("results.patch", results.patch || "", "7848" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 6.3) AppleWebKit/537.36 (KHTML, like Gecko) BoxNotes/1.3.0 Chrome/56.0.2924.87 Electron/1.6.8 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 6.3) AppleWebKit/537.36 (KHTML, like Gecko) BoxNotes/1.3.0 Chrome/56.0.2924.87 Electron/1.6.8 Safari/537.36");
		assert.strictEqual(results.family, "BoxNotes", msg("results.family", results.family, "BoxNotes"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) BoxNotes/1.3.0 Chrome/56.0.2924.87 Electron/1.6.8 Safari/537.36 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) BoxNotes/1.3.0 Chrome/56.0.2924.87 Electron/1.6.8 Safari/537.36");
		assert.strictEqual(results.family, "BoxNotes", msg("results.family", results.family, "BoxNotes"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Box/1.2.93;Windows/10;Intel64 Family 6 Model 158 Stepping 9, GenuineIntel/64bit correctly", function() {
		const results = uaParser("Box/1.2.93;Windows/10;Intel64 Family 6 Model 158 Stepping 9, GenuineIntel/64bit");
		assert.strictEqual(results.family, "Box", msg("results.family", results.family, "Box"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "93" || "", msg("results.patch", results.patch || "", "93" || ""));
	});
	it("parses Evolution/3.26.2.1 correctly", function() {
		const results = uaParser("Evolution/3.26.2.1");
		assert.strictEqual(results.family, "Evolution", msg("results.family", results.family, "Evolution"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "26", msg("results.minor", results.minor || "", "26"));
		assert.equal(results.patch || "", "2.1" || "", msg("results.patch", results.patch || "", "2.1" || ""));
	});
	it("parses RCM CardDAV plugin/2.0.4 correctly", function() {
		const results = uaParser("RCM CardDAV plugin/2.0.4");
		assert.strictEqual(results.family, "RCM CardDAV plugin", msg("results.family", results.family, "RCM CardDAV plugin"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses RCM CardDAV plugin/0.9.2-dev correctly", function() {
		const results = uaParser("RCM CardDAV plugin/0.9.2-dev");
		assert.strictEqual(results.family, "RCM CardDAV plugin", msg("results.family", results.family, "RCM CardDAV plugin"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "2-dev" || "", msg("results.patch", results.patch || "", "2-dev" || ""));
	});
	it("parses DAVdroid/1.9.2-gplay (2017/11/04; dav4android; okhttp3) Android/7.1.2 correctly", function() {
		const results = uaParser("DAVdroid/1.9.2-gplay (2017/11/04; dav4android; okhttp3) Android/7.1.2");
		assert.strictEqual(results.family, "DAVdroid", msg("results.family", results.family, "DAVdroid"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses DAVdroid/1.9-ose (2017/10/19; dav4android; okhttp3) Android/7.1.2 correctly", function() {
		const results = uaParser("DAVdroid/1.9-ose (2017/10/19; dav4android; okhttp3) Android/7.1.2");
		assert.strictEqual(results.family, "DAVdroid", msg("results.family", results.family, "DAVdroid"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Windows) mirall/2.3.2 (build 1) (Nextcloud) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows) mirall/2.3.2 (build 1) (Nextcloud)");
		assert.strictEqual(results.family, "Nextcloud", msg("results.family", results.family, "Nextcloud"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Linux) mirall/2.3.2 (Nextcloud) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux) mirall/2.3.2 (Nextcloud)");
		assert.strictEqual(results.family, "Nextcloud", msg("results.family", results.family, "Nextcloud"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (Android) ownCloud-android/2.0.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Android) ownCloud-android/2.0.0");
		assert.strictEqual(results.family, "Owncloud", msg("results.family", results.family, "Owncloud"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.3; WOW64; Trident/7.0; .NET4.0E; .NET4.0C; InfoPath.3) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.3; WOW64; Trident/7.0; .NET4.0E; .NET4.0C; InfoPath.3)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.2; Trident/6.0; .NET4.0E; .NET4.0C; .NET CLR 3.5.30729; .NET CLR 2.0.50727; .NET CLR 3.0.30729) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.2; Trident/6.0; .NET4.0E; .NET4.0C; .NET CLR 3.5.30729; .NET CLR 2.0.50727; .NET CLR 3.0.30729)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; Trident/5.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; .NET4.0C; .NET4.0E; InfoPath.2) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 6.1; Trident/5.0; SLCC2; .NET CLR 2.0.50727; .NET CLR 3.5.30729; .NET CLR 3.0.30729; Media Center PC 6.0; .NET4.0C; .NET4.0E; InfoPath.2)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Tableau/1.0 (1025794) correctly", function() {
		const results = uaParser("Tableau/1.0 (1025794)");
		assert.strictEqual(results.family, "Tableau", msg("results.family", results.family, "Tableau"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses BUbiNG (+http://law.di.unimi.it/BUbiNG.html) correctly", function() {
		const results = uaParser("BUbiNG (+http://law.di.unimi.it/BUbiNG.html)");
		assert.strictEqual(results.family, "BUbiNG", msg("results.family", results.family, "BUbiNG"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; SemrushBot/1.2~bl; +http://www.semrush.com/bot.html) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; SemrushBot/1.2~bl; +http://www.semrush.com/bot.html)");
		assert.strictEqual(results.family, "SemrushBot", msg("results.family", results.family, "SemrushBot"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Outlook-iOS/665.29827.prod.iphone (2.63.0) correctly", function() {
		const results = uaParser("Outlook-iOS/665.29827.prod.iphone (2.63.0)");
		assert.strictEqual(results.family, "Outlook-iOS", msg("results.family", results.family, "Outlook-iOS"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "63", msg("results.minor", results.minor || "", "63"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.0; LG-TP260 Build/NRD90U; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/64.0.3282.137 Mobile Safari/537.36 Instagram 33.0.0.11.92 Android (24/7.0; 320dpi; 720x1193; LGE/lge; LG-TP260; lv517; lv517; en_US; 93117667) correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.0; LG-TP260 Build/NRD90U; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/64.0.3282.137 Mobile Safari/537.36 Instagram 33.0.0.11.92 Android (24/7.0; 320dpi; 720x1193; LGE/lge; LG-TP260; lv517; lv517; en_US; 93117667)");
		assert.strictEqual(results.family, "Instagram", msg("results.family", results.family, "Instagram"));
		assert.strictEqual(results.major || "", "33", msg("results.major", results.major || "", "33"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.0; SM-G610F Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/63.0.3239.111 Mobile Safari/537.36 Flipboard/4.1.9/4323,4.1.9.4323 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.0; SM-G610F Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/63.0.3239.111 Mobile Safari/537.36 Flipboard/4.1.9/4323,4.1.9.4323");
		assert.strictEqual(results.family, "Flipboard", msg("results.family", results.family, "Flipboard"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.0; SM-G930F Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/64.0.3282.137 Mobile Safari/537.36 Onefootball/Android/9.10.6 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.0; SM-G930F Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/64.0.3282.137 Mobile Safari/537.36 Onefootball/Android/9.10.6");
		assert.strictEqual(results.family, "Onefootball", msg("results.family", results.family, "Onefootball"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "6" || "", msg("results.patch", results.patch || "", "6" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 7.0; SM-A520F Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36 Flipboard-Briefing/2.7.28 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 7.0; SM-A520F Build/NRD90M; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/58.0.3029.83 Mobile Safari/537.36 Flipboard-Briefing/2.7.28");
		assert.strictEqual(results.family, "Flipboard-Briefing", msg("results.family", results.family, "Flipboard-Briefing"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "28" || "", msg("results.patch", results.patch || "", "28" || ""));
	});
	it("parses Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 10.0; WOW64; Trident/8.0; .NET4.0C; .NET4.0E; .NET CLR 2.0.50727; .NET CLR 3.0.30729; .NET CLR 3.5.30729) correctly", function() {
		const results = uaParser("Mozilla/4.0 (compatible; MSIE 7.0; Windows NT 10.0; WOW64; Trident/8.0; .NET4.0C; .NET4.0E; .NET CLR 2.0.50727; .NET CLR 3.0.30729; .NET CLR 3.5.30729)");
		assert.strictEqual(results.family, "IE", msg("results.family", results.family, "IE"));
		assert.strictEqual(results.major || "", "11", msg("results.major", results.major || "", "11"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ESPN Radio/3.2.113 CFNetwork/485.12.30 Darwin/10.4.0 correctly", function() {
		const results = uaParser("ESPN Radio/3.2.113 CFNetwork/485.12.30 Darwin/10.4.0");
		assert.strictEqual(results.family, "ESPN", msg("results.family", results.family, "ESPN"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "113" || "", msg("results.patch", results.patch || "", "113" || ""));
	});
	it("parses ESPN Radio 4.7.4 rv:1032 (iPhone; iPhone OS 9.2.1; en_US) correctly", function() {
		const results = uaParser("ESPN Radio 4.7.4 rv:1032 (iPhone; iPhone OS 9.2.1; en_US)");
		assert.strictEqual(results.family, "ESPN", msg("results.family", results.family, "ESPN"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses ESPN Radio 4.5.1 (iPhone; iPhone OS 5.1.1; en_US) correctly", function() {
		const results = uaParser("ESPN Radio 4.5.1 (iPhone; iPhone OS 5.1.1; en_US)");
		assert.strictEqual(results.family, "ESPN", msg("results.family", results.family, "ESPN"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses ESPN Radio 4.0 (iPhone; iPhone OS 7.1.2; en_AU) correctly", function() {
		const results = uaParser("ESPN Radio 4.0 (iPhone; iPhone OS 7.1.2; en_AU)");
		assert.strictEqual(results.family, "ESPN", msg("results.family", results.family, "ESPN"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:55.0) Gecko/20100101 Firefox/55.2.2 Waterfox/55.2.2 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:55.0) Gecko/20100101 Firefox/55.2.2 Waterfox/55.2.2");
		assert.strictEqual(results.family, "Waterfox", msg("results.family", results.family, "Waterfox"));
		assert.strictEqual(results.major || "", "55", msg("results.major", results.major || "", "55"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses Mozilla/5.0 (X11; Linux x86_64; rv:55.0) Gecko/20100101 Goanna/4.0 Firefox/55.0 Basilisk/20171113 correctly", function() {
		const results = uaParser("Mozilla/5.0 (X11; Linux x86_64; rv:55.0) Gecko/20100101 Goanna/4.0 Firefox/55.0 Basilisk/20171113");
		assert.strictEqual(results.family, "Basilisk", msg("results.family", results.family, "Basilisk"));
		assert.strictEqual(results.major || "", "55", msg("results.major", results.major || "", "55"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "20171113" || "", msg("results.patch", results.patch || "", "20171113" || ""));
	});
	it("parses Go-http-client/1.1 correctly", function() {
		const results = uaParser("Go-http-client/1.1");
		assert.strictEqual(results.family, "Go-http-client", msg("results.family", results.family, "Go-http-client"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses scalaj-http/1.0 correctly", function() {
		const results = uaParser("scalaj-http/1.0");
		assert.strictEqual(results.family, "scalaj-http", msg("results.family", results.family, "scalaj-http"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses reqwest/0.8.1 correctly", function() {
		const results = uaParser("reqwest/0.8.1");
		assert.strictEqual(results.family, "reqwest", msg("results.family", results.family, "reqwest"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses akka-http/10.0.10 correctly", function() {
		const results = uaParser("akka-http/10.0.10");
		assert.strictEqual(results.family, "akka-http", msg("results.family", results.family, "akka-http"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "10" || "", msg("results.patch", results.patch || "", "10" || ""));
	});
	it("parses Python/3.6 aiohttp/3.5.4 correctly", function() {
		const results = uaParser("Python/3.6 aiohttp/3.5.4");
		assert.strictEqual(results.family, "Python/3.6 aiohttp", msg("results.family", results.family, "Python/3.6 aiohttp"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "5", msg("results.minor", results.minor || "", "5"));
		assert.equal(results.patch || "", "4" || "", msg("results.patch", results.patch || "", "4" || ""));
	});
	it("parses unirest-java/1.3.11 correctly", function() {
		const results = uaParser("unirest-java/1.3.11");
		assert.strictEqual(results.family, "unirest-java", msg("results.family", results.family, "unirest-java"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "11" || "", msg("results.patch", results.patch || "", "11" || ""));
	});
	it("parses axios/0.18.0 correctly", function() {
		const results = uaParser("axios/0.18.0");
		assert.strictEqual(results.family, "axios", msg("results.family", results.family, "axios"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "18", msg("results.minor", results.minor || "", "18"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses got/9.6.0 (https://github.com/sindresorhus/got) correctly", function() {
		const results = uaParser("got/9.6.0 (https://github.com/sindresorhus/got)");
		assert.strictEqual(results.family, "got", msg("results.family", results.family, "got"));
		assert.strictEqual(results.major || "", "9", msg("results.major", results.major || "", "9"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses S3Gof3r correctly", function() {
		const results = uaParser("S3Gof3r");
		assert.strictEqual(results.family, "S3Gof3r", msg("results.family", results.family, "S3Gof3r"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses rusoto/0.36.0 rust/1.35.0 linux correctly", function() {
		const results = uaParser("rusoto/0.36.0 rust/1.35.0 linux");
		assert.strictEqual(results.family, "rusoto", msg("results.family", results.family, "rusoto"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "36", msg("results.minor", results.minor || "", "36"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses ibm-cos-sdk-java/2.3.0 Linux/4.9.0-8-amd64 Java_HotSpot(TM)_64-Bit_Server_VM/9.0.4+11/9.0.4 correctly", function() {
		const results = uaParser("ibm-cos-sdk-java/2.3.0 Linux/4.9.0-8-amd64 Java_HotSpot(TM)_64-Bit_Server_VM/9.0.4+11/9.0.4");
		assert.strictEqual(results.family, "ibm-cos-sdk-java", msg("results.family", results.family, "ibm-cos-sdk-java"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses aws-sdk-dotnet-45/3.3.11.0 aws-sdk-dotnet-core/3.3.17.10 .NET_Runtime/4.0 .NET_Framework/4.0 OS/Microsoft_Windows_NT_6.2.9200.0 ClientSync correctly", function() {
		const results = uaParser("aws-sdk-dotnet-45/3.3.11.0 aws-sdk-dotnet-core/3.3.17.10 .NET_Runtime/4.0 .NET_Framework/4.0 OS/Microsoft_Windows_NT_6.2.9200.0 ClientSync");
		assert.strictEqual(results.family, "aws-sdk-dotnet-45", msg("results.family", results.family, "aws-sdk-dotnet-45"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "11" || "", msg("results.patch", results.patch || "", "11" || ""));
	});
	it("parses Boto/2.48.0 Python/2.7.14 Linux/4.2.0-41-generic correctly", function() {
		const results = uaParser("Boto/2.48.0 Python/2.7.14 Linux/4.2.0-41-generic");
		assert.strictEqual(results.family, "Boto", msg("results.family", results.family, "Boto"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "48", msg("results.minor", results.minor || "", "48"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses aws-cli/1.14.9 Python/2.7.12 Linux/4.9.76-3.78.amzn1.x86_64 botocore/1.8.13 correctly", function() {
		const results = uaParser("aws-cli/1.14.9 Python/2.7.12 Linux/4.9.76-3.78.amzn1.x86_64 botocore/1.8.13");
		assert.strictEqual(results.family, "aws-cli", msg("results.family", results.family, "aws-cli"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "14", msg("results.minor", results.minor || "", "14"));
		assert.equal(results.patch || "", "9" || "", msg("results.patch", results.patch || "", "9" || ""));
	});
	it("parses Boto3/1.6.2 Python/3.4.3 Linux/4.4.35-33.55.amzn1.x86_64 Botocore/1.9.2 Resource correctly", function() {
		const results = uaParser("Boto3/1.6.2 Python/3.4.3 Linux/4.4.35-33.55.amzn1.x86_64 Botocore/1.9.2 Resource");
		assert.strictEqual(results.family, "Boto3", msg("results.family", results.family, "Boto3"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "6", msg("results.minor", results.minor || "", "6"));
		assert.equal(results.patch || "", "2" || "", msg("results.patch", results.patch || "", "2" || ""));
	});
	it("parses ElasticMapReduce/1.0.0 emrfs/s3n {}, aws-sdk-java/1.11.129 Linux/4.4.35-33.55.amzn1.x86_64 OpenJDK_64-Bit_Server_VM/25.141-b16/1.8.0_141 scala/2.11.8 correctly", function() {
		const results = uaParser("ElasticMapReduce/1.0.0 emrfs/s3n {}, aws-sdk-java/1.11.129 Linux/4.4.35-33.55.amzn1.x86_64 OpenJDK_64-Bit_Server_VM/25.141-b16/1.8.0_141 scala/2.11.8");
		assert.strictEqual(results.family, "aws-sdk-java", msg("results.family", results.family, "aws-sdk-java"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "11", msg("results.minor", results.minor || "", "11"));
		assert.equal(results.patch || "", "129" || "", msg("results.patch", results.patch || "", "129" || ""));
	});
	it("parses Hadoop 2.6.0-cdh5.14.0, aws-sdk-java/1.11.134 Linux/4.4.0-1052-aws OpenJDK_64-Bit_Server_VM/25.151-b12/1.8.0_151 correctly", function() {
		const results = uaParser("Hadoop 2.6.0-cdh5.14.0, aws-sdk-java/1.11.134 Linux/4.4.0-1052-aws OpenJDK_64-Bit_Server_VM/25.151-b12/1.8.0_151");
		assert.strictEqual(results.family, "aws-sdk-java", msg("results.family", results.family, "aws-sdk-java"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "11", msg("results.minor", results.minor || "", "11"));
		assert.equal(results.patch || "", "134" || "", msg("results.patch", results.patch || "", "134" || ""));
	});
	it("parses Hadoop 2.8.3-amzn-0, aws-sdk-java/1.11.267 Linux/4.9.77-31.58.amzn1.x86_64 OpenJDK_64-Bit_Server_VM/25.161-b14 java/1.8.0_161 scala/2.11.8 correctly", function() {
		const results = uaParser("Hadoop 2.8.3-amzn-0, aws-sdk-java/1.11.267 Linux/4.9.77-31.58.amzn1.x86_64 OpenJDK_64-Bit_Server_VM/25.161-b14 java/1.8.0_161 scala/2.11.8");
		assert.strictEqual(results.family, "aws-sdk-java", msg("results.family", results.family, "aws-sdk-java"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "11", msg("results.minor", results.minor || "", "11"));
		assert.equal(results.patch || "", "267" || "", msg("results.patch", results.patch || "", "267" || ""));
	});
	it("parses aws-sdk-java/1.11.226 Mac_OS_X/10.12.6 Java_HotSpot(TM)_64-Bit_Server_VM/25.131-b11 java/1.8.0_131 scala/2.11.11 correctly", function() {
		const results = uaParser("aws-sdk-java/1.11.226 Mac_OS_X/10.12.6 Java_HotSpot(TM)_64-Bit_Server_VM/25.131-b11 java/1.8.0_131 scala/2.11.11");
		assert.strictEqual(results.family, "aws-sdk-java", msg("results.family", results.family, "aws-sdk-java"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "11", msg("results.minor", results.minor || "", "11"));
		assert.equal(results.patch || "", "226" || "", msg("results.patch", results.patch || "", "226" || ""));
	});
	it("parses aws-sdk-ruby2/2.2.18 ruby/2.1.5 x86_64-linux correctly", function() {
		const results = uaParser("aws-sdk-ruby2/2.2.18 ruby/2.1.5 x86_64-linux");
		assert.strictEqual(results.family, "aws-sdk-ruby2", msg("results.family", results.family, "aws-sdk-ruby2"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "18" || "", msg("results.patch", results.patch || "", "18" || ""));
	});
	it("parses aws-sdk-cpp/1.0.64 Linux/4.4.0-66-generic x86_64 correctly", function() {
		const results = uaParser("aws-sdk-cpp/1.0.64 Linux/4.4.0-66-generic x86_64");
		assert.strictEqual(results.family, "aws-sdk-cpp", msg("results.family", results.family, "aws-sdk-cpp"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "64" || "", msg("results.patch", results.patch || "", "64" || ""));
	});
	it("parses aws-sdk-go/1.4.12 (go1.6; linux; amd64) S3Manager correctly", function() {
		const results = uaParser("aws-sdk-go/1.4.12 (go1.6; linux; amd64) S3Manager");
		assert.strictEqual(results.family, "aws-sdk-go", msg("results.family", results.family, "aws-sdk-go"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "12" || "", msg("results.patch", results.patch || "", "12" || ""));
	});
	it("parses aws-sdk-nodejs/2.141.0 win32/v8.4.0 correctly", function() {
		const results = uaParser("aws-sdk-nodejs/2.141.0 win32/v8.4.0");
		assert.strictEqual(results.family, "aws-sdk-nodejs", msg("results.family", results.family, "aws-sdk-nodejs"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "141", msg("results.minor", results.minor || "", "141"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses JetS3t/0.9.0 (Linux/4.4.0-1044-aws; amd64; en; JVM 1.8.0_131) correctly", function() {
		const results = uaParser("JetS3t/0.9.0 (Linux/4.4.0-1044-aws; amd64; en; JVM 1.8.0_131)");
		assert.strictEqual(results.family, "JetS3t", msg("results.family", results.family, "JetS3t"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "9", msg("results.minor", results.minor || "", "9"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses s3fs/1.80 (commit hash 6be3236; OpenSSL) correctly", function() {
		const results = uaParser("s3fs/1.80 (commit hash 6be3236; OpenSSL)");
		assert.strictEqual(results.family, "s3fs", msg("results.family", results.family, "s3fs"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "80", msg("results.minor", results.minor || "", "80"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Cyberduck/6.3.0.27105 (Windows 10/10.0) (x86) correctly", function() {
		const results = uaParser("Cyberduck/6.3.0.27105 (Windows 10/10.0) (x86)");
		assert.strictEqual(results.family, "Cyberduck", msg("results.family", results.family, "Cyberduck"));
		assert.strictEqual(results.major || "", "6", msg("results.major", results.major || "", "6"));
		assert.strictEqual(results.minor || "", "3", msg("results.minor", results.minor || "", "3"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses S3 Browser 7-4-5 https://s3browser.com correctly", function() {
		const results = uaParser("S3 Browser 7-4-5 https://s3browser.com");
		assert.strictEqual(results.family, "S3 Browser", msg("results.family", results.family, "S3 Browser"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses rclone/v1.34 correctly", function() {
		const results = uaParser("rclone/v1.34");
		assert.strictEqual(results.family, "rclone", msg("results.family", results.family, "rclone"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "34", msg("results.minor", results.minor || "", "34"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses PycURL/7.43.0 libcurl/7.38.0 OpenSSL/1.0.1t zlib/1.2.8 libidn/1.29 libssh2/1.4.3 librtmp/2.3 correctly", function() {
		const results = uaParser("PycURL/7.43.0 libcurl/7.38.0 OpenSSL/1.0.1t zlib/1.2.8 libidn/1.29 libssh2/1.4.3 librtmp/2.3");
		assert.strictEqual(results.family, "PycURL", msg("results.family", results.family, "PycURL"));
		assert.strictEqual(results.major || "", "7", msg("results.major", results.major || "", "7"));
		assert.strictEqual(results.minor || "", "43", msg("results.minor", results.minor || "", "43"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses Axel 2.4 (Linux) correctly", function() {
		const results = uaParser("Axel 2.4 (Linux)");
		assert.strictEqual(results.family, "Axel", msg("results.family", results.family, "Axel"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "4", msg("results.minor", results.minor || "", "4"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses lftp/4.7.7 correctly", function() {
		const results = uaParser("lftp/4.7.7");
		assert.strictEqual(results.family, "lftp", msg("results.family", results.family, "lftp"));
		assert.strictEqual(results.major || "", "4", msg("results.major", results.major || "", "4"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "7" || "", msg("results.patch", results.patch || "", "7" || ""));
	});
	it("parses aria2/1.19.0 correctly", function() {
		const results = uaParser("aria2/1.19.0");
		assert.strictEqual(results.family, "aria2", msg("results.family", results.family, "aria2"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "19", msg("results.minor", results.minor || "", "19"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses SalesforceMobileSDK/5.3.0 android mobile/7.1.1 (XT1635-02) Salesforce1/15.2 Native uid_bef1747905d064c6 ftr_ Cordova/6.2.3 correctly", function() {
		const results = uaParser("SalesforceMobileSDK/5.3.0 android mobile/7.1.1 (XT1635-02) Salesforce1/15.2 Native uid_bef1747905d064c6 ftr_ Cordova/6.2.3");
		assert.strictEqual(results.family, "Salesforce", msg("results.family", results.family, "Salesforce"));
		assert.strictEqual(results.major || "", "15", msg("results.major", results.major || "", "15"));
		assert.strictEqual(results.minor || "", "2", msg("results.minor", results.minor || "", "2"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses YahooMailProxy; https://help.yahoo.com/kb/yahoo-mail-proxy-SLN28749.html correctly", function() {
		const results = uaParser("YahooMailProxy; https://help.yahoo.com/kb/yahoo-mail-proxy-SLN28749.html");
		assert.strictEqual(results.family, "YahooMailProxy", msg("results.family", results.family, "YahooMailProxy"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ViaFree-DK/3.8.3 (com.MTGx.ViaFree.dk; build:7383; iOS 12.1.0) Alamofire/4.7.0 correctly", function() {
		const results = uaParser("ViaFree-DK/3.8.3 (com.MTGx.ViaFree.dk; build:7383; iOS 12.1.0) Alamofire/4.7.0");
		assert.strictEqual(results.family, "ViaFree", msg("results.family", results.family, "ViaFree"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "8", msg("results.minor", results.minor || "", "8"));
		assert.equal(results.patch || "", "3" || "", msg("results.patch", results.patch || "", "3" || ""));
	});
	it("parses Viafree-tvOS-DK/3.7.1 (com.MTGx.ViaFree.dk; build:7341; tvOS 12.1.0) Alamofire/4.7.0 correctly", function() {
		const results = uaParser("Viafree-tvOS-DK/3.7.1 (com.MTGx.ViaFree.dk; build:7341; tvOS 12.1.0) Alamofire/4.7.0");
		assert.strictEqual(results.family, "ViaFree", msg("results.family", results.family, "ViaFree"));
		assert.strictEqual(results.major || "", "3", msg("results.major", results.major || "", "3"));
		assert.strictEqual(results.minor || "", "7", msg("results.minor", results.minor || "", "7"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses OC/15.0.5071.1000 (Skype for Business) correctly", function() {
		const results = uaParser("OC/15.0.5071.1000 (Skype for Business)");
		assert.strictEqual(results.family, "Skype", msg("results.family", results.family, "Skype"));
		assert.strictEqual(results.major || "", "15", msg("results.major", results.major || "", "15"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "5071" || "", msg("results.patch", results.patch || "", "5071" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 8.1.0; TA-1024 Build/OPR1.170623.026; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/71.0.3578.99 Mobile Safari/537.36 GSA/8.65.5.21.arm64 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 8.1.0; TA-1024 Build/OPR1.170623.026; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/71.0.3578.99 Mobile Safari/537.36 GSA/8.65.5.21.arm64");
		assert.strictEqual(results.family, "Google", msg("results.family", results.family, "Google"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "65", msg("results.minor", results.minor || "", "65"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 8.0.0; SM-G960F Build/R16NW; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/71.0.3578.99 Mobile Safari/537.36 GSA/8.65.5.21.arm64 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 8.0.0; SM-G960F Build/R16NW; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/71.0.3578.99 Mobile Safari/537.36 GSA/8.65.5.21.arm64");
		assert.strictEqual(results.family, "Google", msg("results.family", results.family, "Google"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "65", msg("results.minor", results.minor || "", "65"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Mozilla/5.0 (Linux; Android 9; Pixel 3 Build/PQ1A.181205.006; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/71.0.3578.99 Mobile Safari/537.36 GSA/8.65.5.21.arm64 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Linux; Android 9; Pixel 3 Build/PQ1A.181205.006; wv) AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Chrome/71.0.3578.99 Mobile Safari/537.36 GSA/8.65.5.21.arm64");
		assert.strictEqual(results.family, "Google", msg("results.family", results.family, "Google"));
		assert.strictEqual(results.major || "", "8", msg("results.major", results.major || "", "8"));
		assert.strictEqual(results.minor || "", "65", msg("results.minor", results.minor || "", "65"));
		assert.equal(results.patch || "", "5" || "", msg("results.patch", results.patch || "", "5" || ""));
	});
	it("parses Microsoft Office Word 2014 correctly", function() {
		const results = uaParser("Microsoft Office Word 2014");
		assert.strictEqual(results.family, "Word", msg("results.family", results.family, "Word"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Ghost/2.13.1+moya (https://github.com/TryGhost/Ghost) correctly", function() {
		const results = uaParser("Ghost/2.13.1+moya (https://github.com/TryGhost/Ghost)");
		assert.strictEqual(results.family, "Ghost", msg("results.family", results.family, "Ghost"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "13", msg("results.minor", results.minor || "", "13"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses Ghost/2.10.8 (https://github.com/TryGhost/Ghost) correctly", function() {
		const results = uaParser("Ghost/2.10.8 (https://github.com/TryGhost/Ghost)");
		assert.strictEqual(results.family, "Ghost", msg("results.family", results.family, "Ghost"));
		assert.strictEqual(results.major || "", "2", msg("results.major", results.major || "", "2"));
		assert.strictEqual(results.minor || "", "10", msg("results.minor", results.minor || "", "10"));
		assert.equal(results.patch || "", "8" || "", msg("results.patch", results.patch || "", "8" || ""));
	});
	it("parses Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/75.0.3763.0 Safari/537.36 Edg/75.0.131.0 correctly", function() {
		const results = uaParser("Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/75.0.3763.0 Safari/537.36 Edg/75.0.131.0");
		assert.strictEqual(results.family, "Edge", msg("results.family", results.family, "Edge"));
		assert.strictEqual(results.major || "", "75", msg("results.major", results.major || "", "75"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "131" || "", msg("results.patch", results.patch || "", "131" || ""));
	});
	it("parses Snapchat/10.29.1.0 (iPhone10,1; iOS 11.2; gzip) correctly", function() {
		const results = uaParser("Snapchat/10.29.1.0 (iPhone10,1; iOS 11.2; gzip)");
		assert.strictEqual(results.family, "Snapchat", msg("results.family", results.family, "Snapchat"));
		assert.strictEqual(results.major || "", "10", msg("results.major", results.major || "", "10"));
		assert.strictEqual(results.minor || "", "29", msg("results.minor", results.minor || "", "29"));
		assert.equal(results.patch || "", "1" || "", msg("results.patch", results.patch || "", "1" || ""));
	});
	it("parses OgScrper/1.0.0 correctly", function() {
		const results = uaParser("OgScrper/1.0.0");
		assert.strictEqual(results.family, "OgScrper", msg("results.family", results.family, "OgScrper"));
		assert.strictEqual(results.major || "", "1", msg("results.major", results.major || "", "1"));
		assert.strictEqual(results.minor || "", "0", msg("results.minor", results.minor || "", "0"));
		assert.equal(results.patch || "", "0" || "", msg("results.patch", results.patch || "", "0" || ""));
	});
	it("parses abot/0.1 (abot; http://www.abot.com; abot@abot.com) correctly", function() {
		const results = uaParser("abot/0.1 (abot; http://www.abot.com; abot@abot.com)");
		assert.strictEqual(results.family, "abot", msg("results.family", results.family, "abot"));
		assert.strictEqual(results.major || "", "0", msg("results.major", results.major || "", "0"));
		assert.strictEqual(results.minor || "", "1", msg("results.minor", results.minor || "", "1"));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Ace Explorer correctly", function() {
		const results = uaParser("Ace Explorer");
		assert.strictEqual(results.family, "Other", msg("results.family", results.family, "Other"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ACS-NF/3.0 NEC-c616/001.00 correctly", function() {
		const results = uaParser("ACS-NF/3.0 NEC-c616/001.00");
		assert.strictEqual(results.family, "Other", msg("results.family", results.family, "Other"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ACS-NF/3.0 NEC-e616/001.01 (www.proxomitron.de) correctly", function() {
		const results = uaParser("ACS-NF/3.0 NEC-e616/001.01 (www.proxomitron.de)");
		assert.strictEqual(results.family, "Other", msg("results.family", results.family, "Other"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses ActiveBookmark 1.0 correctly", function() {
		const results = uaParser("ActiveBookmark 1.0");
		assert.strictEqual(results.family, "Other", msg("results.family", results.family, "Other"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || "", "", msg("results.minor", results.minor || "", ""));
		assert.equal(results.patch || "", "" || "", msg("results.patch", results.patch || "", "" || ""));
	});
	it("parses Mozilla/5.0 (compatible; AI) correctly", function() {
		const results = uaParser("Mozilla/5.0 (compatible; AI)");
		assert.strictEqual(results.family, "Other", msg("results.family", results.family, "Other"));
		assert.strictEqual(results.major || "", "", msg("results.major", results.major || "", ""));
		assert.strictEqual(results.minor || ""