// The MIT License (MIT)
//
// Copyright (c) 2017 Firebase
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var cloud_functions_1 = require("../cloud-functions");
/** @internal */
exports.provider = 'firebase.auth';
/** Handle events in the Firebase Auth user lifecycle. */
function user() {
    return new UserBuilder('projects/' + process.env.GCLOUD_PROJECT);
}
exports.user = user;
/** Builder used to create Cloud Functions for Firebase Auth user lifecycle events. */
var UserBuilder = (function () {
    /** @internal */
    function UserBuilder(resource) {
        this.resource = resource;
    }
    UserBuilder.dataConstructor = function (raw) {
        // The UserRecord returned here is an interface. The firebase-admin/auth/user-record module
        // also has a class of the same name, which is one implementation of the interface. Here,
        // because our wire format already almost matches the UserRecord interface, we only use the
        // interface, no need to use the class.
        //
        // The one change we need to make to match the interface is to our incoming timestamps. The
        // interface requires them to be Date objects, while they raw payload has them as strings.
        if (raw.data.metadata) {
            var metadata = raw.data.metadata;
            if (metadata.lastSignedInAt && typeof metadata.lastSignedInAt === 'string') {
                metadata.lastSignedInAt = new Date(metadata.lastSignedInAt);
            }
            if (metadata.createdAt && typeof metadata.createdAt === 'string') {
                metadata.createdAt = new Date(metadata.createdAt);
            }
        }
        return raw.data;
    };
    /** Respond to the creation of a Firebase Auth user. */
    UserBuilder.prototype.onCreate = function (handler) {
        return cloud_functions_1.makeCloudFunction({
            provider: exports.provider, handler: handler,
            resource: this.resource,
            eventType: 'user.create',
            dataConstructor: UserBuilder.dataConstructor,
        });
    };
    /** Respond to the deletion of a Firebase Auth user. */
    UserBuilder.prototype.onDelete = function (handler) {
        return cloud_functions_1.makeCloudFunction({
            provider: exports.provider, handler: handler,
            resource: this.resource,
            eventType: 'user.delete',
            dataConstructor: UserBuilder.dataConstructor,
        });
    };
    return UserBuilder;
}());
exports.UserBuilder = UserBuilder;
