// The MIT License (MIT)
//
// Copyright (c) 2017 Firebase
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.
"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var cloud_functions_1 = require("../cloud-functions");
/** @internal */
exports.provider = 'cloud.pubsub';
/** Handle events on a Cloud Pub/Sub topic. */
function topic(topic) {
    if (topic.indexOf('/') !== -1) {
        throw new Error('Topic name may not have a /');
    }
    return new TopicBuilder("projects/" + process.env.GCLOUD_PROJECT + "/topics/" + topic);
}
exports.topic = topic;
/** Builder used to create Cloud Functions for Google Pub/Sub topics. */
var TopicBuilder = (function () {
    /** @internal */
    function TopicBuilder(resource) {
        this.resource = resource;
    }
    /** Handle a Pub/Sub message that was published to a Cloud Pub/Sub topic */
    TopicBuilder.prototype.onPublish = function (handler) {
        return cloud_functions_1.makeCloudFunction({
            provider: exports.provider, handler: handler,
            resource: this.resource,
            eventType: 'topic.publish',
            dataConstructor: function (raw) { return raw.data instanceof Message ? raw.data : new Message(raw.data); },
        });
    };
    return TopicBuilder;
}());
exports.TopicBuilder = TopicBuilder;
/**
 * A Pub/Sub message.
 *
 * This class has an additional .json helper which will correctly deserialize any
 * message that was a JSON object when published with the JS SDK. .json will throw
 * if the message is not a base64 encoded JSON string.
 */
var Message = (function () {
    function Message(data) {
        _a = [data.data, data.attributes || {}, data.json], this.data = _a[0], this.attributes = _a[1], this._json = _a[2];
        var _a;
    }
    Object.defineProperty(Message.prototype, "json", {
        get: function () {
            if (typeof this._json === 'undefined') {
                this._json = JSON.parse(new Buffer(this.data, 'base64').toString('utf8'));
            }
            return this._json;
        },
        enumerable: true,
        configurable: true
    });
    Message.prototype.toJSON = function () {
        return {
            data: this.data,
            attributes: this.attributes,
        };
    };
    return Message;
}());
exports.Message = Message;
