/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { DataSnapshot } from '../../api/DataSnapshot';
import { DataEvent, CancelEvent, Event } from './Event';
import { Path } from '../util/Path';
import { Change } from './Change';
import { Query } from '../../api/Query';
/**
 * An EventRegistration is basically an event type ('value', 'child_added', etc.) and a callback
 * to be notified of that type of event.
 *
 * That said, it can also contain a cancel callback to be notified if the event is canceled.  And
 * currently, this code is organized around the idea that you would register multiple child_ callbacks
 * together, as a single EventRegistration.  Though currently we don't do that.
 */
export interface EventRegistration {
    /**
     * True if this container has a callback to trigger for this event type
     * @param {!string} eventType
     * @return {boolean}
     */
    respondsTo(eventType: string): boolean;
    /**
     * @param {!Change} change
     * @param {!Query} query
     * @return {!Event}
     */
    createEvent(change: Change, query: Query): Event;
    /**
     * Given event data, return a function to trigger the user's callback
     * @param {!Event} eventData
     * @return {function()}
     */
    getEventRunner(eventData: Event): () => void;
    /**
     * @param {!Error} error
     * @param {!Path} path
     * @return {?CancelEvent}
     */
    createCancelEvent(error: Error, path: Path): CancelEvent | null;
    /**
     * @param {!EventRegistration} other
     * @return {boolean}
     */
    matches(other: EventRegistration): boolean;
    /**
     * False basically means this is a "dummy" callback container being used as a sentinel
     * to remove all callback containers of a particular type.  (e.g. if the user does
     * ref.off('value') without specifying a specific callback).
     *
     * (TODO: Rework this, since it's hacky)
     *
     * @return {boolean}
     */
    hasAnyCallback(): boolean;
}
/**
 * Represents registration for 'value' events.
 */
export declare class ValueEventRegistration implements EventRegistration {
    private callback_;
    private cancelCallback_;
    private context_;
    /**
     * @param {?function(!DataSnapshot)} callback_
     * @param {?function(Error)} cancelCallback_
     * @param {?Object} context_
     */
    constructor(callback_: ((d: DataSnapshot) => void) | null, cancelCallback_: ((e: Error) => void) | null, context_: Object | null);
    /**
     * @inheritDoc
     */
    respondsTo(eventType: string): boolean;
    /**
     * @inheritDoc
     */
    createEvent(change: Change, query: Query): DataEvent;
    /**
     * @inheritDoc
     */
    getEventRunner(eventData: CancelEvent | DataEvent): () => void;
    /**
     * @inheritDoc
     */
    createCancelEvent(error: Error, path: Path): CancelEvent | null;
    /**
     * @inheritDoc
     */
    matches(other: EventRegistration): boolean;
    /**
     * @inheritDoc
     */
    hasAnyCallback(): boolean;
}
/**
 * Represents the registration of 1 or more child_xxx events.
 *
 * Currently, it is always exactly 1 child_xxx event, but the idea is we might let you
 * register a group of callbacks together in the future.
 *
 * @constructor
 * @implements {EventRegistration}
 */
export declare class ChildEventRegistration implements EventRegistration {
    private callbacks_;
    private cancelCallback_;
    private context_?;
    /**
     * @param {?Object.<string, function(!DataSnapshot, ?string=)>} callbacks_
     * @param {?function(Error)} cancelCallback_
     * @param {Object=} context_
     */
    constructor(callbacks_: ({
        [k: string]: (d: DataSnapshot, s?: string | null) => void;
    }) | null, cancelCallback_: ((e: Error) => void) | null, context_?: Object);
    /**
     * @inheritDoc
     */
    respondsTo(eventType: string): boolean;
    /**
     * @inheritDoc
     */
    createCancelEvent(error: Error, path: Path): CancelEvent | null;
    /**
     * @inheritDoc
     */
    createEvent(change: Change, query: Query): DataEvent;
    /**
     * @inheritDoc
     */
    getEventRunner(eventData: CancelEvent | DataEvent): () => void;
    /**
     * @inheritDoc
     */
    matches(other: EventRegistration): boolean;
    /**
     * @inheritDoc
     */
    hasAnyCallback(): boolean;
}
