'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

function _interopDefault (ex) { return (ex && (typeof ex === 'object') && 'default' in ex) ? ex['default'] : ex; }

var firebase = _interopDefault(require('@firebase/app'));
var logger = require('@firebase/logger');
var util = require('util');
var tslib_1 = require('tslib');
var grpc = require('grpc');
var path = require('path');

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/** The semver (www.semver.org) version of the SDK. */
var SDK_VERSION = firebase.SDK_VERSION;

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var logClient = new logger.Logger('@firebase/firestore');
var LogLevel;
(function (LogLevel) {
    LogLevel[LogLevel["DEBUG"] = 0] = "DEBUG";
    LogLevel[LogLevel["ERROR"] = 1] = "ERROR";
    LogLevel[LogLevel["SILENT"] = 2] = "SILENT";
})(LogLevel || (LogLevel = {}));
// Helper methods are needed because variables can't be exported as read/write
function getLogLevel() {
    if (logClient.logLevel === logger.LogLevel.DEBUG) {
        return LogLevel.DEBUG;
    }
    else if (logClient.logLevel === logger.LogLevel.SILENT) {
        return LogLevel.SILENT;
    }
    else {
        return LogLevel.ERROR;
    }
}
function setLogLevel(newLevel) {
    /**
     * Map the new log level to the associated Firebase Log Level
     */
    switch (newLevel) {
        case LogLevel.DEBUG:
            logClient.logLevel = logger.LogLevel.DEBUG;
            break;
        case LogLevel.ERROR:
            logClient.logLevel = logger.LogLevel.ERROR;
            break;
        case LogLevel.SILENT:
            logClient.logLevel = logger.LogLevel.SILENT;
            break;
        default:
            logClient.error("Firestore (" + SDK_VERSION + "): Invalid value passed to `setLogLevel`");
    }
}
function debug(tag, msg) {
    var obj = [];
    for (var _i = 2; _i < arguments.length; _i++) {
        obj[_i - 2] = arguments[_i];
    }
    if (logClient.logLevel <= logger.LogLevel.DEBUG) {
        var args = obj.map(argToString);
        logClient.debug.apply(logClient, ["Firestore (" + SDK_VERSION + ") [" + tag + "]: " + msg].concat(args));
    }
}
function error(msg) {
    var obj = [];
    for (var _i = 1; _i < arguments.length; _i++) {
        obj[_i - 1] = arguments[_i];
    }
    if (logClient.logLevel <= logger.LogLevel.ERROR) {
        var args = obj.map(argToString);
        logClient.error.apply(logClient, ["Firestore (" + SDK_VERSION + "): " + msg].concat(args));
    }
}
/**
 * Converts an additional log parameter to a string representation.
 */
function argToString(obj) {
    if (typeof obj === 'string') {
        return obj;
    }
    else {
        var platform = PlatformSupport.getPlatform();
        try {
            return platform.formatJSON(obj);
        }
        catch (e) {
            // Converting to JSON failed, just log the object directly
            return obj;
        }
    }
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Unconditionally fails, throwing an Error with the given message.
 *
 * Returns any so it can be used in expressions:
 * @example
 * let futureVar = fail('not implemented yet');
 */
function fail(failure) {
    // Log the failure in addition to throw an exception, just in case the
    // exception is swallowed.
    var message = "FIRESTORE (" + SDK_VERSION + ") INTERNAL ASSERTION FAILED: " + failure;
    error(message);
    // NOTE: We don't use FirestoreError here because these are internal failures
    // that cannot be handled by the user. (Also it would create a circular
    // dependency between the error and assert modules which doesn't work.)
    throw new Error(message);
}
/**
 * Fails if the given assertion condition is false, throwing an Error with the
 * given message if it did.
 */
function assert(assertion, message) {
    if (!assertion) {
        fail(message);
    }
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Provides singleton helpers where setup code can inject a platform at runtime.
 * setPlatform needs to be set before Firestore is used and must be set exactly
 * once.
 */
var PlatformSupport = /** @class */ (function () {
    function PlatformSupport() {
    }
    PlatformSupport.setPlatform = function (platform) {
        if (PlatformSupport.platform) {
            fail('Platform already defined');
        }
        PlatformSupport.platform = platform;
    };
    PlatformSupport.getPlatform = function () {
        if (!PlatformSupport.platform) {
            fail('Platform not set');
        }
        return PlatformSupport.platform;
    };
    return PlatformSupport;
}());
/**
 * Returns the representation of an empty "proto" byte string for the
 * platform.
 */
function emptyByteString() {
    return PlatformSupport.getPlatform().emptyByteString;
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// TODO(mcg): Change to a string enum once we've upgraded to typescript 2.4.
//  tslint:disable-next-line:variable-name Intended to look like a TS 2.4 enum
var Code = {
    // Causes are copied from:
    // https://github.com/grpc/grpc/blob/bceec94ea4fc5f0085d81235d8e1c06798dc341a/include/grpc%2B%2B/impl/codegen/status_code_enum.h
    /** Not an error; returned on success. */
    OK: 'ok',
    /** The operation was cancelled (typically by the caller). */
    CANCELLED: 'cancelled',
    /** Unknown error or an error from a different error domain. */
    UNKNOWN: 'unknown',
    /**
     * Client specified an invalid argument. Note that this differs from
     * FAILED_PRECONDITION. INVALID_ARGUMENT indicates arguments that are
     * problematic regardless of the state of the system (e.g., a malformed file
     * name).
     */
    INVALID_ARGUMENT: 'invalid-argument',
    /**
     * Deadline expired before operation could complete. For operations that
     * change the state of the system, this error may be returned even if the
     * operation has completed successfully. For example, a successful response
     * from a server could have been delayed long enough for the deadline to
     * expire.
     */
    DEADLINE_EXCEEDED: 'deadline-exceeded',
    /** Some requested entity (e.g., file or directory) was not found. */
    NOT_FOUND: 'not-found',
    /**
     * Some entity that we attempted to create (e.g., file or directory) already
     * exists.
     */
    ALREADY_EXISTS: 'already-exists',
    /**
     * The caller does not have permission to execute the specified operation.
     * PERMISSION_DENIED must not be used for rejections caused by exhausting
     * some resource (use RESOURCE_EXHAUSTED instead for those errors).
     * PERMISSION_DENIED must not be used if the caller can not be identified
     * (use UNAUTHENTICATED instead for those errors).
     */
    PERMISSION_DENIED: 'permission-denied',
    /**
     * The request does not have valid authentication credentials for the
     * operation.
     */
    UNAUTHENTICATED: 'unauthenticated',
    /**
     * Some resource has been exhausted, perhaps a per-user quota, or perhaps the
     * entire file system is out of space.
     */
    RESOURCE_EXHAUSTED: 'resource-exhausted',
    /**
     * Operation was rejected because the system is not in a state required for
     * the operation's execution. For example, directory to be deleted may be
     * non-empty, an rmdir operation is applied to a non-directory, etc.
     *
     * A litmus test that may help a service implementor in deciding
     * between FAILED_PRECONDITION, ABORTED, and UNAVAILABLE:
     *  (a) Use UNAVAILABLE if the client can retry just the failing call.
     *  (b) Use ABORTED if the client should retry at a higher-level
     *      (e.g., restarting a read-modify-write sequence).
     *  (c) Use FAILED_PRECONDITION if the client should not retry until
     *      the system state has been explicitly fixed. E.g., if an "rmdir"
     *      fails because the directory is non-empty, FAILED_PRECONDITION
     *      should be returned since the client should not retry unless
     *      they have first fixed up the directory by deleting files from it.
     *  (d) Use FAILED_PRECONDITION if the client performs conditional
     *      REST Get/Update/Delete on a resource and the resource on the
     *      server does not match the condition. E.g., conflicting
     *      read-modify-write on the same resource.
     */
    FAILED_PRECONDITION: 'failed-precondition',
    /**
     * The operation was aborted, typically due to a concurrency issue like
     * sequencer check failures, transaction aborts, etc.
     *
     * See litmus test above for deciding between FAILED_PRECONDITION, ABORTED,
     * and UNAVAILABLE.
     */
    ABORTED: 'aborted',
    /**
     * Operation was attempted past the valid range. E.g., seeking or reading
     * past end of file.
     *
     * Unlike INVALID_ARGUMENT, this error indicates a problem that may be fixed
     * if the system state changes. For example, a 32-bit file system will
     * generate INVALID_ARGUMENT if asked to read at an offset that is not in the
     * range [0,2^32-1], but it will generate OUT_OF_RANGE if asked to read from
     * an offset past the current file size.
     *
     * There is a fair bit of overlap between FAILED_PRECONDITION and
     * OUT_OF_RANGE. We recommend using OUT_OF_RANGE (the more specific error)
     * when it applies so that callers who are iterating through a space can
     * easily look for an OUT_OF_RANGE error to detect when they are done.
     */
    OUT_OF_RANGE: 'out-of-range',
    /** Operation is not implemented or not supported/enabled in this service. */
    UNIMPLEMENTED: 'unimplemented',
    /**
     * Internal errors. Means some invariants expected by underlying System has
     * been broken. If you see one of these errors, Something is very broken.
     */
    INTERNAL: 'internal',
    /**
     * The service is currently unavailable. This is a most likely a transient
     * condition and may be corrected by retrying with a backoff.
     *
     * See litmus test above for deciding between FAILED_PRECONDITION, ABORTED,
     * and UNAVAILABLE.
     */
    UNAVAILABLE: 'unavailable',
    /** Unrecoverable data loss or corruption. */
    DATA_LOSS: 'data-loss'
};
/**
 * An error class used for Firestore-generated errors. Ideally we should be
 * using FirebaseError, but integrating with it is overly arduous at the moment,
 * so we define our own compatible error class (with a `name` of 'FirebaseError'
 * and compatible `code` and `message` fields.)
 */
var FirestoreError = /** @class */ (function (_super) {
    tslib_1.__extends(FirestoreError, _super);
    function FirestoreError(code, message) {
        var _this = _super.call(this, message) || this;
        _this.code = code;
        _this.message = message;
        _this.name = 'FirebaseError';
        // HACK: We write a toString property directly because Error is not a real
        // class and so inheritance does not work correctly. We could alternatively
        // do the same "back-door inheritance" trick that FirebaseError does.
        _this.toString = function () { return _this.name + ": [code=" + _this.code + "]: " + _this.message; };
        return _this;
    }
    return FirestoreError;
}(Error));

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Helper function to prevent instantiation through the constructor.
 *
 * This method creates a new constructor that throws when it's invoked.
 * The prototype of that constructor is then set to the prototype of the hidden
 * "class" to expose all the prototype methods and allow for instanceof
 * checks.
 *
 * To also make all the static methods available, all properties of the
 * original constructor are copied to the new constructor.
 */
function makeConstructorPrivate(cls, optionalMessage) {
    function PublicConstructor() {
        var error = 'This constructor is private.';
        if (optionalMessage) {
            error += ' ';
            error += optionalMessage;
        }
        throw new FirestoreError(Code.INVALID_ARGUMENT, error);
    }
    // Make sure instanceof checks work and all methods are exposed on the public
    // constructor
    PublicConstructor.prototype = cls.prototype;
    // Copy any static methods/members
    for (var staticProperty in cls) {
        if (cls.hasOwnProperty(staticProperty)) {
            PublicConstructor[staticProperty] = cls[staticProperty];
        }
    }
    return PublicConstructor;
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
function contains(obj, key) {
    return Object.prototype.hasOwnProperty.call(obj, key);
}
/** Returns the given value if it's defined or the defaultValue otherwise. */
function defaulted(value, defaultValue) {
    return value !== undefined ? value : defaultValue;
}
function forEachNumber(obj, fn) {
    for (var key in obj) {
        if (Object.prototype.hasOwnProperty.call(obj, key)) {
            var num = Number(key);
            if (!isNaN(num)) {
                fn(num, obj[key]);
            }
        }
    }
}
function values(obj) {
    var vs = [];
    forEach(obj, function (_, v) { return vs.push(v); });
    return vs;
}
function forEach(obj, fn) {
    for (var key in obj) {
        if (Object.prototype.hasOwnProperty.call(obj, key)) {
            fn(key, obj[key]);
        }
    }
}
function isEmpty(obj) {
    assert(obj != null && typeof obj === 'object', 'isEmpty() expects object parameter.');
    for (var key in obj) {
        if (Object.prototype.hasOwnProperty.call(obj, key)) {
            return false;
        }
    }
    return true;
}
function shallowCopy(obj) {
    assert(obj && typeof obj === 'object', 'shallowCopy() expects object parameter.');
    var result = {};
    for (var key in obj) {
        if (Object.prototype.hasOwnProperty.call(obj, key)) {
            result[key] = obj[key];
        }
    }
    return result;
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Validates that no arguments were passed in the invocation of functionName.
 *
 * Forward the magic "arguments" variable as second parameter on which the
 * parameter validation is performed:
 * validateNoArgs('myFunction', arguments);
 */
function validateNoArgs(functionName, args) {
    if (args.length !== 0) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() does not support arguments, " +
            'but was called with ' +
            formatPlural(args.length, 'argument') +
            '.');
    }
}
/**
 * Validates the invocation of functionName has the exact number of arguments.
 *
 * Forward the magic "arguments" variable as second parameter on which the
 * parameter validation is performed:
 * validateExactNumberOfArgs('myFunction', arguments, 2);
 */
function validateExactNumberOfArgs(functionName, args, numberOfArgs) {
    if (args.length !== numberOfArgs) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires " +
            formatPlural(numberOfArgs, 'argument') +
            ', but was called with ' +
            formatPlural(args.length, 'argument') +
            '.');
    }
}
/**
 * Validates the invocation of functionName has at least the provided number of
 * arguments (but can have many more).
 *
 * Forward the magic "arguments" variable as second parameter on which the
 * parameter validation is performed:
 * validateAtLeastNumberOfArgs('myFunction', arguments, 2);
 */
function validateAtLeastNumberOfArgs(functionName, args, minNumberOfArgs) {
    if (args.length < minNumberOfArgs) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires at least " +
            formatPlural(minNumberOfArgs, 'argument') +
            ', but was called with ' +
            formatPlural(args.length, 'argument') +
            '.');
    }
}
/**
 * Validates the invocation of functionName has number of arguments between
 * the values provided.
 *
 * Forward the magic "arguments" variable as second parameter on which the
 * parameter validation is performed:
 * validateBetweenNumberOfArgs('myFunction', arguments, 2, 3);
 */
function validateBetweenNumberOfArgs(functionName, args, minNumberOfArgs, maxNumberOfArgs) {
    if (args.length < minNumberOfArgs || args.length > maxNumberOfArgs) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires between " + minNumberOfArgs + " and " +
            (maxNumberOfArgs + " arguments, but was called with ") +
            formatPlural(args.length, 'argument') +
            '.');
    }
}
/**
 * Validates the provided argument is an array and has as least the expected
 * number of elements.
 */
function validateNamedArrayAtLeastNumberOfElements(functionName, value, name, minNumberOfElements) {
    if (!(value instanceof Array) || value.length < minNumberOfElements) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires its " + name + " argument to be an " +
            'array with at least ' +
            (formatPlural(minNumberOfElements, 'element') + "."));
    }
}
/**
 * Validates the provided positional argument has the native JavaScript type
 * using typeof checks.
 */
function validateArgType(functionName, type, position, argument) {
    validateType(functionName, type, ordinal(position) + " argument", argument);
}
/**
 * Validates the provided argument has the native JavaScript type using
 * typeof checks or is undefined.
 */
function validateOptionalArgType(functionName, type, position, argument) {
    if (argument !== undefined) {
        validateArgType(functionName, type, position, argument);
    }
}
/**
 * Validates the provided named option has the native JavaScript type using
 * typeof checks.
 */
function validateNamedType(functionName, type, optionName, argument) {
    validateType(functionName, type, optionName + " option", argument);
}
/**
 * Validates the provided named option has the native JavaScript type using
 * typeof checks or is undefined.
 */
function validateNamedOptionalType(functionName, type, optionName, argument) {
    if (argument !== undefined) {
        validateNamedType(functionName, type, optionName, argument);
    }
}
function validateArrayElements(functionName, optionName, typeDescription, argument, validator) {
    if (!(argument instanceof Array)) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires its " + optionName + " " +
            ("option to be an array, but it was: " + valueDescription(argument)));
    }
    for (var i = 0; i < argument.length; ++i) {
        if (!validator(argument[i])) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires all " + optionName + " " +
                ("elements to be " + typeDescription + ", but the value at index " + i + " ") +
                ("was: " + valueDescription(argument[i])));
        }
    }
}
function validateOptionalArrayElements(functionName, optionName, typeDescription, argument, validator) {
    if (argument !== undefined) {
        validateArrayElements(functionName, optionName, typeDescription, argument, validator);
    }
}
/**
 * Validates that the provided named option equals one of the expected values.
 */
function validateNamedPropertyEquals(functionName, inputName, optionName, input, expected) {
    var expectedDescription = [];
    for (var _i = 0, expected_1 = expected; _i < expected_1.length; _i++) {
        var val = expected_1[_i];
        if (val === input) {
            return;
        }
        expectedDescription.push(valueDescription(val));
    }
    var actualDescription = valueDescription(input);
    throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid value " + actualDescription + " provided to function " + functionName + "() for option " +
        ("\"" + optionName + "\". Acceptable values: " + expectedDescription.join(', ')));
}
/**
 * Validates that the provided named option equals one of the expected values or
 * is undefined.
 */
function validateNamedOptionalPropertyEquals(functionName, inputName, optionName, input, expected) {
    if (input !== undefined) {
        validateNamedPropertyEquals(functionName, inputName, optionName, input, expected);
    }
}
/** Helper to validate the type of a provided input. */
function validateType(functionName, type, inputName, input) {
    var valid = false;
    if (type === 'object') {
        valid = isPlainObject(input);
    }
    else if (type === 'non-empty string') {
        valid = typeof input === 'string' && input !== '';
    }
    else {
        valid = typeof input === type;
    }
    if (!valid) {
        var description = valueDescription(input);
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires its " + inputName + " " +
            ("to be of type " + type + ", but it was: " + description));
    }
}
/**
 * Returns true if it's a non-null object without a custom prototype
 * (i.e. excludes Array, Date, etc.).
 */
function isPlainObject(input) {
    return (typeof input === 'object' &&
        input !== null &&
        (Object.getPrototypeOf(input) === Object.prototype ||
            Object.getPrototypeOf(input) === null));
}
/** Returns a string describing the type / value of the provided input. */
function valueDescription(input) {
    if (input === undefined) {
        return 'undefined';
    }
    else if (input === null) {
        return 'null';
    }
    else if (typeof input === 'string') {
        if (input.length > 20) {
            input = input.substring(0, 20) + "...";
        }
        return JSON.stringify(input);
    }
    else if (typeof input === 'number' || typeof input === 'boolean') {
        return '' + input;
    }
    else if (typeof input === 'object') {
        if (input instanceof Array) {
            return 'an array';
        }
        else {
            var customObjectName = tryGetCustomObjectType(input);
            if (customObjectName) {
                return "a custom " + customObjectName + " object";
            }
            else {
                return 'an object';
            }
        }
    }
    else if (typeof input === 'function') {
        return 'a function';
    }
    else {
        return fail('Unknown wrong type: ' + typeof input);
    }
}
/** Hacky method to try to get the constructor name for an object. */
function tryGetCustomObjectType(input) {
    if (input.constructor) {
        var funcNameRegex = /function\s+([^\s(]+)\s*\(/;
        var results = funcNameRegex.exec(input.constructor.toString());
        if (results && results.length > 1) {
            return results[1];
        }
    }
    return null;
}
/** Validates the provided argument is defined. */
function validateDefined(functionName, position, argument) {
    if (argument === undefined) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires a valid " + ordinal(position) + " " +
            "argument, but it was undefined.");
    }
}
/**
 * Validates the provided positional argument is an object, and its keys and
 * values match the expected keys and types provided in optionTypes.
 */
function validateOptionNames(functionName, options, optionNames) {
    forEach(options, function (key, _) {
        if (optionNames.indexOf(key) < 0) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Unknown option '" + key + "' passed to function " + functionName + "(). " +
                'Available options: ' +
                optionNames.join(', '));
        }
    });
}
/**
 * Helper method to throw an error that the provided argument did not pass
 * an instanceof check.
 */
function invalidClassError(functionName, type, position, argument) {
    var description = valueDescription(argument);
    return new FirestoreError(Code.INVALID_ARGUMENT, "Function " + functionName + "() requires its " + ordinal(position) + " " +
        ("argument to be a " + type + ", but it was: " + description));
}
/** Converts a number to its english word representation */
function ordinal(num) {
    switch (num) {
        case 1:
            return 'first';
        case 2:
            return 'second';
        case 3:
            return 'third';
        default:
            return num + 'th';
    }
}
/**
 * Formats the given word as plural conditionally given the preceding number.
 */
function formatPlural(num, str) {
    return num + " " + str + (num === 1 ? '' : 's');
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// tslint:disable-next-line:class-as-namespace
var AutoId = /** @class */ (function () {
    function AutoId() {
    }
    AutoId.newId = function () {
        // Alphanumeric characters
        var chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
        var autoId = '';
        for (var i = 0; i < 20; i++) {
            autoId += chars.charAt(Math.floor(Math.random() * chars.length));
        }
        assert(autoId.length === 20, 'Invalid auto ID: ' + autoId);
        return autoId;
    };
    return AutoId;
}());
function primitiveComparator(left, right) {
    if (left < right)
        return -1;
    if (left > right)
        return 1;
    return 0;
}
/** Helper to compare nullable (or undefined-able) objects using isEqual(). */
function equals(left, right) {
    if (left !== null && left !== undefined) {
        return !!(right && left.isEqual(right));
    }
    else {
        // HACK: Explicitly cast since TypeScript's type narrowing apparently isn't
        // smart enough.
        return left === right;
    }
}
/** Helper to compare arrays using isEqual(). */
function arrayEquals(left, right) {
    if (left.length !== right.length) {
        return false;
    }
    for (var i = 0; i < left.length; i++) {
        if (!left[i].isEqual(right[i])) {
            return false;
        }
    }
    return true;
}
/**
 * Returns the immediate lexicographically-following string. This is useful to
 * construct an inclusive range for indexeddb iterators.
 */
function immediateSuccessor(s) {
    // Return the input string, with an additional NUL byte appended.
    return s + '\0';
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/** Helper function to assert Uint8Array is available at runtime. */
function assertUint8ArrayAvailable() {
    if (typeof Uint8Array === 'undefined') {
        throw new FirestoreError(Code.UNIMPLEMENTED, 'Uint8Arrays are not available in this environment.');
    }
}
/** Helper function to assert Base64 functions are available at runtime. */
function assertBase64Available() {
    if (!PlatformSupport.getPlatform().base64Available) {
        throw new FirestoreError(Code.UNIMPLEMENTED, 'Blobs are unavailable in Firestore in this environment.');
    }
}
/**
 * Immutable class holding a blob (binary data).
 * This class is directly exposed in the public API.
 *
 * Note that while you can't hide the constructor in JavaScript code, we are
 * using the hack above to make sure no-one outside this module can call it.
 */
var Blob = /** @class */ (function () {
    function Blob(binaryString) {
        assertBase64Available();
        this._binaryString = binaryString;
    }
    Blob.fromBase64String = function (base64) {
        validateExactNumberOfArgs('Blob.fromBase64String', arguments, 1);
        validateArgType('Blob.fromBase64String', 'string', 1, base64);
        assertBase64Available();
        try {
            var binaryString = PlatformSupport.getPlatform().atob(base64);
            return new Blob(binaryString);
        }
        catch (e) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Failed to construct Blob from Base64 string: ' + e);
        }
    };
    Blob.fromUint8Array = function (array) {
        validateExactNumberOfArgs('Blob.fromUint8Array', arguments, 1);
        assertUint8ArrayAvailable();
        if (!(array instanceof Uint8Array)) {
            throw invalidClassError('Blob.fromUint8Array', 'Uint8Array', 1, array);
        }
        // We can't call array.map directly because it expects the return type to
        // be a Uint8Array, whereas we can convert it to a regular array by invoking
        // map on the Array prototype.
        var binaryString = Array.prototype.map
            .call(array, function (char) {
            return String.fromCharCode(char);
        })
            .join('');
        return new Blob(binaryString);
    };
    Blob.prototype.toBase64 = function () {
        validateExactNumberOfArgs('Blob.toBase64', arguments, 0);
        assertBase64Available();
        return PlatformSupport.getPlatform().btoa(this._binaryString);
    };
    Blob.prototype.toUint8Array = function () {
        validateExactNumberOfArgs('Blob.toUint8Array', arguments, 0);
        assertUint8ArrayAvailable();
        var buffer = new Uint8Array(this._binaryString.length);
        for (var i = 0; i < this._binaryString.length; i++) {
            buffer[i] = this._binaryString.charCodeAt(i);
        }
        return buffer;
    };
    Blob.prototype.toString = function () {
        return 'Blob(base64: ' + this.toBase64() + ')';
    };
    Blob.prototype.isEqual = function (other) {
        return this._binaryString === other._binaryString;
    };
    /**
     * Actually private to JS consumers of our API, so this function is prefixed
     * with an underscore.
     */
    Blob.prototype._compareTo = function (other) {
        return primitiveComparator(this._binaryString, other._binaryString);
    };
    return Blob;
}());
// Public instance that disallows construction at runtime. This constructor is
// used when exporting Blob on firebase.firestore.Blob and will be called Blob
// publicly. Internally we still use Blob which has a type checked private
// constructor. Note that Blob and PublicBlob can be used interchangeably in
// instanceof checks.
// For our internal TypeScript code PublicBlob doesn't exist as a type, and so
// we need to use Blob as type and export it too.
// tslint:disable-next-line:variable-name We're treating this as a class name.
var PublicBlob = makeConstructorPrivate(Blob, 'Use Blob.fromUint8Array() or Blob.fromBase64String() instead.');

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Immutable class representing a geo point as latitude-longitude pair.
 * This class is directly exposed in the public API, including its constructor.
 */
var GeoPoint = /** @class */ (function () {
    function GeoPoint(latitude, longitude) {
        validateExactNumberOfArgs('GeoPoint', arguments, 2);
        validateArgType('GeoPoint', 'number', 1, latitude);
        validateArgType('GeoPoint', 'number', 2, longitude);
        if (!isFinite(latitude) || latitude < -90 || latitude > 90) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Latitude must be a number between -90 and 90, but was: ' + latitude);
        }
        if (!isFinite(longitude) || longitude < -180 || longitude > 180) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Longitude must be a number between -180 and 180, but was: ' + longitude);
        }
        this._lat = latitude;
        this._long = longitude;
    }
    Object.defineProperty(GeoPoint.prototype, "latitude", {
        /**
         * Returns the latitude of this geo point, a number between -90 and 90.
         */
        get: function () {
            return this._lat;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(GeoPoint.prototype, "longitude", {
        /**
         * Returns the longitude of this geo point, a number between -180 and 180.
         */
        get: function () {
            return this._long;
        },
        enumerable: true,
        configurable: true
    });
    GeoPoint.prototype.isEqual = function (other) {
        return this._lat === other._lat && this._long === other._long;
    };
    /**
     * Actually private to JS consumers of our API, so this function is prefixed
     * with an underscore.
     */
    GeoPoint.prototype._compareTo = function (other) {
        return (primitiveComparator(this._lat, other._lat) ||
            primitiveComparator(this._long, other._long));
    };
    return GeoPoint;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var Timestamp = /** @class */ (function () {
    function Timestamp(seconds, nanoseconds) {
        this.seconds = seconds;
        this.nanoseconds = nanoseconds;
        if (nanoseconds < 0) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Timestamp nanoseconds out of range: ' + nanoseconds);
        }
        if (nanoseconds >= 1e9) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Timestamp nanoseconds out of range: ' + nanoseconds);
        }
        // Midnight at the beginning of 1/1/1 is the earliest Firestore supports.
        if (seconds < -62135596800) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Timestamp seconds out of range: ' + seconds);
        }
        // This will break in the year 10,000.
        if (seconds >= 253402300800) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Timestamp seconds out of range: ' + seconds);
        }
    }
    Timestamp.now = function () {
        return Timestamp.fromMillis(Date.now());
    };
    Timestamp.fromDate = function (date) {
        return Timestamp.fromMillis(date.getTime());
    };
    Timestamp.fromMillis = function (milliseconds) {
        var seconds = Math.floor(milliseconds / 1000);
        var nanos = (milliseconds - seconds * 1000) * 1e6;
        return new Timestamp(seconds, nanos);
    };
    Timestamp.prototype.toDate = function () {
        return new Date(this.toMillis());
    };
    Timestamp.prototype.toMillis = function () {
        return this.seconds * 1000 + this.nanoseconds / 1e6;
    };
    Timestamp.prototype._compareTo = function (other) {
        if (this.seconds === other.seconds) {
            return primitiveComparator(this.nanoseconds, other.nanoseconds);
        }
        return primitiveComparator(this.seconds, other.seconds);
    };
    Timestamp.prototype.isEqual = function (other) {
        return (other.seconds === this.seconds && other.nanoseconds === this.nanoseconds);
    };
    Timestamp.prototype.toString = function () {
        return ('Timestamp(seconds=' +
            this.seconds +
            ', nanoseconds=' +
            this.nanoseconds +
            ')');
    };
    return Timestamp;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var DatabaseInfo = /** @class */ (function () {
    /**
     * Constructs a DatabaseInfo using the provided host, databaseId and
     * persistenceKey.
     *
     * @param databaseId The database to use.
     * @param persistenceKey A unique identifier for this Firestore's local
     * storage (used in conjunction with the databaseId).
     * @param host The Firestore backend host to connect to.
     * @param ssl Whether to use SSL when connecting.
     */
    function DatabaseInfo(databaseId, persistenceKey, host, ssl) {
        this.databaseId = databaseId;
        this.persistenceKey = persistenceKey;
        this.host = host;
        this.ssl = ssl;
    }
    return DatabaseInfo;
}());
/** The default database name for a project. */
var DEFAULT_DATABASE_NAME = '(default)';
/** Represents the database ID a Firestore client is associated with. */
var DatabaseId = /** @class */ (function () {
    function DatabaseId(projectId, database) {
        this.projectId = projectId;
        this.database = database ? database : DEFAULT_DATABASE_NAME;
    }
    Object.defineProperty(DatabaseId.prototype, "isDefaultDatabase", {
        get: function () {
            return this.database === DEFAULT_DATABASE_NAME;
        },
        enumerable: true,
        configurable: true
    });
    DatabaseId.prototype.isEqual = function (other) {
        return (other instanceof DatabaseId &&
            other.projectId === this.projectId &&
            other.database === this.database);
    };
    DatabaseId.prototype.compareTo = function (other) {
        return (primitiveComparator(this.projectId, other.projectId) ||
            primitiveComparator(this.database, other.database));
    };
    return DatabaseId;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var DOCUMENT_KEY_NAME = '__name__';
/**
 * Path represents an ordered sequence of string segments.
 */
var Path = /** @class */ (function () {
    function Path(segments, offset, length) {
        this.init(segments, offset, length);
    }
    /**
     * An initialization method that can be called from outside the constructor.
     * We need this so that we can have a non-static construct method that returns
     * the polymorphic `this` type.
     */
    Path.prototype.init = function (segments, offset, length) {
        if (offset === undefined) {
            offset = 0;
        }
        else if (offset > segments.length) {
            fail('offset ' + offset + ' out of range ' + segments.length);
        }
        if (length === undefined) {
            length = segments.length - offset;
        }
        else if (length > segments.length - offset) {
            fail('length ' + length + ' out of range ' + (segments.length - offset));
        }
        this.segments = segments;
        this.offset = offset;
        this.len = length;
    };
    /**
     * Constructs a new instance of Path using the same concrete type as `this`.
     * We need this instead of using the normal constructor, because polymorphic
     * `this` doesn't work on static methods.
     */
    Path.prototype.construct = function (segments, offset, length) {
        var path = Object.create(Object.getPrototypeOf(this));
        path.init(segments, offset, length);
        return path;
    };
    Object.defineProperty(Path.prototype, "length", {
        get: function () {
            return this.len;
        },
        enumerable: true,
        configurable: true
    });
    Path.prototype.isEqual = function (other) {
        return Path.comparator(this, other) === 0;
    };
    Path.prototype.child = function (nameOrPath) {
        var segments = this.segments.slice(this.offset, this.limit());
        if (nameOrPath instanceof Path) {
            nameOrPath.forEach(function (segment) {
                segments.push(segment);
            });
        }
        else if (typeof nameOrPath === 'string') {
            segments.push(nameOrPath);
        }
        else {
            fail('Unknown parameter type for Path.child(): ' + nameOrPath);
        }
        return this.construct(segments);
    };
    /** The index of one past the last segment of the path. */
    Path.prototype.limit = function () {
        return this.offset + this.length;
    };
    Path.prototype.popFirst = function (size) {
        size = size === undefined ? 1 : size;
        assert(this.length >= size, "Can't call popFirst() with less segments");
        return this.construct(this.segments, this.offset + size, this.length - size);
    };
    Path.prototype.popLast = function () {
        assert(!this.isEmpty(), "Can't call popLast() on empty path");
        return this.construct(this.segments, this.offset, this.length - 1);
    };
    Path.prototype.firstSegment = function () {
        assert(!this.isEmpty(), "Can't call firstSegment() on empty path");
        return this.segments[this.offset];
    };
    Path.prototype.lastSegment = function () {
        return this.get(this.length - 1);
    };
    Path.prototype.get = function (index) {
        assert(index < this.length, 'Index out of range');
        return this.segments[this.offset + index];
    };
    Path.prototype.isEmpty = function () {
        return this.length === 0;
    };
    Path.prototype.isPrefixOf = function (other) {
        if (other.length < this.length) {
            return false;
        }
        for (var i = 0; i < this.length; i++) {
            if (this.get(i) !== other.get(i)) {
                return false;
            }
        }
        return true;
    };
    Path.prototype.isImmediateParentOf = function (potentialChild) {
        if (this.length + 1 !== potentialChild.length) {
            return false;
        }
        for (var i = 0; i < this.length; i++) {
            if (this.get(i) !== potentialChild.get(i)) {
                return false;
            }
        }
        return true;
    };
    Path.prototype.forEach = function (fn) {
        for (var i = this.offset, end = this.limit(); i < end; i++) {
            fn(this.segments[i]);
        }
    };
    Path.prototype.toArray = function () {
        return this.segments.slice(this.offset, this.limit());
    };
    Path.comparator = function (p1, p2) {
        var len = Math.min(p1.length, p2.length);
        for (var i = 0; i < len; i++) {
            var left = p1.get(i);
            var right = p2.get(i);
            if (left < right)
                return -1;
            if (left > right)
                return 1;
        }
        if (p1.length < p2.length)
            return -1;
        if (p1.length > p2.length)
            return 1;
        return 0;
    };
    return Path;
}());
/**
 * A slash-separated path for navigating resources (documents and collections)
 * within Firestore.
 */
var ResourcePath = /** @class */ (function (_super) {
    tslib_1.__extends(ResourcePath, _super);
    function ResourcePath() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    ResourcePath.prototype.canonicalString = function () {
        // NOTE: The client is ignorant of any path segments containing escape
        // sequences (e.g. __id123__) and just passes them through raw (they exist
        // for legacy reasons and should not be used frequently).
        return this.toArray().join('/');
    };
    ResourcePath.prototype.toString = function () {
        return this.canonicalString();
    };
    /**
     * Creates a resource path from the given slash-delimited string.
     */
    ResourcePath.fromString = function (path) {
        // NOTE: The client is ignorant of any path segments containing escape
        // sequences (e.g. __id123__) and just passes them through raw (they exist
        // for legacy reasons and should not be used frequently).
        if (path.indexOf('//') >= 0) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid path (" + path + "). Paths must not contain // in them.");
        }
        // We may still have an empty segment at the beginning or end if they had a
        // leading or trailing slash (which we allow).
        var segments = path.split('/').filter(function (segment) { return segment.length > 0; });
        return new ResourcePath(segments);
    };
    ResourcePath.EMPTY_PATH = new ResourcePath([]);
    return ResourcePath;
}(Path));
var identifierRegExp = /^[_a-zA-Z][_a-zA-Z0-9]*$/;
/** A dot-separated path for navigating sub-objects within a document. */
var FieldPath = /** @class */ (function (_super) {
    tslib_1.__extends(FieldPath, _super);
    function FieldPath() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    /**
     * Returns true if the string could be used as a segment in a field path
     * without escaping.
     */
    FieldPath.isValidIdentifier = function (segment) {
        return identifierRegExp.test(segment);
    };
    FieldPath.prototype.canonicalString = function () {
        return this.toArray()
            .map(function (str) {
            str = str.replace('\\', '\\\\').replace('`', '\\`');
            if (!FieldPath.isValidIdentifier(str)) {
                str = '`' + str + '`';
            }
            return str;
        })
            .join('.');
    };
    FieldPath.prototype.toString = function () {
        return this.canonicalString();
    };
    /**
     * Returns true if this field references the key of a document.
     */
    FieldPath.prototype.isKeyField = function () {
        return this.length === 1 && this.get(0) === DOCUMENT_KEY_NAME;
    };
    /**
     * The field designating the key of a document.
     */
    FieldPath.keyField = function () {
        return new FieldPath([DOCUMENT_KEY_NAME]);
    };
    /**
     * Parses a field string from the given server-formatted string.
     *
     * - Splitting the empty string is not allowed (for now at least).
     * - Empty segments within the string (e.g. if there are two consecutive
     *   separators) are not allowed.
     *
     * TODO(b/37244157): we should make this more strict. Right now, it allows
     * non-identifier path components, even if they aren't escaped.
     */
    FieldPath.fromServerFormat = function (path) {
        var segments = [];
        var current = '';
        var i = 0;
        var addCurrentSegment = function () {
            if (current.length === 0) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid field path (" + path + "). Paths must not be empty, begin " +
                    "with '.', end with '.', or contain '..'");
            }
            segments.push(current);
            current = '';
        };
        var inBackticks = false;
        while (i < path.length) {
            var c = path[i];
            if (c === '\\') {
                if (i + 1 === path.length) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, 'Path has trailing escape character: ' + path);
                }
                var next = path[i + 1];
                if (!(next === '\\' || next === '.' || next === '`')) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, 'Path has invalid escape sequence: ' + path);
                }
                current += next;
                i += 2;
            }
            else if (c === '`') {
                inBackticks = !inBackticks;
                i++;
            }
            else if (c === '.' && !inBackticks) {
                addCurrentSegment();
                i++;
            }
            else {
                current += c;
                i++;
            }
        }
        addCurrentSegment();
        if (inBackticks) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Unterminated ` in path: ' + path);
        }
        return new FieldPath(segments);
    };
    FieldPath.EMPTY_PATH = new FieldPath([]);
    return FieldPath;
}(Path));

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var DocumentKey = /** @class */ (function () {
    function DocumentKey(path) {
        this.path = path;
        assert(DocumentKey.isDocumentKey(path), 'Invalid DocumentKey with an odd number of segments: ' +
            path.toArray().join('/'));
    }
    /** Returns true if the document is in the specified collectionId. */
    DocumentKey.prototype.hasCollectionId = function (collectionId) {
        return (this.path.length >= 2 &&
            this.path.get(this.path.length - 2) === collectionId);
    };
    DocumentKey.prototype.isEqual = function (other) {
        return (other !== null && ResourcePath.comparator(this.path, other.path) === 0);
    };
    DocumentKey.prototype.toString = function () {
        return this.path.toString();
    };
    DocumentKey.comparator = function (k1, k2) {
        return ResourcePath.comparator(k1.path, k2.path);
    };
    DocumentKey.isDocumentKey = function (path) {
        return path.length % 2 === 0;
    };
    /**
     * Creates and returns a new document key with the given segments.
     *
     * @param path The segments of the path to the document
     * @return A new instance of DocumentKey
     */
    DocumentKey.fromSegments = function (segments) {
        return new DocumentKey(new ResourcePath(segments.slice()));
    };
    /**
     * Creates and returns a new document key using '/' to split the string into
     * segments.
     *
     * @param path The slash-separated path string to the document
     * @return A new instance of DocumentKey
     */
    DocumentKey.fromPathString = function (path) {
        return new DocumentKey(ResourcePath.fromString(path));
    };
    DocumentKey.EMPTY = new DocumentKey(new ResourcePath([]));
    return DocumentKey;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * The result of a lookup for a given path may be an existing document or a
 * marker that this document does not exist at a given version.
 */
var MaybeDocument = /** @class */ (function () {
    function MaybeDocument(key, version) {
        this.key = key;
        this.version = version;
    }
    MaybeDocument.compareByKey = function (d1, d2) {
        return DocumentKey.comparator(d1.key, d2.key);
    };
    return MaybeDocument;
}());
/**
 * Represents a document in Firestore with a key, version, data and whether the
 * data has local mutations applied to it.
 */
var Document = /** @class */ (function (_super) {
    tslib_1.__extends(Document, _super);
    function Document(key, version, data, options, 
    /**
     * Memoized serialized form of the document for optimization purposes (avoids repeated
     * serialization). Might be undefined.
     */
    proto) {
        var _this = _super.call(this, key, version) || this;
        _this.data = data;
        _this.proto = proto;
        _this.hasLocalMutations = !!options.hasLocalMutations;
        _this.hasCommittedMutations = !!options.hasCommittedMutations;
        return _this;
    }
    Document.prototype.field = function (path) {
        return this.data.field(path);
    };
    Document.prototype.fieldValue = function (path) {
        var field = this.field(path);
        return field ? field.value() : undefined;
    };
    Document.prototype.value = function () {
        return this.data.value();
    };
    Document.prototype.isEqual = function (other) {
        return (other instanceof Document &&
            this.key.isEqual(other.key) &&
            this.version.isEqual(other.version) &&
            this.data.isEqual(other.data) &&
            this.hasLocalMutations === other.hasLocalMutations &&
            this.hasCommittedMutations === other.hasCommittedMutations);
    };
    Document.prototype.toString = function () {
        return ("Document(" + this.key + ", " + this.version + ", " + this.data.toString() + ", " +
            ("{hasLocalMutations: " + this.hasLocalMutations + "}), ") +
            ("{hasCommittedMutations: " + this.hasCommittedMutations + "})"));
    };
    Object.defineProperty(Document.prototype, "hasPendingWrites", {
        get: function () {
            return this.hasLocalMutations || this.hasCommittedMutations;
        },
        enumerable: true,
        configurable: true
    });
    Document.compareByField = function (field, d1, d2) {
        var v1 = d1.field(field);
        var v2 = d2.field(field);
        if (v1 !== undefined && v2 !== undefined) {
            return v1.compareTo(v2);
        }
        else {
            return fail("Trying to compare documents on fields that don't exist");
        }
    };
    return Document;
}(MaybeDocument));
/**
 * A class representing a deleted document.
 * Version is set to 0 if we don't point to any specific time, otherwise it
 * denotes time we know it didn't exist at.
 */
var NoDocument = /** @class */ (function (_super) {
    tslib_1.__extends(NoDocument, _super);
    function NoDocument(key, version, options) {
        var _this = _super.call(this, key, version) || this;
        _this.hasCommittedMutations = !!(options && options.hasCommittedMutations);
        return _this;
    }
    NoDocument.prototype.toString = function () {
        return "NoDocument(" + this.key + ", " + this.version + ")";
    };
    Object.defineProperty(NoDocument.prototype, "hasPendingWrites", {
        get: function () {
            return this.hasCommittedMutations;
        },
        enumerable: true,
        configurable: true
    });
    NoDocument.prototype.isEqual = function (other) {
        return (other instanceof NoDocument &&
            other.hasCommittedMutations === this.hasCommittedMutations &&
            other.version.isEqual(this.version) &&
            other.key.isEqual(this.key));
    };
    return NoDocument;
}(MaybeDocument));
/**
 * A class representing an existing document whose data is unknown (e.g. a
 * document that was updated without a known base document).
 */
var UnknownDocument = /** @class */ (function (_super) {
    tslib_1.__extends(UnknownDocument, _super);
    function UnknownDocument(key, version) {
        return _super.call(this, key, version) || this;
    }
    UnknownDocument.prototype.toString = function () {
        return "UnknownDocument(" + this.key + ", " + this.version + ")";
    };
    Object.defineProperty(UnknownDocument.prototype, "hasPendingWrites", {
        get: function () {
            return true;
        },
        enumerable: true,
        configurable: true
    });
    UnknownDocument.prototype.isEqual = function (other) {
        return (other instanceof UnknownDocument &&
            other.version.isEqual(this.version) &&
            other.key.isEqual(this.key));
    };
    return UnknownDocument;
}(MaybeDocument));

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// An immutable sorted map implementation, based on a Left-leaning Red-Black
// tree.
var SortedMap = /** @class */ (function () {
    function SortedMap(comparator, root) {
        this.comparator = comparator;
        this.root = root ? root : LLRBNode.EMPTY;
    }
    // Returns a copy of the map, with the specified key/value added or replaced.
    SortedMap.prototype.insert = function (key, value) {
        return new SortedMap(this.comparator, this.root
            .insert(key, value, this.comparator)
            .copy(null, null, LLRBNode.BLACK, null, null));
    };
    // Returns a copy of the map, with the specified key removed.
    SortedMap.prototype.remove = function (key) {
        return new SortedMap(this.comparator, this.root
            .remove(key, this.comparator)
            .copy(null, null, LLRBNode.BLACK, null, null));
    };
    // Returns the value of the node with the given key, or null.
    SortedMap.prototype.get = function (key) {
        var node = this.root;
        while (!node.isEmpty()) {
            var cmp = this.comparator(key, node.key);
            if (cmp === 0) {
                return node.value;
            }
            else if (cmp < 0) {
                node = node.left;
            }
            else if (cmp > 0) {
                node = node.right;
            }
        }
        return null;
    };
    // Returns the index of the element in this sorted map, or -1 if it doesn't
    // exist.
    SortedMap.prototype.indexOf = function (key) {
        // Number of nodes that were pruned when descending right
        var prunedNodes = 0;
        var node = this.root;
        while (!node.isEmpty()) {
            var cmp = this.comparator(key, node.key);
            if (cmp === 0) {
                return prunedNodes + node.left.size;
            }
            else if (cmp < 0) {
                node = node.left;
            }
            else {
                // Count all nodes left of the node plus the node itself
                prunedNodes += node.left.size + 1;
                node = node.right;
            }
        }
        // Node not found
        return -1;
    };
    SortedMap.prototype.isEmpty = function () {
        return this.root.isEmpty();
    };
    Object.defineProperty(SortedMap.prototype, "size", {
        // Returns the total number of nodes in the map.
        get: function () {
            return this.root.size;
        },
        enumerable: true,
        configurable: true
    });
    // Returns the minimum key in the map.
    SortedMap.prototype.minKey = function () {
        return this.root.minKey();
    };
    // Returns the maximum key in the map.
    SortedMap.prototype.maxKey = function () {
        return this.root.maxKey();
    };
    // Traverses the map in key order and calls the specified action function
    // for each key/value pair. If action returns true, traversal is aborted.
    // Returns the first truthy value returned by action, or the last falsey
    // value returned by action.
    SortedMap.prototype.inorderTraversal = function (action) {
        return this.root.inorderTraversal(action);
    };
    SortedMap.prototype.forEach = function (fn) {
        this.inorderTraversal(function (k, v) {
            fn(k, v);
            return false;
        });
    };
    SortedMap.prototype.toString = function () {
        var descriptions = [];
        this.inorderTraversal(function (k, v) {
            descriptions.push(k + ":" + v);
            return false;
        });
        return "{" + descriptions.join(', ') + "}";
    };
    // Traverses the map in reverse key order and calls the specified action
    // function for each key/value pair. If action returns true, traversal is
    // aborted.
    // Returns the first truthy value returned by action, or the last falsey
    // value returned by action.
    SortedMap.prototype.reverseTraversal = function (action) {
        return this.root.reverseTraversal(action);
    };
    // Returns an iterator over the SortedMap.
    SortedMap.prototype.getIterator = function () {
        return new SortedMapIterator(this.root, null, this.comparator, false);
    };
    SortedMap.prototype.getIteratorFrom = function (key) {
        return new SortedMapIterator(this.root, key, this.comparator, false);
    };
    SortedMap.prototype.getReverseIterator = function () {
        return new SortedMapIterator(this.root, null, this.comparator, true);
    };
    SortedMap.prototype.getReverseIteratorFrom = function (key) {
        return new SortedMapIterator(this.root, key, this.comparator, true);
    };
    return SortedMap;
}()); // end SortedMap
// An iterator over an LLRBNode.
var SortedMapIterator = /** @class */ (function () {
    function SortedMapIterator(node, startKey, comparator, isReverse) {
        this.isReverse = isReverse;
        this.nodeStack = [];
        var cmp = 1;
        while (!node.isEmpty()) {
            cmp = startKey ? comparator(node.key, startKey) : 1;
            // flip the comparison if we're going in reverse
            if (isReverse)
                cmp *= -1;
            if (cmp < 0) {
                // This node is less than our start key. ignore it
                if (this.isReverse) {
                    node = node.left;
                }
                else {
                    node = node.right;
                }
            }
            else if (cmp === 0) {
                // This node is exactly equal to our start key. Push it on the stack,
                // but stop iterating;
                this.nodeStack.push(node);
                break;
            }
            else {
                // This node is greater than our start key, add it to the stack and move
                // to the next one
                this.nodeStack.push(node);
                if (this.isReverse) {
                    node = node.right;
                }
                else {
                    node = node.left;
                }
            }
        }
    }
    SortedMapIterator.prototype.getNext = function () {
        assert(this.nodeStack.length > 0, 'getNext() called on iterator when hasNext() is false.');
        var node = this.nodeStack.pop();
        var result = { key: node.key, value: node.value };
        if (this.isReverse) {
            node = node.left;
            while (!node.isEmpty()) {
                this.nodeStack.push(node);
                node = node.right;
            }
        }
        else {
            node = node.right;
            while (!node.isEmpty()) {
                this.nodeStack.push(node);
                node = node.left;
            }
        }
        return result;
    };
    SortedMapIterator.prototype.hasNext = function () {
        return this.nodeStack.length > 0;
    };
    SortedMapIterator.prototype.peek = function () {
        if (this.nodeStack.length === 0)
            return null;
        var node = this.nodeStack[this.nodeStack.length - 1];
        return { key: node.key, value: node.value };
    };
    return SortedMapIterator;
}()); // end SortedMapIterator
// Represents a node in a Left-leaning Red-Black tree.
var LLRBNode = /** @class */ (function () {
    function LLRBNode(key, value, color, left, right) {
        this.key = key;
        this.value = value;
        this.color = color != null ? color : LLRBNode.RED;
        this.left = left != null ? left : LLRBNode.EMPTY;
        this.right = right != null ? right : LLRBNode.EMPTY;
        this.size = this.left.size + 1 + this.right.size;
    }
    // Returns a copy of the current node, optionally replacing pieces of it.
    LLRBNode.prototype.copy = function (key, value, color, left, right) {
        return new LLRBNode(key != null ? key : this.key, value != null ? value : this.value, color != null ? color : this.color, left != null ? left : this.left, right != null ? right : this.right);
    };
    LLRBNode.prototype.isEmpty = function () {
        return false;
    };
    // Traverses the tree in key order and calls the specified action function
    // for each node. If action returns true, traversal is aborted.
    // Returns the first truthy value returned by action, or the last falsey
    // value returned by action.
    LLRBNode.prototype.inorderTraversal = function (action) {
        return (this.left.inorderTraversal(action) ||
            action(this.key, this.value) ||
            this.right.inorderTraversal(action));
    };
    // Traverses the tree in reverse key order and calls the specified action
    // function for each node. If action returns true, traversal is aborted.
    // Returns the first truthy value returned by action, or the last falsey
    // value returned by action.
    LLRBNode.prototype.reverseTraversal = function (action) {
        return (this.right.reverseTraversal(action) ||
            action(this.key, this.value) ||
            this.left.reverseTraversal(action));
    };
    // Returns the minimum node in the tree.
    LLRBNode.prototype.min = function () {
        if (this.left.isEmpty()) {
            return this;
        }
        else {
            return this.left.min();
        }
    };
    // Returns the maximum key in the tree.
    LLRBNode.prototype.minKey = function () {
        return this.min().key;
    };
    // Returns the maximum key in the tree.
    LLRBNode.prototype.maxKey = function () {
        if (this.right.isEmpty()) {
            return this.key;
        }
        else {
            return this.right.maxKey();
        }
    };
    // Returns new tree, with the key/value added.
    LLRBNode.prototype.insert = function (key, value, comparator) {
        var n = this;
        var cmp = comparator(key, n.key);
        if (cmp < 0) {
            n = n.copy(null, null, null, n.left.insert(key, value, comparator), null);
        }
        else if (cmp === 0) {
            n = n.copy(null, value, null, null, null);
        }
        else {
            n = n.copy(null, null, null, null, n.right.insert(key, value, comparator));
        }
        return n.fixUp();
    };
    LLRBNode.prototype.removeMin = function () {
        if (this.left.isEmpty()) {
            return LLRBNode.EMPTY;
        }
        var n = this;
        if (!n.left.isRed() && !n.left.left.isRed())
            n = n.moveRedLeft();
        n = n.copy(null, null, null, n.left.removeMin(), null);
        return n.fixUp();
    };
    // Returns new tree, with the specified item removed.
    LLRBNode.prototype.remove = function (key, comparator) {
        var smallest;
        var n = this;
        if (comparator(key, n.key) < 0) {
            if (!n.left.isEmpty() && !n.left.isRed() && !n.left.left.isRed()) {
                n = n.moveRedLeft();
            }
            n = n.copy(null, null, null, n.left.remove(key, comparator), null);
        }
        else {
            if (n.left.isRed())
                n = n.rotateRight();
            if (!n.right.isEmpty() && !n.right.isRed() && !n.right.left.isRed()) {
                n = n.moveRedRight();
            }
            if (comparator(key, n.key) === 0) {
                if (n.right.isEmpty()) {
                    return LLRBNode.EMPTY;
                }
                else {
                    smallest = n.right.min();
                    n = n.copy(smallest.key, smallest.value, null, null, n.right.removeMin());
                }
            }
            n = n.copy(null, null, null, null, n.right.remove(key, comparator));
        }
        return n.fixUp();
    };
    LLRBNode.prototype.isRed = function () {
        return this.color;
    };
    // Returns new tree after performing any needed rotations.
    LLRBNode.prototype.fixUp = function () {
        var n = this;
        if (n.right.isRed() && !n.left.isRed())
            n = n.rotateLeft();
        if (n.left.isRed() && n.left.left.isRed())
            n = n.rotateRight();
        if (n.left.isRed() && n.right.isRed())
            n = n.colorFlip();
        return n;
    };
    LLRBNode.prototype.moveRedLeft = function () {
        var n = this.colorFlip();
        if (n.right.left.isRed()) {
            n = n.copy(null, null, null, null, n.right.rotateRight());
            n = n.rotateLeft();
            n = n.colorFlip();
        }
        return n;
    };
    LLRBNode.prototype.moveRedRight = function () {
        var n = this.colorFlip();
        if (n.left.left.isRed()) {
            n = n.rotateRight();
            n = n.colorFlip();
        }
        return n;
    };
    LLRBNode.prototype.rotateLeft = function () {
        var nl = this.copy(null, null, LLRBNode.RED, null, this.right.left);
        return this.right.copy(null, null, this.color, nl, null);
    };
    LLRBNode.prototype.rotateRight = function () {
        var nr = this.copy(null, null, LLRBNode.RED, this.left.right, null);
        return this.left.copy(null, null, this.color, null, nr);
    };
    LLRBNode.prototype.colorFlip = function () {
        var left = this.left.copy(null, null, !this.left.color, null, null);
        var right = this.right.copy(null, null, !this.right.color, null, null);
        return this.copy(null, null, !this.color, left, right);
    };
    // For testing.
    LLRBNode.prototype.checkMaxDepth = function () {
        var blackDepth = this.check();
        if (Math.pow(2.0, blackDepth) <= this.size + 1) {
            return true;
        }
        else {
            return false;
        }
    };
    // In a balanced RB tree, the black-depth (number of black nodes) from root to
    // leaves is equal on both sides.  This function verifies that or asserts.
    LLRBNode.prototype.check = function () {
        if (this.isRed() && this.left.isRed()) {
            throw fail('Red node has red child(' + this.key + ',' + this.value + ')');
        }
        if (this.right.isRed()) {
            throw fail('Right child of (' + this.key + ',' + this.value + ') is red');
        }
        var blackDepth = this.left.check();
        if (blackDepth !== this.right.check()) {
            throw fail('Black depths differ');
        }
        else {
            return blackDepth + (this.isRed() ? 0 : 1);
        }
    };
    // tslint:disable-next-line:no-any Empty node is shared between all LLRB trees.
    LLRBNode.EMPTY = null;
    LLRBNode.RED = true;
    LLRBNode.BLACK = false;
    return LLRBNode;
}()); // end LLRBNode
// Represents an empty node (a leaf node in the Red-Black Tree).
var LLRBEmptyNode = /** @class */ (function () {
    function LLRBEmptyNode() {
        this.size = 0;
    }
    // Returns a copy of the current node.
    LLRBEmptyNode.prototype.copy = function (key, value, color, left, right) {
        return this;
    };
    // Returns a copy of the tree, with the specified key/value added.
    LLRBEmptyNode.prototype.insert = function (key, value, comparator) {
        return new LLRBNode(key, value);
    };
    // Returns a copy of the tree, with the specified key removed.
    LLRBEmptyNode.prototype.remove = function (key, comparator) {
        return this;
    };
    LLRBEmptyNode.prototype.isEmpty = function () {
        return true;
    };
    LLRBEmptyNode.prototype.inorderTraversal = function (action) {
        return false;
    };
    LLRBEmptyNode.prototype.reverseTraversal = function (action) {
        return false;
    };
    LLRBEmptyNode.prototype.minKey = function () {
        return null;
    };
    LLRBEmptyNode.prototype.maxKey = function () {
        return null;
    };
    LLRBEmptyNode.prototype.isRed = function () {
        return false;
    };
    // For testing.
    LLRBEmptyNode.prototype.checkMaxDepth = function () {
        return true;
    };
    LLRBEmptyNode.prototype.check = function () {
        return 0;
    };
    return LLRBEmptyNode;
}()); // end LLRBEmptyNode
LLRBNode.EMPTY = new LLRBEmptyNode();

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var TypeOrder;
(function (TypeOrder) {
    // This order is defined by the backend.
    TypeOrder[TypeOrder["NullValue"] = 0] = "NullValue";
    TypeOrder[TypeOrder["BooleanValue"] = 1] = "BooleanValue";
    TypeOrder[TypeOrder["NumberValue"] = 2] = "NumberValue";
    TypeOrder[TypeOrder["TimestampValue"] = 3] = "TimestampValue";
    TypeOrder[TypeOrder["StringValue"] = 4] = "StringValue";
    TypeOrder[TypeOrder["BlobValue"] = 5] = "BlobValue";
    TypeOrder[TypeOrder["RefValue"] = 6] = "RefValue";
    TypeOrder[TypeOrder["GeoPointValue"] = 7] = "GeoPointValue";
    TypeOrder[TypeOrder["ArrayValue"] = 8] = "ArrayValue";
    TypeOrder[TypeOrder["ObjectValue"] = 9] = "ObjectValue";
})(TypeOrder || (TypeOrder = {}));
/** Defines the return value for pending server timestamps. */
var ServerTimestampBehavior;
(function (ServerTimestampBehavior) {
    ServerTimestampBehavior[ServerTimestampBehavior["Default"] = 0] = "Default";
    ServerTimestampBehavior[ServerTimestampBehavior["Estimate"] = 1] = "Estimate";
    ServerTimestampBehavior[ServerTimestampBehavior["Previous"] = 2] = "Previous";
})(ServerTimestampBehavior || (ServerTimestampBehavior = {}));
/** Holds properties that define field value deserialization options. */
var FieldValueOptions = /** @class */ (function () {
    function FieldValueOptions(serverTimestampBehavior, timestampsInSnapshots) {
        this.serverTimestampBehavior = serverTimestampBehavior;
        this.timestampsInSnapshots = timestampsInSnapshots;
    }
    FieldValueOptions.fromSnapshotOptions = function (options, timestampsInSnapshots) {
        switch (options.serverTimestamps) {
            case 'estimate':
                return new FieldValueOptions(ServerTimestampBehavior.Estimate, timestampsInSnapshots);
            case 'previous':
                return new FieldValueOptions(ServerTimestampBehavior.Previous, timestampsInSnapshots);
            case 'none': // Fall-through intended.
            case undefined:
                return new FieldValueOptions(ServerTimestampBehavior.Default, timestampsInSnapshots);
            default:
                return fail('fromSnapshotOptions() called with invalid options.');
        }
    };
    return FieldValueOptions;
}());
/**
 * A field value represents a datatype as stored by Firestore.
 */
var FieldValue = /** @class */ (function () {
    function FieldValue() {
    }
    FieldValue.prototype.toString = function () {
        var val = this.value();
        return val === null ? 'null' : val.toString();
    };
    FieldValue.prototype.defaultCompareTo = function (other) {
        assert(this.typeOrder !== other.typeOrder, 'Default compareTo should not be used for values of same type.');
        var cmp = primitiveComparator(this.typeOrder, other.typeOrder);
        return cmp;
    };
    return FieldValue;
}());
var NullValue = /** @class */ (function (_super) {
    tslib_1.__extends(NullValue, _super);
    function NullValue() {
        var _this = _super.call(this) || this;
        _this.typeOrder = TypeOrder.NullValue;
        // internalValue is unused but we add it to work around
        // https://github.com/Microsoft/TypeScript/issues/15585
        _this.internalValue = null;
        return _this;
    }
    NullValue.prototype.value = function (options) {
        return null;
    };
    NullValue.prototype.isEqual = function (other) {
        return other instanceof NullValue;
    };
    NullValue.prototype.compareTo = function (other) {
        if (other instanceof NullValue) {
            return 0;
        }
        return this.defaultCompareTo(other);
    };
    NullValue.INSTANCE = new NullValue();
    return NullValue;
}(FieldValue));
var BooleanValue = /** @class */ (function (_super) {
    tslib_1.__extends(BooleanValue, _super);
    function BooleanValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.BooleanValue;
        return _this;
    }
    BooleanValue.prototype.value = function (options) {
        return this.internalValue;
    };
    BooleanValue.prototype.isEqual = function (other) {
        return (other instanceof BooleanValue &&
            this.internalValue === other.internalValue);
    };
    BooleanValue.prototype.compareTo = function (other) {
        if (other instanceof BooleanValue) {
            return primitiveComparator(this, other);
        }
        return this.defaultCompareTo(other);
    };
    BooleanValue.of = function (value) {
        return value ? BooleanValue.TRUE : BooleanValue.FALSE;
    };
    BooleanValue.TRUE = new BooleanValue(true);
    BooleanValue.FALSE = new BooleanValue(false);
    return BooleanValue;
}(FieldValue));
/** Base class for IntegerValue and DoubleValue. */
var NumberValue = /** @class */ (function (_super) {
    tslib_1.__extends(NumberValue, _super);
    function NumberValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.NumberValue;
        return _this;
    }
    NumberValue.prototype.value = function (options) {
        return this.internalValue;
    };
    NumberValue.prototype.compareTo = function (other) {
        if (other instanceof NumberValue) {
            return numericComparator(this.internalValue, other.internalValue);
        }
        return this.defaultCompareTo(other);
    };
    return NumberValue;
}(FieldValue));
/** Utility function to compare doubles (using Firestore semantics for NaN). */
function numericComparator(left, right) {
    if (left < right) {
        return -1;
    }
    else if (left > right) {
        return 1;
    }
    else if (left === right) {
        return 0;
    }
    else {
        // one or both are NaN.
        if (isNaN(left)) {
            return isNaN(right) ? 0 : -1;
        }
        else {
            return 1;
        }
    }
}
/**
 * Utility function to check numbers for equality using Firestore semantics
 * (NaN === NaN, -0.0 !== 0.0).
 */
function numericEquals(left, right) {
    // Implemented based on Object.is() polyfill from
    // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Object/is
    if (left === right) {
        // +0 != -0
        return left !== 0 || 1 / left === 1 / right;
    }
    else {
        // NaN == NaN
        return left !== left && right !== right;
    }
}
var IntegerValue = /** @class */ (function (_super) {
    tslib_1.__extends(IntegerValue, _super);
    function IntegerValue(internalValue) {
        return _super.call(this, internalValue) || this;
    }
    IntegerValue.prototype.isEqual = function (other) {
        // NOTE: DoubleValue and IntegerValue instances may compareTo() the same,
        // but that doesn't make them equal via isEqual().
        if (other instanceof IntegerValue) {
            return numericEquals(this.internalValue, other.internalValue);
        }
        else {
            return false;
        }
    };
    return IntegerValue;
}(NumberValue));
var DoubleValue = /** @class */ (function (_super) {
    tslib_1.__extends(DoubleValue, _super);
    function DoubleValue(internalValue) {
        var _this = _super.call(this, internalValue) || this;
        _this.internalValue = internalValue;
        return _this;
    }
    DoubleValue.prototype.isEqual = function (other) {
        // NOTE: DoubleValue and IntegerValue instances may compareTo() the same,
        // but that doesn't make them equal via isEqual().
        if (other instanceof DoubleValue) {
            return numericEquals(this.internalValue, other.internalValue);
        }
        else {
            return false;
        }
    };
    DoubleValue.NAN = new DoubleValue(NaN);
    DoubleValue.POSITIVE_INFINITY = new DoubleValue(Infinity);
    DoubleValue.NEGATIVE_INFINITY = new DoubleValue(-Infinity);
    return DoubleValue;
}(NumberValue));
// TODO(b/37267885): Add truncation support
var StringValue = /** @class */ (function (_super) {
    tslib_1.__extends(StringValue, _super);
    function StringValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.StringValue;
        return _this;
    }
    StringValue.prototype.value = function (options) {
        return this.internalValue;
    };
    StringValue.prototype.isEqual = function (other) {
        return (other instanceof StringValue && this.internalValue === other.internalValue);
    };
    StringValue.prototype.compareTo = function (other) {
        if (other instanceof StringValue) {
            return primitiveComparator(this.internalValue, other.internalValue);
        }
        return this.defaultCompareTo(other);
    };
    return StringValue;
}(FieldValue));
var TimestampValue = /** @class */ (function (_super) {
    tslib_1.__extends(TimestampValue, _super);
    function TimestampValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.TimestampValue;
        return _this;
    }
    TimestampValue.prototype.value = function (options) {
        if (!options || options.timestampsInSnapshots) {
            return this.internalValue;
        }
        else {
            return this.internalValue.toDate();
        }
    };
    TimestampValue.prototype.isEqual = function (other) {
        return (other instanceof TimestampValue &&
            this.internalValue.isEqual(other.internalValue));
    };
    TimestampValue.prototype.compareTo = function (other) {
        if (other instanceof TimestampValue) {
            return this.internalValue._compareTo(other.internalValue);
        }
        else if (other instanceof ServerTimestampValue) {
            // Concrete timestamps come before server timestamps.
            return -1;
        }
        else {
            return this.defaultCompareTo(other);
        }
    };
    return TimestampValue;
}(FieldValue));
/**
 * Represents a locally-applied ServerTimestamp.
 *
 * Notes:
 * - ServerTimestampValue instances are created as the result of applying a
 *   TransformMutation (see TransformMutation.applyTo()). They can only exist in
 *   the local view of a document. Therefore they do not need to be parsed or
 *   serialized.
 * - When evaluated locally (e.g. for snapshot.data()), they by default
 *   evaluate to `null`. This behavior can be configured by passing custom
 *   FieldValueOptions to value().
 * - With respect to other ServerTimestampValues, they sort by their
 *   localWriteTime.
 */
var ServerTimestampValue = /** @class */ (function (_super) {
    tslib_1.__extends(ServerTimestampValue, _super);
    function ServerTimestampValue(localWriteTime, previousValue) {
        var _this = _super.call(this) || this;
        _this.localWriteTime = localWriteTime;
        _this.previousValue = previousValue;
        _this.typeOrder = TypeOrder.TimestampValue;
        return _this;
    }
    ServerTimestampValue.prototype.value = function (options) {
        if (options &&
            options.serverTimestampBehavior === ServerTimestampBehavior.Estimate) {
            return new TimestampValue(this.localWriteTime).value(options);
        }
        else if (options &&
            options.serverTimestampBehavior === ServerTimestampBehavior.Previous) {
            return this.previousValue ? this.previousValue.value(options) : null;
        }
        else {
            return null;
        }
    };
    ServerTimestampValue.prototype.isEqual = function (other) {
        return (other instanceof ServerTimestampValue &&
            this.localWriteTime.isEqual(other.localWriteTime));
    };
    ServerTimestampValue.prototype.compareTo = function (other) {
        if (other instanceof ServerTimestampValue) {
            return this.localWriteTime._compareTo(other.localWriteTime);
        }
        else if (other instanceof TimestampValue) {
            // Server timestamps come after all concrete timestamps.
            return 1;
        }
        else {
            return this.defaultCompareTo(other);
        }
    };
    ServerTimestampValue.prototype.toString = function () {
        return '<ServerTimestamp localTime=' + this.localWriteTime.toString() + '>';
    };
    return ServerTimestampValue;
}(FieldValue));
var BlobValue = /** @class */ (function (_super) {
    tslib_1.__extends(BlobValue, _super);
    function BlobValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.BlobValue;
        return _this;
    }
    BlobValue.prototype.value = function (options) {
        return this.internalValue;
    };
    BlobValue.prototype.isEqual = function (other) {
        return (other instanceof BlobValue &&
            this.internalValue.isEqual(other.internalValue));
    };
    BlobValue.prototype.compareTo = function (other) {
        if (other instanceof BlobValue) {
            return this.internalValue._compareTo(other.internalValue);
        }
        return this.defaultCompareTo(other);
    };
    return BlobValue;
}(FieldValue));
var RefValue = /** @class */ (function (_super) {
    tslib_1.__extends(RefValue, _super);
    function RefValue(databaseId, key) {
        var _this = _super.call(this) || this;
        _this.databaseId = databaseId;
        _this.key = key;
        _this.typeOrder = TypeOrder.RefValue;
        return _this;
    }
    RefValue.prototype.value = function (options) {
        return this.key;
    };
    RefValue.prototype.isEqual = function (other) {
        if (other instanceof RefValue) {
            return (this.key.isEqual(other.key) && this.databaseId.isEqual(other.databaseId));
        }
        else {
            return false;
        }
    };
    RefValue.prototype.compareTo = function (other) {
        if (other instanceof RefValue) {
            var cmp = this.databaseId.compareTo(other.databaseId);
            return cmp !== 0 ? cmp : DocumentKey.comparator(this.key, other.key);
        }
        return this.defaultCompareTo(other);
    };
    return RefValue;
}(FieldValue));
var GeoPointValue = /** @class */ (function (_super) {
    tslib_1.__extends(GeoPointValue, _super);
    function GeoPointValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.GeoPointValue;
        return _this;
    }
    GeoPointValue.prototype.value = function (options) {
        return this.internalValue;
    };
    GeoPointValue.prototype.isEqual = function (other) {
        return (other instanceof GeoPointValue &&
            this.internalValue.isEqual(other.internalValue));
    };
    GeoPointValue.prototype.compareTo = function (other) {
        if (other instanceof GeoPointValue) {
            return this.internalValue._compareTo(other.internalValue);
        }
        return this.defaultCompareTo(other);
    };
    return GeoPointValue;
}(FieldValue));
var ObjectValue = /** @class */ (function (_super) {
    tslib_1.__extends(ObjectValue, _super);
    function ObjectValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.ObjectValue;
        return _this;
    }
    ObjectValue.prototype.value = function (options) {
        var result = {};
        this.internalValue.inorderTraversal(function (key, val) {
            result[key] = val.value(options);
        });
        return result;
    };
    ObjectValue.prototype.forEach = function (action) {
        this.internalValue.inorderTraversal(action);
    };
    ObjectValue.prototype.isEqual = function (other) {
        if (other instanceof ObjectValue) {
            var it1 = this.internalValue.getIterator();
            var it2 = other.internalValue.getIterator();
            while (it1.hasNext() && it2.hasNext()) {
                var next1 = it1.getNext();
                var next2 = it2.getNext();
                if (next1.key !== next2.key || !next1.value.isEqual(next2.value)) {
                    return false;
                }
            }
            return !it1.hasNext() && !it2.hasNext();
        }
        return false;
    };
    ObjectValue.prototype.compareTo = function (other) {
        if (other instanceof ObjectValue) {
            var it1 = this.internalValue.getIterator();
            var it2 = other.internalValue.getIterator();
            while (it1.hasNext() && it2.hasNext()) {
                var next1 = it1.getNext();
                var next2 = it2.getNext();
                var cmp = primitiveComparator(next1.key, next2.key) ||
                    next1.value.compareTo(next2.value);
                if (cmp) {
                    return cmp;
                }
            }
            // Only equal if both iterators are exhausted
            return primitiveComparator(it1.hasNext(), it2.hasNext());
        }
        else {
            return this.defaultCompareTo(other);
        }
    };
    ObjectValue.prototype.set = function (path, to) {
        assert(!path.isEmpty(), 'Cannot set field for empty path on ObjectValue');
        if (path.length === 1) {
            return this.setChild(path.firstSegment(), to);
        }
        else {
            var child = this.child(path.firstSegment());
            if (!(child instanceof ObjectValue)) {
                child = ObjectValue.EMPTY;
            }
            var newChild = child.set(path.popFirst(), to);
            return this.setChild(path.firstSegment(), newChild);
        }
    };
    ObjectValue.prototype.delete = function (path) {
        assert(!path.isEmpty(), 'Cannot delete field for empty path on ObjectValue');
        if (path.length === 1) {
            return new ObjectValue(this.internalValue.remove(path.firstSegment()));
        }
        else {
            // nested field
            var child = this.child(path.firstSegment());
            if (child instanceof ObjectValue) {
                var newChild = child.delete(path.popFirst());
                return new ObjectValue(this.internalValue.insert(path.firstSegment(), newChild));
            }
            else {
                // Don't actually change a primitive value to an object for a delete
                return this;
            }
        }
    };
    ObjectValue.prototype.contains = function (path) {
        return this.field(path) !== undefined;
    };
    ObjectValue.prototype.field = function (path) {
        assert(!path.isEmpty(), "Can't get field of empty path");
        var field = this;
        path.forEach(function (pathSegment) {
            if (field instanceof ObjectValue) {
                field = field.internalValue.get(pathSegment) || undefined;
            }
            else {
                field = undefined;
            }
        });
        return field;
    };
    ObjectValue.prototype.toString = function () {
        return this.internalValue.toString();
    };
    ObjectValue.prototype.child = function (childName) {
        return this.internalValue.get(childName) || undefined;
    };
    ObjectValue.prototype.setChild = function (childName, value) {
        return new ObjectValue(this.internalValue.insert(childName, value));
    };
    ObjectValue.EMPTY = new ObjectValue(new SortedMap(primitiveComparator));
    return ObjectValue;
}(FieldValue));
var ArrayValue = /** @class */ (function (_super) {
    tslib_1.__extends(ArrayValue, _super);
    function ArrayValue(internalValue) {
        var _this = _super.call(this) || this;
        _this.internalValue = internalValue;
        _this.typeOrder = TypeOrder.ArrayValue;
        return _this;
    }
    ArrayValue.prototype.value = function (options) {
        return this.internalValue.map(function (v) { return v.value(options); });
    };
    ArrayValue.prototype.forEach = function (action) {
        this.internalValue.forEach(action);
    };
    ArrayValue.prototype.isEqual = function (other) {
        if (other instanceof ArrayValue) {
            if (this.internalValue.length !== other.internalValue.length) {
                return false;
            }
            for (var i = 0; i < this.internalValue.length; i++) {
                if (!this.internalValue[i].isEqual(other.internalValue[i])) {
                    return false;
                }
            }
            return true;
        }
        return false;
    };
    ArrayValue.prototype.compareTo = function (other) {
        if (other instanceof ArrayValue) {
            var minLength = Math.min(this.internalValue.length, other.internalValue.length);
            for (var i = 0; i < minLength; i++) {
                var cmp = this.internalValue[i].compareTo(other.internalValue[i]);
                if (cmp) {
                    return cmp;
                }
            }
            return primitiveComparator(this.internalValue.length, other.internalValue.length);
        }
        else {
            return this.defaultCompareTo(other);
        }
    };
    ArrayValue.prototype.toString = function () {
        var descriptions = this.internalValue.map(function (v) { return v.toString(); });
        return "[" + descriptions.join(',') + "]";
    };
    return ArrayValue;
}(FieldValue));

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// Untyped Number alias we can use to check for ES6 methods / properties.
// tslint:disable-next-line:no-any variable-name
var NumberAsAny = Number;
/**
 * Minimum safe integer in Javascript because of floating point precision.
 * Added to not rely on ES6 features.
 */
var MIN_SAFE_INTEGER = NumberAsAny.MIN_SAFE_INTEGER || -(Math.pow(2, 53) - 1);
/**
 * Maximum safe integer in Javascript because of floating point precision.
 * Added to not rely on ES6 features.
 */
var MAX_SAFE_INTEGER = NumberAsAny.MAX_SAFE_INTEGER || Math.pow(2, 53) - 1;
/**
 * Returns whether an number is an integer, uses native implementation if
 * available.
 * Added to not rely on ES6 features.
 * @param value The value to test for being an integer
 */
var isInteger = NumberAsAny.isInteger ||
    (function (value) {
        return typeof value === 'number' &&
            isFinite(value) &&
            Math.floor(value) === value;
    });
/**
 * Returns whether a variable is either undefined or null.
 */
function isNullOrUndefined(value) {
    return value === null || value === undefined;
}
/**
 * Returns whether a value is an integer and in the safe integer range
 * @param value The value to test for being an integer and in the safe range
 */
function isSafeInteger(value) {
    return (isInteger(value) &&
        value <= MAX_SAFE_INTEGER &&
        value >= MIN_SAFE_INTEGER);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var Query = /** @class */ (function () {
    /**
     * Initializes a Query with a path and optional additional query constraints.
     * Path must currently be empty if this is a collection group query.
     */
    function Query(path, collectionGroup, explicitOrderBy, filters, limit, startAt, endAt) {
        if (collectionGroup === void 0) { collectionGroup = null; }
        if (explicitOrderBy === void 0) { explicitOrderBy = []; }
        if (filters === void 0) { filters = []; }
        if (limit === void 0) { limit = null; }
        if (startAt === void 0) { startAt = null; }
        if (endAt === void 0) { endAt = null; }
        this.path = path;
        this.collectionGroup = collectionGroup;
        this.explicitOrderBy = explicitOrderBy;
        this.filters = filters;
        this.limit = limit;
        this.startAt = startAt;
        this.endAt = endAt;
        this.memoizedCanonicalId = null;
        this.memoizedOrderBy = null;
        if (this.startAt) {
            this.assertValidBound(this.startAt);
        }
        if (this.endAt) {
            this.assertValidBound(this.endAt);
        }
    }
    Query.atPath = function (path) {
        return new Query(path);
    };
    Object.defineProperty(Query.prototype, "orderBy", {
        get: function () {
            if (this.memoizedOrderBy === null) {
                var inequalityField = this.getInequalityFilterField();
                var firstOrderByField = this.getFirstOrderByField();
                if (inequalityField !== null && firstOrderByField === null) {
                    // In order to implicitly add key ordering, we must also add the
                    // inequality filter field for it to be a valid query.
                    // Note that the default inequality field and key ordering is ascending.
                    if (inequalityField.isKeyField()) {
                        this.memoizedOrderBy = [KEY_ORDERING_ASC];
                    }
                    else {
                        this.memoizedOrderBy = [
                            new OrderBy(inequalityField),
                            KEY_ORDERING_ASC
                        ];
                    }
                }
                else {
                    assert(inequalityField === null ||
                        (firstOrderByField !== null &&
                            inequalityField.isEqual(firstOrderByField)), 'First orderBy should match inequality field.');
                    this.memoizedOrderBy = [];
                    var foundKeyOrdering = false;
                    for (var _i = 0, _a = this.explicitOrderBy; _i < _a.length; _i++) {
                        var orderBy = _a[_i];
                        this.memoizedOrderBy.push(orderBy);
                        if (orderBy.field.isKeyField()) {
                            foundKeyOrdering = true;
                        }
                    }
                    if (!foundKeyOrdering) {
                        // The order of the implicit key ordering always matches the last
                        // explicit order by
                        var lastDirection = this.explicitOrderBy.length > 0
                            ? this.explicitOrderBy[this.explicitOrderBy.length - 1].dir
                            : Direction.ASCENDING;
                        this.memoizedOrderBy.push(lastDirection === Direction.ASCENDING
                            ? KEY_ORDERING_ASC
                            : KEY_ORDERING_DESC);
                    }
                }
            }
            return this.memoizedOrderBy;
        },
        enumerable: true,
        configurable: true
    });
    Query.prototype.addFilter = function (filter) {
        assert(this.getInequalityFilterField() == null ||
            !(filter instanceof RelationFilter) ||
            !filter.isInequality() ||
            filter.field.isEqual(this.getInequalityFilterField()), 'Query must only have one inequality field.');
        assert(!this.isDocumentQuery(), 'No filtering allowed for document query');
        var newFilters = this.filters.concat([filter]);
        return new Query(this.path, this.collectionGroup, this.explicitOrderBy.slice(), newFilters, this.limit, this.startAt, this.endAt);
    };
    Query.prototype.addOrderBy = function (orderBy) {
        assert(!this.startAt && !this.endAt, 'Bounds must be set after orderBy');
        // TODO(dimond): validate that orderBy does not list the same key twice.
        var newOrderBy = this.explicitOrderBy.concat([orderBy]);
        return new Query(this.path, this.collectionGroup, newOrderBy, this.filters.slice(), this.limit, this.startAt, this.endAt);
    };
    Query.prototype.withLimit = function (limit) {
        return new Query(this.path, this.collectionGroup, this.explicitOrderBy.slice(), this.filters.slice(), limit, this.startAt, this.endAt);
    };
    Query.prototype.withStartAt = function (bound) {
        return new Query(this.path, this.collectionGroup, this.explicitOrderBy.slice(), this.filters.slice(), this.limit, bound, this.endAt);
    };
    Query.prototype.withEndAt = function (bound) {
        return new Query(this.path, this.collectionGroup, this.explicitOrderBy.slice(), this.filters.slice(), this.limit, this.startAt, bound);
    };
    /**
     * Helper to convert a collection group query into a collection query at a
     * specific path. This is used when executing collection group queries, since
     * we have to split the query into a set of collection queries at multiple
     * paths.
     */
    Query.prototype.asCollectionQueryAtPath = function (path) {
        return new Query(path, 
        /*collectionGroup=*/ null, this.explicitOrderBy.slice(), this.filters.slice(), this.limit, this.startAt, this.endAt);
    };
    // TODO(b/29183165): This is used to get a unique string from a query to, for
    // example, use as a dictionary key, but the implementation is subject to
    // collisions. Make it collision-free.
    Query.prototype.canonicalId = function () {
        if (this.memoizedCanonicalId === null) {
            var canonicalId = this.path.canonicalString();
            if (this.isCollectionGroupQuery()) {
                canonicalId += '|cg:' + this.collectionGroup;
            }
            canonicalId += '|f:';
            for (var _i = 0, _a = this.filters; _i < _a.length; _i++) {
                var filter = _a[_i];
                canonicalId += filter.canonicalId();
                canonicalId += ',';
            }
            canonicalId += '|ob:';
            // TODO(dimond): make this collision resistant
            for (var _b = 0, _c = this.orderBy; _b < _c.length; _b++) {
                var orderBy = _c[_b];
                canonicalId += orderBy.canonicalId();
                canonicalId += ',';
            }
            if (!isNullOrUndefined(this.limit)) {
                canonicalId += '|l:';
                canonicalId += this.limit;
            }
            if (this.startAt) {
                canonicalId += '|lb:';
                canonicalId += this.startAt.canonicalId();
            }
            if (this.endAt) {
                canonicalId += '|ub:';
                canonicalId += this.endAt.canonicalId();
            }
            this.memoizedCanonicalId = canonicalId;
        }
        return this.memoizedCanonicalId;
    };
    Query.prototype.toString = function () {
        var str = 'Query(' + this.path.canonicalString();
        if (this.isCollectionGroupQuery()) {
            str += ' collectionGroup=' + this.collectionGroup;
        }
        if (this.filters.length > 0) {
            str += ", filters: [" + this.filters.join(', ') + "]";
        }
        if (!isNullOrUndefined(this.limit)) {
            str += ', limit: ' + this.limit;
        }
        if (this.explicitOrderBy.length > 0) {
            str += ", orderBy: [" + this.explicitOrderBy.join(', ') + "]";
        }
        if (this.startAt) {
            str += ', startAt: ' + this.startAt.canonicalId();
        }
        if (this.endAt) {
            str += ', endAt: ' + this.endAt.canonicalId();
        }
        return str + ')';
    };
    Query.prototype.isEqual = function (other) {
        if (this.limit !== other.limit) {
            return false;
        }
        if (this.orderBy.length !== other.orderBy.length) {
            return false;
        }
        for (var i = 0; i < this.orderBy.length; i++) {
            if (!this.orderBy[i].isEqual(other.orderBy[i])) {
                return false;
            }
        }
        if (this.filters.length !== other.filters.length) {
            return false;
        }
        for (var i = 0; i < this.filters.length; i++) {
            if (!this.filters[i].isEqual(other.filters[i])) {
                return false;
            }
        }
        if (this.collectionGroup !== other.collectionGroup) {
            return false;
        }
        if (!this.path.isEqual(other.path)) {
            return false;
        }
        if (this.startAt !== null
            ? !this.startAt.isEqual(other.startAt)
            : other.startAt !== null) {
            return false;
        }
        return this.endAt !== null
            ? this.endAt.isEqual(other.endAt)
            : other.endAt === null;
    };
    Query.prototype.docComparator = function (d1, d2) {
        var comparedOnKeyField = false;
        for (var _i = 0, _a = this.orderBy; _i < _a.length; _i++) {
            var orderBy = _a[_i];
            var comp = orderBy.compare(d1, d2);
            if (comp !== 0)
                return comp;
            comparedOnKeyField = comparedOnKeyField || orderBy.field.isKeyField();
        }
        // Assert that we actually compared by key
        assert(comparedOnKeyField, "orderBy used that doesn't compare on key field");
        return 0;
    };
    Query.prototype.matches = function (doc) {
        return (this.matchesPathAndCollectionGroup(doc) &&
            this.matchesOrderBy(doc) &&
            this.matchesFilters(doc) &&
            this.matchesBounds(doc));
    };
    Query.prototype.hasLimit = function () {
        return !isNullOrUndefined(this.limit);
    };
    Query.prototype.getFirstOrderByField = function () {
        return this.explicitOrderBy.length > 0
            ? this.explicitOrderBy[0].field
            : null;
    };
    Query.prototype.getInequalityFilterField = function () {
        for (var _i = 0, _a = this.filters; _i < _a.length; _i++) {
            var filter = _a[_i];
            if (filter instanceof RelationFilter && filter.isInequality()) {
                return filter.field;
            }
        }
        return null;
    };
    Query.prototype.hasArrayContainsFilter = function () {
        return (this.filters.find(function (filter) {
            return filter instanceof RelationFilter &&
                filter.op === RelationOp.ARRAY_CONTAINS;
        }) !== undefined);
    };
    Query.prototype.isDocumentQuery = function () {
        return (DocumentKey.isDocumentKey(this.path) &&
            this.collectionGroup === null &&
            this.filters.length === 0);
    };
    Query.prototype.isCollectionGroupQuery = function () {
        return this.collectionGroup !== null;
    };
    Query.prototype.matchesPathAndCollectionGroup = function (doc) {
        var docPath = doc.key.path;
        if (this.collectionGroup !== null) {
            // NOTE: this.path is currently always empty since we don't expose Collection
            // Group queries rooted at a document path yet.
            return (doc.key.hasCollectionId(this.collectionGroup) &&
                this.path.isPrefixOf(docPath));
        }
        else if (DocumentKey.isDocumentKey(this.path)) {
            // exact match for document queries
            return this.path.isEqual(docPath);
        }
        else {
            // shallow ancestor queries by default
            return this.path.isImmediateParentOf(docPath);
        }
    };
    /**
     * A document must have a value for every ordering clause in order to show up
     * in the results.
     */
    Query.prototype.matchesOrderBy = function (doc) {
        for (var _i = 0, _a = this.explicitOrderBy; _i < _a.length; _i++) {
            var orderBy = _a[_i];
            // order by key always matches
            if (!orderBy.field.isKeyField() &&
                doc.field(orderBy.field) === undefined) {
                return false;
            }
        }
        return true;
    };
    Query.prototype.matchesFilters = function (doc) {
        for (var _i = 0, _a = this.filters; _i < _a.length; _i++) {
            var filter = _a[_i];
            if (!filter.matches(doc)) {
                return false;
            }
        }
        return true;
    };
    /**
     * Makes sure a document is within the bounds, if provided.
     */
    Query.prototype.matchesBounds = function (doc) {
        if (this.startAt && !this.startAt.sortsBeforeDocument(this.orderBy, doc)) {
            return false;
        }
        if (this.endAt && this.endAt.sortsBeforeDocument(this.orderBy, doc)) {
            return false;
        }
        return true;
    };
    Query.prototype.assertValidBound = function (bound) {
        assert(bound.position.length <= this.orderBy.length, 'Bound is longer than orderBy');
    };
    return Query;
}());
var Filter = /** @class */ (function () {
    function Filter() {
    }
    /**
     * Creates a filter based on the provided arguments.
     */
    Filter.create = function (field, op, value) {
        if (value.isEqual(NullValue.INSTANCE)) {
            if (op !== RelationOp.EQUAL) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid query. You can only perform equals comparisons on null.');
            }
            return new NullFilter(field);
        }
        else if (value.isEqual(DoubleValue.NAN)) {
            if (op !== RelationOp.EQUAL) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid query. You can only perform equals comparisons on NaN.');
            }
            return new NanFilter(field);
        }
        else {
            return new RelationFilter(field, op, value);
        }
    };
    return Filter;
}());
var RelationOp = /** @class */ (function () {
    function RelationOp(name) {
        this.name = name;
    }
    RelationOp.fromString = function (op) {
        switch (op) {
            case '<':
                return RelationOp.LESS_THAN;
            case '<=':
                return RelationOp.LESS_THAN_OR_EQUAL;
            case '==':
                return RelationOp.EQUAL;
            case '>=':
                return RelationOp.GREATER_THAN_OR_EQUAL;
            case '>':
                return RelationOp.GREATER_THAN;
            case 'array-contains':
                return RelationOp.ARRAY_CONTAINS;
            default:
                return fail('Unknown relation: ' + op);
        }
    };
    RelationOp.prototype.toString = function () {
        return this.name;
    };
    RelationOp.prototype.isEqual = function (other) {
        return this.name === other.name;
    };
    RelationOp.LESS_THAN = new RelationOp('<');
    RelationOp.LESS_THAN_OR_EQUAL = new RelationOp('<=');
    RelationOp.EQUAL = new RelationOp('==');
    RelationOp.GREATER_THAN = new RelationOp('>');
    RelationOp.GREATER_THAN_OR_EQUAL = new RelationOp('>=');
    RelationOp.ARRAY_CONTAINS = new RelationOp('array-contains');
    return RelationOp;
}());
var RelationFilter = /** @class */ (function (_super) {
    tslib_1.__extends(RelationFilter, _super);
    function RelationFilter(field, op, value) {
        var _this = _super.call(this) || this;
        _this.field = field;
        _this.op = op;
        _this.value = value;
        return _this;
    }
    RelationFilter.prototype.matches = function (doc) {
        if (this.field.isKeyField()) {
            assert(this.value instanceof RefValue, 'Comparing on key, but filter value not a RefValue');
            assert(this.op !== RelationOp.ARRAY_CONTAINS, "array-contains queries don't make sense on document keys.");
            var refValue = this.value;
            var comparison = DocumentKey.comparator(doc.key, refValue.key);
            return this.matchesComparison(comparison);
        }
        else {
            var val = doc.field(this.field);
            return val !== undefined && this.matchesValue(val);
        }
    };
    RelationFilter.prototype.matchesValue = function (value) {
        var _this = this;
        if (this.op === RelationOp.ARRAY_CONTAINS) {
            return (value instanceof ArrayValue &&
                value.internalValue.find(function (element) { return element.isEqual(_this.value); }) !==
                    undefined);
        }
        else {
            // Only compare types with matching backend order (such as double and int).
            return (this.value.typeOrder === value.typeOrder &&
                this.matchesComparison(value.compareTo(this.value)));
        }
    };
    RelationFilter.prototype.matchesComparison = function (comparison) {
        switch (this.op) {
            case RelationOp.LESS_THAN:
                return comparison < 0;
            case RelationOp.LESS_THAN_OR_EQUAL:
                return comparison <= 0;
            case RelationOp.EQUAL:
                return comparison === 0;
            case RelationOp.GREATER_THAN:
                return comparison > 0;
            case RelationOp.GREATER_THAN_OR_EQUAL:
                return comparison >= 0;
            default:
                return fail('Unknown relation op' + this.op);
        }
    };
    RelationFilter.prototype.isInequality = function () {
        return (this.op !== RelationOp.EQUAL && this.op !== RelationOp.ARRAY_CONTAINS);
    };
    RelationFilter.prototype.canonicalId = function () {
        // TODO(b/29183165): Technically, this won't be unique if two values have
        // the same description, such as the int 3 and the string "3". So we should
        // add the types in here somehow, too.
        return (this.field.canonicalString() + this.op.toString() + this.value.toString());
    };
    RelationFilter.prototype.isEqual = function (other) {
        if (other instanceof RelationFilter) {
            return (this.op.isEqual(other.op) &&
                this.field.isEqual(other.field) &&
                this.value.isEqual(other.value));
        }
        else {
            return false;
        }
    };
    RelationFilter.prototype.toString = function () {
        return this.field.canonicalString() + " " + this.op + " " + this.value.value();
    };
    return RelationFilter;
}(Filter));
/**
 * Filter that matches 'null' values.
 */
var NullFilter = /** @class */ (function (_super) {
    tslib_1.__extends(NullFilter, _super);
    function NullFilter(field) {
        var _this = _super.call(this) || this;
        _this.field = field;
        return _this;
    }
    NullFilter.prototype.matches = function (doc) {
        var val = doc.field(this.field);
        return val !== undefined && val.value() === null;
    };
    NullFilter.prototype.canonicalId = function () {
        return this.field.canonicalString() + ' IS null';
    };
    NullFilter.prototype.toString = function () {
        return this.field.canonicalString() + " IS null";
    };
    NullFilter.prototype.isEqual = function (other) {
        if (other instanceof NullFilter) {
            return this.field.isEqual(other.field);
        }
        else {
            return false;
        }
    };
    return NullFilter;
}(Filter));
/**
 * Filter that matches 'NaN' values.
 */
var NanFilter = /** @class */ (function (_super) {
    tslib_1.__extends(NanFilter, _super);
    function NanFilter(field) {
        var _this = _super.call(this) || this;
        _this.field = field;
        return _this;
    }
    NanFilter.prototype.matches = function (doc) {
        var field = doc.field(this.field);
        var val = field && field.value();
        return typeof val === 'number' && isNaN(val);
    };
    NanFilter.prototype.canonicalId = function () {
        return this.field.canonicalString() + ' IS NaN';
    };
    NanFilter.prototype.toString = function () {
        return this.field.canonicalString() + " IS NaN";
    };
    NanFilter.prototype.isEqual = function (other) {
        if (other instanceof NanFilter) {
            return this.field.isEqual(other.field);
        }
        else {
            return false;
        }
    };
    return NanFilter;
}(Filter));
/**
 * The direction of sorting in an order by.
 */
var Direction = /** @class */ (function () {
    function Direction(name) {
        this.name = name;
    }
    Direction.prototype.toString = function () {
        return this.name;
    };
    Direction.ASCENDING = new Direction('asc');
    Direction.DESCENDING = new Direction('desc');
    return Direction;
}());
/**
 * Represents a bound of a query.
 *
 * The bound is specified with the given components representing a position and
 * whether it's just before or just after the position (relative to whatever the
 * query order is).
 *
 * The position represents a logical index position for a query. It's a prefix
 * of values for the (potentially implicit) order by clauses of a query.
 *
 * Bound provides a function to determine whether a document comes before or
 * after a bound. This is influenced by whether the position is just before or
 * just after the provided values.
 */
var Bound = /** @class */ (function () {
    function Bound(position, before) {
        this.position = position;
        this.before = before;
    }
    Bound.prototype.canonicalId = function () {
        // TODO(b/29183165): Make this collision robust.
        var canonicalId = this.before ? 'b:' : 'a:';
        for (var _i = 0, _a = this.position; _i < _a.length; _i++) {
            var component = _a[_i];
            canonicalId += component.toString();
        }
        return canonicalId;
    };
    /**
     * Returns true if a document sorts before a bound using the provided sort
     * order.
     */
    Bound.prototype.sortsBeforeDocument = function (orderBy, doc) {
        assert(this.position.length <= orderBy.length, "Bound has more components than query's orderBy");
        var comparison = 0;
        for (var i = 0; i < this.position.length; i++) {
            var orderByComponent = orderBy[i];
            var component = this.position[i];
            if (orderByComponent.field.isKeyField()) {
                assert(component instanceof RefValue, 'Bound has a non-key value where the key path is being used.');
                comparison = DocumentKey.comparator(component.key, doc.key);
            }
            else {
                var docValue = doc.field(orderByComponent.field);
                assert(docValue !== undefined, 'Field should exist since document matched the orderBy already.');
                comparison = component.compareTo(docValue);
            }
            if (orderByComponent.dir === Direction.DESCENDING) {
                comparison = comparison * -1;
            }
            if (comparison !== 0) {
                break;
            }
        }
        return this.before ? comparison <= 0 : comparison < 0;
    };
    Bound.prototype.isEqual = function (other) {
        if (other === null) {
            return false;
        }
        if (this.before !== other.before ||
            this.position.length !== other.position.length) {
            return false;
        }
        for (var i = 0; i < this.position.length; i++) {
            var thisPosition = this.position[i];
            var otherPosition = other.position[i];
            return thisPosition.isEqual(otherPosition);
        }
        return true;
    };
    return Bound;
}());
/**
 * An ordering on a field, in some Direction. Direction defaults to ASCENDING.
 */
var OrderBy = /** @class */ (function () {
    function OrderBy(field, dir) {
        this.field = field;
        if (dir === undefined) {
            dir = Direction.ASCENDING;
        }
        this.dir = dir;
        this.isKeyOrderBy = field.isKeyField();
    }
    OrderBy.prototype.compare = function (d1, d2) {
        var comparison = this.isKeyOrderBy
            ? Document.compareByKey(d1, d2)
            : Document.compareByField(this.field, d1, d2);
        switch (this.dir) {
            case Direction.ASCENDING:
                return comparison;
            case Direction.DESCENDING:
                return -1 * comparison;
            default:
                return fail('Unknown direction: ' + this.dir);
        }
    };
    OrderBy.prototype.canonicalId = function () {
        // TODO(b/29183165): Make this collision robust.
        return this.field.canonicalString() + this.dir.toString();
    };
    OrderBy.prototype.toString = function () {
        return this.field.canonicalString() + " (" + this.dir + ")";
    };
    OrderBy.prototype.isEqual = function (other) {
        return this.dir === other.dir && this.field.isEqual(other.field);
    };
    return OrderBy;
}());
var KEY_ORDERING_ASC = new OrderBy(FieldPath.keyField(), Direction.ASCENDING);
var KEY_ORDERING_DESC = new OrderBy(FieldPath.keyField(), Direction.DESCENDING);

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A version of a document in Firestore. This corresponds to the version
 * timestamp, such as update_time or read_time.
 */
var SnapshotVersion = /** @class */ (function () {
    function SnapshotVersion(timestamp) {
        this.timestamp = timestamp;
    }
    // TODO(b/34176344): Once we no longer need to use the old alpha protos,
    // delete this constructor and use a timestamp-backed version everywhere.
    SnapshotVersion.fromMicroseconds = function (value) {
        var seconds = Math.floor(value / 1e6);
        var nanos = (value % 1e6) * 1e3;
        return new SnapshotVersion(new Timestamp(seconds, nanos));
    };
    SnapshotVersion.fromTimestamp = function (value) {
        return new SnapshotVersion(value);
    };
    SnapshotVersion.forDeletedDoc = function () {
        return SnapshotVersion.MIN;
    };
    SnapshotVersion.prototype.compareTo = function (other) {
        return this.timestamp._compareTo(other.timestamp);
    };
    SnapshotVersion.prototype.isEqual = function (other) {
        return this.timestamp.isEqual(other.timestamp);
    };
    /** Returns a number representation of the version for use in spec tests. */
    SnapshotVersion.prototype.toMicroseconds = function () {
        // Convert to microseconds.
        return this.timestamp.seconds * 1e6 + this.timestamp.nanoseconds / 1000;
    };
    SnapshotVersion.prototype.toString = function () {
        return 'SnapshotVersion(' + this.timestamp.toString() + ')';
    };
    SnapshotVersion.prototype.toTimestamp = function () {
        return this.timestamp;
    };
    SnapshotVersion.MIN = new SnapshotVersion(new Timestamp(0, 0));
    return SnapshotVersion;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/** An enumeration of the different purposes we have for queries. */
var QueryPurpose;
(function (QueryPurpose) {
    /** A regular, normal query. */
    QueryPurpose[QueryPurpose["Listen"] = 0] = "Listen";
    /**
     * The query was used to refill a query after an existence filter mismatch.
     */
    QueryPurpose[QueryPurpose["ExistenceFilterMismatch"] = 1] = "ExistenceFilterMismatch";
    /** The query was used to resolve a limbo document. */
    QueryPurpose[QueryPurpose["LimboResolution"] = 2] = "LimboResolution";
})(QueryPurpose || (QueryPurpose = {}));
/**
 * An immutable set of metadata that the local store tracks for each query.
 */
var QueryData = /** @class */ (function () {
    function QueryData(
    /** The query being listened to. */
    query, 
    /**
     * The target ID to which the query corresponds; Assigned by the
     * LocalStore for user listens and by the SyncEngine for limbo watches.
     */
    targetId, 
    /** The purpose of the query. */
    purpose, 
    /** The sequence number of the last transaction during which this query data was modified */
    sequenceNumber, 
    /** The latest snapshot version seen for this target. */
    snapshotVersion, 
    /**
     * An opaque, server-assigned token that allows watching a query to be
     * resumed after disconnecting without retransmitting all the data that
     * matches the query. The resume token essentially identifies a point in
     * time from which the server should resume sending results.
     */
    resumeToken) {
        if (snapshotVersion === void 0) { snapshotVersion = SnapshotVersion.MIN; }
        if (resumeToken === void 0) { resumeToken = emptyByteString(); }
        this.query = query;
        this.targetId = targetId;
        this.purpose = purpose;
        this.sequenceNumber = sequenceNumber;
        this.snapshotVersion = snapshotVersion;
        this.resumeToken = resumeToken;
    }
    /**
     * Creates a new query data instance with an updated snapshot version and
     * resume token.
     */
    QueryData.prototype.copy = function (overwrite) {
        return new QueryData(this.query, this.targetId, this.purpose, overwrite.sequenceNumber === undefined
            ? this.sequenceNumber
            : overwrite.sequenceNumber, overwrite.snapshotVersion === undefined
            ? this.snapshotVersion
            : overwrite.snapshotVersion, overwrite.resumeToken === undefined
            ? this.resumeToken
            : overwrite.resumeToken);
    };
    QueryData.prototype.isEqual = function (other) {
        return (this.targetId === other.targetId &&
            this.purpose === other.purpose &&
            this.sequenceNumber === other.sequenceNumber &&
            this.snapshotVersion.isEqual(other.snapshotVersion) &&
            this.resumeToken === other.resumeToken &&
            this.query.isEqual(other.query));
    };
    return QueryData;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * SortedSet is an immutable (copy-on-write) collection that holds elements
 * in order specified by the provided comparator.
 *
 * NOTE: if provided comparator returns 0 for two elements, we consider them to
 * be equal!
 */
var SortedSet = /** @class */ (function () {
    function SortedSet(comparator) {
        this.comparator = comparator;
        this.data = new SortedMap(this.comparator);
    }
    /**
     * Creates a SortedSet from the keys of the map.
     * This is currently implemented as an O(n) copy.
     */
    SortedSet.fromMapKeys = function (map) {
        var keys = new SortedSet(map.comparator);
        map.forEach(function (key) {
            keys = keys.add(key);
        });
        return keys;
    };
    SortedSet.prototype.has = function (elem) {
        return this.data.get(elem) !== null;
    };
    SortedSet.prototype.first = function () {
        return this.data.minKey();
    };
    SortedSet.prototype.last = function () {
        return this.data.maxKey();
    };
    Object.defineProperty(SortedSet.prototype, "size", {
        get: function () {
            return this.data.size;
        },
        enumerable: true,
        configurable: true
    });
    SortedSet.prototype.indexOf = function (elem) {
        return this.data.indexOf(elem);
    };
    /** Iterates elements in order defined by "comparator" */
    SortedSet.prototype.forEach = function (cb) {
        this.data.inorderTraversal(function (k, v) {
            cb(k);
            return false;
        });
    };
    /** Iterates over `elem`s such that: range[0] <= elem < range[1]. */
    SortedSet.prototype.forEachInRange = function (range, cb) {
        var iter = this.data.getIteratorFrom(range[0]);
        while (iter.hasNext()) {
            var elem = iter.getNext();
            if (this.comparator(elem.key, range[1]) >= 0)
                return;
            cb(elem.key);
        }
    };
    /**
     * Iterates over `elem`s such that: start <= elem until false is returned.
     */
    SortedSet.prototype.forEachWhile = function (cb, start) {
        var iter;
        if (start !== undefined) {
            iter = this.data.getIteratorFrom(start);
        }
        else {
            iter = this.data.getIterator();
        }
        while (iter.hasNext()) {
            var elem = iter.getNext();
            var result = cb(elem.key);
            if (!result)
                return;
        }
    };
    /** Finds the least element greater than or equal to `elem`. */
    SortedSet.prototype.firstAfterOrEqual = function (elem) {
        var iter = this.data.getIteratorFrom(elem);
        return iter.hasNext() ? iter.getNext().key : null;
    };
    SortedSet.prototype.getIterator = function () {
        return new SortedSetIterator(this.data.getIterator());
    };
    SortedSet.prototype.getIteratorFrom = function (key) {
        return new SortedSetIterator(this.data.getIteratorFrom(key));
    };
    /** Inserts or updates an element */
    SortedSet.prototype.add = function (elem) {
        return this.copy(this.data.remove(elem).insert(elem, true));
    };
    /** Deletes an element */
    SortedSet.prototype.delete = function (elem) {
        if (!this.has(elem))
            return this;
        return this.copy(this.data.remove(elem));
    };
    SortedSet.prototype.isEmpty = function () {
        return this.data.isEmpty();
    };
    SortedSet.prototype.unionWith = function (other) {
        var result = this;
        other.forEach(function (elem) {
            result = result.add(elem);
        });
        return result;
    };
    SortedSet.prototype.isEqual = function (other) {
        if (!(other instanceof SortedSet))
            return false;
        if (this.size !== other.size)
            return false;
        var thisIt = this.data.getIterator();
        var otherIt = other.data.getIterator();
        while (thisIt.hasNext()) {
            var thisElem = thisIt.getNext().key;
            var otherElem = otherIt.getNext().key;
            if (this.comparator(thisElem, otherElem) !== 0)
                return false;
        }
        return true;
    };
    SortedSet.prototype.toArray = function () {
        var res = [];
        this.forEach(function (targetId) {
            res.push(targetId);
        });
        return res;
    };
    SortedSet.prototype.toString = function () {
        var result = [];
        this.forEach(function (elem) { return result.push(elem); });
        return 'SortedSet(' + result.toString() + ')';
    };
    SortedSet.prototype.copy = function (data) {
        var result = new SortedSet(this.comparator);
        result.data = data;
        return result;
    };
    return SortedSet;
}());
var SortedSetIterator = /** @class */ (function () {
    function SortedSetIterator(iter) {
        this.iter = iter;
    }
    SortedSetIterator.prototype.getNext = function () {
        return this.iter.getNext().key;
    };
    SortedSetIterator.prototype.hasNext = function () {
        return this.iter.hasNext();
    };
    return SortedSetIterator;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Provides a set of fields that can be used to partially patch a document.
 * FieldMask is used in conjunction with ObjectValue.
 * Examples:
 *   foo - Overwrites foo entirely with the provided value. If foo is not
 *         present in the companion ObjectValue, the field is deleted.
 *   foo.bar - Overwrites only the field bar of the object foo.
 *             If foo is not an object, foo is replaced with an object
 *             containing foo
 */
var FieldMask = /** @class */ (function () {
    function FieldMask(fields) {
        this.fields = fields;
        // TODO(dimond): validation of FieldMask
    }
    FieldMask.fromSet = function (fields) {
        return new FieldMask(fields);
    };
    FieldMask.fromArray = function (fields) {
        var fieldsAsSet = new SortedSet(FieldPath.comparator);
        fields.forEach(function (fieldPath) { return (fieldsAsSet = fieldsAsSet.add(fieldPath)); });
        return new FieldMask(fieldsAsSet);
    };
    /**
     * Verifies that `fieldPath` is included by at least one field in this field
     * mask.
     *
     * This is an O(n) operation, where `n` is the size of the field mask.
     */
    FieldMask.prototype.covers = function (fieldPath) {
        var found = false;
        this.fields.forEach(function (fieldMaskPath) {
            if (fieldMaskPath.isPrefixOf(fieldPath)) {
                found = true;
            }
        });
        return found;
    };
    /**
     * Applies this field mask to the provided object value and returns an object
     * that only contains fields that are specified in both the input object and
     * this field mask.
     */
    FieldMask.prototype.applyTo = function (data) {
        var filteredObject = ObjectValue.EMPTY;
        this.fields.forEach(function (fieldMaskPath) {
            if (fieldMaskPath.isEmpty()) {
                return data;
            }
            else {
                var newValue = data.field(fieldMaskPath);
                if (newValue !== undefined) {
                    filteredObject = filteredObject.set(fieldMaskPath, newValue);
                }
            }
        });
        return filteredObject;
    };
    FieldMask.prototype.isEqual = function (other) {
        return this.fields.isEqual(other.fields);
    };
    return FieldMask;
}());
/** A field path and the TransformOperation to perform upon it. */
var FieldTransform = /** @class */ (function () {
    function FieldTransform(field, transform) {
        this.field = field;
        this.transform = transform;
    }
    Object.defineProperty(FieldTransform.prototype, "isIdempotent", {
        /** Whether this field transform is idempotent. */
        get: function () {
            return this.transform.isIdempotent;
        },
        enumerable: true,
        configurable: true
    });
    FieldTransform.prototype.isEqual = function (other) {
        return (this.field.isEqual(other.field) && this.transform.isEqual(other.transform));
    };
    return FieldTransform;
}());
/** The result of successfully applying a mutation to the backend. */
var MutationResult = /** @class */ (function () {
    function MutationResult(
    /**
     * The version at which the mutation was committed:
     *
     * - For most operations, this is the updateTime in the WriteResult.
     * - For deletes, the commitTime of the WriteResponse (because deletes are
     *   not stored and have no updateTime).
     *
     * Note that these versions can be different: No-op writes will not change
     * the updateTime even though the commitTime advances.
     */
    version, 
    /**
     * The resulting fields returned from the backend after a
     * TransformMutation has been committed. Contains one FieldValue for each
     * FieldTransform that was in the mutation.
     *
     * Will be null if the mutation was not a TransformMutation.
     */
    transformResults) {
        this.version = version;
        this.transformResults = transformResults;
    }
    return MutationResult;
}());
var MutationType;
(function (MutationType) {
    MutationType[MutationType["Set"] = 0] = "Set";
    MutationType[MutationType["Patch"] = 1] = "Patch";
    MutationType[MutationType["Transform"] = 2] = "Transform";
    MutationType[MutationType["Delete"] = 3] = "Delete";
})(MutationType || (MutationType = {}));
/**
 * Encodes a precondition for a mutation. This follows the model that the
 * backend accepts with the special case of an explicit "empty" precondition
 * (meaning no precondition).
 */
var Precondition = /** @class */ (function () {
    function Precondition(updateTime, exists) {
        this.updateTime = updateTime;
        this.exists = exists;
        assert(updateTime === undefined || exists === undefined, 'Precondition can specify "exists" or "updateTime" but not both');
    }
    /** Creates a new Precondition with an exists flag. */
    Precondition.exists = function (exists) {
        return new Precondition(undefined, exists);
    };
    /** Creates a new Precondition based on a version a document exists at. */
    Precondition.updateTime = function (version) {
        return new Precondition(version);
    };
    Object.defineProperty(Precondition.prototype, "isNone", {
        /** Returns whether this Precondition is empty. */
        get: function () {
            return this.updateTime === undefined && this.exists === undefined;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Returns true if the preconditions is valid for the given document
     * (or null if no document is available).
     */
    Precondition.prototype.isValidFor = function (maybeDoc) {
        if (this.updateTime !== undefined) {
            return (maybeDoc instanceof Document &&
                maybeDoc.version.isEqual(this.updateTime));
        }
        else if (this.exists !== undefined) {
            return this.exists === maybeDoc instanceof Document;
        }
        else {
            assert(this.isNone, 'Precondition should be empty');
            return true;
        }
    };
    Precondition.prototype.isEqual = function (other) {
        return (equals(this.updateTime, other.updateTime) &&
            this.exists === other.exists);
    };
    Precondition.NONE = new Precondition();
    return Precondition;
}());
/**
 * A mutation describes a self-contained change to a document. Mutations can
 * create, replace, delete, and update subsets of documents.
 *
 * Mutations not only act on the value of the document but also it version.
 *
 * For local mutations (mutations that haven't been committed yet), we preserve
 * the existing version for Set, Patch, and Transform mutations. For Delete
 * mutations, we reset the version to 0.
 *
 * Here's the expected transition table.
 *
 * MUTATION           APPLIED TO            RESULTS IN
 *
 * SetMutation        Document(v3)          Document(v3)
 * SetMutation        NoDocument(v3)        Document(v0)
 * SetMutation        null                  Document(v0)
 * PatchMutation      Document(v3)          Document(v3)
 * PatchMutation      NoDocument(v3)        NoDocument(v3)
 * PatchMutation      null                  null
 * TransformMutation  Document(v3)          Document(v3)
 * TransformMutation  NoDocument(v3)        NoDocument(v3)
 * TransformMutation  null                  null
 * DeleteMutation     Document(v3)          NoDocument(v0)
 * DeleteMutation     NoDocument(v3)        NoDocument(v0)
 * DeleteMutation     null                  NoDocument(v0)
 *
 * For acknowledged mutations, we use the updateTime of the WriteResponse as
 * the resulting version for Set, Patch, and Transform mutations. As deletes
 * have no explicit update time, we use the commitTime of the WriteResponse for
 * Delete mutations.
 *
 * If a mutation is acknowledged by the backend but fails the precondition check
 * locally, we return an `UnknownDocument` and rely on Watch to send us the
 * updated version.
 *
 * Note that TransformMutations don't create Documents (in the case of being
 * applied to a NoDocument), even though they would on the backend. This is
 * because the client always combines the TransformMutation with a SetMutation
 * or PatchMutation and we only want to apply the transform if the prior
 * mutation resulted in a Document (always true for a SetMutation, but not
 * necessarily for a PatchMutation).
 *
 * ## Subclassing Notes
 *
 * Subclasses of Mutation need to implement applyToRemoteDocument() and
 * applyToLocalView() to implement the actual behavior of applying the mutation
 * to some source document.
 */
var Mutation = /** @class */ (function () {
    function Mutation() {
    }
    Mutation.prototype.verifyKeyMatches = function (maybeDoc) {
        if (maybeDoc != null) {
            assert(maybeDoc.key.isEqual(this.key), 'Can only apply a mutation to a document with the same key');
        }
    };
    /**
     * Returns the version from the given document for use as the result of a
     * mutation. Mutations are defined to return the version of the base document
     * only if it is an existing document. Deleted and unknown documents have a
     * post-mutation version of SnapshotVersion.MIN.
     */
    Mutation.getPostMutationVersion = function (maybeDoc) {
        if (maybeDoc instanceof Document) {
            return maybeDoc.version;
        }
        else {
            return SnapshotVersion.MIN;
        }
    };
    return Mutation;
}());
/**
 * A mutation that creates or replaces the document at the given key with the
 * object value contents.
 */
var SetMutation = /** @class */ (function (_super) {
    tslib_1.__extends(SetMutation, _super);
    function SetMutation(key, value, precondition) {
        var _this = _super.call(this) || this;
        _this.key = key;
        _this.value = value;
        _this.precondition = precondition;
        _this.type = MutationType.Set;
        return _this;
    }
    SetMutation.prototype.applyToRemoteDocument = function (maybeDoc, mutationResult) {
        this.verifyKeyMatches(maybeDoc);
        assert(mutationResult.transformResults == null, 'Transform results received by SetMutation.');
        // Unlike applyToLocalView, if we're applying a mutation to a remote
        // document the server has accepted the mutation so the precondition must
        // have held.
        var version = mutationResult.version;
        return new Document(this.key, version, this.value, {
            hasCommittedMutations: true
        });
    };
    SetMutation.prototype.applyToLocalView = function (maybeDoc, baseDoc, localWriteTime) {
        this.verifyKeyMatches(maybeDoc);
        if (!this.precondition.isValidFor(maybeDoc)) {
            return maybeDoc;
        }
        var version = Mutation.getPostMutationVersion(maybeDoc);
        return new Document(this.key, version, this.value, {
            hasLocalMutations: true
        });
    };
    Object.defineProperty(SetMutation.prototype, "isIdempotent", {
        get: function () {
            return true;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(SetMutation.prototype, "fieldMask", {
        get: function () {
            return null;
        },
        enumerable: true,
        configurable: true
    });
    SetMutation.prototype.isEqual = function (other) {
        return (other instanceof SetMutation &&
            this.key.isEqual(other.key) &&
            this.value.isEqual(other.value) &&
            this.precondition.isEqual(other.precondition));
    };
    return SetMutation;
}(Mutation));
/**
 * A mutation that modifies fields of the document at the given key with the
 * given values. The values are applied through a field mask:
 *
 *  * When a field is in both the mask and the values, the corresponding field
 *    is updated.
 *  * When a field is in neither the mask nor the values, the corresponding
 *    field is unmodified.
 *  * When a field is in the mask but not in the values, the corresponding field
 *    is deleted.
 *  * When a field is not in the mask but is in the values, the values map is
 *    ignored.
 */
var PatchMutation = /** @class */ (function (_super) {
    tslib_1.__extends(PatchMutation, _super);
    function PatchMutation(key, data, fieldMask, precondition) {
        var _this = _super.call(this) || this;
        _this.key = key;
        _this.data = data;
        _this.fieldMask = fieldMask;
        _this.precondition = precondition;
        _this.type = MutationType.Patch;
        return _this;
    }
    PatchMutation.prototype.applyToRemoteDocument = function (maybeDoc, mutationResult) {
        this.verifyKeyMatches(maybeDoc);
        assert(mutationResult.transformResults == null, 'Transform results received by PatchMutation.');
        if (!this.precondition.isValidFor(maybeDoc)) {
            // Since the mutation was not rejected, we know that the  precondition
            // matched on the backend. We therefore must not have the expected version
            // of the document in our cache and return an UnknownDocument with the
            // known updateTime.
            return new UnknownDocument(this.key, mutationResult.version);
        }
        var newData = this.patchDocument(maybeDoc);
        return new Document(this.key, mutationResult.version, newData, {
            hasCommittedMutations: true
        });
    };
    PatchMutation.prototype.applyToLocalView = function (maybeDoc, baseDoc, localWriteTime) {
        this.verifyKeyMatches(maybeDoc);
        if (!this.precondition.isValidFor(maybeDoc)) {
            return maybeDoc;
        }
        var version = Mutation.getPostMutationVersion(maybeDoc);
        var newData = this.patchDocument(maybeDoc);
        return new Document(this.key, version, newData, {
            hasLocalMutations: true
        });
    };
    Object.defineProperty(PatchMutation.prototype, "isIdempotent", {
        get: function () {
            return true;
        },
        enumerable: true,
        configurable: true
    });
    PatchMutation.prototype.isEqual = function (other) {
        return (other instanceof PatchMutation &&
            this.key.isEqual(other.key) &&
            this.fieldMask.isEqual(other.fieldMask) &&
            this.precondition.isEqual(other.precondition));
    };
    /**
     * Patches the data of document if available or creates a new document. Note
     * that this does not check whether or not the precondition of this patch
     * holds.
     */
    PatchMutation.prototype.patchDocument = function (maybeDoc) {
        var data;
        if (maybeDoc instanceof Document) {
            data = maybeDoc.data;
        }
        else {
            data = ObjectValue.EMPTY;
        }
        return this.patchObject(data);
    };
    PatchMutation.prototype.patchObject = function (data) {
        var _this = this;
        this.fieldMask.fields.forEach(function (fieldPath) {
            if (!fieldPath.isEmpty()) {
                var newValue = _this.data.field(fieldPath);
                if (newValue !== undefined) {
                    data = data.set(fieldPath, newValue);
                }
                else {
                    data = data.delete(fieldPath);
                }
            }
        });
        return data;
    };
    return PatchMutation;
}(Mutation));
/**
 * A mutation that modifies specific fields of the document with transform
 * operations. Currently the only supported transform is a server timestamp, but
 * IP Address, increment(n), etc. could be supported in the future.
 *
 * It is somewhat similar to a PatchMutation in that it patches specific fields
 * and has no effect when applied to a null or NoDocument (see comment on
 * Mutation for rationale).
 */
var TransformMutation = /** @class */ (function (_super) {
    tslib_1.__extends(TransformMutation, _super);
    function TransformMutation(key, fieldTransforms) {
        var _this = _super.call(this) || this;
        _this.key = key;
        _this.fieldTransforms = fieldTransforms;
        _this.type = MutationType.Transform;
        // NOTE: We set a precondition of exists: true as a safety-check, since we
        // always combine TransformMutations with a SetMutation or PatchMutation which
        // (if successful) should end up with an existing document.
        _this.precondition = Precondition.exists(true);
        return _this;
    }
    TransformMutation.prototype.applyToRemoteDocument = function (maybeDoc, mutationResult) {
        this.verifyKeyMatches(maybeDoc);
        assert(mutationResult.transformResults != null, 'Transform results missing for TransformMutation.');
        if (!this.precondition.isValidFor(maybeDoc)) {
            // Since the mutation was not rejected, we know that the  precondition
            // matched on the backend. We therefore must not have the expected version
            // of the document in our cache and return an UnknownDocument with the
            // known updateTime.
            return new UnknownDocument(this.key, mutationResult.version);
        }
        var doc = this.requireDocument(maybeDoc);
        var transformResults = this.serverTransformResults(maybeDoc, mutationResult.transformResults);
        var version = mutationResult.version;
        var newData = this.transformObject(doc.data, transformResults);
        return new Document(this.key, version, newData, {
            hasCommittedMutations: true
        });
    };
    TransformMutation.prototype.applyToLocalView = function (maybeDoc, baseDoc, localWriteTime) {
        this.verifyKeyMatches(maybeDoc);
        if (!this.precondition.isValidFor(maybeDoc)) {
            return maybeDoc;
        }
        var doc = this.requireDocument(maybeDoc);
        var transformResults = this.localTransformResults(localWriteTime, baseDoc);
        var newData = this.transformObject(doc.data, transformResults);
        return new Document(this.key, doc.version, newData, {
            hasLocalMutations: true
        });
    };
    Object.defineProperty(TransformMutation.prototype, "isIdempotent", {
        get: function () {
            for (var _i = 0, _a = this.fieldTransforms; _i < _a.length; _i++) {
                var fieldTransform = _a[_i];
                if (!fieldTransform.isIdempotent) {
                    return false;
                }
            }
            return true;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(TransformMutation.prototype, "fieldMask", {
        get: function () {
            var fieldMask = new SortedSet(FieldPath.comparator);
            this.fieldTransforms.forEach(function (transform) { return (fieldMask = fieldMask.add(transform.field)); });
            return new FieldMask(fieldMask);
        },
        enumerable: true,
        configurable: true
    });
    TransformMutation.prototype.isEqual = function (other) {
        return (other instanceof TransformMutation &&
            this.key.isEqual(other.key) &&
            arrayEquals(this.fieldTransforms, other.fieldTransforms) &&
            this.precondition.isEqual(other.precondition));
    };
    /**
     * Asserts that the given MaybeDocument is actually a Document and verifies
     * that it matches the key for this mutation. Since we only support
     * transformations with precondition exists this method is guaranteed to be
     * safe.
     */
    TransformMutation.prototype.requireDocument = function (maybeDoc) {
        assert(maybeDoc instanceof Document, 'Unknown MaybeDocument type ' + maybeDoc);
        var doc = maybeDoc;
        assert(doc.key.isEqual(this.key), 'Can only transform a document with the same key');
        return doc;
    };
    /**
     * Creates a list of "transform results" (a transform result is a field value
     * representing the result of applying a transform) for use after a
     * TransformMutation has been acknowledged by the server.
     *
     * @param baseDoc The document prior to applying this mutation batch.
     * @param serverTransformResults The transform results received by the server.
     * @return The transform results list.
     */
    TransformMutation.prototype.serverTransformResults = function (baseDoc, serverTransformResults) {
        var transformResults = [];
        assert(this.fieldTransforms.length === serverTransformResults.length, "server transform result count (" + serverTransformResults.length + ") " +
            ("should match field transform count (" + this.fieldTransforms.length + ")"));
        for (var i = 0; i < serverTransformResults.length; i++) {
            var fieldTransform = this.fieldTransforms[i];
            var transform = fieldTransform.transform;
            var previousValue = null;
            if (baseDoc instanceof Document) {
                previousValue = baseDoc.field(fieldTransform.field) || null;
            }
            transformResults.push(transform.applyToRemoteDocument(previousValue, serverTransformResults[i]));
        }
        return transformResults;
    };
    /**
     * Creates a list of "transform results" (a transform result is a field value
     * representing the result of applying a transform) for use when applying a
     * TransformMutation locally.
     *
     * @param localWriteTime The local time of the transform mutation (used to
     *     generate ServerTimestampValues).
     * @param baseDoc The document prior to applying this mutation batch.
     * @return The transform results list.
     */
    TransformMutation.prototype.localTransformResults = function (localWriteTime, baseDoc) {
        var transformResults = [];
        for (var _i = 0, _a = this.fieldTransforms; _i < _a.length; _i++) {
            var fieldTransform = _a[_i];
            var transform = fieldTransform.transform;
            var previousValue = null;
            if (baseDoc instanceof Document) {
                previousValue = baseDoc.field(fieldTransform.field) || null;
            }
            transformResults.push(transform.applyToLocalView(previousValue, localWriteTime));
        }
        return transformResults;
    };
    TransformMutation.prototype.transformObject = function (data, transformResults) {
        assert(transformResults.length === this.fieldTransforms.length, 'TransformResults length mismatch.');
        for (var i = 0; i < this.fieldTransforms.length; i++) {
            var fieldTransform = this.fieldTransforms[i];
            var fieldPath = fieldTransform.field;
            data = data.set(fieldPath, transformResults[i]);
        }
        return data;
    };
    return TransformMutation;
}(Mutation));
/** A mutation that deletes the document at the given key. */
var DeleteMutation = /** @class */ (function (_super) {
    tslib_1.__extends(DeleteMutation, _super);
    function DeleteMutation(key, precondition) {
        var _this = _super.call(this) || this;
        _this.key = key;
        _this.precondition = precondition;
        _this.type = MutationType.Delete;
        return _this;
    }
    DeleteMutation.prototype.applyToRemoteDocument = function (maybeDoc, mutationResult) {
        this.verifyKeyMatches(maybeDoc);
        assert(mutationResult.transformResults == null, 'Transform results received by DeleteMutation.');
        // Unlike applyToLocalView, if we're applying a mutation to a remote
        // document the server has accepted the mutation so the precondition must
        // have held.
        return new NoDocument(this.key, mutationResult.version, {
            hasCommittedMutations: true
        });
    };
    DeleteMutation.prototype.applyToLocalView = function (maybeDoc, baseDoc, localWriteTime) {
        this.verifyKeyMatches(maybeDoc);
        if (!this.precondition.isValidFor(maybeDoc)) {
            return maybeDoc;
        }
        if (maybeDoc) {
            assert(maybeDoc.key.isEqual(this.key), 'Can only apply mutation to document with same key');
        }
        return new NoDocument(this.key, SnapshotVersion.forDeletedDoc());
    };
    DeleteMutation.prototype.isEqual = function (other) {
        return (other instanceof DeleteMutation &&
            this.key.isEqual(other.key) &&
            this.precondition.isEqual(other.precondition));
    };
    Object.defineProperty(DeleteMutation.prototype, "isIdempotent", {
        get: function () {
            return true;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(DeleteMutation.prototype, "fieldMask", {
        get: function () {
            return null;
        },
        enumerable: true,
        configurable: true
    });
    return DeleteMutation;
}(Mutation));

/**
 * @license
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/** Transforms a value into a server-generated timestamp. */
var ServerTimestampTransform = /** @class */ (function () {
    function ServerTimestampTransform() {
        this.isIdempotent = true;
    }
    ServerTimestampTransform.prototype.applyToLocalView = function (previousValue, localWriteTime) {
        return new ServerTimestampValue(localWriteTime, previousValue);
    };
    ServerTimestampTransform.prototype.applyToRemoteDocument = function (previousValue, transformResult) {
        return transformResult;
    };
    ServerTimestampTransform.prototype.isEqual = function (other) {
        return other instanceof ServerTimestampTransform;
    };
    ServerTimestampTransform.instance = new ServerTimestampTransform();
    return ServerTimestampTransform;
}());
/** Transforms an array value via a union operation. */
var ArrayUnionTransformOperation = /** @class */ (function () {
    function ArrayUnionTransformOperation(elements) {
        this.elements = elements;
        this.isIdempotent = true;
    }
    ArrayUnionTransformOperation.prototype.applyToLocalView = function (previousValue, localWriteTime) {
        return this.apply(previousValue);
    };
    ArrayUnionTransformOperation.prototype.applyToRemoteDocument = function (previousValue, transformResult) {
        // The server just sends null as the transform result for array operations,
        // so we have to calculate a result the same as we do for local
        // applications.
        return this.apply(previousValue);
    };
    ArrayUnionTransformOperation.prototype.apply = function (previousValue) {
        var result = coercedFieldValuesArray(previousValue);
        var _loop_1 = function (toUnion) {
            if (!result.find(function (element) { return element.isEqual(toUnion); })) {
                result.push(toUnion);
            }
        };
        for (var _i = 0, _a = this.elements; _i < _a.length; _i++) {
            var toUnion = _a[_i];
            _loop_1(toUnion);
        }
        return new ArrayValue(result);
    };
    ArrayUnionTransformOperation.prototype.isEqual = function (other) {
        return (other instanceof ArrayUnionTransformOperation &&
            arrayEquals(other.elements, this.elements));
    };
    return ArrayUnionTransformOperation;
}());
/** Transforms an array value via a remove operation. */
var ArrayRemoveTransformOperation = /** @class */ (function () {
    function ArrayRemoveTransformOperation(elements) {
        this.elements = elements;
        this.isIdempotent = true;
    }
    ArrayRemoveTransformOperation.prototype.applyToLocalView = function (previousValue, localWriteTime) {
        return this.apply(previousValue);
    };
    ArrayRemoveTransformOperation.prototype.applyToRemoteDocument = function (previousValue, transformResult) {
        // The server just sends null as the transform result for array operations,
        // so we have to calculate a result the same as we do for local
        // applications.
        return this.apply(previousValue);
    };
    ArrayRemoveTransformOperation.prototype.apply = function (previousValue) {
        var result = coercedFieldValuesArray(previousValue);
        var _loop_2 = function (toRemove) {
            result = result.filter(function (element) { return !element.isEqual(toRemove); });
        };
        for (var _i = 0, _a = this.elements; _i < _a.length; _i++) {
            var toRemove = _a[_i];
            _loop_2(toRemove);
        }
        return new ArrayValue(result);
    };
    ArrayRemoveTransformOperation.prototype.isEqual = function (other) {
        return (other instanceof ArrayRemoveTransformOperation &&
            arrayEquals(other.elements, this.elements));
    };
    return ArrayRemoveTransformOperation;
}());
/**
 * Implements the backend semantics for locally computed NUMERIC_ADD (increment)
 * transforms. Converts all field values to integers or doubles, but unlike the
 * backend does not cap integer values at 2^63. Instead, JavaScript number
 * arithmetic is used and precision loss can occur for values greater than 2^53.
 */
var NumericIncrementTransformOperation = /** @class */ (function () {
    function NumericIncrementTransformOperation(operand) {
        this.operand = operand;
        this.isIdempotent = false;
    }
    NumericIncrementTransformOperation.prototype.applyToLocalView = function (previousValue, localWriteTime) {
        // PORTING NOTE: Since JavaScript's integer arithmetic is limited to 53 bit
        // precision and resolves overflows by reducing precision, we do not
        // manually cap overflows at 2^63.
        // Return an integer value iff the previous value and the operand is an
        // integer.
        if (previousValue instanceof IntegerValue &&
            this.operand instanceof IntegerValue) {
            var sum = previousValue.internalValue + this.operand.internalValue;
            return new IntegerValue(sum);
        }
        else if (previousValue instanceof NumberValue) {
            var sum = previousValue.internalValue + this.operand.internalValue;
            return new DoubleValue(sum);
        }
        else {
            // If the existing value is not a number, use the value of the transform as
            // the new base value.
            return this.operand;
        }
    };
    NumericIncrementTransformOperation.prototype.applyToRemoteDocument = function (previousValue, transformResult) {
        assert(transformResult !== null, "Didn't receive transformResult for NUMERIC_ADD transform");
        return transformResult;
    };
    NumericIncrementTransformOperation.prototype.isEqual = function (other) {
        return (other instanceof NumericIncrementTransformOperation &&
            this.operand.isEqual(other.operand));
    };
    return NumericIncrementTransformOperation;
}());
function coercedFieldValuesArray(value) {
    if (value instanceof ArrayValue) {
        return value.internalValue.slice();
    }
    else {
        // coerce to empty array.
        return [];
    }
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var ExistenceFilter = /** @class */ (function () {
    // TODO(b/33078163): just use simplest form of existence filter for now
    function ExistenceFilter(count) {
        this.count = count;
    }
    ExistenceFilter.prototype.isEqual = function (other) {
        return other && other.count === this.count;
    };
    return ExistenceFilter;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Error Codes describing the different ways GRPC can fail. These are copied
 * directly from GRPC's sources here:
 *
 * https://github.com/grpc/grpc/blob/bceec94ea4fc5f0085d81235d8e1c06798dc341a/include/grpc%2B%2B/impl/codegen/status_code_enum.h
 *
 * Important! The names of these identifiers matter because the string forms
 * are used for reverse lookups from the webchannel stream. Do NOT change the
 * names of these identifiers.
 */
var RpcCode;
(function (RpcCode) {
    RpcCode[RpcCode["OK"] = 0] = "OK";
    RpcCode[RpcCode["CANCELLED"] = 1] = "CANCELLED";
    RpcCode[RpcCode["UNKNOWN"] = 2] = "UNKNOWN";
    RpcCode[RpcCode["INVALID_ARGUMENT"] = 3] = "INVALID_ARGUMENT";
    RpcCode[RpcCode["DEADLINE_EXCEEDED"] = 4] = "DEADLINE_EXCEEDED";
    RpcCode[RpcCode["NOT_FOUND"] = 5] = "NOT_FOUND";
    RpcCode[RpcCode["ALREADY_EXISTS"] = 6] = "ALREADY_EXISTS";
    RpcCode[RpcCode["PERMISSION_DENIED"] = 7] = "PERMISSION_DENIED";
    RpcCode[RpcCode["UNAUTHENTICATED"] = 16] = "UNAUTHENTICATED";
    RpcCode[RpcCode["RESOURCE_EXHAUSTED"] = 8] = "RESOURCE_EXHAUSTED";
    RpcCode[RpcCode["FAILED_PRECONDITION"] = 9] = "FAILED_PRECONDITION";
    RpcCode[RpcCode["ABORTED"] = 10] = "ABORTED";
    RpcCode[RpcCode["OUT_OF_RANGE"] = 11] = "OUT_OF_RANGE";
    RpcCode[RpcCode["UNIMPLEMENTED"] = 12] = "UNIMPLEMENTED";
    RpcCode[RpcCode["INTERNAL"] = 13] = "INTERNAL";
    RpcCode[RpcCode["UNAVAILABLE"] = 14] = "UNAVAILABLE";
    RpcCode[RpcCode["DATA_LOSS"] = 15] = "DATA_LOSS";
})(RpcCode || (RpcCode = {}));
/**
 * Determines whether an error code represents a permanent error when received
 * in response to a non-write operation.
 *
 * See isPermanentWriteError for classifying write errors.
 */
function isPermanentError(code) {
    switch (code) {
        case Code.OK:
            return fail('Treated status OK as error');
        case Code.CANCELLED:
        case Code.UNKNOWN:
        case Code.DEADLINE_EXCEEDED:
        case Code.RESOURCE_EXHAUSTED:
        case Code.INTERNAL:
        case Code.UNAVAILABLE:
        // Unauthenticated means something went wrong with our token and we need
        // to retry with new credentials which will happen automatically.
        case Code.UNAUTHENTICATED:
            return false;
        case Code.INVALID_ARGUMENT:
        case Code.NOT_FOUND:
        case Code.ALREADY_EXISTS:
        case Code.PERMISSION_DENIED:
        case Code.FAILED_PRECONDITION:
        // Aborted might be retried in some scenarios, but that is dependant on
        // the context and should handled individually by the calling code.
        // See https://cloud.google.com/apis/design/errors.
        case Code.ABORTED:
        case Code.OUT_OF_RANGE:
        case Code.UNIMPLEMENTED:
        case Code.DATA_LOSS:
            return true;
        default:
            return fail('Unknown status code: ' + code);
    }
}
/**
 * Determines whether an error code represents a permanent error when received
 * in response to a write operation.
 *
 * Write operations must be handled specially because as of b/119437764, ABORTED
 * errors on the write stream should be retried too (even though ABORTED errors
 * are not generally retryable).
 *
 * Note that during the initial handshake on the write stream an ABORTED error
 * signals that we should discard our stream token (i.e. it is permanent). This
 * means a handshake error should be classified with isPermanentError, above.
 */
function isPermanentWriteError(code) {
    return isPermanentError(code) && code !== Code.ABORTED;
}
/**
 * Maps an error Code from GRPC status code number, like 0, 1, or 14. These
 * are not the same as HTTP status codes.
 *
 * @returns The Code equivalent to the given GRPC status code. Fails if there
 *     is no match.
 */
function mapCodeFromRpcCode(code) {
    if (code === undefined) {
        // This shouldn't normally happen, but in certain error cases (like trying
        // to send invalid proto messages) we may get an error with no GRPC code.
        error('GRPC error has no .code');
        return Code.UNKNOWN;
    }
    switch (code) {
        case RpcCode.OK:
            return Code.OK;
        case RpcCode.CANCELLED:
            return Code.CANCELLED;
        case RpcCode.UNKNOWN:
            return Code.UNKNOWN;
        case RpcCode.DEADLINE_EXCEEDED:
            return Code.DEADLINE_EXCEEDED;
        case RpcCode.RESOURCE_EXHAUSTED:
            return Code.RESOURCE_EXHAUSTED;
        case RpcCode.INTERNAL:
            return Code.INTERNAL;
        case RpcCode.UNAVAILABLE:
            return Code.UNAVAILABLE;
        case RpcCode.UNAUTHENTICATED:
            return Code.UNAUTHENTICATED;
        case RpcCode.INVALID_ARGUMENT:
            return Code.INVALID_ARGUMENT;
        case RpcCode.NOT_FOUND:
            return Code.NOT_FOUND;
        case RpcCode.ALREADY_EXISTS:
            return Code.ALREADY_EXISTS;
        case RpcCode.PERMISSION_DENIED:
            return Code.PERMISSION_DENIED;
        case RpcCode.FAILED_PRECONDITION:
            return Code.FAILED_PRECONDITION;
        case RpcCode.ABORTED:
            return Code.ABORTED;
        case RpcCode.OUT_OF_RANGE:
            return Code.OUT_OF_RANGE;
        case RpcCode.UNIMPLEMENTED:
            return Code.UNIMPLEMENTED;
        case RpcCode.DATA_LOSS:
            return Code.DATA_LOSS;
        default:
            return fail('Unknown status code: ' + code);
    }
}
/**
 * Maps an RPC code from a Code. This is the reverse operation from
 * mapCodeFromRpcCode and should really only be used in tests.
 */
function mapRpcCodeFromCode(code) {
    if (code === undefined) {
        return RpcCode.OK;
    }
    switch (code) {
        case Code.OK:
            return RpcCode.OK;
        case Code.CANCELLED:
            return RpcCode.CANCELLED;
        case Code.UNKNOWN:
            return RpcCode.UNKNOWN;
        case Code.DEADLINE_EXCEEDED:
            return RpcCode.DEADLINE_EXCEEDED;
        case Code.RESOURCE_EXHAUSTED:
            return RpcCode.RESOURCE_EXHAUSTED;
        case Code.INTERNAL:
            return RpcCode.INTERNAL;
        case Code.UNAVAILABLE:
            return RpcCode.UNAVAILABLE;
        case Code.UNAUTHENTICATED:
            return RpcCode.UNAUTHENTICATED;
        case Code.INVALID_ARGUMENT:
            return RpcCode.INVALID_ARGUMENT;
        case Code.NOT_FOUND:
            return RpcCode.NOT_FOUND;
        case Code.ALREADY_EXISTS:
            return RpcCode.ALREADY_EXISTS;
        case Code.PERMISSION_DENIED:
            return RpcCode.PERMISSION_DENIED;
        case Code.FAILED_PRECONDITION:
            return RpcCode.FAILED_PRECONDITION;
        case Code.ABORTED:
            return RpcCode.ABORTED;
        case Code.OUT_OF_RANGE:
            return RpcCode.OUT_OF_RANGE;
        case Code.UNIMPLEMENTED:
            return RpcCode.UNIMPLEMENTED;
        case Code.DATA_LOSS:
            return RpcCode.DATA_LOSS;
        default:
            return fail('Unknown status code: ' + code);
    }
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var EMPTY_MAYBE_DOCUMENT_MAP = new SortedMap(DocumentKey.comparator);
function maybeDocumentMap() {
    return EMPTY_MAYBE_DOCUMENT_MAP;
}
function nullableMaybeDocumentMap() {
    return maybeDocumentMap();
}
var EMPTY_DOCUMENT_MAP = new SortedMap(DocumentKey.comparator);
function documentMap() {
    return EMPTY_DOCUMENT_MAP;
}
var EMPTY_DOCUMENT_VERSION_MAP = new SortedMap(DocumentKey.comparator);
function documentVersionMap() {
    return EMPTY_DOCUMENT_VERSION_MAP;
}
var EMPTY_DOCUMENT_KEY_SET = new SortedSet(DocumentKey.comparator);
function documentKeySet() {
    var keys = [];
    for (var _i = 0; _i < arguments.length; _i++) {
        keys[_i] = arguments[_i];
    }
    var set = EMPTY_DOCUMENT_KEY_SET;
    for (var _a = 0, keys_1 = keys; _a < keys_1.length; _a++) {
        var key = keys_1[_a];
        set = set.add(key);
    }
    return set;
}
var EMPTY_TARGET_ID_SET = new SortedSet(primitiveComparator);
function targetIdSet() {
    return EMPTY_TARGET_ID_SET;
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * DocumentSet is an immutable (copy-on-write) collection that holds documents
 * in order specified by the provided comparator. We always add a document key
 * comparator on top of what is provided to guarantee document equality based on
 * the key.
 */
var DocumentSet = /** @class */ (function () {
    /** The default ordering is by key if the comparator is omitted */
    function DocumentSet(comp) {
        // We are adding document key comparator to the end as it's the only
        // guaranteed unique property of a document.
        if (comp) {
            this.comparator = function (d1, d2) {
                return comp(d1, d2) || DocumentKey.comparator(d1.key, d2.key);
            };
        }
        else {
            this.comparator = function (d1, d2) {
                return DocumentKey.comparator(d1.key, d2.key);
            };
        }
        this.keyedMap = documentMap();
        this.sortedSet = new SortedMap(this.comparator);
    }
    /**
     * Returns an empty copy of the existing DocumentSet, using the same
     * comparator.
     */
    DocumentSet.emptySet = function (oldSet) {
        return new DocumentSet(oldSet.comparator);
    };
    DocumentSet.prototype.has = function (key) {
        return this.keyedMap.get(key) != null;
    };
    DocumentSet.prototype.get = function (key) {
        return this.keyedMap.get(key);
    };
    DocumentSet.prototype.first = function () {
        return this.sortedSet.minKey();
    };
    DocumentSet.prototype.last = function () {
        return this.sortedSet.maxKey();
    };
    DocumentSet.prototype.isEmpty = function () {
        return this.sortedSet.isEmpty();
    };
    /**
     * Returns the index of the provided key in the document set, or -1 if the
     * document key is not present in the set;
     */
    DocumentSet.prototype.indexOf = function (key) {
        var doc = this.keyedMap.get(key);
        return doc ? this.sortedSet.indexOf(doc) : -1;
    };
    Object.defineProperty(DocumentSet.prototype, "size", {
        get: function () {
            return this.sortedSet.size;
        },
        enumerable: true,
        configurable: true
    });
    /** Iterates documents in order defined by "comparator" */
    DocumentSet.prototype.forEach = function (cb) {
        this.sortedSet.inorderTraversal(function (k, v) {
            cb(k);
            return false;
        });
    };
    /** Inserts or updates a document with the same key */
    DocumentSet.prototype.add = function (doc) {
        // First remove the element if we have it.
        var set = this.delete(doc.key);
        return set.copy(set.keyedMap.insert(doc.key, doc), set.sortedSet.insert(doc, null));
    };
    /** Deletes a document with a given key */
    DocumentSet.prototype.delete = function (key) {
        var doc = this.get(key);
        if (!doc) {
            return this;
        }
        return this.copy(this.keyedMap.remove(key), this.sortedSet.remove(doc));
    };
    DocumentSet.prototype.isEqual = function (other) {
        if (!(other instanceof DocumentSet))
            return false;
        if (this.size !== other.size)
            return false;
        var thisIt = this.sortedSet.getIterator();
        var otherIt = other.sortedSet.getIterator();
        while (thisIt.hasNext()) {
            var thisDoc = thisIt.getNext().key;
            var otherDoc = otherIt.getNext().key;
            if (!thisDoc.isEqual(otherDoc))
                return false;
        }
        return true;
    };
    DocumentSet.prototype.toString = function () {
        var docStrings = [];
        this.forEach(function (doc) {
            docStrings.push(doc.toString());
        });
        if (docStrings.length === 0) {
            return 'DocumentSet ()';
        }
        else {
            return 'DocumentSet (\n  ' + docStrings.join('  \n') + '\n)';
        }
    };
    DocumentSet.prototype.copy = function (keyedMap, sortedSet) {
        var newSet = new DocumentSet();
        newSet.comparator = this.comparator;
        newSet.keyedMap = keyedMap;
        newSet.sortedSet = sortedSet;
        return newSet;
    };
    return DocumentSet;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var ChangeType;
(function (ChangeType) {
    ChangeType[ChangeType["Added"] = 0] = "Added";
    ChangeType[ChangeType["Removed"] = 1] = "Removed";
    ChangeType[ChangeType["Modified"] = 2] = "Modified";
    ChangeType[ChangeType["Metadata"] = 3] = "Metadata";
})(ChangeType || (ChangeType = {}));
var SyncState;
(function (SyncState) {
    SyncState[SyncState["Local"] = 0] = "Local";
    SyncState[SyncState["Synced"] = 1] = "Synced";
})(SyncState || (SyncState = {}));
/**
 * DocumentChangeSet keeps track of a set of changes to docs in a query, merging
 * duplicate events for the same doc.
 */
var DocumentChangeSet = /** @class */ (function () {
    function DocumentChangeSet() {
        this.changeMap = new SortedMap(DocumentKey.comparator);
    }
    DocumentChangeSet.prototype.track = function (change) {
        var key = change.doc.key;
        var oldChange = this.changeMap.get(key);
        if (!oldChange) {
            this.changeMap = this.changeMap.insert(key, change);
            return;
        }
        // Merge the new change with the existing change.
        if (change.type !== ChangeType.Added &&
            oldChange.type === ChangeType.Metadata) {
            this.changeMap = this.changeMap.insert(key, change);
        }
        else if (change.type === ChangeType.Metadata &&
            oldChange.type !== ChangeType.Removed) {
            this.changeMap = this.changeMap.insert(key, {
                type: oldChange.type,
                doc: change.doc
            });
        }
        else if (change.type === ChangeType.Modified &&
            oldChange.type === ChangeType.Modified) {
            this.changeMap = this.changeMap.insert(key, {
                type: ChangeType.Modified,
                doc: change.doc
            });
        }
        else if (change.type === ChangeType.Modified &&
            oldChange.type === ChangeType.Added) {
            this.changeMap = this.changeMap.insert(key, {
                type: ChangeType.Added,
                doc: change.doc
            });
        }
        else if (change.type === ChangeType.Removed &&
            oldChange.type === ChangeType.Added) {
            this.changeMap = this.changeMap.remove(key);
        }
        else if (change.type === ChangeType.Removed &&
            oldChange.type === ChangeType.Modified) {
            this.changeMap = this.changeMap.insert(key, {
                type: ChangeType.Removed,
                doc: oldChange.doc
            });
        }
        else if (change.type === ChangeType.Added &&
            oldChange.type === ChangeType.Removed) {
            this.changeMap = this.changeMap.insert(key, {
                type: ChangeType.Modified,
                doc: change.doc
            });
        }
        else {
            // This includes these cases, which don't make sense:
            // Added->Added
            // Removed->Removed
            // Modified->Added
            // Removed->Modified
            // Metadata->Added
            // Removed->Metadata
            fail('unsupported combination of changes: ' +
                JSON.stringify(change) +
                ' after ' +
                JSON.stringify(oldChange));
        }
    };
    DocumentChangeSet.prototype.getChanges = function () {
        var changes = [];
        this.changeMap.inorderTraversal(function (key, change) {
            changes.push(change);
        });
        return changes;
    };
    return DocumentChangeSet;
}());
var ViewSnapshot = /** @class */ (function () {
    function ViewSnapshot(query, docs, oldDocs, docChanges, mutatedKeys, fromCache, syncStateChanged, excludesMetadataChanges) {
        this.query = query;
        this.docs = docs;
        this.oldDocs = oldDocs;
        this.docChanges = docChanges;
        this.mutatedKeys = mutatedKeys;
        this.fromCache = fromCache;
        this.syncStateChanged = syncStateChanged;
        this.excludesMetadataChanges = excludesMetadataChanges;
    }
    /** Returns a view snapshot as if all documents in the snapshot were added. */
    ViewSnapshot.fromInitialDocuments = function (query, documents, mutatedKeys, fromCache) {
        var changes = [];
        documents.forEach(function (doc) {
            changes.push({ type: ChangeType.Added, doc: doc });
        });
        return new ViewSnapshot(query, documents, DocumentSet.emptySet(documents), changes, mutatedKeys, fromCache, true, false);
    };
    Object.defineProperty(ViewSnapshot.prototype, "hasPendingWrites", {
        get: function () {
            return !this.mutatedKeys.isEmpty();
        },
        enumerable: true,
        configurable: true
    });
    ViewSnapshot.prototype.isEqual = function (other) {
        if (this.fromCache !== other.fromCache ||
            this.syncStateChanged !== other.syncStateChanged ||
            !this.mutatedKeys.isEqual(other.mutatedKeys) ||
            !this.query.isEqual(other.query) ||
            !this.docs.isEqual(other.docs) ||
            !this.oldDocs.isEqual(other.oldDocs)) {
            return false;
        }
        var changes = this.docChanges;
        var otherChanges = other.docChanges;
        if (changes.length !== otherChanges.length) {
            return false;
        }
        for (var i = 0; i < changes.length; i++) {
            if (changes[i].type !== otherChanges[i].type ||
                !changes[i].doc.isEqual(otherChanges[i].doc)) {
                return false;
            }
        }
        return true;
    };
    return ViewSnapshot;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * An event from the RemoteStore. It is split into targetChanges (changes to the
 * state or the set of documents in our watched targets) and documentUpdates
 * (changes to the actual documents).
 */
var RemoteEvent = /** @class */ (function () {
    function RemoteEvent(
    /**
     * The snapshot version this event brings us up to, or MIN if not set.
     */
    snapshotVersion, 
    /**
     * A map from target to changes to the target. See TargetChange.
     */
    targetChanges, 
    /**
     * A set of targets that is known to be inconsistent. Listens for these
     * targets should be re-established without resume tokens.
     */
    targetMismatches, 
    /**
     * A set of which documents have changed or been deleted, along with the
     * doc's new values (if not deleted).
     */
    documentUpdates, 
    /**
     * A set of which document updates are due only to limbo resolution targets.
     */
    resolvedLimboDocuments) {
        this.snapshotVersion = snapshotVersion;
        this.targetChanges = targetChanges;
        this.targetMismatches = targetMismatches;
        this.documentUpdates = documentUpdates;
        this.resolvedLimboDocuments = resolvedLimboDocuments;
    }
    /**
     * HACK: Views require RemoteEvents in order to determine whether the view is
     * CURRENT, but secondary tabs don't receive remote events. So this method is
     * used to create a synthesized RemoteEvent that can be used to apply a
     * CURRENT status change to a View, for queries executed in a different tab.
     */
    // PORTING NOTE: Multi-tab only
    RemoteEvent.createSynthesizedRemoteEventForCurrentChange = function (targetId, current) {
        var _a;
        var targetChanges = (_a = {},
            _a[targetId] = TargetChange.createSynthesizedTargetChangeForCurrentChange(targetId, current),
            _a);
        return new RemoteEvent(SnapshotVersion.MIN, targetChanges, targetIdSet(), maybeDocumentMap(), documentKeySet());
    };
    return RemoteEvent;
}());
/**
 * A TargetChange specifies the set of changes for a specific target as part of
 * a RemoteEvent. These changes track which documents are added, modified or
 * removed, as well as the target's resume token and whether the target is
 * marked CURRENT.
 * The actual changes *to* documents are not part of the TargetChange since
 * documents may be part of multiple targets.
 */
var TargetChange = /** @class */ (function () {
    function TargetChange(
    /**
     * An opaque, server-assigned token that allows watching a query to be resumed
     * after disconnecting without retransmitting all the data that matches the
     * query. The resume token essentially identifies a point in time from which
     * the server should resume sending results.
     */
    resumeToken, 
    /**
     * The "current" (synced) status of this target. Note that "current"
     * has special meaning in the RPC protocol that implies that a target is
     * both up-to-date and consistent with the rest of the watch stream.
     */
    current, 
    /**
     * The set of documents that were newly assigned to this target as part of
     * this remote event.
     */
    addedDocuments, 
    /**
     * The set of documents that were already assigned to this target but received
     * an update during this remote event.
     */
    modifiedDocuments, 
    /**
     * The set of documents that were removed from this target as part of this
     * remote event.
     */
    removedDocuments) {
        this.resumeToken = resumeToken;
        this.current = current;
        this.addedDocuments = addedDocuments;
        this.modifiedDocuments = modifiedDocuments;
        this.removedDocuments = removedDocuments;
    }
    /**
     * HACK: Views require TargetChanges in order to determine whether the view is
     * CURRENT, but secondary tabs don't receive remote events. So this method is
     * used to create a synthesized TargetChanges that can be used to apply a
     * CURRENT status change to a View, for queries executed in a different tab.
     */
    // PORTING NOTE: Multi-tab only
    TargetChange.createSynthesizedTargetChangeForCurrentChange = function (targetId, current) {
        return new TargetChange(emptyByteString(), current, documentKeySet(), documentKeySet(), documentKeySet());
    };
    return TargetChange;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Represents a changed document and a list of target ids to which this change
 * applies.
 *
 * If document has been deleted NoDocument will be provided.
 */
var DocumentWatchChange = /** @class */ (function () {
    function DocumentWatchChange(
    /** The new document applies to all of these targets. */
    updatedTargetIds, 
    /** The new document is removed from all of these targets. */
    removedTargetIds, 
    /** The key of the document for this change. */
    key, 
    /**
     * The new document or NoDocument if it was deleted. Is null if the
     * document went out of view without the server sending a new document.
     */
    newDoc) {
        this.updatedTargetIds = updatedTargetIds;
        this.removedTargetIds = removedTargetIds;
        this.key = key;
        this.newDoc = newDoc;
    }
    return DocumentWatchChange;
}());
var ExistenceFilterChange = /** @class */ (function () {
    function ExistenceFilterChange(targetId, existenceFilter) {
        this.targetId = targetId;
        this.existenceFilter = existenceFilter;
    }
    return ExistenceFilterChange;
}());
var WatchTargetChangeState;
(function (WatchTargetChangeState) {
    WatchTargetChangeState[WatchTargetChangeState["NoChange"] = 0] = "NoChange";
    WatchTargetChangeState[WatchTargetChangeState["Added"] = 1] = "Added";
    WatchTargetChangeState[WatchTargetChangeState["Removed"] = 2] = "Removed";
    WatchTargetChangeState[WatchTargetChangeState["Current"] = 3] = "Current";
    WatchTargetChangeState[WatchTargetChangeState["Reset"] = 4] = "Reset";
})(WatchTargetChangeState || (WatchTargetChangeState = {}));
var WatchTargetChange = /** @class */ (function () {
    function WatchTargetChange(
    /** What kind of change occurred to the watch target. */
    state, 
    /** The target IDs that were added/removed/set. */
    targetIds, 
    /**
     * An opaque, server-assigned token that allows watching a query to be
     * resumed after disconnecting without retransmitting all the data that
     * matches the query. The resume token essentially identifies a point in
     * time from which the server should resume sending results.
     */
    resumeToken, 
    /** An RPC error indicating why the watch failed. */
    cause) {
        if (resumeToken === void 0) { resumeToken = emptyByteString(); }
        if (cause === void 0) { cause = null; }
        this.state = state;
        this.targetIds = targetIds;
        this.resumeToken = resumeToken;
        this.cause = cause;
    }
    return WatchTargetChange;
}());
/** Tracks the internal state of a Watch target. */
var TargetState = /** @class */ (function () {
    function TargetState() {
        /**
         * The number of pending responses (adds or removes) that we are waiting on.
         * We only consider targets active that have no pending responses.
         */
        this.pendingResponses = 0;
        /**
         * Keeps track of the document changes since the last raised snapshot.
         *
         * These changes are continuously updated as we receive document updates and
         * always reflect the current set of changes against the last issued snapshot.
         */
        this.documentChanges = snapshotChangesMap();
        /** See public getters for explanations of these fields. */
        this._resumeToken = emptyByteString();
        this._current = false;
        /**
         * Whether this target state should be included in the next snapshot. We
         * initialize to true so that newly-added targets are included in the next
         * RemoteEvent.
         */
        this._hasPendingChanges = true;
    }
    Object.defineProperty(TargetState.prototype, "current", {
        /**
         * Whether this target has been marked 'current'.
         *
         * 'Current' has special meaning in the RPC protocol: It implies that the
         * Watch backend has sent us all changes up to the point at which the target
         * was added and that the target is consistent with the rest of the watch
         * stream.
         */
        get: function () {
            return this._current;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(TargetState.prototype, "resumeToken", {
        /** The last resume token sent to us for this target. */
        get: function () {
            return this._resumeToken;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(TargetState.prototype, "isPending", {
        /** Whether this target has pending target adds or target removes. */
        get: function () {
            return this.pendingResponses !== 0;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(TargetState.prototype, "hasPendingChanges", {
        /** Whether we have modified any state that should trigger a snapshot. */
        get: function () {
            return this._hasPendingChanges;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Applies the resume token to the TargetChange, but only when it has a new
     * value. Empty resumeTokens are discarded.
     */
    TargetState.prototype.updateResumeToken = function (resumeToken) {
        if (resumeToken.length > 0) {
            this._hasPendingChanges = true;
            this._resumeToken = resumeToken;
        }
    };
    /**
     * Creates a target change from the current set of changes.
     *
     * To reset the document changes after raising this snapshot, call
     * `clearPendingChanges()`.
     */
    TargetState.prototype.toTargetChange = function () {
        var addedDocuments = documentKeySet();
        var modifiedDocuments = documentKeySet();
        var removedDocuments = documentKeySet();
        this.documentChanges.forEach(function (key, changeType) {
            switch (changeType) {
                case ChangeType.Added:
                    addedDocuments = addedDocuments.add(key);
                    break;
                case ChangeType.Modified:
                    modifiedDocuments = modifiedDocuments.add(key);
                    break;
                case ChangeType.Removed:
                    removedDocuments = removedDocuments.add(key);
                    break;
                default:
                    fail('Encountered invalid change type: ' + changeType);
            }
        });
        return new TargetChange(this._resumeToken, this._current, addedDocuments, modifiedDocuments, removedDocuments);
    };
    /**
     * Resets the document changes and sets `hasPendingChanges` to false.
     */
    TargetState.prototype.clearPendingChanges = function () {
        this._hasPendingChanges = false;
        this.documentChanges = snapshotChangesMap();
    };
    TargetState.prototype.addDocumentChange = function (key, changeType) {
        this._hasPendingChanges = true;
        this.documentChanges = this.documentChanges.insert(key, changeType);
    };
    TargetState.prototype.removeDocumentChange = function (key) {
        this._hasPendingChanges = true;
        this.documentChanges = this.documentChanges.remove(key);
    };
    TargetState.prototype.recordPendingTargetRequest = function () {
        this.pendingResponses += 1;
    };
    TargetState.prototype.recordTargetResponse = function () {
        this.pendingResponses -= 1;
    };
    TargetState.prototype.markCurrent = function () {
        this._hasPendingChanges = true;
        this._current = true;
    };
    return TargetState;
}());
/**
 * A helper class to accumulate watch changes into a RemoteEvent.
 */
var WatchChangeAggregator = /** @class */ (function () {
    function WatchChangeAggregator(metadataProvider) {
        this.metadataProvider = metadataProvider;
        /** The internal state of all tracked targets. */
        this.targetStates = {};
        /** Keeps track of the documents to update since the last raised snapshot. */
        this.pendingDocumentUpdates = maybeDocumentMap();
        /** A mapping of document keys to their set of target IDs. */
        this.pendingDocumentTargetMapping = documentTargetMap();
        /**
         * A list of targets with existence filter mismatches. These targets are
         * known to be inconsistent and their listens needs to be re-established by
         * RemoteStore.
         */
        this.pendingTargetResets = new SortedSet(primitiveComparator);
    }
    /**
     * Processes and adds the DocumentWatchChange to the current set of changes.
     */
    WatchChangeAggregator.prototype.handleDocumentChange = function (docChange) {
        for (var _i = 0, _a = docChange.updatedTargetIds; _i < _a.length; _i++) {
            var targetId = _a[_i];
            if (docChange.newDoc instanceof Document) {
                this.addDocumentToTarget(targetId, docChange.newDoc);
            }
            else if (docChange.newDoc instanceof NoDocument) {
                this.removeDocumentFromTarget(targetId, docChange.key, docChange.newDoc);
            }
        }
        for (var _b = 0, _c = docChange.removedTargetIds; _b < _c.length; _b++) {
            var targetId = _c[_b];
            this.removeDocumentFromTarget(targetId, docChange.key, docChange.newDoc);
        }
    };
    /** Processes and adds the WatchTargetChange to the current set of changes. */
    WatchChangeAggregator.prototype.handleTargetChange = function (targetChange) {
        var _this = this;
        this.forEachTarget(targetChange, function (targetId) {
            var targetState = _this.ensureTargetState(targetId);
            switch (targetChange.state) {
                case WatchTargetChangeState.NoChange:
                    if (_this.isActiveTarget(targetId)) {
                        targetState.updateResumeToken(targetChange.resumeToken);
                    }
                    break;
                case WatchTargetChangeState.Added:
                    // We need to decrement the number of pending acks needed from watch
                    // for this targetId.
                    targetState.recordTargetResponse();
                    if (!targetState.isPending) {
                        // We have a freshly added target, so we need to reset any state
                        // that we had previously. This can happen e.g. when remove and add
                        // back a target for existence filter mismatches.
                        targetState.clearPendingChanges();
                    }
                    targetState.updateResumeToken(targetChange.resumeToken);
                    break;
                case WatchTargetChangeState.Removed:
                    // We need to keep track of removed targets to we can post-filter and
                    // remove any target changes.
                    // We need to decrement the number of pending acks needed from watch
                    // for this targetId.
                    targetState.recordTargetResponse();
                    if (!targetState.isPending) {
                        _this.removeTarget(targetId);
                    }
                    assert(!targetChange.cause, 'WatchChangeAggregator does not handle errored targets');
                    break;
                case WatchTargetChangeState.Current:
                    if (_this.isActiveTarget(targetId)) {
                        targetState.markCurrent();
                        targetState.updateResumeToken(targetChange.resumeToken);
                    }
                    break;
                case WatchTargetChangeState.Reset:
                    if (_this.isActiveTarget(targetId)) {
                        // Reset the target and synthesizes removes for all existing
                        // documents. The backend will re-add any documents that still
                        // match the target before it sends the next global snapshot.
                        _this.resetTarget(targetId);
                        targetState.updateResumeToken(targetChange.resumeToken);
                    }
                    break;
                default:
                    fail('Unknown target watch change state: ' + targetChange.state);
            }
        });
    };
    /**
     * Iterates over all targetIds that the watch change applies to: either the
     * targetIds explicitly listed in the change or the targetIds of all currently
     * active targets.
     */
    WatchChangeAggregator.prototype.forEachTarget = function (targetChange, fn) {
        if (targetChange.targetIds.length > 0) {
            targetChange.targetIds.forEach(fn);
        }
        else {
            forEachNumber(this.targetStates, fn);
        }
    };
    /**
     * Handles existence filters and synthesizes deletes for filter mismatches.
     * Targets that are invalidated by filter mismatches are added to
     * `pendingTargetResets`.
     */
    WatchChangeAggregator.prototype.handleExistenceFilter = function (watchChange) {
        var targetId = watchChange.targetId;
        var expectedCount = watchChange.existenceFilter.count;
        var queryData = this.queryDataForActiveTarget(targetId);
        if (queryData) {
            var query = queryData.query;
            if (query.isDocumentQuery()) {
                if (expectedCount === 0) {
                    // The existence filter told us the document does not exist. We deduce
                    // that this document does not exist and apply a deleted document to
                    // our updates. Without applying this deleted document there might be
                    // another query that will raise this document as part of a snapshot
                    // until it is resolved, essentially exposing inconsistency between
                    // queries.
                    var key = new DocumentKey(query.path);
                    this.removeDocumentFromTarget(targetId, key, new NoDocument(key, SnapshotVersion.forDeletedDoc()));
                }
                else {
                    assert(expectedCount === 1, 'Single document existence filter with count: ' + expectedCount);
                }
            }
            else {
                var currentSize = this.getCurrentDocumentCountForTarget(targetId);
                if (currentSize !== expectedCount) {
                    // Existence filter mismatch: We reset the mapping and raise a new
                    // snapshot with `isFromCache:true`.
                    this.resetTarget(targetId);
                    this.pendingTargetResets = this.pendingTargetResets.add(targetId);
                }
            }
        }
    };
    /**
     * Converts the currently accumulated state into a remote event at the
     * provided snapshot version. Resets the accumulated changes before returning.
     */
    WatchChangeAggregator.prototype.createRemoteEvent = function (snapshotVersion) {
        var _this = this;
        var targetChanges = {};
        forEachNumber(this.targetStates, function (targetId, targetState) {
            var queryData = _this.queryDataForActiveTarget(targetId);
            if (queryData) {
                if (targetState.current && queryData.query.isDocumentQuery()) {
                    // Document queries for document that don't exist can produce an empty
                    // result set. To update our local cache, we synthesize a document
                    // delete if we have not previously received the document. This
                    // resolves the limbo state of the document, removing it from
                    // limboDocumentRefs.
                    //
                    // TODO(dimond): Ideally we would have an explicit lookup query
                    // instead resulting in an explicit delete message and we could
                    // remove this special logic.
                    var key = new DocumentKey(queryData.query.path);
                    if (_this.pendingDocumentUpdates.get(key) === null &&
                        !_this.targetContainsDocument(targetId, key)) {
                        _this.removeDocumentFromTarget(targetId, key, new NoDocument(key, snapshotVersion));
                    }
                }
                if (targetState.hasPendingChanges) {
                    targetChanges[targetId] = targetState.toTargetChange();
                    targetState.clearPendingChanges();
                }
            }
        });
        var resolvedLimboDocuments = documentKeySet();
        // We extract the set of limbo-only document updates as the GC logic
        // special-cases documents that do not appear in the query cache.
        //
        // TODO(gsoltis): Expand on this comment once GC is available in the JS
        // client.
        this.pendingDocumentTargetMapping.forEach(function (key, targets) {
            var isOnlyLimboTarget = true;
            targets.forEachWhile(function (targetId) {
                var queryData = _this.queryDataForActiveTarget(targetId);
                if (queryData && queryData.purpose !== QueryPurpose.LimboResolution) {
                    isOnlyLimboTarget = false;
                    return false;
                }
                return true;
            });
            if (isOnlyLimboTarget) {
                resolvedLimboDocuments = resolvedLimboDocuments.add(key);
            }
        });
        var remoteEvent = new RemoteEvent(snapshotVersion, targetChanges, this.pendingTargetResets, this.pendingDocumentUpdates, resolvedLimboDocuments);
        this.pendingDocumentUpdates = maybeDocumentMap();
        this.pendingDocumentTargetMapping = documentTargetMap();
        this.pendingTargetResets = new SortedSet(primitiveComparator);
        return remoteEvent;
    };
    /**
     * Adds the provided document to the internal list of document updates and
     * its document key to the given target's mapping.
     */
    // Visible for testing.
    WatchChangeAggregator.prototype.addDocumentToTarget = function (targetId, document) {
        if (!this.isActiveTarget(targetId)) {
            return;
        }
        var changeType = this.targetContainsDocument(targetId, document.key)
            ? ChangeType.Modified
            : ChangeType.Added;
        var targetState = this.ensureTargetState(targetId);
        targetState.addDocumentChange(document.key, changeType);
        this.pendingDocumentUpdates = this.pendingDocumentUpdates.insert(document.key, document);
        this.pendingDocumentTargetMapping = this.pendingDocumentTargetMapping.insert(document.key, this.ensureDocumentTargetMapping(document.key).add(targetId));
    };
    /**
     * Removes the provided document from the target mapping. If the
     * document no longer matches the target, but the document's state is still
     * known (e.g. we know that the document was deleted or we received the change
     * that caused the filter mismatch), the new document can be provided
     * to update the remote document cache.
     */
    // Visible for testing.
    WatchChangeAggregator.prototype.removeDocumentFromTarget = function (targetId, key, updatedDocument) {
        if (!this.isActiveTarget(targetId)) {
            return;
        }
        var targetState = this.ensureTargetState(targetId);
        if (this.targetContainsDocument(targetId, key)) {
            targetState.addDocumentChange(key, ChangeType.Removed);
        }
        else {
            // The document may have entered and left the target before we raised a
            // snapshot, so we can just ignore the change.
            targetState.removeDocumentChange(key);
        }
        this.pendingDocumentTargetMapping = this.pendingDocumentTargetMapping.insert(key, this.ensureDocumentTargetMapping(key).delete(targetId));
        if (updatedDocument) {
            this.pendingDocumentUpdates = this.pendingDocumentUpdates.insert(key, updatedDocument);
        }
    };
    WatchChangeAggregator.prototype.removeTarget = function (targetId) {
        delete this.targetStates[targetId];
    };
    /**
     * Returns the current count of documents in the target. This includes both
     * the number of documents that the LocalStore considers to be part of the
     * target as well as any accumulated changes.
     */
    WatchChangeAggregator.prototype.getCurrentDocumentCountForTarget = function (targetId) {
        var targetState = this.ensureTargetState(targetId);
        var targetChange = targetState.toTargetChange();
        return (this.metadataProvider.getRemoteKeysForTarget(targetId).size +
            targetChange.addedDocuments.size -
            targetChange.removedDocuments.size);
    };
    /**
     * Increment the number of acks needed from watch before we can consider the
     * server to be 'in-sync' with the client's active targets.
     */
    WatchChangeAggregator.prototype.recordPendingTargetRequest = function (targetId) {
        // For each request we get we need to record we need a response for it.
        var targetState = this.ensureTargetState(targetId);
        targetState.recordPendingTargetRequest();
    };
    WatchChangeAggregator.prototype.ensureTargetState = function (targetId) {
        if (!this.targetStates[targetId]) {
            this.targetStates[targetId] = new TargetState();
        }
        return this.targetStates[targetId];
    };
    WatchChangeAggregator.prototype.ensureDocumentTargetMapping = function (key) {
        var targetMapping = this.pendingDocumentTargetMapping.get(key);
        if (!targetMapping) {
            targetMapping = new SortedSet(primitiveComparator);
            this.pendingDocumentTargetMapping = this.pendingDocumentTargetMapping.insert(key, targetMapping);
        }
        return targetMapping;
    };
    /**
     * Verifies that the user is still interested in this target (by calling
     * `getQueryDataForTarget()`) and that we are not waiting for pending ADDs
     * from watch.
     */
    WatchChangeAggregator.prototype.isActiveTarget = function (targetId) {
        return this.queryDataForActiveTarget(targetId) !== null;
    };
    /**
     * Returns the QueryData for an active target (i.e. a target that the user
     * is still interested in that has no outstanding target change requests).
     */
    WatchChangeAggregator.prototype.queryDataForActiveTarget = function (targetId) {
        var targetState = this.targetStates[targetId];
        return targetState && targetState.isPending
            ? null
            : this.metadataProvider.getQueryDataForTarget(targetId);
    };
    /**
     * Resets the state of a Watch target to its initial state (e.g. sets
     * 'current' to false, clears the resume token and removes its target mapping
     * from all documents).
     */
    WatchChangeAggregator.prototype.resetTarget = function (targetId) {
        var _this = this;
        assert(!this.targetStates[targetId].isPending, 'Should only reset active targets');
        this.targetStates[targetId] = new TargetState();
        // Trigger removal for any documents currently mapped to this target.
        // These removals will be part of the initial snapshot if Watch does not
        // resend these documents.
        var existingKeys = this.metadataProvider.getRemoteKeysForTarget(targetId);
        existingKeys.forEach(function (key) {
            _this.removeDocumentFromTarget(targetId, key, /*updatedDocument=*/ null);
        });
    };
    /**
     * Returns whether the LocalStore considers the document to be part of the
     * specified target.
     */
    WatchChangeAggregator.prototype.targetContainsDocument = function (targetId, key) {
        var existingKeys = this.metadataProvider.getRemoteKeysForTarget(targetId);
        return existingKeys.has(key);
    };
    return WatchChangeAggregator;
}());
function documentTargetMap() {
    return new SortedMap(DocumentKey.comparator);
}
function snapshotChangesMap() {
    return new SortedMap(DocumentKey.comparator);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var DIRECTIONS = (function () {
    var dirs = {};
    dirs[Direction.ASCENDING.name] = 'ASCENDING';
    dirs[Direction.DESCENDING.name] = 'DESCENDING';
    return dirs;
})();
var OPERATORS = (function () {
    var ops = {};
    ops[RelationOp.LESS_THAN.name] = 'LESS_THAN';
    ops[RelationOp.LESS_THAN_OR_EQUAL.name] = 'LESS_THAN_OR_EQUAL';
    ops[RelationOp.GREATER_THAN.name] = 'GREATER_THAN';
    ops[RelationOp.GREATER_THAN_OR_EQUAL.name] = 'GREATER_THAN_OR_EQUAL';
    ops[RelationOp.EQUAL.name] = 'EQUAL';
    ops[RelationOp.ARRAY_CONTAINS.name] = 'ARRAY_CONTAINS';
    return ops;
})();
// A RegExp matching ISO 8601 UTC timestamps with optional fraction.
var ISO_REG_EXP = new RegExp(/^\d{4}-\d\d-\d\dT\d\d:\d\d:\d\d(?:\.(\d+))?Z$/);
function assertPresent(value, description) {
    assert(!isNullOrUndefined(value), description + ' is missing');
}
function parseInt64(value) {
    // TODO(bjornick): Handle int64 greater than 53 bits.
    if (typeof value === 'number') {
        return value;
    }
    else if (typeof value === 'string') {
        return Number(value);
    }
    else {
        return fail("can't parse " + value);
    }
}
/**
 * Generates JsonObject values for the Datastore API suitable for sending to
 * either GRPC stub methods or via the JSON/HTTP REST API.
 * TODO(klimt): We can remove the databaseId argument if we keep the full
 * resource name in documents.
 */
var JsonProtoSerializer = /** @class */ (function () {
    function JsonProtoSerializer(databaseId, options) {
        this.databaseId = databaseId;
        this.options = options;
    }
    JsonProtoSerializer.prototype.emptyByteString = function () {
        if (this.options.useProto3Json) {
            return '';
        }
        else {
            return new Uint8Array(0);
        }
    };
    JsonProtoSerializer.prototype.unsafeCastProtoByteString = function (byteString) {
        // byteStrings can be either string or UInt8Array, but the typings say
        // it's always a string. Cast as string to avoid type check failing
        return byteString;
    };
    JsonProtoSerializer.prototype.fromRpcStatus = function (status) {
        var code = status.code === undefined
            ? Code.UNKNOWN
            : mapCodeFromRpcCode(status.code);
        return new FirestoreError(code, status.message || '');
    };
    /**
     * Returns a value for a number (or undefined) that's appropriate to put into
     * a google.protobuf.Int32Value proto.
     * DO NOT USE THIS FOR ANYTHING ELSE.
     * This method cheats. It's typed as returning "number" because that's what
     * our generated proto interfaces say Int32Value must be. But GRPC actually
     * expects a { value: <number> } struct.
     */
    JsonProtoSerializer.prototype.toInt32Value = function (val) {
        if (!isNullOrUndefined(val)) {
            // tslint:disable-next-line:no-any We need to match generated Proto types.
            return { value: val };
        }
        else {
            return undefined;
        }
    };
    /**
     * Returns a number (or null) from a google.protobuf.Int32Value proto.
     * DO NOT USE THIS FOR ANYTHING ELSE.
     * This method cheats. It's typed as accepting "number" because that's what
     * our generated proto interfaces say Int32Value must be, but it actually
     * accepts { value: number } to match our serialization in toInt32Value().
     */
    JsonProtoSerializer.prototype.fromInt32Value = function (val) {
        var result;
        if (typeof val === 'object') {
            // tslint:disable-next-line:no-any We need to match generated Proto types.
            result = val.value;
        }
        else {
            // We accept raw numbers (without the {value: ... } wrapper) for
            // compatibility with legacy persisted data.
            result = val;
        }
        return isNullOrUndefined(result) ? null : result;
    };
    /**
     * Returns a value for a Date that's appropriate to put into a proto.
     * DO NOT USE THIS FOR ANYTHING ELSE.
     * This method cheats. It's typed as returning "string" because that's what
     * our generated proto interfaces say dates must be. But it's easier and safer
     * to actually return a Timestamp proto.
     */
    JsonProtoSerializer.prototype.toTimestamp = function (timestamp) {
        return {
            seconds: timestamp.seconds,
            nanos: timestamp.nanoseconds
            // tslint:disable-next-line:no-any
        };
    };
    JsonProtoSerializer.prototype.fromTimestamp = function (date) {
        // The json interface (for the browser) will return an iso timestamp string,
        // while the proto js library (for node) will return a
        // google.protobuf.Timestamp instance.
        if (typeof date === 'string') {
            // TODO(b/37282237): Use strings for Proto3 timestamps
            // assert(this.options.useProto3Json,
            //   'The timestamp string format requires Proto3.');
            return this.fromIso8601String(date);
        }
        else {
            assert(!!date, 'Cannot deserialize null or undefined timestamp.');
            // TODO(b/37282237): Use strings for Proto3 timestamps
            // assert(!this.options.useProto3Json,
            //   'The timestamp instance format requires Proto JS.');
            var seconds = parseInt64(date.seconds || '0');
            var nanos = date.nanos || 0;
            return new Timestamp(seconds, nanos);
        }
    };
    JsonProtoSerializer.prototype.fromIso8601String = function (utc) {
        // The date string can have higher precision (nanos) than the Date class
        // (millis), so we do some custom parsing here.
        // Parse the nanos right out of the string.
        var nanos = 0;
        var fraction = ISO_REG_EXP.exec(utc);
        assert(!!fraction, 'invalid timestamp: ' + utc);
        if (fraction[1]) {
            // Pad the fraction out to 9 digits (nanos).
            var nanoStr = fraction[1];
            nanoStr = (nanoStr + '000000000').substr(0, 9);
            nanos = Number(nanoStr);
        }
        // Parse the date to get the seconds.
        var date = new Date(utc);
        var seconds = Math.floor(date.getTime() / 1000);
        return new Timestamp(seconds, nanos);
    };
    /**
     * Returns a value for bytes that's appropriate to put in a proto.
     * DO NOT USE THIS FOR ANYTHING ELSE.
     * This method cheats. It's typed as returning "string" because that's what
     * our generated proto interfaces say bytes must be. But it should return
     * an Uint8Array in Node.
     */
    JsonProtoSerializer.prototype.toBytes = function (bytes) {
        if (this.options.useProto3Json) {
            return bytes.toBase64();
        }
        else {
            // The typings say it's a string, but it needs to be a Uint8Array in Node.
            return this.unsafeCastProtoByteString(bytes.toUint8Array());
        }
    };
    /**
     * Parse the blob from the protos into the internal Blob class. Note that the
     * typings assume all blobs are strings, but they are actually Uint8Arrays
     * on Node.
     */
    JsonProtoSerializer.prototype.fromBlob = function (blob) {
        if (typeof blob === 'string') {
            assert(this.options.useProto3Json, 'Expected bytes to be passed in as Uint8Array, but got a string instead.');
            return Blob.fromBase64String(blob);
        }
        else {
            assert(!this.options.useProto3Json, 'Expected bytes to be passed in as string, but got something else instead.');
            return Blob.fromUint8Array(blob);
        }
    };
    JsonProtoSerializer.prototype.toVersion = function (version) {
        return this.toTimestamp(version.toTimestamp());
    };
    JsonProtoSerializer.prototype.fromVersion = function (version) {
        assert(!!version, "Trying to deserialize version that isn't set");
        return SnapshotVersion.fromTimestamp(this.fromTimestamp(version));
    };
    JsonProtoSerializer.prototype.toResourceName = function (databaseId, path) {
        return this.fullyQualifiedPrefixPath(databaseId)
            .child('documents')
            .child(path)
            .canonicalString();
    };
    JsonProtoSerializer.prototype.fromResourceName = function (name) {
        var resource = ResourcePath.fromString(name);
        assert(this.isValidResourceName(resource), 'Tried to deserialize invalid key ' + resource.toString());
        return resource;
    };
    JsonProtoSerializer.prototype.toName = function (key) {
        return this.toResourceName(this.databaseId, key.path);
    };
    JsonProtoSerializer.prototype.fromName = function (name) {
        var resource = this.fromResourceName(name);
        assert(resource.get(1) === this.databaseId.projectId, 'Tried to deserialize key from different project: ' +
            resource.get(1) +
            ' vs ' +
            this.databaseId.projectId);
        assert((!resource.get(3) && !this.databaseId.database) ||
            resource.get(3) === this.databaseId.database, 'Tried to deserialize key from different database: ' +
            resource.get(3) +
            ' vs ' +
            this.databaseId.database);
        return new DocumentKey(this.extractLocalPathFromResourceName(resource));
    };
    JsonProtoSerializer.prototype.toQueryPath = function (path) {
        return this.toResourceName(this.databaseId, path);
    };
    JsonProtoSerializer.prototype.fromQueryPath = function (name) {
        var resourceName = this.fromResourceName(name);
        // In v1beta1 queries for collections at the root did not have a trailing
        // "/documents". In v1 all resource paths contain "/documents". Preserve the
        // ability to read the v1beta1 form for compatibility with queries persisted
        // in the local query cache.
        if (resourceName.length === 4) {
            return ResourcePath.EMPTY_PATH;
        }
        return this.extractLocalPathFromResourceName(resourceName);
    };
    Object.defineProperty(JsonProtoSerializer.prototype, "encodedDatabaseId", {
        get: function () {
            var path = new ResourcePath([
                'projects',
                this.databaseId.projectId,
                'databases',
                this.databaseId.database
            ]);
            return path.canonicalString();
        },
        enumerable: true,
        configurable: true
    });
    JsonProtoSerializer.prototype.fullyQualifiedPrefixPath = function (databaseId) {
        return new ResourcePath([
            'projects',
            databaseId.projectId,
            'databases',
            databaseId.database
        ]);
    };
    JsonProtoSerializer.prototype.extractLocalPathFromResourceName = function (resourceName) {
        assert(resourceName.length > 4 && resourceName.get(4) === 'documents', 'tried to deserialize invalid key ' + resourceName.toString());
        return resourceName.popFirst(5);
    };
    JsonProtoSerializer.prototype.isValidResourceName = function (path) {
        // Resource names have at least 4 components (project ID, database ID)
        return (path.length >= 4 &&
            path.get(0) === 'projects' &&
            path.get(2) === 'databases');
    };
    JsonProtoSerializer.prototype.toValue = function (val) {
        if (val instanceof NullValue) {
            return { nullValue: 'NULL_VALUE' };
        }
        else if (val instanceof BooleanValue) {
            return { booleanValue: val.value() };
        }
        else if (val instanceof IntegerValue) {
            return { integerValue: '' + val.value() };
        }
        else if (val instanceof DoubleValue) {
            var doubleValue = val.value();
            if (this.options.useProto3Json) {
                // Proto 3 let's us encode NaN and Infinity as string values as
                // expected by the backend. This is currently not checked by our unit
                // tests because they rely on protobuf.js.
                if (isNaN(doubleValue)) {
                    return { doubleValue: 'NaN' };
                }
                else if (doubleValue === Infinity) {
                    return { doubleValue: 'Infinity' };
                }
                else if (doubleValue === -Infinity) {
                    return { doubleValue: '-Infinity' };
                }
            }
            return { doubleValue: val.value() };
        }
        else if (val instanceof StringValue) {
            return { stringValue: val.value() };
        }
        else if (val instanceof ObjectValue) {
            return { mapValue: this.toMapValue(val) };
        }
        else if (val instanceof ArrayValue) {
            return { arrayValue: this.toArrayValue(val) };
        }
        else if (val instanceof TimestampValue) {
            return {
                timestampValue: this.toTimestamp(val.internalValue)
            };
        }
        else if (val instanceof GeoPointValue) {
            return {
                geoPointValue: {
                    latitude: val.value().latitude,
                    longitude: val.value().longitude
                }
            };
        }
        else if (val instanceof BlobValue) {
            return {
                bytesValue: this.toBytes(val.value())
            };
        }
        else if (val instanceof RefValue) {
            return {
                referenceValue: this.toResourceName(val.databaseId, val.key.path)
            };
        }
        else {
            return fail('Unknown FieldValue ' + JSON.stringify(val));
        }
    };
    JsonProtoSerializer.prototype.fromValue = function (obj) {
        var _this = this;
        // tslint:disable-next-line:no-any
        var type = obj['value_type'];
        if (hasTag(obj, type, 'nullValue')) {
            return NullValue.INSTANCE;
        }
        else if (hasTag(obj, type, 'booleanValue')) {
            return BooleanValue.of(obj.booleanValue);
        }
        else if (hasTag(obj, type, 'integerValue')) {
            return new IntegerValue(parseInt64(obj.integerValue));
        }
        else if (hasTag(obj, type, 'doubleValue')) {
            if (this.options.useProto3Json) {
                // Proto 3 uses the string values 'NaN' and 'Infinity'.
                if (obj.doubleValue === 'NaN') {
                    return DoubleValue.NAN;
                }
                else if (obj.doubleValue === 'Infinity') {
                    return DoubleValue.POSITIVE_INFINITY;
                }
                else if (obj.doubleValue === '-Infinity') {
                    return DoubleValue.NEGATIVE_INFINITY;
                }
            }
            return new DoubleValue(obj.doubleValue);
        }
        else if (hasTag(obj, type, 'stringValue')) {
            return new StringValue(obj.stringValue);
        }
        else if (hasTag(obj, type, 'mapValue')) {
            return this.fromFields(obj.mapValue.fields || {});
        }
        else if (hasTag(obj, type, 'arrayValue')) {
            // "values" is not present if the array is empty
            assertPresent(obj.arrayValue, 'arrayValue');
            var values = obj.arrayValue.values || [];
            return new ArrayValue(values.map(function (v) { return _this.fromValue(v); }));
        }
        else if (hasTag(obj, type, 'timestampValue')) {
            assertPresent(obj.timestampValue, 'timestampValue');
            return new TimestampValue(this.fromTimestamp(obj.timestampValue));
        }
        else if (hasTag(obj, type, 'geoPointValue')) {
            assertPresent(obj.geoPointValue, 'geoPointValue');
            var latitude = obj.geoPointValue.latitude || 0;
            var longitude = obj.geoPointValue.longitude || 0;
            return new GeoPointValue(new GeoPoint(latitude, longitude));
        }
        else if (hasTag(obj, type, 'bytesValue')) {
            assertPresent(obj.bytesValue, 'bytesValue');
            var blob = this.fromBlob(obj.bytesValue);
            return new BlobValue(blob);
        }
        else if (hasTag(obj, type, 'referenceValue')) {
            assertPresent(obj.referenceValue, 'referenceValue');
            var resourceName = this.fromResourceName(obj.referenceValue);
            var dbId = new DatabaseId(resourceName.get(1), resourceName.get(3));
            var key = new DocumentKey(this.extractLocalPathFromResourceName(resourceName));
            return new RefValue(dbId, key);
        }
        else {
            return fail('Unknown Value proto ' + JSON.stringify(obj));
        }
    };
    /** Creates an api.Document from key and fields (but no create/update time) */
    JsonProtoSerializer.prototype.toMutationDocument = function (key, fields) {
        return {
            name: this.toName(key),
            fields: this.toFields(fields)
        };
    };
    JsonProtoSerializer.prototype.toDocument = function (document) {
        assert(!document.hasLocalMutations, "Can't serialize documents with mutations.");
        return {
            name: this.toName(document.key),
            fields: this.toFields(document.data),
            updateTime: this.toTimestamp(document.version.toTimestamp())
        };
    };
    JsonProtoSerializer.prototype.fromDocument = function (document, hasCommittedMutations) {
        return new Document(this.fromName(document.name), this.fromVersion(document.updateTime), this.fromFields(document.fields || {}), { hasCommittedMutations: !!hasCommittedMutations });
    };
    JsonProtoSerializer.prototype.toFields = function (fields) {
        var _this = this;
        var result = {};
        fields.forEach(function (key, value) {
            result[key] = _this.toValue(value);
        });
        return result;
    };
    JsonProtoSerializer.prototype.fromFields = function (object) {
        var _this = this;
        // Proto map<string, Value> gets mapped to Object, so cast it.
        var map = object;
        var result = ObjectValue.EMPTY;
        forEach(map, function (key, value) {
            result = result.set(new FieldPath([key]), _this.fromValue(value));
        });
        return result;
    };
    JsonProtoSerializer.prototype.toMapValue = function (map) {
        return {
            fields: this.toFields(map)
        };
    };
    JsonProtoSerializer.prototype.toArrayValue = function (array) {
        var _this = this;
        var result = [];
        array.forEach(function (value) {
            result.push(_this.toValue(value));
        });
        return { values: result };
    };
    JsonProtoSerializer.prototype.fromFound = function (doc) {
        assert(!!doc.found, 'Tried to deserialize a found document from a missing document.');
        assertPresent(doc.found.name, 'doc.found.name');
        assertPresent(doc.found.updateTime, 'doc.found.updateTime');
        var key = this.fromName(doc.found.name);
        var version = this.fromVersion(doc.found.updateTime);
        var fields = this.fromFields(doc.found.fields || {});
        return new Document(key, version, fields, {}, doc.found);
    };
    JsonProtoSerializer.prototype.fromMissing = function (result) {
        assert(!!result.missing, 'Tried to deserialize a missing document from a found document.');
        assert(!!result.readTime, 'Tried to deserialize a missing document without a read time.');
        var key = this.fromName(result.missing);
        var version = this.fromVersion(result.readTime);
        return new NoDocument(key, version);
    };
    JsonProtoSerializer.prototype.fromMaybeDocument = function (result) {
        // tslint:disable-next-line:no-any
        var type = result['result'];
        if (hasTag(result, type, 'found')) {
            return this.fromFound(result);
        }
        else if (hasTag(result, type, 'missing')) {
            return this.fromMissing(result);
        }
        return fail('invalid batch get response: ' + JSON.stringify(result));
    };
    JsonProtoSerializer.prototype.toWatchTargetChangeState = function (state) {
        switch (state) {
            case WatchTargetChangeState.Added:
                return 'ADD';
            case WatchTargetChangeState.Current:
                return 'CURRENT';
            case WatchTargetChangeState.NoChange:
                return 'NO_CHANGE';
            case WatchTargetChangeState.Removed:
                return 'REMOVE';
            case WatchTargetChangeState.Reset:
                return 'RESET';
            default:
                return fail('Unknown WatchTargetChangeState: ' + state);
        }
    };
    JsonProtoSerializer.prototype.toTestWatchChange = function (watchChange) {
        if (watchChange instanceof ExistenceFilterChange) {
            return {
                filter: {
                    count: watchChange.existenceFilter.count,
                    targetId: watchChange.targetId
                }
            };
        }
        if (watchChange instanceof DocumentWatchChange) {
            if (watchChange.newDoc instanceof Document) {
                var doc = watchChange.newDoc;
                return {
                    documentChange: {
                        document: {
                            name: this.toName(doc.key),
                            fields: this.toFields(doc.data),
                            updateTime: this.toVersion(doc.version)
                        },
                        targetIds: watchChange.updatedTargetIds,
                        removedTargetIds: watchChange.removedTargetIds
                    }
                };
            }
            else if (watchChange.newDoc instanceof NoDocument) {
                var doc = watchChange.newDoc;
                return {
                    documentDelete: {
                        document: this.toName(doc.key),
                        readTime: this.toVersion(doc.version),
                        removedTargetIds: watchChange.removedTargetIds
                    }
                };
            }
            else if (watchChange.newDoc === null) {
                return {
                    documentRemove: {
                        document: this.toName(watchChange.key),
                        removedTargetIds: watchChange.removedTargetIds
                    }
                };
            }
        }
        if (watchChange instanceof WatchTargetChange) {
            var cause = undefined;
            if (watchChange.cause) {
                cause = {
                    code: mapRpcCodeFromCode(watchChange.cause.code),
                    message: watchChange.cause.message
                };
            }
            return {
                targetChange: {
                    targetChangeType: this.toWatchTargetChangeState(watchChange.state),
                    targetIds: watchChange.targetIds,
                    resumeToken: this.unsafeCastProtoByteString(watchChange.resumeToken),
                    cause: cause
                }
            };
        }
        return fail('Unrecognized watch change: ' + JSON.stringify(watchChange));
    };
    JsonProtoSerializer.prototype.fromWatchChange = function (change) {
        // tslint:disable-next-line:no-any
        var type = change['response_type'];
        var watchChange;
        if (hasTag(change, type, 'targetChange')) {
            assertPresent(change.targetChange, 'targetChange');
            // proto3 default value is unset in JSON (undefined), so use 'NO_CHANGE'
            // if unset
            var state = this.fromWatchTargetChangeState(change.targetChange.targetChangeType || 'NO_CHANGE');
            var targetIds = change.targetChange.targetIds || [];
            var resumeToken = change.targetChange.resumeToken || this.emptyByteString();
            var causeProto = change.targetChange.cause;
            var cause = causeProto && this.fromRpcStatus(causeProto);
            watchChange = new WatchTargetChange(state, targetIds, resumeToken, cause || null);
        }
        else if (hasTag(change, type, 'documentChange')) {
            assertPresent(change.documentChange, 'documentChange');
            assertPresent(change.documentChange.document, 'documentChange.name');
            assertPresent(change.documentChange.document.name, 'documentChange.document.name');
            assertPresent(change.documentChange.document.updateTime, 'documentChange.document.updateTime');
            var entityChange = change.documentChange;
            var key = this.fromName(entityChange.document.name);
            var version = this.fromVersion(entityChange.document.updateTime);
            var fields = this.fromFields(entityChange.document.fields || {});
            // The document may soon be re-serialized back to protos in order to store it in local
            // persistence. Memoize the encoded form to avoid encoding it again.
            var doc = new Document(key, version, fields, {}, entityChange.document);
            var updatedTargetIds = entityChange.targetIds || [];
            var removedTargetIds = entityChange.removedTargetIds || [];
            watchChange = new DocumentWatchChange(updatedTargetIds, removedTargetIds, doc.key, doc);
        }
        else if (hasTag(change, type, 'documentDelete')) {
            assertPresent(change.documentDelete, 'documentDelete');
            assertPresent(change.documentDelete.document, 'documentDelete.document');
            var docDelete = change.documentDelete;
            var key = this.fromName(docDelete.document);
            var version = docDelete.readTime
                ? this.fromVersion(docDelete.readTime)
                : SnapshotVersion.forDeletedDoc();
            var doc = new NoDocument(key, version);
            var removedTargetIds = docDelete.removedTargetIds || [];
            watchChange = new DocumentWatchChange([], removedTargetIds, doc.key, doc);
        }
        else if (hasTag(change, type, 'documentRemove')) {
            assertPresent(change.documentRemove, 'documentRemove');
            assertPresent(change.documentRemove.document, 'documentRemove');
            var docRemove = change.documentRemove;
            var key = this.fromName(docRemove.document);
            var removedTargetIds = docRemove.removedTargetIds || [];
            watchChange = new DocumentWatchChange([], removedTargetIds, key, null);
        }
        else if (hasTag(change, type, 'filter')) {
            // TODO(dimond): implement existence filter parsing with strategy.
            assertPresent(change.filter, 'filter');
            assertPresent(change.filter.targetId, 'filter.targetId');
            var filter = change.filter;
            var count = filter.count || 0;
            var existenceFilter = new ExistenceFilter(count);
            var targetId = filter.targetId;
            watchChange = new ExistenceFilterChange(targetId, existenceFilter);
        }
        else {
            return fail('Unknown change type ' + JSON.stringify(change));
        }
        return watchChange;
    };
    JsonProtoSerializer.prototype.fromWatchTargetChangeState = function (state) {
        if (state === 'NO_CHANGE') {
            return WatchTargetChangeState.NoChange;
        }
        else if (state === 'ADD') {
            return WatchTargetChangeState.Added;
        }
        else if (state === 'REMOVE') {
            return WatchTargetChangeState.Removed;
        }
        else if (state === 'CURRENT') {
            return WatchTargetChangeState.Current;
        }
        else if (state === 'RESET') {
            return WatchTargetChangeState.Reset;
        }
        else {
            return fail('Got unexpected TargetChange.state: ' + state);
        }
    };
    JsonProtoSerializer.prototype.versionFromListenResponse = function (change) {
        // We have only reached a consistent snapshot for the entire stream if there
        // is a read_time set and it applies to all targets (i.e. the list of
        // targets is empty). The backend is guaranteed to send such responses.
        // tslint:disable-next-line:no-any
        var type = change['response_type'];
        if (!hasTag(change, type, 'targetChange')) {
            return SnapshotVersion.MIN;
        }
        var targetChange = change.targetChange;
        if (targetChange.targetIds && targetChange.targetIds.length) {
            return SnapshotVersion.MIN;
        }
        if (!targetChange.readTime) {
            return SnapshotVersion.MIN;
        }
        return this.fromVersion(targetChange.readTime);
    };
    JsonProtoSerializer.prototype.toMutation = function (mutation) {
        var _this = this;
        var result;
        if (mutation instanceof SetMutation) {
            result = {
                update: this.toMutationDocument(mutation.key, mutation.value)
            };
        }
        else if (mutation instanceof DeleteMutation) {
            result = { delete: this.toName(mutation.key) };
        }
        else if (mutation instanceof PatchMutation) {
            result = {
                update: this.toMutationDocument(mutation.key, mutation.data),
                updateMask: this.toDocumentMask(mutation.fieldMask)
            };
        }
        else if (mutation instanceof TransformMutation) {
            result = {
                transform: {
                    document: this.toName(mutation.key),
                    fieldTransforms: mutation.fieldTransforms.map(function (transform) {
                        return _this.toFieldTransform(transform);
                    })
                }
            };
        }
        else {
            return fail('Unknown mutation type ' + mutation.type);
        }
        if (!mutation.precondition.isNone) {
            result.currentDocument = this.toPrecondition(mutation.precondition);
        }
        return result;
    };
    JsonProtoSerializer.prototype.fromMutation = function (proto) {
        var _this = this;
        var precondition = proto.currentDocument
            ? this.fromPrecondition(proto.currentDocument)
            : Precondition.NONE;
        if (proto.update) {
            assertPresent(proto.update.name, 'name');
            var key = this.fromName(proto.update.name);
            var value = this.fromFields(proto.update.fields || {});
            if (proto.updateMask) {
                var fieldMask = this.fromDocumentMask(proto.updateMask);
                return new PatchMutation(key, value, fieldMask, precondition);
            }
            else {
                return new SetMutation(key, value, precondition);
            }
        }
        else if (proto.delete) {
            var key = this.fromName(proto.delete);
            return new DeleteMutation(key, precondition);
        }
        else if (proto.transform) {
            var key = this.fromName(proto.transform.document);
            var fieldTransforms = proto.transform.fieldTransforms.map(function (transform) {
                return _this.fromFieldTransform(transform);
            });
            assert(precondition.exists === true, 'Transforms only support precondition "exists == true"');
            return new TransformMutation(key, fieldTransforms);
        }
        else {
            return fail('unknown mutation proto: ' + JSON.stringify(proto));
        }
    };
    JsonProtoSerializer.prototype.toPrecondition = function (precondition) {
        assert(!precondition.isNone, "Can't serialize an empty precondition");
        if (precondition.updateTime !== undefined) {
            return {
                updateTime: this.toVersion(precondition.updateTime)
            };
        }
        else if (precondition.exists !== undefined) {
            return { exists: precondition.exists };
        }
        else {
            return fail('Unknown precondition');
        }
    };
    JsonProtoSerializer.prototype.fromPrecondition = function (precondition) {
        if (precondition.updateTime !== undefined) {
            return Precondition.updateTime(this.fromVersion(precondition.updateTime));
        }
        else if (precondition.exists !== undefined) {
            return Precondition.exists(precondition.exists);
        }
        else {
            return Precondition.NONE;
        }
    };
    JsonProtoSerializer.prototype.fromWriteResult = function (proto, commitTime) {
        var _this = this;
        // NOTE: Deletes don't have an updateTime.
        var version = proto.updateTime
            ? this.fromVersion(proto.updateTime)
            : this.fromVersion(commitTime);
        var transformResults = null;
        if (proto.transformResults && proto.transformResults.length > 0) {
            transformResults = proto.transformResults.map(function (result) {
                return _this.fromValue(result);
            });
        }
        return new MutationResult(version, transformResults);
    };
    JsonProtoSerializer.prototype.fromWriteResults = function (protos, commitTime) {
        var _this = this;
        if (protos && protos.length > 0) {
            assert(commitTime !== undefined, 'Received a write result without a commit time');
            return protos.map(function (proto) { return _this.fromWriteResult(proto, commitTime); });
        }
        else {
            return [];
        }
    };
    JsonProtoSerializer.prototype.toFieldTransform = function (fieldTransform) {
        var _this = this;
        var transform = fieldTransform.transform;
        if (transform instanceof ServerTimestampTransform) {
            return {
                fieldPath: fieldTransform.field.canonicalString(),
                setToServerValue: 'REQUEST_TIME'
            };
        }
        else if (transform instanceof ArrayUnionTransformOperation) {
            return {
                fieldPath: fieldTransform.field.canonicalString(),
                appendMissingElements: {
                    values: transform.elements.map(function (v) { return _this.toValue(v); })
                }
            };
        }
        else if (transform instanceof ArrayRemoveTransformOperation) {
            return {
                fieldPath: fieldTransform.field.canonicalString(),
                removeAllFromArray: {
                    values: transform.elements.map(function (v) { return _this.toValue(v); })
                }
            };
        }
        else if (transform instanceof NumericIncrementTransformOperation) {
            return {
                fieldPath: fieldTransform.field.canonicalString(),
                increment: this.toValue(transform.operand)
            };
        }
        else {
            throw fail('Unknown transform: ' + fieldTransform.transform);
        }
    };
    JsonProtoSerializer.prototype.fromFieldTransform = function (proto) {
        var _this = this;
        // tslint:disable-next-line:no-any We need to match generated Proto types.
        var type = proto['transform_type'];
        var transform = null;
        if (hasTag(proto, type, 'setToServerValue')) {
            assert(proto.setToServerValue === 'REQUEST_TIME', 'Unknown server value transform proto: ' + JSON.stringify(proto));
            transform = ServerTimestampTransform.instance;
        }
        else if (hasTag(proto, type, 'appendMissingElements')) {
            var values = proto.appendMissingElements.values || [];
            transform = new ArrayUnionTransformOperation(values.map(function (v) { return _this.fromValue(v); }));
        }
        else if (hasTag(proto, type, 'removeAllFromArray')) {
            var values = proto.removeAllFromArray.values || [];
            transform = new ArrayRemoveTransformOperation(values.map(function (v) { return _this.fromValue(v); }));
        }
        else if (hasTag(proto, type, 'increment')) {
            var operand = this.fromValue(proto.increment);
            assert(operand instanceof NumberValue, 'NUMERIC_ADD transform requires a NumberValue');
            transform = new NumericIncrementTransformOperation(operand);
        }
        else {
            fail('Unknown transform proto: ' + JSON.stringify(proto));
        }
        var fieldPath = FieldPath.fromServerFormat(proto.fieldPath);
        return new FieldTransform(fieldPath, transform);
    };
    JsonProtoSerializer.prototype.toDocumentsTarget = function (query) {
        return { documents: [this.toQueryPath(query.path)] };
    };
    JsonProtoSerializer.prototype.fromDocumentsTarget = function (documentsTarget) {
        var count = documentsTarget.documents.length;
        assert(count === 1, 'DocumentsTarget contained other than 1 document: ' + count);
        var name = documentsTarget.documents[0];
        return Query.atPath(this.fromQueryPath(name));
    };
    JsonProtoSerializer.prototype.toQueryTarget = function (query) {
        // Dissect the path into parent, collectionId, and optional key filter.
        var result = { structuredQuery: {} };
        var path = query.path;
        if (query.collectionGroup !== null) {
            assert(path.length % 2 === 0, 'Collection Group queries should be within a document path or root.');
            result.parent = this.toQueryPath(path);
            result.structuredQuery.from = [
                {
                    collectionId: query.collectionGroup,
                    allDescendants: true
                }
            ];
        }
        else {
            assert(path.length % 2 !== 0, 'Document queries with filters are not supported.');
            result.parent = this.toQueryPath(path.popLast());
            result.structuredQuery.from = [{ collectionId: path.lastSegment() }];
        }
        var where = this.toFilter(query.filters);
        if (where) {
            result.structuredQuery.where = where;
        }
        var orderBy = this.toOrder(query.orderBy);
        if (orderBy) {
            result.structuredQuery.orderBy = orderBy;
        }
        var limit = this.toInt32Value(query.limit);
        if (limit !== undefined) {
            result.structuredQuery.limit = limit;
        }
        if (query.startAt) {
            result.structuredQuery.startAt = this.toCursor(query.startAt);
        }
        if (query.endAt) {
            result.structuredQuery.endAt = this.toCursor(query.endAt);
        }
        return result;
    };
    JsonProtoSerializer.prototype.fromQueryTarget = function (target) {
        var path = this.fromQueryPath(target.parent);
        var query = target.structuredQuery;
        var fromCount = query.from ? query.from.length : 0;
        var collectionGroup = null;
        if (fromCount > 0) {
            assert(fromCount === 1, 'StructuredQuery.from with more than one collection is not supported.');
            var from = query.from[0];
            if (from.allDescendants) {
                collectionGroup = from.collectionId;
            }
            else {
                path = path.child(from.collectionId);
            }
        }
        var filterBy = [];
        if (query.where) {
            filterBy = this.fromFilter(query.where);
        }
        var orderBy = [];
        if (query.orderBy) {
            orderBy = this.fromOrder(query.orderBy);
        }
        var limit = null;
        if (query.limit) {
            limit = this.fromInt32Value(query.limit);
        }
        var startAt = null;
        if (query.startAt) {
            startAt = this.fromCursor(query.startAt);
        }
        var endAt = null;
        if (query.endAt) {
            endAt = this.fromCursor(query.endAt);
        }
        return new Query(path, collectionGroup, orderBy, filterBy, limit, startAt, endAt);
    };
    JsonProtoSerializer.prototype.toListenRequestLabels = function (queryData) {
        var value = this.toLabel(queryData.purpose);
        if (value == null) {
            return null;
        }
        else {
            return {
                'goog-listen-tags': value
            };
        }
    };
    JsonProtoSerializer.prototype.toLabel = function (purpose) {
        switch (purpose) {
            case QueryPurpose.Listen:
                return null;
            case QueryPurpose.ExistenceFilterMismatch:
                return 'existence-filter-mismatch';
            case QueryPurpose.LimboResolution:
                return 'limbo-document';
            default:
                return fail('Unrecognized query purpose: ' + purpose);
        }
    };
    JsonProtoSerializer.prototype.toTarget = function (queryData) {
        var result;
        var query = queryData.query;
        if (query.isDocumentQuery()) {
            result = { documents: this.toDocumentsTarget(query) };
        }
        else {
            result = { query: this.toQueryTarget(query) };
        }
        result.targetId = queryData.targetId;
        if (queryData.resumeToken.length > 0) {
            result.resumeToken = this.unsafeCastProtoByteString(queryData.resumeToken);
        }
        return result;
    };
    JsonProtoSerializer.prototype.toFilter = function (filters) {
        var _this = this;
        if (filters.length === 0)
            return;
        var protos = filters.map(function (filter) {
            return filter instanceof RelationFilter
                ? _this.toRelationFilter(filter)
                : _this.toUnaryFilter(filter);
        });
        if (protos.length === 1) {
            return protos[0];
        }
        return { compositeFilter: { op: 'AND', filters: protos } };
    };
    JsonProtoSerializer.prototype.fromFilter = function (filter) {
        var _this = this;
        if (!filter) {
            return [];
        }
        else if (filter.unaryFilter !== undefined) {
            return [this.fromUnaryFilter(filter)];
        }
        else if (filter.fieldFilter !== undefined) {
            return [this.fromRelationFilter(filter)];
        }
        else if (filter.compositeFilter !== undefined) {
            return filter.compositeFilter
                .filters.map(function (f) { return _this.fromFilter(f); })
                .reduce(function (accum, current) { return accum.concat(current); });
        }
        else {
            return fail('Unknown filter: ' + JSON.stringify(filter));
        }
    };
    JsonProtoSerializer.prototype.toOrder = function (orderBys) {
        var _this = this;
        if (orderBys.length === 0)
            return;
        return orderBys.map(function (order) { return _this.toPropertyOrder(order); });
    };
    JsonProtoSerializer.prototype.fromOrder = function (orderBys) {
        var _this = this;
        return orderBys.map(function (order) { return _this.fromPropertyOrder(order); });
    };
    JsonProtoSerializer.prototype.toCursor = function (cursor) {
        var _this = this;
        return {
            before: cursor.before,
            values: cursor.position.map(function (component) { return _this.toValue(component); })
        };
    };
    JsonProtoSerializer.prototype.fromCursor = function (cursor) {
        var _this = this;
        var before = !!cursor.before;
        var position = cursor.values.map(function (component) { return _this.fromValue(component); });
        return new Bound(position, before);
    };
    // visible for testing
    JsonProtoSerializer.prototype.toDirection = function (dir) {
        return DIRECTIONS[dir.name];
    };
    // visible for testing
    JsonProtoSerializer.prototype.fromDirection = function (dir) {
        switch (dir) {
            case 'ASCENDING':
                return Direction.ASCENDING;
            case 'DESCENDING':
                return Direction.DESCENDING;
            default:
                return undefined;
        }
    };
    // visible for testing
    JsonProtoSerializer.prototype.toOperatorName = function (op) {
        return OPERATORS[op.name];
    };
    JsonProtoSerializer.prototype.fromOperatorName = function (op) {
        switch (op) {
            case 'EQUAL':
                return RelationOp.EQUAL;
            case 'GREATER_THAN':
                return RelationOp.GREATER_THAN;
            case 'GREATER_THAN_OR_EQUAL':
                return RelationOp.GREATER_THAN_OR_EQUAL;
            case 'LESS_THAN':
                return RelationOp.LESS_THAN;
            case 'LESS_THAN_OR_EQUAL':
                return RelationOp.LESS_THAN_OR_EQUAL;
            case 'ARRAY_CONTAINS':
                return RelationOp.ARRAY_CONTAINS;
            case 'OPERATOR_UNSPECIFIED':
                return fail('Unspecified relation');
            default:
                return fail('Unknown relation');
        }
    };
    JsonProtoSerializer.prototype.toFieldPathReference = function (path) {
        return { fieldPath: path.canonicalString() };
    };
    JsonProtoSerializer.prototype.fromFieldPathReference = function (fieldReference) {
        return FieldPath.fromServerFormat(fieldReference.fieldPath);
    };
    // visible for testing
    JsonProtoSerializer.prototype.toPropertyOrder = function (orderBy) {
        return {
            field: this.toFieldPathReference(orderBy.field),
            direction: this.toDirection(orderBy.dir)
        };
    };
    JsonProtoSerializer.prototype.fromPropertyOrder = function (orderBy) {
        return new OrderBy(this.fromFieldPathReference(orderBy.field), this.fromDirection(orderBy.direction));
    };
    // visible for testing
    JsonProtoSerializer.prototype.toRelationFilter = function (filter) {
        if (filter instanceof RelationFilter) {
            return {
                fieldFilter: {
                    field: this.toFieldPathReference(filter.field),
                    op: this.toOperatorName(filter.op),
                    value: this.toValue(filter.value)
                }
            };
        }
        else {
            return fail('Unrecognized filter: ' + JSON.stringify(filter));
        }
    };
    JsonProtoSerializer.prototype.fromRelationFilter = function (filter) {
        return new RelationFilter(this.fromFieldPathReference(filter.fieldFilter.field), this.fromOperatorName(filter.fieldFilter.op), this.fromValue(filter.fieldFilter.value));
    };
    // visible for testing
    JsonProtoSerializer.prototype.toUnaryFilter = function (filter) {
        if (filter instanceof NanFilter) {
            return {
                unaryFilter: {
                    field: this.toFieldPathReference(filter.field),
                    op: 'IS_NAN'
                }
            };
        }
        else if (filter instanceof NullFilter) {
            return {
                unaryFilter: {
                    field: this.toFieldPathReference(filter.field),
                    op: 'IS_NULL'
                }
            };
        }
        else {
            return fail('Unrecognized filter: ' + JSON.stringify(filter));
        }
    };
    JsonProtoSerializer.prototype.fromUnaryFilter = function (filter) {
        switch (filter.unaryFilter.op) {
            case 'IS_NAN':
                var nanField = this.fromFieldPathReference(filter.unaryFilter.field);
                return new NanFilter(nanField);
            case 'IS_NULL':
                var nullField = this.fromFieldPathReference(filter.unaryFilter.field);
                return new NullFilter(nullField);
            case 'OPERATOR_UNSPECIFIED':
                return fail('Unspecified filter');
            default:
                return fail('Unknown filter');
        }
    };
    JsonProtoSerializer.prototype.toDocumentMask = function (fieldMask) {
        var canonicalFields = [];
        fieldMask.fields.forEach(function (field) {
            return canonicalFields.push(field.canonicalString());
        });
        return {
            fieldPaths: canonicalFields
        };
    };
    JsonProtoSerializer.prototype.fromDocumentMask = function (proto) {
        var paths = proto.fieldPaths || [];
        var fields = paths.map(function (path) { return FieldPath.fromServerFormat(path); });
        return FieldMask.fromArray(fields);
    };
    return JsonProtoSerializer;
}());
/**
 * Checks for a specific oneof tag in a protocol buffer message.
 *
 * This intentionally accommodates two distinct cases:
 *
 * 1) Messages containing a type tag: these are the format produced by GRPC in
 * return values. These may contain default-value mappings for all tags in the
 * oneof but the type tag specifies which one was actually set.
 *
 * 2) Messages that don't contain a type tag: these are the format required by
 * GRPC as inputs. If we emitted objects with type tags, ProtoBuf.js would
 * choke claiming that the tags aren't fields in the Message.
 *
 * Allowing both formats here makes the serializer able to consume the outputs
 * it produces: for all messages it supports, fromX(toX(value)) == value.
 *
 * Note that case 2 suffers from ambiguity: if multiple tags are present
 * without a type tag then the callers are structured in such a way that the
 * first invocation will win. Since we only parse in this mode when parsing
 * the output of a serialize method this works, but it's not a general
 * solution.
 *
 * Unfortunately there is no general solution here because proto3 makes it
 * impossible to distinguish unset from explicitly set fields: both have the
 * default value for the type. Without the type tag but multiple value tags
 * it's possible to have default values for each tag in the oneof and not be
 * able to know which was actually in effect.
 */
function hasTag(obj, type, tag) {
    return type === tag || (!type && tag in obj);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Provides a simple helper class that implements the Stream interface to
 * bridge to other implementations that are streams but do not implement the
 * interface. The stream callbacks are invoked with the callOn... methods.
 */
var StreamBridge = /** @class */ (function () {
    function StreamBridge(args) {
        this.sendFn = args.sendFn;
        this.closeFn = args.closeFn;
    }
    StreamBridge.prototype.onOpen = function (callback) {
        assert(!this.wrappedOnOpen, 'Called onOpen on stream twice!');
        this.wrappedOnOpen = callback;
    };
    StreamBridge.prototype.onClose = function (callback) {
        assert(!this.wrappedOnClose, 'Called onClose on stream twice!');
        this.wrappedOnClose = callback;
    };
    StreamBridge.prototype.onMessage = function (callback) {
        assert(!this.wrappedOnMessage, 'Called onMessage on stream twice!');
        this.wrappedOnMessage = callback;
    };
    StreamBridge.prototype.close = function () {
        this.closeFn();
    };
    StreamBridge.prototype.send = function (msg) {
        this.sendFn(msg);
    };
    StreamBridge.prototype.callOnOpen = function () {
        assert(this.wrappedOnOpen !== undefined, 'Cannot call onOpen because no callback was set');
        this.wrappedOnOpen();
    };
    StreamBridge.prototype.callOnClose = function (err) {
        assert(this.wrappedOnClose !== undefined, 'Cannot call onClose because no callback was set');
        this.wrappedOnClose(err);
    };
    StreamBridge.prototype.callOnMessage = function (msg) {
        assert(this.wrappedOnMessage !== undefined, 'Cannot call onMessage because no callback was set');
        this.wrappedOnMessage(msg);
    };
    return StreamBridge;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Utilities for dealing with node.js-style APIs. See nodePromise for more
 * details.
 */
/**
 * Creates a node-style callback that resolves or rejects a new Promise. The
 * callback is passed to the given action which can then use the callback as
 * a parameter to a node-style function.
 *
 * The intent is to directly bridge a node-style function (which takes a
 * callback) into a Promise without manually converting between the node-style
 * callback and the promise at each call.
 *
 * In effect it allows you to convert:
 *
 * @example
 * new Promise((resolve: (value?: fs.Stats) => void,
 *              reject: (error?: any) => void) => {
 *   fs.stat(path, (error?: any, stat?: fs.Stats) => {
 *     if (error) {
 *       reject(error);
 *     } else {
 *       resolve(stat);
 *     }
 *   });
 * });
 *
 * Into
 * @example
 * nodePromise((callback: NodeCallback<fs.Stats>) => {
 *   fs.stat(path, callback);
 * });
 *
 * @param action a function that takes a node-style callback as an argument and
 *     then uses that callback to invoke some node-style API.
 * @return a new Promise which will be rejected if the callback is given the
 *     first Error parameter or will resolve to the value given otherwise.
 */
function nodePromise(action) {
    return new Promise(function (resolve, reject) {
        action(function (error, value) {
            if (error) {
                reject(error);
            }
            else {
                resolve(value);
            }
        });
    });
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var Deferred = /** @class */ (function () {
    function Deferred() {
        var _this = this;
        this.promise = new Promise(function (resolve, reject) {
            _this.resolve = resolve;
            _this.reject = reject;
        });
    }
    return Deferred;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var SDK_VERSION$1 = firebase.SDK_VERSION;
var grpcVersion = require('grpc/package.json').version;
var LOG_TAG = 'Connection';
// TODO(b/38203344): The SDK_VERSION is set independently from Firebase because
// we are doing out-of-band releases. Once we release as part of Firebase, we
// should use the Firebase version instead.
var X_GOOG_API_CLIENT_VALUE = "gl-node/" + process.versions.node + " fire/" + SDK_VERSION$1 + " grpc/" + grpcVersion;
function createMetadata(databaseInfo, token) {
    assert(token === null || token.type === 'OAuth', 'If provided, token must be OAuth');
    var metadata = new grpc.Metadata();
    if (token) {
        for (var header in token.authHeaders) {
            if (token.authHeaders.hasOwnProperty(header)) {
                metadata.set(header, token.authHeaders[header]);
            }
        }
    }
    metadata.set('x-goog-api-client', X_GOOG_API_CLIENT_VALUE);
    // This header is used to improve routing and project isolation by the
    // backend.
    metadata.set('google-cloud-resource-prefix', "projects/" + databaseInfo.databaseId.projectId + "/" +
        ("databases/" + databaseInfo.databaseId.database));
    return metadata;
}
/**
 * A Connection implemented by GRPC-Node.
 */
var GrpcConnection = /** @class */ (function () {
    function GrpcConnection(protos, databaseInfo) {
        this.databaseInfo = databaseInfo;
        // We cache stubs for the most-recently-used token.
        this.cachedStub = null;
        this.firestore = protos['google']['firestore']['v1'];
    }
    GrpcConnection.prototype.ensureActiveStub = function () {
        if (!this.cachedStub) {
            debug(LOG_TAG, 'Creating Firestore stub.');
            var credentials = this.databaseInfo.ssl
                ? grpc.credentials.createSsl()
                : grpc.credentials.createInsecure();
            this.cachedStub = new this.firestore.Firestore(this.databaseInfo.host, credentials, {
                // We do our own connection backoff (that for example is aware of whether or
                // not a write stream error is permanent or not) so we don't want gRPC to do
                // backoff on top of that. 100ms is the minimum value that gRPC allows.
                'grpc.initial_reconnect_backoff_ms': 100,
                'grpc.max_reconnect_backoff_ms': 100
            });
        }
        return this.cachedStub;
    };
    GrpcConnection.prototype.invokeRPC = function (rpcName, request, token) {
        var stub = this.ensureActiveStub();
        var metadata = createMetadata(this.databaseInfo, token);
        return nodePromise(function (callback) {
            debug(LOG_TAG, "RPC '" + rpcName + "' invoked with request:", request);
            return stub[rpcName](request, metadata, function (grpcError, value) {
                if (grpcError) {
                    debug(LOG_TAG, "RPC '" + rpcName + "' failed with error:", grpcError);
                    callback(new FirestoreError(mapCodeFromRpcCode(grpcError.code), grpcError.message));
                }
                else {
                    debug(LOG_TAG, "RPC '" + rpcName + "' completed with response:", value);
                    callback(undefined, value);
                }
            });
        });
    };
    GrpcConnection.prototype.invokeStreamingRPC = function (rpcName, request, token) {
        var results = [];
        var responseDeferred = new Deferred();
        debug(LOG_TAG, "RPC '" + rpcName + "' invoked (streaming) with request:", request);
        var stub = this.ensureActiveStub();
        var metadata = createMetadata(this.databaseInfo, token);
        var stream = stub[rpcName](request, metadata);
        stream.on('data', function (response) {
            debug(LOG_TAG, "RPC " + rpcName + " received result:", response);
            results.push(response);
        });
        stream.on('end', function () {
            debug(LOG_TAG, "RPC '" + rpcName + "' completed.");
            responseDeferred.resolve(results);
        });
        stream.on('error', function (grpcError) {
            debug(LOG_TAG, "RPC '" + rpcName + "' failed with error:", grpcError);
            var code = mapCodeFromRpcCode(grpcError.code);
            responseDeferred.reject(new FirestoreError(code, grpcError.message));
        });
        return responseDeferred.promise;
    };
    // TODO(mikelehen): This "method" is a monster. Should be refactored.
    GrpcConnection.prototype.openStream = function (rpcName, token) {
        var stub = this.ensureActiveStub();
        var metadata = createMetadata(this.databaseInfo, token);
        var grpcStream = stub[rpcName](metadata);
        var closed = false;
        var close;
        var stream = new StreamBridge({
            sendFn: function (msg) {
                if (!closed) {
                    debug(LOG_TAG, 'GRPC stream sending:', msg);
                    try {
                        grpcStream.write(msg);
                    }
                    catch (e) {
                        // This probably means we didn't conform to the proto.  Make sure to
                        // log the message we sent.
                        error('Failure sending:', msg);
                        error('Error:', e);
                        throw e;
                    }
                }
                else {
                    debug(LOG_TAG, 'Not sending because gRPC stream is closed:', msg);
                }
            },
            closeFn: function () {
                debug(LOG_TAG, 'GRPC stream closed locally via close().');
                close();
            }
        });
        close = function (err) {
            if (!closed) {
                closed = true;
                stream.callOnClose(err);
                grpcStream.end();
            }
        };
        grpcStream.on('data', function (msg) {
            if (!closed) {
                debug(LOG_TAG, 'GRPC stream received:', msg);
                stream.callOnMessage(msg);
            }
        });
        grpcStream.on('end', function () {
            debug(LOG_TAG, 'GRPC stream ended.');
            close();
        });
        grpcStream.on('error', function (grpcError) {
            debug(LOG_TAG, 'GRPC stream error. Code:', grpcError.code, 'Message:', grpcError.message);
            var code = mapCodeFromRpcCode(grpcError.code);
            close(new FirestoreError(code, grpcError.message));
        });
        debug(LOG_TAG, 'Opening GRPC stream');
        // TODO(dimond): Since grpc has no explicit open status (or does it?) we
        // simulate an onOpen in the next loop after the stream had it's listeners
        // registered
        setTimeout(function () {
            stream.callOnOpen();
        }, 0);
        return stream;
    };
    return GrpcConnection;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Loads the protocol buffer definitions for Firestore.
 *
 * @returns The GrpcObject representing our protos.
 */
function loadProtos() {
    var options = {
        // Beware that converting fields to camel case does not convert the tag
        // fields in oneof groups (!!!). This will likely be fixed when we upgrade
        // to protobufjs 6.x
        convertFieldsToCamelCase: true
    };
    var root = path.resolve(__dirname, "src/protos");
    var firestoreProtoFile = {
        root: root,
        file: 'google/firestore/v1/firestore.proto'
    };
    return grpc.load(firestoreProtoFile, /*format=*/ 'proto', options);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var NodePlatform = /** @class */ (function () {
    function NodePlatform() {
        this.base64Available = true;
        this.emptyByteString = new Uint8Array(0);
        this.document = null;
    }
    Object.defineProperty(NodePlatform.prototype, "window", {
        get: function () {
            if (process.env.USE_MOCK_PERSISTENCE === 'YES') {
                return window;
            }
            return null;
        },
        enumerable: true,
        configurable: true
    });
    NodePlatform.prototype.loadConnection = function (databaseInfo) {
        var protos = loadProtos();
        return Promise.resolve(new GrpcConnection(protos, databaseInfo));
    };
    NodePlatform.prototype.newSerializer = function (partitionId) {
        return new JsonProtoSerializer(partitionId, { useProto3Json: false });
    };
    NodePlatform.prototype.formatJSON = function (value) {
        // util.inspect() results in much more readable output than JSON.stringify()
        return util.inspect(value, { depth: 100 });
    };
    NodePlatform.prototype.atob = function (encoded) {
        // Node actually doesn't validate base64 strings.
        // A quick sanity check that is not a fool-proof validation
        if (/[^-A-Za-z0-9+/=]/.test(encoded)) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Not a valid Base64 string: ' + encoded);
        }
        return new Buffer(encoded, 'base64').toString('binary');
    };
    NodePlatform.prototype.btoa = function (raw) {
        return new Buffer(raw, 'binary').toString('base64');
    };
    return NodePlatform;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * This code needs to run before Firestore is used. This can be achieved in
 * several ways:
 *   1) Through the JSCompiler compiling this code and then (automatically)
 *      executing it before exporting the Firestore symbols.
 *   2) Through importing this module first in a Firestore main module
 */
PlatformSupport.setPlatform(new NodePlatform());

/**
 * @license
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * `ListenSequence` is a monotonic sequence. It is initialized with a minimum value to
 * exceed. All subsequent calls to next will return increasing values. If provided with a
 * `SequenceNumberSyncer`, it will additionally bump its next value when told of a new value, as
 * well as write out sequence numbers that it produces via `next()`.
 */
var ListenSequence = /** @class */ (function () {
    function ListenSequence(previousValue, sequenceNumberSyncer) {
        var _this = this;
        this.previousValue = previousValue;
        if (sequenceNumberSyncer) {
            sequenceNumberSyncer.sequenceNumberHandler = function (sequenceNumber) {
                return _this.setPreviousValue(sequenceNumber);
            };
            this.writeNewSequenceNumber = function (sequenceNumber) {
                return sequenceNumberSyncer.writeSequenceNumber(sequenceNumber);
            };
        }
    }
    ListenSequence.prototype.setPreviousValue = function (externalPreviousValue) {
        this.previousValue = Math.max(externalPreviousValue, this.previousValue);
        return this.previousValue;
    };
    ListenSequence.prototype.next = function () {
        var nextValue = ++this.previousValue;
        if (this.writeNewSequenceNumber) {
            this.writeNewSequenceNumber(nextValue);
        }
        return nextValue;
    };
    ListenSequence.INVALID = -1;
    return ListenSequence;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Wellknown "timer" IDs used when scheduling delayed operations on the
 * AsyncQueue. These IDs can then be used from tests to check for the presence
 * of operations or to run them early.
 *
 * The string values are used when encoding these timer IDs in JSON spec tests.
 */
var TimerId;
(function (TimerId) {
    /** All can be used with runDelayedOperationsEarly() to run all timers. */
    TimerId["All"] = "all";
    /**
     * The following 4 timers are used in persistent_stream.ts for the listen and
     * write streams. The "Idle" timer is used to close the stream due to
     * inactivity. The "ConnectionBackoff" timer is used to restart a stream once
     * the appropriate backoff delay has elapsed.
     */
    TimerId["ListenStreamIdle"] = "listen_stream_idle";
    TimerId["ListenStreamConnectionBackoff"] = "listen_stream_connection_backoff";
    TimerId["WriteStreamIdle"] = "write_stream_idle";
    TimerId["WriteStreamConnectionBackoff"] = "write_stream_connection_backoff";
    /**
     * A timer used in online_state_tracker.ts to transition from
     * OnlineState.Unknown to Offline after a set timeout, rather than waiting
     * indefinitely for success or failure.
     */
    TimerId["OnlineStateTimeout"] = "online_state_timeout";
    /**
     * A timer used to update the client metadata in IndexedDb, which is used
     * to determine the primary leaseholder.
     */
    TimerId["ClientMetadataRefresh"] = "client_metadata_refresh";
    /** A timer used to periodically attempt LRU Garbage collection */
    TimerId["LruGarbageCollection"] = "lru_garbage_collection";
})(TimerId || (TimerId = {}));
/**
 * Represents an operation scheduled to be run in the future on an AsyncQueue.
 *
 * It is created via DelayedOperation.createAndSchedule().
 *
 * Supports cancellation (via cancel()) and early execution (via skipDelay()).
 */
var DelayedOperation = /** @class */ (function () {
    function DelayedOperation(asyncQueue, timerId, targetTimeMs, op, removalCallback) {
        this.asyncQueue = asyncQueue;
        this.timerId = timerId;
        this.targetTimeMs = targetTimeMs;
        this.op = op;
        this.removalCallback = removalCallback;
        this.deferred = new Deferred();
        this.then = this.deferred.promise.then.bind(this.deferred.promise);
        this.catch = this.deferred.promise.catch.bind(this.deferred.promise);
        // It's normal for the deferred promise to be canceled (due to cancellation)
        // and so we attach a dummy catch callback to avoid
        // 'UnhandledPromiseRejectionWarning' log spam.
        this.deferred.promise.catch(function (err) { });
    }
    /**
     * Creates and returns a DelayedOperation that has been scheduled to be
     * executed on the provided asyncQueue after the provided delayMs.
     *
     * @param asyncQueue The queue to schedule the operation on.
     * @param id A Timer ID identifying the type of operation this is.
     * @param delayMs The delay (ms) before the operation should be scheduled.
     * @param op The operation to run.
     * @param removalCallback A callback to be called synchronously once the
     *   operation is executed or canceled, notifying the AsyncQueue to remove it
     *   from its delayedOperations list.
     *   PORTING NOTE: This exists to prevent making removeDelayedOperation() and
     *   the DelayedOperation class public.
     */
    DelayedOperation.createAndSchedule = function (asyncQueue, timerId, delayMs, op, removalCallback) {
        var targetTime = Date.now() + delayMs;
        var delayedOp = new DelayedOperation(asyncQueue, timerId, targetTime, op, removalCallback);
        delayedOp.start(delayMs);
        return delayedOp;
    };
    /**
     * Starts the timer. This is called immediately after construction by
     * createAndSchedule().
     */
    DelayedOperation.prototype.start = function (delayMs) {
        var _this = this;
        this.timerHandle = setTimeout(function () { return _this.handleDelayElapsed(); }, delayMs);
    };
    /**
     * Queues the operation to run immediately (if it hasn't already been run or
     * canceled).
     */
    DelayedOperation.prototype.skipDelay = function () {
        return this.handleDelayElapsed();
    };
    /**
     * Cancels the operation if it hasn't already been executed or canceled. The
     * promise will be rejected.
     *
     * As long as the operation has not yet been run, calling cancel() provides a
     * guarantee that the operation will not be run.
     */
    DelayedOperation.prototype.cancel = function (reason) {
        if (this.timerHandle !== null) {
            this.clearTimeout();
            this.deferred.reject(new FirestoreError(Code.CANCELLED, 'Operation cancelled' + (reason ? ': ' + reason : '')));
        }
    };
    DelayedOperation.prototype.handleDelayElapsed = function () {
        var _this = this;
        this.asyncQueue.enqueueAndForget(function () {
            if (_this.timerHandle !== null) {
                _this.clearTimeout();
                return _this.op().then(function (result) {
                    return _this.deferred.resolve(result);
                });
            }
            else {
                return Promise.resolve();
            }
        });
    };
    DelayedOperation.prototype.clearTimeout = function () {
        if (this.timerHandle !== null) {
            this.removalCallback(this);
            clearTimeout(this.timerHandle);
            this.timerHandle = null;
        }
    };
    return DelayedOperation;
}());
var AsyncQueue = /** @class */ (function () {
    function AsyncQueue() {
        // The last promise in the queue.
        this.tail = Promise.resolve();
        // Operations scheduled to be queued in the future. Operations are
        // automatically removed after they are run or canceled.
        this.delayedOperations = [];
        // Flag set while there's an outstanding AsyncQueue operation, used for
        // assertion sanity-checks.
        this.operationInProgress = false;
    }
    /**
     * Adds a new operation to the queue without waiting for it to complete (i.e.
     * we ignore the Promise result).
     */
    AsyncQueue.prototype.enqueueAndForget = function (op) {
        // tslint:disable-next-line:no-floating-promises
        this.enqueue(op);
    };
    /**
     * Adds a new operation to the queue. Returns a promise that will be resolved
     * when the promise returned by the new operation is (with its value).
     */
    AsyncQueue.prototype.enqueue = function (op) {
        var _this = this;
        this.verifyNotFailed();
        var newTail = this.tail.then(function () {
            _this.operationInProgress = true;
            return op()
                .catch(function (error$1) {
                _this.failure = error$1;
                _this.operationInProgress = false;
                var message = error$1.stack || error$1.message || '';
                error('INTERNAL UNHANDLED ERROR: ', message);
                // Escape the promise chain and throw the error globally so that
                // e.g. any global crash reporting library detects and reports it.
                // (but not for simulated errors in our tests since this breaks mocha)
                if (message.indexOf('Firestore Test Simulated Error') < 0) {
                    setTimeout(function () {
                        throw error$1;
                    }, 0);
                }
                // Re-throw the error so that this.tail becomes a rejected Promise and
                // all further attempts to chain (via .then) will just short-circuit
                // and return the rejected Promise.
                throw error$1;
            })
                .then(function (result) {
                _this.operationInProgress = false;
                return result;
            });
        });
        this.tail = newTail;
        return newTail;
    };
    /**
     * Schedules an operation to be queued on the AsyncQueue once the specified
     * `delayMs` has elapsed. The returned CancelablePromise can be used to cancel
     * the operation prior to its running.
     */
    AsyncQueue.prototype.enqueueAfterDelay = function (timerId, delayMs, op) {
        var _this = this;
        this.verifyNotFailed();
        assert(delayMs >= 0, "Attempted to schedule an operation with a negative delay of " + delayMs);
        // While not necessarily harmful, we currently don't expect to have multiple
        // ops with the same timer id in the queue, so defensively reject them.
        assert(!this.containsDelayedOperation(timerId), "Attempted to schedule multiple operations with timer id " + timerId + ".");
        var delayedOp = DelayedOperation.createAndSchedule(this, timerId, delayMs, op, function (op) { return _this.removeDelayedOperation(op); });
        this.delayedOperations.push(delayedOp);
        return delayedOp;
    };
    AsyncQueue.prototype.verifyNotFailed = function () {
        if (this.failure) {
            fail('AsyncQueue is already failed: ' +
                (this.failure.stack || this.failure.message));
        }
    };
    /**
     * Verifies there's an operation currently in-progress on the AsyncQueue.
     * Unfortunately we can't verify that the running code is in the promise chain
     * of that operation, so this isn't a foolproof check, but it should be enough
     * to catch some bugs.
     */
    AsyncQueue.prototype.verifyOperationInProgress = function () {
        assert(this.operationInProgress, 'verifyOpInProgress() called when no op in progress on this queue.');
    };
    /**
     * Waits until all currently queued tasks are finished executing. Delayed
     * operations are not run.
     */
    AsyncQueue.prototype.drain = function () {
        return this.enqueue(function () { return Promise.resolve(); });
    };
    /**
     * For Tests: Determine if a delayed operation with a particular TimerId
     * exists.
     */
    AsyncQueue.prototype.containsDelayedOperation = function (timerId) {
        for (var _i = 0, _a = this.delayedOperations; _i < _a.length; _i++) {
            var op = _a[_i];
            if (op.timerId === timerId) {
                return true;
            }
        }
        return false;
    };
    /**
     * For Tests: Runs some or all delayed operations early.
     *
     * @param lastTimerId Delayed operations up to and including this TimerId will
     *  be drained. Throws if no such operation exists. Pass TimerId.All to run
     *  all delayed operations.
     * @returns a Promise that resolves once all operations have been run.
     */
    AsyncQueue.prototype.runDelayedOperationsEarly = function (lastTimerId) {
        var _this = this;
        // Note that draining may generate more delayed ops, so we do that first.
        return this.drain().then(function () {
            assert(lastTimerId === TimerId.All ||
                _this.containsDelayedOperation(lastTimerId), "Attempted to drain to missing operation " + lastTimerId);
            // Run ops in the same order they'd run if they ran naturally.
            _this.delayedOperations.sort(function (a, b) { return a.targetTimeMs - b.targetTimeMs; });
            for (var _i = 0, _a = _this.delayedOperations; _i < _a.length; _i++) {
                var op = _a[_i];
                op.skipDelay();
                if (lastTimerId !== TimerId.All && op.timerId === lastTimerId) {
                    break;
                }
            }
            return _this.drain();
        });
    };
    /** Called once a DelayedOperation is run or canceled. */
    AsyncQueue.prototype.removeDelayedOperation = function (op) {
        // NOTE: indexOf / slice are O(n), but delayedOperations is expected to be small.
        var index = this.delayedOperations.indexOf(op);
        assert(index >= 0, 'Delayed operation not found.');
        this.delayedOperations.splice(index, 1);
    };
    return AsyncQueue;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var escapeChar = '\u0001';
var encodedSeparatorChar = '\u0001';
var encodedNul = '\u0010';
var encodedEscape = '\u0011';
/**
 * Encodes a resource path into a IndexedDb-compatible string form.
 */
function encode(path) {
    var result = '';
    for (var i = 0; i < path.length; i++) {
        if (result.length > 0) {
            result = encodeSeparator(result);
        }
        result = encodeSegment(path.get(i), result);
    }
    return encodeSeparator(result);
}
/** Encodes a single segment of a resource path into the given result */
function encodeSegment(segment, resultBuf) {
    var result = resultBuf;
    var length = segment.length;
    for (var i = 0; i < length; i++) {
        var c = segment.charAt(i);
        switch (c) {
            case '\0':
                result += escapeChar + encodedNul;
                break;
            case escapeChar:
                result += escapeChar + encodedEscape;
                break;
            default:
                result += c;
        }
    }
    return result;
}
/** Encodes a path separator into the given result */
function encodeSeparator(result) {
    return result + escapeChar + encodedSeparatorChar;
}
/**
 * Decodes the given IndexedDb-compatible string form of a resource path into
 * a ResourcePath instance. Note that this method is not suitable for use with
 * decoding resource names from the server; those are One Platform format
 * strings.
 */
function decode(path) {
    // Event the empty path must encode as a path of at least length 2. A path
    // with exactly 2 must be the empty path.
    var length = path.length;
    assert(length >= 2, 'Invalid path ' + path);
    if (length === 2) {
        assert(path.charAt(0) === escapeChar && path.charAt(1) === encodedSeparatorChar, 'Non-empty path ' + path + ' had length 2');
        return ResourcePath.EMPTY_PATH;
    }
    // Escape characters cannot exist past the second-to-last position in the
    // source value.
    var lastReasonableEscapeIndex = length - 2;
    var segments = [];
    var segmentBuilder = '';
    for (var start = 0; start < length;) {
        // The last two characters of a valid encoded path must be a separator, so
        // there must be an end to this segment.
        var end = path.indexOf(escapeChar, start);
        if (end < 0 || end > lastReasonableEscapeIndex) {
            fail('Invalid encoded resource path: "' + path + '"');
        }
        var next = path.charAt(end + 1);
        switch (next) {
            case encodedSeparatorChar:
                var currentPiece = path.substring(start, end);
                var segment = void 0;
                if (segmentBuilder.length === 0) {
                    // Avoid copying for the common case of a segment that excludes \0
                    // and \001
                    segment = currentPiece;
                }
                else {
                    segmentBuilder += currentPiece;
                    segment = segmentBuilder;
                    segmentBuilder = '';
                }
                segments.push(segment);
                break;
            case encodedNul:
                segmentBuilder += path.substring(start, end);
                segmentBuilder += '\0';
                break;
            case encodedEscape:
                // The escape character can be used in the output to encode itself.
                segmentBuilder += path.substring(start, end + 1);
                break;
            default:
                fail('Invalid encoded resource path: "' + path + '"');
        }
        start = end + 2;
    }
    return new ResourcePath(segments);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var BATCHID_UNKNOWN = -1;
/**
 * A batch of mutations that will be sent as one unit to the backend.
 */
var MutationBatch = /** @class */ (function () {
    /**
     * @param batchId The unique ID of this mutation batch.
     * @param localWriteTime The original write time of this mutation.
     * @param baseMutations Mutations that are used to populate the base
     * values when this mutation is applied locally. This can be used to locally
     * overwrite values that are persisted in the remote document cache. Base
     * mutations are never sent to the backend.
     * @param mutations The user-provided mutations in this mutation batch.
     * User-provided mutations are applied both locally and remotely on the
     * backend.
     */
    function MutationBatch(batchId, localWriteTime, baseMutations, mutations) {
        this.batchId = batchId;
        this.localWriteTime = localWriteTime;
        this.baseMutations = baseMutations;
        this.mutations = mutations;
        assert(mutations.length > 0, 'Cannot create an empty mutation batch');
    }
    /**
     * Applies all the mutations in this MutationBatch to the specified document
     * to create a new remote document
     *
     * @param docKey The key of the document to apply mutations to.
     * @param maybeDoc The document to apply mutations to.
     * @param batchResult The result of applying the MutationBatch to the
     * backend.
     */
    MutationBatch.prototype.applyToRemoteDocument = function (docKey, maybeDoc, batchResult) {
        if (maybeDoc) {
            assert(maybeDoc.key.isEqual(docKey), "applyToRemoteDocument: key " + docKey + " should match maybeDoc key\n        " + maybeDoc.key);
        }
        var mutationResults = batchResult.mutationResults;
        assert(mutationResults.length === this.mutations.length, "Mismatch between mutations length\n      (" + this.mutations.length + ") and mutation results length\n      (" + mutationResults.length + ").");
        for (var i = 0; i < this.mutations.length; i++) {
            var mutation = this.mutations[i];
            if (mutation.key.isEqual(docKey)) {
                var mutationResult = mutationResults[i];
                maybeDoc = mutation.applyToRemoteDocument(maybeDoc, mutationResult);
            }
        }
        return maybeDoc;
    };
    /**
     * Computes the local view of a document given all the mutations in this
     * batch.
     *
     * @param docKey The key of the document to apply mutations to.
     * @param maybeDoc The document to apply mutations to.
     */
    MutationBatch.prototype.applyToLocalView = function (docKey, maybeDoc) {
        if (maybeDoc) {
            assert(maybeDoc.key.isEqual(docKey), "applyToLocalDocument: key " + docKey + " should match maybeDoc key\n        " + maybeDoc.key);
        }
        // First, apply the base state. This allows us to apply non-idempotent
        // transform against a consistent set of values.
        for (var _i = 0, _a = this.baseMutations; _i < _a.length; _i++) {
            var mutation = _a[_i];
            if (mutation.key.isEqual(docKey)) {
                maybeDoc = mutation.applyToLocalView(maybeDoc, maybeDoc, this.localWriteTime);
            }
        }
        var baseDoc = maybeDoc;
        // Second, apply all user-provided mutations.
        for (var _b = 0, _c = this.mutations; _b < _c.length; _b++) {
            var mutation = _c[_b];
            if (mutation.key.isEqual(docKey)) {
                maybeDoc = mutation.applyToLocalView(maybeDoc, baseDoc, this.localWriteTime);
            }
        }
        return maybeDoc;
    };
    /**
     * Computes the local view for all provided documents given the mutations in
     * this batch.
     */
    MutationBatch.prototype.applyToLocalDocumentSet = function (maybeDocs) {
        var _this = this;
        // TODO(mrschmidt): This implementation is O(n^2). If we apply the mutations
        // directly (as done in `applyToLocalView()`), we can reduce the complexity
        // to O(n).
        var mutatedDocuments = maybeDocs;
        this.mutations.forEach(function (m) {
            var mutatedDocument = _this.applyToLocalView(m.key, maybeDocs.get(m.key));
            if (mutatedDocument) {
                mutatedDocuments = mutatedDocuments.insert(m.key, mutatedDocument);
            }
        });
        return mutatedDocuments;
    };
    MutationBatch.prototype.keys = function () {
        return this.mutations.reduce(function (keys, m) { return keys.add(m.key); }, documentKeySet());
    };
    MutationBatch.prototype.isEqual = function (other) {
        return (this.batchId === other.batchId &&
            arrayEquals(this.mutations, other.mutations) &&
            arrayEquals(this.baseMutations, other.baseMutations));
    };
    return MutationBatch;
}());
/** The result of applying a mutation batch to the backend. */
var MutationBatchResult = /** @class */ (function () {
    function MutationBatchResult(batch, commitVersion, mutationResults, streamToken, 
    /**
     * A pre-computed mapping from each mutated document to the resulting
     * version.
     */
    docVersions) {
        this.batch = batch;
        this.commitVersion = commitVersion;
        this.mutationResults = mutationResults;
        this.streamToken = streamToken;
        this.docVersions = docVersions;
    }
    /**
     * Creates a new MutationBatchResult for the given batch and results. There
     * must be one result for each mutation in the batch. This static factory
     * caches a document=>version mapping (docVersions).
     */
    MutationBatchResult.from = function (batch, commitVersion, results, streamToken) {
        assert(batch.mutations.length === results.length, 'Mutations sent ' +
            batch.mutations.length +
            ' must equal results received ' +
            results.length);
        var versionMap = documentVersionMap();
        var mutations = batch.mutations;
        for (var i = 0; i < mutations.length; i++) {
            versionMap = versionMap.insert(mutations[i].key, results[i].version);
        }
        return new MutationBatchResult(batch, commitVersion, results, streamToken, versionMap);
    };
    return MutationBatchResult;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * PersistencePromise<> is essentially a re-implementation of Promise<> except
 * it has a .next() method instead of .then() and .next() and .catch() callbacks
 * are executed synchronously when a PersistencePromise resolves rather than
 * asynchronously (Promise<> implementations use setImmediate() or similar).
 *
 * This is necessary to interoperate with IndexedDB which will automatically
 * commit transactions if control is returned to the event loop without
 * synchronously initiating another operation on the transaction.
 *
 * NOTE: .then() and .catch() only allow a single consumer, unlike normal
 * Promises.
 */
var PersistencePromise = /** @class */ (function () {
    function PersistencePromise(callback) {
        var _this = this;
        // NOTE: next/catchCallback will always point to our own wrapper functions,
        // not the user's raw next() or catch() callbacks.
        // tslint:disable-next-line:no-any Accept any result type for the next call in the Promise chain.
        this.nextCallback = null;
        // tslint:disable-next-line:no-any Accept any result type for the error handler.
        this.catchCallback = null;
        // When the operation resolves, we'll set result or error and mark isDone.
        this.result = undefined;
        this.error = undefined;
        this.isDone = false;
        // Set to true when .then() or .catch() are called and prevents additional
        // chaining.
        this.callbackAttached = false;
        callback(function (value) {
            _this.isDone = true;
            _this.result = value;
            if (_this.nextCallback) {
                // value should be defined unless T is Void, but we can't express
                // that in the type system.
                _this.nextCallback(value);
            }
        }, function (error) {
            _this.isDone = true;
            _this.error = error;
            if (_this.catchCallback) {
                _this.catchCallback(error);
            }
        });
    }
    PersistencePromise.prototype.catch = function (fn) {
        return this.next(undefined, fn);
    };
    PersistencePromise.prototype.next = function (nextFn, catchFn) {
        var _this = this;
        if (this.callbackAttached) {
            fail('Called next() or catch() twice for PersistencePromise');
        }
        this.callbackAttached = true;
        if (this.isDone) {
            if (!this.error) {
                return this.wrapSuccess(nextFn, this.result);
            }
            else {
                return this.wrapFailure(catchFn, this.error);
            }
        }
        else {
            return new PersistencePromise(function (resolve, reject) {
                _this.nextCallback = function (value) {
                    _this.wrapSuccess(nextFn, value).next(resolve, reject);
                };
                _this.catchCallback = function (error) {
                    _this.wrapFailure(catchFn, error).next(resolve, reject);
                };
            });
        }
    };
    PersistencePromise.prototype.toPromise = function () {
        var _this = this;
        return new Promise(function (resolve, reject) {
            _this.next(resolve, reject);
        });
    };
    PersistencePromise.prototype.wrapUserFunction = function (fn) {
        try {
            var result = fn();
            if (result instanceof PersistencePromise) {
                return result;
            }
            else {
                return PersistencePromise.resolve(result);
            }
        }
        catch (e) {
            return PersistencePromise.reject(e);
        }
    };
    PersistencePromise.prototype.wrapSuccess = function (nextFn, value) {
        if (nextFn) {
            return this.wrapUserFunction(function () { return nextFn(value); });
        }
        else {
            // If there's no nextFn, then R must be the same as T but we
            // can't express that in the type system.
            // tslint:disable-next-line:no-any
            return PersistencePromise.resolve(value);
        }
    };
    PersistencePromise.prototype.wrapFailure = function (catchFn, error) {
        if (catchFn) {
            return this.wrapUserFunction(function () { return catchFn(error); });
        }
        else {
            return PersistencePromise.reject(error);
        }
    };
    PersistencePromise.resolve = function (result) {
        return new PersistencePromise(function (resolve, reject) {
            resolve(result);
        });
    };
    PersistencePromise.reject = function (error) {
        return new PersistencePromise(function (resolve, reject) {
            reject(error);
        });
    };
    PersistencePromise.waitFor = function (
    // tslint:disable-next-line:no-any Accept all Promise types in waitFor().
    all) {
        return new PersistencePromise(function (resolve, reject) {
            var expectedCount = 0;
            var resolvedCount = 0;
            var done = false;
            all.forEach(function (element) {
                ++expectedCount;
                element.next(function () {
                    ++resolvedCount;
                    if (done && resolvedCount === expectedCount) {
                        resolve();
                    }
                }, function (err) { return reject(err); });
            });
            done = true;
            if (resolvedCount === expectedCount) {
                resolve();
            }
        });
    };
    /**
     * Given an array of predicate functions that asynchronously evaluate to a
     * boolean, implements a short-circuiting `or` between the results. Predicates
     * will be evaluated until one of them returns `true`, then stop. The final
     * result will be whether any of them returned `true`.
     */
    PersistencePromise.or = function (predicates) {
        var p = PersistencePromise.resolve(false);
        var _loop_1 = function (predicate) {
            p = p.next(function (isTrue) {
                if (isTrue) {
                    return PersistencePromise.resolve(isTrue);
                }
                else {
                    return predicate();
                }
            });
        };
        for (var _i = 0, predicates_1 = predicates; _i < predicates_1.length; _i++) {
            var predicate = predicates_1[_i];
            _loop_1(predicate);
        }
        return p;
    };
    /**
     * Given an iterable, call the given function on each element in the
     * collection and wait for all of the resulting concurrent PersistencePromises
     * to resolve.
     */
    PersistencePromise.forEach = function (collection, f) {
        var _this = this;
        var promises = [];
        collection.forEach(function (r, s) {
            promises.push(f.call(_this, r, s));
        });
        return this.waitFor(promises);
    };
    return PersistencePromise;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/** A mutation queue for a specific user, backed by IndexedDB. */
var IndexedDbMutationQueue = /** @class */ (function () {
    function IndexedDbMutationQueue(
    /**
     * The normalized userId (e.g. null UID => "" userId) used to store /
     * retrieve mutations.
     */
    userId, serializer, indexManager, referenceDelegate) {
        this.userId = userId;
        this.serializer = serializer;
        this.indexManager = indexManager;
        this.referenceDelegate = referenceDelegate;
        /**
         * Caches the document keys for pending mutation batches. If the mutation
         * has been removed from IndexedDb, the cached value may continue to
         * be used to retrieve the batch's document keys. To remove a cached value
         * locally, `removeCachedMutationKeys()` should be invoked either directly
         * or through `removeMutationBatches()`.
         *
         * With multi-tab, when the primary client acknowledges or rejects a mutation,
         * this cache is used by secondary clients to invalidate the local
         * view of the documents that were previously affected by the mutation.
         */
        // PORTING NOTE: Multi-tab only.
        this.documentKeysByBatchId = {};
    }
    /**
     * Creates a new mutation queue for the given user.
     * @param user The user for which to create a mutation queue.
     * @param serializer The serializer to use when persisting to IndexedDb.
     */
    IndexedDbMutationQueue.forUser = function (user, serializer, indexManager, referenceDelegate) {
        // TODO(mcg): Figure out what constraints there are on userIDs
        // In particular, are there any reserved characters? are empty ids allowed?
        // For the moment store these together in the same mutations table assuming
        // that empty userIDs aren't allowed.
        assert(user.uid !== '', 'UserID must not be an empty string.');
        var userId = user.isAuthenticated() ? user.uid : '';
        return new IndexedDbMutationQueue(userId, serializer, indexManager, referenceDelegate);
    };
    IndexedDbMutationQueue.prototype.checkEmpty = function (transaction) {
        var empty = true;
        var range = IDBKeyRange.bound([this.userId, Number.NEGATIVE_INFINITY], [this.userId, Number.POSITIVE_INFINITY]);
        return mutationsStore(transaction)
            .iterate({ index: DbMutationBatch.userMutationsIndex, range: range }, function (key, value, control) {
            empty = false;
            control.done();
        })
            .next(function () { return empty; });
    };
    IndexedDbMutationQueue.prototype.acknowledgeBatch = function (transaction, batch, streamToken) {
        return this.getMutationQueueMetadata(transaction).next(function (metadata) {
            metadata.lastStreamToken = convertStreamToken(streamToken);
            return mutationQueuesStore(transaction).put(metadata);
        });
    };
    IndexedDbMutationQueue.prototype.getLastStreamToken = function (transaction) {
        return this.getMutationQueueMetadata(transaction).next(function (metadata) { return metadata.lastStreamToken; });
    };
    IndexedDbMutationQueue.prototype.setLastStreamToken = function (transaction, streamToken) {
        return this.getMutationQueueMetadata(transaction).next(function (metadata) {
            metadata.lastStreamToken = convertStreamToken(streamToken);
            return mutationQueuesStore(transaction).put(metadata);
        });
    };
    IndexedDbMutationQueue.prototype.addMutationBatch = function (transaction, localWriteTime, baseMutations, mutations) {
        var _this = this;
        var documentStore = documentMutationsStore(transaction);
        var mutationStore = mutationsStore(transaction);
        // The IndexedDb implementation in Chrome (and Firefox) does not handle
        // compound indices that include auto-generated keys correctly. To ensure
        // that the index entry is added correctly in all browsers, we perform two
        // writes: The first write is used to retrieve the next auto-generated Batch
        // ID, and the second write populates the index and stores the actual
        // mutation batch.
        // See: https://bugs.chromium.org/p/chromium/issues/detail?id=701972
        // tslint:disable-next-line:no-any We write an empty object to obtain key
        return mutationStore.add({}).next(function (batchId) {
            assert(typeof batchId === 'number', 'Auto-generated key is not a number');
            var batch = new MutationBatch(batchId, localWriteTime, baseMutations, mutations);
            var dbBatch = _this.serializer.toDbMutationBatch(_this.userId, batch);
            _this.documentKeysByBatchId[batchId] = batch.keys();
            var promises = [];
            for (var _i = 0, mutations_1 = mutations; _i < mutations_1.length; _i++) {
                var mutation = mutations_1[_i];
                var indexKey = DbDocumentMutation.key(_this.userId, mutation.key.path, batchId);
                promises.push(mutationStore.put(dbBatch));
                promises.push(documentStore.put(indexKey, DbDocumentMutation.PLACEHOLDER));
                promises.push(_this.indexManager.addToCollectionParentIndex(transaction, mutation.key.path.popLast()));
            }
            return PersistencePromise.waitFor(promises).next(function () { return batch; });
        });
    };
    IndexedDbMutationQueue.prototype.lookupMutationBatch = function (transaction, batchId) {
        var _this = this;
        return mutationsStore(transaction)
            .get(batchId)
            .next(function (dbBatch) {
            if (dbBatch) {
                assert(dbBatch.userId === _this.userId, "Unexpected user '" + dbBatch.userId + "' for mutation batch " + batchId);
                return _this.serializer.fromDbMutationBatch(dbBatch);
            }
            return null;
        });
    };
    IndexedDbMutationQueue.prototype.lookupMutationKeys = function (transaction, batchId) {
        var _this = this;
        if (this.documentKeysByBatchId[batchId]) {
            return PersistencePromise.resolve(this.documentKeysByBatchId[batchId]);
        }
        else {
            return this.lookupMutationBatch(transaction, batchId).next(function (batch) {
                if (batch) {
                    var keys = batch.keys();
                    _this.documentKeysByBatchId[batchId] = keys;
                    return keys;
                }
                else {
                    return null;
                }
            });
        }
    };
    IndexedDbMutationQueue.prototype.getNextMutationBatchAfterBatchId = function (transaction, batchId) {
        var _this = this;
        return this.getMutationQueueMetadata(transaction).next(function (metadata) {
            var nextBatchId = batchId + 1;
            var range = IDBKeyRange.lowerBound([_this.userId, nextBatchId]);
            var foundBatch = null;
            return mutationsStore(transaction)
                .iterate({ index: DbMutationBatch.userMutationsIndex, range: range }, function (key, dbBatch, control) {
                if (dbBatch.userId === _this.userId) {
                    assert(dbBatch.batchId >= nextBatchId, 'Should have found mutation after ' + nextBatchId);
                    foundBatch = _this.serializer.fromDbMutationBatch(dbBatch);
                }
                control.done();
            })
                .next(function () { return foundBatch; });
        });
    };
    IndexedDbMutationQueue.prototype.getAllMutationBatches = function (transaction) {
        var _this = this;
        var range = IDBKeyRange.bound([this.userId, BATCHID_UNKNOWN], [this.userId, Number.POSITIVE_INFINITY]);
        return mutationsStore(transaction)
            .loadAll(DbMutationBatch.userMutationsIndex, range)
            .next(function (dbBatches) {
            return dbBatches.map(function (dbBatch) { return _this.serializer.fromDbMutationBatch(dbBatch); });
        });
    };
    IndexedDbMutationQueue.prototype.getAllMutationBatchesAffectingDocumentKey = function (transaction, documentKey) {
        var _this = this;
        // Scan the document-mutation index starting with a prefix starting with
        // the given documentKey.
        var indexPrefix = DbDocumentMutation.prefixForPath(this.userId, documentKey.path);
        var indexStart = IDBKeyRange.lowerBound(indexPrefix);
        var results = [];
        return documentMutationsStore(transaction)
            .iterate({ range: indexStart }, function (indexKey, _, control) {
            var userID = indexKey[0], encodedPath = indexKey[1], batchId = indexKey[2];
            // Only consider rows matching exactly the specific key of
            // interest. Note that because we order by path first, and we
            // order terminators before path separators, we'll encounter all
            // the index rows for documentKey contiguously. In particular, all
            // the rows for documentKey will occur before any rows for
            // documents nested in a subcollection beneath documentKey so we
            // can stop as soon as we hit any such row.
            var path = decode(encodedPath);
            if (userID !== _this.userId || !documentKey.path.isEqual(path)) {
                control.done();
                return;
            }
            // Look up the mutation batch in the store.
            return mutationsStore(transaction)
                .get(batchId)
                .next(function (mutation) {
                if (!mutation) {
                    throw fail('Dangling document-mutation reference found: ' +
                        indexKey +
                        ' which points to ' +
                        batchId);
                }
                assert(mutation.userId === _this.userId, "Unexpected user '" + mutation.userId + "' for mutation batch " + batchId);
                results.push(_this.serializer.fromDbMutationBatch(mutation));
            });
        })
            .next(function () { return results; });
    };
    IndexedDbMutationQueue.prototype.getAllMutationBatchesAffectingDocumentKeys = function (transaction, documentKeys) {
        var _this = this;
        var uniqueBatchIDs = new SortedSet(primitiveComparator);
        var promises = [];
        documentKeys.forEach(function (documentKey) {
            var indexStart = DbDocumentMutation.prefixForPath(_this.userId, documentKey.path);
            var range = IDBKeyRange.lowerBound(indexStart);
            var promise = documentMutationsStore(transaction).iterate({ range: range }, function (indexKey, _, control) {
                var userID = indexKey[0], encodedPath = indexKey[1], batchID = indexKey[2];
                // Only consider rows matching exactly the specific key of
                // interest. Note that because we order by path first, and we
                // order terminators before path separators, we'll encounter all
                // the index rows for documentKey contiguously. In particular, all
                // the rows for documentKey will occur before any rows for
                // documents nested in a subcollection beneath documentKey so we
                // can stop as soon as we hit any such row.
                var path = decode(encodedPath);
                if (userID !== _this.userId || !documentKey.path.isEqual(path)) {
                    control.done();
                    return;
                }
                uniqueBatchIDs = uniqueBatchIDs.add(batchID);
            });
            promises.push(promise);
        });
        return PersistencePromise.waitFor(promises).next(function () {
            return _this.lookupMutationBatches(transaction, uniqueBatchIDs);
        });
    };
    IndexedDbMutationQueue.prototype.getAllMutationBatchesAffectingQuery = function (transaction, query) {
        var _this = this;
        assert(!query.isDocumentQuery(), "Document queries shouldn't go down this path");
        assert(!query.isCollectionGroupQuery(), 'CollectionGroup queries should be handled in LocalDocumentsView');
        var queryPath = query.path;
        var immediateChildrenLength = queryPath.length + 1;
        // TODO(mcg): Actually implement a single-collection query
        //
        // This is actually executing an ancestor query, traversing the whole
        // subtree below the collection which can be horrifically inefficient for
        // some structures. The right way to solve this is to implement the full
        // value index, but that's not in the cards in the near future so this is
        // the best we can do for the moment.
        //
        // Since we don't yet index the actual properties in the mutations, our
        // current approach is to just return all mutation batches that affect
        // documents in the collection being queried.
        var indexPrefix = DbDocumentMutation.prefixForPath(this.userId, queryPath);
        var indexStart = IDBKeyRange.lowerBound(indexPrefix);
        // Collect up unique batchIDs encountered during a scan of the index. Use a
        // SortedSet to accumulate batch IDs so they can be traversed in order in a
        // scan of the main table.
        var uniqueBatchIDs = new SortedSet(primitiveComparator);
        return documentMutationsStore(transaction)
            .iterate({ range: indexStart }, function (indexKey, _, control) {
            var userID = indexKey[0], encodedPath = indexKey[1], batchID = indexKey[2];
            var path = decode(encodedPath);
            if (userID !== _this.userId || !queryPath.isPrefixOf(path)) {
                control.done();
                return;
            }
            // Rows with document keys more than one segment longer than the
            // query path can't be matches. For example, a query on 'rooms'
            // can't match the document /rooms/abc/messages/xyx.
            // TODO(mcg): we'll need a different scanner when we implement
            // ancestor queries.
            if (path.length !== immediateChildrenLength) {
                return;
            }
            uniqueBatchIDs = uniqueBatchIDs.add(batchID);
        })
            .next(function () { return _this.lookupMutationBatches(transaction, uniqueBatchIDs); });
    };
    IndexedDbMutationQueue.prototype.lookupMutationBatches = function (transaction, batchIDs) {
        var _this = this;
        var results = [];
        var promises = [];
        // TODO(rockwood): Implement this using iterate.
        batchIDs.forEach(function (batchId) {
            promises.push(mutationsStore(transaction)
                .get(batchId)
                .next(function (mutation) {
                if (mutation === null) {
                    throw fail('Dangling document-mutation reference found, ' +
                        'which points to ' +
                        batchId);
                }
                assert(mutation.userId === _this.userId, "Unexpected user '" + mutation.userId + "' for mutation batch " + batchId);
                results.push(_this.serializer.fromDbMutationBatch(mutation));
            }));
        });
        return PersistencePromise.waitFor(promises).next(function () { return results; });
    };
    IndexedDbMutationQueue.prototype.removeMutationBatch = function (transaction, batch) {
        var _this = this;
        return removeMutationBatch(transaction.simpleDbTransaction, this.userId, batch).next(function (removedDocuments) {
            _this.removeCachedMutationKeys(batch.batchId);
            return PersistencePromise.forEach(removedDocuments, function (key) {
                return _this.referenceDelegate.removeMutationReference(transaction, key);
            });
        });
    };
    IndexedDbMutationQueue.prototype.removeCachedMutationKeys = function (batchId) {
        delete this.documentKeysByBatchId[batchId];
    };
    IndexedDbMutationQueue.prototype.performConsistencyCheck = function (txn) {
        var _this = this;
        return this.checkEmpty(txn).next(function (empty) {
            if (!empty) {
                return PersistencePromise.resolve();
            }
            // Verify that there are no entries in the documentMutations index if
            // the queue is empty.
            var startRange = IDBKeyRange.lowerBound(DbDocumentMutation.prefixForUser(_this.userId));
            var danglingMutationReferences = [];
            return documentMutationsStore(txn)
                .iterate({ range: startRange }, function (key, _, control) {
                var userID = key[0];
                if (userID !== _this.userId) {
                    control.done();
                    return;
                }
                else {
                    var path = decode(key[1]);
                    danglingMutationReferences.push(path);
                }
            })
                .next(function () {
                assert(danglingMutationReferences.length === 0, 'Document leak -- detected dangling mutation references when queue is empty. ' +
                    'Dangling keys: ' +
                    danglingMutationReferences.map(function (p) { return p.canonicalString(); }));
            });
        });
    };
    IndexedDbMutationQueue.prototype.containsKey = function (txn, key) {
        return mutationQueueContainsKey(txn, this.userId, key);
    };
    // PORTING NOTE: Multi-tab only (state is held in memory in other clients).
    /** Returns the mutation queue's metadata from IndexedDb. */
    IndexedDbMutationQueue.prototype.getMutationQueueMetadata = function (transaction) {
        var _this = this;
        return mutationQueuesStore(transaction)
            .get(this.userId)
            .next(function (metadata) {
            return (metadata ||
                new DbMutationQueue(_this.userId, BATCHID_UNKNOWN, 
                /*lastStreamToken=*/ ''));
        });
    };
    return IndexedDbMutationQueue;
}());
/**
 * @return true if the mutation queue for the given user contains a pending
 *         mutation for the given key.
 */
function mutationQueueContainsKey(txn, userId, key) {
    var indexKey = DbDocumentMutation.prefixForPath(userId, key.path);
    var encodedPath = indexKey[1];
    var startRange = IDBKeyRange.lowerBound(indexKey);
    var containsKey = false;
    return documentMutationsStore(txn)
        .iterate({ range: startRange, keysOnly: true }, function (key, value, control) {
        var userID = key[0], keyPath = key[1], /*batchID*/ _ = key[2];
        if (userID === userId && keyPath === encodedPath) {
            containsKey = true;
        }
        control.done();
    })
        .next(function () { return containsKey; });
}
/** Returns true if any mutation queue contains the given document. */
function mutationQueuesContainKey(txn, docKey) {
    var found = false;
    return mutationQueuesStore(txn)
        .iterateSerial(function (userId) {
        return mutationQueueContainsKey(txn, userId, docKey).next(function (containsKey) {
            if (containsKey) {
                found = true;
            }
            return PersistencePromise.resolve(!containsKey);
        });
    })
        .next(function () { return found; });
}
/**
 * Delete a mutation batch and the associated document mutations.
 * @return A PersistencePromise of the document mutations that were removed.
 */
function removeMutationBatch(txn, userId, batch) {
    var mutationStore = txn.store(DbMutationBatch.store);
    var indexTxn = txn.store(DbDocumentMutation.store);
    var promises = [];
    var range = IDBKeyRange.only(batch.batchId);
    var numDeleted = 0;
    var removePromise = mutationStore.iterate({ range: range }, function (key, value, control) {
        numDeleted++;
        return control.delete();
    });
    promises.push(removePromise.next(function () {
        assert(numDeleted === 1, 'Dangling document-mutation reference found: Missing batch ' +
            batch.batchId);
    }));
    var removedDocuments = [];
    for (var _i = 0, _a = batch.mutations; _i < _a.length; _i++) {
        var mutation = _a[_i];
        var indexKey = DbDocumentMutation.key(userId, mutation.key.path, batch.batchId);
        promises.push(indexTxn.delete(indexKey));
        removedDocuments.push(mutation.key);
    }
    return PersistencePromise.waitFor(promises).next(function () { return removedDocuments; });
}
function convertStreamToken(token) {
    if (token instanceof Uint8Array) {
        // TODO(b/78771403): Convert tokens to strings during deserialization
        assert(process.env.USE_MOCK_PERSISTENCE === 'YES', 'Persisting non-string stream tokens is only supported with mock persistence.');
        return token.toString();
    }
    else {
        return token;
    }
}
/**
 * Helper to get a typed SimpleDbStore for the mutations object store.
 */
function mutationsStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbMutationBatch.store);
}
/**
 * Helper to get a typed SimpleDbStore for the mutationQueues object store.
 */
function documentMutationsStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbDocumentMutation.store);
}
/**
 * Helper to get a typed SimpleDbStore for the mutationQueues object store.
 */
function mutationQueuesStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbMutationQueue.store);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var RESERVED_BITS = 1;
var GeneratorIds;
(function (GeneratorIds) {
    GeneratorIds[GeneratorIds["QueryCache"] = 0] = "QueryCache";
    GeneratorIds[GeneratorIds["SyncEngine"] = 1] = "SyncEngine"; // The target IDs for limbo detection are odd (end in 1).
})(GeneratorIds || (GeneratorIds = {}));
/**
 * Generates monotonically increasing target IDs for sending targets to the
 * watch stream.
 *
 * The client constructs two generators, one for the query cache (via
 * forQueryCache()), and one for limbo documents (via forSyncEngine()). These
 * two generators produce non-overlapping IDs (by using even and odd IDs
 * respectively).
 *
 * By separating the target ID space, the query cache can generate target IDs
 * that persist across client restarts, while sync engine can independently
 * generate in-memory target IDs that are transient and can be reused after a
 * restart.
 */
// TODO(mrschmidt): Explore removing this class in favor of generating these IDs
// directly in SyncEngine and LocalStore.
var TargetIdGenerator = /** @class */ (function () {
    /**
     * Instantiates a new TargetIdGenerator. If a seed is provided, the generator
     * will use the seed value as the next target ID.
     */
    function TargetIdGenerator(generatorId, seed) {
        this.generatorId = generatorId;
        assert((generatorId & RESERVED_BITS) === generatorId, "Generator ID " + generatorId + " contains more than " + RESERVED_BITS + " reserved bits");
        this.seek(seed !== undefined ? seed : this.generatorId);
    }
    TargetIdGenerator.prototype.next = function () {
        var nextId = this.nextId;
        this.nextId += 1 << RESERVED_BITS;
        return nextId;
    };
    /**
     * Returns the ID that follows the given ID. Subsequent calls to `next()`
     * use the newly returned target ID as their base.
     */
    // PORTING NOTE: Multi-tab only.
    TargetIdGenerator.prototype.after = function (targetId) {
        this.seek(targetId + (1 << RESERVED_BITS));
        return this.next();
    };
    TargetIdGenerator.prototype.seek = function (targetId) {
        assert((targetId & RESERVED_BITS) === this.generatorId, 'Cannot supply target ID from different generator ID');
        this.nextId = targetId;
    };
    TargetIdGenerator.forQueryCache = function () {
        // We seed the query cache generator to return '2' as its first ID, as there
        // is no differentiation in the protocol layer between an unset number and
        // the number '0'. If we were to sent a target with target ID '0', the
        // backend would consider it unset and replace it with its own ID.
        var targetIdGenerator = new TargetIdGenerator(GeneratorIds.QueryCache, 2);
        return targetIdGenerator;
    };
    TargetIdGenerator.forSyncEngine = function () {
        // Sync engine assigns target IDs for limbo document detection.
        return new TargetIdGenerator(GeneratorIds.SyncEngine);
    };
    return TargetIdGenerator;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$1 = 'SimpleDb';
/**
 * Provides a wrapper around IndexedDb with a simplified interface that uses
 * Promise-like return values to chain operations. Real promises cannot be used
 * since .then() continuations are executed asynchronously (e.g. via
 * .setImmediate), which would cause IndexedDB to end the transaction.
 * See PersistencePromise for more details.
 */
var SimpleDb = /** @class */ (function () {
    function SimpleDb(db) {
        this.db = db;
    }
    /**
     * Opens the specified database, creating or upgrading it if necessary.
     *
     * Note that `version` must not be a downgrade. IndexedDB does not support downgrading the schema
     * version. We currently do not support any way to do versioning outside of IndexedDB's versioning
     * mechanism, as only version-upgrade transactions are allowed to do things like create
     * objectstores.
     */
    SimpleDb.openOrCreate = function (name, version, schemaConverter) {
        assert(SimpleDb.isAvailable(), 'IndexedDB not supported in current environment.');
        debug(LOG_TAG$1, 'Opening database:', name);
        return new PersistencePromise(function (resolve, reject) {
            // TODO(mikelehen): Investigate browser compatibility.
            // https://developer.mozilla.org/en-US/docs/Web/API/IndexedDB_API/Using_IndexedDB
            // suggests IE9 and older WebKit browsers handle upgrade
            // differently. They expect setVersion, as described here:
            // https://developer.mozilla.org/en-US/docs/Web/API/IDBVersionChangeRequest/setVersion
            var request = window.indexedDB.open(name, version);
            request.onsuccess = function (event) {
                var db = event.target.result;
                resolve(new SimpleDb(db));
            };
            request.onblocked = function () {
                reject(new FirestoreError(Code.FAILED_PRECONDITION, 'Cannot upgrade IndexedDB schema while another tab is open. ' +
                    'Close all tabs that access Firestore and reload this page to proceed.'));
            };
            request.onerror = function (event) {
                var error = event.target.error;
                if (error.name === 'VersionError') {
                    reject(new FirestoreError(Code.FAILED_PRECONDITION, 'A newer version of the Firestore SDK was previously used and so the persisted ' +
                        'data is not compatible with the version of the SDK you are now using. The SDK ' +
                        'will operate with persistence disabled. If you need persistence, please ' +
                        're-upgrade to a newer version of the SDK or else clear the persisted IndexedDB ' +
                        'data for your app to start fresh.'));
                }
                else {
                    reject(error);
                }
            };
            request.onupgradeneeded = function (event) {
                debug(LOG_TAG$1, 'Database "' + name + '" requires upgrade from version:', event.oldVersion);
                var db = event.target.result;
                // We are provided a version upgrade transaction from the request, so
                // we wrap that in a SimpleDbTransaction to allow use of our friendlier
                // API for schema migration operations.
                var txn = new SimpleDbTransaction(request.transaction);
                schemaConverter
                    .createOrUpgrade(db, txn, event.oldVersion, SCHEMA_VERSION)
                    .next(function () {
                    debug(LOG_TAG$1, 'Database upgrade to version ' + SCHEMA_VERSION + ' complete');
                });
            };
        }).toPromise();
    };
    /** Deletes the specified database. */
    SimpleDb.delete = function (name) {
        debug(LOG_TAG$1, 'Removing database:', name);
        return wrapRequest(window.indexedDB.deleteDatabase(name)).toPromise();
    };
    /** Returns true if IndexedDB is available in the current environment. */
    SimpleDb.isAvailable = function () {
        if (typeof window === 'undefined' || window.indexedDB == null) {
            return false;
        }
        // We extensively use indexed array values and compound keys,
        // which IE and Edge do not support. However, they still have indexedDB
        // defined on the window, so we need to check for them here and make sure
        // to return that persistence is not enabled for those browsers.
        // For tracking support of this feature, see here:
        // https://developer.microsoft.com/en-us/microsoft-edge/platform/status/indexeddbarraysandmultientrysupport/
        // If we are running in Node using the IndexedDBShim, `window` is defined,
        // but `window.navigator` is not. In this case, we support IndexedDB and
        // return `true`.
        if (window.navigator === undefined) {
            return process.env.USE_MOCK_PERSISTENCE === 'YES';
        }
        // Check the UA string to find out the browser.
        // TODO(mikelehen): Move this logic into packages/util/environment.ts
        var ua = window.navigator.userAgent;
        // IE 10
        // ua = 'Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.2; Trident/6.0)';
        // IE 11
        // ua = 'Mozilla/5.0 (Windows NT 6.3; Trident/7.0; rv:11.0) like Gecko';
        // Edge
        // ua = 'Mozilla/5.0 (Windows NT 10.0; WOW64) AppleWebKit/537.36 (KHTML,
        // like Gecko) Chrome/39.0.2171.71 Safari/537.36 Edge/12.0';
        if (ua.indexOf('MSIE ') > 0 ||
            ua.indexOf('Trident/') > 0 ||
            ua.indexOf('Edge/') > 0) {
            return false;
        }
        else {
            return true;
        }
    };
    /** Helper to get a typed SimpleDbStore from a transaction. */
    SimpleDb.getStore = function (txn, store) {
        return txn.store(store);
    };
    SimpleDb.prototype.runTransaction = function (mode, objectStores, transactionFn) {
        var transaction = SimpleDbTransaction.open(this.db, mode, objectStores);
        var transactionFnResult = transactionFn(transaction)
            .catch(function (error) {
            // Abort the transaction if there was an error.
            transaction.abort(error);
            // We cannot actually recover, and calling `abort()` will cause the transaction's
            // completion promise to be rejected. This in turn means that we won't use
            // `transactionFnResult` below. We return a rejection here so that we don't add the
            // possibility of returning `void` to the type of `transactionFnResult`.
            return PersistencePromise.reject(error);
        })
            .toPromise();
        // As noted above, errors are propagated by aborting the transaction. So
        // we swallow any error here to avoid the browser logging it as unhandled.
        transactionFnResult.catch(function () { });
        // Wait for the transaction to complete (i.e. IndexedDb's onsuccess event to
        // fire), but still return the original transactionFnResult back to the
        // caller.
        return transaction.completionPromise.then(function () { return transactionFnResult; });
    };
    SimpleDb.prototype.close = function () {
        this.db.close();
    };
    return SimpleDb;
}());
/**
 * A controller for iterating over a key range or index. It allows an iterate
 * callback to delete the currently-referenced object, or jump to a new key
 * within the key range or index.
 */
var IterationController = /** @class */ (function () {
    function IterationController(dbCursor) {
        this.dbCursor = dbCursor;
        this.shouldStop = false;
        this.nextKey = null;
    }
    Object.defineProperty(IterationController.prototype, "isDone", {
        get: function () {
            return this.shouldStop;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(IterationController.prototype, "skipToKey", {
        get: function () {
            return this.nextKey;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(IterationController.prototype, "cursor", {
        set: function (value) {
            this.dbCursor = value;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * This function can be called to stop iteration at any point.
     */
    IterationController.prototype.done = function () {
        this.shouldStop = true;
    };
    /**
     * This function can be called to skip to that next key, which could be
     * an index or a primary key.
     */
    IterationController.prototype.skip = function (key) {
        this.nextKey = key;
    };
    /**
     * Delete the current cursor value from the object store.
     *
     * NOTE: You CANNOT do this with a keysOnly query.
     */
    IterationController.prototype.delete = function () {
        return wrapRequest(this.dbCursor.delete());
    };
    return IterationController;
}());
/**
 * Wraps an IDBTransaction and exposes a store() method to get a handle to a
 * specific object store.
 */
var SimpleDbTransaction = /** @class */ (function () {
    function SimpleDbTransaction(transaction) {
        var _this = this;
        this.transaction = transaction;
        this.aborted = false;
        /**
         * A promise that resolves with the result of the IndexedDb transaction.
         */
        this.completionDeferred = new Deferred();
        this.transaction.oncomplete = function () {
            _this.completionDeferred.resolve();
        };
        this.transaction.onabort = function () {
            if (transaction.error) {
                _this.completionDeferred.reject(transaction.error);
            }
            else {
                _this.completionDeferred.resolve();
            }
        };
        this.transaction.onerror = function (event) {
            _this.completionDeferred.reject(event.target.error);
        };
    }
    SimpleDbTransaction.open = function (db, mode, objectStoreNames) {
        return new SimpleDbTransaction(db.transaction(objectStoreNames, mode));
    };
    Object.defineProperty(SimpleDbTransaction.prototype, "completionPromise", {
        get: function () {
            return this.completionDeferred.promise;
        },
        enumerable: true,
        configurable: true
    });
    SimpleDbTransaction.prototype.abort = function (error) {
        if (error) {
            this.completionDeferred.reject(error);
        }
        if (!this.aborted) {
            debug(LOG_TAG$1, 'Aborting transaction:', error ? error.message : 'Client-initiated abort');
            this.aborted = true;
            this.transaction.abort();
        }
    };
    /**
     * Returns a SimpleDbStore<KeyType, ValueType> for the specified store. All
     * operations performed on the SimpleDbStore happen within the context of this
     * transaction and it cannot be used anymore once the transaction is
     * completed.
     *
     * Note that we can't actually enforce that the KeyType and ValueType are
     * correct, but they allow type safety through the rest of the consuming code.
     */
    SimpleDbTransaction.prototype.store = function (storeName) {
        var store = this.transaction.objectStore(storeName);
        assert(!!store, 'Object store not part of transaction: ' + storeName);
        return new SimpleDbStore(store);
    };
    return SimpleDbTransaction;
}());
/**
 * A wrapper around an IDBObjectStore providing an API that:
 *
 * 1) Has generic KeyType / ValueType parameters to provide strongly-typed
 * methods for acting against the object store.
 * 2) Deals with IndexedDB's onsuccess / onerror event callbacks, making every
 * method return a PersistencePromise instead.
 * 3) Provides a higher-level API to avoid needing to do excessive wrapping of
 * intermediate IndexedDB types (IDBCursorWithValue, etc.)
 */
var SimpleDbStore = /** @class */ (function () {
    function SimpleDbStore(store) {
        this.store = store;
    }
    SimpleDbStore.prototype.put = function (keyOrValue, value) {
        var request;
        if (value !== undefined) {
            debug(LOG_TAG$1, 'PUT', this.store.name, keyOrValue, value);
            request = this.store.put(value, keyOrValue);
        }
        else {
            debug(LOG_TAG$1, 'PUT', this.store.name, '<auto-key>', keyOrValue);
            request = this.store.put(keyOrValue);
        }
        return wrapRequest(request);
    };
    /**
     * Adds a new value into an Object Store and returns the new key. Similar to
     * IndexedDb's `add()`, this method will fail on primary key collisions.
     *
     * @param value The object to write.
     * @return The key of the value to add.
     */
    SimpleDbStore.prototype.add = function (value) {
        debug(LOG_TAG$1, 'ADD', this.store.name, value, value);
        var request = this.store.add(value);
        return wrapRequest(request);
    };
    /**
     * Gets the object with the specified key from the specified store, or null
     * if no object exists with the specified key.
     *
     * @key The key of the object to get.
     * @return The object with the specified key or null if no object exists.
     */
    SimpleDbStore.prototype.get = function (key) {
        var _this = this;
        var request = this.store.get(key);
        // tslint:disable-next-line:no-any We're doing an unsafe cast to ValueType.
        return wrapRequest(request).next(function (result) {
            // Normalize nonexistence to null.
            if (result === undefined) {
                result = null;
            }
            debug(LOG_TAG$1, 'GET', _this.store.name, key, result);
            return result;
        });
    };
    SimpleDbStore.prototype.delete = function (key) {
        debug(LOG_TAG$1, 'DELETE', this.store.name, key);
        var request = this.store.delete(key);
        return wrapRequest(request);
    };
    /**
     * If we ever need more of the count variants, we can add overloads. For now,
     * all we need is to count everything in a store.
     *
     * Returns the number of rows in the store.
     */
    SimpleDbStore.prototype.count = function () {
        debug(LOG_TAG$1, 'COUNT', this.store.name);
        var request = this.store.count();
        return wrapRequest(request);
    };
    SimpleDbStore.prototype.loadAll = function (indexOrRange, range) {
        var cursor = this.cursor(this.options(indexOrRange, range));
        var results = [];
        return this.iterateCursor(cursor, function (key, value) {
            results.push(value);
        }).next(function () {
            return results;
        });
    };
    SimpleDbStore.prototype.deleteAll = function (indexOrRange, range) {
        debug(LOG_TAG$1, 'DELETE ALL', this.store.name);
        var options = this.options(indexOrRange, range);
        options.keysOnly = false;
        var cursor = this.cursor(options);
        return this.iterateCursor(cursor, function (key, value, control) {
            // NOTE: Calling delete() on a cursor is documented as more efficient than
            // calling delete() on an object store with a single key
            // (https://developer.mozilla.org/en-US/docs/Web/API/IDBObjectStore/delete),
            // however, this requires us *not* to use a keysOnly cursor
            // (https://developer.mozilla.org/en-US/docs/Web/API/IDBCursor/delete). We
            // may want to compare the performance of each method.
            return control.delete();
        });
    };
    SimpleDbStore.prototype.iterate = function (optionsOrCallback, callback) {
        var options;
        if (!callback) {
            options = {};
            callback = optionsOrCallback;
        }
        else {
            options = optionsOrCallback;
        }
        var cursor = this.cursor(options);
        return this.iterateCursor(cursor, callback);
    };
    /**
     * Iterates over a store, but waits for the given callback to complete for
     * each entry before iterating the next entry. This allows the callback to do
     * asynchronous work to determine if this iteration should continue.
     *
     * The provided callback should return `true` to continue iteration, and
     * `false` otherwise.
     */
    SimpleDbStore.prototype.iterateSerial = function (callback) {
        var cursorRequest = this.cursor({});
        return new PersistencePromise(function (resolve, reject) {
            cursorRequest.onerror = function (event) {
                reject(event.target.error);
            };
            cursorRequest.onsuccess = function (event) {
                var cursor = event.target.result;
                if (!cursor) {
                    resolve();
                    return;
                }
                callback(cursor.primaryKey, cursor.value).next(function (shouldContinue) {
                    if (shouldContinue) {
                        cursor.continue();
                    }
                    else {
                        resolve();
                    }
                });
            };
        });
    };
    SimpleDbStore.prototype.iterateCursor = function (cursorRequest, fn) {
        var results = [];
        return new PersistencePromise(function (resolve, reject) {
            cursorRequest.onerror = function (event) {
                reject(event.target.error);
            };
            cursorRequest.onsuccess = function (event) {
                var cursor = event.target.result;
                if (!cursor) {
                    resolve();
                    return;
                }
                var controller = new IterationController(cursor);
                var userResult = fn(cursor.primaryKey, cursor.value, controller);
                if (userResult instanceof PersistencePromise) {
                    var userPromise = userResult.catch(function (err) {
                        controller.done();
                        return PersistencePromise.reject(err);
                    });
                    results.push(userPromise);
                }
                if (controller.isDone) {
                    resolve();
                }
                else if (controller.skipToKey === null) {
                    cursor.continue();
                }
                else {
                    cursor.continue(controller.skipToKey);
                }
            };
        }).next(function () {
            return PersistencePromise.waitFor(results);
        });
    };
    SimpleDbStore.prototype.options = function (indexOrRange, range) {
        var indexName = undefined;
        if (indexOrRange !== undefined) {
            if (typeof indexOrRange === 'string') {
                indexName = indexOrRange;
            }
            else {
                assert(range === undefined, '3rd argument must not be defined if 2nd is a range.');
                range = indexOrRange;
            }
        }
        return { index: indexName, range: range };
    };
    SimpleDbStore.prototype.cursor = function (options) {
        var direction = 'next';
        if (options.reverse) {
            direction = 'prev';
        }
        if (options.index) {
            var index = this.store.index(options.index);
            if (options.keysOnly) {
                return index.openKeyCursor(options.range, direction);
            }
            else {
                return index.openCursor(options.range, direction);
            }
        }
        else {
            return this.store.openCursor(options.range, direction);
        }
    };
    return SimpleDbStore;
}());
/**
 * Wraps an IDBRequest in a PersistencePromise, using the onsuccess / onerror
 * handlers to resolve / reject the PersistencePromise as appropriate.
 */
function wrapRequest(request) {
    return new PersistencePromise(function (resolve, reject) {
        request.onsuccess = function (event) {
            var result = event.target.result;
            resolve(result);
        };
        request.onerror = function (event) {
            reject(event.target.error);
        };
    });
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var IndexedDbQueryCache = /** @class */ (function () {
    function IndexedDbQueryCache(referenceDelegate, serializer) {
        this.referenceDelegate = referenceDelegate;
        this.serializer = serializer;
        // PORTING NOTE: We don't cache global metadata for the query cache, since
        // some of it (in particular `highestTargetId`) can be modified by secondary
        // tabs. We could perhaps be more granular (and e.g. still cache
        // `lastRemoteSnapshotVersion` in memory) but for simplicity we currently go
        // to IndexedDb whenever we need to read metadata. We can revisit if it turns
        // out to have a meaningful performance impact.
        this.targetIdGenerator = TargetIdGenerator.forQueryCache();
    }
    IndexedDbQueryCache.prototype.allocateTargetId = function (transaction) {
        var _this = this;
        return this.retrieveMetadata(transaction).next(function (metadata) {
            metadata.highestTargetId = _this.targetIdGenerator.after(metadata.highestTargetId);
            return _this.saveMetadata(transaction, metadata).next(function () { return metadata.highestTargetId; });
        });
    };
    IndexedDbQueryCache.prototype.getLastRemoteSnapshotVersion = function (transaction) {
        return this.retrieveMetadata(transaction).next(function (metadata) {
            return SnapshotVersion.fromTimestamp(new Timestamp(metadata.lastRemoteSnapshotVersion.seconds, metadata.lastRemoteSnapshotVersion.nanoseconds));
        });
    };
    IndexedDbQueryCache.prototype.getHighestSequenceNumber = function (transaction) {
        return getHighestListenSequenceNumber(transaction.simpleDbTransaction);
    };
    IndexedDbQueryCache.prototype.setTargetsMetadata = function (transaction, highestListenSequenceNumber, lastRemoteSnapshotVersion) {
        var _this = this;
        return this.retrieveMetadata(transaction).next(function (metadata) {
            metadata.highestListenSequenceNumber = highestListenSequenceNumber;
            if (lastRemoteSnapshotVersion) {
                metadata.lastRemoteSnapshotVersion = lastRemoteSnapshotVersion.toTimestamp();
            }
            if (highestListenSequenceNumber > metadata.highestListenSequenceNumber) {
                metadata.highestListenSequenceNumber = highestListenSequenceNumber;
            }
            return _this.saveMetadata(transaction, metadata);
        });
    };
    IndexedDbQueryCache.prototype.addQueryData = function (transaction, queryData) {
        var _this = this;
        return this.saveQueryData(transaction, queryData).next(function () {
            return _this.retrieveMetadata(transaction).next(function (metadata) {
                metadata.targetCount += 1;
                _this.updateMetadataFromQueryData(queryData, metadata);
                return _this.saveMetadata(transaction, metadata);
            });
        });
    };
    IndexedDbQueryCache.prototype.updateQueryData = function (transaction, queryData) {
        return this.saveQueryData(transaction, queryData);
    };
    IndexedDbQueryCache.prototype.removeQueryData = function (transaction, queryData) {
        var _this = this;
        return this.removeMatchingKeysForTargetId(transaction, queryData.targetId)
            .next(function () { return targetsStore(transaction).delete(queryData.targetId); })
            .next(function () { return _this.retrieveMetadata(transaction); })
            .next(function (metadata) {
            assert(metadata.targetCount > 0, 'Removing from an empty query cache');
            metadata.targetCount -= 1;
            return _this.saveMetadata(transaction, metadata);
        });
    };
    /**
     * Drops any targets with sequence number less than or equal to the upper bound, excepting those
     * present in `activeTargetIds`. Document associations for the removed targets are also removed.
     * Returns the number of targets removed.
     */
    IndexedDbQueryCache.prototype.removeTargets = function (txn, upperBound, activeTargetIds) {
        var _this = this;
        var count = 0;
        var promises = [];
        return targetsStore(txn)
            .iterate(function (key, value) {
            var queryData = _this.serializer.fromDbTarget(value);
            if (queryData.sequenceNumber <= upperBound &&
                activeTargetIds[queryData.targetId] === undefined) {
                count++;
                promises.push(_this.removeQueryData(txn, queryData));
            }
        })
            .next(function () { return PersistencePromise.waitFor(promises); })
            .next(function () { return count; });
    };
    /**
     * Call provided function with each `QueryData` that we have cached.
     */
    IndexedDbQueryCache.prototype.forEachTarget = function (txn, f) {
        var _this = this;
        return targetsStore(txn).iterate(function (key, value) {
            var queryData = _this.serializer.fromDbTarget(value);
            f(queryData);
        });
    };
    IndexedDbQueryCache.prototype.retrieveMetadata = function (transaction) {
        return retrieveMetadata(transaction.simpleDbTransaction);
    };
    IndexedDbQueryCache.prototype.saveMetadata = function (transaction, metadata) {
        return globalTargetStore(transaction).put(DbTargetGlobal.key, metadata);
    };
    IndexedDbQueryCache.prototype.saveQueryData = function (transaction, queryData) {
        return targetsStore(transaction).put(this.serializer.toDbTarget(queryData));
    };
    /**
     * In-place updates the provided metadata to account for values in the given
     * QueryData. Saving is done separately. Returns true if there were any
     * changes to the metadata.
     */
    IndexedDbQueryCache.prototype.updateMetadataFromQueryData = function (queryData, metadata) {
        var updated = false;
        if (queryData.targetId > metadata.highestTargetId) {
            metadata.highestTargetId = queryData.targetId;
            updated = true;
        }
        if (queryData.sequenceNumber > metadata.highestListenSequenceNumber) {
            metadata.highestListenSequenceNumber = queryData.sequenceNumber;
            updated = true;
        }
        return updated;
    };
    IndexedDbQueryCache.prototype.getQueryCount = function (transaction) {
        return this.retrieveMetadata(transaction).next(function (metadata) { return metadata.targetCount; });
    };
    IndexedDbQueryCache.prototype.getQueryData = function (transaction, query) {
        var _this = this;
        // Iterating by the canonicalId may yield more than one result because
        // canonicalId values are not required to be unique per target. This query
        // depends on the queryTargets index to be efficient.
        var canonicalId = query.canonicalId();
        var range = IDBKeyRange.bound([canonicalId, Number.NEGATIVE_INFINITY], [canonicalId, Number.POSITIVE_INFINITY]);
        var result = null;
        return targetsStore(transaction)
            .iterate({ range: range, index: DbTarget.queryTargetsIndexName }, function (key, value, control) {
            var found = _this.serializer.fromDbTarget(value);
            // After finding a potential match, check that the query is
            // actually equal to the requested query.
            if (query.isEqual(found.query)) {
                result = found;
                control.done();
            }
        })
            .next(function () { return result; });
    };
    IndexedDbQueryCache.prototype.addMatchingKeys = function (txn, keys, targetId) {
        var _this = this;
        // PORTING NOTE: The reverse index (documentsTargets) is maintained by
        // IndexedDb.
        var promises = [];
        var store = documentTargetStore(txn);
        keys.forEach(function (key) {
            var path = encode(key.path);
            promises.push(store.put(new DbTargetDocument(targetId, path)));
            promises.push(_this.referenceDelegate.addReference(txn, key));
        });
        return PersistencePromise.waitFor(promises);
    };
    IndexedDbQueryCache.prototype.removeMatchingKeys = function (txn, keys, targetId) {
        var _this = this;
        // PORTING NOTE: The reverse index (documentsTargets) is maintained by
        // IndexedDb.
        var store = documentTargetStore(txn);
        return PersistencePromise.forEach(keys, function (key) {
            var path = encode(key.path);
            return PersistencePromise.waitFor([
                store.delete([targetId, path]),
                _this.referenceDelegate.removeReference(txn, key)
            ]);
        });
    };
    IndexedDbQueryCache.prototype.removeMatchingKeysForTargetId = function (txn, targetId) {
        var store = documentTargetStore(txn);
        var range = IDBKeyRange.bound([targetId], [targetId + 1], 
        /*lowerOpen=*/ false, 
        /*upperOpen=*/ true);
        return store.delete(range);
    };
    IndexedDbQueryCache.prototype.getMatchingKeysForTargetId = function (txn, targetId) {
        var range = IDBKeyRange.bound([targetId], [targetId + 1], 
        /*lowerOpen=*/ false, 
        /*upperOpen=*/ true);
        var store = documentTargetStore(txn);
        var result = documentKeySet();
        return store
            .iterate({ range: range, keysOnly: true }, function (key, _, control) {
            var path = decode(key[1]);
            var docKey = new DocumentKey(path);
            result = result.add(docKey);
        })
            .next(function () { return result; });
    };
    IndexedDbQueryCache.prototype.containsKey = function (txn, key) {
        var path = encode(key.path);
        var range = IDBKeyRange.bound([path], [immediateSuccessor(path)], 
        /*lowerOpen=*/ false, 
        /*upperOpen=*/ true);
        var count = 0;
        return documentTargetStore(txn)
            .iterate({
            index: DbTargetDocument.documentTargetsIndex,
            keysOnly: true,
            range: range
        }, function (_a, _, control) {
            var targetId = _a[0], path = _a[1];
            // Having a sentinel row for a document does not count as containing that document;
            // For the query cache, containing the document means the document is part of some
            // target.
            if (targetId !== 0) {
                count++;
                control.done();
            }
        })
            .next(function () { return count > 0; });
    };
    IndexedDbQueryCache.prototype.getQueryDataForTarget = function (transaction, targetId) {
        var _this = this;
        return targetsStore(transaction)
            .get(targetId)
            .next(function (found) {
            if (found) {
                return _this.serializer.fromDbTarget(found);
            }
            else {
                return null;
            }
        });
    };
    return IndexedDbQueryCache;
}());
/**
 * Helper to get a typed SimpleDbStore for the queries object store.
 */
function targetsStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbTarget.store);
}
/**
 * Helper to get a typed SimpleDbStore for the target globals object store.
 */
function globalTargetStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbTargetGlobal.store);
}
function retrieveMetadata(txn) {
    var globalStore = SimpleDb.getStore(txn, DbTargetGlobal.store);
    return globalStore.get(DbTargetGlobal.key).next(function (metadata) {
        assert(metadata !== null, 'Missing metadata row.');
        return metadata;
    });
}
function getHighestListenSequenceNumber(txn) {
    return retrieveMetadata(txn).next(function (targetGlobal) { return targetGlobal.highestListenSequenceNumber; });
}
/**
 * Helper to get a typed SimpleDbStore for the document target object store.
 */
function documentTargetStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbTargetDocument.store);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A map implementation that uses objects as keys. Objects must implement the
 * Equatable interface and must be immutable. Entries in the map are stored
 * together with the key being produced from the mapKeyFn. This map
 * automatically handles collisions of keys.
 */
var ObjectMap = /** @class */ (function () {
    function ObjectMap(mapKeyFn) {
        this.mapKeyFn = mapKeyFn;
        /**
         * The inner map for a key -> value pair. Due to the possibility of
         * collisions we keep a list of entries that we do a linear search through
         * to find an actual match. Note that collisions should be rare, so we still
         * expect near constant time lookups in practice.
         */
        this.inner = {};
    }
    /** Get a value for this key, or undefined if it does not exist. */
    ObjectMap.prototype.get = function (key) {
        var id = this.mapKeyFn(key);
        var matches = this.inner[id];
        if (matches === undefined) {
            return undefined;
        }
        for (var _i = 0, matches_1 = matches; _i < matches_1.length; _i++) {
            var _a = matches_1[_i], otherKey = _a[0], value = _a[1];
            if (otherKey.isEqual(key)) {
                return value;
            }
        }
        return undefined;
    };
    ObjectMap.prototype.has = function (key) {
        return this.get(key) !== undefined;
    };
    /** Put this key and value in the map. */
    ObjectMap.prototype.set = function (key, value) {
        var id = this.mapKeyFn(key);
        var matches = this.inner[id];
        if (matches === undefined) {
            this.inner[id] = [[key, value]];
            return;
        }
        for (var i = 0; i < matches.length; i++) {
            if (matches[i][0].isEqual(key)) {
                matches[i] = [key, value];
                return;
            }
        }
        matches.push([key, value]);
    };
    /**
     * Remove this key from the map. Returns a boolean if anything was deleted.
     */
    ObjectMap.prototype.delete = function (key) {
        var id = this.mapKeyFn(key);
        var matches = this.inner[id];
        if (matches === undefined) {
            return false;
        }
        for (var i = 0; i < matches.length; i++) {
            if (matches[i][0].isEqual(key)) {
                if (matches.length === 1) {
                    delete this.inner[id];
                }
                else {
                    matches.splice(i, 1);
                }
                return true;
            }
        }
        return false;
    };
    ObjectMap.prototype.forEach = function (fn) {
        forEach(this.inner, function (_, entries) {
            for (var _i = 0, entries_1 = entries; _i < entries_1.length; _i++) {
                var _a = entries_1[_i], k = _a[0], v = _a[1];
                fn(k, v);
            }
        });
    };
    ObjectMap.prototype.isEmpty = function () {
        return isEmpty(this.inner);
    };
    return ObjectMap;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * An in-memory buffer of entries to be written to a RemoteDocumentCache.
 * It can be used to batch up a set of changes to be written to the cache, but
 * additionally supports reading entries back with the `getEntry()` method,
 * falling back to the underlying RemoteDocumentCache if no entry is
 * buffered.
 *
 * Entries added to the cache *must* be read first. This is to facilitate
 * calculating the size delta of the pending changes.
 *
 * PORTING NOTE: This class was implemented then removed from other platforms.
 * If byte-counting ends up being needed on the other platforms, consider
 * porting this class as part of that implementation work.
 */
var RemoteDocumentChangeBuffer = /** @class */ (function () {
    function RemoteDocumentChangeBuffer() {
        this.changes = maybeDocumentMap();
        this.documentSizes = new ObjectMap(function (key) {
            return key.toString();
        });
    }
    /** Buffers a `RemoteDocumentCache.addEntry()` call. */
    RemoteDocumentChangeBuffer.prototype.addEntry = function (maybeDocument) {
        var changes = this.assertChanges();
        this.changes = changes.insert(maybeDocument.key, maybeDocument);
    };
    // NOTE: removeEntry() is intentionally omitted. If it needs to be added in
    // the future it must take byte counting into account.
    /**
     * Looks up an entry in the cache. The buffered changes will first be checked,
     * and if no buffered change applies, this will forward to
     * `RemoteDocumentCache.getEntry()`.
     *
     * @param transaction The transaction in which to perform any persistence
     *     operations.
     * @param documentKey The key of the entry to look up.
     * @return The cached Document or NoDocument entry, or null if we have nothing
     * cached.
     */
    RemoteDocumentChangeBuffer.prototype.getEntry = function (transaction, documentKey) {
        var _this = this;
        var changes = this.assertChanges();
        var bufferedEntry = changes.get(documentKey);
        if (bufferedEntry) {
            return PersistencePromise.resolve(bufferedEntry);
        }
        else {
            // Record the size of everything we load from the cache so we can compute a delta later.
            return this.getFromCache(transaction, documentKey).next(function (getResult) {
                if (getResult === null) {
                    _this.documentSizes.set(documentKey, 0);
                    return null;
                }
                else {
                    _this.documentSizes.set(documentKey, getResult.size);
                    return getResult.maybeDocument;
                }
            });
        }
    };
    /**
     * Looks up several entries in the cache, forwarding to
     * `RemoteDocumentCache.getEntry()`.
     *
     * @param transaction The transaction in which to perform any persistence
     *     operations.
     * @param documentKeys The keys of the entries to look up.
     * @return A map of cached `Document`s or `NoDocument`s, indexed by key. If an
     *     entry cannot be found, the corresponding key will be mapped to a null
     *     value.
     */
    RemoteDocumentChangeBuffer.prototype.getEntries = function (transaction, documentKeys) {
        var _this = this;
        // Record the size of everything we load from the cache so we can compute
        // a delta later.
        return this.getAllFromCache(transaction, documentKeys).next(function (_a) {
            var maybeDocuments = _a.maybeDocuments, sizeMap = _a.sizeMap;
            // Note: `getAllFromCache` returns two maps instead of a single map from
            // keys to `DocumentSizeEntry`s. This is to allow returning the
            // `NullableMaybeDocumentMap` directly, without a conversion.
            sizeMap.forEach(function (documentKey, size) {
                _this.documentSizes.set(documentKey, size);
            });
            return maybeDocuments;
        });
    };
    /**
     * Applies buffered changes to the underlying RemoteDocumentCache, using
     * the provided transaction.
     */
    RemoteDocumentChangeBuffer.prototype.apply = function (transaction) {
        var result = this.applyChanges(transaction);
        // We should not buffer any more changes.
        this.changes = null;
        return result;
    };
    /** Helper to assert this.changes is not null and return it. */
    RemoteDocumentChangeBuffer.prototype.assertChanges = function () {
        assert(this.changes !== null, 'Changes have already been applied.');
        return this.changes;
    };
    return RemoteDocumentChangeBuffer;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var REMOTE_DOCUMENT_CHANGE_MISSING_ERR_MSG = 'The remote document changelog no longer contains all changes for all ' +
    'local query views. It may be necessary to rebuild these views.';
var IndexedDbRemoteDocumentCache = /** @class */ (function () {
    /**
     * @param {LocalSerializer} serializer The document serializer.
     * @param {IndexManager} indexManager The query indexes that need to be maintained.
     * @param keepDocumentChangeLog Whether to keep a document change log in
     * IndexedDb. This change log is required for Multi-Tab synchronization, but
     * not needed in clients that don't share access to their remote document
     * cache.
     */
    function IndexedDbRemoteDocumentCache(serializer, indexManager, keepDocumentChangeLog) {
        this.serializer = serializer;
        this.indexManager = indexManager;
        this.keepDocumentChangeLog = keepDocumentChangeLog;
        /** The last id read by `getNewDocumentChanges()`. */
        this._lastProcessedDocumentChangeId = 0;
    }
    Object.defineProperty(IndexedDbRemoteDocumentCache.prototype, "lastProcessedDocumentChangeId", {
        get: function () {
            return this._lastProcessedDocumentChangeId;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Starts up the remote document cache.
     *
     * Reads the ID of the last  document change from the documentChanges store.
     * Existing changes will not be returned as part of
     * `getNewDocumentChanges()`.
     */
    // PORTING NOTE: This is only used for multi-tab synchronization.
    IndexedDbRemoteDocumentCache.prototype.start = function (transaction) {
        var store = SimpleDb.getStore(transaction, DbRemoteDocumentChanges.store);
        return this.synchronizeLastDocumentChangeId(store);
    };
    /**
     * Adds the supplied entries to the cache. Adds the given size delta to the cached size.
     */
    IndexedDbRemoteDocumentCache.prototype.addEntries = function (transaction, entries, sizeDelta) {
        var promises = [];
        if (entries.length > 0) {
            var documentStore = remoteDocumentsStore(transaction);
            var changedKeys = documentKeySet();
            for (var _i = 0, entries_1 = entries; _i < entries_1.length; _i++) {
                var _a = entries_1[_i], key = _a.key, doc = _a.doc;
                promises.push(documentStore.put(dbKey(key), doc));
                changedKeys = changedKeys.add(key);
                promises.push(this.indexManager.addToCollectionParentIndex(transaction, key.path.popLast()));
            }
            if (this.keepDocumentChangeLog) {
                promises.push(documentChangesStore(transaction).put({
                    changes: this.serializer.toDbResourcePaths(changedKeys)
                }));
            }
            promises.push(this.updateSize(transaction, sizeDelta));
        }
        return PersistencePromise.waitFor(promises);
    };
    /**
     * Removes a document from the cache. Note that this method does *not* do any
     * size accounting. It is the responsibility of the caller to count the bytes removed
     * and issue a final updateSize() call after removing documents.
     *
     * @param documentKey The key of the document to remove
     * @return The size of the document that was removed.
     */
    IndexedDbRemoteDocumentCache.prototype.removeEntry = function (transaction, documentKey) {
        // We don't need to keep changelog for these removals since `removeEntry` is
        // only used for garbage collection.
        var store = remoteDocumentsStore(transaction);
        var key = dbKey(documentKey);
        return store.get(key).next(function (document) {
            if (document) {
                return store.delete(key).next(function () { return dbDocumentSize(document); });
            }
            else {
                return PersistencePromise.resolve(0);
            }
        });
    };
    IndexedDbRemoteDocumentCache.prototype.getEntry = function (transaction, documentKey) {
        var _this = this;
        return remoteDocumentsStore(transaction)
            .get(dbKey(documentKey))
            .next(function (dbRemoteDoc) {
            return dbRemoteDoc
                ? _this.serializer.fromDbRemoteDocument(dbRemoteDoc)
                : null;
        });
    };
    /**
     * Looks up an entry in the cache.
     *
     * @param documentKey The key of the entry to look up.
     * @return The cached MaybeDocument entry and its size, or null if we have nothing cached.
     */
    IndexedDbRemoteDocumentCache.prototype.getSizedEntry = function (transaction, documentKey) {
        var _this = this;
        return remoteDocumentsStore(transaction)
            .get(dbKey(documentKey))
            .next(function (dbRemoteDoc) {
            return dbRemoteDoc
                ? {
                    maybeDocument: _this.serializer.fromDbRemoteDocument(dbRemoteDoc),
                    size: dbDocumentSize(dbRemoteDoc)
                }
                : null;
        });
    };
    IndexedDbRemoteDocumentCache.prototype.getEntries = function (transaction, documentKeys) {
        var _this = this;
        var results = nullableMaybeDocumentMap();
        return this.forEachDbEntry(transaction, documentKeys, function (key, dbRemoteDoc) {
            if (dbRemoteDoc) {
                results = results.insert(key, _this.serializer.fromDbRemoteDocument(dbRemoteDoc));
            }
            else {
                results = results.insert(key, null);
            }
        }).next(function () { return results; });
    };
    /**
     * Looks up several entries in the cache.
     *
     * @param documentKeys The set of keys entries to look up.
     * @return A map of MaybeDocuments indexed by key (if a document cannot be
     *     found, the key will be mapped to null) and a map of sizes indexed by
     *     key (zero if the key cannot be found).
     */
    IndexedDbRemoteDocumentCache.prototype.getSizedEntries = function (transaction, documentKeys) {
        var _this = this;
        var results = nullableMaybeDocumentMap();
        var sizeMap = new SortedMap(DocumentKey.comparator);
        return this.forEachDbEntry(transaction, documentKeys, function (key, dbRemoteDoc) {
            if (dbRemoteDoc) {
                results = results.insert(key, _this.serializer.fromDbRemoteDocument(dbRemoteDoc));
                sizeMap = sizeMap.insert(key, dbDocumentSize(dbRemoteDoc));
            }
            else {
                results = results.insert(key, null);
                sizeMap = sizeMap.insert(key, 0);
            }
        }).next(function () {
            return { maybeDocuments: results, sizeMap: sizeMap };
        });
    };
    IndexedDbRemoteDocumentCache.prototype.forEachDbEntry = function (transaction, documentKeys, callback) {
        if (documentKeys.isEmpty()) {
            return PersistencePromise.resolve();
        }
        var range = IDBKeyRange.bound(documentKeys.first().path.toArray(), documentKeys.last().path.toArray());
        var keyIter = documentKeys.getIterator();
        var nextKey = keyIter.getNext();
        return remoteDocumentsStore(transaction)
            .iterate({ range: range }, function (potentialKeyRaw, dbRemoteDoc, control) {
            var potentialKey = DocumentKey.fromSegments(potentialKeyRaw);
            // Go through keys not found in cache.
            while (nextKey && DocumentKey.comparator(nextKey, potentialKey) < 0) {
                callback(nextKey, null);
                nextKey = keyIter.getNext();
            }
            if (nextKey && nextKey.isEqual(potentialKey)) {
                // Key found in cache.
                callback(nextKey, dbRemoteDoc);
                nextKey = keyIter.hasNext() ? keyIter.getNext() : null;
            }
            // Skip to the next key (if there is one).
            if (nextKey) {
                control.skip(nextKey.path.toArray());
            }
            else {
                control.done();
            }
        })
            .next(function () {
            // The rest of the keys are not in the cache. One case where `iterate`
            // above won't go through them is when the cache is empty.
            while (nextKey) {
                callback(nextKey, null);
                nextKey = keyIter.hasNext() ? keyIter.getNext() : null;
            }
        });
    };
    IndexedDbRemoteDocumentCache.prototype.getDocumentsMatchingQuery = function (transaction, query) {
        var _this = this;
        assert(!query.isCollectionGroupQuery(), 'CollectionGroup queries should be handled in LocalDocumentsView');
        var results = documentMap();
        var immediateChildrenPathLength = query.path.length + 1;
        // Documents are ordered by key, so we can use a prefix scan to narrow down
        // the documents we need to match the query against.
        var startKey = query.path.toArray();
        var range = IDBKeyRange.lowerBound(startKey);
        return remoteDocumentsStore(transaction)
            .iterate({ range: range }, function (key, dbRemoteDoc, control) {
            // The query is actually returning any path that starts with the query
            // path prefix which may include documents in subcollections. For
            // example, a query on 'rooms' will return rooms/abc/messages/xyx but we
            // shouldn't match it. Fix this by discarding rows with document keys
            // more than one segment longer than the query path.
            if (key.length !== immediateChildrenPathLength) {
                return;
            }
            var maybeDoc = _this.serializer.fromDbRemoteDocument(dbRemoteDoc);
            if (!query.path.isPrefixOf(maybeDoc.key.path)) {
                control.done();
            }
            else if (maybeDoc instanceof Document && query.matches(maybeDoc)) {
                results = results.insert(maybeDoc.key, maybeDoc);
            }
        })
            .next(function () { return results; });
    };
    IndexedDbRemoteDocumentCache.prototype.getNewDocumentChanges = function (transaction) {
        var _this = this;
        assert(this.keepDocumentChangeLog, 'Can only call getNewDocumentChanges() when document change log is enabled');
        var changedKeys = documentKeySet();
        var changedDocs = maybeDocumentMap();
        var range = IDBKeyRange.lowerBound(this._lastProcessedDocumentChangeId + 1);
        var firstIteration = true;
        var changesStore = documentChangesStore(transaction);
        return changesStore
            .iterate({ range: range }, function (_, documentChange) {
            if (firstIteration) {
                firstIteration = false;
                // If our client was throttled for more than 30 minutes, another
                // client may have garbage collected the remote document changelog.
                if (_this._lastProcessedDocumentChangeId + 1 !== documentChange.id) {
                    // Reset the `lastProcessedDocumentChangeId` to allow further
                    // invocations to successfully return the changes after this
                    // rejection.
                    return _this.synchronizeLastDocumentChangeId(changesStore).next(function () {
                        return PersistencePromise.reject(new FirestoreError(Code.DATA_LOSS, REMOTE_DOCUMENT_CHANGE_MISSING_ERR_MSG));
                    });
                }
            }
            changedKeys = changedKeys.unionWith(_this.serializer.fromDbResourcePaths(documentChange.changes));
            _this._lastProcessedDocumentChangeId = documentChange.id;
        })
            .next(function () {
            var documentPromises = [];
            changedKeys.forEach(function (key) {
                documentPromises.push(_this.getEntry(transaction, key).next(function (maybeDocument) {
                    var doc = maybeDocument ||
                        new NoDocument(key, SnapshotVersion.forDeletedDoc());
                    changedDocs = changedDocs.insert(key, doc);
                }));
            });
            return PersistencePromise.waitFor(documentPromises);
        })
            .next(function () { return changedDocs; });
    };
    /**
     * Removes all changes in the remote document changelog through `changeId`
     * (inclusive).
     */
    IndexedDbRemoteDocumentCache.prototype.removeDocumentChangesThroughChangeId = function (transaction, changeId) {
        var range = IDBKeyRange.upperBound(changeId);
        return documentChangesStore(transaction).delete(range);
    };
    IndexedDbRemoteDocumentCache.prototype.synchronizeLastDocumentChangeId = function (documentChangesStore) {
        var _this = this;
        // If there are no existing changes, we set `lastProcessedDocumentChangeId`
        // to 0 since IndexedDb's auto-generated keys start at 1.
        this._lastProcessedDocumentChangeId = 0;
        return documentChangesStore.iterate({ keysOnly: true, reverse: true }, function (key, value, control) {
            _this._lastProcessedDocumentChangeId = key;
            control.done();
        });
    };
    IndexedDbRemoteDocumentCache.prototype.newChangeBuffer = function () {
        return new IndexedDbRemoteDocumentChangeBuffer(this);
    };
    IndexedDbRemoteDocumentCache.prototype.getSize = function (txn) {
        return this.getMetadata(txn).next(function (metadata) { return metadata.byteSize; });
    };
    IndexedDbRemoteDocumentCache.prototype.getMetadata = function (txn) {
        return documentGlobalStore(txn)
            .get(DbRemoteDocumentGlobal.key)
            .next(function (metadata) {
            assert(!!metadata, 'Missing document cache metadata');
            return metadata;
        });
    };
    IndexedDbRemoteDocumentCache.prototype.setMetadata = function (txn, metadata) {
        return documentGlobalStore(txn).put(DbRemoteDocumentGlobal.key, metadata);
    };
    /**
     * Adds the given delta to the cached current size. Callers to removeEntry *must* call this
     * afterwards to update the size of the cache.
     *
     * @param sizeDelta
     */
    IndexedDbRemoteDocumentCache.prototype.updateSize = function (txn, sizeDelta) {
        var _this = this;
        return this.getMetadata(txn).next(function (metadata) {
            metadata.byteSize += sizeDelta;
            return _this.setMetadata(txn, metadata);
        });
    };
    return IndexedDbRemoteDocumentCache;
}());
function documentGlobalStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbRemoteDocumentGlobal.store);
}
/**
 * Handles the details of adding and updating documents in the IndexedDbRemoteDocumentCache
 */
var IndexedDbRemoteDocumentChangeBuffer = /** @class */ (function (_super) {
    tslib_1.__extends(IndexedDbRemoteDocumentChangeBuffer, _super);
    function IndexedDbRemoteDocumentChangeBuffer(documentCache) {
        var _this = _super.call(this) || this;
        _this.documentCache = documentCache;
        return _this;
    }
    IndexedDbRemoteDocumentChangeBuffer.prototype.applyChanges = function (transaction) {
        var _this = this;
        var changes = this.assertChanges();
        var delta = 0;
        var toApply = [];
        changes.forEach(function (key, maybeDocument) {
            var doc = _this.documentCache.serializer.toDbRemoteDocument(maybeDocument);
            var previousSize = _this.documentSizes.get(key);
            // NOTE: if we ever decide we need to support doing writes without
            // reading first, this assert will need to change to do the read automatically.
            assert(previousSize !== undefined, "Attempting to change document " + key.toString() + " without having read it first");
            var size = dbDocumentSize(doc);
            delta += size - previousSize;
            toApply.push({ key: key, doc: doc });
        });
        return this.documentCache.addEntries(transaction, toApply, delta);
    };
    IndexedDbRemoteDocumentChangeBuffer.prototype.getFromCache = function (transaction, documentKey) {
        return this.documentCache.getSizedEntry(transaction, documentKey);
    };
    IndexedDbRemoteDocumentChangeBuffer.prototype.getAllFromCache = function (transaction, documentKeys) {
        return this.documentCache.getSizedEntries(transaction, documentKeys);
    };
    return IndexedDbRemoteDocumentChangeBuffer;
}(RemoteDocumentChangeBuffer));
function isDocumentChangeMissingError(err) {
    return (err.code === Code.DATA_LOSS &&
        err.message === REMOTE_DOCUMENT_CHANGE_MISSING_ERR_MSG);
}
/**
 * Helper to get a typed SimpleDbStore for the remoteDocuments object store.
 */
function remoteDocumentsStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbRemoteDocument.store);
}
/**
 * Helper to get a typed SimpleDbStore for the remoteDocumentChanges object
 * store.
 */
function documentChangesStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbRemoteDocumentChanges.store);
}
function dbKey(docKey) {
    return docKey.path.toArray();
}
/**
 * Retrusn an approximate size for the given document.
 */
function dbDocumentSize(doc) {
    var value;
    if (doc.document) {
        value = doc.document;
    }
    else if (doc.unknownDocument) {
        value = doc.unknownDocument;
    }
    else if (doc.noDocument) {
        value = doc.noDocument;
    }
    else {
        throw fail('Unknown remote document type');
    }
    return JSON.stringify(value).length;
}

/**
 * Copyright 2019 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * An in-memory implementation of IndexManager.
 */
var MemoryIndexManager = /** @class */ (function () {
    function MemoryIndexManager() {
        this.collectionParentIndex = new MemoryCollectionParentIndex();
    }
    MemoryIndexManager.prototype.addToCollectionParentIndex = function (transaction, collectionPath) {
        this.collectionParentIndex.add(collectionPath);
        return PersistencePromise.resolve();
    };
    MemoryIndexManager.prototype.getCollectionParents = function (transaction, collectionId) {
        return PersistencePromise.resolve(this.collectionParentIndex.getEntries(collectionId));
    };
    return MemoryIndexManager;
}());
/**
 * Internal implementation of the collection-parent index exposed by MemoryIndexManager.
 * Also used for in-memory caching by IndexedDbIndexManager and initial index population
 * in indexeddb_schema.ts
 */
var MemoryCollectionParentIndex = /** @class */ (function () {
    function MemoryCollectionParentIndex() {
        this.index = {};
    }
    // Returns false if the entry already existed.
    MemoryCollectionParentIndex.prototype.add = function (collectionPath) {
        assert(collectionPath.length % 2 === 1, 'Expected a collection path.');
        var collectionId = collectionPath.lastSegment();
        var parentPath = collectionPath.popLast();
        var existingParents = this.index[collectionId] ||
            new SortedSet(ResourcePath.comparator);
        var added = !existingParents.has(parentPath);
        this.index[collectionId] = existingParents.add(parentPath);
        return added;
    };
    MemoryCollectionParentIndex.prototype.getEntries = function (collectionId) {
        var parentPaths = this.index[collectionId] ||
            new SortedSet(ResourcePath.comparator);
        return parentPaths.toArray();
    };
    return MemoryCollectionParentIndex;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Schema Version for the Web client:
 * 1. Initial version including Mutation Queue, Query Cache, and Remote Document
 *    Cache
 * 2. Used to ensure a targetGlobal object exists and add targetCount to it. No
 *    longer required because migration 3 unconditionally clears it.
 * 3. Dropped and re-created Query Cache to deal with cache corruption related
 *    to limbo resolution. Addresses
 *    https://github.com/firebase/firebase-ios-sdk/issues/1548
 * 4. Multi-Tab Support.
 * 5. Removal of held write acks.
 * 6. Create document global for tracking document cache size.
 * 7. Ensure every cached document has a sentinel row with a sequence number.
 * 8. Add collection-parent index for Collection Group queries.
 */
var SCHEMA_VERSION = 8;
/** Performs database creation and schema upgrades. */
var SchemaConverter = /** @class */ (function () {
    function SchemaConverter(serializer) {
        this.serializer = serializer;
    }
    /**
     * Performs database creation and schema upgrades.
     *
     * Note that in production, this method is only ever used to upgrade the schema
     * to SCHEMA_VERSION. Different values of toVersion are only used for testing
     * and local feature development.
     */
    SchemaConverter.prototype.createOrUpgrade = function (db, txn, fromVersion, toVersion) {
        var _this = this;
        assert(fromVersion < toVersion &&
            fromVersion >= 0 &&
            toVersion <= SCHEMA_VERSION, "Unexpected schema upgrade from v" + fromVersion + " to v{toVersion}.");
        if (fromVersion < 1 && toVersion >= 1) {
            createPrimaryClientStore(db);
            createMutationQueue(db);
            createQueryCache(db);
            createRemoteDocumentCache(db);
        }
        // Migration 2 to populate the targetGlobal object no longer needed since
        // migration 3 unconditionally clears it.
        var p = PersistencePromise.resolve();
        if (fromVersion < 3 && toVersion >= 3) {
            // Brand new clients don't need to drop and recreate--only clients that
            // potentially have corrupt data.
            if (fromVersion !== 0) {
                dropQueryCache(db);
                createQueryCache(db);
            }
            p = p.next(function () { return writeEmptyTargetGlobalEntry(txn); });
        }
        if (fromVersion < 4 && toVersion >= 4) {
            if (fromVersion !== 0) {
                // Schema version 3 uses auto-generated keys to generate globally unique
                // mutation batch IDs (this was previously ensured internally by the
                // client). To migrate to the new schema, we have to read all mutations
                // and write them back out. We preserve the existing batch IDs to guarantee
                // consistency with other object stores. Any further mutation batch IDs will
                // be auto-generated.
                p = p.next(function () { return upgradeMutationBatchSchemaAndMigrateData(db, txn); });
            }
            p = p.next(function () {
                createClientMetadataStore(db);
                createRemoteDocumentChangesStore(db);
            });
        }
        if (fromVersion < 5 && toVersion >= 5) {
            p = p.next(function () { return _this.removeAcknowledgedMutations(txn); });
        }
        if (fromVersion < 6 && toVersion >= 6) {
            p = p.next(function () {
                createDocumentGlobalStore(db);
                return _this.addDocumentGlobal(txn);
            });
        }
        if (fromVersion < 7 && toVersion >= 7) {
            p = p.next(function () { return _this.ensureSequenceNumbers(txn); });
        }
        if (fromVersion < 8 && toVersion >= 8) {
            p = p.next(function () { return _this.createCollectionParentIndex(db, txn); });
        }
        return p;
    };
    SchemaConverter.prototype.addDocumentGlobal = function (txn) {
        var byteCount = 0;
        return txn
            .store(DbRemoteDocument.store)
            .iterate(function (_, doc) {
            byteCount += dbDocumentSize(doc);
        })
            .next(function () {
            var metadata = new DbRemoteDocumentGlobal(byteCount);
            return txn
                .store(DbRemoteDocumentGlobal.store)
                .put(DbRemoteDocumentGlobal.key, metadata);
        });
    };
    SchemaConverter.prototype.removeAcknowledgedMutations = function (txn) {
        var _this = this;
        var queuesStore = txn.store(DbMutationQueue.store);
        var mutationsStore = txn.store(DbMutationBatch.store);
        return queuesStore.loadAll().next(function (queues) {
            return PersistencePromise.forEach(queues, function (queue) {
                var range = IDBKeyRange.bound([queue.userId, BATCHID_UNKNOWN], [queue.userId, queue.lastAcknowledgedBatchId]);
                return mutationsStore
                    .loadAll(DbMutationBatch.userMutationsIndex, range)
                    .next(function (dbBatches) {
                    return PersistencePromise.forEach(dbBatches, function (dbBatch) {
                        assert(dbBatch.userId === queue.userId, "Cannot process batch " + dbBatch.batchId + " from unexpected user");
                        var batch = _this.serializer.fromDbMutationBatch(dbBatch);
                        return removeMutationBatch(txn, queue.userId, batch).next(function () { });
                    });
                });
            });
        });
    };
    /**
     * Ensures that every document in the remote document cache has a corresponding sentinel row
     * with a sequence number. Missing rows are given the most recently used sequence number.
     */
    SchemaConverter.prototype.ensureSequenceNumbers = function (txn) {
        var documentTargetStore = txn.store(DbTargetDocument.store);
        var documentsStore = txn.store(DbRemoteDocument.store);
        return getHighestListenSequenceNumber(txn).next(function (currentSequenceNumber) {
            var writeSentinelKey = function (path) {
                return documentTargetStore.put(new DbTargetDocument(0, encode(path), currentSequenceNumber));
            };
            var promises = [];
            return documentsStore
                .iterate(function (key, doc) {
                var path = new ResourcePath(key);
                var docSentinelKey = sentinelKey(path);
                promises.push(documentTargetStore.get(docSentinelKey).next(function (maybeSentinel) {
                    if (!maybeSentinel) {
                        return writeSentinelKey(path);
                    }
                    else {
                        return PersistencePromise.resolve();
                    }
                }));
            })
                .next(function () { return PersistencePromise.waitFor(promises); });
        });
    };
    SchemaConverter.prototype.createCollectionParentIndex = function (db, txn) {
        // Create the index.
        db.createObjectStore(DbCollectionParent.store, {
            keyPath: DbCollectionParent.keyPath
        });
        var collectionParentsStore = txn.store(DbCollectionParent.store);
        // Helper to add an index entry iff we haven't already written it.
        var cache = new MemoryCollectionParentIndex();
        var addEntry = function (collectionPath) {
            if (cache.add(collectionPath)) {
                var collectionId = collectionPath.lastSegment();
                var parentPath = collectionPath.popLast();
                return collectionParentsStore.put({
                    collectionId: collectionId,
                    parent: encode(parentPath)
                });
            }
        };
        // Index existing remote documents.
        return txn
            .store(DbRemoteDocument.store)
            .iterate({ keysOnly: true }, function (pathSegments, _) {
            var path = new ResourcePath(pathSegments);
            return addEntry(path.popLast());
        })
            .next(function () {
            // Index existing mutations.
            return txn
                .store(DbDocumentMutation.store)
                .iterate({ keysOnly: true }, function (_a, _) {
                var userID = _a[0], encodedPath = _a[1], batchId = _a[2];
                var path = decode(encodedPath);
                return addEntry(path.popLast());
            });
        });
    };
    return SchemaConverter;
}());
function sentinelKey(path) {
    return [0, encode(path)];
}
/**
 * Wrapper class to store timestamps (seconds and nanos) in IndexedDb objects.
 */
var DbTimestamp = /** @class */ (function () {
    function DbTimestamp(seconds, nanoseconds) {
        this.seconds = seconds;
        this.nanoseconds = nanoseconds;
    }
    return DbTimestamp;
}());
/**
 * A singleton object to be stored in the 'owner' store in IndexedDb.
 *
 * A given database can have a single primary tab assigned at a given time. That
 * tab must validate that it is still holding the primary lease before every
 * operation that requires locked access. The primary tab should regularly
 * write an updated timestamp to this lease to prevent other tabs from
 * "stealing" the primary lease
 */
var DbPrimaryClient = /** @class */ (function () {
    function DbPrimaryClient(ownerId, 
    /** Whether to allow shared access from multiple tabs. */
    allowTabSynchronization, leaseTimestampMs) {
        this.ownerId = ownerId;
        this.allowTabSynchronization = allowTabSynchronization;
        this.leaseTimestampMs = leaseTimestampMs;
    }
    /**
     * Name of the IndexedDb object store.
     *
     * Note that the name 'owner' is chosen to ensure backwards compatibility with
     * older clients that only supported single locked access to the persistence
     * layer.
     */
    DbPrimaryClient.store = 'owner';
    /**
     * The key string used for the single object that exists in the
     * DbPrimaryClient store.
     */
    DbPrimaryClient.key = 'owner';
    return DbPrimaryClient;
}());
function createPrimaryClientStore(db) {
    db.createObjectStore(DbPrimaryClient.store);
}
/**
 * An object to be stored in the 'mutationQueues' store in IndexedDb.
 *
 * Each user gets a single queue of MutationBatches to apply to the server.
 * DbMutationQueue tracks the metadata about the queue.
 */
var DbMutationQueue = /** @class */ (function () {
    function DbMutationQueue(
    /**
     * The normalized user ID to which this queue belongs.
     */
    userId, 
    /**
     * An identifier for the highest numbered batch that has been acknowledged
     * by the server. All MutationBatches in this queue with batchIds less
     * than or equal to this value are considered to have been acknowledged by
     * the server.
     *
     * NOTE: this is deprecated and no longer used by the code.
     */
    lastAcknowledgedBatchId, 
    /**
     * A stream token that was previously sent by the server.
     *
     * See StreamingWriteRequest in datastore.proto for more details about
     * usage.
     *
     * After sending this token, earlier tokens may not be used anymore so
     * only a single stream token is retained.
     */
    lastStreamToken) {
        this.userId = userId;
        this.lastAcknowledgedBatchId = lastAcknowledgedBatchId;
        this.lastStreamToken = lastStreamToken;
    }
    /** Name of the IndexedDb object store.  */
    DbMutationQueue.store = 'mutationQueues';
    /** Keys are automatically assigned via the userId property. */
    DbMutationQueue.keyPath = 'userId';
    return DbMutationQueue;
}());
/**
 * An object to be stored in the 'mutations' store in IndexedDb.
 *
 * Represents a batch of user-level mutations intended to be sent to the server
 * in a single write. Each user-level batch gets a separate DbMutationBatch
 * with a new batchId.
 */
var DbMutationBatch = /** @class */ (function () {
    function DbMutationBatch(
    /**
     * The normalized user ID to which this batch belongs.
     */
    userId, 
    /**
     * An identifier for this batch, allocated using an auto-generated key.
     */
    batchId, 
    /**
     * The local write time of the batch, stored as milliseconds since the
     * epoch.
     */
    localWriteTimeMs, 
    /**
     * A list of "mutations" that represent a partial base state from when this
     * write batch was initially created. During local application of the write
     * batch, these baseMutations are applied prior to the real writes in order
     * to override certain document fields from the remote document cache. This
     * is necessary in the case of non-idempotent writes (e.g. `increment()`
     * transforms) to make sure that the local view of the modified documents
     * doesn't flicker if the remote document cache receives the result of the
     * non-idempotent write before the write is removed from the queue.
     *
     * These mutations are never sent to the backend.
     */
    baseMutations, 
    /**
     * A list of mutations to apply. All mutations will be applied atomically.
     *
     * Mutations are serialized via JsonProtoSerializer.toMutation().
     */
    mutations) {
        this.userId = userId;
        this.batchId = batchId;
        this.localWriteTimeMs = localWriteTimeMs;
        this.baseMutations = baseMutations;
        this.mutations = mutations;
    }
    /** Name of the IndexedDb object store.  */
    DbMutationBatch.store = 'mutations';
    /** Keys are automatically assigned via the userId, batchId properties. */
    DbMutationBatch.keyPath = 'batchId';
    /** The index name for lookup of mutations by user. */
    DbMutationBatch.userMutationsIndex = 'userMutationsIndex';
    /** The user mutations index is keyed by [userId, batchId] pairs. */
    DbMutationBatch.userMutationsKeyPath = ['userId', 'batchId'];
    return DbMutationBatch;
}());
function createMutationQueue(db) {
    db.createObjectStore(DbMutationQueue.store, {
        keyPath: DbMutationQueue.keyPath
    });
    var mutationBatchesStore = db.createObjectStore(DbMutationBatch.store, {
        keyPath: DbMutationBatch.keyPath,
        autoIncrement: true
    });
    mutationBatchesStore.createIndex(DbMutationBatch.userMutationsIndex, DbMutationBatch.userMutationsKeyPath, { unique: true });
    db.createObjectStore(DbDocumentMutation.store);
}
/**
 * Upgrade function to migrate the 'mutations' store from V1 to V3. Loads
 * and rewrites all data.
 */
function upgradeMutationBatchSchemaAndMigrateData(db, txn) {
    var v1MutationsStore = txn.store(DbMutationBatch.store);
    return v1MutationsStore.loadAll().next(function (existingMutations) {
        db.deleteObjectStore(DbMutationBatch.store);
        var mutationsStore = db.createObjectStore(DbMutationBatch.store, {
            keyPath: DbMutationBatch.keyPath,
            autoIncrement: true
        });
        mutationsStore.createIndex(DbMutationBatch.userMutationsIndex, DbMutationBatch.userMutationsKeyPath, { unique: true });
        var v3MutationsStore = txn.store(DbMutationBatch.store);
        var writeAll = existingMutations.map(function (mutation) {
            return v3MutationsStore.put(mutation);
        });
        return PersistencePromise.waitFor(writeAll);
    });
}
/**
 * An object to be stored in the 'documentMutations' store in IndexedDb.
 *
 * A manually maintained index of all the mutation batches that affect a given
 * document key. The rows in this table are references based on the contents of
 * DbMutationBatch.mutations.
 */
var DbDocumentMutation = /** @class */ (function () {
    function DbDocumentMutation() {
    }
    /**
     * Creates a [userId] key for use in the DbDocumentMutations index to iterate
     * over all of a user's document mutations.
     */
    DbDocumentMutation.prefixForUser = function (userId) {
        return [userId];
    };
    /**
     * Creates a [userId, encodedPath] key for use in the DbDocumentMutations
     * index to iterate over all at document mutations for a given path or lower.
     */
    DbDocumentMutation.prefixForPath = function (userId, path) {
        return [userId, encode(path)];
    };
    /**
     * Creates a full index key of [userId, encodedPath, batchId] for inserting
     * and deleting into the DbDocumentMutations index.
     */
    DbDocumentMutation.key = function (userId, path, batchId) {
        return [userId, encode(path), batchId];
    };
    DbDocumentMutation.store = 'documentMutations';
    /**
     * Because we store all the useful information for this store in the key,
     * there is no useful information to store as the value. The raw (unencoded)
     * path cannot be stored because IndexedDb doesn't store prototype
     * information.
     */
    DbDocumentMutation.PLACEHOLDER = new DbDocumentMutation();
    return DbDocumentMutation;
}());
function createRemoteDocumentCache(db) {
    db.createObjectStore(DbRemoteDocument.store);
}
/**
 * Represents the known absence of a document at a particular version.
 * Stored in IndexedDb as part of a DbRemoteDocument object.
 */
var DbNoDocument = /** @class */ (function () {
    function DbNoDocument(path, readTime) {
        this.path = path;
        this.readTime = readTime;
    }
    return DbNoDocument;
}());
/**
 * Represents a document that is known to exist but whose data is unknown.
 * Stored in IndexedDb as part of a DbRemoteDocument object.
 */
var DbUnknownDocument = /** @class */ (function () {
    function DbUnknownDocument(path, version) {
        this.path = path;
        this.version = version;
    }
    return DbUnknownDocument;
}());
/**
 * An object to be stored in the 'remoteDocuments' store in IndexedDb.
 * It represents either:
 *
 * - A complete document.
 * - A "no document" representing a document that is known not to exist (at
 * some version).
 * - An "unknown document" representing a document that is known to exist (at
 * some version) but whose contents are unknown.
 *
 * Note: This is the persisted equivalent of a MaybeDocument and could perhaps
 * be made more general if necessary.
 */
var DbRemoteDocument = /** @class */ (function () {
    function DbRemoteDocument(
    /**
     * Set to an instance of DbUnknownDocument if the data for a document is
     * not known, but it is known that a document exists at the specified
     * version (e.g. it had a successful update applied to it)
     */
    unknownDocument, 
    /**
     * Set to an instance of a DbNoDocument if it is known that no document
     * exists.
     */
    noDocument, 
    /**
     * Set to an instance of a Document if there's a cached version of the
     * document.
     */
    document, 
    /**
     * Documents that were written to the remote document store based on
     * a write acknowledgment are marked with `hasCommittedMutations`. These
     * documents are potentially inconsistent with the backend's copy and use
     * the write's commit version as their document version.
     */
    hasCommittedMutations) {
        this.unknownDocument = unknownDocument;
        this.noDocument = noDocument;
        this.document = document;
        this.hasCommittedMutations = hasCommittedMutations;
    }
    DbRemoteDocument.store = 'remoteDocuments';
    return DbRemoteDocument;
}());
/**
 * Contains a single entry that has metadata about the remote document cache.
 */
var DbRemoteDocumentGlobal = /** @class */ (function () {
    /**
     * @param byteSize Approximately the total size in bytes of all the documents in the document
     * cache.
     */
    function DbRemoteDocumentGlobal(byteSize) {
        this.byteSize = byteSize;
    }
    DbRemoteDocumentGlobal.store = 'remoteDocumentGlobal';
    DbRemoteDocumentGlobal.key = 'remoteDocumentGlobalKey';
    return DbRemoteDocumentGlobal;
}());
function createDocumentGlobalStore(db) {
    db.createObjectStore(DbRemoteDocumentGlobal.store);
}
/**
 * An object to be stored in the 'targets' store in IndexedDb.
 *
 * This is based on and should be kept in sync with the proto used in the iOS
 * client.
 *
 * Each query the client listens to against the server is tracked on disk so
 * that the query can be efficiently resumed on restart.
 */
var DbTarget = /** @class */ (function () {
    function DbTarget(
    /**
     * An auto-generated sequential numeric identifier for the query.
     *
     * Queries are stored using their canonicalId as the key, but these
     * canonicalIds can be quite long so we additionally assign a unique
     * queryId which can be used by referenced data structures (e.g.
     * indexes) to minimize the on-disk cost.
     */
    targetId, 
    /**
     * The canonical string representing this query. This is not unique.
     */
    canonicalId, 
    /**
     * The last readTime received from the Watch Service for this query.
     *
     * This is the same value as TargetChange.read_time in the protos.
     */
    readTime, 
    /**
     * An opaque, server-assigned token that allows watching a query to be
     * resumed after disconnecting without retransmitting all the data
     * that matches the query. The resume token essentially identifies a
     * point in time from which the server should resume sending results.
     *
     * This is related to the snapshotVersion in that the resumeToken
     * effectively also encodes that value, but the resumeToken is opaque
     * and sometimes encodes additional information.
     *
     * A consequence of this is that the resumeToken should be used when
     * asking the server to reason about where this client is in the watch
     * stream, but the client should use the snapshotVersion for its own
     * purposes.
     *
     * This is the same value as TargetChange.resume_token in the protos.
     */
    resumeToken, 
    /**
     * A sequence number representing the last time this query was
     * listened to, used for garbage collection purposes.
     *
     * Conventionally this would be a timestamp value, but device-local
     * clocks are unreliable and they must be able to create new listens
     * even while disconnected. Instead this should be a monotonically
     * increasing number that's incremented on each listen call.
     *
     * This is different from the queryId since the queryId is an
     * immutable identifier assigned to the Query on first use while
     * lastListenSequenceNumber is updated every time the query is
     * listened to.
     */
    lastListenSequenceNumber, 
    /**
     * The query for this target.
     *
     * Because canonical ids are not unique we must store the actual query. We
     * use the proto to have an object we can persist without having to
     * duplicate translation logic to and from a `Query` object.
     */
    query) {
        this.targetId = targetId;
        this.canonicalId = canonicalId;
        this.readTime = readTime;
        this.resumeToken = resumeToken;
        this.lastListenSequenceNumber = lastListenSequenceNumber;
        this.query = query;
    }
    DbTarget.store = 'targets';
    /** Keys are automatically assigned via the targetId property. */
    DbTarget.keyPath = 'targetId';
    /** The name of the queryTargets index. */
    DbTarget.queryTargetsIndexName = 'queryTargetsIndex';
    /**
     * The index of all canonicalIds to the targets that they match. This is not
     * a unique mapping because canonicalId does not promise a unique name for all
     * possible queries, so we append the targetId to make the mapping unique.
     */
    DbTarget.queryTargetsKeyPath = ['canonicalId', 'targetId'];
    return DbTarget;
}());
/**
 * An object representing an association between a target and a document, or a
 * sentinel row marking the last sequence number at which a document was used.
 * Each document cached must have a corresponding sentinel row before lru
 * garbage collection is enabled.
 *
 * The target associations and sentinel rows are co-located so that orphaned
 * documents and their sequence numbers can be identified efficiently via a scan
 * of this store.
 */
var DbTargetDocument = /** @class */ (function () {
    function DbTargetDocument(
    /**
     * The targetId identifying a target or 0 for a sentinel row.
     */
    targetId, 
    /**
     * The path to the document, as encoded in the key.
     */
    path, 
    /**
     * If this is a sentinel row, this should be the sequence number of the last
     * time the document specified by `path` was used. Otherwise, it should be
     * `undefined`.
     */
    sequenceNumber) {
        this.targetId = targetId;
        this.path = path;
        this.sequenceNumber = sequenceNumber;
        assert((targetId === 0) === (sequenceNumber !== undefined), 
        // tslint:disable-next-line:max-line-length
        'A target-document row must either have targetId == 0 and a defined sequence number, or a non-zero targetId and no sequence number');
    }
    /** Name of the IndexedDb object store.  */
    DbTargetDocument.store = 'targetDocuments';
    /** Keys are automatically assigned via the targetId, path properties. */
    DbTargetDocument.keyPath = ['targetId', 'path'];
    /** The index name for the reverse index. */
    DbTargetDocument.documentTargetsIndex = 'documentTargetsIndex';
    /** We also need to create the reverse index for these properties. */
    DbTargetDocument.documentTargetsKeyPath = ['path', 'targetId'];
    return DbTargetDocument;
}());
/**
 * A record of global state tracked across all Targets, tracked separately
 * to avoid the need for extra indexes.
 *
 * This should be kept in-sync with the proto used in the iOS client.
 */
var DbTargetGlobal = /** @class */ (function () {
    function DbTargetGlobal(
    /**
     * The highest numbered target id across all targets.
     *
     * See DbTarget.targetId.
     */
    highestTargetId, 
    /**
     * The highest numbered lastListenSequenceNumber across all targets.
     *
     * See DbTarget.lastListenSequenceNumber.
     */
    highestListenSequenceNumber, 
    /**
     * A global snapshot version representing the last consistent snapshot we
     * received from the backend. This is monotonically increasing and any
     * snapshots received from the backend prior to this version (e.g. for
     * targets resumed with a resumeToken) should be suppressed (buffered)
     * until the backend has caught up to this snapshot version again. This
     * prevents our cache from ever going backwards in time.
     */
    lastRemoteSnapshotVersion, 
    /**
     * The number of targets persisted.
     */
    targetCount) {
        this.highestTargetId = highestTargetId;
        this.highestListenSequenceNumber = highestListenSequenceNumber;
        this.lastRemoteSnapshotVersion = lastRemoteSnapshotVersion;
        this.targetCount = targetCount;
    }
    /**
     * The key string used for the single object that exists in the
     * DbTargetGlobal store.
     */
    DbTargetGlobal.key = 'targetGlobalKey';
    DbTargetGlobal.store = 'targetGlobal';
    return DbTargetGlobal;
}());
/**
 * An object representing an association between a Collection id (e.g. 'messages')
 * to a parent path (e.g. '/chats/123') that contains it as a (sub)collection.
 * This is used to efficiently find all collections to query when performing
 * a Collection Group query.
 */
var DbCollectionParent = /** @class */ (function () {
    function DbCollectionParent(
    /**
     * The collectionId (e.g. 'messages')
     */
    collectionId, 
    /**
     * The path to the parent (either a document location or an empty path for
     * a root-level collection).
     */
    parent) {
        this.collectionId = collectionId;
        this.parent = parent;
    }
    /** Name of the IndexedDb object store. */
    DbCollectionParent.store = 'collectionParents';
    /** Keys are automatically assigned via the collectionId, parent properties. */
    DbCollectionParent.keyPath = ['collectionId', 'parent'];
    return DbCollectionParent;
}());
function createQueryCache(db) {
    var targetDocumentsStore = db.createObjectStore(DbTargetDocument.store, {
        keyPath: DbTargetDocument.keyPath
    });
    targetDocumentsStore.createIndex(DbTargetDocument.documentTargetsIndex, DbTargetDocument.documentTargetsKeyPath, { unique: true });
    var targetStore = db.createObjectStore(DbTarget.store, {
        keyPath: DbTarget.keyPath
    });
    // NOTE: This is unique only because the TargetId is the suffix.
    targetStore.createIndex(DbTarget.queryTargetsIndexName, DbTarget.queryTargetsKeyPath, { unique: true });
    db.createObjectStore(DbTargetGlobal.store);
}
function dropQueryCache(db) {
    db.deleteObjectStore(DbTargetDocument.store);
    db.deleteObjectStore(DbTarget.store);
    db.deleteObjectStore(DbTargetGlobal.store);
}
/**
 * Creates the target global singleton row.
 *
 * @param {IDBTransaction} txn The version upgrade transaction for indexeddb
 */
function writeEmptyTargetGlobalEntry(txn) {
    var globalStore = txn.store(DbTargetGlobal.store);
    var metadata = new DbTargetGlobal(
    /*highestTargetId=*/ 0, 
    /*lastListenSequenceNumber=*/ 0, SnapshotVersion.MIN.toTimestamp(), 
    /*targetCount=*/ 0);
    return globalStore.put(DbTargetGlobal.key, metadata);
}
/**
 * An object store to store the keys of changed documents. This is used to
 * facilitate storing document changelogs in the Remote Document Cache.
 *
 * PORTING NOTE: This is used for change propagation during multi-tab syncing
 * and not needed on iOS and Android.
 */
var DbRemoteDocumentChanges = /** @class */ (function () {
    function DbRemoteDocumentChanges(
    /** The keys of the changed documents. */
    changes) {
        this.changes = changes;
    }
    /** Name of the IndexedDb object store.  */
    DbRemoteDocumentChanges.store = 'remoteDocumentChanges';
    /** Keys are auto-generated via the `id` property. */
    DbRemoteDocumentChanges.keyPath = 'id';
    return DbRemoteDocumentChanges;
}());
function createRemoteDocumentChangesStore(db) {
    db.createObjectStore(DbRemoteDocumentChanges.store, {
        keyPath: 'id',
        autoIncrement: true
    });
}
/**
 * A record of the metadata state of each client.
 *
 * PORTING NOTE: This is used to synchronize multi-tab state and does not need
 * to be ported to iOS or Android.
 */
var DbClientMetadata = /** @class */ (function () {
    function DbClientMetadata(
    /** The auto-generated client id assigned at client startup. */
    clientId, 
    /** The last time this state was updated. */
    updateTimeMs, 
    /** Whether the client's network connection is enabled. */
    networkEnabled, 
    /** Whether this client is running in a foreground tab. */
    inForeground, 
    /**
     * The last change read from the DbRemoteDocumentChanges store.
     * Can be undefined for backwards compatibility.
     */
    lastProcessedDocumentChangeId) {
        this.clientId = clientId;
        this.updateTimeMs = updateTimeMs;
        this.networkEnabled = networkEnabled;
        this.inForeground = inForeground;
        this.lastProcessedDocumentChangeId = lastProcessedDocumentChangeId;
    }
    /** Name of the IndexedDb object store. */
    DbClientMetadata.store = 'clientMetadata';
    /** Keys are automatically assigned via the clientId properties. */
    DbClientMetadata.keyPath = 'clientId';
    return DbClientMetadata;
}());
function createClientMetadataStore(db) {
    db.createObjectStore(DbClientMetadata.store, {
        keyPath: DbClientMetadata.keyPath
    });
}
// Visible for testing
var V1_STORES = [
    DbMutationQueue.store,
    DbMutationBatch.store,
    DbDocumentMutation.store,
    DbRemoteDocument.store,
    DbTarget.store,
    DbPrimaryClient.store,
    DbTargetGlobal.store,
    DbTargetDocument.store
];
// V2 is no longer usable (see comment at top of file)
// Visible for testing
var V3_STORES = V1_STORES;
// Visible for testing
var V4_STORES = V3_STORES.concat([
    DbClientMetadata.store,
    DbRemoteDocumentChanges.store
]);
// V5 does not change the set of stores.
var V6_STORES = V4_STORES.concat([DbRemoteDocumentGlobal.store]);
// V7 does not change the set of stores.
var V8_STORES = V6_STORES.concat([DbCollectionParent.store]);
/**
 * The list of all default IndexedDB stores used throughout the SDK. This is
 * used when creating transactions so that access across all stores is done
 * atomically.
 */
var ALL_STORES = V8_STORES;

/**
 * Copyright 2019 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A persisted implementation of IndexManager.
 */
var IndexedDbIndexManager = /** @class */ (function () {
    function IndexedDbIndexManager() {
        /**
         * An in-memory copy of the index entries we've already written since the SDK
         * launched. Used to avoid re-writing the same entry repeatedly.
         *
         * This is *NOT* a complete cache of what's in persistence and so can never be used to
         * satisfy reads.
         */
        this.collectionParentsCache = new MemoryCollectionParentIndex();
    }
    IndexedDbIndexManager.prototype.addToCollectionParentIndex = function (transaction, collectionPath) {
        assert(collectionPath.length % 2 === 1, 'Expected a collection path.');
        if (this.collectionParentsCache.add(collectionPath)) {
            assert(collectionPath.length >= 1, 'Invalid collection path.');
            var collectionId = collectionPath.lastSegment();
            var parentPath = collectionPath.popLast();
            return collectionParentsStore(transaction).put({
                collectionId: collectionId,
                parent: encode(parentPath)
            });
        }
        return PersistencePromise.resolve();
    };
    IndexedDbIndexManager.prototype.getCollectionParents = function (transaction, collectionId) {
        var parentPaths = [];
        var range = IDBKeyRange.bound([collectionId, ''], [immediateSuccessor(collectionId), ''], 
        /*lowerOpen=*/ false, 
        /*upperOpen=*/ true);
        return collectionParentsStore(transaction)
            .loadAll(range)
            .next(function (entries) {
            for (var _i = 0, entries_1 = entries; _i < entries_1.length; _i++) {
                var entry = entries_1[_i];
                // This collectionId guard shouldn't be necessary (and isn't as long
                // as we're running in a real browser), but there's a bug in
                // indexeddbshim that breaks our range in our tests running in node:
                // https://github.com/axemclion/IndexedDBShim/issues/334
                if (entry.collectionId !== collectionId) {
                    break;
                }
                parentPaths.push(decode(entry.parent));
            }
            return parentPaths;
        });
    };
    return IndexedDbIndexManager;
}());
/**
 * Helper to get a typed SimpleDbStore for the collectionParents
 * document store.
 */
function collectionParentsStore(txn) {
    return IndexedDbPersistence.getStore(txn, DbCollectionParent.store);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/** Serializer for values stored in the LocalStore. */
var LocalSerializer = /** @class */ (function () {
    function LocalSerializer(remoteSerializer) {
        this.remoteSerializer = remoteSerializer;
    }
    /** Decodes a remote document from storage locally to a Document. */
    LocalSerializer.prototype.fromDbRemoteDocument = function (remoteDoc) {
        if (remoteDoc.document) {
            return this.remoteSerializer.fromDocument(remoteDoc.document, !!remoteDoc.hasCommittedMutations);
        }
        else if (remoteDoc.noDocument) {
            var key = DocumentKey.fromSegments(remoteDoc.noDocument.path);
            var version = this.fromDbTimestamp(remoteDoc.noDocument.readTime);
            return new NoDocument(key, version, {
                hasCommittedMutations: !!remoteDoc.hasCommittedMutations
            });
        }
        else if (remoteDoc.unknownDocument) {
            var key = DocumentKey.fromSegments(remoteDoc.unknownDocument.path);
            var version = this.fromDbTimestamp(remoteDoc.unknownDocument.version);
            return new UnknownDocument(key, version);
        }
        else {
            return fail('Unexpected DbRemoteDocument');
        }
    };
    /** Encodes a document for storage locally. */
    LocalSerializer.prototype.toDbRemoteDocument = function (maybeDoc) {
        if (maybeDoc instanceof Document) {
            var doc = maybeDoc.proto
                ? maybeDoc.proto
                : this.remoteSerializer.toDocument(maybeDoc);
            var hasCommittedMutations = maybeDoc.hasCommittedMutations;
            return new DbRemoteDocument(
            /* unknownDocument= */ null, 
            /* noDocument= */ null, doc, hasCommittedMutations);
        }
        else if (maybeDoc instanceof NoDocument) {
            var path = maybeDoc.key.path.toArray();
            var readTime = this.toDbTimestamp(maybeDoc.version);
            var hasCommittedMutations = maybeDoc.hasCommittedMutations;
            return new DbRemoteDocument(
            /* unknownDocument= */ null, new DbNoDocument(path, readTime), 
            /* document= */ null, hasCommittedMutations);
        }
        else if (maybeDoc instanceof UnknownDocument) {
            var path = maybeDoc.key.path.toArray();
            var readTime = this.toDbTimestamp(maybeDoc.version);
            return new DbRemoteDocument(new DbUnknownDocument(path, readTime), 
            /* noDocument= */ null, 
            /* document= */ null, 
            /* hasCommittedMutations= */ true);
        }
        else {
            return fail('Unexpected MaybeDocumment');
        }
    };
    LocalSerializer.prototype.toDbTimestamp = function (snapshotVersion) {
        var timestamp = snapshotVersion.toTimestamp();
        return new DbTimestamp(timestamp.seconds, timestamp.nanoseconds);
    };
    LocalSerializer.prototype.fromDbTimestamp = function (dbTimestamp) {
        var timestamp = new Timestamp(dbTimestamp.seconds, dbTimestamp.nanoseconds);
        return SnapshotVersion.fromTimestamp(timestamp);
    };
    /** Encodes a batch of mutations into a DbMutationBatch for local storage. */
    LocalSerializer.prototype.toDbMutationBatch = function (userId, batch) {
        var _this = this;
        var serializedBaseMutations = batch.baseMutations.map(function (m) {
            return _this.remoteSerializer.toMutation(m);
        });
        var serializedMutations = batch.mutations.map(function (m) {
            return _this.remoteSerializer.toMutation(m);
        });
        return new DbMutationBatch(userId, batch.batchId, batch.localWriteTime.toMillis(), serializedBaseMutations, serializedMutations);
    };
    /** Decodes a DbMutationBatch into a MutationBatch */
    LocalSerializer.prototype.fromDbMutationBatch = function (dbBatch) {
        var _this = this;
        var baseMutations = (dbBatch.baseMutations || []).map(function (m) {
            return _this.remoteSerializer.fromMutation(m);
        });
        var mutations = dbBatch.mutations.map(function (m) {
            return _this.remoteSerializer.fromMutation(m);
        });
        var timestamp = Timestamp.fromMillis(dbBatch.localWriteTimeMs);
        return new MutationBatch(dbBatch.batchId, timestamp, baseMutations, mutations);
    };
    /*
     * Encodes a set of document keys into an array of EncodedResourcePaths.
     */
    LocalSerializer.prototype.toDbResourcePaths = function (keys) {
        var encodedKeys = [];
        keys.forEach(function (key) {
            encodedKeys.push(encode(key.path));
        });
        return encodedKeys;
    };
    /** Decodes an array of EncodedResourcePaths into a set of document keys. */
    LocalSerializer.prototype.fromDbResourcePaths = function (encodedPaths) {
        var keys = documentKeySet();
        for (var _i = 0, encodedPaths_1 = encodedPaths; _i < encodedPaths_1.length; _i++) {
            var documentKey = encodedPaths_1[_i];
            keys = keys.add(new DocumentKey(decode(documentKey)));
        }
        return keys;
    };
    /** Decodes a DbTarget into QueryData */
    LocalSerializer.prototype.fromDbTarget = function (dbTarget) {
        var version = this.fromDbTimestamp(dbTarget.readTime);
        var query;
        if (isDocumentQuery(dbTarget.query)) {
            query = this.remoteSerializer.fromDocumentsTarget(dbTarget.query);
        }
        else {
            query = this.remoteSerializer.fromQueryTarget(dbTarget.query);
        }
        return new QueryData(query, dbTarget.targetId, QueryPurpose.Listen, dbTarget.lastListenSequenceNumber, version, dbTarget.resumeToken);
    };
    /** Encodes QueryData into a DbTarget for storage locally. */
    LocalSerializer.prototype.toDbTarget = function (queryData) {
        assert(QueryPurpose.Listen === queryData.purpose, 'Only queries with purpose ' +
            QueryPurpose.Listen +
            ' may be stored, got ' +
            queryData.purpose);
        var dbTimestamp = this.toDbTimestamp(queryData.snapshotVersion);
        var queryProto;
        if (queryData.query.isDocumentQuery()) {
            queryProto = this.remoteSerializer.toDocumentsTarget(queryData.query);
        }
        else {
            queryProto = this.remoteSerializer.toQueryTarget(queryData.query);
        }
        var resumeToken;
        if (queryData.resumeToken instanceof Uint8Array) {
            // TODO(b/78771403): Convert tokens to strings during deserialization
            assert(process.env.USE_MOCK_PERSISTENCE === 'YES', 'Persisting non-string stream tokens is only supported with mock persistence .');
            resumeToken = queryData.resumeToken.toString();
        }
        else {
            resumeToken = queryData.resumeToken;
        }
        // lastListenSequenceNumber is always 0 until we do real GC.
        return new DbTarget(queryData.targetId, queryData.query.canonicalId(), dbTimestamp, resumeToken, queryData.sequenceNumber, queryProto);
    };
    return LocalSerializer;
}());
/**
 * A helper function for figuring out what kind of query has been stored.
 */
function isDocumentQuery(dbQuery) {
    return dbQuery.documents !== undefined;
}

/**
 * @license
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
function bufferEntryComparator(_a, _b) {
    var aSequence = _a[0], aIndex = _a[1];
    var bSequence = _b[0], bIndex = _b[1];
    var seqCmp = primitiveComparator(aSequence, bSequence);
    if (seqCmp === 0) {
        // This order doesn't matter, but we can bias against churn by sorting
        // entries created earlier as less than newer entries.
        return primitiveComparator(aIndex, bIndex);
    }
    else {
        return seqCmp;
    }
}
/**
 * Used to calculate the nth sequence number. Keeps a rolling buffer of the
 * lowest n values passed to `addElement`, and finally reports the largest of
 * them in `maxValue`.
 */
var RollingSequenceNumberBuffer = /** @class */ (function () {
    function RollingSequenceNumberBuffer(maxElements) {
        this.maxElements = maxElements;
        this.buffer = new SortedSet(bufferEntryComparator);
        this.previousIndex = 0;
    }
    RollingSequenceNumberBuffer.prototype.nextIndex = function () {
        return ++this.previousIndex;
    };
    RollingSequenceNumberBuffer.prototype.addElement = function (sequenceNumber) {
        var entry = [sequenceNumber, this.nextIndex()];
        if (this.buffer.size < this.maxElements) {
            this.buffer = this.buffer.add(entry);
        }
        else {
            var highestValue = this.buffer.last();
            if (bufferEntryComparator(entry, highestValue) < 0) {
                this.buffer = this.buffer.delete(highestValue).add(entry);
            }
        }
    };
    Object.defineProperty(RollingSequenceNumberBuffer.prototype, "maxValue", {
        get: function () {
            // Guaranteed to be non-empty. If we decide we are not collecting any
            // sequence numbers, nthSequenceNumber below short-circuits. If we have
            // decided that we are collecting n sequence numbers, it's because n is some
            // percentage of the existing sequence numbers. That means we should never
            // be in a situation where we are collecting sequence numbers but don't
            // actually have any.
            return this.buffer.last()[0];
        },
        enumerable: true,
        configurable: true
    });
    return RollingSequenceNumberBuffer;
}());
var GC_DID_NOT_RUN = {
    didRun: false,
    sequenceNumbersCollected: 0,
    targetsRemoved: 0,
    documentsRemoved: 0
};
var LruParams = /** @class */ (function () {
    function LruParams(
    // When we attempt to collect, we will only do so if the cache size is greater than this
    // threshold. Passing `COLLECTION_DISABLED` here will cause collection to always be skipped.
    cacheSizeCollectionThreshold, 
    // The percentage of sequence numbers that we will attempt to collect
    percentileToCollect, 
    // A cap on the total number of sequence numbers that will be collected. This prevents
    // us from collecting a huge number of sequence numbers if the cache has grown very large.
    maximumSequenceNumbersToCollect) {
        this.cacheSizeCollectionThreshold = cacheSizeCollectionThreshold;
        this.percentileToCollect = percentileToCollect;
        this.maximumSequenceNumbersToCollect = maximumSequenceNumbersToCollect;
    }
    LruParams.withCacheSize = function (cacheSize) {
        return new LruParams(cacheSize, LruParams.DEFAULT_COLLECTION_PERCENTILE, LruParams.DEFAULT_MAX_SEQUENCE_NUMBERS_TO_COLLECT);
    };
    LruParams.COLLECTION_DISABLED = -1;
    LruParams.MINIMUM_CACHE_SIZE_BYTES = 1 * 1024 * 1024;
    LruParams.DEFAULT_CACHE_SIZE_BYTES = 40 * 1024 * 1024;
    LruParams.DEFAULT_COLLECTION_PERCENTILE = 10;
    LruParams.DEFAULT_MAX_SEQUENCE_NUMBERS_TO_COLLECT = 1000;
    LruParams.DEFAULT = new LruParams(LruParams.DEFAULT_CACHE_SIZE_BYTES, LruParams.DEFAULT_COLLECTION_PERCENTILE, LruParams.DEFAULT_MAX_SEQUENCE_NUMBERS_TO_COLLECT);
    LruParams.DISABLED = new LruParams(LruParams.COLLECTION_DISABLED, 0, 0);
    return LruParams;
}());
/** How long we wait to try running LRU GC after SDK initialization. */
var INITIAL_GC_DELAY_MS = 1 * 60 * 1000;
/** Minimum amount of time between GC checks, after the first one. */
var REGULAR_GC_DELAY_MS = 5 * 60 * 1000;
/**
 * This class is responsible for the scheduling of LRU garbage collection. It handles checking
 * whether or not GC is enabled, as well as which delay to use before the next run.
 */
var LruScheduler = /** @class */ (function () {
    function LruScheduler(garbageCollector, asyncQueue, localStore) {
        this.garbageCollector = garbageCollector;
        this.asyncQueue = asyncQueue;
        this.localStore = localStore;
        this.gcTask = null;
    }
    LruScheduler.prototype.start = function () {
        assert(this.gcTask === null, 'Cannot start an already started LruScheduler');
        if (this.garbageCollector.params.cacheSizeCollectionThreshold !==
            LruParams.COLLECTION_DISABLED) {
            this.scheduleGC();
        }
    };
    LruScheduler.prototype.stop = function () {
        if (this.gcTask) {
            this.gcTask.cancel();
            this.gcTask = null;
        }
    };
    Object.defineProperty(LruScheduler.prototype, "started", {
        get: function () {
            return this.gcTask !== null;
        },
        enumerable: true,
        configurable: true
    });
    LruScheduler.prototype.scheduleGC = function () {
        var _this = this;
        assert(this.gcTask === null, 'Cannot schedule GC while a task is pending');
        var delay = this.hasRun ? REGULAR_GC_DELAY_MS : INITIAL_GC_DELAY_MS;
        debug('LruGarbageCollector', "Garbage collection scheduled in " + delay + "ms");
        this.gcTask = this.asyncQueue.enqueueAfterDelay(TimerId.LruGarbageCollection, delay, function () {
            _this.gcTask = null;
            _this.hasRun = true;
            return _this.localStore
                .collectGarbage(_this.garbageCollector)
                .then(function () { return _this.scheduleGC(); })
                .catch(ignoreIfPrimaryLeaseLoss);
        });
    };
    return LruScheduler;
}());
/** Implements the steps for LRU garbage collection. */
var LruGarbageCollector = /** @class */ (function () {
    function LruGarbageCollector(delegate, params) {
        this.delegate = delegate;
        this.params = params;
    }
    /** Given a percentile of target to collect, returns the number of targets to collect. */
    LruGarbageCollector.prototype.calculateTargetCount = function (txn, percentile) {
        return this.delegate.getSequenceNumberCount(txn).next(function (targetCount) {
            return Math.floor((percentile / 100.0) * targetCount);
        });
    };
    /** Returns the nth sequence number, counting in order from the smallest. */
    LruGarbageCollector.prototype.nthSequenceNumber = function (txn, n) {
        var _this = this;
        if (n === 0) {
            return PersistencePromise.resolve(ListenSequence.INVALID);
        }
        var buffer = new RollingSequenceNumberBuffer(n);
        return this.delegate
            .forEachTarget(txn, function (target) { return buffer.addElement(target.sequenceNumber); })
            .next(function () {
            return _this.delegate.forEachOrphanedDocumentSequenceNumber(txn, function (sequenceNumber) { return buffer.addElement(sequenceNumber); });
        })
            .next(function () { return buffer.maxValue; });
    };
    /**
     * Removes targets with a sequence number equal to or less than the given upper bound, and removes
     * document associations with those targets.
     */
    LruGarbageCollector.prototype.removeTargets = function (txn, upperBound, activeTargetIds) {
        return this.delegate.removeTargets(txn, upperBound, activeTargetIds);
    };
    /**
     * Removes documents that have a sequence number equal to or less than the upper bound and are not
     * otherwise pinned.
     */
    LruGarbageCollector.prototype.removeOrphanedDocuments = function (txn, upperBound) {
        return this.delegate.removeOrphanedDocuments(txn, upperBound);
    };
    LruGarbageCollector.prototype.collect = function (txn, activeTargetIds) {
        var _this = this;
        if (this.params.cacheSizeCollectionThreshold === LruParams.COLLECTION_DISABLED) {
            debug('LruGarbageCollector', 'Garbage collection skipped; disabled');
            return PersistencePromise.resolve(GC_DID_NOT_RUN);
        }
        return this.getCacheSize(txn).next(function (cacheSize) {
            if (cacheSize < _this.params.cacheSizeCollectionThreshold) {
                debug('LruGarbageCollector', "Garbage collection skipped; Cache size " + cacheSize + " " +
                    ("is lower than threshold " + _this.params.cacheSizeCollectionThreshold));
                return GC_DID_NOT_RUN;
            }
            else {
                return _this.runGarbageCollection(txn, activeTargetIds);
            }
        });
    };
    LruGarbageCollector.prototype.getCacheSize = function (txn) {
        return this.delegate.getCacheSize(txn);
    };
    LruGarbageCollector.prototype.runGarbageCollection = function (txn, activeTargetIds) {
        var _this = this;
        var upperBoundSequenceNumber;
        var sequenceNumbersToCollect, targetsRemoved;
        // Timestamps for various pieces of the process
        var startTs, countedTargetsTs, foundUpperBoundTs, removedTargetsTs, removedDocumentsTs;
        startTs = Date.now();
        return this.calculateTargetCount(txn, this.params.percentileToCollect)
            .next(function (sequenceNumbers) {
            // Cap at the configured max
            if (sequenceNumbers > _this.params.maximumSequenceNumbersToCollect) {
                debug('LruGarbageCollector', 'Capping sequence numbers to collect down ' +
                    ("to the maximum of " + _this.params.maximumSequenceNumbersToCollect + " ") +
                    ("from " + sequenceNumbers));
                sequenceNumbersToCollect = _this.params
                    .maximumSequenceNumbersToCollect;
            }
            else {
                sequenceNumbersToCollect = sequenceNumbers;
            }
            countedTargetsTs = Date.now();
            return _this.nthSequenceNumber(txn, sequenceNumbersToCollect);
        })
            .next(function (upperBound) {
            upperBoundSequenceNumber = upperBound;
            foundUpperBoundTs = Date.now();
            return _this.removeTargets(txn, upperBoundSequenceNumber, activeTargetIds);
        })
            .next(function (numTargetsRemoved) {
            targetsRemoved = numTargetsRemoved;
            removedTargetsTs = Date.now();
            return _this.removeOrphanedDocuments(txn, upperBoundSequenceNumber);
        })
            .next(function (documentsRemoved) {
            removedDocumentsTs = Date.now();
            if (getLogLevel() <= LogLevel.DEBUG) {
                var desc = 'LRU Garbage Collection\n' +
                    ("\tCounted targets in " + (countedTargetsTs - startTs) + "ms\n") +
                    ("\tDetermined least recently used " + sequenceNumbersToCollect + " in ") +
                    (foundUpperBoundTs - countedTargetsTs + "ms\n") +
                    ("\tRemoved " + targetsRemoved + " targets in ") +
                    (removedTargetsTs - foundUpperBoundTs + "ms\n") +
                    ("\tRemoved " + documentsRemoved + " documents in ") +
                    (removedDocumentsTs - removedTargetsTs + "ms\n") +
                    ("Total Duration: " + (removedDocumentsTs - startTs) + "ms");
                debug('LruGarbageCollector', desc);
            }
            return PersistencePromise.resolve({
                didRun: true,
                sequenceNumbersCollected: sequenceNumbersToCollect,
                targetsRemoved: targetsRemoved,
                documentsRemoved: documentsRemoved
            });
        });
    };
    return LruGarbageCollector;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Opaque interface representing a persistence transaction.
 *
 * When you call Persistence.runTransaction(), it will create a transaction and
 * pass it to your callback. You then pass it to any method that operates
 * on persistence.
 */
var PersistenceTransaction = /** @class */ (function () {
    function PersistenceTransaction() {
    }
    return PersistenceTransaction;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$2 = 'IndexedDbPersistence';
/**
 * Oldest acceptable age in milliseconds for client metadata before the client
 * is considered inactive and its associated data (such as the remote document
 * cache changelog) is garbage collected.
 */
var MAX_CLIENT_AGE_MS = 30 * 60 * 1000; // 30 minutes
/**
 * Oldest acceptable metadata age for clients that may participate in the
 * primary lease election. Clients that have not updated their client metadata
 * within 5 seconds are not eligible to receive a primary lease.
 */
var MAX_PRIMARY_ELIGIBLE_AGE_MS = 5000;
/**
 * The interval at which clients will update their metadata, including
 * refreshing their primary lease if held or potentially trying to acquire it if
 * not held.
 *
 * Primary clients may opportunistically refresh their metadata earlier
 * if they're already performing an IndexedDB operation.
 */
var CLIENT_METADATA_REFRESH_INTERVAL_MS = 4000;
/** User-facing error when the primary lease is required but not available. */
var PRIMARY_LEASE_LOST_ERROR_MSG = 'The current tab is not in the required state to perform this operation. ' +
    'It might be necessary to refresh the browser tab.';
var PRIMARY_LEASE_EXCLUSIVE_ERROR_MSG = 'Another tab has exclusive access to the persistence layer. ' +
    'To allow shared access, make sure to invoke ' +
    '`enablePersistence()` with `experimentalTabSynchronization:true` in all tabs.';
var UNSUPPORTED_PLATFORM_ERROR_MSG = 'This platform is either missing' +
    ' IndexedDB or is known to have an incomplete implementation. Offline' +
    ' persistence has been disabled.';
// The format of the LocalStorage key that stores zombied client is:
//     firestore_zombie_<persistence_prefix>_<instance_key>
var ZOMBIED_CLIENTS_KEY_PREFIX = 'firestore_zombie';
var IndexedDbTransaction = /** @class */ (function (_super) {
    tslib_1.__extends(IndexedDbTransaction, _super);
    function IndexedDbTransaction(simpleDbTransaction, currentSequenceNumber) {
        var _this = _super.call(this) || this;
        _this.simpleDbTransaction = simpleDbTransaction;
        _this.currentSequenceNumber = currentSequenceNumber;
        return _this;
    }
    return IndexedDbTransaction;
}(PersistenceTransaction));
var IndexedDbPersistence = /** @class */ (function () {
    // Note that `multiClientParams` must be present to enable multi-client support while multi-tab
    // is still experimental. When multi-client is switched to always on, `multiClientParams` will
    // no longer be optional.
    function IndexedDbPersistence(persistenceKey, clientId, platform, queue, serializer, lruParams, multiClientParams) {
        this.persistenceKey = persistenceKey;
        this.clientId = clientId;
        this.queue = queue;
        this.multiClientParams = multiClientParams;
        this._started = false;
        this.isPrimary = false;
        this.networkEnabled = true;
        this.inForeground = false;
        /** The last time we garbage collected the Remote Document Changelog. */
        this.lastGarbageCollectionTime = Number.NEGATIVE_INFINITY;
        /** A listener to notify on primary state changes. */
        this.primaryStateListener = function (_) { return Promise.resolve(); };
        if (!IndexedDbPersistence.isAvailable()) {
            throw new FirestoreError(Code.UNIMPLEMENTED, UNSUPPORTED_PLATFORM_ERROR_MSG);
        }
        this.referenceDelegate = new IndexedDbLruDelegate(this, lruParams);
        this.dbName = persistenceKey + IndexedDbPersistence.MAIN_DATABASE;
        this.serializer = new LocalSerializer(serializer);
        this.document = platform.document;
        this.allowTabSynchronization = multiClientParams !== undefined;
        this.queryCache = new IndexedDbQueryCache(this.referenceDelegate, this.serializer);
        this.indexManager = new IndexedDbIndexManager();
        this.remoteDocumentCache = new IndexedDbRemoteDocumentCache(this.serializer, this.indexManager, 
        /*keepDocumentChangeLog=*/ this.allowTabSynchronization);
        if (platform.window && platform.window.localStorage) {
            this.window = platform.window;
            this.webStorage = this.window.localStorage;
        }
        else {
            throw new FirestoreError(Code.UNIMPLEMENTED, 'IndexedDB persistence is only available on platforms that support LocalStorage.');
        }
    }
    IndexedDbPersistence.getStore = function (txn, store) {
        if (txn instanceof IndexedDbTransaction) {
            return SimpleDb.getStore(txn.simpleDbTransaction, store);
        }
        else {
            throw fail('IndexedDbPersistence must use instances of IndexedDbTransaction');
        }
    };
    IndexedDbPersistence.createIndexedDbPersistence = function (persistenceKey, clientId, platform, queue, serializer, lruParams) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var persistence;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        persistence = new IndexedDbPersistence(persistenceKey, clientId, platform, queue, serializer, lruParams);
                        return [4 /*yield*/, persistence.start()];
                    case 1:
                        _a.sent();
                        return [2 /*return*/, persistence];
                }
            });
        });
    };
    IndexedDbPersistence.createMultiClientIndexedDbPersistence = function (persistenceKey, clientId, platform, queue, serializer, lruParams, multiClientParams) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var persistence;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        persistence = new IndexedDbPersistence(persistenceKey, clientId, platform, queue, serializer, lruParams, multiClientParams);
                        return [4 /*yield*/, persistence.start()];
                    case 1:
                        _a.sent();
                        return [2 /*return*/, persistence];
                }
            });
        });
    };
    /**
     * Attempt to start IndexedDb persistence.
     *
     * @return {Promise<void>} Whether persistence was enabled.
     */
    IndexedDbPersistence.prototype.start = function () {
        var _this = this;
        assert(!this.started, 'IndexedDbPersistence double-started!');
        assert(this.window !== null, "Expected 'window' to be defined");
        return SimpleDb.openOrCreate(this.dbName, SCHEMA_VERSION, new SchemaConverter(this.serializer))
            .then(function (db) {
            _this.simpleDb = db;
            // NOTE: This is expected to fail sometimes (in the case of another tab already
            // having the persistence lock), so it's the first thing we should do.
            return _this.updateClientMetadataAndTryBecomePrimary();
        })
            .then(function () {
            _this.attachVisibilityHandler();
            _this.attachWindowUnloadHook();
            _this.scheduleClientMetadataAndPrimaryLeaseRefreshes();
            return _this.startRemoteDocumentCache();
        })
            .then(function () {
            return _this.simpleDb.runTransaction('readonly', [DbTargetGlobal.store], function (txn) {
                return getHighestListenSequenceNumber(txn).next(function (highestListenSequenceNumber) {
                    var sequenceNumberSyncer = _this.multiClientParams
                        ? _this.multiClientParams.sequenceNumberSyncer
                        : undefined;
                    _this.listenSequence = new ListenSequence(highestListenSequenceNumber, sequenceNumberSyncer);
                });
            });
        })
            .then(function () {
            _this._started = true;
        })
            .catch(function (reason) {
            _this.simpleDb && _this.simpleDb.close();
            return Promise.reject(reason);
        });
    };
    IndexedDbPersistence.prototype.startRemoteDocumentCache = function () {
        var _this = this;
        return this.simpleDb.runTransaction('readonly', ALL_STORES, function (txn) {
            return _this.remoteDocumentCache.start(txn);
        });
    };
    IndexedDbPersistence.prototype.setPrimaryStateListener = function (primaryStateListener) {
        var _this = this;
        this.primaryStateListener = function (primaryState) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (this.started) {
                    return [2 /*return*/, primaryStateListener(primaryState)];
                }
                return [2 /*return*/];
            });
        }); };
        return primaryStateListener(this.isPrimary);
    };
    IndexedDbPersistence.prototype.setNetworkEnabled = function (networkEnabled) {
        var _this = this;
        if (this.networkEnabled !== networkEnabled) {
            this.networkEnabled = networkEnabled;
            // Schedule a primary lease refresh for immediate execution. The eventual
            // lease update will be propagated via `primaryStateListener`.
            this.queue.enqueueAndForget(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                return tslib_1.__generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            if (!this.started) return [3 /*break*/, 2];
                            return [4 /*yield*/, this.updateClientMetadataAndTryBecomePrimary()];
                        case 1:
                            _a.sent();
                            _a.label = 2;
                        case 2: return [2 /*return*/];
                    }
                });
            }); });
        }
    };
    /**
     * Updates the client metadata in IndexedDb and attempts to either obtain or
     * extend the primary lease for the local client. Asynchronously notifies the
     * primary state listener if the client either newly obtained or released its
     * primary lease.
     */
    IndexedDbPersistence.prototype.updateClientMetadataAndTryBecomePrimary = function () {
        var _this = this;
        return this.simpleDb.runTransaction('readwrite', ALL_STORES, function (txn) {
            var metadataStore = clientMetadataStore(txn);
            return metadataStore
                .put(new DbClientMetadata(_this.clientId, Date.now(), _this.networkEnabled, _this.inForeground, _this.remoteDocumentCache.lastProcessedDocumentChangeId))
                .next(function () {
                if (_this.isPrimary) {
                    return _this.verifyPrimaryLease(txn).next(function (success) {
                        if (!success) {
                            _this.isPrimary = false;
                            _this.queue.enqueueAndForget(function () {
                                return _this.primaryStateListener(false);
                            });
                        }
                    });
                }
            })
                .next(function () { return _this.canActAsPrimary(txn); })
                .next(function (canActAsPrimary) {
                var wasPrimary = _this.isPrimary;
                _this.isPrimary = canActAsPrimary;
                if (wasPrimary !== _this.isPrimary) {
                    _this.queue.enqueueAndForget(function () {
                        return _this.primaryStateListener(_this.isPrimary);
                    });
                }
                if (wasPrimary && !_this.isPrimary) {
                    return _this.releasePrimaryLeaseIfHeld(txn);
                }
                else if (_this.isPrimary) {
                    return _this.acquireOrExtendPrimaryLease(txn);
                }
            });
        });
    };
    IndexedDbPersistence.prototype.verifyPrimaryLease = function (txn) {
        var _this = this;
        var store = primaryClientStore(txn);
        return store.get(DbPrimaryClient.key).next(function (primaryClient) {
            return PersistencePromise.resolve(_this.isLocalClient(primaryClient));
        });
    };
    IndexedDbPersistence.prototype.removeClientMetadata = function (txn) {
        var metadataStore = clientMetadataStore(txn);
        return metadataStore.delete(this.clientId);
    };
    /**
     * If the garbage collection threshold has passed, prunes the
     * RemoteDocumentChanges and the ClientMetadata store based on the last update
     * time of all clients.
     */
    IndexedDbPersistence.prototype.maybeGarbageCollectMultiClientState = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var activeClients_1, inactiveClients_1;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!(this.isPrimary &&
                            !this.isWithinAge(this.lastGarbageCollectionTime, MAX_CLIENT_AGE_MS))) return [3 /*break*/, 2];
                        this.lastGarbageCollectionTime = Date.now();
                        inactiveClients_1 = [];
                        return [4 /*yield*/, this.runTransaction('maybeGarbageCollectMultiClientState', 'readwrite-primary', function (txn) {
                                var metadataStore = IndexedDbPersistence.getStore(txn, DbClientMetadata.store);
                                return metadataStore
                                    .loadAll()
                                    .next(function (existingClients) {
                                    activeClients_1 = _this.filterActiveClients(existingClients, MAX_CLIENT_AGE_MS);
                                    inactiveClients_1 = existingClients.filter(function (client) { return activeClients_1.indexOf(client) === -1; });
                                })
                                    .next(function () {
                                    // Delete metadata for clients that are no longer considered active.
                                    return PersistencePromise.forEach(inactiveClients_1, function (inactiveClient) {
                                        return metadataStore.delete(inactiveClient.clientId);
                                    });
                                })
                                    .next(function () {
                                    // Retrieve the minimum change ID from the set of active clients.
                                    // The primary client doesn't read from the document change log,
                                    // and hence we exclude it when we determine the minimum
                                    // `lastProcessedDocumentChangeId`.
                                    activeClients_1 = activeClients_1.filter(function (client) { return client.clientId !== _this.clientId; });
                                    if (activeClients_1.length > 0) {
                                        var processedChangeIds = activeClients_1.map(function (client) { return client.lastProcessedDocumentChangeId || 0; });
                                        var oldestChangeId = Math.min.apply(Math, processedChangeIds);
                                        return _this.remoteDocumentCache.removeDocumentChangesThroughChangeId(txn, oldestChangeId);
                                    }
                                });
                            })];
                    case 1:
                        _a.sent();
                        // Delete potential leftover entries that may continue to mark the
                        // inactive clients as zombied in LocalStorage.
                        // Ideally we'd delete the IndexedDb and LocalStorage zombie entries for
                        // the client atomically, but we can't. So we opt to delete the IndexedDb
                        // entries first to avoid potentially reviving a zombied client.
                        inactiveClients_1.forEach(function (inactiveClient) {
                            _this.window.localStorage.removeItem(_this.zombiedClientLocalStorageKey(inactiveClient.clientId));
                        });
                        _a.label = 2;
                    case 2: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Schedules a recurring timer to update the client metadata and to either
     * extend or acquire the primary lease if the client is eligible.
     */
    IndexedDbPersistence.prototype.scheduleClientMetadataAndPrimaryLeaseRefreshes = function () {
        var _this = this;
        this.clientMetadataRefresher = this.queue.enqueueAfterDelay(TimerId.ClientMetadataRefresh, CLIENT_METADATA_REFRESH_INTERVAL_MS, function () {
            return _this.updateClientMetadataAndTryBecomePrimary()
                .then(function () { return _this.maybeGarbageCollectMultiClientState(); })
                .then(function () { return _this.scheduleClientMetadataAndPrimaryLeaseRefreshes(); });
        });
    };
    /** Checks whether `client` is the local client. */
    IndexedDbPersistence.prototype.isLocalClient = function (client) {
        return client ? client.ownerId === this.clientId : false;
    };
    /**
     * Evaluate the state of all active clients and determine whether the local
     * client is or can act as the holder of the primary lease. Returns whether
     * the client is eligible for the lease, but does not actually acquire it.
     * May return 'false' even if there is no active leaseholder and another
     * (foreground) client should become leaseholder instead.
     */
    IndexedDbPersistence.prototype.canActAsPrimary = function (txn) {
        var _this = this;
        var store = primaryClientStore(txn);
        return store
            .get(DbPrimaryClient.key)
            .next(function (currentPrimary) {
            var currentLeaseIsValid = currentPrimary !== null &&
                _this.isWithinAge(currentPrimary.leaseTimestampMs, MAX_PRIMARY_ELIGIBLE_AGE_MS) &&
                !_this.isClientZombied(currentPrimary.ownerId);
            // A client is eligible for the primary lease if:
            // - its network is enabled and the client's tab is in the foreground.
            // - its network is enabled and no other client's tab is in the
            //   foreground.
            // - every clients network is disabled and the client's tab is in the
            //   foreground.
            // - every clients network is disabled and no other client's tab is in
            //   the foreground.
            if (currentLeaseIsValid) {
                if (_this.isLocalClient(currentPrimary) && _this.networkEnabled) {
                    return true;
                }
                if (!_this.isLocalClient(currentPrimary)) {
                    if (!currentPrimary.allowTabSynchronization) {
                        // Fail the `canActAsPrimary` check if the current leaseholder has
                        // not opted into multi-tab synchronization. If this happens at
                        // client startup, we reject the Promise returned by
                        // `enablePersistence()` and the user can continue to use Firestore
                        // with in-memory persistence.
                        // If this fails during a lease refresh, we will instead block the
                        // AsyncQueue from executing further operations. Note that this is
                        // acceptable since mixing & matching different `synchronizeTabs`
                        // settings is not supported.
                        //
                        // TODO(b/114226234): Remove this check when `synchronizeTabs` can
                        // no longer be turned off.
                        throw new FirestoreError(Code.FAILED_PRECONDITION, PRIMARY_LEASE_EXCLUSIVE_ERROR_MSG);
                    }
                    return false;
                }
            }
            if (_this.networkEnabled && _this.inForeground) {
                return true;
            }
            return clientMetadataStore(txn)
                .loadAll()
                .next(function (existingClients) {
                // Process all existing clients and determine whether at least one of
                // them is better suited to obtain the primary lease.
                var preferredCandidate = _this.filterActiveClients(existingClients, MAX_PRIMARY_ELIGIBLE_AGE_MS).find(function (otherClient) {
                    if (_this.clientId !== otherClient.clientId) {
                        var otherClientHasBetterNetworkState = !_this.networkEnabled && otherClient.networkEnabled;
                        var otherClientHasBetterVisibility = !_this.inForeground && otherClient.inForeground;
                        var otherClientHasSameNetworkState = _this.networkEnabled === otherClient.networkEnabled;
                        if (otherClientHasBetterNetworkState ||
                            (otherClientHasBetterVisibility &&
                                otherClientHasSameNetworkState)) {
                            return true;
                        }
                    }
                    return false;
                });
                return preferredCandidate === undefined;
            });
        })
            .next(function (canActAsPrimary) {
            if (_this.isPrimary !== canActAsPrimary) {
                debug(LOG_TAG$2, "Client " + (canActAsPrimary ? 'is' : 'is not') + " eligible for a primary lease.");
            }
            return canActAsPrimary;
        });
    };
    IndexedDbPersistence.prototype.shutdown = function (deleteData) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        // The shutdown() operations are idempotent and can be called even when
                        // start() aborted (e.g. because it couldn't acquire the persistence lease).
                        this._started = false;
                        this.markClientZombied();
                        if (this.clientMetadataRefresher) {
                            this.clientMetadataRefresher.cancel();
                        }
                        this.detachVisibilityHandler();
                        this.detachWindowUnloadHook();
                        return [4 /*yield*/, this.simpleDb.runTransaction('readwrite', [DbPrimaryClient.store, DbClientMetadata.store], function (txn) {
                                return _this.releasePrimaryLeaseIfHeld(txn).next(function () {
                                    return _this.removeClientMetadata(txn);
                                });
                            })];
                    case 1:
                        _a.sent();
                        this.simpleDb.close();
                        // Remove the entry marking the client as zombied from LocalStorage since
                        // we successfully deleted its metadata from IndexedDb.
                        this.removeClientZombiedEntry();
                        if (!deleteData) return [3 /*break*/, 3];
                        return [4 /*yield*/, SimpleDb.delete(this.dbName)];
                    case 2:
                        _a.sent();
                        _a.label = 3;
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Returns clients that are not zombied and have an updateTime within the
     * provided threshold.
     */
    IndexedDbPersistence.prototype.filterActiveClients = function (clients, activityThresholdMs) {
        var _this = this;
        return clients.filter(function (client) {
            return _this.isWithinAge(client.updateTimeMs, activityThresholdMs) &&
                !_this.isClientZombied(client.clientId);
        });
    };
    IndexedDbPersistence.prototype.getActiveClients = function () {
        var _this = this;
        return this.simpleDb.runTransaction('readonly', [DbClientMetadata.store], function (txn) {
            return clientMetadataStore(txn)
                .loadAll()
                .next(function (clients) {
                return _this.filterActiveClients(clients, MAX_CLIENT_AGE_MS).map(function (clientMetadata) { return clientMetadata.clientId; });
            });
        });
    };
    Object.defineProperty(IndexedDbPersistence.prototype, "started", {
        get: function () {
            return this._started;
        },
        enumerable: true,
        configurable: true
    });
    IndexedDbPersistence.prototype.getMutationQueue = function (user) {
        assert(this.started, 'Cannot initialize MutationQueue before persistence is started.');
        return IndexedDbMutationQueue.forUser(user, this.serializer, this.indexManager, this.referenceDelegate);
    };
    IndexedDbPersistence.prototype.getQueryCache = function () {
        assert(this.started, 'Cannot initialize QueryCache before persistence is started.');
        return this.queryCache;
    };
    IndexedDbPersistence.prototype.getRemoteDocumentCache = function () {
        assert(this.started, 'Cannot initialize RemoteDocumentCache before persistence is started.');
        return this.remoteDocumentCache;
    };
    IndexedDbPersistence.prototype.getIndexManager = function () {
        assert(this.started, 'Cannot initialize IndexManager before persistence is started.');
        return this.indexManager;
    };
    IndexedDbPersistence.prototype.runTransaction = function (action, mode, transactionOperation) {
        var _this = this;
        debug(LOG_TAG$2, 'Starting transaction:', action);
        // Do all transactions as readwrite against all object stores, since we
        // are the only reader/writer.
        return this.simpleDb.runTransaction(mode === 'readonly' ? 'readonly' : 'readwrite', ALL_STORES, function (simpleDbTxn) {
            if (mode === 'readwrite-primary') {
                // While we merely verify that we have (or can acquire) the lease
                // immediately, we wait to extend the primary lease until after
                // executing transactionOperation(). This ensures that even if the
                // transactionOperation takes a long time, we'll use a recent
                // leaseTimestampMs in the extended (or newly acquired) lease.
                return _this.verifyPrimaryLease(simpleDbTxn)
                    .next(function (success) {
                    if (!success) {
                        error("Failed to obtain primary lease for action '" + action + "'.");
                        _this.isPrimary = false;
                        _this.queue.enqueueAndForget(function () {
                            return _this.primaryStateListener(false);
                        });
                        throw new FirestoreError(Code.FAILED_PRECONDITION, PRIMARY_LEASE_LOST_ERROR_MSG);
                    }
                    return transactionOperation(new IndexedDbTransaction(simpleDbTxn, _this.listenSequence.next()));
                })
                    .next(function (result) {
                    return _this.acquireOrExtendPrimaryLease(simpleDbTxn).next(function () { return result; });
                });
            }
            else {
                return _this.verifyAllowTabSynchronization(simpleDbTxn).next(function () {
                    return transactionOperation(new IndexedDbTransaction(simpleDbTxn, _this.listenSequence.next()));
                });
            }
        });
    };
    /**
     * Verifies that the current tab is the primary leaseholder or alternatively
     * that the leaseholder has opted into multi-tab synchronization.
     */
    // TODO(b/114226234): Remove this check when `synchronizeTabs` can no longer
    // be turned off.
    IndexedDbPersistence.prototype.verifyAllowTabSynchronization = function (txn) {
        var _this = this;
        var store = primaryClientStore(txn);
        return store.get(DbPrimaryClient.key).next(function (currentPrimary) {
            var currentLeaseIsValid = currentPrimary !== null &&
                _this.isWithinAge(currentPrimary.leaseTimestampMs, MAX_PRIMARY_ELIGIBLE_AGE_MS) &&
                !_this.isClientZombied(currentPrimary.ownerId);
            if (currentLeaseIsValid && !_this.isLocalClient(currentPrimary)) {
                if (!currentPrimary.allowTabSynchronization) {
                    throw new FirestoreError(Code.FAILED_PRECONDITION, PRIMARY_LEASE_EXCLUSIVE_ERROR_MSG);
                }
            }
        });
    };
    /**
     * Obtains or extends the new primary lease for the local client. This
     * method does not verify that the client is eligible for this lease.
     */
    IndexedDbPersistence.prototype.acquireOrExtendPrimaryLease = function (txn) {
        var newPrimary = new DbPrimaryClient(this.clientId, this.allowTabSynchronization, Date.now());
        return primaryClientStore(txn).put(DbPrimaryClient.key, newPrimary);
    };
    IndexedDbPersistence.isAvailable = function () {
        return SimpleDb.isAvailable();
    };
    /**
     * Generates a string used as a prefix when storing data in IndexedDB and
     * LocalStorage.
     */
    IndexedDbPersistence.buildStoragePrefix = function (databaseInfo) {
        // Use two different prefix formats:
        //
        //   * firestore / persistenceKey / projectID . databaseID / ...
        //   * firestore / persistenceKey / projectID / ...
        //
        // projectIDs are DNS-compatible names and cannot contain dots
        // so there's no danger of collisions.
        var database = databaseInfo.databaseId.projectId;
        if (!databaseInfo.databaseId.isDefaultDatabase) {
            database += '.' + databaseInfo.databaseId.database;
        }
        return 'firestore/' + databaseInfo.persistenceKey + '/' + database + '/';
    };
    /** Checks the primary lease and removes it if we are the current primary. */
    IndexedDbPersistence.prototype.releasePrimaryLeaseIfHeld = function (txn) {
        var _this = this;
        var store = primaryClientStore(txn);
        return store.get(DbPrimaryClient.key).next(function (primaryClient) {
            if (_this.isLocalClient(primaryClient)) {
                debug(LOG_TAG$2, 'Releasing primary lease.');
                return store.delete(DbPrimaryClient.key);
            }
            else {
                return PersistencePromise.resolve();
            }
        });
    };
    /** Verifies that `updateTimeMs` is within `maxAgeMs`. */
    IndexedDbPersistence.prototype.isWithinAge = function (updateTimeMs, maxAgeMs) {
        var now = Date.now();
        var minAcceptable = now - maxAgeMs;
        var maxAcceptable = now;
        if (updateTimeMs < minAcceptable) {
            return false;
        }
        else if (updateTimeMs > maxAcceptable) {
            error("Detected an update time that is in the future: " + updateTimeMs + " > " + maxAcceptable);
            return false;
        }
        return true;
    };
    IndexedDbPersistence.prototype.attachVisibilityHandler = function () {
        var _this = this;
        if (this.document !== null &&
            typeof this.document.addEventListener === 'function') {
            this.documentVisibilityHandler = function () {
                _this.queue.enqueueAndForget(function () {
                    _this.inForeground = _this.document.visibilityState === 'visible';
                    return _this.updateClientMetadataAndTryBecomePrimary();
                });
            };
            this.document.addEventListener('visibilitychange', this.documentVisibilityHandler);
            this.inForeground = this.document.visibilityState === 'visible';
        }
    };
    IndexedDbPersistence.prototype.detachVisibilityHandler = function () {
        if (this.documentVisibilityHandler) {
            assert(this.document !== null &&
                typeof this.document.addEventListener === 'function', "Expected 'document.addEventListener' to be a function");
            this.document.removeEventListener('visibilitychange', this.documentVisibilityHandler);
            this.documentVisibilityHandler = null;
        }
    };
    /**
     * Attaches a window.unload handler that will synchronously write our
     * clientId to a "zombie client id" location in LocalStorage. This can be used
     * by tabs trying to acquire the primary lease to determine that the lease
     * is no longer valid even if the timestamp is recent. This is particularly
     * important for the refresh case (so the tab correctly re-acquires the
     * primary lease). LocalStorage is used for this rather than IndexedDb because
     * it is a synchronous API and so can be used reliably from  an unload
     * handler.
     */
    IndexedDbPersistence.prototype.attachWindowUnloadHook = function () {
        var _this = this;
        if (typeof this.window.addEventListener === 'function') {
            this.windowUnloadHandler = function () {
                // Note: In theory, this should be scheduled on the AsyncQueue since it
                // accesses internal state. We execute this code directly during shutdown
                // to make sure it gets a chance to run.
                _this.markClientZombied();
                _this.queue.enqueueAndForget(function () {
                    // Attempt graceful shutdown (including releasing our primary lease),
                    // but there's no guarantee it will complete.
                    return _this.shutdown();
                });
            };
            this.window.addEventListener('unload', this.windowUnloadHandler);
        }
    };
    IndexedDbPersistence.prototype.detachWindowUnloadHook = function () {
        if (this.windowUnloadHandler) {
            assert(typeof this.window.removeEventListener === 'function', "Expected 'window.removeEventListener' to be a function");
            this.window.removeEventListener('unload', this.windowUnloadHandler);
            this.windowUnloadHandler = null;
        }
    };
    /**
     * Returns whether a client is "zombied" based on its LocalStorage entry.
     * Clients become zombied when their tab closes without running all of the
     * cleanup logic in `shutdown()`.
     */
    IndexedDbPersistence.prototype.isClientZombied = function (clientId) {
        try {
            var isZombied = this.webStorage.getItem(this.zombiedClientLocalStorageKey(clientId)) !==
                null;
            debug(LOG_TAG$2, "Client '" + clientId + "' " + (isZombied ? 'is' : 'is not') + " zombied in LocalStorage");
            return isZombied;
        }
        catch (e) {
            // Gracefully handle if LocalStorage isn't working.
            error(LOG_TAG$2, 'Failed to get zombied client id.', e);
            return false;
        }
    };
    /**
     * Record client as zombied (a client that had its tab closed). Zombied
     * clients are ignored during primary tab selection.
     */
    IndexedDbPersistence.prototype.markClientZombied = function () {
        try {
            this.webStorage.setItem(this.zombiedClientLocalStorageKey(this.clientId), String(Date.now()));
        }
        catch (e) {
            // Gracefully handle if LocalStorage isn't available / working.
            error('Failed to set zombie client id.', e);
        }
    };
    /** Removes the zombied client entry if it exists. */
    IndexedDbPersistence.prototype.removeClientZombiedEntry = function () {
        try {
            this.webStorage.removeItem(this.zombiedClientLocalStorageKey(this.clientId));
        }
        catch (e) {
            // Ignore
        }
    };
    IndexedDbPersistence.prototype.zombiedClientLocalStorageKey = function (clientId) {
        return ZOMBIED_CLIENTS_KEY_PREFIX + "_" + this.persistenceKey + "_" + clientId;
    };
    /**
     * The name of the main (and currently only) IndexedDB database. this name is
     * appended to the prefix provided to the IndexedDbPersistence constructor.
     */
    IndexedDbPersistence.MAIN_DATABASE = 'main';
    return IndexedDbPersistence;
}());
function isPrimaryLeaseLostError(err) {
    return (err.code === Code.FAILED_PRECONDITION &&
        err.message === PRIMARY_LEASE_LOST_ERROR_MSG);
}
/**
 * Verifies the error thrown by a LocalStore operation. If a LocalStore
 * operation fails because the primary lease has been taken by another client,
 * we ignore the error (the persistence layer will immediately call
 * `applyPrimaryLease` to propagate the primary state change). All other errors
 * are re-thrown.
 *
 * @param err An error returned by a LocalStore operation.
 * @return A Promise that resolves after we recovered, or the original error.
 */
function ignoreIfPrimaryLeaseLoss(err) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        return tslib_1.__generator(this, function (_a) {
            if (isPrimaryLeaseLostError(err)) {
                debug(LOG_TAG$2, 'Unexpectedly lost primary lease');
            }
            else {
                throw err;
            }
            return [2 /*return*/];
        });
    });
}
/**
 * Helper to get a typed SimpleDbStore for the primary client object store.
 */
function primaryClientStore(txn) {
    return txn.store(DbPrimaryClient.store);
}
/**
 * Helper to get a typed SimpleDbStore for the client metadata object store.
 */
function clientMetadataStore(txn) {
    return txn.store(DbClientMetadata.store);
}
/** Provides LRU functionality for IndexedDB persistence. */
var IndexedDbLruDelegate = /** @class */ (function () {
    function IndexedDbLruDelegate(db, params) {
        this.db = db;
        this.garbageCollector = new LruGarbageCollector(this, params);
    }
    IndexedDbLruDelegate.prototype.getSequenceNumberCount = function (txn) {
        var docCountPromise = this.orphanedDocmentCount(txn);
        var targetCountPromise = this.db.getQueryCache().getQueryCount(txn);
        return targetCountPromise.next(function (targetCount) {
            return docCountPromise.next(function (docCount) { return targetCount + docCount; });
        });
    };
    IndexedDbLruDelegate.prototype.orphanedDocmentCount = function (txn) {
        var orphanedCount = 0;
        return this.forEachOrphanedDocumentSequenceNumber(txn, function (_) {
            orphanedCount++;
        }).next(function () { return orphanedCount; });
    };
    IndexedDbLruDelegate.prototype.forEachTarget = function (txn, f) {
        return this.db.getQueryCache().forEachTarget(txn, f);
    };
    IndexedDbLruDelegate.prototype.forEachOrphanedDocumentSequenceNumber = function (txn, f) {
        return this.forEachOrphanedDocument(txn, function (docKey, sequenceNumber) {
            return f(sequenceNumber);
        });
    };
    IndexedDbLruDelegate.prototype.setInMemoryPins = function (inMemoryPins) {
        this.inMemoryPins = inMemoryPins;
    };
    IndexedDbLruDelegate.prototype.addReference = function (txn, key) {
        return writeSentinelKey(txn, key);
    };
    IndexedDbLruDelegate.prototype.removeReference = function (txn, key) {
        return writeSentinelKey(txn, key);
    };
    IndexedDbLruDelegate.prototype.removeTargets = function (txn, upperBound, activeTargetIds) {
        return this.db
            .getQueryCache()
            .removeTargets(txn, upperBound, activeTargetIds);
    };
    IndexedDbLruDelegate.prototype.removeMutationReference = function (txn, key) {
        return writeSentinelKey(txn, key);
    };
    /**
     * Returns true if anything would prevent this document from being garbage
     * collected, given that the document in question is not present in any
     * targets and has a sequence number less than or equal to the upper bound for
     * the collection run.
     */
    IndexedDbLruDelegate.prototype.isPinned = function (txn, docKey) {
        if (this.inMemoryPins.containsKey(docKey)) {
            return PersistencePromise.resolve(true);
        }
        else {
            return mutationQueuesContainKey(txn, docKey);
        }
    };
    IndexedDbLruDelegate.prototype.removeOrphanedDocuments = function (txn, upperBound) {
        var _this = this;
        var count = 0;
        var bytesRemoved = 0;
        var promises = [];
        var iteration = this.forEachOrphanedDocument(txn, function (docKey, sequenceNumber) {
            if (sequenceNumber <= upperBound) {
                var p = _this.isPinned(txn, docKey).next(function (isPinned) {
                    if (!isPinned) {
                        count++;
                        return _this.removeOrphanedDocument(txn, docKey).next(function (documentBytes) {
                            bytesRemoved += documentBytes;
                        });
                    }
                });
                promises.push(p);
            }
        });
        // Wait for iteration first to make sure we have a chance to add all of the
        // removal promises to the array.
        return iteration
            .next(function () { return PersistencePromise.waitFor(promises); })
            .next(function () {
            return _this.db.getRemoteDocumentCache().updateSize(txn, -bytesRemoved);
        })
            .next(function () { return count; });
    };
    /**
     * Clears a document from the cache. The document is assumed to be orphaned, so target-document
     * associations are not queried. We remove it from the remote document cache, as well as remove
     * its sentinel row.
     */
    IndexedDbLruDelegate.prototype.removeOrphanedDocument = function (txn, docKey) {
        var totalBytesRemoved = 0;
        var documentCache = this.db.getRemoteDocumentCache();
        return PersistencePromise.waitFor([
            documentTargetStore(txn).delete(sentinelKey$1(docKey)),
            documentCache.removeEntry(txn, docKey).next(function (bytesRemoved) {
                totalBytesRemoved += bytesRemoved;
            })
        ]).next(function () { return totalBytesRemoved; });
    };
    IndexedDbLruDelegate.prototype.removeTarget = function (txn, queryData) {
        var updated = queryData.copy({
            sequenceNumber: txn.currentSequenceNumber
        });
        return this.db.getQueryCache().updateQueryData(txn, updated);
    };
    IndexedDbLruDelegate.prototype.updateLimboDocument = function (txn, key) {
        return writeSentinelKey(txn, key);
    };
    /**
     * Call provided function for each document in the cache that is 'orphaned'. Orphaned
     * means not a part of any target, so the only entry in the target-document index for
     * that document will be the sentinel row (targetId 0), which will also have the sequence
     * number for the last time the document was accessed.
     */
    IndexedDbLruDelegate.prototype.forEachOrphanedDocument = function (txn, f) {
        var store = documentTargetStore(txn);
        var nextToReport = ListenSequence.INVALID;
        var nextPath;
        return store
            .iterate({
            index: DbTargetDocument.documentTargetsIndex
        }, function (_a, _b) {
            var targetId = _a[0], docKey = _a[1];
            var path = _b.path, sequenceNumber = _b.sequenceNumber;
            if (targetId === 0) {
                // if nextToReport is valid, report it, this is a new key so the
                // last one must not be a member of any targets.
                if (nextToReport !== ListenSequence.INVALID) {
                    f(new DocumentKey(decode(nextPath)), nextToReport);
                }
                // set nextToReport to be this sequence number. It's the next one we
                // might report, if we don't find any targets for this document.
                // Note that the sequence number must be defined when the targetId
                // is 0.
                nextToReport = sequenceNumber;
                nextPath = path;
            }
            else {
                // set nextToReport to be invalid, we know we don't need to report
                // this one since we found a target for it.
                nextToReport = ListenSequence.INVALID;
            }
        })
            .next(function () {
            // Since we report sequence numbers after getting to the next key, we
            // need to check if the last key we iterated over was an orphaned
            // document and report it.
            if (nextToReport !== ListenSequence.INVALID) {
                f(new DocumentKey(decode(nextPath)), nextToReport);
            }
        });
    };
    IndexedDbLruDelegate.prototype.getCacheSize = function (txn) {
        return this.db.getRemoteDocumentCache().getSize(txn);
    };
    return IndexedDbLruDelegate;
}());
function sentinelKey$1(key) {
    return [0, encode(key.path)];
}
/**
 * @return A value suitable for writing a sentinel row in the target-document
 * store.
 */
function sentinelRow(key, sequenceNumber) {
    return new DbTargetDocument(0, encode(key.path), sequenceNumber);
}
function writeSentinelKey(txn, key) {
    return documentTargetStore(txn).put(sentinelRow(key, txn.currentSequenceNumber));
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A readonly view of the local state of all documents we're tracking (i.e. we
 * have a cached version in remoteDocumentCache or local mutations for the
 * document). The view is computed by applying the mutations in the
 * MutationQueue to the RemoteDocumentCache.
 */
var LocalDocumentsView = /** @class */ (function () {
    function LocalDocumentsView(remoteDocumentCache, mutationQueue, indexManager) {
        this.remoteDocumentCache = remoteDocumentCache;
        this.mutationQueue = mutationQueue;
        this.indexManager = indexManager;
    }
    /**
     * Get the local view of the document identified by `key`.
     *
     * @return Local view of the document or null if we don't have any cached
     * state for it.
     */
    LocalDocumentsView.prototype.getDocument = function (transaction, key) {
        var _this = this;
        return this.mutationQueue
            .getAllMutationBatchesAffectingDocumentKey(transaction, key)
            .next(function (batches) { return _this.getDocumentInternal(transaction, key, batches); });
    };
    /** Internal version of `getDocument` that allows reusing batches. */
    LocalDocumentsView.prototype.getDocumentInternal = function (transaction, key, inBatches) {
        return this.remoteDocumentCache.getEntry(transaction, key).next(function (doc) {
            for (var _i = 0, inBatches_1 = inBatches; _i < inBatches_1.length; _i++) {
                var batch = inBatches_1[_i];
                doc = batch.applyToLocalView(key, doc);
            }
            return doc;
        });
    };
    // Returns the view of the given `docs` as they would appear after applying
    // all mutations in the given `batches`.
    LocalDocumentsView.prototype.applyLocalMutationsToDocuments = function (transaction, docs, batches) {
        var results = nullableMaybeDocumentMap();
        docs.forEach(function (key, localView) {
            for (var _i = 0, batches_1 = batches; _i < batches_1.length; _i++) {
                var batch = batches_1[_i];
                localView = batch.applyToLocalView(key, localView);
            }
            results = results.insert(key, localView);
        });
        return results;
    };
    /**
     * Gets the local view of the documents identified by `keys`.
     *
     * If we don't have cached state for a document in `keys`, a NoDocument will
     * be stored for that key in the resulting set.
     */
    LocalDocumentsView.prototype.getDocuments = function (transaction, keys) {
        var _this = this;
        return this.remoteDocumentCache
            .getEntries(transaction, keys)
            .next(function (docs) { return _this.getLocalViewOfDocuments(transaction, docs); });
    };
    /**
     * Similar to `getDocuments`, but creates the local view from the given
     * `baseDocs` without retrieving documents from the local store.
     */
    LocalDocumentsView.prototype.getLocalViewOfDocuments = function (transaction, baseDocs) {
        var _this = this;
        return this.mutationQueue
            .getAllMutationBatchesAffectingDocumentKeys(transaction, baseDocs)
            .next(function (batches) {
            var docs = _this.applyLocalMutationsToDocuments(transaction, baseDocs, batches);
            var results = maybeDocumentMap();
            docs.forEach(function (key, maybeDoc) {
                // TODO(http://b/32275378): Don't conflate missing / deleted.
                if (!maybeDoc) {
                    maybeDoc = new NoDocument(key, SnapshotVersion.forDeletedDoc());
                }
                results = results.insert(key, maybeDoc);
            });
            return results;
        });
    };
    /** Performs a query against the local view of all documents. */
    LocalDocumentsView.prototype.getDocumentsMatchingQuery = function (transaction, query) {
        if (query.isDocumentQuery()) {
            return this.getDocumentsMatchingDocumentQuery(transaction, query.path);
        }
        else if (query.isCollectionGroupQuery()) {
            return this.getDocumentsMatchingCollectionGroupQuery(transaction, query);
        }
        else {
            return this.getDocumentsMatchingCollectionQuery(transaction, query);
        }
    };
    LocalDocumentsView.prototype.getDocumentsMatchingDocumentQuery = function (transaction, docPath) {
        // Just do a simple document lookup.
        return this.getDocument(transaction, new DocumentKey(docPath)).next(function (maybeDoc) {
            var result = documentMap();
            if (maybeDoc instanceof Document) {
                result = result.insert(maybeDoc.key, maybeDoc);
            }
            return result;
        });
    };
    LocalDocumentsView.prototype.getDocumentsMatchingCollectionGroupQuery = function (transaction, query) {
        var _this = this;
        assert(query.path.isEmpty(), 'Currently we only support collection group queries at the root.');
        var collectionId = query.collectionGroup;
        var results = documentMap();
        return this.indexManager
            .getCollectionParents(transaction, collectionId)
            .next(function (parents) {
            // Perform a collection query against each parent that contains the
            // collectionId and aggregate the results.
            return PersistencePromise.forEach(parents, function (parent) {
                var collectionQuery = query.asCollectionQueryAtPath(parent.child(collectionId));
                return _this.getDocumentsMatchingCollectionQuery(transaction, collectionQuery).next(function (r) {
                    r.forEach(function (key, doc) {
                        results = results.insert(key, doc);
                    });
                });
            }).next(function () { return results; });
        });
    };
    LocalDocumentsView.prototype.getDocumentsMatchingCollectionQuery = function (transaction, query) {
        var _this = this;
        // Query the remote documents and overlay mutations.
        var results;
        return this.remoteDocumentCache
            .getDocumentsMatchingQuery(transaction, query)
            .next(function (queryResults) {
            results = queryResults;
            return _this.mutationQueue.getAllMutationBatchesAffectingQuery(transaction, query);
        })
            .next(function (matchingMutationBatches) {
            for (var _i = 0, matchingMutationBatches_1 = matchingMutationBatches; _i < matchingMutationBatches_1.length; _i++) {
                var batch = matchingMutationBatches_1[_i];
                for (var _a = 0, _b = batch.mutations; _a < _b.length; _a++) {
                    var mutation = _b[_a];
                    var key = mutation.key;
                    // Only process documents belonging to the collection.
                    if (!query.path.isImmediateParentOf(key.path)) {
                        continue;
                    }
                    var baseDoc = results.get(key);
                    var mutatedDoc = mutation.applyToLocalView(baseDoc, baseDoc, batch.localWriteTime);
                    if (mutatedDoc instanceof Document) {
                        results = results.insert(key, mutatedDoc);
                    }
                    else {
                        results = results.remove(key);
                    }
                }
            }
        })
            .next(function () {
            // Finally, filter out any documents that don't actually match
            // the query.
            results.forEach(function (key, doc) {
                if (!query.matches(doc)) {
                    results = results.remove(key);
                }
            });
            return results;
        });
    };
    return LocalDocumentsView;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A collection of references to a document from some kind of numbered entity
 * (either a target ID or batch ID). As references are added to or removed from
 * the set corresponding events are emitted to a registered garbage collector.
 *
 * Each reference is represented by a DocumentReference object. Each of them
 * contains enough information to uniquely identify the reference. They are all
 * stored primarily in a set sorted by key. A document is considered garbage if
 * there's no references in that set (this can be efficiently checked thanks to
 * sorting by key).
 *
 * ReferenceSet also keeps a secondary set that contains references sorted by
 * IDs. This one is used to efficiently implement removal of all references by
 * some target ID.
 */
var ReferenceSet = /** @class */ (function () {
    function ReferenceSet() {
        // A set of outstanding references to a document sorted by key.
        this.refsByKey = new SortedSet(DocReference.compareByKey);
        // A set of outstanding references to a document sorted by target id.
        this.refsByTarget = new SortedSet(DocReference.compareByTargetId);
    }
    /** Returns true if the reference set contains no references. */
    ReferenceSet.prototype.isEmpty = function () {
        return this.refsByKey.isEmpty();
    };
    /** Adds a reference to the given document key for the given ID. */
    ReferenceSet.prototype.addReference = function (key, id) {
        var ref = new DocReference(key, id);
        this.refsByKey = this.refsByKey.add(ref);
        this.refsByTarget = this.refsByTarget.add(ref);
    };
    /** Add references to the given document keys for the given ID. */
    ReferenceSet.prototype.addReferences = function (keys, id) {
        var _this = this;
        keys.forEach(function (key) { return _this.addReference(key, id); });
    };
    /**
     * Removes a reference to the given document key for the given
     * ID.
     */
    ReferenceSet.prototype.removeReference = function (key, id) {
        this.removeRef(new DocReference(key, id));
    };
    ReferenceSet.prototype.removeReferences = function (keys, id) {
        var _this = this;
        keys.forEach(function (key) { return _this.removeReference(key, id); });
    };
    /**
     * Clears all references with a given ID. Calls removeRef() for each key
     * removed.
     */
    ReferenceSet.prototype.removeReferencesForId = function (id) {
        var _this = this;
        var emptyKey = DocumentKey.EMPTY;
        var startRef = new DocReference(emptyKey, id);
        var endRef = new DocReference(emptyKey, id + 1);
        var keys = [];
        this.refsByTarget.forEachInRange([startRef, endRef], function (ref) {
            _this.removeRef(ref);
            keys.push(ref.key);
        });
        return keys;
    };
    ReferenceSet.prototype.removeAllReferences = function () {
        var _this = this;
        this.refsByKey.forEach(function (ref) { return _this.removeRef(ref); });
    };
    ReferenceSet.prototype.removeRef = function (ref) {
        this.refsByKey = this.refsByKey.delete(ref);
        this.refsByTarget = this.refsByTarget.delete(ref);
    };
    ReferenceSet.prototype.referencesForId = function (id) {
        var emptyKey = DocumentKey.EMPTY;
        var startRef = new DocReference(emptyKey, id);
        var endRef = new DocReference(emptyKey, id + 1);
        var keys = documentKeySet();
        this.refsByTarget.forEachInRange([startRef, endRef], function (ref) {
            keys = keys.add(ref.key);
        });
        return keys;
    };
    ReferenceSet.prototype.containsKey = function (key) {
        var ref = new DocReference(key, 0);
        var firstRef = this.refsByKey.firstAfterOrEqual(ref);
        return firstRef !== null && key.isEqual(firstRef.key);
    };
    return ReferenceSet;
}());
var DocReference = /** @class */ (function () {
    function DocReference(key, targetOrBatchId) {
        this.key = key;
        this.targetOrBatchId = targetOrBatchId;
    }
    /** Compare by key then by ID */
    DocReference.compareByKey = function (left, right) {
        return (DocumentKey.comparator(left.key, right.key) ||
            primitiveComparator(left.targetOrBatchId, right.targetOrBatchId));
    };
    /** Compare by ID then by key */
    DocReference.compareByTargetId = function (left, right) {
        return (primitiveComparator(left.targetOrBatchId, right.targetOrBatchId) ||
            DocumentKey.comparator(left.key, right.key));
    };
    return DocReference;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$3 = 'LocalStore';
/**
 * Local storage in the Firestore client. Coordinates persistence components
 * like the mutation queue and remote document cache to present a
 * latency-compensated view of stored data.
 *
 * The LocalStore is responsible for accepting mutations from the Sync Engine.
 * Writes from the client are put into a queue as provisional Mutations until
 * they are processed by the RemoteStore and confirmed as having been written
 * to the server.
 *
 * The local store provides the local version of documents that have been
 * modified locally. It maintains the constraint:
 *
 *   LocalDocument = RemoteDocument + Active(LocalMutations)
 *
 * (Active mutations are those that are enqueued and have not been previously
 * acknowledged or rejected).
 *
 * The RemoteDocument ("ground truth") state is provided via the
 * applyChangeBatch method. It will be some version of a server-provided
 * document OR will be a server-provided document PLUS acknowledged mutations:
 *
 *   RemoteDocument' = RemoteDocument + Acknowledged(LocalMutations)
 *
 * Note that this "dirty" version of a RemoteDocument will not be identical to a
 * server base version, since it has LocalMutations added to it pending getting
 * an authoritative copy from the server.
 *
 * Since LocalMutations can be rejected by the server, we have to be able to
 * revert a LocalMutation that has already been applied to the LocalDocument
 * (typically done by replaying all remaining LocalMutations to the
 * RemoteDocument to re-apply).
 *
 * The LocalStore is responsible for the garbage collection of the documents it
 * contains. For now, it every doc referenced by a view, the mutation queue, or
 * the RemoteStore.
 *
 * It also maintains the persistence of mapping queries to resume tokens and
 * target ids. It needs to know this data about queries to properly know what
 * docs it would be allowed to garbage collect.
 *
 * The LocalStore must be able to efficiently execute queries against its local
 * cache of the documents, to provide the initial set of results before any
 * remote changes have been received.
 *
 * Note: In TypeScript, most methods return Promises since the implementation
 * may rely on fetching data from IndexedDB which is async.
 * These Promises will only be rejected on an I/O error or other internal
 * (unexpected) failure (e.g. failed assert) and always represent an
 * unrecoverable error (should be caught / reported by the async_queue).
 */
var LocalStore = /** @class */ (function () {
    function LocalStore(
    /** Manages our in-memory or durable persistence. */
    persistence, initialUser) {
        this.persistence = persistence;
        /**
         * The set of document references maintained by any local views.
         */
        this.localViewReferences = new ReferenceSet();
        /** Maps a targetID to data about its query. */
        this.queryDataByTarget = {};
        assert(persistence.started, 'LocalStore was passed an unstarted persistence implementation');
        this.persistence.referenceDelegate.setInMemoryPins(this.localViewReferences);
        this.mutationQueue = persistence.getMutationQueue(initialUser);
        this.remoteDocuments = persistence.getRemoteDocumentCache();
        this.queryCache = persistence.getQueryCache();
        this.localDocuments = new LocalDocumentsView(this.remoteDocuments, this.mutationQueue, this.persistence.getIndexManager());
    }
    /**
     * Tells the LocalStore that the currently authenticated user has changed.
     *
     * In response the local store switches the mutation queue to the new user and
     * returns any resulting document changes.
     */
    // PORTING NOTE: Android and iOS only return the documents affected by the
    // change.
    LocalStore.prototype.handleUserChange = function (user) {
        var _this = this;
        return this.persistence.runTransaction('Handle user change', 'readonly', function (txn) {
            // Swap out the mutation queue, grabbing the pending mutation batches
            // before and after.
            var oldBatches;
            return _this.mutationQueue
                .getAllMutationBatches(txn)
                .next(function (promisedOldBatches) {
                oldBatches = promisedOldBatches;
                _this.mutationQueue = _this.persistence.getMutationQueue(user);
                // Recreate our LocalDocumentsView using the new
                // MutationQueue.
                _this.localDocuments = new LocalDocumentsView(_this.remoteDocuments, _this.mutationQueue, _this.persistence.getIndexManager());
                return _this.mutationQueue.getAllMutationBatches(txn);
            })
                .next(function (newBatches) {
                var removedBatchIds = [];
                var addedBatchIds = [];
                // Union the old/new changed keys.
                var changedKeys = documentKeySet();
                for (var _i = 0, oldBatches_1 = oldBatches; _i < oldBatches_1.length; _i++) {
                    var batch = oldBatches_1[_i];
                    removedBatchIds.push(batch.batchId);
                    for (var _a = 0, _b = batch.mutations; _a < _b.length; _a++) {
                        var mutation = _b[_a];
                        changedKeys = changedKeys.add(mutation.key);
                    }
                }
                for (var _c = 0, newBatches_1 = newBatches; _c < newBatches_1.length; _c++) {
                    var batch = newBatches_1[_c];
                    addedBatchIds.push(batch.batchId);
                    for (var _d = 0, _e = batch.mutations; _d < _e.length; _d++) {
                        var mutation = _e[_d];
                        changedKeys = changedKeys.add(mutation.key);
                    }
                }
                // Return the set of all (potentially) changed documents and the list
                // of mutation batch IDs that were affected by change.
                return _this.localDocuments
                    .getDocuments(txn, changedKeys)
                    .next(function (affectedDocuments) {
                    return {
                        affectedDocuments: affectedDocuments,
                        removedBatchIds: removedBatchIds,
                        addedBatchIds: addedBatchIds
                    };
                });
            });
        });
    };
    /* Accept locally generated Mutations and commit them to storage. */
    LocalStore.prototype.localWrite = function (mutations) {
        var _this = this;
        var localWriteTime = Timestamp.now();
        var keys = mutations.reduce(function (keys, m) { return keys.add(m.key); }, documentKeySet());
        return this.persistence.runTransaction('Locally write mutations', 'readwrite', function (txn) {
            // Load and apply all existing mutations. This lets us compute the
            // current base state for all non-idempotent transforms before applying
            // any additional user-provided writes.
            return _this.localDocuments
                .getDocuments(txn, keys)
                .next(function (existingDocs) {
                // For non-idempotent mutations (such as `FieldValue.increment()`),
                // we record the base state in a separate patch mutation. This is
                // later used to guarantee consistent values and prevents flicker
                // even if the backend sends us an update that already includes our
                // transform.
                var baseMutations = [];
                for (var _i = 0, mutations_1 = mutations; _i < mutations_1.length; _i++) {
                    var mutation = mutations_1[_i];
                    var maybeDoc = existingDocs.get(mutation.key);
                    if (!mutation.isIdempotent) {
                        // Theoretically, we should only include non-idempotent fields
                        // in this field mask as this mask is used to populate the base
                        // state for all DocumentTransforms.  By  including all fields,
                        // we incorrectly prevent rebasing of idempotent transforms
                        // (such as `arrayUnion()`) when any non-idempotent transforms
                        // are present.
                        var fieldMask = mutation.fieldMask;
                        if (fieldMask) {
                            var baseValues = maybeDoc instanceof Document
                                ? fieldMask.applyTo(maybeDoc.data)
                                : ObjectValue.EMPTY;
                            // NOTE: The base state should only be applied if there's some
                            // existing document to override, so use a Precondition of
                            // exists=true
                            baseMutations.push(new PatchMutation(mutation.key, baseValues, fieldMask, Precondition.exists(true)));
                        }
                    }
                }
                return _this.mutationQueue
                    .addMutationBatch(txn, localWriteTime, baseMutations, mutations)
                    .next(function (batch) {
                    var changes = batch.applyToLocalDocumentSet(existingDocs);
                    return { batchId: batch.batchId, changes: changes };
                });
            });
        });
    };
    /** Returns the local view of the documents affected by a mutation batch. */
    // PORTING NOTE: Multi-tab only.
    LocalStore.prototype.lookupMutationDocuments = function (batchId) {
        var _this = this;
        return this.persistence.runTransaction('Lookup mutation documents', 'readonly', function (txn) {
            return _this.mutationQueue
                .lookupMutationKeys(txn, batchId)
                .next(function (keys) {
                if (keys) {
                    return _this.localDocuments.getDocuments(txn, keys);
                }
                else {
                    return PersistencePromise.resolve(null);
                }
            });
        });
    };
    /**
     * Acknowledge the given batch.
     *
     * On the happy path when a batch is acknowledged, the local store will
     *
     *  + remove the batch from the mutation queue;
     *  + apply the changes to the remote document cache;
     *  + recalculate the latency compensated view implied by those changes (there
     *    may be mutations in the queue that affect the documents but haven't been
     *    acknowledged yet); and
     *  + give the changed documents back the sync engine
     *
     * @returns The resulting (modified) documents.
     */
    LocalStore.prototype.acknowledgeBatch = function (batchResult) {
        var _this = this;
        return this.persistence.runTransaction('Acknowledge batch', 'readwrite-primary', function (txn) {
            var affected = batchResult.batch.keys();
            var documentBuffer = _this.remoteDocuments.newChangeBuffer();
            return _this.mutationQueue
                .acknowledgeBatch(txn, batchResult.batch, batchResult.streamToken)
                .next(function () {
                return _this.applyWriteToRemoteDocuments(txn, batchResult, documentBuffer);
            })
                .next(function () { return documentBuffer.apply(txn); })
                .next(function () { return _this.mutationQueue.performConsistencyCheck(txn); })
                .next(function () { return _this.localDocuments.getDocuments(txn, affected); });
        });
    };
    /**
     * Remove mutations from the MutationQueue for the specified batch;
     * LocalDocuments will be recalculated.
     *
     * @returns The resulting modified documents.
     */
    LocalStore.prototype.rejectBatch = function (batchId) {
        var _this = this;
        return this.persistence.runTransaction('Reject batch', 'readwrite-primary', function (txn) {
            var affectedKeys;
            return _this.mutationQueue
                .lookupMutationBatch(txn, batchId)
                .next(function (batch) {
                assert(batch !== null, 'Attempt to reject nonexistent batch!');
                affectedKeys = batch.keys();
                return _this.mutationQueue.removeMutationBatch(txn, batch);
            })
                .next(function () {
                return _this.mutationQueue.performConsistencyCheck(txn);
            })
                .next(function () {
                return _this.localDocuments.getDocuments(txn, affectedKeys);
            });
        });
    };
    /** Returns the last recorded stream token for the current user. */
    LocalStore.prototype.getLastStreamToken = function () {
        var _this = this;
        return this.persistence.runTransaction('Get last stream token', 'readonly', function (txn) {
            return _this.mutationQueue.getLastStreamToken(txn);
        });
    };
    /**
     * Sets the stream token for the current user without acknowledging any
     * mutation batch. This is usually only useful after a stream handshake or in
     * response to an error that requires clearing the stream token.
     */
    LocalStore.prototype.setLastStreamToken = function (streamToken) {
        var _this = this;
        return this.persistence.runTransaction('Set last stream token', 'readwrite-primary', function (txn) {
            return _this.mutationQueue.setLastStreamToken(txn, streamToken);
        });
    };
    /**
     * Returns the last consistent snapshot processed (used by the RemoteStore to
     * determine whether to buffer incoming snapshots from the backend).
     */
    LocalStore.prototype.getLastRemoteSnapshotVersion = function () {
        var _this = this;
        return this.persistence.runTransaction('Get last remote snapshot version', 'readonly', function (txn) { return _this.queryCache.getLastRemoteSnapshotVersion(txn); });
    };
    /**
     * Update the "ground-state" (remote) documents. We assume that the remote
     * event reflects any write batches that have been acknowledged or rejected
     * (i.e. we do not re-apply local mutations to updates from this event).
     *
     * LocalDocuments are re-calculated if there are remaining mutations in the
     * queue.
     */
    LocalStore.prototype.applyRemoteEvent = function (remoteEvent) {
        var _this = this;
        var documentBuffer = this.remoteDocuments.newChangeBuffer();
        return this.persistence.runTransaction('Apply remote event', 'readwrite-primary', function (txn) {
            var promises = [];
            var authoritativeUpdates = documentKeySet();
            forEachNumber(remoteEvent.targetChanges, function (targetId, change) {
                // Do not ref/unref unassigned targetIds - it may lead to leaks.
                var queryData = _this.queryDataByTarget[targetId];
                if (!queryData)
                    return;
                // When a global snapshot contains updates (either add or modify) we
                // can completely trust these updates as authoritative and blindly
                // apply them to our cache (as a defensive measure to promote
                // self-healing in the unfortunate case that our cache is ever somehow
                // corrupted / out-of-sync).
                //
                // If the document is only updated while removing it from a target
                // then watch isn't obligated to send the absolute latest version: it
                // can send the first version that caused the document not to match.
                change.addedDocuments.forEach(function (key) {
                    authoritativeUpdates = authoritativeUpdates.add(key);
                });
                change.modifiedDocuments.forEach(function (key) {
                    authoritativeUpdates = authoritativeUpdates.add(key);
                });
                promises.push(_this.queryCache
                    .removeMatchingKeys(txn, change.removedDocuments, targetId)
                    .next(function () {
                    return _this.queryCache.addMatchingKeys(txn, change.addedDocuments, targetId);
                }));
                // Update the resume token if the change includes one. Don't clear
                // any preexisting value.
                var resumeToken = change.resumeToken;
                if (resumeToken.length > 0) {
                    var oldQueryData = queryData;
                    queryData = queryData.copy({
                        resumeToken: resumeToken,
                        snapshotVersion: remoteEvent.snapshotVersion
                    });
                    _this.queryDataByTarget[targetId] = queryData;
                    if (LocalStore.shouldPersistQueryData(oldQueryData, queryData, change)) {
                        promises.push(_this.queryCache.updateQueryData(txn, queryData));
                    }
                }
            });
            var changedDocs = maybeDocumentMap();
            var updatedKeys = documentKeySet();
            remoteEvent.documentUpdates.forEach(function (key, doc) {
                updatedKeys = updatedKeys.add(key);
            });
            // Each loop iteration only affects its "own" doc, so it's safe to get all the remote
            // documents in advance in a single call.
            promises.push(documentBuffer.getEntries(txn, updatedKeys).next(function (existingDocs) {
                remoteEvent.documentUpdates.forEach(function (key, doc) {
                    var existingDoc = existingDocs.get(key);
                    // If a document update isn't authoritative, make sure we don't
                    // apply an old document version to the remote cache. We make an
                    // exception for SnapshotVersion.MIN which can happen for
                    // manufactured events (e.g. in the case of a limbo document
                    // resolution failing).
                    if (existingDoc == null ||
                        doc.version.isEqual(SnapshotVersion.MIN) ||
                        (authoritativeUpdates.has(doc.key) &&
                            !existingDoc.hasPendingWrites) ||
                        doc.version.compareTo(existingDoc.version) >= 0) {
                        documentBuffer.addEntry(doc);
                        changedDocs = changedDocs.insert(key, doc);
                    }
                    else {
                        debug(LOG_TAG$3, 'Ignoring outdated watch update for ', key, '. Current version:', existingDoc.version, ' Watch version:', doc.version);
                    }
                    if (remoteEvent.resolvedLimboDocuments.has(key)) {
                        promises.push(_this.persistence.referenceDelegate.updateLimboDocument(txn, key));
                    }
                });
            }));
            // HACK: The only reason we allow a null snapshot version is so that we
            // can synthesize remote events when we get permission denied errors while
            // trying to resolve the state of a locally cached document that is in
            // limbo.
            var remoteVersion = remoteEvent.snapshotVersion;
            if (!remoteVersion.isEqual(SnapshotVersion.MIN)) {
                var updateRemoteVersion = _this.queryCache
                    .getLastRemoteSnapshotVersion(txn)
                    .next(function (lastRemoteVersion) {
                    assert(remoteVersion.compareTo(lastRemoteVersion) >= 0, 'Watch stream reverted to previous snapshot?? ' +
                        remoteVersion +
                        ' < ' +
                        lastRemoteVersion);
                    return _this.queryCache.setTargetsMetadata(txn, txn.currentSequenceNumber, remoteVersion);
                });
                promises.push(updateRemoteVersion);
            }
            return PersistencePromise.waitFor(promises)
                .next(function () { return documentBuffer.apply(txn); })
                .next(function () {
                return _this.localDocuments.getLocalViewOfDocuments(txn, changedDocs);
            });
        });
    };
    /**
     * Returns true if the newQueryData should be persisted during an update of
     * an active target. QueryData should always be persisted when a target is
     * being released and should not call this function.
     *
     * While the target is active, QueryData updates can be omitted when nothing
     * about the target has changed except metadata like the resume token or
     * snapshot version. Occasionally it's worth the extra write to prevent these
     * values from getting too stale after a crash, but this doesn't have to be
     * too frequent.
     */
    LocalStore.shouldPersistQueryData = function (oldQueryData, newQueryData, change) {
        // Avoid clearing any existing value
        if (newQueryData.resumeToken.length === 0)
            return false;
        // Any resume token is interesting if there isn't one already.
        if (oldQueryData.resumeToken.length === 0)
            return true;
        // Don't allow resume token changes to be buffered indefinitely. This
        // allows us to be reasonably up-to-date after a crash and avoids needing
        // to loop over all active queries on shutdown. Especially in the browser
        // we may not get time to do anything interesting while the current tab is
        // closing.
        var timeDelta = newQueryData.snapshotVersion.toMicroseconds() -
            oldQueryData.snapshotVersion.toMicroseconds();
        if (timeDelta >= this.RESUME_TOKEN_MAX_AGE_MICROS)
            return true;
        // Otherwise if the only thing that has changed about a target is its resume
        // token it's not worth persisting. Note that the RemoteStore keeps an
        // in-memory view of the currently active targets which includes the current
        // resume token, so stream failure or user changes will still use an
        // up-to-date resume token regardless of what we do here.
        var changes = change.addedDocuments.size +
            change.modifiedDocuments.size +
            change.removedDocuments.size;
        return changes > 0;
    };
    /**
     * Notify local store of the changed views to locally pin documents.
     */
    LocalStore.prototype.notifyLocalViewChanges = function (viewChanges) {
        var _this = this;
        return this.persistence.runTransaction('notifyLocalViewChanges', 'readwrite', function (txn) {
            return PersistencePromise.forEach(viewChanges, function (viewChange) {
                _this.localViewReferences.addReferences(viewChange.addedKeys, viewChange.targetId);
                _this.localViewReferences.removeReferences(viewChange.removedKeys, viewChange.targetId);
                return PersistencePromise.forEach(viewChange.removedKeys, function (key) {
                    return _this.persistence.referenceDelegate.removeReference(txn, key);
                });
            });
        });
    };
    /**
     * Gets the mutation batch after the passed in batchId in the mutation queue
     * or null if empty.
     * @param afterBatchId If provided, the batch to search after.
     * @returns The next mutation or null if there wasn't one.
     */
    LocalStore.prototype.nextMutationBatch = function (afterBatchId) {
        var _this = this;
        return this.persistence.runTransaction('Get next mutation batch', 'readonly', function (txn) {
            if (afterBatchId === undefined) {
                afterBatchId = BATCHID_UNKNOWN;
            }
            return _this.mutationQueue.getNextMutationBatchAfterBatchId(txn, afterBatchId);
        });
    };
    /**
     * Read the current value of a Document with a given key or null if not
     * found - used for testing.
     */
    LocalStore.prototype.readDocument = function (key) {
        var _this = this;
        return this.persistence.runTransaction('read document', 'readonly', function (txn) {
            return _this.localDocuments.getDocument(txn, key);
        });
    };
    /**
     * Assigns the given query an internal ID so that its results can be pinned so
     * they don't get GC'd. A query must be allocated in the local store before
     * the store can be used to manage its view.
     */
    LocalStore.prototype.allocateQuery = function (query) {
        var _this = this;
        return this.persistence.runTransaction('Allocate query', 'readwrite', function (txn) {
            var queryData;
            return _this.queryCache
                .getQueryData(txn, query)
                .next(function (cached) {
                if (cached) {
                    // This query has been listened to previously, so reuse the
                    // previous targetID.
                    // TODO(mcg): freshen last accessed date?
                    queryData = cached;
                    return PersistencePromise.resolve();
                }
                else {
                    return _this.queryCache.allocateTargetId(txn).next(function (targetId) {
                        queryData = new QueryData(query, targetId, QueryPurpose.Listen, txn.currentSequenceNumber);
                        return _this.queryCache.addQueryData(txn, queryData);
                    });
                }
            })
                .next(function () {
                assert(!_this.queryDataByTarget[queryData.targetId], 'Tried to allocate an already allocated query: ' + query);
                _this.queryDataByTarget[queryData.targetId] = queryData;
                return queryData;
            });
        });
    };
    /**
     * Unpin all the documents associated with the given query. If
     * `keepPersistedQueryData` is set to false and Eager GC enabled, the method
     * directly removes the associated query data from the query cache.
     */
    // PORTING NOTE: `keepPersistedQueryData` is multi-tab only.
    LocalStore.prototype.releaseQuery = function (query, keepPersistedQueryData) {
        var _this = this;
        var mode = keepPersistedQueryData ? 'readwrite' : 'readwrite-primary';
        return this.persistence.runTransaction('Release query', mode, function (txn) {
            return _this.queryCache
                .getQueryData(txn, query)
                .next(function (queryData) {
                assert(queryData != null, 'Tried to release nonexistent query: ' + query);
                var targetId = queryData.targetId;
                var cachedQueryData = _this.queryDataByTarget[targetId];
                // References for documents sent via Watch are automatically removed when we delete a
                // query's target data from the reference delegate. Since this does not remove references
                // for locally mutated documents, we have to remove the target associations for these
                // documents manually.
                var removed = _this.localViewReferences.removeReferencesForId(targetId);
                delete _this.queryDataByTarget[targetId];
                if (!keepPersistedQueryData) {
                    return PersistencePromise.forEach(removed, function (key) {
                        return _this.persistence.referenceDelegate.removeReference(txn, key);
                    }).next(function () {
                        return _this.persistence.referenceDelegate.removeTarget(txn, cachedQueryData);
                    });
                }
                else {
                    return PersistencePromise.resolve();
                }
            });
        });
    };
    /**
     * Runs the specified query against all the documents in the local store and
     * returns the results.
     */
    LocalStore.prototype.executeQuery = function (query) {
        var _this = this;
        return this.persistence.runTransaction('Execute query', 'readonly', function (txn) {
            return _this.localDocuments.getDocumentsMatchingQuery(txn, query);
        });
    };
    /**
     * Returns the keys of the documents that are associated with the given
     * target id in the remote table.
     */
    LocalStore.prototype.remoteDocumentKeys = function (targetId) {
        var _this = this;
        return this.persistence.runTransaction('Remote document keys', 'readonly', function (txn) {
            return _this.queryCache.getMatchingKeysForTargetId(txn, targetId);
        });
    };
    // PORTING NOTE: Multi-tab only.
    LocalStore.prototype.getActiveClients = function () {
        return this.persistence.getActiveClients();
    };
    // PORTING NOTE: Multi-tab only.
    LocalStore.prototype.removeCachedMutationBatchMetadata = function (batchId) {
        this.mutationQueue.removeCachedMutationKeys(batchId);
    };
    // PORTING NOTE: Multi-tab only.
    LocalStore.prototype.setNetworkEnabled = function (networkEnabled) {
        this.persistence.setNetworkEnabled(networkEnabled);
    };
    LocalStore.prototype.applyWriteToRemoteDocuments = function (txn, batchResult, documentBuffer) {
        var _this = this;
        var batch = batchResult.batch;
        var docKeys = batch.keys();
        var promiseChain = PersistencePromise.resolve();
        docKeys.forEach(function (docKey) {
            promiseChain = promiseChain
                .next(function () {
                return documentBuffer.getEntry(txn, docKey);
            })
                .next(function (remoteDoc) {
                var doc = remoteDoc;
                var ackVersion = batchResult.docVersions.get(docKey);
                assert(ackVersion !== null, 'ackVersions should contain every doc in the write.');
                if (!doc || doc.version.compareTo(ackVersion) < 0) {
                    doc = batch.applyToRemoteDocument(docKey, doc, batchResult);
                    if (!doc) {
                        assert(!remoteDoc, 'Mutation batch ' +
                            batch +
                            ' applied to document ' +
                            remoteDoc +
                            ' resulted in null');
                    }
                    else {
                        documentBuffer.addEntry(doc);
                    }
                }
            });
        });
        return promiseChain.next(function () {
            return _this.mutationQueue.removeMutationBatch(txn, batch);
        });
    };
    LocalStore.prototype.collectGarbage = function (garbageCollector) {
        var _this = this;
        return this.persistence.runTransaction('Collect garbage', 'readwrite-primary', function (txn) { return garbageCollector.collect(txn, _this.queryDataByTarget); });
    };
    // PORTING NOTE: Multi-tab only.
    LocalStore.prototype.getQueryForTarget = function (targetId) {
        var _this = this;
        if (this.queryDataByTarget[targetId]) {
            return Promise.resolve(this.queryDataByTarget[targetId].query);
        }
        else {
            return this.persistence.runTransaction('Get query data', 'readonly', function (txn) {
                return _this.queryCache
                    .getQueryDataForTarget(txn, targetId)
                    .next(function (queryData) { return (queryData ? queryData.query : null); });
            });
        }
    };
    // PORTING NOTE: Multi-tab only.
    LocalStore.prototype.getNewDocumentChanges = function () {
        var _this = this;
        return this.persistence.runTransaction('Get new document changes', 'readonly', function (txn) {
            return _this.remoteDocuments.getNewDocumentChanges(txn);
        });
    };
    /**
     * The maximum time to leave a resume token buffered without writing it out.
     * This value is arbitrary: it's long enough to avoid several writes
     * (possibly indefinitely if updates come more frequently than this) but
     * short enough that restarting after crashing will still have a pretty
     * recent resume token.
     */
    LocalStore.RESUME_TOKEN_MAX_AGE_MICROS = 5 * 60 * 1e6;
    return LocalStore;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var MemoryMutationQueue = /** @class */ (function () {
    function MemoryMutationQueue(indexManager, referenceDelegate) {
        this.indexManager = indexManager;
        this.referenceDelegate = referenceDelegate;
        /**
         * The set of all mutations that have been sent but not yet been applied to
         * the backend.
         */
        this.mutationQueue = [];
        /** Next value to use when assigning sequential IDs to each mutation batch. */
        this.nextBatchId = 1;
        /** The last received stream token from the server, used to acknowledge which
         * responses the client has processed. Stream tokens are opaque checkpoint
         * markers whose only real value is their inclusion in the next request.
         */
        this.lastStreamToken = emptyByteString();
        /** An ordered mapping between documents and the mutations batch IDs. */
        this.batchesByDocumentKey = new SortedSet(DocReference.compareByKey);
    }
    MemoryMutationQueue.prototype.checkEmpty = function (transaction) {
        return PersistencePromise.resolve(this.mutationQueue.length === 0);
    };
    MemoryMutationQueue.prototype.acknowledgeBatch = function (transaction, batch, streamToken) {
        var batchId = batch.batchId;
        var batchIndex = this.indexOfExistingBatchId(batchId, 'acknowledged');
        assert(batchIndex === 0, 'Can only acknowledge the first batch in the mutation queue');
        // Verify that the batch in the queue is the one to be acknowledged.
        var check = this.mutationQueue[batchIndex];
        assert(batchId === check.batchId, 'Queue ordering failure: expected batch ' +
            batchId +
            ', got batch ' +
            check.batchId);
        this.lastStreamToken = streamToken;
        return PersistencePromise.resolve();
    };
    MemoryMutationQueue.prototype.getLastStreamToken = function (transaction) {
        return PersistencePromise.resolve(this.lastStreamToken);
    };
    MemoryMutationQueue.prototype.setLastStreamToken = function (transaction, streamToken) {
        this.lastStreamToken = streamToken;
        return PersistencePromise.resolve();
    };
    MemoryMutationQueue.prototype.addMutationBatch = function (transaction, localWriteTime, baseMutations, mutations) {
        assert(mutations.length !== 0, 'Mutation batches should not be empty');
        var batchId = this.nextBatchId;
        this.nextBatchId++;
        if (this.mutationQueue.length > 0) {
            var prior = this.mutationQueue[this.mutationQueue.length - 1];
            assert(prior.batchId < batchId, 'Mutation batchIDs must be monotonically increasing order');
        }
        var batch = new MutationBatch(batchId, localWriteTime, baseMutations, mutations);
        this.mutationQueue.push(batch);
        // Track references by document key and index collection parents.
        for (var _i = 0, mutations_1 = mutations; _i < mutations_1.length; _i++) {
            var mutation = mutations_1[_i];
            this.batchesByDocumentKey = this.batchesByDocumentKey.add(new DocReference(mutation.key, batchId));
            this.indexManager.addToCollectionParentIndex(transaction, mutation.key.path.popLast());
        }
        return PersistencePromise.resolve(batch);
    };
    MemoryMutationQueue.prototype.lookupMutationBatch = function (transaction, batchId) {
        return PersistencePromise.resolve(this.findMutationBatch(batchId));
    };
    MemoryMutationQueue.prototype.lookupMutationKeys = function (transaction, batchId) {
        var mutationBatch = this.findMutationBatch(batchId);
        assert(mutationBatch != null, 'Failed to find local mutation batch.');
        return PersistencePromise.resolve(mutationBatch.keys());
    };
    MemoryMutationQueue.prototype.getNextMutationBatchAfterBatchId = function (transaction, batchId) {
        var nextBatchId = batchId + 1;
        // The requested batchId may still be out of range so normalize it to the
        // start of the queue.
        var rawIndex = this.indexOfBatchId(nextBatchId);
        var index = rawIndex < 0 ? 0 : rawIndex;
        return PersistencePromise.resolve(this.mutationQueue.length > index ? this.mutationQueue[index] : null);
    };
    MemoryMutationQueue.prototype.getAllMutationBatches = function (transaction) {
        return PersistencePromise.resolve(this.mutationQueue.slice());
    };
    MemoryMutationQueue.prototype.getAllMutationBatchesAffectingDocumentKey = function (transaction, documentKey) {
        var _this = this;
        var start = new DocReference(documentKey, 0);
        var end = new DocReference(documentKey, Number.POSITIVE_INFINITY);
        var result = [];
        this.batchesByDocumentKey.forEachInRange([start, end], function (ref) {
            assert(documentKey.isEqual(ref.key), "Should only iterate over a single key's batches");
            var batch = _this.findMutationBatch(ref.targetOrBatchId);
            assert(batch !== null, 'Batches in the index must exist in the main table');
            result.push(batch);
        });
        return PersistencePromise.resolve(result);
    };
    MemoryMutationQueue.prototype.getAllMutationBatchesAffectingDocumentKeys = function (transaction, documentKeys) {
        var _this = this;
        var uniqueBatchIDs = new SortedSet(primitiveComparator);
        documentKeys.forEach(function (documentKey) {
            var start = new DocReference(documentKey, 0);
            var end = new DocReference(documentKey, Number.POSITIVE_INFINITY);
            _this.batchesByDocumentKey.forEachInRange([start, end], function (ref) {
                assert(documentKey.isEqual(ref.key), "For each key, should only iterate over a single key's batches");
                uniqueBatchIDs = uniqueBatchIDs.add(ref.targetOrBatchId);
            });
        });
        return PersistencePromise.resolve(this.findMutationBatches(uniqueBatchIDs));
    };
    MemoryMutationQueue.prototype.getAllMutationBatchesAffectingQuery = function (transaction, query) {
        assert(!query.isCollectionGroupQuery(), 'CollectionGroup queries should be handled in LocalDocumentsView');
        // Use the query path as a prefix for testing if a document matches the
        // query.
        var prefix = query.path;
        var immediateChildrenPathLength = prefix.length + 1;
        // Construct a document reference for actually scanning the index. Unlike
        // the prefix the document key in this reference must have an even number of
        // segments. The empty segment can be used a suffix of the query path
        // because it precedes all other segments in an ordered traversal.
        var startPath = prefix;
        if (!DocumentKey.isDocumentKey(startPath)) {
            startPath = startPath.child('');
        }
        var start = new DocReference(new DocumentKey(startPath), 0);
        // Find unique batchIDs referenced by all documents potentially matching the
        // query.
        var uniqueBatchIDs = new SortedSet(primitiveComparator);
        this.batchesByDocumentKey.forEachWhile(function (ref) {
            var rowKeyPath = ref.key.path;
            if (!prefix.isPrefixOf(rowKeyPath)) {
                return false;
            }
            else {
                // Rows with document keys more than one segment longer than the query
                // path can't be matches. For example, a query on 'rooms' can't match
                // the document /rooms/abc/messages/xyx.
                // TODO(mcg): we'll need a different scanner when we implement
                // ancestor queries.
                if (rowKeyPath.length === immediateChildrenPathLength) {
                    uniqueBatchIDs = uniqueBatchIDs.add(ref.targetOrBatchId);
                }
                return true;
            }
        }, start);
        return PersistencePromise.resolve(this.findMutationBatches(uniqueBatchIDs));
    };
    MemoryMutationQueue.prototype.findMutationBatches = function (batchIDs) {
        var _this = this;
        // Construct an array of matching batches, sorted by batchID to ensure that
        // multiple mutations affecting the same document key are applied in order.
        var result = [];
        batchIDs.forEach(function (batchId) {
            var batch = _this.findMutationBatch(batchId);
            if (batch !== null) {
                result.push(batch);
            }
        });
        return result;
    };
    MemoryMutationQueue.prototype.removeMutationBatch = function (transaction, batch) {
        var _this = this;
        // Find the position of the first batch for removal. This need not be the
        // first entry in the queue.
        var batchIndex = this.indexOfExistingBatchId(batch.batchId, 'removed');
        assert(batchIndex === 0, 'Can only remove the first entry of the mutation queue');
        this.mutationQueue.shift();
        var references = this.batchesByDocumentKey;
        return PersistencePromise.forEach(batch.mutations, function (mutation) {
            var ref = new DocReference(mutation.key, batch.batchId);
            references = references.delete(ref);
            return _this.referenceDelegate.removeMutationReference(transaction, mutation.key);
        }).next(function () {
            _this.batchesByDocumentKey = references;
        });
    };
    MemoryMutationQueue.prototype.removeCachedMutationKeys = function (batchId) {
        // No-op since the memory mutation queue does not maintain a separate cache.
    };
    MemoryMutationQueue.prototype.containsKey = function (txn, key) {
        var ref = new DocReference(key, 0);
        var firstRef = this.batchesByDocumentKey.firstAfterOrEqual(ref);
        return PersistencePromise.resolve(key.isEqual(firstRef && firstRef.key));
    };
    MemoryMutationQueue.prototype.performConsistencyCheck = function (txn) {
        if (this.mutationQueue.length === 0) {
            assert(this.batchesByDocumentKey.isEmpty(), 'Document leak -- detected dangling mutation references when queue is empty.');
        }
        return PersistencePromise.resolve();
    };
    /**
     * Finds the index of the given batchId in the mutation queue and asserts that
     * the resulting index is within the bounds of the queue.
     *
     * @param batchId The batchId to search for
     * @param action A description of what the caller is doing, phrased in passive
     * form (e.g. "acknowledged" in a routine that acknowledges batches).
     */
    MemoryMutationQueue.prototype.indexOfExistingBatchId = function (batchId, action) {
        var index = this.indexOfBatchId(batchId);
        assert(index >= 0 && index < this.mutationQueue.length, 'Batches must exist to be ' + action);
        return index;
    };
    /**
     * Finds the index of the given batchId in the mutation queue. This operation
     * is O(1).
     *
     * @return The computed index of the batch with the given batchId, based on
     * the state of the queue. Note this index can be negative if the requested
     * batchId has already been remvoed from the queue or past the end of the
     * queue if the batchId is larger than the last added batch.
     */
    MemoryMutationQueue.prototype.indexOfBatchId = function (batchId) {
        if (this.mutationQueue.length === 0) {
            // As an index this is past the end of the queue
            return 0;
        }
        // Examine the front of the queue to figure out the difference between the
        // batchId and indexes in the array. Note that since the queue is ordered
        // by batchId, if the first batch has a larger batchId then the requested
        // batchId doesn't exist in the queue.
        var firstBatchId = this.mutationQueue[0].batchId;
        return batchId - firstBatchId;
    };
    /**
     * A version of lookupMutationBatch that doesn't return a promise, this makes
     * other functions that uses this code easier to read and more efficent.
     */
    MemoryMutationQueue.prototype.findMutationBatch = function (batchId) {
        var index = this.indexOfBatchId(batchId);
        if (index < 0 || index >= this.mutationQueue.length) {
            return null;
        }
        var batch = this.mutationQueue[index];
        assert(batch.batchId === batchId, 'If found batch must match');
        return batch;
    };
    return MemoryMutationQueue;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var MemoryQueryCache = /** @class */ (function () {
    function MemoryQueryCache(persistence) {
        this.persistence = persistence;
        /**
         * Maps a query to the data about that query
         */
        this.queries = new ObjectMap(function (q) { return q.canonicalId(); });
        /** The last received snapshot version. */
        this.lastRemoteSnapshotVersion = SnapshotVersion.MIN;
        /** The highest numbered target ID encountered. */
        this.highestTargetId = 0;
        /** The highest sequence number encountered. */
        this.highestSequenceNumber = 0;
        /**
         * A ordered bidirectional mapping between documents and the remote target
         * IDs.
         */
        this.references = new ReferenceSet();
        this.targetCount = 0;
        this.targetIdGenerator = TargetIdGenerator.forQueryCache();
    }
    MemoryQueryCache.prototype.getTargetCount = function (txn) {
        return PersistencePromise.resolve(this.targetCount);
    };
    MemoryQueryCache.prototype.forEachTarget = function (txn, f) {
        this.queries.forEach(function (_, queryData) { return f(queryData); });
        return PersistencePromise.resolve();
    };
    MemoryQueryCache.prototype.getLastRemoteSnapshotVersion = function (transaction) {
        return PersistencePromise.resolve(this.lastRemoteSnapshotVersion);
    };
    MemoryQueryCache.prototype.getHighestSequenceNumber = function (transaction) {
        return PersistencePromise.resolve(this.highestSequenceNumber);
    };
    MemoryQueryCache.prototype.allocateTargetId = function (transaction) {
        var nextTargetId = this.targetIdGenerator.after(this.highestTargetId);
        this.highestTargetId = nextTargetId;
        return PersistencePromise.resolve(nextTargetId);
    };
    MemoryQueryCache.prototype.setTargetsMetadata = function (transaction, highestListenSequenceNumber, lastRemoteSnapshotVersion) {
        if (lastRemoteSnapshotVersion) {
            this.lastRemoteSnapshotVersion = lastRemoteSnapshotVersion;
        }
        if (highestListenSequenceNumber > this.highestSequenceNumber) {
            this.highestSequenceNumber = highestListenSequenceNumber;
        }
        return PersistencePromise.resolve();
    };
    MemoryQueryCache.prototype.saveQueryData = function (queryData) {
        this.queries.set(queryData.query, queryData);
        var targetId = queryData.targetId;
        if (targetId > this.highestTargetId) {
            this.highestTargetId = targetId;
        }
        if (queryData.sequenceNumber > this.highestSequenceNumber) {
            this.highestSequenceNumber = queryData.sequenceNumber;
        }
    };
    MemoryQueryCache.prototype.addQueryData = function (transaction, queryData) {
        assert(!this.queries.has(queryData.query), 'Adding a query that already exists');
        this.saveQueryData(queryData);
        this.targetCount += 1;
        return PersistencePromise.resolve();
    };
    MemoryQueryCache.prototype.updateQueryData = function (transaction, queryData) {
        assert(this.queries.has(queryData.query), 'Updating a non-existent query');
        this.saveQueryData(queryData);
        return PersistencePromise.resolve();
    };
    MemoryQueryCache.prototype.removeQueryData = function (transaction, queryData) {
        assert(this.targetCount > 0, 'Removing a target from an empty cache');
        assert(this.queries.has(queryData.query), 'Removing a non-existent target from the cache');
        this.queries.delete(queryData.query);
        this.references.removeReferencesForId(queryData.targetId);
        this.targetCount -= 1;
        return PersistencePromise.resolve();
    };
    MemoryQueryCache.prototype.removeTargets = function (transaction, upperBound, activeTargetIds) {
        var _this = this;
        var count = 0;
        var removals = [];
        this.queries.forEach(function (key, queryData) {
            if (queryData.sequenceNumber <= upperBound &&
                !activeTargetIds[queryData.targetId]) {
                _this.queries.delete(key);
                removals.push(_this.removeMatchingKeysForTargetId(transaction, queryData.targetId));
                count++;
            }
        });
        return PersistencePromise.waitFor(removals).next(function () { return count; });
    };
    MemoryQueryCache.prototype.getQueryCount = function (transaction) {
        return PersistencePromise.resolve(this.targetCount);
    };
    MemoryQueryCache.prototype.getQueryData = function (transaction, query) {
        var queryData = this.queries.get(query) || null;
        return PersistencePromise.resolve(queryData);
    };
    MemoryQueryCache.prototype.getQueryDataForTarget = function (transaction, targetId) {
        // This method is only needed for multi-tab and we can't implement it
        // efficiently without additional data structures.
        return fail('Not yet implemented.');
    };
    MemoryQueryCache.prototype.addMatchingKeys = function (txn, keys, targetId) {
        this.references.addReferences(keys, targetId);
        var referenceDelegate = this.persistence.referenceDelegate;
        var promises = [];
        if (referenceDelegate) {
            keys.forEach(function (key) {
                promises.push(referenceDelegate.addReference(txn, key));
            });
        }
        return PersistencePromise.waitFor(promises);
    };
    MemoryQueryCache.prototype.removeMatchingKeys = function (txn, keys, targetId) {
        this.references.removeReferences(keys, targetId);
        var referenceDelegate = this.persistence.referenceDelegate;
        var promises = [];
        if (referenceDelegate) {
            keys.forEach(function (key) {
                promises.push(referenceDelegate.removeReference(txn, key));
            });
        }
        return PersistencePromise.waitFor(promises);
    };
    MemoryQueryCache.prototype.removeMatchingKeysForTargetId = function (txn, targetId) {
        this.references.removeReferencesForId(targetId);
        return PersistencePromise.resolve();
    };
    MemoryQueryCache.prototype.getMatchingKeysForTargetId = function (txn, targetId) {
        var matchingKeys = this.references.referencesForId(targetId);
        return PersistencePromise.resolve(matchingKeys);
    };
    MemoryQueryCache.prototype.containsKey = function (txn, key) {
        return PersistencePromise.resolve(this.references.containsKey(key));
    };
    return MemoryQueryCache;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
function documentSizeMap() {
    return new SortedMap(DocumentKey.comparator);
}
var MemoryRemoteDocumentCache = /** @class */ (function () {
    /**
     * @param sizer Used to assess the size of a document. For eager GC, this is expected to just
     * return 0 to avoid unnecessarily doing the work of calculating the size.
     */
    function MemoryRemoteDocumentCache(indexManager, sizer) {
        this.indexManager = indexManager;
        this.sizer = sizer;
        this.docs = documentSizeMap();
        this.newDocumentChanges = documentKeySet();
        this.size = 0;
    }
    /**
     * Adds the supplied entries to the cache. Adds the given size delta to the cached size.
     */
    MemoryRemoteDocumentCache.prototype.addEntries = function (transaction, entries, sizeDelta) {
        var promises = [];
        for (var _i = 0, entries_1 = entries; _i < entries_1.length; _i++) {
            var entry = entries_1[_i];
            var key = entry.maybeDocument.key;
            this.docs = this.docs.insert(key, entry);
            this.newDocumentChanges = this.newDocumentChanges.add(key);
            promises.push(this.indexManager.addToCollectionParentIndex(transaction, key.path.popLast()));
        }
        this.size += sizeDelta;
        return PersistencePromise.waitFor(promises);
    };
    /**
     * Removes the specified entry from the cache and updates the size as appropriate.
     */
    MemoryRemoteDocumentCache.prototype.removeEntry = function (transaction, documentKey) {
        var entry = this.docs.get(documentKey);
        if (entry) {
            this.docs = this.docs.remove(documentKey);
            this.size -= entry.size;
            return PersistencePromise.resolve(entry.size);
        }
        else {
            return PersistencePromise.resolve(0);
        }
    };
    MemoryRemoteDocumentCache.prototype.getEntry = function (transaction, documentKey) {
        var entry = this.docs.get(documentKey);
        return PersistencePromise.resolve(entry ? entry.maybeDocument : null);
    };
    /**
     * Looks up an entry in the cache.
     *
     * @param documentKey The key of the entry to look up.
     * @return The cached MaybeDocument entry and its size, or null if we have nothing cached.
     */
    MemoryRemoteDocumentCache.prototype.getSizedEntry = function (transaction, documentKey) {
        return PersistencePromise.resolve(this.docs.get(documentKey));
    };
    MemoryRemoteDocumentCache.prototype.getEntries = function (transaction, documentKeys) {
        var _this = this;
        var results = nullableMaybeDocumentMap();
        documentKeys.forEach(function (documentKey) {
            var entry = _this.docs.get(documentKey);
            results = results.insert(documentKey, entry ? entry.maybeDocument : null);
        });
        return PersistencePromise.resolve(results);
    };
    /**
     * Looks up several entries in the cache.
     *
     * @param documentKeys The set of keys entries to look up.
     * @return A map of MaybeDocuments indexed by key (if a document cannot be
     *     found, the key will be mapped to null) and a map of sizes indexed by
     *     key (zero if the key cannot be found).
     */
    MemoryRemoteDocumentCache.prototype.getSizedEntries = function (transaction, documentKeys) {
        var _this = this;
        var results = nullableMaybeDocumentMap();
        var sizeMap = new SortedMap(DocumentKey.comparator);
        documentKeys.forEach(function (documentKey) {
            var entry = _this.docs.get(documentKey);
            results = results.insert(documentKey, entry ? entry.maybeDocument : null);
            sizeMap = sizeMap.insert(documentKey, entry ? entry.size : 0);
        });
        return PersistencePromise.resolve({ maybeDocuments: results, sizeMap: sizeMap });
    };
    MemoryRemoteDocumentCache.prototype.getDocumentsMatchingQuery = function (transaction, query) {
        assert(!query.isCollectionGroupQuery(), 'CollectionGroup queries should be handled in LocalDocumentsView');
        var results = documentMap();
        // Documents are ordered by key, so we can use a prefix scan to narrow down
        // the documents we need to match the query against.
        var prefix = new DocumentKey(query.path.child(''));
        var iterator = this.docs.getIteratorFrom(prefix);
        while (iterator.hasNext()) {
            var _a = iterator.getNext(), key = _a.key, maybeDocument = _a.value.maybeDocument;
            if (!query.path.isPrefixOf(key.path)) {
                break;
            }
            if (maybeDocument instanceof Document && query.matches(maybeDocument)) {
                results = results.insert(maybeDocument.key, maybeDocument);
            }
        }
        return PersistencePromise.resolve(results);
    };
    MemoryRemoteDocumentCache.prototype.forEachDocumentKey = function (transaction, f) {
        return PersistencePromise.forEach(this.docs, function (key) { return f(key); });
    };
    MemoryRemoteDocumentCache.prototype.getNewDocumentChanges = function (transaction) {
        var _this = this;
        var changedDocs = maybeDocumentMap();
        this.newDocumentChanges.forEach(function (key) {
            var entry = _this.docs.get(key);
            var changedDoc = entry
                ? entry.maybeDocument
                : new NoDocument(key, SnapshotVersion.forDeletedDoc());
            changedDocs = changedDocs.insert(key, changedDoc);
        });
        this.newDocumentChanges = documentKeySet();
        return PersistencePromise.resolve(changedDocs);
    };
    MemoryRemoteDocumentCache.prototype.newChangeBuffer = function () {
        return new MemoryRemoteDocumentChangeBuffer(this.sizer, this);
    };
    MemoryRemoteDocumentCache.prototype.getSize = function (txn) {
        return PersistencePromise.resolve(this.size);
    };
    return MemoryRemoteDocumentCache;
}());
/**
 * Handles the details of adding and updating documents in the MemoryRemoteDocumentCache.
 */
var MemoryRemoteDocumentChangeBuffer = /** @class */ (function (_super) {
    tslib_1.__extends(MemoryRemoteDocumentChangeBuffer, _super);
    function MemoryRemoteDocumentChangeBuffer(sizer, documentCache) {
        var _this = _super.call(this) || this;
        _this.sizer = sizer;
        _this.documentCache = documentCache;
        return _this;
    }
    MemoryRemoteDocumentChangeBuffer.prototype.applyChanges = function (transaction) {
        var _this = this;
        var changes = this.assertChanges();
        var delta = 0;
        var docs = [];
        changes.forEach(function (key, maybeDocument) {
            var previousSize = _this.documentSizes.get(key);
            assert(previousSize !== undefined, "Attempting to change document " + key.toString() + " without having read it first");
            var size = _this.sizer(maybeDocument);
            delta += size - previousSize;
            docs.push({ maybeDocument: maybeDocument, size: size });
        });
        return this.documentCache.addEntries(transaction, docs, delta);
    };
    MemoryRemoteDocumentChangeBuffer.prototype.getFromCache = function (transaction, documentKey) {
        return this.documentCache.getSizedEntry(transaction, documentKey);
    };
    MemoryRemoteDocumentChangeBuffer.prototype.getAllFromCache = function (transaction, documentKeys) {
        return this.documentCache.getSizedEntries(transaction, documentKeys);
    };
    return MemoryRemoteDocumentChangeBuffer;
}(RemoteDocumentChangeBuffer));

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$4 = 'MemoryPersistence';
/**
 * A memory-backed instance of Persistence. Data is stored only in RAM and
 * not persisted across sessions.
 */
var MemoryPersistence = /** @class */ (function () {
    /**
     * The constructor accepts a factory for creating a reference delegate. This
     * allows both the delegate and this instance to have strong references to
     * each other without having nullable fields that would then need to be
     * checked or asserted on every access.
     */
    function MemoryPersistence(clientId, referenceDelegateFactory) {
        var _this = this;
        this.clientId = clientId;
        this.mutationQueues = {};
        this.listenSequence = new ListenSequence(0);
        this._started = false;
        this._started = true;
        this.referenceDelegate = referenceDelegateFactory(this);
        this.queryCache = new MemoryQueryCache(this);
        var sizer = function (doc) {
            return _this.referenceDelegate.documentSize(doc);
        };
        this.indexManager = new MemoryIndexManager();
        this.remoteDocumentCache = new MemoryRemoteDocumentCache(this.indexManager, sizer);
    }
    MemoryPersistence.createLruPersistence = function (clientId, serializer, params) {
        var factory = function (p) {
            return new MemoryLruDelegate(p, new LocalSerializer(serializer), params);
        };
        return new MemoryPersistence(clientId, factory);
    };
    MemoryPersistence.createEagerPersistence = function (clientId) {
        var factory = function (p) {
            return new MemoryEagerDelegate(p);
        };
        return new MemoryPersistence(clientId, factory);
    };
    MemoryPersistence.prototype.shutdown = function (deleteData) {
        // No durable state to ensure is closed on shutdown.
        this._started = false;
        return Promise.resolve();
    };
    Object.defineProperty(MemoryPersistence.prototype, "started", {
        get: function () {
            return this._started;
        },
        enumerable: true,
        configurable: true
    });
    MemoryPersistence.prototype.getActiveClients = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                return [2 /*return*/, [this.clientId]];
            });
        });
    };
    MemoryPersistence.prototype.setPrimaryStateListener = function (primaryStateListener) {
        // All clients using memory persistence act as primary.
        return primaryStateListener(true);
    };
    MemoryPersistence.prototype.setNetworkEnabled = function (networkEnabled) {
        // No op.
    };
    MemoryPersistence.prototype.getIndexManager = function () {
        return this.indexManager;
    };
    MemoryPersistence.prototype.getMutationQueue = function (user) {
        var queue = this.mutationQueues[user.toKey()];
        if (!queue) {
            queue = new MemoryMutationQueue(this.indexManager, this.referenceDelegate);
            this.mutationQueues[user.toKey()] = queue;
        }
        return queue;
    };
    MemoryPersistence.prototype.getQueryCache = function () {
        return this.queryCache;
    };
    MemoryPersistence.prototype.getRemoteDocumentCache = function () {
        return this.remoteDocumentCache;
    };
    MemoryPersistence.prototype.runTransaction = function (action, mode, transactionOperation) {
        var _this = this;
        debug(LOG_TAG$4, 'Starting transaction:', action);
        var txn = new MemoryTransaction(this.listenSequence.next());
        this.referenceDelegate.onTransactionStarted();
        return transactionOperation(txn)
            .next(function (result) {
            return _this.referenceDelegate
                .onTransactionCommitted(txn)
                .next(function () { return result; });
        })
            .toPromise();
    };
    MemoryPersistence.prototype.mutationQueuesContainKey = function (transaction, key) {
        return PersistencePromise.or(values(this.mutationQueues)
            .map(function (queue) { return function () { return queue.containsKey(transaction, key); }; }));
    };
    return MemoryPersistence;
}());
/**
 * Memory persistence is not actually transactional, but future implementations
 * may have transaction-scoped state.
 */
var MemoryTransaction = /** @class */ (function () {
    function MemoryTransaction(currentSequenceNumber) {
        this.currentSequenceNumber = currentSequenceNumber;
    }
    return MemoryTransaction;
}());
var MemoryEagerDelegate = /** @class */ (function () {
    function MemoryEagerDelegate(persistence) {
        this.persistence = persistence;
    }
    MemoryEagerDelegate.prototype.setInMemoryPins = function (inMemoryPins) {
        this.inMemoryPins = inMemoryPins;
    };
    MemoryEagerDelegate.prototype.addReference = function (txn, key) {
        this.orphanedDocuments.delete(key);
        return PersistencePromise.resolve();
    };
    MemoryEagerDelegate.prototype.removeReference = function (txn, key) {
        this.orphanedDocuments.add(key);
        return PersistencePromise.resolve();
    };
    MemoryEagerDelegate.prototype.removeMutationReference = function (txn, key) {
        this.orphanedDocuments.add(key);
        return PersistencePromise.resolve();
    };
    MemoryEagerDelegate.prototype.removeTarget = function (txn, queryData) {
        var _this = this;
        var cache = this.persistence.getQueryCache();
        return cache
            .getMatchingKeysForTargetId(txn, queryData.targetId)
            .next(function (keys) {
            keys.forEach(function (key) { return _this.orphanedDocuments.add(key); });
        })
            .next(function () { return cache.removeQueryData(txn, queryData); });
    };
    MemoryEagerDelegate.prototype.onTransactionStarted = function () {
        this.orphanedDocuments = new Set();
    };
    MemoryEagerDelegate.prototype.onTransactionCommitted = function (txn) {
        var _this = this;
        var cache = this.persistence.getRemoteDocumentCache();
        return PersistencePromise.forEach(this.orphanedDocuments, function (key) {
            return _this.isReferenced(txn, key).next(function (isReferenced) {
                if (!isReferenced) {
                    // Since this is the eager delegate and memory persistence,
                    // we don't care about the size of documents. We don't track
                    // the size of the cache for eager GC.
                    return cache.removeEntry(txn, key).next(function () { });
                }
                return PersistencePromise.resolve();
            });
        });
    };
    MemoryEagerDelegate.prototype.updateLimboDocument = function (txn, key) {
        var _this = this;
        return this.isReferenced(txn, key).next(function (isReferenced) {
            if (isReferenced) {
                _this.orphanedDocuments.delete(key);
            }
            else {
                _this.orphanedDocuments.add(key);
            }
        });
    };
    MemoryEagerDelegate.prototype.documentSize = function (doc) {
        // For eager GC, we don't care about the document size, there are no size thresholds.
        return 0;
    };
    MemoryEagerDelegate.prototype.isReferenced = function (txn, key) {
        var _this = this;
        return PersistencePromise.or([
            function () { return _this.persistence.getQueryCache().containsKey(txn, key); },
            function () { return _this.persistence.mutationQueuesContainKey(txn, key); },
            function () { return PersistencePromise.resolve(_this.inMemoryPins.containsKey(key)); }
        ]);
    };
    return MemoryEagerDelegate;
}());
var MemoryLruDelegate = /** @class */ (function () {
    function MemoryLruDelegate(persistence, serializer, lruParams) {
        this.persistence = persistence;
        this.serializer = serializer;
        this.orphanedSequenceNumbers = new ObjectMap(function (k) { return encode(k.path); });
        this.garbageCollector = new LruGarbageCollector(this, lruParams);
    }
    // No-ops, present so memory persistence doesn't have to care which delegate
    // it has.
    MemoryLruDelegate.prototype.onTransactionStarted = function () { };
    MemoryLruDelegate.prototype.onTransactionCommitted = function (txn) {
        return PersistencePromise.resolve();
    };
    MemoryLruDelegate.prototype.forEachTarget = function (txn, f) {
        return this.persistence.getQueryCache().forEachTarget(txn, f);
    };
    MemoryLruDelegate.prototype.getSequenceNumberCount = function (txn) {
        var docCountPromise = this.orphanedDocumentCount(txn);
        var targetCountPromise = this.persistence
            .getQueryCache()
            .getTargetCount(txn);
        return targetCountPromise.next(function (targetCount) {
            return docCountPromise.next(function (docCount) { return targetCount + docCount; });
        });
    };
    MemoryLruDelegate.prototype.orphanedDocumentCount = function (txn) {
        var orphanedCount = 0;
        return this.forEachOrphanedDocumentSequenceNumber(txn, function (_) {
            orphanedCount++;
        }).next(function () { return orphanedCount; });
    };
    MemoryLruDelegate.prototype.forEachOrphanedDocumentSequenceNumber = function (txn, f) {
        var _this = this;
        return PersistencePromise.forEach(this.orphanedSequenceNumbers, function (key, sequenceNumber) {
            // Pass in the exact sequence number as the upper bound so we know it won't be pinned by
            // being too recent.
            return _this.isPinned(txn, key, sequenceNumber).next(function (isPinned) {
                if (!isPinned) {
                    return f(sequenceNumber);
                }
                else {
                    return PersistencePromise.resolve();
                }
            });
        });
    };
    MemoryLruDelegate.prototype.setInMemoryPins = function (inMemoryPins) {
        this.inMemoryPins = inMemoryPins;
    };
    MemoryLruDelegate.prototype.removeTargets = function (txn, upperBound, activeTargetIds) {
        return this.persistence
            .getQueryCache()
            .removeTargets(txn, upperBound, activeTargetIds);
    };
    MemoryLruDelegate.prototype.removeOrphanedDocuments = function (txn, upperBound) {
        var _this = this;
        var count = 0;
        var cache = this.persistence.getRemoteDocumentCache();
        var p = cache.forEachDocumentKey(txn, function (key) {
            return _this.isPinned(txn, key, upperBound).next(function (isPinned) {
                if (isPinned) {
                    return PersistencePromise.resolve();
                }
                else {
                    count++;
                    // The memory remote document cache does its own byte
                    // accounting on removal. This is ok because updating the size
                    // for memory persistence does not incur IO.
                    return cache.removeEntry(txn, key).next();
                }
            });
        });
        return p.next(function () { return count; });
    };
    MemoryLruDelegate.prototype.removeMutationReference = function (txn, key) {
        this.orphanedSequenceNumbers.set(key, txn.currentSequenceNumber);
        return PersistencePromise.resolve();
    };
    MemoryLruDelegate.prototype.removeTarget = function (txn, queryData) {
        var updated = queryData.copy({
            sequenceNumber: txn.currentSequenceNumber
        });
        return this.persistence.getQueryCache().updateQueryData(txn, updated);
    };
    MemoryLruDelegate.prototype.addReference = function (txn, key) {
        this.orphanedSequenceNumbers.set(key, txn.currentSequenceNumber);
        return PersistencePromise.resolve();
    };
    MemoryLruDelegate.prototype.removeReference = function (txn, key) {
        this.orphanedSequenceNumbers.set(key, txn.currentSequenceNumber);
        return PersistencePromise.resolve();
    };
    MemoryLruDelegate.prototype.updateLimboDocument = function (txn, key) {
        this.orphanedSequenceNumbers.set(key, txn.currentSequenceNumber);
        return PersistencePromise.resolve();
    };
    MemoryLruDelegate.prototype.documentSize = function (maybeDoc) {
        var remoteDocument = this.serializer.toDbRemoteDocument(maybeDoc);
        var value;
        if (remoteDocument.document) {
            value = remoteDocument.document;
        }
        else if (remoteDocument.unknownDocument) {
            value = remoteDocument.unknownDocument;
        }
        else if (remoteDocument.noDocument) {
            value = remoteDocument.noDocument;
        }
        else {
            throw fail('Unknown remote document type');
        }
        return JSON.stringify(value).length;
    };
    MemoryLruDelegate.prototype.isPinned = function (txn, key, upperBound) {
        var _this = this;
        return PersistencePromise.or([
            function () { return _this.persistence.mutationQueuesContainKey(txn, key); },
            function () { return PersistencePromise.resolve(_this.inMemoryPins.containsKey(key)); },
            function () { return _this.persistence.getQueryCache().containsKey(txn, key); },
            function () {
                var orphanedAt = _this.orphanedSequenceNumbers.get(key);
                return PersistencePromise.resolve(orphanedAt !== undefined && orphanedAt > upperBound);
            }
        ]);
    };
    MemoryLruDelegate.prototype.getCacheSize = function (txn) {
        return this.persistence.getRemoteDocumentCache().getSize(txn);
    };
    return MemoryLruDelegate;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$5 = 'ExponentialBackoff';
/**
 * A helper for running delayed tasks following an exponential backoff curve
 * between attempts.
 *
 * Each delay is made up of a "base" delay which follows the exponential
 * backoff curve, and a +/- 50% "jitter" that is calculated and added to the
 * base delay. This prevents clients from accidentally synchronizing their
 * delays causing spikes of load to the backend.
 */
var ExponentialBackoff = /** @class */ (function () {
    function ExponentialBackoff(
    /**
     * The AsyncQueue to run backoff operations on.
     */
    queue, 
    /**
     * The ID to use when scheduling backoff operations on the AsyncQueue.
     */
    timerId, 
    /**
     * The initial delay (used as the base delay on the first retry attempt).
     * Note that jitter will still be applied, so the actual delay could be as
     * little as 0.5*initialDelayMs.
     */
    initialDelayMs, 
    /**
     * The multiplier to use to determine the extended base delay after each
     * attempt.
     */
    backoffFactor, 
    /**
     * The maximum base delay after which no further backoff is performed.
     * Note that jitter will still be applied, so the actual delay could be as
     * much as 1.5*maxDelayMs.
     */
    maxDelayMs) {
        this.queue = queue;
        this.timerId = timerId;
        this.initialDelayMs = initialDelayMs;
        this.backoffFactor = backoffFactor;
        this.maxDelayMs = maxDelayMs;
        this.timerPromise = null;
        /** The last backoff attempt, as epoch milliseconds. */
        this.lastAttemptTime = Date.now();
        this.reset();
    }
    /**
     * Resets the backoff delay.
     *
     * The very next backoffAndWait() will have no delay. If it is called again
     * (i.e. due to an error), initialDelayMs (plus jitter) will be used, and
     * subsequent ones will increase according to the backoffFactor.
     */
    ExponentialBackoff.prototype.reset = function () {
        this.currentBaseMs = 0;
    };
    /**
     * Resets the backoff delay to the maximum delay (e.g. for use after a
     * RESOURCE_EXHAUSTED error).
     */
    ExponentialBackoff.prototype.resetToMax = function () {
        this.currentBaseMs = this.maxDelayMs;
    };
    /**
     * Returns a promise that resolves after currentDelayMs, and increases the
     * delay for any subsequent attempts. If there was a pending backoff operation
     * already, it will be canceled.
     */
    ExponentialBackoff.prototype.backoffAndRun = function (op) {
        var _this = this;
        // Cancel any pending backoff operation.
        this.cancel();
        // First schedule using the current base (which may be 0 and should be
        // honored as such).
        var desiredDelayWithJitterMs = Math.floor(this.currentBaseMs + this.jitterDelayMs());
        // Guard against lastAttemptTime being in the future due to a clock change.
        var delaySoFarMs = Math.max(0, Date.now() - this.lastAttemptTime);
        // Guard against the backoff delay already being past.
        var remainingDelayMs = Math.max(0, desiredDelayWithJitterMs - delaySoFarMs);
        if (this.currentBaseMs > 0) {
            debug(LOG_TAG$5, "Backing off for " + remainingDelayMs + " ms " +
                ("(base delay: " + this.currentBaseMs + " ms, ") +
                ("delay with jitter: " + desiredDelayWithJitterMs + " ms, ") +
                ("last attempt: " + delaySoFarMs + " ms ago)"));
        }
        this.timerPromise = this.queue.enqueueAfterDelay(this.timerId, remainingDelayMs, function () {
            _this.lastAttemptTime = Date.now();
            return op();
        });
        // Apply backoff factor to determine next delay and ensure it is within
        // bounds.
        this.currentBaseMs *= this.backoffFactor;
        if (this.currentBaseMs < this.initialDelayMs) {
            this.currentBaseMs = this.initialDelayMs;
        }
        if (this.currentBaseMs > this.maxDelayMs) {
            this.currentBaseMs = this.maxDelayMs;
        }
    };
    ExponentialBackoff.prototype.cancel = function () {
        if (this.timerPromise !== null) {
            this.timerPromise.cancel();
            this.timerPromise = null;
        }
    };
    /** Returns a random value in the range [-currentBaseMs/2, currentBaseMs/2] */
    ExponentialBackoff.prototype.jitterDelayMs = function () {
        return (Math.random() - 0.5) * this.currentBaseMs;
    };
    return ExponentialBackoff;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$6 = 'PersistentStream';
/**
 * PersistentStream can be in one of 5 states (each described in detail below)
 * based on the following state transition diagram:
 *
 *          start() called             auth & connection succeeded
 * INITIAL ----------------> STARTING -----------------------------> OPEN
 *                             ^  |                                   |
 *                             |  |                    error occurred |
 *                             |  \-----------------------------v-----/
 *                             |                                |
 *                    backoff  |                                |
 *                    elapsed  |              start() called    |
 *                             \--- BACKOFF <---------------- ERROR
 *
 * [any state] --------------------------> INITIAL
 *               stop() called or
 *               idle timer expired
 */
var PersistentStreamState;
(function (PersistentStreamState) {
    /**
     * The streaming RPC is not yet running and there's no error condition.
     * Calling start() will start the stream immediately without backoff.
     * While in this state isStarted() will return false.
     */
    PersistentStreamState[PersistentStreamState["Initial"] = 0] = "Initial";
    /**
     * The stream is starting, either waiting for an auth token or for the stream
     * to successfully open. While in this state, isStarted() will return true but
     * isOpen() will return false.
     */
    PersistentStreamState[PersistentStreamState["Starting"] = 1] = "Starting";
    /**
     * The streaming RPC is up and running. Requests and responses can flow
     * freely. Both isStarted() and isOpen() will return true.
     */
    PersistentStreamState[PersistentStreamState["Open"] = 2] = "Open";
    /**
     * The stream encountered an error. The next start attempt will back off.
     * While in this state isStarted() will return false.
     */
    PersistentStreamState[PersistentStreamState["Error"] = 3] = "Error";
    /**
     * An in-between state after an error where the stream is waiting before
     * re-starting. After waiting is complete, the stream will try to open.
     * While in this state isStarted() will return true but isOpen() will return
     * false.
     */
    PersistentStreamState[PersistentStreamState["Backoff"] = 4] = "Backoff";
})(PersistentStreamState || (PersistentStreamState = {}));
/**
 * Initial backoff time in milliseconds after an error.
 * Set to 1s according to https://cloud.google.com/apis/design/errors.
 */
var BACKOFF_INITIAL_DELAY_MS = 1000;
/** Maximum backoff time in milliseconds */
var BACKOFF_MAX_DELAY_MS = 60 * 1000;
var BACKOFF_FACTOR = 1.5;
/** The time a stream stays open after it is marked idle. */
var IDLE_TIMEOUT_MS = 60 * 1000;
/**
 * A PersistentStream is an abstract base class that represents a streaming RPC
 * to the Firestore backend. It's built on top of the connections own support
 * for streaming RPCs, and adds several critical features for our clients:
 *
 *   - Exponential backoff on failure
 *   - Authentication via CredentialsProvider
 *   - Dispatching all callbacks into the shared worker queue
 *   - Closing idle streams after 60 seconds of inactivity
 *
 * Subclasses of PersistentStream implement serialization of models to and
 * from the JSON representation of the protocol buffers for a specific
 * streaming RPC.
 *
 * ## Starting and Stopping
 *
 * Streaming RPCs are stateful and need to be start()ed before messages can
 * be sent and received. The PersistentStream will call the onOpen() function
 * of the listener once the stream is ready to accept requests.
 *
 * Should a start() fail, PersistentStream will call the registered onClose()
 * listener with a FirestoreError indicating what went wrong.
 *
 * A PersistentStream can be started and stopped repeatedly.
 *
 * Generic types:
 *  SendType: The type of the outgoing message of the underlying
 *    connection stream
 *  ReceiveType: The type of the incoming message of the underlying
 *    connection stream
 *  ListenerType: The type of the listener that will be used for callbacks
 */
var PersistentStream = /** @class */ (function () {
    function PersistentStream(queue, connectionTimerId, idleTimerId, connection, credentialsProvider, listener) {
        this.queue = queue;
        this.idleTimerId = idleTimerId;
        this.connection = connection;
        this.credentialsProvider = credentialsProvider;
        this.listener = listener;
        this.state = PersistentStreamState.Initial;
        /**
         * A close count that's incremented every time the stream is closed; used by
         * getCloseGuardedDispatcher() to invalidate callbacks that happen after
         * close.
         */
        this.closeCount = 0;
        this.idleTimer = null;
        this.stream = null;
        this.backoff = new ExponentialBackoff(queue, connectionTimerId, BACKOFF_INITIAL_DELAY_MS, BACKOFF_FACTOR, BACKOFF_MAX_DELAY_MS);
    }
    /**
     * Returns true if start() has been called and no error has occurred. True
     * indicates the stream is open or in the process of opening (which
     * encompasses respecting backoff, getting auth tokens, and starting the
     * actual RPC). Use isOpen() to determine if the stream is open and ready for
     * outbound requests.
     */
    PersistentStream.prototype.isStarted = function () {
        return (this.state === PersistentStreamState.Starting ||
            this.state === PersistentStreamState.Open ||
            this.state === PersistentStreamState.Backoff);
    };
    /**
     * Returns true if the underlying RPC is open (the onOpen() listener has been
     * called) and the stream is ready for outbound requests.
     */
    PersistentStream.prototype.isOpen = function () {
        return this.state === PersistentStreamState.Open;
    };
    /**
     * Starts the RPC. Only allowed if isStarted() returns false. The stream is
     * not immediately ready for use: onOpen() will be invoked when the RPC is
     * ready for outbound requests, at which point isOpen() will return true.
     *
     * When start returns, isStarted() will return true.
     */
    PersistentStream.prototype.start = function () {
        if (this.state === PersistentStreamState.Error) {
            this.performBackoff();
            return;
        }
        assert(this.state === PersistentStreamState.Initial, 'Already started');
        this.auth();
    };
    /**
     * Stops the RPC. This call is idempotent and allowed regardless of the
     * current isStarted() state.
     *
     * When stop returns, isStarted() and isOpen() will both return false.
     */
    PersistentStream.prototype.stop = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!this.isStarted()) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.close(PersistentStreamState.Initial)];
                    case 1:
                        _a.sent();
                        _a.label = 2;
                    case 2: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * After an error the stream will usually back off on the next attempt to
     * start it. If the error warrants an immediate restart of the stream, the
     * sender can use this to indicate that the receiver should not back off.
     *
     * Each error will call the onClose() listener. That function can decide to
     * inhibit backoff if required.
     */
    PersistentStream.prototype.inhibitBackoff = function () {
        assert(!this.isStarted(), 'Can only inhibit backoff in a stopped state');
        this.state = PersistentStreamState.Initial;
        this.backoff.reset();
    };
    /**
     * Marks this stream as idle. If no further actions are performed on the
     * stream for one minute, the stream will automatically close itself and
     * notify the stream's onClose() handler with Status.OK. The stream will then
     * be in a !isStarted() state, requiring the caller to start the stream again
     * before further use.
     *
     * Only streams that are in state 'Open' can be marked idle, as all other
     * states imply pending network operations.
     */
    PersistentStream.prototype.markIdle = function () {
        var _this = this;
        // Starts the idle time if we are in state 'Open' and are not yet already
        // running a timer (in which case the previous idle timeout still applies).
        if (this.isOpen() && this.idleTimer === null) {
            this.idleTimer = this.queue.enqueueAfterDelay(this.idleTimerId, IDLE_TIMEOUT_MS, function () { return _this.handleIdleCloseTimer(); });
        }
    };
    /** Sends a message to the underlying stream. */
    PersistentStream.prototype.sendRequest = function (msg) {
        this.cancelIdleCheck();
        this.stream.send(msg);
    };
    /** Called by the idle timer when the stream should close due to inactivity. */
    PersistentStream.prototype.handleIdleCloseTimer = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (this.isOpen()) {
                    // When timing out an idle stream there's no reason to force the stream into backoff when
                    // it restarts so set the stream state to Initial instead of Error.
                    return [2 /*return*/, this.close(PersistentStreamState.Initial)];
                }
                return [2 /*return*/];
            });
        });
    };
    /** Marks the stream as active again. */
    PersistentStream.prototype.cancelIdleCheck = function () {
        if (this.idleTimer) {
            this.idleTimer.cancel();
            this.idleTimer = null;
        }
    };
    /**
     * Closes the stream and cleans up as necessary:
     *
     * * closes the underlying GRPC stream;
     * * calls the onClose handler with the given 'error';
     * * sets internal stream state to 'finalState';
     * * adjusts the backoff timer based on the error
     *
     * A new stream can be opened by calling start().
     *
     * @param finalState the intended state of the stream after closing.
     * @param error the error the connection was closed with.
     */
    PersistentStream.prototype.close = function (finalState, error$1) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        assert(this.isStarted(), 'Only started streams should be closed.');
                        assert(finalState === PersistentStreamState.Error || isNullOrUndefined(error$1), "Can't provide an error when not in an error state.");
                        // Cancel any outstanding timers (they're guaranteed not to execute).
                        this.cancelIdleCheck();
                        this.backoff.cancel();
                        // Invalidates any stream-related callbacks (e.g. from auth or the
                        // underlying stream), guaranteeing they won't execute.
                        this.closeCount++;
                        if (finalState !== PersistentStreamState.Error) {
                            // If this is an intentional close ensure we don't delay our next connection attempt.
                            this.backoff.reset();
                        }
                        else if (error$1 && error$1.code === Code.RESOURCE_EXHAUSTED) {
                            // Log the error. (Probably either 'quota exceeded' or 'max queue length reached'.)
                            error(error$1.toString());
                            error('Using maximum backoff delay to prevent overloading the backend.');
                            this.backoff.resetToMax();
                        }
                        else if (error$1 && error$1.code === Code.UNAUTHENTICATED) {
                            // "unauthenticated" error means the token was rejected. Try force refreshing it in case it
                            // just expired.
                            this.credentialsProvider.invalidateToken();
                        }
                        // Clean up the underlying stream because we are no longer interested in events.
                        if (this.stream !== null) {
                            this.tearDown();
                            this.stream.close();
                            this.stream = null;
                        }
                        // This state must be assigned before calling onClose() to allow the callback to
                        // inhibit backoff or otherwise manipulate the state in its non-started state.
                        this.state = finalState;
                        // Notify the listener that the stream closed.
                        return [4 /*yield*/, this.listener.onClose(error$1)];
                    case 1:
                        // Notify the listener that the stream closed.
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Can be overridden to perform additional cleanup before the stream is closed.
     * Calling super.tearDown() is not required.
     */
    PersistentStream.prototype.tearDown = function () { };
    PersistentStream.prototype.auth = function () {
        var _this = this;
        assert(this.state === PersistentStreamState.Initial, 'Must be in initial state to auth');
        this.state = PersistentStreamState.Starting;
        var dispatchIfNotClosed = this.getCloseGuardedDispatcher(this.closeCount);
        // TODO(mikelehen): Just use dispatchIfNotClosed, but see TODO below.
        var closeCount = this.closeCount;
        this.credentialsProvider.getToken().then(function (token) {
            // Stream can be stopped while waiting for authentication.
            // TODO(mikelehen): We really should just use dispatchIfNotClosed
            // and let this dispatch onto the queue, but that opened a spec test can
            // of worms that I don't want to deal with in this PR.
            if (_this.closeCount === closeCount) {
                // Normally we'd have to schedule the callback on the AsyncQueue.
                // However, the following calls are safe to be called outside the
                // AsyncQueue since they don't chain asynchronous calls
                _this.startStream(token);
            }
        }, function (error) {
            dispatchIfNotClosed(function () {
                var rpcError = new FirestoreError(Code.UNKNOWN, 'Fetching auth token failed: ' + error.message);
                return _this.handleStreamClose(rpcError);
            });
        });
    };
    PersistentStream.prototype.startStream = function (token) {
        var _this = this;
        assert(this.state === PersistentStreamState.Starting, 'Trying to start stream in a non-starting state');
        var dispatchIfNotClosed = this.getCloseGuardedDispatcher(this.closeCount);
        this.stream = this.startRpc(token);
        this.stream.onOpen(function () {
            dispatchIfNotClosed(function () {
                assert(_this.state === PersistentStreamState.Starting, 'Expected stream to be in state Starting, but was ' + _this.state);
                _this.state = PersistentStreamState.Open;
                return _this.listener.onOpen();
            });
        });
        this.stream.onClose(function (error) {
            dispatchIfNotClosed(function () {
                return _this.handleStreamClose(error);
            });
        });
        this.stream.onMessage(function (msg) {
            dispatchIfNotClosed(function () {
                return _this.onMessage(msg);
            });
        });
    };
    PersistentStream.prototype.performBackoff = function () {
        var _this = this;
        assert(this.state === PersistentStreamState.Error, 'Should only perform backoff when in Error state');
        this.state = PersistentStreamState.Backoff;
        this.backoff.backoffAndRun(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                assert(this.state === PersistentStreamState.Backoff, 'Backoff elapsed but state is now: ' + this.state);
                this.state = PersistentStreamState.Initial;
                this.start();
                assert(this.isStarted(), 'PersistentStream should have started');
                return [2 /*return*/];
            });
        }); });
    };
    // Visible for tests
    PersistentStream.prototype.handleStreamClose = function (error) {
        assert(this.isStarted(), "Can't handle server close on non-started stream");
        debug(LOG_TAG$6, "close with error: " + error);
        this.stream = null;
        // In theory the stream could close cleanly, however, in our current model
        // we never expect this to happen because if we stop a stream ourselves,
        // this callback will never be called. To prevent cases where we retry
        // without a backoff accidentally, we set the stream to error in all cases.
        return this.close(PersistentStreamState.Error, error);
    };
    /**
     * Returns a "dispatcher" function that dispatches operations onto the
     * AsyncQueue but only runs them if closeCount remains unchanged. This allows
     * us to turn auth / stream callbacks into no-ops if the stream is closed /
     * re-opened, etc.
     */
    PersistentStream.prototype.getCloseGuardedDispatcher = function (startCloseCount) {
        var _this = this;
        return function (fn) {
            _this.queue.enqueueAndForget(function () {
                if (_this.closeCount === startCloseCount) {
                    return fn();
                }
                else {
                    debug(LOG_TAG$6, 'stream callback skipped by getCloseGuardedDispatcher.');
                    return Promise.resolve();
                }
            });
        };
    };
    return PersistentStream;
}());
/**
 * A PersistentStream that implements the Listen RPC.
 *
 * Once the Listen stream has called the onOpen() listener, any number of
 * listen() and unlisten() calls can be made to control what changes will be
 * sent from the server for ListenResponses.
 */
var PersistentListenStream = /** @class */ (function (_super) {
    tslib_1.__extends(PersistentListenStream, _super);
    function PersistentListenStream(queue, connection, credentials, serializer, listener) {
        var _this = _super.call(this, queue, TimerId.ListenStreamConnectionBackoff, TimerId.ListenStreamIdle, connection, credentials, listener) || this;
        _this.serializer = serializer;
        return _this;
    }
    PersistentListenStream.prototype.startRpc = function (token) {
        return this.connection.openStream('Listen', token);
    };
    PersistentListenStream.prototype.onMessage = function (watchChangeProto) {
        // A successful response means the stream is healthy
        this.backoff.reset();
        var watchChange = this.serializer.fromWatchChange(watchChangeProto);
        var snapshot = this.serializer.versionFromListenResponse(watchChangeProto);
        return this.listener.onWatchChange(watchChange, snapshot);
    };
    /**
     * Registers interest in the results of the given query. If the query
     * includes a resumeToken it will be included in the request. Results that
     * affect the query will be streamed back as WatchChange messages that
     * reference the targetId.
     */
    PersistentListenStream.prototype.watch = function (queryData) {
        var request = {};
        request.database = this.serializer.encodedDatabaseId;
        request.addTarget = this.serializer.toTarget(queryData);
        var labels = this.serializer.toListenRequestLabels(queryData);
        if (labels) {
            request.labels = labels;
        }
        this.sendRequest(request);
    };
    /**
     * Unregisters interest in the results of the query associated with the
     * given targetId.
     */
    PersistentListenStream.prototype.unwatch = function (targetId) {
        var request = {};
        request.database = this.serializer.encodedDatabaseId;
        request.removeTarget = targetId;
        this.sendRequest(request);
    };
    return PersistentListenStream;
}(PersistentStream));
/**
 * A Stream that implements the Write RPC.
 *
 * The Write RPC requires the caller to maintain special streamToken
 * state in between calls, to help the server understand which responses the
 * client has processed by the time the next request is made. Every response
 * will contain a streamToken; this value must be passed to the next
 * request.
 *
 * After calling start() on this stream, the next request must be a handshake,
 * containing whatever streamToken is on hand. Once a response to this
 * request is received, all pending mutations may be submitted. When
 * submitting multiple batches of mutations at the same time, it's
 * okay to use the same streamToken for the calls to writeMutations.
 *
 * TODO(b/33271235): Use proto types
 */
var PersistentWriteStream = /** @class */ (function (_super) {
    tslib_1.__extends(PersistentWriteStream, _super);
    function PersistentWriteStream(queue, connection, credentials, serializer, listener) {
        var _this = _super.call(this, queue, TimerId.WriteStreamConnectionBackoff, TimerId.WriteStreamIdle, connection, credentials, listener) || this;
        _this.serializer = serializer;
        _this.handshakeComplete_ = false;
        return _this;
    }
    Object.defineProperty(PersistentWriteStream.prototype, "handshakeComplete", {
        /**
         * Tracks whether or not a handshake has been successfully exchanged and
         * the stream is ready to accept mutations.
         */
        get: function () {
            return this.handshakeComplete_;
        },
        enumerable: true,
        configurable: true
    });
    // Override of PersistentStream.start
    PersistentWriteStream.prototype.start = function () {
        this.handshakeComplete_ = false;
        _super.prototype.start.call(this);
    };
    PersistentWriteStream.prototype.tearDown = function () {
        if (this.handshakeComplete_) {
            this.writeMutations([]);
        }
    };
    PersistentWriteStream.prototype.startRpc = function (token) {
        return this.connection.openStream('Write', token);
    };
    PersistentWriteStream.prototype.onMessage = function (responseProto) {
        // Always capture the last stream token.
        assert(!!responseProto.streamToken, 'Got a write response without a stream token');
        this.lastStreamToken = responseProto.streamToken;
        if (!this.handshakeComplete_) {
            // The first response is always the handshake response
            assert(!responseProto.writeResults || responseProto.writeResults.length === 0, 'Got mutation results for handshake');
            this.handshakeComplete_ = true;
            return this.listener.onHandshakeComplete();
        }
        else {
            // A successful first write response means the stream is healthy,
            // Note, that we could consider a successful handshake healthy, however,
            // the write itself might be causing an error we want to back off from.
            this.backoff.reset();
            var results = this.serializer.fromWriteResults(responseProto.writeResults, responseProto.commitTime);
            var commitVersion = this.serializer.fromVersion(responseProto.commitTime);
            return this.listener.onMutationResult(commitVersion, results);
        }
    };
    /**
     * Sends an initial streamToken to the server, performing the handshake
     * required to make the StreamingWrite RPC work. Subsequent
     * calls should wait until onHandshakeComplete was called.
     */
    PersistentWriteStream.prototype.writeHandshake = function () {
        assert(this.isOpen(), 'Writing handshake requires an opened stream');
        assert(!this.handshakeComplete_, 'Handshake already completed');
        // TODO(dimond): Support stream resumption. We intentionally do not set the
        // stream token on the handshake, ignoring any stream token we might have.
        var request = {};
        request.database = this.serializer.encodedDatabaseId;
        this.sendRequest(request);
    };
    /** Sends a group of mutations to the Firestore backend to apply. */
    PersistentWriteStream.prototype.writeMutations = function (mutations) {
        var _this = this;
        assert(this.isOpen(), 'Writing mutations requires an opened stream');
        assert(this.handshakeComplete_, 'Handshake must be complete before writing mutations');
        assert(this.lastStreamToken.length > 0, 'Trying to write mutation without a token');
        var request = {
            // Protos are typed with string, but we support UInt8Array on Node
            // tslint:disable-next-line:no-any
            streamToken: this.lastStreamToken,
            writes: mutations.map(function (mutation) { return _this.serializer.toMutation(mutation); })
        };
        this.sendRequest(request);
    };
    return PersistentWriteStream;
}(PersistentStream));

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Datastore is a wrapper around the external Google Cloud Datastore grpc API,
 * which provides an interface that is more convenient for the rest of the
 * client SDK architecture to consume.
 */
var Datastore = /** @class */ (function () {
    function Datastore(queue, connection, credentials, serializer) {
        this.queue = queue;
        this.connection = connection;
        this.credentials = credentials;
        this.serializer = serializer;
    }
    Datastore.prototype.newPersistentWriteStream = function (listener) {
        return new PersistentWriteStream(this.queue, this.connection, this.credentials, this.serializer, listener);
    };
    Datastore.prototype.newPersistentWatchStream = function (listener) {
        return new PersistentListenStream(this.queue, this.connection, this.credentials, this.serializer, listener);
    };
    Datastore.prototype.commit = function (mutations) {
        var _this = this;
        var params = {
            database: this.serializer.encodedDatabaseId,
            writes: mutations.map(function (m) { return _this.serializer.toMutation(m); })
        };
        return this.invokeRPC('Commit', params).then(function (response) {
            return _this.serializer.fromWriteResults(response.writeResults, response.commitTime);
        });
    };
    Datastore.prototype.lookup = function (keys) {
        var _this = this;
        var params = {
            database: this.serializer.encodedDatabaseId,
            documents: keys.map(function (k) { return _this.serializer.toName(k); })
        };
        return this.invokeStreamingRPC('BatchGetDocuments', params).then(function (response) {
            var docs = maybeDocumentMap();
            response.forEach(function (proto) {
                var doc = _this.serializer.fromMaybeDocument(proto);
                docs = docs.insert(doc.key, doc);
            });
            var result = [];
            keys.forEach(function (key) {
                var doc = docs.get(key);
                assert(!!doc, 'Missing entity in write response for ' + key);
                result.push(doc);
            });
            return result;
        });
    };
    /** Gets an auth token and invokes the provided RPC. */
    Datastore.prototype.invokeRPC = function (rpcName, request) {
        var _this = this;
        return this.credentials
            .getToken()
            .then(function (token) {
            return _this.connection.invokeRPC(rpcName, request, token);
        })
            .catch(function (error) {
            if (error.code === Code.UNAUTHENTICATED) {
                _this.credentials.invalidateToken();
            }
            throw error;
        });
    };
    /** Gets an auth token and invokes the provided RPC with streamed results. */
    Datastore.prototype.invokeStreamingRPC = function (rpcName, request) {
        var _this = this;
        return this.credentials
            .getToken()
            .then(function (token) {
            return _this.connection.invokeStreamingRPC(rpcName, request, token);
        })
            .catch(function (error) {
            if (error.code === Code.UNAUTHENTICATED) {
                _this.credentials.invalidateToken();
            }
            throw error;
        });
    };
    return Datastore;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Internal transaction object responsible for accumulating the mutations to
 * perform and the base versions for any documents read.
 */
var Transaction = /** @class */ (function () {
    function Transaction(datastore) {
        this.datastore = datastore;
        // The version of each document that was read during this transaction.
        this.readVersions = documentVersionMap();
        this.mutations = [];
        this.committed = false;
    }
    Transaction.prototype.recordVersion = function (doc) {
        var docVersion;
        if (doc instanceof Document) {
            docVersion = doc.version;
        }
        else if (doc instanceof NoDocument) {
            // For deleted docs, we must use baseVersion 0 when we overwrite them.
            docVersion = SnapshotVersion.forDeletedDoc();
        }
        else {
            throw fail('Document in a transaction was a ' + doc.constructor.name);
        }
        var existingVersion = this.readVersions.get(doc.key);
        if (existingVersion !== null) {
            if (!docVersion.isEqual(existingVersion)) {
                // This transaction will fail no matter what.
                throw new FirestoreError(Code.ABORTED, 'Document version changed between two reads.');
            }
        }
        else {
            this.readVersions = this.readVersions.insert(doc.key, docVersion);
        }
    };
    Transaction.prototype.lookup = function (keys) {
        var _this = this;
        if (this.committed) {
            return Promise.reject('Transaction has already completed.');
        }
        if (this.mutations.length > 0) {
            return Promise.reject('Transactions lookups are invalid after writes.');
        }
        return this.datastore.lookup(keys).then(function (docs) {
            docs.forEach(function (doc) {
                if (doc instanceof NoDocument || doc instanceof Document) {
                    _this.recordVersion(doc);
                }
                else {
                    fail('Document in a transaction was a ' + doc.constructor.name);
                }
            });
            return docs;
        });
    };
    Transaction.prototype.write = function (mutations) {
        if (this.committed) {
            throw new FirestoreError(Code.FAILED_PRECONDITION, 'Transaction has already completed.');
        }
        this.mutations = this.mutations.concat(mutations);
    };
    /**
     * Returns the version of this document when it was read in this transaction,
     * as a precondition, or no precondition if it was not read.
     */
    Transaction.prototype.precondition = function (key) {
        var version = this.readVersions.get(key);
        if (version) {
            return Precondition.updateTime(version);
        }
        else {
            return Precondition.NONE;
        }
    };
    /**
     * Returns the precondition for a document if the operation is an update.
     */
    Transaction.prototype.preconditionForUpdate = function (key) {
        var version = this.readVersions.get(key);
        if (version && version.isEqual(SnapshotVersion.forDeletedDoc())) {
            // The document doesn't exist, so fail the transaction.
            throw new FirestoreError(Code.FAILED_PRECONDITION, "Can't update a document that doesn't exist.");
        }
        else if (version) {
            // Document exists, base precondition on document update time.
            return Precondition.updateTime(version);
        }
        else {
            // Document was not read, so we just use the preconditions for a blind
            // update.
            return Precondition.exists(true);
        }
    };
    Transaction.prototype.set = function (key, data) {
        this.write(data.toMutations(key, this.precondition(key)));
    };
    Transaction.prototype.update = function (key, data) {
        this.write(data.toMutations(key, this.preconditionForUpdate(key)));
    };
    Transaction.prototype.delete = function (key) {
        this.write([new DeleteMutation(key, this.precondition(key))]);
        // Since the delete will be applied before all following writes, we need to
        // ensure that the precondition for the next write will be exists: false.
        this.readVersions = this.readVersions.insert(key, SnapshotVersion.forDeletedDoc());
    };
    Transaction.prototype.commit = function () {
        var _this = this;
        var unwritten = this.readVersions;
        // For each mutation, note that the doc was written.
        this.mutations.forEach(function (mutation) {
            unwritten = unwritten.remove(mutation.key);
        });
        if (!unwritten.isEmpty()) {
            return Promise.reject(Error('Every document read in a transaction must also be written.'));
        }
        return this.datastore.commit(this.mutations).then(function () {
            _this.committed = true;
        });
    };
    return Transaction;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Describes the online state of the Firestore client. Note that this does not
 * indicate whether or not the remote store is trying to connect or not. This is
 * primarily used by the View / EventManager code to change their behavior while
 * offline (e.g. get() calls shouldn't wait for data from the server and
 * snapshot events should set metadata.isFromCache=true).
 */
var OnlineState;
(function (OnlineState) {
    /**
     * The Firestore client is in an unknown online state. This means the client
     * is either not actively trying to establish a connection or it is currently
     * trying to establish a connection, but it has not succeeded or failed yet.
     * Higher-level components should not operate in offline mode.
     */
    OnlineState[OnlineState["Unknown"] = 0] = "Unknown";
    /**
     * The client is connected and the connections are healthy. This state is
     * reached after a successful connection and there has been at least one
     * successful message received from the backends.
     */
    OnlineState[OnlineState["Online"] = 1] = "Online";
    /**
     * The client is either trying to establish a connection but failing, or it
     * has been explicitly marked offline via a call to disableNetwork().
     * Higher-level components should operate in offline mode.
     */
    OnlineState[OnlineState["Offline"] = 2] = "Offline";
})(OnlineState || (OnlineState = {}));
/** The source of an online state event. */
var OnlineStateSource;
(function (OnlineStateSource) {
    OnlineStateSource[OnlineStateSource["RemoteStore"] = 0] = "RemoteStore";
    OnlineStateSource[OnlineStateSource["SharedClientState"] = 1] = "SharedClientState";
})(OnlineStateSource || (OnlineStateSource = {}));

/**
 * @license
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$7 = 'OnlineStateTracker';
// To deal with transient failures, we allow multiple stream attempts before
// giving up and transitioning from OnlineState.Unknown to Offline.
// TODO(mikelehen): This used to be set to 2 as a mitigation for b/66228394.
// @jdimond thinks that bug is sufficiently fixed so that we can set this back
// to 1. If that works okay, we could potentially remove this logic entirely.
var MAX_WATCH_STREAM_FAILURES = 1;
// To deal with stream attempts that don't succeed or fail in a timely manner,
// we have a timeout for OnlineState to reach Online or Offline.
// If the timeout is reached, we transition to Offline rather than waiting
// indefinitely.
var ONLINE_STATE_TIMEOUT_MS = 10 * 1000;
/**
 * A component used by the RemoteStore to track the OnlineState (that is,
 * whether or not the client as a whole should be considered to be online or
 * offline), implementing the appropriate heuristics.
 *
 * In particular, when the client is trying to connect to the backend, we
 * allow up to MAX_WATCH_STREAM_FAILURES within ONLINE_STATE_TIMEOUT_MS for
 * a connection to succeed. If we have too many failures or the timeout elapses,
 * then we set the OnlineState to Offline, and the client will behave as if
 * it is offline (get()s will return cached data, etc.).
 */
var OnlineStateTracker = /** @class */ (function () {
    function OnlineStateTracker(asyncQueue, onlineStateHandler) {
        this.asyncQueue = asyncQueue;
        this.onlineStateHandler = onlineStateHandler;
        /** The current OnlineState. */
        this.state = OnlineState.Unknown;
        /**
         * A count of consecutive failures to open the stream. If it reaches the
         * maximum defined by MAX_WATCH_STREAM_FAILURES, we'll set the OnlineState to
         * Offline.
         */
        this.watchStreamFailures = 0;
        /**
         * A timer that elapses after ONLINE_STATE_TIMEOUT_MS, at which point we
         * transition from OnlineState.Unknown to OnlineState.Offline without waiting
         * for the stream to actually fail (MAX_WATCH_STREAM_FAILURES times).
         */
        this.onlineStateTimer = null;
        /**
         * Whether the client should log a warning message if it fails to connect to
         * the backend (initially true, cleared after a successful stream, or if we've
         * logged the message already).
         */
        this.shouldWarnClientIsOffline = true;
    }
    /**
     * Called by RemoteStore when a watch stream is started (including on each
     * backoff attempt).
     *
     * If this is the first attempt, it sets the OnlineState to Unknown and starts
     * the onlineStateTimer.
     */
    OnlineStateTracker.prototype.handleWatchStreamStart = function () {
        var _this = this;
        if (this.watchStreamFailures === 0) {
            this.setAndBroadcast(OnlineState.Unknown);
            assert(this.onlineStateTimer === null, "onlineStateTimer shouldn't be started yet");
            this.onlineStateTimer = this.asyncQueue.enqueueAfterDelay(TimerId.OnlineStateTimeout, ONLINE_STATE_TIMEOUT_MS, function () {
                _this.onlineStateTimer = null;
                assert(_this.state === OnlineState.Unknown, 'Timer should be canceled if we transitioned to a different state.');
                _this.logClientOfflineWarningIfNecessary("Backend didn't respond within " + ONLINE_STATE_TIMEOUT_MS / 1000 + " " +
                    "seconds.");
                _this.setAndBroadcast(OnlineState.Offline);
                // NOTE: handleWatchStreamFailure() will continue to increment
                // watchStreamFailures even though we are already marked Offline,
                // but this is non-harmful.
                return Promise.resolve();
            });
        }
    };
    /**
     * Updates our OnlineState as appropriate after the watch stream reports a
     * failure. The first failure moves us to the 'Unknown' state. We then may
     * allow multiple failures (based on MAX_WATCH_STREAM_FAILURES) before we
     * actually transition to the 'Offline' state.
     */
    OnlineStateTracker.prototype.handleWatchStreamFailure = function (error) {
        if (this.state === OnlineState.Online) {
            this.setAndBroadcast(OnlineState.Unknown);
            // To get to OnlineState.Online, set() must have been called which would
            // have reset our heuristics.
            assert(this.watchStreamFailures === 0, 'watchStreamFailures must be 0');
            assert(this.onlineStateTimer === null, 'onlineStateTimer must be null');
        }
        else {
            this.watchStreamFailures++;
            if (this.watchStreamFailures >= MAX_WATCH_STREAM_FAILURES) {
                this.clearOnlineStateTimer();
                this.logClientOfflineWarningIfNecessary("Connection failed " + MAX_WATCH_STREAM_FAILURES + " " +
                    ("times. Most recent error: " + error.toString()));
                this.setAndBroadcast(OnlineState.Offline);
            }
        }
    };
    /**
     * Explicitly sets the OnlineState to the specified state.
     *
     * Note that this resets our timers / failure counters, etc. used by our
     * Offline heuristics, so must not be used in place of
     * handleWatchStreamStart() and handleWatchStreamFailure().
     */
    OnlineStateTracker.prototype.set = function (newState) {
        this.clearOnlineStateTimer();
        this.watchStreamFailures = 0;
        if (newState === OnlineState.Online) {
            // We've connected to watch at least once. Don't warn the developer
            // about being offline going forward.
            this.shouldWarnClientIsOffline = false;
        }
        this.setAndBroadcast(newState);
    };
    OnlineStateTracker.prototype.setAndBroadcast = function (newState) {
        if (newState !== this.state) {
            this.state = newState;
            this.onlineStateHandler(newState);
        }
    };
    OnlineStateTracker.prototype.logClientOfflineWarningIfNecessary = function (details) {
        var message = "Could not reach Cloud Firestore backend. " + details + "\n" +
            "This typically indicates that your device does not have a healthy " +
            "Internet connection at the moment. The client will operate in offline " +
            "mode until it is able to successfully connect to the backend.";
        if (this.shouldWarnClientIsOffline) {
            error(message);
            this.shouldWarnClientIsOffline = false;
        }
        else {
            debug(LOG_TAG$7, message);
        }
    };
    OnlineStateTracker.prototype.clearOnlineStateTimer = function () {
        if (this.onlineStateTimer !== null) {
            this.onlineStateTimer.cancel();
            this.onlineStateTimer = null;
        }
    };
    return OnlineStateTracker;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$8 = 'RemoteStore';
// TODO(b/35853402): Negotiate this with the stream.
var MAX_PENDING_WRITES = 10;
/**
 * RemoteStore - An interface to remotely stored data, basically providing a
 * wrapper around the Datastore that is more reliable for the rest of the
 * system.
 *
 * RemoteStore is responsible for maintaining the connection to the server.
 * - maintaining a list of active listens.
 * - reconnecting when the connection is dropped.
 * - resuming all the active listens on reconnect.
 *
 * RemoteStore handles all incoming events from the Datastore.
 * - listening to the watch stream and repackaging the events as RemoteEvents
 * - notifying SyncEngine of any changes to the active listens.
 *
 * RemoteStore takes writes from other components and handles them reliably.
 * - pulling pending mutations from LocalStore and sending them to Datastore.
 * - retrying mutations that failed because of network problems.
 * - acking mutations to the SyncEngine once they are accepted or rejected.
 */
var RemoteStore = /** @class */ (function () {
    function RemoteStore(
    /**
     * The local store, used to fill the write pipeline with outbound mutations.
     */
    localStore, 
    /** The client-side proxy for interacting with the backend. */
    datastore, asyncQueue, onlineStateHandler) {
        this.localStore = localStore;
        this.datastore = datastore;
        /**
         * A list of up to MAX_PENDING_WRITES writes that we have fetched from the
         * LocalStore via fillWritePipeline() and have or will send to the write
         * stream.
         *
         * Whenever writePipeline.length > 0 the RemoteStore will attempt to start or
         * restart the write stream. When the stream is established the writes in the
         * pipeline will be sent in order.
         *
         * Writes remain in writePipeline until they are acknowledged by the backend
         * and thus will automatically be re-sent if the stream is interrupted /
         * restarted before they're acknowledged.
         *
         * Write responses from the backend are linked to their originating request
         * purely based on order, and so we can just shift() writes from the front of
         * the writePipeline as we receive responses.
         */
        this.writePipeline = [];
        /**
         * A mapping of watched targets that the client cares about tracking and the
         * user has explicitly called a 'listen' for this target.
         *
         * These targets may or may not have been sent to or acknowledged by the
         * server. On re-establishing the listen stream, these targets should be sent
         * to the server. The targets removed with unlistens are removed eagerly
         * without waiting for confirmation from the listen stream.
         */
        this.listenTargets = {};
        this.watchChangeAggregator = null;
        /**
         * Set to true by enableNetwork() and false by disableNetwork() and indicates
         * the user-preferred network state.
         */
        this.networkEnabled = false;
        this.isPrimary = false;
        this.onlineStateTracker = new OnlineStateTracker(asyncQueue, onlineStateHandler);
        // Create streams (but note they're not started yet).
        this.watchStream = this.datastore.newPersistentWatchStream({
            onOpen: this.onWatchStreamOpen.bind(this),
            onClose: this.onWatchStreamClose.bind(this),
            onWatchChange: this.onWatchStreamChange.bind(this)
        });
        this.writeStream = this.datastore.newPersistentWriteStream({
            onOpen: this.onWriteStreamOpen.bind(this),
            onClose: this.onWriteStreamClose.bind(this),
            onHandshakeComplete: this.onWriteHandshakeComplete.bind(this),
            onMutationResult: this.onMutationResult.bind(this)
        });
    }
    /**
     * Starts up the remote store, creating streams, restoring state from
     * LocalStore, etc.
     */
    RemoteStore.prototype.start = function () {
        return this.enableNetwork();
    };
    /** Re-enables the network. Idempotent. */
    RemoteStore.prototype.enableNetwork = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        this.networkEnabled = true;
                        if (!this.canUseNetwork()) return [3 /*break*/, 3];
                        _a = this.writeStream;
                        return [4 /*yield*/, this.localStore.getLastStreamToken()];
                    case 1:
                        _a.lastStreamToken = _b.sent();
                        if (this.shouldStartWatchStream()) {
                            this.startWatchStream();
                        }
                        else {
                            this.onlineStateTracker.set(OnlineState.Unknown);
                        }
                        // This will start the write stream if necessary.
                        return [4 /*yield*/, this.fillWritePipeline()];
                    case 2:
                        // This will start the write stream if necessary.
                        _b.sent();
                        _b.label = 3;
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Temporarily disables the network. The network can be re-enabled using
     * enableNetwork().
     */
    RemoteStore.prototype.disableNetwork = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.networkEnabled = false;
                        return [4 /*yield*/, this.disableNetworkInternal()];
                    case 1:
                        _a.sent();
                        // Set the OnlineState to Offline so get()s return from cache, etc.
                        this.onlineStateTracker.set(OnlineState.Offline);
                        return [2 /*return*/];
                }
            });
        });
    };
    RemoteStore.prototype.disableNetworkInternal = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.writeStream.stop()];
                    case 1:
                        _a.sent();
                        return [4 /*yield*/, this.watchStream.stop()];
                    case 2:
                        _a.sent();
                        if (this.writePipeline.length > 0) {
                            debug(LOG_TAG$8, "Stopping write stream with " + this.writePipeline.length + " pending writes");
                            this.writePipeline = [];
                        }
                        this.cleanUpWatchStreamState();
                        return [2 /*return*/];
                }
            });
        });
    };
    RemoteStore.prototype.shutdown = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        debug(LOG_TAG$8, 'RemoteStore shutting down.');
                        this.networkEnabled = false;
                        return [4 /*yield*/, this.disableNetworkInternal()];
                    case 1:
                        _a.sent();
                        // Set the OnlineState to Unknown (rather than Offline) to avoid potentially
                        // triggering spurious listener events with cached data, etc.
                        this.onlineStateTracker.set(OnlineState.Unknown);
                        return [2 /*return*/];
                }
            });
        });
    };
    /** Starts new listen for the given query. Uses resume token if provided */
    RemoteStore.prototype.listen = function (queryData) {
        assert(!contains(this.listenTargets, queryData.targetId), 'listen called with duplicate targetId!');
        // Mark this as something the client is currently listening for.
        this.listenTargets[queryData.targetId] = queryData;
        if (this.shouldStartWatchStream()) {
            // The listen will be sent in onWatchStreamOpen
            this.startWatchStream();
        }
        else if (this.watchStream.isOpen()) {
            this.sendWatchRequest(queryData);
        }
    };
    /** Removes the listen from server */
    RemoteStore.prototype.unlisten = function (targetId) {
        assert(contains(this.listenTargets, targetId), 'unlisten called without assigned target ID!');
        delete this.listenTargets[targetId];
        if (this.watchStream.isOpen()) {
            this.sendUnwatchRequest(targetId);
        }
        if (isEmpty(this.listenTargets)) {
            if (this.watchStream.isOpen()) {
                this.watchStream.markIdle();
            }
            else if (this.canUseNetwork()) {
                // Revert to OnlineState.Unknown if the watch stream is not open and we
                // have no listeners, since without any listens to send we cannot
                // confirm if the stream is healthy and upgrade to OnlineState.Online.
                this.onlineStateTracker.set(OnlineState.Unknown);
            }
        }
    };
    /** {@link TargetMetadataProvider.getQueryDataForTarget} */
    RemoteStore.prototype.getQueryDataForTarget = function (targetId) {
        return this.listenTargets[targetId] || null;
    };
    /** {@link TargetMetadataProvider.getRemoteKeysForTarget} */
    RemoteStore.prototype.getRemoteKeysForTarget = function (targetId) {
        return this.syncEngine.getRemoteKeysForTarget(targetId);
    };
    /**
     * We need to increment the the expected number of pending responses we're due
     * from watch so we wait for the ack to process any messages from this target.
     */
    RemoteStore.prototype.sendWatchRequest = function (queryData) {
        this.watchChangeAggregator.recordPendingTargetRequest(queryData.targetId);
        this.watchStream.watch(queryData);
    };
    /**
     * We need to increment the expected number of pending responses we're due
     * from watch so we wait for the removal on the server before we process any
     * messages from this target.
     */
    RemoteStore.prototype.sendUnwatchRequest = function (targetId) {
        this.watchChangeAggregator.recordPendingTargetRequest(targetId);
        this.watchStream.unwatch(targetId);
    };
    RemoteStore.prototype.startWatchStream = function () {
        assert(this.shouldStartWatchStream(), 'startWatchStream() called when shouldStartWatchStream() is false.');
        this.watchChangeAggregator = new WatchChangeAggregator(this);
        this.watchStream.start();
        this.onlineStateTracker.handleWatchStreamStart();
    };
    /**
     * Returns whether the watch stream should be started because it's necessary
     * and has not yet been started.
     */
    RemoteStore.prototype.shouldStartWatchStream = function () {
        return (this.canUseNetwork() &&
            !this.watchStream.isStarted() &&
            !isEmpty(this.listenTargets));
    };
    RemoteStore.prototype.canUseNetwork = function () {
        return this.isPrimary && this.networkEnabled;
    };
    RemoteStore.prototype.cleanUpWatchStreamState = function () {
        this.watchChangeAggregator = null;
    };
    RemoteStore.prototype.onWatchStreamOpen = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                forEachNumber(this.listenTargets, function (targetId, queryData) {
                    _this.sendWatchRequest(queryData);
                });
                return [2 /*return*/];
            });
        });
    };
    RemoteStore.prototype.onWatchStreamClose = function (error) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (error === undefined) {
                    // Graceful stop (due to stop() or idle timeout). Make sure that's
                    // desirable.
                    assert(!this.shouldStartWatchStream(), 'Watch stream was stopped gracefully while still needed.');
                }
                this.cleanUpWatchStreamState();
                // If we still need the watch stream, retry the connection.
                if (this.shouldStartWatchStream()) {
                    this.onlineStateTracker.handleWatchStreamFailure(error);
                    this.startWatchStream();
                }
                else {
                    // No need to restart watch stream because there are no active targets.
                    // The online state is set to unknown because there is no active attempt
                    // at establishing a connection
                    this.onlineStateTracker.set(OnlineState.Unknown);
                }
                return [2 /*return*/];
            });
        });
    };
    RemoteStore.prototype.onWatchStreamChange = function (watchChange, snapshotVersion) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var lastRemoteSnapshotVersion;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        // Mark the client as online since we got a message from the server
                        this.onlineStateTracker.set(OnlineState.Online);
                        if (watchChange instanceof WatchTargetChange &&
                            watchChange.state === WatchTargetChangeState.Removed &&
                            watchChange.cause) {
                            // There was an error on a target, don't wait for a consistent snapshot
                            // to raise events
                            return [2 /*return*/, this.handleTargetError(watchChange)];
                        }
                        if (watchChange instanceof DocumentWatchChange) {
                            this.watchChangeAggregator.handleDocumentChange(watchChange);
                        }
                        else if (watchChange instanceof ExistenceFilterChange) {
                            this.watchChangeAggregator.handleExistenceFilter(watchChange);
                        }
                        else {
                            assert(watchChange instanceof WatchTargetChange, 'Expected watchChange to be an instance of WatchTargetChange');
                            this.watchChangeAggregator.handleTargetChange(watchChange);
                        }
                        if (!!snapshotVersion.isEqual(SnapshotVersion.MIN)) return [3 /*break*/, 3];
                        return [4 /*yield*/, this.localStore.getLastRemoteSnapshotVersion()];
                    case 1:
                        lastRemoteSnapshotVersion = _a.sent();
                        if (!(snapshotVersion.compareTo(lastRemoteSnapshotVersion) >= 0)) return [3 /*break*/, 3];
                        // We have received a target change with a global snapshot if the snapshot
                        // version is not equal to SnapshotVersion.MIN.
                        return [4 /*yield*/, this.raiseWatchSnapshot(snapshotVersion)];
                    case 2:
                        // We have received a target change with a global snapshot if the snapshot
                        // version is not equal to SnapshotVersion.MIN.
                        _a.sent();
                        _a.label = 3;
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Takes a batch of changes from the Datastore, repackages them as a
     * RemoteEvent, and passes that on to the listener, which is typically the
     * SyncEngine.
     */
    RemoteStore.prototype.raiseWatchSnapshot = function (snapshotVersion) {
        var _this = this;
        assert(!snapshotVersion.isEqual(SnapshotVersion.MIN), "Can't raise event for unknown SnapshotVersion");
        var remoteEvent = this.watchChangeAggregator.createRemoteEvent(snapshotVersion);
        // Update in-memory resume tokens. LocalStore will update the
        // persistent view of these when applying the completed RemoteEvent.
        forEachNumber(remoteEvent.targetChanges, function (targetId, change) {
            if (change.resumeToken.length > 0) {
                var queryData = _this.listenTargets[targetId];
                // A watched target might have been removed already.
                if (queryData) {
                    _this.listenTargets[targetId] = queryData.copy({
                        resumeToken: change.resumeToken,
                        snapshotVersion: snapshotVersion
                    });
                }
            }
        });
        // Re-establish listens for the targets that have been invalidated by
        // existence filter mismatches.
        remoteEvent.targetMismatches.forEach(function (targetId) {
            var queryData = _this.listenTargets[targetId];
            if (!queryData) {
                // A watched target might have been removed already.
                return;
            }
            // Clear the resume token for the query, since we're in a known mismatch
            // state.
            _this.listenTargets[targetId] = queryData.copy({
                resumeToken: emptyByteString()
            });
            // Cause a hard reset by unwatching and rewatching immediately, but
            // deliberately don't send a resume token so that we get a full update.
            _this.sendUnwatchRequest(targetId);
            // Mark the query we send as being on behalf of an existence filter
            // mismatch, but don't actually retain that in listenTargets. This ensures
            // that we flag the first re-listen this way without impacting future
            // listens of this target (that might happen e.g. on reconnect).
            var requestQueryData = new QueryData(queryData.query, targetId, QueryPurpose.ExistenceFilterMismatch, queryData.sequenceNumber);
            _this.sendWatchRequest(requestQueryData);
        });
        // Finally raise remote event
        return this.syncEngine.applyRemoteEvent(remoteEvent);
    };
    /** Handles an error on a target */
    RemoteStore.prototype.handleTargetError = function (watchChange) {
        var _this = this;
        assert(!!watchChange.cause, 'Handling target error without a cause');
        var error = watchChange.cause;
        var promiseChain = Promise.resolve();
        watchChange.targetIds.forEach(function (targetId) {
            promiseChain = promiseChain.then(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                return tslib_1.__generator(this, function (_a) {
                    // A watched target might have been removed already.
                    if (contains(this.listenTargets, targetId)) {
                        delete this.listenTargets[targetId];
                        this.watchChangeAggregator.removeTarget(targetId);
                        return [2 /*return*/, this.syncEngine.rejectListen(targetId, error)];
                    }
                    return [2 /*return*/];
                });
            }); });
        });
        return promiseChain;
    };
    /**
     * Attempts to fill our write pipeline with writes from the LocalStore.
     *
     * Called internally to bootstrap or refill the write pipeline and by
     * SyncEngine whenever there are new mutations to process.
     *
     * Starts the write stream if necessary.
     */
    RemoteStore.prototype.fillWritePipeline = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var lastBatchIdRetrieved, batch;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!this.canAddToWritePipeline()) return [3 /*break*/, 4];
                        lastBatchIdRetrieved = this.writePipeline.length > 0
                            ? this.writePipeline[this.writePipeline.length - 1].batchId
                            : BATCHID_UNKNOWN;
                        return [4 /*yield*/, this.localStore.nextMutationBatch(lastBatchIdRetrieved)];
                    case 1:
                        batch = _a.sent();
                        if (!(batch === null)) return [3 /*break*/, 2];
                        if (this.writePipeline.length === 0) {
                            this.writeStream.markIdle();
                        }
                        return [3 /*break*/, 4];
                    case 2:
                        this.addToWritePipeline(batch);
                        return [4 /*yield*/, this.fillWritePipeline()];
                    case 3:
                        _a.sent();
                        _a.label = 4;
                    case 4:
                        if (this.shouldStartWriteStream()) {
                            this.startWriteStream();
                        }
                        return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Returns true if we can add to the write pipeline (i.e. the network is
     * enabled and the write pipeline is not full).
     */
    RemoteStore.prototype.canAddToWritePipeline = function () {
        return (this.canUseNetwork() && this.writePipeline.length < MAX_PENDING_WRITES);
    };
    // For testing
    RemoteStore.prototype.outstandingWrites = function () {
        return this.writePipeline.length;
    };
    /**
     * Queues additional writes to be sent to the write stream, sending them
     * immediately if the write stream is established.
     */
    RemoteStore.prototype.addToWritePipeline = function (batch) {
        assert(this.canAddToWritePipeline(), 'addToWritePipeline called when pipeline is full');
        this.writePipeline.push(batch);
        if (this.writeStream.isOpen() && this.writeStream.handshakeComplete) {
            this.writeStream.writeMutations(batch.mutations);
        }
    };
    RemoteStore.prototype.shouldStartWriteStream = function () {
        return (this.canUseNetwork() &&
            !this.writeStream.isStarted() &&
            this.writePipeline.length > 0);
    };
    RemoteStore.prototype.startWriteStream = function () {
        assert(this.shouldStartWriteStream(), 'startWriteStream() called when shouldStartWriteStream() is false.');
        this.writeStream.start();
    };
    RemoteStore.prototype.onWriteStreamOpen = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                this.writeStream.writeHandshake();
                return [2 /*return*/];
            });
        });
    };
    RemoteStore.prototype.onWriteHandshakeComplete = function () {
        var _this = this;
        // Record the stream token.
        return this.localStore
            .setLastStreamToken(this.writeStream.lastStreamToken)
            .then(function () {
            // Send the write pipeline now that the stream is established.
            for (var _i = 0, _a = _this.writePipeline; _i < _a.length; _i++) {
                var batch = _a[_i];
                _this.writeStream.writeMutations(batch.mutations);
            }
        })
            .catch(ignoreIfPrimaryLeaseLoss);
    };
    RemoteStore.prototype.onMutationResult = function (commitVersion, results) {
        var _this = this;
        // This is a response to a write containing mutations and should be
        // correlated to the first write in our write pipeline.
        assert(this.writePipeline.length > 0, 'Got result for empty write pipeline');
        var batch = this.writePipeline.shift();
        var success = MutationBatchResult.from(batch, commitVersion, results, this.writeStream.lastStreamToken);
        return this.syncEngine.applySuccessfulWrite(success).then(function () {
            // It's possible that with the completion of this mutation another
            // slot has freed up.
            return _this.fillWritePipeline();
        });
    };
    RemoteStore.prototype.onWriteStreamClose = function (error) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var errorHandling;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                if (error === undefined) {
                    // Graceful stop (due to stop() or idle timeout). Make sure that's
                    // desirable.
                    assert(!this.shouldStartWriteStream(), 'Write stream was stopped gracefully while still needed.');
                }
                // If the write stream closed due to an error, invoke the error callbacks if
                // there are pending writes.
                if (error && this.writePipeline.length > 0) {
                    errorHandling = void 0;
                    if (this.writeStream.handshakeComplete) {
                        // This error affects the actual write.
                        errorHandling = this.handleWriteError(error);
                    }
                    else {
                        // If there was an error before the handshake has finished, it's
                        // possible that the server is unable to process the stream token
                        // we're sending. (Perhaps it's too old?)
                        errorHandling = this.handleHandshakeError(error);
                    }
                    return [2 /*return*/, errorHandling.then(function () {
                            // The write stream might have been started by refilling the write
                            // pipeline for failed writes
                            if (_this.shouldStartWriteStream()) {
                                _this.startWriteStream();
                            }
                        })];
                }
                return [2 /*return*/];
            });
        });
    };
    RemoteStore.prototype.handleHandshakeError = function (error) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                // Reset the token if it's a permanent error, signaling the write stream is
                // no longer valid. Note that the handshake does not count as a write: see
                // comments on isPermanentWriteError for details.
                if (isPermanentError(error.code)) {
                    debug(LOG_TAG$8, 'RemoteStore error before completed handshake; resetting stream token: ', this.writeStream.lastStreamToken);
                    this.writeStream.lastStreamToken = emptyByteString();
                    return [2 /*return*/, this.localStore
                            .setLastStreamToken(emptyByteString())
                            .catch(ignoreIfPrimaryLeaseLoss)];
                }
                return [2 /*return*/];
            });
        });
    };
    RemoteStore.prototype.handleWriteError = function (error) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var batch;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                // Only handle permanent errors here. If it's transient, just let the retry
                // logic kick in.
                if (isPermanentWriteError(error.code)) {
                    batch = this.writePipeline.shift();
                    // In this case it's also unlikely that the server itself is melting
                    // down -- this was just a bad request so inhibit backoff on the next
                    // restart.
                    this.writeStream.inhibitBackoff();
                    return [2 /*return*/, this.syncEngine
                            .rejectFailedWrite(batch.batchId, error)
                            .then(function () {
                            // It's possible that with the completion of this mutation
                            // another slot has freed up.
                            return _this.fillWritePipeline();
                        })];
                }
                return [2 /*return*/];
            });
        });
    };
    RemoteStore.prototype.createTransaction = function () {
        return new Transaction(this.datastore);
    };
    RemoteStore.prototype.handleCredentialChange = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!this.canUseNetwork()) return [3 /*break*/, 3];
                        // Tear down and re-create our network streams. This will ensure we get a fresh auth token
                        // for the new user and re-fill the write pipeline with new mutations from the LocalStore
                        // (since mutations are per-user).
                        debug(LOG_TAG$8, 'RemoteStore restarting streams for new credential');
                        this.networkEnabled = false;
                        return [4 /*yield*/, this.disableNetworkInternal()];
                    case 1:
                        _a.sent();
                        this.onlineStateTracker.set(OnlineState.Unknown);
                        return [4 /*yield*/, this.enableNetwork()];
                    case 2:
                        _a.sent();
                        _a.label = 3;
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Toggles the network state when the client gains or loses its primary lease.
     */
    RemoteStore.prototype.applyPrimaryState = function (isPrimary) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.isPrimary = isPrimary;
                        if (!(isPrimary && this.networkEnabled)) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.enableNetwork()];
                    case 1:
                        _a.sent();
                        return [3 /*break*/, 4];
                    case 2:
                        if (!!isPrimary) return [3 /*break*/, 4];
                        return [4 /*yield*/, this.disableNetworkInternal()];
                    case 3:
                        _a.sent();
                        this.onlineStateTracker.set(OnlineState.Unknown);
                        _a.label = 4;
                    case 4: return [2 /*return*/];
                }
            });
        });
    };
    return RemoteStore;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Holds the listeners and the last received ViewSnapshot for a query being
 * tracked by EventManager.
 */
var QueryListenersInfo = /** @class */ (function () {
    function QueryListenersInfo() {
        this.listeners = [];
    }
    return QueryListenersInfo;
}());
/**
 * EventManager is responsible for mapping queries to query event emitters.
 * It handles "fan-out". -- Identical queries will re-use the same watch on the
 * backend.
 */
var EventManager = /** @class */ (function () {
    function EventManager(syncEngine) {
        this.syncEngine = syncEngine;
        this.queries = new ObjectMap(function (q) {
            return q.canonicalId();
        });
        this.onlineState = OnlineState.Unknown;
        this.syncEngine.subscribe(this);
    }
    EventManager.prototype.listen = function (listener) {
        var query = listener.query;
        var firstListen = false;
        var queryInfo = this.queries.get(query);
        if (!queryInfo) {
            firstListen = true;
            queryInfo = new QueryListenersInfo();
            this.queries.set(query, queryInfo);
        }
        queryInfo.listeners.push(listener);
        listener.applyOnlineStateChange(this.onlineState);
        if (queryInfo.viewSnap)
            listener.onViewSnapshot(queryInfo.viewSnap);
        if (firstListen) {
            return this.syncEngine.listen(query).then(function (targetId) {
                queryInfo.targetId = targetId;
                return targetId;
            });
        }
        else {
            return Promise.resolve(queryInfo.targetId);
        }
    };
    EventManager.prototype.unlisten = function (listener) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var query, lastListen, queryInfo, i;
            return tslib_1.__generator(this, function (_a) {
                query = listener.query;
                lastListen = false;
                queryInfo = this.queries.get(query);
                if (queryInfo) {
                    i = queryInfo.listeners.indexOf(listener);
                    if (i >= 0) {
                        queryInfo.listeners.splice(i, 1);
                        lastListen = queryInfo.listeners.length === 0;
                    }
                }
                if (lastListen) {
                    this.queries.delete(query);
                    return [2 /*return*/, this.syncEngine.unlisten(query)];
                }
                return [2 /*return*/];
            });
        });
    };
    EventManager.prototype.onWatchChange = function (viewSnaps) {
        for (var _i = 0, viewSnaps_1 = viewSnaps; _i < viewSnaps_1.length; _i++) {
            var viewSnap = viewSnaps_1[_i];
            var query = viewSnap.query;
            var queryInfo = this.queries.get(query);
            if (queryInfo) {
                for (var _a = 0, _b = queryInfo.listeners; _a < _b.length; _a++) {
                    var listener = _b[_a];
                    listener.onViewSnapshot(viewSnap);
                }
                queryInfo.viewSnap = viewSnap;
            }
        }
    };
    EventManager.prototype.onWatchError = function (query, error) {
        var queryInfo = this.queries.get(query);
        if (queryInfo) {
            for (var _i = 0, _a = queryInfo.listeners; _i < _a.length; _i++) {
                var listener = _a[_i];
                listener.onError(error);
            }
        }
        // Remove all listeners. NOTE: We don't need to call syncEngine.unlisten()
        // after an error.
        this.queries.delete(query);
    };
    EventManager.prototype.onOnlineStateChange = function (onlineState) {
        this.onlineState = onlineState;
        this.queries.forEach(function (_, queryInfo) {
            for (var _i = 0, _a = queryInfo.listeners; _i < _a.length; _i++) {
                var listener = _a[_i];
                listener.applyOnlineStateChange(onlineState);
            }
        });
    };
    return EventManager;
}());
/**
 * QueryListener takes a series of internal view snapshots and determines
 * when to raise the event.
 *
 * It uses an Observer to dispatch events.
 */
var QueryListener = /** @class */ (function () {
    function QueryListener(query, queryObserver, options) {
        this.query = query;
        this.queryObserver = queryObserver;
        /**
         * Initial snapshots (e.g. from cache) may not be propagated to the wrapped
         * observer. This flag is set to true once we've actually raised an event.
         */
        this.raisedInitialEvent = false;
        this.onlineState = OnlineState.Unknown;
        this.options = options || {};
    }
    QueryListener.prototype.onViewSnapshot = function (snap) {
        assert(snap.docChanges.length > 0 || snap.syncStateChanged, 'We got a new snapshot with no changes?');
        if (!this.options.includeMetadataChanges) {
            // Remove the metadata only changes.
            var docChanges = [];
            for (var _i = 0, _a = snap.docChanges; _i < _a.length; _i++) {
                var docChange = _a[_i];
                if (docChange.type !== ChangeType.Metadata) {
                    docChanges.push(docChange);
                }
            }
            snap = new ViewSnapshot(snap.query, snap.docs, snap.oldDocs, docChanges, snap.mutatedKeys, snap.fromCache, snap.syncStateChanged, 
            /* excludesMetadataChanges= */ true);
        }
        if (!this.raisedInitialEvent) {
            if (this.shouldRaiseInitialEvent(snap, this.onlineState)) {
                this.raiseInitialEvent(snap);
            }
        }
        else if (this.shouldRaiseEvent(snap)) {
            this.queryObserver.next(snap);
        }
        this.snap = snap;
    };
    QueryListener.prototype.onError = function (error) {
        this.queryObserver.error(error);
    };
    QueryListener.prototype.applyOnlineStateChange = function (onlineState) {
        this.onlineState = onlineState;
        if (this.snap &&
            !this.raisedInitialEvent &&
            this.shouldRaiseInitialEvent(this.snap, onlineState)) {
            this.raiseInitialEvent(this.snap);
        }
    };
    QueryListener.prototype.shouldRaiseInitialEvent = function (snap, onlineState) {
        assert(!this.raisedInitialEvent, 'Determining whether to raise first event but already had first event');
        // Always raise the first event when we're synced
        if (!snap.fromCache) {
            return true;
        }
        // NOTE: We consider OnlineState.Unknown as online (it should become Offline
        // or Online if we wait long enough).
        var maybeOnline = onlineState !== OnlineState.Offline;
        // Don't raise the event if we're online, aren't synced yet (checked
        // above) and are waiting for a sync.
        if (this.options.waitForSyncWhenOnline && maybeOnline) {
            assert(snap.fromCache, 'Waiting for sync, but snapshot is not from cache');
            return false;
        }
        // Raise data from cache if we have any documents or we are offline
        return !snap.docs.isEmpty() || onlineState === OnlineState.Offline;
    };
    QueryListener.prototype.shouldRaiseEvent = function (snap) {
        // We don't need to handle includeDocumentMetadataChanges here because
        // the Metadata only changes have already been stripped out if needed.
        // At this point the only changes we will see are the ones we should
        // propagate.
        if (snap.docChanges.length > 0) {
            return true;
        }
        var hasPendingWritesChanged = this.snap && this.snap.hasPendingWrites !== snap.hasPendingWrites;
        if (snap.syncStateChanged || hasPendingWritesChanged) {
            return this.options.includeMetadataChanges === true;
        }
        // Generally we should have hit one of the cases above, but it's possible
        // to get here if there were only metadata docChanges and they got
        // stripped out.
        return false;
    };
    QueryListener.prototype.raiseInitialEvent = function (snap) {
        assert(!this.raisedInitialEvent, 'Trying to raise initial events for second time');
        snap = ViewSnapshot.fromInitialDocuments(snap.query, snap.docs, snap.mutatedKeys, snap.fromCache);
        this.raisedInitialEvent = true;
        this.queryObserver.next(snap);
    };
    return QueryListener;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * A set of changes to what documents are currently in view and out of view for
 * a given query. These changes are sent to the LocalStore by the View (via
 * the SyncEngine) and are used to pin / unpin documents as appropriate.
 */
var LocalViewChanges = /** @class */ (function () {
    function LocalViewChanges(targetId, addedKeys, removedKeys) {
        this.targetId = targetId;
        this.addedKeys = addedKeys;
        this.removedKeys = removedKeys;
    }
    LocalViewChanges.fromSnapshot = function (targetId, viewSnapshot) {
        var addedKeys = documentKeySet();
        var removedKeys = documentKeySet();
        for (var _i = 0, _a = viewSnapshot.docChanges; _i < _a.length; _i++) {
            var docChange = _a[_i];
            switch (docChange.type) {
                case ChangeType.Added:
                    addedKeys = addedKeys.add(docChange.doc.key);
                    break;
                case ChangeType.Removed:
                    removedKeys = removedKeys.add(docChange.doc.key);
                    break;
                default:
                // do nothing
            }
        }
        return new LocalViewChanges(targetId, addedKeys, removedKeys);
    };
    return LocalViewChanges;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var AddedLimboDocument = /** @class */ (function () {
    function AddedLimboDocument(key) {
        this.key = key;
    }
    return AddedLimboDocument;
}());
var RemovedLimboDocument = /** @class */ (function () {
    function RemovedLimboDocument(key) {
        this.key = key;
    }
    return RemovedLimboDocument;
}());
/**
 * View is responsible for computing the final merged truth of what docs are in
 * a query. It gets notified of local and remote changes to docs, and applies
 * the query filters and limits to determine the most correct possible results.
 */
var View = /** @class */ (function () {
    function View(query, 
    /** Documents included in the remote target */
    _syncedDocuments) {
        this.query = query;
        this._syncedDocuments = _syncedDocuments;
        this.syncState = null;
        /**
         * A flag whether the view is current with the backend. A view is considered
         * current after it has seen the current flag from the backend and did not
         * lose consistency within the watch stream (e.g. because of an existence
         * filter mismatch).
         */
        this.current = false;
        /** Documents in the view but not in the remote target */
        this.limboDocuments = documentKeySet();
        /** Document Keys that have local changes */
        this.mutatedKeys = documentKeySet();
        this.documentSet = new DocumentSet(query.docComparator.bind(query));
    }
    Object.defineProperty(View.prototype, "syncedDocuments", {
        /**
         * The set of remote documents that the server has told us belongs to the target associated with
         * this view.
         */
        get: function () {
            return this._syncedDocuments;
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Iterates over a set of doc changes, applies the query limit, and computes
     * what the new results should be, what the changes were, and whether we may
     * need to go back to the local cache for more results. Does not make any
     * changes to the view.
     * @param docChanges The doc changes to apply to this view.
     * @param previousChanges If this is being called with a refill, then start
     *        with this set of docs and changes instead of the current view.
     * @return a new set of docs, changes, and refill flag.
     */
    View.prototype.computeDocChanges = function (docChanges, previousChanges) {
        var _this = this;
        var changeSet = previousChanges
            ? previousChanges.changeSet
            : new DocumentChangeSet();
        var oldDocumentSet = previousChanges
            ? previousChanges.documentSet
            : this.documentSet;
        var newMutatedKeys = previousChanges
            ? previousChanges.mutatedKeys
            : this.mutatedKeys;
        var newDocumentSet = oldDocumentSet;
        var needsRefill = false;
        // Track the last doc in a (full) limit. This is necessary, because some
        // update (a delete, or an update moving a doc past the old limit) might
        // mean there is some other document in the local cache that either should
        // come (1) between the old last limit doc and the new last document, in the
        // case of updates, or (2) after the new last document, in the case of
        // deletes. So we keep this doc at the old limit to compare the updates to.
        //
        // Note that this should never get used in a refill (when previousChanges is
        // set), because there will only be adds -- no deletes or updates.
        var lastDocInLimit = this.query.hasLimit() && oldDocumentSet.size === this.query.limit
            ? oldDocumentSet.last()
            : null;
        docChanges.inorderTraversal(function (key, newMaybeDoc) {
            var oldDoc = oldDocumentSet.get(key);
            var newDoc = newMaybeDoc instanceof Document ? newMaybeDoc : null;
            if (newDoc) {
                assert(key.isEqual(newDoc.key), 'Mismatching keys found in document changes: ' +
                    key +
                    ' != ' +
                    newDoc.key);
                newDoc = _this.query.matches(newDoc) ? newDoc : null;
            }
            var oldDocHadPendingMutations = oldDoc
                ? _this.mutatedKeys.has(oldDoc.key)
                : false;
            var newDocHasPendingMutations = newDoc
                ? newDoc.hasLocalMutations ||
                    // We only consider committed mutations for documents that were
                    // mutated during the lifetime of the view.
                    (_this.mutatedKeys.has(newDoc.key) && newDoc.hasCommittedMutations)
                : false;
            var changeApplied = false;
            // Calculate change
            if (oldDoc && newDoc) {
                var docsEqual = oldDoc.data.isEqual(newDoc.data);
                if (!docsEqual) {
                    if (!_this.shouldWaitForSyncedDocument(oldDoc, newDoc)) {
                        changeSet.track({
                            type: ChangeType.Modified,
                            doc: newDoc
                        });
                        changeApplied = true;
                        if (lastDocInLimit &&
                            _this.query.docComparator(newDoc, lastDocInLimit) > 0) {
                            // This doc moved from inside the limit to after the limit.
                            // That means there may be some doc in the local cache that's
                            // actually less than this one.
                            needsRefill = true;
                        }
                    }
                }
                else if (oldDocHadPendingMutations !== newDocHasPendingMutations) {
                    changeSet.track({ type: ChangeType.Metadata, doc: newDoc });
                    changeApplied = true;
                }
            }
            else if (!oldDoc && newDoc) {
                changeSet.track({ type: ChangeType.Added, doc: newDoc });
                changeApplied = true;
            }
            else if (oldDoc && !newDoc) {
                changeSet.track({ type: ChangeType.Removed, doc: oldDoc });
                changeApplied = true;
                if (lastDocInLimit) {
                    // A doc was removed from a full limit query. We'll need to
                    // requery from the local cache to see if we know about some other
                    // doc that should be in the results.
                    needsRefill = true;
                }
            }
            if (changeApplied) {
                if (newDoc) {
                    newDocumentSet = newDocumentSet.add(newDoc);
                    if (newDocHasPendingMutations) {
                        newMutatedKeys = newMutatedKeys.add(key);
                    }
                    else {
                        newMutatedKeys = newMutatedKeys.delete(key);
                    }
                }
                else {
                    newDocumentSet = newDocumentSet.delete(key);
                    newMutatedKeys = newMutatedKeys.delete(key);
                }
            }
        });
        if (this.query.hasLimit()) {
            while (newDocumentSet.size > this.query.limit) {
                var oldDoc = newDocumentSet.last();
                newDocumentSet = newDocumentSet.delete(oldDoc.key);
                newMutatedKeys = newMutatedKeys.delete(oldDoc.key);
                changeSet.track({ type: ChangeType.Removed, doc: oldDoc });
            }
        }
        assert(!needsRefill || !previousChanges, 'View was refilled using docs that themselves needed refilling.');
        return {
            documentSet: newDocumentSet,
            changeSet: changeSet,
            needsRefill: needsRefill,
            mutatedKeys: newMutatedKeys
        };
    };
    View.prototype.shouldWaitForSyncedDocument = function (oldDoc, newDoc) {
        // We suppress the initial change event for documents that were modified as
        // part of a write acknowledgment (e.g. when the value of a server transform
        // is applied) as Watch will send us the same document again.
        // By suppressing the event, we only raise two user visible events (one with
        // `hasPendingWrites` and the final state of the document) instead of three
        // (one with `hasPendingWrites`, the modified document with
        // `hasPendingWrites` and the final state of the document).
        return (oldDoc.hasLocalMutations &&
            newDoc.hasCommittedMutations &&
            !newDoc.hasLocalMutations);
    };
    /**
     * Updates the view with the given ViewDocumentChanges and optionally updates
     * limbo docs and sync state from the provided target change.
     * @param docChanges The set of changes to make to the view's docs.
     * @param updateLimboDocuments Whether to update limbo documents based on this
     *        change.
     * @param targetChange A target change to apply for computing limbo docs and
     *        sync state.
     * @return A new ViewChange with the given docs, changes, and sync state.
     */
    // PORTING NOTE: The iOS/Android clients always compute limbo document changes.
    View.prototype.applyChanges = function (docChanges, updateLimboDocuments, targetChange) {
        var _this = this;
        assert(!docChanges.needsRefill, 'Cannot apply changes that need a refill');
        var oldDocs = this.documentSet;
        this.documentSet = docChanges.documentSet;
        this.mutatedKeys = docChanges.mutatedKeys;
        // Sort changes based on type and query comparator
        var changes = docChanges.changeSet.getChanges();
        changes.sort(function (c1, c2) {
            return (compareChangeType(c1.type, c2.type) ||
                _this.query.docComparator(c1.doc, c2.doc));
        });
        this.applyTargetChange(targetChange);
        var limboChanges = updateLimboDocuments
            ? this.updateLimboDocuments()
            : [];
        var synced = this.limboDocuments.size === 0 && this.current;
        var newSyncState = synced ? SyncState.Synced : SyncState.Local;
        var syncStateChanged = newSyncState !== this.syncState;
        this.syncState = newSyncState;
        if (changes.length === 0 && !syncStateChanged) {
            // no changes
            return { limboChanges: limboChanges };
        }
        else {
            var snap = new ViewSnapshot(this.query, docChanges.documentSet, oldDocs, changes, docChanges.mutatedKeys, newSyncState === SyncState.Local, syncStateChanged, 
            /* excludesMetadataChanges= */ false);
            return {
                snapshot: snap,
                limboChanges: limboChanges
            };
        }
    };
    /**
     * Applies an OnlineState change to the view, potentially generating a
     * ViewChange if the view's syncState changes as a result.
     */
    View.prototype.applyOnlineStateChange = function (onlineState) {
        if (this.current && onlineState === OnlineState.Offline) {
            // If we're offline, set `current` to false and then call applyChanges()
            // to refresh our syncState and generate a ViewChange as appropriate. We
            // are guaranteed to get a new TargetChange that sets `current` back to
            // true once the client is back online.
            this.current = false;
            return this.applyChanges({
                documentSet: this.documentSet,
                changeSet: new DocumentChangeSet(),
                mutatedKeys: this.mutatedKeys,
                needsRefill: false
            }, 
            /* updateLimboDocuments= */ false);
        }
        else {
            // No effect, just return a no-op ViewChange.
            return { limboChanges: [] };
        }
    };
    /**
     * Returns whether the doc for the given key should be in limbo.
     */
    View.prototype.shouldBeInLimbo = function (key) {
        // If the remote end says it's part of this query, it's not in limbo.
        if (this._syncedDocuments.has(key)) {
            return false;
        }
        // The local store doesn't think it's a result, so it shouldn't be in limbo.
        if (!this.documentSet.has(key)) {
            return false;
        }
        // If there are local changes to the doc, they might explain why the server
        // doesn't know that it's part of the query. So don't put it in limbo.
        // TODO(klimt): Ideally, we would only consider changes that might actually
        // affect this specific query.
        if (this.documentSet.get(key).hasLocalMutations) {
            return false;
        }
        // Everything else is in limbo.
        return true;
    };
    /**
     * Updates syncedDocuments, current, and limbo docs based on the given change.
     * Returns the list of changes to which docs are in limbo.
     */
    View.prototype.applyTargetChange = function (targetChange) {
        var _this = this;
        if (targetChange) {
            targetChange.addedDocuments.forEach(function (key) { return (_this._syncedDocuments = _this._syncedDocuments.add(key)); });
            targetChange.modifiedDocuments.forEach(function (key) {
                return assert(_this._syncedDocuments.has(key), "Modified document " + key + " not found in view.");
            });
            targetChange.removedDocuments.forEach(function (key) { return (_this._syncedDocuments = _this._syncedDocuments.delete(key)); });
            this.current = targetChange.current;
        }
    };
    View.prototype.updateLimboDocuments = function () {
        var _this = this;
        // We can only determine limbo documents when we're in-sync with the server.
        if (!this.current) {
            return [];
        }
        // TODO(klimt): Do this incrementally so that it's not quadratic when
        // updating many documents.
        var oldLimboDocuments = this.limboDocuments;
        this.limboDocuments = documentKeySet();
        this.documentSet.forEach(function (doc) {
            if (_this.shouldBeInLimbo(doc.key)) {
                _this.limboDocuments = _this.limboDocuments.add(doc.key);
            }
        });
        // Diff the new limbo docs with the old limbo docs.
        var changes = [];
        oldLimboDocuments.forEach(function (key) {
            if (!_this.limboDocuments.has(key)) {
                changes.push(new RemovedLimboDocument(key));
            }
        });
        this.limboDocuments.forEach(function (key) {
            if (!oldLimboDocuments.has(key)) {
                changes.push(new AddedLimboDocument(key));
            }
        });
        return changes;
    };
    /**
     * Update the in-memory state of the current view with the state read from
     * persistence.
     *
     * We update the query view whenever a client's primary status changes:
     * - When a client transitions from primary to secondary, it can miss
     *   LocalStorage updates and its query views may temporarily not be
     *   synchronized with the state on disk.
     * - For secondary to primary transitions, the client needs to update the list
     *   of `syncedDocuments` since secondary clients update their query views
     *   based purely on synthesized RemoteEvents.
     *
     * @param localDocs - The documents that match the query according to the
     * LocalStore.
     * @param remoteKeys - The keys of the documents that match the query
     * according to the backend.
     *
     * @return The ViewChange that resulted from this synchronization.
     */
    // PORTING NOTE: Multi-tab only.
    View.prototype.synchronizeWithPersistedState = function (localDocs, remoteKeys) {
        this._syncedDocuments = remoteKeys;
        this.limboDocuments = documentKeySet();
        var docChanges = this.computeDocChanges(localDocs);
        return this.applyChanges(docChanges, /*updateLimboDocuments=*/ true);
    };
    /**
     * Returns a view snapshot as if this query was just listened to. Contains
     * a document add for every existing document and the `fromCache` and
     * `hasPendingWrites` status of the already established view.
     */
    // PORTING NOTE: Multi-tab only.
    View.prototype.computeInitialSnapshot = function () {
        return ViewSnapshot.fromInitialDocuments(this.query, this.documentSet, this.mutatedKeys, this.syncState === SyncState.Local);
    };
    return View;
}());
function compareChangeType(c1, c2) {
    var order = function (change) {
        switch (change) {
            case ChangeType.Added:
                return 1;
            case ChangeType.Modified:
                return 2;
            case ChangeType.Metadata:
                // A metadata change is converted to a modified change at the public
                // api layer.  Since we sort by document key and then change type,
                // metadata and modified changes must be sorted equivalently.
                return 2;
            case ChangeType.Removed:
                return 0;
            default:
                return fail('Unknown ChangeType: ' + change);
        }
    };
    return order(c1) - order(c2);
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$9 = 'SyncEngine';
/**
 * QueryView contains all of the data that SyncEngine needs to keep track of for
 * a particular query.
 */
var QueryView = /** @class */ (function () {
    function QueryView(
    /**
     * The query itself.
     */
    query, 
    /**
     * The target number created by the client that is used in the watch
     * stream to identify this query.
     */
    targetId, 
    /**
     * The view is responsible for computing the final merged truth of what
     * docs are in the query. It gets notified of local and remote changes,
     * and applies the query filters and limits to determine the most correct
     * possible results.
     */
    view) {
        this.query = query;
        this.targetId = targetId;
        this.view = view;
    }
    return QueryView;
}());
/** Tracks a limbo resolution. */
var LimboResolution = /** @class */ (function () {
    function LimboResolution(key) {
        this.key = key;
    }
    return LimboResolution;
}());
/**
 * SyncEngine is the central controller in the client SDK architecture. It is
 * the glue code between the EventManager, LocalStore, and RemoteStore. Some of
 * SyncEngine's responsibilities include:
 * 1. Coordinating client requests and remote events between the EventManager
 *    and the local and remote data stores.
 * 2. Managing a View object for each query, providing the unified view between
 *    the local and remote data stores.
 * 3. Notifying the RemoteStore when the LocalStore has new mutations in its
 *    queue that need sending to the backend.
 *
 * The SyncEngine’s methods should only ever be called by methods running in the
 * global async queue.
 */
var SyncEngine = /** @class */ (function () {
    function SyncEngine(localStore, remoteStore, 
    // PORTING NOTE: Manages state synchronization in multi-tab environments.
    sharedClientState, currentUser) {
        this.localStore = localStore;
        this.remoteStore = remoteStore;
        this.sharedClientState = sharedClientState;
        this.currentUser = currentUser;
        this.syncEngineListener = null;
        this.queryViewsByQuery = new ObjectMap(function (q) {
            return q.canonicalId();
        });
        this.queryViewsByTarget = {};
        this.limboTargetsByKey = new SortedMap(DocumentKey.comparator);
        this.limboResolutionsByTarget = {};
        this.limboDocumentRefs = new ReferenceSet();
        /** Stores user completion handlers, indexed by User and BatchId. */
        this.mutationUserCallbacks = {};
        this.limboTargetIdGenerator = TargetIdGenerator.forSyncEngine();
        // The primary state is set to `true` or `false` immediately after Firestore
        // startup. In the interim, a client should only be considered primary if
        // `isPrimary` is true.
        this.isPrimary = undefined;
        this.onlineState = OnlineState.Unknown;
    }
    Object.defineProperty(SyncEngine.prototype, "isPrimaryClient", {
        // Only used for testing.
        get: function () {
            return this.isPrimary === true;
        },
        enumerable: true,
        configurable: true
    });
    /** Subscribes to SyncEngine notifications. Has to be called exactly once. */
    SyncEngine.prototype.subscribe = function (syncEngineListener) {
        assert(syncEngineListener !== null, 'SyncEngine listener cannot be null');
        assert(this.syncEngineListener === null, 'SyncEngine already has a subscriber.');
        this.syncEngineListener = syncEngineListener;
    };
    /**
     * Initiates the new listen, resolves promise when listen enqueued to the
     * server. All the subsequent view snapshots or errors are sent to the
     * subscribed handlers. Returns the targetId of the query.
     */
    SyncEngine.prototype.listen = function (query) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var targetId, viewSnapshot, queryView, queryData, status_1;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.assertSubscribed('listen()');
                        queryView = this.queryViewsByQuery.get(query);
                        if (!queryView) return [3 /*break*/, 1];
                        // PORTING NOTE: With Multi-Tab Web, it is possible that a query view
                        // already exists when EventManager calls us for the first time. This
                        // happens when the primary tab is already listening to this query on
                        // behalf of another tab and the user of the primary also starts listening
                        // to the query. EventManager will not have an assigned target ID in this
                        // case and calls `listen` to obtain this ID.
                        targetId = queryView.targetId;
                        this.sharedClientState.addLocalQueryTarget(targetId);
                        viewSnapshot = queryView.view.computeInitialSnapshot();
                        return [3 /*break*/, 4];
                    case 1: return [4 /*yield*/, this.localStore.allocateQuery(query)];
                    case 2:
                        queryData = _a.sent();
                        status_1 = this.sharedClientState.addLocalQueryTarget(queryData.targetId);
                        targetId = queryData.targetId;
                        return [4 /*yield*/, this.initializeViewAndComputeSnapshot(queryData, status_1 === 'current')];
                    case 3:
                        viewSnapshot = _a.sent();
                        if (this.isPrimary) {
                            this.remoteStore.listen(queryData);
                        }
                        _a.label = 4;
                    case 4:
                        this.syncEngineListener.onWatchChange([viewSnapshot]);
                        return [2 /*return*/, targetId];
                }
            });
        });
    };
    /**
     * Registers a view for a previously unknown query and computes its initial
     * snapshot.
     */
    SyncEngine.prototype.initializeViewAndComputeSnapshot = function (queryData, current) {
        var _this = this;
        var query = queryData.query;
        return this.localStore.executeQuery(query).then(function (docs) {
            return _this.localStore
                .remoteDocumentKeys(queryData.targetId)
                .then(function (remoteKeys) {
                var view = new View(query, remoteKeys);
                var viewDocChanges = view.computeDocChanges(docs);
                // tslint:disable-next-line:max-line-length Prettier formats this exceed 100 characters.
                var synthesizedTargetChange = TargetChange.createSynthesizedTargetChangeForCurrentChange(queryData.targetId, current && _this.onlineState !== OnlineState.Offline);
                var viewChange = view.applyChanges(viewDocChanges, 
                /* updateLimboDocuments= */ _this.isPrimary === true, synthesizedTargetChange);
                assert(viewChange.limboChanges.length === 0, 'View returned limbo docs before target ack from the server.');
                assert(!!viewChange.snapshot, 'applyChanges for new view should always return a snapshot');
                var data = new QueryView(query, queryData.targetId, view);
                _this.queryViewsByQuery.set(query, data);
                _this.queryViewsByTarget[queryData.targetId] = data;
                return viewChange.snapshot;
            });
        });
    };
    /**
     * Reconcile the list of synced documents in an existing view with those
     * from persistence.
     */
    // PORTING NOTE: Multi-tab only.
    SyncEngine.prototype.synchronizeViewAndComputeSnapshot = function (queryView) {
        var _this = this;
        return this.localStore.executeQuery(queryView.query).then(function (docs) {
            return _this.localStore
                .remoteDocumentKeys(queryView.targetId)
                .then(function (remoteKeys) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                var viewSnapshot;
                return tslib_1.__generator(this, function (_a) {
                    viewSnapshot = queryView.view.synchronizeWithPersistedState(docs, remoteKeys);
                    if (this.isPrimary) {
                        this.updateTrackedLimbos(queryView.targetId, viewSnapshot.limboChanges);
                    }
                    return [2 /*return*/, viewSnapshot];
                });
            }); });
        });
    };
    /** Stops listening to the query. */
    SyncEngine.prototype.unlisten = function (query) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var queryView, targetRemainsActive;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.assertSubscribed('unlisten()');
                        queryView = this.queryViewsByQuery.get(query);
                        assert(!!queryView, 'Trying to unlisten on query not found:' + query);
                        if (!this.isPrimary) return [3 /*break*/, 3];
                        // We need to remove the local query target first to allow us to verify
                        // whether any other client is still interested in this target.
                        this.sharedClientState.removeLocalQueryTarget(queryView.targetId);
                        targetRemainsActive = this.sharedClientState.isActiveQueryTarget(queryView.targetId);
                        if (!!targetRemainsActive) return [3 /*break*/, 2];
                        return [4 /*yield*/, this.localStore
                                .releaseQuery(query, /*keepPersistedQueryData=*/ false)
                                .then(function () {
                                _this.sharedClientState.clearQueryState(queryView.targetId);
                                _this.remoteStore.unlisten(queryView.targetId);
                                _this.removeAndCleanupQuery(queryView);
                            })
                                .catch(ignoreIfPrimaryLeaseLoss)];
                    case 1:
                        _a.sent();
                        _a.label = 2;
                    case 2: return [3 /*break*/, 5];
                    case 3:
                        this.removeAndCleanupQuery(queryView);
                        return [4 /*yield*/, this.localStore.releaseQuery(query, 
                            /*keepPersistedQueryData=*/ true)];
                    case 4:
                        _a.sent();
                        _a.label = 5;
                    case 5: return [2 /*return*/];
                }
            });
        });
    };
    /**
     * Initiates the write of local mutation batch which involves adding the
     * writes to the mutation queue, notifying the remote store about new
     * mutations and raising events for any changes this write caused.
     *
     * The promise returned by this call is resolved when the above steps
     * have completed, *not* when the write was acked by the backend. The
     * userCallback is resolved once the write was acked/rejected by the
     * backend (or failed locally for any other reason).
     */
    SyncEngine.prototype.write = function (batch, userCallback) {
        var _this = this;
        this.assertSubscribed('write()');
        return this.localStore
            .localWrite(batch)
            .then(function (result) {
            _this.sharedClientState.addPendingMutation(result.batchId);
            _this.addMutationCallback(result.batchId, userCallback);
            return _this.emitNewSnapsAndNotifyLocalStore(result.changes);
        })
            .then(function () {
            return _this.remoteStore.fillWritePipeline();
        });
    };
    // TODO(klimt): Wrap the given error in a standard Firestore error object.
    SyncEngine.prototype.wrapUpdateFunctionError = function (error) {
        return error;
    };
    /**
     * Takes an updateFunction in which a set of reads and writes can be performed
     * atomically. In the updateFunction, the client can read and write values
     * using the supplied transaction object. After the updateFunction, all
     * changes will be committed. If some other client has changed any of the data
     * referenced, then the updateFunction will be called again. If the
     * updateFunction still fails after the given number of retries, then the
     * transaction will be rejection.
     *
     * The transaction object passed to the updateFunction contains methods for
     * accessing documents and collections. Unlike other datastore access, data
     * accessed with the transaction will not reflect local changes that have not
     * been committed. For this reason, it is required that all reads are
     * performed before any writes. Transactions must be performed while online.
     *
     * The promise returned is resolved when the transaction is fully committed.
     */
    SyncEngine.prototype.runTransaction = function (updateFunction, retries) {
        var _this = this;
        if (retries === void 0) { retries = 5; }
        assert(retries >= 0, 'Got negative number of retries for transaction.');
        var transaction = this.remoteStore.createTransaction();
        var wrappedUpdateFunction = function () {
            try {
                var userPromise = updateFunction(transaction);
                if (isNullOrUndefined(userPromise) ||
                    !userPromise.catch ||
                    !userPromise.then) {
                    return Promise.reject(Error('Transaction callback must return a Promise'));
                }
                return userPromise.catch(function (e) {
                    return Promise.reject(_this.wrapUpdateFunctionError(e));
                });
            }
            catch (e) {
                return Promise.reject(_this.wrapUpdateFunctionError(e));
            }
        };
        return wrappedUpdateFunction().then(function (result) {
            return transaction
                .commit()
                .then(function () {
                return result;
            })
                .catch(function (error) {
                if (retries === 0) {
                    return Promise.reject(error);
                }
                // TODO(klimt): Put in a retry delay?
                return _this.runTransaction(updateFunction, retries - 1);
            });
        });
    };
    SyncEngine.prototype.applyRemoteEvent = function (remoteEvent) {
        var _this = this;
        this.assertSubscribed('applyRemoteEvent()');
        return this.localStore
            .applyRemoteEvent(remoteEvent)
            .then(function (changes) {
            // Update `receivedDocument` as appropriate for any limbo targets.
            forEach(remoteEvent.targetChanges, function (targetId, targetChange) {
                var limboResolution = _this.limboResolutionsByTarget[targetId];
                if (limboResolution) {
                    // Since this is a limbo resolution lookup, it's for a single document
                    // and it could be added, modified, or removed, but not a combination.
                    assert(targetChange.addedDocuments.size +
                        targetChange.modifiedDocuments.size +
                        targetChange.removedDocuments.size <=
                        1, 'Limbo resolution for single document contains multiple changes.');
                    if (targetChange.addedDocuments.size > 0) {
                        limboResolution.receivedDocument = true;
                    }
                    else if (targetChange.modifiedDocuments.size > 0) {
                        assert(limboResolution.receivedDocument, 'Received change for limbo target document without add.');
                    }
                    else if (targetChange.removedDocuments.size > 0) {
                        assert(limboResolution.receivedDocument, 'Received remove for limbo target document without add.');
                        limboResolution.receivedDocument = false;
                    }
                }
            });
            return _this.emitNewSnapsAndNotifyLocalStore(changes, remoteEvent);
        })
            .catch(ignoreIfPrimaryLeaseLoss);
    };
    /**
     * Applies an OnlineState change to the sync engine and notifies any views of
     * the change.
     */
    SyncEngine.prototype.applyOnlineStateChange = function (onlineState, source) {
        // If we are the secondary client, we explicitly ignore the remote store's
        // online state (the local client may go offline, even though the primary
        // tab remains online) and only apply the primary tab's online state from
        // SharedClientState.
        if ((this.isPrimary && source === OnlineStateSource.RemoteStore) ||
            (!this.isPrimary && source === OnlineStateSource.SharedClientState)) {
            var newViewSnapshots_1 = [];
            this.queryViewsByQuery.forEach(function (query, queryView) {
                var viewChange = queryView.view.applyOnlineStateChange(onlineState);
                assert(viewChange.limboChanges.length === 0, 'OnlineState should not affect limbo documents.');
                if (viewChange.snapshot) {
                    newViewSnapshots_1.push(viewChange.snapshot);
                }
            });
            this.syncEngineListener.onOnlineStateChange(onlineState);
            this.syncEngineListener.onWatchChange(newViewSnapshots_1);
            this.onlineState = onlineState;
            if (this.isPrimary) {
                this.sharedClientState.setOnlineState(onlineState);
            }
        }
    };
    SyncEngine.prototype.rejectListen = function (targetId, err) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var limboResolution, limboKey, documentUpdates, resolvedLimboDocuments, event_1, queryView_1;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.assertSubscribed('rejectListens()');
                        // PORTING NOTE: Multi-tab only.
                        this.sharedClientState.updateQueryState(targetId, 'rejected', err);
                        limboResolution = this.limboResolutionsByTarget[targetId];
                        limboKey = limboResolution && limboResolution.key;
                        if (!limboKey) return [3 /*break*/, 1];
                        // Since this query failed, we won't want to manually unlisten to it.
                        // So go ahead and remove it from bookkeeping.
                        this.limboTargetsByKey = this.limboTargetsByKey.remove(limboKey);
                        delete this.limboResolutionsByTarget[targetId];
                        documentUpdates = new SortedMap(DocumentKey.comparator);
                        documentUpdates = documentUpdates.insert(limboKey, new NoDocument(limboKey, SnapshotVersion.forDeletedDoc()));
                        resolvedLimboDocuments = documentKeySet().add(limboKey);
                        event_1 = new RemoteEvent(SnapshotVersion.MIN, 
                        /* targetChanges= */ {}, 
                        /* targetMismatches= */ new SortedSet(primitiveComparator), documentUpdates, resolvedLimboDocuments);
                        return [2 /*return*/, this.applyRemoteEvent(event_1)];
                    case 1:
                        queryView_1 = this.queryViewsByTarget[targetId];
                        assert(!!queryView_1, 'Unknown targetId: ' + targetId);
                        return [4 /*yield*/, this.localStore
                                .releaseQuery(queryView_1.query, /* keepPersistedQueryData */ false)
                                .then(function () { return _this.removeAndCleanupQuery(queryView_1); })
                                .catch(ignoreIfPrimaryLeaseLoss)];
                    case 2:
                        _a.sent();
                        this.syncEngineListener.onWatchError(queryView_1.query, err);
                        _a.label = 3;
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    // PORTING NOTE: Multi-tab only
    SyncEngine.prototype.applyBatchState = function (batchId, batchState, error) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var documents;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.assertSubscribed('applyBatchState()');
                        return [4 /*yield*/, this.localStore.lookupMutationDocuments(batchId)];
                    case 1:
                        documents = _a.sent();
                        if (documents === null) {
                            // A throttled tab may not have seen the mutation before it was completed
                            // and removed from the mutation queue, in which case we won't have cached
                            // the affected documents. In this case we can safely ignore the update
                            // since that means we didn't apply the mutation locally at all (if we
                            // had, we would have cached the affected documents), and so we will just
                            // see any resulting document changes via normal remote document updates
                            // as applicable.
                            debug(LOG_TAG$9, 'Cannot apply mutation batch with id: ' + batchId);
                            return [2 /*return*/];
                        }
                        if (!(batchState === 'pending')) return [3 /*break*/, 3];
                        // If we are the primary client, we need to send this write to the
                        // backend. Secondary clients will ignore these writes since their remote
                        // connection is disabled.
                        return [4 /*yield*/, this.remoteStore.fillWritePipeline()];
                    case 2:
                        // If we are the primary client, we need to send this write to the
                        // backend. Secondary clients will ignore these writes since their remote
                        // connection is disabled.
                        _a.sent();
                        return [3 /*break*/, 4];
                    case 3:
                        if (batchState === 'acknowledged' || batchState === 'rejected') {
                            // NOTE: Both these methods are no-ops for batches that originated from
                            // other clients.
                            this.processUserCallback(batchId, error ? error : null);
                            this.localStore.removeCachedMutationBatchMetadata(batchId);
                        }
                        else {
                            fail("Unknown batchState: " + batchState);
                        }
                        _a.label = 4;
                    case 4: return [4 /*yield*/, this.emitNewSnapsAndNotifyLocalStore(documents)];
                    case 5:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    SyncEngine.prototype.applySuccessfulWrite = function (mutationBatchResult) {
        var _this = this;
        this.assertSubscribed('applySuccessfulWrite()');
        var batchId = mutationBatchResult.batch.batchId;
        // The local store may or may not be able to apply the write result and
        // raise events immediately (depending on whether the watcher is caught
        // up), so we raise user callbacks first so that they consistently happen
        // before listen events.
        this.processUserCallback(batchId, /*error=*/ null);
        return this.localStore
            .acknowledgeBatch(mutationBatchResult)
            .then(function (changes) {
            _this.sharedClientState.updateMutationState(batchId, 'acknowledged');
            return _this.emitNewSnapsAndNotifyLocalStore(changes);
        })
            .catch(ignoreIfPrimaryLeaseLoss);
    };
    SyncEngine.prototype.rejectFailedWrite = function (batchId, error) {
        var _this = this;
        this.assertSubscribed('rejectFailedWrite()');
        // The local store may or may not be able to apply the write result and
        // raise events immediately (depending on whether the watcher is caught up),
        // so we raise user callbacks first so that they consistently happen before
        // listen events.
        this.processUserCallback(batchId, error);
        return this.localStore
            .rejectBatch(batchId)
            .then(function (changes) {
            _this.sharedClientState.updateMutationState(batchId, 'rejected', error);
            return _this.emitNewSnapsAndNotifyLocalStore(changes);
        })
            .catch(ignoreIfPrimaryLeaseLoss);
    };
    SyncEngine.prototype.addMutationCallback = function (batchId, callback) {
        var newCallbacks = this.mutationUserCallbacks[this.currentUser.toKey()];
        if (!newCallbacks) {
            newCallbacks = new SortedMap(primitiveComparator);
        }
        newCallbacks = newCallbacks.insert(batchId, callback);
        this.mutationUserCallbacks[this.currentUser.toKey()] = newCallbacks;
    };
    /**
     * Resolves or rejects the user callback for the given batch and then discards
     * it.
     */
    SyncEngine.prototype.processUserCallback = function (batchId, error) {
        var newCallbacks = this.mutationUserCallbacks[this.currentUser.toKey()];
        // NOTE: Mutations restored from persistence won't have callbacks, so it's
        // okay for there to be no callback for this ID.
        if (newCallbacks) {
            var callback = newCallbacks.get(batchId);
            if (callback) {
                assert(batchId === newCallbacks.minKey(), 'Mutation callbacks processed out-of-order?');
                if (error) {
                    callback.reject(error);
                }
                else {
                    callback.resolve();
                }
                newCallbacks = newCallbacks.remove(batchId);
            }
            this.mutationUserCallbacks[this.currentUser.toKey()] = newCallbacks;
        }
    };
    SyncEngine.prototype.removeAndCleanupQuery = function (queryView) {
        var _this = this;
        this.sharedClientState.removeLocalQueryTarget(queryView.targetId);
        this.queryViewsByQuery.delete(queryView.query);
        delete this.queryViewsByTarget[queryView.targetId];
        if (this.isPrimary) {
            var limboKeys = this.limboDocumentRefs.referencesForId(queryView.targetId);
            this.limboDocumentRefs.removeReferencesForId(queryView.targetId);
            limboKeys.forEach(function (limboKey) {
                var isReferenced = _this.limboDocumentRefs.containsKey(limboKey);
                if (!isReferenced) {
                    // We removed the last reference for this key
                    _this.removeLimboTarget(limboKey);
                }
            });
        }
    };
    SyncEngine.prototype.removeLimboTarget = function (key) {
        // It's possible that the target already got removed because the query failed. In that case,
        // the key won't exist in `limboTargetsByKey`. Only do the cleanup if we still have the target.
        var limboTargetId = this.limboTargetsByKey.get(key);
        if (limboTargetId === null) {
            // This target already got removed, because the query failed.
            return;
        }
        this.remoteStore.unlisten(limboTargetId);
        this.limboTargetsByKey = this.limboTargetsByKey.remove(key);
        delete this.limboResolutionsByTarget[limboTargetId];
    };
    SyncEngine.prototype.updateTrackedLimbos = function (targetId, limboChanges) {
        for (var _i = 0, limboChanges_1 = limboChanges; _i < limboChanges_1.length; _i++) {
            var limboChange = limboChanges_1[_i];
            if (limboChange instanceof AddedLimboDocument) {
                this.limboDocumentRefs.addReference(limboChange.key, targetId);
                this.trackLimboChange(limboChange);
            }
            else if (limboChange instanceof RemovedLimboDocument) {
                debug(LOG_TAG$9, 'Document no longer in limbo: ' + limboChange.key);
                this.limboDocumentRefs.removeReference(limboChange.key, targetId);
                var isReferenced = this.limboDocumentRefs.containsKey(limboChange.key);
                if (!isReferenced) {
                    // We removed the last reference for this key
                    this.removeLimboTarget(limboChange.key);
                }
            }
            else {
                fail('Unknown limbo change: ' + JSON.stringify(limboChange));
            }
        }
    };
    SyncEngine.prototype.trackLimboChange = function (limboChange) {
        var key = limboChange.key;
        if (!this.limboTargetsByKey.get(key)) {
            debug(LOG_TAG$9, 'New document in limbo: ' + key);
            var limboTargetId = this.limboTargetIdGenerator.next();
            var query = Query.atPath(key.path);
            this.limboResolutionsByTarget[limboTargetId] = new LimboResolution(key);
            this.remoteStore.listen(new QueryData(query, limboTargetId, QueryPurpose.LimboResolution, ListenSequence.INVALID));
            this.limboTargetsByKey = this.limboTargetsByKey.insert(key, limboTargetId);
        }
    };
    // Visible for testing
    SyncEngine.prototype.currentLimboDocs = function () {
        return this.limboTargetsByKey;
    };
    SyncEngine.prototype.emitNewSnapsAndNotifyLocalStore = function (changes, remoteEvent) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var newSnaps, docChangesInAllViews, queriesProcessed;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        newSnaps = [];
                        docChangesInAllViews = [];
                        queriesProcessed = [];
                        this.queryViewsByQuery.forEach(function (_, queryView) {
                            queriesProcessed.push(Promise.resolve()
                                .then(function () {
                                var viewDocChanges = queryView.view.computeDocChanges(changes);
                                if (!viewDocChanges.needsRefill) {
                                    return viewDocChanges;
                                }
                                // The query has a limit and some docs were removed, so we need
                                // to re-run the query against the local store to make sure we
                                // didn't lose any good docs that had been past the limit.
                                return _this.localStore.executeQuery(queryView.query).then(function (docs) {
                                    return queryView.view.computeDocChanges(docs, viewDocChanges);
                                });
                            })
                                .then(function (viewDocChanges) {
                                var targetChange = remoteEvent && remoteEvent.targetChanges[queryView.targetId];
                                var viewChange = queryView.view.applyChanges(viewDocChanges, 
                                /* updateLimboDocuments= */ _this.isPrimary === true, targetChange);
                                _this.updateTrackedLimbos(queryView.targetId, viewChange.limboChanges);
                                if (viewChange.snapshot) {
                                    if (_this.isPrimary) {
                                        _this.sharedClientState.updateQueryState(queryView.targetId, viewChange.snapshot.fromCache ? 'not-current' : 'current');
                                    }
                                    newSnaps.push(viewChange.snapshot);
                                    var docChanges = LocalViewChanges.fromSnapshot(queryView.targetId, viewChange.snapshot);
                                    docChangesInAllViews.push(docChanges);
                                }
                            }));
                        });
                        return [4 /*yield*/, Promise.all(queriesProcessed)];
                    case 1:
                        _a.sent();
                        this.syncEngineListener.onWatchChange(newSnaps);
                        return [4 /*yield*/, this.localStore.notifyLocalViewChanges(docChangesInAllViews)];
                    case 2:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    SyncEngine.prototype.assertSubscribed = function (fnName) {
        assert(this.syncEngineListener !== null, 'Trying to call ' + fnName + ' before calling subscribe().');
    };
    SyncEngine.prototype.handleCredentialChange = function (user) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var userChanged, result;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        userChanged = !this.currentUser.isEqual(user);
                        this.currentUser = user;
                        if (!userChanged) return [3 /*break*/, 3];
                        return [4 /*yield*/, this.localStore.handleUserChange(user)];
                    case 1:
                        result = _a.sent();
                        // TODO(b/114226417): Consider calling this only in the primary tab.
                        this.sharedClientState.handleUserChange(user, result.removedBatchIds, result.addedBatchIds);
                        return [4 /*yield*/, this.emitNewSnapsAndNotifyLocalStore(result.affectedDocuments)];
                    case 2:
                        _a.sent();
                        _a.label = 3;
                    case 3: return [4 /*yield*/, this.remoteStore.handleCredentialChange()];
                    case 4:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        });
    };
    // PORTING NOTE: Multi-tab only
    SyncEngine.prototype.applyPrimaryState = function (isPrimary) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var activeTargets, activeQueries, _i, activeQueries_1, queryData, activeTargets_1, p_1;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!(isPrimary === true && this.isPrimary !== true)) return [3 /*break*/, 3];
                        this.isPrimary = true;
                        return [4 /*yield*/, this.remoteStore.applyPrimaryState(true)];
                    case 1:
                        _a.sent();
                        activeTargets = this.sharedClientState.getAllActiveQueryTargets();
                        return [4 /*yield*/, this.synchronizeQueryViewsAndRaiseSnapshots(activeTargets.toArray())];
                    case 2:
                        activeQueries = _a.sent();
                        for (_i = 0, activeQueries_1 = activeQueries; _i < activeQueries_1.length; _i++) {
                            queryData = activeQueries_1[_i];
                            this.remoteStore.listen(queryData);
                        }
                        return [3 /*break*/, 7];
                    case 3:
                        if (!(isPrimary === false && this.isPrimary !== false)) return [3 /*break*/, 7];
                        this.isPrimary = false;
                        activeTargets_1 = [];
                        p_1 = Promise.resolve();
                        forEachNumber(this.queryViewsByTarget, function (targetId, queryView) {
                            if (_this.sharedClientState.isLocalQueryTarget(targetId)) {
                                activeTargets_1.push(targetId);
                            }
                            else {
                                p_1 = p_1.then(function () { return _this.unlisten(queryView.query); });
                            }
                            _this.remoteStore.unlisten(queryView.targetId);
                        });
                        return [4 /*yield*/, p_1];
                    case 4:
                        _a.sent();
                        return [4 /*yield*/, this.synchronizeQueryViewsAndRaiseSnapshots(activeTargets_1)];
                    case 5:
                        _a.sent();
                        this.resetLimboDocuments();
                        return [4 /*yield*/, this.remoteStore.applyPrimaryState(false)];
                    case 6:
                        _a.sent();
                        _a.label = 7;
                    case 7: return [2 /*return*/];
                }
            });
        });
    };
    // PORTING NOTE: Multi-tab only.
    SyncEngine.prototype.resetLimboDocuments = function () {
        var _this = this;
        forEachNumber(this.limboResolutionsByTarget, function (targetId) {
            _this.remoteStore.unlisten(targetId);
        });
        this.limboDocumentRefs.removeAllReferences();
        this.limboResolutionsByTarget = [];
        this.limboTargetsByKey = new SortedMap(DocumentKey.comparator);
    };
    /**
     * Reconcile the query views of the provided query targets with the state from
     * persistence. Raises snapshots for any changes that affect the local
     * client and returns the updated state of all target's query data.
     */
    // PORTING NOTE: Multi-tab only.
    SyncEngine.prototype.synchronizeQueryViewsAndRaiseSnapshots = function (targets) {
        var _this = this;
        var p = Promise.resolve();
        var activeQueries = [];
        var newViewSnapshots = [];
        var _loop_1 = function (targetId) {
            p = p.then(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                var queryData, queryView, viewChange, query;
                return tslib_1.__generator(this, function (_a) {
                    switch (_a.label) {
                        case 0:
                            queryView = this.queryViewsByTarget[targetId];
                            if (!queryView) return [3 /*break*/, 4];
                            // For queries that have a local View, we need to update their state
                            // in LocalStore (as the resume token and the snapshot version
                            // might have changed) and reconcile their views with the persisted
                            // state (the list of syncedDocuments may have gotten out of sync).
                            return [4 /*yield*/, this.localStore.releaseQuery(queryView.query, 
                                /*keepPersistedQueryData=*/ true)];
                        case 1:
                            // For queries that have a local View, we need to update their state
                            // in LocalStore (as the resume token and the snapshot version
                            // might have changed) and reconcile their views with the persisted
                            // state (the list of syncedDocuments may have gotten out of sync).
                            _a.sent();
                            return [4 /*yield*/, this.localStore.allocateQuery(queryView.query)];
                        case 2:
                            queryData = _a.sent();
                            return [4 /*yield*/, this.synchronizeViewAndComputeSnapshot(queryView)];
                        case 3:
                            viewChange = _a.sent();
                            if (viewChange.snapshot) {
                                newViewSnapshots.push(viewChange.snapshot);
                            }
                            return [3 /*break*/, 8];
                        case 4:
                            assert(this.isPrimary === true, 'A secondary tab should never have an active query without an active view.');
                            return [4 /*yield*/, this.localStore.getQueryForTarget(targetId)];
                        case 5:
                            query = _a.sent();
                            assert(!!query, "Query data for target " + targetId + " not found");
                            return [4 /*yield*/, this.localStore.allocateQuery(query)];
                        case 6:
                            queryData = _a.sent();
                            return [4 /*yield*/, this.initializeViewAndComputeSnapshot(queryData, 
                                /*current=*/ false)];
                        case 7:
                            _a.sent();
                            _a.label = 8;
                        case 8:
                            activeQueries.push(queryData);
                            return [2 /*return*/];
                    }
                });
            }); });
        };
        for (var _i = 0, targets_1 = targets; _i < targets_1.length; _i++) {
            var targetId = targets_1[_i];
            _loop_1(targetId);
        }
        return p.then(function () {
            _this.syncEngineListener.onWatchChange(newViewSnapshots);
            return activeQueries;
        });
    };
    // PORTING NOTE: Multi-tab only
    SyncEngine.prototype.getActiveClients = function () {
        return this.localStore.getActiveClients();
    };
    // PORTING NOTE: Multi-tab only
    SyncEngine.prototype.applyTargetState = function (targetId, state, error) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _a, queryView;
            var _this = this;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (this.isPrimary) {
                            // If we receive a target state notification via WebStorage, we are
                            // either already secondary or another tab has taken the primary lease.
                            debug(LOG_TAG$9, 'Ignoring unexpected query state notification.');
                            return [2 /*return*/];
                        }
                        if (!this.queryViewsByTarget[targetId]) return [3 /*break*/, 5];
                        _a = state;
                        switch (_a) {
                            case 'current': return [3 /*break*/, 1];
                            case 'not-current': return [3 /*break*/, 1];
                            case 'rejected': return [3 /*break*/, 2];
                        }
                        return [3 /*break*/, 4];
                    case 1:
                        {
                            return [2 /*return*/, this.localStore.getNewDocumentChanges().then(function (changes) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                                    var synthesizedRemoteEvent;
                                    return tslib_1.__generator(this, function (_a) {
                                        switch (_a.label) {
                                            case 0:
                                                synthesizedRemoteEvent = RemoteEvent.createSynthesizedRemoteEventForCurrentChange(targetId, state === 'current');
                                                return [4 /*yield*/, this.emitNewSnapsAndNotifyLocalStore(changes, synthesizedRemoteEvent)];
                                            case 1:
                                                _a.sent();
                                                return [2 /*return*/];
                                        }
                                    });
                                }); }, function (err) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                                    var activeTargets_2;
                                    return tslib_1.__generator(this, function (_a) {
                                        switch (_a.label) {
                                            case 0:
                                                if (!isDocumentChangeMissingError(err)) return [3 /*break*/, 2];
                                                activeTargets_2 = [];
                                                forEachNumber(this.queryViewsByTarget, function (target) {
                                                    return activeTargets_2.push(target);
                                                });
                                                return [4 /*yield*/, this.synchronizeQueryViewsAndRaiseSnapshots(activeTargets_2)];
                                            case 1:
                                                _a.sent();
                                                return [3 /*break*/, 3];
                                            case 2: throw err;
                                            case 3: return [2 /*return*/];
                                        }
                                    });
                                }); })];
                        }
                        _b.label = 2;
                    case 2:
                        queryView = this.queryViewsByTarget[targetId];
                        this.removeAndCleanupQuery(queryView);
                        return [4 /*yield*/, this.localStore.releaseQuery(queryView.query, 
                            /*keepPersistedQueryData=*/ true)];
                    case 3:
                        _b.sent();
                        this.syncEngineListener.onWatchError(queryView.query, error);
                        return [3 /*break*/, 5];
                    case 4:
                        fail('Unexpected target state: ' + state);
                        _b.label = 5;
                    case 5: return [2 /*return*/];
                }
            });
        });
    };
    // PORTING NOTE: Multi-tab only
    SyncEngine.prototype.applyActiveTargetsChange = function (added, removed) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var _i, added_1, targetId, query, queryData, _loop_2, this_1, _a, removed_1, targetId;
            var _this = this;
            return tslib_1.__generator(this, function (_b) {
                switch (_b.label) {
                    case 0:
                        if (!this.isPrimary) {
                            return [2 /*return*/];
                        }
                        _i = 0, added_1 = added;
                        _b.label = 1;
                    case 1:
                        if (!(_i < added_1.length)) return [3 /*break*/, 6];
                        targetId = added_1[_i];
                        assert(!this.queryViewsByTarget[targetId], 'Trying to add an already active target');
                        return [4 /*yield*/, this.localStore.getQueryForTarget(targetId)];
                    case 2:
                        query = _b.sent();
                        assert(!!query, "Query data for active target " + targetId + " not found");
                        return [4 /*yield*/, this.localStore.allocateQuery(query)];
                    case 3:
                        queryData = _b.sent();
                        return [4 /*yield*/, this.initializeViewAndComputeSnapshot(queryData, 
                            /*current=*/ false)];
                    case 4:
                        _b.sent();
                        this.remoteStore.listen(queryData);
                        _b.label = 5;
                    case 5:
                        _i++;
                        return [3 /*break*/, 1];
                    case 6:
                        _loop_2 = function (targetId) {
                            var queryView;
                            return tslib_1.__generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        queryView = this_1.queryViewsByTarget[targetId];
                                        if (!queryView) return [3 /*break*/, 2];
                                        return [4 /*yield*/, this_1.localStore
                                                .releaseQuery(queryView.query, /*keepPersistedQueryData=*/ false)
                                                .then(function () {
                                                _this.remoteStore.unlisten(targetId);
                                                _this.removeAndCleanupQuery(queryView);
                                            })
                                                .catch(ignoreIfPrimaryLeaseLoss)];
                                    case 1:
                                        _a.sent();
                                        _a.label = 2;
                                    case 2: return [2 /*return*/];
                                }
                            });
                        };
                        this_1 = this;
                        _a = 0, removed_1 = removed;
                        _b.label = 7;
                    case 7:
                        if (!(_a < removed_1.length)) return [3 /*break*/, 10];
                        targetId = removed_1[_a];
                        return [5 /*yield**/, _loop_2(targetId)];
                    case 8:
                        _b.sent();
                        _b.label = 9;
                    case 9:
                        _a++;
                        return [3 /*break*/, 7];
                    case 10: return [2 /*return*/];
                }
            });
        });
    };
    // PORTING NOTE: Multi-tab only. In other clients, LocalStore is unaware of
    // the online state.
    SyncEngine.prototype.enableNetwork = function () {
        this.localStore.setNetworkEnabled(true);
        return this.remoteStore.enableNetwork();
    };
    // PORTING NOTE: Multi-tab only. In other clients, LocalStore is unaware of
    // the online state.
    SyncEngine.prototype.disableNetwork = function () {
        this.localStore.setNetworkEnabled(false);
        return this.remoteStore.disableNetwork();
    };
    SyncEngine.prototype.getRemoteKeysForTarget = function (targetId) {
        var limboResolution = this.limboResolutionsByTarget[targetId];
        if (limboResolution && limboResolution.receivedDocument) {
            return documentKeySet().add(limboResolution.key);
        }
        else {
            return this.queryViewsByTarget[targetId]
                ? this.queryViewsByTarget[targetId].view.syncedDocuments
                : documentKeySet();
        }
    };
    return SyncEngine;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Simple wrapper around a nullable UID. Mostly exists to make code more
 * readable.
 */
var User = /** @class */ (function () {
    function User(uid) {
        this.uid = uid;
    }
    User.prototype.isAuthenticated = function () {
        return this.uid != null;
    };
    /**
     * Returns a key representing this user, suitable for inclusion in a
     * dictionary.
     */
    User.prototype.toKey = function () {
        if (this.isAuthenticated()) {
            return 'uid:' + this.uid;
        }
        else {
            return 'anonymous-user';
        }
    };
    User.prototype.isEqual = function (otherUser) {
        return otherUser.uid === this.uid;
    };
    /** A user with a null UID. */
    User.UNAUTHENTICATED = new User(null);
    // TODO(mikelehen): Look into getting a proper uid-equivalent for
    // non-FirebaseAuth providers.
    User.GOOGLE_CREDENTIALS = new User('google-credentials-uid');
    User.FIRST_PARTY = new User('first-party-uid');
    return User;
}());

/**
 * @license
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$a = 'SharedClientState';
// The format of the LocalStorage key that stores the client state is:
//     firestore_clients_<persistence_prefix>_<instance_key>
var CLIENT_STATE_KEY_PREFIX = 'firestore_clients';
// The format of the WebStorage key that stores the mutation state is:
//     firestore_mutations_<persistence_prefix>_<batch_id>
//     (for unauthenticated users)
// or: firestore_mutations_<persistence_prefix>_<batch_id>_<user_uid>
//
// 'user_uid' is last to avoid needing to escape '_' characters that it might
// contain.
var MUTATION_BATCH_KEY_PREFIX = 'firestore_mutations';
// The format of the WebStorage key that stores a query target's metadata is:
//     firestore_targets_<persistence_prefix>_<target_id>
var QUERY_TARGET_KEY_PREFIX = 'firestore_targets';
// The WebStorage prefix that stores the primary tab's online state. The
// format of the key is:
//     firestore_online_state_<persistence_prefix>
var ONLINE_STATE_KEY_PREFIX = 'firestore_online_state';
// The WebStorage key prefix for the key that stores the last sequence number allocated. The key
// looks like 'firestore_sequence_number_<persistence_prefix>'.
var SEQUENCE_NUMBER_KEY_PREFIX = 'firestore_sequence_number';
/**
 * Holds the state of a mutation batch, including its user ID, batch ID and
 * whether the batch is 'pending', 'acknowledged' or 'rejected'.
 */
// Visible for testing
var MutationMetadata = /** @class */ (function () {
    function MutationMetadata(user, batchId, state, error) {
        this.user = user;
        this.batchId = batchId;
        this.state = state;
        this.error = error;
        assert((error !== undefined) === (state === 'rejected'), "MutationMetadata must contain an error iff state is 'rejected'");
    }
    /**
     * Parses a MutationMetadata from its JSON representation in WebStorage.
     * Logs a warning and returns null if the format of the data is not valid.
     */
    MutationMetadata.fromWebStorageEntry = function (user, batchId, value) {
        var mutationBatch = JSON.parse(value);
        var validData = typeof mutationBatch === 'object' &&
            ['pending', 'acknowledged', 'rejected'].indexOf(mutationBatch.state) !==
                -1 &&
            (mutationBatch.error === undefined ||
                typeof mutationBatch.error === 'object');
        var firestoreError = undefined;
        if (validData && mutationBatch.error) {
            validData =
                typeof mutationBatch.error.message === 'string' &&
                    typeof mutationBatch.error.code === 'string';
            if (validData) {
                firestoreError = new FirestoreError(mutationBatch.error.code, mutationBatch.error.message);
            }
        }
        if (validData) {
            return new MutationMetadata(user, batchId, mutationBatch.state, firestoreError);
        }
        else {
            error(LOG_TAG$a, "Failed to parse mutation state for ID '" + batchId + "': " + value);
            return null;
        }
    };
    MutationMetadata.prototype.toWebStorageJSON = function () {
        var batchMetadata = {
            state: this.state,
            updateTimeMs: Date.now() // Modify the existing value to trigger update.
        };
        if (this.error) {
            batchMetadata.error = {
                code: this.error.code,
                message: this.error.message
            };
        }
        return JSON.stringify(batchMetadata);
    };
    return MutationMetadata;
}());
/**
 * Holds the state of a query target, including its target ID and whether the
 * target is 'not-current', 'current' or 'rejected'.
 */
// Visible for testing
var QueryTargetMetadata = /** @class */ (function () {
    function QueryTargetMetadata(targetId, state, error) {
        this.targetId = targetId;
        this.state = state;
        this.error = error;
        assert((error !== undefined) === (state === 'rejected'), "QueryTargetMetadata must contain an error iff state is 'rejected'");
    }
    /**
     * Parses a QueryTargetMetadata from its JSON representation in WebStorage.
     * Logs a warning and returns null if the format of the data is not valid.
     */
    QueryTargetMetadata.fromWebStorageEntry = function (targetId, value) {
        var targetState = JSON.parse(value);
        var validData = typeof targetState === 'object' &&
            ['not-current', 'current', 'rejected'].indexOf(targetState.state) !==
                -1 &&
            (targetState.error === undefined ||
                typeof targetState.error === 'object');
        var firestoreError = undefined;
        if (validData && targetState.error) {
            validData =
                typeof targetState.error.message === 'string' &&
                    typeof targetState.error.code === 'string';
            if (validData) {
                firestoreError = new FirestoreError(targetState.error.code, targetState.error.message);
            }
        }
        if (validData) {
            return new QueryTargetMetadata(targetId, targetState.state, firestoreError);
        }
        else {
            error(LOG_TAG$a, "Failed to parse target state for ID '" + targetId + "': " + value);
            return null;
        }
    };
    QueryTargetMetadata.prototype.toWebStorageJSON = function () {
        var targetState = {
            state: this.state,
            updateTimeMs: Date.now() // Modify the existing value to trigger update.
        };
        if (this.error) {
            targetState.error = {
                code: this.error.code,
                message: this.error.message
            };
        }
        return JSON.stringify(targetState);
    };
    return QueryTargetMetadata;
}());
/**
 * This class represents the immutable ClientState for a client read from
 * WebStorage, containing the list of active query targets.
 */
var RemoteClientState = /** @class */ (function () {
    function RemoteClientState(clientId, activeTargetIds) {
        this.clientId = clientId;
        this.activeTargetIds = activeTargetIds;
    }
    /**
     * Parses a RemoteClientState from the JSON representation in WebStorage.
     * Logs a warning and returns null if the format of the data is not valid.
     */
    RemoteClientState.fromWebStorageEntry = function (clientId, value) {
        var clientState = JSON.parse(value);
        var validData = typeof clientState === 'object' &&
            clientState.activeTargetIds instanceof Array;
        var activeTargetIdsSet = targetIdSet();
        for (var i = 0; validData && i < clientState.activeTargetIds.length; ++i) {
            validData = isSafeInteger(clientState.activeTargetIds[i]);
            activeTargetIdsSet = activeTargetIdsSet.add(clientState.activeTargetIds[i]);
        }
        if (validData) {
            return new RemoteClientState(clientId, activeTargetIdsSet);
        }
        else {
            error(LOG_TAG$a, "Failed to parse client data for instance '" + clientId + "': " + value);
            return null;
        }
    };
    return RemoteClientState;
}());
/**
 * This class represents the online state for all clients participating in
 * multi-tab. The online state is only written to by the primary client, and
 * used in secondary clients to update their query views.
 */
var SharedOnlineState = /** @class */ (function () {
    function SharedOnlineState(clientId, onlineState) {
        this.clientId = clientId;
        this.onlineState = onlineState;
    }
    /**
     * Parses a SharedOnlineState from its JSON representation in WebStorage.
     * Logs a warning and returns null if the format of the data is not valid.
     */
    SharedOnlineState.fromWebStorageEntry = function (value) {
        var onlineState = JSON.parse(value);
        var validData = typeof onlineState === 'object' &&
            OnlineState[onlineState.onlineState] !== undefined &&
            typeof onlineState.clientId === 'string';
        if (validData) {
            return new SharedOnlineState(onlineState.clientId, OnlineState[onlineState.onlineState]);
        }
        else {
            error(LOG_TAG$a, "Failed to parse online state: " + value);
            return null;
        }
    };
    return SharedOnlineState;
}());
/**
 * Metadata state of the local client. Unlike `RemoteClientState`, this class is
 * mutable and keeps track of all pending mutations, which allows us to
 * update the range of pending mutation batch IDs as new mutations are added or
 * removed.
 *
 * The data in `LocalClientState` is not read from WebStorage and instead
 * updated via its instance methods. The updated state can be serialized via
 * `toWebStorageJSON()`.
 */
// Visible for testing.
var LocalClientState = /** @class */ (function () {
    function LocalClientState() {
        this.activeTargetIds = targetIdSet();
    }
    LocalClientState.prototype.addQueryTarget = function (targetId) {
        assert(!this.activeTargetIds.has(targetId), "Target with ID '" + targetId + "' already active.");
        this.activeTargetIds = this.activeTargetIds.add(targetId);
    };
    LocalClientState.prototype.removeQueryTarget = function (targetId) {
        this.activeTargetIds = this.activeTargetIds.delete(targetId);
    };
    /**
     * Converts this entry into a JSON-encoded format we can use for WebStorage.
     * Does not encode `clientId` as it is part of the key in WebStorage.
     */
    LocalClientState.prototype.toWebStorageJSON = function () {
        var data = {
            activeTargetIds: this.activeTargetIds.toArray(),
            updateTimeMs: Date.now() // Modify the existing value to trigger update.
        };
        return JSON.stringify(data);
    };
    return LocalClientState;
}());
/**
 * `WebStorageSharedClientState` uses WebStorage (window.localStorage) as the
 * backing store for the SharedClientState. It keeps track of all active
 * clients and supports modifications of the local client's data.
 */
var WebStorageSharedClientState = /** @class */ (function () {
    function WebStorageSharedClientState(queue, platform, persistenceKey, localClientId, initialUser) {
        this.queue = queue;
        this.platform = platform;
        this.persistenceKey = persistenceKey;
        this.localClientId = localClientId;
        this.syncEngine = null;
        this.onlineStateHandler = null;
        this.sequenceNumberHandler = null;
        this.activeClients = {};
        this.storageListener = this.handleWebStorageEvent.bind(this);
        this.started = false;
        /**
         * Captures WebStorage events that occur before `start()` is called. These
         * events are replayed once `WebStorageSharedClientState` is started.
         */
        this.earlyEvents = [];
        if (!WebStorageSharedClientState.isAvailable(this.platform)) {
            throw new FirestoreError(Code.UNIMPLEMENTED, 'LocalStorage is not available on this platform.');
        }
        // Escape the special characters mentioned here:
        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions
        var escapedPersistenceKey = persistenceKey.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        this.storage = this.platform.window.localStorage;
        this.currentUser = initialUser;
        this.localClientStorageKey = this.toWebStorageClientStateKey(this.localClientId);
        this.sequenceNumberKey = SEQUENCE_NUMBER_KEY_PREFIX + "_" + persistenceKey;
        this.activeClients[this.localClientId] = new LocalClientState();
        this.clientStateKeyRe = new RegExp("^" + CLIENT_STATE_KEY_PREFIX + "_" + escapedPersistenceKey + "_([^_]*)$");
        this.mutationBatchKeyRe = new RegExp("^" + MUTATION_BATCH_KEY_PREFIX + "_" + escapedPersistenceKey + "_(\\d+)(?:_(.*))?$");
        this.queryTargetKeyRe = new RegExp("^" + QUERY_TARGET_KEY_PREFIX + "_" + escapedPersistenceKey + "_(\\d+)$");
        this.onlineStateKey = ONLINE_STATE_KEY_PREFIX + "_" + persistenceKey;
        // Rather than adding the storage observer during start(), we add the
        // storage observer during initialization. This ensures that we collect
        // events before other components populate their initial state (during their
        // respective start() calls). Otherwise, we might for example miss a
        // mutation that is added after LocalStore's start() processed the existing
        // mutations but before we observe WebStorage events.
        this.platform.window.addEventListener('storage', this.storageListener);
    }
    /** Returns 'true' if WebStorage is available in the current environment. */
    WebStorageSharedClientState.isAvailable = function (platform) {
        return !!(platform.window && platform.window.localStorage != null);
    };
    WebStorageSharedClientState.prototype.start = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            var existingClients, _i, existingClients_1, clientId, storageItem, clientState, onlineStateJSON, onlineState, _a, _b, event_1;
            var _this = this;
            return tslib_1.__generator(this, function (_c) {
                switch (_c.label) {
                    case 0:
                        assert(!this.started, 'WebStorageSharedClientState already started');
                        assert(this.syncEngine !== null, 'syncEngine property must be set before calling start()');
                        assert(this.onlineStateHandler !== null, 'onlineStateHandler property must be set before calling start()');
                        return [4 /*yield*/, this.syncEngine.getActiveClients()];
                    case 1:
                        existingClients = _c.sent();
                        for (_i = 0, existingClients_1 = existingClients; _i < existingClients_1.length; _i++) {
                            clientId = existingClients_1[_i];
                            if (clientId === this.localClientId) {
                                continue;
                            }
                            storageItem = this.getItem(this.toWebStorageClientStateKey(clientId));
                            if (storageItem) {
                                clientState = RemoteClientState.fromWebStorageEntry(clientId, storageItem);
                                if (clientState) {
                                    this.activeClients[clientState.clientId] = clientState;
                                }
                            }
                        }
                        this.persistClientState();
                        onlineStateJSON = this.storage.getItem(this.onlineStateKey);
                        if (onlineStateJSON) {
                            onlineState = this.fromWebStorageOnlineState(onlineStateJSON);
                            if (onlineState) {
                                this.handleOnlineStateEvent(onlineState);
                            }
                        }
                        for (_a = 0, _b = this.earlyEvents; _a < _b.length; _a++) {
                            event_1 = _b[_a];
                            this.handleWebStorageEvent(event_1);
                        }
                        this.earlyEvents = [];
                        // Register a window unload hook to remove the client metadata entry from
                        // WebStorage even if `shutdown()` was not called.
                        this.platform.window.addEventListener('unload', function () { return _this.shutdown(); });
                        this.started = true;
                        return [2 /*return*/];
                }
            });
        });
    };
    WebStorageSharedClientState.prototype.writeSequenceNumber = function (sequenceNumber) {
        this.setItem(this.sequenceNumberKey, JSON.stringify(sequenceNumber));
    };
    WebStorageSharedClientState.prototype.getAllActiveQueryTargets = function () {
        var activeTargets = targetIdSet();
        forEach(this.activeClients, function (key, value) {
            activeTargets = activeTargets.unionWith(value.activeTargetIds);
        });
        return activeTargets;
    };
    WebStorageSharedClientState.prototype.isActiveQueryTarget = function (targetId) {
        // This is not using `obj.forEach` since `forEach` doesn't support early
        // return.
        for (var clientId in this.activeClients) {
            if (this.activeClients.hasOwnProperty(clientId)) {
                if (this.activeClients[clientId].activeTargetIds.has(targetId)) {
                    return true;
                }
            }
        }
        return false;
    };
    WebStorageSharedClientState.prototype.addPendingMutation = function (batchId) {
        this.persistMutationState(batchId, 'pending');
    };
    WebStorageSharedClientState.prototype.updateMutationState = function (batchId, state, error) {
        this.persistMutationState(batchId, state, error);
        // Once a final mutation result is observed by other clients, they no longer
        // access the mutation's metadata entry. Since WebStorage replays events
        // in order, it is safe to delete the entry right after updating it.
        this.removeMutationState(batchId);
    };
    WebStorageSharedClientState.prototype.addLocalQueryTarget = function (targetId) {
        var queryState = 'not-current';
        // Lookup an existing query state if the target ID was already registered
        // by another tab
        if (this.isActiveQueryTarget(targetId)) {
            var storageItem = this.storage.getItem(this.toWebStorageQueryTargetMetadataKey(targetId));
            if (storageItem) {
                var metadata = QueryTargetMetadata.fromWebStorageEntry(targetId, storageItem);
                if (metadata) {
                    queryState = metadata.state;
                }
            }
        }
        this.localClientState.addQueryTarget(targetId);
        this.persistClientState();
        return queryState;
    };
    WebStorageSharedClientState.prototype.removeLocalQueryTarget = function (targetId) {
        this.localClientState.removeQueryTarget(targetId);
        this.persistClientState();
    };
    WebStorageSharedClientState.prototype.isLocalQueryTarget = function (targetId) {
        return this.localClientState.activeTargetIds.has(targetId);
    };
    WebStorageSharedClientState.prototype.clearQueryState = function (targetId) {
        this.removeItem(this.toWebStorageQueryTargetMetadataKey(targetId));
    };
    WebStorageSharedClientState.prototype.updateQueryState = function (targetId, state, error) {
        this.persistQueryTargetState(targetId, state, error);
    };
    WebStorageSharedClientState.prototype.handleUserChange = function (user, removedBatchIds, addedBatchIds) {
        var _this = this;
        removedBatchIds.forEach(function (batchId) {
            _this.removeMutationState(batchId);
        });
        this.currentUser = user;
        addedBatchIds.forEach(function (batchId) {
            _this.addPendingMutation(batchId);
        });
    };
    WebStorageSharedClientState.prototype.setOnlineState = function (onlineState) {
        this.persistOnlineState(onlineState);
    };
    WebStorageSharedClientState.prototype.shutdown = function () {
        if (this.started) {
            this.platform.window.removeEventListener('storage', this.storageListener);
            this.removeItem(this.localClientStorageKey);
            this.started = false;
        }
    };
    WebStorageSharedClientState.prototype.getItem = function (key) {
        var value = this.storage.getItem(key);
        debug(LOG_TAG$a, 'READ', key, value);
        return value;
    };
    WebStorageSharedClientState.prototype.setItem = function (key, value) {
        debug(LOG_TAG$a, 'SET', key, value);
        this.storage.setItem(key, value);
    };
    WebStorageSharedClientState.prototype.removeItem = function (key) {
        debug(LOG_TAG$a, 'REMOVE', key);
        this.storage.removeItem(key);
    };
    WebStorageSharedClientState.prototype.handleWebStorageEvent = function (event) {
        var _this = this;
        if (event.storageArea === this.storage) {
            debug(LOG_TAG$a, 'EVENT', event.key, event.newValue);
            if (event.key === this.localClientStorageKey) {
                error('Received WebStorage notification for local change. Another client might have ' +
                    'garbage-collected our state');
                return;
            }
            this.queue.enqueueAndForget(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                var clientState, clientId, mutationMetadata, queryTargetMetadata, onlineState, sequenceNumber;
                return tslib_1.__generator(this, function (_a) {
                    if (!this.started) {
                        this.earlyEvents.push(event);
                        return [2 /*return*/];
                    }
                    if (event.key === null) {
                        return [2 /*return*/];
                    }
                    if (this.clientStateKeyRe.test(event.key)) {
                        if (event.newValue != null) {
                            clientState = this.fromWebStorageClientState(event.key, event.newValue);
                            if (clientState) {
                                return [2 /*return*/, this.handleClientStateEvent(clientState.clientId, clientState)];
                            }
                        }
                        else {
                            clientId = this.fromWebStorageClientStateKey(event.key);
                            return [2 /*return*/, this.handleClientStateEvent(clientId, null)];
                        }
                    }
                    else if (this.mutationBatchKeyRe.test(event.key)) {
                        if (event.newValue !== null) {
                            mutationMetadata = this.fromWebStorageMutationMetadata(event.key, event.newValue);
                            if (mutationMetadata) {
                                return [2 /*return*/, this.handleMutationBatchEvent(mutationMetadata)];
                            }
                        }
                    }
                    else if (this.queryTargetKeyRe.test(event.key)) {
                        if (event.newValue !== null) {
                            queryTargetMetadata = this.fromWebStorageQueryTargetMetadata(event.key, event.newValue);
                            if (queryTargetMetadata) {
                                return [2 /*return*/, this.handleQueryTargetEvent(queryTargetMetadata)];
                            }
                        }
                    }
                    else if (event.key === this.onlineStateKey) {
                        if (event.newValue !== null) {
                            onlineState = this.fromWebStorageOnlineState(event.newValue);
                            if (onlineState) {
                                return [2 /*return*/, this.handleOnlineStateEvent(onlineState)];
                            }
                        }
                    }
                    else if (event.key === this.sequenceNumberKey) {
                        assert(!!this.sequenceNumberHandler, 'Missing sequenceNumberHandler');
                        sequenceNumber = fromWebStorageSequenceNumber(event.newValue);
                        if (sequenceNumber !== ListenSequence.INVALID) {
                            this.sequenceNumberHandler(sequenceNumber);
                        }
                    }
                    return [2 /*return*/];
                });
            }); });
        }
    };
    Object.defineProperty(WebStorageSharedClientState.prototype, "localClientState", {
        get: function () {
            return this.activeClients[this.localClientId];
        },
        enumerable: true,
        configurable: true
    });
    WebStorageSharedClientState.prototype.persistClientState = function () {
        this.setItem(this.localClientStorageKey, this.localClientState.toWebStorageJSON());
    };
    WebStorageSharedClientState.prototype.persistMutationState = function (batchId, state, error) {
        var mutationState = new MutationMetadata(this.currentUser, batchId, state, error);
        var mutationKey = this.toWebStorageMutationBatchKey(batchId);
        this.setItem(mutationKey, mutationState.toWebStorageJSON());
    };
    WebStorageSharedClientState.prototype.removeMutationState = function (batchId) {
        var mutationKey = this.toWebStorageMutationBatchKey(batchId);
        this.removeItem(mutationKey);
    };
    WebStorageSharedClientState.prototype.persistOnlineState = function (onlineState) {
        var entry = {
            clientId: this.localClientId,
            onlineState: OnlineState[onlineState]
        };
        this.storage.setItem(this.onlineStateKey, JSON.stringify(entry));
    };
    WebStorageSharedClientState.prototype.persistQueryTargetState = function (targetId, state, error) {
        var targetKey = this.toWebStorageQueryTargetMetadataKey(targetId);
        var targetMetadata = new QueryTargetMetadata(targetId, state, error);
        this.setItem(targetKey, targetMetadata.toWebStorageJSON());
    };
    /** Assembles the key for a client state in WebStorage */
    WebStorageSharedClientState.prototype.toWebStorageClientStateKey = function (clientId) {
        assert(clientId.indexOf('_') === -1, "Client key cannot contain '_', but was '" + clientId + "'");
        return CLIENT_STATE_KEY_PREFIX + "_" + this.persistenceKey + "_" + clientId;
    };
    /** Assembles the key for a query state in WebStorage */
    WebStorageSharedClientState.prototype.toWebStorageQueryTargetMetadataKey = function (targetId) {
        return QUERY_TARGET_KEY_PREFIX + "_" + this.persistenceKey + "_" + targetId;
    };
    /** Assembles the key for a mutation batch in WebStorage */
    WebStorageSharedClientState.prototype.toWebStorageMutationBatchKey = function (batchId) {
        var mutationKey = MUTATION_BATCH_KEY_PREFIX + "_" + this.persistenceKey + "_" + batchId;
        if (this.currentUser.isAuthenticated()) {
            mutationKey += "_" + this.currentUser.uid;
        }
        return mutationKey;
    };
    /**
     * Parses a client state key in WebStorage. Returns null if the key does not
     * match the expected key format.
     */
    WebStorageSharedClientState.prototype.fromWebStorageClientStateKey = function (key) {
        var match = this.clientStateKeyRe.exec(key);
        return match ? match[1] : null;
    };
    /**
     * Parses a client state in WebStorage. Returns 'null' if the value could not
     * be parsed.
     */
    WebStorageSharedClientState.prototype.fromWebStorageClientState = function (key, value) {
        var clientId = this.fromWebStorageClientStateKey(key);
        assert(clientId !== null, "Cannot parse client state key '" + key + "'");
        return RemoteClientState.fromWebStorageEntry(clientId, value);
    };
    /**
     * Parses a mutation batch state in WebStorage. Returns 'null' if the value
     * could not be parsed.
     */
    WebStorageSharedClientState.prototype.fromWebStorageMutationMetadata = function (key, value) {
        var match = this.mutationBatchKeyRe.exec(key);
        assert(match !== null, "Cannot parse mutation batch key '" + key + "'");
        var batchId = Number(match[1]);
        var userId = match[2] !== undefined ? match[2] : null;
        return MutationMetadata.fromWebStorageEntry(new User(userId), batchId, value);
    };
    /**
     * Parses a query target state from WebStorage. Returns 'null' if the value
     * could not be parsed.
     */
    WebStorageSharedClientState.prototype.fromWebStorageQueryTargetMetadata = function (key, value) {
        var match = this.queryTargetKeyRe.exec(key);
        assert(match !== null, "Cannot parse query target key '" + key + "'");
        var targetId = Number(match[1]);
        return QueryTargetMetadata.fromWebStorageEntry(targetId, value);
    };
    /**
     * Parses an online state from WebStorage. Returns 'null' if the value
     * could not be parsed.
     */
    WebStorageSharedClientState.prototype.fromWebStorageOnlineState = function (value) {
        return SharedOnlineState.fromWebStorageEntry(value);
    };
    WebStorageSharedClientState.prototype.handleMutationBatchEvent = function (mutationBatch) {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (mutationBatch.user.uid !== this.currentUser.uid) {
                    debug(LOG_TAG$a, "Ignoring mutation for non-active user " + mutationBatch.user.uid);
                    return [2 /*return*/];
                }
                return [2 /*return*/, this.syncEngine.applyBatchState(mutationBatch.batchId, mutationBatch.state, mutationBatch.error)];
            });
        });
    };
    WebStorageSharedClientState.prototype.handleQueryTargetEvent = function (targetMetadata) {
        return this.syncEngine.applyTargetState(targetMetadata.targetId, targetMetadata.state, targetMetadata.error);
    };
    WebStorageSharedClientState.prototype.handleClientStateEvent = function (clientId, clientState) {
        var _this = this;
        var existingTargets = this.getAllActiveQueryTargets();
        if (clientState) {
            this.activeClients[clientId] = clientState;
        }
        else {
            delete this.activeClients[clientId];
        }
        var newTargets = this.getAllActiveQueryTargets();
        var addedTargets = [];
        var removedTargets = [];
        newTargets.forEach(function (targetId) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!existingTargets.has(targetId)) {
                    addedTargets.push(targetId);
                }
                return [2 /*return*/];
            });
        }); });
        existingTargets.forEach(function (targetId) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                if (!newTargets.has(targetId)) {
                    removedTargets.push(targetId);
                }
                return [2 /*return*/];
            });
        }); });
        return this.syncEngine.applyActiveTargetsChange(addedTargets, removedTargets);
    };
    WebStorageSharedClientState.prototype.handleOnlineStateEvent = function (onlineState) {
        // We check whether the client that wrote this online state is still active
        // by comparing its client ID to the list of clients kept active in
        // IndexedDb. If a client does not update their IndexedDb client state
        // within 5 seconds, it is considered inactive and we don't emit an online
        // state event.
        if (this.activeClients[onlineState.clientId]) {
            this.onlineStateHandler(onlineState.onlineState);
        }
    };
    return WebStorageSharedClientState;
}());
function fromWebStorageSequenceNumber(seqString) {
    var sequenceNumber = ListenSequence.INVALID;
    if (seqString != null) {
        try {
            var parsed = JSON.parse(seqString);
            assert(typeof parsed === 'number', 'Found non-numeric sequence number');
            sequenceNumber = parsed;
        }
        catch (e) {
            error(LOG_TAG$a, 'Failed to read sequence number from WebStorage', e);
        }
    }
    return sequenceNumber;
}
/**
 * `MemorySharedClientState` is a simple implementation of SharedClientState for
 * clients using memory persistence. The state in this class remains fully
 * isolated and no synchronization is performed.
 */
var MemorySharedClientState = /** @class */ (function () {
    function MemorySharedClientState() {
        this.localState = new LocalClientState();
        this.queryState = {};
        this.syncEngine = null;
        this.onlineStateHandler = null;
        this.sequenceNumberHandler = null;
    }
    MemorySharedClientState.prototype.addPendingMutation = function (batchId) {
        // No op.
    };
    MemorySharedClientState.prototype.updateMutationState = function (batchId, state, error) {
        // No op.
    };
    MemorySharedClientState.prototype.addLocalQueryTarget = function (targetId) {
        this.localState.addQueryTarget(targetId);
        return this.queryState[targetId] || 'not-current';
    };
    MemorySharedClientState.prototype.updateQueryState = function (targetId, state, error) {
        this.queryState[targetId] = state;
    };
    MemorySharedClientState.prototype.removeLocalQueryTarget = function (targetId) {
        this.localState.removeQueryTarget(targetId);
    };
    MemorySharedClientState.prototype.isLocalQueryTarget = function (targetId) {
        return this.localState.activeTargetIds.has(targetId);
    };
    MemorySharedClientState.prototype.clearQueryState = function (targetId) {
        delete this.queryState[targetId];
    };
    MemorySharedClientState.prototype.getAllActiveQueryTargets = function () {
        return this.localState.activeTargetIds;
    };
    MemorySharedClientState.prototype.isActiveQueryTarget = function (targetId) {
        return this.localState.activeTargetIds.has(targetId);
    };
    MemorySharedClientState.prototype.start = function () {
        this.localState = new LocalClientState();
        return Promise.resolve();
    };
    MemorySharedClientState.prototype.handleUserChange = function (user, removedBatchIds, addedBatchIds) {
        // No op.
    };
    MemorySharedClientState.prototype.setOnlineState = function (onlineState) {
        // No op.
    };
    MemorySharedClientState.prototype.shutdown = function () { };
    MemorySharedClientState.prototype.writeSequenceNumber = function (sequenceNumber) { };
    return MemorySharedClientState;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var LOG_TAG$b = 'FirestoreClient';
/** The DOMException code for an aborted operation. */
var DOM_EXCEPTION_ABORTED = 20;
/** The DOMException code for quota exceeded. */
var DOM_EXCEPTION_QUOTA_EXCEEDED = 22;
var IndexedDbPersistenceSettings = /** @class */ (function () {
    function IndexedDbPersistenceSettings(cacheSizeBytes, experimentalTabSynchronization) {
        this.cacheSizeBytes = cacheSizeBytes;
        this.experimentalTabSynchronization = experimentalTabSynchronization;
    }
    IndexedDbPersistenceSettings.prototype.lruParams = function () {
        return LruParams.withCacheSize(this.cacheSizeBytes);
    };
    return IndexedDbPersistenceSettings;
}());
var MemoryPersistenceSettings = /** @class */ (function () {
    function MemoryPersistenceSettings() {
    }
    return MemoryPersistenceSettings;
}());
/**
 * FirestoreClient is a top-level class that constructs and owns all of the
 * pieces of the client SDK architecture. It is responsible for creating the
 * async queue that is shared by all of the other components in the system.
 */
var FirestoreClient = /** @class */ (function () {
    function FirestoreClient(platform, databaseInfo, credentials, 
    /**
     * Asynchronous queue responsible for all of our internal processing. When
     * we get incoming work from the user (via public API) or the network
     * (incoming GRPC messages), we should always schedule onto this queue.
     * This ensures all of our work is properly serialized (e.g. we don't
     * start processing a new operation while the previous one is waiting for
     * an async I/O to complete).
     */
    asyncQueue) {
        this.platform = platform;
        this.databaseInfo = databaseInfo;
        this.credentials = credentials;
        this.asyncQueue = asyncQueue;
        this.clientId = AutoId.newId();
    }
    /**
     * Starts up the FirestoreClient, returning only whether or not enabling
     * persistence succeeded.
     *
     * The intent here is to "do the right thing" as far as users are concerned.
     * Namely, in cases where offline persistence is requested and possible,
     * enable it, but otherwise fall back to persistence disabled. For the most
     * part we expect this to succeed one way or the other so we don't expect our
     * users to actually wait on the firestore.enablePersistence Promise since
     * they generally won't care.
     *
     * Of course some users actually do care about whether or not persistence
     * was successfully enabled, so the Promise returned from this method
     * indicates this outcome.
     *
     * This presents a problem though: even before enablePersistence resolves or
     * rejects, users may have made calls to e.g. firestore.collection() which
     * means that the FirestoreClient in there will be available and will be
     * enqueuing actions on the async queue.
     *
     * Meanwhile any failure of an operation on the async queue causes it to
     * panic and reject any further work, on the premise that unhandled errors
     * are fatal.
     *
     * Consequently the fallback is handled internally here in start, and if the
     * fallback succeeds we signal success to the async queue even though the
     * start() itself signals failure.
     *
     * @param persistenceSettings Settings object to configure offline
     *     persistence.
     * @returns A deferred result indicating the user-visible result of enabling
     *     offline persistence. This method will reject this if IndexedDB fails to
     *     start for any reason. If usePersistence is false this is
     *     unconditionally resolved.
     */
    FirestoreClient.prototype.start = function (persistenceSettings) {
        var _this = this;
        // We defer our initialization until we get the current user from
        // setChangeListener(). We block the async queue until we got the initial
        // user and the initialization is completed. This will prevent any scheduled
        // work from happening before initialization is completed.
        //
        // If initializationDone resolved then the FirestoreClient is in a usable
        // state.
        var initializationDone = new Deferred();
        // If usePersistence is true, certain classes of errors while starting are
        // recoverable but only by falling back to persistence disabled.
        //
        // If there's an error in the first case but not in recovery we cannot
        // reject the promise blocking the async queue because this will cause the
        // async queue to panic.
        var persistenceResult = new Deferred();
        var initialized = false;
        this.credentials.setChangeListener(function (user) {
            if (!initialized) {
                initialized = true;
                _this.initializePersistence(persistenceSettings, persistenceResult, user)
                    .then(function (maybeLruGc) { return _this.initializeRest(user, maybeLruGc); })
                    .then(initializationDone.resolve, initializationDone.reject);
            }
            else {
                _this.asyncQueue.enqueueAndForget(function () {
                    return _this.handleCredentialChange(user);
                });
            }
        });
        // Block the async queue until initialization is done
        this.asyncQueue.enqueueAndForget(function () {
            return initializationDone.promise;
        });
        // Return only the result of enabling persistence. Note that this does not
        // need to await the completion of initializationDone because the result of
        // this method should not reflect any other kind of failure to start.
        return persistenceResult.promise;
    };
    /** Enables the network connection and requeues all pending operations. */
    FirestoreClient.prototype.enableNetwork = function () {
        var _this = this;
        return this.asyncQueue.enqueue(function () {
            return _this.syncEngine.enableNetwork();
        });
    };
    /**
     * Initializes persistent storage, attempting to use IndexedDB if
     * usePersistence is true or memory-only if false.
     *
     * If IndexedDB fails because it's already open in another tab or because the
     * platform can't possibly support our implementation then this method rejects
     * the persistenceResult and falls back on memory-only persistence.
     *
     * @param persistenceSettings Settings object to configure offline persistence
     * @param persistenceResult A deferred result indicating the user-visible
     *     result of enabling offline persistence. This method will reject this if
     *     IndexedDB fails to start for any reason. If usePersistence is false
     *     this is unconditionally resolved.
     * @returns a Promise indicating whether or not initialization should
     *     continue, i.e. that one of the persistence implementations actually
     *     succeeded.
     */
    FirestoreClient.prototype.initializePersistence = function (persistenceSettings, persistenceResult, user) {
        var _this = this;
        if (persistenceSettings instanceof IndexedDbPersistenceSettings) {
            return this.startIndexedDbPersistence(user, persistenceSettings)
                .then(function (maybeLruGc) {
                persistenceResult.resolve();
                return maybeLruGc;
            })
                .catch(function (error) {
                // Regardless of whether or not the retry succeeds, from an user
                // perspective, offline persistence has failed.
                persistenceResult.reject(error);
                // An unknown failure on the first stage shuts everything down.
                if (!_this.canFallback(error)) {
                    throw error;
                }
                console.warn('Error enabling offline storage. Falling back to' +
                    ' storage disabled: ' +
                    error);
                return _this.startMemoryPersistence();
            });
        }
        else {
            // When usePersistence == false, enabling offline persistence is defined
            // to unconditionally succeed. This allows start() to have the same
            // signature for both cases, despite the fact that the returned promise
            // is only used in the enablePersistence call.
            persistenceResult.resolve();
            return this.startMemoryPersistence();
        }
    };
    /**
     * Decides whether the provided error allows us to gracefully disable
     * persistence (as opposed to crashing the client).
     */
    FirestoreClient.prototype.canFallback = function (error) {
        if (error instanceof FirestoreError) {
            return (error.code === Code.FAILED_PRECONDITION ||
                error.code === Code.UNIMPLEMENTED);
        }
        else if (typeof DOMException !== 'undefined' &&
            error instanceof DOMException) {
            // We fall back to memory persistence if we cannot write the primary
            // lease. This can happen can during a schema migration, or if we run out
            // of quota when we try to write the primary lease.
            // For both the `QuotaExceededError` and the  `AbortError`, it is safe to
            // fall back to memory persistence since all modifications to IndexedDb
            // failed to commit.
            return (error.code === DOM_EXCEPTION_QUOTA_EXCEEDED ||
                error.code === DOM_EXCEPTION_ABORTED);
        }
        return true;
    };
    /**
     * Starts IndexedDB-based persistence.
     *
     * @returns A promise indicating success or failure.
     */
    FirestoreClient.prototype.startIndexedDbPersistence = function (user, settings) {
        var _this = this;
        // TODO(http://b/33384523): For now we just disable garbage collection
        // when persistence is enabled.
        var storagePrefix = IndexedDbPersistence.buildStoragePrefix(this.databaseInfo);
        // Opt to use proto3 JSON in case the platform doesn't support Uint8Array.
        var serializer = new JsonProtoSerializer(this.databaseInfo.databaseId, {
            useProto3Json: true
        });
        return Promise.resolve().then(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            var persistence, lruParams;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (settings.experimentalTabSynchronization &&
                            !WebStorageSharedClientState.isAvailable(this.platform)) {
                            throw new FirestoreError(Code.UNIMPLEMENTED, 'IndexedDB persistence is only available on platforms that support LocalStorage.');
                        }
                        lruParams = settings.lruParams();
                        if (!settings.experimentalTabSynchronization) return [3 /*break*/, 2];
                        this.sharedClientState = new WebStorageSharedClientState(this.asyncQueue, this.platform, storagePrefix, this.clientId, user);
                        return [4 /*yield*/, IndexedDbPersistence.createMultiClientIndexedDbPersistence(storagePrefix, this.clientId, this.platform, this.asyncQueue, serializer, lruParams, { sequenceNumberSyncer: this.sharedClientState })];
                    case 1:
                        persistence = _a.sent();
                        return [3 /*break*/, 4];
                    case 2:
                        this.sharedClientState = new MemorySharedClientState();
                        return [4 /*yield*/, IndexedDbPersistence.createIndexedDbPersistence(storagePrefix, this.clientId, this.platform, this.asyncQueue, serializer, lruParams)];
                    case 3:
                        persistence = _a.sent();
                        _a.label = 4;
                    case 4:
                        this.persistence = persistence;
                        return [2 /*return*/, persistence.referenceDelegate.garbageCollector];
                }
            });
        }); });
    };
    /**
     * Starts Memory-backed persistence. In practice this cannot fail.
     *
     * @returns A promise that will successfully resolve.
     */
    FirestoreClient.prototype.startMemoryPersistence = function () {
        this.persistence = MemoryPersistence.createEagerPersistence(this.clientId);
        this.sharedClientState = new MemorySharedClientState();
        return Promise.resolve(null);
    };
    /**
     * Initializes the rest of the FirestoreClient, assuming the initial user
     * has been obtained from the credential provider and some persistence
     * implementation is available in this.persistence.
     */
    FirestoreClient.prototype.initializeRest = function (user, maybeLruGc) {
        var _this = this;
        debug(LOG_TAG$b, 'Initializing. user=', user.uid);
        return this.platform
            .loadConnection(this.databaseInfo)
            .then(function (connection) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            var serializer, datastore, remoteStoreOnlineStateChangedHandler, sharedClientStateOnlineStateChangedHandler;
            var _this = this;
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        this.localStore = new LocalStore(this.persistence, user);
                        if (maybeLruGc) {
                            // We're running LRU Garbage collection. Set up the scheduler.
                            this.lruScheduler = new LruScheduler(maybeLruGc, this.asyncQueue, this.localStore);
                        }
                        serializer = this.platform.newSerializer(this.databaseInfo.databaseId);
                        datastore = new Datastore(this.asyncQueue, connection, this.credentials, serializer);
                        remoteStoreOnlineStateChangedHandler = function (onlineState) {
                            return _this.syncEngine.applyOnlineStateChange(onlineState, OnlineStateSource.RemoteStore);
                        };
                        sharedClientStateOnlineStateChangedHandler = function (onlineState) {
                            return _this.syncEngine.applyOnlineStateChange(onlineState, OnlineStateSource.SharedClientState);
                        };
                        this.remoteStore = new RemoteStore(this.localStore, datastore, this.asyncQueue, remoteStoreOnlineStateChangedHandler);
                        this.syncEngine = new SyncEngine(this.localStore, this.remoteStore, this.sharedClientState, user);
                        this.sharedClientState.onlineStateHandler = sharedClientStateOnlineStateChangedHandler;
                        // Set up wiring between sync engine and other components
                        this.remoteStore.syncEngine = this.syncEngine;
                        this.sharedClientState.syncEngine = this.syncEngine;
                        this.eventMgr = new EventManager(this.syncEngine);
                        // PORTING NOTE: LocalStore doesn't need an explicit start() on the Web.
                        return [4 /*yield*/, this.sharedClientState.start()];
                    case 1:
                        // PORTING NOTE: LocalStore doesn't need an explicit start() on the Web.
                        _a.sent();
                        return [4 /*yield*/, this.remoteStore.start()];
                    case 2:
                        _a.sent();
                        // NOTE: This will immediately call the listener, so we make sure to
                        // set it after localStore / remoteStore are started.
                        return [4 /*yield*/, this.persistence.setPrimaryStateListener(function (isPrimary) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                                return tslib_1.__generator(this, function (_a) {
                                    switch (_a.label) {
                                        case 0: return [4 /*yield*/, this.syncEngine.applyPrimaryState(isPrimary)];
                                        case 1:
                                            _a.sent();
                                            if (this.lruScheduler) {
                                                if (isPrimary && !this.lruScheduler.started) {
                                                    this.lruScheduler.start();
                                                }
                                                else if (!isPrimary) {
                                                    this.lruScheduler.stop();
                                                }
                                            }
                                            return [2 /*return*/];
                                    }
                                });
                            }); })];
                    case 3:
                        // NOTE: This will immediately call the listener, so we make sure to
                        // set it after localStore / remoteStore are started.
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        }); });
    };
    FirestoreClient.prototype.handleCredentialChange = function (user) {
        this.asyncQueue.verifyOperationInProgress();
        debug(LOG_TAG$b, 'Credential Changed. Current user: ' + user.uid);
        return this.syncEngine.handleCredentialChange(user);
    };
    /** Disables the network connection. Pending operations will not complete. */
    FirestoreClient.prototype.disableNetwork = function () {
        var _this = this;
        return this.asyncQueue.enqueue(function () {
            return _this.syncEngine.disableNetwork();
        });
    };
    FirestoreClient.prototype.shutdown = function (options) {
        var _this = this;
        return this.asyncQueue.enqueue(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        // PORTING NOTE: LocalStore does not need an explicit shutdown on web.
                        if (this.lruScheduler) {
                            this.lruScheduler.stop();
                        }
                        return [4 /*yield*/, this.remoteStore.shutdown()];
                    case 1:
                        _a.sent();
                        return [4 /*yield*/, this.sharedClientState.shutdown()];
                    case 2:
                        _a.sent();
                        return [4 /*yield*/, this.persistence.shutdown(options && options.purgePersistenceWithDataLoss)];
                    case 3:
                        _a.sent();
                        // `removeChangeListener` must be called after shutting down the
                        // RemoteStore as it will prevent the RemoteStore from retrieving
                        // auth tokens.
                        this.credentials.removeChangeListener();
                        return [2 /*return*/];
                }
            });
        }); });
    };
    FirestoreClient.prototype.listen = function (query, observer, options) {
        var _this = this;
        var listener = new QueryListener(query, observer, options);
        this.asyncQueue.enqueueAndForget(function () {
            return _this.eventMgr.listen(listener);
        });
        return listener;
    };
    FirestoreClient.prototype.unlisten = function (listener) {
        var _this = this;
        this.asyncQueue.enqueueAndForget(function () {
            return _this.eventMgr.unlisten(listener);
        });
    };
    FirestoreClient.prototype.getDocumentFromLocalCache = function (docKey) {
        var _this = this;
        return this.asyncQueue
            .enqueue(function () {
            return _this.localStore.readDocument(docKey);
        })
            .then(function (maybeDoc) {
            if (maybeDoc instanceof Document) {
                return maybeDoc;
            }
            else if (maybeDoc instanceof NoDocument) {
                return null;
            }
            else {
                throw new FirestoreError(Code.UNAVAILABLE, 'Failed to get document from cache. (However, this document may ' +
                    "exist on the server. Run again without setting 'source' in " +
                    'the GetOptions to attempt to retrieve the document from the ' +
                    'server.)');
            }
        });
    };
    FirestoreClient.prototype.getDocumentsFromLocalCache = function (query) {
        var _this = this;
        return this.asyncQueue
            .enqueue(function () {
            return _this.localStore.executeQuery(query);
        })
            .then(function (docs) {
            var remoteKeys = documentKeySet();
            var view = new View(query, remoteKeys);
            var viewDocChanges = view.computeDocChanges(docs);
            return view.applyChanges(viewDocChanges, 
            /* updateLimboDocuments= */ false).snapshot;
        });
    };
    FirestoreClient.prototype.write = function (mutations) {
        var _this = this;
        var deferred = new Deferred();
        this.asyncQueue.enqueueAndForget(function () {
            return _this.syncEngine.write(mutations, deferred);
        });
        return deferred.promise;
    };
    FirestoreClient.prototype.databaseId = function () {
        return this.databaseInfo.databaseId;
    };
    FirestoreClient.prototype.transaction = function (updateFunction) {
        var _this = this;
        // We have to wait for the async queue to be sure syncEngine is initialized.
        return this.asyncQueue
            .enqueue(function () { return tslib_1.__awaiter(_this, void 0, void 0, function () { return tslib_1.__generator(this, function (_a) {
            return [2 /*return*/];
        }); }); })
            .then(function () { return _this.syncEngine.runTransaction(updateFunction); });
    };
    return FirestoreClient;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * A wrapper implementation of Observer<T> that will dispatch events
 * asynchronously. To allow immediate silencing, a mute call is added which
 * causes events scheduled to no longer be raised.
 */
var AsyncObserver = /** @class */ (function () {
    function AsyncObserver(observer) {
        this.observer = observer;
        /**
         * When set to true, will not raise future events. Necessary to deal with
         * async detachment of listener.
         */
        this.muted = false;
    }
    AsyncObserver.prototype.next = function (value) {
        this.scheduleEvent(this.observer.next, value);
    };
    AsyncObserver.prototype.error = function (error) {
        this.scheduleEvent(this.observer.error, error);
    };
    AsyncObserver.prototype.mute = function () {
        this.muted = true;
    };
    AsyncObserver.prototype.scheduleEvent = function (eventHandler, event) {
        var _this = this;
        if (!this.muted) {
            setTimeout(function () {
                if (!_this.muted) {
                    eventHandler(event);
                }
            }, 0);
        }
    };
    return AsyncObserver;
}());

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// The objects that are a part of this API are exposed to third-parties as
// compiled javascript so we want to flag our private members with a leading
// underscore to discourage their use.
// tslint:disable:strip-private-property-underscore
/**
 * A FieldPath refers to a field in a document. The path may consist of a single
 * field name (referring to a top-level field in the document), or a list of
 * field names (referring to a nested field in the document).
 */
var FieldPath$1 = /** @class */ (function () {
    /**
     * Creates a FieldPath from the provided field names. If more than one field
     * name is provided, the path will point to a nested field in a document.
     *
     * @param fieldNames A list of field names.
     */
    function FieldPath$1() {
        var fieldNames = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            fieldNames[_i] = arguments[_i];
        }
        validateNamedArrayAtLeastNumberOfElements('FieldPath', fieldNames, 'fieldNames', 1);
        for (var i = 0; i < fieldNames.length; ++i) {
            validateArgType('FieldPath', 'string', i, fieldNames[i]);
            if (fieldNames[i].length === 0) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid field name at argument $(i + 1). " +
                    'Field names must not be empty.');
            }
        }
        this._internalPath = new FieldPath(fieldNames);
    }
    FieldPath$1.documentId = function () {
        return FieldPath$1._DOCUMENT_ID;
    };
    FieldPath$1.prototype.isEqual = function (other) {
        if (!(other instanceof FieldPath$1)) {
            throw invalidClassError('isEqual', 'FieldPath', 1, other);
        }
        return this._internalPath.isEqual(other._internalPath);
    };
    /**
     * Internal Note: The backend doesn't technically support querying by
     * document ID. Instead it queries by the entire document name (full path
     * included), but in the cases we currently support documentId(), the net
     * effect is the same.
     */
    FieldPath$1._DOCUMENT_ID = new FieldPath$1(FieldPath.keyField().canonicalString());
    return FieldPath$1;
}());
/**
 * Matches any characters in a field path string that are reserved.
 */
var RESERVED = new RegExp('[~\\*/\\[\\]]');
/**
 * Parses a field path string into a FieldPath, treating dots as separators.
 */
function fromDotSeparatedString(path) {
    var found = path.search(RESERVED);
    if (found >= 0) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid field path (" + path + "). Paths must not contain " +
            "'~', '*', '/', '[', or ']'");
    }
    try {
        return new (FieldPath$1.bind.apply(FieldPath$1, [void 0].concat(path.split('.'))))();
    }
    catch (e) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid field path (" + path + "). Paths must not be empty, " +
            "begin with '.', end with '.', or contain '..'");
    }
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var OAuthToken = /** @class */ (function () {
    function OAuthToken(value, user) {
        this.user = user;
        this.type = 'OAuth';
        this.authHeaders = { Authorization: "Bearer " + value };
    }
    return OAuthToken;
}());
/** A CredentialsProvider that always yields an empty token. */
var EmptyCredentialsProvider = /** @class */ (function () {
    function EmptyCredentialsProvider() {
        /**
         * Stores the listener registered with setChangeListener()
         * This isn't actually necessary since the UID never changes, but we use this
         * to verify the listen contract is adhered to in tests.
         */
        this.changeListener = null;
    }
    EmptyCredentialsProvider.prototype.getToken = function () {
        return Promise.resolve(null);
    };
    EmptyCredentialsProvider.prototype.invalidateToken = function () { };
    EmptyCredentialsProvider.prototype.setChangeListener = function (changeListener) {
        assert(!this.changeListener, 'Can only call setChangeListener() once.');
        this.changeListener = changeListener;
        // Fire with initial user.
        changeListener(User.UNAUTHENTICATED);
    };
    EmptyCredentialsProvider.prototype.removeChangeListener = function () {
        assert(this.changeListener !== null, 'removeChangeListener() when no listener registered');
        this.changeListener = null;
    };
    return EmptyCredentialsProvider;
}());
var FirebaseCredentialsProvider = /** @class */ (function () {
    function FirebaseCredentialsProvider(app) {
        var _this = this;
        this.app = app;
        /**
         * The auth token listener registered with FirebaseApp, retained here so we
         * can unregister it.
         */
        this.tokenListener = null;
        /**
         * Counter used to detect if the token changed while a getToken request was
         * outstanding.
         */
        this.tokenCounter = 0;
        /** The listener registered with setChangeListener(). */
        this.changeListener = null;
        this.forceRefresh = false;
        this.tokenListener = function () {
            _this.tokenCounter++;
            _this.currentUser = _this.getUser();
            if (_this.changeListener) {
                _this.changeListener(_this.currentUser);
            }
        };
        this.tokenCounter = 0;
        // Will fire at least once where we set this.currentUser
        this.app.INTERNAL.addAuthTokenListener(this.tokenListener);
    }
    FirebaseCredentialsProvider.prototype.getToken = function () {
        var _this = this;
        assert(this.tokenListener != null, 'getToken cannot be called after listener removed.');
        // Take note of the current value of the tokenCounter so that this method
        // can fail (with an ABORTED error) if there is a token change while the
        // request is outstanding.
        var initialTokenCounter = this.tokenCounter;
        var forceRefresh = this.forceRefresh;
        this.forceRefresh = false;
        return this.app.INTERNAL.getToken(forceRefresh).then(function (tokenData) {
            // Cancel the request since the token changed while the request was
            // outstanding so the response is potentially for a previous user (which
            // user, we can't be sure).
            if (_this.tokenCounter !== initialTokenCounter) {
                throw new FirestoreError(Code.ABORTED, 'getToken aborted due to token change.');
            }
            else {
                if (tokenData) {
                    assert(typeof tokenData.accessToken === 'string', 'Invalid tokenData returned from getToken():' + tokenData);
                    return new OAuthToken(tokenData.accessToken, _this.currentUser);
                }
                else {
                    return null;
                }
            }
        });
    };
    FirebaseCredentialsProvider.prototype.invalidateToken = function () {
        this.forceRefresh = true;
    };
    FirebaseCredentialsProvider.prototype.setChangeListener = function (changeListener) {
        assert(!this.changeListener, 'Can only call setChangeListener() once.');
        this.changeListener = changeListener;
        // Fire the initial event, but only if we received the initial user
        if (this.currentUser) {
            changeListener(this.currentUser);
        }
    };
    FirebaseCredentialsProvider.prototype.removeChangeListener = function () {
        assert(this.tokenListener != null, 'removeChangeListener() called twice');
        assert(this.changeListener !== null, 'removeChangeListener() called when no listener registered');
        this.app.INTERNAL.removeAuthTokenListener(this.tokenListener);
        this.tokenListener = null;
        this.changeListener = null;
    };
    FirebaseCredentialsProvider.prototype.getUser = function () {
        var currentUid = this.app.INTERNAL.getUid();
        assert(currentUid === null || typeof currentUid === 'string', 'Received invalid UID: ' + currentUid);
        return new User(currentUid);
    };
    return FirebaseCredentialsProvider;
}());
/*
 * FirstPartyToken provides a fresh token each time its value
 * is requested, because if the token is too old, requests will be rejected.
 * Technically this may no longer be necessary since the SDK should gracefully
 * recover from unauthenticated errors (see b/33147818 for context), but it's
 * safer to keep the implementation as-is.
 */
var FirstPartyToken = /** @class */ (function () {
    function FirstPartyToken(gapi, sessionIndex) {
        this.gapi = gapi;
        this.sessionIndex = sessionIndex;
        this.type = 'FirstParty';
        this.user = User.FIRST_PARTY;
    }
    Object.defineProperty(FirstPartyToken.prototype, "authHeaders", {
        get: function () {
            var headers = {
                'X-Goog-AuthUser': this.sessionIndex
            };
            var authHeader = this.gapi.auth.getAuthHeaderValueForFirstParty([]);
            if (authHeader) {
                headers['Authorization'] = authHeader;
            }
            return headers;
        },
        enumerable: true,
        configurable: true
    });
    return FirstPartyToken;
}());
/*
 * Provides user credentials required for the Firestore JavaScript SDK
 * to authenticate the user, using technique that is only available
 * to applications hosted by Google.
 */
var FirstPartyCredentialsProvider = /** @class */ (function () {
    function FirstPartyCredentialsProvider(gapi, sessionIndex) {
        this.gapi = gapi;
        this.sessionIndex = sessionIndex;
    }
    FirstPartyCredentialsProvider.prototype.getToken = function () {
        return Promise.resolve(new FirstPartyToken(this.gapi, this.sessionIndex));
    };
    FirstPartyCredentialsProvider.prototype.setChangeListener = function (changeListener) {
        // Fire with initial uid.
        changeListener(User.FIRST_PARTY);
    };
    FirstPartyCredentialsProvider.prototype.removeChangeListener = function () { };
    FirstPartyCredentialsProvider.prototype.invalidateToken = function () { };
    return FirstPartyCredentialsProvider;
}());
/**
 * Builds a CredentialsProvider depending on the type of
 * the credentials passed in.
 */
function makeCredentialsProvider(credentials) {
    if (!credentials) {
        return new EmptyCredentialsProvider();
    }
    switch (credentials.type) {
        case 'gapi':
            var client = credentials.client;
            // Make sure this is a Gapi client.
            assert(!!(typeof client === 'object' &&
                client !== null &&
                client['auth'] &&
                client['auth']['getAuthHeaderValueForFirstParty']), 'unexpected gapi interface');
            return new FirstPartyCredentialsProvider(client, credentials.sessionIndex || '0');
        case 'provider':
            return credentials.client;
        default:
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'makeCredentialsProvider failed due to invalid credential type');
    }
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
function isPartialObserver(obj) {
    return implementsAnyMethods(obj, ['next', 'error', 'complete']);
}
/**
 * Returns true if obj is an object and contains at least one of the specified
 * methods.
 */
function implementsAnyMethods(obj, methods) {
    if (typeof obj !== 'object' || obj === null) {
        return false;
    }
    var object = obj;
    for (var _i = 0, methods_1 = methods; _i < methods_1.length; _i++) {
        var method = methods_1[_i];
        if (method in object && typeof object[method] === 'function') {
            return true;
        }
    }
    return false;
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * An opaque base class for FieldValue sentinel objects in our public API,
 * with public static methods for creating said sentinel objects.
 */
// tslint:disable-next-line:class-as-namespace  We use this as a base class.
var FieldValueImpl = /** @class */ (function () {
    function FieldValueImpl(_methodName) {
        this._methodName = _methodName;
    }
    FieldValueImpl.delete = function () {
        validateNoArgs('FieldValue.delete', arguments);
        return DeleteFieldValueImpl.instance;
    };
    FieldValueImpl.serverTimestamp = function () {
        validateNoArgs('FieldValue.serverTimestamp', arguments);
        return ServerTimestampFieldValueImpl.instance;
    };
    FieldValueImpl.arrayUnion = function () {
        var elements = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            elements[_i] = arguments[_i];
        }
        validateAtLeastNumberOfArgs('FieldValue.arrayUnion', arguments, 1);
        // NOTE: We don't actually parse the data until it's used in set() or
        // update() since we need access to the Firestore instance.
        return new ArrayUnionFieldValueImpl(elements);
    };
    FieldValueImpl.arrayRemove = function () {
        var elements = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            elements[_i] = arguments[_i];
        }
        validateAtLeastNumberOfArgs('FieldValue.arrayRemove', arguments, 1);
        // NOTE: We don't actually parse the data until it's used in set() or
        // update() since we need access to the Firestore instance.
        return new ArrayRemoveFieldValueImpl(elements);
    };
    FieldValueImpl.increment = function (n) {
        validateArgType('FieldValue.increment', 'number', 1, n);
        validateExactNumberOfArgs('FieldValue.increment', arguments, 1);
        return new NumericIncrementFieldValueImpl(n);
    };
    FieldValueImpl.prototype.isEqual = function (other) {
        return this === other;
    };
    return FieldValueImpl;
}());
var DeleteFieldValueImpl = /** @class */ (function (_super) {
    tslib_1.__extends(DeleteFieldValueImpl, _super);
    function DeleteFieldValueImpl() {
        return _super.call(this, 'FieldValue.delete') || this;
    }
    /** Singleton instance. */
    DeleteFieldValueImpl.instance = new DeleteFieldValueImpl();
    return DeleteFieldValueImpl;
}(FieldValueImpl));
var ServerTimestampFieldValueImpl = /** @class */ (function (_super) {
    tslib_1.__extends(ServerTimestampFieldValueImpl, _super);
    function ServerTimestampFieldValueImpl() {
        return _super.call(this, 'FieldValue.serverTimestamp') || this;
    }
    /** Singleton instance. */
    ServerTimestampFieldValueImpl.instance = new ServerTimestampFieldValueImpl();
    return ServerTimestampFieldValueImpl;
}(FieldValueImpl));
var ArrayUnionFieldValueImpl = /** @class */ (function (_super) {
    tslib_1.__extends(ArrayUnionFieldValueImpl, _super);
    function ArrayUnionFieldValueImpl(_elements) {
        var _this = _super.call(this, 'FieldValue.arrayUnion') || this;
        _this._elements = _elements;
        return _this;
    }
    return ArrayUnionFieldValueImpl;
}(FieldValueImpl));
var ArrayRemoveFieldValueImpl = /** @class */ (function (_super) {
    tslib_1.__extends(ArrayRemoveFieldValueImpl, _super);
    function ArrayRemoveFieldValueImpl(_elements) {
        var _this = _super.call(this, 'FieldValue.arrayRemove') || this;
        _this._elements = _elements;
        return _this;
    }
    return ArrayRemoveFieldValueImpl;
}(FieldValueImpl));
var NumericIncrementFieldValueImpl = /** @class */ (function (_super) {
    tslib_1.__extends(NumericIncrementFieldValueImpl, _super);
    function NumericIncrementFieldValueImpl(_operand) {
        var _this = _super.call(this, 'FieldValue.increment') || this;
        _this._operand = _operand;
        return _this;
    }
    return NumericIncrementFieldValueImpl;
}(FieldValueImpl));
// Public instance that disallows construction at runtime. This constructor is
// used when exporting FieldValueImpl on firebase.firestore.FieldValue and will
// be called FieldValue publicly. Internally we still use FieldValueImpl which
// has a type-checked private constructor. Note that FieldValueImpl and
// PublicFieldValue can be used interchangeably in instanceof checks.
// For our internal TypeScript code PublicFieldValue doesn't exist as a type,
// and so we need to use FieldValueImpl as type and export it too.
// tslint:disable-next-line:variable-name  We treat this as a class name.
var PublicFieldValue = makeConstructorPrivate(FieldValueImpl, 'Use FieldValue.<field>() instead.');

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var RESERVED_FIELD_REGEX = /^__.*__$/;
/** The result of parsing document data (e.g. for a setData call). */
var ParsedSetData = /** @class */ (function () {
    function ParsedSetData(data, fieldMask, fieldTransforms) {
        this.data = data;
        this.fieldMask = fieldMask;
        this.fieldTransforms = fieldTransforms;
    }
    ParsedSetData.prototype.toMutations = function (key, precondition) {
        var mutations = [];
        if (this.fieldMask !== null) {
            mutations.push(new PatchMutation(key, this.data, this.fieldMask, precondition));
        }
        else {
            mutations.push(new SetMutation(key, this.data, precondition));
        }
        if (this.fieldTransforms.length > 0) {
            mutations.push(new TransformMutation(key, this.fieldTransforms));
        }
        return mutations;
    };
    return ParsedSetData;
}());
/** The result of parsing "update" data (i.e. for an updateData call). */
var ParsedUpdateData = /** @class */ (function () {
    function ParsedUpdateData(data, fieldMask, fieldTransforms) {
        this.data = data;
        this.fieldMask = fieldMask;
        this.fieldTransforms = fieldTransforms;
    }
    ParsedUpdateData.prototype.toMutations = function (key, precondition) {
        var mutations = [
            new PatchMutation(key, this.data, this.fieldMask, precondition)
        ];
        if (this.fieldTransforms.length > 0) {
            mutations.push(new TransformMutation(key, this.fieldTransforms));
        }
        return mutations;
    };
    return ParsedUpdateData;
}());
/*
 * Represents what type of API method provided the data being parsed; useful
 * for determining which error conditions apply during parsing and providing
 * better error messages.
 */
var UserDataSource;
(function (UserDataSource) {
    UserDataSource[UserDataSource["Set"] = 0] = "Set";
    UserDataSource[UserDataSource["Update"] = 1] = "Update";
    UserDataSource[UserDataSource["MergeSet"] = 2] = "MergeSet";
    /**
     * Indicates the source is a where clause, cursor bound, arrayUnion()
     * element, etc. Of note, isWrite(source) will return false.
     */
    UserDataSource[UserDataSource["Argument"] = 3] = "Argument";
})(UserDataSource || (UserDataSource = {}));
function isWrite(dataSource) {
    switch (dataSource) {
        case UserDataSource.Set: // fall through
        case UserDataSource.MergeSet: // fall through
        case UserDataSource.Update:
            return true;
        case UserDataSource.Argument:
            return false;
        default:
            throw fail("Unexpected case for UserDataSource: " + dataSource);
    }
}
/** A "context" object passed around while parsing user data. */
var ParseContext = /** @class */ (function () {
    /**
     * Initializes a ParseContext with the given source and path.
     *
     * @param dataSource Indicates what kind of API method this data came from.
     * @param methodName The name of the method the user called to create this
     *     ParseContext.
     * @param path A path within the object being parsed. This could be an empty
     *     path (in which case the context represents the root of the data being
     *     parsed), or a nonempty path (indicating the context represents a nested
     *     location within the data).
     * @param arrayElement Whether or not this context corresponds to an element
     *     of an array.
     * @param fieldTransforms A mutable list of field transforms encountered while
     *     parsing the data.
     * @param fieldMask A mutable list of field paths encountered while parsing
     *     the data.
     *
     * TODO(b/34871131): We don't support array paths right now, so path can be
     * null to indicate the context represents any location within an array (in
     * which case certain features will not work and errors will be somewhat
     * compromised).
     */
    function ParseContext(dataSource, methodName, path, arrayElement, fieldTransforms, fieldMask) {
        this.dataSource = dataSource;
        this.methodName = methodName;
        this.path = path;
        this.arrayElement = arrayElement;
        // Minor hack: If fieldTransforms is undefined, we assume this is an
        // external call and we need to validate the entire path.
        if (fieldTransforms === undefined) {
            this.validatePath();
        }
        this.arrayElement = arrayElement !== undefined ? arrayElement : false;
        this.fieldTransforms = fieldTransforms || [];
        this.fieldMask = fieldMask || [];
    }
    ParseContext.prototype.childContextForField = function (field) {
        var childPath = this.path == null ? null : this.path.child(field);
        var context = new ParseContext(this.dataSource, this.methodName, childPath, 
        /*arrayElement=*/ false, this.fieldTransforms, this.fieldMask);
        context.validatePathSegment(field);
        return context;
    };
    ParseContext.prototype.childContextForFieldPath = function (field) {
        var childPath = this.path == null ? null : this.path.child(field);
        var context = new ParseContext(this.dataSource, this.methodName, childPath, 
        /*arrayElement=*/ false, this.fieldTransforms, this.fieldMask);
        context.validatePath();
        return context;
    };
    ParseContext.prototype.childContextForArray = function (index) {
        // TODO(b/34871131): We don't support array paths right now; so make path
        // null.
        return new ParseContext(this.dataSource, this.methodName, 
        /*path=*/ null, 
        /*arrayElement=*/ true, this.fieldTransforms, this.fieldMask);
    };
    ParseContext.prototype.createError = function (reason) {
        var fieldDescription = this.path === null || this.path.isEmpty()
            ? ''
            : " (found in field " + this.path.toString() + ")";
        return new FirestoreError(Code.INVALID_ARGUMENT, "Function " + this.methodName + "() called with invalid data. " +
            reason +
            fieldDescription);
    };
    /** Returns 'true' if 'fieldPath' was traversed when creating this context. */
    ParseContext.prototype.contains = function (fieldPath) {
        return (this.fieldMask.find(function (field) { return fieldPath.isPrefixOf(field); }) !== undefined ||
            this.fieldTransforms.find(function (transform) {
                return fieldPath.isPrefixOf(transform.field);
            }) !== undefined);
    };
    ParseContext.prototype.validatePath = function () {
        // TODO(b/34871131): Remove null check once we have proper paths for fields
        // within arrays.
        if (this.path === null) {
            return;
        }
        for (var i = 0; i < this.path.length; i++) {
            this.validatePathSegment(this.path.get(i));
        }
    };
    ParseContext.prototype.validatePathSegment = function (segment) {
        if (isWrite(this.dataSource) && RESERVED_FIELD_REGEX.test(segment)) {
            throw this.createError('Document fields cannot begin and end with __');
        }
    };
    return ParseContext;
}());
/**
 * A placeholder object for DocumentReferences in this file, in order to
 * avoid a circular dependency. See the comments for `DataPreConverter` for
 * the full context.
 */
var DocumentKeyReference = /** @class */ (function () {
    function DocumentKeyReference(databaseId, key) {
        this.databaseId = databaseId;
        this.key = key;
    }
    return DocumentKeyReference;
}());
/**
 * Helper for parsing raw user input (provided via the API) into internal model
 * classes.
 */
var UserDataConverter = /** @class */ (function () {
    function UserDataConverter(preConverter) {
        this.preConverter = preConverter;
    }
    /** Parse document data from a non-merge set() call. */
    UserDataConverter.prototype.parseSetData = function (methodName, input) {
        var context = new ParseContext(UserDataSource.Set, methodName, FieldPath.EMPTY_PATH);
        validatePlainObject('Data must be an object, but it was:', context, input);
        var updateData = this.parseData(input, context);
        return new ParsedSetData(updateData, 
        /* fieldMask= */ null, context.fieldTransforms);
    };
    /** Parse document data from a set() call with '{merge:true}'. */
    UserDataConverter.prototype.parseMergeData = function (methodName, input, fieldPaths) {
        var context = new ParseContext(UserDataSource.MergeSet, methodName, FieldPath.EMPTY_PATH);
        validatePlainObject('Data must be an object, but it was:', context, input);
        var updateData = this.parseData(input, context);
        var fieldMask;
        var fieldTransforms;
        if (!fieldPaths) {
            fieldMask = FieldMask.fromArray(context.fieldMask);
            fieldTransforms = context.fieldTransforms;
        }
        else {
            var validatedFieldPaths = new SortedSet(FieldPath.comparator);
            for (var _i = 0, fieldPaths_1 = fieldPaths; _i < fieldPaths_1.length; _i++) {
                var stringOrFieldPath = fieldPaths_1[_i];
                var fieldPath = void 0;
                if (stringOrFieldPath instanceof FieldPath$1) {
                    fieldPath = stringOrFieldPath._internalPath;
                }
                else if (typeof stringOrFieldPath === 'string') {
                    fieldPath = fieldPathFromDotSeparatedString(methodName, stringOrFieldPath);
                }
                else {
                    throw fail('Expected stringOrFieldPath to be a string or a FieldPath');
                }
                if (!context.contains(fieldPath)) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, "Field '" + fieldPath + "' is specified in your field mask but missing from your input data.");
                }
                validatedFieldPaths = validatedFieldPaths.add(fieldPath);
            }
            fieldMask = FieldMask.fromSet(validatedFieldPaths);
            fieldTransforms = context.fieldTransforms.filter(function (transform) {
                return fieldMask.covers(transform.field);
            });
        }
        return new ParsedSetData(updateData, fieldMask, fieldTransforms);
    };
    /** Parse update data from an update() call. */
    UserDataConverter.prototype.parseUpdateData = function (methodName, input) {
        var _this = this;
        var context = new ParseContext(UserDataSource.Update, methodName, FieldPath.EMPTY_PATH);
        validatePlainObject('Data must be an object, but it was:', context, input);
        var fieldMaskPaths = new SortedSet(FieldPath.comparator);
        var updateData = ObjectValue.EMPTY;
        forEach(input, function (key, value) {
            var path = fieldPathFromDotSeparatedString(methodName, key);
            var childContext = context.childContextForFieldPath(path);
            value = _this.runPreConverter(value, childContext);
            if (value instanceof DeleteFieldValueImpl) {
                // Add it to the field mask, but don't add anything to updateData.
                fieldMaskPaths = fieldMaskPaths.add(path);
            }
            else {
                var parsedValue = _this.parseData(value, childContext);
                if (parsedValue != null) {
                    fieldMaskPaths = fieldMaskPaths.add(path);
                    updateData = updateData.set(path, parsedValue);
                }
            }
        });
        var mask = FieldMask.fromSet(fieldMaskPaths);
        return new ParsedUpdateData(updateData, mask, context.fieldTransforms);
    };
    /** Parse update data from a list of field/value arguments. */
    UserDataConverter.prototype.parseUpdateVarargs = function (methodName, field, value, moreFieldsAndValues) {
        var context = new ParseContext(UserDataSource.Update, methodName, FieldPath.EMPTY_PATH);
        var keys = [fieldPathFromArgument(methodName, field)];
        var values = [value];
        if (moreFieldsAndValues.length % 2 !== 0) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + methodName + "() needs to be called with an even number " +
                'of arguments that alternate between field names and values.');
        }
        for (var i = 0; i < moreFieldsAndValues.length; i += 2) {
            keys.push(fieldPathFromArgument(methodName, moreFieldsAndValues[i]));
            values.push(moreFieldsAndValues[i + 1]);
        }
        var fieldMaskPaths = new SortedSet(FieldPath.comparator);
        var updateData = ObjectValue.EMPTY;
        for (var i = 0; i < keys.length; ++i) {
            var path = keys[i];
            var childContext = context.childContextForFieldPath(path);
            var value_1 = this.runPreConverter(values[i], childContext);
            if (value_1 instanceof DeleteFieldValueImpl) {
                // Add it to the field mask, but don't add anything to updateData.
                fieldMaskPaths = fieldMaskPaths.add(path);
            }
            else {
                var parsedValue = this.parseData(value_1, childContext);
                if (parsedValue != null) {
                    fieldMaskPaths = fieldMaskPaths.add(path);
                    updateData = updateData.set(path, parsedValue);
                }
            }
        }
        var mask = FieldMask.fromSet(fieldMaskPaths);
        return new ParsedUpdateData(updateData, mask, context.fieldTransforms);
    };
    /**
     * Parse a "query value" (e.g. value in a where filter or a value in a cursor
     * bound).
     */
    UserDataConverter.prototype.parseQueryValue = function (methodName, input) {
        var context = new ParseContext(UserDataSource.Argument, methodName, FieldPath.EMPTY_PATH);
        var parsed = this.parseData(input, context);
        assert(parsed != null, 'Parsed data should not be null.');
        assert(context.fieldTransforms.length === 0, 'Field transforms should have been disallowed.');
        return parsed;
    };
    /** Sends data through this.preConverter, handling any thrown errors. */
    UserDataConverter.prototype.runPreConverter = function (input, context) {
        try {
            return this.preConverter(input);
        }
        catch (e) {
            var message = errorMessage(e);
            throw context.createError(message);
        }
    };
    /**
     * Internal helper for parsing user data.
     *
     * @param input Data to be parsed.
     * @param context A context object representing the current path being parsed,
     * the source of the data being parsed, etc.
     * @return The parsed value, or null if the value was a FieldValue sentinel
     * that should not be included in the resulting parsed data.
     */
    UserDataConverter.prototype.parseData = function (input, context) {
        input = this.runPreConverter(input, context);
        if (looksLikeJsonObject(input)) {
            validatePlainObject('Unsupported field value:', context, input);
            return this.parseObject(input, context);
        }
        else if (input instanceof FieldValueImpl) {
            // FieldValues usually parse into transforms (except FieldValue.delete())
            // in which case we do not want to include this field in our parsed data
            // (as doing so will overwrite the field directly prior to the transform
            // trying to transform it). So we don't add this location to
            // context.fieldMask and we return null as our parsing result.
            this.parseSentinelFieldValue(input, context);
            return null;
        }
        else {
            // If context.path is null we are inside an array and we don't support
            // field mask paths more granular than the top-level array.
            if (context.path) {
                context.fieldMask.push(context.path);
            }
            if (input instanceof Array) {
                // TODO(b/34871131): Include the path containing the array in the error
                // message.
                if (context.arrayElement) {
                    throw context.createError('Nested arrays are not supported');
                }
                return this.parseArray(input, context);
            }
            else {
                return this.parseScalarValue(input, context);
            }
        }
    };
    UserDataConverter.prototype.parseObject = function (obj, context) {
        var _this = this;
        var result = new SortedMap(primitiveComparator);
        if (isEmpty(obj)) {
            // If we encounter an empty object, we explicitly add it to the update
            // mask to ensure that the server creates a map entry.
            if (context.path && context.path.length > 0) {
                context.fieldMask.push(context.path);
            }
        }
        else {
            forEach(obj, function (key, val) {
                var parsedValue = _this.parseData(val, context.childContextForField(key));
                if (parsedValue != null) {
                    result = result.insert(key, parsedValue);
                }
            });
        }
        return new ObjectValue(result);
    };
    UserDataConverter.prototype.parseArray = function (array, context) {
        var result = [];
        var entryIndex = 0;
        for (var _i = 0, array_1 = array; _i < array_1.length; _i++) {
            var entry = array_1[_i];
            var parsedEntry = this.parseData(entry, context.childContextForArray(entryIndex));
            if (parsedEntry == null) {
                // Just include nulls in the array for fields being replaced with a
                // sentinel.
                parsedEntry = NullValue.INSTANCE;
            }
            result.push(parsedEntry);
            entryIndex++;
        }
        return new ArrayValue(result);
    };
    /**
     * "Parses" the provided FieldValueImpl, adding any necessary transforms to
     * context.fieldTransforms.
     */
    UserDataConverter.prototype.parseSentinelFieldValue = function (value, context) {
        // Sentinels are only supported with writes, and not within arrays.
        if (!isWrite(context.dataSource)) {
            throw context.createError(value._methodName + "() can only be used with update() and set()");
        }
        if (context.path === null) {
            throw context.createError(value._methodName + "() is not currently supported inside arrays");
        }
        if (value instanceof DeleteFieldValueImpl) {
            if (context.dataSource === UserDataSource.MergeSet) {
                // No transform to add for a delete, but we need to add it to our
                // fieldMask so it gets deleted.
                context.fieldMask.push(context.path);
            }
            else if (context.dataSource === UserDataSource.Update) {
                assert(context.path.length > 0, 'FieldValue.delete() at the top level should have already' +
                    ' been handled.');
                throw context.createError('FieldValue.delete() can only appear at the top level ' +
                    'of your update data');
            }
            else {
                // We shouldn't encounter delete sentinels for queries or non-merge set() calls.
                throw context.createError('FieldValue.delete() cannot be used with set() unless you pass ' +
                    '{merge:true}');
            }
        }
        else if (value instanceof ServerTimestampFieldValueImpl) {
            context.fieldTransforms.push(new FieldTransform(context.path, ServerTimestampTransform.instance));
        }
        else if (value instanceof ArrayUnionFieldValueImpl) {
            var parsedElements = this.parseArrayTransformElements(value._methodName, value._elements);
            var arrayUnion = new ArrayUnionTransformOperation(parsedElements);
            context.fieldTransforms.push(new FieldTransform(context.path, arrayUnion));
        }
        else if (value instanceof ArrayRemoveFieldValueImpl) {
            var parsedElements = this.parseArrayTransformElements(value._methodName, value._elements);
            var arrayRemove = new ArrayRemoveTransformOperation(parsedElements);
            context.fieldTransforms.push(new FieldTransform(context.path, arrayRemove));
        }
        else if (value instanceof NumericIncrementFieldValueImpl) {
            var operand = this.parseQueryValue('FieldValue.increment', value._operand);
            var numericIncrement = new NumericIncrementTransformOperation(operand);
            context.fieldTransforms.push(new FieldTransform(context.path, numericIncrement));
        }
        else {
            fail('Unknown FieldValue type: ' + value);
        }
    };
    /**
     * Helper to parse a scalar value (i.e. not an Object, Array, or FieldValue)
     *
     * @return The parsed value
     */
    UserDataConverter.prototype.parseScalarValue = function (value, context) {
        if (value === null) {
            return NullValue.INSTANCE;
        }
        else if (typeof value === 'number') {
            if (isSafeInteger(value)) {
                return new IntegerValue(value);
            }
            else {
                return new DoubleValue(value);
            }
        }
        else if (typeof value === 'boolean') {
            return BooleanValue.of(value);
        }
        else if (typeof value === 'string') {
            return new StringValue(value);
        }
        else if (value instanceof Date) {
            return new TimestampValue(Timestamp.fromDate(value));
        }
        else if (value instanceof Timestamp) {
            // Firestore backend truncates precision down to microseconds. To ensure
            // offline mode works the same with regards to truncation, perform the
            // truncation immediately without waiting for the backend to do that.
            return new TimestampValue(new Timestamp(value.seconds, Math.floor(value.nanoseconds / 1000) * 1000));
        }
        else if (value instanceof GeoPoint) {
            return new GeoPointValue(value);
        }
        else if (value instanceof Blob) {
            return new BlobValue(value);
        }
        else if (value instanceof DocumentKeyReference) {
            return new RefValue(value.databaseId, value.key);
        }
        else {
            throw context.createError("Unsupported field value: " + valueDescription(value));
        }
    };
    UserDataConverter.prototype.parseArrayTransformElements = function (methodName, elements) {
        var _this = this;
        return elements.map(function (element, i) {
            // Although array transforms are used with writes, the actual elements
            // being unioned or removed are not considered writes since they cannot
            // contain any FieldValue sentinels, etc.
            var context = new ParseContext(UserDataSource.Argument, methodName, FieldPath.EMPTY_PATH);
            return _this.parseData(element, context.childContextForArray(i));
        });
    };
    return UserDataConverter;
}());
/**
 * Checks whether an object looks like a JSON object that should be converted
 * into a struct. Normal class/prototype instances are considered to look like
 * JSON objects since they should be converted to a struct value. Arrays, Dates,
 * GeoPoints, etc. are not considered to look like JSON objects since they map
 * to specific FieldValue types other than ObjectValue.
 */
function looksLikeJsonObject(input) {
    return (typeof input === 'object' &&
        input !== null &&
        !(input instanceof Array) &&
        !(input instanceof Date) &&
        !(input instanceof Timestamp) &&
        !(input instanceof GeoPoint) &&
        !(input instanceof Blob) &&
        !(input instanceof DocumentKeyReference) &&
        !(input instanceof FieldValueImpl));
}
function validatePlainObject(message, context, input) {
    if (!looksLikeJsonObject(input) || !isPlainObject(input)) {
        var description = valueDescription(input);
        if (description === 'an object') {
            // Massage the error if it was an object.
            throw context.createError(message + ' a custom object');
        }
        else {
            throw context.createError(message + ' ' + description);
        }
    }
}
/**
 * Helper that calls fromDotSeparatedString() but wraps any error thrown.
 */
function fieldPathFromArgument(methodName, path) {
    if (path instanceof FieldPath$1) {
        return path._internalPath;
    }
    else if (typeof path === 'string') {
        return fieldPathFromDotSeparatedString(methodName, path);
    }
    else {
        var message = 'Field path arguments must be of type string or FieldPath.';
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + methodName + "() called with invalid data. " + message);
    }
}
/**
 * Wraps fromDotSeparatedString with an error message about the method that
 * was thrown.
 * @param methodName The publicly visible method name
 * @param path The dot-separated string form of a field path which will be split
 * on dots.
 */
function fieldPathFromDotSeparatedString(methodName, path) {
    try {
        return fromDotSeparatedString(path)._internalPath;
    }
    catch (e) {
        var message = errorMessage(e);
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Function " + methodName + "() called with invalid data. " + message);
    }
}
/**
 * Extracts the message from a caught exception, which should be an Error object
 * though JS doesn't guarantee that.
 */
function errorMessage(error) {
    return error instanceof Error ? error.message : error.toString();
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
// The objects that are a part of this API are exposed to third-parties as
// compiled javascript so we want to flag our private members with a leading
// underscore to discourage their use.
// tslint:disable:strip-private-property-underscore
// settings() defaults:
var DEFAULT_HOST = 'firestore.googleapis.com';
var DEFAULT_SSL = true;
var DEFAULT_TIMESTAMPS_IN_SNAPSHOTS = true;
/**
 * Constant used to indicate the LRU garbage collection should be disabled.
 * Set this value as the `cacheSizeBytes` on the settings passed to the
 * `Firestore` instance.
 */
var CACHE_SIZE_UNLIMITED = LruParams.COLLECTION_DISABLED;
// enablePersistence() defaults:
var DEFAULT_SYNCHRONIZE_TABS = false;
/**
 * A concrete type describing all the values that can be applied via a
 * user-supplied firestore.Settings object. This is a separate type so that
 * defaults can be supplied and the value can be checked for equality.
 */
var FirestoreSettings = /** @class */ (function () {
    function FirestoreSettings(settings) {
        if (settings.host === undefined) {
            if (settings.ssl !== undefined) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, "Can't provide ssl option if host option is not set");
            }
            this.host = DEFAULT_HOST;
            this.ssl = DEFAULT_SSL;
        }
        else {
            validateNamedType('settings', 'non-empty string', 'host', settings.host);
            this.host = settings.host;
            validateNamedOptionalType('settings', 'boolean', 'ssl', settings.ssl);
            this.ssl = defaulted(settings.ssl, DEFAULT_SSL);
        }
        validateOptionNames('settings', settings, [
            'host',
            'ssl',
            'credentials',
            'timestampsInSnapshots',
            'cacheSizeBytes'
        ]);
        validateNamedOptionalType('settings', 'object', 'credentials', settings.credentials);
        this.credentials = settings.credentials;
        validateNamedOptionalType('settings', 'boolean', 'timestampsInSnapshots', settings.timestampsInSnapshots);
        // Nobody should set timestampsInSnapshots anymore, but the error depends on
        // whether they set it to true or false...
        if (settings.timestampsInSnapshots === true) {
            error("\n  The timestampsInSnapshots setting now defaults to true and you no\n  longer need to explicitly set it. In a future release, the setting\n  will be removed entirely and so it is recommended that you remove it\n  from your firestore.settings() call now.");
        }
        else if (settings.timestampsInSnapshots === false) {
            error("\n  The timestampsInSnapshots setting will soon be removed. YOU MUST UPDATE\n  YOUR CODE.\n\n  To hide this warning, stop using the timestampsInSnapshots setting in your\n  firestore.settings({ ... }) call.\n\n  Once you remove the setting, Timestamps stored in Cloud Firestore will be\n  read back as Firebase Timestamp objects instead of as system Date objects.\n  So you will also need to update code expecting a Date to instead expect a\n  Timestamp. For example:\n\n  // Old:\n  const date = snapshot.get('created_at');\n  // New:\n  const timestamp = snapshot.get('created_at'); const date =\n  timestamp.toDate();\n\n  Please audit all existing usages of Date when you enable the new\n  behavior.");
        }
        this.timestampsInSnapshots = defaulted(settings.timestampsInSnapshots, DEFAULT_TIMESTAMPS_IN_SNAPSHOTS);
        validateNamedOptionalType('settings', 'number', 'cacheSizeBytes', settings.cacheSizeBytes);
        if (settings.cacheSizeBytes === undefined) {
            this.cacheSizeBytes = LruParams.DEFAULT_CACHE_SIZE_BYTES;
        }
        else {
            if (settings.cacheSizeBytes !== CACHE_SIZE_UNLIMITED &&
                settings.cacheSizeBytes < LruParams.MINIMUM_CACHE_SIZE_BYTES) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, "cacheSizeBytes must be at least " + LruParams.MINIMUM_CACHE_SIZE_BYTES);
            }
            else {
                this.cacheSizeBytes = settings.cacheSizeBytes;
            }
        }
    }
    FirestoreSettings.prototype.isEqual = function (other) {
        return (this.host === other.host &&
            this.ssl === other.ssl &&
            this.timestampsInSnapshots === other.timestampsInSnapshots &&
            this.credentials === other.credentials &&
            this.cacheSizeBytes === other.cacheSizeBytes);
    };
    return FirestoreSettings;
}());
var FirestoreConfig = /** @class */ (function () {
    function FirestoreConfig() {
    }
    return FirestoreConfig;
}());
/**
 * The root reference to the database.
 */
var Firestore = /** @class */ (function () {
    function Firestore(databaseIdOrApp) {
        var _this = this;
        // Public for use in tests.
        // TODO(mikelehen): Use modularized initialization instead.
        this._queue = new AsyncQueue();
        this.INTERNAL = {
            delete: function (options) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                return tslib_1.__generator(this, function (_a) {
                    if (this._firestoreClient) {
                        return [2 /*return*/, this._firestoreClient.shutdown(options)];
                    }
                    return [2 /*return*/];
                });
            }); }
        };
        var config = new FirestoreConfig();
        if (typeof databaseIdOrApp.options === 'object') {
            // This is very likely a Firebase app object
            // TODO(b/34177605): Can we somehow use instanceof?
            var app = databaseIdOrApp;
            config.firebaseApp = app;
            config.databaseId = Firestore.databaseIdFromApp(app);
            config.persistenceKey = config.firebaseApp.name;
            config.credentials = new FirebaseCredentialsProvider(app);
        }
        else {
            var external_1 = databaseIdOrApp;
            if (!external_1.projectId) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, 'Must provide projectId');
            }
            config.databaseId = new DatabaseId(external_1.projectId, external_1.database);
            // Use a default persistenceKey that lines up with FirebaseApp.
            config.persistenceKey = '[DEFAULT]';
            config.credentials = new EmptyCredentialsProvider();
        }
        config.settings = new FirestoreSettings({});
        this._config = config;
        this._databaseId = config.databaseId;
    }
    Firestore.prototype.settings = function (settingsLiteral) {
        validateExactNumberOfArgs('Firestore.settings', arguments, 1);
        validateArgType('Firestore.settings', 'object', 1, settingsLiteral);
        if (contains(settingsLiteral, 'persistence')) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, '"persistence" is now specified with a separate call to ' +
                'firestore.enablePersistence().');
        }
        var newSettings = new FirestoreSettings(settingsLiteral);
        if (this._firestoreClient && !this._config.settings.isEqual(newSettings)) {
            throw new FirestoreError(Code.FAILED_PRECONDITION, 'Firestore has already been started and its settings can no longer ' +
                'be changed. You can only call settings() before calling any other ' +
                'methods on a Firestore object.');
        }
        this._config.settings = newSettings;
        if (newSettings.credentials !== undefined) {
            this._config.credentials = makeCredentialsProvider(newSettings.credentials);
        }
    };
    Firestore.prototype.enableNetwork = function () {
        this.ensureClientConfigured();
        return this._firestoreClient.enableNetwork();
    };
    Firestore.prototype.disableNetwork = function () {
        this.ensureClientConfigured();
        return this._firestoreClient.disableNetwork();
    };
    Firestore.prototype.enablePersistence = function (settings) {
        if (this._firestoreClient) {
            throw new FirestoreError(Code.FAILED_PRECONDITION, 'Firestore has already been started and persistence can no longer ' +
                'be enabled. You can only call enablePersistence() before calling ' +
                'any other methods on a Firestore object.');
        }
        return this.configureClient(new IndexedDbPersistenceSettings(this._config.settings.cacheSizeBytes, settings !== undefined &&
            defaulted(settings.experimentalTabSynchronization, DEFAULT_SYNCHRONIZE_TABS)));
    };
    Firestore.prototype.ensureClientConfigured = function () {
        if (!this._firestoreClient) {
            // Kick off starting the client but don't actually wait for it.
            // tslint:disable-next-line:no-floating-promises
            this.configureClient(new MemoryPersistenceSettings());
        }
        return this._firestoreClient;
    };
    Firestore.prototype.configureClient = function (persistenceSettings) {
        var _this = this;
        assert(!!this._config.settings.host, 'FirestoreSettings.host cannot be falsey');
        assert(!this._firestoreClient, 'configureClient() called multiple times');
        var databaseInfo = new DatabaseInfo(this._config.databaseId, this._config.persistenceKey, this._config.settings.host, this._config.settings.ssl);
        var preConverter = function (value) {
            if (value instanceof DocumentReference) {
                var thisDb = _this._config.databaseId;
                var otherDb = value.firestore._config.databaseId;
                if (!otherDb.isEqual(thisDb)) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, 'Document reference is for database ' +
                        (otherDb.projectId + "/" + otherDb.database + " but should be ") +
                        ("for database " + thisDb.projectId + "/" + thisDb.database));
                }
                return new DocumentKeyReference(_this._config.databaseId, value._key);
            }
            else {
                return value;
            }
        };
        this._dataConverter = new UserDataConverter(preConverter);
        this._firestoreClient = new FirestoreClient(PlatformSupport.getPlatform(), databaseInfo, this._config.credentials, this._queue);
        return this._firestoreClient.start(persistenceSettings);
    };
    Firestore.databaseIdFromApp = function (app) {
        var options = app.options;
        if (!contains(options, 'projectId')) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, '"projectId" not provided in firebase.initializeApp.');
        }
        var projectId = options['projectId'];
        if (!projectId || typeof projectId !== 'string') {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'projectId must be a string in FirebaseApp.options');
        }
        return new DatabaseId(projectId);
    };
    Object.defineProperty(Firestore.prototype, "app", {
        get: function () {
            if (!this._config.firebaseApp) {
                throw new FirestoreError(Code.FAILED_PRECONDITION, "Firestore was not initialized using the Firebase SDK. 'app' is " +
                    'not available');
            }
            return this._config.firebaseApp;
        },
        enumerable: true,
        configurable: true
    });
    Firestore.prototype.collection = function (pathString) {
        validateExactNumberOfArgs('Firestore.collection', arguments, 1);
        validateArgType('Firestore.collection', 'non-empty string', 1, pathString);
        this.ensureClientConfigured();
        return new CollectionReference(ResourcePath.fromString(pathString), this);
    };
    Firestore.prototype.doc = function (pathString) {
        validateExactNumberOfArgs('Firestore.doc', arguments, 1);
        validateArgType('Firestore.doc', 'non-empty string', 1, pathString);
        this.ensureClientConfigured();
        return DocumentReference.forPath(ResourcePath.fromString(pathString), this);
    };
    // TODO(b/116617988): Fix name, uncomment d.ts definitions, and update CHANGELOG.md.
    Firestore.prototype._collectionGroup = function (collectionId) {
        validateExactNumberOfArgs('Firestore.collectionGroup', arguments, 1);
        validateArgType('Firestore.collectionGroup', 'non-empty string', 1, collectionId);
        if (collectionId.indexOf('/') >= 0) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid collection ID '" + collectionId + "' passed to function " +
                "Firestore.collectionGroup(). Collection IDs must not contain '/'.");
        }
        this.ensureClientConfigured();
        return new Query$1(new Query(ResourcePath.EMPTY_PATH, collectionId), this);
    };
    Firestore.prototype.runTransaction = function (updateFunction) {
        var _this = this;
        validateExactNumberOfArgs('Firestore.runTransaction', arguments, 1);
        validateArgType('Firestore.runTransaction', 'function', 1, updateFunction);
        return this.ensureClientConfigured().transaction(function (transaction) {
            return updateFunction(new Transaction$1(_this, transaction));
        });
    };
    Firestore.prototype.batch = function () {
        this.ensureClientConfigured();
        return new WriteBatch(this);
    };
    Object.defineProperty(Firestore, "logLevel", {
        get: function () {
            switch (getLogLevel()) {
                case LogLevel.DEBUG:
                    return 'debug';
                case LogLevel.ERROR:
                    return 'error';
                case LogLevel.SILENT:
                    return 'silent';
                default:
                    return fail('Unknown log level: ' + getLogLevel());
            }
        },
        enumerable: true,
        configurable: true
    });
    Firestore.setLogLevel = function (level) {
        validateExactNumberOfArgs('Firestore.setLogLevel', arguments, 1);
        validateArgType('Firestore.setLogLevel', 'non-empty string', 1, level);
        switch (level) {
            case 'debug':
                setLogLevel(LogLevel.DEBUG);
                break;
            case 'error':
                setLogLevel(LogLevel.ERROR);
                break;
            case 'silent':
                setLogLevel(LogLevel.SILENT);
                break;
            default:
                throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid log level: ' + level);
        }
    };
    // Note: this is not a property because the minifier can't work correctly with
    // the way TypeScript compiler outputs properties.
    Firestore.prototype._areTimestampsInSnapshotsEnabled = function () {
        return this._config.settings.timestampsInSnapshots;
    };
    return Firestore;
}());
/**
 * A reference to a transaction.
 */
var Transaction$1 = /** @class */ (function () {
    function Transaction(_firestore, _transaction) {
        this._firestore = _firestore;
        this._transaction = _transaction;
    }
    Transaction.prototype.get = function (documentRef) {
        var _this = this;
        validateExactNumberOfArgs('Transaction.get', arguments, 1);
        var ref = validateReference('Transaction.get', documentRef, this._firestore);
        return this._transaction
            .lookup([ref._key])
            .then(function (docs) {
            if (!docs || docs.length !== 1) {
                return fail('Mismatch in docs returned from document lookup.');
            }
            var doc = docs[0];
            if (doc instanceof NoDocument) {
                return new DocumentSnapshot(_this._firestore, ref._key, null, 
                /* fromCache= */ false, 
                /* hasPendingWrites= */ false);
            }
            else if (doc instanceof Document) {
                return new DocumentSnapshot(_this._firestore, ref._key, doc, 
                /* fromCache= */ false, 
                /* hasPendingWrites= */ false);
            }
            else {
                throw fail("BatchGetDocumentsRequest returned unexpected document type: " + doc.constructor.name);
            }
        });
    };
    Transaction.prototype.set = function (documentRef, value, options) {
        validateBetweenNumberOfArgs('Transaction.set', arguments, 2, 3);
        var ref = validateReference('Transaction.set', documentRef, this._firestore);
        options = validateSetOptions('Transaction.set', options);
        var parsed = options.merge || options.mergeFields
            ? this._firestore._dataConverter.parseMergeData('Transaction.set', value, options.mergeFields)
            : this._firestore._dataConverter.parseSetData('Transaction.set', value);
        this._transaction.set(ref._key, parsed);
        return this;
    };
    Transaction.prototype.update = function (documentRef, fieldOrUpdateData, value) {
        var moreFieldsAndValues = [];
        for (var _i = 3; _i < arguments.length; _i++) {
            moreFieldsAndValues[_i - 3] = arguments[_i];
        }
        var ref;
        var parsed;
        if (typeof fieldOrUpdateData === 'string' ||
            fieldOrUpdateData instanceof FieldPath$1) {
            validateAtLeastNumberOfArgs('Transaction.update', arguments, 3);
            ref = validateReference('Transaction.update', documentRef, this._firestore);
            parsed = this._firestore._dataConverter.parseUpdateVarargs('Transaction.update', fieldOrUpdateData, value, moreFieldsAndValues);
        }
        else {
            validateExactNumberOfArgs('Transaction.update', arguments, 2);
            ref = validateReference('Transaction.update', documentRef, this._firestore);
            parsed = this._firestore._dataConverter.parseUpdateData('Transaction.update', fieldOrUpdateData);
        }
        this._transaction.update(ref._key, parsed);
        return this;
    };
    Transaction.prototype.delete = function (documentRef) {
        validateExactNumberOfArgs('Transaction.delete', arguments, 1);
        var ref = validateReference('Transaction.delete', documentRef, this._firestore);
        this._transaction.delete(ref._key);
        return this;
    };
    return Transaction;
}());
var WriteBatch = /** @class */ (function () {
    function WriteBatch(_firestore) {
        this._firestore = _firestore;
        this._mutations = [];
        this._committed = false;
    }
    WriteBatch.prototype.set = function (documentRef, value, options) {
        validateBetweenNumberOfArgs('WriteBatch.set', arguments, 2, 3);
        this.verifyNotCommitted();
        var ref = validateReference('WriteBatch.set', documentRef, this._firestore);
        options = validateSetOptions('WriteBatch.set', options);
        var parsed = options.merge || options.mergeFields
            ? this._firestore._dataConverter.parseMergeData('WriteBatch.set', value, options.mergeFields)
            : this._firestore._dataConverter.parseSetData('WriteBatch.set', value);
        this._mutations = this._mutations.concat(parsed.toMutations(ref._key, Precondition.NONE));
        return this;
    };
    WriteBatch.prototype.update = function (documentRef, fieldOrUpdateData, value) {
        var moreFieldsAndValues = [];
        for (var _i = 3; _i < arguments.length; _i++) {
            moreFieldsAndValues[_i - 3] = arguments[_i];
        }
        this.verifyNotCommitted();
        var ref;
        var parsed;
        if (typeof fieldOrUpdateData === 'string' ||
            fieldOrUpdateData instanceof FieldPath$1) {
            validateAtLeastNumberOfArgs('WriteBatch.update', arguments, 3);
            ref = validateReference('WriteBatch.update', documentRef, this._firestore);
            parsed = this._firestore._dataConverter.parseUpdateVarargs('WriteBatch.update', fieldOrUpdateData, value, moreFieldsAndValues);
        }
        else {
            validateExactNumberOfArgs('WriteBatch.update', arguments, 2);
            ref = validateReference('WriteBatch.update', documentRef, this._firestore);
            parsed = this._firestore._dataConverter.parseUpdateData('WriteBatch.update', fieldOrUpdateData);
        }
        this._mutations = this._mutations.concat(parsed.toMutations(ref._key, Precondition.exists(true)));
        return this;
    };
    WriteBatch.prototype.delete = function (documentRef) {
        validateExactNumberOfArgs('WriteBatch.delete', arguments, 1);
        this.verifyNotCommitted();
        var ref = validateReference('WriteBatch.delete', documentRef, this._firestore);
        this._mutations = this._mutations.concat(new DeleteMutation(ref._key, Precondition.NONE));
        return this;
    };
    WriteBatch.prototype.commit = function () {
        return tslib_1.__awaiter(this, void 0, void 0, function () {
            return tslib_1.__generator(this, function (_a) {
                this.verifyNotCommitted();
                this._committed = true;
                if (this._mutations.length > 0) {
                    return [2 /*return*/, this._firestore.ensureClientConfigured().write(this._mutations)];
                }
                return [2 /*return*/];
            });
        });
    };
    WriteBatch.prototype.verifyNotCommitted = function () {
        if (this._committed) {
            throw new FirestoreError(Code.FAILED_PRECONDITION, 'A write batch can no longer be used after commit() ' +
                'has been called.');
        }
    };
    return WriteBatch;
}());
/**
 * A reference to a particular document in a collection in the database.
 */
var DocumentReference = /** @class */ (function () {
    function DocumentReference(_key, firestore) {
        this._key = _key;
        this.firestore = firestore;
        this._firestoreClient = this.firestore.ensureClientConfigured();
    }
    DocumentReference.forPath = function (path, firestore) {
        if (path.length % 2 !== 0) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid document reference. Document ' +
                'references must have an even number of segments, but ' +
                (path.canonicalString() + " has " + path.length));
        }
        return new DocumentReference(new DocumentKey(path), firestore);
    };
    Object.defineProperty(DocumentReference.prototype, "id", {
        get: function () {
            return this._key.path.lastSegment();
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(DocumentReference.prototype, "parent", {
        get: function () {
            return new CollectionReference(this._key.path.popLast(), this.firestore);
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(DocumentReference.prototype, "path", {
        get: function () {
            return this._key.path.canonicalString();
        },
        enumerable: true,
        configurable: true
    });
    DocumentReference.prototype.collection = function (pathString) {
        validateExactNumberOfArgs('DocumentReference.collection', arguments, 1);
        validateArgType('DocumentReference.collection', 'non-empty string', 1, pathString);
        if (!pathString) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Must provide a non-empty collection name to collection()');
        }
        var path = ResourcePath.fromString(pathString);
        return new CollectionReference(this._key.path.child(path), this.firestore);
    };
    DocumentReference.prototype.isEqual = function (other) {
        if (!(other instanceof DocumentReference)) {
            throw invalidClassError('isEqual', 'DocumentReference', 1, other);
        }
        return this.firestore === other.firestore && this._key.isEqual(other._key);
    };
    DocumentReference.prototype.set = function (value, options) {
        validateBetweenNumberOfArgs('DocumentReference.set', arguments, 1, 2);
        options = validateSetOptions('DocumentReference.set', options);
        var parsed = options.merge || options.mergeFields
            ? this.firestore._dataConverter.parseMergeData('DocumentReference.set', value, options.mergeFields)
            : this.firestore._dataConverter.parseSetData('DocumentReference.set', value);
        return this._firestoreClient.write(parsed.toMutations(this._key, Precondition.NONE));
    };
    DocumentReference.prototype.update = function (fieldOrUpdateData, value) {
        var moreFieldsAndValues = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            moreFieldsAndValues[_i - 2] = arguments[_i];
        }
        var parsed;
        if (typeof fieldOrUpdateData === 'string' ||
            fieldOrUpdateData instanceof FieldPath$1) {
            validateAtLeastNumberOfArgs('DocumentReference.update', arguments, 2);
            parsed = this.firestore._dataConverter.parseUpdateVarargs('DocumentReference.update', fieldOrUpdateData, value, moreFieldsAndValues);
        }
        else {
            validateExactNumberOfArgs('DocumentReference.update', arguments, 1);
            parsed = this.firestore._dataConverter.parseUpdateData('DocumentReference.update', fieldOrUpdateData);
        }
        return this._firestoreClient.write(parsed.toMutations(this._key, Precondition.exists(true)));
    };
    DocumentReference.prototype.delete = function () {
        validateExactNumberOfArgs('DocumentReference.delete', arguments, 0);
        return this._firestoreClient.write([
            new DeleteMutation(this._key, Precondition.NONE)
        ]);
    };
    DocumentReference.prototype.onSnapshot = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        validateBetweenNumberOfArgs('DocumentReference.onSnapshot', arguments, 1, 4);
        var options = {
            includeMetadataChanges: false
        };
        var observer;
        var currArg = 0;
        if (typeof args[currArg] === 'object' &&
            !isPartialObserver(args[currArg])) {
            options = args[currArg];
            validateOptionNames('DocumentReference.onSnapshot', options, [
                'includeMetadataChanges'
            ]);
            validateNamedOptionalType('DocumentReference.onSnapshot', 'boolean', 'includeMetadataChanges', options.includeMetadataChanges);
            currArg++;
        }
        var internalOptions = {
            includeMetadataChanges: options.includeMetadataChanges
        };
        if (isPartialObserver(args[currArg])) {
            observer = args[currArg];
        }
        else {
            validateArgType('DocumentReference.onSnapshot', 'function', currArg, args[currArg]);
            validateOptionalArgType('DocumentReference.onSnapshot', 'function', currArg + 1, args[currArg + 1]);
            validateOptionalArgType('DocumentReference.onSnapshot', 'function', currArg + 2, args[currArg + 2]);
            observer = {
                next: args[currArg],
                error: args[currArg + 1],
                complete: args[currArg + 2]
            };
        }
        return this.onSnapshotInternal(internalOptions, observer);
    };
    DocumentReference.prototype.onSnapshotInternal = function (options, observer) {
        var _this = this;
        var errHandler = function (err) {
            console.error('Uncaught Error in onSnapshot:', err);
        };
        if (observer.error) {
            errHandler = observer.error.bind(observer);
        }
        var asyncObserver = new AsyncObserver({
            next: function (snapshot) {
                if (observer.next) {
                    assert(snapshot.docs.size <= 1, 'Too many documents returned on a document query');
                    var doc = snapshot.docs.get(_this._key);
                    observer.next(new DocumentSnapshot(_this.firestore, _this._key, doc, snapshot.fromCache, snapshot.hasPendingWrites));
                }
            },
            error: errHandler
        });
        var internalListener = this._firestoreClient.listen(Query.atPath(this._key.path), asyncObserver, options);
        return function () {
            asyncObserver.mute();
            _this._firestoreClient.unlisten(internalListener);
        };
    };
    DocumentReference.prototype.get = function (options) {
        var _this = this;
        validateBetweenNumberOfArgs('DocumentReference.get', arguments, 0, 1);
        validateGetOptions('DocumentReference.get', options);
        return new Promise(function (resolve, reject) {
            if (options && options.source === 'cache') {
                _this.firestore
                    .ensureClientConfigured()
                    .getDocumentFromLocalCache(_this._key)
                    .then(function (doc) {
                    resolve(new DocumentSnapshot(_this.firestore, _this._key, doc, 
                    /*fromCache=*/ true, doc instanceof Document ? doc.hasLocalMutations : false));
                }, reject);
            }
            else {
                _this.getViaSnapshotListener(resolve, reject, options);
            }
        });
    };
    DocumentReference.prototype.getViaSnapshotListener = function (resolve, reject, options) {
        var unlisten = this.onSnapshotInternal({
            includeMetadataChanges: true,
            waitForSyncWhenOnline: true
        }, {
            next: function (snap) {
                // Remove query first before passing event to user to avoid
                // user actions affecting the now stale query.
                unlisten();
                if (!snap.exists && snap.metadata.fromCache) {
                    // TODO(dimond): If we're online and the document doesn't
                    // exist then we resolve with a doc.exists set to false. If
                    // we're offline however, we reject the Promise in this
                    // case. Two options: 1) Cache the negative response from
                    // the server so we can deliver that even when you're
                    // offline 2) Actually reject the Promise in the online case
                    // if the document doesn't exist.
                    reject(new FirestoreError(Code.UNAVAILABLE, 'Failed to get document because the client is ' + 'offline.'));
                }
                else if (snap.exists &&
                    snap.metadata.fromCache &&
                    options &&
                    options.source === 'server') {
                    reject(new FirestoreError(Code.UNAVAILABLE, 'Failed to get document from server. (However, this ' +
                        'document does exist in the local cache. Run again ' +
                        'without setting source to "server" to ' +
                        'retrieve the cached document.)'));
                }
                else {
                    resolve(snap);
                }
            },
            error: reject
        });
    };
    return DocumentReference;
}());
var SnapshotMetadata = /** @class */ (function () {
    function SnapshotMetadata(hasPendingWrites, fromCache) {
        this.hasPendingWrites = hasPendingWrites;
        this.fromCache = fromCache;
    }
    SnapshotMetadata.prototype.isEqual = function (other) {
        return (this.hasPendingWrites === other.hasPendingWrites &&
            this.fromCache === other.fromCache);
    };
    return SnapshotMetadata;
}());
var DocumentSnapshot = /** @class */ (function () {
    function DocumentSnapshot(_firestore, _key, _document, _fromCache, _hasPendingWrites) {
        this._firestore = _firestore;
        this._key = _key;
        this._document = _document;
        this._fromCache = _fromCache;
        this._hasPendingWrites = _hasPendingWrites;
    }
    DocumentSnapshot.prototype.data = function (options) {
        validateBetweenNumberOfArgs('DocumentSnapshot.data', arguments, 0, 1);
        options = validateSnapshotOptions('DocumentSnapshot.data', options);
        return !this._document
            ? undefined
            : this.convertObject(this._document.data, FieldValueOptions.fromSnapshotOptions(options, this._firestore._areTimestampsInSnapshotsEnabled()));
    };
    DocumentSnapshot.prototype.get = function (fieldPath, options) {
        validateBetweenNumberOfArgs('DocumentSnapshot.get', arguments, 1, 2);
        options = validateSnapshotOptions('DocumentSnapshot.get', options);
        if (this._document) {
            var value = this._document.data.field(fieldPathFromArgument('DocumentSnapshot.get', fieldPath));
            if (value !== undefined) {
                return this.convertValue(value, FieldValueOptions.fromSnapshotOptions(options, this._firestore._areTimestampsInSnapshotsEnabled()));
            }
        }
        return undefined;
    };
    Object.defineProperty(DocumentSnapshot.prototype, "id", {
        get: function () {
            return this._key.path.lastSegment();
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(DocumentSnapshot.prototype, "ref", {
        get: function () {
            return new DocumentReference(this._key, this._firestore);
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(DocumentSnapshot.prototype, "exists", {
        get: function () {
            return this._document !== null;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(DocumentSnapshot.prototype, "metadata", {
        get: function () {
            return new SnapshotMetadata(this._hasPendingWrites, this._fromCache);
        },
        enumerable: true,
        configurable: true
    });
    DocumentSnapshot.prototype.isEqual = function (other) {
        if (!(other instanceof DocumentSnapshot)) {
            throw invalidClassError('isEqual', 'DocumentSnapshot', 1, other);
        }
        return (this._firestore === other._firestore &&
            this._fromCache === other._fromCache &&
            this._key.isEqual(other._key) &&
            (this._document === null
                ? other._document === null
                : this._document.isEqual(other._document)));
    };
    DocumentSnapshot.prototype.convertObject = function (data, options) {
        var _this = this;
        var result = {};
        data.forEach(function (key, value) {
            result[key] = _this.convertValue(value, options);
        });
        return result;
    };
    DocumentSnapshot.prototype.convertValue = function (value, options) {
        if (value instanceof ObjectValue) {
            return this.convertObject(value, options);
        }
        else if (value instanceof ArrayValue) {
            return this.convertArray(value, options);
        }
        else if (value instanceof RefValue) {
            var key = value.value(options);
            var database = this._firestore.ensureClientConfigured().databaseId();
            if (!value.databaseId.isEqual(database)) {
                // TODO(b/64130202): Somehow support foreign references.
                error("Document " + this._key.path + " contains a document " +
                    "reference within a different database (" +
                    (value.databaseId.projectId + "/" + value.databaseId.database + ") which is not ") +
                    "supported. It will be treated as a reference in the current " +
                    ("database (" + database.projectId + "/" + database.database + ") ") +
                    "instead.");
            }
            return new DocumentReference(key, this._firestore);
        }
        else {
            return value.value(options);
        }
    };
    DocumentSnapshot.prototype.convertArray = function (data, options) {
        var _this = this;
        return data.internalValue.map(function (value) {
            return _this.convertValue(value, options);
        });
    };
    return DocumentSnapshot;
}());
var QueryDocumentSnapshot = /** @class */ (function (_super) {
    tslib_1.__extends(QueryDocumentSnapshot, _super);
    function QueryDocumentSnapshot(firestore, key, document, fromCache, hasPendingWrites) {
        return _super.call(this, firestore, key, document, fromCache, hasPendingWrites) || this;
    }
    QueryDocumentSnapshot.prototype.data = function (options) {
        var data = _super.prototype.data.call(this, options);
        assert(typeof data === 'object', 'Document in a QueryDocumentSnapshot should exist');
        return data;
    };
    return QueryDocumentSnapshot;
}(DocumentSnapshot));
var Query$1 = /** @class */ (function () {
    function Query(_query, firestore) {
        this._query = _query;
        this.firestore = firestore;
    }
    Query.prototype.where = function (field, opStr, value) {
        validateExactNumberOfArgs('Query.where', arguments, 3);
        validateArgType('Query.where', 'non-empty string', 2, opStr);
        validateDefined('Query.where', 3, value);
        var fieldValue;
        var fieldPath = fieldPathFromArgument('Query.where', field);
        var relationOp = RelationOp.fromString(opStr);
        if (fieldPath.isKeyField()) {
            if (relationOp === RelationOp.ARRAY_CONTAINS) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid Query. You can't perform array-contains queries on " +
                    'FieldPath.documentId() since document IDs are not arrays.');
            }
            if (typeof value === 'string') {
                if (value === '') {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, 'Function Query.where() requires its third parameter to be a ' +
                        'valid document ID if the first parameter is ' +
                        'FieldPath.documentId(), but it was an empty string.');
                }
                if (!this._query.isCollectionGroupQuery() &&
                    value.indexOf('/') !== -1) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid third parameter to Query.where(). When querying a collection by " +
                        "FieldPath.documentId(), the value provided must be a plain document ID, but " +
                        ("'" + value + "' contains a slash."));
                }
                var path = this._query.path.child(ResourcePath.fromString(value));
                if (!DocumentKey.isDocumentKey(path)) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid third parameter to Query.where(). When querying a collection group by " +
                        "FieldPath.documentId(), the value provided must result in a valid document path, " +
                        ("but '" + path + "' is not because it has an odd number of segments (" + path.length + ")."));
                }
                fieldValue = new RefValue(this.firestore._databaseId, new DocumentKey(path));
            }
            else if (value instanceof DocumentReference) {
                var ref = value;
                fieldValue = new RefValue(this.firestore._databaseId, ref._key);
            }
            else {
                throw new FirestoreError(Code.INVALID_ARGUMENT, "Function Query.where() requires its third parameter to be a " +
                    "string or a DocumentReference if the first parameter is " +
                    "FieldPath.documentId(), but it was: " +
                    (valueDescription(value) + "."));
            }
        }
        else {
            fieldValue = this.firestore._dataConverter.parseQueryValue('Query.where', value);
        }
        var filter = Filter.create(fieldPath, relationOp, fieldValue);
        this.validateNewFilter(filter);
        return new Query(this._query.addFilter(filter), this.firestore);
    };
    Query.prototype.orderBy = function (field, directionStr) {
        validateBetweenNumberOfArgs('Query.orderBy', arguments, 1, 2);
        validateOptionalArgType('Query.orderBy', 'non-empty string', 2, directionStr);
        var direction;
        if (directionStr === undefined || directionStr === 'asc') {
            direction = Direction.ASCENDING;
        }
        else if (directionStr === 'desc') {
            direction = Direction.DESCENDING;
        }
        else {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Function Query.orderBy() has unknown direction '" + directionStr + "', " +
                "expected 'asc' or 'desc'.");
        }
        if (this._query.startAt !== null) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid query. You must not call Query.startAt() or ' +
                'Query.startAfter() before calling Query.orderBy().');
        }
        if (this._query.endAt !== null) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid query. You must not call Query.endAt() or ' +
                'Query.endBefore() before calling Query.orderBy().');
        }
        var fieldPath = fieldPathFromArgument('Query.orderBy', field);
        var orderBy = new OrderBy(fieldPath, direction);
        this.validateNewOrderBy(orderBy);
        return new Query(this._query.addOrderBy(orderBy), this.firestore);
    };
    Query.prototype.limit = function (n) {
        validateExactNumberOfArgs('Query.limit', arguments, 1);
        validateArgType('Query.limit', 'number', 1, n);
        if (n <= 0) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid Query. Query limit (" + n + ") is invalid. Limit must be " +
                'positive.');
        }
        return new Query(this._query.withLimit(n), this.firestore);
    };
    Query.prototype.startAt = function (docOrField) {
        var fields = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            fields[_i - 1] = arguments[_i];
        }
        validateAtLeastNumberOfArgs('Query.startAt', arguments, 1);
        var bound = this.boundFromDocOrFields('Query.startAt', docOrField, fields, 
        /*before=*/ true);
        return new Query(this._query.withStartAt(bound), this.firestore);
    };
    Query.prototype.startAfter = function (docOrField) {
        var fields = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            fields[_i - 1] = arguments[_i];
        }
        validateAtLeastNumberOfArgs('Query.startAfter', arguments, 1);
        var bound = this.boundFromDocOrFields('Query.startAfter', docOrField, fields, 
        /*before=*/ false);
        return new Query(this._query.withStartAt(bound), this.firestore);
    };
    Query.prototype.endBefore = function (docOrField) {
        var fields = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            fields[_i - 1] = arguments[_i];
        }
        validateAtLeastNumberOfArgs('Query.endBefore', arguments, 1);
        var bound = this.boundFromDocOrFields('Query.endBefore', docOrField, fields, 
        /*before=*/ true);
        return new Query(this._query.withEndAt(bound), this.firestore);
    };
    Query.prototype.endAt = function (docOrField) {
        var fields = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            fields[_i - 1] = arguments[_i];
        }
        validateAtLeastNumberOfArgs('Query.endAt', arguments, 1);
        var bound = this.boundFromDocOrFields('Query.endAt', docOrField, fields, 
        /*before=*/ false);
        return new Query(this._query.withEndAt(bound), this.firestore);
    };
    Query.prototype.isEqual = function (other) {
        if (!(other instanceof Query)) {
            throw invalidClassError('isEqual', 'Query', 1, other);
        }
        return (this.firestore === other.firestore && this._query.isEqual(other._query));
    };
    /** Helper function to create a bound from a document or fields */
    Query.prototype.boundFromDocOrFields = function (methodName, docOrField, fields, before) {
        validateDefined(methodName, 1, docOrField);
        if (docOrField instanceof DocumentSnapshot) {
            if (fields.length > 0) {
                throw new FirestoreError(Code.INVALID_ARGUMENT, "Too many arguments provided to " + methodName + "().");
            }
            var snap = docOrField;
            if (!snap.exists) {
                throw new FirestoreError(Code.NOT_FOUND, "Can't use a DocumentSnapshot that doesn't exist for " +
                    (methodName + "()."));
            }
            return this.boundFromDocument(methodName, snap._document, before);
        }
        else {
            var allFields = [docOrField].concat(fields);
            return this.boundFromFields(methodName, allFields, before);
        }
    };
    /**
     * Create a Bound from a query and a document.
     *
     * Note that the Bound will always include the key of the document
     * and so only the provided document will compare equal to the returned
     * position.
     *
     * Will throw if the document does not contain all fields of the order by
     * of the query or if any of the fields in the order by are an uncommitted
     * server timestamp.
     */
    Query.prototype.boundFromDocument = function (methodName, doc, before) {
        var components = [];
        // Because people expect to continue/end a query at the exact document
        // provided, we need to use the implicit sort order rather than the explicit
        // sort order, because it's guaranteed to contain the document key. That way
        // the position becomes unambiguous and the query continues/ends exactly at
        // the provided document. Without the key (by using the explicit sort
        // orders), multiple documents could match the position, yielding duplicate
        // results.
        for (var _i = 0, _a = this._query.orderBy; _i < _a.length; _i++) {
            var orderBy = _a[_i];
            if (orderBy.field.isKeyField()) {
                components.push(new RefValue(this.firestore._databaseId, doc.key));
            }
            else {
                var value = doc.field(orderBy.field);
                if (value instanceof ServerTimestampValue) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid query. You are trying to start or end a query using a ' +
                        'document for which the field "' +
                        orderBy.field +
                        '" is an uncommitted server timestamp. (Since the value of ' +
                        'this field is unknown, you cannot start/end a query with it.)');
                }
                else if (value !== undefined) {
                    components.push(value);
                }
                else {
                    var field = orderBy.field.canonicalString();
                    throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid query. You are trying to start or end a query using a " +
                        ("document for which the field '" + field + "' (used as the ") +
                        "orderBy) does not exist.");
                }
            }
        }
        return new Bound(components, before);
    };
    /**
     * Converts a list of field values to a Bound for the given query.
     */
    Query.prototype.boundFromFields = function (methodName, values, before) {
        // Use explicit order by's because it has to match the query the user made
        var orderBy = this._query.explicitOrderBy;
        if (values.length > orderBy.length) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Too many arguments provided to " + methodName + "(). " +
                "The number of arguments must be less than or equal to the " +
                "number of Query.orderBy() clauses");
        }
        var components = [];
        for (var i = 0; i < values.length; i++) {
            var rawValue = values[i];
            var orderByComponent = orderBy[i];
            if (orderByComponent.field.isKeyField()) {
                if (typeof rawValue !== 'string') {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid query. Expected a string for document ID in " +
                        (methodName + "(), but got a " + typeof rawValue));
                }
                if (!this._query.isCollectionGroupQuery() &&
                    rawValue.indexOf('/') !== -1) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid query. When querying a collection and ordering by FieldPath.documentId(), " +
                        ("the value passed to " + methodName + "() must be a plain document ID, but ") +
                        ("'" + rawValue + "' contains a slash."));
                }
                var path = this._query.path.child(ResourcePath.fromString(rawValue));
                if (!DocumentKey.isDocumentKey(path)) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid query. When querying a collection group and ordering by " +
                        ("FieldPath.documentId(), the value passed to " + methodName + "() must result in a ") +
                        ("valid document path, but '" + path + "' is not because it contains an odd number ") +
                        "of segments.");
                }
                var key = new DocumentKey(path);
                components.push(new RefValue(this.firestore._databaseId, key));
            }
            else {
                var wrapped = this.firestore._dataConverter.parseQueryValue(methodName, rawValue);
                components.push(wrapped);
            }
        }
        return new Bound(components, before);
    };
    Query.prototype.onSnapshot = function () {
        var args = [];
        for (var _i = 0; _i < arguments.length; _i++) {
            args[_i] = arguments[_i];
        }
        validateBetweenNumberOfArgs('Query.onSnapshot', arguments, 1, 4);
        var options = {};
        var observer;
        var currArg = 0;
        if (typeof args[currArg] === 'object' &&
            !isPartialObserver(args[currArg])) {
            options = args[currArg];
            validateOptionNames('Query.onSnapshot', options, [
                'includeMetadataChanges'
            ]);
            validateNamedOptionalType('Query.onSnapshot', 'boolean', 'includeMetadataChanges', options.includeMetadataChanges);
            currArg++;
        }
        if (isPartialObserver(args[currArg])) {
            observer = args[currArg];
        }
        else {
            validateArgType('Query.onSnapshot', 'function', currArg, args[currArg]);
            validateOptionalArgType('Query.onSnapshot', 'function', currArg + 1, args[currArg + 1]);
            validateOptionalArgType('Query.onSnapshot', 'function', currArg + 2, args[currArg + 2]);
            observer = {
                next: args[currArg],
                error: args[currArg + 1],
                complete: args[currArg + 2]
            };
        }
        return this.onSnapshotInternal(options, observer);
    };
    Query.prototype.onSnapshotInternal = function (options, observer) {
        var _this = this;
        var errHandler = function (err) {
            console.error('Uncaught Error in onSnapshot:', err);
        };
        if (observer.error) {
            errHandler = observer.error.bind(observer);
        }
        var asyncObserver = new AsyncObserver({
            next: function (result) {
                if (observer.next) {
                    observer.next(new QuerySnapshot(_this.firestore, _this._query, result));
                }
            },
            error: errHandler
        });
        var firestoreClient = this.firestore.ensureClientConfigured();
        var internalListener = firestoreClient.listen(this._query, asyncObserver, options);
        return function () {
            asyncObserver.mute();
            firestoreClient.unlisten(internalListener);
        };
    };
    Query.prototype.get = function (options) {
        var _this = this;
        validateBetweenNumberOfArgs('Query.get', arguments, 0, 1);
        validateGetOptions('Query.get', options);
        return new Promise(function (resolve, reject) {
            if (options && options.source === 'cache') {
                _this.firestore
                    .ensureClientConfigured()
                    .getDocumentsFromLocalCache(_this._query)
                    .then(function (viewSnap) {
                    resolve(new QuerySnapshot(_this.firestore, _this._query, viewSnap));
                }, reject);
            }
            else {
                _this.getViaSnapshotListener(resolve, reject, options);
            }
        });
    };
    Query.prototype.getViaSnapshotListener = function (resolve, reject, options) {
        var unlisten = this.onSnapshotInternal({
            includeMetadataChanges: true,
            waitForSyncWhenOnline: true
        }, {
            next: function (result) {
                // Remove query first before passing event to user to avoid
                // user actions affecting the now stale query.
                unlisten();
                if (result.metadata.fromCache &&
                    options &&
                    options.source === 'server') {
                    reject(new FirestoreError(Code.UNAVAILABLE, 'Failed to get documents from server. (However, these ' +
                        'documents may exist in the local cache. Run again ' +
                        'without setting source to "server" to ' +
                        'retrieve the cached documents.)'));
                }
                else {
                    resolve(result);
                }
            },
            error: reject
        });
    };
    Query.prototype.validateNewFilter = function (filter) {
        if (filter instanceof RelationFilter) {
            if (filter.isInequality()) {
                var existingField = this._query.getInequalityFilterField();
                if (existingField !== null && !existingField.isEqual(filter.field)) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid query. All where filters with an inequality' +
                        ' (<, <=, >, or >=) must be on the same field. But you have' +
                        (" inequality filters on '" + existingField.toString() + "'") +
                        (" and '" + filter.field.toString() + "'"));
                }
                var firstOrderByField = this._query.getFirstOrderByField();
                if (firstOrderByField !== null) {
                    this.validateOrderByAndInequalityMatch(filter.field, firstOrderByField);
                }
            }
            else if (filter.op === RelationOp.ARRAY_CONTAINS) {
                if (this._query.hasArrayContainsFilter()) {
                    throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid query. Queries only support a single array-contains ' +
                        'filter.');
                }
            }
        }
    };
    Query.prototype.validateNewOrderBy = function (orderBy) {
        if (this._query.getFirstOrderByField() === null) {
            // This is the first order by. It must match any inequality.
            var inequalityField = this._query.getInequalityFilterField();
            if (inequalityField !== null) {
                this.validateOrderByAndInequalityMatch(inequalityField, orderBy.field);
            }
        }
    };
    Query.prototype.validateOrderByAndInequalityMatch = function (inequality, orderBy) {
        if (!orderBy.isEqual(inequality)) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid query. You have a where filter with an inequality " +
                ("(<, <=, >, or >=) on field '" + inequality.toString() + "' ") +
                ("and so you must also use '" + inequality.toString() + "' ") +
                "as your first Query.orderBy(), but your first Query.orderBy() " +
                ("is on field '" + orderBy.toString() + "' instead."));
        }
    };
    return Query;
}());
var QuerySnapshot = /** @class */ (function () {
    function QuerySnapshot(_firestore, _originalQuery, _snapshot) {
        this._firestore = _firestore;
        this._originalQuery = _originalQuery;
        this._snapshot = _snapshot;
        this._cachedChanges = null;
        this._cachedChangesIncludeMetadataChanges = null;
        this.metadata = new SnapshotMetadata(_snapshot.hasPendingWrites, _snapshot.fromCache);
    }
    Object.defineProperty(QuerySnapshot.prototype, "docs", {
        get: function () {
            var result = [];
            this.forEach(function (doc) { return result.push(doc); });
            return result;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(QuerySnapshot.prototype, "empty", {
        get: function () {
            return this._snapshot.docs.isEmpty();
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(QuerySnapshot.prototype, "size", {
        get: function () {
            return this._snapshot.docs.size;
        },
        enumerable: true,
        configurable: true
    });
    QuerySnapshot.prototype.forEach = function (callback, thisArg) {
        var _this = this;
        validateBetweenNumberOfArgs('QuerySnapshot.forEach', arguments, 1, 2);
        validateArgType('QuerySnapshot.forEach', 'function', 1, callback);
        this._snapshot.docs.forEach(function (doc) {
            callback.call(thisArg, _this.convertToDocumentImpl(doc));
        });
    };
    Object.defineProperty(QuerySnapshot.prototype, "query", {
        get: function () {
            return new Query$1(this._originalQuery, this._firestore);
        },
        enumerable: true,
        configurable: true
    });
    QuerySnapshot.prototype.docChanges = function (options) {
        if (options) {
            validateOptionNames('QuerySnapshot.docChanges', options, [
                'includeMetadataChanges'
            ]);
            validateNamedOptionalType('QuerySnapshot.docChanges', 'boolean', 'includeMetadataChanges', options.includeMetadataChanges);
        }
        var includeMetadataChanges = !!(options && options.includeMetadataChanges);
        if (includeMetadataChanges && this._snapshot.excludesMetadataChanges) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'To include metadata changes with your document changes, you must ' +
                'also pass { includeMetadataChanges:true } to onSnapshot().');
        }
        if (!this._cachedChanges ||
            this._cachedChangesIncludeMetadataChanges !== includeMetadataChanges) {
            this._cachedChanges = changesFromSnapshot(this._firestore, includeMetadataChanges, this._snapshot);
            this._cachedChangesIncludeMetadataChanges = includeMetadataChanges;
        }
        return this._cachedChanges;
    };
    /** Check the equality. The call can be very expensive. */
    QuerySnapshot.prototype.isEqual = function (other) {
        if (!(other instanceof QuerySnapshot)) {
            throw invalidClassError('isEqual', 'QuerySnapshot', 1, other);
        }
        return (this._firestore === other._firestore &&
            this._originalQuery.isEqual(other._originalQuery) &&
            this._snapshot.isEqual(other._snapshot));
    };
    QuerySnapshot.prototype.convertToDocumentImpl = function (doc) {
        return new QueryDocumentSnapshot(this._firestore, doc.key, doc, this.metadata.fromCache, this._snapshot.mutatedKeys.has(doc.key));
    };
    return QuerySnapshot;
}());
// TODO(2018/11/01): As of 2018/04/17 we're changing docChanges from an array
// into a method. Because this is a runtime breaking change and somewhat subtle
// (both Array and Function have a .length, etc.), we'll replace commonly-used
// properties (including Symbol.iterator) to throw a custom error message. In
// ~6 months we can delete the custom error as most folks will have hopefully
// migrated.
function throwDocChangesMethodError() {
    throw new FirestoreError(Code.INVALID_ARGUMENT, 'QuerySnapshot.docChanges has been changed from a property into a ' +
        'method, so usages like "querySnapshot.docChanges" should become ' +
        '"querySnapshot.docChanges()"');
}
var docChangesPropertiesToOverride = [
    'length',
    'forEach',
    'map'
].concat((typeof Symbol !== 'undefined' ? [Symbol.iterator] : []));
docChangesPropertiesToOverride.forEach(function (property) {
    /**
     * We are (re-)defining properties on QuerySnapshot.prototype.docChanges which
     * is a Function. This could fail, in particular in the case of 'length' which
     * already exists on Function.prototype and on IE11 is improperly defined with
     * `{ configurable: false }`. So we wrap this in a try/catch to ensure that we
     * still have a functional SDK.
     */
    try {
        Object.defineProperty(QuerySnapshot.prototype.docChanges, property, {
            get: function () { return throwDocChangesMethodError(); }
        });
    }
    catch (err) { } // Ignore this failure intentionally
});
var CollectionReference = /** @class */ (function (_super) {
    tslib_1.__extends(CollectionReference, _super);
    function CollectionReference(path, firestore) {
        var _this = _super.call(this, Query.atPath(path), firestore) || this;
        if (path.length % 2 !== 1) {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Invalid collection reference. Collection ' +
                'references must have an odd number of segments, but ' +
                (path.canonicalString() + " has " + path.length));
        }
        return _this;
    }
    Object.defineProperty(CollectionReference.prototype, "id", {
        get: function () {
            return this._query.path.lastSegment();
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(CollectionReference.prototype, "parent", {
        get: function () {
            var parentPath = this._query.path.popLast();
            if (parentPath.isEmpty()) {
                return null;
            }
            else {
                return new DocumentReference(new DocumentKey(parentPath), this.firestore);
            }
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(CollectionReference.prototype, "path", {
        get: function () {
            return this._query.path.canonicalString();
        },
        enumerable: true,
        configurable: true
    });
    CollectionReference.prototype.doc = function (pathString) {
        validateBetweenNumberOfArgs('CollectionReference.doc', arguments, 0, 1);
        // We allow omission of 'pathString' but explicitly prohibit passing in both
        // 'undefined' and 'null'.
        if (arguments.length === 0) {
            pathString = AutoId.newId();
        }
        validateArgType('CollectionReference.doc', 'non-empty string', 1, pathString);
        if (pathString === '') {
            throw new FirestoreError(Code.INVALID_ARGUMENT, 'Document path must be a non-empty string');
        }
        var path = ResourcePath.fromString(pathString);
        return DocumentReference.forPath(this._query.path.child(path), this.firestore);
    };
    CollectionReference.prototype.add = function (value) {
        validateExactNumberOfArgs('CollectionReference.add', arguments, 1);
        validateArgType('CollectionReference.add', 'object', 1, value);
        var docRef = this.doc();
        return docRef.set(value).then(function () { return docRef; });
    };
    return CollectionReference;
}(Query$1));
function validateSetOptions(methodName, options) {
    if (options === undefined) {
        return {
            merge: false
        };
    }
    validateOptionNames(methodName, options, ['merge', 'mergeFields']);
    validateNamedOptionalType(methodName, 'boolean', 'merge', options.merge);
    validateOptionalArrayElements(methodName, 'mergeFields', 'a string or a FieldPath', options.mergeFields, function (element) {
        return typeof element === 'string' || element instanceof FieldPath$1;
    });
    if (options.mergeFields !== undefined && options.merge !== undefined) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, "Invalid options passed to function " + methodName + "(): You cannot specify both \"merge\" " +
            "and \"mergeFields\".");
    }
    return options;
}
function validateSnapshotOptions(methodName, options) {
    if (options === undefined) {
        return {};
    }
    validateOptionNames(methodName, options, ['serverTimestamps']);
    validateNamedOptionalPropertyEquals(methodName, 'options', 'serverTimestamps', options.serverTimestamps, ['estimate', 'previous', 'none']);
    return options;
}
function validateGetOptions(methodName, options) {
    validateOptionalArgType(methodName, 'object', 1, options);
    if (options) {
        validateOptionNames(methodName, options, ['source']);
        validateNamedOptionalPropertyEquals(methodName, 'options', 'source', options.source, ['default', 'server', 'cache']);
    }
}
function validateReference(methodName, documentRef, firestore) {
    if (!(documentRef instanceof DocumentReference)) {
        throw invalidClassError(methodName, 'DocumentReference', 1, documentRef);
    }
    else if (documentRef.firestore !== firestore) {
        throw new FirestoreError(Code.INVALID_ARGUMENT, 'Provided document reference is from a different Firestore instance.');
    }
    else {
        return documentRef;
    }
}
/**
 * Calculates the array of firestore.DocumentChange's for a given ViewSnapshot.
 *
 * Exported for testing.
 */
function changesFromSnapshot(firestore, includeMetadataChanges, snapshot) {
    if (snapshot.oldDocs.isEmpty()) {
        // Special case the first snapshot because index calculation is easy and
        // fast
        var lastDoc_1;
        var index_1 = 0;
        return snapshot.docChanges.map(function (change) {
            var doc = new QueryDocumentSnapshot(firestore, change.doc.key, change.doc, snapshot.fromCache, snapshot.mutatedKeys.has(change.doc.key));
            assert(change.type === ChangeType.Added, 'Invalid event type for first snapshot');
            assert(!lastDoc_1 || snapshot.query.docComparator(lastDoc_1, change.doc) < 0, 'Got added events in wrong order');
            lastDoc_1 = change.doc;
            return {
                type: 'added',
                doc: doc,
                oldIndex: -1,
                newIndex: index_1++
            };
        });
    }
    else {
        // A DocumentSet that is updated incrementally as changes are applied to use
        // to lookup the index of a document.
        var indexTracker_1 = snapshot.oldDocs;
        return snapshot.docChanges
            .filter(function (change) { return includeMetadataChanges || change.type !== ChangeType.Metadata; })
            .map(function (change) {
            var doc = new QueryDocumentSnapshot(firestore, change.doc.key, change.doc, snapshot.fromCache, snapshot.mutatedKeys.has(change.doc.key));
            var oldIndex = -1;
            var newIndex = -1;
            if (change.type !== ChangeType.Added) {
                oldIndex = indexTracker_1.indexOf(change.doc.key);
                assert(oldIndex >= 0, 'Index for document not found');
                indexTracker_1 = indexTracker_1.delete(change.doc.key);
            }
            if (change.type !== ChangeType.Removed) {
                indexTracker_1 = indexTracker_1.add(change.doc);
                newIndex = indexTracker_1.indexOf(change.doc.key);
            }
            return { type: resultChangeType(change.type), doc: doc, oldIndex: oldIndex, newIndex: newIndex };
        });
    }
}
function resultChangeType(type) {
    switch (type) {
        case ChangeType.Added:
            return 'added';
        case ChangeType.Modified:
        case ChangeType.Metadata:
            return 'modified';
        case ChangeType.Removed:
            return 'removed';
        default:
            return fail('Unknown change type: ' + type);
    }
}
// Export the classes with a private constructor (it will fail if invoked
// at runtime). Note that this still allows instanceof checks.
// We're treating the variables as class names, so disable checking for lower
// case variable names.
// tslint:disable:variable-name
var PublicFirestore = makeConstructorPrivate(Firestore, 'Use firebase.firestore() instead.');
var PublicTransaction = makeConstructorPrivate(Transaction$1, 'Use firebase.firestore().runTransaction() instead.');
var PublicWriteBatch = makeConstructorPrivate(WriteBatch, 'Use firebase.firestore().batch() instead.');
var PublicDocumentReference = makeConstructorPrivate(DocumentReference, 'Use firebase.firestore().doc() instead.');
var PublicDocumentSnapshot = makeConstructorPrivate(DocumentSnapshot);
var PublicQueryDocumentSnapshot = makeConstructorPrivate(QueryDocumentSnapshot);
var PublicQuery = makeConstructorPrivate(Query$1);
var PublicQuerySnapshot = makeConstructorPrivate(QuerySnapshot);
var PublicCollectionReference = makeConstructorPrivate(CollectionReference, 'Use firebase.firestore().collection() instead.');
// tslint:enable:variable-name

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var firestoreNamespace = {
    Firestore: PublicFirestore,
    GeoPoint: GeoPoint,
    Timestamp: Timestamp,
    Blob: PublicBlob,
    Transaction: PublicTransaction,
    WriteBatch: PublicWriteBatch,
    DocumentReference: PublicDocumentReference,
    DocumentSnapshot: PublicDocumentSnapshot,
    Query: PublicQuery,
    QueryDocumentSnapshot: PublicQueryDocumentSnapshot,
    QuerySnapshot: PublicQuerySnapshot,
    CollectionReference: PublicCollectionReference,
    FieldPath: FieldPath$1,
    FieldValue: PublicFieldValue,
    setLogLevel: Firestore.setLogLevel,
    CACHE_SIZE_UNLIMITED: CACHE_SIZE_UNLIMITED
};
/**
 * Configures Firestore as part of the Firebase SDK by calling registerService.
 */
function configureForFirebase(firebase) {
    firebase.INTERNAL.registerService('firestore', function (app) { return new Firestore(app); }, shallowCopy(firestoreNamespace));
}

/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
function registerFirestore(instance) {
    configureForFirebase(instance);
}
registerFirestore(firebase);

exports.registerFirestore = registerFirestore;
//# sourceMappingURL=index.node.cjs.js.map
