/**
 * @license
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { Document } from '../model/document';
import { FieldValue } from '../model/field_value';
import { FieldPath, ResourcePath } from '../model/path';
export declare class Query {
    readonly path: ResourcePath;
    readonly collectionGroup: string | null;
    readonly explicitOrderBy: OrderBy[];
    readonly filters: Filter[];
    readonly limit: number | null;
    readonly startAt: Bound | null;
    readonly endAt: Bound | null;
    static atPath(path: ResourcePath): Query;
    private memoizedCanonicalId;
    private memoizedOrderBy;
    /**
     * Initializes a Query with a path and optional additional query constraints.
     * Path must currently be empty if this is a collection group query.
     */
    constructor(path: ResourcePath, collectionGroup?: string | null, explicitOrderBy?: OrderBy[], filters?: Filter[], limit?: number | null, startAt?: Bound | null, endAt?: Bound | null);
    readonly orderBy: OrderBy[];
    addFilter(filter: Filter): Query;
    addOrderBy(orderBy: OrderBy): Query;
    withLimit(limit: number | null): Query;
    withStartAt(bound: Bound): Query;
    withEndAt(bound: Bound): Query;
    /**
     * Helper to convert a collection group query into a collection query at a
     * specific path. This is used when executing collection group queries, since
     * we have to split the query into a set of collection queries at multiple
     * paths.
     */
    asCollectionQueryAtPath(path: ResourcePath): Query;
    canonicalId(): string;
    toString(): string;
    isEqual(other: Query): boolean;
    docComparator(d1: Document, d2: Document): number;
    matches(doc: Document): boolean;
    hasLimit(): boolean;
    getFirstOrderByField(): FieldPath | null;
    getInequalityFilterField(): FieldPath | null;
    hasArrayContainsFilter(): boolean;
    isDocumentQuery(): boolean;
    isCollectionGroupQuery(): boolean;
    private matchesPathAndCollectionGroup;
    /**
     * A document must have a value for every ordering clause in order to show up
     * in the results.
     */
    private matchesOrderBy;
    private matchesFilters;
    /**
     * Makes sure a document is within the bounds, if provided.
     */
    private matchesBounds;
    private assertValidBound;
}
export declare abstract class Filter {
    abstract matches(doc: Document): boolean;
    abstract canonicalId(): string;
    abstract isEqual(filter: Filter): boolean;
    /**
     * Creates a filter based on the provided arguments.
     */
    static create(field: FieldPath, op: RelationOp, value: FieldValue): Filter;
}
export declare class RelationOp {
    name: string;
    static LESS_THAN: RelationOp;
    static LESS_THAN_OR_EQUAL: RelationOp;
    static EQUAL: RelationOp;
    static GREATER_THAN: RelationOp;
    static GREATER_THAN_OR_EQUAL: RelationOp;
    static ARRAY_CONTAINS: RelationOp;
    static fromString(op: string): RelationOp;
    constructor(name: string);
    toString(): string;
    isEqual(other: RelationOp): boolean;
}
export declare class RelationFilter extends Filter {
    field: FieldPath;
    op: RelationOp;
    value: FieldValue;
    constructor(field: FieldPath, op: RelationOp, value: FieldValue);
    matches(doc: Document): boolean;
    private matchesValue;
    private matchesComparison;
    isInequality(): boolean;
    canonicalId(): string;
    isEqual(other: Filter): boolean;
    toString(): string;
}
/**
 * Filter that matches 'null' values.
 */
export declare class NullFilter extends Filter {
    field: FieldPath;
    constructor(field: FieldPath);
    matches(doc: Document): boolean;
    canonicalId(): string;
    toString(): string;
    isEqual(other: Filter): boolean;
}
/**
 * Filter that matches 'NaN' values.
 */
export declare class NanFilter extends Filter {
    field: FieldPath;
    constructor(field: FieldPath);
    matches(doc: Document): boolean;
    canonicalId(): string;
    toString(): string;
    isEqual(other: Filter): boolean;
}
/**
 * The direction of sorting in an order by.
 */
export declare class Direction {
    name: string;
    static ASCENDING: Direction;
    static DESCENDING: Direction;
    private constructor();
    toString(): string;
}
/**
 * Represents a bound of a query.
 *
 * The bound is specified with the given components representing a position and
 * whether it's just before or just after the position (relative to whatever the
 * query order is).
 *
 * The position represents a logical index position for a query. It's a prefix
 * of values for the (potentially implicit) order by clauses of a query.
 *
 * Bound provides a function to determine whether a document comes before or
 * after a bound. This is influenced by whether the position is just before or
 * just after the provided values.
 */
export declare class Bound {
    readonly position: FieldValue[];
    readonly before: boolean;
    constructor(position: FieldValue[], before: boolean);
    canonicalId(): string;
    /**
     * Returns true if a document sorts before a bound using the provided sort
     * order.
     */
    sortsBeforeDocument(orderBy: OrderBy[], doc: Document): boolean;
    isEqual(other: Bound | null): boolean;
}
/**
 * An ordering on a field, in some Direction. Direction defaults to ASCENDING.
 */
export declare class OrderBy {
    readonly field: FieldPath;
    readonly dir: Direction;
    private readonly isKeyOrderBy;
    constructor(field: FieldPath, dir?: Direction);
    compare(d1: Document, d2: Document): number;
    canonicalId(): string;
    toString(): string;
    isEqual(other: OrderBy): boolean;
}
