/**
 * @license
 * Copyright 2018 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { Timestamp } from '../api/timestamp';
import { FieldValue, NumberValue } from './field_value';
/** Represents a transform within a TransformMutation. */
export interface TransformOperation {
    /** Whether this field transform is idempotent. */
    readonly isIdempotent: boolean;
    /**
     * Computes the local transform result against the provided `previousValue`,
     * optionally using the provided localWriteTime.
     */
    applyToLocalView(previousValue: FieldValue | null, localWriteTime: Timestamp): FieldValue;
    /**
     * Computes a final transform result after the transform has been acknowledged
     * by the server, potentially using the server-provided transformResult.
     */
    applyToRemoteDocument(previousValue: FieldValue | null, transformResult: FieldValue | null): FieldValue;
    isEqual(other: TransformOperation): boolean;
}
/** Transforms a value into a server-generated timestamp. */
export declare class ServerTimestampTransform implements TransformOperation {
    readonly isIdempotent = true;
    private constructor();
    static instance: ServerTimestampTransform;
    applyToLocalView(previousValue: FieldValue | null, localWriteTime: Timestamp): FieldValue;
    applyToRemoteDocument(previousValue: FieldValue | null, transformResult: FieldValue | null): FieldValue;
    isEqual(other: TransformOperation): boolean;
}
/** Transforms an array value via a union operation. */
export declare class ArrayUnionTransformOperation implements TransformOperation {
    readonly elements: FieldValue[];
    readonly isIdempotent = true;
    constructor(elements: FieldValue[]);
    applyToLocalView(previousValue: FieldValue | null, localWriteTime: Timestamp): FieldValue;
    applyToRemoteDocument(previousValue: FieldValue | null, transformResult: FieldValue | null): FieldValue;
    private apply;
    isEqual(other: TransformOperation): boolean;
}
/** Transforms an array value via a remove operation. */
export declare class ArrayRemoveTransformOperation implements TransformOperation {
    readonly elements: FieldValue[];
    readonly isIdempotent = true;
    constructor(elements: FieldValue[]);
    applyToLocalView(previousValue: FieldValue | null, localWriteTime: Timestamp): FieldValue;
    applyToRemoteDocument(previousValue: FieldValue | null, transformResult: FieldValue | null): FieldValue;
    private apply;
    isEqual(other: TransformOperation): boolean;
}
/**
 * Implements the backend semantics for locally computed NUMERIC_ADD (increment)
 * transforms. Converts all field values to integers or doubles, but unlike the
 * backend does not cap integer values at 2^63. Instead, JavaScript number
 * arithmetic is used and precision loss can occur for values greater than 2^53.
 */
export declare class NumericIncrementTransformOperation implements TransformOperation {
    readonly operand: NumberValue;
    readonly isIdempotent = false;
    constructor(operand: NumberValue);
    applyToLocalView(previousValue: FieldValue | null, localWriteTime: Timestamp): FieldValue;
    applyToRemoteDocument(previousValue: FieldValue | null, transformResult: FieldValue | null): FieldValue;
    isEqual(other: TransformOperation): boolean;
}
