(function (global, factory) {
	typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory(require('sanctuary-type-classes'), require('inspect-f'), require('sanctuary-type-identifiers'), require('denque'), require('concurrify')) :
	typeof define === 'function' && define.amd ? define(['sanctuary-type-classes', 'inspect-f', 'sanctuary-type-identifiers', 'denque', 'concurrify'], factory) :
	(global.Fluture = factory(global.sanctuaryTypeClasses,global.inspectf,global.sanctuaryTypeIdentifiers,global.Denque,global.concurrify));
}(this, (function (Z,inspectf,type,Denque,concurrify) { 'use strict';

Z = Z && Z.hasOwnProperty('default') ? Z['default'] : Z;
inspectf = inspectf && inspectf.hasOwnProperty('default') ? inspectf['default'] : inspectf;
type = type && type.hasOwnProperty('default') ? type['default'] : type;
Denque = Denque && Denque.hasOwnProperty('default') ? Denque['default'] : Denque;
concurrify = concurrify && concurrify.hasOwnProperty('default') ? concurrify['default'] : concurrify;

/* istanbul ignore file: environment-specific */

/* eslint-disable no-undef */
const scope = typeof self === 'object' ? self :
                     typeof global === 'object' ? global :
                     typeof window === 'object' ? window :
                     {};
/* eslint-enable no-undef */

const setImmediate = typeof scope.setImmediate === 'function' ?
                            scope.setImmediate :
                            function setImmediate(f, x){ return setTimeout(f, 0, x) };

function noop(){}
function moop(){ return this }
var show = Z.toString;
function padf(sf, s){ return s.replace(/^/gm, sf).replace(sf, '') }
function showf(f){ return padf('  ', inspectf(2, f)) }

function mapArray(xs, f){
  var l = xs.length, ys = new Array(l);
  for(var i = 0; i < l; i++) ys[i] = f(xs[i], i, xs);
  return ys;
}

function partial1(f, a){
  return function bound1(b, c, d){
    switch(arguments.length){
      case 1: return f(a, b);
      case 2: return f(a, b, c);
      default: return f(a, b, c, d);
    }
  };
}

function partial2(f, a, b){
  return function bound2(c, d){
    return arguments.length === 1 ? f(a, b, c) : f(a, b, c, d);
  };
}

function partial3(f, a, b, c){
  return function bound3(d){
    return f(a, b, c, d);
  };
}

function immediately(f){
  return function immediate(x){ return setImmediate(f, x) };
}

var FL = {
  map: 'fantasy-land/map',
  bimap: 'fantasy-land/bimap',
  chain: 'fantasy-land/chain',
  chainRec: 'fantasy-land/chainRec',
  ap: 'fantasy-land/ap',
  of: 'fantasy-land/of',
  zero: 'fantasy-land/zero'
};

var ordinal = ['first', 'second', 'third', 'fourth', 'fifth'];

var namespace = 'fluture';
var name = 'Future';
var version = 3;

var $$type = namespace + '/' + name + '@' + version;

function error(message){
  throw new Error(message);
}

function typeError(message){
  throw new TypeError(message);
}

function invalidArgument(it, at, expected, actual){
  typeError(
    it + ' expects its ' + ordinal[at] + ' argument to ' + expected + '\n  Actual: ' + show(actual)
  );
}

function invalidContext(it, actual){
  typeError(
    it + ' was invoked outside the context of a Future. You might want to use'
  + ' a dispatcher instead\n  Called on: ' + (show(actual))
  );
}

function invalidNamespace(m, x){
  return (
    'The Future was not created by ' + namespace + '. '
  + 'Make sure you transform other Futures to ' + namespace + ' Futures. '
  + 'Got ' + (x ? ('a Future from ' + x) : 'an unscoped Future') + '.'
  + '\n  See: https://github.com/fluture-js/Fluture#casting-futures'
  );
}

function invalidVersion(m, x){
  return (
    'The Future was created by ' + (x < version ? 'an older' : 'a newer')
  + ' version of ' + namespace + '. '
  + 'This means that one of the sources which creates Futures is outdated. '
  + 'Update this source, or transform its created Futures to be compatible.'
  + '\n  See: https://github.com/fluture-js/Fluture#casting-futures'
  );
}

function invalidFuture(it, at, m, s){
  var id = type.parse(type(m));
  var info = id.name === name ? '\n' + (
    id.namespace !== namespace ? invalidNamespace(m, id.namespace)
  : id.version !== version ? invalidVersion(m, id.version)
  : 'Nothing seems wrong. Contact the Fluture maintainers.') : '';
  typeError(
    it + ' expects ' + (ordinal[at] ? 'its ' + ordinal[at] + ' argument to be a valid Future' : at)
  + '.' + info + '\n  Actual: ' + (show(m)) + ' :: ' + (id.name) + (s || '')
  );
}

function isFunction(f){
  return typeof f === 'function';
}

function isThenable(m){
  return m instanceof Promise || Boolean(m) && isFunction(m.then);
}

function isBoolean(f){
  return typeof f === 'boolean';
}

function isNumber(f){
  return typeof f === 'number';
}

function isUnsigned(n){
  return (n === Infinity || isNumber(n) && n > 0 && n % 1 === 0);
}

function isObject(o){
  return o !== null && typeof o === 'object';
}

function isIterator(i){
  return isObject(i) && isFunction(i.next);
}

function isArray(x){
  return Array.isArray(x);
}

/*eslint no-cond-assign:0, no-constant-condition:0 */

function interpreter(rej, res){

  //This is the primary queue of actions. All actions in here will be "cold",
  //meaning they haven't had the chance yet to run concurrent computations.
  var cold = new Denque(this._actions.size);

  //This is the secondary queue of actions. All actions in here will be "hot",
  //meaning they have already had a chance to run a concurrent computation.
  var queue = new Denque(this._actions.size);

  //These combined variables define our current state.
  // future  = the future we are currently forking
  // action  = the action to be informed when the future settles
  // cancel  = the cancel function of the current future
  // settled = a boolean indicating whether a new tick should start
  // async   = a boolean indicating whether we are awaiting a result asynchronously
  var future, action, cancel = noop, settled, async = true, it;

  //This function is called with a future to use in the next tick.
  //Here we "flatten" the actions of another Sequence into our own actions,
  //this is the magic that allows for infinitely stack safe recursion because
  //actions like ChainAction will return a new Sequence.
  //If we settled asynchronously, we call drain() directly to run the next tick.
  function settle(m){
    settled = true;
    future = m;

    if(future._spawn){
      var tail = future._actions;

      while(!tail.isEmpty){
        cold.unshift(tail.head);
        tail = tail.tail;
      }

      future = future._spawn;
    }

    if(async) drain();
  }

  //This function serves as a rejection handler for our current future.
  //It will tell the current action that the future rejected, and it will
  //settle the current tick with the action's answer to that.
  function rejected(x){
    settle(action.rejected(x));
  }

  //This function serves as a resolution handler for our current future.
  //It will tell the current action that the future resolved, and it will
  //settle the current tick with the action's answer to that.
  function resolved(x){
    settle(action.resolved(x));
  }

  //This function is passed into actions when they are "warmed up".
  //If the action decides that it has its result, without the need to await
  //anything else, then it can call this function to force "early termination".
  //When early termination occurs, all actions which were queued prior to the
  //terminator will be skipped. If they were already hot, they will also receive
  //a cancel signal so they can cancel their own concurrent computations, as
  //their results are no longer needed.
  function early(m, terminator){
    cancel();
    cold.clear();

    if(async && action !== terminator){
      action.cancel();
      while((it = queue.shift()) && it !== terminator) it.cancel();
    }

    settle(m);
  }

  //This function serves to kickstart concurrent computations.
  //Takes all actions from the cold queue *back-to-front*, and calls run() on
  //each of them, passing them the "early" function. If any of them settles (by
  //calling early()), we abort. After warming up all actions in the cold queue,
  //we warm up the current action as well.
  function warmupActions(){
    while(it = cold.pop()){
      it = it.run(early);
      if(settled) return;
      queue.unshift(it);
    }

    action = action.run(early);
  }

  //This function represents our main execution loop.
  //When we refer to a "tick", we mean the execution of the body inside the
  //primary while-loop of this function.
  //Every tick follows the following algorithm:
  // 1. We try to take an action from the cold queue, if we fail, go to step 2.
  //      1a. We fork the future.
  //      1b. We warmupActions() if the we haven't settled yet.
  // 2. We try to take an action from the hot queue, if we fail, go to step 3.
  //      2a. We fork the Future, if settles, we continue to the next tick.
  // 3. If we couldn't take actions from either queues, we fork the Future into
  //    the user provided continuations. This is the end of the interpretation.
  // 4. If we did take an action from one of queues, but none of the steps
  //    caused a settle(), it means we are asynchronously waiting for something
  //    to settle and start the next tick, so we return from the function.
  function drain(){
    async = false;

    while(true){
      settled = false;
      if(action = cold.shift()){
        cancel = future._fork(rejected, resolved);
        if(!settled) warmupActions();
      }else if(action = queue.shift()){
        cancel = future._fork(rejected, resolved);
      }else break;
      if(settled) continue;
      async = true;
      return;
    }

    cancel = future._fork(rej, res);
  }

  //Start the execution loop.
  settle(this);

  //Return a cancellation function. It will cancel the current Future, the
  //current action, and all queued hot actions.
  return function Sequence$cancel(){
    cancel();
    action && action.cancel();
    while(it = queue.shift()) it.cancel();
  };

}

var empty = ({isEmpty: true, size: 0, head: null, tail: null});

function cons(head, tail){
  return {isEmpty: false, size: tail.size + 1, head: head, tail: tail};
}

function throwRejection(x){
  error('Future#value was called on a rejected Future\n  Actual: Future.reject(' + show(x) + ')');
}

function Future(computation){
  if(!isFunction(computation)) invalidArgument('Future', 0, 'be a Function', computation);
  return new Computation(computation);
}

function isFuture(x){
  return x instanceof Future || type(x) === $$type;
}

Future['@@type'] = $$type;

Future.prototype[FL.ap] = function Future$FL$ap(other){
  return other._ap(this);
};

Future.prototype[FL.map] = function Future$FL$map(mapper){
  return this._map(mapper);
};

Future.prototype[FL.bimap] = function Future$FL$bimap(lmapper, rmapper){
  return this._bimap(lmapper, rmapper);
};

Future.prototype[FL.chain] = function Future$FL$chain(mapper){
  return this._chain(mapper);
};

Future.prototype.ap = function Future$ap(other){
  if(!isFuture(this)) invalidContext('Future#ap', this);
  if(!isFuture(other)) invalidFuture('Future#ap', 0, other);
  return this._ap(other);
};

Future.prototype.map = function Future$map(mapper){
  if(!isFuture(this)) invalidContext('Future#map', this);
  if(!isFunction(mapper)) invalidArgument('Future#map', 0, 'to be a Function', mapper);
  return this._map(mapper);
};

Future.prototype.bimap = function Future$bimap(lmapper, rmapper){
  if(!isFuture(this)) invalidContext('Future#bimap', this);
  if(!isFunction(lmapper)) invalidArgument('Future#bimap', 0, 'to be a Function', lmapper);
  if(!isFunction(rmapper)) invalidArgument('Future#bimap', 1, 'to be a Function', rmapper);
  return this._bimap(lmapper, rmapper);
};

Future.prototype.chain = function Future$chain(mapper){
  if(!isFuture(this)) invalidContext('Future#chain', this);
  if(!isFunction(mapper)) invalidArgument('Future#chain', 0, 'to be a Function', mapper);
  return this._chain(mapper);
};

Future.prototype.mapRej = function Future$mapRej(mapper){
  if(!isFuture(this)) invalidContext('Future#mapRej', this);
  if(!isFunction(mapper)) invalidArgument('Future#mapRej', 0, 'to be a Function', mapper);
  return this._mapRej(mapper);
};

Future.prototype.chainRej = function Future$chainRej(mapper){
  if(!isFuture(this)) invalidContext('Future#chainRej', this);
  if(!isFunction(mapper)) invalidArgument('Future#chainRej', 0, 'to be a Function', mapper);
  return this._chainRej(mapper);
};

Future.prototype.race = function Future$race(other){
  if(!isFuture(this)) invalidContext('Future#race', this);
  if(!isFuture(other)) invalidFuture('Future#race', 0, other);
  return this._race(other);
};

Future.prototype.both = function Future$both(other){
  if(!isFuture(this)) invalidContext('Future#both', this);
  if(!isFuture(other)) invalidFuture('Future#both', 0, other);
  return this._both(other);
};

Future.prototype.and = function Future$and(other){
  if(!isFuture(this)) invalidContext('Future#and', this);
  if(!isFuture(other)) invalidFuture('Future#and', 0, other);
  return this._and(other);
};

Future.prototype.or = function Future$or(other){
  if(!isFuture(this)) invalidContext('Future#or', this);
  if(!isFuture(other)) invalidFuture('Future#or', 0, other);
  return this._or(other);
};

Future.prototype.swap = function Future$swap(){
  if(!isFuture(this)) invalidContext('Future#ap', this);
  return this._swap();
};

Future.prototype.fold = function Future$fold(lmapper, rmapper){
  if(!isFuture(this)) invalidContext('Future#ap', this);
  if(!isFunction(lmapper)) invalidArgument('Future#fold', 0, 'to be a Function', lmapper);
  if(!isFunction(rmapper)) invalidArgument('Future#fold', 1, 'to be a Function', rmapper);
  return this._fold(lmapper, rmapper);
};

Future.prototype.finally = function Future$finally(other){
  if(!isFuture(this)) invalidContext('Future#finally', this);
  if(!isFuture(other)) invalidFuture('Future#finally', 0, other);
  return this._finally(other);
};

Future.prototype.lastly = function Future$lastly(other){
  if(!isFuture(this)) invalidContext('Future#lastly', this);
  if(!isFuture(other)) invalidFuture('Future#lastly', 0, other);
  return this._finally(other);
};

Future.prototype.fork = function Future$fork(rej, res){
  if(!isFuture(this)) invalidContext('Future#fork', this);
  if(!isFunction(rej)) invalidArgument('Future#fork', 0, 'to be a Function', rej);
  if(!isFunction(res)) invalidArgument('Future#fork', 0, 'to be a Function', res);
  return this._fork(rej, res);
};

Future.prototype.value = function Future$value(res){
  if(!isFuture(this)) invalidContext('Future#value', this);
  if(!isFunction(res)) invalidArgument('Future#value', 0, 'to be a Function', res);
  return this._fork(throwRejection, res);
};

Future.prototype.done = function Future$done(callback){
  if(!isFuture(this)) invalidContext('Future#done', this);
  if(!isFunction(callback)) invalidArgument('Future#done', 0, 'to be a Function', callback);
  return this._fork(function Future$done$rej(x){ callback(x); },
                    function Future$done$res(x){ callback(null, x); });
};

Future.prototype.promise = function Future$promise(){
  var _this = this;
  return new Promise(function Future$promise$computation(res, rej){
    _this._fork(rej, res);
  });
};

Future.prototype.isRejected = function Future$isRejected(){
  return false;
};

Future.prototype.isResolved = function Future$isResolved(){
  return false;
};

Future.prototype.isSettled = function Future$isSettled(){
  return this.isRejected() || this.isResolved();
};

Future.prototype.extractLeft = function Future$extractLeft(){
  return [];
};

Future.prototype.extractRight = function Future$extractRight(){
  return [];
};

var Core = Object.create(Future.prototype);

Core._ap = function Core$ap(other){
  return new Sequence(this)._ap(other);
};

Core._map = function Core$map(mapper){
  return new Sequence(this)._map(mapper);
};

Core._bimap = function Core$bimap(lmapper, rmapper){
  return new Sequence(this)._bimap(lmapper, rmapper);
};

Core._chain = function Core$chain(mapper){
  return new Sequence(this)._chain(mapper);
};

Core._mapRej = function Core$mapRej(mapper){
  return new Sequence(this)._mapRej(mapper);
};

Core._chainRej = function Core$chainRej(mapper){
  return new Sequence(this)._chainRej(mapper);
};

Core._race = function Core$race(other){
  return new Sequence(this)._race(other);
};

Core._both = function Core$both(other){
  return new Sequence(this)._both(other);
};

Core._and = function Core$and(other){
  return new Sequence(this)._and(other);
};

Core._or = function Core$or(other){
  return new Sequence(this)._or(other);
};

Core._swap = function Core$swap(){
  return new Sequence(this)._swap();
};

Core._fold = function Core$fold(lmapper, rmapper){
  return new Sequence(this)._fold(lmapper, rmapper);
};

Core._finally = function Core$finally(other){
  return new Sequence(this)._finally(other);
};

function check$fork(f, c){
  if(!(f === undefined || (isFunction(f) && f.length === 0))) typeError(
    'Future expected its computation to return a nullary function or void'
    + '\n  Actual: ' + show(f) + '\n  From calling: ' + showf(c)
  );
}

function Computation(computation){
  this._computation = computation;
}

Computation.prototype = Object.create(Core);

Computation.prototype._fork = function Computation$_fork(rej, res){
  var open = true;
  var cancel = this._computation(function Computation$rej(x){
    if(open){
      open = false;
      rej(x);
    }
  }, function Computation$res(x){
    if(open){
      open = false;
      res(x);
    }
  });
  check$fork(cancel, this._computation);

  return function Computation$cancel(){
    if(open){
      open = false;
      cancel && cancel();
    }
  };
};

Computation.prototype.toString = function Computation$toString(){
  return 'Future(' + showf(this._computation) + ')';
};

function Rejected(value){
  this._value = value;
}

Rejected.prototype = Object.create(Core);

Rejected.prototype._ap = moop;
Rejected.prototype._map = moop;
Rejected.prototype._chain = moop;
Rejected.prototype._race = moop;
Rejected.prototype._both = moop;
Rejected.prototype._and = moop;

Rejected.prototype._or = function Rejected$or(other){
  return other;
};

Rejected.prototype._finally = function Rejected$finally(other){
  return other._and(this);
};

Rejected.prototype._swap = function Rejected$swap(){
  return new Resolved(this._value);
};

Rejected.prototype._fork = function Rejected$_fork(rej){
  rej(this._value);
  return noop;
};

Rejected.prototype.isRejected = function Rejected$isRejected(){
  return true;
};

Rejected.prototype.extractLeft = function Rejected$extractLeft(){
  return [this._value];
};

Rejected.prototype.toString = function Rejected$toString(){
  return 'Future.reject(' + show(this._value) + ')';
};

function reject(x){
  return new Rejected(x);
}

function Resolved(value){
  this._value = value;
}

Resolved.prototype = Object.create(Core);

Resolved.prototype._race = moop;
Resolved.prototype._mapRej = moop;
Resolved.prototype._or = moop;

Resolved.prototype._and = function Resolved$and(other){
  return other;
};

Resolved.prototype._both = function Resolved$both(other){
  var left = this._value;
  return other._map(function Resolved$both$mapper(right){
    return [left, right];
  });
};

Resolved.prototype._swap = function Resolved$swap(){
  return new Rejected(this._value);
};

Resolved.prototype._finally = function Resolved$finally(other){
  var value = this._value;
  return other._map(function Resolved$finally$mapper(){
    return value;
  });
};

Resolved.prototype._fork = function _fork(rej, res){
  res(this._value);
  return noop;
};

Resolved.prototype.isResolved = function Resolved$isResolved(){
  return true;
};

Resolved.prototype.extractRight = function Resolved$extractRight(){
  return [this._value];
};

Resolved.prototype.toString = function Resolved$toString(){
  return 'Future.of(' + show(this._value) + ')';
};

function of(x){
  return new Resolved(x);
}

function Never(){
  this._isNever = true;
}

Never.prototype = Object.create(Future.prototype);

Never.prototype._ap = moop;
Never.prototype._map = moop;
Never.prototype._bimap = moop;
Never.prototype._chain = moop;
Never.prototype._mapRej = moop;
Never.prototype._chainRej = moop;
Never.prototype._both = moop;
Never.prototype._or = moop;
Never.prototype._swap = moop;
Never.prototype._fold = moop;
Never.prototype._finally = moop;

Never.prototype._race = function Never$race(other){
  return other;
};

Never.prototype._fork = function Never$_fork(){
  return noop;
};

Never.prototype.toString = function Never$toString(){
  return 'Future.never';
};

var never = new Never();

function isNever(x){
  return isFuture(x) && x._isNever === true;
}

function Eager(future){
  var _this = this;
  _this.rej = noop;
  _this.res = noop;
  _this.rejected = false;
  _this.resolved = false;
  _this.value = null;
  _this.cancel = future._fork(function Eager$reject(x){
    _this.value = x;
    _this.rejected = true;
    _this.cancel = noop;
    _this.rej(x);
  }, function Eager$resolve(x){
    _this.value = x;
    _this.resolved = true;
    _this.cancel = noop;
    _this.res(x);
  });
}

Eager.prototype = Object.create(Core);

Eager.prototype._fork = function Eager$_fork(rej, res){
  if(this.rejected) rej(this.value);
  else if(this.resolved) res(this.value);
  else{
    this.rej = rej;
    this.res = res;
  }
  return this.cancel;
};

function check$ap(f){
  return isFunction(f) ? f : typeError(
    'Future#ap expects its first argument to be a Future of a Function'
    + '\n  Actual: Future.of(' + show(f) + ')'
  );
}

function check$chain(m, f, x){
  return isFuture(m) ? m : invalidFuture(
    'Future#chain',
    'the function it\'s given to return a Future',
    m,
    '\n  From calling: ' + showf(f) + '\n  With: ' + show(x)
  );
}

function check$chainRej(m, f, x){
  return isFuture(m) ? m : invalidFuture(
    'Future#chainRej',
    'the function it\'s given to return a Future',
    m,
    '\n  From calling: ' + showf(f) + '\n  With: ' + show(x)
  );
}

var Action = {
  rejected: function Action$rejected(x){ this.cancel(); return new Rejected(x) },
  resolved: function Action$resolved(x){ this.cancel(); return new Resolved(x) },
  run: function Action$run(){ return this },
  cancel: function Action$cancel(){}
};

function ApAction(other){ this.other = other; }
ApAction.prototype = Object.create(Action);

ApAction.prototype.resolved = function ApAction$resolved(f){
  check$ap(f);
  return this.other._map(function ApAction$resolved$mapper(x){ return f(x) });
};

ApAction.prototype.toString = function ApAction$toString(){
  return 'ap(' + this.other.toString() + ')';
};

function MapAction(mapper){ this.mapper = mapper; }
MapAction.prototype = Object.create(Action);

MapAction.prototype.resolved = function MapAction$resolved(x){
  return new Resolved(this.mapper(x));
};

MapAction.prototype.toString = function MapAction$toString(){
  return 'map(' + showf(this.mapper) + ')';
};

function BimapAction(lmapper, rmapper){ this.lmapper = lmapper; this.rmapper = rmapper; }
BimapAction.prototype = Object.create(Action);

BimapAction.prototype.rejected = function BimapAction$rejected(x){
  return new Rejected(this.lmapper(x));
};

BimapAction.prototype.resolved = function BimapAction$resolved(x){
  return new Resolved(this.rmapper(x));
};

BimapAction.prototype.toString = function BimapAction$toString(){
  return 'bimap(' + showf(this.lmapper) + ', ' + showf(this.rmapper) + ')';
};

function ChainAction(mapper){ this.mapper = mapper; }
ChainAction.prototype = Object.create(Action);

ChainAction.prototype.resolved = function ChainAction$resolved(x){
  return check$chain(this.mapper(x), this.mapper, x);
};

ChainAction.prototype.toString = function ChainAction$toString(){
  return 'chain(' + showf(this.mapper) + ')';
};

function MapRejAction(mapper){ this.mapper = mapper; }
MapRejAction.prototype = Object.create(Action);

MapRejAction.prototype.rejected = function MapRejAction$rejected(x){
  return new Rejected(this.mapper(x));
};

MapRejAction.prototype.toString = function MapRejAction$toString(){
  return 'mapRej(' + showf(this.mapper) + ')';
};

function ChainRejAction(mapper){ this.mapper = mapper; }
ChainRejAction.prototype = Object.create(Action);

ChainRejAction.prototype.rejected = function ChainRejAction$rejected(x){
  return check$chainRej(this.mapper(x), this.mapper, x);
};

ChainRejAction.prototype.toString = function ChainRejAction$toString(){
  return 'chainRej(' + showf(this.mapper) + ')';
};

function SwapAction(){}
SwapAction.prototype = Object.create(Action);

SwapAction.prototype.rejected = function SwapAction$rejected(x){
  return new Resolved(x);
};

SwapAction.prototype.resolved = function SwapAction$resolved(x){
  return new Rejected(x);
};

SwapAction.prototype.toString = function SwapAction$toString(){
  return 'swap()';
};

function FoldAction(lmapper, rmapper){ this.lmapper = lmapper; this.rmapper = rmapper; }
FoldAction.prototype = Object.create(Action);

FoldAction.prototype.rejected = function FoldAction$rejected(x){
  return new Resolved(this.lmapper(x));
};

FoldAction.prototype.resolved = function FoldAction$resolved(x){
  return new Resolved(this.rmapper(x));
};

FoldAction.prototype.toString = function FoldAction$toString(){
  return 'fold(' + showf(this.lmapper) + ', ' + showf(this.rmapper) + ')';
};

function FinallyAction(other){ this.other = other; }
FinallyAction.prototype = Object.create(Action);

FinallyAction.prototype.rejected = function FinallyAction$rejected(x){
  return this.other._and(new Rejected(x));
};

FinallyAction.prototype.resolved = function FinallyAction$resolved(x){
  return this.other._map(function FoldAction$resolved$mapper(){ return x });
};

FinallyAction.prototype.cancel = function FinallyAction$cancel(){
  this.other._fork(noop, noop)();
};

FinallyAction.prototype.toString = function FinallyAction$toString(){
  return 'finally(' + this.other.toString() + ')';
};

function AndAction(other){ this.other = other; }
AndAction.prototype = Object.create(Action);

AndAction.prototype.resolved = function AndAction$resolved(){
  return this.other;
};

AndAction.prototype.toString = function AndAction$toString(){
  return 'and(' + this.other.toString() + ')';
};

function OrAction(other){ this.other = other; }
OrAction.prototype = Object.create(Action);

OrAction.prototype.rejected = function OrAction$rejected(){
  return this.other;
};

OrAction.prototype.toString = function OrAction$toString(){
  return 'or(' + this.other.toString() + ')';
};

function RaceAction(other){ this.other = other; }
RaceAction.prototype = Object.create(Action);

RaceAction.prototype.run = function RaceAction$run(early){
  return new RaceActionState(early, new Eager(this.other));
};

RaceAction.prototype.toString = function RaceAction$toString(){
  return 'race(' + this.other.toString() + ')';
};

function BothAction(other){ this.other = other; }
BothAction.prototype = Object.create(Action);

BothAction.prototype.resolved = function BothAction$resolved(x){
  return this.other._map(function BothAction$resolved$mapper(y){ return [x, y] });
};

BothAction.prototype.run = function BothAction$run(early){
  return new BothActionState(early, new Eager(this.other));
};

BothAction.prototype.toString = function BothAction$toString(){
  return 'both(' + this.other.toString() + ')';
};

function RaceActionState(early, other){
  var _this = this;
  _this.other = other;
  _this.cancel = other._fork(
    function RaceActionState$rej(x){ early(new Rejected(x), _this); },
    function RaceActionState$res(x){ early(new Resolved(x), _this); }
  );
}

RaceActionState.prototype = Object.create(RaceAction.prototype);

function BothActionState(early, other){
  var _this = this;
  _this.other = other;
  _this.cancel = other._fork(
    function BothActionState$rej(x){ early(new Rejected(x), _this); },
    noop
  );
}

BothActionState.prototype = Object.create(BothAction.prototype);

function Sequence(spawn, actions){
  this._spawn = spawn;
  this._actions = actions || empty;
}

Sequence.prototype = Object.create(Future.prototype);

Sequence.prototype._transform = function Sequence$_transform(action){
  return new Sequence(this._spawn, cons(action, this._actions));
};

Sequence.prototype._ap = function Sequence$ap(other){
  return this._transform(new ApAction(other));
};

Sequence.prototype._map = function Sequence$map(mapper){
  return this._transform(new MapAction(mapper));
};

Sequence.prototype._bimap = function Sequence$bimap(lmapper, rmapper){
  return this._transform(new BimapAction(lmapper, rmapper));
};

Sequence.prototype._chain = function Sequence$chain(mapper){
  return this._transform(new ChainAction(mapper));
};

Sequence.prototype._mapRej = function Sequence$mapRej(mapper){
  return this._transform(new MapRejAction(mapper));
};

Sequence.prototype._chainRej = function Sequence$chainRej(mapper){
  return this._transform(new ChainRejAction(mapper));
};

Sequence.prototype._race = function Sequence$race(other){
  return isNever(other) ? this : this._transform(new RaceAction(other));
};

Sequence.prototype._both = function Sequence$both(other){
  return this._transform(new BothAction(other));
};

Sequence.prototype._and = function Sequence$and(other){
  return this._transform(new AndAction(other));
};

Sequence.prototype._or = function Sequence$or(other){
  return this._transform(new OrAction(other));
};

Sequence.prototype._swap = function Sequence$swap(){
  return this._transform(new SwapAction);
};

Sequence.prototype._fold = function Sequence$fold(lmapper, rmapper){
  return this._transform(new FoldAction(lmapper, rmapper));
};

Sequence.prototype._finally = function Sequence$finally(other){
  return this._transform(new FinallyAction(other));
};

Sequence.prototype._fork = interpreter;

Sequence.prototype.toString = function Sequence$toString(){
  var str = '', tail = this._actions;

  while(!tail.isEmpty){
    str = '.' + tail.head.toString() + str;
    tail = tail.tail;
  }

  return this._spawn.toString() + str;
};

function Next(x){
  return {done: false, value: x};
}

function Done(x){
  return {done: true, value: x};
}

function isIteration(x){
  return isObject(x) && isBoolean(x.done);
}

var Undetermined = 0;
var Synchronous = 1;
var Asynchronous = 2;

function ChainRec(step, init){
  this._step = step;
  this._init = init;
}

ChainRec.prototype = Object.create(Core);

ChainRec.prototype._fork = function ChainRec$_fork(rej, res){

  var _step = this._step;
  var _init = this._init;
  var timing = Undetermined, cancel = noop, state = Next(_init);

  function resolved(it){
    state = it;
    timing = timing === Undetermined ? Synchronous : drain();
  }

  function drain(){
    while(!state.done){
      timing = Undetermined;
      var m = _step(Next, Done, state.value);
      cancel = m._fork(rej, resolved);

      if(timing !== Synchronous){
        timing = Asynchronous;
        return;
      }
    }

    res(state.value);
  }

  drain();

  return function Future$chainRec$cancel(){ cancel(); };

};

ChainRec.prototype.toString = function ChainRec$toString(){
  return 'Future.chainRec(' + showf(this._step) + ', ' + show(this._init) + ')';
};

function chainRec(step, init){
  return new ChainRec(step, init);
}

function ap$mval(mval, mfunc){
  if(!Z.Apply.test(mfunc)) invalidArgument('Future.ap', 1, 'be an Apply', mfunc);
  return Z.ap(mval, mfunc);
}

function ap(mval, mfunc){
  if(!Z.Apply.test(mval)) invalidArgument('Future.ap', 0, 'be an Apply', mval);
  if(arguments.length === 1) return partial1(ap$mval, mval);
  return ap$mval(mval, mfunc);
}

function alt$left(left, right){
  if(!Z.Alt.test(right)) invalidArgument('alt', 1, 'be an Alt', right);
  return Z.alt(left, right);
}

function alt(left, right){
  if(!Z.Alt.test(left)) invalidArgument('alt', 0, 'be an Alt', left);
  if(arguments.length === 1) return partial1(alt$left, left);
  return alt$left(left, right);
}

function map$mapper(mapper, m){
  if(!Z.Functor.test(m)) invalidArgument('Future.map', 1, 'be a Functor', m);
  return Z.map(mapper, m);
}

function map(mapper, m){
  if(!isFunction(mapper)) invalidArgument('Future.map', 0, 'be a Function', mapper);
  if(arguments.length === 1) return partial1(map$mapper, mapper);
  return map$mapper(mapper, m);
}

function bimap$lmapper$rmapper(lmapper, rmapper, m){
  if(!Z.Bifunctor.test(m)) invalidArgument('Future.bimap', 2, 'be a Bifunctor', m);
  return Z.bimap(lmapper, rmapper, m);
}

function bimap$lmapper(lmapper, rmapper, m){
  if(!isFunction(rmapper)) invalidArgument('Future.bimap', 1, 'be a Function', rmapper);
  if(arguments.length === 2) return partial2(bimap$lmapper$rmapper, lmapper, rmapper);
  return bimap$lmapper$rmapper(lmapper, rmapper, m);
}

function bimap(lmapper, rmapper, m){
  if(!isFunction(lmapper)) invalidArgument('Future.bimap', 0, 'be a Function', lmapper);
  if(arguments.length === 1) return partial1(bimap$lmapper, lmapper);
  if(arguments.length === 2) return bimap$lmapper(lmapper, rmapper);
  return bimap$lmapper(lmapper, rmapper, m);
}

function chain$chainer(chainer, m){
  if(!Z.Chain.test(m)) invalidArgument('Future.chain', 1, 'be a Chain', m);
  return Z.chain(chainer, m);
}

function chain(chainer, m){
  if(!isFunction(chainer)) invalidArgument('Future.chain', 0, 'be a Function', chainer);
  if(arguments.length === 1) return partial1(chain$chainer, chainer);
  return chain$chainer(chainer, m);
}

function mapRej$mapper(mapper, m){
  if(!isFuture(m)) invalidFuture('Future.mapRej', 1, m);
  return m.mapRej(mapper);
}

function mapRej(mapper, m){
  if(!isFunction(mapper)) invalidArgument('Future.mapRej', 0, 'be a Function', mapper);
  if(arguments.length === 1) return partial1(mapRej$mapper, mapper);
  return mapRej$mapper(mapper, m);
}

function chainRej$chainer(chainer, m){
  if(!isFuture(m)) invalidFuture('Future.chainRej', 1, m);
  return m.chainRej(chainer);
}

function chainRej(chainer, m){
  if(!isFunction(chainer)) invalidArgument('Future.chainRej', 0, 'be a Function', chainer);
  if(arguments.length === 1) return partial1(chainRej$chainer, chainer);
  return chainRej$chainer(chainer, m);
}

function lastly$right(right, left){
  if(!isFuture(left)) invalidFuture('Future.finally', 1, left);
  return left.finally(right);
}

function lastly(right, left){
  if(!isFuture(right)) invalidFuture('Future.finally', 0, right);
  if(arguments.length === 1) return partial1(lastly$right, right);
  return lastly$right(right, left);
}

function and$left(left, right){
  if(!isFuture(right)) invalidFuture('Future.and', 1, right);
  return left.and(right);
}

function and(left, right){
  if(!isFuture(left)) invalidFuture('Future.and', 0, left);
  if(arguments.length === 1) return partial1(and$left, left);
  return and$left(left, right);
}

function both$left(left, right){
  if(!isFuture(right)) invalidFuture('Future.both', 1, right);
  return left.both(right);
}

function both(left, right){
  if(!isFuture(left)) invalidFuture('Future.both', 0, left);
  if(arguments.length === 1) return partial1(both$left, left);
  return both$left(left, right);
}

function or$left(left, right){
  if(!isFuture(right)) invalidFuture('Future.or', 1, right);
  return left.or(right);
}

function or(left, right){
  if(!isFuture(left)) invalidFuture('Future.or', 0, left);
  if(arguments.length === 1) return partial1(or$left, left);
  return or$left(left, right);
}

function race$right(right, left){
  if(!isFuture(left)) invalidFuture('Future.race', 1, left);
  return left.race(right);
}

function race(right, left){
  if(!isFuture(right)) invalidFuture('Future.race', 0, right);
  if(arguments.length === 1) return partial1(race$right, right);
  return race$right(right, left);
}

function swap(m){
  if(!isFuture(m)) invalidFuture('Future.swap', 0, m);
  return m.swap();
}

function fold$f$g(f, g, m){
  if(!isFuture(m)) invalidFuture('Future.fold', 2, m);
  return m.fold(f, g);
}

function fold$f(f, g, m){
  if(!isFunction(g)) invalidArgument('Future.fold', 1, 'be a function', g);
  if(arguments.length === 2) return partial2(fold$f$g, f, g);
  return fold$f$g(f, g, m);
}

function fold(f, g, m){
  if(!isFunction(f)) invalidArgument('Future.fold', 0, 'be a function', f);
  if(arguments.length === 1) return partial1(fold$f, f);
  if(arguments.length === 2) return fold$f(f, g);
  return fold$f(f, g, m);
}

function done$callback(callback, m){
  if(!isFuture(m)) invalidFuture('Future.done', 1, m);
  return m.done(callback);
}

function done(callback, m){
  if(!isFunction(callback)) invalidArgument('Future.done', 0, 'be a Function', callback);
  if(arguments.length === 1) return partial1(done$callback, callback);
  return done$callback(callback, m);
}

function fork$f$g(f, g, m){
  if(!isFuture(m)) invalidFuture('Future.fork', 2, m);
  return m._fork(f, g);
}

function fork$f(f, g, m){
  if(!isFunction(g)) invalidArgument('Future.fork', 1, 'be a function', g);
  if(arguments.length === 2) return partial2(fork$f$g, f, g);
  return fork$f$g(f, g, m);
}

function fork(f, g, m){
  if(!isFunction(f)) invalidArgument('Future.fork', 0, 'be a function', f);
  if(arguments.length === 1) return partial1(fork$f, f);
  if(arguments.length === 2) return fork$f(f, g);
  return fork$f(f, g, m);
}

function promise(m){
  if(!isFuture(m)) invalidFuture('Future.promise', 0, m);
  return m.promise();
}

function value$cont(cont, m){
  if(!isFuture(m)) invalidFuture('Future.value', 1, m);
  return m.value(cont);
}

function value(cont, m){
  if(!isFunction(cont)) invalidArgument('Future.value', 0, 'be a Function', cont);
  if(arguments.length === 1) return partial1(value$cont, cont);
  return value$cont(cont, m);
}

function extractLeft(m){
  if(!isFuture(m)) invalidFuture('Future.extractLeft', 0, m);
  return m.extractLeft();
}

function extractRight(m){
  if(!isFuture(m)) invalidFuture('Future.extractRight', 0, m);
  return m.extractRight();
}

function check$ap$f(f){
  if(!isFunction(f)) typeError(
    'Future#ap expects its first argument to be a Future of a Function'
    + '\n  Actual: Future.of(' + show(f) + ')'
  );
}

function ParallelAp(mval, mfunc){
  this._mval = mval;
  this._mfunc = mfunc;
}

ParallelAp.prototype = Object.create(Core);

ParallelAp.prototype._fork = function ParallelAp$fork(rej, res){
  var func, val, okval = false, okfunc = false, rejected = false, c1, c2;

  function ParallelAp$rej(x){
    if(!rejected){
      rejected = true;
      rej(x);
    }
  }

  c1 = this._mval._fork(ParallelAp$rej, function ParallelAp$fork$resVal(x){
    c1 = noop;
    if(!okval) return void (okfunc = true, val = x);
    res(func(x));
  });
  c2 = this._mfunc._fork(ParallelAp$rej, function ParallelAp$fork$resFunc(f){
    c2 = noop;
    check$ap$f(f);
    if(!okfunc) return void (okval = true, func = f);
    res(f(val));
  });

  return function ParallelAp$fork$cancel(){
    c1();
    c2();
  };
};

ParallelAp.prototype.toString = function ParallelAp$toString(){
  return 'new ParallelAp(' + this._mval.toString() + ', ' + this._mfunc.toString() + ')';
};

function parallelAp(mval, mfunc){
  return new ParallelAp(mval, mfunc);
}

function After$race(other){
  return other.isSettled()
       ? other
       : isNever(other)
       ? this
       : typeof other._time === 'number'
       ? other._time < this._time ? other : this
       : Core._race.call(this, other);
}

function After(time, value){
  this._time = time;
  this._value = value;
}

After.prototype = Object.create(Core);

After.prototype._race = After$race;

After.prototype._swap = function After$swap(){
  return new RejectAfter(this._time, this._value);
};

After.prototype._fork = function After$_fork(rej, res){
  var id = setTimeout(res, this._time, this._value);
  return function After$cancel(){ clearTimeout(id); };
};

After.prototype.extractRight = function After$extractRight(){
  return [this._value];
};

After.prototype.toString = function After$toString(){
  return 'Future.after(' + show(this._time) + ', ' + show(this._value) + ')';
};

function RejectAfter(time, value){
  this._time = time;
  this._value = value;
}

RejectAfter.prototype = Object.create(Core);

RejectAfter.prototype._race = After$race;

RejectAfter.prototype._swap = function RejectAfter$swap(){
  return new After(this._time, this._value);
};

RejectAfter.prototype._fork = function RejectAfter$_fork(rej){
  var id = setTimeout(rej, this._time, this._value);
  return function RejectAfter$cancel(){ clearTimeout(id); };
};

RejectAfter.prototype.extractLeft = function RejectAfter$extractLeft(){
  return [this._value];
};

RejectAfter.prototype.toString = function RejectAfter$toString(){
  return 'Future.rejectAfter(' + show(this._time) + ', ' + show(this._value) + ')';
};

function after$time(time, value){
  return time === Infinity ? never : new After(time, value);
}

function after(time, value){
  if(!isUnsigned(time)) invalidArgument('Future.after', 0, 'be a positive integer', time);
  if(arguments.length === 1) return partial1(after$time, time);
  return after$time(time, value);
}

function rejectAfter$time(time, reason){
  return time === Infinity ? never : new RejectAfter(time, reason);
}

function rejectAfter(time, reason){
  if(!isUnsigned(time)) invalidArgument('Future.rejectAfter', 0, 'be a positive integer', time);
  if(arguments.length === 1) return partial1(rejectAfter$time, time);
  return rejectAfter$time(time, reason);
}

function Attempt(fn){
  this._fn = fn;
}

Attempt.prototype = Object.create(Core);

Attempt.prototype._fork = function Attempt$fork(rej, res){
  var r;
  try{ r = this._fn(); }catch(e){ rej(e); return noop }
  res(r);
  return noop;
};

Attempt.prototype.toString = function Attempt$toString(){
  return 'Future.try(' + showf(this._fn) + ')';
};

function attempt(f){
  if(!isFunction(f)) invalidArgument('Future.try', 0, 'be a function', f);
  return new Attempt(f);
}

var Cold = Cached.Cold = 0;
var Pending = Cached.Pending = 1;
var Rejected$1 = Cached.Rejected = 2;
var Resolved$1 = Cached.Resolved = 3;

function Queued(rej, res){
  this[Rejected$1] = rej;
  this[Resolved$1] = res;
}

function Cached(pure){
  this._pure = pure;
  this.reset();
}

Cached.prototype = Object.create(Core);

Cached.prototype.isRejected = function Cached$isRejected(){
  return this._state === Rejected$1;
};

Cached.prototype.isResolved = function Cached$isResolved(){
  return this._state === Resolved$1;
};

Cached.prototype.extractLeft = function Cached$extractLeft(){
  return this.isRejected() ? [this._value] : [];
};

Cached.prototype.extractRight = function Cached$extractRight(){
  return this.isResolved() ? [this._value] : [];
};

Cached.prototype._addToQueue = function Cached$addToQueue(rej, res){
  var _this = this;
  if(_this._state > Pending) return noop;
  var i = _this._queue.push(new Queued(rej, res)) - 1;
  _this._queued = _this._queued + 1;

  return function Cached$removeFromQueue(){
    if(_this._state > Pending) return;
    _this._queue[i] = undefined;
    _this._queued = _this._queued - 1;
    if(_this._queued === 0) _this.reset();
  };
};

Cached.prototype._drainQueue = function Cached$drainQueue(){
  if(this._state <= Pending) return;
  if(this._queued === 0) return;
  var queue = this._queue;
  var length = queue.length;
  var state = this._state;
  var value = this._value;

  for(var i = 0; i < length; i++){
    queue[i] && queue[i][state](value);
    queue[i] = undefined;
  }

  this._queue = undefined;
  this._queued = 0;
};

Cached.prototype.reject = function Cached$reject(reason){
  if(this._state > Pending) return;
  this._value = reason;
  this._state = Rejected$1;
  this._drainQueue();
};

Cached.prototype.resolve = function Cached$resolve(value){
  if(this._state > Pending) return;
  this._value = value;
  this._state = Resolved$1;
  this._drainQueue();
};

Cached.prototype.run = function Cached$run(){
  var _this = this;
  if(_this._state > Cold) return;
  _this._state = Pending;
  _this._cancel = _this._pure._fork(
    function Cached$fork$rej(x){ _this.reject(x); },
    function Cached$fork$res(x){ _this.resolve(x); }
  );
};

Cached.prototype.reset = function Cached$reset(){
  if(this._state === Cold) return;
  if(this._state > Pending) this._cancel();
  this._cancel = noop;
  this._queue = [];
  this._queued = 0;
  this._value = undefined;
  this._state = Cold;
};

Cached.prototype._fork = function Cached$_fork(rej, res){
  var cancel = noop;

  switch(this._state){
    case Pending: cancel = this._addToQueue(rej, res); break;
    case Rejected$1: rej(this._value); break;
    case Resolved$1: res(this._value); break;
    default: cancel = this._addToQueue(rej, res); this.run();
  }

  return cancel;
};

Cached.prototype.toString = function Cached$toString(){
  return 'Future.cache(' + this._pure.toString() + ')';
};

function cache(m){
  if(!isFuture(m)) invalidFuture('Future.cache', 0, m);
  return new Cached(m);
}

function Encase(fn, a){
  this._fn = fn;
  this._a = a;
}

Encase.prototype = Object.create(Core);

Encase.prototype._fork = function Encase$fork(rej, res){
  var r;
  try{ r = this._fn(this._a); }catch(e){ rej(e); return noop }
  res(r);
  return noop;
};

Encase.prototype.toString = function Encase$toString(){
  return 'Future.encase(' + showf(this._fn) + ', ' + show(this._a) + ')';
};

function encase(f, x){
  if(!isFunction(f)) invalidArgument('Future.encase', 0, 'be a function', f);
  if(arguments.length === 1) return partial1(encase, f);
  return new Encase(f, x);
}

function Encase2(fn, a, b){
  this._fn = fn;
  this._a = a;
  this._b = b;
}

Encase2.prototype = Object.create(Core);

Encase2.prototype._fork = function Encase2$fork(rej, res){
  var r;
  try{ r = this._fn(this._a, this._b); }catch(e){ rej(e); return noop }
  res(r);
  return noop;
};

Encase2.prototype.toString = function Encase2$toString(){
  return 'Future.encase2(' + showf(this._fn) + ', ' + show(this._a) + ', ' + show(this._b) + ')';
};

function encase2(f, x, y){
  if(!isFunction(f)) invalidArgument('Future.encase2', 0, 'be a function', f);

  switch(arguments.length){
    case 1: return partial1(encase2, f);
    case 2: return partial2(encase2, f, x);
    default: return new Encase2(f, x, y);
  }
}

function Encase3(fn, a, b, c){
  this._fn = fn;
  this._a = a;
  this._b = b;
  this._c = c;
}

Encase3.prototype = Object.create(Core);

Encase3.prototype._fork = function Encase3$fork(rej, res){
  var r;
  try{ r = this._fn(this._a, this._b, this._c); }catch(e){ rej(e); return noop }
  res(r);
  return noop;
};

Encase3.prototype.toString = function Encase3$toString(){
  return 'Future.encase3('
       + showf(this._fn)
       + ', '
       + show(this._a)
       + ', '
       + show(this._b)
       + ', '
       + show(this._c)
       + ')';
};

function encase3(f, x, y, z){
  if(!isFunction(f)) invalidArgument('Future.encase3', 0, 'be a function', f);

  switch(arguments.length){
    case 1: return partial1(encase3, f);
    case 2: return partial2(encase3, f, x);
    case 3: return partial3(encase3, f, x, y);
    default: return new Encase3(f, x, y, z);
  }
}

function EncaseN(fn, a){
  this._fn = fn;
  this._a = a;
}

EncaseN.prototype = Object.create(Core);

EncaseN.prototype._fork = function EncaseN$fork(rej, res){
  var open = true;
  this._fn(this._a, function EncaseN$done(err, val){
    if(open){
      open = false;
      err ? rej(err) : res(val);
    }
  });
  return function EncaseN$cancel(){ open = false; };
};

EncaseN.prototype.toString = function EncaseN$toString(){
  return 'Future.encaseN(' + showf(this._fn) + ', ' + show(this._a) + ')';
};

function encaseN(f, x){
  if(!isFunction(f)) invalidArgument('Future.encaseN', 0, 'be a function', f);
  if(arguments.length === 1) return partial1(encaseN, f);
  return new EncaseN(f, x);
}

function EncaseN2(fn, a, b){
  this._fn = fn;
  this._a = a;
  this._b = b;
}

EncaseN2.prototype = Object.create(Core);

EncaseN2.prototype._fork = function EncaseN2$fork(rej, res){
  var open = true;
  this._fn(this._a, this._b, function EncaseN2$done(err, val){
    if(open){
      open = false;
      err ? rej(err) : res(val);
    }
  });
  return function EncaseN2$cancel(){ open = false; };
};

EncaseN2.prototype.toString = function EncaseN2$toString(){
  return 'Future.encaseN2(' + showf(this._fn) + ', ' + show(this._a) + ', ' + show(this._b) + ')';
};

function encaseN2(f, x, y){
  if(!isFunction(f)) invalidArgument('Future.encaseN2', 0, 'be a function', f);

  switch(arguments.length){
    case 1: return partial1(encaseN2, f);
    case 2: return partial2(encaseN2, f, x);
    default: return new EncaseN2(f, x, y);
  }
}

function EncaseN$1(fn, a, b, c){
  this._fn = fn;
  this._a = a;
  this._b = b;
  this._c = c;
}

EncaseN$1.prototype = Object.create(Core);

EncaseN$1.prototype._fork = function EncaseN$3$fork(rej, res){
  var open = true;
  this._fn(this._a, this._b, this._c, function EncaseN$3$done(err, val){
    if(open){
      open = false;
      err ? rej(err) : res(val);
    }
  });
  return function EncaseN$3$cancel(){ open = false; };
};

EncaseN$1.prototype.toString = function EncaseN$3$toString(){
  return 'Future.encaseN3('
       + showf(this._fn)
       + ', '
       + show(this._a)
       + ', '
       + show(this._b)
       + ', '
       + show(this._c)
       + ')';
};

function encaseN3(f, x, y, z){
  if(!isFunction(f)) invalidArgument('Future.encaseN3', 0, 'be a function', f);

  switch(arguments.length){
    case 1: return partial1(encaseN3, f);
    case 2: return partial2(encaseN3, f, x);
    case 3: return partial3(encaseN3, f, x, y);
    default: return new EncaseN$1(f, x, y, z);
  }
}

function check$promise(p, f, a){
  return isThenable(p) ? p : typeError(
    'Future.encaseP expects the function it\'s given to return a Promise/Thenable'
    + '\n  Actual: ' + (show(p)) + '\n  From calling: ' + (showf(f))
    + '\n  With: ' + (show(a))
  );
}

function EncaseP(fn, a){
  this._fn = fn;
  this._a = a;
}

EncaseP.prototype = Object.create(Core);

EncaseP.prototype._fork = function EncaseP$fork(rej, res){
  var _fn = this._fn;
  var _a = this._a;
  var open = true;
  check$promise(_fn(_a), _fn, _a).then(immediately(function EncaseP$res(x){
    if(open){
      open = false;
      res(x);
    }
  }), immediately(function EncaseP$rej(x){
    if(open){
      open = false;
      rej(x);
    }
  }));
  return function EncaseP$cancel(){ open = false; };
};

EncaseP.prototype.toString = function EncaseP$toString(){
  return 'Future.encaseP(' + showf(this._fn) + ', ' + show(this._a) + ')';
};

function encaseP(f, x){
  if(!isFunction(f)) invalidArgument('Future.encaseP', 0, 'be a function', f);
  if(arguments.length === 1) return partial1(encaseP, f);
  return new EncaseP(f, x);
}

function check$promise$1(p, f, a, b){
  return isThenable(p) ? p : typeError(
    'Future.encaseP2 expects the function it\'s given to return a Promise/Thenable'
    + '\n  Actual: ' + (show(p)) + '\n  From calling: ' + (showf(f))
    + '\n  With 1: ' + (show(a))
    + '\n  With 2: ' + (show(b))
  );
}

function EncaseP2(fn, a, b){
  this._fn = fn;
  this._a = a;
  this._b = b;
}

EncaseP2.prototype = Object.create(Core);

EncaseP2.prototype._fork = function EncaseP2$fork(rej, res){
  var _fn = this._fn;
  var _a = this._a;
  var _b = this._b;
  var open = true;
  check$promise$1(_fn(_a, _b), _fn, _a, _b).then(immediately(function EncaseP2$res(x){
    if(open){
      open = false;
      res(x);
    }
  }), immediately(function EncaseP2$rej(x){
    if(open){
      open = false;
      rej(x);
    }
  }));
  return function EncaseP2$cancel(){ open = false; };
};

EncaseP2.prototype.toString = function EncaseP2$toString(){
  return 'Future.encaseP2(' + showf(this._fn) + ', ' + show(this._a) + ', ' + show(this._b) + ')';
};

function encaseP2(f, x, y){
  if(!isFunction(f)) invalidArgument('Future.encaseP2', 0, 'be a function', f);

  switch(arguments.length){
    case 1: return partial1(encaseP2, f);
    case 2: return partial2(encaseP2, f, x);
    default: return new EncaseP2(f, x, y);
  }
}

function check$promise$2(p, f, a, b, c){
  return isThenable(p) ? p : typeError(
    'Future.encaseP3 expects the function it\'s given to return a Promise/Thenable'
    + '\n  Actual: ' + (show(p)) + '\n  From calling: ' + (showf(f))
    + '\n  With 1: ' + (show(a))
    + '\n  With 2: ' + (show(b))
    + '\n  With 3: ' + (show(c))
  );
}

function EncaseP3(fn, a, b, c){
  this._fn = fn;
  this._a = a;
  this._b = b;
  this._c = c;
}

EncaseP3.prototype = Object.create(Core);

EncaseP3.prototype._fork = function EncaseP3$fork(rej, res){
  var _fn = this._fn;
  var _a = this._a;
  var _b = this._b;
  var _c = this._c;
  var open = true;
  check$promise$2(_fn(_a, _b, _c), _fn, _a, _b, _c).then(immediately(function EncaseP3$res(x){
    if(open){
      open = false;
      res(x);
    }
  }), immediately(function EncaseP3$rej(x){
    if(open){
      open = false;
      rej(x);
    }
  }));
  return function EncaseP3$cancel(){ open = false; };
};

EncaseP3.prototype.toString = function EncaseP3$toString(){
  return 'Future.encaseP3('
       + showf(this._fn)
       + ', '
       + show(this._a)
       + ', '
       + show(this._b)
       + ', '
       + show(this._c)
       + ')';
};

function encaseP3(f, x, y, z){
  if(!isFunction(f)) invalidArgument('Future.encaseP3', 0, 'be a function', f);

  switch(arguments.length){
    case 1: return partial1(encaseP3, f);
    case 2: return partial2(encaseP3, f, x);
    case 3: return partial3(encaseP3, f, x, y);
    default: return new EncaseP3(f, x, y, z);
  }
}

/*eslint consistent-return: 0*/

function check$iterator(g){
  return isIterator(g) ? g : invalidArgument(
    'Future.do', 0, 'return an iterator, maybe you forgot the "*"', g
  );
}

function check$iteration(o){
  if(!isIteration(o)) typeError(
    'Future.do was given an invalid generator:'
    + ' Its iterator did not return a valid iteration from iterator.next()'
    + '\n  Actual: ' + show(o)
  );
  if(o.done || isFuture(o.value)) return o;
  return invalidFuture(
    'Future.do',
    'the iterator to produce only valid Futures',
    o.value,
    '\n  Tip: If you\'re using a generator, make sure you always yield a Future'
  );
}

function Go(generator){
  this._generator = generator;
}

Go.prototype = Object.create(Core);

Go.prototype._fork = function Go$_fork(rej, res){

  var iterator = check$iterator(this._generator());

  var timing = Undetermined, cancel = noop, state, value;

  function resolved(x){
    value = x;
    if(timing === Asynchronous) return drain();
    timing = Synchronous;
    state = check$iteration(iterator.next(value));
  }

  function drain(){
    state = check$iteration(iterator.next(value));

    while(!state.done){
      timing = Undetermined;
      cancel = state.value._fork(rej, resolved);

      if(timing !== Synchronous){
        timing = Asynchronous;
        return;
      }
    }

    res(state.value);
  }

  drain();

  return function Go$cancel(){ cancel(); };

};

Go.prototype.toString = function Go$toString(){
  return 'Future.do(' + showf(this._generator) + ')';
};

function go(generator){
  if(!isFunction(generator)) invalidArgument('Future.do', 0, 'be a Function', generator);
  return new Go(generator);
}

function check$dispose(m, f, x){
  if(!isFuture(m)) invalidFuture(
    'Future.hook',
    'the first function it\'s given to return a Future',
    m,
    '\n  From calling: ' + showf(f) + '\n  With: ' + show(x)
  );
}

function check$consume(m, f, x){
  if(!isFuture(m)) invalidFuture(
    'Future.hook',
    'the second function it\'s given to return a Future',
    m,
    '\n  From calling: ' + showf(f) + '\n  With: ' + show(x)
  );
}

function Hook(acquire, dispose, consume){
  this._acquire = acquire;
  this._dispose = dispose;
  this._consume = consume;
}

Hook.prototype = Object.create(Core);

Hook.prototype._fork = function Hook$fork(rej, res){

  var _acquire = this._acquire, _dispose = this._dispose, _consume = this._consume;
  var cancel, cancelAcquire = noop, cancelConsume = noop, resource, value, cont = noop;

  function Hook$done(){
    cont(value);
  }

  function Hook$dispose(){
    var disposal = _dispose(resource);
    check$dispose(disposal, _dispose, resource);
    cancel = disposal._fork(rej, Hook$done);
    return cancel;
  }

  function Hook$cancelConsuption(){
    cancelConsume();
    Hook$dispose()();
  }

  function Hook$consumptionRejected(x){
    cont = rej;
    value = x;
    Hook$dispose();
  }

  function Hook$consumptionResolved(x){
    cont = res;
    value = x;
    Hook$dispose();
  }

  function Hook$acquireResolved(x){
    resource = x;
    var consumption = _consume(resource);
    check$consume(consumption, _consume, resource);
    cancel = Hook$cancelConsuption;
    cancelConsume = consumption._fork(Hook$consumptionRejected, Hook$consumptionResolved);
  }

  cancelAcquire = _acquire._fork(rej, Hook$acquireResolved);

  cancel = cancel || cancelAcquire;

  return function Hook$fork$cancel(){ cancel(); };

};

Hook.prototype.toString = function Hook$toString(){
  return 'Future.hook('
       + this._acquire.toString()
       + ', '
       + showf(this._dispose)
       + ', '
       + showf(this._consume)
       + ')';
};

function hook$acquire$cleanup(acquire, cleanup, consume){
  if(!isFunction(consume)) invalidArgument('Future.hook', 2, 'be a Future', consume);
  return new Hook(acquire, cleanup, consume);
}

function hook$acquire(acquire, cleanup, consume){
  if(!isFunction(cleanup)) invalidArgument('Future.hook', 1, 'be a function', cleanup);
  if(arguments.length === 2) return partial2(hook$acquire$cleanup, acquire, cleanup);
  return hook$acquire$cleanup(acquire, cleanup, consume);
}

function hook(acquire, cleanup, consume){
  if(!isFuture(acquire)) invalidFuture('Future.hook', 0, acquire);
  if(arguments.length === 1) return partial1(hook$acquire, acquire);
  if(arguments.length === 2) return hook$acquire(acquire, cleanup);
  return hook$acquire(acquire, cleanup, consume);
}

function Node(fn){
  this._fn = fn;
}

Node.prototype = Object.create(Core);

Node.prototype._fork = function Node$fork(rej, res){
  var open = true;
  this._fn(function Node$done(err, val){
    if(open){
      open = false;
      err ? rej(err) : res(val);
    }
  });
  return function Node$cancel(){ open = false; };
};

Node.prototype.toString = function Node$toString(){
  return 'Future.node(' + showf(this._fn) + ')';
};

function node(f){
  if(!isFunction(f)) invalidArgument('Future.node', 0, 'be a function', f);
  return new Node(f);
}

function check$parallel(m, i){
  return isFuture(m) ? m : invalidFuture(
    'Future.parallel',
    'its second argument to be an array of valid Futures. '
  + 'The value at position ' + i + ' in the array is not a Future',
    m
  );
}

function Parallel(max, futures){
  this._futures = futures;
  this._length = futures.length;
  this._max = Math.min(this._length, max);
}

Parallel.prototype = Object.create(Core);

Parallel.prototype._fork = function Parallel$_fork(rej, res){

  var _futures = this._futures, _length = this._length, _max = this._max;
  var cancels = new Array(_length), out = new Array(_length);
  var cursor = 0, running = 0, blocked = false;

  function Parallel$cancel(){
    cursor = _length;
    for(var n = 0; n < _length; n++) cancels[n] && cancels[n]();
  }

  function Parallel$run(idx){
    running++;
    cancels[idx] = _futures[idx]._fork(function Parallel$rej(reason){
      cancels[idx] = noop;
      Parallel$cancel();
      rej(reason);
    }, function Parallel$res(value){
      cancels[idx] = noop;
      out[idx] = value;
      running--;
      if(cursor === _length && running === 0) res(out);
      else if(blocked) Parallel$drain();
    });
  }

  function Parallel$drain(){
    blocked = false;
    while(cursor < _length && running < _max) Parallel$run(cursor++);
    blocked = true;
  }

  Parallel$drain();

  return Parallel$cancel;

};

Parallel.prototype.toString = function Parallel$toString(){
  return 'Future.parallel(' + this._max + ', ' + show(this._futures) + ')';
};

var emptyArray = new Resolved([]);

function parallel$max(max, xs){
  if(!isArray(xs)) invalidArgument('Future.parallel', 1, 'be an array', xs);
  var futures = mapArray(xs, check$parallel);
  return futures.length === 0 ? emptyArray : new Parallel(max, futures);
}

function parallel(max, xs){
  if(!isUnsigned(max)) invalidArgument('Future.parallel', 0, 'be a positive integer', max);
  if(arguments.length === 1) return partial1(parallel$max, max);
  return parallel$max(max, xs);
}

function check$promise$3(p, f){
  return isThenable(p) ? p : typeError(
    'Future.tryP expects the function it\'s given to return a Promise/Thenable'
    + '\n  Actual: ' + show(p) + '\n  From calling: ' + showf(f)
  );
}

function TryP(fn){
  this._fn = fn;
}

TryP.prototype = Object.create(Core);

TryP.prototype._fork = function TryP$fork(rej, res){
  var open = true;
  check$promise$3(this._fn(), this._fn).then(immediately(function TryP$res(x){
    if(open){
      open = false;
      res(x);
    }
  }), immediately(function TryP$rej(x){
    if(open){
      open = false;
      rej(x);
    }
  }));
  return function TryP$cancel(){ open = false; };
};

TryP.prototype.toString = function TryP$toString(){
  return 'Future.tryP(' + show(this._fn) + ')';
};

function tryP(f){
  if(!isFunction(f)) invalidArgument('Future.tryP', 0, 'be a function', f);
  return new TryP(f);
}

if(typeof Object.create !== 'function') error('Please polyfill Object.create to use Fluture');
if(typeof Object.assign !== 'function') error('Please polyfill Object.assign to use Fluture');
if(typeof Array.isArray !== 'function') error('Please polyfill Array.isArray to use Fluture');

Future.of = Future[FL.of] = of;
Future.chainRec = Future[FL.chainRec] = chainRec;
Future.reject = reject;
Future.ap = ap;
Future.map = map;
Future.bimap = bimap;
Future.chain = chain;

var Par = concurrify(Future, never, race, parallelAp);
Par.of = Par[FL.of];
Par.zero = Par[FL.zero];
Par.map = map;
Par.ap = ap;
Par.alt = alt;

function isParallel(x){
  return x instanceof Par || type(x) === Par['@@type'];
}

function seq(par){
  if(!isParallel(par)) invalidArgument('Future.seq', 0, 'to be a Par', par);
  return par.sequential;
}




var Fluture = Object.freeze({
	Future: Future,
	default: Future,
	Par: Par,
	isParallel: isParallel,
	seq: seq,
	isFuture: isFuture,
	reject: reject,
	of: of,
	never: never,
	isNever: isNever,
	after: after,
	rejectAfter: rejectAfter,
	attempt: attempt,
	try: attempt,
	cache: cache,
	encase: encase,
	encase2: encase2,
	encase3: encase3,
	encaseN: encaseN,
	encaseN2: encaseN2,
	encaseN3: encaseN3,
	encaseP: encaseP,
	encaseP2: encaseP2,
	encaseP3: encaseP3,
	go: go,
	do: go,
	hook: hook,
	node: node,
	parallel: parallel,
	tryP: tryP,
	ap: ap,
	alt: alt,
	map: map,
	bimap: bimap,
	chain: chain,
	mapRej: mapRej,
	chainRej: chainRej,
	lastly: lastly,
	finally: lastly,
	and: and,
	both: both,
	or: or,
	race: race,
	swap: swap,
	fold: fold,
	done: done,
	fork: fork,
	promise: promise,
	value: value,
	extractLeft: extractLeft,
	extractRight: extractRight
});

var index_cjs = Object.assign(Future, Fluture);

return index_cjs;

})));
