/**
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * @providesModule FluxContainerSubscriptions
 * 
 */

'use strict';

function _classCallCheck(instance, Constructor) { if (!(instance instanceof Constructor)) { throw new TypeError('Cannot call a class as a function'); } }

var FluxStoreGroup = require('./FluxStoreGroup');

function shallowArrayEqual(a, b) {
  if (a === b) {
    return true;
  }
  if (a.length !== b.length) {
    return false;
  }
  for (var i = 0; i < a.length; i++) {
    if (a[i] !== b[i]) {
      return false;
    }
  }
  return true;
}

var FluxContainerSubscriptions = (function () {
  function FluxContainerSubscriptions() {
    _classCallCheck(this, FluxContainerSubscriptions);

    this._callbacks = [];
  }

  FluxContainerSubscriptions.prototype.setStores = function setStores(stores) {
    var _this = this;

    if (this._stores && shallowArrayEqual(this._stores, stores)) {
      return;
    }
    this._stores = stores;
    this._resetTokens();
    this._resetStoreGroup();

    var changed = false;
    var changedStores = [];

    if (process.env.NODE_ENV !== 'production') {
      // Keep track of the stores that changed for debugging purposes only
      this._tokens = stores.map(function (store) {
        return store.addListener(function () {
          changed = true;
          changedStores.push(store);
        });
      });
    } else {
      (function () {
        var setChanged = function () {
          changed = true;
        };
        _this._tokens = stores.map(function (store) {
          return store.addListener(setChanged);
        });
      })();
    }

    var callCallbacks = function () {
      if (changed) {
        _this._callbacks.forEach(function (fn) {
          return fn();
        });
        changed = false;
        if (process.env.NODE_ENV !== 'production') {
          // Uncomment this to print the stores that changed.
          // console.log(changedStores);
          changedStores = [];
        }
      }
    };
    this._storeGroup = new FluxStoreGroup(stores, callCallbacks);
  };

  FluxContainerSubscriptions.prototype.addListener = function addListener(fn) {
    this._callbacks.push(fn);
  };

  FluxContainerSubscriptions.prototype.reset = function reset() {
    this._resetTokens();
    this._resetStoreGroup();
    this._resetCallbacks();
    this._resetStores();
  };

  FluxContainerSubscriptions.prototype._resetTokens = function _resetTokens() {
    if (this._tokens) {
      this._tokens.forEach(function (token) {
        return token.remove();
      });
      this._tokens = null;
    }
  };

  FluxContainerSubscriptions.prototype._resetStoreGroup = function _resetStoreGroup() {
    if (this._storeGroup) {
      this._storeGroup.release();
      this._storeGroup = null;
    }
  };

  FluxContainerSubscriptions.prototype._resetStores = function _resetStores() {
    this._stores = null;
  };

  FluxContainerSubscriptions.prototype._resetCallbacks = function _resetCallbacks() {
    this._callbacks = [];
  };

  return FluxContainerSubscriptions;
})();

module.exports = FluxContainerSubscriptions;