import { FormDataLike } from "./FormDataLike";
import { FileLike } from "./FileLike";
export declare class Encoder {
    #private;
    /**
     * Returns boundary string
     */
    readonly boundary: string;
    /**
     * Returns Content-Type header for multipart/form-data
     */
    readonly contentType: string;
    /**
     * Returns headers object with Content-Type and Content-Length header
     */
    readonly headers: {
        "Content-Type": string;
        "Content-Length": number;
    };
    /**
     * Creates a multipart/form-data encoder.
     *
     * @param form - FormData object to encode. This object must be a spec-compatible FormData implementation.
     * @param boundary - An optional boundary string that will be used by the encoder. If there's no boundary string is present, Encoder will generate it automatically.
     *
     * @example
     *
     * import {Encoder} from "form-data-encoder"
     * import {FormData} from "formdata-node"
     *
     * const fd = new FormData()
     *
     * fd.set("greeting", "Hello, World!")
     *
     * const encoder = new Encoder(fd)
     */
    constructor(form: FormDataLike, boundary?: string);
    /**
     * Returns form-data content length
     */
    getContentLength(): number;
    /**
     * Creates an iterator allowing to go through form-data parts (with metadata).
     * This method **will not** read the files.
     *
     * Using this method, you can convert form-data content into Blob:
     *
     * @example
     *
     * import {Readable} from "stream"
     *
     * import {Encoder} from "form-data-encoder"
     *
     * import {FormData} from "formdata-polyfill/esm-min.js"
     * import {fileFrom} from "fetch-blob/form.js"
     * import {File} from "fetch-blob/file.js"
     * import {Blob} from "fetch-blob"
     *
     * import fetch from "node-fetch"
     *
     * const fd = new FormData()
     *
     * fd.set("field", "Just a random string")
     * fd.set("file", new File(["Using files is class amazing"]))
     * fd.set("fileFromPath", await fileFrom("path/to/a/file.txt"))
     *
     * const encoder = new Encoder(fd)
     *
     * const options = {
     *   method: "post",
     *   body: new Blob(encoder, {type: encoder.contentType})
     * }
     *
     * const response = await fetch("https://httpbin.org/post", options)
     *
     * console.log(await response.json())
     */
    values(): Generator<Uint8Array | FileLike, void, undefined>;
    /**
     * Creates an async iterator allowing to perform the encoding by portions.
     * This method **will** also read files.
     *
     * @example
     *
     * import {Readable} from "stream"
     *
     * import {FormData, File, fileFromPath} from "formdata-node"
     * import {Encoder} from "form-data-encoder"
     *
     * import fetch from "node-fetch"
     *
     * const fd = new FormData()
     *
     * fd.set("field", "Just a random string")
     * fd.set("file", new File(["Using files is class amazing"]))
     * fd.set("fileFromPath", await fileFromPath("path/to/a/file.txt"))
     *
     * const encoder = new Encoder(fd)
     *
     * const options = {
     *   method: "post",
     *   headers: encoder.headers,
     *   body: Readable.from(encoder.encode()) // or Readable.from(encoder)
     * }
     *
     * const response = await fetch("https://httpbin.org/post", options)
     *
     * console.log(await response.json())
     */
    encode(): AsyncGenerator<Uint8Array, void, undefined>;
    /**
     * Creates an iterator allowing to read through the encoder data using for...of loops
     */
    [Symbol.iterator](): Generator<Uint8Array | FileLike, void, undefined>;
    /**
     * Creates an **async** iterator allowing to read through the encoder data using for-await...of loops
     */
    [Symbol.asyncIterator](): AsyncGenerator<Uint8Array, void, undefined>;
}
