import { FORMKIT_VERSION } from '@formkit/core';
import { Command } from 'commander';
import chalk from 'chalk';
import axios from 'axios';
import { inputs } from '@formkit/inputs';
import { token } from '@formkit/utils';
import { resolve, isAbsolute, relative, dirname } from 'path';
import { writeFile, mkdir, access, readFile, readdir } from 'fs/promises';
import { execa, execaCommand } from 'execa';
import { cwd } from 'node:process';
import prompts from 'prompts';
import ora from 'ora';
import http from 'http';
import url, { fileURLToPath } from 'url';
import open from 'open';
import { existsSync, constants } from 'fs';

/**
 * @internal
 */
async function exportInput(inputOption, options = {}) {
    const input = await requireInput(inputOption);
    const lang = await requireLang(options.lang);
    const exportData = await requireInputCode(input, lang);
    const sourceCode = transformSource(exportData, input);
    const [absoluteDir, relativeDir] = await requireOutputDir(options.dir);
    const validDir = await upsertDir(absoluteDir);
    if (validDir === false)
        return error(`${relativeDir} is not a writable directory.`);
    if (!validDir)
        return;
    const outFile = resolve(absoluteDir, `${input}.${lang}`);
    if (!existsSync(outFile)) {
        await writeFile(outFile, sourceCode);
    }
    else {
        return error('Did not export input because that file already exists.');
    }
    green(`Success! Exported ${relativeDir}/${input}.${lang}`);
    console.log(`To use it pass it to your FormKit configuration:
  // ...
  import { ${input} } from '${relativeDir}/${input}'
  // ...
  const config = defaultConfig({
    inputs: {
      ${input}
    }
  })
  `);
}
/**
 * Checks if a given directory is writable, if it doesn't exist, create it.
 * @param dir - A directory to create if it doesn't exist.
 * @returns
 */
async function upsertDir(dir) {
    if (!existsSync(dir)) {
        const local = '.' + dir.replace(process.cwd(), '');
        const { confirm } = await prompts({
            type: 'confirm',
            name: 'confirm',
            initial: true,
            message: `Export directory does not exist (${local}) does not exist. Create it?`,
        });
        if (!confirm)
            return info('Directory not created — no input was exported.');
        try {
            await mkdir(dir, { recursive: true });
        }
        catch {
            return error(`Unable to create directory ${dir}.`);
        }
    }
    try {
        await access(dir, constants.W_OK);
        return true;
    }
    catch (err) {
        return false;
    }
}
/**
 * Attempts to intelligently determine the directory to export to.
 * @param dir - The directory to export the input to.
 * @returns
 */
async function requireOutputDir(dir) {
    if (dir && isAbsolute(dir)) {
        return [dir, relative(process.cwd(), dir)];
    }
    else if (dir) {
        const abs = resolve(process.cwd(), dir);
        return [abs, relative(process.cwd(), abs)];
    }
    const rel = await prompts({
        type: 'text',
        name: 'dir',
        message: 'Where should the input be exported to (relative to the current directory)?',
        initial: guessDir(),
    }).then((res) => res.dir);
    const abs = resolve(process.cwd(), rel);
    return [abs, rel];
}
function guessDir() {
    const srcDir = resolve(process.cwd(), 'src');
    if (existsSync(srcDir)) {
        return './src/inputs';
    }
    return './inputs';
}
/**
 * Transforms the source code of the inputs to be used locally.
 * @param exportData - The code to export.
 * @returns
 */
function transformSource(exportData, type) {
    if (exportData) {
        // Change the exports from relative to npm package based.
        exportData = exportData.replace(/(}\sfrom\s['"])\.\.\/(?:index)?(['"])?/g, '$1@formkit/inputs$2');
        const memoKey = token();
        exportData = exportData.replace(/(schemaMemoKey:\s?['"])[a-zA-Z0-9]+(['"])/g, `$1${memoKey}$2`);
        // Inject the forceTypeProp in the definition.
        exportData = exportData.replace(/^  props: \[(.*)\],/gm, `  props: [$1],
  /**
   * Forces node.props.type to be this explicit value.
   */
  forceTypeProp: '${type}',`);
    }
    else {
        error('Unable to export the input file because it cannot be located.');
    }
    return exportData;
}
/**
 * Determine the language the user wants to export.
 * @param lang - The language to export the input to.
 * @returns
 */
async function requireLang(lang) {
    if (!lang) {
        const guessedLang = guessLang();
        lang = await prompts({
            type: 'select',
            name: 'lang',
            message: 'What language should be used?',
            choices: [
                guessedLang === 'ts'
                    ? { title: 'TypeScript', value: 'ts' }
                    : { title: 'JavaScript', value: 'js' },
                guessedLang === 'ts'
                    ? { title: 'JavaScript', value: 'js' }
                    : { title: 'TypeScript', value: 'ts' },
            ],
        }).then((val) => val.lang);
        if (!lang) {
            error('No language selected, exiting.');
        }
    }
    return lang;
}
/**
 * Fetch the input name that the user wants to export.
 * @param inputName - The name of the input to load.
 * @returns
 */
async function requireInput(inputName) {
    if (!inputName) {
        const res = await prompts({
            type: 'autocomplete',
            name: 'inputName',
            message: 'What input do you want to export?',
            choices: Object.keys(inputs).map((i) => ({
                title: i,
                value: i,
            })),
        });
        inputName = res.inputName;
    }
    if (!inputName || !(inputName in inputs)) {
        error(`Cannot export “${inputName}” because it is not part of the @formkit/inputs package.`);
    }
    return inputName;
}
/**
 * Loads the string data of an input that should be exported.
 * @param name - The name of the input to load.
 * @param lang - The language to load the input in.
 * @returns
 */
async function requireInputCode(name, lang) {
    var _a;
    lang = !lang ? guessLang() : lang;
    const localFile = resolve(__dirname, `../../inputs/dist/exports/${name}.${lang}`);
    let fileData = null;
    if (existsSync(localFile)) {
        fileData = await readFile(localFile, { encoding: 'utf8' });
    }
    else {
        warning(`Unable to locate ${localFile}`);
        const cdnUrl = `https://cdn.jsdelivr.net/npm/@formkit/inputs@${FORMKIT_VERSION}/dist/exports/${name}.${lang}`;
        try {
            const res = await axios.get(cdnUrl);
            fileData = res.data;
        }
        catch (e) {
            if (e && ((_a = e === null || e === void 0 ? void 0 : e.response) === null || _a === void 0 ? void 0 : _a.status)) {
                error(`${e.response.status} — unable to load ${localFile}`);
            }
            else {
                error('Unable to load input file — probably a network error. Are you online?');
            }
        }
    }
    if (!fileData) {
        info('Checking CDN for an exportable input.');
    }
    if (!fileData) {
        error(`Unable to load export ${name}.${lang}`);
    }
    else {
        return fileData;
    }
}
/**
 * Guess the language the user is leveraging on their project.
 */
function guessLang() {
    const tsconfig = resolve(process.cwd(), 'tsconfig.json');
    return existsSync(tsconfig) ? 'ts' : 'js';
}

const APP_URL = 'https://pro.formkit.com';
async function login() {
    const spinner = ora(`To login visit: ${APP_URL}/cli-login`).start();
    await open(`${APP_URL}/cli-login`);
    let server;
    const token = await new Promise((resolve, reject) => {
        server = http
            .createServer((req, res) => {
            const urlObj = url.parse(req.url, true);
            const token = urlObj.query.token;
            if (token) {
                resolve(token);
                res.writeHead(302, {
                    Location: `${APP_URL}/cli-login?success=true`,
                });
                res.end();
            }
            else {
                res.writeHead(302, {
                    Location: `${APP_URL}/cli-login?success=false`,
                });
                reject('Login failed.');
            }
        })
            .listen(5479);
    });
    server === null || server === void 0 ? void 0 : server.close();
    spinner.stop();
    return token;
}
async function createTeam(token) {
    const res = await prompts({
        type: 'text',
        name: 'name',
        message: 'Please enter a new team name:',
        initial: 'My team',
    });
    const response = await fetch(`${APP_URL}/api/teams`, {
        method: 'POST',
        headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
            Accept: 'application/json',
        },
        body: JSON.stringify({
            name: res.name,
        }),
    });
    return await response.json();
}
async function createProject(token, team) {
    const res = await prompts({
        type: 'text',
        name: 'name',
        message: 'Please enter a new project name:',
        initial: 'My new project',
    });
    const response = await fetch(`${APP_URL}/api/teams/${team}/projects`, {
        method: 'POST',
        headers: {
            Authorization: `Bearer ${token}`,
            'Content-Type': 'application/json',
            Accept: 'application/json',
        },
        body: JSON.stringify({
            name: res.name,
            license: 'development',
        }),
    });
    if (!response.ok) {
        error('Failed to create project.');
    }
    const data = await response.json();
    if (data.data) {
        info(`Your project was created successfully with a development license — to upgrade to a production license visit ${APP_URL}`);
    }
    return data.data;
}
async function selectProProject() {
    try {
        const token = (await login());
        const spinner = ora('Fetching account...').start();
        const response = await fetch(`${APP_URL}/api/account`, {
            headers: {
                Authorization: `Bearer ${token}`,
                'Content-Type': 'application/json',
                Accept: 'application/json',
            },
        });
        const data = await response.json();
        spinner.stop();
        const res = await prompts({
            type: 'select',
            name: 'team',
            message: 'Select a team:',
            choices: data.teams
                .map((team) => ({
                title: team.name,
                value: team,
            }))
                .concat([{ title: 'Create a new team', value: 'new' }]),
        });
        if (res.team === 'new') {
            const team = await createTeam(token);
            const project = await createProject(token, team.id);
            return project.api_key;
        }
        else {
            let { project } = await prompts({
                type: 'select',
                name: 'project',
                message: 'Select a project:',
                choices: res.team.projects
                    .map((team) => ({
                    title: team.name,
                    value: team,
                }))
                    .concat([{ title: 'Create a new project', value: 'new' }]),
            });
            if (project === 'new') {
                project = await createProject(token, res.team.id);
            }
            return project.api_key;
        }
    }
    catch (err) {
        console.log(err);
        error('Login failed.');
    }
    return 'test';
}
async function createApp(appName, options = {}) {
    var _a, _b;
    if (!appName) {
        const res = await prompts({
            type: 'text',
            name: 'name',
            message: 'Please enter a directory name for the project:',
            initial: 'formkit-app',
        });
        appName = res.name;
    }
    const isEmpty = await isDirEmpty(resolve(cwd(), `./${appName}`));
    if (!isEmpty) {
        error('Directory is not empty. Please choose a different name.');
    }
    if (!options.framework) {
        const res = await prompts({
            type: 'select',
            name: 'framework',
            message: 'What framework would you like to use?',
            choices: [
                { title: 'Vite', value: 'vite' },
                { title: 'Nuxt', value: 'nuxt' },
            ],
            initial: 1,
        });
        options.framework = res.framework;
    }
    if (!options.lang && options.framework === 'vite') {
        const res = await prompts({
            type: 'select',
            name: 'lang',
            message: 'What language should be used?',
            choices: [
                { title: 'TypeScript', value: 'ts' },
                { title: 'JavaScript', value: 'js' },
            ],
            initial: 1,
        });
        options.lang = res.lang;
    }
    if (!options.pro) {
        const res = await prompts([
            {
                type: 'toggle',
                name: 'install_pro',
                message: 'Would you like to install FormKit Pro?',
                active: 'yes',
                initial: true,
                inactive: 'no',
            },
        ]);
        if (res.install_pro) {
            options.pro = await selectProProject();
        }
    }
    if (options.framework === 'vite') {
        await execa('npx', [
            '--yes',
            'create-vite',
            appName,
            '--template',
            `vue${options.lang === 'ts' ? '-ts' : ''}`,
        ]);
        // TODO: add better version matching here:
        await addDependency(appName, '@formkit/vue', 'latest');
        await addDependency(appName, '@formkit/icons', 'latest');
        if (options.pro) {
            await addDependency(appName, '@formkit/pro');
        }
        await addInitialApp(appName, 'src/App.vue', !!options.pro);
        await writeFile(resolve(cwd(), `./${appName}/src/formkit.config.${options.lang}`), buildFormKitConfig(options));
        await writeFile(resolve(cwd(), `./${appName}/src/main.${options.lang}`), buildMain());
    }
    else {
        options.lang = 'ts';
        info('Fetching nuxi cli...');
        const subprocess = execaCommand(`npx --yes nuxi@latest init --no-install $APP_NAME`, {
            cwd: process.cwd(),
            shell: true,
            stdio: 'inherit',
            env: { APP_NAME: appName },
        });
        (_a = subprocess.stdout) === null || _a === void 0 ? void 0 : _a.pipe(process.stdout);
        (_b = subprocess.stderr) === null || _b === void 0 ? void 0 : _b.pipe(process.stderr);
        await subprocess;
        await writeFile(resolve(cwd(), `./${appName}/formkit.config.ts`), buildFormKitConfig(options));
        await addDependency(appName, '@formkit/nuxt', 'latest');
        await addDependency(appName, '@formkit/icons', 'latest');
        if (options.pro) {
            await addDependency(appName, '@formkit/pro');
        }
        await addNuxtModule(appName);
        await addInitialApp(appName, 'app.vue', !!options.pro);
    }
    green(`Created ${appName}!

To run your new app:
📁 cd ${appName}
✅ npm install
🚀 npm run dev
`);
}
async function addInitialApp(dirName, component, pro) {
    const appPath = resolve(cwd(), `./${dirName}/${component}`);
    await writeFile(appPath, `<script setup>
async function submit() {
  await new Promise(r => setTimeout(r, 1000))
  alert('Submitted! 🎉')
}
</script>

<template>
  <div class="your-first-form">
    <img
      src="https://pro.formkit.com/logo.svg"
      alt="FormKit Logo"
      width="244"
      height="50"
      class="logo"
    >
    <FormKit
      type="form"
      #default="{ value }"
      @submit="submit"
    >
      <FormKit
        type="text"
        name="name"
        label="Name"
        help="What do people call you?"
      />
      <FormKit
        type="checkbox"
        name="flavors"
        label="Favorite ice cream flavors"
        :options="{
          'vanilla': 'Vanilla',
          'chocolate': 'Chocolate',
          'strawberry': 'Strawberry',
          'mint-chocolate-chip': 'Mint Chocolate Chip',
          'rocky-road': 'Rocky Road',
          'cookie-dough': 'Cookie Dough',
          'pistachio': 'Pistachio',
        }"
        validation="required|min:2"
      />
      ${(pro &&
        `
      <FormKit
        type="repeater"
        name="invitees"
        label="Invitees"
        help="Who else should we invite to FormKit?"
      >
        <FormKit
          type="text"
          name="email"
          label="Email"
          validation="required|email"
        />
      </FormKit>`) ||
        ''}
      <FormKit
        type="checkbox"
        name="agree"
        label="I agree FormKit is the best form authoring framework."
      />
      <pre>{{ value }}</pre>
    </FormKit>
  </div>
</template>

<style scoped>
.your-first-form {
  width: calc(100% - 2em);
  max-width: 480px;
  box-sizing: border-box;
  padding: 2em;
  box-shadow: 0 0 1em rgba(0, 0, 0, .1);
  border-radius: .5em;
  margin: 4em auto;
}

.logo {
  width: 150px;
  height: auto;
  display: block;
  margin: 0 auto 2em auto;
}
pre {
  background-color: rgba(0, 100, 250, .1);
  padding: 1em;
}
</style>
`);
}
/**
 * Adds a dependency to a new project’s package.json file.
 * @param dirName - The directory to find a package.json
 * @param dependency - An npm dependency to add.
 */
async function addDependency(dirName, dependency, version = 'latest') {
    const packageJsonPath = resolve(cwd(), `./${dirName}/package.json`);
    const raw = await readFile(packageJsonPath, 'utf-8');
    const packageJson = JSON.parse(raw);
    if (!('dependencies' in packageJson)) {
        packageJson.dependencies = {};
    }
    packageJson.dependencies[dependency] = version;
    await writeFile(packageJsonPath, JSON.stringify(packageJson, null, 2));
}
function buildMain() {
    return `import { createApp } from 'vue'
import { plugin, defaultConfig } from '@formkit/vue'
import App from './App.vue'
import formKitConfig from './formkit.config'

const app = createApp(App)
app.use(plugin, defaultConfig(formKitConfig))
app.mount('#app')
`;
}
async function addNuxtModule(dirName) {
    const nuxtConfigPath = resolve(cwd(), `./${dirName}/nuxt.config.ts`);
    const raw = await readFile(nuxtConfigPath, 'utf-8');
    const configWithFormKit = raw.replace(/(defineNuxtConfig\({\n).*?(\n}\))/g, "$1  modules: ['@formkit/nuxt'],\n  formkit: {\n    autoImport: true\n  }$2");
    await writeFile(nuxtConfigPath, configWithFormKit);
}
/**
 * Builds the formkit.config.ts file.
 * @param options - Build the formkit.config.ts file for a Nuxt project.
 * @returns
 */
function buildFormKitConfig(options) {
    const imports = [
        'import "@formkit/themes/genesis"',
        'import { genesisIcons } from "@formkit/icons"',
    ];
    if (options.lang === 'ts' && options.framework === 'vite') {
        imports.push("import { DefaultConfigOptions } from '@formkit/vue'");
    }
    else if (options.lang === 'ts' && options.framework === 'nuxt') {
        imports.push("import { defineFormKitConfig } from '@formkit/vue'");
    }
    const setup = [];
    let config = '';
    if (options.pro) {
        imports.push("import { createProPlugin, inputs } from '@formkit/pro'");
        imports.push("import '@formkit/pro/genesis'");
        setup.push('');
        setup.push(`const pro = createProPlugin('${options.pro}', inputs)`);
        setup.push('');
        config += `  plugins: [pro]`;
    }
    config += `${config ? ',\n' : ''}  icons: { ...genesisIcons }`;
    const viteExport = `const config${options.lang === 'ts' ? ': DefaultConfigOptions' : ''} = {
${config}
}

export default config`;
    const nuxtExport = `export default defineFormKitConfig({
${config}
})`;
    let defaultExport = '';
    if (options.framework === 'nuxt') {
        defaultExport = nuxtExport;
    }
    else if (options.framework === 'vite') {
        defaultExport = viteExport;
    }
    const rawConfig = `${imports.join('\n')}
${setup.join('\n')}
${defaultExport}
`;
    return rawConfig;
}
async function isDirEmpty(path) {
    try {
        const entries = await readdir(path);
        return entries.length === 0;
    }
    catch (error) {
        return true;
    }
}

const __filename = fileURLToPath(import.meta.url);
/**
 * @internal
 */
const __dirname = dirname(__filename);
/**
 * @internal
 */
const red = (m) => console.log(`${chalk.red.bold('[FormKit Error]: ')} ${chalk.red(m)}`);
/**
 * @internal
 */
const info = (m) => {
    console.log(`${chalk.blue(m)}`);
};
/**
 * @internal
 */
const warning = (m) => {
    console.log(`${chalk.yellow.bold('[FormKit Warn]: ')} ${chalk.yellow(m)}`);
};
/**
 * @internal
 */
const green = (m) => {
    console.log(chalk.greenBright(m));
};
const program = new Command();
program
    .name('FormKit CLI')
    .description('The official FormKit command line utility.')
    .version(FORMKIT_VERSION);
program
    .command('export')
    .argument('[input]', 'An input to export (from @formkit/inputs, like "text" or "select")')
    .description('Export an input from @formkit/inputs to a local file for modification.')
    .option('-d, --dir <dir>', 'The directory to export inputs to')
    .option('-l, --lang <ts|js>', 'Export as TypeScript (ts) or JavaScript (js)')
    .action(exportInput);
program
    .command('create-app')
    .argument('[name]', 'Creates a vite application with this name.')
    .option('-l, --lang <ts|js>', 'Create a TypeScript (ts) or JavaScript (js) app.')
    .option('-f, --framework <vite|nuxt>', 'Create a new Vite or Nuxt app.')
    .option('--pro <key>', 'Creates the project with FormKit pro installed if a project key is provided.')
    .description('Creates a new Vue 3 Vite application with FormKit installed.')
    .action(createApp);
/**
 * @internal
 */
async function main() {
    const res = await execa('npx', ['--version']);
    const [major] = res.stdout.trim().split('.');
    if (Number(major) < 7) {
        error(`Your npm version must be 7 or higher (found ${res.stdout.trim()}).`);
    }
    program.parse();
}
/**
 * @internal
 */
function error(message) {
    red(message);
    process.exit(1);
}

export { __dirname, main as default, error, green, info, red, warning };
