'use strict';

var validation$K = require('@formkit/validation');
var utils = require('@formkit/utils');

/**
 * Given a string, convert it to sentence case.
 *
 * @param str - The string to sentence case.
 *
 * @returns `string`
 *
 * @public
 */
function sentence(str) {
    return str[0].toUpperCase() + str.substr(1);
}
/**
 * Creates an oxford-comma separated list of items.
 *
 * @param items - the items to list out.
 * @param conjunction - in the list "x, y, and z", "and" is the conjunction.
 * Defaults to "or".
 *
 * @returns `string`
 *
 * @public
 */
function list(items, conjunction = 'or') {
    return items.reduce((oxford, item, index) => {
        oxford += item;
        if (index <= items.length - 2 && items.length > 2) {
            oxford += ', ';
        }
        if (index === items.length - 2) {
            oxford += `${items.length === 2 ? ' ' : ''}${conjunction} `;
        }
        return oxford;
    }, '');
}
/**
 * Given a string or a date, return a nice human-readable version.
 *
 * @param date - A string or a date.
 *
 * @returns `string`
 *
 * @public
 */
function date(date) {
    const dateTime = typeof date === 'string' ? new Date(Date.parse(date)) : date;
    if (!(dateTime instanceof Date)) {
        return '(unknown)';
    }
    return new Intl.DateTimeFormat(undefined, {
        dateStyle: 'medium',
        timeZone: 'UTC'
    }).format(dateTime);
}
/**
 * Orders two variables from smallest to largest.
 *
 * @param first - The first number or string.
 * @param second - The second number or string.
 *
 * @returns `[smaller: number | string, larger: number | string]`
 *
 * @public
 */
function order(first, second) {
    return Number(first) >= Number(second) ? [second, first] : [first, second];
}

/**
 * Standard language for interface features.
 * @public
 */
const ui$J = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'إضافة',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'إزالة',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'إزالة الكل',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'عذرا، لم يتم تعبئة جميع الحقول بشكل صحيح.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'إرسال',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'لا يوجد ملف مختار',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'تحرك لأعلى',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'انتقل لأسفل',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'يتم الآن التحميل...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'تحميل المزيد',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'التالي',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'السابق',
    /**
     * Shown when transferring items between lists.
     */
    addAllValues: 'أضف جميع القيم',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'إضافة قيم محددة',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'قم بإزالة جميع القيم',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'إزالة القيم المحددة',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'اختر التاريخ',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'تاريخ التغيير',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'التاريخ المحدد غير صالح.',
    /**
     * Shown when there is something to close
     */
    close: 'أغلق',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$J = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `الرجاء قبول ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `يجب أن يكون ${sentence(name)} بعد ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `يجب أن يكون ${sentence(name)} في المستقبل.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `يمكن أن يحتوي ${sentence(name)} على أحرف أبجدية فقط.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `يمكن أن يحتوي ${sentence(name)} على أحرف وأرقام فقط.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `يمكن أن تحتوي ${sentence(name)} على أحرف ومسافات فقط.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `يجب أن يحتوي ${sentence(name)} على أحرف أبجدية.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `يجب أن يحتوي ${sentence(name)} على أحرف أو أرقام.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `يجب أن يحتوي ${sentence(name)} على أحرف أو مسافات.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `يجب أن يحتوي ${sentence(name)} على رمز.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `يجب أن يحتوي ${sentence(name)} على أحرف كبيرة.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `يجب أن يحتوي ${sentence(name)} على أحرف صغيرة.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `يجب أن يحتوي ${sentence(name)} على أرقام.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `يجب أن يكون ${sentence(name)} رمزًا.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `يمكن أن يحتوي ${sentence(name)} على أحرف كبيرة فقط.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `يمكن أن يحتوي ${sentence(name)} على أحرف صغيرة فقط.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `يجب أن يكون ${sentence(name)} قبل ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `يجب أن يكون ${sentence(name)} في الماضي.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `تمت تهيئة هذا الحقل بشكل غير صحيح ولا يمكن إرساله.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `يجب أن يكون ${sentence(name)} ما بين ${a} و ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} غير متطابق.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} ليس تاريخًا صالحًا ، يرجى استخدام التنسيق ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'تمت تهيئة هذا الحقل بشكل غير صحيح ولا يمكن إرساله';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `يجب أن يكون ${sentence(name)} بين ${date(args[0])} و ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'الرجاء أدخال بريد إليكتروني صالح.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `لا ينتهي ${sentence(name)} بـ ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} ليست قيمة مسموح بها.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `يجب أن يكون ${sentence(name)} حرفًا واحدًا على الأقل.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `يجب أن يكون ${sentence(name)} أقل من أو يساوي ${max} حرفًا.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `يجب أن يتكون ${sentence(name)} من الأحرف ${max}.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `يجب أن يكون ${sentence(name)} أكبر من أو يساوي ${min} حرفًا.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `يجب أن يكون ${sentence(name)} بين ${min} و ${max} حرفًا.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} ليست قيمة مسموح بها.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `لا يمكن أن يكون أكثر من ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `يجب أن يكون ${sentence(name)} أقل من أو يساوي ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'لا يسمح بتنسيقات الملفات.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `يجب أن يكون ${sentence(name)} من النوع: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `لا يمكن أن يكون أقل من ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `يجب أن يكون ${sentence(name)} على الأقل ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” ليس ${name} مسموحًا به.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} يجب ان يكون رقماً`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' أو ')} مطلوب.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} مطلوب.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `لا يبدأ ${sentence(name)} بـ ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `يرجى إدخال عنوان URL صالح.`;
        /* </i18n> */
    },
};

var ar = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$J,
    validation: validation$J
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$I = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'əlavə edin',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'çıxarmaq',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Hamısını silin',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Üzr istəyirik, bütün sahələr düzgün doldurulmayıb.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Təqdim et',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Heç bir fayl seçilməyib',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'yuxarı hərəkət',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Aşağı hərəkət',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Yükləmə...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Daha çox yüklə',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Növbəti',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Əvvəlki',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Bütün dəyərləri əlavə edin',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Seçilmiş dəyərləri əlavə edin',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Bütün dəyərləri sil',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Seçilmiş dəyərləri sil',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Tarixi seçin',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Tarixi dəyişdirin',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Seçilmiş tarix etibarsızdır.',
    /**
     * Shown when there is something to close
     */
    close: 'Bağlayın',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$I = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `${name} qəbul edin.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])} sonra olmalıdır.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} gələcəkdə olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} yalnız əlifba sırası simvollarından ibarət ola bilər.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} yalnız hərf və rəqəmlərdən ibarət ola bilər.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} yalnız hərflərdən və boşluqlardan ibarət ola bilər.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} əlifba sırası simvolları ehtiva etməlidir.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} hərfləri və ya nömrələri ehtiva etməlidir.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} hərfləri və ya boşluqları ehtiva etməlidir.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} simvolu ehtiva etməlidir.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} böyük olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} kiçik olmalıdır.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} nömrələri ehtiva etməlidir.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} simvol olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} yalnız böyük hərfləri ehtiva edə bilər.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} yalnız kiçik hərfləri ehtiva edə bilər.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])} əvvəl olmalıdır.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} keçmişdə olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Bu sahə səhv konfiqurasiya edilib və onu təqdim etmək mümkün deyil.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} ${a} və ${b} arasında olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} uyğun gəlmir.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} etibarlı tarix deyil, ${args[0]} formatından istifadə edin`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Bu sahə səhv konfiqurasiya edilib və onu təqdim etmək mümkün deyil';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} ${date(args[0])} və ${date(args[1])} arasında olmalıdır`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Etibarlı e-poçt ünvanı daxil edin.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)} ilə bitmir.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} icazə verilən dəyər deyil.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} ən azı bir simvol olmalıdır.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} ${max} simvoldan kiçik və ya ona bərabər olmalıdır.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} ${max} simvol uzunluğunda olmalıdır.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} ${min} simvoldan böyük və ya ona bərabər olmalıdır.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} ${min} və ${max} simvol arasında olmalıdır.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} icazə verilən dəyər deyil.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${args[0]} ${name}-dən çox ola bilməz.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} ${args[0]} dəyərindən kiçik və ya ona bərabər olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Fayl formatlarına icazə verilmir.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} aşağıdakı tipdə olmalıdır: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${args[0]} ${name}-dən az ola bilməz.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} ən azı ${args[0]} olmalıdır.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” icazə verilən ${name} deyil.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} rəqəm olmalıdır.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' və ya ')} tələb olunur.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} tələb olunur.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)} ilə başlamır.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Xahiş edirik, düzgün URL daxil edin.`;
        /* </i18n> */
    },
};

var az = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$I,
    validation: validation$I
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$H = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Добави',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Премахни',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Премахни всички',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Извинете, не всички полета са попълнени правилно.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Изпрати',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Няма избран файл',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Преместване нагоре',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Преместете се надолу',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Зареждане...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Заредете повече',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Следващ',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Предишен',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Добавете всички стойности',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Добавяне на избрани стойности',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Премахнете всички стойности',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Премахване на избраните стойности',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Избери дата',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Промяна на датата',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Избраната дата е невалидна.',
    /**
     * Shown when there is something to close
     */
    close: 'Затвори',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$H = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Моля приемете ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} трябва да е след ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} трябва да бъде в бъдещето.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} може да съдържа само букви.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} може да съдържа само букви и цифри.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} може да съдържа само букви и интервали.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} трябва да съдържа азбучни знаци.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} трябва да съдържа букви или цифри.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} трябва да съдържа букви или интервали.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} трябва да съдържа символ.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} трябва да съдържа главни букви.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} трябва да съдържа малки букви.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} трябва да съдържа числа.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} трябва да бъде символ.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} може да съдържа само главни букви.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} може да съдържа само малки букви.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} трябва да е преди ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} трябва да бъде в миналото.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Това поле е конфигурирано неправилно и не може да бъде изпратено`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} трябва да бъде между ${a} и ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} не съвпада.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} е невалидна дата. Моля, използвайте формата ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Това поле е конфигурирано неправилно и не може да бъде изпратено';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} трябва да бъде между ${date(args[0])} и ${date(args[1])}.`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Моля, въведете валиден имейл адрес.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} не завършва на ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} е неразрешена стойност.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} трябва да има поне един символ.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} трябва да бъде по-малко или равно на ${max} символа.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} трябва да бъде ${max} символи дълго.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} трябва да бъде по-голямо или равно на ${min} символа.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} трябва да бъде между ${min} и ${max} символа.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} е неразрешена стойност.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Не може да има повече от ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} трябва да бъде по-малко или равно на ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Не са разрешени никакви файлови формати.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} трябва да бъде от тип: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Не може да има по-малко от ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} трябва да бъде поне ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” е неразрешен ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} трябва да бъде число.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' или ')} изисква се.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} е задължително.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} не започва с ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Моля, въведете валиден URL адрес.`;
        /* </i18n> */
    },
};

var bg = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$H,
    validation: validation$H
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$G = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Dodaj',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Ukloni',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Ukloni sve',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Nažalost, nisu sva polja ispravno popunjena.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Pošalji',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nije odabran nijedan fajl',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Pomjeri gore',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Pomjeri dole',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Učitavanje...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Učitaj više',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Sljedeći',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Prethodni',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Dodajte sve vrijednosti',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Dodajte odabrane vrijednosti',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Uklonite sve vrijednosti',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Uklonite odabrane vrijednosti',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Odaberite datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Promjenite datum',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Odabrani datum je nevažeći.',
    /**
     * Shown when there is something to close
     */
    close: 'Zatvori',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$G = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Molimo prihvatite ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} mora biti poslije ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti u budućnosti.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} može sadržavati samo abecedne karaktere.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} može sadržavati samo slova i brojeve.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} može sadržavati samo slova i razmake.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} mora sadržavati abecedne karaktere.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} mora sadržavati slova ili brojeve.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} mora sadržavati slova ili razmake.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} mora sadržavati simbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} mora sadržavati veliko slovo.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} mora sadržavati malo slovo.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} mora sadržavati brojeve.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} mora biti simbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} može sadržavati samo velika slova.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} može sadržavati samo mala slova.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} mora biti prije ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti u prošlosti.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Ovo polje je pogrešno konfigurirano i ne može se poslati.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} mora biti između ${a} i ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} ne podudara se.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} nije ispravan datum, molimo koristite format ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Ovo polje je pogrešno konfigurirano i ne može se poslati';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} mora biti između ${date(args[0])} i ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Molimo Vas da unesete validnu email adresu.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} ne završava sa ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} nije dozvoljena vrijednost.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} mora biti najmanje jedan karakter.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} mora biti manje od ili jednako ${max} karaktera.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} treba imati ${max} karaktera.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} mora biti veći od ili jednak ${min} karaktera.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} mora biti između ${min} i ${max} karaktera.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nije dozvoljena vrijednost.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Ne može imati više od ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} mora biti manji ili jednak ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Nisu dozvoljeni formati datoteka.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} mora biti tipa: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Ne može imati manje od ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `Mora biti barem ${args[0]} ${name} .`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” nije dozvoljeno ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} mora biti broj.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' ili ')} je obavezno.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} je obavezno.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ne počinje sa ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Unesite važeći link.`;
        /* </i18n> */
    },
};

var bs = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$G,
    validation: validation$G
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$F = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Afegir',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Eliminar',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Eliminar tot',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Disculpi, no tots els camps estan omplerts correctament.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Enviar',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Cap fitxer triat',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Moure amunt',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Moure avall',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Carregant...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Carregar més',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Següent',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Anterior',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Afegir tots els valors',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Afegeix els valors seleccionats',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Eliminar tots els valors',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Elimina els valors seleccionats',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Trieu la data',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Canviar data',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'La data seleccionada no és vàlida.',
    /**
     * Shown when there is something to close
     */
    close: 'Tancar',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$F = {
    /**
     * The value is not an accepted value.
     * @see {@link https://docs.formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Si us plau accepti ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://docs.formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} ha de ser posterior a ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} ha de succeïr al futur.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://docs.formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} només pot contenir caràcters alfabètics.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://docs.formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} només pot contenir lletres i números.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://docs.formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} només pot contenir lletres i espais.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} ha de contenir caràcters alfabètics.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} ha de contenir lletres o números.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} ha de contenir lletres o espais.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} ha de contenir símbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} ha de contenir majúscules.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} ha de contenir minúscules.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} ha de contenir números.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} ha de ser un símbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} només pot contenir lletres majúscules.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} només pot contenir lletres minúscules.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://docs.formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} ha de ser anterior a ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} ha d'estar al passat.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://docs.formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Aquest camp està configurat incorrectament i no pot ésser enviat.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} ha d'estar entre ${a} i ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://docs.formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} no concorda.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://docs.formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} no és una data vàlida, si us plau empri el format ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Aquest camp està configurat incorrectament i no pot ésser enviat';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://docs.formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} ha d'estar entre ${date(args[0])} i ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://docs.formkit.com/essentials/validation#email}
     */
    email: `Si us plau, entri una adreça d'e-mail vàlida.`,
    /**
     * Does not end with the specified value
     * @see {@link https://docs.formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} no acaba amb ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://docs.formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} no és un valor acceptat.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://docs.formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} ha de tenir com a mínim un caràcter.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} ha de ser inferior o igual a ${max} caràcters.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} ha de tenir una longitud de ${max} caràcters.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} ha de ser major que o igual a ${min} caràcters.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} ha d'estar entre ${min} i ${max} caràcters.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://docs.formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} no és un valor permès.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://docs.formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `No pot tenir més de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} ha de ser menys que o igual a ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://docs.formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'No hi ha cap format de fitxer acceptat.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} ha de ser del tipus: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://docs.formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `No pot tenir menys de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} ha de ser com a mínim ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://docs.formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” no s'accepta com a ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://docs.formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} ha de ser un número.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' o ')} es requereix.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://docs.formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} és obligatori.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://docs.formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} no comença amb ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://docs.formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Si us plau inclogui una url vàlida.`;
        /* </i18n> */
    },
};

var ca = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$F,
    validation: validation$F
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$E = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Přidat',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Odebrat',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Odebrat vše',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Pardon, ale ne všechna pole jsou vyplněna správně.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Odeslat',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Žádný soubor nebyl vybrán',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Pohyb nahoru',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Posunout dolů',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Načítání...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Načíst více',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Další',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Předchozí',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Přidat všechny hodnoty',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Přidání vybraných hodnot',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Odstraňte všechny hodnoty',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Odstranění vybraných hodnot',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Zvolte datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Změnit datum',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Vybrané datum je neplatné.',
    /**
     * Shown when there is something to close
     */
    close: 'Zavřít',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$E = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Prosím, zaškrtněte ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} musí být po ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} musí být v budoucnosti.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} může obsahovat pouze písmena.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} může obsahovat pouze písmena a čísla.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} musí obsahovat abecední znaky.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} musí obsahovat písmena nebo číslice.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} musí obsahovat písmena nebo mezery.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} musí obsahovat symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} musí obsahovat velká písmena.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} musí obsahovat malá písmena.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} musí obsahovat čísla.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} musí být symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} může obsahovat pouze velká písmena.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} může obsahovat pouze malá písmena.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} musí být před ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} musí být v minulosti.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Toto pole bylo špatně nakonfigurováno a nemůže být odesláno.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} musí být mezi ${a} a ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} nejsou shodná.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} není platné datum, prosím, použijte formát ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Toto pole bylo špatně nakonfigurováno a nemůže být odesláno.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} musí být mezi ${date(args[0])} a ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Prosím, zadejte platnou e-mailovou adresu.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} nekončí na ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} není povolená hodnota.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} musí mít nejméně jeden znak.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} může mít maximálně ${max} znaků.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} by mělo být ${max} znaků dlouhé.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} musí obsahovat minimálně ${min} znaků.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} musí být dlouhé ${min} až ${max} znaků.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} není povolená hodnota.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Nelze použít více než ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} musí mít menší nebo rovno než ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Nejsou nakonfigurovány povolené typy souborů.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} musí být typu: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Nelze mít méně než ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} musí být minimálně ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” není dovolená hodnota pro ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} musí být číslo.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' nebo ')} je vyžadován.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} je povinné.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} nezačíná na ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Zadejte prosím platnou adresu URL.`;
        /* </i18n> */
    },
};

var cs = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$E,
    validation: validation$E
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$D = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Tilføj',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Fjern',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Fjern alle',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Alle felter er ikke korrekt udfyldt.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Send',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Ingen filer valgt',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Flyt op',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Flyt ned',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Indlæser...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Indlæs mere',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Næste',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Forrige',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Tilføj alle værdier',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Tilføj valgte værdier',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Fjern alle værdier',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Fjern valgte værdier',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Vælg dato',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Skift dato',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Den valgte dato er ugyldig.',
    /**
     * Shown when there is something to close
     */
    close: 'Luk',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$D = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Accepter venligst ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} skal være senere end ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} skal være i fremtiden.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} kan kun indeholde bogstaver.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} kan kun indeholde bogstaver og tal.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} kan kun indeholde bogstaver og mellemrum.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} skal indeholde alfabetiske tegn.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} skal indeholde bogstaver eller tal.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} skal indeholde bogstaver eller mellemrum.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} skal indeholde symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} skal indeholde store bogstaver.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} skal indeholde små bogstaver.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} skal indeholde tal.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} skal være et symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} kan kun indeholde store bogstaver.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} kan kun indeholde små bogstaver.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} skal være før ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} skal være før i dag.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Dette felt er ikke konfigureret korrekt og kan derfor ikke blive sendt.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} skal være mellem ${a} og ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} matcher ikke.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} er ikke gyldig, brug venligst formatet ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Dette felt er ikke konfigureret korrekt og kan derfor ikke blive sendt.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} skal være mellem ${date(args[0])} og ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Indtast venligst en gyldig email-adresse.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} slutter ikke med ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} er ikke en gyldig værdi.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} skal være på mindst ét tegn.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} skal være på højst ${max} tegn.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} skal være ${max} tegn lange.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} skal være på mindst ${min} tegn.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} skal være på mindst ${min} og højst ${max} tegn.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} er ikke en gyldig værdi.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Kan ikke have flere end ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} skal være mindre eller lig med ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Ingen filformater tilladt.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} skal være af filtypen: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Kan ikke have mindre end ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} skal være mindst ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” er ikke en tilladt ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} skal være et tal.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' eller ')} er påkrævet.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} er påkrævet.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} starter ikke med ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Indtast en gyldig URL.`;
        /* </i18n> */
    },
};

var da = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$D,
    validation: validation$D
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$C = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Hinzufügen',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Entfernen',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Alles entfernen',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Entschuldigung, nicht alle Felder wurden korrekt ausgefüllt.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Senden',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Keine Datei ausgewählt',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Gehe nach oben',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Gehen Sie nach unten',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Wird geladen...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Mehr laden',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Weiter',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Voriges',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Alle Werte hinzufügen',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Ausgewählte Werte hinzufügen',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Alle Werte entfernen',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Ausgewählte Werte entfernen',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Datum wählen',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Datum ändern',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Das gewählte Datum ist ungültig.',
    /**
     * Shown when there is something to close
     */
    close: 'Schliessen',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$C = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Bitte ${name} akzeptieren.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} muss nach dem ${date(args[0])} liegen.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} muss in der Zukunft liegen.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} darf nur Buchstaben enthalten.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} darf nur Buchstaben und Zahlen enthalten.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} dürfen nur Buchstaben und Leerzeichen enthalten.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} muss alphabetische Zeichen enthalten.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} muss Buchstaben oder Zahlen enthalten.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} muss Buchstaben oder Leerzeichen enthalten.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} muss ein Symbol enthalten.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} muss Großbuchstaben enthalten.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} muss Kleinbuchstaben enthalten.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} muss Zahlen enthalten.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} muss ein Symbol sein.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} kann nur Großbuchstaben enthalten.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} kann nur Kleinbuchstaben enthalten.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} muss vor dem ${date(args[0])} liegen.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} muss in der Vergangenheit liegen.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Dieses Feld wurde falsch konfiguriert und kann nicht übermittelt werden.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} muss zwischen ${args[0]} und ${args[1]} sein.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} stimmt nicht überein.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} ist kein gültiges Datum im Format ${args[0]}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Dieses Feld wurde falsch konfiguriert und kann nicht übermittelt werden.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} muss zwischen ${date(args[0])} und ${date(args[1])} liegen.`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'E-Mail Adresse ist ungültig.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} endet nicht mit ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} enthält einen ungültigen Wert.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = first <= second ? first : second;
        const max = second >= first ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} muss mindestens ein Zeichen enthalten.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} darf maximal ${max} Zeichen enthalten.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} sollte ${max} Zeichen lang sein.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} muss mindestens ${min} Zeichen enthalten.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} muss zwischen ${min} und ${max} Zeichen enthalten.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} enthält einen ungültigen Wert.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Darf maximal ${args[0]} ${name} haben.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} darf maximal ${args[0]} sein.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Keine Dateiformate konfiguriert.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} muss vom Typ ${args[0]} sein.`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Mindestens ${args[0]} ${name} erforderlich.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} muss mindestens ${args[0]} sein.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” ist kein gültiger Wert für ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} muss eine Zahl sein.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' oder ')} ist erforderlich.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} ist erforderlich.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} beginnt nicht mit ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Bitte geben Sie eine gültige URL ein.`;
        /* </i18n> */
    },
};

var de = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$C,
    validation: validation$C
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$B = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Προσθήκη',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Αφαίρεση',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Αφαίρεση όλων',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Λυπούμαστε, υπάρχουν πεδία που δεν έχουν συμπληρωθεί σωστά.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Υποβολή',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Κανένα αρχείο δεν έχει επιλεγεί',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Προς τα επάνω',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Προς τα κάτω',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Φορτώνει...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Φόρτωση περισσότερων',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Επόμενη',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Προηγούμενο',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Προσθήκη όλων των τιμών',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Προσθήκη επιλεγμένων τιμών',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Κατάργηση όλων των τιμών',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Αφαίρεση επιλεγμένων τιμών',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Επιλέξτε ημερομηνία',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Αλλαγή ημερομηνίας',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Η επιλεγμένη ημερομηνία δεν είναι έγκυρη.',
    /**
     * Shown when there is something to close
     */
    close: 'Κλείσιμο',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$B = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Παρακαλώ αποδεχτείτε το ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} πρέπει να είναι μετά της ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} πρέπει να είναι στο μέλλον.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} μπορεί να περιέχει μόνο αλφαβητικούς χαρακτήρες.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} μπορεί να περιέχει μόνο γράμματα και αριθμούς.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} μπορεί να περιέχει μόνο γράμματα και κενά.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `Το ${sentence(name)} πρέπει να περιέχει αλφαβητικούς χαρακτήρες.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `Το ${sentence(name)} πρέπει να περιέχει γράμματα ή αριθμούς.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} πρέπει να περιέχει γράμματα ή κενά.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `Το ${sentence(name)} πρέπει να περιέχει το σύμβολο.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `Το ${sentence(name)} πρέπει να περιέχει κεφαλαία γράμματα.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `Το ${sentence(name)} πρέπει να περιέχει πεζά γράμματα.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `Το ${sentence(name)} πρέπει να περιέχει αριθμούς.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `Το ${sentence(name)} πρέπει να είναι ένα σύμβολο.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `Το ${sentence(name)} μπορεί να περιέχει μόνο κεφαλαία γράμματα.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `Το ${sentence(name)} μπορεί να περιέχει μόνο πεζά γράμματα.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} πρέπει να είναι πριν της ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} πρέπει να είναι στο παρελθόν.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Αυτό το πεδίο έχει ρυθμιστεί λανθασμένα και δεν μπορεί να υποβληθεί.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} πρέπει να είναι μεταξύ ${a} και ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} δεν ταιριάζει.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} δεν είναι έγυρη ημερομηνία, παρακαλώ ακολουθήστε την διαμόρφωση ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Αυτό το πεδίο έχει ρυθμιστεί λανθασμένα και δεν μπορεί να υποβληθεί';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} πρέπει να είναι μεταξύ ${date(args[0])} και ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Παρακαλώ πληκτρολογήστε μια έγκυρη email διεύθυνση. ',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} δεν καταλήγει με ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} δεν είναι μια επιτρεπτή τιμή.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} πρέπει να είναι τουλάχιστον ενός χαρακτήρα.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} πρέπει να είναι λιγότεροι ή ίσοι με ${max} χαρακτήρες.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `Το ${sentence(name)} θα πρέπει να έχει μήκος ${max} χαρακτήρες.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} πρέπει να είναι περισσότεροι ή ίσοι με ${min} χαρακτήρες.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} πρέπει να είναι μεταξύ ${min} και ${max} χαρακτήρες.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} δεν είναι μια επιτρεπτή τιμή.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Δεν μπορεί να έχει παραπάνω από ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} πρέπει αν είναι λιγότερο ή ίσο με το ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Δεν επιτρέπονται αρχεία.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} πρέπει να είναι τύπου: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Δεν μπορεί να είναι λιγότερο από ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} πρέπει να είναι τουλάχιστον ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” δεν είναι μια επιτρεπτή ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} πρέπει να είναι αριθμός.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' ή ')} απαιτείται.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} είναι υποχρεωτικό.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} δεν αρχίζει με ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Παρακαλώ εισάγετε ένα έγκυρο URL.`;
        /* </i18n> */
    },
};

var el = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$B,
    validation: validation$B
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$A = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Add',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Remove',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Remove all',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Sorry, not all fields are filled out correctly.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Submit',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'No file chosen',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Move up',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Move down',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Loading...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Load more',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Next',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Previous',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Add all values',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Add selected values',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Remove all values',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Remove selected values',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Choose date',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Change date',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'The selected date is invalid.',
    /**
     * Shown above error summaries when someone attempts to submit a form with
     * errors and the developer has implemented `<FormKitSummary />`.
     */
    summaryHeader: 'There were errors in your form.',
    /*
     * Shown when there is something to close
     */
    close: 'Close',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$A = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Please accept the ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} must be after ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} must be in the future.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} can only contain alphabetical characters.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} can only contain letters and numbers.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} can only contain letters and spaces.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} must contain alphabetical characters.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} must contain letters or numbers.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} must contain letters or spaces.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} must contain a symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} must contain an uppercase letter.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} must contain a lowercase letter.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} must contain numbers.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} must be a symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} can only contain uppercase letters.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} can only contain lowercase letters.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} must be before ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} must be in the past.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `This field was configured incorrectly and can’t be submitted.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} must be between ${a} and ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} does not match.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} is not a valid date, please use the format ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'This field was configured incorrectly and can’t be submitted';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} must be between ${date(args[0])} and ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Please enter a valid email address.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} doesn’t end with ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} is not an allowed value.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} must be at least one character.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} must be less than or equal to ${max} characters.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} should be ${max} characters long.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} must be greater than or equal to ${min} characters.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} must be between ${min} and ${max} characters.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} is not an allowed value.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Cannot have more than ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} must be less than or equal to ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'No file formats allowed.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} must be of the type: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Cannot have fewer than ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `Must be at least ${args[0]} ${name} .`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” is not an allowed ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} must be a number.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' or ')} is required.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} is required.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} doesn’t start with ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Please enter a valid URL.`;
        /* </i18n> */
    },
};

var en = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$A,
    validation: validation$A
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$z = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Añadir',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Quitar',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Quitar todos',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Discúlpe, los campos no fueron completados correctamente.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Enviar',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Archivo no seleccionado',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Moverse hacia arriba',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Moverse hacia abajo',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Cargando...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Cargar más',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Próximo',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Anterior',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Añadir todos los valores',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Añadir valores seleccionados',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Eliminar todos los valores',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Eliminar los valores seleccionados',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Elige fecha',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Cambiar fecha',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'La fecha seleccionada no es válida.',
    /**
     * Shown when there is something to close
     */
    close: 'Cerrar',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$z = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Acepte el ${name} por favor.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} debe ser posterior a ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} debe ser una fecha futura.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} debe contener solo caractéres alfabéticos.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} debe ser alfanumérico.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} espacios alfa solo pueden contener letras y espacios.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} debe contener caracteres alfabéticos.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} debe contener letras o números.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} debe contener letras o espacios.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} debe contener un símbolo.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} debe estar en mayúsculas.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} debe contener minúsculas.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} debe contener números.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} debe ser un símbolo.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} solo puede contener letras mayúsculas.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} solo puede contener letras minúsculas.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} debe ser anterior a ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} debe ser una fecha pasada.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `El campo no fue completado correctamente y no puede ser enviado.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} debe estar entre ${a} y ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} no coincide.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} no es una fecha válida, por favor utilice el formato ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'El campo no fue completado correctamente y no puede ser enviado.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} debe estar entre ${date(args[0])} y ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Ingrese una dirección de correo electrónico válida por favor.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} no termina con ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} no es un valor permitido.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} debe tener al menos una letra.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} debe tener como máximo ${max} caractéres.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} debe tener ${max} caracteres.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} debe tener como mínimo ${min} caractéres.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} debe tener entre ${min} y ${max} caractéres.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} no es un valor permitido.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `No puede tener más de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} debe ser menor o igual a ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'No existen formatos de archivos permitidos.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} debe ser del tipo: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `No puede tener menos de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} debe ser de al menos ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” no es un valor permitido de ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} debe ser un número.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' o ')} se requiere estar.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} es requerido.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} debe comenzar con ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Introduce una URL válida.`;
        /* </i18n> */
    },
};

var es = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$z,
    validation: validation$z
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$y = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'افزودن',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'حذف',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'همه را حذف کنید',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'همه فیلدها به‌درستی پر نشده‌اند',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'ثبت',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'هیچ فایلی انتخاب نشده است',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'حرکت به بالا',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'حرکت به پایین',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'در حال بارگذاری...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'بارگذاری بیشتر',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'بعدی',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'قبلی',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'تمام مقادیر را اضافه کنید',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'مقادیر انتخاب شده را اضافه کنید',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'تمام مقادیر را حذف کنید',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'حذف مقادیر انتخاب شده',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'تاریخ را انتخاب کنید',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'تغییر تاریخ',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'تاریخ انتخاب شده نامعتبر است',
    /**
     * Shown when there is something to close
     */
    close: 'بستن',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$y = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `لطفاً ${name} را بپذیرید.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} باید بعد از تاریخ ${date(args[0])} باشد.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} باید مربوط به آینده باشد.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} فقط میتواند شامل حروف الفبا باشد.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} فقط میتواند شامل حروف و اعداد باشد.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} فقط می تواند شامل حروف و فاصله باشد.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} باید حاوی حروف الفبا باشد.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} باید حاوی حروف یا اعداد باشد.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} باید حاوی حروف یا فاصله باشد.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} باید حاوی نماد باشد.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} باید دارای حروف بزرگ باشد.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} باید حاوی حروف کوچک باشد.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} باید حاوی اعداد باشد.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} باید یک نماد باشد.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} تنها می‌تواند شامل حروف بزرگ باشد.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} تنها می‌تواند شامل حروف کوچک باشد.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} باید قبل از تاریخ ${date(args[0])} باشد.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} باید مربوط به گذشته باشد.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `این فیلد به اشتباه پیکربندی شده است و قابل ارسال نیست`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} باید بین ${args[0]} و ${args[1]} باشد.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} مطابقت ندارد.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} تاریخ معتبری نیست، لطفاً از قالب ${args[0]} استفاده کنید
`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'این فیلد به اشتباه پیکربندی شده است و قابل ارسال نیست';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} باید بین ${date(args[0])} و ${date(args[1])} باشد.`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'لطفا آدرس ایمیل معتبر وارد کنید.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} باید به ${list(args)} ختم شود.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} مجاز نیست.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = first <= second ? first : second;
        const max = second >= first ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} باید حداقل یک کاراکتر باشد.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} باید کمتر یا برابر با ${max} کاراکتر باشد.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} باید ${max} کاراکتر طولانی باشد.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} باید بزرگتر یا برابر با ${min} کاراکتر باشد.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} باید بین ${min} و ${max} کاراکتر باشد.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} مجاز نیست.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${name} نمی تواند بیش از ${args[0]} باشد.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} باید کمتر یا برابر با ${args[0]} باشد.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'فرمت فایل مجاز نیست.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} باید از این نوع باشد: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${name} نمی تواند کمتر از ${args[0]} باشد.
`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} باید حداقل ${args[0]} باشد.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" یک ${name} مجاز نیست.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} باید عدد باشد.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' یا ')} مورد نیاز است.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `پر کردن ${sentence(name)} اجباری است.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} باید با ${list(args)} شروع شود.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `لطفا یک URL معتبر وارد کنید.`;
        /* </i18n> */
    },
};

var fa = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$y,
    validation: validation$y
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$x = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Lisää',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Poista',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Poista kaikki',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Kaikkia kenttiä ei ole täytetty oikein.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Tallenna',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Ei valittuja tiedostoja',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Siirrä ylös',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Siirrä alas',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Ladataan...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Lataa lisää',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Seuraava',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Edellinen',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Lisää kaikki arvot',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Lisää valitut arvot',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Poista kaikki arvot',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Poista valitut arvot',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Valitse päivämäärä',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Vaihda päivämäärä',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Valittu päivämäärä on virheellinen.',
    /**
     * Shown when there is something to close
     */
    close: 'Sulje',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$x = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Ole hyvä ja hyväksy ${name}`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} tulee olla ${date(args[0])} jälkeen.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} on oltava tulevaisuudessa.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} saa sisältää vain kirjaimia.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} saa sisältää vain kirjaimia ja numeroita.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} voivat sisältää vain kirjaimia ja välilyöntejä.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} on sisällettävä aakkoselliset merkit.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} täytyy sisältää kirjaimia tai numeroita.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} täytyy sisältää kirjaimia tai välilyöntejä.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} täytyy sisältää symboli.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} täytyy sisältää isoja kirjaimia.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} täytyy sisältää pieniä kirjaimia.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} täytyy sisältää numeroita.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} on oltava symboli.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} voi sisältää vain isoja kirjaimia.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} voi sisältää vain pieniä kirjaimia.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} tulee olla ennen: ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} on oltava menneisyydessä.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Tämä kenttä on täytetty virheellisesti joten sitä ei voitu lähettää.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} on oltava välillä ${a} - ${b} `;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} ei täsmää.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} ei ole validi päivämäärä, ole hyvä ja syötä muodossa: ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Tämä kenttä on täytetty virheellisesti joten sitä ei voitu lähettää.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} on oltava välillä ${date(args[0])} - ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Syötä validi sähköpostiosoite.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} tulee päättyä ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} ei ole sallittu vaihtoehto.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} on oltava vähintään yksi merkki.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} on oltava ${max} tai alle merkkiä.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} pitäisi olla ${max} merkkiä pitkä.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} on oltava vähintään ${min} merkkiä.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} on oltava vähintään ${min}, enintään ${max} merkkiä.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} ei ole sallittu arvo.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Valitse enintään ${args[0]} ${name} vaihtoehtoa.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} on oltava ${args[0]} tai alle.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Tiedostoja ei sallita.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} tulee olla ${args[0]}-tiedostotyyppiä.`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Valitse vähintään ${args[0]} ${name} vaihtoehtoa.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} tulee olla ${args[0]} tai suurempi.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” ei ole sallittu ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `Kentän ${sentence(name)} tulee olla numero.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' tai ')} vaaditaan.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} vaaditaan.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} on alettava ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Anna kelvollinen URL-osoite.`;
        /* </i18n> */
    },
};

var fi = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$x,
    validation: validation$x
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$w = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Ajouter',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Supprimer',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Enlever tout',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Désolé, tous les champs ne sont pas remplis correctement.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Valider',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Aucun fichier choisi',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Déplacez-vous',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Déplacez-vous',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Chargement...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Chargez plus',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Suivant',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Précédent',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Ajouter toutes les valeurs',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Ajouter les valeurs sélectionnées',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Supprimer toutes les valeurs',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Supprimer les valeurs sélectionnées',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Choisissez la date',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Modifier la date',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: `La date sélectionnée n'est pas valide.`,
    /**
     * Shown when there is something to close
     */
    close: 'Fermer',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$w = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Veuillez accepter le ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} doit être postérieure au ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} doit être dans le futur.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} ne peut contenir que des caractères alphabétiques.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} ne peut contenir que des lettres et des chiffres.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} ne peuvent contenir que des lettres et des espaces.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} doit contenir des caractères alphabétiques.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} doit contenir au moins un lettre ou nombre.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} doit contenir des lettres ou des espaces.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} doit contenir un symbole.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} doit contenir au moins une majuscule.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} doit contenir au moins une minuscule.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} doit contenir des chiffres.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} doit être un symbole.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} ne peuvent contenir que des majuscules.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} ne peut contenir que des lettres minuscules.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} doit être antérieure au ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} doit être dans le passé.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Ce champ a été configuré de manière incorrecte et ne peut pas être soumis.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} doit être comprise entre ${a} et ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} ne correspond pas.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} n'est pas une date valide, veuillez utiliser le format ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Ce champ a été configuré de manière incorrecte et ne peut pas être soumis.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} doit être comprise entre ${date(args[0])} et ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Veuillez saisir une adresse email valide.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} ne se termine pas par ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} n'est pas une valeur autorisée.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} doit comporter au moins un caractère.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} doit être inférieur ou égal à ${max} caractères.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} doit contenir ${max} caractères.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} doit être supérieur ou égal à ${min} caractères.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} doit être comprise entre ${min} et ${max} caractères.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} n'est pas une valeur autorisée.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Ne peut pas avoir plus de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} doit être inférieur ou égal à ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Aucun format de fichier n’est autorisé';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} doit être du type: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Ne peut pas avoir moins de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} doit être au moins de ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” n'est pas un ${name} autorisé.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} doit être un nombre.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' ou ')} est requis.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} est requis.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ne commence pas par ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Entrez une URL valide.`;
        /* </i18n> */
    },
};

var fr = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$w,
    validation: validation$w
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$v = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Foeg ta',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Ferwider',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Ferwider alles',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Sorry, net alle fjilden binne korrekt ynfolle.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Ferstjoere',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Gjin bestân keazen',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Gean omheech',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Nei ûnderen',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Lade…',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Mear lade',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Folgende',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Foarige',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Foegje alle wearden ta',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Foegje selektearre wearden ta',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Fuortsmite alle wearden',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Fuortsmite selektearre wearden',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Kies datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Feroarje datum',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'De selektearre datum is ûnjildich.',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$v = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Akseptearje de ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} moat nei ${date(args[0])} wêze.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} moat yn de takomst lizze.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} mei allinne alfabetyske tekens befetsje.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} mei allinne letters en sifers befetsje.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} mei allinne letters en spaasjes befetsje.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} must contain alphabetical characters.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} must contain letters and numbers.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} must contain letters and spaces.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} must contain symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} must contain uppercase.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} must contain lowercase.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} must contain number.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} can only contain symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} can only contain uppercase.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} can only contain lowercase.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} moat foar ${date(args[0])} falle.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} moat yn it ferline wêze.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Dit fjild is ferkeard konfigurearre en kin net ferstjoerd wurde.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} moat tusken ${a} en ${b} lizze.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} komt net oerien.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} is gjin jildige datum, brûk de notaasje ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Dit fjild is ferkeard konfigurearre en kin net ferstjoerd wurde';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} moat tusken ${date(args[0])} en ${date(args[1])} lizze`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Folje in jildich e-mailadres yn.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} einiget net mei ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} is gjin tastiene wearde.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} moat minimaal ien teken wêze.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} moat lytser wêze as of gelyk wêze oan ${max} tekens.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} moat ${max} tekens lang wêze.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} moat grutter wêze as of gelyk wêze oan ${min} tekens.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} moat tusken de ${min} en ${max} tekens befetsje.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} is gjin tastiene wearde.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Mei net mear as ${args[0]} ${name} hawwe.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} moat lytser wêze as of gelyk wêze oan ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Gjin bestânsnotaasjes tastien.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} moat fan it type: ${args[0]} wêze`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Mei net minder as ${args[0]} ${name} hawwe.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} moat minimaal ${args[0]} wêze.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" is gjin tastiene ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} moat in getal wêze.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames.map(name => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        }).filter(name => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' of ')} is ferplichte.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} is ferplicht.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} begjint net mei ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Doch der in jildige url by.`;
        /* </i18n> */
    },
};

var fy = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$v,
    validation: validation$v
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$u = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'הוסף',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'מחק',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'מחק הכל',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'שים לב, לא כל השדות מלאים כראוי.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'שלח',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'לא נבחר קובץ..',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'הזז למעלה',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'הזז למטה',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'טוען...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'טען יותר',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'הבא',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'הקודם',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'הוסף את כל הערכים',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'הוספת ערכים נבחרים',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'הסר את כל הערכים',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'הסר ערכים נבחרים',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'בחר תאריך',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'שינוי תאריך',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'התאריך שנבחר אינו חוקי.',
    /**
     * Shown when there is something to close
     */
    close: 'סגור',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$u = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `אנא אשר את ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} חייב להיות אחרי ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} חייב להיות בעתיד.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} חייב להכיל אותיות אלפבת.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} יכול להכיל רק מספרים ואותיות.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} יכול להכיל רק אותיות ורווחים.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} חייב להכיל תווים אלפביתיים.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} חייב להכיל אותיות או מספרים.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} חייב להכיל אותיות או רווחים.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} חייב להכיל סמל.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} חייב להכיל אותיות רישיות.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} חייב להכיל אותיות קטנות.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} חייב להכיל מספרים.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} חייב להיות סמל.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} יכול להכיל אותיות רישיות בלבד.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} יכול להכיל רק אותיות קטנות.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} חייב להיות לפני ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} חייב להיות בעבר`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `שדה זה לא הוגדר כראוי ולא יכול להישלח.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} חייב להיות בין ${a} ו- ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} לא מתאים.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} הוא לא תאריך תקין, אנא השתמש בפורמט ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'שדה זה לא הוגדר כראוי ולא יכול להישלח.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} חייב להיות בין ${date(args[0])} ו- ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'אנא הקלד אימייל תקין.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} לא מסתיים ב- ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} הוא לא ערך מורשה.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} חייב להיות לפחות תו אחד.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} חייב להיות פחות או שווה ל- ${max} תווים.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} צריך להיות ${max} תווים ארוכים.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} חייב להיות גדול או שווה ל- ${min} תווים.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} חייב להיות בין ${min} ו- ${max} תווים.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} הוא לא ערך תקין.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${name} לא יכול להיות עם יותר מ- ${args[0]}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} חייב להיות פחות או שווה ל- ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'פורמט הקובץ לא מורשה.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} חייב להיות מסוג: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${name} לא יכול להיות עם פחות מ- ${args[0]}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} חייב להיות לפחות ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” לא מתאים ל- ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} חייב להיות מספר.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' או ')} נדרש.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} הינו חובה.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} לא מתחיל ב- ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `הזן כתובת URL חוקית.`;
        /* </i18n> */
    },
};

var he = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$u,
    validation: validation$u
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$t = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Dodaj',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Ukloni',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Pojedina polja nisu ispravno ispunjena.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Predaj',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Pomaknite se gore',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Pomakni se dolje',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Učitavanje...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Učitaj više',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Sljedeći',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'prijašnji',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Dodajte sve vrijednosti',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Dodavanje odabranih vrijednosti',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Ukloni sve vrijednosti',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Ukloni odabrane vrijednosti',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Odaberite datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Promijeni datum',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Odabrani datum je nevažeći.',
    /**
     * Shown when there is something to close
     */
    close: 'Zatvoriti',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$t = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Potrebno je potvrditi ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} mora biti u periodu poslije ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti u budućnosti.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} mora sadržavati samo slova.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} mora sadržavati slova i brojeve.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} mogu sadržavati samo slova i razmake..`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} mora sadržavati abecedne znakove.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} mora sadržavati slova ili brojeve.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} mora sadržavati slova ili razmake.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} mora sadržavati simbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} mora sadržavati velika slova.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} mora sadržavati mala slova.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} mora sadržavati brojeve.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} mora biti simbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} može sadržavati samo velika slova.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} može sadržavati samo mala slova.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} mora biti prije ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti u prošlosti.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Format sadržaja nije ispravan i ne može biti predan.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} mora biti između ${args[0]} i ${args[1]}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} ne odgovara zadanoj vrijednosti.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} nije ispravan format datuma. Molimo koristite sljedeći format: ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Ovo polje nije ispravno postavljeno i ne može biti predano.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} mora biti vrijednost između ${date(args[0])} i ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Molimo upišite ispravnu email adresu.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} ne završava s ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} nije dopuštena vrijednost.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = first <= second ? first : second;
        const max = second >= first ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} mora sadržavati barem jedan znak.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} mora imati ${max} ili manje znakova.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} trebao bi biti dugačak ${max} znakova.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} mora imati barem ${min} znakova.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `Broj znakova za polje ${sentence(name)} mora biti između ${min} i ${max}.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nije dozvoljena vrijednost.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Ne smije imati više od ${args[0]} ${name} polja.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} mora imati vrijednost manju ili jednaku ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Format datoteke nije dozvoljen.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `Format datoteke na polju ${sentence(name)} mora odgovarati: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Broj upisanih vrijednosti na polju ${name} mora biti barem ${args[0]}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} mora biti barem ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” nije dozvoljena vrijednost na polju ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} mora biti broj.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' ili ')} je potreban.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} je obavezno.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ne počinje s ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Unesite valjanu URL adresu.`;
        /* </i18n> */
    },
};

var hr = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$t,
    validation: validation$t
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$s = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Hozzáadás',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Eltávolítás',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Összes eltávolítása',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Sajnáljuk, nem minden mező lett helyesen kitöltve.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Beküldés',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nincs fájl kiválasztva',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Mozgás felfelé',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Mozgás lefelé',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Betöltés...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Töltsön be többet',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Következő',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Előző',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Adja hozzá az összes értéket',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Kiválasztott értékek hozzáadása',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Távolítsa el az összes értéket',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'A kiválasztott értékek eltávolítása',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Válassza ki a dátumot',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Dátum módosítása',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'A kiválasztott dátum érvénytelen.',
    /**
     * Shown when there is something to close
     */
    close: 'Bezárás',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$s = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Fogadja el a ${name} mezőt.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} mezőnek ${date(args[0])} után kell lennie.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mezőnek a jövőben kell lennie.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} csak alfanumerikus karaktereket tartalmazhat.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} csak betűket és számokat tartalmazhat.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} csak betűket és szóközöket tartalmazhat.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `A ${sentence(name)} betűrendes karaktereket kell tartalmaznia.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `A ${sentence(name)} betűket vagy számokat kell tartalmaznia.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `A ${sentence(name)} betűket vagy szóközöket kell tartalmaznia.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `A ${sentence(name)} szimbólumot kell tartalmaznia.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `A ${sentence(name)} nagybetűt kell tartalmaznia.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `A ${sentence(name)} kisbetűt kell tartalmaznia.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `A ${sentence(name)} számot kell tartalmaznia.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `A ${sentence(name)} szimbólumnak kell lennie.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `A ${sentence(name)} csak nagybetűket tartalmazhat.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `A ${sentence(name)} csak kisbetűket tartalmazhat.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} mezőnek ${date(args[0])} előtt kell lennie.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mezőnek a múltban kell lennie.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Ez a mező hibásan lett konfigurálva, így nem lehet beküldeni.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `A ${sentence(name)} mezőnek ${a} és ${b} között kell lennie.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} nem egyezik.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} nem érvényes dátum, ${args[0]} formátumot használj`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Ez a mező hibásan lett konfigurálva, így nem lehet beküldeni.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} mezőnek ${date(args[0])} és ${args[1]} között kell lennie`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Kérjük, érvényes email címet adjon meg.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} mező nem a kijelölt (${list(args)}) módon ér véget.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} nem engedélyezett érték.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} mezőnek legalább egy karakteresnek kell lennie.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} mezőnek maximum ${max} karakteresnek kell lennie.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} ${max} karakter hosszúnak kell lennie.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} mezőnek minimum ${min} karakteresnek kell lennie.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} mezőnek ${min} és ${max} karakter között kell lennie.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nem engedélyezett érték.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Nem lehet több mint ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} nem lehet nagyobb, mint ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Nincsenek támogatott fájlformátumok.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)}-nak/nek a következőnek kell lennie: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Nem lehet kevesebb, mint ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)}-nak/nek minimum ${args[0]}-nak/nek kell lennie.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" nem engedélyezett ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} mezőnek számnak kell lennie.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' vagy ')} szükséges.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} mező kötelező.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} nem a következővel kezdődik: ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Kérjük, adjon meg egy érvényes URL-t.`;
        /* </i18n> */
    },
};

var hu = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$s,
    validation: validation$s
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$r = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Tambah',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Hapus',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Hapus semua',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Maaf, tidak semua bidang formulir terisi dengan benar',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Kirim',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Tidak ada file yang dipilih',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Pindah ke atas',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Pindah ke bawah',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Memuat...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Muat lebih',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Berikutnya',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Sebelumnya',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Tambahkan semua nilai',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Tambahkan nilai yang dipilih',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Hapus semua nilai',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Hapus nilai yang dipilih',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Pilih tanggal',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Ubah tanggal',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Tanggal yang dipilih tidak valid.',
    /**
     * Shown when there is something to close
     */
    close: 'Tutup',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$r = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Tolong terima kolom ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} nilainya harus lebih dari waktu ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} harus berisi waktu di masa depan.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} hanya bisa diisi huruf alfabet.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} hanya bisa diisi huruf dan angka.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} hanya boleh berisi huruf dan spasi..`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} harus berisi karakter abjad.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} harus mengandung huruf atau angka.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} harus berisi huruf atau spasi.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} harus berisi simbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} harus berisi huruf besar.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} harus berisi huruf kecil.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} harus berisi angka.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} harus berupa simbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} hanya dapat berisi huruf besar.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} hanya dapat berisi huruf kecil.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} nilainya harus kurang dari waktu ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} harus berisi waktu yang sudah lampau.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Kolom ini tidak diisi dengan benar sehingga tidak bisa dikirim`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} harus bernilai diantara ${a} dan ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} nilainya tidak cocok.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} waktu tidak cocok, mohon gunakan format waktu ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Kolom ini tidak diisi dengan benar sehingga tidak bisa dikirim';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} harus diantara waktu ${date(args[0])} dan waktu ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Tolong tulis alamat email yang benar.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} nilainya tidak berakhiran dengan ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} adalah nilai yang tidak diizinkan.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} nilainya setidaknya berisi satu karakter.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} jumlah karakternya harus kurang dari atau sama dengan ${max} karakter.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} harus ${max} karakter panjang.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} jumlah karakternya harus lebih dari atau sama dengan ${min} karakter.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} jumlah karakternya hanya bisa antara ${min} dan ${max} karakter.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nilainya tidak diizinkan.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Tidak bisa memiliki lebih dari ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} harus lebih kecil atau sama dengan ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Format file tidak diizinkan';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} hanya bisa bertipe: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Tidak boleh kurang dari ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} setidaknya harus berisi ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” adalah nilai yang tidak diperbolehkan untuk ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} harus berupa angka.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' atau ')} diperlukan`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} harus diisi.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} tidak dimulai dengan ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Harap masukkan URL yang valid.`;
        /* </i18n> */
    },
};

var id = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$r,
    validation: validation$r
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$q = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Bæta við',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Fjarlægja',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Fjarlægja allt',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Því miður, það er ekki búið að fylla rétt inn í alla reiti.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Senda',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Engin skrá valin',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Færa upp',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Færa niður',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Hleðsla...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Sækja meira',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Áfram',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Til baka',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Bæta við öllum gildum',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Bæta við völdum gildum',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Fjarlægja öll gildi',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Fjarlægja valin gildi',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Velja dagsetningu',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Breyta dagsetningu',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Valin dagsetning er ógild',
    /**
     * Shown when there is something to close
     */
    close: 'Loka',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$q = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Vinsamlegast samþykktu ${name}`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} þarf að vera eftir ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} þarf að vera í framtíðinni.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} má einungis innihalda bókstafi.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} má einungis innihalda bókstafi og tölur.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} má einungis innihalda bókstafi og bil.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} verður að innihalda bókstafi.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} verður að innihalda bókstafi eða tölur.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} verður að innihalda bókstafi eða bil.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} verður að innihalda tákn.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} verður að innihalda hástaf.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} verður að innihalda lágstaf.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} verður að innihalda tölur.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} verður að vera tákn.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} má einungis innihalda hástafi.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} má einungis innihalda lágstafi.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} þarf að vera á undan ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} þarf að vera liðin.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Þessi reitur var ekki rétt stilltur.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} þarf að vera á milli ${a} og ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} passar ekki.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} er ekki gild dagsetning, vinsamlegast hafið dagsetninguna í formi ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Þessi reitur var ekki rétt stilltur';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} þarf að vera á milli ${date(args[0])} og ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Vinsamlegast sláðu inn gilt netfang.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} endar ekki á ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} er ekki leyfilegt gildi.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} þarf að vera að minnsta kosti einn stafur.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} þarf að vera færri en eða nákvæmlega ${max} stafir.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} þarf að vera ${max} stafir að lengd.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} þarf að vera fleiri en eða nákvæmlega ${min} stafir.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} þarf að vera á milli ${min} og ${max} stafir.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} er ekki leyfilegt gildi.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Ekki hægt að hafa fleiri en ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} þarf að vera minna en eða nákvæmlega ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Ekki leyfileg tegund skráar.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} þarf að vera af tegundinni: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Ekki hægt að hafa færri en ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `Þarf að vera að minnsta kosti ${args[0]} ${name}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” er ekki leyfilegt fyrir ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} þarf að vera tala.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' or ')} is required.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} er skilyrt.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} byrjar ekki á ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Vinsamlegast sláðu inn gilda slóð.`;
        /* </i18n> */
    },
};

var is = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$q,
    validation: validation$q
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$p = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Inserisci',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Rimuovi',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Rimuovi tutti',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Ci dispiace, non tutti i campi sono compilati correttamente.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Invia',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nessun file selezionato',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Sposta in alto',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Sposta giù',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Caricamento...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Carica altro',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Prossimo',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Precedente',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Aggiungi tutti i valori',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Aggiungi valori selezionati',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Rimuovi tutti i valori',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Rimuovi i valori selezionati',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Scegli la data',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Cambia data',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'La data selezionata non è valida.',
    /**
     * Shown when there is something to close
     */
    close: 'Chiudi',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$p = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Si prega di accettare ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `la data ${sentence(name)} deve essere successiva ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `la data ${sentence(name)} deve essere nel futuro.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} può contenere solo caratteri alfanumerici.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} può contenere solo lettere e numeri.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} può contenere solo lettere e spazi.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} deve contenere caratteri alfabetici.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} deve contenere lettere o numeri.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} deve contenere lettere o spazi.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} deve contenere un simbolo.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} must contain uppercase.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} deve contenere lettere minuscole.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} deve contenere numeri.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} deve essere un simbolo.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} può contenere solo lettere maiuscole.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} può contenere solo lettere minuscole.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `la data ${sentence(name)} deve essere antecedente ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} deve essere nel passato.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Questo campo è stato configurato male e non può essere inviato.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} deve essere tra ${a} e ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} non corrisponde.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} non è una data valida, per favore usa il formato ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Questo campo è stato configurato in modo errato e non può essere inviato.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} deve essere tra ${date(args[0])} e ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Per favore inserire un indirizzo email valido.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} non termina con ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} non è un valore consentito.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} deve contenere almeno un carattere.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} deve essere minore o uguale a ${max} caratteri.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} deve contenere ${max} caratteri.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} deve essere maggiore o uguale a ${min} caratteri.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} deve essere tra ${min} e ${max} caratteri.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} non è un valore consentito.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Non può avere più di ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} deve essere minore o uguale a ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Formato file non consentito.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} deve essere di tipo: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Non può avere meno di ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} deve essere almeno ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" non è un ${name} consentito.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} deve essere un numero.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' o ')} è richiesto.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} è richiesto.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} non inizia con ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Inserisci un URL valido.`;
        /* </i18n> */
    },
};

var it = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$p,
    validation: validation$p
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$o = {
    /**
     * Shown on buttons for adding new items.
     */
    add: '追加',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: '削除',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: '全て削除',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: '項目が正しく入力されていません。',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: '送信',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'ファイルが選択されていません',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: '上に移動',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: '下へ移動',
    /**
     * Shown when something is actively loading.
     */
    isLoading: '読み込み中...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'さらに読み込む',
    /**
     * Shown on buttons that navigate state forward
     */
    next: '[次へ]',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: '前へ',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'すべての値を追加',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: '選択した値を追加',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'すべての値を削除',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: '選択した値を削除',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: '日付を選択',
    /**
     * Shown when there is a date to change.
     */
    changeDate: '日付を変更',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: '選択した日付は無効です。',
    /**
     * Shown when there is something to close
     */
    close: '閉じる',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$o = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `${name}の同意が必要です。`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)}は${date(args[0])}より後の日付である必要があります。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)}は将来の日付でなければなりません。`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)}には英字のみを含めることができます。`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)}には、文字と数字のみを含めることができます。`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)}には、文字とスペースのみを含めることができます。`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} にはアルファベット文字が含まれている必要があります。`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} には文字または数字を含める必要があります。`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} には文字またはスペースを含める必要があります。`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} には記号が含まれている必要があります。`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} には大文字を含める必要があります。`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} には小文字を含める必要があります。`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} には数字が含まれている必要があります。`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} はシンボルでなければなりません。`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} には大文字しか使用できません`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} には小文字しか使用できません。`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)}は${date(args[0])}より前の日付である必要があります。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)}は過去の日付である必要があります。`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `このフィールドは正しく構成されていないため、送信できません。`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)}は${a}と${b}の間にある必要があります。`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)}が一致しません。`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)}は有効な日付ではありません。${args[0]}の形式を使用してください。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'このフィールドは正しく構成されておらず、送信できません。';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)}は${date(args[0])}と${date(args[1])}の間にある必要があります。`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: '有効なメールアドレスを入力してください。',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)}は${list(args)}で終わっていません。`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)}は許可された値ではありません。`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)}は少なくとも1文字である必要があります。`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)}は${max}文字以下である必要があります。`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} の長さは ${max} 文字でなければなりません。`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)}は${min}文字以上である必要があります。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)}は${min}から${max}文字の間でなければなりません。`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)}は許可された値ではありません。`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${name}は${args[0]}を超えることはできません。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)}は${args[0]}以下である必要があります。`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'ファイル形式は許可されていません。';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)}は${args[0]}である必要があります。`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${name}は${args[0]}未満にすることはできません。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)}は少なくとも${args[0]}である必要があります。`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}”は許可された${name}ではありません。`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)}は数値でなければなりません。`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join('または')}${labels}が必要です。`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)}は必須です。`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)}は${list(args)}で始まっていません。`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `有効な URL を入力してください。`;
        /* </i18n> */
    },
};

var ja = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$o,
    validation: validation$o
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$n = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'қосу',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Жою',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Барлығын жою',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Кешіріңіз, барлық өрістер дұрыс толтырылмаған.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Жіберу',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Ешбір файл таңдалмады',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Жоғары жылжу',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Төмен жылжытыңыз',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Жүктеу...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Көбірек жүктеңіз',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Келесі',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Алдыңғы',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Барлық мәндерді қосыңыз',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Таңдалған мәндерді қосыңыз',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Барлық мәндерді алып тастаңыз',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Таңдалған мәндерді алып тастаңыз',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Күнді таңдаңыз',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Өзгерту күні',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Таңдалған күн жарамсыз.',
    /**
     * Shown when there is something to close
     */
    close: 'Жабу',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$n = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `қабылдаңыз ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} кейін болуы керек ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} болашақта болуы керек.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} тек алфавиттік таңбаларды қамтуы мүмкін.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} тек әріптер мен сандардан тұруы мүмкін.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} тек әріптер мен бос орындар болуы мүмкін.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} алфавиттік таңбалардан тұруы керек.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} құрамында әріптер немесе сандар болуы керек.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} құрамында әріптер немесе бос орындар болуы керек.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} символы болуы керек.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} құрамында бас әріптер болуы керек.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} кіші әріп болуы керек.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} сандардан тұруы керек.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} символы болуы керек.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} тек бас әріптерден тұруы мүмкін.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} құрамында тек кіші әріптер болуы мүмкін.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} бұрын болуы керек ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} өткенде болуы керек.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Бұл өріс қате конфигурацияланған және оны жіберу мүмкін емес.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} арасында болуы керек ${a} және ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} сәйкес келмейді.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} жарамды күн емес, пішімді пайдаланыңыз ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Бұл өріс қате конфигурацияланған және оны жіберу мүмкін емес';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} арасында болуы керек ${date(args[0])} және ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Өтінеміз қолданыстағы электронды пошта адресін енгізіңіз.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} -мен бітпейді ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} рұқсат етілген мән емес.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} кем дегенде бір таңба болуы керек.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} кем немесе тең болуы керек ${max} кейіпкерлер.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} ${max} таңбалары болуы керек.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} артық немесе тең болуы керек ${min} кейіпкерлер.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} арасында болуы керек ${min} және ${max} кейіпкерлер.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} рұқсат етілген мән емес.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `артық болуы мүмкін емес ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} кем немесе тең болуы керек ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Файл пішімдері рұқсат етілмейді.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} типте болуы керек: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `кем болуы мүмкін емес ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} кем дегенде болуы керек ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” рұқсат етілмейді ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} сан болуы керек.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' не ')} қажет.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} талап етіледі.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} -ден басталмайды ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Жарамды URL мекенжайын енгізіңіз.`;
        /* </i18n> */
    },
};

var kk = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$n,
    validation: validation$n
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$m = {
    /**
     * Shown on buttons for adding new items.
     */
    add: '추가',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: '제거',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: '모두 제거',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: '모든 값을 채워주세요',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: '제출하기',
    /**
     * Shown when no files are selected.
     */
    noFiles: '선택된 파일이 없습니다',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: '위로 이동',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: '아래로 이동',
    /**
     * Shown when something is actively loading.
     */
    isLoading: '로드 중...',
    /**
     * Shown when there is more to load.
     */
    loadMore: '더 불러오기',
    /**
     * Shown on buttons that navigate state forward
     */
    next: '다음',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: '이전',
    /**
     * Shown when adding all values.
     */
    addAllValues: '모든 값 추가',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: '선택한 값 추가',
    /**
     * Shown when removing all values.
     */
    removeAllValues: '모든 값 제거',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: '선택한 값 제거',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: '날짜 선택',
    /**
     * Shown when there is a date to change.
     */
    changeDate: '날짜 변경',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: '선택한 날짜가 잘못되었습니다.',
    /**
     * Shown when there is something to close
     */
    close: '닫기',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$m = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `${name} 올바른 값을 선택 해주세요`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])} 이후여야 합니다`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} 미래의 날짜여야합니다`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} 알파벳 문자만 포함할 수 있습니다`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} 문자와 숫자만 포함될 수 있습니다`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} 문자와 공백만 포함할 수 있습니다.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} 에는 알파벳 문자가 포함되어야 합니다.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} 에는 문자나 숫자가 포함되어야 합니다.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} 에는 문자나 공백이 포함되어야 합니다.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} 에는 기호를 포함해야 합니다.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} 는 대문자를 포함해야 합니다.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} 는 소문자를 포함해야 합니다.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} 에는 숫자가 포함되어야 합니다.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} 는 기호여야 합니다.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} 는 대문자만 포함할 수 있습니다.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} 는 소문자만 포함할 수 있습니다.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])} 이전여야 합니다`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} 과거의 날짜여야합니다`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `잘못된 구성으로 제출할 수 없습니다`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} ${a}와 ${b} 사이여야 합니다`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} 일치하지 않습니다`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} 유효한 날짜가 아닙니다. ${args[0]}과 같은 형식을 사용해주세요`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return '잘못된 구성으로 제출할 수 없습니다';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} ${date(args[0])}에서 ${date(args[1])} 사이여야 합니다`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: '올바른 이메일 주소를 입력해주세요',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)}로 끝나지 않습니다`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} 허용되는 값이 아닙니다`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} 하나 이상의 문자여야 합니다`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} ${max}자 이하여야 합니다`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} 는 ${max} 자 길이여야 합니다.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} ${min} 문자보다 크거나 같아야 합니다`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} ${min}에서 ${max}자 사이여야 합니다`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} 허용되는 값이 아닙니다`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${args[0]} ${name} 초과할 수 없습니다`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} ${args[0]}보다 작거나 같아야 합니다`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return '파일 형식이 허용되지 않습니다';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} ${args[0]} 유형이어야 합니다`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${args[0]} ${name}보다 작을 수 없습니다`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} ${args[0]} 이상이어야 합니다`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `${value}" 허용되지 않는 ${name}입니다`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} 숫자여야 합니다`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' 또는 ')}가 필요합니다.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} 필수 값입니다`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)}로 시작하지 않습니다`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `유효한 URL을 입력하십시오.`;
        /* </i18n> */
    },
};

var ko = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$m,
    validation: validation$m
});

/**
 * By the given quantity it returns singular or plural with the correct suffix in Lithuanian
 */
function getByQuantity(quantity, vienetas, vienetai, vienetu) {
    const lastTwoDigits = quantity.toString().slice(-2);
    const parsedQuantity = parseInt(lastTwoDigits);
    if ((parsedQuantity > 10 && parsedQuantity < 20) ||
        parsedQuantity % 10 === 0) {
        return vienetu;
    }
    if (parsedQuantity === 1 || parsedQuantity % 10 === 1) {
        return vienetas;
    }
    return vienetai;
}
/**
 * Standard language for interface features.
 * @public
 */
const ui$l = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Pridėti',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Pašalinti',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Pašalinti visus',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Atsiprašome, ne visi laukai užpildyti teisingai.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Pateikti',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nepasirinktas joks failas',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Aukštyn',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Žemyn',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Kraunama...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Daugiau',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Kitas',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Ankstesnis',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Pridėti visas reikšmes',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Pridėti pasirinktas reikšmes',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Pašalinti visas reikšmes',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Pašalinti pasirinktas reikšmes',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Pasirinkti datą',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Pakeisti datą',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Pasirinkta data yra netinkama.',
    /**
     * Shown when there is something to close
     */
    close: 'Uždaryti',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$l = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Prašome priimti ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} turi būti po ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} turi būti ateityje.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} gali būti tik abėcėlės simboliai.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} gali būti tik raidės ir skaičiai.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} gali būti tik raidės ir tarpai.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} turi būti abėcėlės simbolių.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} turi būti raidžių arba skaičių.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} turi būti raidžių arba tarpų.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} turi būti simbolių.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} turi būti didžioji raidė.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} turi būti mažoji raidė.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} turi būti skaičių.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} turi būti simbolis.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} turi būti tik didžiosios raidės.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} turi būti tik mažosios raidės.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} turi būti prieš ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} turi būti praeityje.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Šis laukas buvo sukonfigūruotas neteisingai ir jo negalima pateikti.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} turi būti tarp ${a} ir ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} nesutampa.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} nėra tinkama data, naudokite formatą ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Šis laukas buvo sukonfigūruotas neteisingai ir jo negalima pateikti';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} turi būti tarp ${date(args[0])} ir ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Įveskite teisingą el. pašto adresą.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} nesibaigia su ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} nėra leistina reikšmė.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} turi būti bent vienas simbolis.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} turi būti mažiau arba lygiai ${max} ${getByQuantity(max, 'simbolis', 'simboliai', 'simbolių')}.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} turi būti iš ${max} ${getByQuantity(max, 'simbolio', 'simbolių', 'simbolių')}.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} turi būti daugiau arba lygiai ${min} ${getByQuantity(min, 'simbolis', 'simboliai', 'simbolių')}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} turi būti tarp ${min} ir ${max} simbolių.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nėra leistina reikšmė.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Negali turėti daugiau nei ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} turi būti mažiau arba lygiai ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Neleidžiami jokie failų formatai.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} turi būti tokio tipo: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Negali turėti mažiau nei ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `Turi būti bent ${args[0]} ${name} .`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” nėra leidžiamas ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} turi būti skaičius.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' arba ')} yra privaloma.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} yra privaloma.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} neprasideda su ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Įveskite teisingą URL.`;
        /* </i18n> */
    },
};

var lt = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$l,
    validation: validation$l
});

/**
Here we can import additional helper functions to assist in formatting our
language. Feel free to add additional helper methods to libs/formats if it
assists in creating good validation messages for your locale.
*/
/**
 * Standard language for interface features.
 * @public
 */
const ui$k = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Pievienot',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Noņemt',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Noņemt visus',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Piedodiet, visi lauki nav aizpildīti.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Iesniegt',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nav izvēlēts fails',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Uz augšu',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Uz leju',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Ielādējas...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Ielādēt vēl',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Tālāk',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Atpakaļ',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Pievienot visas vērtības',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Pievienojiet izvēlēto vērtību',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Noņemt visas vērtības',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Noņemt izvēlētās vērtības',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Izvēlieties datumu',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Mainīt datumu',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Izvēlētais datums ir nepareizs.',
    /**
     * Shown when there is something to close
     */
    close: 'Aizvērt',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$k = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Lūdzu apstipriniet ${name}`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${name} jābūt pēc ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${name} jābūt pēc šodienas datuma.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${name} var saturēt tikai alfabētiskās rakstzīmes.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${name} var saturēt tikai burtus un ciparus.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${name} var saturēt tikai burtus un atstarpes.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${name} jābūt pirms ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${name} jābūt pirms šodienas datuma.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Šis lauks tika aizpildīts nepareizi un nevar tikt iesniegts.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${name} jābūt starp ${a} un ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${name} nesakrīt.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${name} nav pareizs datums, lūdzu lietojiet formātu ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Šis lauks tika aizpildīts nepareizi un nevar tikt iesniegts.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${name} jābūt starp ${date(args[0])} un ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Lūdzu ierakstiet pareizu e-pasta adresi.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${name} nebeidzas ar ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${name} nav atļauta vērtība.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${name} jābūt vismaz vienam simbolam.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${name} jābūt mazāk par vai ${max} simboliem.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${name} jābūt ${max} simbolu garumā.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${name} jābūt vismaz ${min} simboliem.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${name} jābūt starp ${min} un ${max} simboliem.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${name} nav atļauta vērtība.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Nevar būt vairāk par ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${name} nevar būt mazāk par ${args[0]} vai ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Atļauti nenoteikti faila formāti.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} faila formāti var būt šādi: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Nevar būt mazāk par ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `Jābūt vismaz ${args[0]} ${name}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” nav atļauta vērtība iekš ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${name} jābūt ciparam.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${name} ir obligāti jāaizpilda`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${name} nesākas ar ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Lūdzu pievienojiet pareizu URL.`;
        /* </i18n> */
    },
};

var lv = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$k,
    validation: validation$k
});

/**
 * Here we can import additional helper functions to assist in formatting our
 * language. Feel free to add additional helper methods to libs/formats if it
 * assists in creating good validation messages for your locale.
 */
/**
 * Standard language for interface features.
 * @public
 */
const ui$j = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Нэмэх',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Хасах',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Бүгдийг хасах',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Уучлраарай, зарим нүдэн дахь өгөгдөл дутуу байна.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Илгээх',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Файл сонгоогүй байна',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Дээшээ',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Доошоо',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Ачааллаж байна...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Нэмж ачааллах',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Дараагийн',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Өмнөх',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Бүх утгуудыг нэмэх',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Сонгогдсон утгуудыг нэмэх',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Бүх утгуудыг устгах',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Сонгогдсон утгуудыг хасах',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Огноо сонгох',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Огноо өөрчлөх',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Сонгосон огноо буруу байна.',
    /**
     * Shown when there is something to close
     */
    close: 'Хаах',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$j = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `${name} утгыг зөвшөөрнө үү.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} нь ${date(args[0])}-ны дараа орох ёстой.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} утга ирээдүйг заах ёстой.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} зөвхөн үсэг агуулах ёстой.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} зөвхөн үсэг болон тоог агуулах ёстой.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} зөвхөн үсэг болон зай агуулах ёстой.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} нь ${date(args[0])}-ны өмнө байх ёстой.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} өнгөрсөн огноо байх ёстой.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Энэ нүдэн дэхь өгөгдөл буруу учраас илгээх боломжгүй.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} нь заавал ${a}, ${b} хоёрын дунд байх ёстой.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} таарахгүй байна.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} нь хүчинтэй огноо биш тул ${args[0]} гэсэн огноог ашиглаарай.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Энэхүү нүд буруу тул цааш илгээх боломжгүй.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} нь заавал ${date(args[0])}, ${date(args[1])} хоёр огноон дунд байх ёстой.`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Та хүчинтэй имейл хаягаа оруулна уу.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} нь ${list(args)} гэсэн утгаар төгсөөгүй байна.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} нь зөвшөөрөгдөх утга биш.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} дээр хаяж нэг үсэг байх ёстой`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)}-н урт нь ${max}-ээс ихгүй байх ёстой.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} нь ${max} урт байвал зүгээр.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)}-н урт нь ${min}-ээс их буюу тэнцүү байж болно.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)}-н урт нь ${min}, ${max} хоёрын дунд байх ёстой.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} нь зөвшөөрөгдөх утга биш.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${name} нь ${args[0]}-аас их байж болохгүй.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} нь ${args[0]}-тай тэнцүү эсвэл бага байх ёстой.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Файлын формат буруу.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} төрөл нь ${args[0]} байх ёстой.`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${name} нь ${args[0]}-аас их байж болохгүй.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${name} нь дор хаяж ${args[0]}-тай тэнцүү байх ёстой.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” бол зөвшөөрөгдөх ${name} гэсэн утга биш.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} зөвхөн тоо байх ёстой.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} байх ёстой.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} нь ${list(args)}-ээр эхлээгүй байна.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Хүчннтай URL оруулна уу.`;
        /* </i18n> */
    },
};

var mn = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$j,
    validation: validation$j
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$i = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Legg til',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Fjern',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Fjern alle',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Beklager, noen felter er ikke fylt ut korrekt.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Send inn',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Ingen fil valgt',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Flytt opp',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Flytt ned',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Laster...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Last mer',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Neste',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Forrige',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Legg til alle verdier',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Legg til valgte verdier',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Fjern alle verdier',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Fjern valgte verdier',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Velg dato',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Endre dato',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Den valgte datoen er ugyldig.',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$i = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Vennligst aksepter ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} må være senere enn ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} må være i fremtiden.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} kan bare inneholde alfabetiske tegn.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} kan bare inneholde bokstaver og tall.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} kan bare inneholde bokstaver og mellomrom.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} must contain alphabetical characters.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} must contain letters and numbers.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} must contain letters and spaces.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} must contain symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} must contain uppercase.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} must contain lowercase.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} must contain number.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} can only contain symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} can only contain uppercase.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} can only contain lowercase.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} må være tidligere enn ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} må være i fortiden.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Dette feltet er feilkonfigurert og kan ikke innsendes.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} må være mellom ${a} og ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} stemmer ikke overens.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} er ikke en gyldig dato, vennligst bruk formatet ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Dette feltet er feilkonfigurert og kan ikke innsendes.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} må være mellom ${date(args[0])} og ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Vennligst oppgi en gyldig epostadresse.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} slutter ikke med ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} er ikke en tillatt verdi.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} må ha minst ett tegn.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} må ha mindre enn eller nøyaktig ${max} tegn.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} skal være ${max} tegn langt.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} må ha mer enn eller nøyaktig ${min} tegn.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} må ha mellom ${min} og ${max} tegn.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} er ikke en tillatt verdi.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Kan ikke ha mer enn ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} må være mindre enn eller nøyaktig ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Ingen tillatte filformater.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} må være av typen: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Kan ikke ha mindre enn ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} må være minst ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” er ikke en tillatt ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} må være et tall.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames.map(name => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        }).filter(name => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' eller ')} er nødvendig.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} er påkrevd.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} starter ikke med ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Vennligst inkluder en gyldig url.`;
        /* </i18n> */
    },
};

var nb = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$i,
    validation: validation$i
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$h = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Toevoegen',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Verwijderen',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Alles verwijderen',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Sorry, niet alle velden zijn correct ingevuld.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Versturen',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Geen bestand gekozen',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Naar boven gaan',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Naar beneden verplaatsen',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Aan het laden...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Meer laden',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Vervolgens',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Voorgaand',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Alle waarden toevoegen',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Geselecteerde waarden toevoegen',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Alle waarden verwijderen',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Geselecteerde waarden verwijderen',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Kies een datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Datum wijzigen',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'De geselecteerde datum is ongeldig.',
    /**
     * Shown when there is something to close
     */
    close: 'Sluiten',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$h = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Accepteer de ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} moet na ${date(args[0])} zijn.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} moet in de toekomst liggen.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} mag alleen alfabetische tekens bevatten.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} mag alleen letters en cijfers bevatten.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} mag alleen letters en spaties bevatten.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} moet alfabetische tekens bevatten.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} moet letters of cijfers bevatten.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} moet letters of spaties bevatten.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} moet een symbool bevatten.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} moet hoofdletters bevatten.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} moet kleine letters bevatten.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} moet cijfers bevatten.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} moet een symbool zijn.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} mag alleen hoofdletters bevatten.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} mag alleen kleine letters bevatten.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} moet vóór ${date(args[0])} vallen.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} moet in het verleden liggen.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Dit veld is onjuist geconfigureerd en kan niet worden verzonden.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} moet tussen ${a} en ${b} liggen.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} komt niet overeen.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} is geen geldige datum, gebruik de notatie ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Dit veld is onjuist geconfigureerd en kan niet worden verzonden';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} moet tussen ${date(args[0])} en ${date(args[1])} liggen`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Vul een geldig e-mailadres in.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} eindigt niet met ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} is geen toegestane waarde.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} moet minimaal één teken zijn.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} mag maximaal ${max} tekens lang zijn.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} moet ${max} tekens lang zijn.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} moet minimaal ${min} tekens lang zijn.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} moet tussen de ${min} en ${max} tekens zijn.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} is geen toegestane waarde.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Mag niet meer dan ${args[0]} ${name} hebben.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} moet kleiner zijn dan of gelijk zijn aan ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Geen bestandsformaten toegestaan.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} moet van het type: ${args[0]} zijn`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Mag niet minder dan ${args[0]} ${name} hebben.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} moet minimaal ${args[0]} zijn.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" is geen toegestane ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} moet een getal zijn.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' of ')} is vereist.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} is verplicht.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} begint niet met ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Voer een geldige URL in.`;
        /* </i18n> */
    },
};

var nl = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$h,
    validation: validation$h
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$g = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Dodaj',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Usuń',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Usuń wszystko',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Nie wszystkie pola zostały wypełnione poprawnie.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Wyślij',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nie wybrano pliku',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Przesuń w górę',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Przesuń w dół',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Ładowanie...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Załaduj więcej',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Kolejny',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Poprzednia',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Dodaj wszystkie wartości',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Dodaj wybrane wartości',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Usuń wszystkie wartości',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Usuń wybrane wartości',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Wybierz datę',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Zmień datę',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Wybrana data jest nieprawidłowa.',
    /**
     * Shown when there is something to close
     */
    close: 'Zamknij',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$g = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Proszę zaakceptować ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} musi być po ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} musi być w przyszłości.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `Pole ${sentence(name)} może zawierać tylko znaki alfabetyczne.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `Pole ${sentence(name)} może zawierać tylko znaki alfanumeryczne.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `Pole ${sentence(name)} mogą zawierać tylko litery i spacje.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} musi zawierać znaki alfabetyczne.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} musi zawierać litery lub cyfry.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} musi zawierać litery lub spacje.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} musi zawierać symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} musi zawierać wielkie litery.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} musi zawierać małe litery.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} musi zawierać liczby.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} musi być symbolem.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} może zawierać tylko wielkie litery.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} może zawierać tylko małe litery.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} musi być przed ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} musi być w przeszłości.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Pole zostało wypełnione niepoprawnie i nie może zostać wysłane.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `Wartość pola ${sentence(name)} musi być pomiędzy ${a} i ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} nie pokrywa się.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `Wartość pola ${sentence(name)} nie jest poprawną datą, proszę użyć formatu ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'To pole zostało wypełnione niepoprawnie i nie może zostać wysłane';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `Data w polu ${sentence(name)} musi być pomiędzy ${date(args[0])} i ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Proszę wpisać poprawny adres email.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `Pole ${sentence(name)} nie kończy się na ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `Pole ${sentence(name)} nie jest dozwoloną wartością.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `Pole ${sentence(name)} musi posiadać minimum jeden znak.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `Pole ${sentence(name)} musi zawierać ${max} lub mniej znaków.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `Pole ${sentence(name)} musi zawierać ${min} lub więcej znaków.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `Pole ${sentence(name)} musi mieć ${min} znaków.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `Pole ${sentence(name)} musi mieć ${min}-${max} znaków.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `Pole ${sentence(name)} zawiera niedozwolone znaki.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Nie można mieć więcej niż ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `Wartość pola ${sentence(name)} musi być mniejsza lub równa ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Nie podano dozwolonych typów plików.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} musi być typem: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Musisz podać więcej niż ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return ` Musisz podać conajmniej ${args[0]} ${sentence(name)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `Wartość pola ${name} jest niedozwolona.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} musi być numerem.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' lub ')} wymagany jest.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `Pole ${sentence(name)} jest wymagane.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `Wartośc pola ${sentence(name)} nie zaczyna się od ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Wprowadź prawidłowy adres URL.`;
        /* </i18n> */
    },
};

var pl = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$g,
    validation: validation$g
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$f = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Incluir',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Remover',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Remover todos',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Desculpe, alguns campos não foram preenchidos corretamente.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Enviar',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nenhum arquivo selecionado.',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Mover para cima',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Mover para baixo',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Carregando...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Carregar mais',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Próximo',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Anterior',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Adicione todos os valores',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Adicionar valores selecionados',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Remover todos os valores',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Remover valores selecionados',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Escolha a data',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Data da alteração',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'A data selecionada é inválida.',
    /**
     * Shown when there is something to close
     */
    close: 'Fechar',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$f = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Por favor aceite o ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} precisa ser depois de ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} precisa ser no futuro.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} precisa conter apenas letras.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} pode conter apenas letras e números.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} pode conter apenas números e espaços.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} deve conter caracteres alfabéticos.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} deve conter letras ou números.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} deve conter letras ou espaços.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} deve conter um símbolo.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} deve conter letras maiúsculas.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} deve conter letras minúsculas.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} deve conter números.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} deve ser um símbolo.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} só pode conter letras maiúsculas.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} só pode conter letras minúsculas.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} precisa ser antes de ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} precisa ser no passado.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Este campo não foi configurado corretamente e não pode ser submetido.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} precisa ser entre ${a} e ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} não é igual.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} não é uma data válida, por favor use este formato ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Este campo não foi configurado corretamente e não pode ser submetido.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} precisa ser entre ${date(args[0])} e ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Por favor, insira um endereço de email válido.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} não termina com ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} não é um valor permitido.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = first <= second ? first : second;
        const max = second >= first ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} precisa conter ao menos um caractere.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} precisa ser menor ou igual a ${max} caracteres.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} precisa conter ${max} caracteres.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} precisa ser maior ou igual a ${min} caracteres.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} precisa ter entre ${min} e ${max} caracteres.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} não é um valor permitido.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Não pode ter mais de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} precisa ser menor ou igual a ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Não há formatos de arquivos permitidos.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} precisa ser do tipo: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Não pode ter menos de ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} precisa ser pelo menos ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” não é um(a) ${name} permitido(a).`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} precisa ser um número.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' ou ')} é necessário.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} é obrigatório.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} não começa com ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Por favor, insira uma url válida.`;
        /* </i18n> */
    },
};

var pt = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$f,
    validation: validation$f
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$e = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Adăugare',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Elimină',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Elimină tot',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Pare rău, unele câmpuri nu sunt corect completate.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Trimite',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nu este selectat nici un fișier',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Mutare în sus',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Mutare în jos',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Se încarcă...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Încărcați mai mult',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Urmatorul',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Precedent',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Adăugați toate valorile',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Adăugarea valorilor selectate',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Eliminați toate valorile',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Eliminați valorile selectate',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Alege data',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Modificați data',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Data selectată este nevalidă.',
    /**
     * Shown when there is something to close
     */
    close: 'Închide',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$e = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Te rog acceptă ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} trebuie să fie după ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} trebuie sa fie în viitor.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} poate conține doar caractere alafetice.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} poate conține doar litere și numere.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} poate conține doar litere și spații.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} trebuie să conțină caractere alfabetice.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} trebuie să conțină litere sau numere.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} trebuie să conțină litere sau spații.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} trebuie să conțină simbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} trebuie să conțină majuscule.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} trebuie să conțină litere mici.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} trebuie să conțină numere.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} trebuie să fie un simbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} poate conține doar litere mari.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} poate conține doar litere mici.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} trebuie să preceadă ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} trebuie să fie în trecut.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Câmpul a fost configurat incorect și nu poate fi trimis.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} trebuie să fie între ${a} și ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} nu se potrivește.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} nu este validă, te rog foloște formatul ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Câmpul a fost incorect configurat și nu poate fi trimis.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} trebuie să fie între ${date(args[0])} și ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Te rog folosește o adresă de email validă.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} nu se termină cu ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} nu este o valoare acceptată.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} trebuie sa conțină cel puțin un caracter.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} trebuie sa aibă cel mult ${max} caractere.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} ar trebui să aibă ${max} caractere lungi.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} trebuie să aibă cel puțin ${min} caractere.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} trebuie să aibă între ${min} și ${max} caractere.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nu este o valoare acceptată.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Nu poate avea mai mult decat ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} trebuie să fie cel mult egal cu ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Tipul de fișier neacceptat.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} trebuie să fie de tipul: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Nu poate avea mai puțin decât ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} trebuie să fie cel puțin ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” nu este o valoare acceptă pentru ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} trebuie să fie un număr.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' sau ')} este necesar.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} este necesar.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} nu începe cu ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Introduceți o adresă URL validă.`;
        /* </i18n> */
    },
};

var ro = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$e,
    validation: validation$e
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$d = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Добавить',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Удалить',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Убрать все',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Извините, не все поля заполнены верно.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Отправить',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Файл не выбран',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Переместить вверх',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Переместить вниз',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Загрузка...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Загрузить больше',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Следующий',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Предыдущий',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Добавить все значения',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Добавить выбранные значения',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Удалить все значения',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Удалить выбранные значения',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Выберите дату',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Изменить дату',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Выбранная дата недействительна.',
    /**
     * Shown when there is something to close
     */
    close: 'Закрыть',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$d = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Пожалуйста, примите ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `Дата ${sentence(name)} должна быть позже ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `Дата ${sentence(name)} должна быть в будущем.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `Поле ${sentence(name)} может содержать только буквы.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `Поле ${sentence(name)} может содержать только буквы и цифры.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} могут содержать только буквы и пробелы.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} должен содержать алфавитные символы.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} должен содержать буквы или цифры.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} должно содержать буквы или пробелы.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} должен содержать символ.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} должно содержать прописные буквы.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} должно содержать строчные буквы.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} должен содержать числа.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} должен быть символом.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} может содержать только прописные буквы.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} может содержать только буквы нижнего регистра.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `Дата ${sentence(name)} должна быть раньше ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `Дата ${sentence(name)} должна быть в прошлом.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Это поле заполнено неверно и не может быть отправлено.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `Поле ${sentence(name)} должно быть между ${a} и ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `Поле ${sentence(name)} не совпадает.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `Поле ${sentence(name)} имеет неверную дату. Пожалуйста, используйте формат ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Это поле заполнено неверно и не может быть отправлено.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `Дата ${sentence(name)} должна быть между ${date(args[0])} и ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Пожалуйста, введите действительный электронный адрес.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `Поле ${sentence(name)} не должно заканчиваться на ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `Поле ${sentence(name)} имеет неподустимое значение.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `Поле ${sentence(name)} должно содержать минимум один символ.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `Длина поля ${sentence(name)} должна быть меньше или равна ${max} символам.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `Длина ${sentence(name)} должна составлять ${max} символов.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `Длина поля ${sentence(name)} должна быть больше или равна ${min} символам.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `Длина поля ${sentence(name)} должна быть между ${min} и ${max} символами.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `Поле ${sentence(name)} имеет недопустимое значение.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Не может быть выбрано больше, чем ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `Поле ${sentence(name)} должно быть меньше или равно ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Не указаны поддержиавемые форматы файла.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `Формат файла в поле ${sentence(name)} должен быть: ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Не может быть выбрано меньше, чем ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `Поле ${sentence(name)} должно быть не менее, чем ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” не поддерживается в поле ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `Поле ${sentence(name)} должно быть числом.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' или ')} требуется.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `Поле ${sentence(name)} обязательно для заполнения.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `Поле ${sentence(name)} должно начинаться с ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Пожалуйста, введите действительный URL-адрес.`;
        /* </i18n> */
    },
};

var ru = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$d,
    validation: validation$d
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$c = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Pridať',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Odstrániť',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Odstrániť všetko',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Prepáčte, ale nie všetky polia sú vyplnené správne.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Odoslať',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nebol vybraný žiadny súbor',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Posunúť hore',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Posunúť dole',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Načítavanie...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Načítať viac',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Ďalšie',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Predošlý',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Pridajte všetky hodnoty',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Pridajte vybrané hodnoty',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Odstrániť všetky hodnoty',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Odstrániť vybrané hodnoty',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Vyberte dátum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Zmena dátumu',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Vybraný dátum je neplatný.',
    /**
     * Shown when there is something to close
     */
    close: 'Zavrieť',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$c = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Prosím zaškrtnite ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} musí byť za ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} musí byť v budúcnosti.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} môže obsahovať iba písmená.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} môže obsahovať iba písmená a čísla.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} môže obsahovať iba písmená a medzery.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} musí obsahovať abecedné znaky.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} musí obsahovať písmená alebo číslice.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} musí obsahovať písmená alebo medzery.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} musí obsahovať symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} musí obsahovať veľké písmená.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} musí obsahovať malé písmená.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} musí obsahovať čísla.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} musí byť symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} môže obsahovať iba veľké písmená.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} môže obsahovať len malé písmená.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} musí byť pred ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} musí byť v minulosti.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Toto pole bolo nesprávne nakonfigurované a nemôže byť odoslané.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} musí byť medzi ${a} and ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} does not match.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} nie je platným dátumom, prosím, použite formát ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Toto pole bolo nesprávne nakonfigurované a nemôže byť odoslané.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} musí byť medzi ${date(args[0])} a ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Prosím, zadajte platnú emailovú adresu.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} nekončí na ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} nie je povolená hodnota.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} musí mať najmenej jeden znak.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} musí byť menšie alebo rovné ako ${max} znakov.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} by mala mať dĺžku ${max} znakov.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} musí byť väčšie alebo rovné ako ${min} znakov.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} musí byť medzi ${min} až ${max} znakov.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nie je povolená hodnota.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Nie je možné použiť viac než ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} musí byť menšie alebo rovné ako ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Nie sú povolené formáty súborov.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} musí byť typu: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Nemôže byť menej než ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} musí byť minimálne ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” nie je povolené hodnota pre ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} musí byť číslo.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' alebo ')} je potrebný.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} je povinné.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} nezačíná s ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Zadajte platnú adresu URL.`;
        /* </i18n> */
    },
};

var sk = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$c,
    validation: validation$c
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$b = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Dodaj',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Odstrani',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Odstrani vse',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Nekatera polja niso pravilno izpolnjena.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Pošlji',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Nobena datoteka ni izbrana',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Premakni se navzgor',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Premakni se navzdol',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Nalaganje...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Naloži več',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Naslednji',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Prejšnji',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Dodajte vse vrednosti',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Dodajanje izbranih vrednosti',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Odstranite vse vrednosti',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Odstrani izbrane vrednosti',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Izberite datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Spremeni datum',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Izbrani datum je neveljaven.',
    /**
     * Shown when there is something to close
     */
    close: 'Zapri',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$b = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Prosimo popravite ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} mora biti po ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti v prihodnosti.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} lahko vsebuje samo znake abecede.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} lahko vsebuje samo črke in številke.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} lahko vsebuje samo črke in presledke.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} mora vsebovati abecedne znake.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} mora vsebovati črke ali številke.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} mora vsebovati črke ali presledke.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} mora vsebovati simbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} mora vsebovati velike črke.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} mora vsebovati male črke.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} mora vsebovati številke.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} mora biti simbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} lahko vsebuje le velike črke.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} lahko vsebuje le male črke.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} mora biti pred ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti v preteklosti.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `To polje je narobe nastavljeno in ne mora biti izpolnjeno.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} mora biti med ${a} in ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} se ne ujema.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} ni pravilen datum, prosimo uporabite format ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'To polje je narobe nastavljeno in ne mora biti izpolnjeno.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} mora biti med ${date(args[0])} in ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Vnesite veljaven e-poštni naslov.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} se mora kočati z ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} ni dovoljena vrednost.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} mora vsebovati vsaj en znak.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} mora vsebovati največ ${max} znakov.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} mora biti dolg ${max} znakov.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} mora vsebovati vsaj ${min} znakov.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} mora vsebovati med ${min} in ${max} znakov.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} ni dovoljena vrednost.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Največ je ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} je lahko največ ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Nobena vrsta datoteke ni dovoljena.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} mora biti tipa: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Najmanj ${args[0]} ${name} je dovoljenih.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} mora biti vsaj ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” ni dovoljen(a/o) ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} mora biti številka.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' ali ')} zahtevan je.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} je zahtevan(o/a).`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} se mora začeti z ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Vnesite veljaven URL.`;
        /* </i18n> */
    },
};

var sl = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$b,
    validation: validation$b
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$a = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Dodaj',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Ukloni',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Ukloni sve',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Pojedina polja nisu ispravno ispunjena.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Pošalji',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Fajl nije odabran',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Pomerite se gore',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Pomeri se dole',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Učitavanje...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Učitaj više',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Sledeća',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Prethodna',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Dodajte sve vrednosti',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Dodajte izabrane vrednosti',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Uklonite sve vrednosti',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Uklonite izabrane vrednosti',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Izaberite datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Promenite datum',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Izabrani datum je nevažeći.',
    /**
     * Shown when there is something to close
     */
    close: 'Zatvori',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$a = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Molimo prihvatite ${name}`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} mora biti posle ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti u budućnosti.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} može da sadrži samo abecedne znakove.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} može da sadrži samo slova i brojeve.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} može da sadrži samo slova i razmake.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} mora da sadrži abecedne znakove.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} mora da sadrži slova ili brojeve.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} mora da sadrži slova ili razmake.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} mora da sadrži simbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} mora da sadrži velika slova.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} mora da sadrži mala slova.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} mora da sadrži brojeve.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} mora biti simbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} može da sadrži samo velika slova.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} može da sadrži samo mala slova.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} mora biti pre ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} mora biti u prošlosti.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Ovo polje je pogrešno konfigurisano i ne može se poslati.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} mora biti između ${a} i ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} se ne podudara.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} nije važeći datum, molimo Vas koristite format ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Ovo polje je pogrešno konfigurisano i ne može se poslati';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} mora biti između ${date(args[0])} i ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Unesite ispravnu e-mail adresu.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} se ne završava sa ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} nije dozvoljena vrednost`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} mora biti najmanje jedan karakter.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} mora biti manji ili jednaki od ${max} karaktera.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} treba da bude ${max} znakova dugačak.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} mora biti veći ili jednaki od ${min} karaktera.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} mora biti između ${min} i ${max} karaktera.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} nije dozvoljena vrednost.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Ne može imati više od ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} mora biti manji ili jednaki od ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Nisu dozvoljeni formati datoteka.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} mora biti tipa: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Ne može imati manje od ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} mora da ima najmanje ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” nije dozvoljeno ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} mora biti broj.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' или ')} потребан је.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} je obavezno polje.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ne počinje sa ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Unesite važeću URL adresu.`;
        /* </i18n> */
    },
};

var sr = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$a,
    validation: validation$a
});

// Please copy and paste the file your just downloaded here
/**
 * Standard language for interface features.
 * @public
 */
const ui$9 = {
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Ta bort',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Ta bort alla',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Tyvärr är inte alla fält korrekt ifyllda',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Skicka',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Ingen fil vald',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Flytta upp',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Flytta ner',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Laddar...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Ladda mer',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Nästa',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Föregående',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Lägg till alla värden',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Lägg till valda värden',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Ta bort alla värden',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Ta bort valda värden',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Välj datum',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Ändra datum',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Det valda datumet är ogiltigt.',
    /**
     * Shown when there is something to close
     */
    close: 'Stäng',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$9 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Var god acceptera ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} måste vara efter ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} måste vara framåt i tiden.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} kan enbart innehålla bokstäver i alfabetet.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} kan bara innehålla bokstäver och siffror.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} kan bara innehålla bokstäver och blanksteg.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} måste innehålla alfabetiska tecken.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} måste innehålla bokstäver eller siffror.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} måste innehålla bokstäver eller mellanslag.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} måste innehålla symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} måste innehålla versaler.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} måste innehålla gemener.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} måste innehålla siffror.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} måste vara en symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} kan bara innehålla versaler.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} kan bara innehålla små bokstäver.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} måste vara före ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} måste vara bakåt i tiden.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Det här fältet ställdes inte in korrekt och kan inte användas.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} måste vara mellan ${a} och ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} matchar inte.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} är inte ett giltigt datum, var god använd formatet ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Det här fältet ställdes inte in korrekt och kan inte användas';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} måste vara mellan ${date(args[0])} och ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Var god fyll i en giltig e-postadress.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} slutar inte med ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} är inte ett godkänt värde.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} måste ha minst ett tecken.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} måste vara ${max} tecken eller färre.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} bör vara ${max} tecken långa.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} måste vara ${min} tecken eller fler.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} måste vara mellan ${min} och ${max} tecken.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} är inte ett godkänt värde.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Kan inte ha mer än ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} måste vara ${args[0]} eller mindre.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Inga filtyper tillåtna.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} måste vara av filtypen: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Kan inte ha mindre än ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} måste vara minst ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” är inte ett godkänt ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} måste vara en siffra.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' eller ')} krävs.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} är obligatoriskt.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} börjar inte med ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Ange en giltig URL.`;
        /* </i18n> */
    },
};

var sv = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$9,
    validation: validation$9
});

/**
 * Here we can import additional helper functions to assist in formatting our
 * language. Feel free to add additional helper methods to libs/formats if it
 * assists in creating good validation messages for your locale.
 */
/**
 * Standard language for interface features.
 * @public
 */
const ui$8 = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Aumenta',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Hasai',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Hasai Hotu',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Deskulpa, iha informasaun neebe sala iha formuláriu',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Submete',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Seidauk hili file',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Muda ba leten',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Muda ba kotuk',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Hein lai...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Foti tan',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Ba oin',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Ba kotuk',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Aumenta hotu',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Aumenta buat neebe hili ona',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Hasai hotu',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Hasai buat neebe hili ona',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Hili data',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Troka data',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Data la loos.',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$8 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Favor ida simu ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} tenki depoid ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} tenki iha futuru.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} bele uza letra deit.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} bele uza letra ka numeru deit.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} bele uza letra ka numeru deit.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} tenki antes ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} tenki antes ohin loron.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Informasaun nee la loos no la bele submete.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} tenki iha klaran entre ${a} no ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} la hanesan.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} la loos, favor ida hakerek tuir ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Informasaun nee la loos no la bele submete.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} tenki iha ${date(args[0])} no ${date(args[1])} nia klaran`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Favor hakerek endresu email neebe loos.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} la remata ho ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `la bele uza ${sentence(name)}.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} tenki iha letra ida ka liu.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} tenki badak liu ${max} letra.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} tenki iha letra ${max}.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} tenki iha letra ${min} ka liu.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} tenki iha letra ${min} too ${max}.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `la bele uza ${sentence(name)}.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `La bele iha ${args[0]} ka liu ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} tenki kiik liu ka hanesan ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return `La bele simu 'format' ida.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} tenki iha tipo: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Presiza ${args[0]} ${name} ka liu.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${name} tenki ${args[0]} ka liu.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `La bele uza “${value}” ba ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} tenki numeru.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `Presiza ${sentence(name)}.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} la komesa ho ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Favor hakerek URL neebe loos.`;
        /* </i18n> */
    },
};

var tet = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$8,
    validation: validation$8
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$7 = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Илова кардан',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Хориҷ кардан',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Ҳамаро хориҷ кунед',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Бубахшед, на ҳама майдонҳо дуруст пур карда шудаанд.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Пешниҳод кунед',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Ягон файл интихоб нашудааст',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Ба боло ҳаракат кунед',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Ба поён ҳаракат кунед',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Дар ҳоли боргузорӣ',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Бештар бор кунед',
    /**
     * Show on buttons that navigate state forward
     */
    next: 'Баъдӣ',
    /**
     * Show on buttons that navigate state backward
     */
    prev: 'Гузашта',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Ҳама арзишҳоро илова кунед',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Илова кардани арзишҳои интихобшуда',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Ҳама арзишҳоро хориҷ кунед',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Арзишҳои интихобшударо хориҷ кунед',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Сана интихоб кунед',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Тағйир додани сана',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Санаи интихобшуда нодуруст аст.',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$7 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Лутфан ${name}-ро қабул кунед`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} бояд пас аз ${date(args[0])} бошад.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} бояд дар оянда бошад.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} метавонад танҳо аломатҳои алифборо дар бар гирад.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} метавонад танҳо ҳарфҳо ва рақамҳоро дар бар гирад.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} метавонад танҳо ҳарфҳо ва фосилаҳоро дар бар гирад.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} must contain alphabetical characters.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} must contain letters and numbers.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} must contain letters and spaces.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} must contain symbol.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} must contain uppercase.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} must contain lowercase.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} must contain number.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} can only contain symbol.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} can only contain uppercase.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} can only contain lowercase.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} бояд пеш аз ${date(args[0])} бошад.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} бояд дар гузашта бошад.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Ин майдон нодуруст танзим шудааст ва онро пешниҳод кардан ғайриимкон аст.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} бояд дар байни ${a} ва ${b} бошад.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} мувофиқат намекунад.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} санаи дуруст нест, лутфан формати ${args[0]}-ро истифода баред`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Ин майдон нодуруст танзим шудааст ва онро пешниҳод кардан ғайриимкон аст';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} бояд дар байни ${date(args[0])} ва ${date(args[1])} бошад`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Лутфан нишонаи имейли амалкунандаро ворид намоед.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} бо ${list(args)} ба охир намерасад.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} арзиши иҷозатдодашуда нест.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} бояд ҳадди аққал як аломат бошад.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} бояд аз ${max} аломат камтар ё баробар бошад.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} бояд ${max} аломат бошад.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} бояд аз ${min} аломат зиёд ё баробар бошад.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} бояд дар байни ${min} ва ${max} аломат бошад.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} арзиши иҷозатдодашуда нест.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Зиёда аз ${args[0]} ${name} дошта наметавонад.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} бояд аз ${args[0]} камтар ё баробар бошад.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Ягон формати файл иҷозат дода намешавад.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} бояд чунин намуд бошад: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Камтар аз ${args[0]} ${name} дошта наметавонад.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} бояд ҳадди аққал ${args[0]} бошад.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" ${name} иҷозат дода намешавад.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} бояд рақам бошад.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames.map(name => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        }).filter(name => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' ё ')} зарур а`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} лозим аст.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} бо ${list(args)} оғоз намешавад.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Лутфан URL-и дурустро дохил кунед.`;
        /* </i18n> */
    },
};

var tg = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$7,
    validation: validation$7
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$6 = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'เพิ่ม',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'เอาออก',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'เอาออกทั้งหมด',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'ขออภัย ข้อมูลบางช่องที่กรอกไม่ถูกต้อง',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'ส่ง',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'ยังไม่ได้เลือกไฟล์',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'เลื่อนขึ้น',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'เลื่อนลง',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'กำลังโหลด...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'โหลดเพิ่มเติม',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'ถัดไป',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'ก่อนหน้า',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'เพิ่มค่าทั้งหมด',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'เพิ่มค่าที่เลือก',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'ลบค่าทั้งหมด',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'ลบค่าที่เลือก',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'เลือกวันที่',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'เปลี่ยนวันที่',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'วันที่ที่เลือกไม่ถูกต้อง',
    /**
     * Shown when there is something to close
     */
    close: 'ปิด',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$6 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `กรุณายอมรับ ${name}`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} จะต้องเป็นวันที่หลังจาก ${date(args[0])}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} จะต้องเป็นวันที่ที่ยังไม่มาถึง`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} สามารถเป็นได้แค่ตัวอักษรเท่านั้น`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} สามารถเป็นได้แค่ตัวอักษรและตัวเลขเท่านั้น`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} สามารถเป็นได้แค่ตัวอักษรและเว้นวรรคเท่านั้น`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} ต้องมีตัวอักษรตัวอักษร`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} ต้องมีตัวอักษรหรือตัวเลข`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} ต้องมีตัวอักษรหรือช่องว่าง`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} ต้องมีสัญลักษณ์`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} ต้องมีตัวพิมพ์ใหญ่`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} ต้องมีตัวพิมพ์เล็ก`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} ต้องมีตัวเลข`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} จะต้องเป็นสัญลักษณ์`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} เท่านั้นที่สามารถมีตัวอักษรตัวพิมพ์ใหญ่`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} เท่านั้นที่สามารถมีตัวอักษรตัวพิมพ์เล็ก`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} จะต้องเป็นวันที่ที่มาก่อน ${date(args[0])}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} จะต้องเป็นวันที่ที่ผ่านมาแล้ว`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `ช่องนี้ถูกตั้งค่าอย่างไม่ถูกต้อง และจะไม่สามารถส่งข้อมูลได้`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} จะต้องเป็นค่าระหว่าง ${a} และ ${b}`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} ไม่ตรงกัน`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} ไม่อยู่ในรูปแบบวันที่ที่ถูกต้อง กรุณากรอกตามรูปแบบ ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'ช่องนี้ถูกตั้งค่าอย่างไม่ถูกต้อง และจะไม่สามารถส่งข้อมูลได้';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} จะต้องเป็นวันที่ระหว่าง ${date(args[0])} และ ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'กรุณากรอกที่อยู่อีเมลทีถูกต้อง',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} จะต้องลงท้ายด้วย ${list(args)}`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} ไม่ใช่ค่าที่อนุญาตให้กรอก`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} จะต้องมีความยาวอย่างน้อยหนึ่งตัวอักษร`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} จะต้องมีความยาวไม่เกิน ${max} ตัวอักษร`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} ควรจะเป็น ${max} ตัวอักษรยาว`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} จะต้องมีความยาว ${min} ตัวอักษรขึ้นไป`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} จะต้องมีความยาวระหว่าง ${min} และ ${max} ตัวอักษร`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} ไม่ใช่ค่าที่อนุญาตให้กรอก`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `ไม่สามารถเลือกมากกว่า ${args[0]} ${name} ได้`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} จะต้องมีค่าไม่เกิน ${args[0]}`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'ไม่มีประเภทของไฟล์ที่อนุญาต';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} จะต้องเป็นไฟล์ประเภท ${args[0]} เท่านั้น`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `ไม่สามารถเลือกน้อยกว่า ${args[0]} ${name} ได้`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} จะต้องมีค่าอย่างน้อย ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” ไม่ใช่ค่า ${name} ที่อนุญาตให้กรอก`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} จะต้องเป็นตัวเลขเท่านั้น`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' หรือ ')} ต้องการ.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `กรุณากรอก ${sentence(name)}`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} จะต้องเริ่มต้นด้วย ${list(args)}`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `กรุณาระบุที่อยู่ลิงก์ให้ถูกต้อง`;
        /* </i18n> */
    },
};

var th = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$6,
    validation: validation$6
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$5 = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Ekle',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Kaldır',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Hepsini kaldır',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Maalesef, tüm alanlar doğru doldurulmadı.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Gönder',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Dosya yok',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Yukarı Taşı',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Aşağı taşı',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Yükleniyor...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Daha fazla yükle',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Sonraki',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Önceki',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Tüm değerleri ekle',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Seçili değerleri ekle',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Tüm değerleri kaldır',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Seçili değerleri kaldır',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Tarih seçin',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Tarihi değiştir',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Seçilen tarih geçersiz.',
    /**
     * Shown when there is something to close
     */
    close: 'Kapat',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$5 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Lütfen ${name}'yi kabul edin.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])}'den sonra olmalıdır.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} gelecekte bir zaman olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} sadece alfabetik karakterler içerebilir.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} sadece alfabetik karakterler ve sayı içerebilir.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} yalnızca harf ve boşluk içerebilir.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} alfabetik karakterler içermelidir.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} harf veya rakamı içermelidir.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} harf veya boşluk içermelidir.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} sembol içermelidir.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} büyük harf içermelidir.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} küçük harf içermelidir.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} sayı içermelidir.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} bir sembol olmalıdır.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} yalnızca büyük harfler içerebilir.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} yalnızca küçük harfler içerebilir.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])} tarihinden önce olmalı.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} geçmişte olmalı.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Alan yanlış yapılandırılmış ve gönderilemez.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} ${a} ve ${b} aralığında olmalı.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} eşleşmiyor.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} geçerli bir tarih değil, lütfen ${args[0]} biçimini kullanın.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Alan yanlış yapılandırılmış ve gönderilemez.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)}, ${date(args[0])} ve ${date(args[1])} aralığında olmalı.`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Lütfen geçerli bir e-mail adresi girin.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)} ile bitmiyor.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} izin verilen bir değer değil.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} en azından bir karakter uzunluğunda olmalı.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} ${max}'e eşit veya daha küçük olmalı.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} ${max} karakter uzunluğunda olmalıdır.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} ${min}'e eşit veya daha büyük olmalı.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)}, ${min} ve ${max} karakter uzunluğu aralığında olmalı.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} izin verilen bir değer değil.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${name}'in uzunluğu ${args[0]}'dan daha uzun olamaz.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} en azından ${args[0]} uzunluğunda veya ona eşit olmalı.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Hiçbir dosya türüne izin verilmez.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} şu tiplerden biri olmalı: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${name}'in uzunluğu ${args[0]}'dan daha kısa olamaz.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} en azından ${args[0]} uzunluğunda olmalı.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” ${name} olamaz.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} sayı olmalı.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' veya ')} gereklidir.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} gerekli.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)} ile başlamıyor.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Lütfen geçerli bir URL girin.`;
        /* </i18n> */
    },
};

var tr = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$5,
    validation: validation$5
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$4 = {
    /**
     * Shown on a button for adding additional items.
     */
    add: 'Додати',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Видалити',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Видалити все',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Вибачте, не всі поля заповнені правильно.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Відправити',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Файл не вибрано',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Рухатися вгору',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Пересунути вниз',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Завантаження...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Завантажте більше',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Наступний',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Попередній',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Додати всі значення',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Додати вибрані значення',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Вилучити всі значення',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Вилучити вибрані значення',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Виберіть дату',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Змінити дату',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Вибрана дата недійсна.',
    /**
     * Shown when there is something to close
     */
    close: 'Закрити',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$4 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Будь ласка, прийміть ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `Дата ${sentence(name)} повинна бути пізніше за ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `Дата ${sentence(name)} має бути в майбутньому.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `Поле ${sentence(name)} може містити лише літери.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `Поле ${sentence(name)} може містити лише літери та цифри.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `Поле ${sentence(name)} може містити лише літери та пробіли.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} повинен містити алфавітні символи.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} повинен містити букви або цифри.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} повинен містити літери або пробіли.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} повинен містити символ.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} повинен містити великі регістри.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} повинен містити малі регістри.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} повинен містити цифри.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} має бути символом.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} може містити лише великі літери.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} може містити лише малі літери.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `Дата ${sentence(name)} повинна бути раніше за ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `Дата ${sentence(name)} повинна бути в минулому.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Це поле заповнено неправильно і не може бути надіслано.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `Поле ${sentence(name)} повинно бути між ${a} та ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `Поле ${sentence(name)} не збігається.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `Поле ${sentence(name)} має неправильну дату. Будь ласка, використовуйте формат ${args[0]}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Це поле заповнено неправильно і не може бути надіслано.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `Дата ${sentence(name)} повинна бути між ${date(args[0])} та ${date(args[1])}.`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Будь ласка, введіть дійсну електронну адресу.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `Поле ${sentence(name)} не повинно закінчуватися на ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `Поле ${sentence(name)} має неприпустиме значення.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `Поле ${sentence(name)} має містити щонайменше один символ.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `Довжина поля ${sentence(name)} повинна бути меншою або дорівнювати ${max} символам.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} має бути довжиною ${max} символів.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `Довжина поля ${sentence(name)} повинна бути більшою або дорівнювати ${min} символам.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `Довжина поля ${sentence(name)} повинна бути між ${min} та ${max} символами.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `Поле ${sentence(name)} має неприпустиме значення.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `Не може бути вибрано більше ніж ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `Поле ${sentence(name)} має бути менше або дорівнювати ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Не вказано дозволені типи файлів.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `Тип файлу в полі ${sentence(name)} має бути: ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `Не може бути вибрано менше ніж ${args[0]} ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `Поле ${sentence(name)} має бути не менше ніж ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” не дозволено в полі ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `Поле ${sentence(name)} має бути числом.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' або ')} потрібно.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `Поле ${sentence(name)} є обов'язковим.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `Поле ${sentence(name)} має починатися з ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Будь ласка, введіть коректну URL-адресу.`;
        /* </i18n> */
    },
};

var uk = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$4,
    validation: validation$4
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$3 = {
    /**
     * Shown on a button for adding additional items.
     */
    add: "Qo'shish",
    /**
     * Shown when a button to remove items is visible.
     */
    remove: "O'chirish",
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: "Hammasini o'chirish",
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: "Kechirasiz, barcha maydonlar to'g'ri to'ldirilmagan.",
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Yuborish',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Hech qanday fayl tanlanmagan',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Yuqoriga ko’taring',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Pastga siljish',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Yuklanmoqda...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Ko’proq yuklang',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Keyingi',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Oldingi',
    /**
     * Shown when adding all values.
     */
    addAllValues: `Barcha qiymatlarni qo'shish`,
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: `Tanlangan qiymatlarni qoʻshish`,
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Barcha qiymatlarni olib tashlang',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Tanlangan qiymatlarni olib tashlash',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Sanani tanlang',
    /**
     * Shown when there is a date to change.
     */
    changeDate: `O'zgartirish sanasi`,
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Tanlangan sana yaroqsiz.',
    /**
     * Shown when there is something to close
     */
    close: 'Yopish',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$3 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `${name} ni qabul qiling.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])} dan keyin bo'lishi kerak.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} kelajakda bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} faqat alifbo tartibidagi belgilardan iborat bo'lishi mumkin.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} faqat harflar va raqamlardan iborat bo'lishi mumkin.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} faqat harf va bo'shliqlardan iborat bo'lishi mumkin.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} alfavit belgilarini o'z ichiga olishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} harflar yoki raqamlarni o'z ichiga olishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} harflar yoki bo'shliqlar bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} belgisi bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} katta harfni o'z ichiga olishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} kichik harflarni o'z ichiga olishi kerak.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} raqamlarini o'z ichiga olishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} belgisi bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} faqat katta harflarni o'z ichiga olishi mumkin.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} faqat kichik harflarni o'z ichiga olishi mumkin.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} ${date(args[0])} dan oldin bo'lishi kerak`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} o'tmishda bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Bu maydon noto'g'ri sozlangan va uni yuborib bo'lmaydi.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} ${a} va ${b} orasida bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} mos emas.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} haqiqiy sana emas, iltimos ${args[0]} formatidan foydalaning`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return "Bu maydon noto'g'ri sozlangan va uni yuborib bo'lmaydi";
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} ${date(args[0])} va ${date(args[1])} oralig'ida bo'lishi kerak`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Iltimos amaldagi e-mail manzilingizni kiriting.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)} bilan tugamaydi.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} ruxsat etilgan qiymat emas.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} kamida bitta belgidan iborat bo'lishi kerak.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} ${max} ta belgidan kam yoki teng bo'lishi kerak.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} bo'lishi kerak ${max} belgilar uzun.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} ${min} ta belgidan ko'p yoki teng bo'lishi kerak.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} ${min} va ${max} gacha belgilardan iborat bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} ruxsat etilgan qiymat emas.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${args[0]} ${name} dan ortiq bo'lishi mumkin emas.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} ${args[0]} dan kichik yoki teng bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Fayl formatlariga ruxsat berilmagan.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} quyidagi turdagi bo'lishi kerak: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${args[0]} ${name} dan kam bo'lmasligi kerak.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} kamida ${args[0]} bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" ruxsat berilmagan ${name}.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} raqam bo'lishi kerak.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' yoki ')} kerak.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} talab qilinadi.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} ${list(args)} bilan boshlanmaydi.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Iltimos, tegishli URL manzilini kiriting.`;
        /* </i18n> */
    },
};

var uz = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$3,
    validation: validation$3
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$2 = {
    /**
     * Shown on buttons for adding new items.
     */
    add: 'Thêm',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: 'Xoá',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: 'Xoá tất cả',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: 'Xin lỗi, không phải tất cả các trường đều được nhập đúng.',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: 'Gửi',
    /**
     * Shown when no files are selected.
     */
    noFiles: 'Chưa chọn file',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: 'Di chuyển lên',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: 'Di chuyển xuống',
    /**
     * Shown when something is actively loading.
     */
    isLoading: 'Đang tải...',
    /**
     * Shown when there is more to load.
     */
    loadMore: 'Tải thêm',
    /**
     * Shown on buttons that navigate state forward
     */
    next: 'Tiếp',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: 'Trước',
    /**
     * Shown when adding all values.
     */
    addAllValues: 'Thêm tất cả các giá trị',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: 'Thêm các giá trị đã chọn',
    /**
     * Shown when removing all values.
     */
    removeAllValues: 'Loại bỏ tất cả các giá trị',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: 'Loại bỏ các giá trị đã chọn',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: 'Chọn ngày',
    /**
     * Shown when there is a date to change.
     */
    changeDate: 'Thay đổi ngày',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: 'Ngày đã chọn không hợp lệ.',
    /**
     * Shown when there is something to close
     */
    close: 'Đóng',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$2 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `Hãy đồng ý với ${name}.`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} phải sau ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} phải trong tương lai.`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} có thể chỉ bao gồm các chữ cái alphabet.`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} có thể chỉ bao gồm các chữ cái và chữ số.`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} chỉ có thể chứa các chữ cái và khoảng trắng.`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} phải chứa các ký tự chữ cái.`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} phải chứa chữ cái hoặc số.`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} phải chứa chữ cái hoặc dấu cách.`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} phải chứa ký hiệu.`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} phải chứa chữ hoa.`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} phải chứa chữ thường.`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} phải chứa số.`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} phải là một ký hiệu.`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} chỉ có thể chứa chữ hoa.`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} chỉ có thể chứa chữ thường.`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} phải trước ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} phải trong quá khứ.`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `Trường này đã được thiết lập sai và không thể gửi.`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} phải ở giữa ${a} và ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} không khớp.`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} không phải ngày hợp lệ, hãy sử dụng định dạng ${args[0]}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return 'Trường này đã được thiết lập sai và không thể gửi.';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} phải ở giữa khoảng từ ${date(args[0])} đến ${date(args[1])}.`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: 'Hãy nhập một địa chỉ email hợp lệ.',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} không kết thúc với ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} không phải một giá trị được cho phép.`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} phải có độ dài tối thiểu một ký tự.`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} phải có độ dài tối đa ${max} ký tự.`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} nên dài ${max} ký tự.`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} phải có độ dài tối thiểu ${min} ký tự.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} phải có độ dài tối đa trong khoảng từ ${min} đến ${max} ký tự.`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} không phải một giá trị được cho phép.`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${name} không thể lớn hơn ${args[0]}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} phải tối đa bằng ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return 'Định dạng tệp tin này không được phép.';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} phải là một trong các dạng: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${name} không thể nhỏ hơn ${args[0]}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)} phải tối thiểu bằng ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}" không phải giá trị ${name} được phép.`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} phải là một số.`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join(' hoặc ')} cần có.`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} là bắt buộc.`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} không bắt đầu với ${list(args)}.`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `Vui lòng nhập một URL hợp lệ.`;
        /* </i18n> */
    },
};

var vi = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$2,
    validation: validation$2
});

/**
 * Standard language for interface features.
 * @public
 */
const ui$1 = {
    /**
     * Shown on a button for adding additional items.
     */
    add: '添加',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: '移除',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: '移除全部',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: '抱歉，部分字段未被正确填写。',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: '提交',
    /**
     * Shown when no files are selected.
     */
    noFiles: '未选择文件',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: '上移',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: '下移',
    /**
     * Shown when something is actively loading.
     */
    isLoading: '加载中...',
    /**
     * Shown when there is more to load.
     */
    loadMore: '加载更多',
    /**
     * Shown on buttons that navigate state forward
     */
    next: '下一步',
    /**
     * Shown on buttons that navigate state backward
     */
    prev: '上一步',
    /**
     * Shown when adding all values.
     */
    addAllValues: '添加所有值',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: '添加所选值',
    /**
     * Shown when removing all values.
     */
    removeAllValues: '移除所有值',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: '移除所选值',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: '选择日期',
    /**
     * Shown when there is a date to change.
     */
    changeDate: '更改日期',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: '所选日期无效。',
    /**
     * Shown when there is something to close
     */
    close: '关闭',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation$1 = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `请接受${name}。`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)}必须晚于${date(args[0])}。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)}必须是未来的日期。`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)}只能包含英文字母。`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)}只能包含字母和数字。`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)}只能包含字母和空格。`;
        /* </i18n> */
    },
    /**
     * The value have no letter.
     * @see {@link https://formkit.com/essentials/validation#contains_alpha}
     */
    contains_alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical characters."> */
        return `${sentence(name)} 必须包含字母字符`;
        /* </i18n> */
    },
    /**
     * The value have no alphanumeric
     * @see {@link https://formkit.com/essentials/validation#contains_alphanumeric}
     */
    contains_alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphanumeric characters."> */
        return `${sentence(name)} 必须包含字母或数字。`;
        /* </i18n> */
    },
    /**
     * The value have no letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#contains_alpha-spaces}
     */
    contains_alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical and non-space characters."> */
        return `${sentence(name)} 必须包含字母或空格。`;
        /* </i18n> */
    },
    /**
     * The value have no symbol
     * @see {@link https://formkit.com/essentials/validation#contains_symbol}
     */
    contains_symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains only alphanumeric and space characters."> */
        return `${sentence(name)} 必须包含符号。`;
        /* </i18n> */
    },
    /**
     * The value have no uppercase
     * @see {@link https://formkit.com/essentials/validation#contains_uppercase}
     */
    contains_uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-uppercase characters."> */
        return `${sentence(name)} 必须包含大写字母。`;
        /* </i18n> */
    },
    /**
     * The value have no lowercase
     * @see {@link https://formkit.com/essentials/validation#contains_lowercase}
     */
    contains_lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains only non-alphabetical-lowercase characters."> */
        return `${sentence(name)} 必须包含小写字母。`;
        /* </i18n> */
    },
    /**
     *  The value have no numeric
     * @see {@link https://formkit.com/essentials/validation#contains_numeric}
     */
    contains_numeric({ name }) {
        /* <i18n case="Shown when the user-provided value have no numeric."> */
        return `${sentence(name)} 必须包含数字。`;
        /* </i18n> */
    },
    /**
     * The value is not symbol
     * @see {@link https://formkit.com/essentials/validation#symbol}
     */
    symbol({ name }) {
        /* <i18n case="Shown when the user-provided value contains alphanumeric and space characters."> */
        return `${sentence(name)} 必须是符号。`;
        /* </i18n> */
    },
    /**
     * The value is not uppercase
     * @see {@link https://formkit.com/essentials/validation#uppercase}
     */
    uppercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-uppercase characters."> */
        return `${sentence(name)} 只能包含大写字母。`;
        /* </i18n> */
    },
    /**
     * The value is not lowercase
     * @see {@link https://formkit.com/essentials/validation#lowercase}
     */
    lowercase({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical-lowercase characters."> */
        return `${sentence(name)} 只能包含小写字母。`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)}必须早于${date(args[0])}。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)}必须是过去的日期。`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `该字段未被正确设置而无法提交。`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)}必须在${a}和${b}之间。`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)}不匹配。`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)}不是合法日期，请使用 ${args[0]} 格式`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return '该字段未被正确设置而无法提交';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)}必须在${date(args[0])}和${date(args[1])}之间`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: '请输入合法的电子邮件地址。',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)}必须以${list(args)}结尾。`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)}是不允许的。`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)}至少要有一个字符。`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)}必须少于或等于${max}个字符。`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)}必须包含${max}个字符。`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)}必须多于或等于${min}个字符。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)}必须介于${min}和${max}个字符之间。`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)}是不允许的。`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `${name}不得多于${args[0]}个值。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${name}不得大于${args[0]}。`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return '没有允许的文件格式。';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)}的类型必须为：${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `${name}不得少于${args[0]}个值。`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${sentence(name)}不得小于${args[0]}。`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `"${value}"不是一个合法的${name}。`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)}必须为数字。`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join('或')}${labels}需要。`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)}不得留空。`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)}必须以${list(args)}开头。`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `请输入有效的 URL。`;
        /* </i18n> */
    },
};

var zh = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui$1,
    validation: validation$1
});

/**
 * Standard language for interface features.
 * @public
 */
const ui = {
    /**
     * Shown on a button for adding additional items.
     */
    add: '新增',
    /**
     * Shown when a button to remove items is visible.
     */
    remove: '移除',
    /**
     * Shown when there are multiple items to remove at the same time.
     */
    removeAll: '移除全部',
    /**
     * Shown when all fields are not filled out correctly.
     */
    incomplete: '很抱歉，部分欄位填寫錯誤',
    /**
     * Shown in a button inside a form to submit the form.
     */
    submit: '提交',
    /**
     * Shown when no files are selected.
     */
    noFiles: '尚未選取檔案',
    /**
     * Shown on buttons that move fields up in a list.
     */
    moveUp: '上移',
    /**
     * Shown on buttons that move fields down in a list.
     */
    moveDown: '下移',
    /**
     * Shown when something is actively loading.
     */
    isLoading: '載入中...',
    /**
     * Shown when there is more to load.
     */
    loadMore: '載入更多',
    /**
     * Show on buttons that navigate state forward
     */
    next: '下一個',
    /**
     * Show on buttons that navigate state backward
     */
    prev: '上一個',
    /**
     * Shown when adding all values.
     */
    addAllValues: '加入全部的值',
    /**
     * Shown when adding selected values.
     */
    addSelectedValues: '加入選取的值',
    /**
     * Shown when removing all values.
     */
    removeAllValues: '移除全部的值',
    /**
     * Shown when removing selected values.
     */
    removeSelectedValues: '移除選取的值',
    /**
     * Shown when there is a date to choose.
     */
    chooseDate: '選擇日期',
    /**
     * Shown when there is a date to change.
     */
    changeDate: '變更日期',
    /**
     * Shown when the date is invalid.
     */
    invalidDate: '選取的日期無效',
    /**
     * Shown when there is something to close
     */
    close: '關閉',
};
/**
 * These are all the possible strings that pertain to validation messages.
 * @public
 */
const validation = {
    /**
     * The value is not an accepted value.
     * @see {@link https://formkit.com/essentials/validation#accepted}
     */
    accepted({ name }) {
        /* <i18n case="Shown when the user-provided value is not a valid 'accepted' value."> */
        return `請接受 ${name}`;
        /* </i18n> */
    },
    /**
     * The date is not after
     * @see {@link https://formkit.com/essentials/validation#date-after}
     */
    date_after({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not after the date supplied to the rule."> */
            return `${sentence(name)} 必須晚於 ${date(args[0])}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not after today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} 必須晚於今日`;
        /* </i18n> */
    },
    /**
     * The value is not a letter.
     * @see {@link https://formkit.com/essentials/validation#alpha}
     */
    alpha({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical characters."> */
        return `${sentence(name)} 欄位儘能填寫英文字母`;
        /* </i18n> */
    },
    /**
     * The value is not alphanumeric
     * @see {@link https://formkit.com/essentials/validation#alphanumeric}
     */
    alphanumeric({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphanumeric characters."> */
        return `${sentence(name)} 欄位僅能填寫英文字母與數字`;
        /* </i18n> */
    },
    /**
     * The value is not letter and/or spaces
     * @see {@link https://formkit.com/essentials/validation#alpha-spaces}
     */
    alpha_spaces({ name }) {
        /* <i18n case="Shown when the user-provided value contains non-alphabetical and non-space characters."> */
        return `${sentence(name)} 欄位儘能填寫英文字母與空白`;
        /* </i18n> */
    },
    /**
     * The date is not before
     * @see {@link https://formkit.com/essentials/validation#date-before}
     */
    date_before({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date is not before the date supplied to the rule."> */
            return `${sentence(name)} 必須早於 ${date(args[0])}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided date is not before today's date, since no date was supplied to the rule."> */
        return `${sentence(name)} 必須早於今日`;
        /* </i18n> */
    },
    /**
     * The value is not between two numbers
     * @see {@link https://formkit.com/essentials/validation#between}
     */
    between({ name, args }) {
        if (isNaN(args[0]) || isNaN(args[1])) {
            /* <i18n case="Shown when any of the arguments supplied to the rule were not a number."> */
            return `欄位值錯誤，無法提交`;
            /* </i18n> */
        }
        const [a, b] = order(args[0], args[1]);
        /* <i18n case="Shown when the user-provided value is not between two numbers."> */
        return `${sentence(name)} 必須介於 ${a} 和 ${b}.`;
        /* </i18n> */
    },
    /**
     * The confirmation field does not match
     * @see {@link https://formkit.com/essentials/validation#confirm}
     */
    confirm({ name }) {
        /* <i18n case="Shown when the user-provided value does not equal the value of the matched input."> */
        return `${sentence(name)} 與目標不一致`;
        /* </i18n> */
    },
    /**
     * The value is not a valid date
     * @see {@link https://formkit.com/essentials/validation#date-format}
     */
    date_format({ name, args }) {
        if (Array.isArray(args) && args.length) {
            /* <i18n case="Shown when the user-provided date does not satisfy the date format supplied to the rule."> */
            return `${sentence(name)} 不是有效的日期，請使用 ${args[0]} 格式`;
            /* </i18n> */
        }
        /* <i18n case="Shown when no date argument was supplied to the rule."> */
        return '欄位值錯誤，無法提交';
        /* </i18n> */
    },
    /**
     * Is not within expected date range
     * @see {@link https://formkit.com/essentials/validation#date-between}
     */
    date_between({ name, args }) {
        /* <i18n case="Shown when the user-provided date is not between the start and end dates supplied to the rule. "> */
        return `${sentence(name)} 必須介於 ${date(args[0])} 和 ${date(args[1])}`;
        /* </i18n> */
    },
    /**
     * Shown when the user-provided value is not a valid email address.
     * @see {@link https://formkit.com/essentials/validation#email}
     */
    email: '請輸入有效的 email',
    /**
     * Does not end with the specified value
     * @see {@link https://formkit.com/essentials/validation#ends-with}
     */
    ends_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not end with the substring supplied to the rule."> */
        return `${sentence(name)} 的結尾必須是 ${list(args)}`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#is}
     */
    is({ name }) {
        /* <i18n case="Shown when the user-provided value is not one of the values supplied to the rule."> */
        return `${sentence(name)} 欄位的值不合規則`;
        /* </i18n> */
    },
    /**
     * Does not match specified length
     * @see {@link https://formkit.com/essentials/validation#length}
     */
    length({ name, args: [first = 0, second = Infinity] }) {
        const min = Number(first) <= Number(second) ? first : second;
        const max = Number(second) >= Number(first) ? second : first;
        if (min == 1 && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is not at least one character."> */
            return `${sentence(name)} 欄位必須至少包含一個字`;
            /* </i18n> */
        }
        if (min == 0 && max) {
            /* <i18n case="Shown when first argument supplied to the rule is 0, and the user-provided value is longer than the max (the 2nd argument) supplied to the rule."> */
            return `${sentence(name)} 的字數必須小於等於 ${max}`;
            /* </i18n> */
        }
        if (min === max) {
            /* <i18n case="Shown when first and second argument supplied to the rule are the same, and the user-provided value is not any of the arguments supplied to the rule."> */
            return `${sentence(name)} 的字數必須為 ${max}`;
            /* </i18n> */
        }
        if (min && max === Infinity) {
            /* <i18n case="Shown when the length of the user-provided value is less than the minimum supplied to the rule and there is no maximum supplied to the rule."> */
            return `${sentence(name)} 的字數必須大於等於 ${min}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the length of the user-provided value is between the two lengths supplied to the rule."> */
        return `${sentence(name)} 的字數必須介於 ${min} 和 ${max}`;
        /* </i18n> */
    },
    /**
     * Value is not a match
     * @see {@link https://formkit.com/essentials/validation#matches}
     */
    matches({ name }) {
        /* <i18n case="Shown when the user-provided value does not match any of the values or RegExp patterns supplied to the rule. "> */
        return `${sentence(name)} 欄位的值無效`;
        /* </i18n> */
    },
    /**
     * Exceeds maximum allowed value
     * @see {@link https://formkit.com/essentials/validation#max}
     */
    max({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is longer than the max supplied to the rule."> */
            return `不能超過 ${args[0]} 個 ${name}.`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is greater than the maximum number supplied to the rule."> */
        return `${sentence(name)} 必須小於等於 ${args[0]}.`;
        /* </i18n> */
    },
    /**
     * The (field-level) value does not match specified mime type
     * @see {@link https://formkit.com/essentials/validation#mime}
     */
    mime({ name, args }) {
        if (!args[0]) {
            /* <i18n case="Shown when no file formats were supplied to the rule."> */
            return '非有效的檔案格式';
            /* </i18n> */
        }
        /* <i18n case="Shown when the mime type of user-provided file does not match any mime types supplied to the rule."> */
        return `${sentence(name)} 可接受的檔案格式為: ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Does not fulfill minimum allowed value
     * @see {@link https://formkit.com/essentials/validation#min}
     */
    min({ name, node: { value }, args }) {
        if (Array.isArray(value)) {
            /* <i18n case="Shown when the length of the array of user-provided values is shorter than the min supplied to the rule."> */
            return `不可少於 ${args[0]} 個 ${name}`;
            /* </i18n> */
        }
        /* <i18n case="Shown when the user-provided value is less than the minimum number supplied to the rule."> */
        return `${name} 必須大於等於 ${args[0]}`;
        /* </i18n> */
    },
    /**
     * Is not an allowed value
     * @see {@link https://formkit.com/essentials/validation#not}
     */
    not({ name, node: { value } }) {
        /* <i18n case="Shown when the user-provided value matches one of the values supplied to (and thus disallowed by) the rule."> */
        return `“${value}” 不是 ${name} 欄位可接受的值`;
        /* </i18n> */
    },
    /**
     *  Is not a number
     * @see {@link https://formkit.com/essentials/validation#number}
     */
    number({ name }) {
        /* <i18n case="Shown when the user-provided value is not a number."> */
        return `${sentence(name)} 欄位必須是數字`;
        /* </i18n> */
    },
    /**
     * Require one field.
     * @see {@link https://formkit.com/essentials/validation#require-one}
     */
    require_one: ({ name, node, args: inputNames }) => {
        const labels = inputNames
            .map((name) => {
            const dependentNode = node.at(name);
            if (dependentNode) {
                return validation$K.createMessageName(dependentNode);
            }
            return false;
        })
            .filter((name) => !!name);
        labels.unshift(name);
        /* <i18n case="Shown when the user-provided has not provided a value for at least one of the required fields."> */
        return `${labels.join('或')}${labels}需要。`;
        /* </i18n> */
    },
    /**
     * Required field.
     * @see {@link https://formkit.com/essentials/validation#required}
     */
    required({ name }) {
        /* <i18n case="Shown when a user does not provide a value to a required input."> */
        return `${sentence(name)} 是必要欄位`;
        /* </i18n> */
    },
    /**
     * Does not start with specified value
     * @see {@link https://formkit.com/essentials/validation#starts-with}
     */
    starts_with({ name, args }) {
        /* <i18n case="Shown when the user-provided value does not start with the substring supplied to the rule."> */
        return `${sentence(name)} 的開頭必須是 ${list(args)}`;
        /* </i18n> */
    },
    /**
     * Is not a url
     * @see {@link https://formkit.com/essentials/validation#url}
     */
    url() {
        /* <i18n case="Shown when the user-provided value is not a valid url."> */
        return `請輸入有效的 url`;
        /* </i18n> */
    },
};

var zhTW = /*#__PURE__*/Object.freeze({
    __proto__: null,
    ui: ui,
    validation: validation
});

/**
 * Create a new internationalization plugin for FormKit.
 *
 * @param registry - The registry of {@link @formkit/i18n#FormKitLocaleRegistry | FormKitLocales}.
 *
 * @returns {@link @formkit/core#FormKitPlugin | FormKitPlugin}
 *
 * @public
 */
function createI18nPlugin(registry) {
    return function i18nPlugin(node) {
        let localeKey = parseLocale(node.config.locale, registry);
        let locale = localeKey ? registry[localeKey] : {};
        /* If the locale prop changes, update the active locale */
        node.on('prop:locale', ({ payload: lang }) => {
            localeKey = parseLocale(lang, registry);
            locale = localeKey ? registry[localeKey] : {};
            // Run through all the messages in the store and update with new locale
            node.store.touch();
        });
        node.on('prop:label', () => node.store.touch());
        node.on('prop:validationLabel', () => node.store.touch());
        /**
         * Hook into the core text or t() hook to perform localization on the
         * output of core functionality.
         */
        node.hook.text((fragment, next) => {
            var _a, _b;
            const key = ((_a = fragment.meta) === null || _a === void 0 ? void 0 : _a.messageKey) || fragment.key;
            if (utils.has(locale, fragment.type) && utils.has(locale[fragment.type], key)) {
                const t = locale[fragment.type][key];
                if (typeof t === 'function') {
                    fragment.value = Array.isArray((_b = fragment.meta) === null || _b === void 0 ? void 0 : _b.i18nArgs)
                        ? t(...fragment.meta.i18nArgs) // eslint-disable-line @typescript-eslint/no-non-null-assertion
                        : t(fragment);
                }
                else {
                    fragment.value = t;
                }
            }
            return next(fragment);
        });
    };
}
/**
 * Parse ISO 639-1 and 639-2 to a valid locale key.
 *
 * @param locale - An ISO 639-1 and (optionally) ISO 639-2 language tag. For these are valid locale keys:
 * zh
 * zh-CN
 * zh-HK
 * en
 * en-GB
 *
 * @param availableLocales - An array of locales that may be valid.
 *
 * @public
 */
function parseLocale(locale, availableLocales) {
    if (utils.has(availableLocales, locale)) {
        return locale;
    }
    const [lang] = locale.split('-');
    if (utils.has(availableLocales, lang)) {
        return lang;
    }
    for (const locale in availableLocales) {
        return locale;
    }
    return false;
}

/**
 * The official FormKit internationalization (i18n) plugin. This package
 * contains the locales and the plugin that integrates FormKit with these
 * locales. Read the {@link https://formkit.com/essentials/internationalization
 * | internationalization documentation} for usage instructions.
 *
 * @packageDocumentation
 */
/**
 * Export all the available locales at once.
 *
 * @public
 */
const locales = {
    ar,
    az,
    bg,
    bs,
    ca,
    cs,
    da,
    de,
    el,
    en,
    es,
    fa,
    fi,
    fr,
    fy,
    he,
    hr,
    hu,
    id,
    it,
    ja,
    kk,
    ko,
    lt,
    lv,
    nb,
    nl,
    pl,
    pt,
    ro,
    ru,
    sk,
    sl,
    sr,
    sv,
    tet,
    tg,
    th,
    tr,
    uk,
    uz,
    vi,
    zh,
    'zh-TW': zhTW,
    is,
    mn,
};

exports.ar = ar;
exports.az = az;
exports.bg = bg;
exports.bs = bs;
exports.ca = ca;
exports.createI18nPlugin = createI18nPlugin;
exports.cs = cs;
exports.da = da;
exports.date = date;
exports.de = de;
exports.el = el;
exports.en = en;
exports.es = es;
exports.fa = fa;
exports.fi = fi;
exports.fr = fr;
exports.fy = fy;
exports.he = he;
exports.hr = hr;
exports.hu = hu;
exports.id = id;
exports.is = is;
exports.it = it;
exports.ja = ja;
exports.kk = kk;
exports.ko = ko;
exports.list = list;
exports.locales = locales;
exports.lt = lt;
exports.lv = lv;
exports.mn = mn;
exports.nb = nb;
exports.nl = nl;
exports.order = order;
exports.pl = pl;
exports.pt = pt;
exports.ro = ro;
exports.ru = ru;
exports.sentence = sentence;
exports.sk = sk;
exports.sl = sl;
exports.sr = sr;
exports.sv = sv;
exports.tet = tet;
exports.tg = tg;
exports.th = th;
exports.tr = tr;
exports.uk = uk;
exports.uz = uz;
exports.vi = vi;
exports.zh = zh;
exports.zhTW = zhTW;
