import { has } from '@formkit/utils';
import { isNode } from '@formkit/core';

/**
 * FormKit Observer is a utility to wrap a FormKitNode in a dependency tracking observer proxy.
 *
 * @packageDocumentation
 */
/**
 * A registry of all revoked observers.
 */
const revokedObservers = new WeakSet();
/**
 * Creates the observer.
 * @param node - The {@link @formkit/core#FormKitNode | FormKitNode} to observe.
 * @param dependencies - The dependent nodes and the events that are required to
 * watch for changes.
 * @returns Returns a {@link @formkit/observer#FormKitObservedNode | FormKitObservedNode}.
 * @public
 */
function createObserver(node, dependencies) {
    // The dependencies touched during tracking
    const deps = dependencies || Object.assign(new Map(), { active: false });
    // A registry of event receipts returned by the event system
    const receipts = new Map();
    /**
     * Simple function to add a dependency to the deps map.
     * @param event - The name of the event type (like commit/input etc)
     */
    const addDependency = function (event) {
        var _a;
        if (!deps.active)
            return;
        if (!deps.has(node))
            deps.set(node, new Set());
        (_a = deps.get(node)) === null || _a === void 0 ? void 0 : _a.add(event);
    };
    /**
     * Proxies the props of a node so we know which ones were messed with, could
     * potentially be more generalized in the future if we want to support
     * more sub-objects.
     * @param props - The props object from a node
     * @returns
     */
    const observeProps = function (props) {
        return new Proxy(props, {
            get(...args) {
                typeof args[1] === 'string' && addDependency(`prop:${args[1]}`);
                return Reflect.get(...args);
            },
        });
    };
    /**
     * Observes the FormKit ledger "value".
     * @param ledger - A formkit ledger counter.
     */
    const observeLedger = function (ledger) {
        return new Proxy(ledger, {
            get(...args) {
                if (args[1] === 'value') {
                    return (key) => {
                        addDependency(`count:${key}`);
                        return ledger.value(key);
                    };
                }
                return Reflect.get(...args);
            },
        });
    };
    /**
     * Return values from our observer proxy first pass through this function
     * which gives us a chance to listen sub-dependencies and properties.
     */
    const observe = function (value, property) {
        if (isNode(value)) {
            return createObserver(value, deps);
        }
        if (property === 'value')
            addDependency('commit');
        if (property === '_value')
            addDependency('input');
        if (property === 'props')
            return observeProps(value);
        if (property === 'ledger')
            return observeLedger(value);
        return value;
    };
    /**
     * The actual proxy object of the original node.
     */
    const { proxy: observed, revoke, } = Proxy.revocable(node, {
        get(...args) {
            switch (args[1]) {
                case '_node':
                    return node;
                case 'deps':
                    return deps;
                case 'watch':
                    return (block, after) => watch(observed, block, after);
                case 'observe':
                    return () => {
                        const old = new Map(deps);
                        deps.clear();
                        deps.active = true;
                        return old;
                    };
                case 'stopObserve':
                    return () => {
                        const newDeps = new Map(deps);
                        deps.active = false;
                        return newDeps;
                    };
                case 'receipts':
                    return receipts;
                case 'kill':
                    return () => {
                        removeListeners(receipts);
                        revokedObservers.add(args[2]);
                        revoke();
                        return undefined;
                    };
            }
            const value = Reflect.get(...args);
            // If we're dealing with a function, we need to sub-call the function
            // get that return value, and pass it through the same logic.
            if (typeof value === 'function') {
                return (...subArgs) => {
                    const subValue = value(...subArgs);
                    return observe(subValue, args[1]);
                };
            }
            return observe(value, args[1]);
        },
    });
    return observed;
}
/**
 * Given two maps (`toAdd` and `toRemove`), apply the dependencies as event
 * listeners on the underlying nodes.
 * @param node - The node to apply dependencies to.
 * @param deps - A tuple of toAdd and toRemove FormKitDependencies maps.
 * @param callback - The callback to add or remove.
 * @internal
 */
function applyListeners(node, [toAdd, toRemove], callback) {
    toAdd.forEach((events, depNode) => {
        events.forEach((event) => {
            var _a;
            node.receipts.has(depNode) || node.receipts.set(depNode, {});
            node.receipts.set(depNode, Object.assign((_a = node.receipts.get(depNode)) !== null && _a !== void 0 ? _a : {}, {
                [event]: depNode.on(event, callback),
            }));
        });
    });
    toRemove.forEach((events, depNode) => {
        events.forEach((event) => {
            if (node.receipts.has(depNode)) {
                const nodeReceipts = node.receipts.get(depNode);
                if (nodeReceipts && has(nodeReceipts, event)) {
                    depNode.off(nodeReceipts[event]);
                    delete nodeReceipts[event];
                    node.receipts.set(depNode, nodeReceipts);
                }
            }
        });
    });
}
/**
 * Remove all the receipts from the observed node and subtree.
 * @param receipts - The FormKit observer receipts to remove.
 * @public
 */
function removeListeners(receipts) {
    receipts.forEach((events, node) => {
        for (const event in events) {
            node.off(events[event]);
        }
    });
}
/**
 * Observes a chunk of code to dependencies, and then re-calls that chunk of
 * code when those dependencies are manipulated.
 * @param node - The node to observer
 * @param block - The block of code to observe
 * @param after - A function to call after a effect has been run.
 * @public
 */
function watch(node, block, after) {
    const doAfterObservation = (res) => {
        const newDeps = node.stopObserve();
        applyListeners(node, diffDeps(oldDeps, newDeps), () => watch(node, block, after));
        if (after)
            after(res);
    };
    const oldDeps = new Map(node.deps);
    node.observe();
    const res = block(node);
    if (res instanceof Promise)
        res.then((val) => doAfterObservation(val));
    else
        doAfterObservation(res);
}
/**
 * Determines which nodes should be added as dependencies and which should be
 * removed.
 * @param previous - The previous watcher dependencies.
 * @param current - The new/current watcher dependencies.
 * @returns A tuple of maps: `toAdd` and `toRemove`.
 * @public
 */
function diffDeps(previous, current) {
    const toAdd = new Map();
    const toRemove = new Map();
    current.forEach((events, node) => {
        if (!previous.has(node)) {
            toAdd.set(node, events);
        }
        else {
            const eventsToAdd = new Set();
            const previousEvents = previous.get(node);
            events.forEach((event) => !(previousEvents === null || previousEvents === void 0 ? void 0 : previousEvents.has(event)) && eventsToAdd.add(event));
            toAdd.set(node, eventsToAdd);
        }
    });
    previous.forEach((events, node) => {
        if (!current.has(node)) {
            toRemove.set(node, events);
        }
        else {
            const eventsToRemove = new Set();
            const newEvents = current.get(node);
            events.forEach((event) => !(newEvents === null || newEvents === void 0 ? void 0 : newEvents.has(event)) && eventsToRemove.add(event));
            toRemove.set(node, eventsToRemove);
        }
    });
    return [toAdd, toRemove];
}
/**
 * Checks if the given node is revoked.
 * @param node - Any observed node to check.
 * @returns A `boolean` indicating if the node is revoked.
 * @public
 */
function isKilled(node) {
    return revokedObservers.has(node);
}

export { applyListeners, createObserver, diffDeps, isKilled, removeListeners };
